#include <string.h>
#include <time.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "cdialog.h"
#include "obj.h"
#include "win.h"
#include "wincb.h"
#include "winlist.h"
#include "winfio.h"
#include "windnd.h"
#include "core.h"
#include "config.h"


static void WinDragDataReceivedNexus(
	win_struct *win,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	obj_struct *tar_obj_parent, obj_struct *tar_obj_sibling
);

void WinListDNDSetIcon(
	win_struct *win, gint row, gint column
);
void WinListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
);
void WinListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,                       
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
);
void WinListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);

void WinTreeDNDSetIcon(win_struct *win, GtkCTreeNode *node);
void WinTreeDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void WinTreeDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
void WinTreeDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Win "drag_data_received" signal nexus.
 */
static void WinDragDataReceivedNexus(
	win_struct *win,
	GdkDragContext *dc, guint info, GtkSelectionData *selection_data,
	obj_struct *tar_obj_parent, obj_struct *tar_obj_sibling
)
{
	gboolean obj_list_is_ref = TRUE;
	gint tar_sibling_index;
	gulong cur_time;
	GList *glist, *obj_list = NULL, *obj_del_list = NULL;
	GtkWidget *toplevel = win->toplevel;
	obj_struct *obj, *src_obj, *obj_parent;

#define FREE_ALL		{	\
 g_list_free(obj_del_list);		\
					\
 if(obj_list != NULL) {			\
  /* Delete each source Object if they	\
   * are not references			\
   */					\
  if(!obj_list_is_ref)			\
   g_list_foreach(			\
    obj_list, (GFunc)ObjDelete, NULL	\
   );					\
  g_list_free(obj_list);		\
 }					\
					\
}

	if(!OBJ_IS_GROUP(tar_obj_parent))
	    return;

	/* Get index of the specified sibling Object in the parent
	 * Object's list of child Objects
	 */
	if(tar_obj_sibling != NULL)
	{
	    glist = OBJ_CHILDREN(tar_obj_parent);
	    tar_sibling_index = (glist != NULL) ?
		g_list_index(glist, tar_obj_sibling) : -1;
	}
	else
	{
	    tar_sibling_index = -1;
	}


	WinSetBusy(win, TRUE);

	/* Parse the buffer into a list of Objects based on the
	 * target type
	 */
	/* Menu Editor Item pointer */
	if(info == DND_INFO_MEDIT_ITEM)
	{
	    /* Parse buffer into a list of references to Objects */
	    obj_list = ObjDDEBufferParse(
		(const guint8 *)selection_data->data,
		selection_data->length
	    );
	}
	/* Location string */
	else if((info == DND_INFO_TEXT_PLAIN) ||
	        (info == DND_INFO_TEXT_URI_LIST) ||
		(info == DND_INFO_STRING)
	)
	{
	    /* Parse buffer and create a list of new Objects from the
	     * data string
	     */
	    const gchar *s = (const gchar *)selection_data->data,
			*s_end = (const gchar *)(s + selection_data->length),
			*s2;
	    gchar *v;
	    gint len;

	    /* Iterate through string */
	    while(s < s_end)
	    {
		/* Seek s2 to the end of this string segment */
		for(s2 = s; s2 < s_end; s2++)
		{
		    if(*s2 == '\0')
			break;
		}

		/* Get length of this string segment and copy it to
		 * a new string v
		 */
		len = (gint)(s2 - s);
		v = (len > 0) ?
		    (gchar *)g_malloc((len + 1) * sizeof(gchar)) : NULL;
		if(v != NULL)
		{
		    gchar *name, *cmd, *v2, *v3;

		    memcpy(v, s, len);
		    v[len] = '\0';

		    /* Get command */
		    if(g_strcasepfx(v, "file://"))
		    {
			for(v2 = (gchar *)(v + STRLEN("file://"));
			    *v2 != '\0';
			    v2++
			)
			{
			    if(*v2 == '/')
				break;
			}
		    }
		    else
			v2 = v;
		    cmd = STRDUP(v2);

		    /* Get name */
		    v2 = STRDUP(cmd);
		    v3 = strpbrk(v2, " \t");
		    if(v3 != NULL)
			*v3 = '\0';
		    v3 = strrchr(v2, '/');
		    name = STRDUP((v3 != NULL) ? (v3 + 1) : v2);
		    g_free(v2);

		    /* Create new Object */
		    obj = ObjNew();
		    if(obj != NULL)
		    {
			obj->type = OBJ_TYPE_ITEM;
			obj->name = STRDUP(name);
			obj->value = STRDUP(cmd);
			WinFIOObjLoadIcons(win, obj, NULL);

			obj_list = g_list_append(obj_list, obj);
		    }
		    g_free(cmd);
		    g_free(name);
		    g_free(v);
		}

		s = s2 + 1;	/* Seek to next string segment */
	    }

	    /* Mark the source Objects list as not being references so
	     * they will be deleted
	     */
	    obj_list_is_ref = FALSE;
	}
	else
	{
            /* Unsupported data */
            gchar	*target_name = gdk_atom_name(
                selection_data->target
            ),
			*type_name = gdk_atom_name(
		selection_data->type
	    ),
			*msg = g_strdup_printf(
"Data \"%s\" of type \"%s\",\n\
is not supported by this application.",
                target_name,
                type_name
            );
            g_free(target_name);
            g_free(type_name);
            CDialogSetTransientFor(toplevel);
            CDialogGetResponse(
                "Unsupported Data",
                msg,
                NULL,
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK,
                CDIALOG_BTNFLAG_OK
            );
            g_free(msg);
            CDialogSetTransientFor(NULL);

	    FREE_ALL
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Get current time to update time stamps with */
	cur_time = (gulong)time(NULL);

	/* Iterate through list of Objects */
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    src_obj = OBJ(glist->data);
	    if((src_obj == NULL) || (src_obj == tar_obj_parent))
		continue;

	    /* If the source Object is a reference then make sure
	     * that it exists
	     */
	    if(obj_list_is_ref)
	    {
		if(!ObjIsDescendent(win->obj_menu_toplevel, src_obj) &&
		   !ObjIsDescendent(win->obj_tool_bar_toplevel, src_obj)
		)
		    continue;

		/* Make sure we are not copying/moving a parent
		 * Object into a child Object
		 */
		if(ObjIsAncestor(tar_obj_parent, src_obj))
		{
		    gchar *buf = g_strdup_printf(
"Unable to %s parent \"%s\" into child \"%s\".",
			(dc->action == GDK_ACTION_COPY) ? "copy" : "move",
			src_obj->name, tar_obj_parent->name
		    );
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Operation Failed", buf, NULL,
			CDIALOG_ICON_ERROR,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		    g_free(buf);
		    continue;
		}
	    }

	    /* Parent does not allow Group Objects as childs? */
	    if(OBJ_IS_GROUP(src_obj) && OBJ_NO_SUBGROUPS(tar_obj_parent))
	    {
		gchar *buf = g_strdup_printf(
"The group \"%s\" does not permit subgroups.",
		    tar_obj_parent->name
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Operation Failed", buf, NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		continue;
	    }

	    /* Create new object based on the drag action */
	    switch(dc->action)
	    {
	      case GDK_ACTION_DEFAULT:
	      case GDK_ACTION_MOVE:
		/* Copy the source Object and all its child Objects */
		obj = ObjCopy(src_obj);
		if(obj != NULL)
		{
		    /* Set parent */
		    obj->parent = tar_obj_parent;

		    /* Add to the parent's list of child Objects */
		    if(tar_sibling_index > -1)
		    {
			tar_obj_parent->children = g_list_insert(
			    tar_obj_parent->children, obj,
			    tar_sibling_index
			);
			tar_sibling_index++;
		    }
		    else
			tar_obj_parent->children = g_list_append(
			    tar_obj_parent->children, obj
			);

		    /* Notify the Win about this new Object */
		    WinObjAddedCB(win, obj);

		    /* If the source Object is a reference then add
		     * it to the delete list since this is a move
		     */
		    if(obj_list_is_ref)
			obj_del_list = g_list_append(
			    obj_del_list, src_obj
			);
		}
		break;

	      case GDK_ACTION_COPY:
		/* Copy the source Object and all its child Objects */
		obj = ObjCopy(src_obj);
		if(obj != NULL)
		{
		    /* Set parent */
		    obj->parent = tar_obj_parent;

		    /* Add to the parent's list of child Objects */
		    if(tar_sibling_index > -1)
		    {
			tar_obj_parent->children = g_list_insert(
			    tar_obj_parent->children, obj,
			    tar_sibling_index
			);
			tar_sibling_index++;
		    }
		    else
			tar_obj_parent->children = g_list_append(
			    tar_obj_parent->children, obj
			);

		    /* Modify the Object's name since the drag action
		     * is to copy
		     */
		    if(!OBJ_IS_SEPARATOR(obj) &&
		       obj_list_is_ref
		    )
		    {
			g_free(obj->name);
			obj->name = g_strdup_printf(
			    "Copy of %s",
			    src_obj->name
			);
		    }

		    /* Update last modified time */
		    obj->last_modified = cur_time;

		    /* Notify the Win about this new Object */
		    WinObjAddedCB(win, obj);
		}
		break;

	      case GDK_ACTION_LINK:
		/* Create a new Link */
		obj = ObjNew();
		if(obj != NULL)
		{
		    obj->type = OBJ_TYPE_ITEM_LINK;

		    /* Set parent */
		    obj->parent = tar_obj_parent;

		    /* Add to the parent's list of child Objects */
		    if(tar_sibling_index > -1)
		    {
			tar_obj_parent->children = g_list_insert(
			    tar_obj_parent->children, obj,
			    tar_sibling_index
			);
			tar_sibling_index++;
		    }
		    else
			tar_obj_parent->children = g_list_append(
			    tar_obj_parent->children, obj
			);

		    /* Set name, icon, and background */
		    g_free(obj->name);
		    obj->name = g_strdup_printf(
			"Link to %s",
			src_obj->name
		    );
		    g_free(obj->icon_name);
		    obj->icon_name = STRDUP(src_obj->icon_name);
		    g_free(obj->bg_name);
		    obj->bg_name = STRDUP(src_obj->bg_name);

		    /* Set link destination as the path to the source
		     * Object
		     */
		    g_free(obj->value);
		    obj->value = ObjGetPathFromObj(src_obj);

		    obj->name_orientation = GTK_ORIENTATION_HORIZONTAL;
		    obj->name_justify = GTK_JUSTIFY_LEFT;

		    /* Load icon */
		    WinFIOObjLoadIcons(win, obj, obj->icon_name);

		    obj->accel_key = '\0';
		    obj->accel_mods = 0;

		    /* Update last modified time */
		    obj->last_modified = cur_time;

		    /* Notify the Win about this new Object */
		    WinObjAddedCB(win, obj);
		}
		break;

	      case GDK_ACTION_PRIVATE:
	      case GDK_ACTION_ASK:
		break;
	    }
	}

	/* Delete Objects that were added in the delete Objects list */
	for(glist = obj_del_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = OBJ(glist->data);
	    if(obj == NULL)
		continue;

	    /* Notify the Win about the removal of this Object */
	    WinObjRemovedCB(win, obj);

	    /* Remove this Object from its parent's list of children */
	    obj_parent = obj->parent;
	    if(obj_parent != NULL)
		obj_parent->children = g_list_remove(
		    obj_parent->children, obj
		);

	    /* Delete this Object and all its children */
	    ObjDelete(obj);
	}

	FREE_ALL
	WinSetBusy(win, FALSE);
#undef FREE_ALL
}


/*
 *	Sets the DND icon based on the specified cell on the Win's
 *	GtkCList.
 */
void WinListDNDSetIcon(
	win_struct *win, gint row, gint column   
)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return;

	if((row >= 0) && (row < clist->rows))
	{
	    gint i;
	    gchar *text = NULL;
	    guint8 spacing = 0;
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;

	    /* Search for the first cell that has a useable pixmap
	     * for the drag icon
	     */
	    for(i = 0; i < clist->columns; i++)
	    {
		switch(gtk_clist_get_cell_type(clist, row, i))
		{
		  case GTK_CELL_PIXMAP:
		    gtk_clist_get_pixmap(
			clist, row, i,
			&pixmap, &mask
		    );
		    break;
		  case GTK_CELL_PIXTEXT:
		    gtk_clist_get_pixtext(
			clist, row, i,
			&text, &spacing, &pixmap, &mask
		    );
		    break; 
		  case GTK_CELL_TEXT:
		  case GTK_CELL_WIDGET:
		  case GTK_CELL_EMPTY:
		    break;
		}  
		if(pixmap != NULL) 
		    break;
	    }

	    /* If we got a pixmap then set it as the drag icon */
	    if(pixmap != NULL)
	    {
		gint width, height;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height / 2
		);
	    }     
	}
}


/*
 *	Win GtkCList "drag_data_get" signal callback.
 */
void WinListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint row, buf_len = 0;
	guint8 *buf = NULL;
	GList *glist;
	GtkCList *clist;
	core_struct *core;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	core = CORE(win->core);

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)
	    return;

	/* Iterate through selected rows */
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    row = (gint)glist->data;
	    buf = ObjDDEBufferAppend(
		buf, &buf_len,
		OBJ(gtk_clist_get_row_data(clist, row))
	    );
	}
	if(buf != NULL)
	{
	    gtk_selection_data_set( 
		selection_data,
		core->menu_item_ptr_atom,
		8,                      /* Bits Per Character */
		buf, buf_len
	    );
	    data_sent = TRUE;
	    g_free(buf);
	}

	/* If no data was sent out then send an error response */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    ); 
	    data_sent = TRUE;
	}
}

/*
 *	Win GtkCList "drag_data_received" signal callback.
 */
void WinListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkCList *clist;
	obj_struct *tar_obj_parent, *tar_obj_sibling;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)
	    return;

	/* Find the row and column from the specified coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,    
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))               
	{
	    row = -1;
	    column = 0;
	}

	/* Get parent and sibling Objects */
	tar_obj_parent = WinTreeGetLocation(win);
	tar_obj_sibling = OBJ(gtk_clist_get_row_data(clist, row));

	/* Process the received drag data */
	WinDragDataReceivedNexus(
	    win, dc, info, selection_data,
	    tar_obj_parent, tar_obj_sibling
	);                                         
}

/*
 *	Win GtkCList "drag_data_delete" signal callback.
 */
void WinListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	GtkCList *clist;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	clist = GTK_IS_CLIST(widget) ? GTK_CLIST(widget) : NULL;
	if(clist == NULL)                                       
	    return;

}


/*
 *	Sets the DND icon based on the specified node on the Win's
 *	GtkCTree.
 */
void WinTreeDNDSetIcon(win_struct *win, GtkCTreeNode *node)
{
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	if(ctree == NULL)
	    return;

	if(node != NULL)
	{
	    GdkPixmap *pixmap = NULL;
	    GdkBitmap *mask = NULL;
	    GtkCTreeRow *row_ptr = GTK_CTREE_ROW(node);
	    if(row_ptr != NULL)
	    {
		if(row_ptr->expanded)
		{
		    pixmap = row_ptr->pixmap_opened;
		    mask = row_ptr->mask_opened;
		    if(pixmap == NULL)
		    {
			pixmap = row_ptr->pixmap_closed;
			mask = row_ptr->mask_closed;
		    }
		}
		else
		{   
		    pixmap = row_ptr->pixmap_closed;
		    mask = row_ptr->mask_closed;
		}
	    }    

	    /* If we got a pixmap then set it as the drag icon */
	    if(pixmap != NULL)
	    {
		gint width, height;
		gdk_window_get_size(pixmap, &width, &height);
		GUIDNDSetDragIcon(
		    pixmap, mask,
		    width / 2, height / 2
		);
	    }
	}
}

/*
 *	Win GtkCTree "drag_data_get" signal callback.
 */
void WinTreeDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint buf_len = 0;
	guint8 *buf = NULL;
	GList *glist;
	GtkCTreeNode *node;
	GtkCList *clist;
	GtkCTree *ctree;
	core_struct *core;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	core = CORE(win->core);

	ctree = GTK_IS_CTREE(widget) ? GTK_CTREE(widget) : NULL;
	if(ctree == NULL)
	    return;  

	clist = GTK_CLIST(widget);

	/* Iterate through selected rows */
	for(glist = clist->selection;      
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    node = (GtkCTreeNode *)glist->data;
	    buf = ObjDDEBufferAppend(
		buf, &buf_len,
		OBJ(gtk_ctree_node_get_row_data(ctree, node))
	    );
	}
	if(buf != NULL)
	{
	    gtk_selection_data_set( 
		selection_data,
		core->menu_item_ptr_atom,
		8,                      /* Bits Per Character */
		buf, buf_len                   
	    );
	    data_sent = TRUE;
	    g_free(buf);
	}

	/* If no data was sent out then send an error response */
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)		/* Length */
	    ); 
	    data_sent = TRUE;
	}
}

/*
 *	Win GtkCTree "drag_data_received" signal callback.
 */
void WinTreeDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkCList *clist;
	GtkCTree *ctree;
	obj_struct *tar_obj_parent, *tar_obj_sibling;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	ctree = GTK_IS_CTREE(widget) ? GTK_CTREE(widget) : NULL;
	if((ctree == NULL) ||
	   ((selection_data != NULL) ? (selection_data->length <= 0) : TRUE)
	)
	    return;
		   
	clist = GTK_CLIST(widget);

	/* Find the row and column from the specified coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))
	{
	    row = -1;
	    column = 0;
	}

	/* Get parent and sibling Objects */
	tar_obj_parent = OBJ((row > -1) ?
	    gtk_clist_get_row_data(clist, row) : NULL
	);
	tar_obj_sibling = NULL;

	/* Process the received drag data */
	WinDragDataReceivedNexus(
	    win, dc, info, selection_data,
	    tar_obj_parent, tar_obj_sibling
	);
}

/*
 *	Win GtkCTree "drag_data_delete" signal callback.
 */
void WinTreeDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	GtkCList *clist;
	GtkCTree *ctree;
	win_struct *win = WIN(data);
	if((widget == NULL) || (dc == NULL) || (win == NULL))
	    return;

	ctree = GTK_IS_CTREE(widget) ? GTK_CTREE(widget) : NULL;
	if(ctree == NULL)
	    return;

	clist = GTK_CLIST(ctree);

}
