#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "guiutils.h"
#include "imgfio.h"
#include "obj.h"
#include "win.h"
#include "wincb.h"
#include "winlist.h"
#include "winfio.h"
#include "core.h"
#include "config.h"

#include "images/icon_executable_20x20.xpm"
#include "images/icon_folder_closed_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_folder_menus_20x20.xpm"
#include "images/icon_folder_menus_opened_20x20.xpm"
#include "images/icon_folder_tool_bar_20x20.xpm"
#include "images/icon_folder_tool_bar_opened_20x20.xpm"
#include "images/icon_link2_20x20.xpm"
#include "images/icon_hsep_20x20.xpm"


static gboolean IS_STRING_IN_LIST(GList *glist, const gchar *s);
static gchar *COMPLETE_PATH(
	gchar **path_list, const gchar *name,
	const gchar *home
);
static gchar *COMPLETE_PATH_LIST(
	GList *paths, const gchar *name,
	const gchar *home
);
static void OBJ_SET_PIXMAP(
	obj_struct *obj,
	GdkPixmap *pixmap, GdkBitmap *mask,
	GdkPixmap *pixmap_opened, GdkBitmap *mask_opened
);

/* Icon Paths */
static GList *WinFIOGetIconPathsIceWM(win_struct *win);
GList *WinFIOGetIconPaths(win_struct *win);

/* Icon Name Parsing */
static gchar *WinFIOObjGetIconNameFromFullPathIceWM(
	win_struct *win, const gchar *full_path
);
gchar *WinFIOObjGetIconNameFromFullPath(
	win_struct *win, const gchar *full_path
);

/* Open Icon */
static GdkPixmap *WinFIOOpenIconIceWM(
	win_struct *win, const gchar *icon_name, menu_icon_size size,
	GdkBitmap **mask_rtn
);
GdkPixmap *WinFIOOpenIcon(
	win_struct *win, const gchar *icon_name, menu_icon_size size,
	GdkBitmap **mask_rtn
);

/* Load Object Icons */
static gint WinFIOObjLoadIconsIceWM(
	win_struct *win, obj_struct *obj, const gchar *icon_name
);
gint WinFIOObjLoadIcons(
	win_struct *win, obj_struct *obj, const gchar *icon_name
);

/* Open Menu Configuration */
static gint WinFIOOpenIceWMFile(
	win_struct *win, const gchar *path, FILE *fp,
	obj_struct *parent_obj, GtkCTreeNode *parent_node,
	gboolean verbose, gboolean interactive            
);
static gint WinFIOOpenIceWM(
	win_struct *win, const gchar *path,
	gboolean verbose, gboolean interactive
);
gint WinFIOOpen(
	win_struct *win, const gchar *path,
	menu_format format,
	gboolean verbose, gboolean interactive
);

/* Save Menu Configuration */
static void WinFIOSaveIceWMIterate(
	win_struct *win, const gchar *path, FILE *fp,
	obj_struct *obj, gint *level,
	gboolean verbose, gboolean interactive
);
static gint WinFIOSaveIceWM(
	win_struct *win, const gchar *path,
	gboolean verbose, gboolean interactive
);
gint WinFIOSave(
	win_struct *win, const gchar *path, menu_format format,
	gboolean verbose, gboolean interactive
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#ifndef ISCR
# define ISCR(c)	(((c) == '\n') || ((c) == '\r'))
#endif


/*
 *	Checks if the specified string is in the list.
 */
static gboolean IS_STRING_IN_LIST(GList *glist, const gchar *s)
{
	if(s == NULL)
	    return(FALSE);

	while(glist != NULL)
	{
	    if((glist->data != NULL) ?
		!strcmp((char *)glist->data, (const char *)s) : FALSE
	    )
		return(TRUE);

	    glist = g_list_next(glist);
	}

	return(FALSE);
}


/*
 *	Returns a copy of the completed path by looking for an
 *	object that exists when the specified name is postfixed to
 *	each path in the path list.
 *
 *	If name is an absolute path then a copy of name is returned.
 *
 *	For any path in the path list that is not an absolute path,
 *	the home directory will be prefixed to it.
 *
 *	The calling function must delete the returned pointer.
 */
static gchar *COMPLETE_PATH(
	gchar **path_list, const gchar *name,
	const gchar *home
)
{
	gint i;
	gchar *full_path = NULL;
	const gchar *path_pfx;

	if((path_list == NULL) || (name == NULL))
	    return(NULL);

	if(ISPATHABSOLUTE(name))
	    return(STRDUP(name));

	/* Iterate through the paths list, prefixing each path to the
	 * specified name and check if the object exists
	 */
	for(i = 0; path_list[i] != NULL; i++)
	{
	    /* Get path prefix from the paths list */
	    path_pfx = path_list[i];

	    /* Format full path with the path prefix and the specified
	     * name (and home directory as needed)
	     */
	    if(ISPATHABSOLUTE(path_pfx))
		full_path = g_strdup_printf(
		    "%s%c%s",
		    path_pfx, DIR_DELIMINATOR, name
		);
	    else
		full_path = g_strdup_printf(        
		    "%s%c%s%c%s",
		    home, DIR_DELIMINATOR, path_pfx, DIR_DELIMINATOR, name
		);
	    if(full_path == NULL)
		continue;

	    /* Object exists? */
	    if(!access((char *)full_path, F_OK))
		break;

	    g_free(full_path);
	    full_path = NULL;
	}

	return(full_path);
}

/*
 *	Same as COMPLETE_PATH() except that it uses a GList for a list
 *	of paths.
 */
static gchar *COMPLETE_PATH_LIST(
	GList *paths, const gchar *name,
	const gchar *home                    
)
{
	gchar *full_path = NULL;
	const gchar *path_pfx;
	GList *glist;

	if((paths == NULL) || (name == NULL))
	    return(NULL);

	if(ISPATHABSOLUTE(name))
	    return(STRDUP(name));

	/* Iterate through the paths list, prefixing each path to the
	 * specified name and check if the object exists
	 */
	for(glist = paths; glist != NULL; glist = g_list_next(glist))
	{
	    /* Get path prefix from the paths list */
	    path_pfx = (gchar *)glist->data;
	    if(path_pfx == NULL)
		continue;

	    /* Format full path with the path prefix and the specified
	     * name (and home directory as needed)
	     */
	    if(ISPATHABSOLUTE(path_pfx))
		full_path = g_strdup_printf(
		    "%s%c%s",
		    path_pfx, DIR_DELIMINATOR, name
		);
	    else
		full_path = g_strdup_printf(
		    "%s%c%s%c%s",
		    home, DIR_DELIMINATOR, path_pfx, DIR_DELIMINATOR, name
		);
	    if(full_path == NULL)
		continue;

	    /* Object exists? */
	    if(!access((char *)full_path, F_OK))
		break;

	    g_free(full_path);
	    full_path = NULL;
	}

	return(full_path);
}

/*
 *	Sets the specified pixmaps & masks to the specified Object,
 *	and unrefs the existing pixmaps & masks on the Object.
 */
static void OBJ_SET_PIXMAP(
	obj_struct *obj,
	GdkPixmap *pixmap, GdkBitmap *mask,
	GdkPixmap *pixmap_opened, GdkBitmap *mask_opened
)
{
	if(obj == NULL)
	    return;

	/* Unref existing pixmaps & masks */
	GDK_PIXMAP_UNREF(obj->pixmap)
	obj->pixmap = NULL;
	GDK_BITMAP_UNREF(obj->mask)
	obj->mask = NULL;

	GDK_PIXMAP_UNREF(obj->pixmap_opened)
	obj->pixmap_opened = NULL;
	GDK_BITMAP_UNREF(obj->mask_opened)
	obj->mask_opened = NULL;

	/* Set new pixmaps & masks */
	if(pixmap != NULL)
	{
	    gdk_pixmap_ref(pixmap);
	    obj->pixmap = pixmap;
	}
	if(mask != NULL)
	{
	    gdk_bitmap_ref(mask);
	    obj->mask = mask;
	}

	if(pixmap_opened != NULL)
	{
	    gdk_pixmap_ref(pixmap_opened);
	    obj->pixmap_opened = pixmap_opened;
	}
	if(mask_opened != NULL)
	{
	    gdk_bitmap_ref(mask_opened);
	    obj->mask_opened = mask_opened;
	}
}


/*
 *	Returns a list of strings describing the IceWM icon paths.
 */
static GList *WinFIOGetIconPathsIceWM(win_struct *win)
{
	gchar *icewm_data_paths[] = ICEWM_DATA_PATHS;
	const gchar *home_dir;
	gchar *preferences_path;
	FILE *fp;
	GList *paths = NULL;
	core_struct *core = CORE(win->core);

	/* Get home dir */
	home_dir = core->home_dir;

	/* Get path to IceWM preferences file which specifies the
	 * icon paths and open it
	 */
	preferences_path = COMPLETE_PATH(
	    icewm_data_paths, "preferences", home_dir
	);
	fp = FOpen(preferences_path, "rb");
	if(fp == NULL)
	{
	    g_free(preferences_path);
	    return(paths);
	}

	/* Seek fp to the IconPath parameter's value */
	if(!FSeekToParm(fp, "IconPath", '#', '='))
	{
	    gint i;
	    gchar *s, *s2, *path, *full_path, **strv;
	    gchar *v = (gchar *)FGetString(fp);

	    /* Seek s past initial '"' character */
	    s = v;
	    if(*s == '"')
		s++;

	    /* Remove tailing '"' character */
	    s2 = (gchar *)strchr((char *)s, '"');
	    if(s2 != NULL)
		*s2 = '\0';

	    /* Explode strings and add them to the paths list,
	     * complete the path with the home dir as needed
	     */
	    strv = g_strsplit(s, ":", -1);
	    if(strv != NULL)
	    {
		for(i = 0; strv[i] != NULL; i++)
		{
		    path = strv[i];

		    /* Get full path */
		    if(ISPATHABSOLUTE(path))
			full_path = STRDUP(path);
		    else
			full_path = STRDUP(PrefixPaths(home_dir, path));
		    if(full_path == NULL)
			continue;

		    /* Check if the path is already in the list */
		    if(IS_STRING_IN_LIST(paths, full_path))
		    {
			g_free(full_path);
			continue;
		    }

		    /* Check if the path does not exist */
		    if(access((char *)full_path, F_OK))
		    {
			g_free(full_path);
			continue;
		    }

		    /* Add path to the paths list */
		    paths = g_list_append(paths, full_path);
		}

		g_strfreev(strv);
	    }

	    g_free(v);
	}

	if(TRUE)
	{
	    /* Append default paths that exist */
	    gint i;
	    gchar *path, *full_path;
	    gchar *icewm_icon_paths[] = ICEWM_ICON_PATHS;

	    for(i = 0; icewm_icon_paths[i] != NULL; i++)
	    {
		path = icewm_icon_paths[i];

		/* Get full path */
		if(ISPATHABSOLUTE(path))
		    full_path = STRDUP(path);
		else
		    full_path = STRDUP(PrefixPaths(home_dir, path));

		/* Check if the path is already in the list */
		if(IS_STRING_IN_LIST(paths, full_path))
		{
		    g_free(full_path);
		    continue;
		}

		/* Check if the path does not exist */
		if(access(full_path, F_OK))
		{
		    g_free(full_path);
		    continue;
		}

		/* Add full path to the paths list */
		paths = g_list_append(paths, full_path);
	    }
	}

	/* Close the IceWM Preferences file */
	FClose(fp);

	g_free(preferences_path);

#if 0
if(paths != NULL) {
 GList *glist = paths;
 while(glist != NULL) {
  g_print("Path: \"%s\"\n", (const char *)glist->data);
  glist = g_list_next(glist);
 }
}
#endif

	return(paths);
}

/*
 *	Returns a list of strings describing the icon paths.
 */
GList *WinFIOGetIconPaths(win_struct *win)
{
	GList *paths = NULL;

	if(win == NULL)
	    return(paths);

	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:  
	    paths = NULL;
	    break;             

	  case MENU_FORMAT_ICEWM:
	    paths = WinFIOGetIconPathsIceWM(win);
	    break;
	}

	return(paths);
}


/*
 *	Returns a copy of the IceWM icon name obtained from the
 *	specified full path.
 */
static gchar *WinFIOObjGetIconNameFromFullPathIceWM(
	win_struct *win, const gchar *full_path
)
{
	gint len;
	gchar *icon_name;
	const gchar *s, *name = strrchr(full_path, DIR_DELIMINATOR);
	if(name != NULL)
	    name++;
	else
	    name = full_path;

	/* Seek s to the first '_' or '.' character in the name */
	for(s = name; *s != '\0'; s++)
	{
	    if((*s == '_') || (*s == '.'))
		break;
	}

	/* Get length of name to copy */
	len = (gint)(s - name);
	if(len <= 0)
	    return(NULL);

	/* Copy the icon name segment of name to icon_name */
	icon_name = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(icon_name != NULL)
	{
	    memcpy(icon_name, name, len);
	    icon_name[len] = '\0';
	}

	return(icon_name);
}

/*
 *	Returns a copy of the icon name obtained from the specified
 *	full path.
 *
 *	The calling function must delete the returned string.
 */
gchar *WinFIOObjGetIconNameFromFullPath(
	win_struct *win, const gchar *full_path
)
{
	gchar *icon_name = NULL;

	if((win == NULL) || (full_path == NULL))
	    return(icon_name);

	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    icon_name = NULL;
	    break;

	  case MENU_FORMAT_ICEWM:
	    icon_name = WinFIOObjGetIconNameFromFullPathIceWM(win, full_path);
	    break;
	}

	return(icon_name);
}


/*
 *	Opens the IceWM Icon specified by the icon_name.
 */
static GdkPixmap *WinFIOOpenIconIceWM(
	win_struct *win, const gchar *icon_name, menu_icon_size size,
	GdkBitmap **mask_rtn
)
{
	gint width, height;
	gchar *name, *full_path;
	GList *icon_paths = win->icon_paths;
	GdkPixmap *pixmap = NULL;
	menu_icon_size cur_size = size;
	core_struct *core = CORE(win->core);

	if(STRISEMPTY(icon_name))
	    return(pixmap);

	while(pixmap == NULL)
	{
#define FMT_NAME(_size_,_name_,_ext_)	{	\
 const gchar *size_str = NULL;		\
 switch(_size_) {			\
  case MENU_ICON_SIZE_MINI:		\
   size_str = "16x16";			\
   break;				\
  case MENU_ICON_SIZE_SMALL:		\
   size_str = "20x20";			\
   break;				\
  case MENU_ICON_SIZE_SMALL_MEDIUM:	\
   size_str = "24x24";			\
   break;				\
  case MENU_ICON_SIZE_MEDIUM:		\
   size_str = "32x32";			\
   break;				\
  case MENU_ICON_SIZE_MEDIUM_LARGE:	\
   size_str = "40x40";			\
   break;				\
  case MENU_ICON_SIZE_LARGE:		\
   size_str = "48x48";			\
   break;				\
  case MENU_ICON_SIZE_XLARGE:		\
   size_str = "64x64";			\
   break;				\
 }					\
 if(size_str != NULL)			\
  name = g_strdup_printf(		\
   "%s_%s%s",				\
   (_name_), size_str, (_ext_)		\
  );					\
 else					\
  name = NULL;				\
}

	    /* Get the full path to the icon file from the specified
	     * icon name and size
	     */
	    FMT_NAME(cur_size, icon_name, ".xpm");
	    full_path = COMPLETE_PATH_LIST(
		icon_paths, name, core->home_dir
	    );
	    g_free(name);
	    if(full_path == NULL)
	    {
		FMT_NAME(cur_size, icon_name, ".png");
		full_path = COMPLETE_PATH_LIST(
		    icon_paths, name, core->home_dir
		);
		g_free(name);
	    }
	    if(full_path == NULL)
	    {
		FMT_NAME(cur_size, icon_name, ".tga");
		full_path = COMPLETE_PATH_LIST(
		    icon_paths, name, core->home_dir
		);
		g_free(name);
	    }
#undef FMT_NAME

	    /* Attempt to open the icon */
	    pixmap = ImgOpenPixmap(
		full_path,
		&width, &height,
		mask_rtn,
#ifdef HAVE_IMLIB
		imlib_handle
#else
		NULL
#endif
	    );
	    g_free(full_path);

	    /* If failed to open the icon then try a different size */
	    if(pixmap == NULL)
	    {
		/* Give up when there are no remaining sizes to try */
		if(cur_size == MENU_ICON_SIZE_SMALL)
		    break;

		cur_size--;
	    }
	}

	/* If no pixmap was opened then try to open one with an
	 * unspecified size
	 */
	if(pixmap == NULL)
	{
	    name = g_strdup_printf(
		"%s.xpm",
		icon_name
	    );
	    full_path = COMPLETE_PATH_LIST(
		icon_paths, name, core->home_dir
	    );
	    g_free(name);
	    if(full_path == NULL)
	    {
		name = g_strdup_printf(
		    "%s.png",
		    icon_name
		);
		full_path = COMPLETE_PATH_LIST(
		    icon_paths, name, core->home_dir
		);
		g_free(name);
	    }
	    if(full_path == NULL)
	    {
		name = g_strdup_printf(
		    "%s.tga",
		    icon_name
		);
		full_path = COMPLETE_PATH_LIST(
		    icon_paths, name, core->home_dir
		);
		g_free(name);
	    }

	    /* Attempt to open the icon */
	    pixmap = ImgOpenPixmap(
		full_path,
		&width, &height,
		mask_rtn,
#ifdef HAVE_IMLIB
		imlib_handle
#else
		NULL
#endif
	    );
	    g_free(full_path);
	}

	return(pixmap);
}

/*
 *	Opens the Icon specified by the icon_name.
 */
GdkPixmap *WinFIOOpenIcon(
	win_struct *win, const gchar *icon_name, menu_icon_size size,
	GdkBitmap **mask_rtn
)
{
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;

	if(mask_rtn != NULL)
	    *mask_rtn = mask;

	if(win == NULL)
	    return(pixmap);

	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    pixmap = NULL;
	    break;

	  case MENU_FORMAT_ICEWM:
	    pixmap = WinFIOOpenIconIceWM(win, icon_name, size, mask_rtn);
	    break;
	}

	return(pixmap);
}


/*
 *	Loads the IceWM Icon specified by the icon_name to the
 *	Object.
 */
static gint WinFIOObjLoadIconsIceWM(
	win_struct *win, obj_struct *obj, const gchar *icon_name  
)
{
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap_opened = NULL;
	GdkBitmap *mask_opened = NULL;


	/* Use default icon? */
	if(STRISEMPTY(icon_name))
	{
	    /* Load default icon based on the Object's type */
	    switch(OBJ_TYPE(obj))
	    {
	      case OBJ_TYPE_ITEM:
	      case OBJ_TYPE_ITEM_SPECIAL:
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask, 
		    (guint8 **)icon_executable_20x20_xpm
		);
		break;

	      case OBJ_TYPE_ITEM_LINK:
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask,
		    (guint8 **)icon_link2_20x20_xpm
		);
		break;

	      case OBJ_TYPE_ITEM_SEPARATOR:
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask,
		    (guint8 **)icon_hsep_20x20_xpm  
		);
		break;         

	      case OBJ_TYPE_GROUP:
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask,
		    (guint8 **)icon_folder_closed_20x20_xpm
		);
		pixmap_opened = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask_opened,
		    (guint8 **)icon_folder_opened_20x20_xpm
		);
		break;
	    }
	}
	else
	{
	    gint width, height;
	    gchar *name, *full_path = NULL;
	    GList *icon_paths = win->icon_paths;
	    menu_icon_size size;
	    core_struct *core = CORE(win->core);

	    /* Get the full path to the icon file from the specified
	     * icon name
	     */
	    for(size = MENU_ICON_SIZE_SMALL; (gint)size >= 0; size--)
	    {
#define FMT_NAME(_size_,_name_,_ext_)	{	\
 const gchar *size_str;			\
					\
 /* Objects may only use the small or	\
  * mini size icons			\
  */					\
 switch(_size_) {			\
  case MENU_ICON_SIZE_MINI:		\
   size_str = "16x16";			\
   break;				\
  case MENU_ICON_SIZE_SMALL:		\
   size_str = "20x20";			\
   break;				\
  default:				\
   size_str = NULL;			\
   break;				\
 }					\
 if(size_str != NULL)			\
  name = g_strdup_printf(		\
   "%s_%s%s",				\
   (_name_), size_str, (_ext_)		\
  );					\
 else					\
  name = NULL;				\
}
		/* Get icon file name based on the size */
		FMT_NAME(size, icon_name, ".xpm");
		if(name == NULL)
		    break;
		full_path = COMPLETE_PATH_LIST( 
		    icon_paths, name, core->home_dir
		);
		g_free(name);
		if(full_path == NULL)
		{
		    FMT_NAME(size, icon_name, ".png");
		    if(name == NULL)
			break;
		    full_path = COMPLETE_PATH_LIST(
			icon_paths, name, core->home_dir
		    );
		    g_free(name);
		}
		if(full_path == NULL)
		{
		    FMT_NAME(size, icon_name, ".tga");
		    if(name == NULL)
			break;
		    full_path = COMPLETE_PATH_LIST(
			icon_paths, name, core->home_dir
		    );
		    g_free(name);
		}
#undef FMT_NAME
		if(full_path != NULL)
		    break;
	    }

	    /* If unable to find an icon file with the appropriate
	     * size then try an unspecified size
	     */
	    if(full_path == NULL)
	    {
		name = g_strdup_printf(
		    "%s.xpm",
		    icon_name
		);
		full_path = COMPLETE_PATH_LIST(
		    icon_paths, name, core->home_dir
		);
		g_free(name);
		if(full_path == NULL)
		{
		    name = g_strdup_printf(
			"%s.png",
			icon_name
		    );
		    full_path = COMPLETE_PATH_LIST(
			icon_paths, name, core->home_dir
		    );
		    g_free(name);
		}
		if(full_path == NULL)
		{
		    name = g_strdup_printf(
			"%s.tga",
			icon_name
		    );
		    full_path = COMPLETE_PATH_LIST(
			icon_paths, name, core->home_dir
		    );
		    g_free(name);
		}
	    }

	    /* Open the icon based on the Object's type, substituting
	     * certain default icons where appropriate
	     */
	    switch(OBJ_TYPE(obj))
	    {
	      case OBJ_TYPE_ITEM:
		if(!g_strcasecmp(icon_name, ICEWM_DEF_ICON_NAME_ITEM))
		{
		    g_free(full_path);
		    return(WinFIOObjLoadIconsIceWM(win, obj, NULL));
		}
		pixmap = ImgOpenPixmap(
		    full_path,
		    &width, &height,
		    &mask,
#ifdef HAVE_IMLIB
		    imlib_handle
#else
		    NULL
#endif
		);
		break;

	      case OBJ_TYPE_ITEM_LINK:
	      case OBJ_TYPE_ITEM_SPECIAL:
		pixmap = ImgOpenPixmap(
		    full_path,
		    &width, &height,
		    &mask,
#ifdef HAVE_IMLIB
		    imlib_handle        
#else           
		    NULL
#endif
		);
		break;

	      case OBJ_TYPE_ITEM_SEPARATOR:
		pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask,
		    (guint8 **)icon_hsep_20x20_xpm
		);
		break; 

	      case OBJ_TYPE_GROUP:
		if(!g_strcasecmp(icon_name, ICEWM_DEF_ICON_NAME_GROUP))
		{
		    g_free(full_path);
		    return(WinFIOObjLoadIconsIceWM(win, obj, NULL));
		}
		pixmap = ImgOpenPixmap(
		    full_path,
		    &width, &height,
		    &mask,
#ifdef HAVE_IMLIB
		    imlib_handle        
#else           
		    NULL
#endif
		);
		pixmap_opened = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		    &mask_opened,
		    (guint8 **)icon_folder_opened_20x20_xpm
		);
		break;
	    }
	    g_free(full_path);
	}

	/* Set the opened icons to the Object */
	OBJ_SET_PIXMAP(
	    obj,
	    pixmap, mask,
	    pixmap_opened, mask_opened
	);

	GDK_PIXMAP_UNREF(pixmap)
	GDK_BITMAP_UNREF(mask)

	GDK_PIXMAP_UNREF(pixmap_opened)
	GDK_BITMAP_UNREF(mask_opened)

	return(0);
}

/*
 *	Loads the icon specified by the icon_name to the Object.
 *
 *	Existing existing pixmaps & masks on the Object will be
 *	unref'ed before loading the new pixmaps & masks.
 *
 *	If icon_name is NULL then default icons will be loaded.
 */
gint WinFIOObjLoadIcons(
	win_struct *win, obj_struct *obj, const gchar *icon_name
)
{
	gint status = -1;

	if((win == NULL) || (obj == NULL))
	    return(status);
	     
	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    status = -2;
	    break;

	  case MENU_FORMAT_ICEWM:
	    status = WinFIOObjLoadIconsIceWM(win, obj, icon_name);
	    break;
	}

	return(status);
}


/*
 *	Opens the Objects from the IceWM configuration file
 *	fp.
 *
 *	All opened Objects will be parented to the specified
 *	parent Object.
 */
static gint WinFIOOpenIceWMFile(
	win_struct *win, const gchar *path, FILE *fp,
	obj_struct *parent_obj, GtkCTreeNode *parent_node,
	gboolean verbose, gboolean interactive
)
{
	gboolean read_only = FALSE;
	gint c, status = 0, lines_read = 0;
	gchar id[1024];
	gulong file_size = 0l, last_modified = 0l;
	struct stat stat_buf;
	obj_struct *obj;
	core_struct *core = CORE(win->core);


	if(fp == NULL)
	    return(0);

	if(!OBJ_IS_GROUP(parent_obj))
	    return(-1);

	/* Get configuration file stats */
	if(!fstat(fileno(fp), &stat_buf))
	{
	    gint euid = core->euid;
	    const mode_t m = stat_buf.st_mode;

	    /* Get file size */
	    file_size = (gulong)stat_buf.st_size;

	    /* Get last modified time and update the parent Object's
	     * last modified time
	     */
	    last_modified = (gulong)stat_buf.st_mtime;
	    parent_obj->last_modified = last_modified;

	    /* Check if effective user id is allowed to write */
	    if(euid == 0)
		read_only = FALSE;
	    else if((uid_t)euid == stat_buf.st_uid)
		read_only = (m & S_IWUSR) ? FALSE : TRUE;
	    else
		read_only = (m & S_IWOTH) ? FALSE : TRUE;
	    win->read_only = read_only;
	}
	else
	{
	    if(verbose && !STRISEMPTY(path))
		g_printerr(
"%s: Unable to get file statistics.\n",
		    path
		);
	    status = -1;
	}

	/* Begin reading the IceWM configuration file */
	while(TRUE)
	{
#define GET_IDENTIFIER {		\
 gboolean in_quotes = FALSE;		\
 gchar *s = id,				\
       *s_end = id + sizeof(id) - 1;	\
					\
 /* Seek fp past spaces */		\
 do {					\
  c = fgetc(fp);			\
  if(c == EOF)				\
   break;				\
  if(ISCR(c))				\
   lines_read++;			\
 } while(ISSPACE(c));			\
 if(c == EOF) {				\
  *s = '\0';				\
  break;				\
 }					\
 fseek(fp, -1l, SEEK_CUR);		\
					\
 /* Read this identifier until next	\
  * space is encountered		\
  */					\
 while(s < s_end) {			\
  c = fgetc(fp);			\
  if(c == EOF)				\
   break;				\
  if(ISCR(c))				\
   lines_read++;			\
					\
  if(ISSPACE(c) && !in_quotes)		\
   break;				\
  else if(c == '"') {			\
   if(in_quotes)			\
    break;				\
   else					\
    in_quotes = TRUE;			\
  }					\
  else					\
   *s++ = c;				\
 }					\
 *s = '\0';				\
}
#define GET_IDENTIFIER_WHOLE_LINE {	\
 gchar *s = id,				\
       *s_end = id + sizeof(id) - 1;	\
					\
 /* Seek fp past spaces */              \
 do {                                   \
  c = fgetc(fp);                        \
  if(c == EOF)                          \
   break;                               \
  if(ISCR(c))                           \
   lines_read++;                        \
 } while(ISSPACE(c));                   \
 if(c == EOF) {				\
  *s = '\0';				\
  break;				\
 }					\
 fseek(fp, -1l, SEEK_CUR);		\
					\
 /* Read rest of line until newline	\
  * while ignoring quotes and storing	\
  * them in the identifier		\
  */					\
 while(s < s_end) {			\
  c = fgetc(fp);			\
  if(c == EOF)				\
   break;				\
  if(ISCR(c))				\
   lines_read++;			\
					\
  if(ISCR(c))				\
   break;				\
  else					\
   *s++ = c;				\
 }					\
 *s = '\0';				\
}

	    /* Get next identifier value */
	    GET_IDENTIFIER

	    /* Skip empty identifiers */
	    if(*id == '\0')
	    {
		if(c == EOF)
		    break;
		else
		    continue;
	    }

	    /* Skip comments */
	    if(*id == '#')
	    {
		fseek(fp, -1l, SEEK_CUR);
		FSeekNextLine(fp);
		continue;
	    }

	    /* Report progress */
	    if(verbose)
		WinStatusProgress(
		    win,
		    (file_size > 0l) ?
			((gfloat)ftell(fp) / (gfloat)file_size) : -1.0f,
		    TRUE
		);

	    /* Handle by identifier */
	    /* Item */
	    if(!g_strcasecmp(id, "prog"))
	    {
		gchar *s, *name, *icon_name, *cmd;

		/* Get name */
		GET_IDENTIFIER
		name = STRDUP(id);

		/* Get icon name */  
		GET_IDENTIFIER
		icon_name = STRDUP(id);

		/* Get subsequent identifiers as the command until the
		 * newline is reached
		 */
		GET_IDENTIFIER_WHOLE_LINE
		cmd = STRDUP(id);

		/* If the icon name has an extension, remove it */
		s = (gchar *)strchr((char *)icon_name, '.');
		if(s != NULL)
		    *s = '\0';

		/* Create new Item Object */
		if(parent_obj != NULL)
		{
		    obj = ObjNewWithValues(
			OBJ_TYPE_ITEM,
			(read_only ? OBJ_FLAG_READ_ONLY : 0),
			parent_obj, NULL,
			name, icon_name, NULL, cmd, NULL,
			GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
			NULL, NULL, NULL, NULL,
			'\0', 0,
			last_modified
		    );
		    if(obj != NULL)
		    {
			WinFIOObjLoadIconsIceWM(win, obj, icon_name);

			/* Add this new Object to the parent's list of
			 * child Objects
			 */
			parent_obj->children = g_list_append(
			    OBJ_CHILDREN(parent_obj), obj
			);
		    }
		    else
		    {
			if(verbose && !STRISEMPTY(path))
			    g_printerr(
"%s: Line %i: Memory allocation error.\n",
				path, lines_read
			);
			status = -3;
		    }
		}

		g_free(name);
		g_free(icon_name);
		g_free(cmd);
	    }
	    /* Group Start */
	    else if(!g_strcasecmp(id, "menu"))
	    {
		gchar *s, *name, *icon_name;

		/* Get name */
		GET_IDENTIFIER
		name = STRDUP(id);

		/* Get icon name */
		GET_IDENTIFIER
		icon_name = STRDUP(id);

		/* Seek past subsequent identifiers until the {
		 * identifier is reached
		 */
		do {
		    GET_IDENTIFIER
		} while((c != EOF) && g_strcasecmp(id, "{"));

		/* If the icon name has an extension, remove it */
		s = (gchar *)strchr((char *)icon_name, '.');
		if(s != NULL)
		    *s = '\0';

		/* Create new Group Object */
		if(parent_obj != NULL)
		{
		    obj = ObjNewWithValues(
			OBJ_TYPE_GROUP,
			(read_only ? OBJ_FLAG_READ_ONLY : 0),
			parent_obj, NULL,
			name, icon_name, NULL, NULL, NULL,
			GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
			NULL, NULL, NULL, NULL,
			'\0', 0,    
			last_modified
		    );
		    if(obj != NULL)
		    {
			WinFIOObjLoadIconsIceWM(win, obj, icon_name);

			/* Append this Group to the parent's list of
			 * child Objects
			 */
			parent_obj->children = g_list_append(
			    OBJ_CHILDREN(parent_obj), obj
			);

			/* Append new node to the Tree and set this
			 * Group Object as the new parent Object
			 */
			parent_node = WinTreeInsert(win, parent_node, NULL, obj);
			parent_obj = obj;
			if(parent_node == NULL)
			{
			    if(verbose && !STRISEMPTY(path))
				g_printerr(
"%s: Line %i: Unable to insert Group \"%s\" to the Tree.\n",
				    path, lines_read, name
				);
			    status = -1;
			}
		    }
		    else
		    {
			if(verbose && !STRISEMPTY(path))
			    g_printerr(
"%s: Line %i: Memory allocation error.\n",
				path, lines_read
			);
			status = -3;

			/* Must break since subsequent parsing will be
			 * referencing an invalid parent Object
			 */
			g_free(name);
			g_free(icon_name);
			break;
		    }
		}

		g_free(name);
		g_free(icon_name);
	    }
	    /* Group End */
	    else if(!g_strcasecmp(id, "}"))
	    {
		/* Get the parents of the parent object and node and
		 * set them back in context
		 */
		GtkCTreeRow *row;

		/* Stack underflow? */
		if((parent_obj == NULL) || (parent_node == NULL))
		{
		    if(verbose && !STRISEMPTY(path))
			g_printerr(
"%s: Line %i: Stack underflow.\n",
			    path, lines_read + 1
			);
		    status = -1;
		    break;
		}

		parent_obj = parent_obj->parent;
		row = GTK_CTREE_ROW(parent_node);
		parent_node = (row != NULL) ? row->parent : NULL;
	    }
	    /* Separator */
	    else if(!g_strcasecmp(id, "separator"))
	    {
		/* Append Separator to the parent object's
		 * list of child objects
		 */
		if(parent_obj != NULL)
		{
		    obj = ObjNewWithValues(
			OBJ_TYPE_ITEM_SEPARATOR,
			(read_only ? OBJ_FLAG_READ_ONLY : 0),
			parent_obj, NULL,
			"Separator", NULL, NULL, NULL, NULL,
			GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
			NULL, NULL, NULL, NULL,
			'\0', 0,
			last_modified
		    );
		    if(obj != NULL)
		    {
			WinFIOObjLoadIconsIceWM(win, obj, NULL);

			/* Add this new Object to the parent's list of
			 * child Objects
			 */
			parent_obj->children = g_list_append(
			    OBJ_CHILDREN(parent_obj), obj
			);
		    }
		    else
		    {
			if(verbose && !STRISEMPTY(path))
			    g_printerr(
"%s: Line %i: Memory allocation error.\n",
				path, lines_read
			);
			status = -3;
		    }
		}
	    }
	    /* Special */
	    else if(!g_strcasecmp(id, "startup") ||
		    !g_strcasecmp(id, "restart") ||
		    !g_strcasecmp(id, "shutdown") ||
		    !g_strcasecmp(id, "menuprog") ||
		    !g_strcasecmp(id, "menufile")
	    )
	    {
		gchar *name, *value;

		name = STRDUP(id);
		GET_IDENTIFIER_WHOLE_LINE
		value = STRDUP(id);

		/* Append Special Object to the parent Object's
		 * list of child objects
		 */
		if(parent_obj != NULL)
		{
		    obj = ObjNewWithValues(
			OBJ_TYPE_ITEM_SPECIAL,
			(read_only ? OBJ_FLAG_READ_ONLY : 0),
			parent_obj, NULL,
			name, ICEWM_DEF_ICON_NAME_SPECIAL, NULL, value, NULL,
			GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
			NULL, NULL, NULL, NULL,
			'\0', 0,
			last_modified
		    );
		    if(obj != NULL)
		    {
			WinFIOObjLoadIconsIceWM(win, obj, NULL);

			/* Add this new Object to the parent's list of
			 * child Objects
			 */
			parent_obj->children = g_list_append(
			    OBJ_CHILDREN(parent_obj), obj
			);
		    }
		    else
		    {
			if(verbose && !STRISEMPTY(path))
			    g_printerr(
"%s: Line %i: Memory allocation error.\n",
				path, lines_read
			);
			status = -3;
		    }
		}

		g_free(name);
		g_free(value);
	    }
	    /* Unknown identifier */
	    else
	    {
		if(verbose && !STRISEMPTY(path))
		    g_printerr(
"%s: Line %i: Unsupported identifier \"%s\".\n",
			path, lines_read + 1, id
		    );
		status = -1;
	    }

	    if(c == EOF)
		break;

#undef GET_IDENTIFIER
#undef GET_IDENTIFIER_WHOLE_LINE
	}

	return(status);
}


/*
 *	Opens the IceWM Menu configuration.
 */
static gint WinFIOOpenIceWM(
	win_struct *win, const gchar *path,
	gboolean verbose, gboolean interactive
)
{
	gchar	*menu_file = NULL,
		*tool_bar_file = NULL,
		*parent_dir = NULL;
	GtkCTreeNode *parent_node = NULL;
	GtkCTree *ctree = GTK_CTREE(win->ctree);
	obj_struct *obj, *parent_obj = NULL;
	core_struct *core = CORE(win->core);

#define FREE_ALL	{		\
 g_free(menu_file);			\
 menu_file = NULL;			\
 g_free(tool_bar_file);			\
 tool_bar_file = NULL;			\
 g_free(parent_dir);			\
 parent_dir = NULL;			\
}

	/* Get path to menu configuration file
	 *
	 * Use alternate menu configuration file if specified, otherwise
	 * use the menu configuration file in the user's home directory
	 * regardless if it exists or not
	 */
	if(path != NULL)
	{
	    if(ISPATHDIR(path))
		menu_file = STRDUP(PrefixPaths(path, "menu"));
	    else
		menu_file = STRDUP(path);
	}
	else
	{
	    menu_file = STRDUP(PrefixPaths(
		core->home_dir, ".icewm/menu"
	    ));
	}
	if(menu_file == NULL)
	{
	    if(verbose && (path != NULL))
		g_printerr(
"%s: Unable to parse path.\n",
		    path
		);
	    FREE_ALL
	    return(-1);
	}

	/* Parse parent directory and tool bar file from the menu
	 * file path
	 */
	parent_dir = STRDUP(GetParentDir(menu_file));
	tool_bar_file = STRDUP(PrefixPaths(parent_dir, "toolbar"));
	if((tool_bar_file == NULL) || (parent_dir == NULL))
	{
	    if(verbose)
		g_printerr(
"%s: Unable to parse path.\n",
		    menu_file
		);
	    FREE_ALL
	    return(-1);
	}

	/* Record the new menu configuration file on the Win */
	g_free(win->filename);
	win->filename = STRDUP(menu_file);

	/* Report initial progress */
	if(verbose)
	{
	    gchar *buf = g_strdup_printf(
		"Opening menu configuration: %s",
		menu_file
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	    WinStatusProgress(win, -1.0f, TRUE);
	}

	/* Get icon paths */
	g_list_foreach(win->icon_paths, (GFunc)g_free, NULL);
	g_list_free(win->icon_paths);
	win->icon_paths = WinFIOGetIconPathsIceWM(win);

	/* Create toplevel Start Menu Object */
	win->obj_menu_toplevel = parent_obj = obj = ObjNewWithValues(
	    OBJ_TYPE_GROUP,
	    OBJ_FLAG_READ_ONLY,
	    NULL, NULL,
	    "Start Menu",
	    ICEWM_DEF_ICON_NAME_GROUP,
	    NULL, NULL, NULL,
	    GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
	    NULL, NULL, NULL, NULL,
	    '\0', 0,
	    0l
	);
	if(obj != NULL)
	{
	    obj->pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		&obj->mask,
		(guint8 **)icon_folder_menus_20x20_xpm
	    );
	    obj->pixmap_opened = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		&obj->mask_opened,
		(guint8 **)icon_folder_menus_opened_20x20_xpm
	    );
	    parent_node = WinTreeInsert(win, NULL, NULL, obj);
	}
	if((parent_obj != NULL) && (parent_node != NULL))
	{
	    /* Open the Menu configuration file */
	    FILE *fp = FOpen(menu_file, "rb");
	    if(fp != NULL)
	    {
		WinFIOOpenIceWMFile(
		    win, menu_file, fp,
		    parent_obj, parent_node,
		    verbose, interactive
	        );
		FClose(fp);
	    }

	    /* Expand and select the parent node */
	    gtk_ctree_expand(ctree, parent_node);
	    gtk_ctree_select(ctree, parent_node);
	}

	/* Create toplevel Tool Bar Object */
	win->obj_tool_bar_toplevel = parent_obj = obj = ObjNewWithValues(
	    OBJ_TYPE_GROUP,
	    OBJ_FLAG_READ_ONLY | OBJ_FLAG_NO_SUBGROUPS,
	    NULL, NULL,
	    "Tool Bar",
	    ICEWM_DEF_ICON_NAME_GROUP,
	    NULL, NULL, NULL,
	    GTK_ORIENTATION_HORIZONTAL, GTK_JUSTIFY_LEFT,
	    NULL, NULL, NULL, NULL,
	    '\0', 0,
	    0l
	);
	if(obj != NULL)
	{
	    obj->pixmap = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		&obj->mask,
		(guint8 **)icon_folder_tool_bar_20x20_xpm
	    );
	    obj->pixmap_opened = GDK_PIXMAP_NEW_FROM_XPM_DATA(
		&obj->mask_opened,
		(guint8 **)icon_folder_tool_bar_opened_20x20_xpm
	    );
	    parent_node = WinTreeInsert(win, NULL, NULL, obj);
	}
	if((parent_obj != NULL) && (parent_node != NULL))
	{
	    /* Open the Tool Bar configuration file */
	    FILE *fp = FOpen(tool_bar_file, "rb");
	    if(fp != NULL)
	    {
		WinFIOOpenIceWMFile(
		    win, tool_bar_file, fp,
		    parent_obj, parent_node,
		    verbose, interactive
		);
		FClose(fp);
	    }

	    /* Expand the parent node */
	    gtk_ctree_expand(ctree, parent_node);
	}


	FREE_ALL

	return(0);
#undef FREE_ALL
}

/*
 *	Opens the menus to the Win from file.
 *
 *	If a path is specified then this may override the default
 *	path to the menu configuration file or directory for certain
 *	menu formats.
 */
gint WinFIOOpen(
	win_struct *win, const gchar *path,
	menu_format format,
	gboolean verbose, gboolean interactive
)
{
	gint status = -1;
	obj_struct *obj;

	if(win == NULL)
	    return(status);

	/* Report initial progress */
	if(verbose)
	{
	    WinStatusMessage(win, "Opening menu configuration...", FALSE);
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	/* Delete toplevel Objects */
	obj = win->obj_tool_bar_toplevel;
	if(obj != NULL)
	{
	    WinObjRemovedCB(win, obj);
	    ObjDelete(obj);
	    win->obj_tool_bar_toplevel = obj = NULL;
	}
	obj = win->obj_menu_toplevel;
	if(obj != NULL)
	{
	    WinObjRemovedCB(win, obj);
	    ObjDelete(obj);
	    win->obj_menu_toplevel = obj = NULL;
	}
	

	/* Record the new format and reset the file name
	 *
	 * The file name will be set later when the exact location
	 * is determined
	 */
	win->format = format;

	g_free(win->filename);
	win->filename = NULL;

	win->read_only = FALSE;

	g_list_foreach(win->icon_paths, (GFunc)g_free, NULL);
	g_list_free(win->icon_paths);
	win->icon_paths = NULL;

	/* Open menu configuration by format */
	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    status = -2;
	    break;

	  case MENU_FORMAT_ICEWM:
	    status = WinFIOOpenIceWM(win, path, verbose, interactive);
	    break;
	}

	/* Report final progress */
	if(verbose)
	{
	    WinStatusMessage(
		win,
		((status == 0) || (status == -4)) ?
		    "Menu configuration opened" :
		    "Error opening menu configuration",
		FALSE
	    );
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	return(status);
}


/*
 *	Called by WinFIOSaveIceWM().
 *
 *	Saves the specified Object and all its child Objects.
 */
static void WinFIOSaveIceWMIterate(
	win_struct *win, const gchar *path, FILE *fp,
	obj_struct *obj, gint *level,
	gboolean verbose, gboolean interactive
)
{
	const gchar *name, *icon_name;
	GList *glist;
	obj_struct *dest_obj;

	if(obj == NULL)
	    return;

#define WRITE_INDENT(_l_)	{	\
 gint i, m = (_l_);			\
 for(i = 0; i < m; i++)			\
  fputc('\t', fp);			\
}

	/* Report progress */
	if(verbose)
	    WinStatusProgress(win, -1.0f, TRUE);

	/* Get the name of the Object, use the default name if the
	 * Object does not have a name
	 */
	name = obj->name;
	if(STRISEMPTY(name))
	    name = "Untitled";

	/* Get the icon name */
	icon_name = obj->icon_name;

	/* Save Object by its type */
	switch(OBJ_TYPE(obj))
	{
	  case OBJ_TYPE_ITEM:
	    WRITE_INDENT(*level);
	    if((obj->accel_key != '\0') && verbose)
		g_printerr(
"%s: Item \"%s\": IceWM does not support accelerator keys.\n",
		    path, name
		);
	    fprintf(
		fp,
		"prog \"%s\" \"%s\" %s\n",
		name,
		!STRISEMPTY(icon_name) ? icon_name : ICEWM_DEF_ICON_NAME_ITEM,
		obj->value
	    );
	    break;

	  case OBJ_TYPE_ITEM_LINK:
	    /* Save dereferenced location (get link destination Object
	     * and save it)
	     */
	    dest_obj = WinObjGetLinkObj(win, obj);
	    if((dest_obj != NULL) ? !OBJ_IS_LINK(dest_obj) : FALSE)
	    {
		if(verbose)
		    g_printerr(
"%s: Dereferencing link \"%s\" and saving destination object \"%s\".\n",
			path, name, dest_obj->name
		    );
		WinFIOSaveIceWMIterate(
		    win, path, fp, dest_obj, level,
		    verbose, interactive
		);
	    }
	    else
	    {
		if(verbose)
		    g_printerr(
"%s: Link \"%s\" destination is not valid.\n",
			path, name
		    );
	    }
	    break;

	  case OBJ_TYPE_ITEM_SEPARATOR:
	    WRITE_INDENT(*level);
	    fprintf(fp, "separator\n");
	    break;

	  case OBJ_TYPE_ITEM_SPECIAL:
	    if(!g_strcasecmp(name, "startup") ||
	       !g_strcasecmp(name, "restart") ||
	       !g_strcasecmp(name, "shutdown") ||
	       !g_strcasecmp(name, "menuprog") ||
	       !g_strcasecmp(name, "menufile")
	    )
	    {
		WRITE_INDENT(*level);
		if((obj->accel_key != '\0') && verbose)
		    g_printerr(
"%s: Special object \"%s\": IceWM does not support accelerator keys.\n",
			path, name
		    );
		fprintf(
		    fp,
		    "%s %s\n",
		    name,		/* Name is the type identifier */
		    obj->value
		);
	    }
	    else
	    {
		if(verbose)
		    g_printerr(
"%s: Special object \"%s\" is not supported by IceWM.\n",
			path, name
		    );  
	    }
	    break;

	  case OBJ_TYPE_GROUP:
	    WRITE_INDENT(*level);
	    fprintf(
		fp,
		"menu \"%s\" \"%s\" {\n",
		name,
		!STRISEMPTY(icon_name) ? icon_name : ICEWM_DEF_ICON_NAME_GROUP
	    );

	    /* Push level and save all child Objects in this Group */
	    *level = (*level) + 1;
	    for(glist = OBJ_CHILDREN(obj); glist != NULL; glist = g_list_next(glist))
		WinFIOSaveIceWMIterate(
		    win, path, fp, OBJ(glist->data),
		    level,
		    verbose, interactive
		);
	    *level = (*level) - 1;

	    WRITE_INDENT(*level);
	    fprintf(fp, "}\n");

	    break;
	}

#undef WRITE_INDENT
}

/*
 *	Saves the IceWM Menu configuration.
 */
static gint WinFIOSaveIceWM(
	win_struct *win, const gchar *path,
	gboolean verbose, gboolean interactive
)
{
	gint status = 0;
	gchar	*menu_file = NULL,
		*tool_bar_file = NULL,
		*parent_dir = NULL;
	FILE *fp;

#define FREE_ALL        {               \
 g_free(menu_file);                     \
 menu_file = NULL;                      \
 g_free(tool_bar_file);                 \
 tool_bar_file = NULL;                  \
 g_free(parent_dir);			\
 parent_dir = NULL;			\
}

	if(STRISEMPTY(path))
	{
	    FREE_ALL
	    return(-1);
	}

	/* Get menu configuration file, tool bar file, and the parent
	 * directory from the specified path
	 *
	 * The specified path must be the menu configuration file
	 */
	menu_file = STRDUP(path);
	parent_dir = STRDUP(GetParentDir(menu_file));
	tool_bar_file = STRDUP(PrefixPaths(parent_dir, "toolbar"));	    
	if((menu_file == NULL) || (tool_bar_file == NULL) ||
	   (parent_dir == NULL)
	)
	{
	    if(verbose)
		g_printerr(
"%s: Unable to parse path\n",
		    path
		);
	    FREE_ALL
	    return(-1);
	}

	/* Create parent directory and any of its compoents as needed */
	if(rmkdir(parent_dir, S_IRUSR | S_IWUSR | S_IXUSR))
	{
	    if(verbose)
		g_printerr(
"%s: Unable to create directory.\n",
		    parent_dir
		);
	    FREE_ALL
	    return(-1);
	}

	/* Open the Menu configuration for writing, create as needed,
	 * and save all menu Objects
	 */
	fp = FOpen(menu_file, "wb");
	if(fp != NULL)
	{
	    obj_struct *obj = win->obj_menu_toplevel;
	    if(OBJ_IS_GROUP(obj))
	    {
		gint level = 0;
		GList *glist;

		for(glist = OBJ_CHILDREN(obj);
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		    WinFIOSaveIceWMIterate(
			win, menu_file, fp, OBJ(glist->data),
			&level,
			verbose, interactive
		    );
	    }
	    FClose(fp);
	}
	else
	{
	    if(verbose)
		g_printerr(
"%s: Unable to open file for writing.\n",
		    menu_file
		);
	    status = -1;
	}

	/* Open the Tool Bar configuration for writing, create as
	 * needed, and save all menu Objects
	 */
	fp = FOpen(tool_bar_file, "wb");
	if(fp != NULL)
	{
	    obj_struct *obj = win->obj_tool_bar_toplevel;
	    if(OBJ_IS_GROUP(obj))
	    {
		gint level = 0;
		GList *glist;

		for(glist = OBJ_CHILDREN(obj);
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		    WinFIOSaveIceWMIterate(
			win, tool_bar_file, fp, OBJ(glist->data),
			&level,
			verbose, interactive
		    );
	    }
	    FClose(fp);
	}
	else
	{
	    if(verbose)
		g_printerr(
"%s: Unable to open file for writing.\n",
		    tool_bar_file
		);
	    status = -1;
	}

	FREE_ALL

	return(status);
#undef FREE_ALL
}

/*
 *	Saves the menus on the Win to file.
 *
 *      If a path is specified then this may override the default
 *      path to the menu configuration file or directory for certain
 *      menu formats.
 */
gint WinFIOSave(
	win_struct *win, const gchar *path,
	menu_format format,
	gboolean verbose, gboolean interactive
)
{
	gint status = -1;

	if(win == NULL)
	    return(status);

	/* Report initial progress */
	if(verbose)
	{
	    WinStatusMessage(win, "Saving menu configuration...", FALSE);
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	/* Save menu configuration by format */
	switch(win->format)
	{
	  case MENU_FORMAT_ENDEAVOUR2:
	    status = -2;
	    break;

	  case MENU_FORMAT_ICEWM:
	    status = WinFIOSaveIceWM(win, path, verbose, interactive);
	    break;
	}

	/* Report final progress */
	if(verbose)
	{
	    WinStatusMessage(
		win,
		((status == 0) || (status == -4)) ?
		    "Menu configuration saved" :
		    "Error saving menu configuration",
		FALSE
	    );
	    WinStatusProgress(win, 0.0f, TRUE);
	}

	return(status);
}
