#include <time.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "obj.h"
#include "win.h"
#include "winlist.h"
#include "config.h"


static gchar *WinTimeStr(gulong t, gulong ct);


/* List Add, Update, and Remove */
gint WinListInsert(
	win_struct *win, gint row,
	obj_struct *obj
);
gint WinListAppend(
	win_struct *win,
	obj_struct *obj
);
void WinListRemove(win_struct *win, gint row);
void WinListClear(win_struct *win);
void WinListUpdateRow(
	win_struct *win, gint row, obj_struct *obj
);
void WinListUpdate(win_struct *win, obj_struct *obj_group);

/* List Utils */
gboolean WinListFindSelectObj(win_struct *win, obj_struct *obj);


/* Tree Add, Update, and Remove */
GtkCTreeNode *WinTreeInsert(
	win_struct *win, GtkCTreeNode *parent, GtkCTreeNode *sibling,
	obj_struct *obj
);
void WinTreeClear(win_struct *win);
void WinTreeUpdateNode(     
	win_struct *win, GtkCTreeNode *node, obj_struct *obj
);

/* Tree Utils */
GtkCTreeNode *WinTreeGetToplevelNode(win_struct *win);
obj_struct *WinTreeGetLocation(win_struct *win);
gboolean WinTreeFindSelectObj(win_struct *win, obj_struct *obj);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a statically allocated string describing the christian
 *	time t using current time ct to abbriviate.
 */
static gchar *WinTimeStr(gulong t, gulong ct)
{
	time_t tv = (time_t)t;
	const gchar *fmt;
	const struct tm *tm_ptr;
	static gchar buf[80];

	*buf = '\0';

	if(t == 0)
	    return(buf);

	/* Less than 6 months old? */
	if((ct - t) < (3600 * 24 * 28 * 6))
	    fmt = "%b %d %H:%M";
	else
	    fmt = "%b %d %Y";

	/* Format time value */
	tm_ptr = localtime(&tv);

	/* Format time string */
	if(strftime(buf, sizeof(buf), fmt, tm_ptr) <= 0)
	    *buf = '\0';

	return(buf);
}

/*
 *	Inserts the Object to the Win's List at the specified row.
 *
 *	The Object will be transfered and should not be referenced
 *	after this call.
 */
gint WinListInsert(
	win_struct *win, gint row,
	obj_struct *obj                 /* Transfered */
)
{
	gint i, columns, new_row;
	gchar **strv;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return(-1);

	columns = clist->columns;
	if(columns <= 0)
	    return(-1);    

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)
	    strv[i] = "";

	/* Insert or append new row */
	if((row >= 0) && (row < clist->rows))
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete row cell values */
	g_free(strv);

	/* Failed to create new row? */
	if(new_row < 0)
	    return(-1);    

	/* Set the Object as the row data */
	gtk_clist_set_row_data(clist, new_row, obj);

	/* Update the row cell values with the new object */
	WinListUpdateRow(win, new_row, obj);

	return(new_row);
}

/*
 *	Appends the Object to the Win's List.
 *
 *	The Object will be transfered and should not be referenced
 *	after this call.
 */
gint WinListAppend(
	win_struct *win,
	obj_struct *obj                 /* Transfered */
)
{
	return(WinListInsert(win, -1, obj));
}

/*
 *	Removes the specified row on the Win's List.
 */
void WinListRemove(win_struct *win, gint row)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_remove(clist, row);
}

/*
 *	Deletes all rows on the Win's List.
 */
void WinListClear(win_struct *win)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return;

	gtk_clist_clear(clist);
}

/*
 *	Updates the specified row on the Win's List with the values
 *	specified in the Object.
 */
void WinListUpdateRow(win_struct *win, gint row, obj_struct *obj)
{
	gint column, columns;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return;

	if((row < 0) || (row >= clist->rows))
	    return;

	columns = clist->columns;
	if(columns <= 0)
	    return;

	/* Name */
	column = 0;
	if(column < columns)
	{
	    gchar *s = (obj != NULL) ? obj->name : NULL;
	    if((obj != NULL) ? (obj->pixmap != NULL) : FALSE)
		gtk_clist_set_pixtext(
		    clist, row, column,
		    (s != NULL) ? s : "",
		    WIN_LIST_PIXMAP_TEXT_SPACING,
		    obj->pixmap, obj->mask
		);
	    else
		gtk_clist_set_text(
		    clist, row, column,
		    (s != NULL) ? s : ""
		);
	}

	/* Type */
	column = 1;
	if(column < columns)
	{
	    gchar *s = NULL;
	    switch(obj->type)
	    {
	      case OBJ_TYPE_ITEM:
		s = "Item";
		break;
	      case OBJ_TYPE_ITEM_LINK:
		s = "Link";
		break;
	      case OBJ_TYPE_ITEM_SEPARATOR:
		s = "Separator";
		break;
	      case OBJ_TYPE_ITEM_SPECIAL:
		s = "Special";
		break;
	      case OBJ_TYPE_GROUP:
		s = "Group";
		break;
	    }
	    gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	    );    
	}     

	/* Value */
	column = 2;
	if(column < columns)
	{
	    gchar *s = (obj != NULL) ? obj->value : NULL;
	    gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	    );
	}

	/* Date Modified */
	column = 3;
	if(column < columns) 
	{
	    gchar *s = WinTimeStr(obj->last_modified, (gulong)time(NULL));
	    gtk_clist_set_text(
		clist, row, column,
		(s != NULL) ? s : ""
	    );
	}

}

/*
 *	Clears and updates the Win's List with the child objects in
 *	the specified object group.
 */
void WinListUpdate(win_struct *win, obj_struct *obj_group)
{
	GList *glist;
	obj_struct *obj;
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->clist : NULL;
	if(clist == NULL)
	    return;

	/* Delete all rows */
	gtk_clist_clear(clist);

	/* No specified Object or Object is not a Group? */
	if(!OBJ_IS_GROUP(obj_group))
	    return;

	/* Add the Group's child Objects to the list */
	for(glist = OBJ_CHILDREN(obj_group);
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    obj = OBJ(glist->data);
	    if(obj == NULL)
		continue;

	    WinListAppend(win, obj);
	}
}


/*
 *	Searches for the Win's List for the specified Object
 *	and selects the Object if found.
 *
 *	Returns TRUE if the Object was found.
 */
gboolean WinListFindSelectObj(win_struct *win, obj_struct *obj)
{
	gint row;
	GtkCList *clist = (win != NULL) ? 
	    (GtkCList *)win->clist : NULL;
	if((clist == NULL) || (obj == NULL))
	    return(FALSE);

	row = gtk_clist_find_row_from_data(clist, obj);
	if(row < 0)
	    return(FALSE);

	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(clist, row, 0);

	return(TRUE);
}


/*
 *	Inserts the Object to the Win's Tree to the specified parent
 *	node after the specified sibling node.
 *
 *	The Object will be transfered and should not be referenced
 *	after this call.
 */
GtkCTreeNode *WinTreeInsert(
	win_struct *win, GtkCTreeNode *parent, GtkCTreeNode *sibling,
	obj_struct *obj
)
{
	gint i, columns;
	gchar **strv;
	GtkCTreeNode *node;
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	if(ctree == NULL)
	    return(NULL);

	columns = GTK_CLIST(ctree)->columns;
	if(columns <= 0)
	    return(NULL);

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)                         
	    strv[i] = "";

	/* Insert new node */
	node = gtk_ctree_insert_node(
	    ctree, parent, sibling,
	    strv,
	    WIN_LIST_PIXMAP_TEXT_SPACING,
	    NULL, NULL,
	    NULL, NULL,
	    FALSE,			/* Is Leaf? */
	    FALSE			/* Expanded? */
	);

	/* Delete row cell values */
	g_free(strv);               
		     
	/* Failed to create new node? */
	if(node == NULL)
	    return(NULL);

	/* Set the Object as the row data */
	gtk_ctree_node_set_row_data(ctree, node, obj);

	/* Update the row cell values with the new object */
	WinTreeUpdateNode(win, node, obj);

	return(node);

}

/*
 *	Deletes all nodes on the Win's Tree.
 */
void WinTreeClear(win_struct *win)
{
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	GtkCTreeNode *node = WinTreeGetToplevelNode(win);
	if(node != NULL)
	    gtk_ctree_remove_node(ctree, node);
}

/*
 *	Updates the specified node on the Win's Tree with the values
 *	specified in the Object.
 */
void WinTreeUpdateNode(     
	win_struct *win, GtkCTreeNode *node, obj_struct *obj
)
{
	gint column, columns;
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	if((ctree == NULL) || (node == NULL) || (obj == NULL))
	    return;

	columns = GTK_CLIST(ctree)->columns;
	if(columns <= 0)
	    return;

	column = 0;
	if(column < columns)
	{
	    gchar *s = (obj != NULL) ? obj->name : NULL;
	    gtk_ctree_set_node_info(
		ctree, node,
		(s != NULL) ? s : "",
		WIN_LIST_PIXMAP_TEXT_SPACING,
		obj->pixmap, obj->mask,
		obj->pixmap_opened, obj->mask_opened,
		FALSE,			/* Is Leaf? */
		FALSE			/* Expanded? */
	    );
	}     
}


/*
 *      Returns the Win Tree's toplevel node.
 */
GtkCTreeNode *WinTreeGetToplevelNode(win_struct *win)
{
	GtkCList *clist = (win != NULL) ?
	    (GtkCList *)win->ctree : NULL;  
	return((clist != NULL) ?
	    (GtkCTreeNode *)g_list_nth(clist->row_list, 0) : NULL
	);
}

/*
 *	Returns the Win Tree's currently selected Object.
 */
obj_struct *WinTreeGetLocation(win_struct *win)
{
	GList *glist;
	GtkCTreeNode *node;
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	if(ctree == NULL)
	    return(NULL);

	glist = GTK_CLIST(ctree)->selection_end;
	node = (GtkCTreeNode *)((glist != NULL) ? glist->data : NULL);
	return(OBJ(gtk_ctree_node_get_row_data(ctree, node)));
}

/*
 *	Searches for the Win's Tree for the specified Object, expanding
 *	branches as needed and selecting the Object if found.
 *
 *	Returns TRUE if the Object was found.
 */
gboolean WinTreeFindSelectObj(win_struct *win, obj_struct *obj)
{
	GtkCTreeRow *row;
	GtkCTreeNode *node, *matched_node;
	GtkCTree *ctree = (win != NULL) ?
	    (GtkCTree *)win->ctree : NULL;
	if((ctree == NULL) || (obj == NULL))
	    return(FALSE);

	/* Search for node who's data matches the specified Object */
	matched_node = node = gtk_ctree_find_by_row_data(ctree, NULL, obj);
	if(node == NULL)
	    return(FALSE);

	/* Expand all parent nodes as needed */
	row = GTK_CTREE_ROW(node);
	node = (row != NULL) ? row->parent : NULL;
	while(node != NULL)
	{
	    row = GTK_CTREE_ROW(node);
	    if(row == NULL)
		break;

	    if(!row->expanded)
		gtk_ctree_expand(ctree, node);
	    node = row->parent;
	}

	gtk_ctree_select(ctree, matched_node);

	return(TRUE);
}
