//////////////////////////////////////////////////////////////////////////////
// newstepstyle.cc
// -------------------
// A style for KDE
// -------------------
// Copyright (c) 2004-2005 David Johnson
// Please see the header file for copyright and license information.
//////////////////////////////////////////////////////////////////////////////

#include <kdemacros.h>
#include <kdrawutil.h>
#include <kpixmap.h>
#include <kpixmapeffect.h>

#include <qapplication.h>
#include <qintdict.h>
#include <qpainter.h>
#include <qpointarray.h>
#include <qsettings.h>

#include <qcombobox.h>
#include <qheader.h>
#include <qmainwindow.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qprogressbar.h>
#include <qpushbutton.h>
#include <qscrollbar.h>
#include <qslider.h>
#include <qstyleplugin.h>
#include <qtabbar.h>
#include <qtabwidget.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>

#include "newstepstyle.h"
#include "bitmaps.h"

// some convenient constants

static const int MINICONSIZE     = 16;
static const int MAXGRADIENTSIZE = 64;

static const char* QTOOLBAREXTENSION  = "QToolBarExtensionWidget";
static const char* KTOOLBARWIDGET     = "kde toolbar widget";

static const int ITEMFRAME       = 1; // menuitem stuff
static const int ITEMHMARGIN     = 3;
static const int ITEMVMARGIN     = 0;
static const int ARROWMARGIN     = 6;
static const int RIGHTBORDER     = 6;

static bool lightscheme       = false;
static unsigned contrast      = 100;
static bool nextscroll        = true;
QMap<unsigned int, QIntDict<GradientSet> > gradients; // gradients[color][size]

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction, Initialization                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// NewstepStyle()
// -----------
// Constructor

NewstepStyle::NewstepStyle(bool classic)
    : KStyle(FilledFrameWorkaround | AllowMenuTransparency), hover_(0)
{
    QSettings settings;

    // "classic" theme overrides all options
    if (classic) {
        nexttabs_ = true;
        nextmenus_ = true;
        pedantic_ = true;
        highlights_ = false;
        gradients_ = false;
    } else {
        pedantic_ =
            settings.readBoolEntry("/newstepstyle/Settings/pedantic", false);
        nexttabs_ =
            settings.readBoolEntry("/newstepstyle/Settings/nexttabs", false);
        nextmenus_ =
            settings.readBoolEntry("/newstepstyle/Settings/nextmenus", false);
        highlights_ =
            settings.readBoolEntry("/newstepstyle/Settings/highlights", true);
        gradients_ = (QPixmap::defaultDepth() > 8) &&
            settings.readBoolEntry("/newstepstyle/Settings/gradients", true);
    }

    if (gradients_) {
        contrast = 100 + settings.readNumEntry("/Qt/KDE/contrast", 5);
    }

    // nextmenus_ doesn't work with translucent menus
    if (nextmenus_ &&
        settings.readEntry("/KStyle/Settings/MenuTransparencyEngine") != "Disabled") {
        nextmenus_ = false;
    }

    reverse_ = QApplication::reverseLayout();
    nextscroll = ((pedantic_ && !reverse_) || (!pedantic_ && reverse_));

    // create bitmaps
    radio_shadow = QBitmap(radiosz, radiosz, radio_shadow_bits, true);
    radiooff_dark = QBitmap(radiosz, radiosz, radiooff_dark_bits, true);
    radiooff_mid = QBitmap(radiosz, radiosz, radiooff_mid_bits, true);
    radiooff_light = QBitmap(radiosz, radiosz, radiooff_light_bits, true);
    radioon_dark = QBitmap(radiosz, radiosz, radioon_dark_bits, true);
    radioon_mid = QBitmap(radiosz, radiosz, radioon_mid_bits, true);
    radioon_light = QBitmap(radiosz, radiosz, radioon_light_bits, true);
    radioon_light.setMask(radioon_light);
    radiomask = QBitmap(radiosz, radiosz, radiomask_bits, true);
    radiomask.setMask(radiomask);

    dot_shadow = QBitmap(dotsz, dotsz, dot_shadow_bits, true);
    dot_dark = QBitmap(dotsz, dotsz, dot_dark_bits, true);
    dot_light = QBitmap(dotsz, dotsz, dot_light_bits, true);

    check_shadow = QBitmap(checksz, checksz, check_shadow_bits, true);
    check_dark = QBitmap(checksz, checksz, check_dark_bits, true);
    check_light = QBitmap(checksz, checksz, check_light_bits, true);

    uarrow_shadow = QBitmap(arrowsz, arrowsz, uarrow_shadow_bits, true);
    uarrow_dark = QBitmap(arrowsz, arrowsz, uarrow_dark_bits, true);
    larrow_shadow = QBitmap(arrowsz, arrowsz, larrow_shadow_bits, true);
    larrow_dark = QBitmap(arrowsz, arrowsz, larrow_dark_bits, true);
    darrow_shadow = QBitmap(arrowsz, arrowsz, darrow_shadow_bits, true);
    darrow_dark = QBitmap(arrowsz, arrowsz, darrow_dark_bits, true);
    rarrow_shadow = QBitmap(arrowsz, arrowsz, rarrow_shadow_bits, true);
    rarrow_dark = QBitmap(arrowsz, arrowsz, rarrow_dark_bits, true);

    usunkarr_shadow = QBitmap(arrowsz, arrowsz, usunkarr_shadow_bits, true);
    usunkarr_dark = QBitmap(arrowsz, arrowsz, usunkarr_dark_bits, true);
    usunkarr_light = QBitmap(arrowsz, arrowsz, usunkarr_light_bits, true);
    lsunkarr_shadow = QBitmap(arrowsz, arrowsz, lsunkarr_shadow_bits, true);
    lsunkarr_dark = QBitmap(arrowsz, arrowsz, lsunkarr_dark_bits, true);
    lsunkarr_light = QBitmap(arrowsz, arrowsz, lsunkarr_light_bits, true);
    dsunkarr_shadow = QBitmap(arrowsz, arrowsz, dsunkarr_shadow_bits, true);
    dsunkarr_dark = QBitmap(arrowsz, arrowsz, dsunkarr_dark_bits, true);
    dsunkarr_light = QBitmap(arrowsz, arrowsz, dsunkarr_light_bits, true);
    rsunkarr_shadow = QBitmap(arrowsz, arrowsz, rsunkarr_shadow_bits, true);
    rsunkarr_dark = QBitmap(arrowsz, arrowsz, rsunkarr_dark_bits, true);
    rsunkarr_light = QBitmap(arrowsz, arrowsz, rsunkarr_light_bits, true);

    usmarrow_shadow = QBitmap(sarrowsz, sarrowsz, usmarrow_shadow_bits, true);
    usmarrow_dark = QBitmap(sarrowsz, sarrowsz, usmarrow_dark_bits, true);
    lsmarrow_shadow = QBitmap(sarrowsz, sarrowsz, lsmarrow_shadow_bits, true);
    lsmarrow_dark = QBitmap(sarrowsz, sarrowsz, lsmarrow_dark_bits, true);
    dsmarrow_shadow = QBitmap(sarrowsz, sarrowsz, dsmarrow_shadow_bits, true);
    dsmarrow_dark = QBitmap(sarrowsz, sarrowsz, dsmarrow_dark_bits, true);
    rsmarrow_shadow = QBitmap(sarrowsz, sarrowsz, rsmarrow_shadow_bits, true);
    rsmarrow_dark = QBitmap(sarrowsz, sarrowsz, rsmarrow_dark_bits, true);

    return_shadow = QBitmap(returnw, returnh, return_shadow_bits, true);
    return_dark = QBitmap(returnw, returnh, return_dark_bits, true);
    return_light = QBitmap(returnw, returnh, return_light_bits, true);
}

NewstepStyle::~NewstepStyle() { ; }

//////////////////////////////////////////////////////////////////////////////
// Polishing                                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// polish()
// --------
// Initialize the appearance of a widget

void NewstepStyle::polish(QWidget *widget)
{
    if (::qt_cast<QMenuBar*>(widget) ||
        ::qt_cast<QPopupMenu*>(widget)) {
        widget->setBackgroundMode(NoBackground);
    } else if (highlights_ && // highlighting
               (::qt_cast<QPushButton*>(widget) ||
                ::qt_cast<QComboBox*>(widget) ||
                ::qt_cast<QSpinWidget*>(widget) ||
                ::qt_cast<QSlider*>(widget))) {
        widget->installEventFilter(this);
    } else if (widget->inherits(QTOOLBAREXTENSION) ||
               (!qstrcmp(widget->name(), KTOOLBARWIDGET))) {
        widget->installEventFilter(this);
    }

    KStyle::polish(widget);
}

//////////////////////////////////////////////////////////////////////////////
// unPolish()
// ----------
// Undo the initialization of a widget's appearance

void NewstepStyle::unPolish(QWidget *widget)
{
    if (::qt_cast<QMenuBar*>(widget) ||
        ::qt_cast<QPopupMenu*>(widget)) {
        widget->setBackgroundMode(PaletteBackground);
    } else if (highlights_ && // highlighting
               (::qt_cast<QPushButton*>(widget) ||
                ::qt_cast<QComboBox*>(widget) ||
                ::qt_cast<QSpinWidget*>(widget) ||
                ::qt_cast<QSlider*>(widget))) {
        widget->removeEventFilter(this);
    } else if (widget->inherits(QTOOLBAREXTENSION) ||
               (!qstrcmp(widget->name(), KTOOLBARWIDGET))) {
        widget->removeEventFilter(this);
    }

    KStyle::unPolish(widget);
}

//////////////////////////////////////////////////////////////////////////////
// polish()
// --------
// Initialize the palette

void NewstepStyle::polish(QPalette &pal)
{
    gradients.clear(); // clear out gradients on a color change

    int h, s, v;
    pal.color(QPalette::Active, QColorGroup::Background).getHsv(&h, &s, &v);
    lightscheme = (v > 224);

    // ensure white/black for light/shadow
    pal.setColor(QPalette::Disabled, QColorGroup::Light, Qt::white);
    pal.setColor(QPalette::Active, QColorGroup::Light, Qt::white);
    pal.setColor(QPalette::Inactive, QColorGroup::Light, Qt::white);
    pal.setColor(QPalette::Disabled, QColorGroup::Shadow, Qt::black);
    pal.setColor(QPalette::Active, QColorGroup::Shadow, Qt::black);
    pal.setColor(QPalette::Inactive, QColorGroup::Shadow, Qt::black);

    QStyle::polish(pal);
}

//////////////////////////////////////////////////////////////////////////////
// Drawing                                                                  //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// drawNewstepGradient()
// ------------------
// Draw gradient

void NewstepStyle::drawNewstepGradient(QPainter *painter,
                                       const QRect &rect,
                                       QColor color,
                                       bool horizontal,
                                       int px, int py,
                                       int pw, int ph) const
{
    if (!gradients_) {
        painter->fillRect(rect, color);
        return;
    }

    // px, py, pw, ph used for parent-relative pixmaps
    int size;
    if (horizontal)
        size = (pw > 0) ? pw : rect.width();
    else
        size = (ph > 0) ? ph : rect.height();

    if (size > MAXGRADIENTSIZE) { // keep it sensible
        painter->fillRect(rect, color);
    } else {
        // lazy allocation
        GradientSet *set = gradients[color.rgb()][size];
        if (!set) {
            set = new GradientSet(color, size);
            gradients[color.rgb()].setAutoDelete(true);
            gradients[color.rgb()].insert(size, set);
        }
        painter->drawTiledPixmap(rect, *set->gradient(horizontal),
                                 QPoint(px, py));
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawNewstepBevel()
// ------------------
// Draw the basic Newstep bevel, button

void NewstepStyle::drawNewstepBevel(QPainter *painter,
                                    int x, int y, int w, int h,
                                    const QColorGroup &group,
                                    bool sunken,
                                    bool horizontal,
                                    bool flat,
                                    const QBrush *fill) const
{
    int x2 = x + w - 1;
    int y2 = y + h - 1;
    painter->save();

    // fill in the bevel
    if (flat) {
        painter->fillRect(x+1, y+1, w-2, h-2, fill ? *fill : group.button());
    } else {
        drawNewstepGradient(painter, QRect(x+1, y+1, w-2, h-2),
                            fill ? fill->color() : group.button(), horizontal,
                            0, 0, 0, 0);
    }

    // draw bevel edges
    if (sunken) {
        painter->setPen(group.shadow());
        painter->drawLine(x, y, x2-1, y);
        painter->drawLine(x, y, x, y2-1);
        painter->setPen(group.light());
        painter->drawLine(x, y2, x2, y2);
        painter->drawLine(x2, y, x2, y2);
    } else {
        painter->setPen(group.light());
        painter->drawLine(x, y, x2-1, y);
        painter->drawLine(x, y, x, y2-1);
        painter->setPen(group.dark());
        painter->drawLine(x+1, y2-1, x2-1, y2-1);
        painter->drawLine(x2-1, y+1, x2-1, y2-1);
        painter->setPen(group.shadow());
        painter->drawLine(x, y2, x2, y2);
        painter->drawLine(x2, y, x2, y2);
    }

    painter->restore();
}

//////////////////////////////////////////////////////////////////////////////
// drawNewstepPanel()
// ------------------
// Draw the basic Newstep panel. Like bevel with different fill behavior

void NewstepStyle::drawNewstepPanel(QPainter *painter,
                                    int x, int y, int w, int h,
                                    const QColorGroup &group,
                                    bool sunken,
                                    const QBrush *fill) const
{
    int x2 = x + w - 1;
    int y2 = y + h - 1;

    painter->save();

    if (sunken) {
        painter->setPen(group.shadow());
        painter->drawLine(x+1, y+1, x2-2, y+1);
        painter->drawLine(x+1, y+1, x+1, y2-2);
        painter->setPen(group.dark());
        painter->drawLine(x, y, x2-1, y);
        painter->drawLine(x, y, x, y2-1);
        painter->setPen(group.light());
        painter->drawLine(x, y2, x2, y2);
        painter->drawLine(x2, y, x2, y2);
    } else {
        painter->setPen(group.light());
        painter->drawLine(x, y, x2-1, y);
        painter->drawLine(x, y, x, y2-1);
        painter->setPen(group.dark());
        painter->drawLine(x+1, y2-1, x2-1, y2-1);
        painter->drawLine(x2-1, y+1, x2-1, y2-1);
        painter->setPen(group.shadow());
        painter->drawLine(x, y2, x2, y2);
        painter->drawLine(x2, y, x2, y2);
    }

    if (fill) {
        painter->fillRect(x+1, y+1, w-2, h-2, fill->color());
    }
    painter->restore();
}

//////////////////////////////////////////////////////////////////////////////
// drawNewstepTab()
// -------------
// Draw a Newstep style tab

void NewstepStyle::drawNewstepTab(QPainter *painter,
                                  int x, int y, int w, int h,
                                  const QColorGroup &group,
                                  unsigned shape,
                                  const QTabBar *bar,
                                  const QStyleOption &option,
                                  const SFlags flags) const
{
    const QTabWidget *tabwidget;
    bool selected = (flags & Style_Selected);
    bool edge; // tab is at edge of bar
    const int x2 = x + w - 1;
    const int y2 = y + h - 1;
    QPointArray parray;

    painter->save();

    // what position is the tab?
    if ((bar->count() == 1)
        || (bar->indexOf(option.tab()->identifier()) == 0)) {
        edge = true;
    } else {
        edge = false;
    }

    switch (QTabBar::Shape(shape)) {
      case QTabBar::RoundedAbove:
          if (nexttabs_) { // next style rounded tabs
              tabwidget = dynamic_cast<QTabWidget*>(bar->parent());
              if (edge && tabwidget
                  && tabwidget->cornerWidget(reverse_ ?
                                             Qt::TopRight : Qt::TopLeft)) {
                  edge = false;
              }

              // fill in the tab
              painter->setPen((selected) ? group.background() : group.mid());
              painter->setBrush((selected) ? group.background() : group.mid());
              parray.putPoints(0, 16,
                               x+3,y2-1,  x+4,y2-2,  x+5,y2-4,  x+6,y2-7,
                               x+7,y+8,   x+8,y+5,   x+9,y+3,   x+11,y+1,
                               x2-11,y+1, x2-9,y+3,  x2-8,y+5,  x2-7,y+8,
                               x2-6,y2-7, x2-5,y2-4, x2-4,y2-2, x2-3,y2-1);
              painter->drawPolygon(parray);

              // draw tab
              if (selected) painter->setPen(group.light());
              else painter->setPen(group.midlight().light(110));
              parray.putPoints(0, 13,
                               x+3,y2-2, x+4,y2-3, x+4,y2-4, x+5,y2-5,
                               x+5,y2-7, x+6,y2-8, x+6,y+8,  x+7,y+7,
                               x+7,y+5,  x+8,y+4,  x+8,y+3,  x+11,y,
                               x2-12,y);
              painter->drawPolyline(parray, 0, 13);

              painter->setPen(group.dark());
              painter->drawLine(x2-11, y, x2-9, y+2);
              painter->drawPoint(x2-2, y2-1);

              painter->setPen(group.shadow());
              parray.putPoints(0, 11,
                               x2-8,y+3,  x2-8,y+4,  x2-7,y+5,  x2-7,y+7,
                               x2-6,y+8,  x2-6,y2-8, x2-5,y2-7, x2-5,y2-5,
                               x2-4,y2-4, x2-4,y2-3, x2-2,y2-1);
              painter->drawPolyline(parray, 0, 11);

              // finish off bottom
              painter->setPen(group.light());
              if (selected) {
                  painter->drawLine(x, y2-1, x+2, y2-1);
                  painter->drawLine(x2-1, y2-1, x2, y2-1);
              } else {
                  painter->drawLine(x, y2-1, x2, y2-1);
              }
              if (!reverse_ && edge) {
                  // painter->setPen(group.light());
                  painter->drawPoint(x, y2);
              } else if (reverse_ && edge) {
                  painter->setPen(group.shadow());
                  painter->drawLine(x2, y2-1, x2, y2);
                  painter->setPen(group.dark());
                  painter->drawPoint(x2-1, y2);
              }
              break;

          } else {
              // not nexttabs_
              // we fall though to TrianglularAbove below
          }

      case QTabBar::TriangularAbove: {
          tabwidget = dynamic_cast<QTabWidget*>(bar->parent());
          if (edge && tabwidget
              && tabwidget->cornerWidget(reverse_ ?
                                         Qt::TopRight : Qt::TopLeft)) {
              edge = false;
          }

          painter->setBrush((selected) ? group.background() : group.mid());
          painter->setPen(Qt::NoPen);
          painter->fillRect(x+1, y+1, w-2, h-2, painter->brush());

          // draw tab
          painter->setPen(group.light());
          painter->drawLine(x, y, x, y2-1);
          painter->drawLine(x+1, y, x2-1, y);

          painter->setPen(group.dark());
          painter->drawLine(x2-1, y+1, x2-1, y2-1);

          painter->setPen(group.shadow());
          painter->drawLine(x2, y, x2, y2-2);

          // finish off bottom
          painter->setPen(group.light());
          if (selected) {
              painter->drawPoint(x, y2-1);
              painter->drawPoint(x2, y2-1);
          } else {
              painter->drawLine(x, y2-1, x2, y2-1);
          }
          if (!reverse_ && edge) {
              // painter->setPen(group.light());
              painter->drawPoint(x, y2);
          } else if (reverse_ && edge) {
              painter->setPen(group.shadow());
              painter->drawLine(x2, y2-1, x2, y2);
              painter->setPen(group.dark());
              painter->drawPoint(x2-1, y2);
          }
          break;
      }

      case QTabBar::RoundedBelow:
      case QTabBar::TriangularBelow: {
          tabwidget = dynamic_cast<QTabWidget*>(bar->parent());
          if (edge && tabwidget
              && tabwidget->cornerWidget(reverse_ ?
                                         Qt::BottomRight : Qt::BottomLeft)) {
              edge = false;
          }

          painter->setBrush((selected) ? group.background() : group.mid());
          painter->setPen(Qt::NoPen);
          painter->fillRect(x+1, y+1, w-2, h-2, painter->brush());

          // draw tab
          painter->setPen(group.light());
          painter->drawLine(x, y+1, x, y2);

          painter->setPen(group.dark());
          painter->drawLine(x2-1, y+1, x2-1, y2-1);
          painter->drawLine(x2-2, y2-1, x+1, y2-1);

          painter->setPen(group.shadow());
          painter->drawLine(x, y2, x2, y2);
          painter->drawLine(x2, y+1, x2, y2);

          // finish off top
          if (selected) {
              painter->setPen(group.dark());
              painter->drawLine(x2, y, x2-1, y);
          } else {
              painter->setPen(group.shadow());
              painter->drawLine(x, y+1, x2, y+1);
              painter->setPen(group.dark());
              painter->drawLine(x, y, x2, y);
          }
          if (!reverse_ && edge) {
              painter->setPen(group.light());
              painter->drawPoint(x, y);
          } else if (reverse_ && edge) {
              painter->setPen(group.shadow());
              painter->drawPoint(x2, y);
          }
          break;
      }
    }
    painter->restore();
}

//////////////////////////////////////////////////////////////////////////////
// drawPrimitive()
// ---------------
// Draw the element

void NewstepStyle::drawPrimitive(PrimitiveElement element,
                                 QPainter *painter,
                                 const QRect &rect,
                                 const QColorGroup &group,
                                 SFlags flags,
                                 const QStyleOption &option) const
{
    // common locals
    bool down    = flags & Style_Down;
    bool on      = flags & Style_On;
    bool depress = (down || on);
    bool horiz   = flags & Style_Horizontal;
    int  x, y, w, h, x2, y2;

    rect.rect(&x, &y, &w, &h);
    x2 = rect.right();
    y2 = rect.bottom();

    switch(element) {
      case PE_ButtonDefault:
      case PE_ButtonDropDown:
      case PE_ButtonTool:
          drawNewstepPanel(painter, x, y, w, h, group, depress,
                           &group.brush(QColorGroup::Button));
          break;

      case PE_ButtonBevel:
          drawNewstepBevel(painter, x,y,w,h, group, depress, false, depress);
          break;

      case PE_ButtonCommand: {
          QBrush const *brush = ((flags & Style_MouseOver) ?
                                 &group.brush(QColorGroup::Midlight) :
                                 &group.brush(QColorGroup::Button));
          if (pedantic_ && depress)
              brush = &group.brush(QColorGroup::Light);
          drawNewstepBevel(painter, x, y, w, h, group,
                           depress, false, depress, brush);
          break;
      }

      case PE_HeaderSection: {
          // covers kicker taskbar buttons and menu titles
          const QBrush *fill = 0;
          horiz = true;

          QHeader* header = dynamic_cast<QHeader*>(painter->device());
          QWidget* widget = dynamic_cast<QWidget*>(painter->device());

          if (header) {
              horiz = (header->orientation() == Horizontal);
              fill = &group.brush(QColorGroup::Mid);
          }

          if ((widget) && ((widget->inherits("QPopupMenu")) ||
                           (widget->inherits("KPopupTitle")))) {
              // kicker/kdesktop menu titles
              if (nextmenus_) {
                  drawNewstepBevel(painter, x, y, w, h, group,
                                   depress, !horiz, true, fill);
              } else {
                  painter->setPen(group.dark());
                  painter->setBrush(group.brush(QColorGroup::Mid));
                  painter->drawRect(rect);
              }
          } else {
              // taskbar buttons and other headers
              if (depress && !fill) fill = &group.brush(QColorGroup::Mid);
              drawNewstepBevel(painter, x, y, w, h, group,
                               depress, !horiz, false, fill);
          }
          break;
      }

      case PE_HeaderArrow:
          if (flags & Style_Up)
              drawPrimitive(PE_ArrowUp, painter, rect, group, Style_Sunken);
          else
              drawPrimitive(PE_ArrowDown, painter, rect, group, Style_Sunken);
          break;

      case PE_FocusRect:
          if (!pedantic_) {
              QPen old = painter->pen();
              painter->setPen(group.mid());
              painter->drawRect(rect);
              painter->setPen(old);
          }
          break;

      case PE_ScrollBarSlider:
          if (horiz) {
              drawNewstepBevel(painter, x, y+2, w, h-4, group, false,
                               false, false);
              painter->setPen(group.background());
              painter->drawLine(x, y+1, x2, y+1);
              painter->drawLine(x, y2-1, x2, y2-1);
              painter->setPen(group.shadow());
              painter->drawLine(x, y, x2, y);
              painter->drawLine(x, y2, x2, y2);
          } else {
              drawNewstepBevel(painter, x+2, y, w-4, h, group, false,
                               true, false);
              painter->setPen(group.background());
              painter->drawLine(x+1, y, x+1, y2);
              painter->drawLine(x2-1, y, x2-1, y2);
              painter->setPen(group.shadow());
              painter->drawLine(x, y, x, y2);
              painter->drawLine(x2, y, x2, y2);
          }
          // is there room to draw the dot?
          if ((w >= 10) && (h >= 10)) {
              kColorBitmaps(painter, group, x+w/2-4, y+h/2-4,
                            &dot_light, 0, 0, &dot_dark,
                            &dot_shadow, 0);
          }
          break;

      case PE_ScrollBarAddPage:
      case PE_ScrollBarSubPage:
          if (h) { // has a height, thus visible
              if (pedantic_) {
                  painter->fillRect(rect, group.background());
                  painter->fillRect(rect, QBrush(group.dark(), Dense4Pattern));
              } else {
                  painter->fillRect(rect, group.mid());
              }

              if (horiz) {
                  painter->setPen(group.background());
                  painter->drawLine(x, y+1, x2, y+1);
                  painter->drawLine(x, y2-1, x2, y2-1);
                  if (nextscroll && (element == PE_ScrollBarAddPage))
                      painter->drawLine(x2-1, y+1, x2-1, y2-1);
                  else if (!nextscroll && (element == PE_ScrollBarSubPage))
                      painter->drawLine(x+1, y+1, x+1, y2-1);
                  painter->setPen(group.shadow());
                  painter->drawLine(x, y, x2, y);
                  painter->drawLine(x, y2, x2, y2);
                  if (nextscroll && (element == PE_ScrollBarAddPage))
                      painter->drawLine(x2, y+1, x2, y2-1);
                  else if (!nextscroll && (element == PE_ScrollBarSubPage))
                      painter->drawLine(x, y+1, x, y2-1);
              } else {
                  painter->setPen(group.background());
                  painter->drawLine(x+1, y, x+1, y2);
                  painter->drawLine(x2-1, y, x2-1, y2);
                  if (element == PE_ScrollBarSubPage) {
                      painter->drawLine(x+1, y+1, x2-1, y+1);
                  }
                  painter->setPen(group.shadow());
                  painter->drawLine(x, y, x, y2);
                  painter->drawLine(x2, y, x2, y2);
                  if (element == PE_ScrollBarSubPage) {
                      painter->drawLine(x+1, y, x2-1, y);
                  }
              }
          }
          break;

      case PE_ScrollBarAddLine:
      case PE_ScrollBarSubLine: {
          const QBrush *brush = down ? &group.brush(QColorGroup::Light)
              : &group.brush(QColorGroup::Button);
          if (horiz) {
              if (nextscroll) { // next style
                  if (element == PE_ScrollBarAddLine) {
                      drawNewstepBevel(painter, x+1, y+2, w-2, h-4,
                                       group, false, false, false, brush);
                      painter->setPen(group.background());
                      painter->drawRect(x, y+1, w, h-2);
                      painter->setPen(group.shadow());
                      painter->drawLine(x, y, x2, y);
                      painter->drawLine(x, y2, x2, y2);
                      drawPrimitive(PE_ArrowRight, painter, rect, group, flags);
                  } else { // subline
                      drawNewstepBevel(painter, x+2, y+2, w-2, h-4,
                                       group, false, false, false, brush);
                      painter->setPen(group.background());
                      painter->drawLine(x+1, y+1, x2, y+1);
                      painter->drawLine(x+1, y2-1, x2, y2-1);
                      painter->drawLine(x+1, y+2, x+1, y2-2);
                      painter->setPen(group.shadow());
                      painter->drawLine(x, y, x2, y);
                      painter->drawLine(x, y2, x2, y2);
                      painter->drawLine(x, y+1, x, y2-1);
                      drawPrimitive(PE_ArrowLeft,
                                    painter, QRect(x+1, y, w, h), group, flags);
                  }
              } else { // platinum style
                  if (element == PE_ScrollBarAddLine) {
                      drawNewstepBevel(painter, x, y+2, w-2, h-4,
                                       group, false, false, false, brush);
                      painter->setPen(group.background());
                      painter->drawLine(x, y+1, x2-1, y+1);
                      painter->drawLine(x, y2-1, x2-1, y2-1);
                      painter->drawLine(x2-1, y+2, x2-1, y2-2);
                      painter->setPen(group.shadow());
                      painter->drawLine(x, y, x2, y);
                      painter->drawLine(x, y2, x2, y2);
                      painter->drawLine(x2, y+1, x2, y2-1);
                      drawPrimitive(PE_ArrowRight,
                                    painter, QRect(x-1, y, w, h), group, flags);
                  } else { // subline
                      drawNewstepBevel(painter, x+1, y+2, w-2, h-4,
                                       group, false, false, false, brush);
                      painter->setPen(group.background());
                      painter->drawRect(x, y+1, w, h-2);
                      painter->setPen(group.shadow());
                      painter->drawLine(x, y, x2, y);
                      painter->drawLine(x, y2, x2, y2);
                      drawPrimitive(PE_ArrowLeft, painter, rect, group, flags);
                  }
              }
          } else { // vertical
              if (element == PE_ScrollBarAddLine) {
                  drawNewstepBevel(painter, x+2, y, w-4, h-2, group, false,
                                   true, false, brush);
                  painter->setPen(group.background());
                  painter->drawLine(x+1, y, x+1, y2-1);
                  painter->drawLine(x2-1, y, x2-1, y2-1);
                  painter->drawLine(x+2, y2-1, x2-2, y2-1);
                  painter->setPen(group.shadow());
                  painter->drawLine(x, y, x, y2);
                  painter->drawLine(x2, y, x2, y2);
                  painter->drawLine(x+1, y2, x2-1, y2);
                  drawPrimitive(PE_ArrowDown, painter,
                                QRect(x, y-1, w, h), group, flags);
              } else { // subline
                  drawNewstepBevel(painter, x+2, y+1, w-4, h-2, group, false,
                                   true, false, brush);
                  painter->setPen(group.background());
                  painter->drawRect(x+1, y, w-2, h);
                  painter->setPen(group.shadow());
                  painter->drawLine(x, y, x, y2);
                  painter->drawLine(x2, y, x2, y2);
                  drawPrimitive(PE_ArrowUp, painter, rect, group, flags);
              }
          }
          break;
      }

      case PE_Indicator:
          drawNewstepBevel(painter, x, y, w, h, group,
                           false, true, true);
          if (on) {
              kColorBitmaps(painter, group, x+1, y+1,
                            &check_light, 0, 0, &check_dark,
                            &check_shadow, 0);
          }
          break;

      case PE_ExclusiveIndicator:
          painter->fillRect(QRect(x+1, y+1, w-2, h-2), group.background());
          if (on) {
              kColorBitmaps(painter, group, x, y, &radioon_light,
                            (lightscheme ? &radioon_mid : 0),
                            0, &radioon_dark,  &radio_shadow, 0);
          } else {
              kColorBitmaps(painter, group, x, y,
                            &radiooff_light,
                            (lightscheme ? &radiooff_mid : 0),
                            0, &radiooff_dark, &radio_shadow, 0);
          }
          break;

      case PE_ExclusiveIndicatorMask:
          painter->setBrush(Qt::color1);
          painter->setPen(Qt::color1);
          painter->drawPixmap(x, y, radiomask);
          break;

      case PE_DockWindowResizeHandle:
      case PE_Splitter:
          drawNewstepBevel(painter, x, y, w, h, group,
                           false, true, true);
          if (element == PE_Splitter) {
              kColorBitmaps(painter, group, x+w/2-3, y+h/2-3,
                            &dot_light, 0, 0, &dot_dark,
                            &dot_shadow, 0);
          }
          break;

      case PE_Panel:
      case PE_PanelLineEdit:
      case PE_PanelTabWidget:
      case PE_TabBarBase:
          drawNewstepPanel(painter, x, y, w, h, group,
                           (flags & Style_Sunken), NULL);
          break;

      case PE_PanelPopup:
          // we don't draw the popup menu if using next style menus,
          // since the menu items will cover entire surface
          if ((!nextmenus_)
              && dynamic_cast<const QPopupMenu*>(painter->device())) {
              drawNewstepPanel(painter, x, y, w, h, group, false,
                               &group.brush(QColorGroup::Background));
          }
          break;

      case PE_WindowFrame:
          drawNewstepPanel(painter, x, y, w, h, group, false, NULL);
          break;

      case PE_StatusBarSection:
          painter->setPen(group.dark());
          painter->drawLine(x, y,  x2-1, y);
          painter->drawLine(x, y+1, x, y2-1);
          painter->setPen(group.light());
          painter->drawLine(x, y2, x2, y2);
          painter->drawLine(x2, y, x2, y2-1);
          break;

      case PE_PanelMenuBar:
      case PE_PanelDockWindow:
          drawNewstepBevel(painter, x, y, w, h, group, false, (w < h), false);
          break;

      case PE_DockWindowSeparator: {
          QWidget *widget = dynamic_cast<QWidget*>(painter->device());
          bool flat = true;

          if (widget && widget->parent() &&
              widget->parent()->inherits("QToolBar")) {
              QToolBar *toolbar = dynamic_cast<QToolBar*>(widget->parent());
              if (toolbar) {
                  // toolbar not floating or in a QMainWindow
                  flat = flatToolbar(toolbar);
              }
          }

          if (flat)
              painter->fillRect(rect, group.button());
          else
              drawNewstepGradient(painter, rect, group.button(),
                                  !(horiz), 0, 0, -1, -1);
          if (horiz) {
              painter->setPen(group.dark());
              painter->drawLine(w/2, 0, w/2, h-1);
              if (!flat) painter->drawLine(0, h-1, w-1, h-1);

              painter->setPen(group.light());
              painter->drawLine(w/2+1, 0, w/2+1, h-2);
          } else {
              painter->setPen(group.dark());
              painter->drawLine(0, h/2, w-1, h/2);
              if (!flat) painter->drawLine(w-1, 0, w-1, h-1);

              painter->setPen(group.light());
              painter->drawLine(0, h/2+1, w-2, h/2+1);
          }
          break;
      }

      case PE_CheckMark:
          kColorBitmaps(painter, group, x+w/2-7, y+h/2-7,
                        &check_light, 0, 0, &check_dark,
                        &check_shadow, 0);
          break;

          // NOTE: drawPolygon doesn't quite work, so we use bitmaps
      case PE_SpinWidgetUp:
      case PE_ArrowUp:
          if (w < arrowsz) // small arrow
              kColorBitmaps(painter, group, x+w/2-3, y+h/2-3,
                            0, 0, 0, &usmarrow_dark,
                            &usmarrow_shadow, 0);
          else if (flags & Style_Sunken) // sunken arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            &usunkarr_light, 0, 0, &usunkarr_dark,
                            &usunkarr_shadow, 0);
          else // large arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            0, 0, 0, &uarrow_dark,
                            &uarrow_shadow, 0);
          break;

      case PE_SpinWidgetDown:
      case PE_ArrowDown: {
          if (w < arrowsz) // small arrow
              kColorBitmaps(painter, group, x+w/2-3, y+h/2-3,
                            0, 0, 0, &dsmarrow_dark,
                            &dsmarrow_shadow, 0);
          else if (flags & Style_Sunken) // sunken arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            &dsunkarr_light, 0, 0, &dsunkarr_dark,
                            &dsunkarr_shadow, 0);
          else // large arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            0, 0, 0, &darrow_dark,
                            &darrow_shadow, 0);
          break;
      }

      case PE_ArrowLeft:
          if (w < arrowsz) // small arrow
              kColorBitmaps(painter, group, x+w/2-3, y+h/2-3,
                            0, 0, 0, &lsmarrow_dark,
                            &lsmarrow_shadow, 0);
          else if (flags & Style_Sunken) // sunken arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            &lsunkarr_light, 0, 0, &lsunkarr_dark,
                            &lsunkarr_shadow, 0);
          else // large arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            0, 0, 0, &larrow_dark,
                            &larrow_shadow, 0);
          break;

      case PE_ArrowRight:
          if (w < arrowsz) // small arrow
              kColorBitmaps(painter, group, x+w/2-3, y+h/2-3,
                            0, 0, 0, &rsmarrow_dark,
                            &rsmarrow_shadow, 0);
          else if (flags & Style_Sunken) // sunken arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            &rsunkarr_light, 0, 0, &rsunkarr_dark,
                            &rsunkarr_shadow, 0);
          else // large arrow
              kColorBitmaps(painter, group, x+w/2-5, y+h/2-5,
                            0, 0, 0, &rarrow_dark,
                            &rarrow_shadow, 0);
          break;

      default:
          KStyle::drawPrimitive(element, painter, rect, group, flags, option);
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawKStylePrimitive()
// ---------------------
// Draw the element

void NewstepStyle::drawKStylePrimitive(KStylePrimitive element,
                                       QPainter *painter,
                                       const QWidget *widget,
                                       const QRect &rect,
                                       const QColorGroup &group,
                                       SFlags flags,
                                       const QStyleOption &option) const
{
    bool horiz = flags & Style_Horizontal;
    int x, y, w, h, x2, y2;

    rect.rect(&x, &y, &w, &h);
    x2 = rect.right();
    y2 = rect.bottom();

    switch (element) {
      case KPE_GeneralHandle:
          drawNewstepBevel(painter, x+1, y+1, w-2, h-2,
                           group, false, !horiz);
          break;

      case KPE_ToolBarHandle:
          if (horiz) { // horizontal
              drawNewstepGradient(painter, rect, group.button(), !horiz,
                                  0, 0, 0, 0);
              painter->setPen(group.shadow());
              painter->drawLine(x2-1, y, x2-1, y2);

              painter->setPen(group.dark());
              painter->drawLine(x2-2, y, x2-2, y2);
              painter->drawLine(x2-2, y2, x, y2);

              painter->setPen(group.light());
              painter->drawLine(x2, y, x2, y2);
          } else { // vertical
              drawNewstepGradient(painter, rect, group.button(), !horiz,
                                  0, 0, 0, 0);
              painter->setPen(group.shadow());
              painter->drawLine(x, y2-1, x2, y2-1);

              painter->setPen(group.dark());
              painter->drawLine(x, y2-2, x2, y2-2);
              painter->drawLine(x2, y, x2, y2-2);

              painter->setPen(group.light());
              painter->drawLine(x, y2, x2, y2);
          }
          break;

      case KPE_SliderGroove:
          if (pedantic_) {
              painter->fillRect(rect, group.background());
              painter->fillRect(rect, QBrush(group.dark(), Dense4Pattern));
          } else {
              painter->fillRect(rect, group.mid());
          }

          painter->setPen(group.shadow());
          painter->drawLine(x, y, x2, y);
          painter->drawLine(x, y+1, x, y2);
          painter->setPen(group.light());
          painter->drawLine(x+1, y2, x2, y2);
          painter->drawLine(x2, y+1, x2, y2-1);
          break;

      case KPE_SliderHandle: {
          const QSlider* slider = dynamic_cast<const QSlider*>(widget);
          if (slider) {
              int xc = x + w/2 - 1; int yc = y + h/2 - 1;
              QBrush brush = (widget==hover_)
                  ? QBrush(group.button().light(contrast))
                  : group.brush(QColorGroup::Button);

              if (slider->orientation() == Horizontal) {
                  drawNewstepBevel(painter, x+1, y+1, w-3, h-2,
                                   group, false, false, false, &brush);
                  painter->setPen(group.light());
                  painter->drawLine(x+2, y+2, x+2, y2-3);
                  painter->drawLine(xc, y+2, xc, y2-3);
                  painter->setPen(group.dark());
                  painter->drawLine(x2-1, y+1, x2-1, y2-1);
                  painter->drawLine(xc-1, y+2, xc-1, y2-3);
                  painter->setPen(group.button());
                  painter->drawPoint(xc, y2-2);
                  painter->drawPoint(xc-1, y+1);
              } else {
                  drawNewstepBevel(painter, x+1, y+1, w-2, h-3,
                                   group, false, true, false, &brush);
                  painter->setPen(group.light());
                  painter->drawLine(x+2, y+2, x2-3, y+2);
                  painter->drawLine(x+2, yc, x2-3, yc);
                  painter->setPen(group.dark());
                  painter->drawLine(x+1, y2-1, x2-1, y2-1);
                  painter->drawLine(x+2, yc-1, x2-3, yc-1);
                  painter->setPen(group.button());
                  painter->drawPoint(x2-2, yc);
                  painter->drawPoint(x+1, yc-1);
              }
          }
          break;
      }

      default:
          KStyle::drawKStylePrimitive(element, painter, widget, rect,
                                      group, flags, option);
          break;
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawControl()
// -------------
// Draw the control

void NewstepStyle::drawControl(ControlElement element,
                               QPainter *painter,
                               const QWidget *widget,
                               const QRect &rect,
                               const QColorGroup &group,
                               SFlags flags,
                               const QStyleOption &option ) const
{
    bool active, enabled;
    int x, y, w, h, dx;
    QMenuItem *mi;
    QIconSet::Mode mode;
    QIconSet::State state;
    QPixmap pixmap;

    rect.rect(&x, &y, &w, &h);

    switch (element) {
      case CE_PushButton:
          if (widget == hover_) flags |= Style_MouseOver;
          drawPrimitive(PE_ButtonCommand, painter, rect, group, flags);

          if (!pedantic_ && (flags & Style_HasFocus)) { // draw focus
              drawPrimitive(PE_FocusRect, painter,
                            QStyle::visualRect(subRect(SR_PushButtonFocusRect,
                                                       widget), widget),
                            group, flags);
          }
          break;

      case CE_PushButtonLabel: {
          const QPushButton* button = dynamic_cast<const QPushButton*>(widget);
          if (!button) {
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }

          if (button->isOn() || button->isDown()) { // shift contents
              x++;  y++;
              flags |= Style_Sunken;
          }

          if (button->iconSet() && !button->iconSet()->isNull()) { // draw icon
              if (button->isEnabled()) {
                  if (button->hasFocus()) {
                      mode = QIconSet::Active;
                  } else {
                      mode = QIconSet::Normal;
                  }
              } else {
                  mode = QIconSet::Disabled;
              }

              if (button->isToggleButton() && button->isOn()) {
                  state = QIconSet::On;
              } else {
                  state = QIconSet::Off;
              }

              pixmap = button->iconSet()->pixmap(QIconSet::Small, mode, state);
              if (button->text().isEmpty() && !button->pixmap()) {
                  painter->drawPixmap(x+w/2 - pixmap.width()/2,
                                      y+h/2 - pixmap.height()/2, pixmap);
              } else {
                  painter->drawPixmap(x+4, y+h/2 - pixmap.height()/2, pixmap);
              }
              x += pixmap.width() + 4;
              w -= pixmap.width() + 4;
          }

          if (button->isDefault()) { // default
              int xx = x + w - returnw - 3;
              int yy = y + ((h - returnh) / 2);
              kColorBitmaps(painter, group, xx, yy,
                            &return_light, 0, 0, &return_dark,
                            &return_shadow, 0);
              w -= returnw; // adjust remaining text
          }

          drawItem(painter, QRect(x, y, w, h),
                   AlignCenter | ShowPrefix,
                   button->colorGroup(),
                   button->isEnabled(),
                   button->pixmap(),
                   button->text(), -1,
                   &button->colorGroup().buttonText());
          break;
      }

      case CE_CheckBoxLabel:
      case CE_RadioButtonLabel: {
          const QButton *b = dynamic_cast<const QButton*>(widget);
          if (!b) return;

          int alignment = reverse_ ? AlignRight : AlignLeft;
          drawItem(painter, rect, alignment | AlignVCenter | ShowPrefix,
                   group, flags & Style_Enabled, b->pixmap(), b->text());

          // only draw focus if content (forms on html won't)
          if ((flags & Style_HasFocus) && ((b->text()) || b->pixmap())) {
              drawPrimitive(PE_FocusRect, painter,
                            visualRect(subRect(SR_RadioButtonFocusRect,
                                               widget), widget),
                            group, flags);
          }
          break;
      }

      case CE_DockWindowEmptyArea:  {
          const QToolBar *tb = dynamic_cast<const QToolBar*>(widget);
          if (tb) {
              // toolbar not floating or in a QMainWindow
              if (flatToolbar(tb)) {
                  painter->fillRect(rect, tb->paletteBackgroundColor());
              }
          }
          break;
      }

      case CE_MenuBarEmptyArea:
          drawNewstepGradient(painter, QRect(x, y, w, h), group.button(),
                              (w<h), 0, 0, 0, 0);
          break;

      case CE_MenuBarItem: {
          const QMenuBar *mbar = dynamic_cast<const QMenuBar*>(widget);
          if (!mbar) {
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }
          mi = option.menuItem();
          QRect prect = mbar->rect();

          if ((flags & Style_Active) && (flags & Style_HasFocus)) {
              drawNewstepBevel(painter, x, y, w, h, group,
                               (flags & Style_Down), false);
          } else {
              drawNewstepGradient(painter, rect, group.button(), false, x, y,
                                  prect.width()-2, prect.height()-2);
          }
          drawItem(painter, rect,
                   AlignCenter | AlignVCenter |
                   DontClip | ShowPrefix | SingleLine,
                   group, flags & Style_Enabled,
                   mi->pixmap(), mi->text());
          break;
      }

      case CE_PopupMenuItem: {
          const QPopupMenu *popup = dynamic_cast<const QPopupMenu*>(widget);
          if (!popup) {
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }

          mi = option.menuItem();
          if (!mi) {
              painter->fillRect(rect, group.background());
              break;
          }

          bool topmi     = (mi->id() == popup->idAt(0));
          int tabwidth   = option.tabWidth();
          int iconwidth  = option.maxIconWidth();
          active         = flags & Style_Active;
          enabled        = mi->isEnabled();
          QRect vrect;

          if (!nextmenus_ && mi->isSeparator()) { // separator
              painter->setPen(group.dark());
              painter->drawLine(x, y, x+w, y);
              painter->setPen(group.light());
              painter->drawLine(x, y+1, x+w, y+1);
              break;
          }

          // adjust for next style menus
          if (nextmenus_) {
              x++; w--; // shift one pixel right
              if (topmi) {
                  y++; h--; // shift down one pixel
              }
          }

          // draw background
          if (active && enabled) {
              if (nextmenus_) {
                  drawNewstepBevel(painter, x, y, w, h, group,
                                   false, false, true,
                                   &group.brush(QColorGroup::Base));
                  painter->setPen(group.dark());
                  painter->drawLine(x-1, y, x-1, y+h);
                  if (topmi) painter->drawLine(x-1, y-1, x+w, y-1);
              } else {
                  painter->fillRect(rect, group.base());
              }
          } else if (widget->erasePixmap() &&
                     !widget->erasePixmap()->isNull()) {
              painter->drawPixmap(x, y, *widget->erasePixmap(), x, y, w, h);
          } else {
              if (nextmenus_) {
                  drawNewstepBevel(painter, x, y, w, h, group,
                                   false, false, true,
                                   &group.brush(QColorGroup::Background));
                  painter->setPen(group.dark());
                  painter->drawLine(x-1, y, x-1, y+h);
                  if (topmi) painter->drawLine(x-1, y-1, x+w, y-1);
              } else {
                  painter->fillRect(rect, group.background());
              }
          }

          // draw icon
          if (mi->iconSet() && !mi->isChecked() && !pedantic_) {
              if (active)
                  mode = enabled ? QIconSet::Active : QIconSet::Disabled;
              else
                  mode = enabled ? QIconSet::Normal : QIconSet::Disabled;

              pixmap = mi->iconSet()->pixmap(QIconSet::Small, mode);
              QRect pmrect(0, 0, pixmap.width(), pixmap.height());
              vrect = visualRect(QRect(x, y, iconwidth, h), rect);
              pmrect.moveCenter(vrect.center());
              painter->drawPixmap(pmrect.topLeft(), pixmap);
          }

          // draw check/arrow
          if (mi->isChecked() || mi->popup()) {
              PrimitiveElement elem = PE_CheckMark;
              if (mi->popup()) elem = reverse_ ? PE_ArrowLeft : PE_ArrowRight;;

              vrect = visualRect(QRect(x + w - 20 - ITEMHMARGIN, y + h/2 - 10,
                                       20, 20), rect);
              drawPrimitive(elem, painter, vrect, group, Style_Sunken |
                            (enabled ? Style_Enabled : Style_Default));
          }

          // draw text
          int xm = ITEMFRAME + ITEMHMARGIN*2;
          if (!pedantic_) xm += iconwidth;
          int xp = reverse_ ?
              x + tabwidth + RIGHTBORDER + ITEMHMARGIN + ITEMFRAME - 1 :
              x + xm;
          int tw = w - xm - tabwidth - ARROWMARGIN - ITEMHMARGIN * 3
              - ITEMFRAME + 1;

          painter->setPen(enabled ? group.text() : group.mid());

          if (mi->custom()) { // draws own label
              painter->save();
              mi->custom()->paint(painter, group, active, enabled,
                                  xp, y+ITEMVMARGIN, tw, h-2*ITEMVMARGIN);
              painter->restore();
          } else { // draw label
              QString text = mi->text();

              if (!text.isNull()) {
                  int tflags = AlignVCenter | DontClip | SingleLine |
                      (pedantic_ ? NoAccel : ShowPrefix) |
                      (reverse_ ? AlignRight : AlignLeft);

                  int t = text.find('\t');
                  if (t >= 0) {
                      int tabx = reverse_ ?
                          x + RIGHTBORDER + ITEMHMARGIN + ITEMFRAME :
                          x + w - tabwidth - RIGHTBORDER - ITEMHMARGIN
                          - ITEMFRAME;

                      // draw right label (accerator)
                      painter->drawText(tabx, y+ITEMVMARGIN,
                                        tabwidth, h-2*ITEMVMARGIN,
                                        tflags, text.mid(t+1));
                      text = text.left(t);
                  }

                  // draw left label
                  painter->drawText(xp, y+ITEMVMARGIN,
                                    tw, h-2*ITEMVMARGIN,
                                    tflags, text, t);
              }
              else if (mi->pixmap()) { // pixmap as label
                  pixmap = *mi->pixmap();
                  if (pixmap.depth() == 1)
                      painter->setBackgroundMode(OpaqueMode);

                  dx = ((w - pixmap.width()) / 2) + ((w - pixmap.width()) % 2);
                  painter->drawPixmap(x+dx, y+ITEMFRAME, pixmap);

                  if (pixmap.depth() == 1)
                      painter->setBackgroundMode(TransparentMode);
              }
          }
          break;
      }

      case CE_TabBarTab: {
          const QTabBar* tb = dynamic_cast<const QTabBar*>(widget);
          if (tb) {
              // this guy can get complicated, we we do it elsewhere
              drawNewstepTab(painter, x, y, w, h, group, tb->shape(), tb,
                             option, (flags & Style_Selected));
          } else { // not a tabbar
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }
          break;
      }

      case CE_TabBarLabel: {
          const QTabBar* tabbar = dynamic_cast<const QTabBar*>(widget);
          const QTab* tab = dynamic_cast<const QTab*>(option.tab());
          if (!tabbar || !tab) {
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }

          bool selected = (tab->identifier() == tabbar->currentTab());
          const QColor *tcolor = selected ? &group.buttonText() : &group.text();
          drawItem(painter, rect,
                   AlignCenter | ShowPrefix, group,
                   flags & Style_Enabled, 0, tab->text(), -1, tcolor);

          if ((flags & Style_HasFocus) && !tab->text().isEmpty())
              drawPrimitive(PE_FocusRect, painter, rect, group);
          break;
      }

      case CE_ProgressBarGroove: {
          drawNewstepPanel(painter, x+1, y+1, w-2, h-2, group, true,
                           &group.brush(QColorGroup::Background));
          break;
      }

      case CE_ProgressBarContents: {
          const QProgressBar* pbar = dynamic_cast<const QProgressBar*>(widget);
          if (!pbar) {
              KStyle::drawControl(element, painter, widget, rect, group,
                                  flags, option);
              return;
          }
          subRect(SR_ProgressBarContents, widget).rect(&x, &y, &w, &h);

          if (!pbar->totalSteps()) {
              // busy indicator
              int bar = pixelMetric(PM_ProgressBarChunkWidth, widget) + 2;
              int progress = pbar->progress() % ((w-bar) * 2);
              if (progress > (w-bar)) progress = 2 * (w-bar) - progress;
              painter->fillRect(x+progress, y, bar, h, group.highlight());
          } else {
              double progress = static_cast<double>(pbar->progress()) /
                  static_cast<double>(pbar->totalSteps());
              dx = static_cast<int>(w * progress);
              if (dx < 4) break;
              if (reverse_) x += w - dx;
              painter->fillRect(x, y, dx, h, group.highlight());
          }
          break;
      }

      default:
          KStyle::drawControl(element, painter, widget, rect, group,
                              flags, option);
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawControlMask()
// -----------------
// Draw a bitmask for the element

void NewstepStyle::drawControlMask(ControlElement element,
                                   QPainter *painter,
                                   const QWidget *widget,
                                   const QRect &rect,
                                   const QStyleOption &option) const
{
    switch (element) {
      case CE_PushButton: {
          painter->fillRect(rect, Qt::color1);
          painter->setPen(Qt::color0);
          break;
      }

      default:
          KStyle::drawControlMask(element, painter, widget, rect, option);
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawComplexControl()
// --------------------
// Draw a complex control

void NewstepStyle::drawComplexControl(ComplexControl control,
                                      QPainter *painter,
                                      const QWidget *widget,
                                      const QRect &rect,
                                      const QColorGroup &group,
                                      SFlags flags,
                                      SCFlags controls,
                                      SCFlags active,
                                      const QStyleOption &option) const
{
    bool down = flags & Style_Down;
    bool on = flags & Style_On;
    bool raised = flags & Style_Raised;
    bool pressed;
    QRect subrect;
    int x, y, w, h, x2, y2, yy;
    rect.rect(&x, &y, &w, &h);

    switch (control) {
      case CC_ComboBox: {
          const QComboBox * combo = dynamic_cast<const QComboBox*>(widget);
          if (!combo) {
              KStyle::drawComplexControl(control, painter, widget, rect,
                                         group, flags, controls, active,
                                         option);
              return;
          }

          QBrush brush;
          if (active == SC_ComboBoxArrow) {
              brush = group.brush(QColorGroup::Light); // pressed
          } else if (widget==hover_) {
              brush = QBrush(group.button().light(contrast)); // highlighted
          } else {
              brush = group.brush(QColorGroup::Button); // normal
          }

          drawNewstepBevel(painter, x, y, w, h, group,
                           false, false, false, &brush);

          if (controls & SC_ComboBoxArrow) { // draw arrow box
              yy = (h-6)/2 - 1;
              if (reverse_) {
                  painter->fillRect(x+6, yy+2, 9, 6,
                                    group.brush(QColorGroup::Dark));
                  drawNewstepBevel(painter, x+4, yy, 9, 6, group,
                                   false, false, false, &brush);
              } else {
                  painter->fillRect(w-13, yy+2, 9, 6,
                                    group.brush(QColorGroup::Dark));
                  drawNewstepBevel(painter, w-15, yy, 9, 6, group,
                                   false, false, false, &brush);
              }
          }

          if (controls & SC_ComboBoxEditField) { // draw edit box
              if (combo->hasFocus()) { // non editable box
                  subrect = QStyle::visualRect(subRect(SR_ComboBoxFocusRect,
                                                       combo), widget);
                  drawPrimitive(PE_FocusRect, painter, subrect, group,
                                Style_FocusAtBorder,
                                QStyleOption(group.highlight()));
              }
          }
          painter->setPen(group.foreground()); // for subsequent text
          break;
      }

      case CC_SpinWidget: {
          const QSpinWidget *spin = dynamic_cast<const QSpinWidget*>(widget);
          if (!spin) {
              KStyle::drawComplexControl(control, painter, widget, rect,
                                         group, flags, controls, active,
                                         option);
              return;
          }

          PrimitiveElement element;

          // draw frame
          if (controls & SC_SpinWidgetFrame) {
              drawNewstepPanel(painter, x, y, w, h, group, true,
                               &group.brush(QColorGroup::Background));
              // fill in gap between frame and buttons
              painter->setPen(group.background());
              if (reverse_) {
                  painter->drawLine(x-1, y, x-1, y+h);
              } else {
                  painter->drawLine(x+w, y, x+w, y+h);
              }
          }

          // draw arrow box
          if (controls & SC_SpinWidgetButtonField) {
              subrect = querySubControlMetrics(CC_SpinWidget, widget,
                                               SC_SpinWidgetButtonField,
                                               option);
              if (reverse_) subrect.moveLeft(spin->upRect().left());
              subrect.coords(&x, &y, &x2, &y2);
              int c = spin->upRect().bottom();
              drawNewstepGradient(painter, subrect, (widget==hover_)
                                  ? group.button().light(contrast)
                                  : group.button(),
                                  false);

              painter->setPen(group.light());
              painter->drawLine(x, y, x2-1, y);
              painter->drawLine(x, y+1, x, y2-1);
              painter->drawLine(x+1, c+1, x2-1, c+1);

              painter->setPen(group.dark());
              painter->drawLine(x+1, y2-1, x2-1, y2-1);
              painter->drawLine(x2-1, y+1, x2-1, c);
              painter->drawLine(x2-1, c+2, x2-1, y2-2);
              painter->drawLine(x+1, c, x2-2, c);

              painter->setPen(group.shadow());
              painter->drawLine(x2, y, x2, y2);
              painter->drawLine(x, y2, x2-1, y2);
          }

          // draw up arrow
          if (controls & SC_SpinWidgetUp) {
              subrect = spin->upRect();
              subrect.moveBy(0, 1);
              pressed = (active == SC_SpinWidgetUp);
              if (spin->buttonSymbols() == QSpinWidget::PlusMinus)
                  element = PE_SpinWidgetPlus;
              else
                  element = PE_SpinWidgetUp;

              drawPrimitive(element, painter, subrect, group, flags
                            | Style_Enabled | Style_Sunken
                            | (pressed ? Style_On : Style_Raised));
          }

          // draw down button
          if (controls & SC_SpinWidgetDown) {
              subrect = spin->downRect();
              pressed = (active == SC_SpinWidgetDown);
              if (spin->buttonSymbols() == QSpinWidget::PlusMinus)
                  element = PE_SpinWidgetMinus;
              else
                  element = PE_SpinWidgetDown;

              drawPrimitive(element, painter, subrect, group, flags
                            | Style_Enabled | Style_Sunken
                            | (pressed ? Style_On : Style_Raised));
          }
          break;
      }

      case CC_ToolButton: {
          const QToolButton *btn = dynamic_cast<const QToolButton*>(widget);
          if (!btn) {
              KStyle::drawComplexControl(control, painter, widget, rect, group,
                                         flags, controls, active, option);
              return;
          }

          QToolBar *toolbar;
          bool horiz = true; // orientation of toolbar
          bool normal = !(down || on || raised); // normal button state

          x2 = rect.right();
          y2 = rect.bottom();

          // check for QToolBar parent
          if (btn->parent() && btn->parent()->inherits("QToolBar")) {
              toolbar = dynamic_cast<QToolBar*>(btn->parent());
              if (toolbar) {
                  horiz = (toolbar->orientation() == Qt::Horizontal);
                  if (normal) { // draw background
                      if (flatToolbar(toolbar)) {
                          // toolbar not floating or in a QMainWindow
                          painter->fillRect(rect, group.button());
                      } else {
                          drawNewstepGradient(painter, rect, group.button(),
                                              !horiz, 0, 0,
                                              toolbar->width()-2,
                                              toolbar->height()-2);
                          painter->setPen(group.dark());
                          if (horiz) {
                              painter->drawLine(x, y2, x2, y2);
                          } else {
                              painter->drawLine(x2, y, x2, y2);
                          }
                      }
                  }
              }
          }
          // QToolBarExtensionWidget parent
          else if (btn->parent() &&
                   btn->parent()->inherits(QTOOLBAREXTENSION)) {
              // button part of QToolBarExtension Widget
              QWidget *extension;
              if ((extension = dynamic_cast<QWidget*>(btn->parent()))) {
                  toolbar = dynamic_cast<QToolBar*>(extension->parent());
                  if (toolbar){
                      horiz = (toolbar->orientation() == Qt::Horizontal);
                      if (normal) { // draw background
                          drawNewstepGradient(painter, rect, group.button(),
                                              !horiz, 0, 0, toolbar->width()-2,
                                              toolbar->height()-2);
                      }
                  }
              }
          }
          // check for background pixmap
          else if (normal && btn->parentWidget() &&
                   btn->parentWidget()->backgroundPixmap() &&
                   !btn->parentWidget()->backgroundPixmap()->isNull()) {
              QPixmap pixmap = *(btn->parentWidget()->backgroundPixmap());
              painter->drawTiledPixmap(rect, pixmap, btn->pos());
          } else if (normal) { // everything else
              painter->fillRect(rect, group.button());
          }

          // now draw active buttons
          if (down || on) {
              drawNewstepPanel(painter, x, y, w, h, group, true,
                               &group.brush(QColorGroup::Button));
          } else if (raised) {
              drawNewstepBevel(painter, x, y, w, h, group, false, !horiz);
          }
          painter->setPen(group.text());
          break;
      }

      default:
          KStyle::drawComplexControl(control, painter, widget, rect, group,
                                     flags, controls, active, option);
          break;
    }
}

//////////////////////////////////////////////////////////////////////////////
// drawComplexControlMask()
// ------------------------
// Draw a bitmask for the control

void NewstepStyle::drawComplexControlMask(ComplexControl control,
                                          QPainter *painter,
                                          const QWidget *widget,
                                          const QRect &rect,
                                          const QStyleOption &option) const
{
    switch (control) {
      case CC_ComboBox:
      case CC_ToolButton: {
          painter->fillRect(rect, Qt::color1);
          painter->setPen(Qt::color0);
          break;
      }

      default:
          KStyle::drawComplexControlMask(control,painter,widget,rect,option);
    }
}

//////////////////////////////////////////////////////////////////////////////
// pixelMetric()
// -------------
// Get the pixel metric for metric

int NewstepStyle::pixelMetric(PixelMetric metric, const QWidget *widget) const
{
    switch (metric) {
      case PM_ButtonMargin:             // space betweeen frame and label
          return 4;

      case PM_ButtonDefaultIndicator:   // size of default indicator
          return 0;

      case PM_DefaultFrameWidth:
          if (nextmenus_ && dynamic_cast<const QPopupMenu*>(widget)) {
              return 0;
          }
          return 2;

      case PM_ExclusiveIndicatorWidth:  // radiobutton size
      case PM_ExclusiveIndicatorHeight:
          return radiosz;

      case PM_IndicatorWidth:           // checkbox size
      case PM_IndicatorHeight:
          return 15;

      case PM_MenuButtonIndicator:      // arrow width
          return arrowsz;

      case PM_ScrollBarExtent:          // width of scrollbars
          return KStyle::pixelMetric(metric, widget) + 4;

      case PM_ScrollBarSliderMin:       // minimum width of slider
          return 12;

      case PM_SliderLength:
          return 20;

      case PM_SliderThickness:
          return 16;

      case PM_SplitterWidth:            // width of splitter
          return 9;

      case PM_TabBarTabOverlap:         // amount of tab overlap
          if (nexttabs_) {
              if (const QTabBar *tb = dynamic_cast<const QTabBar*>(widget)) {
                  if (tb->shape() == QTabBar::RoundedAbove)
                      return 13;
              }
          }
          return 0;

      case PM_TabBarTabHSpace:          // extra tab spacing
          if (nexttabs_) {
              return 32;
          } else {
              return 24;
          }

      case PM_TabBarTabVSpace:
          if (dynamic_cast<const QTabBar*>(widget)) {
              return (nexttabs_) ? 4 : 6;
          }
          return 0;

      default:
          return KStyle::pixelMetric(metric, widget);
    }
}

//////////////////////////////////////////////////////////////////////////////
// querySubControlMetrics()
// ------------------------
// Return the metrics for complex controls (specifically, scrollbars)

QRect NewstepStyle::querySubControlMetrics(ComplexControl control,
                                           const QWidget* widget,
                                           SubControl subcontrol,
                                           const QStyleOption &option) const
{
    QRect rect;

    switch (control) {
      case CC_ScrollBar: {
          // NOTE: SC_ScrollBarFirst and SC_ScrollBarLast don't exist
          const QScrollBar *sb = dynamic_cast<const QScrollBar*>(widget);
          if (!sb) break;

          bool horizontal; // orientation of scrollbar
          int length;      // length of scrollbar minus buttons
          int extent;     // thickness of the scrollbar
          int btnlen;      // length of the add/sub btns
          int sliderstart; // starting position of the slider
          int sliderlen;   // length of the slider

          horizontal = (sb->orientation() == Qt::Horizontal);
          extent = pixelMetric(PM_ScrollBarExtent, widget);
          btnlen = extent - 2; // since add/sub buttons are truly square
          length = (horizontal ? sb->width() : sb->height()) - (btnlen*2);
          sliderstart = sb->sliderStart();

          // calculate slider length
          if (sb->maxValue() != sb->minValue()) {
              int range = sb->maxValue() - sb->minValue();
              sliderlen = (sb->pageStep() * length)
                  / (range + sb->pageStep());

              int slidermin = pixelMetric(PM_ScrollBarSliderMin, widget);
              if ((sliderlen < slidermin) || (range > INT_MAX / 2))
                  sliderlen = slidermin;
              if (sliderlen > length)
                  sliderlen = length;
          } else {
              sliderlen = length;
          }

          // subcontrols
          switch (subcontrol) {
            case SC_ScrollBarSubLine: // top/left button
                if (nextscroll) { // next scroll
                    if (horizontal)
                        rect.setRect(0, 0, btnlen, extent);
                    else
                        rect.setRect(0, length, extent, btnlen);
                } else { // platinum scroll
                    if (horizontal)
                        rect.setRect(length, 0, btnlen, extent);
                    else
                        rect.setRect(0, length, extent, btnlen);
                }
                break;

            case SC_ScrollBarAddLine: // bottom/right button
                if (nextscroll) { // next scroll
                    if (horizontal)
                        rect.setRect(btnlen, 0, btnlen, extent);
                    else
                        rect.setRect(0, length + btnlen, extent, btnlen);
                } else { // platinum scroll
                    if (horizontal)
                        rect.setRect(length+btnlen, 0, btnlen, extent);
                    else
                        rect.setRect(0, length+btnlen, extent, btnlen);
                }
                break;

            case SC_ScrollBarSubPage: // between top/left button and slider
                if (nextscroll) { // next scroll
                    if (horizontal)
                        rect.setRect(btnlen*2, 0,
                                     sliderstart - 2*btnlen, extent);
                    else
                        rect.setRect(0, 0, extent, sliderstart);
                } else { // platinum scroll
                    if (horizontal)
                        rect.setRect(0, 0, sliderstart, extent);
                    else
                        rect.setRect(0, 0, extent, sliderstart);
                }
                break;

            case SC_ScrollBarAddPage: // between bottom/right btn and thumb
                if (horizontal)
                    rect.setRect(sliderstart + sliderlen, 0,
                                 length - sliderstart - sliderlen
                                 + ((nextscroll) ? 2*btnlen : 0),  extent);
                else
                    rect.setRect(0, sliderstart + sliderlen, extent,
                                 length - sliderstart - sliderlen);
                break;

            case SC_ScrollBarGroove: // scrollbar groove
                // we give ourselves an extra two pixels at the "top"
                if (horizontal) {
                    if (nextscroll)
                        rect.setRect(btnlen*2, 0, length - 2, sb->height());
                    else
                        rect.setRect(2, 0, length - 2, sb->height());
                } else {
                    rect.setRect(0, 2, sb->width(), length - 2);
                }
                break;

            case SC_ScrollBarSlider: // scrollbar slider/thumb
                if (horizontal)
                    rect.setRect(sliderstart, 0, sliderlen, extent);
                else
                    rect.setRect(0, sliderstart, extent, sliderlen);
                break;

            default: // everything else
                rect = KStyle::querySubControlMetrics(control, widget,
                                                      subcontrol,
                                                      option);
                break;
          }
          break;
      }

      case CC_SpinWidget: {
          int frame = 2;
          QSize btn;
          btn.setHeight(widget->height() / 2);
          btn.setWidth(QMIN(btn.height() * 4 / 3, widget->width() / 5));
          btn = btn.expandedTo(QApplication::globalStrut());
          int xr = widget->width() - btn.width() - 1;

          switch (subcontrol) {
            case SC_SpinWidgetUp:
                rect.setRect(xr+1, 0,  btn.width(), btn.height());
                break;

            case SC_SpinWidgetDown:
                rect.setRect(xr+1, btn.height(),
                             btn.width(), widget->height() - btn.height());
                break;

            case SC_SpinWidgetButtonField:
                rect.setRect(xr+1, 0, btn.width(), widget->height());
                break;

            case SC_SpinWidgetEditField:
                rect.setRect(frame, frame,
                             xr - frame*2, widget->height() - frame*2);
                break;

            case SC_SpinWidgetFrame:
                rect.setRect(0, 0, xr, widget->height());
                break;

            default:
                rect = KStyle::querySubControlMetrics(control, widget,
                                                      subcontrol, option);
                break;
          }
          break;
      }

      default:
          rect = KStyle::querySubControlMetrics(control, widget,
                                                subcontrol, option);
    }

    return rect;
}

//////////////////////////////////////////////////////////////////////////////
// sizeFromContents()
// ------------------
// Returns the size of widget based on the contentsize

QSize NewstepStyle::sizeFromContents(ContentsType contents,
                                     const QWidget* widget,
                                     const QSize &contentsize,
                                     const QStyleOption &option ) const
{
    int w = contentsize.width();
    int h = contentsize.height();

    switch (contents) {
      case CT_PushButton: {
          const QPushButton* button = dynamic_cast<const QPushButton*>(widget);
          if (!button) {
              return KStyle::sizeFromContents(contents, widget,
                                              contentsize, option);
          }
          int margin = pixelMetric(PM_ButtonMargin, widget)
              + pixelMetric(PM_DefaultFrameWidth, widget) * 2;

          w += margin + 4; // add room for bold font
          h += margin;

          // standard width and heights.
          if (button->isDefault() || button->autoDefault()) {
              if (w < 70 && !button->pixmap()) w = 70;
          }
          if (h < 22) h = 22;
          return QSize(w, h);
      }

      case CT_PopupMenuItem: {
          if (!widget || option.isDefault()) return contentsize;
          const QPopupMenu *popup = dynamic_cast<const QPopupMenu*>(widget);
          if (!popup) {
              return KStyle::sizeFromContents(contents, widget,
                                              contentsize, option);
          }
          QMenuItem *item = option.menuItem();

          if (item->custom()) {
              w = item->custom()->sizeHint().width();
              h = item->custom()->sizeHint().height();
              if (!item->custom()->fullSpan())
                  h += ITEMVMARGIN*2 + ITEMFRAME*2;
          } else if (item->widget()) {    // a menu item that is a widget
              w = contentsize.width();
              h = contentsize.height();
          } else if (item->isSeparator()) {
              if (nextmenus_) w = h = 0;
              else w = h = 2;
          } else {
              if (item->pixmap()) {
                  h = QMAX(h, item->pixmap()->height() + ITEMFRAME*2);
              } else {
                  h = QMAX(h, MINICONSIZE + ITEMFRAME*2);
                  h = QMAX(h, popup->fontMetrics().height()
                           + ITEMVMARGIN*2 + ITEMFRAME*2);
              }
              if (item->iconSet())
                  h = QMAX(h, item->iconSet()->
                           pixmap(QIconSet::Small, QIconSet::Normal).height()
                           + ITEMFRAME*2);
          }

          if (!item->text().isNull() && item->text().find('\t') >= 0)
              w += 12;
          else if (item->popup())
              w += 2 * ARROWMARGIN;

          if (option.maxIconWidth() || popup->isCheckable()) {
              w += QMAX(option.maxIconWidth(),
                        QIconSet::iconSize(QIconSet::Small).width())
                  + ITEMHMARGIN*2;
          }
          w += RIGHTBORDER;

          // adjust for next style menu
          if (nextmenus_) {
              if (item->id() == popup->idAt(0))
                  h += ITEMFRAME;
              w += ITEMFRAME;
          }

          return QSize(w, h);
      }

      default:
          return KStyle::sizeFromContents(contents, widget, contentsize,
                                          option);
    }
}

//////////////////////////////////////////////////////////////////////////////
// styleHint()
// ------------------
// Returns the style hint stylehint for widget
// contributed by Martin Dietze <di@fh-wedel.de>

int NewstepStyle::styleHint(StyleHint stylehint, 
			    const QWidget* widget, 
                            const QStyleOption &opt, 
			    QStyleHintReturn* returnData) const
{
    switch (stylehint) {
      case SH_MenuBar_MouseTracking:
      case SH_PopupMenu_MouseTracking:
          if (pedantic_) return 0;
          else           return 1;
          break;

      default:
          return KStyle::styleHint(stylehint, widget, opt, returnData);
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// flatToolbar()
// -------------
// Is the toolbar "flat"

bool NewstepStyle::flatToolbar(const QToolBar *toolbar) const
{
    if (!toolbar) return true; // not on a toolbar
    if (!toolbar->isMovingEnabled()) return true; // immobile toolbars are flat
    if (!toolbar->area()) return true; // not docked
    if (toolbar->place() == QDockWindow::OutsideDock) return true; // ditto
    if (!toolbar->mainWindow()) return true; // not in a main window
    return false;
}

//////////////////////////////////////////////////////////////////////////////
// eventFilter()
// -------------
// Grab events we are interested in. Most of this routine is to handle the
// KDE exceptions to the normal styling rules.

bool NewstepStyle::eventFilter(QObject *object, QEvent *event)
{
    if (KStyle::eventFilter(object, event)) return true;
    if (!object->isWidgetType()) return false;

    bool horiz;
    int x, y, w, h;
    QToolBar *toolbar;
    QWidget *widget;

    // painting events
    if (event->type() == QEvent::Paint) {
        // make sure we do the most specific stuff first

        // KDE Toolbar Widget
        // patch by Daniel Brownlees <dbrownlees@paradise.net.nz>
        if (object->parent() && !qstrcmp(object->name(), KTOOLBARWIDGET)) {
            if (0 == (widget = dynamic_cast<QWidget*>(object))) return false;
            QWidget *parent = dynamic_cast<QWidget*>(object->parent());
            int px = widget->x(), py = widget->y();
            // find the toolbar
            while (parent && parent->parent()
                   && !dynamic_cast<QToolBar*>(parent)) {
                px += parent->x();
                py += parent->y();
                parent = dynamic_cast<QWidget*>(parent->parent());
            }
            if (!parent) return false;
            widget->rect().rect(&x, &y, &w, &h);
            QRect prect = parent->rect();

            toolbar = dynamic_cast<QToolBar*>(parent);
            horiz = (toolbar) ? (toolbar->orientation() == Qt::Horizontal)
                : (prect.height() < prect.width());

            QPainter painter(widget);
            if (flatToolbar(toolbar)) {
                painter.fillRect(widget->rect(),
                                  parent->colorGroup().background());
            } else {
                drawNewstepGradient(&painter, widget->rect(),
                                 parent->colorGroup().button(),
                                 !horiz, px, py,
                                 prect.width(), prect.height());
                if (horiz && (h==prect.height()-2)) {
                    painter.setPen(parent->colorGroup().mid());
                    painter.drawLine(x, h-1, w-1, h-1);
                } else if (!horiz && (w==prect.width()-2)) {
                    painter.setPen(parent->colorGroup().mid());
                    painter.drawLine(w-1, y, w-1, h-1);
                }
            }
        }

        // QToolBarExtensionWidget
        else if ((object->parent()) &&
                 ((toolbar = dynamic_cast<QToolBar*>(object->parent())))) {
            if (event->type() == QEvent::Paint) {
                QWidget *widget = static_cast<QWidget*>(object);
                horiz = (toolbar->orientation() == Qt::Horizontal);
                QPainter painter(widget);
                QRect wrect = widget->rect();
                wrect.rect(&x, &y, &w, &h);
                // draw the extension
                drawNewstepGradient(&painter, wrect,
                                    toolbar->colorGroup().button(),
                                    !horiz, x, y, w-1, h-1);
                if (horiz) {
                    painter.setPen(toolbar->colorGroup().dark());
                    painter.drawLine(w-1, 0, w-1, h-1);
                    painter.drawLine(x, h-1, w-2, h-1);
                    painter.drawLine(x, y, x, h-2);
                    painter.setPen(toolbar->colorGroup().light());
                    painter.drawLine(x+1, y, x+1, h-2);
                } else {
                    painter.setPen(toolbar->colorGroup().dark());
                    painter.drawLine(0, h-1, w-1, h-1);
                    painter.drawLine(w-1, y, w-1, h-2);
                    painter.drawLine(x, y, w-2, y);
                    painter.setPen(toolbar->colorGroup().light());
                    painter.drawLine(x, y+1, w-2, y+1);
                }
            }
        }

    } else if (highlights_) { // "mouseover" events
        if (::qt_cast<QPushButton*>(object) ||
            ::qt_cast<QComboBox*>(object) ||
            ::qt_cast<QSpinWidget*>(object) ||
            ::qt_cast<QSlider*>(object)) {
            if (event->type() == QEvent::Enter) {
                if ((widget = dynamic_cast<QWidget*>(object)) &&
                    widget->isEnabled()) {
                    hover_ = widget;
                    widget->repaint(false);
                }
            } else if (event->type() == QEvent::Leave) {
                if ((widget = dynamic_cast<QWidget*>(object)) &&
                    widget->isEnabled()) {
                    hover_ = 0;
                    widget->repaint(false);
                }
            }
        }
    }

    return false;
}

//////////////////////////////////////////////////////////////////////////////
// GradientSet                                                              //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GradientSet()
// -------------
// Constructor

GradientSet::GradientSet(const QColor &color, int size)
    : color_(color), size_(size)
{
    set[false] = 0; set[true] = 0;
}

//////////////////////////////////////////////////////////////////////////////
// ~GradientSet()
// --------------
// Destructor

GradientSet::~GradientSet()
{
    delete set[false]; delete set[true];
}

//////////////////////////////////////////////////////////////////////////////
// gradient()
// ----------
// Return the appropriate gradient pixmap

KPixmap* GradientSet::gradient(bool horizontal)
{
    // lazy allocate
    if (set[horizontal]) {
        return (set[horizontal]);
    }

    set[horizontal] = new KPixmap();
    if (horizontal) {
        set[true]->resize(size_, 16);
        KPixmapEffect::gradient(*set[true],
                                color_.light(contrast),
                                color_.dark(contrast),
                                KPixmapEffect::HorizontalGradient);
    } else {
        set[false]->resize(16, size_);
        KPixmapEffect::gradient(*set[false],
                                color_.light(contrast),
                                color_.dark(contrast),
                                KPixmapEffect::VerticalGradient);
    }

    return (set[horizontal]);
}

//////////////////////////////////////////////////////////////////////////////
// Plugin Stuff                                                             //
//////////////////////////////////////////////////////////////////////////////

class NewstepStylePlugin : public QStylePlugin
{
 public:
    NewstepStylePlugin();
    QStringList keys() const;
    QStyle *create(const QString &key);
};

NewstepStylePlugin::NewstepStylePlugin() : QStylePlugin() { ; }

QStringList NewstepStylePlugin::keys() const
{
    return QStringList() << "newstep" << "newstepclassic";
}

QStyle* NewstepStylePlugin::create(const QString& key)
{
    if (key.lower() == "newstep")
        return new NewstepStyle(false);
    if (key.lower() == "newstepclassic")
        return new NewstepStyle(true);
    return 0;
}

KDE_Q_EXPORT_PLUGIN(NewstepStylePlugin);

#include "newstepstyle.moc"
