"Main module - controls initial launch and GUI."

import general
import sys
import os
import threading
import gtk
import gtk.glade
import getopt


class Gui: 
    "Main GTK interface and associated management."

    def __init__(self, icon_load=""):
        "Run initial setup routine."
        self.w_tree = gtk.glade.XML(general.RESOURCES_PATH + "tango-generator.glade",
                    "wizard")
        self.icons_view = self.w_tree.get_widget("icons_view") 
        self.c_theme_order = [] #Component theme order
        self.source = {}
        self.target = {}
        self.app_cb = []
        self.checked_change = []
        self.distro_name = ""
        self.emblem_method = 0
        self.original_checked = []
        self.checked = []
        self.__connect_glade()
        self.__widget_setup()
        self.setup_list_view()
        self.setup_application_list()
        self.discover_version() #Find Gnome version
        self.__discover_distro()  #Find distribution name
        self.quitting = 0
        if icon_load != "":
            self.load_iconthemes_path(icon_load)
        install_loop = InstallCheck(self)
        install_loop.start()
        
    def __connect_glade(self):
        "Connect widgets to glade file."
        signal_pairs = {    "on_btn_back_clicked" : self.go_back, 
            "on_btn_forward_clicked" : self.go_forward, 
            "on_btn_cancel_clicked" : self.quit, 
            "on_icons_view_cursor_changed" : self.select_icon_view, 
            "on_btn_up_clicked" : self.btn_up_clicked, 
            "on_btn_down_clicked" : self.btn_down_clicked, 
            "on_btn_save_clicked" : self.btn_save_clicked, 
            "on_btn_open_clicked" : self.btn_open_clicked
                   }
        self.w_tree.signal_autoconnect(signal_pairs)
        #Destroy app with window
        self.w_tree.get_widget("wizard").connect("destroy", self.destroy)

        
    def __return_app_names(self, source_xml):
        "Read application names from argument and return as list."
        mainchild = general.return_main_child(source_xml)
        applications = []
        for application_node in mainchild.getElementsByTagName("application"):
            applications.append(application_node.getAttribute("name"))
        return applications
           
    def __calculate_table(self, total_items, desired_rows):
        """Given an ideal number of rows and a number of items, calculate
        the required table size."""
        min_rows = int(round(total_items/3.0))
        columns = 3
        rows = min_rows
        if total_items <=  desired_rows:
            columns = 1
            rows = desired_rows
        elif total_items <=  (2*desired_rows):
            columns = 2     
            rows = (2*desired_rows)
        columns *=  2
        return columns, rows
    
    def set_sensitive(self, widget_name, sensitive):
        "Shortcut to set a widget's sensitivity"
        self.w_tree.get_widget(widget_name).set_sensitive(sensitive)
        
    def __widget_setup(self):
        """Assign aliases to important widgets, and make necessary
        changes to glade file."""
         #Hide notebook tabs
        self.w_tree.get_widget("notebook").set_show_tabs(0) 
        #Load the logo.
        self.w_tree.get_widget("tg-logo").set_from_file(general.PIXMAPS_PATH + "tg-icon.svg")
        self.set_sensitive("btn_back", 0)        
        self.w_tree.get_widget("progressbar").set_pulse_step(0.01)
        
    #def __add_column(self, name, renderer
    def setup_list_view(self):
        """Create the necessary model for the icons list view and hook it
        up."""
        icons_model = gtk.ListStore(bool, str, str)
        self.icons_view.set_model(icons_model)
        
        toggle_renderer = gtk.CellRendererToggle()
        toggle_renderer.connect('toggled', 
        self.list_toggle, icons_model)
        column_enabled = gtk.TreeViewColumn("Theme Name", 
        toggle_renderer, active = 0)
        column_enabled.set_resizable(False)
        column_enabled.set_sort_column_id(0)
        self.icons_view.append_column(column_enabled)

        column_name = gtk.TreeViewColumn("Theme Name", 
        gtk.CellRendererText(), text = 1)
        column_name.set_resizable(True)        
        column_name.set_sort_column_id(1)
        self.icons_view.append_column(column_name)

        column_description = gtk.TreeViewColumn("Theme Description", 
        gtk.CellRendererText(), text = 2)
        column_description.set_resizable(True)        
        column_description.set_sort_column_id(2)
        column_description.set_visible(0)
        self.icons_view.append_column(column_description)
        
        mainchild = general.return_main_child(general.THEMES_PATH)
        for theme_node in mainchild.getElementsByTagName("theme"):
            for description_node in \
            theme_node.getElementsByTagName("description"):
                description = description_node.childNodes[0].data
                theme_name = theme_node.getAttribute("name")
                icons_model.append( [1, theme_name, description] )
            selection = self.icons_view.get_selection()
            selection.select_path(0)
            self.select_icon_view("")
        self.icons_view.set_headers_clickable(0)

    def setup_application_list(self):
        "Load data from the applications list."
        application_names = self.__return_app_names \
        (general.APPLICATIONS_PATH)
        no_columns, no_rows = self.__calculate_table(len(application_names), 5)
        app_table = self.w_tree.get_widget("app_table")
        app_table.resize(no_rows, no_columns)
        self.app_cb = []
        for i in range(len(application_names)):
            column_number = ((i)/no_rows)
            row_number = ((i) % no_rows)
            icon_column = 2*column_number
            button_column = (2*column_number)+1
            
            app_name = application_names[i]
            
            icon_location = general.PIXMAPS_PATH + "app-icons/%s.png" % \
                            app_name.replace(" ", "-").lower()
            app_icon = gtk.Image()
            app_icon.set_from_file(icon_location)
            app_table.attach(app_icon, icon_column, 
            icon_column+1, row_number, row_number+1, 
            gtk.FILL, gtk.FILL, 8)
            app_icon.show()
            
            import installation
            app_button = gtk.CheckButton(app_name)
            app_button.connect("toggled", self.box_toggled)
            installed, tangoified, source, store = installation.check(app_name)
            app_button.set_sensitive(installed)
            app_button.set_active(tangoified)
            self.original_checked.append(tangoified)
            self.app_cb.append(app_button)
            app_table.attach(app_button, button_column, 
            button_column+1, row_number, row_number+1, 
            gtk.EXPAND|gtk.FILL, gtk.EXPAND, gtk.FILL, 8)
            app_button.show()
    
    def box_toggled(self, widget):
        if self.w_tree.get_widget("notebook").get_current_page() != 1:
            return
        changed = 0
        for i in range(len(self.app_cb)):
            if self.original_checked[i] != self.app_cb[i].get_active():
                changed = 1
        if changed == 0:
            self.w_tree.get_widget("hbox_warning").set_child_visible(1)
            self.set_sensitive("btn_forward", 0)
        else:
            self.w_tree.get_widget("hbox_warning").set_child_visible(0)
            self.set_sensitive("btn_forward", 1)
          
    def __discover_distro(self):
        branding_store = gtk.ListStore(str)
        combo_branding = self.w_tree.get_widget("combo_branding")
        combo_branding.set_model(branding_store)
        mainchild = general.return_main_child(general.THEMES_PATH)
        i = 0
        for branding_node in mainchild.getElementsByTagName("distro_branding"):
            for distro_node in branding_node.getElementsByTagName("distro"):
                name = distro_node.childNodes[0].data
                combo_branding.append_text(name)
                if name == "Gnome":
                    combo_branding.set_active(i)
                i += 1
        
    
    def discover_version(self):
        """Attempt to discover the version of gnome and set the
        appropriate combobox."""
        target_store = gtk.ListStore(str)
        combo_version = self.w_tree.get_widget("combo_version")
        combo_version.set_model(target_store)
        versions = ("Gnome 2.18 and earlier", "Gnome 2.20 and later", 
        "Other")
        for version in versions:
            combo_version.append_text(version)
        import subprocess
        point_version = -1
        try:
            version_string = subprocess.Popen(["gnome-about",
            "--version"], 
            -1, stdout = subprocess.PIPE).stdout.read()
            point_version = int(version_string.split()[-1] \
            .split(".")[1])
        except OSError:
            combo_version.set_active(2)
        except:
            try:
                version_string = subprocess.Popen(["gnome-about",
                "--gnome-version"], 
                -1, stdout = subprocess.PIPE).stdout.read()
                point_version = int(version_string.splitlines()[0] \
                .split(":")[1].split(".")[1])
            except:
                combo_version.set_active(2)
        if point_version > 18:
            combo_version.set_active(1)
        else:
            combo_version.set_active(0)
        return point_version

    def destroy(self, widget = None, data = None):
        "Quit GTK interface when application ends."
        self.quitting = 1
        gtk.main_quit()

    def list_toggle(self, cell, path, model = None):
        "Make sure items are actually ticked when selected in listview."
        row = model.get_iter(path)
        model.set_value(row, 0, not cell.get_active())

    def select_icon_view(self, widget):
        "Change description based on selected row."
        try:
            selectedrow = widget.get_selection()
        except AttributeError:
            return
        model, row = selectedrow.get_selected()
        selected_index = model.get_path(row)[0]
        self.w_tree.get_widget("lbl_description") \
        .set_text(model[selected_index][2])

    def go_forward(self, widget):
        "Move the notebook forward one page."
        notebook = self.w_tree.get_widget("notebook")
        page_no = notebook.get_current_page()
        notebook.set_current_page(page_no + 1)
        if page_no ==  0:
            changed = 0
            self.set_sensitive("btn_back", 1)
            for i in range(len(self.app_cb)):
                if self.original_checked[i] != self.app_cb[i].get_active():
                    changed = 1
            if changed == 0:
                self.set_sensitive("btn_forward", 0)
            else:
                self.set_sensitive("btn_forward", 1)
                
        elif page_no ==  1:
            for i in range(len(self.app_cb)):
                if self.original_checked[i] != self.app_cb[i].get_active():
                    self.checked_change.append((self.app_cb[i].get_label(), 
                    self.app_cb[i].get_active()))
                if self.app_cb[i].get_active():
                    self.checked.append((self.app_cb[i].get_label(), 
                    1))
            self.set_sensitive("btn_back", 1)
        elif page_no ==  2:
            self.save_iconthemes_path()
        elif page_no ==  3:
            self.distro_name = general.get_active_text(self.w_tree.get_widget \
            ("combo_branding"))
            self.emblem_method = self.w_tree.get_widget("combo_version") \
            .get_active()
            self.set_sensitive("btn_back", 0)
            self.set_sensitive("btn_forward", 0)
            self.set_sensitive("btn_cancel", 0)
            self.__start_install()        
        elif page_no ==  4:
            self.w_tree.get_widget("lbl_btn_forward").set_text("Finish")
            self.set_sensitive("btn_forward", 0)
        elif page_no ==  5:
            self.quit("")

        
    def go_back(self, widget):
        """Move the notebook back one page."""
        notebook = self.w_tree.get_widget("notebook")
        page_no = notebook.get_current_page()
        notebook.set_current_page(page_no - 1)

        if page_no ==  1:
            #We can go back now.
            self.set_sensitive("btn_back", 0)
            self.set_sensitive("btn_forward", 1)

    def save_iconthemes_path(self, target = ""):
        """Save the contents of the icon THEMES_PATH listview to a file
        path specified as the argument."""
        model = self.icons_view.get_model()
        no_rows = len(model)
        for i in range(no_rows):
            if model[i][0] ==  1:
                self.c_theme_order.append(model[i][1])
        if target !=  "":
            try:
                output = "[Tango Generator Configuration]\n%s\n" \
                % "\n".join(self.c_theme_order)
                output_file = open(target, "w")
                output_file.write(output)
                output_file.close()
            except IOError:
                io_exception = sys.exc_info()
                general.show_error("%s: \n %s" % (io_exception[0].__doc__, 
                io_exception[1][1]))
            
    def load_iconthemes_path(self, target = ""):
        """Load the contents of the file specified as the argument into
        the icon THEMES_PATH listview."""
        input_file = open(target, "r")
        theme_list = input_file.readlines()
        input_file.close()
        
        if theme_list[0] !=  "[Tango Generator Configuration]\n":
            general.show_error("Invalid theme configuration.")
            return
            
        for i in range(len(theme_list)):
            theme_list[i] = theme_list[i][:-1]
    
        theme_list = theme_list[1:]
            
        model = self.icons_view.get_model()
        no_rows = len(model)
        
        rows_moved_to_top = 0
        for i_2 in range(len(theme_list)):
            for i in range(no_rows):
                row = model.get_iter(i)
                icon_theme = model.get_value(row, 1)
                icon_description = model.get_value(row, 2)
                if icon_theme == theme_list[i_2]:
                    model.remove(row)
                    model.insert(rows_moved_to_top, (1, icon_theme, 
                    icon_description))
                    rows_moved_to_top += 1
        rows_not_matched = no_rows-rows_moved_to_top
        for i in range(rows_not_matched):
            row_number = no_rows-i-1
            row = model.get_iter(row_number)
            icon_theme = model.get_value(row, 1)
            icon_description = model.get_value(row, 2)    
            model.remove(row)
            model.insert(row_number, (0, icon_theme, icon_description))
                

        
    def btn_save_clicked(self, widget):
        """Get the filename from the save dialogue and pass it on."""
        chooser = gtk.FileChooserDialog("Save As...", None, 
        gtk.FILE_CHOOSER_ACTION_SAVE, 
        (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, gtk.STOCK_SAVE,
        gtk.RESPONSE_OK))
        top_theme = self.icons_view.get_model()[0][1]
        file_name = "%s.tgc" % top_theme.replace(" ", "-")
        chooser.set_current_folder(os.path.expanduser("~"))
        chooser.set_current_name(file_name)
        tgc_filter = gtk.FileFilter()
        tgc_filter.set_name("Tango Generator Configuration")
        tgc_filter.add_pattern("*.tgc")
        chooser.add_filter(tgc_filter)
        selection = chooser.run()
        if selection == gtk.RESPONSE_OK:
            save_name = chooser.get_filename()
            if save_name[-4:] !=  ".tgc":
                save_name +=  ".tgc"
            self.save_iconthemes_path(save_name)
        chooser.destroy()
    
    def btn_open_clicked(self, widget):
        """Get the filename from the open dialogue and pass it on."""
        chooser = gtk.FileChooserDialog(title = None, 
        action = gtk.FILE_CHOOSER_ACTION_SAVE, 
        buttons = (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL, 
        gtk.STOCK_OPEN, gtk.RESPONSE_OK))
        chooser.set_current_folder(os.path.expanduser("~"))
        tgc_filter = gtk.FileFilter()
        tgc_filter.set_name("Tango Generator Configuration")
        tgc_filter.add_pattern("*.tgc")
        chooser.add_filter(tgc_filter)
        selection = chooser.run()
        if selection == gtk.RESPONSE_OK:
            self.load_iconthemes_path(chooser.get_filename())                        
        chooser.destroy()

        
                 
    def btn_up_clicked(self, widget):
        """Move the selected row up one place."""
        selectedrow = self.icons_view.get_selection()
        model, row = selectedrow.get_selected()
        selected_index = model.get_path(row)[0]
        selected_index -= 1
        if selected_index >=  0:
            row_new = model.get_iter(selected_index)
            checked = model.get_value(row_new, 0)
            icon_theme = model.get_value(row_new, 1)
            icon_description = model.get_value(row_new, 2)
            model.remove(row_new)
            model.insert_after(row_new, 
            [checked, icon_theme, icon_description])
            
    def btn_down_clicked(self, widget):
        """Move the selected row down one place."""
        selectedrow = self.icons_view.get_selection()
        model, row = selectedrow.get_selected()
        selected_index = model.get_path(row)[0]
        if (selected_index+1) < len(model):
            checked = model.get_value(row, 0)
            icon_theme = model.get_value(row, 1)
            icon_description = model.get_value(row, 2)
            model.remove(row)
            row_new = model.get_iter(selected_index)
            model.insert_after(row_new, 
            [checked, icon_theme, icon_description])
            selectedrow.select_path((selected_index+1))
            
    def quit(self, widget):
        """Pass press of cancel button onto the destory function."""
        self.destroy()

    def __start_install(self):
        """Call the processing backend, running asynchrously."""
         
        import processing
        generator_backend = processing.Backend(self.w_tree, 
        self.c_theme_order, self.checked_change, self.checked,
        self.distro_name, self.emblem_method,
        self.w_tree.get_widget("entry_name").get_text())
        generator_backend.start()
        


class InstallCheck(threading.Thread):
    """Background thread to check that all necessary
    commands are available."""
    import time
    
    def __init__(self, gui):
        """Recive and set copy of gui, before running the thread."""
        self.gui = gui
        threading.Thread.__init__(self)
    
    def run(self):
        """Call the command check and act appropriately."""
        import time
        total_working = 0
        sensitive = 1
        while 1:
            total_working = self.check_installs()
            if total_working == 2:
                gtk.gdk.threads_enter()
                self.gui.set_sensitive("btn_forward", 1)
                gtk.gdk.threads_leave()   
                return
            elif self.gui.quitting == 1:
                return
            else:
                time.sleep(2)    

        
    def check_installs(self):
        """Perform specific test for each required command to see if it's
        available."""
        total_working = 0
        
#        locate_result = os.system("locate > /dev/null")
#        if locate_result ==  32512:
#            self.gui.w_tree.get_widget("lbl_slocate") \
#            .set_text("A locate indexer is not installed.")
#            self.gui.w_tree.get_widget("img_slocate") \
#            .set_from_stock(gtk.STOCK_NO, gtk.ICON_SIZE_BUTTON)
#        elif locate_result ==  256:
#            self.gui.w_tree.get_widget("lbl_slocate") \
#            .set_text("A locate indexer was found.")
#            self.gui.w_tree.get_widget("img_slocate") \
#            .set_from_stock(gtk.STOCK_YES, gtk.ICON_SIZE_BUTTON)
#            total_working +=  1
        
#        gksu_result = os.system("gksu --help > /dev/null")
#        if gksu_result ==  32512:
#            self.gui.w_tree.get_widget("lbl_gksu").set_text \
#            ("Gksu is not installed.")
#            self.gui.w_tree.get_widget("img_gksu").set_from_stock \
#            (gtk.STOCK_NO, gtk.ICON_SIZE_BUTTON)
#        elif gksu_result ==  0:
#            self.gui.w_tree.get_widget("lbl_gksu").set_text \
#            ("Gksu was found.")
#            self.gui.w_tree.get_widget("img_gksu").set_from_stock \
#            (gtk.STOCK_YES, gtk.ICON_SIZE_BUTTON)
#            total_working +=  1

        rsvg_result = os.system("rsvg-convert --help > /dev/null")
        magick_result = os.system("convert --help > /dev/null")
	gtk.gdk.threads_enter()
        if rsvg_result ==  32512:
            self.gui.w_tree.get_widget("lbl_rsvg").set_text \
            ("Rsvg-convert from librsvg2-bin is not installed.")
            self.gui.w_tree.get_widget("img_rsvg").set_from_stock \
            (gtk.STOCK_NO, gtk.ICON_SIZE_BUTTON)
        elif rsvg_result ==  0:
            self.gui.w_tree.get_widget("lbl_rsvg").set_text \
            ("Rsvg-convert was found.")
            self.gui.w_tree.get_widget("img_rsvg").set_from_stock \
            (gtk.STOCK_YES, gtk.ICON_SIZE_BUTTON)
            total_working +=  1
        if magick_result ==  32512:
            self.gui.w_tree.get_widget("lbl_magick").set_text \
            ("Imagemagick is not installed.")
            self.gui.w_tree.get_widget("img_magick").set_from_stock \
            (gtk.STOCK_NO, gtk.ICON_SIZE_BUTTON)
        elif magick_result ==  0:
            self.gui.w_tree.get_widget("lbl_magick").set_text \
            ("Imagemagick was found.")
            self.gui.w_tree.get_widget("img_magick").set_from_stock \
            (gtk.STOCK_YES, gtk.ICON_SIZE_BUTTON)
            total_working +=  1
	gtk.gdk.threads_leave()
        return total_working



def main():
    """Parse arguments and options, display the GUI."""
    try:
        opts, args = getopt.getopt(sys.argv[1:], "h", ["help"])
    except getopt.error, msg:
        print msg
        print "For help use --help"
        sys.exit(2)
    for o, a in opts:
        if o in ("-h", "--help"):
            print __doc__
            sys.exit(0)
    for arg in args:
        print arg
    gtk.gdk.threads_init()
    app = Gui()
    gtk.gdk.threads_enter()
    gtk.main()
    gtk.gdk.threads_leave()

if __name__ == "__main__":
    main()

