"""The core processing backend - where the actual theme
creation takes place."""

import general
import gtk
import threading
import random
import sys
import urllib2
import os
import shutil
import socket
    
class Backend(threading.Thread):
    """Prossess the theme data and create the theme."""
    def __init__( self, w_tree, c_theme_order, checked, checked_change,
        distro_name, emblem_method, theme_name):
        "Initialise module variables, start thread."
        self.w_tree = w_tree
        self.c_theme_order = c_theme_order
        self.checked_change = checked_change
        self.distro_name = distro_name
        self.emblem_method = emblem_method
        self.progress = [0.0, 0.0]
        self.checked = self.checked_change
        self.theme_name = theme_name
        threading.Thread.__init__(self)
        
    def run(self):
        "Pass execution on to the relevant functions."
        pulser = PBar_Pulse(self.w_tree.get_widget("progressbar"))
        pulser.start()
        if os.path.exists(general.TMP_PATH):
            shutil.rmtree(general.TMP_PATH)
        self.set_timeout(5)
        self.__estimate_times()
        self.__compile_theme()
        self.install_themes()
        self.finish_up()
    
    def set_timeout(self,timeout):
        socket.setdefaulttimeout(timeout)
        
    def __compile_theme(self):
        """Create the meta-theme from selected component themes."""
        mainchild = general.return_main_child(general.THEMES_PATH)
        for required_theme in self.c_theme_order:
            for theme_node in mainchild.getElementsByTagName("theme"):
                if theme_node.getAttribute("name") == required_theme:
                    self.__unpack_c_theme(theme_node)
        self.add_distro_branding()
        self.update_progress_bar(1)
        self.generate_emblems()
        self.update_progress_bar(2)
        self.generate_emblem_placement()
        self.execute_final_theme(mainchild.getElementsByTagName( \
                                "final_theme")[0])
        self.execute_hard_icon()
        os.system("gtk-update-icon-cache %sfinal/" % general.TMP_PATH)
                    
    def __unpack_c_theme(self, theme_node):
        """Unpack and execute relevant functions on an individual theme."""
        theme_name = theme_node.getAttribute("name")
        for download_node in theme_node.getElementsByTagName("download"): 
            self.download_d_node(download_node)
        for move_node in theme_node.getElementsByTagName("move"):
            self.execute_move_node(move_node)
            self.update_progress_bar(0.01)
        for copy_node in theme_node.getElementsByTagName("copy"):
            self.execute_copy_node(copy_node)
            self.update_progress_bar(0.01)
        for delete_node in theme_node.getElementsByTagName("delete"):
            self.execute_delete_node(delete_node)
            self.update_progress_bar(0.01)
        for link_node in theme_node.getElementsByTagName("link"):
            self.execute_link_node(link_node)
            self.update_progress_bar(0.01)
        result_directory = general.TMP_PATH + \
            theme_node.getElementsByTagName("result")[0].childNodes[0].data \
            + "/"
        self.fill_missing_sizes(result_directory)
        self.update_progress_bar(20)
        if len(theme_node.getElementsByTagName("organise")) > 0:
            self.organise_c_theme(result_directory)
            self.update_progress_bar(20)
        self.merge(result_directory)
        self.update_progress_bar(5)
            
    def execute_final_theme(self, theme_node):
        theme_name = theme_node.getAttribute("name")
        for move_node in theme_node.getElementsByTagName("move"):
            self.execute_move_node(move_node)
        for copy_node in theme_node.getElementsByTagName("copy"):
            self.execute_copy_node(copy_node)
        for delete_node in theme_node.getElementsByTagName("delete"):
            self.execute_delete_node(delete_node)
        for link_node in theme_node.getElementsByTagName("link"):
            self.execute_link_node(link_node)
                 
    def __check_local(self, unsub_local):
        """Check if any paths from an unsubstituted string exist."""
        possible_targets = general.execute_substitutions([unsub_local])
        for possible_target in possible_targets:
            if os.path.exists(possible_target):
                return 1
        return 0
    def __download_theme(self, source, target, iterative):
        """Download the theme from the specified source to target."""
        return self.download_file(source, target, 1, iterative)
    
    def download_d_node(self, download_node):
        already_exists = self.__check_local( \
                            download_node.childNodes[0].data)
        archives = general.execute_substitutions([ \
                    download_node.childNodes[0].data])
        if already_exists == 0:
            sources = general.execute_substitutions([ \
                        download_node.getAttribute("source")])
            iterative = 0
            if len(download_node.getAttribute("iterative")) > 0:
                if download_node.getAttribute("iterative")[0] > 0:
                    iterative = 1
            if iterative == 0:
                self.update_progress_bar(5)
            for i in range(len(sources)):
                self.__download_theme(sources[i],
                                archives[i], iterative)
        for archived in archives:
            if os.path.exists(archived): self.__unpack(archived)
        if general.archive_supported(archives[0]):
            self.update_progress_bar(10)
        else:
            self.update_progress_bar(0.01)
                     
    def download_file(self, source, target, restrict_types = 1, iterative=0,
        ):
        """Download file from source to target."""
        try:
            remote_file = urllib2.urlopen(source)

            content_type = remote_file.info()["Content-Type"]
            #Don't download un 404'd not founds if we're being restrictive
            if (content_type.split(";")[0].split("/")[0] == "text") and \
                restrict_types == 1:
                #Attempt to allow poorly identified svgs
                if not ((content_type.split(";")[0].split("/")[1] == "plain" or 
                (content_type.split(";")[0].split("/")[1].find("xml") > 0)) and
                (source.find("svg") > 0)):
                    return -1
            if os.path.exists(target) == 0:
                if iterative:
                    try:
                        self.__iterative_download(remote_file, target)
                        return 1
                    except:
                        pass
                else:
                    pass
            else:
                return 0
        except urllib2.HTTPError:
            return -1
        except urllib2.URLError:
            return -2
        except KeyError:
            pass
            
        import urllib
        self.__single_download(urllib.urlopen(source), target)
        return 1
    
    def __single_download(self, remote_file, target):
        """Download a file in one action."""
        data = remote_file.read()
        general.make_parent(target)
        local_file = open(target, "w")
        local_file.write(data)
        local_file.close()
        
    def __iterative_download(self, remote_file, target):
        """Download a file iteratively, while updating the progress bar."""
        data = ""
        file_size = int(remote_file.info()["Content-Length"])
        cycle_bytes = 1024 * 5
        progress = 0
        for i in range((file_size/cycle_bytes)+1):
            data += remote_file.read(cycle_bytes)
            self.update_progress_bar(0.5)
            progress += 0.5
        local_file = open(target, "w")
        local_file.write(data)
        local_file.close()
        
    def __unpack(self, file_name):
        tmp_file = file_name.replace(general.ARCHIVE_PATH,
                general.TMP_PATH)
        general.make_parent(tmp_file)
        if general.archive_supported(file_name):
            tmp_parent = os.path.dirname(tmp_file) + "/"
            general.decompress(file_name, tmp_parent)
        else:
            shutil.copy(file_name, tmp_file)
            
    def execute_move_node(self, move_node):
        sources = general.execute_substitutions \
                    ([move_node.getAttribute("source")])
        destinations = general.execute_substitutions \
                    ([move_node.getAttribute("destination")])
        for i in range(len(sources)):
            if os.path.exists(sources[i]):
                general.remove_and_parent(destinations[i])
                shutil.move(sources[i], destinations[i])
        return 1
                
    def execute_copy_node(self, copy_node):
        sources = general.execute_substitutions \
                    ([copy_node.getAttribute("source")])
        destinations = general.execute_substitutions \
                    ([copy_node.getAttribute("destination")])
        for i in range(len(sources)):
            if os.path.exists(sources[i]):
                general.remove_and_parent(destinations[i])
                if os.path.isdir(sources[i]):
                    shutil.copytree(sources[i], destinations[i], 1)
                else:
                    shutil.copy(sources[i], destinations[i])    
        return 1
    
    def execute_delete_node(self, delete_node):
        deletables = general.execute_substitutions([ \
                    delete_node.childNodes[0].data])
        for i in range(len(deletables)):
            if os.path.exists(deletables[i]):
                if os.path.isdir(deletables[i]):
                    shutil.rmtree(deletables[i])
                else:
                    os.remove(deletables[i])
            else:
                try:
                    #Catch broken symbolic links
                    os.remove(deletables[i])
                    return 0
                except OSError:
                    pass
        return 1
        
    def execute_link_node(self, link_node):
        targets = general.execute_substitutions \
                    ([link_node.getAttribute("target")])
        names = general.execute_substitutions \
                    ([link_node.getAttribute("name")])
        for i in range(len(targets)):
            absolute_target = "%s/%s" % (os.path.dirname(names[i]),
                                        targets[i])
            if os.path.exists(absolute_target):
                general.remove_and_parent(names[i])
                os.symlink(targets[i], names[i])
        return 1
    
    def organise_c_theme(self, directory):
        preferred_directories = []
        for line in open(general.RESOURCES_PATH + "file_names.dat", "r").readlines():
            preferred_directories.append(line[:-1])
        sizes = ("8x8", "16x16", "22x22", "24x24", "32x32", "48x48", "scalable")
        for size in sizes:
            component_directory = directory + size + "/"
            for root, dirs, files in os.walk(component_directory):
                for icon_file in files:
                    file_location = root + "/" + icon_file
                    if os.path.islink(file_location) == 0:
                        for preferred_location in preferred_directories:
                            preferred_name = preferred_location.split("/")[-1]
                            actual_name = file_location.split("/")[-1][:-4]
                            if actual_name == preferred_name:
                                intended_file = component_directory \
                                + preferred_location + file_location[-4:]
                                if os.path.exists(intended_file) == 0 and intended_file != file_location:
                                    general.make_parent(intended_file)
                                    shutil.move(file_location, intended_file)
                                elif intended_file != file_location:
                                    os.remove(file_location)
                    else:
                        link_target = os.readlink(file_location)
                        for preferred_location in preferred_directories:
                            preferred_link = preferred_location.split("/")[-1]
                            linked_name = link_target[:-4]
                            if linked_name == preferred_link:
                                preferred_directory = \
                                    "/".join(preferred_location.split("/")[:-1]) + "/"
                                preferred_base = file_location.split("/")[-1]
                                preferred_name = directory + size + "/" + preferred_directory + preferred_base
                                if os.path.exists(preferred_name) == 0 and preferred_name != file_location:
                                    general.make_parent(preferred_name)
                                    os.symlink(link_target, 
                                        preferred_name)
                                    os.remove(file_location)
        
        
    def fill_missing_sizes(self, directory):
        conversions = (("22x22", "24x24"), ("24x24", "22x22"), ("scalable",
                        "48x48"))
        for argument_pair in conversions:
            self.create_sizes(directory, argument_pair[0], argument_pair[1])
    
    def create_sizes(self, directory, orig_dir, result_dir):
        """Given a base directory and two size directories within it,
        resize all originals to resulting sizes."""
        import subprocess
        #Cannot convert to scalable
        if result_dir == "scalable":
            return 0
        convert_command = self.calc_resize_command(orig_dir, result_dir)
        orig_full = directory + orig_dir + "/"
        result_full = directory + result_dir + "/"
        if os.path.exists(result_full) == 0:
            os.mkdir(result_full)
        if os.path.exists(orig_full):
            for root, dirs, files in os.walk(orig_full):
                for directory in dirs:
                    rel_directory = (root + "/" + directory)[len(orig_full):]
                    if (result_full + rel_directory).find("/animations/") == -1:
                        if os.path.exists(result_full + rel_directory) == 0:
                            general.make_parent(result_full + rel_directory)
                            os.mkdir(result_full + rel_directory)
                    else:
                        print result_full + rel_directory
                for file_name in files:
                    rel_file = (root + "/" + file_name)[len(orig_full):]
                    if (result_full + rel_file).find("/animations/") == -1:
                        if os.path.exists(result_full + rel_file) == 0:
                            general.make_parent(result_full + rel_file)
                            if os.path.islink(orig_full + rel_file):
                                new_link = os.readlink(orig_full + rel_file)
                                if new_link[-3:] == "svg":
                                    new_link = new_link[:-3] + "png"
                                link_name = result_full + rel_file
                                if link_name[-3:] == "svg":
                                    link_name = link_name[:-3] + "png"
                                general.delete_if_existing(link_name)
                                os.symlink(new_link, link_name)
                            elif file_name[-4:] == "icon":
                                icon_info = open(orig_full + rel_file).read()
                                orig_size = 0
                                if orig_dir == "scalable":
                                    orig_size = 1000
                                else:
                                    orig_size = int(orig_dir.split("x")[0])
                                result_size = int(result_dir.split("x")[0])
                                new_info = self.extrapolate_icon_info(icon_info, 
                                            orig_size, result_size)
                                open(result_full + rel_file, "w").write( \
                                                                new_info)
                            elif file_name[-3:] == "png" or \
                                                    file_name[-3:] == "svg":
                                general.remove_and_parent(result_full + \
                                                                    rel_file)
                                final_result = result_full + \
                                        rel_file[:-3] + "png"
                                specific_convert_command = \
                                    convert_command.replace("%o",
                                    orig_full + rel_file).replace("%fr",
                                    final_result)
                                os.system(specific_convert_command)
                    
    def calc_resize_command(self, orig_dir, result_dir):
        if orig_dir == "scalable":
            result_size = result_dir.split("x")[0]
            convert_command = "rsvg-convert -w %s -h %s %o > %fr".replace("%s",
                                result_size)
        else:
            result_size = int(result_dir.split("x")[0])
            orig_size = int(orig_dir.split("x")[0])
            if result_size > orig_size:
                border_size = (result_size - orig_size)/2.0
                convert_command = \
                    "convert -bordercolor Transparent -border %sx%s %o %fr". \
                    replace("%s", str(border_size))
            else:
                crop_size = int((orig_size - result_size)/2.0)
                convert_command = \
                    "convert -crop %sx%s+%c+%c %o %fr".replace("%s",
                    str(result_size)).replace("%c", str(crop_size))
        return convert_command
                        
                        
    def extrapolate_icon_info(self, data, orig_size, result_size):
        string_start = data.find("AttachPoints=")
        if string_start < 0:
            string_start = data.find("EmbeddedTextRectangle=")
        if string_start < 0:
            return data
        scaling_start = data.find("=", string_start) + 1
        scaling_end = data.find("\n", scaling_start)
        scaling_text = data[scaling_start:scaling_end]
        #Tango icon theme uses x instead of , as a seperator
        scaling_text = scaling_text.replace("x", ",")
        new_points = ""
        for coordinate in scaling_text.split(","):
            if coordinate != "" and coordinate != "\n":
                if coordinate.find("|") > -1:
                    numbers = coordinate.split("|")
                    for number in numbers:
                        new_number = (float(number) / orig_size) * result_size
                        new_points +=  str(int(new_number)) + "|"
                    new_points = new_points[:-1] + ","
                else:
                    new_number = (float(coordinate) / orig_size) * result_size
                    new_points +=  str(int(new_number)) + ","
        new_points = new_points[:-1]
        adjusted_str = data[:scaling_start] + new_points + data[scaling_end:]
        return adjusted_str
          
    def merge(self, theme_dir):
        final_dir = general.FINAL_PATH
        if os.path.exists(final_dir) == 0:   os.mkdir(final_dir)
        for root, dirs, files in os.walk(theme_dir):
            for icon_file in files:
                try:
                    file_location = root + "/" + icon_file
                    target_location = final_dir + file_location[len(theme_dir):]
                    if os.path.exists(target_location) == 0:
                        general.make_parent(target_location)
                        if os.path.islink(file_location):
                            os.symlink(os.readlink(file_location), target_location)
                        elif os.path.isfile(file_location):
                            shutil.copy(file_location, target_location)
                except OSError:
                    sys.exc_info()[1][1]
        shutil.rmtree(theme_dir)
                    
    def update_progress_bar(self, increase):
        """Update the progress percentage of the progress bar."""
        self.progress[0] += increase
        gtk.gdk.threads_enter()
        self.w_tree.get_widget("progressbar").set_fraction( \
                                self.progress[0]/self.progress[1])
        gtk.gdk.threads_leave()
        
    def __set_progress_label(self, new_label):
        "Update the progress label to the given argument."
        gtk.gdk.threads_enter()
        self.w_tree.get_widget("lbl_progress").set_label(new_label)
        gtk.gdk.threads_leave()
    
    def __estimate_times(self):
        mainchild = general.return_main_child(general.THEMES_PATH)
        for required_theme in self.c_theme_order:
            for theme_node in mainchild.getElementsByTagName("theme"):
                if theme_node.getAttribute("name") == required_theme:
                    self.__estimate_c_theme(theme_node)
        for distro_node in mainchild.getElementsByTagName("distro_branding"):
            download_node = distro_node.getElementsByTagName("download")[0]
            self.estimate_d_node(download_node)
        self.progress[1] += 1
        self.progress[1] += 2
                    
    def __estimate_c_theme(self, theme_node):
        """Estimate the time (arbitrary units) to complete the
        generation."""
        for download_node in theme_node.getElementsByTagName("download"): 
            self.estimate_d_node(download_node)
        for move_node in theme_node.getElementsByTagName("move"):
            self.progress[1] += 0.01
        for copy_node in theme_node.getElementsByTagName("copy"):
            self.progress[1] += 0.01
        for delete_node in theme_node.getElementsByTagName("delete"):
            self.progress[1] += 0.01
        for link_node in theme_node.getElementsByTagName("link"):
            self.progress[1] += 0.01
        self.progress[1] += 20        
        if len(theme_node.getElementsByTagName("organise")) > 0:
            self.progress[1] += 20
        self.progress[1] += 5
                
    def estimate_d_node(self, download_node):
        already_exists = self.__check_local( \
                            download_node.childNodes[0].data)
        if already_exists == 0:
            sources = general.execute_substitutions([ \
                        download_node.getAttribute("source")])
            iterative = 0
            if len(download_node.getAttribute("iterative")) > 0:
                if download_node.getAttribute("iterative")[0] > 0:
                    iterative = 1
            if iterative == 1:
                for source in sources:
                    try:
                        remote_file = urllib2.urlopen(source)
                        file_size = \
                        int(remote_file.info()["Content-Length"])
                        cycle_bytes = 5 * 1024
                        self.progress[1] += ((file_size/cycle_bytes)+1)*0.5
                    except:
                        self.progress[1] += 5
            else:
                self.progress[1] += 5
        archives = general.execute_substitutions([ \
                download_node.childNodes[0].data])
        if general.archive_supported(archives[0]):
            self.progress[1] += 10
        else:
            self.progress[1] += 0.01 
        
    def add_distro_branding(self):
        mainchild = general.return_main_child(general.THEMES_PATH)
        for distro_node in mainchild.getElementsByTagName("distro_branding"):
            download_node = distro_node.getElementsByTagName("download")[0]
            self.download_d_node(download_node)
            for branding_node in distro_node.getElementsByTagName("distro"):
                if branding_node.childNodes[0].data == self.distro_name:
                    path_base = branding_node.getAttribute("path_base")
                    copy_suffix = ("-16.png", "-22.png", "-24.png", "-32.png",
                                ".png", ".svg")
                    target_dir = ("16x16", "22x22", "24x24", "32x32",
                                "48x48", "scalable")
                    target_ext = (".png", ".png", ".png", ".png",
                                ".png", ".svg")                   
                    for i in range(len(copy_suffix)):
                        source = general.TMP_PATH + "menu-icons/" + \
                            path_base + copy_suffix[i]
                        target = general.FINAL_PATH + target_dir[i] + \
                            "/places/start-here" + target_ext[i]
                        if os.path.exists(target): os.remove(target)
                        shutil.copy(source, target)
        shutil.rmtree(general.TMP_PATH + "menu-icons/")
        
    def generate_emblems(self):
        if self.emblem_method:
            self.generate_new_emblems()
        else:
            self.generate_old_emblems()
    
    def generate_new_emblems(self):
        mainchild = general.return_main_child(general.THEMES_PATH)
        for emblem_node in mainchild.getElementsByTagName( \
                                    "emblem_mappings_new"):
            for link_node in emblem_node.getElementsByTagName("mapping"):
                sources = general.execute_substitutions([ \
                                link_node.getAttribute("source")])
                names = general.execute_substitutions([ \
                        "%f%s/emblems/" + link_node.getAttribute("name")])
                old_emblems = \
                general.execute_substitutions(["%f%s/emblems_old"])
                for i in range(len(sources)):
                    source = sources[i]
                    name = names[i]
                    if os.path.exists(source) == 1 and \
                    os.path.exists(name) == 0:
                        general.make_parent(name)
                        shutil.copy(source, name)
                    if os.path.exists(old_emblems[i]):
                        shutil.rmtree(old_emblems[i])
        emblem_dirs = general.execute_substitutions(["%f%s/emblems/"])
        icon_files = []
        for emblem_dir in emblem_dirs:
            for root, dirs, files in os.walk(emblem_dir):
                for icon_file in files:
                    if icon_file[-5:] == ".icon":
                        icon_files.append(emblem_dir + icon_file)
        for icon_file in icon_files:
            if os.path.exists(icon_file):
                for emblem_dir in emblem_dirs:
                    if os.path.exists(emblem_dir + \
                    os.path.basename(icon_file)) == 0:
                        shutil.copy(icon_file,
                        emblem_dir + os.path.basename(icon_file))
        for emblem_dir in emblem_dirs:
            for root, dirs, files in os.walk(emblem_dir):
                for icon in files:
                    if icon[-4:] == ".png" or icon[-4:] == ".svg":
                        icon_file = emblem_dir + icon[:-4] + ".icon"
                        if os.path.exists(icon_file) == 0:
                            english_name = " ".join(icon[:-4].split("-"))
                            english_name = " ".join(english_name.split("_"))
                            capitalised_name = ""
                            for word in english_name.split(" "):
                                if word != "emblem":
                                    capitalised_name += word[:1].upper() + \
                                        word[1:] + " "
                            capitalised_name = capitalised_name[:-1]
                            open(icon_file, "w").write( \
                                "\n[Icon Data]\n\nDisplayName=%s\n" % \
                                capitalised_name)
    
    def generate_old_emblems(self):
        mainchild = general.return_main_child(general.THEMES_PATH)
        for emblem_node in \
        mainchild.getElementsByTagName("emblem_mappings_old"):            
            for link_node in emblem_node.getElementsByTagName("mapping") :
                emblem_source = link_node.getAttribute("source")
                emblem_name = link_node.getAttribute("name")
                emblem_size = int(link_node.getAttribute("source_size"))
                if os.path.exists(general.FINAL_PATH + emblem_name):
                    os.remove(general.FINAL_PATH + emblem_name)
                if os.path.isfile(general.FINAL_PATH + emblem_source):
                        border_width=int((36.0-emblem_size )/2)
                        border_string="%sx%s" % (border_width, border_width)
                        result=os.system( \
                        "convert -bordercolor Transparent -border %s %s %s" \
                        % (border_string, general.FINAL_PATH + emblem_source,
                        general.FINAL_PATH + emblem_name))
        sizes=("8x8","16x16","22x22","24x24","32x32","48x48","scalable")
        for size in general.execute_substitutions(["%s"]):
            if os.path.exists(general.FINAL_PATH + size + "/emblems_old"):
                shutil.rmtree(general.FINAL_PATH + size + "/emblems_old")    
            #Don't delete nautilus, evolution or list view emblems
            if (size != "48x48") & (size != "8x8") & (size != "16x16"):
                if os.path.exists(general.FINAL_PATH + size + "/emblems"):
                    shutil.rmtree(general.FINAL_PATH + size + "/emblems")
    
    def generate_emblem_placement(self):
        sizes=("8x8","16x16","22x22","24x24","32x32","48x48","scalable")
        extensions=("png","png","png","png","png","png","svg")
        for i in range(len(sizes)):
            placement_source=general.FINAL_PATH + "%s/places/folder.icon" % \
                                                    sizes[i]
            if os.path.exists(placement_source):
                placement_contents = open(placement_source).read()
                placement_coordinates = self.cont_to_coord(placement_contents)
                if os.path.exists(general.FINAL_PATH + "%s/places/folder.icon" % sizes[i]):
                    for root, dirs, files in os.walk(general.FINAL_PATH + sizes[i]):
                        for file_name in files:
                            self.add_emblem_placements(root, file_name, 
                                placement_source, placement_coordinates,
                                placement_contents)

    def add_emblem_placements(self, root, file_name, placement_source,
                                placement_coordinates, placement_contents):
        full_file_name=root+"/"+file_name
        if (full_file_name != placement_source) and (full_file_name[-4:] != "icon"):
            relevant_status = (full_file_name.find("/status/") > -1) and \
            (((full_file_name.find("folder") > -1) or \
            (full_file_name.find("trash") > -1) or \
            (full_file_name.find("directory") > -1) or \
            (full_file_name.find("gnome-fs") > -1) or \
            full_file_name.find("stock_open") > -1) or \
            (full_file_name.find("image") > -1))
            relevant_place = (full_file_name.find("/places/") > -1) and \
            (full_file_name.find("main-menu") == -1) and \
            (full_file_name.find("novell-button") == -1) and \
            (full_file_name.find("start-here") == -1) and \
            (full_file_name.find("distributor-logo") == -1)
            relevant_mimetype = (full_file_name.find("/mimetypes/") > -1)
            if relevant_place or relevant_status or relevant_mimetype:
                icon_file = full_file_name.replace(".svg",
                            ".icon").replace(".png", ".icon")
                new_file_contents = ""
                if os.path.exists(icon_file):
                    file_contents = open(icon_file).read()
                    current_coordinates = \
                                self.cont_to_coord(file_contents)
                    if current_coordinates == "":
                        new_file_contents = file_contents + \
                                "\nAttachPoints=%s\n" % placement_coordinates
                    else:
                        new_file_contents = \
                        file_contents.replace(current_coordinates,
                        placement_coordinates)
                else:
                    new_file_contents=placement_contents
                file_object=open(icon_file, "w")
                file_object.write(new_file_contents)
                
    def cont_to_coord(self, file_contents):
        scaling_start=file_contents.find("AttachPoints=") + 13
        scaling_end=file_contents.find("\n", scaling_start) + 1
        scaling_coordinates=file_contents[scaling_start:scaling_end]
        if file_contents.find("AttachPoints=")<0:
            scaling_coordinates=""
        return scaling_coordinates                             
        
    def execute_hard_icon(self):
        mainchild = general.return_main_child(general.THEMES_PATH)
        replacements={}
        for replacement_node in mainchild.getElementsByTagName( \
                                                    "hard_replacement"):
            for hard_node in replacement_node.getElementsByTagName("hard") :
                hard_new = hard_node.getAttribute("new")
                replacements[hard_node.getAttribute("old")] = hard_new
        desktop_dirs = []
        if os.path.exists("/usr/share/applications/"):
        	desktop_dirs.append("/usr/share/applications/")
        if os.path.exists("/usr/share/control-center-2.0/capplets/"):
        	desktop_dirs.append("/usr/share/control-center-2.0/capplets/")
        for desktop_dir in desktop_dirs:
            for icon_name in os.listdir(desktop_dir):
                if os.path.isfile(desktop_dir + icon_name):
                    file=open(desktop_dir + icon_name, "r")
                    file_contents=file.read()
                    new_contents=file_contents
                    icons_replaced=[]
                    for i in range(len(replacements)):
                        old=replacements.keys()[i]
                        new=replacements[replacements.keys()[i]]
                        try:
                            if new_contents != new_contents.replace(old,new):
                                icons_replaced.append(old)
                                new_contents = new_contents.replace(old,new)
                        except:
                            pass
                    if file_contents != new_contents:
                        self.execute_hard_replacement(icons_replaced, icon_name,
                            replacements, new_contents)
                        
    def execute_hard_replacement(self, icons_replaced, icon_name, replacements,
                                    new_contents):
        for replaced_icon in icons_replaced:
            if os.path.exists(os.path.expanduser( \
            "~/.local/share/applications/" + icon_name)) == 0:
                if os.path.isfile(replaced_icon):
                    new_location=os.path.expanduser( \
                    "~/.icons/hicolor/48x48/apps/") + \
                    replacements[replaced_icon] + "." + \
                    replaced_icon.split(".")[-1]
                    general.make_parent(new_location)
                    open(new_location, "w").write(open(replaced_icon, "r").read())
                    if os.path.exists(os.path.expanduser( \
                                        "~/.icons/hicolor/index.theme")) == 0:
                        shutil.copy(general.RESOURCES_PATH + "hicolor.theme", 
                        os.path.expanduser("~/.icons/hicolor/index.theme"))
                general.make_parent(os.path.expanduser( \
                            "~/.local/share/applications/")+icon_name)
                f = open(os.path.expanduser( \
                            "~/.local/share/applications/"+icon_name), "w")
                f.write(new_contents)
                
    def install_themes(self):
        import installation
        for name, checked in self.checked_change:
            installation.install(name, self.theme_name)
                                    
    def finish_up(self):
        """Move the notebook on to the final page."""
        notebook = self.w_tree.get_widget("notebook")
        gtk.gdk.threads_enter()
        page_no = notebook.get_current_page()
        notebook.set_current_page(page_no + 1)
        self.w_tree.get_widget("lbl_btn_forward").set_text("Finish")
        self.w_tree.get_widget("btn_forward").set_sensitive(1)
        gtk.gdk.threads_leave()       

class PBar_Pulse(threading.Thread):
    def __init__(self, pbar):
        """Recive and set copy of gui, before running the thread."""
        self.pbar = pbar
        threading.Thread.__init__(self)
    def run(self):
        import time
        while 1:
            if self.pbar.get_fraction() > 0:
                break
            else:
            
                gtk.gdk.threads_enter()
                self.pbar.pulse()
                gtk.gdk.threads_leave()
                time.sleep(0.05)    

        
