/****************************************************************************
**
** Copyright (C) 2006-2008 fullmetalcoder <fullmetalcoder@hotmail.fr>
**
** This file is part of the Edyuk project <http://edyuk.org>
** 
** This file may be used under the terms of the GNU General Public License
** version 3 as published by the Free Software Foundation and appearing in the
** file GPL.txt included in the packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#ifndef _QEDITOR_H_
#define _QEDITOR_H_

#include "qce-config.h"

#include <QPointer>
#include <QScrollBar>
#include <QBasicTimer>
#include <QAbstractScrollArea>

#include "qdocument.h"
#include "qdocumentcursor.h"

#ifdef _QMDI_
	#include "qmdiclient.h"
#endif

class QMenu;
class QAction;
class QMimeData;
class QActionGroup;
class QFileSystemWatcher;

class QMarker;
class QMatcher;
class QIndenter;
class QHighlighter;
class QCodeCompletionEngine;
class QLanguageDefinition;
class QDocumentLineHandle;

class QCE_EXPORT QEditor : public QAbstractScrollArea
#ifdef _QMDI_
, public qmdiClient
#endif
{
	friend class QEditConfig;
	friend class QEditorFactory;
	
	Q_OBJECT
	
	public:
		enum EditFlag
		{
			None			= 0,
			
			Overwrite		= 1,
			CursorOn		= 2,
			ReadOnly		= 4,
			MousePressed	= 8,
			
			Persistent		= 16,
			Multiline		= 32,
			
			MaybeDrag		= 64,
			Selection		= 128,
			HideCursor		= 256,
			IgnoreScrollBarAdjustement	= 512,
			
			LineWrap		= 1024,
			CtrlNavigation	= 2048,
			AutoCloseChars	= 4096,
			
			FoldedCursor	= 8192,
			
			ReplaceTabs		= 16384,
			RemoveTrailing	= 32768,
			PreserveTrailingIndent = 65536
		};
		
		class InputBinding
		{
			public:
				virtual ~InputBinding() {}
				
				virtual QString id() const = 0;
				virtual QString name() const = 0;
				
				virtual bool keyPressEvent(QKeyEvent *event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
				
				virtual bool inputMethodEvent(QInputMethodEvent* event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
				
				virtual bool mouseMoveEvent(QMouseEvent *event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
				
				virtual bool mousePressEvent(QMouseEvent *event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
				
				virtual bool mouseReleaseEvent(QMouseEvent *event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
				
				virtual bool mouseDoubleClickEvent(QMouseEvent *event, QEditor *editor)
				{
					Q_UNUSED(event)
					Q_UNUSED(editor)
					return false;
				}
		};
		
		//typedef QFlags<EditFlag> State;
		Q_DECLARE_FLAGS(State, EditFlag)
		
		QEditor(QWidget *p = 0);
		QEditor(const QString& s, QWidget *p = 0);
		virtual ~QEditor();
		
		bool flag(EditFlag) const;
		
		QString text() const;
		
		QDocument* document() const;
		InputBinding* inputBinding() const;
		
		QDocumentCursor cursor() const;
		
		QMarker* marker() const;
		QMatcher* matcher() const;
		QIndenter* indenter() const;
		QHighlighter* highlighter() const;
		
		QLanguageDefinition* languageDefinition() const;
		
		QCodeCompletionEngine* completionEngine() const;
		
		virtual QRect cursorRect() const;
		virtual QRect selectionRect() const;
		virtual QRect lineRect(int line) const;
		virtual QRect lineRect(const QDocumentLine& l) const;
		virtual QRect cursorRect(const QDocumentCursor& c) const;
		
		#ifndef _QMDI_
		QString name() const;
		QString fileName() const;
		
		bool isContentModified() const;
		#endif
		
		bool isInConflict() const;
		
		inline int horizontalOffset() const
		{ return horizontalScrollBar()->value(); }
		inline int verticalOffset() const
		{ return verticalScrollBar()->value(); }
		
		inline QPoint mapToContents(const QPoint &point) const
		{
			return QPoint(	point.x() + horizontalOffset(),
							point.y() + verticalOffset() );
		}
		
		static QStringList inputBindings();
		static QString defaultInputBinding();
		static void addInputBinding(InputBinding *b);
		static void removeInputBinding(InputBinding *b);
		static void setDefaultInputBinding(InputBinding *b);
		static void setDefaultInputBinding(const QString& b);
		
	public slots:
		void undo();
		void redo();
		
		void cut();
		void copy();
		void paste();
		
		void selectAll();
		
		void find();
		void findNext();
		void replace();
		
		void gotoLine();
		
		void indentSelection();
		void unindentSelection();
		
		void commentSelection();
		void uncommentSelection();
		
		virtual void save();
		virtual void print();
		
		virtual void retranslate();
		
		virtual void write(const QString& s);
		
		void addAction(QAction *a, const QString& menu, const QString& toolbar = QString());
		void removeAction(QAction *a, const QString& menu, const QString& toolbar = QString());
		
		void load(const QString& file);
		void setText(const QString& s);
		
		void setDocument(QDocument *d);
		void setInputBinding(InputBinding *b);
		
		void setCursor(const QDocumentCursor& c);
		
		void setMarker(QMarker *m);
		void setMatcher(QMatcher *m);
		void setIndenter(QIndenter *m);
		void setHighlighter(QHighlighter *m);
		
		void setLanguageDefinition(QLanguageDefinition *d);
		
		void setCompletionEngine(QCodeCompletionEngine *e);
		
		void zoom(int n);
		
		inline void setPanelMargins(int l, int t, int r,int b)
		{ setViewportMargins(l, t, r, b); }
		
		void setTitle(const QString& title);
		
		void highlight();
		
		virtual void setFileName(const QString& f);
		
	signals:
		void loaded(QEditor *e, const QString& s);
		void saved(QEditor *e, const QString& s);
		
		void contentModified(bool y);
		void titleChanged(const QString& title);
		
		void cursorPositionChanged();
		
		void copyAvailable(bool y);
		
		void undoAvailable(bool y);
		void redoAvailable(bool y);
		
		void markChanged(const QString& f, QDocumentLineHandle *l, int mark, bool on);
		
	protected slots:
		void checkClipboard();
		void reconnectWatcher();
		void fileChanged(const QString& f);
		
		void setContentClean(bool y);
		
		virtual void setContentModified(bool y);
		
	protected:
		virtual void paintEvent(QPaintEvent *e);
		virtual void timerEvent(QTimerEvent *e);
		
		virtual void keyPressEvent(QKeyEvent *e);
		
		virtual void inputMethodEvent(QInputMethodEvent* e);
		
		virtual void mouseMoveEvent(QMouseEvent *e);
		virtual void mousePressEvent(QMouseEvent *e);
		virtual void mouseReleaseEvent(QMouseEvent *e);
		virtual void mouseDoubleClickEvent(QMouseEvent *e);
		
		virtual void dragEnterEvent(QDragEnterEvent *e);
		virtual void dragLeaveEvent(QDragLeaveEvent *e);
		virtual void dragMoveEvent(QDragMoveEvent *e);
		virtual void dropEvent(QDropEvent *e);
		
		virtual void changeEvent(QEvent *e);
		virtual void showEvent(QShowEvent *);
		virtual void wheelEvent(QWheelEvent *e);
		virtual void resizeEvent(QResizeEvent *e);
		virtual void focusInEvent(QFocusEvent *e);
		virtual void focusOutEvent(QFocusEvent *e);
		
		virtual void contextMenuEvent(QContextMenuEvent *e);
		
		virtual void closeEvent(QCloseEvent *e);
		
		virtual bool focusNextPrevChild(bool next);
		
		virtual bool moveKeyEvent(QDocumentCursor& c, QKeyEvent *e, bool *leave);
		virtual bool isProcessingKeyEvent(QKeyEvent *e);
		virtual bool processCursor(QDocumentCursor& c, QKeyEvent *e, bool& b);
		
		virtual void startDrag();
		virtual QMimeData* createMimeDataFromSelection() const;
		virtual void insertFromMimeData(const QMimeData *d);
		
		virtual void scrollContentsBy(int dx, int dy);
		
		// got to make it public for bindings
	public:
		void setFlag(EditFlag f, bool b);
		
		void pageUp(QDocumentCursor::MoveMode moveMode);
		void pageDown(QDocumentCursor::MoveMode moveMode);
		
		void selectionChange(bool force = false);
		
		void repaintCursor();
		void ensureCursorVisible();
		void ensureVisible(int line);
		void ensureVisible(const QRect &rect);
		
		void insertText(QDocumentCursor& c, const QString& text);
		
		QDocumentLine lineAtPosition(const QPoint& p) const;
		QDocumentCursor cursorForPosition(const QPoint& p) const;
		
		void setClipboardSelection();
		void setCursorPosition(const QPoint& p);
		
		void emitCursorPositionChanged();
		
		void clearCursorMirrors();
		void addCursorMirror(const QDocumentCursor& c);
		
	private slots:
		void documentWidthChanged(int newWidth);
		void documentHeightChanged(int newWidth);
		
		void repaintContent(int i, int n);
		void updateContent (int i, int n);
		
		void markChanged(QDocumentLineHandle *l, int mark, bool on);
		
		void bindingSelected(QAction *a);
		
	private:
		enum SaveState
		{
			Undefined,
			Saving,
			Saved,
			Conflict
		};
		
		void init();
		void updateBindingsMenu();
		
		#ifndef _QMDI_
		QString m_name, m_fileName;
		#endif
		
		QMenu *pMenu;
		QAction *aFocus,
				*aUndo, *aRedo,
				*aCut, *aCopy, *aPaste,
				*aSelectAll,
				*aFind, *aFindNext, *aReplace,
				*aGoto,
				*aIndent, *aUnindent,
				*aComment, *aUncomment
				;
		
		QMenu *m_bindingsMenu;
		QAction *aDefaultBinding;
		QActionGroup *m_bindingsActions;
		
		char m_saveState;
		QFileSystemWatcher *m_watcher;
		
		QDocument *m_doc;
		InputBinding *m_binding;
		
		QPointer<QMarker> m_marker;
		QPointer<QMatcher> m_matcher;
		QPointer<QIndenter> m_indenter;
		QPointer<QHighlighter> m_highlighter;
		QPointer<QLanguageDefinition> m_definition;
		QPointer<QCodeCompletionEngine> m_completionEngine;
		
		QDocumentCursor m_cursor, m_doubleClick, m_dragAndDrop;
		
		QList<QDocumentCursor> m_mirrors;
		
		State m_state;
		QRect m_crect;
		bool m_selection;
		QPoint m_clickPoint, m_dragPoint;
		QBasicTimer m_blink, m_scroll, m_click, m_drag;
		
		static QList<QEditor*> m_editors;
		static InputBinding *m_defaultBinding;
		static QHash<QString, InputBinding*> m_bindings;
};

Q_DECLARE_OPERATORS_FOR_FLAGS(QEditor::State);

#endif
