/*
   FALCON - The Falcon Programming Language.
   FILE: flc_memory.h

   User overridable basic memory allocation
   -------------------------------------------------------------------
   Author: Giancarlo Niccolai
   Begin: mar ago 3 2004

   -------------------------------------------------------------------
   (C) Copyright 2004: the FALCON developers (see list in AUTHORS file)

   See LICENSE file for licensing details.
*/

#ifndef flc_MEMORY_H
#define flc_MEMORY_H

#include <falcon/setup.h>
#include <stdlib.h>

/** \file
   This files contains the user overridable memory allocation systems.

   Memory is alwas allocated via memAlloc() and released via memFree()
   function pointers. These are initially set at DflMemAlloc() and
   DflMemFree() functions, but they may be overridden with functions
   of the same kind at every point.

   Falcon memory management is twofold: Memory used for internal storage
   (i.e. module loading, symbol tables, file names, configurations etc)
   is not accounted nor controlled, and never garbaged; the only control
   is during allocation and releasing via the basic functions, which
   print an error message and quit in case of allocation errors.

   Items used by the VM (generated by scripts or returned to scripts by
   extensions) are created via the MemPool object; the memory
   allocated by this object is stored in the memory pool and may be garbaged,
   recycled or corrected.

   In either case, Falcon never makes any assumption on the given memory. The
   embedding application may safely add any kind of memory management &
   accounting it likes.
*/

namespace Falcon {

FALCON_DYN_SYM void * DflMemAlloc( size_t amount );
FALCON_DYN_SYM void DflMemFree( void *mem );
FALCON_DYN_SYM void * DflMemRealloc( void *mem, size_t amount );

FALCON_DYN_SYM extern void * (*memAlloc) ( size_t );
FALCON_DYN_SYM extern void (*memFree) ( void * );
FALCON_DYN_SYM extern void * (*memRealloc) ( void *,  size_t );

}

#endif

/* end of flc_memory.h */
