/**
 *   Copyright (C) 2004 by JAVIER FERNANDO VARGAS G <fernandovargas@cable.net.co>
 *                 2007 - Alain Portal <aportal AT univ-montp2 DOT fr>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * */

#include "mainwidgetpl.h"
#include <qlayout.h>
#include <qlabel.h>
#include <qtextedit.h>
#include <qlineedit.h>
#include <qstring.h>
#include <qhbox.h>
#include <qvbox.h>
#include <qcombobox.h>
#include <qhgroupbox.h>
#include <qvgroupbox.h>
#include <qtooltip.h>
#include <qdatetime.h>

#include <klocale.h>
#include <krestrictedline.h>


#include "gqtextedit.h"

MainWidgetPL::MainWidgetPL(QWidget *parent, const char *name)
        : QVBox(parent, name)
{
    QFont LabelFont;
    loopValues = new PiKDelay(4.00, 1.00);
    statusMessage = i18n("Actual delay: %1 s  (%2 clock cycles).") ;

    setSpacing(3) ;
    setMargin(6) ;

    // input
    QHGroupBox *grinput = new QHGroupBox(i18n("Input data") , this) ;
    new QLabel(i18n("Frequency (MHz)"), grinput) ;
    clock = new QComboBox(grinput) ;
    static const char* items[] = { "1.000000", "2.000000", "4.000000", "8.000000",
				   "10.000000", "16.000000", "20.000000", "24.000000",
				   "32.000000", "40.000000", 0};
    clock->insertStrList( items );
    clock->setEditable(true) ;
    clock->setCurrentItem(2) ;
    QToolTip::add(clock,i18n("Select a standard device clock, or type a custom clock frequency.")) ;

    (new QHBox(grinput))->setMinimumWidth(40) ;
    new QLabel(i18n("Needed delay (s)"), grinput) ;
    delay = new KRestrictedLine  (grinput,"delay editor","0123456789.") ;
    delay->setText("1");
    delay->setMinimumWidth(60);

    // output
    QVGroupBox *groutput = new  QVGroupBox( i18n("Ouput data") , this) ;
    CodeText = new GQTextEdit(groutput,"code displayer");
    QToolTip::add(CodeText,i18n("Copy and paste this code, or simply drag and drop it to your editor."));
    CodeText->setReadOnly(true);

    cycles = new QLabel(statusMessage
    .arg(loopValues->getTimeDelay(),0,'f',8)
    .arg((loopValues->getMachineCycles()),0,'f',0) , groutput ) ;

    connect(clock, SIGNAL(activated(const QString &)), this, SLOT(ClockChanged(const QString &)));
    connect(delay, SIGNAL(returnPressed()), this, SLOT(TimeDelayChanged()));
    connect(delay, SIGNAL(lostFocus()), this, SLOT(TimeDelayChanged()));

    refreshWidget();

}


MainWidgetPL::~MainWidgetPL()
{

}


/*!
    \fn MainWidgetPL::refreshWidget
 */
void MainWidgetPL::refreshWidget()
{
    // set label names
    QString routineName = i18n("delay_") ;
    routineName += delay->text() ;
    routineName += i18n("_sec") ;
    QString loopName = routineName + i18n("_loop") ;
    QString subRoutineName = i18n("sub_delay_") ;
    subRoutineName += delay->text() ;
    subRoutineName += i18n("_sec") ;
    QString loopSubRoutineName = subRoutineName + i18n("_loop") ;

    // generate comments
    QString tmpCode = "; " ;
    tmpCode += i18n("Delay code generated by PikLoops") ;
    tmpCode += " (" ;
    tmpCode += QDateTime::currentDateTime().toString(i18n("ddd yyyy-MMM-dd hh:mm:ss")) ;
    tmpCode += ")\n; " ;

    tmpCode += QString(i18n("Time Delay = %1s  with  Osc = %2MHz\n\n")).
      arg(loopValues->getTimeDelay(),0,'f',8).arg(loopValues->getClockFrequency(),0,'f',8);

    tmpCode += routineName ;
    tmpCode += "\n" ;

    if ( loopValues->getCounterC() == -1 &&
         loopValues->getCounterB() == -1 &&
         loopValues->getCounterA() == -1 ) {
	tmpCode += "\n; ";
	tmpCode += i18n("PikLoops was unable to calculate the delay you requested.\n");
	if ( loopValues->isOverflow() == 1 ) {
	    double maxDelay = ( 4.0 * MACHINE_CYCLE_MAX ) / ( loopValues->getClockFrequency() * 1000000.0 ) ;
	    tmpCode += "; ";
	    tmpCode += QString(i18n("The greatest delay you can get don't have to exceed %1s").arg(maxDelay,0,'f',8)) ;
	}
	else {
	    double minDelay = ( 4.0 * MACHINE_CYCLE_MIN ) / ( loopValues->getClockFrequency() * 1000000.0 ) ;
	    tmpCode += "; ";
	    tmpCode += QString(i18n("The lowest delay you can get have to be greater than %1s").arg(minDelay,0,'f',8)) ;
	}
    }
    else {
	if ( loopValues->getCounterC() != -1 ) {
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterC(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterC"));
	    tmpCode += "\n";
	    tmpCode += loopName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tcall\t");
	    tmpCode += subRoutineName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterC"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopName);
	    tmpCode += QString("\treturn\n");
	    tmpCode += "\n" ;
	    tmpCode += subRoutineName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterB(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterB"));
	    tmpCode += "\n";
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterA(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += "\n";
	    tmpCode += loopSubRoutineName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopSubRoutineName);
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterB"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopSubRoutineName);
	    tmpCode += QString("\treturn\n");
	}
	else if ( loopValues->getCounterB() !=-1 ) {
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterB(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterB"));
	    tmpCode += "\n";
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterA(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += "\n";
	    tmpCode += loopName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopName);
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterB"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopName);
	    tmpCode += QString("\treturn\n");
	}
	else if ( loopValues->getCounterA() !=-1 ) {
	    tmpCode += QString("\tmovlw\tD'%1'\n").arg(loopValues->getCounterA(),0,10);
	    tmpCode += QString("\tmovwf\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += "\n";
	    tmpCode += loopName ;
	    tmpCode += "\n" ;
	    tmpCode += QString("\tdecfsz\t");
	    tmpCode += QString(i18n("CounterA"));
	    tmpCode += QString(",1\n\tgoto\t%1\n").arg(loopName);
	    tmpCode += QString("\treturn\n");
	}
    }
    // update status line
    cycles->setText(statusMessage.arg(loopValues->getTimeDelay(),0,'f',8)
    .arg((loopValues->getMachineCycles()),0,'f',0) ) ;

    CodeText->setText(tmpCode);
    CodeText->setFocus();
}


/*!
    \fn MainWidgetPL::setValues()
 */
void MainWidgetPL::setValues()
{
    loopValues->setClockFrequency(clock->currentText().toDouble());
    loopValues->setTimeDelay(delay->text().toDouble());
    loopValues->Refresh();
    refreshWidget();
}


/*!
    \fn MainWidgetPL::ClockChanged()
 */
void MainWidgetPL::ClockChanged(const QString& s)
{
    loopValues->setClockFrequency(s.toDouble()) ;
    loopValues->Refresh();
    refreshWidget();
}


/*!
    \fn MainWidgetPL::TimeDelayChanged()
 */
void MainWidgetPL::TimeDelayChanged()
{
    loopValues->setTimeDelay(delay->text().toDouble()) ;
    loopValues->Refresh();
    refreshWidget();
}

#include "mainwidgetpl.moc"


/*!
    \fn MainWidgetPL::copyCode()
 */
void MainWidgetPL::copyCode()
{
    CodeText->selectAll(true);
    CodeText->copy();
    CodeText->selectAll(false);
}
