/**************************************************************************
   fotoxx     	image edit program

   Free software licensed under GNU General Public License v.2
   source URL:  kornelix.squarespace.com
***************************************************************************/

#include <fcntl.h>
#include "zfuncs.h"

#define fversion "fotoxx v.5.7"                                            //  version v.5.7
#define flicense "Free software - GNU General Public License v.2"
#define fsource "http://kornelix.squarespace.com/fotoxx"
#define ftranslators "\n Stanislas Zeller, Antonio Sánchez, Miguel Bouzada" \
                     "\n The Hamsters, Helge Soencksen"
#define ftrash "Desktop/fotoxx-trash"                                      //  trash folder location
#define mega (1048576.0)                                                   //  1024 * 1024
#define undomax 20                                                         //  undo stack size
#define maxtag1 20                                                         //  max tag cc for one tag
#define maxtag2 200                                                        //  max tag cc for one image file
#define maxtag3 9000                                                       //  max tag cc for all image files
#define maxtag4 100                                                        //  max tag cc for search tags
#define maxtag5 100                                                        //  max tag cc for recent tags
#define maxntags 1000                                                      //  max tag count for all images
#define exif_tags_key_read "Exif comment"                                  //  Exif image tags (exiv2 read)
#define exif_tags_key_write "Exif.Photo.UserComment"                       //  Exif image tags (exiv2 write)
#define exif_date_key_read "Image timestamp"                               //  Exif image date (exiv2 read)
#define exif_date_key_write "Exif.Photo.DateTimeOriginal"                  //  Exif image date (exiv2 write)

#define nodither GDK_RGB_DITHER_NONE,0,0                                   //  GDK stuff
#define colorspace GDK_COLORSPACE_RGB
#define BILINEAR GDK_INTERP_BILINEAR
#define HYPER GDK_INTERP_HYPER
#define dottedline GDK_LINE_DOUBLE_DASH,GDK_CAP_BUTT,GDK_JOIN_MITER
#define autoscroll GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC               //  scroll window policy
#define textwin GTK_TEXT_WINDOW_TEXT                                       //  GDK window of GTK text view

#define pixbuf_new(a,b,c,d,e) gdk_pixbuf_new(a,b,c,d,e); incr_pixbufs();
#define pixbuf_scale_simple(a,b,c,d) gdk_pixbuf_scale_simple(a,b,c,d);
#define pixbuf_new_from_file(a,b) gdk_pixbuf_new_from_file(a,b); incr_pixbufs();
#define pixbuf_copy(a) gdk_pixbuf_copy(a); incr_pixbufs();
#define pixbuf_rotate(a,b) gdk_pixbuf_rotate(a,b); incr_pixbufs();
#define pixbuf_copy_area(a,b,c,d,e,f,g,h) gdk_pixbuf_copy_area(a,b,c,d,e,f,g,h);
#define pixbuf_draw(a,b,c,d,e,f,g,h,i) gdk_draw_pixbuf(a,b,c,d,e,f,g,h,i,nodither);
#define pixbuf_test(ID) if (!pxb##ID) zappcrash("memory allocation failure");

#define pixbuf_poop(ID)                                                    \
   ww##ID = gdk_pixbuf_get_width(pxb##ID);                                 \
   hh##ID = gdk_pixbuf_get_height(pxb##ID);                                \
   rs##ID = gdk_pixbuf_get_rowstride(pxb##ID);                             \
   ppix##ID = gdk_pixbuf_get_pixels(pxb##ID);

#define getparm(pname) parm_##pname = getParm(#pname);
#define setparm(pname,value) parm_##pname = value; setParm(#pname,parm_##pname);

extern char    JPGquality[4];                                              //  quality param for gdk_pixbuf_save()

namespace      image_navi { 
extern int     xwinW, xwinH;                                               //  thumbnail index window size
extern int     thumbsize;                                                  //  thumbnail image size
}

GtkWidget      *mWin, *dWin;                                               //  main and drawing window
GtkWidget      *mVbox, *mScroll; 
GtkWidget      *mmbar, *mtbar, *stbar;                                     //  menu bar, tool bar, status bar
GtkWidget      *postmessage;
GError         **gerror = 0;
GdkGC          *gdkgc = 0;                                                 //  graphics context
GdkColor       black, white;
GdkColormap    *colormap = 0;
uint           maxcolor = 0xffff;
GdkCursor      *arrowcursor = 0;
GdkCursor      *dragcursor = 0;
int            Fexiv2 = 0;                                                 //  exiv2 program availability   v.5.1
int            Fprintoxx = 0;                                              //  printoxx program availability  v.5.2

//  pixbufs: pxb1, pxb2 = unchanged input images
//           pxb3 = modified output image
//           pxbM = main/drawing window image
//  pxb3 * Rscale >> pxbM >> dWin

GdkPixbuf   *pxb1 = 0, *pxb2 = 0;                                          //  input image1 and image2
GdkPixbuf   *pxb3 = 0;                                                     //  modified / output image3
GdkPixbuf   *pxbM = 0;                                                     //  main window drawing image
GdkPixbuf   *pxb1A = 0, *pxb2A = 0;                                        //  alignment images
GdkPixbuf   *pxb1C = 0, *pxb2C = 0;                                        //  alignment images, curved

int      nch, nbits;                                                       //  pixbuf/image attributes
int      ww1, hh1, rs1;                                                    //  pxb1  width, height, rowstride
int      ww2, hh2, rs2;                                                    //  pxb2  width, height, rowstride
int      ww3, hh3, rs3;                                                    //  pxb3  width, height, rowstride
int      wwM, hhM, rsM;                                                    //  pxbM  width, height, rowstride
int      wwD = 1000, hhD = 700;                                            //  drawing window width, height
int      ww1A, hh1A, rs1A;                                                 //  pxb1A width, height, rowstride
int      ww2A, hh2A, rs2A;                                                 //  pxb2A width, height, rowstride
int      ww1C, hh1C, rs1C;                                                 //  pxb1C width, height, rowstride
int      ww2C, hh2C, rs2C;                                                 //  pxb2C width, height, rowstride

int      orgMx = 0, orgMy = 0;                                             //  pxbM origin in drawing window
int      Mrefresh = 0;                                                     //  flag: refresh/rescale main window
int      Mfreeze = 0;                                                      //  flag: freeze window updates
double   Mscale = 0;                                                       //  pxbM scale: 0 (fit window), 1x, 2x
double   Rscale = 1;                                                       //  curr. pxb3 to pxbM scaling factor
int      Fscale = 0;                                                       //  flag: fit window always (slide show)
int      Fredraw = 0;                                                      //  flag: main window was updated

int      mpDx, mpDy;                                                       //  mouse position in drawing window
int      mpMx, mpMy;                                                       //  corresp. position in pxbM
int      mp3x, mp3y;                                                       //  corresp. position in pxb3
int      LMclick = 0, RMclick = 0;                                         //  L/R-mouse button clicked
int      LMdown = 0;                                                       //  L-mouse button down (drag underway)
int      Mcapture = 0;                                                     //  mouse clicks handled by function
int      clickMx, clickMy;                                                 //  mouse click position in pxbM
int      click3x, click3y;                                                 //  corresp. position in pxb3
int      zoom3x, zoom3y;                                                   //  current zoom center, pxb3 space
int      mdMx1 = 0, mdMy1 = 0, mdMx2 = 0, mdMy2 = 0;                       //  mouse drag vector in pxbM space
int      md3x1 = 0, md3y1 = 0, md3x2 = 0, md3y2 = 0;                       //  corresp. vector in pxb3 space
int      KBkey = 0;                                                        //  keyboard key

typedef  uchar *pixel;                                                     //  3 RGB values, 0-255 each
pixel    ppix1, ppix2, ppix3;                                              //  pxb1, pxb2, pxb3  pixels
pixel    ppix1A, ppix2A, ppix1C, ppix2C;                                   //  pxb1A, pxb2A, pxb1C, pxb2C  pixels
pixel    ppixM;                                                            //  pxbM pixels

char     clfile[maxfcc] = "";                                              //  command line image file
char     cldirk[maxfcc] = "";                                              //  command line image directory
char     imagedirk[maxfcc] = "";                                           //  current image file directory
char     topdirk[maxfcc] = "";                                             //  top-level image directoryt
char     asstagsfile[200];                                                 //  /home/user/.fotoxx/assigned_tags
char     *file1 = 0, *file2 = 0;                                           //  image1 and image2 pathnames
char     fname1[100] = "", fname3[100] = "";                               //  image1 and image3 file names
int      f1size, f3size;                                                   //  image1 and image3 file sizes
int      f1posn, f1count;                                                  //  image1 position in directory

int      debug = 0;                                                        //  set by command line -d parameter
int      fdestroy = 0;                                                     //  quit / destroy flag
int      fullSize, alignSize;                                              //  full and align image sizes
int      Fmod3 = 0;                                                        //  image3 has been modified (edited)
int      func_busy = 0;                                                    //  function active counter
int      kill_func = 0;                                                    //  function kill flag

zdialog  *zdedit = null;                                                   //  image edit dialog
zdialog  *zdsela = null;                                                   //  select area dialog
zdialog  *zdtags = null;                                                   //  edit tags dialog

GdkPixbuf   *undostack[undomax];                                           //  undo/redo stack of saved images
int         Nundos;                                                        //  count

int      Nalign = 0;                                                       //  alignment progress counter
int      showRedpix = 0;                                                   //  flag, highlight alignment pixels
int      Fautolens = 0;                                                    //  flag, autolens function running
int      pxL, pxH, pyL, pyH;                                               //  image overlap rectangle
int      pixsamp;                                                          //  pixel sample size (HDR, pano)
double   rotate_angle = 0;                                                 //  image3 rotatation vs. image1

bitmap   *BMpixels = 0;                                                    //  flags edge pixels for image align

double   Bratios1[3][256], Bratios2[3][256];                               //  brightness ratios /color /brightness
double   colormatch1R[256], colormatch1G[256], colormatch1B[256];          //  image1 color adjustment factors 
double   colormatch2R[256], colormatch2G[256], colormatch2B[256];          //  image2 color adjustment factors
double   Radjust, Gadjust, Badjust;                                        //  RGB manual adjustmants (1 = neutral)
double   lens_curve, lens_mm, lens_bow, lens_mmB, lens_bowB;               //  lens parameters: focal length, bow
double   xoff, yoff, toff, yst1, yst2;                                     //  align offsets: x/y/theta/y-stretch
double   xoffB, yoffB, toffB, yst1B, yst2B;                                //  align offsets: current best values
double   xystep, tstep, xylim, tlim, yststep, ystlim;                      //  align step size and search range
double   matchlev, matchB;                                                 //  image alignment match level
double   blend;                                                            //  image blend width (pano)

//  GTK functions
int   gtkinitfunc(void *data);                                             //  GTK initz. function
void  mwpaint();                                                           //  window repaint - expose event
void  mwpaint2();                                                          //  window repaint - image modified
void  update_status_bar();                                                 //  update status bar
int   KBpress(GtkWidget *, GdkEventKey *, void *);                         //  KB key press event function
int   KBrelease(GtkWidget *, GdkEventKey *, void *);                       //  KB key release event
void  mouse_event(GtkWidget *, GdkEventButton *, void *);                  //  mouse event function
void  draw_dotline(int x1, int y1, int x2, int y2);                        //  draw dotted-line, pxb3 space
void  erase_dotline(int x1, int y1, int x2, int y2);                       //  erase dotted-line
void  menufunc(GtkWidget *, const char *menu);                             //  menu function, main window
int   delete_event();                                                      //  main window delete_event signal
void  destroy();                                                           //  main window destroy signal

//  mouse and keyboard handler functions
typedef void CBfunc();                                                     //  function type
CBfunc   *mouseCBfunc = 0;                                                 //  current mouse handler function
CBfunc   *KBkeyCBfunc = 0;                                                 //  current KB key handler function

//  file functions
void  m_index();                                                           //  show thumbnail index window
void  m_open(char *file);                                                  //  open image file
void  m_save();                                                            //  save modified image
void  m_trash();                                                           //  move image to trash
void  m_print();                                                           //  print image file(s)
void  m_prev();                                                            //  open previous file
void  m_next();                                                            //  open next file
void  m_quit();                                                            //  exit application

//  Etc. functions
void  m_parms();                                                           //  adjust parameters
void  m_thumbs();                                                          //  generate thumbnails
void  m_montest();                                                         //  check monitor
void  m_slideshow();                                                       //  slideshow mode
void  m_clone();                                                           //  start another fotoxx instance

//  help functions
void  m_help(const char *menu);                                            //  display user guide, README, etc.

//  tag functions
void  m_edit_tags();                                                       //  modify image tags menu function
void  edit_tags_dialog();                                                  //  start edit tags dialog
void  load_filetags(char *file);                                           //  file Exif key >> tags_filetags
void  update_filetags(char *file);                                         //  tags_filetags >> file Exif key
void  load_asstags();                                                      //  asstagsfile >> tags_asstags (all tags)
void  update_asstags(char *file, int del = 0);                             //  file + tags_filetags >> asstagsfile
void  m_search_tags();                                                     //  search images for matching tags
void  m_build_tags_index();                                                //  rebuild assigned tags file

//  Exif functions
void  m_exif();                                                            //  show Exif data
int   exif_fetch(char *file);                                              //  image Exif data  >>  file.exv
int   exif_stuff(char *file1, char *file3, int ww, int hh);                //  file.exv  >>  image Exif data
int   set_exif_data(const char *file, const char *key, const char *text);  //  set Exif key data
char * get_exif_data(const char *file, const char *key);                   //  get Exif key data

//  edit functions
void  m_select();                                                          //  select image area to edit
void  m_flatten();                                                         //  flatten brightness distribution
void  m_tune();                                                            //  adjust brightness/colors
void  m_color_dep();                                                       //  set image color depth
void  m_color_int();                                                       //  set image color intensity
void  m_rgb_spread();                                                      //  set image RGB spread
void  m_sharp();                                                           //  sharpen image
void  m_blur();                                                            //  blur image
void  m_denoise();                                                         //  image noise reduction
void  m_redeye();                                                          //  red-eye removal  v.5.5
void  m_trim();                                                            //  trim image
void  m_rotate(double angle);                                              //  rotate image
void  m_resize();                                                          //  resize image
void  m_unbend();                                                          //  unbend panorama image
void  m_warp();                                                            //  warp/distort image area
void  m_HDR();                                                             //  make HDR image
void  m_pano();                                                            //  make panorama image

//  edit support functions
void  m_undo();                                                            //  undo changes
void  m_redo();                                                            //  redo changes
void  m_zoom(const char *which);                                           //  zoom image in/out
void  m_kill();                                                            //  kill running function
void  m_RGB();                                                             //  show RGB values at mouse click

//  pano and HDR common functions
pixel  vpixel(pixel, double px, double py, int, int, int);                 //  get virtual pixel (px,py)
int    sigdiff(double d1, double d2, double signf);                        //  test for significant difference
void   get_overlap_region(GdkPixbuf *pxb1, GdkPixbuf *pxb2);               //  get image overlap rectangle
void   get_Bratios(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                      //  get brightness ratios for overlap 
void   set_colormatch(int state);                                          //  set color matching factors on/off
void   flag_edge_pixels(GdkPixbuf *pixbuf);                                //  flag high-contrast pixels in overlap 
double match_images(GdkPixbuf *pxb1, GdkPixbuf *pxb2);                     //  match images in overlap region
double match_pixels(pixel pix1, pixel pix2);                               //  match two pixels

//  suppporting functions
double brightness(pixel);                                                  //  get pixel brightness
double redness(pixel);                                                     //  get pixel redness
void   setredness(pixel, double redness);                                  //  set pixel redness
int    mod_keep();                                                         //  query to discard mods
void   save_fotoxx_state();                                                //  save state data at exit
void   load_fotoxx_state();                                                //  reload state data at startup
void   push_image3();                                                      //  push image3 into undo stack
void   pull_image3();                                                      //  pull image3 from undo stack (undo)
void   prior_image3();                                                     //  get last image3, no stack removal
void   redo_image3();                                                      //  advance undo stack without push
void   clearmem_image3();                                                  //  clear undo stack
void   select_area_clear();                                                //  clear selected image area
GdkPixbuf * load_pixbuf(const char *file);                                 //  load pixbuf from file, remove alpha 
void   incr_pixbufs();                                                     //  track pixbufs allocated
void   pixbuf_free(GdkPixbuf *&pixbuf);                                    //  track pixbufs freed

//  dialog buttons
const char  *BOK;                                                          //  gettext strings used > 1x
const char  *Bstart;
const char  *Bfinish;
const char  *Bshow;
const char  *Bclear;
const char  *Bdelete;
const char  *Binvert;
const char  *Bcancel;
const char  *Bdone;
const char  *Bundo;
const char  *Bredo;
const char  *Bgraph;
const char  *Bapply;
const char  *Breset;
const char  *Btrim;
const char  *Bproceed;
const char  *Bsearch;
const char  *Bwidth;
const char  *Bheight;
const char  *Bpercent;
const char  *Bpreset;
const char  *Bred;
const char  *Bgreen;
const char  *Bblue;
const char  *Bbrightness;
const char  *Bwhiteness;
const char  *Bblendwidth;

//  messages
const char  *M_totalTagsExceed;
const char  *M_asstagsError;
const char  *M_noexiv2;
const char  *M_seltopdir;

//  adjustable parameters
double   parm_pixel_sample_size;
double   parm_jpg_save_quality;
double   parm_pano_lens_mm;
double   parm_pano_lens_bow;
double   parm_pano_prealign_size;
double   parm_pano_mouse_leverage;
double   parm_pano_align_size_increase;
double   parm_pano_blend_reduction;
double   parm_pano_minimum_blend;
double   parm_pano_image_stretch;

/**************************************************************************/

//  main program

int main(int argc, char *argv[])
{
   GtkWidget   *mFile, *mEdit, *mTags, *mHelp;
   int8        dashes[2] = { 2, 2 };
   int         ii;
   char        lang[4] = "";

   gtk_init(&argc,&argv);                                                  //  GTK command line options
   if (! g_thread_supported()) g_thread_init(null);                        //  initz. GTK for threads
   gdk_threads_init();
   zlockInit();

   initz_appfiles("fotoxx","parameters",null);                             //  get app directories, parameter file
   load_fotoxx_state();                                                    //  get state data from last session

   for (ii = 1; ii < argc; ii++)                                           //  fotoxx command line options
   {
      if (strEqu(argv[ii],"-d"))                                           //  -d (debug flag)
            debug = 1;
      else if (strEqu(argv[ii],"-v")) {                                    //  -v print version and exit   v.5.2
            printf(fversion "\n"); return 0; }
      else if (strEqu(argv[ii],"-l") && argc > ii+1)                       //  -l language code
            strncpy0(lang,argv[++ii],3);
      else if (strEqu(argv[ii],"-i") && argc > ii+1)                       //  -i directory
            strcpy(cldirk,argv[++ii]);
      else if (strEqu(argv[ii],"-f") && argc > ii+1)                       //  -f file
            strcpy(clfile,argv[++ii]);
      else strcpy(clfile,argv[ii]);                                        //  file
   }
   
   ZTXinit(lang);                                                          //  setup translations   v.5.7

   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mWin),fversion);
   gtk_window_set_position(GTK_WINDOW(mWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mWin),wwD,hhD+97);               //  (+ menu + toolbar)

   mVbox = gtk_vbox_new(0,0);                                              //  add vert. packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);

   mmbar = create_menubar(mVbox,16);                                       //  add menu bar and menu items

   mFile = add_menubar_item(mmbar,ZTX("File"),menufunc);
      add_submenu_item(mFile,ZTX("Open"),"m-open.png",menufunc);
      add_submenu_item(mFile,ZTX("Save"),"m-save.png",menufunc);
      add_submenu_item(mFile,ZTX("Trash"),"m-trash.png",menufunc);
      add_submenu_item(mFile,ZTX("Print"),"m-print.png",menufunc);
      add_submenu_item(mFile,ZTX("Quit"),"m-quit.png",menufunc);
      add_submenu_item(mFile,ZTX("Edit Parameters"),"m-params.png",menufunc);
      add_submenu_item(mFile,ZTX("Create Thumbnails"),"m-create-thumbs.png",menufunc);
      add_submenu_item(mFile,ZTX("Check Monitor"),"m-montest.png",menufunc);
      add_submenu_item(mFile,ZTX("Slide Show"),"m-slideshow.png",menufunc);
      add_submenu_item(mFile,ZTX("Clone"),"m-clone.png",menufunc);

   mEdit = add_menubar_item(mmbar,ZTX("Edit"),menufunc);
      add_submenu_item(mEdit,ZTX("Select Area"),"m-select.png",menufunc);
      add_submenu_item(mEdit,ZTX("Brightness Distribution"),"m-flatten.png",menufunc);
      add_submenu_item(mEdit,ZTX("Brightness/Contrast/Color"),"m-tune.png",menufunc);
      add_submenu_item(mEdit,ZTX("Color Depth"),"m-color-depth.png",menufunc);
      add_submenu_item(mEdit,ZTX("Color Intensity"),"m-color-inten.png",menufunc);
      add_submenu_item(mEdit,ZTX("RGB Spread"),"m-rgb-spread.png",menufunc);
      add_submenu_item(mEdit,ZTX("Sharpen"),"m-sharpen.png",menufunc);
      add_submenu_item(mEdit,ZTX("Blur"),"m-blur.png",menufunc);
      add_submenu_item(mEdit,ZTX("Reduce Noise"),"m-noise.png",menufunc);
      add_submenu_item(mEdit,ZTX("Red Eye"),"m-redeye.png",menufunc);
      add_submenu_item(mEdit,ZTX("Trim"),"m-trim.png",menufunc);
      add_submenu_item(mEdit,ZTX("Rotate"),"m-rotate.png",menufunc);
      add_submenu_item(mEdit,ZTX("Resize"),"m-resize.png",menufunc);
      add_submenu_item(mEdit,ZTX("Unbend"),"m-unbend.png",menufunc);
      add_submenu_item(mEdit,ZTX("Warp"),"m-warp.png",menufunc);
      add_submenu_item(mEdit,ZTX("HDR"),"m-hdr.png",menufunc);
      add_submenu_item(mEdit,ZTX("Panorama"),"m-pano.png",menufunc);

   mTags = add_menubar_item(mmbar,ZTX("Tags"),menufunc);
      add_submenu_item(mTags,ZTX("Edit Tags"),"m-tags.png",menufunc);
      add_submenu_item(mTags,ZTX("Search Tags"),"m-tags.png",menufunc);
      add_submenu_item(mTags,ZTX("Build Tags Index"),"m-tags.png",menufunc);
      add_submenu_item(mTags,ZTX("View Exif Data"),"m-exif.png",menufunc);

   mHelp = add_menubar_item(mmbar,ZTX("Help"),menufunc);
      add_submenu_item(mHelp,ZTX("About"),"m-about.png",menufunc);
      add_submenu_item(mHelp,ZTX("User Guide"),"m-userguide.png",menufunc);
      add_submenu_item(mHelp,ZTX("README"),"m-readme.png",menufunc);
      add_submenu_item(mHelp,ZTX("Change Log"),"m-changelog.png",menufunc);
      add_submenu_item(mHelp,ZTX("Error Log"),"m-errorlog.png",menufunc);
      add_submenu_item(mHelp,ZTX("Translations"),"m-readme.png",menufunc);

   mtbar = create_toolbar(mVbox,24);                                       //  add toolbar and buttons  
      add_toolbar_button(mtbar,ZTX("index"),ZTX("thumbnail index"),"index.png",menufunc);
      add_toolbar_button(mtbar,ZTX("open"),ZTX("open image file"),"file.png",menufunc);
      add_toolbar_button(mtbar,ZTX("toolbar::save"),ZTX("save image to a file"),"save.png",menufunc);
      add_toolbar_button(mtbar,ZTX("trash"),ZTX("move image file to trash"),"trash.png",menufunc);
      add_toolbar_button(mtbar,ZTX("prev"),ZTX("open previous image file"),"prev.png",menufunc);
      add_toolbar_button(mtbar,ZTX("next"),ZTX("open next image file"),"next.png",menufunc);
      add_toolbar_button(mtbar,ZTX("undo"),ZTX("undo image changes"),"undo.png",menufunc);
      add_toolbar_button(mtbar,ZTX("redo"),ZTX("redo image changes"),"redo.png",menufunc);
      add_toolbar_button(mtbar,ZTX("zoom+"),ZTX("bigger image"),"zoomin.png",menufunc);
      add_toolbar_button(mtbar,ZTX("zoom-"),ZTX("smaller image"),"zoomout.png",menufunc);
      add_toolbar_button(mtbar,ZTX("kill"),ZTX("kill running function"),"kill.png",menufunc);
      add_toolbar_button(mtbar,ZTX("quit"),ZTX("quit fotoxx"),"quit.png",menufunc);

   mScroll = gtk_scrolled_window_new(0,0);                                 //  add scrolled window    v.5.0
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mScroll),
                          GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
   gtk_box_pack_start(GTK_BOX(mVbox),mScroll,1,1,0);
   
   dWin = gtk_layout_new(0,0);                                             //  drawing window
   gtk_container_add(GTK_CONTAINER(mScroll),dWin);                         //  add to scrolled window

   stbar = create_stbar(mVbox);                                            //  add status bar

   G_SIGNAL(mWin,"delete_event",delete_event,0)                            //  connect signals to window
   G_SIGNAL(mWin,"destroy",destroy,0)
   G_SIGNAL(dWin,"expose-event",mwpaint,0)

   G_SIGNAL(mWin,"key-press-event",KBpress,0)                           	//  connect KB events
   G_SIGNAL(mWin,"key-release-event",KBrelease,0)

   gtk_widget_add_events(dWin,GDK_BUTTON_PRESS_MASK);                      //  connect mouse events
   gtk_widget_add_events(dWin,GDK_BUTTON_RELEASE_MASK);
   gtk_widget_add_events(dWin,GDK_BUTTON_MOTION_MASK);
   gtk_widget_add_events(dWin,GDK_POINTER_MOTION_MASK);
   G_SIGNAL(dWin,"button-press-event",mouse_event,0)
   G_SIGNAL(dWin,"button-release-event",mouse_event,0)
   G_SIGNAL(dWin,"motion-notify-event",mouse_event,0)
   
   gtk_widget_show_all(mWin);                                              //  show all widgets

   gdkgc = gdk_gc_new(dWin->window);                                       //  initz. graphics context

   black.red = black.green = black.blue = 0;                               //  set up colors black, white
   white.red = white.green = white.blue = maxcolor;
   colormap = gtk_widget_get_colormap(dWin);
   gdk_rgb_find_color(colormap,&black);
   gdk_rgb_find_color(colormap,&white);

   gdk_gc_set_foreground(gdkgc,&black);                                    //  set up to draw dotted lines
   gdk_gc_set_background(gdkgc,&white);
   gdk_gc_set_dashes(gdkgc,0,(gint8 *) dashes,2);
   gdk_gc_set_line_attributes(gdkgc,1,dottedline);

   arrowcursor = gdk_cursor_new(GDK_TOP_LEFT_ARROW);                       //  cursor for selection
   dragcursor = gdk_cursor_new(GDK_CROSSHAIR);                             //  cursor for dragging

   gtk_init_add((GtkFunction) gtkinitfunc,0);                              //  setup initz. call from gtk_main()

   gdk_threads_enter();
   gtk_main();                                                             //  process window events
   gdk_threads_leave();

   return 0;
}


//  initial function called from gtk_main() at startup

int gtkinitfunc(void * data)
{
   char     *pp;
   int      err, qual;

   BOK = ZTX("OK");
   Bstart = ZTX("start");
   Bfinish = ZTX("finish");
   Bshow = ZTX("show");
   Bclear = ZTX("clear");
   Bdelete = ZTX("delete");
   Binvert = ZTX("invert");
   Bcancel = ZTX("cancel");
   Bdone = ZTX("done");
   Bundo = ZTX("undo");
   Bredo = ZTX("redo");
   Bgraph = ZTX("graph");
   Bapply = ZTX("apply");
   Breset = ZTX("reset");
   Btrim = ZTX("Trim");
   Bproceed = ZTX("proceed");
   Bsearch = ZTX("search");
   Bwidth = ZTX("width");
   Bheight = ZTX("height");
   Bpercent = ZTX("percent");
   Bpreset = ZTX("presets");
   Bred = ZTX("red");
   Bgreen = ZTX("green");
   Bblue = ZTX("blue");
   Bbrightness = ZTX("brightness");
   Bwhiteness = ZTX("whiteness");
   Bblendwidth = ZTX("blend width");

   M_noexiv2 = ZTX("exiv2 package is required");
   M_seltopdir = ZTX("select top image directory");
   M_totalTagsExceed = ZTX("total tags exceed %d characters");
   M_asstagsError = ZTX("assigned tags file error %s");

   err = system("exiv2 -V > /dev/null");                                   //  check for exiv2           v.5.1
   if (err) zmessageACK(ZTX("exiv2 package is not installed \n"
                           "(edited images will lose EXIF data)"));        //  exiv2 warning             v.5.7
   else Fexiv2 = 1;
   
   err = system("printoxx -v > /dev/null");                                //  check for printoxx        v.5.2
   if (err) printf("printoxx not installed \n");
   else Fprintoxx = 1;

   snprintf(asstagsfile,199,"%s/assigned_tags",get_zuserdir());            //  assigned tags file: ~/.fotoxx

   initParmlist(20);                                                       //  set default parameters
   setparm(pixel_sample_size,5)
   setparm(jpg_save_quality,85)                                            //  default JPG quality   v.5.2
   setparm(pano_lens_mm,40)                                                //  default 40mm lens
   setparm(pano_lens_bow,0)                                                //  default zero bow
   setparm(pano_prealign_size,500)
   setparm(pano_mouse_leverage,2)
   setparm(pano_align_size_increase,1.6)
   setparm(pano_blend_reduction,0.7)
   setparm(pano_minimum_blend,10)
   setparm(pano_image_stretch,1)                                           //  enable by default   v.5.3
   
   initz_userParms();                                                      //  load or initz. user parm file

   getparm(pixel_sample_size)                                              //  get final parm values
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)
   
   qual = int(parm_jpg_save_quality);                                      //  set JPG file save quality
   if (qual < 0 || qual > 100) qual = 80;
   snprintf(JPGquality,4,"%d",qual);
   
   if (*cldirk) strcpy(imagedirk,cldirk);                                  //  command line image directory

   if (*clfile) {                                                          //  command line image file
      if (*clfile != '/') {
         pp = zmalloc(strlen(imagedirk)+strlen(clfile)+2);                 //  clfile is relative to imagedirk
         strcpy(pp,imagedirk);
         strcat(pp,"/");
         strcat(pp,clfile);
      }
      else  pp = strdupz(clfile);                                          //  use clfile
      m_open(pp);                                                          //  open command line file
   }
   
   else if (! blank_null(fname1)) {                                        //  try for last file opened  v.5.5
      pp = zmalloc(strlen(imagedirk)+strlen(fname1)+2);
      strcpy(pp,imagedirk);
      strcat(pp,"/");
      strcat(pp,fname1);
      m_open(pp);
   }
   
   return 0;
}


//  paint window when created, exposed, or resized
//  uses pxb3 (current modified image3)
//  no zlock(), called only from main thread

void mwpaint()                                                             //  overhauled for scrolling   v.5.0
{
   static 
   GtkPolicyType     hpolicy, vpolicy;
   GtkAdjustment     *adjustment;
   static int        wwP = 0, hhP = 0;
   int               wsize, hsize;
   double            scalew, scaleh;
   double            scrollx, scrolly, centerx, centery;
   
   if (fdestroy) return;                                                   //  shutdown underway
   
   if (! pxb3) {                                                           //  no image to show
      gdk_window_clear(GTK_LAYOUT(dWin)->bin_window);
      return;
   }

   //  save current image center position in window, before resize or rescale    v.5.3

   adjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(mScroll));
   scrollx = gtk_adjustment_get_value(adjustment);      
   centerx = (scrollx + 0.5 * wwD) * ww3 / wwM;
   adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mScroll));
   scrolly = gtk_adjustment_get_value(adjustment);
   centery = (scrolly + 0.5 * hhD) * hh3 / hhM;
   
   wwD = mScroll->allocation.width;                                        //  layout area window size
   hhD = mScroll->allocation.height;                                       //  (without scrollbars)

   if (wwD != wwP || hhD != hhP) {
      Mrefresh = 1;                                                        //  if changed, force refresh
      wwP = wwD;
      hhP = hhD;
   }
   
   if (! Mrefresh) {                                                       //  draw unchanged image to window
      pixbuf_draw(GTK_LAYOUT(dWin)->bin_window,0,pxbM,0,0,orgMx,orgMy,-1,-1);
      ++Fredraw;                                                           //  flag: function may need to update
      return;
   }
   
   //  pxb3 image or window size changed, must rescale image to new scrolling area

   gdk_window_freeze_updates(GTK_LAYOUT(dWin)->bin_window);                //  stop updates from scroll settings

   pixbuf_free(pxbM);
   
   if (Mscale == 0)                                                        //  scale to fit window
   {
      if (ww3 <= wwD && hh3 <= hhD && ! Fscale) {                          //  image < window, scale to 100%
         Rscale = 1;
         wwM = ww3;
         hhM = hh3;
         pxbM = pixbuf_copy(pxb3);
      }
      else {                                                               //  else scale to window size
         scalew = 1.0 * wwD / ww3;
         scaleh = 1.0 * hhD / hh3;
         if (scalew < scaleh) Rscale = scalew;
         else Rscale = scaleh;
         wwM = int(Rscale * ww3 + 0.5);
         hhM = int(Rscale * hh3 + 0.5);
         pxbM = pixbuf_scale_simple(pxb3,wwM,hhM,BILINEAR);
      }
   }
   
   else if (Mscale == 1) {                                                 //  scale to 100%
      Rscale = 1;
      wwM = ww3;
      hhM = hh3;
      pxbM = pixbuf_copy(pxb3);
   }

   else {                                                                  //  scale to current zoom setting
      Rscale = Mscale;
      wwM = int(ww3 * Rscale);
      hhM = int(hh3 * Rscale);
      pxbM = pixbuf_scale_simple(pxb3,wwM,hhM,BILINEAR);
   }
   
   pixbuf_test(M);                                                         //  check new scaled image
   pixbuf_poop(M);

   if (wwM > wwD) wsize = wwM;                                             //  scroll region is whole image
   else wsize = wwD;                                                       //    or drawing window size
   if (hhM > hhD) hsize = hhM;
   else hsize = hhD;
   gtk_layout_set_size(GTK_LAYOUT(dWin),wsize,hsize);

   orgMx = orgMy = 0;
   if (wwM < wwD) orgMx = (wwD - wwM) / 2;                                 //  if image < window, center image
   if (hhM < hhD) orgMy = (hhD - hhM) / 2;

   if (orgMx || orgMy) gdk_window_clear(GTK_LAYOUT(dWin)->bin_window);     //  clear window if image < window

   pixbuf_draw(GTK_LAYOUT(dWin)->bin_window,0,pxbM,0,0,orgMx,orgMy,-1,-1); //  draw scaled image to window

   if (wwM > wwD) hpolicy = GTK_POLICY_ALWAYS;                             //  set scrollbars present/absent
   else hpolicy = GTK_POLICY_NEVER;
   if (hhM > hhD) vpolicy = GTK_POLICY_ALWAYS;
   else vpolicy = GTK_POLICY_NEVER;
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(mScroll),hpolicy,vpolicy);

   if (wwM > wwD) {                                                        //  v.5.3
      if (zoom3x) centerx = zoom3x;                                        //  center image on mouse click
      centerx = centerx / ww3 * wwM;                                       //  or keep old image center
      scrollx = (centerx - 0.5 * wwD) / (wwM - wwD);
      if (scrollx < 0) scrollx = 0;
      if (scrollx > 1) scrollx = 1;
      scrollx = scrollx * (wwM - wwD);
      adjustment = gtk_scrolled_window_get_hadjustment(GTK_SCROLLED_WINDOW(mScroll));
      gtk_adjustment_set_value(adjustment,scrollx);
   }

   if (hhM > hhD) {
      if (zoom3y) centery = zoom3y;
      centery = centery / hh3 * hhM;
      scrolly = (centery - 0.5 * hhD) / (hhM - hhD);
      if (scrolly < 0) scrolly = 0;
      if (scrolly > 1) scrolly = 1;
      scrolly = scrolly * (hhM - hhD);
      adjustment = gtk_scrolled_window_get_vadjustment(GTK_SCROLLED_WINDOW(mScroll));
      gtk_adjustment_set_value(adjustment,scrolly);
   }
   
   gdk_window_thaw_updates(GTK_LAYOUT(dWin)->bin_window);                  //  do accumulated window updates

   zoom3x = zoom3y = 0;                                                    //  forget clicked position  v.5.3
   Mrefresh = 0;                                                           //  reset refresh flag
   ++Fredraw;                                                              //  flag: function may need to update
   update_status_bar();                                                    //  update status bar
   return;
}


//  invalidate window - cause window to get repainted immediately
//  can be called from main() or from a thread

void mwpaint2()
{
   if (fdestroy) return;                                                   //  shutdown underway
   Mrefresh++;                                                             //  force refresh from image3
   zlock();
   gdk_window_invalidate_rect(dWin->window,0,1);                           //  invalidate whole window  v.5.5
   zunlock();
   return;
}


//  update main window status bar

void update_status_bar()                                                   //  v.5.0
{
   char        msg[200], msg1[30], msg2[30], msg3[90], msg4[30];
   const char  *fmt3 = "  aligning: %d %+.1f %+.1f %+.4f  stretch: %+.1f %+.1f  match: %.4f";
   int         pct = int(100 * Rscale + 0.4);
   
   if (fdestroy) return;
   
   *msg1 = *msg2 = *msg3 = *msg4 = 0;
   if (file1) snprintf(msg1,29,"%dx%d  %.2fMB  %d%c",ww3,hh3,f3size/mega,pct,'%');
   if (Fmod3) snprintf(msg2,29,"  modified, undo=%d",Nundos);
   if (Nalign) snprintf(msg3,89,fmt3,Nalign,xoffB,yoffB,toffB,yst1B,yst2B,matchB);
   if (Fautolens) snprintf(msg4,29,"  lens mm: %.1f  bow: %.2f",lens_mmB,lens_bowB);
   snprintf(msg,199,"%s%s%s%s",msg1,msg2,msg3,msg4);
   stbar_message(stbar,msg);
   return;
}


//  keyboard event function - some toolbar buttons have KB equivalents
//  GDK key symbols: /usr/include/gtk-2.0/gdk/gdkkeysyms.h

int KBpress(GtkWidget *win, GdkEventKey *event, void *)                    //  prevent propagation of key-press
{                                                                          //    events to toolbar buttons  v.5.7
   return 1;
}


int KBrelease(GtkWidget *win, GdkEventKey *event, void *)
{
   KBkey = event->keyval;

   if (KBkeyCBfunc) {                                                      //  pass to handler function   v.5.5
      CBfunc *myfunc = KBkeyCBfunc;
      myfunc();
      return 1;
   }

   if (func_busy) return 1;                                                //  let function handle it

   if (KBkey == GDK_Left) m_prev();                                        //  arrow keys  >>  prev/next image
   if (KBkey == GDK_Right) m_next();
   if (KBkey == GDK_plus) m_zoom("+");                                     //  +/- keys  >>  zoom in/out  v.5.0
   if (KBkey == GDK_equal) m_zoom("+");
   if (KBkey == GDK_minus) m_zoom("-");
   if (KBkey == GDK_Z) m_zoom("Z");                                        //  Z key: zoom in/out
   if (KBkey == GDK_z) m_zoom("Z");
   if (KBkey == GDK_R) m_rotate(+90);                                      //  keys L, R  >>  rotate
   if (KBkey == GDK_r) m_rotate(+90);
   if (KBkey == GDK_L) m_rotate(-90);
   if (KBkey == GDK_l) m_rotate(-90);
   if (KBkey == GDK_C) m_RGB();                                            //  show color data in status bar   v.5.6
   if (KBkey == GDK_c) m_RGB();
   if (KBkey == GDK_T) m_index();                                          //  key T  >>  thumbnail index
   if (KBkey == GDK_t) m_index();                                          
   if (KBkey == GDK_X) m_index();                                          //  key X  >>  thumbnail index
   if (KBkey == GDK_x) m_index();                                          
   if (KBkey == GDK_Escape) m_slideshow();                                 //  escape  >>  exit slideshow  v.5.0
   if (KBkey == GDK_Delete) m_trash();                                     //  delete  >>  trash
   return 1;
}


//  mouse event function - capture buttons and drag movements

void mouse_event(GtkWidget *, GdkEventButton *event, void *)
{
   static int     bdtime = 0, butime = 0;
   static int     fmouse = 0, fdrag = 0, mbusy = 0;

   if (mbusy) return;                                                      //  deplete pending mouse events   v.5.7
   mbusy++;
   zmainloop();
   mbusy = 0;
   
   if (event->type == GDK_BUTTON_PRESS) {
      bdtime = event->time;
      mpDx = int(event->x);                                                //  track mouse position
      mpDy = int(event->y);
      if (event->button == 1) LMdown = 1;                                  //  left mouse pressed
      fmouse++;
   }

   if (event->type == GDK_MOTION_NOTIFY) {
      mpDx = int(event->x);                                                //  mouse movement
      mpDy = int(event->y);
      fmouse++;
      if (LMdown) fdrag++;                                                 //  drag underway
   }
   
   if (fmouse) {                                                           //  new mouse position
      fmouse = 0;
      mpMx = mpDx - orgMx;                                                 //  corresp. pxbM position   v.5.0
      mpMy = mpDy - orgMy;
      if (mpMx < 0) mpMx = 0;                                              //  stay inside image
      if (mpMy < 0) mpMy = 0;
      if (mpMx > wwM-1) mpMx = wwM-1;
      if (mpMy > hhM-1) mpMy = hhM-1;
      mp3x = int(mpMx / Rscale + 0.5);                                     //  corresp. pxb3 position
      mp3y = int(mpMy / Rscale + 0.5);
   }

   if (event->type == GDK_BUTTON_PRESS) {
      clickMx = mpMx;                                                      //  capture click position   v.5.0
      clickMy = mpMy;                                                      //    at button down moment
      click3x = mp3x;
      click3y = mp3y;
   }

   if (event->type == GDK_BUTTON_RELEASE) {
      butime = event->time;
      LMdown = 0;                                                          //  cancel left mouse down status
      if (fdrag) {
         fdrag = 0;                                                        //  drag underway, end it
         mdMx1 = mdMy1 = mdMx2 = mdMy2 = 0;
         md3x1 = md3y1 = md3x2 = md3y2 = 0;
         clickMx = mpMx;                                                   //  capture mouse position   v.5.0
         clickMy = mpMy;                                                   //    at drag end moment
         click3x = mp3x;
         click3y = mp3y;
      }
      else {
         if (butime - bdtime < 400) {                                      //  ignore > 400 ms
            if (event->button == 1) LMclick++;                             //  left mouse click
            if (event->button == 3) RMclick++;                             //  right mouse click
         }
      }
   }
   
   if (fdrag) {
      if (! mdMx1) { 
         mdMx1 = clickMx;                                                  //  new drag start, pxbM space
         mdMy1 = clickMy; 
         md3x1 = click3x;                                                  //  corresp. pxb3 space
         md3y1 = click3y; 
      }
      mdMx2 = mpMx; mdMy2 = mpMy;                                          //  capture drag extent
      md3x2 = mp3x; md3y2 = mp3y;
   }
   
   if (mouseCBfunc) {                                                      //  pass to handler function  v.5.5
      CBfunc *myfunc = mouseCBfunc;
      myfunc();
   }

   if (LMclick && ! Mcapture) m_zoom("+");                                 //  if not captured by function,
   if (RMclick && ! Mcapture) m_zoom("-");                                 //  assume zoom intended  v.5.0

   return;
}


//  draw a dotted line. coordinates are in image3/pxb3 space.              //  v.5.0

void draw_dotline(int x1, int y1, int x2, int y2)
{
   x1 = int(Rscale * x1 + orgMx + 0.5);                                    //  convert to pxbM space
   y1 = int(Rscale * y1 + orgMy + 0.5);
   x2 = int(Rscale * x2 + orgMx + 0.5);
   y2 = int(Rscale * y2 + orgMy + 0.5);
   
   zlock();
   gdk_draw_line(GTK_LAYOUT(dWin)->bin_window,gdkgc,x1,y1,x2,y2);
   zunlock();
   return;
}


//  erase a dotted line. refresh line path from pxb3 pixels.               //  v.5.0

void erase_dotline(int x1, int y1, int x2, int y2)
{
   double   slope;
   int      pxm, pym, pxn, pyn, px3, py3;

   GOFUNC_DATA(1)
   
   x1 = int(Rscale * x1 + 0.5);                                            //  convert to pxbM space
   y1 = int(Rscale * y1 + 0.5);                                            //  (without orgMx, orgMy)
   x2 = int(Rscale * x2 + 0.5);
   y2 = int(Rscale * y2 + 0.5);
   
   if (abs(y2 - y1) > abs(x2 - x1)) {
      slope = 1.0 * (x2 - x1) / (y2 - y1);
      if (y2 > y1) {
         for (pym = y1; pym <= y2; pym++) {
            pxm = int(x1 + slope * (pym - y1) + 0.5);
            GOFUNC(draw_pixel);
         }
      }
      else {
         for (pym = y1; pym >= y2; pym--) {
            pxm = int(x1 + slope * (pym - y1) + 0.5);
            GOFUNC(draw_pixel);
         }
      }
   }
   else {
      slope = 1.0 * (y2 - y1) / (x2 - x1);
      if (x2 > x1) {
         for (pxm = x1; pxm <= x2; pxm++) {
            pym = int(y1 + slope * (pxm - x1) + 0.5);
            GOFUNC(draw_pixel);
         }
      }
      else {
         for (pxm = x1; pxm >= x2; pxm--) {
            pym = int(y1 + slope * (pxm - x1) + 0.5);
            GOFUNC(draw_pixel);
         }
      }
   }

   return;

draw_pixel:
   {
      zlock();

      for (int ii = -1; ii < 2; ii++)
      for (int jj = -1; jj < 2; jj++)
      {
         px3 = int((pxm+ii) / Rscale + 0.5);                               //  pxb3 space
         py3 = int((pym+jj) / Rscale + 0.5);
         if (px3 < 0 || px3 > ww3 - 1) continue;
         if (py3 < 0 || py3 > hh3 - 1) continue;
         
         pxn = pxm + ii + orgMx;                                           //  pxbM space
         pyn = pym + jj + orgMy;                                           //  remove hhD / wwD limit check   v.5.1
         
         pixbuf_draw(GTK_LAYOUT(dWin)->bin_window,0,pxb3,px3,py3,pxn,pyn,1,1);
      }

      zunlock();
      RETURN
   }
}


//  process main window menu and toolbar events

void menufunc(GtkWidget *, const char *menu)
{
   kill_func = 0;                                                          //  in case prior func killed   v.5.6

   if (strEqu(menu,ZTX("File"))) return;                                   //  ignore top-level menus
   if (strEqu(menu,ZTX("Tags"))) return;
   if (strEqu(menu,ZTX("Edit"))) return;
   if (strEqu(menu,ZTX("Help"))) return;

   if (strEqu(menu,ZTX("quit"))) { m_quit(); return; }                     //  functions that can run parallel
   if (strEqu(menu,ZTX("Quit"))) { m_quit(); return; }
   if (strEqu(menu,ZTX("zoom+"))) { m_zoom("+"); return; }
   if (strEqu(menu,ZTX("zoom-"))) { m_zoom("-"); return; }
   if (strEqu(menu,ZTX("kill"))) { m_kill(); return; }
   if (strEqu(menu,ZTX("Print"))) { m_print(); return; }
   if (strEqu(menu,ZTX("print"))) m_print();
   if (strEqu(menu,ZTX("Select Area"))) { m_select(); return; }
   if (strEqu(menu,ZTX("About"))) { m_help("about"); return; }
   if (strEqu(menu,ZTX("User Guide"))) { m_help("user guide"); return; }
   if (strEqu(menu,ZTX("README"))) { m_help("README"); return; }
   if (strEqu(menu,ZTX("Change Log"))) { m_help("changelog"); return; }
   if (strEqu(menu,ZTX("Error Log"))) { m_help("errorlog"); return; }
   if (strEqu(menu,ZTX("Translations"))) { m_help("translations"); return; }
   if (strEqu(menu,ZTX("Clone"))) { m_clone(); return; }

   if (func_busy || zdedit) {
      zmessageACK(ZTX("prior function still running"));                    //  functions that cannot run parallel
      return;
   }

   if (strEqu(menu,ZTX("Edit Parameters"))) m_parms();
   if (strEqu(menu,ZTX("Create Thumbnails"))) m_thumbs();
   if (strEqu(menu,ZTX("Check Monitor"))) m_montest();
   if (strEqu(menu,ZTX("Slide Show"))) m_slideshow();

   if (strEqu(menu,ZTX("Edit Tags"))) m_edit_tags();
   if (strEqu(menu,ZTX("Search Tags"))) m_search_tags();
   if (strEqu(menu,ZTX("Build Tags Index"))) m_build_tags_index();
   if (strEqu(menu,ZTX("View Exif Data"))) m_exif();

   if (strEqu(menu,ZTX("Brightness Distribution"))) m_flatten();
   if (strEqu(menu,ZTX("Brightness/Contrast/Color"))) m_tune();
   if (strEqu(menu,ZTX("Color Depth"))) m_color_dep();
   if (strEqu(menu,ZTX("Color Intensity"))) m_color_int();
   if (strEqu(menu,ZTX("RGB Spread"))) m_rgb_spread();
   if (strEqu(menu,ZTX("Sharpen"))) m_sharp();
   if (strEqu(menu,ZTX("Blur"))) m_blur();
   if (strEqu(menu,ZTX("Reduce Noise"))) m_denoise();
   if (strEqu(menu,ZTX("Red Eye"))) m_redeye();
   if (strEqu(menu,ZTX("Trim"))) m_trim();
   if (strEqu(menu,ZTX("Rotate"))) m_rotate(0);
   if (strEqu(menu,ZTX("Resize"))) m_resize();
   if (strEqu(menu,ZTX("Unbend"))) m_unbend();
   if (strEqu(menu,ZTX("Warp"))) m_warp();
   if (strEqu(menu,ZTX("HDR"))) m_HDR();
   if (strEqu(menu,ZTX("Panorama"))) m_pano();
   
   if (strEqu(menu,ZTX("index"))) m_index();
   if (strEqu(menu,ZTX("open"))) { m_open(null); return; }                 //  same translations   v.5.6
   if (strEqu(menu,ZTX("Open"))) m_open(null);
   if (strEqu(menu,ZTX("prev"))) m_prev();
   if (strEqu(menu,ZTX("next"))) m_next();
   if (strEqu(menu,ZTX("undo"))) m_undo();
   if (strEqu(menu,ZTX("redo"))) m_redo();
   if (strEqu(menu,ZTX("toolbar::save"))) { m_save(); return; }
   if (strEqu(menu,ZTX("Save"))) m_save();
   if (strEqu(menu,ZTX("Trash"))) { m_trash(); return; }
   if (strEqu(menu,ZTX("trash"))) m_trash();

   return;
}


//  main window delete_event and destroy signals

int delete_event()
{
   if (mod_keep()) return 1;                                               //  allow user bailout    v.5.1
   m_kill();                                                               //  tell function to stop
   save_fotoxx_state();                                                    //  save state data for next session
   fdestroy++;                                                             //  shutdown in progress
   return 0;
}

void destroy()
{
   printf("main window destroyed \n");
   gtk_main_quit();   
   return;
}


/**************************************************************************
      file functions
***************************************************************************/

//  display a thumbnail index of images in a separate window

void m_index()
{
   if (file1) image_xthumbs(file1,"paint1",0,m_open);                      //  v.5.1
   else {
      image_xthumbs(imagedirk,"init",0,m_open);
      image_xthumbs(0,"paint1");
   }
   return;
}


//  open a new image file - make images pxb1 (keep) and pxb3 (modified)

void m_open(char *filez)
{
   char           *newfile, *pp, wtitle[100];
   struct stat    f1stat;
   const char     *openmess = ZTX("open new image file");
   int            newindex = 0;
   
   if (mod_keep()) return;                                                 //  keep modifications
   if (file1) update_filetags(file1);                                      //  commit tag changes, if any  v.5.7

   if (filez) newfile = filez;                                             //  use passed file
   else {                                                                  //  or user-selected file
      if (file1) newfile = zgetfile(openmess,file1,"open");                //  assume same directory as prior 
      else newfile = zgetfile(openmess,imagedirk,"open");                  //  or curr. image directory
      newindex = 1;
   }
   if (! newfile) return;                                                  //  user cancel

   if (file1) zfree(file1);                                                //  set new file1
   file1 = newfile;
   
   pixbuf_free(pxb1);
   pxb1 = load_pixbuf(file1);                                              //  validate file, load image pxb1
   if (! pxb1) {
      zmessageACK(ZTX("file cannot be read: \n %s"),file1);                //  clean missing/non-working file
      zfree(file1);
      file1 = 0;                                                           //  v.34  bugfix
      return;
   }

   pixbuf_poop(1)                                                          //  get pxb1 attributes

   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(pxb1);                                               //  copy to image3 for editing
   pixbuf_test(3);
   pixbuf_poop(3);
   
   stat(file1,&f1stat);                                                    //  get file size
   f1size = f3size = f1stat.st_size;
   
   pp = image_xthumbs(file1,"find",0);
   if (! pp || newindex) {                                                 //  file not in current list   v.5.1
      image_xthumbs(file1,"init");                                         //  reset image file list
      image_xthumbs(0,"paint2");                                           //  refresh index window if active
   }
   if (pp) zfree(pp);
   
   strcpy(imagedirk,file1);                                                //  set new image dirk and file name
   pp = strrchr(imagedirk,'/');
   strncpy0(fname1,pp+1,99);
   strncpy0(fname3,pp+1,99);
   *pp = 0;                                                                //  v.5.5

   image_position(file1,f1posn,f1count);                                   //  position and count in image directory

   snprintf(wtitle,99,"%s  %d/%d  %s",fname1,f1posn,f1count,imagedirk);    //  window title           v.5.7
   if (strlen(wtitle) > 97) strcpy(wtitle+96,"...");                       //  filename  N/NN  /image/directory...
   gtk_window_set_title(GTK_WINDOW(mWin),wtitle);

   clearmem_image3();                                                      //  clear undo stack, mod status
   rotate_angle = 0;                                                       //  no net rotation
   select_area_clear();                                                    //  no selected area
   Mscale = 0;                                                             //  scale to window
   mwpaint2();                                                             //  repaint window
   gtk_window_present(GTK_WINDOW(mWin));                                   //  bring to foreground
   
   if (zdtags) edit_tags_dialog();                                         //  update tags dialog     v.5.1

   return;
}


//  save pxb3 (modified image3) to a file

void m_save()
{
   char           *newfile;
   int            err;
   struct stat    f3stat;
   const char     *title = ZTX("save image to a file");
   
   if (! pxb3) return;

   newfile = zgetfile(title,file1,"save","quality");                       //  get new file name from user
   if (! newfile) return;

   update_filetags(file1);                                                 //  commit tag changes, if any
   if (Fexiv2) exif_fetch(file1);                                          //  get exif data from original file

   gdk_pixbuf_save(pxb3,newfile,"jpeg",gerror,"quality",JPGquality,null);  //  save image3 to new or same file

   if (Fexiv2) {
      err = exif_stuff(file1,newfile,ww3,hh3);                             //  modify for new pixel dimensions
      if (err) zmessageACK(ZTX("unable to copy Exif data"));               //    and stuff exif data into new file
   }
   
   load_filetags(newfile);                                                 //  update assigned tags file  v.5.1
   update_asstags(newfile);

   if (strNeq(newfile,file1) && (samedirk(file1,newfile))) {               //  if new file in current directory,
      image_xthumbs(newfile,"init");                                       //  reset image file list
      image_xthumbs(0,"paint2");                                           //  refresh index window if active
      gtk_window_present(GTK_WINDOW(mWin));                                //  bring to foreground    v.5.2.2
   }

   if (strEqu(newfile,file1)) {                                            //  if same file replaced,      v.5.0
      stat(newfile,&f3stat);                                               //  update size in status bar
      f3size = f3stat.st_size;
      update_status_bar();
   }

   zfree(newfile);
   Fmod3 = 0;                                                              //  clear modified status after save
   return;
}


//  Delete image file - move file1 to trash                                v.43
//  Trash has no standard location, so use private trash folder

void m_trash()
{
   int            err, yn;
   char           command[1000], trashdir[100];
   struct stat    trstat;

   if (! file1) return;                                                    //  nothing to trash
   
   err = stat(file1,&trstat);                                              //  get file status
   if (err) {
      zmessLogACK(strerror(errno));
      return;
   }

   if (! (trstat.st_mode & S_IWUSR)) {                                     //  check permission  v.5.6
      yn = zmessageYN(ZTX("move read-only file to trash?"));
      if (! yn) return;
      trstat.st_mode |= S_IWUSR;
      chmod(file1,trstat.st_mode);
   }
   
   snprintf(trashdir,99,"%s/%s",getenv("HOME"),ftrash);                    //  get full fotoxx trash file
   
   trstat.st_mode = 0;
   err = stat(trashdir,&trstat);
   if (! S_ISDIR(trstat.st_mode)) {
      snprintf(command,999,"mkdir -m 0750 \"%s\"",trashdir);
      err = system(command);
      if (err) {
         zmessLogACK(ZTX("cannot create trash folder: %s"),wstrerror(err));
         return;
      }
   }

   snprintf(command,999,"cp \"%s\" \"%s\" ",file1,trashdir);               //  move image file to trash directory
   err = system(command);
   if (err) {
      zmessLogACK(ZTX("error: %s"),wstrerror(err));
      return;
   }

   snprintf(command,999,"rm \"%s\"",file1);                                //  delete image file
   err = system(command);
   if (err) {
      zmessLogACK(ZTX("error: %s"),wstrerror(err));
      return;
   }
   
   clearmem_image3();                                                      //  clear undo stack, mod counts
   update_asstags(file1,1);                                                //  delete in assigned tags file  v.5.1

   image_xthumbs(file1,"init");                                            //  reset image file list
   image_xthumbs(0,"paint2");                                              //  refresh index window if active
   m_next();                                                               //  step to next file if there

   return;
}


//  print image files

void m_print()                                                             //  v.5.2
{
   int      err;
   char     command[maxfcc];
   
   if (! file1) return;
   if (! Fprintoxx) {
      zmessageACK("printoxx program not found (see user guide)");
      return;
   }
   snprintf(command,maxfcc,"printoxx \"%s\" &",file1);                     //  send curr. file to printoxx
   err = system(command);
   if (err) zmessLogACK(wstrerror(err));
   return;
}


//  open previous image file in same directory as last file opened

void m_prev()
{
   char        *fileP;
   
   if (! file1) return;
   fileP = image_xthumbs(file1,"prev");
   if (image_file_type(fileP) == 2) m_open(fileP);                         //  v.4.8
   return;
}


//  open next image file in same directory as last file opened

void m_next()
{
   char        *fileN;
   
   if (! file1) return;
   fileN = image_xthumbs(file1,"next");
   if (! fileN) return;
   if (image_file_type(fileN) == 2) m_open(fileN);                         //  v.4.8
   return;
}


//  quit - exit program
//  give bailout chance if unsaved changes

void m_quit()
{
   if (mod_keep()) return;
   if (file1) update_filetags(file1);                                      //  commit tag changes, if any  v.5.7
   printf("quit \n");
   fdestroy++;                                                             //  stop window painting
   save_fotoxx_state();                                                    //  save state data for next session
   gtk_main_quit();                                                        //  gone forever
   return;
}


//  edit parameters and get their new values

void m_parms()
{
   int np = editParms(0,0);                                                //  edit parms
   if (! np) return;

   getparm(pixel_sample_size)                                              //  set new values
   getparm(jpg_save_quality)
   getparm(pano_lens_mm)
   getparm(pano_lens_bow)
   getparm(pano_prealign_size)
   getparm(pano_mouse_leverage)
   getparm(pano_align_size_increase)
   getparm(pano_blend_reduction)
   getparm(pano_minimum_blend)
   getparm(pano_image_stretch)
   return;
}


//  create thumbnail images for all image files under chosen directory

void m_thumbs()                                                            //  process entire tree   v.4.8
{
   GdkPixbuf   *pixbuf;
   char        *subdirk, *pp, stbartext[200];
   char        *filespec1, *filespec2, *thumbdirk;
   int         err, contx = 0, fcount = 0;

   pp = zgetfile(M_seltopdir,topdirk,"folder");                            //  select top image directory
   if (! pp) return;
   strcpy(topdirk,pp);
   zfree(pp);

   func_busy++;
   postmessage = zmessage_post(ZTX("computing"));

   while ((subdirk = command_output(contx,"find %s -type d",topdirk)))     //  get all subdirectories
   {
      pp = strrchr(subdirk,'/');
      if (pp && strEqu(pp,"/.thumbnails")) {                               //  do not process thumbnails
         zfree(subdirk);
         continue;
      }

      image_xthumbs(subdirk,"init");
      filespec1 = image_xthumbs(subdirk,"first");
      thumbdirk = 0;

      while (filespec1)                                                    //  get all image files in directory
      {
         zmainloop();                                                      //  check for user kill   v.5.6
         if (kill_func) goto thumbs_quit;

         if (image_file_type(filespec1) == 2) {
            if (! thumbdirk) {                                             //  when first image file found,
               thumbdirk = strdupz(subdirk,20);                            //    create .thumbnails if not already
               strcat(thumbdirk,"/.thumbnails");
               err = mkdir(thumbdirk,0751);
               if (err && errno != EEXIST) {
                  zmessage_kill(postmessage);
                  zmessLogACK(ZTX("cannot create %s \n %s"),thumbdirk,strerror(errno));
                  goto thumbs_quit;
               }
            }

            pixbuf = image_thumbnail(filespec1);                           //  create thumbnail file for image file
            if (pixbuf) g_object_unref(pixbuf);
            
            snprintf(stbartext,199,"%4d %s",++fcount,filespec1);           //  track progress on status bar
            stbar_message(stbar,stbartext);
         }
         
         filespec2 = image_xthumbs(filespec1,"next");
         zfree(filespec1);
         filespec1 = filespec2;
      }

      zfree(subdirk);
   }

thumbs_quit:   
   zmessage_kill(postmessage);                                             //  computation complete
   func_busy--;
   return;
}


//  monitor test function

void m_montest()                                                           //  v.40
{
   pixel       pix3;
   int         red, green, blue;
   int         row, col, row1, row2;
   int         ww = wwD, hh = hhD;
   
   if (mod_keep()) return;

   clearmem_image3();                                                      //  clear undo stack, mod counts

   if (file1) zfree(file1);                                                //  set no file active   v.5.0
   file1 = 0;
   pixbuf_free(pxb1);
   pixbuf_free(pxb3);

   pxb3 = pixbuf_new(colorspace,0,8,ww,hh);
   pixbuf_test(3);
   pixbuf_poop(3);
   nch = 3;

   gtk_window_set_title(GTK_WINDOW(mWin),"monitor check");

   for (red = 0; red <= 1; red++)
   for (green = 0; green <= 1; green++)
   for (blue = 0; blue <= 1; blue++)
   {
      row1 = 4 * red + 2 * green + blue;
      row1 = row1 * hh / 8;
      row2 = row1 + hh / 8;
      
      for (row = row1; row < row2; row++)
      for (col = 0; col < ww; col++)
      {
         pix3 = ppix3 + row * rs3 + col * nch;
         pix3[0] = red * 256 * col / ww;
         pix3[1] = green * 256 * col / ww;
         pix3[2] = blue * 256 * col / ww;
      }
   }

   Mscale = 0;                                                             //  scale to window
   mwpaint2();                                                             //  repaint window
   
   return;
}


//  enter or leave slideshow mode

void m_slideshow()                                                         //  v.41
{
   int            key;
   static int     Fmode = 0, ww, hh;
   
   key = KBkey;
   KBkey = 0;

   if (! Fmode) 
   {
      if (key == GDK_Escape) return;
      gtk_window_get_size(GTK_WINDOW(mWin),&ww,&hh);
      gtk_widget_hide_all(GTK_WIDGET(mmbar));                              //  enter slide show mode
      gtk_widget_hide_all(GTK_WIDGET(mtbar));
      gtk_widget_hide_all(GTK_WIDGET(stbar));
      gtk_window_fullscreen(GTK_WINDOW(mWin));
   }

   if (Fmode) 
   {
      gtk_window_unfullscreen(GTK_WINDOW(mWin));                           //  leave slide show mode
      gtk_window_resize(GTK_WINDOW(mWin),ww,hh);
      gtk_widget_show_all(GTK_WIDGET(mmbar));
      gtk_widget_show_all(GTK_WIDGET(mtbar));
      gtk_widget_show_all(GTK_WIDGET(stbar));
   }

   Fmode = 1 - Fmode;                                                      //  toggle mode
   Mscale = 0;                                                             //  v.5.0
   Fscale = Fmode;                                                         //  zoom-in small images  v.5.0
   mwpaint2(); 
   return;
}


//  start a new instance of fotoxx in parallel

void m_clone()
{
   char     command[200];
   int      ignore;
   
   if (blank_null(file1)) strcpy(command,"fotoxx &");
   else snprintf(command,199,"fotoxx \"%s\" &",file1);                     //  pass current file
   ignore = system(command);
   return;
}


/**************************************************************************
      help functions
***************************************************************************/

//  functions to display help files (some run parallel)

void m_help(const char *menu)                                              //  menu function
{
   if (strEqu(menu,"about")) 
      zmessageACK(" %s \n %s \n %s \n %s",fversion,flicense,fsource,ftranslators);

   if (strEqu(menu,"user guide")) 
      showz_helpfile();                                                    //  launch help file in new process  

   if (strEqu(menu,"README"))
      showz_readme();

   if (strEqu(menu,"changelog"))
      showz_changelog();

   if (strEqu(menu,"errorlog"))                                            //  v.5.1
      showz_errlog();

   if (strEqu(menu,"translations"))                                        //  v.5.7
      showz_translations();

   return;
}


/**************************************************************************
      tag functions
***************************************************************************/

void  edit_tags_fixup(cchar * widgetname);                                 //  fixup tag selection widgets
void  edit_tags_mouse(GtkTextView *, GdkEventButton *, cchar *);           //  select tag via mouse click
int   add_unique_tag(cchar *tag, char *taglist, int maxcc);                //  add tag if unique and enough space
void  add_new_filetag();                                                   //  add tags_atag to tags_filetags
void  delete_filetag();                                                    //  remove tags_atag from tags_filetags
void  add_new_recentag();                                                  //  add tags_atag to tags_recentags

char     tags_pdate[12] = "";                                              //  photo date, yyyymmdd
int      tags_pstars = 0;                                                  //  photo rating in "stars"   v.5.7
char     tags_atag[maxtag1] = "";                                          //  one tag
char     tags_filetags[maxtag2] = "";                                      //  tags for one file
char     tags_asstags[maxtag3] = "";                                       //  all assigned tags
char     tags_searchtags[maxtag4] = "";                                    //  image search tags
char     tags_recentags[maxtag5] = "";                                     //  recently added tags
int      tags_changed = 0;                                                 //  tags have been changed


//  edit tags menu function

void m_edit_tags()
{
   if (! Fexiv2) {                                                         //  exiv2 package is required
      zmessageACK(M_noexiv2);
      return;
   }
   
   edit_tags_dialog();
   return;
}


//  activate edit tags dialog, stuff data from current file

void edit_tags_dialog()
{
   int edit_tags_dialog_event(zdialog *zd, const char *event);
   int edit_tags_dialog_compl(zdialog *zd, int zstat);

   char     *ppv, pstarsN[12];

   if (! file1) return;

   if (! zdtags)                                                           //  (re) start tag edit dialog 
   {
      load_asstags();                                                      //  get all assigned tags

      zdtags = zdialog_new(ZTX("Edit Tags"),mWin,Bdone,Bcancel,0);         //  tag edit dialog

      zdialog_add_widget(zdtags,"hbox","hb1","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","labfile","hb1",ZTX("file:"),"space=10");
      zdialog_add_widget(zdtags,"label","file","hb1");

      zdialog_add_widget(zdtags,"hbox","hb2","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","lab21","hb2",ZTX("photo date (yyyymmdd)"),"space=10");
      zdialog_add_widget(zdtags,"entry","pdate","hb2",0,"scc=12");

      zdialog_add_widget(zdtags,"hbox","hb3","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"label","labstars","hb3",ZTX("photo stars"),"space=10");
      zdialog_add_widget(zdtags,"vbox","vb3","hb3");
      zdialog_add_widget(zdtags,"hbox","hb31","vb3",0,"homog");
      zdialog_add_widget(zdtags,"hbox","hb32","vb3",0,"homog");
      zdialog_add_widget(zdtags,"label","lab30","hb31","0");
      zdialog_add_widget(zdtags,"label","lab31","hb31","1");
      zdialog_add_widget(zdtags,"label","lab32","hb31","2");
      zdialog_add_widget(zdtags,"label","lab33","hb31","3");
      zdialog_add_widget(zdtags,"label","lab34","hb31","4");
      zdialog_add_widget(zdtags,"label","lab35","hb31","5");
      zdialog_add_widget(zdtags,"radio","pstars0","hb32",0);               //  v.5.7
      zdialog_add_widget(zdtags,"radio","pstars1","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars2","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars3","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars4","hb32",0);
      zdialog_add_widget(zdtags,"radio","pstars5","hb32",0);

      zdialog_add_widget(zdtags,"hbox","hb4","dialog","space=5");
      zdialog_add_widget(zdtags,"label","lab4","hb4",ZTX("current tags"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame4","hb4",0,"expand");
      zdialog_add_widget(zdtags,"edit","filetags","frame4",0,"expand");

      zdialog_add_widget(zdtags,"hbox","hb5","dialog","space=5");
      zdialog_add_widget(zdtags,"label","recent","hb5",ZTX("recently added"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame5","hb5",0,"expand");
      zdialog_add_widget(zdtags,"edit","recentags","frame5",0,"expand");

      zdialog_add_widget(zdtags,"hbox","hb6","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"button","add new tag","hb6",ZTX("create tag"),"space=10");
      zdialog_add_widget(zdtags,"entry","atag","hb6",0);

      zdialog_add_widget(zdtags,"hbox","hb7","dialog",0,"space=5");
      zdialog_add_widget(zdtags,"hbox","hb8","dialog");
      zdialog_add_widget(zdtags,"label","labasstags","hb8",ZTX("assigned tags"),"space=10");
      zdialog_add_widget(zdtags,"frame","frame8","dialog",0,"expand");
      zdialog_add_widget(zdtags,"edit","asstags","frame8",0,"expand");

      zdialog_resize(zdtags,400,300);
      zdialog_run(zdtags,edit_tags_dialog_event,edit_tags_dialog_compl);   //  start dialog
      
      edit_tags_fixup("filetags");                                         //  setup for mouse tag selection
      edit_tags_fixup("asstags");
      edit_tags_fixup("recentags");
   }

   load_filetags(file1);                                                   //  get file tags from exif data

   ppv = strrchr(file1,'/');
   zdialog_stuff(zdtags,"file",ppv+1);                                     //  stuff dialog file name

   zdialog_stuff(zdtags,"pdate",tags_pdate);                               //  stuff dialog data
   if (tags_pstars > 5) { tags_pstars = 5; tags_changed++; }
   sprintf(pstarsN,"pstars%d",tags_pstars);
   zdialog_stuff(zdtags,pstarsN,1);
   zdialog_stuff(zdtags,"filetags",tags_filetags);
   zdialog_stuff(zdtags,"asstags",tags_asstags);
   zdialog_stuff(zdtags,"recentags",tags_recentags);

   return;
}


//  setup tag display widget for tag selection using mouse clicks

void edit_tags_fixup(const char * widgetname)
{
   GtkWidget         *widget;
   GdkWindow         *gdkwin;

   widget = zdialog_widget(zdtags,widgetname);                             //  make widget wrap text
   gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(widget),GTK_WRAP_WORD);
   gtk_text_view_set_editable(GTK_TEXT_VIEW(widget),0);                    //  disable widget editing

   gdkwin = gtk_text_view_get_window(GTK_TEXT_VIEW(widget),textwin);       //  cursor for tag selection
   gdk_window_set_cursor(gdkwin,arrowcursor);

   gtk_widget_add_events(widget,GDK_BUTTON_PRESS_MASK);                    //  connect mouse-click event
   G_SIGNAL(widget,"button-press-event",edit_tags_mouse,widgetname)
}


//  edit tags mouse-click event function
//  get clicked tag and add to or remove from tags_filetags

void edit_tags_mouse(GtkTextView *widget, GdkEventButton *event, const char *widgetname)
{
   GtkTextIter    iter;
   int            mpx, mpy, tbx, tby, offset, cc;
   char           *ptext, *pp1, *pp2;

   if (event->type != GDK_BUTTON_PRESS) return;
   mpx = int(event->x);                                                    //  mouse click position
   mpy = int(event->y);

   gtk_text_view_window_to_buffer_coords(widget,GTK_TEXT_WINDOW_TEXT,mpx,mpy,&tbx,&tby);
   gtk_text_view_get_iter_at_location(widget,&iter,tbx,tby);
   offset = gtk_text_iter_get_offset(&iter);                               //  graphic position in widget text

   ptext = 0;   
   if (strEqu(widgetname,"filetags")) ptext = tags_filetags;               //  get corresponding text
   if (strEqu(widgetname,"asstags")) ptext = tags_asstags;
   if (strEqu(widgetname,"recentags")) ptext = tags_recentags;
   if (strEqu(widgetname,"asstags2")) ptext = tags_asstags;
   if (! ptext) return;

   pp1 = ptext + utf8_position(ptext,offset);                              //  graphic position to byte position
   if (! *pp1 || *pp1 == ' ') return;                                      //  reject ambiguity
   while (pp1 > ptext && *pp1 != ' ') pp1--;                               //  find preceeding delimiter
   if (*pp1 == ' ') pp1++;

   pp2 = strchr(pp1,' ');                                                  //  find delimiter following
   if (pp2) cc = pp2 - pp1;
   else cc = strlen(pp1);
   if (cc >= maxtag1) return;                                              //  reject tag too big
   strncpy0(tags_atag,pp1,cc+1);                                           //  tags_atag = selected tag
   
   if (strEqu(widgetname,"filetags")) {
      delete_filetag();                                                    //  remove tag from file tags
      zdialog_stuff(zdtags,"filetags",tags_filetags);                      //  update dialog widgets
   }
   
   if (strEqu(widgetname,"asstags")) {
      add_new_filetag();                                                   //  add assigned tag to file tags
      add_new_recentag();
      zdialog_stuff(zdtags,"filetags",tags_filetags);
   }

   if (strEqu(widgetname,"recentags")) {
      add_new_filetag();                                                   //  add recent tag to file tags
      zdialog_stuff(zdtags,"filetags",tags_filetags);
   }

   if (strEqu(widgetname,"asstags2")) {                                    //  search dialog:
      zdialog_fetch(zdtags,"searchtags",tags_searchtags,maxtag4);          //  add assigned tag to search tags
      strncatv(tags_searchtags,maxtag4," ",tags_atag,0);
      zdialog_stuff(zdtags,"searchtags",tags_searchtags);
   }

   return;
}


//  edit tags dialog event function

int edit_tags_dialog_event(zdialog *zd, const char *event)
{
   int      err;
   
   if (strEqu(event,"pdate")) {                                            //  photo date revised    v.5.6
      err = zdialog_fetch(zd,"pdate",tags_pdate,11);
      if (err) return 1;
      tags_changed++;
   }

   if (strnEqu(event,"pstars",6)) {                                        //  stars revised   v.5.7
      tags_pstars = event[6] - '0';
      tags_changed++;
   }

   if (strEqu(event,"add new tag")) {
      err = zdialog_fetch(zd,"atag",tags_atag,maxtag1);                    //  add new tag to file
      if (err) return 1;                                                   //  reject too big tag
      add_new_filetag();
      add_new_recentag();
      zdialog_stuff(zd,"filetags",tags_filetags);                          //  update dialog widgets
      zdialog_stuff(zd,"asstags",tags_asstags);
      zdialog_stuff(zd,"atag","");
   }

   return 0;
}


int edit_tags_dialog_compl(zdialog *zd, int zstat)                         //  v.5.6
{
   if (zstat == 1) update_filetags(file1);                                 //  done, update file Exif
   zdialog_free(zdtags);                                                   //  kill dialog
   zdtags = null;
   return 0;
}


//  add input tag to output tag list if not already there and enough room
//  returns:   0 = added OK     1 = not unique (case ignored)
//             2 = overflow     3 = bad utf8 characters

int add_unique_tag(const char *tag, char *taglist, int maxcc)
{
   char     *pp1, *pp2, *ppv, temptag1[maxtag1], temptag2[maxtag1];
   int      atcc, cc1, cc2;

   strncpy0(temptag1,tag,maxtag1);                                         //  remove leading and trailing blanks
   atcc = strTrim2(temptag2,temptag1);
   if (! atcc) return 0;
   if (strEqu(temptag2,"(null)")) return 0;                                //  ignore the null tag   v.5.6

   for (ppv = temptag2; *ppv; ppv++)                                       //  replace imbedded blanks with '_'
      if (*ppv == ' ') *ppv = '_';
   
   if (utf8_check(temptag2)) {                                             //  check for valid utf8 encoding
      printf("bad utf8 characters: %s \n",temptag2);
      return 3;
   }
   
   pp1 = taglist;
   cc1 = strlen(temptag2);

   while (true)                                                            //  check if already in tag list
   {
      while (*pp1 == ' ') pp1++;
      if (! *pp1) break;
      pp2 = pp1 + 1;
      while (*pp2 && *pp2 != ' ') pp2++;
      cc2 = pp2 - pp1;
      if (cc2 == cc1 && strncaseEqu(temptag2,pp1,cc1)) return 1;
      pp1 = pp2;
   }
   
   cc2 = strlen(taglist);                                                  //  append to tag list if space enough
   if (cc1 + cc2 + 1 >= maxcc) return 2;
   strcpy(taglist + cc2,temptag2);
   strcpy(taglist + cc2 + cc1," ");
   return 0;
}


//  image file Exif tags >> tags_pdate, tags_pstars, tags_filetags in memory

void load_filetags(char *file)
{
   char        *ppv;
   const char  *pp;
   int         ii, jj, cc;

   *tags_filetags = *tags_pdate = 0;
   tags_pstars = 0;

   ppv = get_exif_data(file1,exif_date_key_read);
   if (ppv) {                                                              //  get photo date from Exif data
      strncpy(tags_pdate,ppv,4);
      strncpy(tags_pdate+4,ppv+5,2);
      strncpy(tags_pdate+6,ppv+8,2);
      tags_pdate[8] = 0;
   }

   ppv = get_exif_data(file,exif_tags_key_read);                           //  get file tags from Exif data
   if (ppv) 
   {
      for (ii = 1; ; ii++)
      {
         pp = strField(ppv,' ',ii);                                        //  assume blank delimited tags
         if (! pp) break;
         cc = strlen(pp);
         if (cc >= maxtag1) continue;                                      //  reject tags too big
         for (jj = 0; jj < cc; jj++)
            if (pp[jj] > 0 && pp[jj] < ' ') break;                         //  reject tags with control characters
         if (jj < cc) continue;
         
         if (strnEqu(pp,"stars=",6)) {                                     //  "stars=N" tag    v.5.7
            convSI(pp+6,tags_pstars,0,99);
            continue;
         }

         strcpy(tags_atag,pp);                                             //  add to file tags if unique
         add_new_filetag();
      }

      zfree(ppv);
   }
   
   tags_changed = 0;
   return;
}


//  tags_pdate, tags_pstars, tags_filetags in memory >> image file Exif tags

void update_filetags(char *file)
{
   char     pdate2[12];

   if (! tags_changed) return;

   strncpy(pdate2,tags_pdate,4);                                           //  yyyymmdd >> yyyy:mm:dd
   strncpy(pdate2+5,tags_pdate+4,2);
   strncpy(pdate2+8,tags_pdate+6,2);
   pdate2[4] = pdate2[7] = ':';
   pdate2[10] = 0;
   
   if (tags_pstars > 0) {                                                  //  add "stars=N" tag   v.5.7
      sprintf(tags_atag,"stars=%d",tags_pstars);
      add_new_filetag();
   }

   set_exif_data(file,exif_date_key_write,pdate2);                         //  update Exif data
   set_exif_data(file,exif_tags_key_write,tags_filetags);
   
   update_asstags(file);                                                   //  update assigned tags file
   tags_changed = 0;
   return;
}


//  add new tag to file tags, if not already and enough space.

void add_new_filetag()
{
   int         err;

   err = add_unique_tag(tags_atag,tags_filetags,maxtag2);
   if (err == 2) { 
      zmessageACK(ZTX("file tags exceed %d characters"),maxtag2);
      return;
   }
   
   tags_changed++;
   return;
}


//  add new tag to recent tags, if not already.
//  remove oldest to make space if needed.

void add_new_recentag()
{
   int         err;
   char        *ppv, temp_recentags[maxtag5];

   if (strnEqu(tags_atag,"stars=",6)) return;                              //  omit this tag   v.5.7

   err = add_unique_tag(tags_atag,tags_recentags,maxtag5);                 //  add tag to recent tags

   while (err == 2)
   {
      strncpy0(temp_recentags,tags_recentags,maxtag5);                     //  remove oldest to make room
      ppv = temp_recentags;
      while (*ppv && *ppv == ' ') ppv++;
      while (*ppv && *ppv != ' ') ppv++;
      while (*ppv && *ppv == ' ') ppv++;
      strcpy(tags_recentags,ppv);
      err = add_unique_tag(tags_atag,tags_recentags,maxtag5);
   }

   zdialog_stuff(zdtags,"recentags",tags_recentags);                       //  update dialog
   return;
}


//  delete a tag from file tags, if present

void delete_filetag()
{
   int         ii, ftcc, atcc;
   char        temp_filetags[maxtag2];
   const char  *pp;
   
   strncpy0(temp_filetags,tags_filetags,maxtag2);
   *tags_filetags = 0;
   ftcc = 0;
   
   for (ii = 1; ; ii++)
   {
      pp = strField(temp_filetags,' ',ii);
      if (! pp) break;
      if (strcaseEqu(pp,tags_atag)) continue;
      atcc = strlen(pp);
      strcpy(tags_filetags + ftcc, pp);
      ftcc += atcc;
      tags_filetags[ftcc] = ' ';
      ftcc++;
      tags_filetags[ftcc] = 0;
   }

   tags_changed++;
   return;
}


//  load assigned tags file >> tags_asstags in memory
//  create list of all assigned tags with no duplicates

void load_asstags()
{
   FILE        *fid;
   int         ntags = 0, ntcc, atcc, ii, err;
   char        *ppv, buff[maxfcc];
   const char  *pp1;
   char        *tags[maxntags];
   
   ntcc = 0;
   *tags_asstags = 0;

   fid = fopen(asstagsfile,"r");
   if (! fid) return;                                                      //  no tags
   
   while (true)                                                            //  read assigned tags file
   {
      ppv = fgets_trim(buff,maxfcc-1,fid);
      if (! ppv) break;
      if (strnNeq(buff,"tags: ",6)) continue;

      for (ii = 1; ; ii++)                                                 //  add file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(buff+6,' ',ii);
         if (! pp1) break;
         if (strnEqu(pp1,"stars=",6)) continue;                            //  omit this tag    v.5.7
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) goto overflow;
      }
   }

   err = fclose(fid);
   if (err) goto tagsfileerr;
   
   for (ii = 1; ; ii++)                                                    //  build sort list
   {
      pp1 = strField(tags_asstags,' ',ii);
      if (! pp1) break;
      tags[ntags] = strdupz(pp1);
      ntags++;
      if (ntags == maxntags) goto toomanytags;
   }
   
   HeapSort(tags,ntags);                                                   //  sort alphabetically
   
   ntcc = 0;
   *tags_asstags = 0;

   for (ii = 0; ii < ntags; ii++)                                          //  build sorted assigned tags list
   {
      atcc = strlen(tags[ii]);
      if (ntcc + atcc + 1 > maxtag3) goto overflow;
      strcpy(tags_asstags + ntcc,tags[ii]);
      ntcc += atcc;
      tags_asstags[ntcc] = ' ';
      ntcc++;
      zfree(tags[ii]);
   }

   tags_asstags[ntcc] = 0;
   
   return;

overflow:
   zmessageACK(M_totalTagsExceed,maxtag3);                                 //  total tags exceed %d characters
   return;

toomanytags:
   zmessageACK(ZTX("too many tags: %d"),maxntags);
   return;

tagsfileerr:
   zmessLogACK(M_asstagsError,strerror(errno));                            //  assigned tags file error: %s
   return;
}


//  update tags_asstags in memory from tags_filetags
//  update assigned tags file (add or replace changed file and its tags)

void update_asstags(char *file, int del)
{
   char        *ppv, temp_asstagsfile[maxfcc], pdate2[12];
   char        datebuff[maxfcc], tagsbuff[maxfcc], filebuff[maxfcc];
   const char  *pp1;
   int         ii, ntcc, err;
   FILE        *fidr, *fidw;

   ntcc = strlen(tags_asstags);
   
   if (! del)                                                              //  unless deleted
   {
      for (ii = 1; ; ii++)                                                 //  add file tags to assigned tags
      {                                                                    //    unless already present
         pp1 = strField(tags_filetags,' ',ii);
         if (! pp1) break;
         if (strnEqu(pp1,"stars=",6)) continue;                            //  omit this tag    v.5.7
         
         err = add_unique_tag(pp1,tags_asstags,maxtag3);
         if (err == 2) {
            zmessageACK(M_totalTagsExceed,maxtag3);                        //  total tags exceed %d characters
            break;
         }
      }
   }

   strcpy(temp_asstagsfile,asstagsfile);                                   //  temp tag file
   strcat(temp_asstagsfile,"_temp");

   fidr = fopen(asstagsfile,"r");                                          //  read tag file
   
   fidw = fopen(temp_asstagsfile,"w");                                     //  write temp tag file
   if (! fidw) goto tagserror;

   if (fidr) {   
      while (true)                                                         //  copy assigned tags file to temp file,
      {                                                                    //    omitting this image file
         ppv = fgets_trim(datebuff,maxfcc,fidr);
         if (! ppv) break;
         if (strnNeq(datebuff,"date: ",6)) continue;                       //  date added    v.5.6

         ppv = fgets_trim(tagsbuff,maxfcc,fidr);
         if (! ppv) break;
         if (strnNeq(tagsbuff,"tags: ",6)) continue;

         ppv = fgets_trim(filebuff,maxfcc,fidr);
         if (! ppv) break;
         if (strnNeq(filebuff,"file: ",6)) continue;

         if (strEqu(filebuff+6,file)) continue;                            //  if my file, skip copy

         fprintf(fidw,"%s\n",datebuff);                                    //  copy to temp file
         fprintf(fidw,"%s\n",tagsbuff);
         fprintf(fidw,"%s\n\n",filebuff);
      }
   }
   
   if (! del)                                                              //  unless deleted, append 
   {                                                                       //    revised file data to temp file
      strncpy(pdate2,tags_pdate,4);
      strncpy(pdate2+5,tags_pdate+4,2);                                    //  yyyymmdd >> yyyy:mm:dd    v.5.6
      strncpy(pdate2+8,tags_pdate+6,2);
      pdate2[4] = pdate2[7] = ':';
      pdate2[10] = 0;
      err = fprintf(fidw,"date: %s\n",pdate2);                             //  photo date tag
      err = fprintf(fidw,"tags: %s\n",tags_filetags);                      //  file tags
      err = fprintf(fidw,"file: %s\n\n",file);                             //  filespec
      if (err <= 0) goto tagserror;
   }

   if (fidr) {
      err = fclose(fidr);
      if (err) goto tagserror;
   }

   err = fclose(fidw);
   if (err) goto tagserror;
   
   err = rename(temp_asstagsfile,asstagsfile);                             //  replace tag file with temp file
   if (err) goto tagserror;
   
   return;
   
tagserror:
   zmessLogACK(M_asstagsError,strerror(errno));                            //  assigned tags file error: %s
   return;
}


//  search image tags for matching images                                  //  v.5.1

char     searchDateFrom[12] = "";                                          //  image search date range   v.5.6
char     searchDateTo[12] = "";
int      searchStarsFrom = 0;                                              //  image search stars range  v.5.7
int      searchStarsTo = 0;

void m_search_tags()
{
   int search_tags_dialog_event(zdialog*, const char *event);
   int search_tags_dialog_compl(zdialog*, int zstat);

   if (zdtags) {
      zdialog_free(zdtags);
      zdtags = null;
   }
   
   zdtags = zdialog_new(ZTX("search images by tags"),mWin,Bsearch,Bcancel,0);

   zdialog_add_widget(zdtags,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdtags,"label","labdate","hb1",ZTX("date range (yyyymmdd)"));
   zdialog_add_widget(zdtags,"entry","datefrom","hb1",0,"scc=10");
   zdialog_add_widget(zdtags,"entry","dateto","hb1",0,"scc=10");

   zdialog_add_widget(zdtags,"hbox","hb2","dialog",0);
   zdialog_add_widget(zdtags,"label","labstars","hb2",ZTX("stars range"));
   zdialog_add_widget(zdtags,"entry","starsfrom","hb2",0,"scc=2|space=5");
   zdialog_add_widget(zdtags,"entry","starsto","hb2",0,"scc=2|space=5");

   zdialog_add_widget(zdtags,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdtags,"label","labsrch","hb3",ZTX("search tags"));
   zdialog_add_widget(zdtags,"entry","searchtags","hb3",0,"expand");
   zdialog_add_widget(zdtags,"button","clear","hb3",Bclear);

   zdialog_add_widget(zdtags,"hbox","hb4","dialog");
   zdialog_add_widget(zdtags,"radio","rmall","hb4",ZTX("match all tags"));
   zdialog_add_widget(zdtags,"label","lspace","hb4","","space=10");
   zdialog_add_widget(zdtags,"radio","rmany","hb4",ZTX("match any tag"));

   zdialog_add_widget(zdtags,"hbox","hb5","dialog",0,"space=10");
   zdialog_add_widget(zdtags,"hbox","hb6","dialog");
   zdialog_add_widget(zdtags,"label","labasstags","hb6",ZTX("assigned tags"));
   zdialog_add_widget(zdtags,"frame","frame6","dialog",0,"expand");
   zdialog_add_widget(zdtags,"edit","asstags2","frame6",0,"expand");

   zdialog_resize(zdtags,400,0);                                           //  start dialog
   zdialog_run(zdtags,search_tags_dialog_event,search_tags_dialog_compl);

   edit_tags_fixup("asstags2");                                            //  setup tag selection via mouse
   
   zdialog_stuff(zdtags,"datefrom",searchDateFrom);                        //  stuff previous date range    v.5.6
   zdialog_stuff(zdtags,"dateto",searchDateTo);
   zdialog_stuff(zdtags,"searchtags",tags_searchtags);                     //  stuff previous search tags
   zdialog_stuff(zdtags,"rmall",1);                                        //  default is match all tags    v.5.6
   zdialog_stuff(zdtags,"rmany",0);
   load_asstags();                                                         //  stuff assigned tags
   zdialog_stuff(zdtags,"asstags2",tags_asstags);
   
   return;
}

  
int search_tags_dialog_event(zdialog *zd, const char *event)               //  dialog event function
{
   if (strEqu(event,"clear")) zdialog_stuff(zd,"searchtags","");
   return 0;
}


int search_tags_dialog_compl(zdialog *zd, int zstat)                       //  dialog completion function
{
   const char  *pps, *ppf;
   char        resultsfile[200];
   char        *ppv, *file, rbuff[maxfcc];
   char        date1[12], date2[12];
   int         err, nfiles, iis, iif, stars, nmatch, nfail;
   int         date1cc, date2cc, Fmall, Fdates, Ftags, Fstars;
   FILE        *fidr, *fidw;
   struct stat statbuf;

   if (zstat == 1) {
      zdialog_fetch(zd,"datefrom",searchDateFrom,10);                      //  get search date range    v.5.6
      zdialog_fetch(zd,"dateto",searchDateTo,10);
      zdialog_fetch(zd,"starsfrom",searchStarsFrom);                       //  get search stars range   v.5.7
      zdialog_fetch(zd,"starsto",searchStarsTo);
      zdialog_fetch(zd,"searchtags",tags_searchtags,maxtag4);              //  get search tags
      zdialog_fetch(zd,"rmall",Fmall);                                     //  get match all/any option
   }
   
   zdialog_free(zdtags);                                                   //  kill dialog
   zdtags = null;
   if (zstat != 1) return 0;                                               //  cancelled

   *date1 = *date2 = 0;                                                    //  convert search dates to file format
   Fdates = date1cc = date2cc = 0;                                         //    yyyymmdd >> yyyy:mm:dd
   if (*searchDateFrom) {
      strncpy(date1,searchDateFrom,4);
      strncpy(date1+5,searchDateFrom+4,2);
      strncpy(date1+8,searchDateFrom+6,2);
      date1[4] = date1[7] = ':';
      date1cc = strlen(date1);
      Fdates++;                                                            //  from date given
   }
   if (*searchDateTo) {
      strncpy(date2,searchDateTo,4);
      strncpy(date2+5,searchDateTo+4,2);
      strncpy(date2+8,searchDateTo+6,2);
      date2[4] = date2[7] = ':';
      date2cc = strlen(date2);
      Fdates++;                                                            //  to date given
   }

   Fstars = 0;
   if (searchStarsFrom || searchStarsTo) Fstars = 1;                       //  stars given

   Ftags = 0;
   if (! blank_null(tags_searchtags)) Ftags = 1;                           //  search tags given

   if (! Ftags && ! Fdates && ! Fstars) {                                  //  no search criteria was given,
      strcpy(tags_searchtags,"(null)");                                    //    find images with no tags
      Ftags = 1;
   }
   
   snprintf(resultsfile,199,"%s/search_results",get_zuserdir());
   fidw = fopen(resultsfile,"w");                                          //  search results output file
   if (! fidw) goto writerror;

   fidr = fopen(asstagsfile,"r");                                          //  read assigned tags file
   if (! fidr) goto noasstags;
   
   nfiles = 0;                                                             //  count matching files found

   while (true)
   {
      ppv = fgets_trim(rbuff,maxfcc-1,fidr);                               //  next assigned tags record
      if (! ppv) break;
      if (! strnEqu(ppv,"date: ",6)) continue;                             //  date: yyyy:mm:dd 

      if (Fdates) {      
         if (strncmp(ppv+6,date1,date1cc) < 0) continue;                   //  check search date range
         if (strncmp(ppv+6,date2,date2cc) > 0) continue;
      }

      ppv = fgets_trim(rbuff,maxfcc-1,fidr);                               //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"tags: ",6)) continue;                             //  tags: xxxx xxxxx ...

      if (Ftags)
      {                                                                    //  tag search
         nmatch = nfail = 0;

         for (iis = 1; ; iis++)
         {
            pps = strField(tags_searchtags,' ',iis);                       //  step thru search tags
            if (! pps) break;

            for (iif = 1; ; iif++)                                         //  step thru file tags
            {
               ppf = strField(ppv+6,' ',iif);
               if (! ppf) { nfail++; break; }                              //  count matches and fails
               if (strcaseEqu(ppf,pps)) { nmatch++; break; }
            }
         }

         if (nmatch == 0) continue;                                        //  no match to any tag
         if (Fmall && nfail) continue;                                     //  no match to all tags
      }
      
      if (Fstars)
      {                                                                    //  stars search   v.5.7
         nfail = 0;

         for (iif = 1; ; iif++)                                            //  step thru file tags
         {
            ppf = strField(ppv+6,' ',iif);
            if (! ppf) { nfail++; break; }
            if (! strnEqu(ppf,"stars=",6)) continue;
            stars = atoi(ppf+6);
            if (stars < searchStarsFrom) nfail++;
            if (searchStarsTo && stars > searchStarsTo) nfail++;
            break;
         }
         
         if (nfail) continue;
      }

      ppv = fgets_trim(rbuff,maxfcc-1,fidr,1);                             //  next record
      if (! ppv) break;
      if (! strnEqu(ppv,"file: ",6)) continue;                             //  file: /dirks.../file.jpg
      file = ppv+6;
      err=stat(file,&statbuf);                                             //  check file exists      v.5.6
      if (err) continue;
      if (! S_ISREG(statbuf.st_mode)) continue;

      fprintf(fidw,"%s\n",file);                                           //  write matching file
      nfiles++;
   }

   fclose(fidr);
   
   err = fclose(fidw);
   if (err) goto writerror;
   
   if (! nfiles) {
      zmessageACK(ZTX("no matching images found"));
      return 0;
   }
   
   image_xthumbs(resultsfile,"initF",0,m_open);                            //  generate index of matching files
   file = image_xthumbs(0,"first",0);
   m_open(file);                                                           //  show first matching file
   image_xthumbs(0,"paint1");                                              //  show new thumbnail index window

   return 0;

noasstags:
   zmessageACK(ZTX("no assigned tags index file"));
   return 0;

writerror:
   zmessLogACK(ZTX("search results file error %s"),strerror(errno));
   return 0;
}


//  rebuild assigned tags file by reading all image Exif data              //  v.5.1

void m_build_tags_index()
{
   char        *subdirk, *ppv, stbartext[200];
   char        *filespec1, *filespec2;
   char        *photodate, *phototags;
   int         contx = 0, err, cc, fcount = 0;
   FILE        *fid;

   ppv = zgetfile(M_seltopdir,topdirk,"folder");                           //  select top image directory
   if (! ppv) return;
   strcpy(topdirk,ppv);
   zfree(ppv);

   func_busy++;

   fid = fopen(asstagsfile,"w");                                           //  open assigned tags file
   if (! fid) goto tags_error;

   postmessage = zmessage_post(ZTX("computing"));
   
   while ((subdirk = command_output(contx,"find %s -type d",topdirk)))     //  get all subdirectories
   {
      ppv = strrchr(subdirk,'/');
      if (ppv && strEqu(ppv,"/.thumbnails")) {                             //  ignore .thumbnails
         zfree(subdirk);
         continue;
      }

      image_xthumbs(subdirk,"init");                                       //  get all image files in directory
      filespec1 = image_xthumbs(subdirk,"first");

      while (filespec1)
      {
         zmainloop();
         if (kill_func) goto tags_quit;                                    //  check for user kill  v.5.6

         if (image_file_type(filespec1) == 2) 
         {
            photodate = get_exif_data(filespec1,exif_date_key_read);       //  output photo date      v.5.6
            if (photodate) photodate[10] = 0;                              //  (truncate to yyyy:mm:dd)
            fprintf(fid,"date: %s\n",photodate);
            phototags = get_exif_data(filespec1,exif_tags_key_read);       //  output photo tags
            fprintf(fid,"tags: %s\n",phototags);
            cc = fprintf(fid,"file: %s\n\n",filespec1);                    //  output image filespec
            if (! cc) goto tags_error;
            snprintf(stbartext,199,"%5d %s",++fcount,filespec1);           //  update status bar
            stbar_message(stbar,stbartext);
         }
         
         filespec2 = image_xthumbs(filespec1,"next");                      //  next image file
         zfree(filespec1);
         filespec1 = filespec2;
      }

      zfree(subdirk);
   }

tags_quit:   
   err = fclose(fid);                                                      //  close tag file
   if (err) goto tags_error;
   zmessage_kill(postmessage);                                             //  computation complete
   func_busy--;
   return;

tags_error:
   zmessage_kill(postmessage);                                             //  computation complete
   zmessLogACK(M_asstagsError,strerror(errno));                            //  assigned tags file error: %s
   func_busy--;
   return;
}


/**************************************************************************
   Exif data management
***************************************************************************/

//  show image Exif data if available

void m_exif()
{
   int      err;
   char     command[1000];

   if (! Fexiv2) {
      zmessageACK(M_noexiv2);                                              //  exiv2 package is required
      return;
   }

   if (! file1) return;
   snprintf(command,999,"exiv2 pr \"%s\" 2>&1",file1);
   err = popup_command(command,400,450);
   return;
}


//  get image Exif data, save in temp file filename.exv

int exif_fetch(char *file)
{
   char     command[maxfcc];
   int      err = 0;

   snprintf(command,maxfcc,"exiv2 -f ex \"%s\" ",file);
   err = system(command);
   return err;
}


//  stuff Exif data from original image file into new (edited) image file

int exif_stuff(char *file1, char *file3, int ww, int hh)
{
   void  modsuff(char *ofile, cchar *ifile, cchar *suff);                  //  private function

   char     efile1[maxfcc], efile3[maxfcc];
   char     command[2*maxfcc];
   int      ignore, err = 0;

   modsuff(efile1,file1,".exv");                                           //  get .exv file names from image files
   modsuff(efile3,file3,".exv");

   if (strNeq(efile1,efile3)) {                                            //  if images files are not the same,
      snprintf(command,1999,"mv -f \"%s\" \"%s\" ",efile1,efile3);         //    move and rename the .exv file
      err = system(command);
      if (err) return err;
   }
   
   snprintf(command,999,"exiv2 in \"%s\" ",file3);                         //  load Exif data into image file
   err = system(command);

   snprintf(command,999,"rm -f \"%s\" ",efile3);                           //  delete Exif data file
   ignore = system(command);
   if (err) return err;

   snprintf(command,999,"exiv2 -M \"set Exif.Photo.PixelXDimension %d\" "  //  modify Exif data in image file
                             " -M \"set Exif.Photo.PixelYDimension %d\" "  //    for new pixel dimensions
                             " \"%s\" ", ww, hh, file3);
   err = system(command);
   return err;
}

void modsuff(char *outfile, cchar *infile, cchar *suff)                    //  modify file suffix
{
   char     *pp;

   if (strlen(infile) > 990) zappcrash("file name overflow");
   strcpy(outfile,infile);
   pp = strrchr(outfile,'/');
   if (! pp) pp = outfile;
   pp = strrchr(pp,'.');
   if (pp) *pp = 0;
   strcat(outfile,suff);
   return;
}


//  set Exif metadata for given image file and key                         //  v.5.1

int set_exif_data(const char *file, const char *key, const char *text)
{
   char           command[maxfcc], *text2;
   int            err;
   
   snprintf(command,maxfcc-1,"exiv2 -M \"del %s\" \"%s\" ",key,file);      //  delete old value
   err = system(command);                                                  //  (old exiv2 needs this)   v.5.6

   if (! blank_null(text)) {                                               //  add new value if present
      text2 = strdupz(text,100);
      repl_1str(text,text2,"\"","\\\"");                                   //  if quotes present, escape them
      snprintf(command,maxfcc-1,"exiv2 -M \"set %s %s\" \"%s\" ",key,text2,file);
      zfree(text2);
   }
   
   err = system(command);
   return err;
}


//  get Exif metadata for given image file and key                         //  v.5.1
//  returned string belongs to caller, is subject for zfree()

char * get_exif_data(const char *file, const char *key)
{
   char           *commandout = 0, *pp;
   static char    *rettext;
   int            contx = 0, err;

   rettext = 0;

   while (true) 
   {
      commandout = command_output(contx,"exiv2 \"%s\" 2>/dev/null",file);             
      if (! commandout) break;
      pp = strstr(commandout,key);
      if (pp) {
         pp = pp + strlen(key);
         pp = strchr(pp,':');
         if (pp && strlen(pp) > 2) strdupz(pp+2,rettext);
      }
      zfree(commandout);
   }
   
   err = command_status(contx);
   return rettext;
}


/**************************************************************************
      image editing functions
***************************************************************************/

/**************************************************************************

   Select an area within the current image within which subsequent         //  overhauled  v.5.5
   edit functions will be carried out. If no selected area has been
   defined, edit functions will apply to the entire image.
   
   Edit zdialog receives event "blendwidth" if this widget is changed.
   
***************************************************************************/

int      sa_npg, sa_maxpg = 1000, sa_pgx[1000], sa_pgy[1000];              //  select area polygon points
int      sa_blend;                                                         //  select area blend width
int      sa_stat;                                                          //  select area status
int      sa_exist = 0;                                                     //  select area exists or not
int      sa_invert = 0;                                                    //  select area inverted or not

struct   sa_pix1 { int16  px, py, dist; };                                 //  list of pixels in select area
sa_pix1  *sa_pix = 0;
int      sa_npix;

void   select_mousefunc();                                                 //  select area mouse event handler
void   select_area_show();                                                 //  draw select area dotted-lines
int    select_dialog_event(zdialog*, const char *event);
int    select_dialog_compl(zdialog*, int zstat);

int    sa_mdown = 0;                                                       //  remember mouse down


//  dialog to outline an image area - may run parallel with edit functions

void m_select()
{
   const char  *title = ZTX("image area for following edits");
   const char  *helptext = ZTX("Use mouse drag and click to enclose an area.\n"
                                 "Use right click to undo the last line segment");

   if (! pxb3) return;                                                     //  no image
   if (zdsela) return;                                                     //  already active
   
   zdsela = zdialog_new(title,mWin,BOK,Bcancel,null);
   zdialog_add_widget(zdsela,"label","labhelp","dialog",helptext,"space=10");
   zdialog_add_widget(zdsela,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"button","start","hb1",Bstart);
   zdialog_add_widget(zdsela,"button","finish","hb1",Bfinish);
   zdialog_add_widget(zdsela,"button","show","hb1",Bshow);
   zdialog_add_widget(zdsela,"button","delete","hb1",Bdelete);
   zdialog_add_widget(zdsela,"button","invert","hb1",Binvert);
   zdialog_add_widget(zdsela,"hsep","hsep1","dialog");
   zdialog_add_widget(zdsela,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"label","labblend","hb2",Bblendwidth);
   zdialog_add_widget(zdsela,"hscale","blendwidth","hb2","0|200|1|0","expand");

   zdialog_run(zdsela,select_dialog_event,select_dialog_compl);            //  run dialog - parallel
   return;
}

int select_dialog_compl(zdialog *zd, int zstat)
{
   mouseCBfunc = 0;                                                        //  disconnect mouse function
   gdk_window_set_cursor(dWin->window,0);                                  //  restore normal cursor
   if (zstat != 1) select_area_clear();                                    //  cancel, remove select area
   if (sa_npg < 3) select_area_clear();                                    //  not enough points
   zdialog_free(zdsela);                                                   //  kill dialog
   zdsela = null;
   return 0;
}

int select_dialog_event(zdialog *zd, const char *event)
{
   zdialog_event * callbackfunc = 0;
   int sel_area_inside(int px, int py);                                    //  is point inside select area
   int sel_area_distance(int px, int py);                                  //  distance to nearest polygon edge

   const char  *delwarn = ZTX("delete existing area?");
   int         ii, yn;
   int         inside, px, py, dist;
   int         minpgx, minpgy, maxpgx, maxpgy;
   const char  *eventx = event;

   if (strEqu(eventx,"start")) 
   {
      if (sa_stat != 0) {
         yn = zmessageYN(delwarn);
         if (! yn) return 0;
      }
      select_area_clear();
      mouseCBfunc = select_mousefunc;                                      //  connect mouse function  v.5.5
      gdk_window_set_cursor(dWin->window,dragcursor);                      //  set drag cursor
      sa_mdown = 0;
      return 1;
   }

   if (strEqu(eventx,"show")) {
      mwpaint();
      select_area_show();
      return 1;
   }

   if (strEqu(eventx,"delete")) 
   {
      if (sa_npg > 20) {
         yn = zmessageYN(delwarn);                                         //  warn if significant work
         if (! yn) return 0;                                               //    will be deleted
      }
      select_area_clear();
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      gdk_window_set_cursor(dWin->window,0);                               //  restore normal cursor
      mwpaint();
      return 1;
   }
   
   if (strEqu(eventx,"invert") && sa_invert)                               //  invert selected area      v.5.7
   {
      zfree(sa_pix);                                                       //  if already inverted, free memory
      sa_pix = 0;                                                          //    and handle as "finish"
      sa_invert = 0;
      eventx = "finish";
   }
         
   if (strEqu(eventx,"finish")) 
   {
      mouseCBfunc = 0;                                                     //  disconnect mouse function
      gdk_window_set_cursor(dWin->window,0);                               //  restore normal cursor

      if (sa_npg < 3) {
         select_area_clear();                                              //  not enough points
         return 0;
      }

      ii = sa_npg - 1;
      if (sa_pgx[ii] != sa_pgx[0] || sa_pgy[ii] != sa_pgy[0]) {            //  if not connected already
         ii++;                                                             //  make the connection    bugfix  5.4.1
         sa_pgx[ii] = sa_pgx[0];                                           //  last line end = 1st point
         sa_pgy[ii] = sa_pgy[0];
      }

      sa_npg = ii;                                                         //  line count
      sa_pgx[ii+1] = sa_pgy[ii+1] = 0;                                     //  last point + 1 = zeros

      postmessage = zmessage_post(ZTX("computing"));                       //  must wait a while   v.5.2.2

      minpgx = maxpgx = sa_pgx[0];
      minpgy = maxpgy = sa_pgy[0];

      for (ii = 0; ii < sa_npg; ii++)                                      //  get rectangle enclosing polygon
      {
         if (sa_pgx[ii] < minpgx) minpgx = sa_pgx[ii];
         if (sa_pgx[ii] > maxpgx) maxpgx = sa_pgx[ii];
         if (sa_pgy[ii] < minpgy) minpgy = sa_pgy[ii];
         if (sa_pgy[ii] > maxpgy) maxpgy = sa_pgy[ii];
      }
      
      ii = (maxpgx - minpgx) * (maxpgy - minpgy);                          //  allocate enough memory for 
      sa_pix = (sa_pix1 *) zmalloc(ii * sizeof(sa_pix1));                  //    all pixels in rectangle

      sa_npix = 0;

      for (px = minpgx; px < maxpgx; px++)
      for (py = minpgy; py < maxpgy; py++)                                 //  loop all pixels in rectangle
      {
         zmainloop(1000);
         inside = sel_area_inside(px,py);                                  //  check if pixel inside polygon
         if (! inside) continue;
         dist = sel_area_distance(px,py);                                  //  distance to nearest edge
         sa_pix[sa_npix].px = px;
         sa_pix[sa_npix].py = py;                                          //  save pixel and distance
         sa_pix[sa_npix].dist = dist;
         sa_npix++;
      }

      zmessage_kill(postmessage);                                          //  computation complete

      sa_stat = 3;                                                         //  area select finished
      sa_exist = 1;
      select_area_show();
      return 1;
   }

   if (strEqu(eventx,"blendwidth"))                                        //  blend width changed 
   {
      zdialog_fetch(zd,"blendwidth",sa_blend);
      if (zdedit && zdedit->sentinel == zdsentinel && zdedit->eventCB) {   //  image edit dialog is active
         callbackfunc = (zdialog_event *) zdedit->eventCB;
         callbackfunc(zdedit,event);                                       //  notify dialog
      }
      return 1;
   }
   
   if (strEqu(eventx,"invert"))                                            //  invert selected area   v.5.7
   {
      if (! sa_exist) {
         zmessageACK("complete area select process");
         return 0;
      }

      postmessage = zmessage_post(ZTX("computing"));                       //  must wait a while   v.5.2.2

      ii = hh3 * ww3;                                                      //  allocate enough memory for 
      sa_pix = (sa_pix1 *) zmalloc(ii * sizeof(sa_pix1));                  //    all pixels in image

      sa_npix = 0;

      for (py = 0; py < hh3; py++)
      for (px = 0; px < ww3; px++)
      {
         zmainloop(1000);
         inside = sel_area_inside(px,py);                                  //  exclude pixels inside polygon
         if (inside) continue;
         dist = sel_area_distance(px,py);                                  //  distance to nearest edge
         sa_pix[sa_npix].px = px;
         sa_pix[sa_npix].py = py;                                          //  save pixel and distance
         sa_pix[sa_npix].dist = dist;
         sa_npix++;
      }

      zmessage_kill(postmessage);                                          //  computation complete

      sa_invert = 1;
      select_area_show();
      return 1;
   }

   return 0;
}


//  select area mouse function - draw lines, compute enclosed pixels

void select_mousefunc()
{
   int         ii, jj;
   
   if (mp3x < 0 || mp3x > ww3-1 || mp3y < 0 || mp3y > hh3-1)               //  ignore if outside image area
      return;

   if (LMdown)                                                             //  left mouse down
   {
      sa_mdown = 1;                                                        //  remember mouse is down

      if (md3x1 == 0 && md3y1 == 0) return;                                //  ignore unless drag started
      if (md3x1 == md3x2 && md3y1 == md3y2) return;                        //  no progress
      if (md3x2 + md3y2 == 0) return;                                      //  went off image        v.5.2.2

      if (sa_npg == 0) {                                                   //  set 1st point if none
         sa_pgx[0] = md3x1;
         sa_pgy[0] = md3y1;
         sa_npg = 1;
         sa_pgx[1] = sa_pgy[1] = -1;                                       //  no drag line
      }
      else {
         ii = sa_npg - 1;                                                  //  drag underway
         jj = sa_npg;
         if (sa_pgx[jj] != -1)                                             //  erase old drag line
            erase_dotline(sa_pgx[ii],sa_pgy[ii],sa_pgx[jj],sa_pgy[jj]);
         sa_pgx[jj] = md3x2;                                               //  next point = curr. mouse posn.
         sa_pgy[jj] = md3y2;
         draw_dotline(sa_pgx[ii],sa_pgy[ii],sa_pgx[jj],sa_pgy[jj]);        //  draw new drag line
      }
   }

   if ((! LMdown && sa_mdown) || LMclick)                                  //  mouse up after down, or left click
   {
      LMclick = RMclick = sa_mdown = 0;                                    //  reset clicks, mouse down   v.5.1

      if (sa_npg > sa_maxpg-2) {
         zmessageACK(ZTX("too many points"));
         return;
      }

      sa_pgx[sa_npg] = click3x;                                            //  position at click or drag end
      sa_pgy[sa_npg] = click3y;

      sa_npg++;                                                            //  new point added
      
      if (sa_npg > 1) {
         ii = sa_npg - 1;                                                  //  eliminate duplication
         jj = sa_npg - 2;
         if (sa_pgx[ii] == sa_pgx[jj] && sa_pgy[ii] == sa_pgy[jj]) 
            sa_npg--;
      }

      if (sa_npg > 1) {
         ii = sa_npg - 2;                                                  //  draw new last line
         jj = sa_npg - 1;
         draw_dotline(sa_pgx[ii],sa_pgy[ii],sa_pgx[jj],sa_pgy[jj]);
      }
      
      ii = sa_npg;
      sa_pgx[ii] = sa_pgy[ii] = -1;                                        //  no drag line
   }

   if (RMclick)                                                            //  right mouse click 
   {
      RMclick = LMclick = sa_mdown = 0;                                    //  reset clicks, mouse down   v.5.1

      ii = sa_npg - 2;
      jj = sa_npg - 1;
      if (sa_npg > 1)                                                      //  erase last line
         erase_dotline(sa_pgx[ii],sa_pgy[ii],sa_pgx[jj],sa_pgy[jj]);

      sa_npg--;                                                            //  reduce line count
      if (sa_npg <= 1) sa_npg = 0;

      ii = sa_npg;
      sa_pgx[ii] = sa_pgy[ii] = -1;                                        //  no drag line
   }
      
   if (Fredraw) select_area_show();                                        //  draw all lines
   Fredraw = 0;                                                            //  v.5.1

   return;
}


//  clear selected image area, free memory

void select_area_clear()
{
   sa_npg = sa_npix = sa_stat = sa_exist = sa_invert = sa_blend = 0;
   if (sa_pix) zfree(sa_pix);
   sa_pix = 0;
   mwpaint2();
   return;
}


//  draw all dotted lines around selected area

void select_area_show()
{
   int         ii, jj;

   for (ii = 0; ii < sa_npg; ii++)
   {
      jj = ii + 1;
      if (sa_pgx[jj] == -1) break;
      draw_dotline(sa_pgx[ii],sa_pgy[ii],sa_pgx[jj],sa_pgy[jj]);           //  last or dragging line
   }
   
   return;
}


//  determine if pixel at (px,py) is inside the area selection polygon

int sel_area_inside(int px, int py)
{
   int sel_area_inside2(int px, int py, int nth);
   int      ii, inside = 0;

   for (ii = 0; ii < sa_npg; ii++)                                         //  count times a vertical line up from
      if (sel_area_inside2(px,py,ii) == 2) inside = 1 - inside;            //    (px,py) intersects polygon face

   return inside;
}

int sel_area_inside2(int px, int py, int nth)                              //  determine if point (px,py) is
{                                                                          //    above or below nth polygon face
   int      x1, y1, x2, y2;
   double   yp;
   
   x1 = sa_pgx[nth];                                                       //  line segment of nth face
   y1 = sa_pgy[nth];
   x2 = sa_pgx[nth+1];
   y2 = sa_pgy[nth+1];
   
   if (x1 == x2) return 0;                                                 //  line is vertical

   if (x1 < x2) {
      if (px <= x1) return 0;                                              //  point is left or right of line
      if (px > x2) return 0;
   }
   else { 
      if (px <= x2) return 0;
      if (px > x1) return 0;
   }

   if (py <= y1 && py <= y2) return 1;                                     //  point is above or below both ends
   if (py >= y1 && py >= y2) return 2;
   
   yp = (px*(y2-y1) + (x2*y1-x1*y2))/(1.0*(x2-x1));                        //  intersect of vertical line x = px
   if (py < yp) return 1;                                                  //  above
   if (py > yp) return 2;                                                  //  below
   return 0;                                                               //  dead on
}


//  get distance from pixel (px,py) to nearest polygon face

int sel_area_distance(int px, int py)
{
   int sel_area_distance2(int px, int py, int nth);
   int      ii, dd, mindd;
   
   mindd = sel_area_distance2(px,py,0);                                    //  distance^2 to 0th polygon face

   for (ii = 1; ii < sa_npg; ii++)
   {
      dd = sel_area_distance2(px,py,ii);                                   //  distance^2 to Nth polygon face
      if (dd < mindd) mindd = dd;                                          //  remember smallest distance^2
   }
   
   mindd = int(sqrt(mindd));                                               //  return smallest distance
   return  mindd;
}

int sel_area_distance2(int px, int py, int nth)                            //  get distance^2 from point (px,py)
{                                                                          //    to nth polygon face
   int      x1, y1, x2, y2;
   int      a, b, c, d;
   int      h1, h2, ls, temp;
   int      dist2;
   int      huge = 1000000;                                                //  distance = 1000

   x1 = sa_pgx[nth];                                                       //  line segment of nth face
   y1 = sa_pgy[nth];
   x2 = sa_pgx[nth+1];
   y2 = sa_pgy[nth+1];
   
   if (x1 < 5 && x2 < 5) return huge;                                      //  v.37
   if (y1 < 5 && y2 < 5) return huge;                                      //  if line segment hugs image edge,
   if (ww1 - x1 < 5 && ww1 - x2 < 5) return huge;                          //    then it is logically distant
   if (hh1 - y1 < 5 && hh1 - y2 < 5) return huge;
   
   a = y1-y2;                                                              //  get min. distance ^2 from point
   b = x2-x1;                                                              //    to entire line of line segment
   c = x1*y2 - y1*x2;                                                      //  (intersect may be outside segment)
   d = (a*px + b*py + c);
   dist2 = int(1.0 * d*d / (a*a + b*b));                                   //  (can overflow 32 bits)
   
   h1 = (x1-px)*(x1-px) + (y1-py)*(y1-py);                                 //  point to (x1,y1) ^2
   h2 = (x2-px)*(x2-px) + (y2-py)*(y2-py);                                 //  point to (x2,y2) ^2
   ls = (x1-x2)*(x1-x2) + (y1-y2)*(y1-y2);                                 //  segment length ^2
   if (h1 > h2) { temp = h2; h2 = h1; h1 = temp; }                         //  re-order, h1 <= h2
   
   if (h1 < 0 || dist2 < 0) {
      printf("px py nth: %d %d %d \n",px,py,nth);                          //  algebra failure     ***
      printf("x1 y1 x2 y2: %d %d %d %d \n",x1,y1,x2,y2);
      printf("h1 dist2: %d %d \n",h1,dist2);
      printf("sel_area_distance2() failure \n");
      return 0;
   }

   if (h2 - dist2 < ls) return dist2;                                      //  intersect within segment, use dist2
   else return h1;                                                         //  distance ^2 to nearest end point
}


/**************************************************************************/

//  flatten brightness distribution             v.4.9

double   flatten_value = 0;                                                //  flatten value, 0 - 100%

int    flatten_dialog_event(zdialog* zd, const char *event);
int    flatten_dialog_compl(zdialog* zd, int zstat);

GtkWidget   *distgraph = 0;                                                //  brightness distribution graph
GtkWidget   *drdistgraph = 0;
int         dist_maxbin = 0;

void flatten_func();
void distgraph_create();
void distgraph_paint();
void distgraph_destroy();


void m_flatten()
{
   const char  *title = ZTX("Flatten Brightness Distribution");

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   flatten_value = 0;

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);                    //  create flatten distribution dialog
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space1","hb1",0,"space=10");
   zdialog_add_widget(zdedit,"label","labfd","hb1",ZTX("Flatten"));
   zdialog_add_widget(zdedit,"hscale","flatten","hb1","0|100|1|0","expand");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space3","hb3",0,"space=10");
   zdialog_add_widget(zdedit,"button","undo","hb3",Bundo);
   zdialog_add_widget(zdedit,"button","redo","hb3",Bredo);
   zdialog_add_widget(zdedit,"button","graph","hb3",Bgraph);               //  v.5.4

   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,flatten_dialog_event,flatten_dialog_compl);          //  run dialog - parallel

   return;
}


int flatten_dialog_event(zdialog *zd, const char *event)                   //  flatten dialog event function
{
   if (strEqu(event,"graph")) {                                            //  graph brightness distribution  v.5.4
      distgraph_create();
      return 0;
   }

   zdialog_fetch(zd,"flatten",flatten_value);                              //  slider value, 0 - 100

   if (strEqu(event,"undo")){                                              //  undo button
      prior_image3();
      distgraph_paint();
      return 0;
   }

   flatten_func();                                                         //  flatten image
   distgraph_paint();                                                      //  update distribution graph  v.5.4

   return 1;
}


int flatten_dialog_compl(zdialog *zd, int zstat)                           //  flatten dialog completion function
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}


void flatten_func()                                                        //  flatten brightness distribution
{
   int         px, py, ii, dist;
   double      bright1, bright3, brdist[2560];
   double      red1, green1, blue1, red3, green3, blue3;
   double      fold = 1, fnew = 0, dold, dnew, cmax;
   pixel       pix1, pix3;
   
   GOFUNC_DATA(1)
   
   for (ii = 0; ii < 2560; ii++)                                           //  clear brightness distribution data
      brdist[ii] = 0;

   fnew = 0.01 * flatten_value;                                            //  0.0 - 1.0  how much to flatten
   fold = 1.0 - fnew;                                                      //  1.0 - 0.0  how much to retain

   if (! sa_exist)                                                         //  no select area, process whole image
   {
      for (py = 0; py < hh1; py++)                                         //  compute brightness distribution
      for (px = 0; px < ww1; px++)
      {
         pix1 = ppix1 + py * rs1 + px * nch;
         bright1 = brightness(pix1);
         brdist[int(10*bright1)]++;
      }
      
      for (ii = 1; ii < 2560; ii++)                                        //  cumulative brightness distribution
         brdist[ii] += brdist[ii-1];                                       //   0 ... (ww1 * hh1)

      for (ii = 0; ii < 2560; ii++)
         brdist[ii] = brdist[ii] / (ww1 * hh1) * 2560.0 / (ii + 1);        //  multiplier per brightness level
         
      dist = sa_blend = 0;

      for (py = 0; py < hh1; py++)                                         //  flatten brightness distribution
      for (px = 0; px < ww1; px++)
         GOFUNC(flatfunc)
   }

   if (sa_exist)                                                           //  process selected area
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;                                               //  compute brightness distribution
         py = sa_pix[ii].py;
         pix1 = ppix1 + py * rs1 + px * nch;
         bright1 = brightness(pix1);
         brdist[int(10*bright1)]++;
      }
      
      for (ii = 1; ii < 2560; ii++)                                        //  cumulative brightness distribution
         brdist[ii] += brdist[ii-1];                                       //   0 ... sa_npix

      for (ii = 0; ii < 2560; ii++)
         brdist[ii] = brdist[ii] / sa_npix * 2560.0 / (ii + 1);            //  multiplier per brightness level
         
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;                                               //  flatten brightness distribution
         py = sa_pix[ii].py;
         dist = sa_pix[ii].dist;
         GOFUNC(flatfunc)
      }
   }

   mwpaint2();                                                             //  update window

   if (fnew > 0) Fmod3 = 1;                                                //  track modified status
   return;


//  process one pixel: flatten and blend

flatfunc:
   {
      zmainloop(1000);

      pix1 = ppix1 + py * rs1 + px * nch;                                  //  input pixel
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright1 = brightness(pix1);                                          //  input brightness
      bright3 = brdist[int(10*bright1)];                                   //  output brightness adjustment

      red3 = bright3 * red1;                                               //  flattened brightness
      green3 = bright3 * green1;
      blue3 = bright3 * blue1;

      red3 = fnew * red3 + fold * red1;                                    //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      if (dist < sa_blend) {                                               //  blend over distance sa_blend
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                         //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.0) {
         cmax = 255.0 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = int(red3 + 0.5);
      pix3[1] = int(green3 + 0.5);
      pix3[2] = int(blue3 + 0.5);
      
      RETURN
   }
}


//  draw a brightness distribution graph     v.5.4

void distgraph_create()
{
   dist_maxbin = 0;                                                        //  force rescale

   if (distgraph) {
      distgraph_paint();                                                   //  update existing graph
      return;
   }

   distgraph = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_window_set_title(GTK_WINDOW(distgraph),ZTX("brightness distribution"));
   gtk_window_set_transient_for(GTK_WINDOW(distgraph),GTK_WINDOW(mWin));
   gtk_window_set_default_size(GTK_WINDOW(distgraph),300,200);
   gtk_window_set_position(GTK_WINDOW(distgraph),GTK_WIN_POS_MOUSE);
   
   drdistgraph = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(distgraph),drdistgraph);

   G_SIGNAL(distgraph,"destroy",distgraph_destroy,0)
   G_SIGNAL(drdistgraph,"expose-event",distgraph_paint,0)

   gtk_widget_show_all(distgraph);
   
   return;
}

void distgraph_paint()
{
   int         brdist[20], nbins = 20;
   int         px, py, ii;
   int         winww, winhh;
   int         ww, hh, orgx, orgy;
   pixel       pix3;
   double      bright;

   if (! distgraph) return;

   for (ii = 0; ii < nbins; ii++)                                          //  clear brightness distribution
      brdist[ii] = 0;

   if (! sa_exist)                                                         //  no select area, process whole image
   {
      for (py = 0; py < hh1; py++)                                         //  compute brightness distribution
      for (px = 0; px < ww1; px++)
      {
         pix3 = ppix3 + py * rs1 + px * nch;
         bright = brightness(pix3);                                        //  0 to 255
         brdist[int(bright / 256.0 * nbins)]++;                            //  0 to nbins
      }
   }

   if (sa_exist)                                                           //  process selected area
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;                                               //  compute brightness distribution
         py = sa_pix[ii].py;
         pix3 = ppix3 + py * rs1 + px * nch;                               //  bugfix  v.5.4.1
         bright = brightness(pix3);
         brdist[int(bright / 256.0 * nbins)]++;
      }
   }

   zlock();   
   gdk_window_clear(drdistgraph->window);
   zunlock();

   winww = drdistgraph->allocation.width;
   winhh = drdistgraph->allocation.height;
   
   for (ii = 0; ii < nbins; ii++)
      if (brdist[ii] > dist_maxbin) dist_maxbin = brdist[ii];

   for (ii = 0; ii < nbins; ii++)
   {
      ww = winww / nbins;
      hh = int(0.9 * winhh * brdist[ii] / dist_maxbin);
      orgx = ii * ww;
      orgy = winhh - hh;
      zlock();
      gdk_draw_rectangle(drdistgraph->window,gdkgc,1,orgx,orgy,ww,hh);
      zunlock();
   }
   
   return;
}

void distgraph_destroy()
{
   if (distgraph) gtk_widget_destroy(distgraph);
   distgraph = 0;
   return;
}


/**************************************************************************/

//  tune - adjust image brightness distribution and color 

int    tune_dialog_event(zdialog* zd, const char *event);
int    tune_dialog_compl(zdialog* zd, int zstat);
int    tune_func();

double   tuneBR[9], tuneWH[9], tuneRGB[3];                                 //  tune dialog widget values


void m_tune()
{
   int         ii;
   const char  *title = ZTX("adjust brightness / whiteness / color");

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   for (ii = 0; ii < 9; ii++) tuneBR[ii] = tuneWH[ii] = 0;                 //  initialize
   tuneRGB[0] = tuneRGB[1] = tuneRGB[2] = 0;

   zdedit = zdialog_new(title,mWin,Bdone,Bcancel,null);                    //  create tune dialog

   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","space1","hb1",0,"space=70");
   zdialog_add_widget(zdedit,"label","labdark","hb1",ZTX("darker areas"));
   zdialog_add_widget(zdedit,"label","expand1","hb1",0,"expand");
   zdialog_add_widget(zdedit,"label","labbright","hb1",ZTX("brighter areas"));
   
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"expand|space=8");          //  brightness buttons
   zdialog_add_widget(zdedit,"vbox","vb2","hb2",0,"space=8");
   zdialog_add_widget(zdedit,"label","bright","vb2",Bbrightness);
   zdialog_add_widget(zdedit,"hbox","hb21","vb2");
   zdialog_add_widget(zdedit,"vbox","vb22","hb21",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb23","hb21",0,"space=5");
   zdialog_add_widget(zdedit,"button","br +++","vb22","+++");
   zdialog_add_widget(zdedit,"button","br ---","vb23"," - - - ");
   zdialog_add_widget(zdedit,"button","br +-", "vb22"," +  - ");
   zdialog_add_widget(zdedit,"button","br -+", "vb23"," -  + ");
   zdialog_add_widget(zdedit,"button","br +-+","vb22"," + - + ");
   zdialog_add_widget(zdedit,"button","br -+-","vb23"," - + - ");

   zdialog_add_widget(zdedit,"vscale","brval0","hb2","-10|10|0.1|0","expand");   //  brightness sliders
   zdialog_add_widget(zdedit,"vscale","brval1","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval2","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval3","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval4","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval5","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval6","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval7","hb2","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","brval8","hb2","-10|10|0.1|0","expand");

   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"expand|space=8");          //  whiteness buttons
   zdialog_add_widget(zdedit,"vbox","vb3","hb3",0,"space=8");
   zdialog_add_widget(zdedit,"label","white","vb3",Bwhiteness);
   zdialog_add_widget(zdedit,"hbox","hb31","vb3");
   zdialog_add_widget(zdedit,"vbox","vb32","hb31",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb33","hb31",0,"space=5");
   zdialog_add_widget(zdedit,"button","wh +++","vb32","+++");
   zdialog_add_widget(zdedit,"button","wh ---","vb33"," - - - ");
   zdialog_add_widget(zdedit,"button","wh +-", "vb32"," +  - ");
   zdialog_add_widget(zdedit,"button","wh -+", "vb33"," -  + ");
   zdialog_add_widget(zdedit,"button","wh +-+","vb32"," + - + ");
   zdialog_add_widget(zdedit,"button","wh -+-","vb33"," - + - ");

   zdialog_add_widget(zdedit,"vscale","whval0","hb3","-10|10|0.1|0","expand");   //  whiteness sliders
   zdialog_add_widget(zdedit,"vscale","whval1","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval2","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval3","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval4","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval5","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval6","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval7","hb3","-10|10|0.1|0","expand");
   zdialog_add_widget(zdedit,"vscale","whval8","hb3","-10|10|0.1|0","expand");

   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=10");                //  RGB spin buttons
   zdialog_add_widget(zdedit,"label","labred","hb4",ZTX("  adjust red"));
   zdialog_add_widget(zdedit,"spin","spred","hb4","-10|10|0.1|0");
   zdialog_add_widget(zdedit,"label","labgreen","hb4",ZTX("  green"));
   zdialog_add_widget(zdedit,"spin","spgreen","hb4","-10|10|0.1|0");
   zdialog_add_widget(zdedit,"label","labblue","hb4",ZTX("  blue"));
   zdialog_add_widget(zdedit,"spin","spblue","hb4","-10|10|0.1|0");

   zdialog_add_widget(zdedit,"hbox","hb6","dialog",0,"space=10");                //  undo, redo, reset buttons
   zdialog_add_widget(zdedit,"label","space6","hb6",0,"space=10");
   zdialog_add_widget(zdedit,"button","undo","hb6",Bundo);
   zdialog_add_widget(zdedit,"button","redo","hb6",Bredo);
   zdialog_add_widget(zdedit,"button","reset","hb6",Breset);
   
   zdialog_resize(zdedit,400,0);
   zdialog_run(zdedit,tune_dialog_event,tune_dialog_compl);                //  run dialog - parallel
   return;
}


int tune_dialog_compl(zdialog *zd, int zstat)                              //  tune dialog completion function
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}


int tune_dialog_event(zdialog *zd, const char *event)                      //  tune dialog event function
{
   int         ii;
   char        vsname[20];
   double      wvalue, brinc[9], whinc[9], pi = 3.14159;

   if (strEqu(event,"reset")) {                                            //  reset button
      for (ii = 0; ii < 9; ii++) 
      {
         strcpy(vsname,"brval0");                                          //  set all sliders back home
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_stuff(zd,vsname,0);
      }

      zdialog_stuff(zd,"spred",0);                                         //  reset RGB spin buttons
      zdialog_stuff(zd,"spgreen",0);
      zdialog_stuff(zd,"spblue",0);
   }

   if (strEqu(event,"undo") || strEqu(event,"reset")) {                    //  undo or reset button
      for (ii = 0; ii < 9; ii++) 
         tuneBR[ii] = tuneWH[ii] = 0;                                      //  reset all adjustments
      tuneRGB[0] = tuneRGB[1] = tuneRGB[2] = 0;
      prior_image3();                                                      //  restore prior image3
      return 1;
   }

   for (ii = 0; ii < 9; ii++)                                              //  clear all increments
      brinc[ii] = whinc[ii] = 0;

   if (strEqu(event,"br +++"))                                             //  convenience buttons "+++" etc.
      for (ii = 0; ii < 9; ii++)                                           //  get increments for all sliders
         brinc[ii] = 0.5;
   if (strEqu(event,"br ---")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5;
   if (strEqu(event,"br +-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br -+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"br +-+")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"br -+-")) 
      for (ii = 0; ii < 9; ii++) 
         brinc[ii] = 0.5 * sin(pi*ii/8.0);

   if (strEqu(event,"wh +++")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5;
   if (strEqu(event,"wh ---")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5;
   if (strEqu(event,"wh +-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh -+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * cos(pi*ii/8.0);
   if (strEqu(event,"wh +-+")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = -0.5 * sin(pi*ii/8.0);
   if (strEqu(event,"wh -+-")) 
      for (ii = 0; ii < 9; ii++) 
         whinc[ii] = 0.5 * sin(pi*ii/8.0);

   for (ii = 0; ii < 9; ii++)                                              //  add increments to sliders
   {
      if (brinc[ii]) {
         strcpy(vsname,"brval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= brinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }

      if (whinc[ii]) {
         strcpy(vsname,"whval0");
         vsname[5] = '0' + ii;
         zdialog_fetch(zd,vsname,wvalue);
         wvalue -= whinc[ii];
         if (wvalue < -10) wvalue = -10;
         if (wvalue > 10) wvalue = 10;
         zdialog_stuff(zd,vsname,wvalue);
      }
   }

   for (ii = 0; ii < 9; ii++)                                              //  get all slider settings
   {
      strcpy(vsname,"brval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneBR[ii] = -wvalue;                                                //  set bright values, -10 to +10

      strcpy(vsname,"whval0");
      vsname[5] = '0' + ii;
      zdialog_fetch(zd,vsname,wvalue);
      if (wvalue < -10) wvalue = -10;
      if (wvalue > 10) wvalue = 10;
      tuneWH[ii] = -wvalue;                                                //  set white values, -10 to +10
   }

   zdialog_fetch(zd,"spred",tuneRGB[0]);                                   //  red/green/blue, -10 to +10
   zdialog_fetch(zd,"spgreen",tuneRGB[1]);
   zdialog_fetch(zd,"spblue",tuneRGB[2]);

   tune_func();                                                            //  compute pixel updates
   Fmod3 = 1;                                                              //  image3 modified
   return 1;
}


int tune_func()                                                            //  modify image based on dialog inputs
{
   void   tune_pixel(pixel pix1, pixel pix2, int dist);                    //  tune one pixel in image

   int         ii, px, py, dist;
   pixel       pix1, pix3;
   
   if (sa_exist)                                                           //  process selected area
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         dist = sa_pix[ii].dist;
         pix1 = ppix1 + py * rs1 + px * nch;                               //  input pixel
         pix3 = ppix3 + py * rs3 + px * nch;                               //  output pixel
         tune_pixel(pix1,pix3,dist);                                       //  modify
      }
   }

   if (! sa_exist)                                                         //  process whole image
   {
      for (py = 0; py < hh1; py++)                                         //  loop all pixels
      for (px = 0; px < ww1; px++)
      {
         pix1 = ppix1 + py * rs1 + px * nch;                               //  input pixel
         pix3 = ppix3 + py * rs3 + px * nch;                               //  output pixel
         tune_pixel(pix1,pix3,0);                                          //  modify
      }
   }

   mwpaint2();                                                             //  update window
   return 0;
}

void tune_pixel(pixel pix1, pixel pix3, int dist)                          //  modify one pixel according to
{                                                                          //    brightness & whiteness setpoints
   int         ii;
   double      red, green, blue, white;
   double      brin, brin8, brvalx, whvalx, brout;
   double      f1, f2;
   
   zmainloop(1000);
   
   red = pix1[0];                                                          //  input pixel
   green = pix1[1];
   blue = pix1[2];
   
   brin = red;                                                             //  get brightest color
   if (green > brin) brin = green;
   if (blue > brin) brin = blue;
   
   brin = 0.003891 * (brin + 1);                                           //  0.00389 to 0.9961   tweak v.5.5

   brin8 = 8.0 * brin;                                                     //  brightness zone 0..8
   ii = int(brin8);

   if (ii < 8)                                                             //  interpolate brightness value
      brvalx = tuneBR[ii] + (tuneBR[ii+1] - tuneBR[ii]) * (brin8 - ii);
   else brvalx = tuneBR[8];
   brvalx = brvalx * 0.1;                                                  //  normalize -1..+1

   if (ii < 8)                                                             //  interpolate whiteness value
      whvalx = tuneWH[ii] + (tuneWH[ii+1] - tuneWH[ii]) * (brin8 - ii);
   else whvalx = tuneWH[8];
   whvalx = whvalx * 0.1;                                                  //  normalize -1..+1

   white = red;                                                            //  white content
   if (green < white) white = green;                                       //   = min(red,green,blue)
   if (blue < white) white = blue;
   white = white * whvalx;                                                 //  -100% to +100%
   red += white;
   green += white;
   blue += white;
   
   brout = brin + brvalx * sqrt(0.5 - fabs(brin - 0.5));                   //  convert input brightness
   if (brout > 0) brout = brout / brin;                                    //    to output brightness
   else brout = 0;
   if (brout * brin > 1.0) brout = 1.0 / brin;                             //  stop overflows
   red = red * brout + 0.49;                                               //  apply to all colors
   green = green * brout + 0.49;
   blue = blue * brout + 0.49;

   if (tuneRGB[0]) red += red * tuneRGB[0] / 20;                           //  get RGB adjustments
   if (tuneRGB[1]) green += green * tuneRGB[1] / 20;                       //  -10 to +10 >> -50% to +50%
   if (tuneRGB[2]) blue += blue * tuneRGB[2] / 20;

   if (red > 255) red = 255;                                               //  stop overflows
   if (green > 255) green = 255;
   if (blue > 255) blue = 255;
   
   if (sa_exist && dist < sa_blend)                                        //  if area selection, blend pixel
   {                                                                       //    changes over distance sa_blend
      f1 = 1.0 * dist / sa_blend;
      f2 = 1.0 - f1;
      red = f1 * red + f2 * pix1[0];
      green = f1 * green + f2 * pix1[1];
      blue = f1 * blue + f2 * pix1[2];
   }

   pix3[0] = int(red);                                                     //  output pixel
   pix3[1] = int(green);
   pix3[2] = int(blue);

   return;
}


/**************************************************************************/

//  image color-depth reduction

int    color_dep_dialog_event(zdialog *zd, const char *event);
int    color_dep_dialog_compl(zdialog *zd, int zstat);
int    color_dep_image();

int      color_depth;


void m_color_dep()
{
   const char  *colmess = ZTX(" Set color depth to 1-8 bits per color. \n"
                                " Press [apply] to update the image.");

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("set color depth"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",colmess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"spin","colors","hb1","1|8|1|8","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb1",Bapply,"space=5");

   zdialog_run(zdedit,color_dep_dialog_event,color_dep_dialog_compl);      //  run dialog
   return;
}


//  colors dialog event and completion callback functions

int color_dep_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   mwpaint2(); 
   return 0;
}

int color_dep_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"apply")) 
   {
      zdialog_fetch(zd,"colors",color_depth);
      postmessage = zmessage_post(ZTX("computing"));                       //  wait a while   v.5.5
      color_dep_image();                                                   //  process image
      zmessage_kill(postmessage);                                          //  computation complete
      Fmod3 = 1;                                                           //  image3 modified
      mwpaint2(); 
   }
   return 1;
}


//  image color depth function

int color_dep_image()
{
   int         ii, px, py, rgb, dist;
   int         mask[8] = { 128, 192, 224, 240, 248, 252, 254, 255 };
   int         mask1, mask2, val1, val2;
   double      f1, f2;
   pixel       pix1, pix3;
   
   mask1 = mask[color_depth - 1];
   mask2 = 128 >> color_depth;

   if (sa_exist)                                                           //  process select area
   {
      for (ii = 0; ii < sa_npix; ii++)
      {
         zmainloop(1000);
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         dist = sa_pix[ii].dist;
         pix1 = ppix1 + py * rs1 + px * nch;                               //  input pixel
         pix3 = ppix3 + py * rs3 + px * nch;                               //  output pixel

         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            val2 = val1 + mask2;
            if (val2 > 255) val2 = 255;
            val2 = val2 & mask1;
            if (dist < sa_blend) {                                         //  if area selection, blend pixel
               f1 = 1.0 * dist / sa_blend;                                 //    changes over distance sa_blend
               f2 = 1.0 - f1;                                              //       v.5.6
               val2 = int(f1 * val2 + f2 * val1);
            }
            pix3[rgb] = val2;
         }
      }
   }

   if (! sa_exist)                                                         //  process entire image3
   {
      for (px = 0; px < ww3; px++)
      for (py = 0; py < hh3; py++)
      {
         zmainloop(1000);
         pix1 = ppix1 + py * rs1 + px * nch;
         pix3 = ppix3 + py * rs3 + px * nch;
         
         for (rgb = 0; rgb < 3; rgb++)
         {
            val1 = pix1[rgb];
            val2 = val1 + mask2;
            if (val2 > 255) val2 = 255;
            val2 = val2 & mask1;
            pix3[rgb] = val2;
         }
      }
   }

   return 1;
}


/**************************************************************************/

//  image color-intensity adjustment
//
//  dialog color value input:
//        0 = no color (grey scale)  
//       50 = normal (initial color of unmodified image)
//      100 = full color
//
//  49 >> 0: decrease RGB spread until all are the same
//  51 >> 100: increase RGB values by same factor until biggest = 255

int   color_int_dialog_event(zdialog *zd, const char *event);
int   color_int_dialog_compl(zdialog *zd, int zstat);
int   color_int_func();

int      color_int_value;


void m_color_int()
{
   const char  *satmessage = ZTX("   left: no color (grey) \n"
                                   " middle: normal (unmodified) \n"
                                   "  right: maximum color");
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("set color intensity"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",satmessage,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space1","hb1",0,"space=10");
   zdialog_add_widget(zdedit,"label","labint","hb1",ZTX("color intensity"));
   zdialog_add_widget(zdedit,"hscale","color_int","hb1","0|100|1|50","expand");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space3","hb3",0,"space=10");
   zdialog_add_widget(zdedit,"button","undo","hb3",Bundo);
   zdialog_add_widget(zdedit,"button","redo","hb3",Bredo);

   color_int_value = 50;
   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,color_int_dialog_event,color_int_dialog_compl);      //  run dialog - parallel
   return;
}


//  color_int dialog event and completion callback functions

int color_int_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}

int color_int_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"undo")) {                                             //  undo button
      prior_image3();
      return 0;
   }

   zdialog_fetch(zd,"color_int",color_int_value);                          //  updated slider value
   color_int_func();
   Fmod3 = 1;                                                              //  image3 modified
   return 1;
}


//  image color intensity function

int color_int_func()
{
   int         ii, px, py, dist;
   int         red50, green50, blue50;
   int         red100, green100, blue100;
   int         rgb0, max50, min50;
   int         red, green, blue;
   double      color, bright, ramper, f1, f2;
   pixel       pix1, pix3;

   GOFUNC_DATA(1)
   
   
   if (sa_exist)                                                           //  process select area
   {
      for (ii = 0; ii < sa_npix; ii++)
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         dist = sa_pix[ii].dist;
         GOFUNC(do1pixel)
      }
   }
   
   if (! sa_exist)
   {
      dist = 0;
      for (px = 0; px < ww3; px++)                                         //  loop all image3 pixels
      for (py = 0; py < hh3; py++)
         GOFUNC(do1pixel)
   }

   mwpaint2();                                                             //  repaint
   return 0;

do1pixel:
   {
      zmainloop(1000);

      pix1 = ppix1 + py * rs1 + px * nch;                                  //  source pixel in image1
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3

      red50 = pix1[0];                                                     //  50%  color values (normal)
      green50 = pix1[1];
      blue50 = pix1[2];
      
      rgb0 = (red50 + green50 + blue50) / 3;                               //  0%  color values (grey scale)
         
      max50 = min50 = red50;
      if (green50 > max50) max50 = green50;                                //  get max/min normal color values
      else if (green50 < min50) min50 = green50;
      if (blue50 > max50) max50 = blue50;
      else if (blue50 < min50) min50 = blue50;
      
      color = (max50 - min50) * 1.0 / (max50 + 1);                         //  gray .. color       0 .. 1
      color = sqrt(color);                                                 //  accelerated curve   0 .. 1
      bright = max50 / 255.0;                                              //  dark .. bright      0 .. 1
      bright = sqrt(bright);                                               //  accelerated curve   0 .. 1
      ramper = 1 - color + bright * color;
      ramper = 1.0 / ramper;

      red100 = int(red50 * ramper);                                        //  100%  color values (max)
      green100 = int(green50 * ramper);
      blue100 = int(blue50 * ramper);
      
      if (color_int_value < 50) 
      {
         red = rgb0 + (color_int_value) * (red50 - rgb0) / 50;             //  compute new color value
         green = rgb0 + (color_int_value) * (green50 - rgb0) / 50;
         blue = rgb0 + (color_int_value) * (blue50 - rgb0) / 50;
      }
      else 
      {
         red = red50 + (color_int_value - 50) * (red100 - red50) / 50;
         green = green50 + (color_int_value - 50) * (green100 - green50) / 50;
         blue = blue50 + (color_int_value - 50) * (blue100 - blue50) / 50;
      }
      
      if (sa_exist && dist < sa_blend) {                                   //  if area selection, blend pixel
         f1 = 1.0 * dist / sa_blend;                                       //    changes over distance sa_blend
         f2 = 1.0 - f1;                                                    //       v.5.6
         red = int(f1 * red + f2 * red50);
         green = int(f1 * green + f2 * green50);
         blue = int(f1 * blue + f2 * blue50);
      }
      
      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
      
      RETURN
   }
}


/**************************************************************************/

//  image RGB spread adjustment                                            //  new in v.5.6
//
//  dialog RGB spread input:
//        0 = no RGB spread (gray scale)
//       50 = normal (initial unmodified RGB)
//      100 = max. RGB spread
//
//  50 >> 0: decrease RGB spread until all are the same
//  50 >> 100: increase RGB spread until one color is 0 or 255
//  in both cases, the average of RGB is not changed

int   rgb_spread_dialog_event(zdialog *zd, const char *event);
int   rgb_spread_dialog_compl(zdialog *zd, int zstat);
int   rgb_spread_func();

int    rgb_spread_value;


void m_rgb_spread()
{
   const char  *rgbmessage = ZTX("   left: no RGB spread (grey) \n"
                                   " middle: normal (unmodified) \n"
                                   "  right: maximum RGB spread");
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("set RGB spread"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",rgbmessage,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space1","hb1",0,"space=10");
   zdialog_add_widget(zdedit,"label","labrgb","hb1",ZTX("RGB spread"));
   zdialog_add_widget(zdedit,"hscale","rgb_spread","hb1","0|100|1|50","expand");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","space3","hb3",0,"space=10");
   zdialog_add_widget(zdedit,"button","undo","hb3",Bundo);
   zdialog_add_widget(zdedit,"button","redo","hb3",Bredo);

   rgb_spread_value = 50;
   zdialog_resize(zdedit,300,0);
   zdialog_run(zdedit,rgb_spread_dialog_event,rgb_spread_dialog_compl);    //  run dialog - parallel
   return;
}


//  rgb_spread dialog event and completion callback functions

int rgb_spread_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}

int rgb_spread_dialog_event(zdialog * zd, const char *event)
{
   if (strEqu(event,"undo")) {                                             //  undo button
      prior_image3();
      return 0;
   }

   zdialog_fetch(zd,"rgb_spread",rgb_spread_value);                        //  update slider value
   rgb_spread_func();
   Fmod3 = 1;                                                              //  image3 modified
   return 1;
}


//  RGB spread function

int rgb_spread_func()
{
   int         ii, px, py, dist;
   int         rgb0, red50, green50, blue50;
   int         red100, green100, blue100;
   int         red, green, blue;
   int         rinc, ginc, binc;
   double      scale, spread, spread1, spread2, f1, f2;
   pixel       pix1, pix3;
   
   GOFUNC_DATA(1)

   spread = rgb_spread_value - 50;                                         //  -50  to  0  to  50
   spread1 = 0.02 * spread + 1;                                            //    0  to  1  to   2
   spread2 = spread1 - 1.0;                                                //   -1  to  0  to   1
   
   if (sa_exist)                                                           //  process select area
   {
      for (ii = 0; ii < sa_npix; ii++)
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         dist = sa_pix[ii].dist;
         GOFUNC(do1pixel)
      }
   }
   
   if (! sa_exist)
   {
      dist = 0;
      for (px = 0; px < ww3; px++)                                         //  loop all image3 pixels
      for (py = 0; py < hh3; py++)
         GOFUNC(do1pixel)
   }
   
   mwpaint2();                                                             //  repaint
   return 0;

do1pixel:
   {
      zmainloop(1000);

      pix1 = ppix1 + py * rs1 + px * nch;                                  //  source pixel in image1
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3

      red50 = pix1[0];                                                     //  50%  color values (normal)
      green50 = pix1[1];
      blue50 = pix1[2];
      
      rgb0 = (red50 + green50 + blue50 + 1) / 3;

      rinc = red50 - rgb0;
      ginc = green50 - rgb0;
      binc = blue50 - rgb0;
      
      scale = 1.0;

   again:
      rinc = int(scale * rinc);
      ginc = int(scale * ginc);
      binc = int(scale * binc);

      red100 = red50 + rinc;
      green100 = green50 + ginc;
      blue100 = blue50 + binc;

      if (red100 > 255) { scale = (255.0 - red50) / rinc;  goto again; }
      if (red100 < 0) { scale = -1.0 * red50 / rinc; goto again; }
      if (green100 > 255) { scale = (255.0 - green50) / ginc; goto again; }
      if (green100 < 0) { scale = -1.0 * green50 / ginc; goto again; }
      if (blue100 > 255) { scale = (255.0 - blue50) / binc; goto again; }
      if (blue100 < 0) { scale = -1.0 * blue50 / binc; goto again; }

      if (spread < 0) {                                                    //  make mid-scale == original RGB  v.5.7
         red = int(rgb0 + spread1 * (red50 - rgb0));
         green = int(rgb0 + spread1 * (green50 - rgb0));
         blue = int(rgb0 + spread1 * (blue50 - rgb0));
      }
      else {
         red = int(red50 + spread2 * (red100 - red50));
         green = int(green50 + spread2 * (green100 - green50));
         blue = int(blue50 + spread2 * (blue100 - blue50));
      }

      if (sa_exist && dist < sa_blend) {                                   //  if area selection, blend pixel
         f1 = 1.0 * dist / sa_blend;                                       //    changes over distance sa_blend
         f2 = 1.0 - f1;
         red = int(f1 * red + f2 * red50);
         green = int(f1 * green + f2 * green50);
         blue = int(f1 * blue + f2 * blue50);
      }

      pix3[0] = red;
      pix3[1] = green;
      pix3[2] = blue;
      
      RETURN
   }
}


/**************************************************************************/

//  image sharpening function

int    sharp_dialog_event(zdialog *zd, const char *event);
int    sharp_dialog_compl(zdialog *zd, int zstat);
int    sharp_imageED();
int    sharp_imageUM();
int    sharp_imageLP();

int      sharpED_cycles;
int      sharpED_reduce;
int      sharpED_thresh;
int      sharpUM_radius;
int      sharpUM_amount;
int      sharpUM_thresh;
int      sharpLP_radius;
int      sharpLP_amount;


void m_sharp()
{
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("sharpen image"),mWin,Bdone,Bcancel,null);     //  dialog for user inputs
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","buttundo","hb1",Bundo,"space=20");

   zdialog_add_widget(zdedit,"hsep","sep2","dialog");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb21","hb2",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb22","hb2",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb23","hb2",0,"homog|space=5");

   zdialog_add_widget(zdedit,"button","buttED","vb21",ZTX("edge detection"),"space=5");
   zdialog_add_widget(zdedit,"label","lab21","vb22",ZTX("cycles"),"space=5");
   zdialog_add_widget(zdedit,"label","lab22","vb22",ZTX("reduce"),"space=5");
   zdialog_add_widget(zdedit,"label","lab23","vb22",ZTX("threshold"));
   zdialog_add_widget(zdedit,"spin","cyclesED","vb23","1|30|1|10","space=5");
   zdialog_add_widget(zdedit,"spin","reduceED","vb23","50|95|1|80","space=5");
   zdialog_add_widget(zdedit,"spin","threshED","vb23","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep4","dialog");
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb41","hb4",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb42","hb4",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb43","hb4",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","buttUM","vb41",ZTX("unsharp mask"),"space=5");
   zdialog_add_widget(zdedit,"label","lab41","vb42",ZTX("radius"));
   zdialog_add_widget(zdedit,"label","lab42","vb42",ZTX("amount"));
   zdialog_add_widget(zdedit,"label","lab43","vb42",ZTX("threshold"));
   zdialog_add_widget(zdedit,"spin","radiusUM","vb43","1|9|1|2");
   zdialog_add_widget(zdedit,"spin","amountUM","vb43","1|200|1|100");
   zdialog_add_widget(zdedit,"spin","threshUM","vb43","1|99|1|1");

   zdialog_add_widget(zdedit,"hsep","sep5","dialog");
   zdialog_add_widget(zdedit,"hbox","hb5","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb51","hb5",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb52","hb5",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb53","hb5",0,"homog|space=5");
   zdialog_add_widget(zdedit,"button","buttLP","vb51","Laplacian","space=5");
   zdialog_add_widget(zdedit,"label","lab51","vb52",ZTX("radius"));
   zdialog_add_widget(zdedit,"label","lab52","vb52",ZTX("amount"));
   zdialog_add_widget(zdedit,"spin","radiusLP","vb53","1|9|1|1");
   zdialog_add_widget(zdedit,"spin","amountLP","vb53","1|100|1|50");

   zdialog_run(zdedit,sharp_dialog_event,sharp_dialog_compl);              //  run dialog, parallel
   return;
}


//  sharpen dialog event and completion callback functions

int sharp_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}

int sharp_dialog_event(zdialog * zd, const char *event)
{
   int      ii, px, py;
   pixel    pix1, pix3;

   zdialog_fetch(zd,"cyclesED",sharpED_cycles);                            //  get all input values
   zdialog_fetch(zd,"reduceED",sharpED_reduce);
   zdialog_fetch(zd,"threshED",sharpED_thresh);
   zdialog_fetch(zd,"radiusUM",sharpUM_radius);
   zdialog_fetch(zd,"amountUM",sharpUM_amount);
   zdialog_fetch(zd,"threshUM",sharpUM_thresh);
   zdialog_fetch(zd,"radiusLP",sharpLP_radius);
   zdialog_fetch(zd,"amountLP",sharpLP_amount);
                                                                           //  check buttons
   if (strEqu(event,"buttED")) {
      postmessage = zmessage_post(ZTX("computing"));
      sharp_imageED();                                                     //  edge detection
      zmessage_kill(postmessage);
      Fmod3 = 1;
      mwpaint2(); 
   }

   if (strEqu(event,"buttUM")) {
      postmessage = zmessage_post(ZTX("computing"));
      sharp_imageUM();                                                     //  unsharp mask
      zmessage_kill(postmessage);
      Fmod3 = 1;
      mwpaint2(); 
   }

   if (strEqu(event,"buttLP")) {
      postmessage = zmessage_post(ZTX("computing"));
      sharp_imageLP();                                                     //  Laplacian
      zmessage_kill(postmessage);
      Fmod3 = 1;
      mwpaint2(); 
   }

   if (strEqu(event,"buttundo")) 
   {                                                                       //  undo
      if (sa_exist) 
      {                                                                    //  restore selected area
         for (ii = 0; ii < sa_npix; ii++)
         {
            px = sa_pix[ii].px;
            py = sa_pix[ii].py;
            pix1 = ppix1 + py * rs1 + px * nch;                            //  image1 pixel >> image3
            pix3 = ppix3 + py * rs3 + px * nch;
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
         mwpaint2();                                                       //  refresh window
      }
      else  prior_image3();                                                //  restore entire image3
   }

   return 1;
}


//  image sharpen function by edge detection and compression

int sharp_imageED()
{
   int      sharp_thresh1 = 100;                                           //  initial threshold
   double   sharp_thresh2 = 0.01 * sharpED_reduce;                         //  decline rate       v.4.6
   int      ii, px, py, cycles, thresh;
   
   GOFUNC_DATA(1)
   
   prior_image3();                                                         //  refresh image3
   func_busy++;
   thresh = sharp_thresh1;
    
   for (cycles = 0; cycles < sharpED_cycles; cycles++)
   {
      if (cycles > 0) thresh = int(thresh * sharp_thresh2);
      
      if (! sa_exist)                                                      //  process entire image
      {
         for (px = 2; px < ww3-2; px++)                                    //  loop all image3 pixels
         for (py = 2; py < hh3-2; py++)
         {
            GOFUNC(sharp_pixel)
            zmainloop(1000);
            if (kill_func) break;
         }
      }

      if (sa_exist)                                                        //  selected area to sharpen
      {
         for (ii = 0; ii < sa_npix; ii++)                                  //  process all enclosed pixels
         {
            px = sa_pix[ii].px;
            py = sa_pix[ii].py;
            if (px < 2 || px >= ww3-2) continue;                           //  omit pixels on edge
            if (py < 2 || py >= hh3-2) continue;
            GOFUNC(sharp_pixel)
            zmainloop(1000);
            if (kill_func) break;
         }
      }
   }

   func_busy--;   
   return 1;

sharp_pixel:                                                               //  revised  v.4.6
   {
      int      dd, rgb, pthresh;
      int      dx[4] = { -1, 0, 1, 1 };                                    //  4 directions: NW N NE E
      int      dy[4] = { -1, -1, -1, 0 };
      int      pv3, pv3u, pv3d, pv3uu, pv3dd, pvdiff;
      pixel    pix3, pix3u, pix3d, pix3uu, pix3dd;
      
      pthresh = sharpED_thresh;                                            //  pthresh = larger
      if (thresh > pthresh) pthresh = thresh;

      pix3 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3

      for (dd = 0; dd < 4; dd++)                                           //  4 directions
      {
         pix3u = pix3 + dy[dd] * rs3 + dx[dd] * nch;                       //  upstream pixel
         pix3d = pix3 - dy[dd] * rs3 - dx[dd] * nch;                       //  downstream pixel

         for (rgb = 0; rgb < 3; rgb++)                                     //  loop 3 RGB colors
         {
            pv3 = pix3[rgb];
            pv3u = pix3u[rgb];                                             //  brightness difference
            pv3d = pix3d[rgb];                                             //    across target pixel

            pvdiff = pv3d - pv3u;
            if (pvdiff < 0) pvdiff = -pvdiff;
            if (pvdiff < pthresh) continue;                                //  brightness slope < threshold

            if (pv3u < pv3 && pv3 < pv3d)                                  //  slope up, monotone
            {
               pix3uu = pix3u + dy[dd] * rs3 + dx[dd] * nch;               //  upstream of upstream pixel
               pix3dd = pix3d - dy[dd] * rs3 - dx[dd] * nch;               //  downstream of downstream
               pv3uu = pix3uu[rgb];
               pv3dd = pix3dd[rgb];

               if (pv3uu >= pv3u) {                                        //  shift focus of changes to
                  pix3u = pix3;                                            //    avoid up/down/up jaggies
                  pv3u = pv3;                                              //             v.4.6
               }
               
               if (pv3dd <= pv3d) {
                  pix3d = pix3;
                  pv3d = pv3;
               }
                  
               if (pv3u > 0) pv3u--;
               if (pv3d < 255) pv3d++;
            }
            
            else if (pv3u > pv3 && pv3 > pv3d)                             //  slope down, monotone
            {
               pix3uu = pix3u + dy[dd] * rs3 + dx[dd] * nch;
               pix3dd = pix3d - dy[dd] * rs3 - dx[dd] * nch;
               pv3uu = pix3uu[rgb];
               pv3dd = pix3dd[rgb];

               if (pv3uu <= pv3u) {
                  pix3u = pix3;
                  pv3u = pv3;
               }
               
               if (pv3dd >= pv3d) {
                  pix3d = pix3;
                  pv3d = pv3;
               }

               if (pv3d > 0) pv3d--;
               if (pv3u < 255) pv3u++;
            }

            else continue;                                                 //  slope too small

            pix3u[rgb] = pv3u;                                             //  modified brightness values
            pix3d[rgb] = pv3d;                                             //    >> image3 pixel
         }
      }
      RETURN
   }
}


//  image sharpen function using unsharp mask                              //  v.4.5

int sharp_imageUM()
{
   int         ii, px, py, dx, dy, rgb;
   int         incr, cval, mean;
   int         rad = sharpUM_radius;
   pixel       pix1, pix3, pixN;
   double      rad2;
   
   GOFUNC_DATA(1)
   
   prior_image3();                                                         //  refresh image3
   func_busy++;

   rad2 = 2 * rad + 1;
   rad2 = 1.0 / (rad2 * rad2);                                             //  1 / area of unsharp mask
   
   if (! sa_exist)                                                         //  process entire image
   {
      for (px = rad; px < ww3-rad; px++)                                   //  loop all image3 pixels
      for (py = rad; py < hh3-rad; py++)
      {
         GOFUNC(sharp_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   if (sa_exist)                                                           //  selected area to sharpen
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         if (px < rad || px >= ww3-rad) continue;                          //  omit pixels on edge
         if (py < rad || py >= hh3-rad) continue;
         GOFUNC(sharp_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   func_busy--;   
   return 1;

sharp_pixel:
   {
      pix1 = ppix1 + py * rs1 + px * nch;                                  //  input pixel
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  output pixel

      for (rgb = 0; rgb < 3; rgb++)                                        //  loop 3 RGB colors
      {
         mean = 0;

         for (dx = -rad; dx <= rad; dx++)                                  //  loop neighbor pixels within radius
         for (dy = -rad; dy <= rad; dy++)
         {
            pixN = pix1 + dy * rs1 + dx * nch;                             //  compute mean brightness
            mean += pixN[rgb];
         }

         mean = int(mean * rad2 + 0.5);
         
         cval = pix3[rgb];                                                 //  pixel brightness
         incr = cval - mean;                                               //  - mean of neighbors
         if (abs(incr) < sharpUM_thresh) continue;                         //  if < threshold, skip
         incr = incr * sharpUM_amount / 100;                               //  reduce to sharp_amount
         cval = cval + incr;                                               //  add back to pixel
         if (cval < 0) cval = 0;
         if (cval > 255) cval = 255;
         pix3[rgb] = cval;
      }
      
      RETURN
   }
}


//  image sharpen function using Lapacian method                           //  v.5.1

int sharp_imageLP()
{
   int         ii, jj, px, py, dx, dy, rgb;
   int         rad = sharpLP_radius;
   int         sumpix[3], maxrgb;
   double      rad2, kern, kernsum;
   double      scale, scale1, scale2;
   double      kernel[19][19];
   pixel       pix1, pix3, pixN;
   
   GOFUNC_DATA(1)
   
   prior_image3();                                                         //  refresh image3
   func_busy++;

   rad2 = rad * rad;
   kernsum = 0;

   for (dx = -rad; dx <= rad; dx++)                                        //  kernel, Gaussian distribution
   for (dy = -rad; dy <= rad; dy++)
   {
      ii = dx + rad;
      jj = dy + rad;
      kern = (0.5 / rad2) * exp( -(dx*dx + dy*dy) / (2 * rad2));
      if (dx || dy) {
         kernel[ii][jj] = -kern;                                           //  surrounding cells < 0
         kernsum += kern;
      }
   }
   
   kernel[rad][rad] = kernsum + 1.0;                                       //  middle cell, total = +1
   
   scale1 = (100 - sharpLP_amount) / 100.0;

   if (! sa_exist)                                                         //  process entire image
   {
      for (px = rad; px < ww3-rad; px++)                                   //  loop all image3 pixels
      for (py = rad; py < hh3-rad; py++)
      {
         GOFUNC(sharp_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   if (sa_exist)                                                           //  selected area to sharpen
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         if (px < rad || px >= ww3-rad) continue;                          //  omit pixels on edge
         if (py < rad || py >= hh3-rad) continue;
         GOFUNC(sharp_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }
   
   func_busy--;
   return 1;

sharp_pixel:
   {
      pix1 = ppix1 + py * rs1 + px * nch;                                  //  input pixel
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  output pixel

      sumpix[0] = sumpix[1] = sumpix[2] = 0;      

      for (dx = -rad; dx <= rad; dx++)                                     //  loop surrounding block of pixels
      for (dy = -rad; dy <= rad; dy++)
      {
         pixN = pix1 + dy * rs1 + dx * nch;
         kern = kernel[dx+rad][dy+rad];
         for (rgb = 0; rgb < 3; rgb++)
            sumpix[rgb] += int(kern * pixN[rgb] - 0.5);                    //  round, v.5.2.1
      }
      
      maxrgb = sumpix[0];
      if (sumpix[1] > maxrgb) maxrgb = sumpix[1];
      if (sumpix[2] > maxrgb) maxrgb = sumpix[2];
      
      if (maxrgb > 255) scale = 255.0 / maxrgb;
      else scale = 1.0;
      
      if (sumpix[0] < 0) sumpix[0] = 0;
      if (sumpix[1] < 0) sumpix[1] = 0;
      if (sumpix[2] < 0) sumpix[2] = 0;
      
      scale2 = (1.0 - scale1) * scale;

      pix3[0] = int(scale1 * pix1[0] + scale2 * sumpix[0]);
      pix3[1] = int(scale1 * pix1[1] + scale2 * sumpix[1]);
      pix3[2] = int(scale1 * pix1[2] + scale2 * sumpix[2]);
      
      RETURN
   }
}


/**************************************************************************/

//  image blur function 

int    blur_dialog_event(zdialog *zd, const char *event);
int    blur_dialog_compl(zdialog *zd, int zstat);
int    blur_image();

int      blur_radius;


void m_blur()
{

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("set blur radius"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","undo","hb1",Bundo,"space=20");
   zdialog_add_widget(zdedit,"hsep","sep1","dialog");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"label","labrad","hb2",ZTX("blur radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","0|9|1|1","space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");

   zdialog_run(zdedit,blur_dialog_event,blur_dialog_compl);                //  run dialog
   return;
}


//  blur dialog event and completion callback functions

int blur_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}

int blur_dialog_event(zdialog * zd, const char *event)
{
   int      ii, px, py;
   pixel    pix1, pix3;

   zdialog_fetch(zd,"radius",blur_radius);

   if (strEqu(event,"apply")) {
      postmessage = zmessage_post(ZTX("computing"));
      blur_image();                                                        //  blur function
      zmessage_kill(postmessage);                                          //  computation complete
      if (blur_radius > 0) Fmod3 = 1;                                      //  image3 modified
      mwpaint2(); 
   }

   if (strEqu(event,"undo")) {
      if (sa_exist)                                                        //  selected area exists
      {
         for (ii = 0; ii < sa_npix; ii++)                                  //  process all enclosed pixels
         {
            px = sa_pix[ii].px;
            py = sa_pix[ii].py;
            pix1 = ppix1 + py * rs1 + px * nch;                            //  image1 pixel >> image3
            pix3 = ppix3 + py * rs3 + px * nch;
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
         mwpaint2();                                                       //  refresh window
      }
      else  prior_image3();                                                //  restore entire image3
   }

   return 1;
}


//  image blur function

int blur_image()
{
   int         ii, px, py, dx, dy, adx, ady, rad;
   double      rad2, red, green, blue;
   double      m, d, w, sum, weight[10][10];                               //  up to blur radius = 9
   pixel       pix1, pix3, pixN;
   
   GOFUNC_DATA(1)

   if (blur_radius == 0) {
      prior_image3();                                                      //  zero blur, restore image3
      return 1;
   }
   
   func_busy++;

   rad = blur_radius;
   rad2 = rad * rad;

   for (dx = 0; dx <= rad; dx++)                                           //  clear weights array
   for (dy = 0; dy <= rad; dy++)
      weight[dx][dy] = 0;

   for (dx = -rad; dx <= rad; dx++)                                        //  weight[dx][dy] = no. of pixels
   for (dy = -rad; dy <= rad; dy++)                                        //    at distance (dx,dy) from center
      ++weight[abs(dx)][abs(dy)];

   m = sqrt(rad2 + rad2);                                                  //  corner pixel distance from center
   sum = 0;

   for (dx = 0; dx <= rad; dx++)                                           //  compute weight of pixel
   for (dy = 0; dy <= rad; dy++)                                           //    at distance dx, dy
   {
      d = sqrt(dx*dx + dy*dy);
      w = (m + 1 - d) / m;
      w = w * w;
      sum += weight[dx][dy] * w;
      weight[dx][dy] = w;
   }

   for (dx = 0; dx <= rad; dx++)                                           //  make weights add up to 1.0
   for (dy = 0; dy <= rad; dy++)
      weight[dx][dy] = weight[dx][dy] / sum;

   if (! sa_exist)                                                         //  process entire image
   {
      for (px = rad; px < ww3-rad; px++)                                   //  loop all image3 pixels
      for (py = rad; py < hh3-rad; py++)
      {
         GOFUNC(blur_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   if (sa_exist)                                                           //  process selected area
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         if (px < rad || px >= ww3-rad) continue;                          //  omit pixels on edge
         if (py < rad || py >= hh3-rad) continue;
         GOFUNC(blur_pixel)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   func_busy--;
   return 1;
   
blur_pixel:
   {
      pix1 = ppix1 + py * rs1 + px * nch;                                  //  source pixel in image1
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  target pixel in image3
      
      red = green = blue = 0;
      
      for (dx = -rad; dx <= rad; dx++)                                     //  loop neighbor pixels within radius
      for (dy = -rad; dy <= rad; dy++)
      {
         adx = abs(dx);
         ady = abs(dy);
         pixN = pix1 + dy * rs1 + dx * nch;
         red += pixN[0] * weight[adx][ady];                                //  accumulate contributions
         green += pixN[1] * weight[adx][ady];
         blue += pixN[2] * weight[adx][ady];
      }
      
      pix3[0] = int(red);
      pix3[1] = int(green);
      pix3[2] = int(blue);
      
      RETURN
   }
}


/**************************************************************************/

//  image noise reduction                                                  v.5.0

int denoise_dialog_event(zdialog *zd, const char *event);                  //  dialog event function
int denoise_dialog_compl(zdialog *zd, int zstat);                          //  dialog completion function
int denoise_image();

int      denoise_method = 1;
int      denoise_radius = 2;

void m_denoise()
{
   const char  *denoise_message = ZTX(" Press the reduce button to \n"
                                    " reduce noise in small steps. \n"
                                    " Use reset to start over.");
   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)
   
   zdedit = zdialog_new(ZTX("noise reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",denoise_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labalg","hb1",ZTX("algorithm"),"space=5");
   zdialog_add_widget(zdedit,"combo","method","hb1",0,"space=5|expand");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","labrad","hb2",ZTX("radius"),"space=5");
   zdialog_add_widget(zdedit,"spin","radius","hb2","1|9|1|1","space=5");
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","reduce","hb3",ZTX("reduce"),"space=5");
   zdialog_add_widget(zdedit,"button","reset","hb3",Breset,"space=5");
   
   zdialog_cb_app(zdedit,"method","flatten outlyers by color");
   zdialog_cb_app(zdedit,"method","flatten outlyers by brightness");
   zdialog_cb_app(zdedit,"method","Gaussian smoothing");
   zdialog_cb_app(zdedit,"method","set median brightness by color");
   zdialog_stuff(zdedit,"method","flatten outlyers by color");

   zdialog_run(zdedit,denoise_dialog_event,denoise_dialog_compl);          //  run dialog
   return;
}


//  denoise dialog event and completion callback functions

int denoise_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}

int denoise_dialog_event(zdialog * zd, const char *event)
{
   int      ii, px, py;
   pixel    pix1, pix3;
   char     method[40];

   zdialog_fetch(zd,"radius",denoise_radius);
   zdialog_fetch(zd,"method",method,39);
   if (strEqu(method,"flatten outlyers by color")) denoise_method = 1;
   if (strEqu(method,"flatten outlyers by brightness")) denoise_method = 2;
   if (strEqu(method,"Gaussian smoothing")) denoise_method = 3;
   if (strEqu(method,"set median brightness by color")) denoise_method = 4;

   if (strEqu(event,"reduce")) {
      postmessage = zmessage_post(ZTX("computing"));                       //  must wait a while
      denoise_image();                                                     //  do the work
      zmessage_kill(postmessage);                                          //  computation complete
      mwpaint2(); 
   }

   if (strEqu(event,"reset")) 
   {
      if (sa_exist)                                                        //  selected area exists
      {
         for (ii = 0; ii < sa_npix; ii++)                                  //  process all enclosed pixels
         {
            px = sa_pix[ii].px;
            py = sa_pix[ii].py;
            pix1 = ppix1 + py * rs1 + px * nch;                            //  image1 pixel >> image3
            pix3 = ppix3 + py * rs3 + px * nch;
            pix3[0] = pix1[0];
            pix3[1] = pix1[1];
            pix3[2] = pix1[2];
         }
         mwpaint2(); 
      }
      else  prior_image3();                                                //  restore entire image3
   }

   return 1;
}


//  image noise reduction function

int denoise_image()
{
   GdkPixbuf   *pxb9;
   pixel       pixN, pix3, pix9, ppix9;
   int         ww9, hh9, rs9;
   int         ii, dist, px, py, rad;

   GOFUNC_DATA(1)
   
   func_busy++;
   
   pxb9 = pixbuf_copy(pxb3);                                               //  copy image3 to image9 
   pixbuf_test(9);                                                         //  image3 is reference source
   pixbuf_poop(9);                                                         //  image9 will be modified
   
   rad = denoise_radius;

   if (sa_exist)
   {
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all pixels in select area
      {
         zmainloop(1000);
         dist = sa_pix[ii].dist;
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         if (px < rad || px >= ww3-rad) continue;                          //  fix bug   v.5.1
         if (py < rad || py >= hh3-rad) continue;
         pix3 = ppix3 + py * rs3 + px * nch;                               //  source pixel
         pix9 = ppix9 + py * rs9 + px * nch;                               //  target pixel
         if (denoise_method == 1) GOFUNC(denoise_func_1)
         if (denoise_method == 2) GOFUNC(denoise_func_2)
         if (denoise_method == 3) GOFUNC(denoise_func_3)
         if (denoise_method == 4) GOFUNC(denoise_func_4)
         zmainloop(1000);
         if (kill_func) break;
      }
   }

   if (! sa_exist)
   {
      for (px = rad; px < ww3-rad; px++)                                   //  loop all image3 pixels
      for (py = rad; py < hh3-rad; py++)
      {
         zmainloop(1000);
         pix3 = ppix3 + py * rs3 + px * nch;                               //  source pixel
         pix9 = ppix9 + py * rs9 + px * nch;                               //  target pixel
         if (denoise_method == 1) GOFUNC(denoise_func_1)
         if (denoise_method == 2) GOFUNC(denoise_func_2)
         if (denoise_method == 3) GOFUNC(denoise_func_3)
         if (denoise_method == 4) GOFUNC(denoise_func_4)
         zmainloop(1000);
         if (kill_func) break;
      }
   }
   
   pixbuf_free(pxb3);                                                      //  new image3 = image9
   pxb3 = pxb9;
   pixbuf_poop(3);

   Fmod3 = 1;                                                              //  image3 modified
   func_busy--;
   return 1;

//  noise reduction algorithms

   int         dy, dx, bright, oldbright, newbright;
   int         min0, min1, min2, max0, max1, max2, minb, maxb;
   int         kk, sumkk, sumpix[3];
   int         ns, rgb, bsortN[400];
   double      Rbright;

//  flatten outlyers within radius, by color 

denoise_func_1:
   {
      min0 = min1 = min2 = 255;
      max0 = max1 = max2 = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) continue;                                 //  skip self

         pixN = pix3 + dy * rs3 + dx * nch;
         if (pixN[0] < min0) min0 = pixN[0];                               //  find min and max per color
         if (pixN[0] > max0) max0 = pixN[0];
         if (pixN[1] < min1) min1 = pixN[1];
         if (pixN[1] > max1) max1 = pixN[1];
         if (pixN[2] < min2) min2 = pixN[2];
         if (pixN[2] > max2) max2 = pixN[2];
      }
      
      if (pix3[0] <= min0 && min0 < 255) pix9[0] = min0 + 1;               //  if outlier, flatten
      if (pix3[0] >= max0 && max0 > 0) pix9[0] = max0 - 1;
      if (pix3[1] <= min1 && min1 < 255) pix9[1] = min1 + 1;
      if (pix3[1] >= max1 && max1 > 0) pix9[1] = max1 - 1;
      if (pix3[2] <= min2 && min2 < 255) pix9[2] = min2 + 1;
      if (pix3[2] >= max2 && max2 > 0) pix9[2] = max2 - 1;
      
      RETURN
   }

//  flatten outlyers within radius, by overall brightness

denoise_func_2:
   {
      minb = 2550;
      maxb = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) continue;                                 //  skip self

         pixN = pix3 + dy * rs3 + dx * nch;
         bright = int(2.5 * pixN[0] + 6.5 * pixN[1] + 1.0 * pixN[2]);      //  find max. and min. brightness
         if (bright < minb) minb = bright;
         if (bright > maxb) maxb = bright;
      }

      oldbright = int(2.5 * pix3[0] + 6.5 * pix3[1] + 1.0 * pix3[2]);      //  compare pixel to max/min
      newbright = oldbright;
      if (oldbright <= minb) newbright++;
      if (oldbright >= maxb) newbright--;
      
      if (newbright != oldbright) {                                        //  if outlyer, flatten
         maxb = pix3[0];
         if (pix3[1] > maxb) maxb = pix3[1];
         if (pix3[2] > maxb) maxb = pix3[2];
         
         bright = (newbright - oldbright);
         Rbright = 1.0 * (maxb + 4 * bright) / maxb;                       //  faster flatten   v.5.1

         bright = int(round(Rbright * pix3[0]));
         if (bright < 0) bright = 0;                                       //  stop underflow   v.5.1
         if (bright > 255) bright = 255;
         pix9[0] = bright;
         bright = int(round(Rbright * pix3[1]));
         if (bright < 0) bright = 0;
         if (bright > 255) bright = 255;
         pix9[1] = bright;
         bright = int(round(Rbright * pix3[2]));
         if (bright < 0) bright = 0;
         if (bright > 255) bright = 255;
         pix9[2] = bright;
      }

      RETURN
   }

//  Gaussian smoothing - fast and blurry                                   //  v.5.1

denoise_func_3:
   {
      sumkk = 0;

      for (rgb = 0; rgb < 3; rgb++) 
         sumpix[rgb] = 0;

      for (dy = -rad; dy <= rad; dy++)                                     //  loop surrounding pixels
      for (dx = -rad; dx <= rad; dx++)
      {
         if (dy == 0 && dx == 0) kk = 4;                                   //  kernel:    1 2 1
         else if (dy == 0 || dx == 0) kk = 2;                              //  (rad=1)    2 4 2
         else kk = 1;                                                      //             1 2 1
         
         sumkk += kk;
         
         pixN = pix3 + dy * rs3 + dx * nch;
         
         for (rgb = 0; rgb < 3; rgb++) 
            sumpix[rgb] += pixN[rgb] * kk;
      }
      
      for (rgb = 0; rgb < 3; rgb++)
         pix9[rgb] = int(1.0 * sumpix[rgb] / sumkk + 0.5);

      RETURN
   }

//  use median brightness for pixels within radius                         //  v.5.1

denoise_func_4:
   {
      for (rgb = 0; rgb < 3; rgb++)                                        //  loop all RGB colors
      {
         ns = 0;

         for (dy = -rad; dy <= rad; dy++)                                  //  loop surrounding pixels
         for (dx = -rad; dx <= rad; dx++)                                  //  get brightness values
         {
            pixN = pix3 + dy * rs3 + dx * nch;
            bsortN[ns] = pixN[rgb];
            ns++;
         }

         HeapSort(bsortN,ns);
         pix9[rgb] = bsortN[ns/2];                                         //  median brightness of 2*ns pixels
      }

      RETURN
   }
}


/**************************************************************************/

//  red eye removal      v.5.5  overhaul

int      redeye_dialog_compl(zdialog *zd, int zstat);
void     redeye_mousefunc();                                               //  handle mouse clicks and drag
int      redeye_createF(int px, int py);                                   //  create 1-click red-eye (type F)
int      redeye_createR(int px, int py, int ww, int hh);                   //  create robust red-eye (type R)
void     redeye_darken(int ii);                                            //  darken red-eye
void     redeye_distr(int ii);                                             //  build pixel redness distribution
int      redeye_find(int px, int py);                                      //  find red-eye at mouse position
void     redeye_remove(int ii);                                            //  remove red-eye at mouse position
int      redeye_radlim(int cx, int cy);                                    //  compute red-eye radius limit

struct sredmem {                                                           //  red-eye struct in memory
   char        type, space[3];
   int         cx, cy, ww, hh, rad, clicks;
   double      thresh, tstep;
};
sredmem  redmem[100];                                                      //  store up to 100 red-eyes
int      Nredmem = 0, maxredmem = 100;


//  menu function

void m_redeye()
{
   const char  *redeye_message 
         = ZTX( "Method 1:\n"
              "  Left-click on red-eye to darken.\n"
              "Method 2:\n"
              "  Drag down and right to enclose red-eye.\n"
              "  Left-click on red-eye to darken.\n"
              "Undo red-eye:\n"
              "  Right-click on red-eye.");

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("red eye reduction"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",redeye_message);
   zdialog_run(zdedit,0,redeye_dialog_compl);                              //  run dialog, parallel mode

   mouseCBfunc = redeye_mousefunc;                                         //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   Nredmem = 0;
   return;
}


//  dialog completion callback function

int redeye_dialog_compl(zdialog *zd, int zstat)
{
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   if (zstat != 1) pull_image3();                                          //  cancel - restore image3
   return 0;
}


//  mouse function create, darken, remove red-eye corrections

void redeye_mousefunc()
{
   int         ii, px, py, ww, hh;

   if (Nredmem == maxredmem) {
      zmessageACK("%d red-eye limit reached",maxredmem);                   //  too many red-eyes
      return;
   }

   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;

      px = click3x;                                                        //  click position
      py = click3y;
      if (px < 0 || px > ww3-1 || py < 0 || py > hh3-1) return;            //  outside image area

      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii < 0) ii = redeye_createF(px,py);                              //  or create new type F
      redeye_darken(ii);                                                   //  darken red-eye
      Fmod3 = 1;
      mwpaint2();
   }
   
   if (RMclick)                                                            //  right mouse click
   {
      RMclick = 0;
      px = click3x;                                                        //  click position
      py = click3y;
      ii = redeye_find(px,py);                                             //  find red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  if found, remove
      mwpaint2();
   }

   if (md3x1 || md3y1)                                                     //  mouse drag underway
   {
      px = md3x1;                                                          //  initial position
      py = md3y1;
      ww = md3x2 - md3x1;                                                  //  increment
      hh = md3y2 - md3y1;
      if (ww < 2 && hh < 2) return;                                        //  v.5.6
      if (ww < 2) ww = 2;
      if (hh < 2) hh = 2;
      if (px < 1) px = 1;                                                  //  keep within image area
      if (py < 1) py = 1;      
      if (px + ww > ww3-1) ww = ww3-1 - px;
      if (py + hh > hh3-1) hh = hh3-1 - py;
      ii = redeye_find(px,py);                                             //  find existing red-eye
      if (ii >= 0) redeye_remove(ii);                                      //  remove it
      mwpaint();
      ii = redeye_createR(px,py,ww,hh);                                    //  create new red-eye type R
      Fmod3 = 1;
   }
}


//  create type F redeye (1-click automatic)

int redeye_createF(int cx, int cy)
{
   pixel       ppix;
   int         cx0, cy0, cx1, cy1, px, py, rad, radlim;
   int         loops, ii, sumx, sumy;
   int         Tnpix, Rnpix, R2npix, npix;
   double      rd, rcx, rcy, redpart, maxred, thresh;
   double      Tsum, Rsum, R2sum, Tavg, Ravg, R2avg;
   
   cx0 = cx;                                                               //  v.5.6
   cy0 = cy;
   
   for (loops = 0; loops < 5; loops++)
   {
      cx1 = cx;
      cy1 = cy;

      radlim = redeye_radlim(cx,cy);                                       //  radius limit (image edge)
      Tsum = Tavg = Ravg = maxred = Tnpix = 0;

      for (rad = 0; rad < radlim-2; rad++)                                 //  find red-eye radius from (cx,cy)
      {
         Rsum = Rnpix = 0;
         R2sum = R2npix = 0;

         for (px = cx-rad-2; px <= cx+rad+2; px++)
         for (py = cy-rad-2; py <= cy+rad+2; py++)
         {
            rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
            ppix = ppix1 + py * rs1 + px * nch;
            redpart = redness(ppix);
            if (redpart > maxred) maxred = redpart;

            if (rd <= rad + 0.5 && rd > rad - 0.5) {                       //  accum. redness at rad
               Rsum += redpart;
               Rnpix++;
            }
            else if (rd <= rad + 2.5 && rd > rad + 1.5) {                  //  accum. redness at rad + 2
               R2sum += redpart;
               R2npix++;
            }
         }
         
         Tsum += Rsum;                                                     //  accum. redness over 0-rad
         Tnpix += Rnpix;
         Tavg = Tsum / Tnpix;
         Ravg = Rsum / Rnpix;
         R2avg = R2sum / R2npix;
         if (R2avg > Ravg || Ravg > Tavg) continue;
         if (Tavg > Ravg && Ravg > R2avg && (Ravg - R2avg) < 0.1 * (Tavg - Ravg)) break;
      }
      
      sumx = sumy = npix = 0;
      rad = int(1.2 * rad + 1);
      if (rad > radlim) rad = radlim;
      thresh = 0.5 * (Tavg + maxred);
      
      for (px = cx-rad; px <= cx+rad; px++)                                //  compute center of gravity for
      for (py = cy-rad; py <= cy+rad; py++)                                //   pixels within rad of (cx,cy)
      {                                                                    //    with high redness
         rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
         if (rd > rad + 0.5) continue;
         ppix = ppix1 + py * rs1 + px * nch;
         if (redness(ppix) < Tavg) continue;
         sumx += (px - cx);
         sumy += (py - cy);
         npix++;
      }

      rcx = cx + 1.0 * sumx / npix;                                        //  new center of red-eye
      rcy = cy + 1.0 * sumy / npix;
      if (fabs(cx0 - rcx) > 10) break;                                     //  give up if big movement   v.5.6
      if (fabs(cy0 - rcy) > 10) break;
      cx = int(rcx + 0.5);
      cy = int(rcy + 0.5);
      if (cx == cx1 && cy == cy1) break;                                   //  done if no change
   }

   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   ii = Nredmem++;                                                         //  add red-eye to memory
   redmem[ii].type = 'F';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  create type R red-eye (drag an ellipse over red-eye area)

int redeye_createR(int cx, int cy, int ww, int hh)
{
   int      mpx, mpy, mww, mhh;
   int      rad, radlim;

   mpx = int((cx-ww/2) * Rscale + orgMx);                                  //  convert to pxbM space
   mpy = int((cy-hh/2) * Rscale + orgMy);
   mww = int(ww * Rscale);                                                 //  draw new ellipse
   mhh = int(hh * Rscale);
   gdk_draw_arc(GTK_LAYOUT(dWin)->bin_window,gdkgc,0,mpx,mpy,mww,mhh,0,64*360);

   if (ww > hh) rad = ww / 2;
   else rad = hh / 2;
   radlim = redeye_radlim(cx,cy);
   if (rad > radlim) rad = radlim;

   int ii = Nredmem++;                                                     //  add red-eye to memory
   redmem[ii].type = 'R';
   redmem[ii].cx = cx;
   redmem[ii].cy = cy;
   redmem[ii].ww = ww;
   redmem[ii].hh = hh;
   redmem[ii].rad = rad;
   redmem[ii].clicks = 0;
   redmem[ii].thresh = 0;
   return ii;
}


//  darken a red-eye and increase click count

void redeye_darken(int ii)
{
   int      cx, cy, ww, hh, px, py, rad, clicks;
   double   rd, thresh;
   char     type;
   pixel    ppix;

   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   thresh = redmem[ii].thresh;
   tstep = redmem[ii].tstep;
   clicks = redmem[ii].clicks++;
   
   if (thresh == 0)                                                        //  1st click 
   {
      redeye_distr(ii);                                                    //  get pixel redness distribution
      thresh = redmem[ii].thresh;                                          //  initial redness threshhold
      tstep = redmem[ii].tstep;                                            //  redness step size
   }

   tstep = (thresh - tstep) / thresh;                                      //  convert to reduction factor
   thresh = thresh * pow(tstep,clicks);                                    //  reduce threshhold by total clicks

   for (px = cx-rad; px <= cx+rad; px++)                                   //  darken pixels over threshhold
   for (py = cy-rad; py <= cy+rad; py++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = ppix3 + py * rs3 + px * nch;
      if (redness(ppix) > thresh) {
         ppix = ppix3 + py * rs3 + px * nch;                               //  set pixel redness = threshhold
         setredness(ppix,thresh);
      }
   }

   return;
}


//  Build a distribution of redness for a red-eye. Use this information 
//  to set initial threshhold and step size for stepwise darkening.

void redeye_distr(int ii)
{
   int         cx, cy, ww, hh, rad, px, py;
   int         bin, npix, dbins[20], bsum, blim;
   double      rd, maxred, minred, redpart, dbase, dstep;
   char        type;
   pixel       ppix;
   
   type = redmem[ii].type;
   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   ww = redmem[ii].ww;
   hh = redmem[ii].hh;
   rad = redmem[ii].rad;
   
   maxred = 0;
   minred = 100;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = ppix1 + py * rs1 + px * nch;
      redpart = redness(ppix);
      if (redpart > maxred) maxred = redpart;
      if (redpart < minred) minred = redpart;
   }
   
   dbase = minred;
   dstep = (maxred - minred) / 19.99;

   for (bin = 0; bin < 20; bin++) dbins[bin] = 0;
   npix = 0;

   for (px = cx-rad; px <= cx+rad; px++)
   for (py = cy-rad; py <= cy+rad; py++)
   {
      if (type == 'R') {
         if (px < cx - ww/2) continue;
         if (px > cx + ww/2) continue;
         if (py < cy - hh/2) continue;
         if (py > cy + hh/2) continue;
      }
      rd = sqrt((px-cx)*(px-cx) + (py-cy)*(py-cy));
      if (rd > rad + 0.5) continue;
      ppix = ppix1 + py * rs1 + px * nch;
      redpart = redness(ppix);
      bin = int((redpart - dbase) / dstep);
      ++dbins[bin];
      ++npix;
   }
   
   bsum = 0;
   blim = int(0.5 * npix);

   for (bin = 0; bin < 20; bin++)                                          //  find redness level for 50% of
   {                                                                       //    pixels within red-eye radius
      bsum += dbins[bin];
      if (bsum > blim) break;
   }

   redmem[ii].thresh = dbase + dstep * bin;                                //  initial redness threshhold
   redmem[ii].tstep = 0.6 * dstep;                                         //  redness step (3% range)  v.5.6

   return;
}


//  find a red-eye (nearly) overlapping the mouse click position

int redeye_find(int cx, int cy)
{
   for (int ii = 0; ii < Nredmem; ii++)
   {
      if (cx > redmem[ii].cx - 2 * redmem[ii].rad && 
          cx < redmem[ii].cx + 2 * redmem[ii].rad &&
          cy > redmem[ii].cy - 2 * redmem[ii].rad && 
          cy < redmem[ii].cy + 2 * redmem[ii].rad) 
            return ii;                                                     //  found
   }
   return -1;                                                              //  not found
}


//  remove a red-eye from memory

void redeye_remove(int ii)
{
   int      cx, cy, rad, npix;

   cx = redmem[ii].cx;
   cy = redmem[ii].cy;
   rad = redmem[ii].rad;
   npix = 2 * rad + 1;
   pixbuf_copy_area(pxb1,cx-rad,cy-rad,npix,npix,pxb3,cx-rad,cy-rad);
   for (ii++; ii < Nredmem; ii++) redmem[ii-1] = redmem[ii];
   Nredmem--;
}


//  compute red-eye radius limit: smaller of 100 and nearest image edge

int redeye_radlim(int cx, int cy)
{
   int radlim = 100;
   if (cx < 100) radlim = cx;
   if (ww1-1 - cx < 100) radlim = ww1-1 - cx;
   if (cy < 100) radlim = cy;
   if (hh1-1 - cy < 100) radlim = hh1-1 - cy;
   return radlim;
}


/**************************************************************************/

//  trim image - use mouse to select image region to retain

void   trim_mousefunc();
int    trim_dialog_compl(zdialog *zd, int zstat);

int      trimx1, trimy1, trimx2, trimy2;                                   //  trim rectangle
int      trimww, trimhh, trimwwp, trimhhp;
double   trimR;																				//  trim image ratio


void m_trim()
{
   const char     *trim_message = ZTX("click or drag trim margins");

   if (! pxb3) return;                                                     //  nothing to trim
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("trim image"),mWin,Btrim,Bcancel,null);        //  dialog to get user inuts
   zdialog_add_widget(zdedit,"label","lab1","dialog",trim_message);
   zdialog_add_widget(zdedit,"hsep","sep1","dialog");
   zdialog_add_widget(zdedit,"label","labw","dialog","width: xxx");
   zdialog_add_widget(zdedit,"label","labh","dialog","height: xxx");
   zdialog_add_widget(zdedit,"label","labr","dialog","ratio: x.xx");

   zdialog_run(zdedit,0,trim_dialog_compl);                                //  run dialog, parallel

   mouseCBfunc = trim_mousefunc;                                           //  connect mouse function  v.5.5
   gdk_window_set_cursor(dWin->window,dragcursor);                         //  set drag cursor

   trimx1 = int(0.2 * ww3);                                                //  start with 20% trim margins
   trimy1 = int(0.2 * hh3);
   trimx2 = int(0.8 * ww3);
   trimy2 = int(0.8 * hh3);
   trimww = trimwwp = trimx2 - trimx1;
   trimhh = trimhhp = trimy2 - trimy1;
   Fredraw++;

   return;
}


//  trim dialog completion callback function

int trim_dialog_compl(zdialog *zd, int zstat)
{
   GdkPixbuf      *pxbT = 0;

   mouseCBfunc = 0;                                                        //  disconnect mouse
   gdk_window_set_cursor(dWin->window,0);                                  //  restore normal cursor
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;

   if (zstat != 1) {
      pull_image3();                                                       //  cancel, restore image3
      return 0;
   }

   if (trimx1 + trimww > ww3) trimww = ww3 - trimx1;
   if (trimy1 + trimhh > hh3) trimhh = hh3 - trimy1;

   pxbT = pixbuf_new(colorspace,0,8,trimww,trimhh);                        //  trim image3 in pxb3
   pixbuf_test(T);
   pixbuf_copy_area(pxb3,trimx1,trimy1,trimww,trimhh,pxbT,0,0);
   pixbuf_free(pxb3);
   pxb3 = pxbT;
   pixbuf_poop(3)                                                          //  get new pxb3 attributes

   select_area_clear();                                                    //  select area invalidated  v.5.1
   Fmod3 = 1;                                                              //  image3 modified
   mwpaint2();                                                             //  update window
   return 0;
}


//  trim mouse function

void trim_mousefunc()
{
   int         xx, yy, ww, hh;
   char        text[20];
   
   if (LMclick || LMdown)                                                  //  mouse click or drag
   {
      LMclick = 0;

      if (abs(mp3x - trimx1) < abs(mp3x - trimx2)) trimx1 = mp3x;          //  new trim rectangle from mouse posn.
      else trimx2 = mp3x;
      if (abs(mp3y - trimy1) < abs(mp3y - trimy2)) trimy1 = mp3y;
      else trimy2 = mp3y;

      if (trimx1 > trimx2-10) trimx1 = trimx2-10;                          //  sanity limits
      if (trimy1 > trimy2-10) trimy1 = trimy2-10;
      if (trimx1 < 0) trimx1 = 0;
      if (trimy1 < 0) trimy1 = 0;
      if (trimx2 > ww3) trimx2 = ww3;
      if (trimy2 > hh3) trimy2 = hh3;
      trimww = trimx2 - trimx1;
      trimhh = trimy2 - trimy1;

      if (trimww != trimwwp || trimhh != trimhhp) {                        //  detect changes
         trimwwp = trimww;
         trimhhp = trimhh;
         Fredraw++;
      }
   }

   if (Fredraw)                                                            //  window needs update
   {
      mwpaint();
      Fredraw = 0;

      xx = int(Rscale * trimx1 + orgMx);                                   //  draw new
      yy = int(Rscale * trimy1 + orgMy);
      ww = int(Rscale * trimww);
      hh = int(Rscale * trimhh);
      gdk_draw_rectangle(GTK_LAYOUT(dWin)->bin_window,gdkgc,0,xx,yy,ww,hh);

      if (trimww > trimhh) trimR = 1.0 * trimww / trimhh;                  //  update dialog data   v.5.4
      else  trimR = 1.0 * trimhh / trimww;
      snprintf(text,19,"width: %d",trimww);
      zdialog_stuff(zdedit,"labw",text);
      snprintf(text,19,"height: %d",trimhh);
      zdialog_stuff(zdedit,"labh",text);
      snprintf(text,19,"ratio: %.2f",trimR);
      zdialog_stuff(zdedit,"labr",text);
   }

   return;
}


/**************************************************************************/

//  rotate image through any arbitrary angle

void * rotate_thread(void *);
int    rotate_dialog_event(zdialog *zd, const char * event);
int    rotate_dialog_compl(zdialog *zd, int zstat);

int         rotate_thread_stat;
double      rotate_delta = 0;


void m_rotate(double angle)
{
   pthread_t   tid;

   if (! pxb3) return;                                                     //  nothing to rotate
   
   if (angle)                                                              //  rotate via keyboard
   {                                                                       //  temp rotate, no modified status
      rotate_angle += angle;
      if (rotate_angle >= 360) rotate_angle -=360;                         //  accumulate rotation
      if (rotate_angle <= -360) rotate_angle +=360;

      zlock();
      pixbuf_free(pxb3);
      pxb3 = pixbuf_rotate(pxb1,rotate_angle);                             //  rotate to new angle
      pixbuf_test(3);
      pixbuf_poop(3);
      zunlock();
      
      mwpaint2();                                                          //  update window 
      return;
   }

   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("rotate image"),mWin,Bdone,Bcancel,null);      //  dialog: get rotation from user
   zdialog_add_widget(zdedit,"label","labdeg","dialog",ZTX("degrees"),"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"vbox","vb4","hb1",0,"space=5");
   zdialog_add_widget(zdedit,"button"," +0.1  ","vb1"," + 0.1 ");          //  button name is increment to use
   zdialog_add_widget(zdedit,"button"," -0.1  ","vb1"," - 0.1 ");
   zdialog_add_widget(zdedit,"button"," +1.0  ","vb2"," + 1   ");
   zdialog_add_widget(zdedit,"button"," -1.0  ","vb2"," - 1   ");
   zdialog_add_widget(zdedit,"button"," +10.0 ","vb3"," + 10  ");
   zdialog_add_widget(zdedit,"button"," -10.0 ","vb3"," - 10  ");
   zdialog_add_widget(zdedit,"button"," +90.0 ","vb4"," + 90  ");
   zdialog_add_widget(zdedit,"button"," -90.0 ","vb4"," - 90  ");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","atrim","hb2","auto trim","space=10");
   zdialog_add_widget(zdedit,"button","undo","hb2","undo","space=10");

   zdialog_run(zdedit,rotate_dialog_event,rotate_dialog_compl);            //  run dialog - parallel
   pthread_create(&tid,0,rotate_thread,0);                                 //  start computation thread
   return;
}


//  rotate dialog event and completion callback functions

int rotate_dialog_compl(zdialog *zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   rotate_thread_stat = 0;                                                 //  tell thread to exit
   rotate_angle = 0;                                                       //  v.5.6
   return 0;
}

int rotate_dialog_event(zdialog *zd, const char * event)
{
   GdkPixbuf   *pxbT = 0;
   int         err, wwcut, hhcut, trimww, trimhh;
   double      radians, incr;
   
   if (strEqu(event,"undo")) {                                             //  undo                v.5.7
      prior_image3();
      rotate_angle = 0;
      zdialog_stuff(zdedit,"labdeg","degrees");
      return 1;
   }
   
   if (strEqu(event,"atrim")) {                                            //  auto trim           v.5.7
      if (! rotate_angle) return 1;
      radians = fabs(rotate_angle / 57.296);
      wwcut = int(hh3 * sin(radians) + 1);                                 //  amount to trim
      hhcut = int(ww3 * sin(radians) + 1);
      trimww = ww3 - 2 * wwcut;
      trimhh = hh3 - 2 * hhcut;
      if (trimww < 10 || trimhh < 10) return 0;                            //  sanity check
      pxbT = pixbuf_new(colorspace,0,8,trimww,trimhh);
      pixbuf_test(T);
      pixbuf_copy_area(pxb3,wwcut,hhcut,trimww,trimhh,pxbT,0,0);
      pixbuf_free(pxb3);
      pxb3 = pxbT;
      pixbuf_poop(3);                                                      //  get new pxb3 attributes
      rotate_angle = 0;
      zdialog_stuff(zdedit,"labdeg","degrees");
      Fmod3 = 1;
      mwpaint2();                                                          //  update window
      return 1;
   }
   
   err = convSD(event,incr);                                               //  button name is increment to use
   if (err) return 0;
   rotate_delta += incr;
   return 1;
}


//  rotate thread function

void * rotate_thread(void *)
{
   char        text[20];
   
   rotate_thread_stat = 1;
   func_busy++;
   
   while (true)
   {
      zsleep(0.1);
      if (kill_func) break;
      if (rotate_thread_stat == 0) break;

      if (rotate_delta) 
      {
         rotate_angle += rotate_delta;                                     //  accum. net rotation   
         rotate_delta = 0;                                                 //    from dialog widget

         if (rotate_angle >= 360) rotate_angle -=360;
         if (rotate_angle <= -360) rotate_angle +=360;

         zdialog_stuff(zdedit,"labdeg","(computing)");
         zsleep(0.1);

         zlock();
         pixbuf_free(pxb3);
         pxb3 = pixbuf_rotate(pxb1,rotate_angle);                          //  rotate to new angle
         pixbuf_test(3);
         pixbuf_poop(3);
         zunlock();

         if (fabs(rotate_angle) < 0.01) rotate_angle = 0;

         if (rotate_angle) {
            Fmod3 = 1;                                                     //  image3 modified
            select_area_clear();                                           //  select area invalidated  v.5.1
         }

         sprintf(text,ZTX("degrees: %.1f"),rotate_angle);                  //  update dialog angle display
         zdialog_stuff(zdedit,"labdeg",text);
         mwpaint2();                                                       //  update window
      }
   }
   
   func_busy--;
   return 0;
}


/**************************************************************************/

//  resize image

int      resize_dialog_event(zdialog *zd, const char * event);

int      wpix0, hpix0, wpix1, hpix1;                                       //  old, new pixel dimensions
double   wpct1, hpct1;

void m_resize()
{
   GdkPixbuf   *pxb9;
   int         zstat;
   const char  *lockmess = ZTX("lock width/height ratio");

   if (! pxb3) return;                                                     //  nothing to resize
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("resize image"),mWin,Bapply,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zdedit,"label","placeholder","vb11",0);              //             pixels       percent
   zdialog_add_widget(zdedit,"label","labw","vb11",Bwidth);                //    width    [______]     [______]
   zdialog_add_widget(zdedit,"label","labh","vb11",Bheight);               //    height   [______]     [______]
   zdialog_add_widget(zdedit,"label","labpix","vb12","pixels");            //
   zdialog_add_widget(zdedit,"spin","wpix","vb12","20|9999|1|0");          //    presets  [2/3] [1/2] [1/3] [1/4] 
   zdialog_add_widget(zdedit,"spin","hpix","vb12","20|9999|1|0");          //
   zdialog_add_widget(zdedit,"label","labpct","vb13",Bpercent);            //    [_] lock width/height ratio
   zdialog_add_widget(zdedit,"spin","wpct","vb13","1|500|0.1|100");        //
   zdialog_add_widget(zdedit,"spin","hpct","vb13","1|500|0.1|100");        //       [ apply ]  [ cancel ]  
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"label","preset","hb2",Bpreset,"space=5");
   zdialog_add_widget(zdedit,"button","b 3/4","hb2"," 3/4 ");
   zdialog_add_widget(zdedit,"button","b 2/3","hb2"," 2/3 ");
   zdialog_add_widget(zdedit,"button","b 1/2","hb2"," 1/2 ");
   zdialog_add_widget(zdedit,"button","b 1/3","hb2"," 1/3 ");
   zdialog_add_widget(zdedit,"button","b 1/4","hb2"," 1/4 ");
   zdialog_add_widget(zdedit,"check","lock","dialog",lockmess);

   wpix0 = ww3;                                                            //  original image3 width, height
   hpix0 = hh3;
   zdialog_stuff(zdedit,"wpix",wpix0);
   zdialog_stuff(zdedit,"hpix",hpix0);
   zdialog_stuff(zdedit,"lock",1);
   
   zstat = zdialog_run(zdedit,resize_dialog_event,0);                      //  run dialog, blocking mode
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;

   if (zstat != 1) {                                                       //  resize cancelled
      pull_image3();
      return;
   }

   if (wpix1 != wpix0 || hpix1 != hpix0) {                                 //  modified
      zlock();
      pxb9 = pixbuf_scale_simple(pxb3,wpix1,hpix1,HYPER);                  //  scale to size (high quality)  v.5.1
      pixbuf_test(9);
      pixbuf_free(pxb3);
      pxb3 = pxb9;
      pixbuf_poop(3);
      zunlock();
      Fmod3 = 1;                                                           //  image3 modified
      select_area_clear();                                                 //  select area invalidated  v.5.1
      mwpaint2();                                                          //  update window
   }
   
   return;
}

int resize_dialog_event(zdialog *zd, const char * event)
{
   int         lock;

   zdialog_fetch(zd,"wpix",wpix1);                                         //  get all widget values
   zdialog_fetch(zd,"hpix",hpix1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);
   
   if (strEqu(event,"b 3/4")) {
      wpix1 = (3 * wpix0 + 3) / 4;
      hpix1 = (3 * hpix0 + 3) / 4;
   }
   
   if (strEqu(event,"b 2/3")) {
      wpix1 = (2 * wpix0 + 2) / 3;
      hpix1 = (2 * hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/2")) {
      wpix1 = (wpix0 + 1) / 2;
      hpix1 = (hpix0 + 1) / 2;
   }
   
   if (strEqu(event,"b 1/3")) {
      wpix1 = (wpix0 + 2) / 3;
      hpix1 = (hpix0 + 2) / 3;
   }
   
   if (strEqu(event,"b 1/4")) {
      wpix1 = (wpix0 + 3) / 4;
      hpix1 = (hpix0 + 3) / 4;
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      wpix1 = int(wpct1 / 100.0 * wpix0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      hpix1 = int(hpct1 / 100.0 * hpix0 + 0.5);
   
   if (lock) {                                                             //  if lock, set other dimension
      if (event[0] == 'w')                                                 //    to preserve width/height ratio
         hpix1 = int(wpix1 * (1.0 * hpix0 / wpix0) + 0.5);
      if (event[0] == 'h') 
         wpix1 = int(hpix1 * (1.0 * wpix0 / hpix0) + 0.5);
   }
   
   hpct1 = 100.0 * hpix1 / hpix0;                                          //  set percents to match pixels
   wpct1 = 100.0 * wpix1 / wpix0;
   
   zdialog_stuff(zd,"wpix",wpix1);                                         //  synch. all widget values
   zdialog_stuff(zd,"hpix",hpix1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);
   
   return 1;
}


/**************************************************************************/

//  unbend an image
//  straighten curvature added by pano or improve perspective

int   unbend_dialog_event(zdialog* zd, const char *event);
int   unbend_dialog_compl(zdialog* zd, int zstat);
void  unbend_mousefunc();

int      unb_hx1, unb_hy1, unb_hx2, unb_hy2;                               //  dotted line end points
int      unb_vx1, unb_vy1, unb_vx2, unb_vy2;
int      unb_horz, unb_vert;                                               //  unbend values from dialog

void m_unbend()
{
   if (! pxb3) return;                                                     //  nothing to unbend
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   zdedit = zdialog_new(ZTX("unbend panorama image"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog|space=10");
   zdialog_add_widget(zdedit,"spin","spvert","vb1","0|30|1|0");
   zdialog_add_widget(zdedit,"spin","sphorz","vb1","-20|20|1|0");
   zdialog_add_widget(zdedit,"button","apply","vb1",Bapply);
   zdialog_add_widget(zdedit,"label","labvert","vb2",ZTX("vertical unbend"));
   zdialog_add_widget(zdedit,"label","labhorz","vb2",ZTX("horizontal unbend"));
   zdialog_add_widget(zdedit,"label","labcomp","vb2"," ");
   
   zdialog_resize(zdedit,260,0);
   zdialog_run(zdedit,unbend_dialog_event,unbend_dialog_compl);            //  run dialog, parallel

   unb_hx1 = 0;                                                            //  initial horz. axis
   unb_hy1 = hh3 / 2;
   unb_hx2 = ww3;
   unb_hy2 = hh3 / 2;
   
   unb_vx1 = ww3 / 2;                                                      //  initial vert. axis
   unb_vy1 = 0;
   unb_vx2 = ww3 / 2;
   unb_vy2 = hh3;
   
   Fredraw++;
   Mcapture = 1;
   mouseCBfunc = unbend_mousefunc;                                         //  connect mouse function  v.5.5
   return;
}


//  callback functions for unbend dialog event and completion

int unbend_dialog_event(zdialog *zd, const char *event)
{
   int            vert, horz;
   int            px3, py3, cx3, cy3;
   double         px1, py1, dispx, dispx2, dispy;
   pixel          pix1, pix3;

   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button
   zdialog_fetch(zd,"spvert",unb_vert);                                    //  get new unbend values
   zdialog_fetch(zd,"sphorz",unb_horz);

   zdialog_stuff(zdedit,"labcomp","(computing)");

   vert = int(unb_vert * 0.01 * hh3);                                      //  convert % to pixels
   horz = int(unb_horz * 0.005 * ww3);
   
   for (py3 = 0; py3 < hh3; py3++)                                         //  step through pxb3 pixels
   for (px3 = 0; px3 < ww3; px3++)
   {
      pix3 = ppix3 + py3 * rs3 + px3 * nch;                                //  output pixel

      cx3 = unb_vx1 + (unb_vx2 - unb_vx1) * py3 / hh3;                     //  center of unbend
      cy3 = unb_hy1 + (unb_hy2 - unb_hy1) * px3 / ww3;
      dispx = 2.0 * (px3 - cx3) / ww3;                                     //  -1.0 ..  0.0 .. +1.0 (roughly)
      dispy = 2.0 * (py3 - cy3) / hh3;                                     //  -1.0 ..  0.0 .. +1.0
      dispx2 = dispx * dispx - 0.5;                                        //  +0.5 .. -0.5 .. +0.5  curved

      px1 = px3 + dispx * dispy * horz;                                    //  input virtual pixel, x
      py1 = py3 - dispy * dispx2 * vert;                                   //  input virtual pixel, y
      pix1 = vpixel(ppix1,px1,py1,ww1,hh1,rs1);                            //  input virtual pixel

      if (pix1) {
         pix3[0] = pix1[0];                                                //  input pixel >> output pixel
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
      else pix3[0] = pix3[1] = pix3[2] = 0;
   }
   
   zdialog_stuff(zdedit,"labcomp"," ");

   mwpaint2();                                                             //  update window
   return 1;
}


int unbend_dialog_compl(zdialog *zd, int zstat)
{
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   mouseCBfunc = 0;                                                        //  disconnect mouse
   Mcapture = 0;
   LMclick = RMclick = 0;

   if (zstat != 1) {
      pull_image3();                                                       //  cancelled, restore image3
      return 0;
   }

   if (unb_vert || unb_horz) {
      Fmod3 = 1;                                                           //  image3 modified
      select_area_clear();                                                 //  select area invalidated  v.5.1
   }
   
   return 1;
}


//  unbend mouse function                                                  //  adjustable axes   v.39

void unbend_mousefunc()   
{
   const char  *close;
   int         dist1, dist2;

   if (LMdown)                                                             //  left mouse down
   {
      if (mpMx < 0.2 * wwM || mpMx > 0.8 * wwM) {                          //  check reasonable mouse position
         if (mpMy < 0.1 * hhM || mpMy > 0.9 * hhM) return;
      }
      else if (mpMy < 0.2 * hhM || mpMy > 0.8 * hhM) {
         if (mpMx < 0.1 * wwM || mpMx > 0.9 * wwM) return;
      }
      else return;

      close = "?";                                                         //  find the closest axis end-point
      dist1 = ww3 * hh3;

      dist2 = (mp3x-unb_hx1)*(mp3x-unb_hx1) 
            + (mp3y-unb_hy1)*(mp3y-unb_hy1);
      if (dist2 < dist1) {
         dist1 = dist2;
         close = "left";
      }

      dist2 = (mp3x-unb_hx2)*(mp3x-unb_hx2) 
            + (mp3y-unb_hy2)*(mp3y-unb_hy2);
      if (dist2 < dist1) {
         dist1 = dist2;
         close = "right";
      }

      dist2 = (mp3x-unb_vx1)*(mp3x-unb_vx1) 
            + (mp3y-unb_vy1)*(mp3y-unb_vy1);
      if (dist2 < dist1) {
         dist1 = dist2;
         close = "top";
      }

      dist2 = (mp3x-unb_vx2)*(mp3x-unb_vx2) 
            + (mp3y-unb_vy2)*(mp3y-unb_vy2);
      if (dist2 < dist1) {
         dist1 = dist2;
         close = "bottom";
      }
      
      if (strEqu(close,"left")) unb_hy1 = mp3y;                            //  set new axis end-point
      if (strEqu(close,"right")) unb_hy2 = mp3y;
      if (strEqu(close,"top")) unb_vx1 = mp3x;
      if (strEqu(close,"bottom")) unb_vx2 = mp3x;

      Fredraw = 1;
   }

   if (Fredraw) {  
      mwpaint();                                                           //  clear old lines
      draw_dotline(unb_hx1,unb_hy1,unb_hx2,unb_hy2);                       //  draw new axes dotted lines
      draw_dotline(unb_vx1,unb_vy1,unb_vx2,unb_vy2);
      Fredraw = 0;
   }

   return ;
}


/**************************************************************************/

//  image warp/distort - select image area and pull with mouse

int warp_dialog_event(zdialog *zd, const char *event);                     //  dialog event function
int warp_dialog_compl(zdialog *zd, int zstat);                             //  dialog completion function
int warp_image(int drx, int dry, int drw, int drh, int accum);             //  do the image warp
void * warp_thread(void *);

int         warp_thread_stat;
float       *warpx, *warpy;                                                //  memory of all displaced pixels
int         Nwarp, warpmem[4][100];                                        //  undo memory, last 100 warps
int         warpdx, warpdy, warpdw, warpdh;                                //  mouse drag rectangle

void m_warp()
{
   const char  *warp_message = 
         ZTX(" Select an area to warp using toolbar [select] button. \n"
               " Press [start warp] and pull area with mouse. \n"
               " Make multiple mouse pulls until satisfied. \n"
               " When finished, select another area or press [done]."); 
   
   int         px, py, ii;
   pthread_t   tid;

   if (! pxb3) return;                                                     //  no image
   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   warpx = (float *) zmalloc(ww3 * hh3 * sizeof(float));                   //  get memory for pixel displacements
   warpy = (float *) zmalloc(ww3 * hh3 * sizeof(float));
   
   for (px = 0; px < ww3; px++)                                            //  set all pixel displacements to zero
   for (py = 0; py < hh3; py++)
   {
      ii = py * ww3 + px;
      warpx[ii] = warpy[ii] = 0.0;
   }

   Nwarp = 0;                                                              //  no warp data
   warpdx = warpdy = warpdw = warpdh = 0;

   zdedit = zdialog_new(ZTX("warp image in selected area"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","lab1","dialog",warp_message,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");
   zdialog_add_widget(zdedit,"button","swarp","hb1",ZTX("start warp"),"space=5");
   zdialog_add_widget(zdedit,"button","uone","hb1",ZTX("undo last"),"space=5");
   zdialog_add_widget(zdedit,"button","uall","hb1",ZTX("undo all"),"space=5");

   zdialog_run(zdedit,warp_dialog_event,warp_dialog_compl);                //  run dialog
   pthread_create(&tid,0,warp_thread,0);                                   //  start computation thread
   return;
}


//  warp dialog event and completion callback functions

int warp_dialog_compl(zdialog * zd, int zstat)
{
   if (zstat != 1) pull_image3();                                          //  cancelled, restore image3
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   warp_thread_stat = 0;                                                   //  tell thread to exit
   gdk_window_set_cursor(dWin->window,0);                                  //  restore normal cursor
   zfree(warpx);                                                           //  release undo memory
   zfree(warpy);
   return 0;
}

int warp_dialog_event(zdialog * zd, const char *event)
{
   int         px, py, ii;
   pixel       pix1, pix3;

   if (strEqu(event,"swarp")) {                                            //  start warp
      if (! sa_exist) {
         zmessageACK(ZTX("must select area first"));                       //  v.5.5
         return 0;
      }
      gdk_window_set_cursor(dWin->window,dragcursor);                      //  set drag cursor
      warp_thread_stat = 2;                                                //  start warp thread
   }

   if (strEqu(event,"uone")) {
      if (Nwarp) {                                                         //  remove most recent warp from list
         ii = --Nwarp;
         warpdx = warpmem[0][ii];
         warpdy = warpmem[1][ii];
         warpdw = warpmem[2][ii];
         warpdh = warpmem[3][ii];
         warp_image(warpdx,warpdy,-warpdw,-warpdh,0);                      //  undo warp
         warp_image(warpdx,warpdy,-warpdw,-warpdh,1);
         mwpaint2(); 
      }
   }

   if (strEqu(event,"uall")) {                                             //  undo all warps
      for (ii = 0; ii < sa_npix; ii++)                                     //  process all enclosed pixels
      {
         px = sa_pix[ii].px;
         py = sa_pix[ii].py;
         pix1 = ppix1 + py * rs1 + px * nch;                               //  image1 pixel >> image3
         pix3 = ppix3 + py * rs3 + px * nch;
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }

      for (px = 0; px < ww3; px++)                                         //  reset pixel displacements
      for (py = 0; py < hh3; py++)
      {
         ii = py * ww3 + px;
         warpx[ii] = warpy[ii] = 0.0;
      }

      Nwarp = 0;                                                           //  erase undo memory
      mwpaint2(); 
   }

   return 1;
}


//  warp thread function

void * warp_thread(void *)                                                 //  modify image based on dialog inputs
{
   int      ii, warped = 0;

   warp_thread_stat = 1;
   func_busy++;

   while (warp_thread_stat == 1)                                           //  0: exit  1: idle  2: process
   {
      zsleep(0.1);
      if (kill_func) break;
   }
   
   while (true)
   {
      zsleep(0.1);
      if (warp_thread_stat == 0) break;
      if (kill_func) break;
      if (! sa_exist) continue;                                            //  select area erased

      while (md3x1 || md3y1)                                               //  mouse drag underway
      {
         warpdx = md3x1;                                                   //  drag origin, image3 coordinates
         warpdy = md3y1;
         warpdw = md3x2 - md3x1;                                           //  drag increment
         warpdh = md3y2 - md3y1;
         warp_image(warpdx,warpdy,warpdw,warpdh,0);                        //  warp image
         warped = 1;
         mwpaint2(); 
      }

      if (warped)                                                          //  drag done, warp done
      {
         warped = 0;
         warp_image(warpdx,warpdy,warpdw,warpdh,1);                        //  accumulate sum of all warps

         if (Nwarp == 100) {                                               //  if full, throw away oldest
            Nwarp = 99;
            for (ii = 0; ii < Nwarp; ii++)
            {
               warpmem[0][ii] = warpmem[0][ii+1];
               warpmem[1][ii] = warpmem[1][ii+1];
               warpmem[2][ii] = warpmem[2][ii+1];
               warpmem[3][ii] = warpmem[3][ii+1];
            }
         }

         ii = Nwarp;
         warpmem[0][ii] = warpdx;                                          //  save latest warp
         warpmem[1][ii] = warpdy;
         warpmem[2][ii] = warpdw;
         warpmem[3][ii] = warpdh;
         Nwarp++;
         Fmod3 = 1;                                                        //  image3 modified
      }
   }
   
   func_busy--;
   return 0;
}


//  warp image within select area according to mouse position and displacement

int warp_image(int drx, int dry, int drw, int drh, int accum)
{
   int            ii, jj, px, py;
   double         ddx, ddy, dpe, dpm, mag, dispx, dispy;
   pixel          pix1, pix3;
   
   for (ii = 0; ii < sa_npix; ii++)                                        //  process all enclosed pixels
   {
      px = sa_pix[ii].px;
      py = sa_pix[ii].py;
      dpe = sa_pix[ii].dist;                                               //  distance from area edge

      ddx = (px - drx);                                                    //  distance from drag origin
      ddy = (py - dry);
      dpm = sqrt(ddx*ddx + ddy*ddy);

      if (dpm < 1) mag = 1;
      else mag = dpe / (dpe + dpm);                                        //  magnification, 0...1
      mag = mag * mag;

      dispx = -drw * mag;                                                  //  warp = drag * magnification
      dispy = -drh * mag;
      
      jj = py * ww3 + px;

      if (accum) {                                                         //  mouse drag done, accumulate warp
         warpx[jj] += dispx;
         warpy[jj] += dispy;
         continue;
      }

      dispx += warpx[jj];                                                  //  add this warp to prior
      dispy += warpy[jj];

      pix1 = vpixel(ppix1,px+dispx,py+dispy,ww1,hh1,rs1);                  //  input virtual pixel
      if (! pix1) continue;
      pix3 = ppix3 + py * rs3 + px * nch;                                  //  output pixel
      pix3[0] = pix1[0];
      pix3[1] = pix1[1];
      pix3[2] = pix1[2];
   }

   return 1;
}


/**************************************************************************/

//  Make an HDR (high dynamic range) image from two images of the same subject 
//  with different exposure levels. The HDR image has expanded gradations of 
//  intensity (visibility of detail) in both the brightest and darkest areas.

void *   HDR_thread(void *);
void     HDR_combine_images(int ww);
void     HDR_level_adjust();
void *   HDR_combine_thread(void *);

double   hdrweight[9];                                                     //  image weights for 8 zones
int      HDR_combine_thread_stat;


void m_HDR()
{
   pthread_t   tid;

   if (! pxb3) return;                                                     //  no image

   if (file2) zfree(file2);
   file2 = zgetfile(ZTX("select image to combine"),file1,"open");          //  get 2nd HDR image in pxb2
   if (! file2) return;
   pxb2 = load_pixbuf(file2);                                              //  v.33
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   if (ww2 != ww3) goto err_nomatch;                                       //  validate compatibility
   if (hh2 != hh3) goto err_nomatch;
   if (rs2 != rs3) goto err_nomatch;

   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   pthread_create(&tid,0,HDR_thread,0);                                    //  start thread to combine images
   return;                                                                 //    pxb1 + pxb2 >> pxb3

err_nomatch:
   zmessageACK(ZTX("2nd file is not compatible with 1st"));
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3

void * HDR_thread(void *)
{
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   int         firstpass = 1, lastpass = 0;

   func_busy++;

   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 1;                                                         //  highlight alignment pixels

   Mscale = 0;                                                             //  scale image to window
   Fscale = 1;                                                             //  even if > 100%
   
   fullSize = ww1;                                                         //  full image size
   if (hh1 > ww1) fullSize = hh1;                                          //  (largest dimension)

   alignSize = 140;                                                        //  initial alignment image size
   if (alignSize > fullSize) alignSize = fullSize;

   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample

   xoff = yoff = toff = 0;                                                 //  initial offsets = 0
   blend = 0;                                                              //  blend region is 100%

   yst1 = yst2 = yst1B = yst2B = 0;                                        //  disable stuff only for pano
   lens_curve = 0;

   while (true)                                                            //  next alignment stage / image size
   {   
      ww1A = ww1 * alignSize / fullSize;                                   //  align width, height in same ratio
      hh1A = hh1 * alignSize / fullSize;
      ww2A = ww1A;
      hh2A = hh1A;

      if (! lastpass) 
      {
         zlock();
         pixbuf_free(pxb1A);                                               //  create alignment images
         pixbuf_free(pxb2A);
         pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,BILINEAR);
         pixbuf_test(1A);
         pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,BILINEAR);
         pixbuf_test(2A);
         pixbuf_free(pxb3);
         pxb3 = pixbuf_copy(pxb1A);                                        //  make pxb3 compatible
         pixbuf_test(3);

         pixbuf_poop(1A)                                                   //  get pxb1A attributes
         pixbuf_poop(2A)                                                   //  get pxb2A attributes
         pixbuf_poop(3)                                                    //  get pxb3 attributes
         zunlock();

         get_overlap_region(pxb1A,pxb2A);                                  //  get image overlap area
         get_Bratios(pxb1A,pxb2A);                                         //  get image brightness ratios
         set_colormatch(1);                                                //  compute color matching factors
         flag_edge_pixels(pxb1A);                                          //  flag high-contrast pixels
      }

      xylim = 2;                                                           //  search range from prior stage:
      xystep = 1;                                                          //    -2 -1 0 +1 +2 pixels

      if (firstpass) xylim = 0.05 * alignSize;                             //  1st stage search range, huge

      if (lastpass) {
         xylim = 1;                                                        //  final stage search range:
         xystep = 0.5;                                                     //    -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians    v.5.3
      tstep = xystep / alignSize / 2;                                      //  theta step size              v.5.3

      HDR_combine_images(0);                                               //  combine images >> pxb3
      mwpaint2();                                                          //  update window

      matchlev = match_images(pxb1A,pxb2A);                                //  set base match level
      matchB = matchlev;                                                   //   = best matching offsets

      xfL = xoff - xylim;
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      
      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test all offset dimensions
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1A,pxb2A);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
         update_status_bar();
         if (kill_func) goto HDR_exit;
      }

      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;

      HDR_combine_images(0);                                               //  combine images >> pxb3
      mwpaint2();                                                          //  update window
     
      firstpass = 0;
      if (lastpass) break;                                                 //  done

      if (alignSize == fullSize) {                                         //  full size image was aligned
         lastpass++;                                                       //  one more pass
         continue;
      }

      double R = alignSize;
      alignSize = 2 * alignSize;                                           //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      R = alignSize / R;                                                   //  ratio of new / old image size
      xoff = R * xoff;                                                     //  adjust offsets for image size
      yoff = R * yoff;
   }

   HDR_level_adjust();                                                     //  dialog, adjust input image weights

HDR_exit:

   Nalign = 0;                                                             //  alignment done
   if (kill_func) pull_image3();                                           //  killed

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   zunlock();

   Fscale = 0;                                                             //  reset forced image scaling
   mwpaint2();                                                             //  repaint window (pxb3)

   func_busy--;
   return 0;
}


//  combine images pxb1A and pxb2A using weights in hdrweight[9]
//  hdrweight[0]-[1] for darkest zone, hdrweight[7]-[8] for brightest zone
//  output is in pxb3

void HDR_combine_images(int weight)
{
   int      px3, py3, ii;
   double   px1, py1, px2, py2;
   double   sintf = sin(toff), costf = cos(toff);
   double   br1, br2, brm;
   pixel    pix1, pix2, pix3;

   for (py3 = 0; py3 < hh1A; py3++)                                        //  step through pxb1A pixels
   for (px3 = 0; px3 < ww1A; px3++)
   {
      px1 = costf * px3 - sintf * (py3 - yoff);                            //  pxb1A pixel, after offsets   v.5.3 
      py1 = costf * py3 + sintf * (px3 - xoff);
      pix1 = vpixel(ppix1A,px1,py1,ww1A,hh1A,rs1A);

      px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                   //  corresponding pxb2A pixel
      py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
      pix2 = vpixel(ppix2A,px2,py2,ww2A,hh2A,rs2A);

      pix3 = ppix3 + py3 * rs3 + px3 * nch;                                //  output pxb3 pixel

      if (! pix1 || ! pix2) {                                              //  if non-overlapping pixel,   v.5.3
         pix3[0] = pix3[1] = pix3[2] = 0;                                  //    set output pixel black
         continue;
      }
      
      if (weight) {
         br1 = pix1[0] + pix1[1] + pix1[2];                                //  image 1 pixel brightness
         br2 = pix2[0] + pix2[1] + pix2[2];                                //  image 2
         brm = (br1 + br2) / 2;                                            //  mean, 0 to 765

         ii = int(brm / 96);                                               //  brightness band, 0 .. 7
         br1 = hdrweight[ii];                                              //  weight at low end of band
         br2 = hdrweight[ii+1];                                            //  weight at high end of band
         brm = br1 + (br2 - br1) * (brm - 96 * ii) / 96;                   //  interpolate within band
         
         br1 = brm / 80.0;                                                 //  image 1 weight: 0 .. 1.0
         br2 = 1.0 - br1;                                                  //  image 2 weight
         
         pix3[0] = int(pix1[0] * br1 + pix2[0] * br2);                     //  build output pixel
         pix3[1] = int(pix1[1] * br1 + pix2[1] * br2);
         pix3[2] = int(pix1[2] * br1 + pix2[2] * br2);
      }

      else {
         pix3[0] = (pix1[0] + pix2[0]) / 2;                                //  output pixel is simple average
         pix3[1] = (pix1[1] + pix2[1]) / 2;
         pix3[2] = (pix1[2] + pix2[2]) / 2;
      }

      if (showRedpix && pix2) {                                            //  highlight alignment pixels
         ii = py3 * ww1A + px3;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   return;
}


//  dialog to adjust image brightness levels
//  pxb1A + pxb2A >> pxb3 under control of scale bars

void HDR_level_adjust()
{
   int      HDR_level_dialog_event(zdialog *zd, const char *name);

   const char  *weightmess = ZTX("\n image weights per brightness band \n");
   int         ii, zstat;
   double      bravg, wval;
   char        vsname[4] = "vs0";
   pthread_t   tid;
   
   zdedit = zdialog_new(ZTX("HDR image weights"),mWin,Bdone,Bcancel,null);
   zdialog_add_widget(zdedit,"label","labt","dialog",weightmess);
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"expand");
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"expand|space=10");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"expand|space=10");
   zdialog_add_widget(zdedit,"vbox","vb3","hb1",0,"expand|space=10");
   zdialog_add_widget(zdedit,"label","lab11","vb1",ZTX("input image 2"));
   zdialog_add_widget(zdedit,"label","lab12","vb1",0,"expand");
   zdialog_add_widget(zdedit,"label","lab13","vb1",ZTX("input image 1"));
   zdialog_add_widget(zdedit,"label","lab14","vb1",ZTX("output image"));
   zdialog_add_widget(zdedit,"label","lab21","vb2"," 100 % ");
   zdialog_add_widget(zdedit,"label","lab22","vb2"," 50/50 ","expand");
   zdialog_add_widget(zdedit,"label","lab23","vb2"," 100 % ");
   zdialog_add_widget(zdedit,"label","lab24","vb2");
   zdialog_add_widget(zdedit,"hbox","hb2","vb3",0,"expand|space=5");
   zdialog_add_widget(zdedit,"vscale","vs0","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs1","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs2","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs3","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs4","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs5","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs6","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs7","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"vscale","vs8","hb2","0|80|1|0|");
   zdialog_add_widget(zdedit,"hbox","hb3","vb3");
   zdialog_add_widget(zdedit,"label","lab31","hb3",ZTX("darker areas"));
   zdialog_add_widget(zdedit,"label","lab32","hb3",0,"expand");
   zdialog_add_widget(zdedit,"label","lab33","hb3",ZTX("brighter areas"));
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=15");
   zdialog_add_widget(zdedit,"label","lab41","hb4",0,"space=50");
   zdialog_add_widget(zdedit,"button","100/0","hb4","100/0");
   zdialog_add_widget(zdedit,"button","0/100","hb4","0/100");
   zdialog_add_widget(zdedit,"button","50/50","hb4","50/50");
   zdialog_add_widget(zdedit,"button","+/-","hb4","  +/-  ");
   zdialog_add_widget(zdedit,"button","-/+","hb4","  -/+  ");

   bravg = 0;                                                              //  get mean brightness ratio
   for (ii = 0; ii < 256; ii++)
      bravg = bravg + Bratios2[0][ii] + Bratios2[1][ii] + Bratios2[2][ii];
   bravg = bravg / 256 / 3;
   
   for (ii = 0; ii < 9; ii++)                                              //  initial ramp: use brighter
   {                                                                       //    image for darker bands, etc.
      vsname[2] = '0' + ii;
      if (bravg < 1) wval = 10 * ii;                                       //  ramp up
      else  wval = 80 - 10 * ii;                                           //  ramp down
      zdialog_stuff(zdedit,vsname,wval);
      hdrweight[ii] = wval;
   }

   showRedpix = 0;                                                         //  stop highlights
   HDR_combine_images(1);                                                  //  combine final images >> pxb3
   mwpaint2();                                                             //  update window
   
   pthread_create(&tid,0,HDR_combine_thread,0);                            //  start compute thread

   zdialog_resize(zdedit,0,400);
   zstat = zdialog_run(zdedit,HDR_level_dialog_event,0);                   //  run dialog - blocking mode
   zdialog_free(zdedit);
   zdedit = null;

   if (zstat == 1) {
      Fmod3 = 1;                                                           //  done, image3 modified
      select_area_clear();                                                 //  select area invalidated  v.5.1
   }
   else  pull_image3();                                                    //  cancel, restore image3
   
   HDR_combine_thread_stat = 0;                                            //  tell thread to exit
   return;
}


//  event callback function for HDR image level adjustment

int HDR_level_dialog_event(zdialog *zd, const char *event)
{
   char        vsname[4] = "vs0";
   double      vsval;
   
   for (int ii = 0; ii < 9; ii++)                                          //  process dialog sliders
   {
      vsname[2] = '0' + ii;
      zdialog_fetch(zd,vsname,vsval);                                      //  get slider ii value

      if (strEqu(event,"100/0")) vsval = 0;                                //  button updates
      if (strEqu(event,"0/100")) vsval = 80;
      if (strEqu(event,"50/50")) vsval = 40;
      if (strEqu(event,"+/-"))   vsval += ii - 4;
      if (strEqu(event,"-/+"))   vsval += 4 - ii;

      if (vsval < 0) vsval = 0;                                            //  stay within limits
      if (vsval > 80) vsval = 80;

      zdialog_stuff(zd,vsname,vsval);                                      //  update dialog slider
      hdrweight[ii] = vsval;                                               //  and corresp. weight
   }

   HDR_combine_thread_stat = 2;                                            //  tell thread to update
   return 1;
}


//  thread to update image runs asynchronously to dialog updates

void * HDR_combine_thread(void *)
{
   HDR_combine_thread_stat = 1;
   
   while (true)
   {
      while (HDR_combine_thread_stat == 1) zsleep(0.1);                    //  0: exit  1: idle  2: process
      if (HDR_combine_thread_stat == 0) return 0;
      HDR_combine_thread_stat = 1;
      HDR_combine_images(1);                                               //  combine images >> pxb3
      mwpaint2();                                                          //  update window
   }
}


/**************************************************************************/

//  panorama function - stitch two images together

void *   pano_thread(void *);                                              //  thread function
void     pano_get_align_images();                                          //  rescale images for alignment
void     pano_curve_images(int newf, int strf);                            //  curve/stretch image during align
void     pano_prealign();                                                  //  manual pre-align
void     pano_final_adjust();                                              //  dialog, adjust color and brightness
void     pano_show_images(int fcolor);                                     //  show combined images

int      pano_align_stat;                                                  //  alignment dialog status
int      pxM, pxmL, pxmH;                                                  //  pano blend stripe
int      pano_automatch;                                                   //  auto color matching on/off
int      pano_adjust_stat;                                                 //  adjust dialog status


void m_pano()
{
   pthread_t   tid;

   if (! pxb3) return;                                                     //  no 1st image

   if (file2) zfree(file2);
   file2 = zgetfile(ZTX("select image to combine"),file1,"open");          //  get 2nd pano image in pxb2
   if (! file2) return;
   pxb2 = load_pixbuf(file2);                                              //  v.33
   if (! pxb2) goto err_nomatch;
   pixbuf_poop(2)

   push_image3();                                                          //  save image3 undo point

   pixbuf_free(pxb1);                                                      //  image3 >> image1 (with prior mods)
   pxb1 = pixbuf_copy(pxb3);                                               //  (image1 = base image for new mods)
   pixbuf_test(1);
   pixbuf_poop(1)

   pthread_create(&tid,0,pano_thread,0);                                   //  start thread to combine images
   return;                                                                 //    pxb1 + pxb2 >> pxb3

err_nomatch:
   zmessageACK(ZTX("2nd file is not compatible with 1st"));
   if (file2) zfree(file2);
   file2 = 0;
   pixbuf_free(pxb2);
   pxb2 = 0;
   return;
}


//  Thread function for combining pxb1 + pxb2 >> pxb3.

void * pano_thread(void *)
{
   int         firstpass = 1, lastpass = 0;
   double      minblend, alignR;
   double      xfL, xfH, yfL, yfH, tfL, tfH;
   double      yst1L, yst1H, yst2L, yst2H;
   
   func_busy++;

   Radjust = Gadjust = Badjust = 1.0;                                      //  no manual color adjustments
   Nalign = 1;                                                             //  alignment in progress
   showRedpix = 0;                                                         //  no pixel highlight (yet)
   pixsamp = 1000 * int(parm_pixel_sample_size);                           //  no. pixels to sample
   pano_align_stat = pano_adjust_stat = 0;

   Mscale = 0;                                                             //  scale image to window
   Fscale = 1;                                                             //  even if > 100%

   fullSize = hh2;                                                         //  full size = image2 height
   alignSize = int(parm_pano_prealign_size);                               //  prealign image size
   pano_get_align_images();                                                //  get prealign images

   xoff = xoffB = 0.8 * ww1A;                                              //  initial x offset (20% overlap)
   yoff = yoffB = toff = toffB = 0;                                        //  initial y and theta offsets
   yst1 = yst1B = yst2 = yst2B = 0;                                        //  initial y-stretch offsets
   lens_mm = parm_pano_lens_mm;                                            //  initial image curvature
   lens_bow = parm_pano_lens_bow;                                          //  initial image bow (barrel dist.)
   blend = ww1A - xoff;                                                    //  initial blend width = overlap area
   
   pano_curve_images(1,0);                                                 //  make curved alignment images
   get_overlap_region(pxb1C,pxb2C);                                        //  get image overlap area
   pano_show_images(0);                                                    //  show combined images

   pano_prealign();                                                        //  manual pre-align - get new offsets
   if (! pano_align_stat) goto pano_exit;                                  //  user canceled

   blend = ww1A - xoff;                                                    //  new blend from pre-align
   if (blend < 20) {
      zmessageACK(ZTX("too little overlap, cannot align"));
      pano_align_stat = 0;
      goto pano_exit;
   }

   showRedpix = 1;                                                         //  highlight alignment pixels

   while (true)
   {
      alignR = alignSize;
      if (firstpass) alignSize = 140;
      else if (lastpass) alignSize = fullSize;
      else  alignSize = int(parm_pano_align_size_increase * alignSize);    //  next larger image size
      if (alignSize > 0.8 * fullSize) alignSize = fullSize;                //  if near goal, jump to it now
      alignR = alignSize / alignR;                                         //  ratio of new / old image size

      xoff = alignR * xoff;                                                //  adjust offsets for new image size
      yoff = alignR * yoff;
      toff = toff;
      yst1 = alignR * yst1;
      yst2 = alignR * yst2;
      blend = alignR * blend;

      xoffB = xoff;                                                        //  initial offsets = best so far
      yoffB = yoff;
      toffB = toff;
      yst1B = yst1;
      yst2B = yst2;

      if (! firstpass) {
         blend = blend * parm_pano_blend_reduction;                        //  reduce image comparison width
         minblend = 0.01 * parm_pano_minimum_blend * alignSize;
         if (blend < minblend) blend = minblend;                           //  keep above minimum
         if (blend > 0.2 * alignSize) blend = 0.2 * alignSize;             //  keep below 20% of image
      }
      
      xylim = 3;                                                           //  +/- search range, centered on
      xystep = 1;                                                          //    results from prior stage
      
      if (firstpass) {
         xylim = alignSize * 0.04;                                         //  tolerate 4 % error in pre-alignment
         xystep = 0.8;
      }

      if (lastpass) {
         xylim = 1;                                                        //  final stage pixel search steps
         xystep = 0.5;                                                     //   -1.0 -0.5 0.0 +0.5 +1.0
      }

      tlim = xylim / alignSize / 2;                                        //  theta max offset, radians  v.5.3
      tstep = xystep / alignSize / 2;                                      //  theta step size            v.5.3

      if (! lastpass) {
         pano_get_align_images();                                          //  get new alignment images
         pano_curve_images(1,0);                                           //  make new curved alignment images
      }

      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get image bright. ratios in overlap 
      set_colormatch(1);                                                   //  compute color matching factors
      flag_edge_pixels(pxb1C);                                             //  flag high-contrast pixels in blend

      pano_show_images(0);                                                 //  show combined images
      
      xfL = xoff - xylim;                                                  //  set x/y/t search ranges, step sizes
      xfH = xoff + xylim + xystep/2;
      yfL = yoff - xylim;
      yfH = yoff + xylim + xystep/2;
      tfL = toff - tlim;
      tfH = toff + tlim + tstep/2;

      matchB = match_images(pxb1C,pxb2C);                                  //  set base match level

      for (xoff = xfL; xoff < xfH; xoff += xystep)                         //  test x, y, theta offsets
      for (yoff = yfL; yoff < yfH; yoff += xystep)                         //    in all possible combinations
      for (toff = tfL; toff < tfH; toff += tstep)
      {
         matchlev = match_images(pxb1C,pxb2C);
         if (sigdiff(matchlev,matchB,0.00001) > 0) {                       //  remember best alignment and offsets
            matchB = matchlev;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
         }

         Nalign++;
         update_status_bar();
         if (kill_func) goto pano_exit;
      }
      
      xoff = xoffB;                                                        //  recover best offsets
      yoff = yoffB;
      toff = toffB;
      
      pano_show_images(0);                                                 //  update window

      if (parm_pano_image_stretch && ! firstpass)                          //  do image2 y-stretch alignment
      {
         ystlim = hh2A * 0.001 * parm_pano_image_stretch;                  //  v.32
         yststep = 0.5;

         yst1L = yst1 - ystlim;                                            //  y-stretch search range, upper half
         yst1H = yst1 + ystlim + yststep/2;

         for (yst1 = yst1L; yst1 < yst1H; yst1 += yststep)                 //  search upper image half
         {
            pano_curve_images(0,1);                                        //  curve and y-stretch
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {                    //  remember best align and y-stretch 
               matchB = matchlev;
               yst1B = yst1;
            }

            Nalign++;
            update_status_bar();
            if (kill_func) goto pano_exit;
         }
         
         yst1 = yst1B;                                                     //  restore best y-stretch offset
         pano_curve_images(0,1);

         yst2L = yst2 - ystlim;                                            //  lower half search range
         yst2H = yst2 + ystlim + yststep/2;
         
         for (yst2 = yst2L; yst2 < yst2H; yst2 += yststep)                 //  search lower half
         {
            pano_curve_images(0,2);
            matchlev = match_images(pxb1C,pxb2C);
            if (sigdiff(matchlev,matchB,0.00001) > 0) {
               matchB = matchlev;
               yst2B = yst2;
            }

            Nalign++;
            update_status_bar();
            if (kill_func) goto pano_exit;
         }
         
         yst2 = yst2B;                                                     //  restore best offset
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  update window
      }

      firstpass = 0;
      if (lastpass) break;
      if (alignSize == fullSize) lastpass = 1;                             //  one more pass, reduced step size
   }

   pano_final_adjust();                                                    //  user color / brightness adjust

pano_exit:

   Nalign = 0;                                                             //  alignment done
   
   if (! kill_func && pano_align_stat && pano_adjust_stat) {               //  all done OK
      Fmod3 = 1;                                                           //  image3 modified
      select_area_clear();                                                 //  select area invalidated  v.5.1
   }
   else  pull_image3();                                                    //  killed or canceled

   zlock();
   if (file2) zfree(file2);                                                //  free 2nd input file
   file2 = 0;
   pixbuf_free(pxb2);                                                      //  free 2nd input image
   pixbuf_free(pxb1A);                                                     //  free alignment images
   pixbuf_free(pxb2A);
   pixbuf_free(pxb1C);
   pixbuf_free(pxb2C);
   zunlock();

   Fscale = 0;                                                             //  reset forced image scaling
   Mscale = 0;                                                             //  scale to window
   mwpaint2(); 
   
   func_busy--;
   return 0;
}


//  scale images into new pixbufs for alignment
//  scaled images in pxb1A/2A, scaled and curved images in pxb1C/2C

void pano_get_align_images()
{
   ww1A = ww1 * alignSize / fullSize;                                      //  size of alignment images
   hh1A = hh1 * alignSize / fullSize;
   ww2A = ww2 * alignSize / fullSize;
   hh2A = hh2 * alignSize / fullSize;

   zlock();
   pixbuf_free(pxb1A);                                                     //  create alignment images
   pixbuf_free(pxb2A);
   pxb1A = pixbuf_scale_simple(pxb1,ww1A,hh1A,BILINEAR);
   pixbuf_test(1A);
   pxb2A = pixbuf_scale_simple(pxb2,ww2A,hh2A,BILINEAR);
   pixbuf_test(2A);
   pixbuf_poop(1A)                                                         //  get pxb1A attributes
   pixbuf_poop(2A)                                                         //  get pxb2A attributes
   zunlock();
   return;
}


//  curve and stretch images according to lens_mm, lens_bow, yst1, yst2
//  pxb1A, pxb2A  >>  pxb1C, pxb2C
//  strf:  0: curve both images, y-stretch both halves of image2
//         1: curve image2 only, y-stretch upper half only
//         2: curve image2 only, y-stretch lower half only
//  v.32: curve entire image instead of right or left half

void pano_curve_images(int newf, int strf)
{
   void     pano_interp_curve();                                           //  convert lens mm to % curve

   pixel       pix, pixc;
   int         pxc, pyc, pxcL, pxcH, pycL, pycH;
   double      px, py, yst;
   double      xdisp, ydisp, xshrink, yshrink;
   double      costf = cos(toff), sintf = sin(toff);
   double      pcurve, pyadj, pbow, R;

   if (newf)                                                               //  new alignment images are present
   {
      zlock();
      pixbuf_free(pxb1C);                                                  //  make new curved alignment images
      pxb1C = pixbuf_copy(pxb1A);
      pixbuf_test(1C);
      pixbuf_free(pxb2C);
      pxb2C = pixbuf_copy(pxb2A);
      pixbuf_test(2C);
      pixbuf_poop(1C)                                                      //  get attributes
      pixbuf_poop(2C)
      zunlock();
   }
   
   pano_interp_curve();                                                    //  convert lens_mm to curve
   pcurve = lens_curve * 0.01 * ww2C;                                      //  curve % to pixels
   pyadj = 1.41;                                                           //  y adjustment factor
   pbow = 0.01 * ww2C * lens_bow;                                          //  lens_bow % to pixels

   R = 1.0 * hh2C / ww2C;
   if (R > 1) pcurve = pcurve / R / R;                                     //  adjust for vertical format

   if (strf == 0 && ww2C > 0.7 * ww1C)                                     //  no curve if 3rd+ image in pano
   {   
      for (pyc = 0; pyc < hh1C; pyc++)                                     //  curve image1 (both halves)
      for (pxc = 0; pxc < ww1C; pxc++)
      {
         xdisp = (pxc - ww1C/2.0) / (ww1C/2.0);                            //  -1 .. 0 .. +1
         ydisp = (pyc - hh1C/2.0) / (ww1C/2.0);                            //  -h/w .. 0 .. +h/w
         xshrink = xdisp * xdisp * xdisp;
         yshrink = pyadj * ydisp * xdisp * xdisp;
         
         px = pxc + pcurve * xshrink;                                      //  x shrink pixels
         py = pyc + pcurve * yshrink;                                      //  y shrink pixels
         px -= pbow * xdisp * ydisp * ydisp;                               //  x bow pixels

         pix = vpixel(ppix1A,px,py,ww1A,hh1A,rs1A);                        //  virtual pixel in uncurved image
         pixc = ppix1C + pyc * rs1C + pxc * nch;                           //  destination pixel in curved image
         if (pix) {  
            pixc[0] = pix[0];
            pixc[1] = pix[1];
            pixc[2] = pix[2];
         }
         else pixc[0] = pixc[1] = pixc[2] = 0;
      }
   }
   
   pycL = 0;
   pycH = hh2C;
   if (strf == 1) pycH = hh2C / 2;                                         //  upper half only
   if (strf == 2) pycL = hh2C / 2;                                         //  lower half only

   pxcL = 0;
   pxcH = ww2C;

   if (strf) {                                                             //  during y-stretch alignment,
      pxcL = int((pxmL - xoff) * costf + (pyL - yoff) * sintf);            //    stay within blend stripe
      if (pxcL < 0) pxcL = 0;
      pxcH = int((pxmH - xoff) * costf + (pyH - yoff) * sintf);
      if (pxcH > ww2C/2) pxcH = ww2C/2;
   }

   for (pyc = pycL; pyc < pycH; pyc++)                                     //  curve and y-stretch image2
   for (pxc = pxcL; pxc < pxcH; pxc++)
   {
      xdisp = (pxc - ww2C/2.0) / (ww2C/2.0);
      ydisp = (pyc - hh2C/2.0) / (ww2C/2.0);
      xshrink = xdisp * xdisp * xdisp;
      yshrink = pyadj * ydisp * xdisp * xdisp;

      px = pxc + pcurve * xshrink;
      py = pyc + pcurve * yshrink;
      px -= pbow * xdisp * ydisp * ydisp;

      if (pyc < hh2C / 2) yst = yst1;                                      //  do y-stretch
      else yst = yst2;
      if (yst) py += yst * ydisp;                                          //  linear better than quadratic

      pix = vpixel(ppix2A,px,py,ww2A,hh2A,rs2A);
      pixc = ppix2C + pyc * rs2C + pxc * nch;
      if (pix) {  
         pixc[0] = pix[0];
         pixc[1] = pix[1];
         pixc[2] = pix[2];
      }
      else pixc[0] = pixc[1] = pixc[2] = 0;
   }

   return;
}


//  convert lens focal length (mm) to image percent curve

void pano_interp_curve()
{
   double curvetab[] = { 25, 10.3, 26, 8.4, 28, 6.78, 32, 5.26,            //  25mm = 10.3 % etc.   v.32
                         40, 3.42, 48, 2.32, 60, 1.7, 80, 1.35, 
                         100, 1.2, 200, 1.1, 0 };
   int      ii;
   double   x0, x1, y0, y1;
   
   for (ii = 0; curvetab[ii]; ii += 2)                                     //  find >= table entry
      if (lens_mm <= curvetab[ii]) break;

   if (ii == 0) {                                                          //  1st entry
      lens_curve = curvetab[1];
      return;
   }

   if (curvetab[ii] == 0) {                                                //  EOL
      lens_curve = 1.0;
      return;
   }
   
   x0 = curvetab[ii-2];                                                    //  interpolate
   x1 = curvetab[ii];
   y0 = curvetab[ii-1];
   y1 = curvetab[ii+1];
   lens_curve = y0 + (y1 - y0) * (lens_mm - x0) / (x1 - x0);

   return;
}


//  perform manual pre-align of image2 to image1
//  return offsets: xoff, yoff, toff, lens_mm, lens_bow

void pano_prealign()   
{  
   void     pano_autolens();                                               //  optimize lens parameters

   int   pano_prealign_event(zdialog *zd, const char *event);              //  dialog event function
   int   pano_prealign_compl(zdialog *zd, int zstat);                      //  dialog completion function

   int         mx0, my0, mx, my, key;                                      //  mouse drag origin, position
   double      lens_mm0, lens_bow0;
   double      mlever = 1.0 / parm_pano_mouse_leverage;
   double      dtoff;

   const char  *align_mess = ZTX("drag right image into rough alignment with left \n"
                                      " to rotate, drag right edge up or down");
   const char  *proceed_mess = ZTX("merge the images together");
   const char  *search_mess = ZTX("auto-search lens mm and bow");

   zdedit = zdialog_new(ZTX("pre-align images"),mWin,Bcancel,null);          //  dialog for alignment
   zdialog_add_widget(zdedit,"label","lab1","dialog",align_mess,"space=5");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=5");             //  lens mm, 25-200 mm
   zdialog_add_widget(zdedit,"spin","spmm","hb1","25|200|0.1|35","space=5");
   zdialog_add_widget(zdedit,"label","labmm","hb1",ZTX("lens mm"));
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");             //  lens bow, -9 to +9 %
   zdialog_add_widget(zdedit,"spin","spbow","hb2","-9|9|0.01|0","space=5");
   zdialog_add_widget(zdedit,"label","labbow","hb2",ZTX("lens bow"));
   zdialog_add_widget(zdedit,"hbox","hb3","dialog",0,"space=5");             //  next button
   zdialog_add_widget(zdedit,"button","proceed","hb3",Bproceed,"space=5");
   zdialog_add_widget(zdedit,"label","labproceed","hb3",proceed_mess);
   zdialog_add_widget(zdedit,"hbox","hb4","dialog",0,"space=5");             //  search button     v.32
   zdialog_add_widget(zdedit,"button","search","hb4",Bsearch,"space=5");
   zdialog_add_widget(zdedit,"label","labsearch","hb4",search_mess);
   zdialog_stuff(zdedit,"spmm",lens_mm);                                     //  pre-load current data
   zdialog_stuff(zdedit,"spbow",lens_bow);

   zdialog_run(zdedit,pano_prealign_event,pano_prealign_compl);            //  run dialog

   lens_mm0 = lens_mm;                                                     //  to detect changes
   lens_bow0 = lens_bow;
   pano_align_stat = 1;                                                    //  status = pre-align in progress
   mx0 = my0 = 0;                                                          //  no drag in progress

   while (pano_align_stat == 1)                                            //  loop while in align status
   {
      if (kill_func) break;

      if (Fautolens) {
         pano_autolens();                                                  //  get lens parameters
         if (pano_align_stat != 1) break;
         zdialog_stuff(zdedit,"spmm",lens_mm);                             //  update dialog
         zdialog_stuff(zdedit,"spbow",lens_bow);
      }
         
      if (lens_mm != lens_mm0 || lens_bow != lens_bow0) {                  //  change in lens parameters
         lens_mm0 = lens_mm;
         lens_bow0 = lens_bow;
         pano_curve_images(0,0);
         pano_show_images(0);                                              //  show combined images
      }
      
      if (KBkey) {                                                         //  KB input
         key = KBkey;
         KBkey = 0;
         if (key == GDK_Left)  xoff -= 0.2;                                //  tweak alignment offsets
         if (key == GDK_Right) xoff += 0.2;
         if (key == GDK_Up)    yoff -= 0.2;
         if (key == GDK_Down)  yoff += 0.2;
         if (key == GDK_r)     toff += 0.0002;
         if (key == GDK_l)     toff -= 0.0002;
         if (key == GDK_p) printf(" %.1f %.2f \n",lens_mm,lens_curve);     //  print curve value

         blend = ww1C - xoff;                                              //  entire overlap
         get_overlap_region(pxb1C,pxb2C);
         pano_show_images(0);                                              //  show combined images
      }

      if (! mdMx1 && ! mdMy1) {
         mx0 = my0 = 0;                                                    //  no drag in progress
         goto idle;
      }

      mx = mp3x;                                                           //  mouse position in image3
      my = mp3y;

      if (mx < xoff || mx > xoff + ww2C || my > hh3) {                     //  if mouse not in image2 area,
         mx0 = my0 = 0;                                                    //    no drag in progress
         goto idle;
      }

      if (mx0 == 0 && my0 == 0) {                                          //  new drag origin
         mx0 = mx;
         my0 = my;
         goto idle;
      }
      
      if (mx == mx0 && my == my0) goto idle;                               //  no movement

      if (mx > xoff + 0.8 * ww2C) {                                        //  near right edge, theta drag 
         dtoff = mlever * (my - my0) / ww2C;                               //  delta theta, radians
         toff += dtoff;
         xoff += dtoff * (hh1C + yoff);                                    //  change center of rotation   v.5.3
         yoff -= dtoff * (ww1C - xoff);                                    //    to middle of overlap area
      }
      else  {                                                              //  x/y drag
         xoff += mlever * (mx - mx0);                                      //  image2 offsets / mouse leverage
         yoff += mlever * (my - my0);
      }

      mx0 = mx;                                                            //  reset drag origin
      my0 = my;

      if (xoff > ww1C) xoff = ww1C;                                        //  limit nonsense
      if (xoff < 0.3 * ww1C) xoff = 0.3 * ww1C;
      if (yoff < -0.5 * hh2C) yoff = -0.5 * hh2C;
      if (yoff > 0.5 * hh1C) yoff = 0.5 * hh1C;
      if (toff < -0.20) toff = -0.20;
      if (toff > 0.20) toff = 0.20;

      blend = ww1C - xoff;                                                 //  entire overlap
      get_overlap_region(pxb1C,pxb2C);
      pano_show_images(0);                                                 //  show combined images
      zsleep(0.05);
      continue;

idle:                                                                      //  spare time activity
      get_overlap_region(pxb1C,pxb2C);                                     //  get image overlap area
      get_Bratios(pxb1C,pxb2C);                                            //  get brightness ratios in overlap 
      set_colormatch(1);                                                   //  compute color matching factors
      flag_edge_pixels(pxb1C);                                             //  flag edge pixels in overlap 
      matchB = match_images(pxb1C,pxb2C);                                  //  match images
      xoffB = xoff;
      yoffB = yoff;
      toffB = toff;
      update_status_bar();
      zsleep(0.1);
   }

   Mscale = 0;                                                             //  reset in case user zoomed in
   return;
}

int pano_prealign_event(zdialog *zd, const char *event)                    //  dialog event callback function
{
   zdialog_fetch(zd,"spmm",lens_mm);                                       //  get revised lens data
   zdialog_fetch(zd,"spbow",lens_bow);
   if (strEqu(event,"search")) Fautolens = 1;                              //  trigger auto-lens function

   if (strEqu(event,"proceed")) {                                          //  proceed with pano
      pano_align_stat = 2;
      zdialog_free(zdedit);                                                //  kill dialog
      zdedit = null;
   }
   return 0;
}

int pano_prealign_compl(zdialog *zd, int zstat)                            //  dialog completion callback function
{                                                                          //  (cancel only)
   pano_align_stat = 0;
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 0;
}


//  optimize lens parameters
//  assumes a good starting point since search ranges are limited
//  inputs and outputs: lens_mm, lens_bow, xoff, yoff, toff

void pano_autolens()
{
   double   mm_range, bow_range, xoff_range, yoff_range, toff_range;
   double   squeeze, xoff_rfinal, rnum;
   int      counter = 0;
   
   mm_range = 0.1 * lens_mm;                                               //  set initial search ranges
   bow_range = 0.5 * lens_bow;
   if (bow_range < 1) bow_range = 1;
   xoff_range = 7;
   yoff_range = 7;
   toff_range = 0.01;

   xoff_rfinal = 0.2;                                                      //  final xoff range - when to quit
   
   Nalign = 0;
   showRedpix = 1;
   pano_curve_images(0,0);
   get_overlap_region(pxb1C,pxb2C);
   get_Bratios(pxb1C,pxb2C);
   set_colormatch(1);
   flag_edge_pixels(pxb1C);
   pano_show_images(0);

   lens_mmB = lens_mm;                                                     //  initial best fit = current data
   lens_bowB = lens_bow;
   xoffB = xoff;
   yoffB = yoff;
   toffB = toff;
   matchB = match_images(pxb1C,pxb2C);
   
   while (true)
   {
      srand48(time(0) + counter++);
      squeeze = 0.9;                                                       //  search range reduction factor

      lens_mm = lens_mmB + mm_range * (drand48() - 0.5);                   //  new random lens factors
      lens_bow = lens_bowB + bow_range * (drand48() - 0.5);                //     within search range
      pano_curve_images(0,0);
         
      for (int ii = 0; ii < 300; ii++)
      {                                                                    
         rnum = drand48();

         if (rnum < 0.33)                                                  //  random change some alignment offset 
            xoff = xoffB + xoff_range * (drand48() - 0.5);                 //    within search range
         else if (rnum < 0.67)
            yoff = yoffB + yoff_range * (drand48() - 0.5);
         else
            toff = toffB + toff_range * (drand48() - 0.5);
      
         matchlev = match_images(pxb1C,pxb2C);                             //  test quality of image alignment
         if (sigdiff(matchlev,matchB,0.00001) > 0) {
            get_overlap_region(pxb1C,pxb2C);                               //  better: reset alignment data
            get_Bratios(pxb1C,pxb2C);
            set_colormatch(1);
            flag_edge_pixels(pxb1C);
            matchB = match_images(pxb1C,pxb2C);                            //  save new best fit
            lens_mmB = lens_mm;
            lens_bowB = lens_bow;
            xoffB = xoff;
            yoffB = yoff;
            toffB = toff;
            pano_show_images(0);                                           //  update window
            squeeze = 1;                                                   //  keep same search range as long
            break;                                                         //    as improvements are found
         }

         Nalign++;
         update_status_bar();
         if (kill_func) goto done;
         if (pano_align_stat != 1) goto done;
      }

      if (xoff_range < xoff_rfinal) goto done;

      mm_range = squeeze * mm_range;                                       //  reduce search range if no 
      if (mm_range < 0.02 * lens_mmB) mm_range = 0.02 * lens_mmB;          //    improvements were found
      bow_range = squeeze * bow_range;
      if (bow_range < 0.1 * lens_bowB) bow_range = 0.1 * lens_bowB;
      if (bow_range < 0.2) bow_range = 0.2;
      xoff_range = squeeze * xoff_range;
      yoff_range = squeeze * yoff_range;
      toff_range = squeeze * toff_range;
   }

done:
   lens_mm = lens_mmB;                                                     //  set best alignment found
   lens_bow = lens_bowB;
   xoff = xoffB;
   yoff = yoffB;
   toff = toffB;
   pano_curve_images(0,0);
   pano_show_images(0);
   update_status_bar();

   zmessageACK("lens mm: %.1f  bow: %.2f",lens_mm,lens_bow);
   Fautolens = 0;
   showRedpix = 0;
   return;
}


//  combine images and update window during alignment
//    pxb1C + pxb2C  >>  pxb3

void pano_show_images(int fcolor)
{
   int            px3, py3, ii, max;
   int            red1, green1, blue1;
   int            red2, green2, blue2;
   int            red3, green3, blue3;
   pixel          pix1, pix2, pix3;
   double         ww, px1, py1, px2, py2, f1, f2;
   double         costf = cos(toff), sintf = sin(toff);

   ww = xoff + ww2C;                                                       //  combined width
   if (toff < 0) ww -= hh2C * toff;                                        //  adjust for theta
   ww3 = int(ww+1);
   hh3 = hh1A;
   
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_new(colorspace,0,8,ww3,hh3);
   pixbuf_test(3);
   pixbuf_poop(3)                                                          //  get pxb3 attributes
   zunlock();
   
   red1 = green1 = blue1 = 0;                                              //  suppress compiler warnings
   red2 = green2 = blue2 = 0;
   
   for (py3 = 0; py3 < hh3; py3++)                                         //  step through pxb3 rows
   for (px3 = 0; px3 < ww3; px3++)                                         //  step through pxb3 pixels in row
   {
      pix1 = pix2 = 0;
      red3 = green3 = blue3 = 0;

      if (px3 < pxmH) {
         px1 = costf * px3 - sintf * (py3 - yoff);                         //  pxb1C pixel, after offsets   v.5.3 
         py1 = costf * py3 + sintf * (px3 - xoff);
         pix1 = vpixel(ppix1C,px1,py1,ww1C,hh1C,rs1C);
      }

      if (px3 >= pxmL) {
         px2 = costf * (px3 - xoff) + sintf * (py3 - yoff);                //  pxb2C pixel, after offsets
         py2 = costf * (py3 - yoff) - sintf * (px3 - xoff);
         pix2 = vpixel(ppix2C,px2,py2,ww2C,hh2C,rs2C);
      }

      if (pix1) {
         red1 = pix1[0];
         green1 = pix1[1];
         blue1 = pix1[2];
         if (!red1 && !green1 && !blue1) pix1 = 0;                         //  ignore black pixels
      }

      if (pix2) {
         red2 = pix2[0];
         green2 = pix2[1];
         blue2 = pix2[2];
         if (!red2 && !green2 && !blue2) pix2 = 0;
      }

      if (fcolor) {                                                        //  brightness compensation       v.43
         if (pix1) {                                                       //  (auto + manual adjustments)
            red1 = int(colormatch1R[red1]);
            green1 = int(colormatch1G[green1]);
            blue1 = int(colormatch1B[blue1]);
            if (red1 > 255 || green1 > 255 || blue1 > 255) {
               max = red1;
               if (green1 > max) max = green1;
               if (blue1 > max) max = blue1;
               f1 = 255.0 / max;
               red1 = int(red1 * f1);
               green1 = int(green1 * f1);
               blue1 = int(blue1 * f1);
            }
         }

         if (pix2) {                                                       //  adjust both images
            red2 = int(colormatch2R[red2]);                                //    in opposite directions      v.43
            green2 = int(colormatch2G[green2]);
            blue2 = int(colormatch2B[blue2]);
            if (red2 > 255 || green2 > 255 || blue2 > 255) {
               max = red2;
               if (green2 > max) max = green2;
               if (blue2 > max) max = blue2;
               f1 = 255.0 / max;
               red2 = int(red2 * f1);
               green2 = int(green2 * f1);
               blue2 = int(blue2 * f1);
            }
         }
      }

      if (pix1) {
         if (! pix2) {
            red3 = red1;                                                   //  use image1 pixel
            green3 = green1;
            blue3 = blue1; 
         }
         else {                                                            //  use blended pixel
            if (fcolor) {
               if (blend == 0) f1 = 1.0;
               else f1 = (pxmH - px3) / blend;                             //  use progressive blend
               f2 = 1.0 - f1;
               red3 = int(f1 * red1 + f2 * red2);
               green3 = int(f1 * green1 + f2 * green2);
               blue3 = int(f1 * blue1 + f2 * blue2);
            }
            else {                                                         //  use 50/50 mix
               red3 = (red1 + red2) / 2;
               green3 = (green1 + green2) / 2;
               blue3 = (blue1 + blue2) / 2;
            }
         }
      }

      else if (pix2) {
            red3 = red2;                                                   //  use image2 pixel
            green3 = green2;
            blue3 = blue2; 
      }

      pix3 = ppix3 + py3 * rs3 + px3 * nch;                                //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
      
      if (showRedpix && pix1 && pix2) {                                    //  highlight alignment pixels
         ii = py3 * ww1C + px3;
         if (bitmap_get(BMpixels,ii)) {
            pix3[0] = 255;
            pix3[1] = pix3[2] = 0;
         }
      }
   }

   mwpaint2();                                                             //  update window
   return;
}


//  adjust color, brightness, blend, and recombine images

void pano_final_adjust()
{
   int pano_adjust_event(zdialog *zd, const char *event);                  //  dialog event function
   int pano_adjust_compl(zdialog *zd, int zstat);                          //  dialog completion function

   const char  *adjmessage = ZTX("\n match brightness and color");

   pano_automatch = 1;                                                     //  init. auto color match on     v.5.3
   blend = 1;                                                              //  show color diffs at joint
   showRedpix = 0;                                                         //  no alignment pixel highlights

   get_overlap_region(pxb1C,pxb2C);
   get_Bratios(pxb1C,pxb2C);                                               //  get overlap area color ratios
   set_colormatch(pano_automatch);                                         //  set initial color matching
   pano_show_images(1);                                                    //  show final results

   zdedit = zdialog_new(ZTX("match images"),mWin,Bdone,Bcancel,null);        //  color adjustment dialog

   zdialog_add_widget(zdedit,"label","lab0","dialog",adjmessage,"space=10");
   zdialog_add_widget(zdedit,"hbox","hb1","dialog",0,"space=10");            //  match brightness and color
   zdialog_add_widget(zdedit,"vbox","vb1","hb1",0,"homog");
   zdialog_add_widget(zdedit,"vbox","vb2","hb1",0,"homog");                  //  red           [ 100 ]
   zdialog_add_widget(zdedit,"label","lab1","vb1",Bred,"space=7");           //  green         [ 100 ]
   zdialog_add_widget(zdedit,"label","lab2","vb1",Bgreen,"space=7");         //  blue          [ 100 ]
   zdialog_add_widget(zdedit,"label","lab3","vb1",Bblue,"space=7");          //  brightness    [ 100 ]
   zdialog_add_widget(zdedit,"label","lab4","vb1",Bbrightness,"space=7");    //  blend width   [  0  ]
   zdialog_add_widget(zdedit,"label","lab5","vb1",Bblendwidth,"space=7");    //
   zdialog_add_widget(zdedit,"spin","spred","vb2","50|200|0.1|100");         //  [ apply ]  [ auto ]  off
   zdialog_add_widget(zdedit,"spin","spgreen","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblue","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spbright","vb2","50|200|0.1|100");
   zdialog_add_widget(zdedit,"spin","spblend","vb2","1|200|1|1");
   zdialog_add_widget(zdedit,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zdedit,"button","apply","hb2",Bapply,"space=5");
   zdialog_add_widget(zdedit,"button","auto","hb2",ZTX("auto"));
   zdialog_add_widget(zdedit,"label","labauto","hb2","on");
   
   pano_adjust_stat = 1;                                                   //  adjust dialog is busy
   zdialog_run(zdedit,pano_adjust_event,pano_adjust_compl);                //  run dialog, parallel
   
   while (true)
   {
      zsleep(0.1);
      if (pano_adjust_stat != 1) return;
      if (kill_func) return;                                               //  v.5.6
   }
}


//  color adjust dialog completion function

int pano_adjust_compl(zdialog *zd, int zstat)
{
   if (zstat == 1) pano_adjust_stat = 2;                                   //  done
   else pano_adjust_stat = 0;                                              //  canceled
   zdialog_free(zdedit);                                                   //  kill dialog
   zdedit = null;
   return 1;
}


//  callback function for adjust color / brightness dialog
//  pxb1C + pxb2C >> pxb3 under control of spin buttons

int pano_adjust_event(zdialog *zd, const char *event)
{
   double      red, green, blue, bright, bright2;
   
   if (strEqu(event,"auto")) 
   {
      pano_automatch = 1 - pano_automatch;                                 //  toggle color automatch state   v.43
      set_colormatch(pano_automatch);                                      //  set corresp. color matching factors
      pano_show_images(1);                                                 //  combine images and update window
      if (pano_automatch) zdialog_stuff(zd,"labauto","on");
      else zdialog_stuff(zd,"labauto","off");
      return 1;
   }

   if (strNeq(event,"apply")) return 0;                                    //  wait for apply button

   zdialog_fetch(zd,"spred",red);                                          //  get color adjustments
   zdialog_fetch(zd,"spgreen",green);
   zdialog_fetch(zd,"spblue",blue);
   zdialog_fetch(zd,"spbright",bright);                                    //  brightness adjustment
   zdialog_fetch(zd,"spblend",blend);                                      //  blend width

   bright2 = (red + green + blue) / 3;                                     //  RGB brightness
   bright = bright / bright2;                                              //  bright setpoint / RGB brightness
   red = red * bright;                                                     //  adjust RGB brightness
   green = green * bright;
   blue = blue * bright;
   
   bright = (red + green + blue) / 3;
   zdialog_stuff(zd,"spred",red);                                          //  force back into consistency
   zdialog_stuff(zd,"spgreen",green);
   zdialog_stuff(zd,"spblue",blue);
   zdialog_stuff(zd,"spbright",bright);

   Radjust = red / 100;                                                    //  normalize 0.5 ... 2.0
   Gadjust = green / 100;
   Badjust = blue / 100;

   get_overlap_region(pxb1C,pxb2C);
   get_Bratios(pxb1C,pxb2C);                                               //  get color ratios for overlap region
   set_colormatch(pano_automatch);                                         //  set corresp. color matching factors
   pano_show_images(1);                                                    //  combine and update window
   return 1;
}


/**************************************************************************
         edit support functions
***************************************************************************/

//  undo last change - restore image3 from undo stack

void m_undo()
{
   pull_image3();                                                          //  restore previous image3
   return;
}


//  redo last undo - advance undo stack if another image is available

void m_redo()
{
   redo_image3();                                                          //  restore undone image3
   return;
}


//  set new image zoom level magnification
//  set up mwpaint() to center image at clicked position

void m_zoom(const char *which)                                             //  v.5.0
{
   int      ii;
   double   scalew, scaleh, oldscale, fitscale, scales[10];
   double   root2 = sqrt(2.0);
   
   scales[9] = 2.0;                                                        //  max scale 200%
   for (ii = 8; ii >= 0; ii--)                                             //  descending scale: 2.0 1.41 1.0
      scales[ii] = scales[ii+1] / root2;                                   //    0.71 0.5 0.35 0.25 0.177 0.125
   
   if (ww3 > wwD || hh3 > hhD) {                                           //  get window fit scale
      scalew = 1.0 * wwD / ww3;
      scaleh = 1.0 * hhD / hh3;
      if (scalew < scaleh) fitscale = scalew;
      else fitscale = scaleh;
   }
   else fitscale = 1.0;                                                    //  if image < window use 100%
   
   oldscale = Rscale;                                                      //  current scale

   if (*which == '+') Mscale = oldscale * root2;                           //  new scale: +41%
   if (*which == '-') Mscale = oldscale / root2;                           //         or  -29%

   if (*which == 'Z') {                                                    //  keyboard Z:
      if (oldscale < 1.0) Mscale = 1.0;                                    //  toggle 100% and fit window   v.5.1
      else Mscale = 0;
   }

   for (ii = 0; ii < 10; ii++)
      if (Mscale < 1.01 * scales[ii]) break;                               //  find next size scale in table
   if (ii == 10) ii = 9;
   Mscale = scales[ii];
   
   if (oldscale < 1.0 && Mscale > 1.0) Mscale = 1.0;                       //  make 100% sticky
   if (Mscale > 0.9 && Mscale < 1.1) Mscale = 1.0;
   
   if (Mscale < fitscale) Mscale = 0;                                      //  fit window size
   
   if (LMclick || RMclick) {                                               //  click position = zoom center
      zoom3x = click3x;
      zoom3y = click3y;
      LMclick = RMclick = 0;
   }
   else zoom3x = zoom3y = 0;                                               //  button or KB zoom, no position

   mwpaint2();                                                             //  refresh window
   return;
}


//  kill busy dialog and/or running function

void m_kill()
{
   printf("kill \n");
   
   kill_func++;                                                            //  tell function to quit

   if (zdedit) {
      zdialog_free(zdedit);                                                //  kill open dialogs
      zdedit = null;
   }

   if (zdsela) {
      zdialog_free(zdsela);
      zdsela = null;
   }

   if (zdtags) {
      zdialog_free(zdtags);
      zdtags = null;
   }

   gdk_window_set_cursor(dWin->window,0);                                  //  restore normal cursor

   return;
}


//  show RGB values for pixel at mouse click

CBfunc   *RGB_func_save;
int      RGB_mcap_save;

void m_RGB()                                                               //  v.5.6
{
   void RGB_mousefunc();

   if (! pxb3) return;                                                     //  no image
   RGB_func_save = mouseCBfunc;
   RGB_mcap_save = Mcapture;
   mouseCBfunc = RGB_mousefunc;                                            //  connect mouse function
   Mcapture = 1;                                                           //  capture mouse clicks
   return;
}

void RGB_mousefunc()
{
   int         px, py, red, green, blue;
   char        work[90];
   pixel       ppix;

   if (LMclick)                                                            //  left mouse click
   {
      LMclick = 0;
      px = click3x;                                                        //  click position
      py = click3y;
      if (px < 0 || px > ww3-1 || py < 0 || py > hh3-1) return;            //  ignore if outside image area
      ppix = ppix3 + py * rs3 + px * nch;
      red = ppix[0];
      green = ppix[1];
      blue = ppix[2];
      snprintf(work,89,"RGB: %d %d %d  brightness: %.0f  redness: %.0f",
                    red, green, blue, brightness(ppix), redness(ppix));
      stbar_message(stbar,work);
   }
   
   if (RMclick)
   {
      Mcapture = RGB_mcap_save;
      mouseCBfunc = RGB_func_save;                                         //  disconnect mouse
      Mcapture = 0;
   }
   
   return;
}


/**************************************************************************
    HDR and pano shared functions (image matching, alignment, overlay)
***************************************************************************/

//  get a virtual pixel at location (px,py) (real values).
//  get the overlapping real pixels and build a composite.

pixel vpixel(pixel ppix, double px, double py, int ww, int hh, int rs)
{
   int            px0, py0;
   pixel          pix0, pix1, pix2, pix3;
   double         f0, f1, f2, f3;
   double         red, green, blue;
   static uchar   rpix[8];
   static int     which = 0;

   which = 4 - which;                                                      //  2 alternating return pixels

   px0 = int(px);                                                          //  pixel containing (px,py)
   py0 = int(py);

   if (px0 < 1 || py0 < 1) return 0;                                       //  sharpen the border    v.32
   if (px0 > ww-3 || py0 > hh-3) return 0;
   
   pix0 = ppix + py0 * rs + px0 * nch;                                     //  4 pixels based at (px0,py0)
   pix1 = pix0 + rs;
   pix2 = pix0 + nch;
   pix3 = pix0 + rs + nch;

   f0 = (px0+1 - px) * (py0+1 - py);                                       //  overlap of (px,py)
   f1 = (px0+1 - px) * (py - py0);                                         //   in each of the 4 pixels
   f2 = (px - px0) * (py0+1 - py);
   f3 = (px - px0) * (py - py0);
   
   red =   f0 * pix0[0] + f1 * pix1[0] + f2 * pix2[0] + f3 * pix3[0];      //  sum the weighted inputs
   green = f0 * pix0[1] + f1 * pix1[1] + f2 * pix2[1] + f3 * pix3[1];
   blue =  f0 * pix0[2] + f1 * pix1[2] + f2 * pix2[2] + f3 * pix3[2];
   
   rpix[which] = int(red + 0.5);
   rpix[which+1] = int(green + 0.5);
   rpix[which+2] = int(blue + 0.5);

   return rpix + which;
}


//  compare two doubles for significant difference
//  return:  0  difference not significant
//          +1  d1 > d2
//          -1  d1 < d2

int sigdiff(double d1, double d2, double signf)
{
   double diff = fabs(d1-d2);
   if (diff == 0.0) return 0;
   diff = diff / (fabs(d1) + fabs(d2));
   if (diff < signf) return 0;
   if (d1 > d2) return 1;
   else return -1;
}


//  Get the rectangle containing the overlap region of two images
//  outputs:   pxL pxH pyL pyH   total image overlap rectangle
//             pxmL pxmH         reduced to width of blend stripe

void get_overlap_region(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         pxL2, pyL2;
   pixel       ppix91, ppix92;

   pixbuf_poop(91)
   pixbuf_poop(92)

   pxL = 0;
   if (xoff > 0) pxL = int(xoff);

   pxH = ww91;
   if (pxH > xoff + ww92) pxH = int(xoff + ww92);
   
   pyL = 0;
   if (yoff > 0) pyL = int(yoff);
   
   pyH = hh91;
   if (pyH > yoff + hh92) pyH = int(yoff + hh92);

   if (toff > 0) {
      pyL2 = int(yoff + toff * (pxH - pxL));
      if (pyL2 > pyL) pyL = pyL2;
   }

   if (toff < 0) {   
      pxL2 = int(xoff - toff * (pyH - pyL));
      if (pxL2 > pxL) pxL = pxL2;
   }
   
   if (blend > 0) {
      pxM = (pxL + pxH) / 2;                                               //  midpoint of blend stripe   v.5.3
      pxmL = pxM - int(blend/2);                                           //  blend stripe pixel x-range
      pxmH = pxM + int(blend/2);
      if (pxmL < pxL) pxmL = pxL;
      if (pxmH > pxH) pxmH = pxH;
   }
   else {                                                                  //  no blend stripe, use whole range
      pxmL = pxL;
      pxmH = pxH;
   }

   return;
}


//  Compute brightness ratio by color for overlapping image areas.         //  compute both image directions  v.43
//    (image2 is overlayed on image1, offset by xoff, yoff, toff)
//
//  Outputs: 
//    Bratios1[kk][ii] = image2/image1 brightness ratio for color kk
//                       and image1 brightness ii
//    Bratios2[kk][ii] = image1/image2 brightness ratio for color kk
//                       and image2 brightness ii

void get_Bratios(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px9, py9;
   int         ii, jj, kk, npix, npix1, npix2, npix3;
   int         brdist1[3][256], brdist2[3][256];
   double      px1, py1, px2, py2;
   double      brlev1[3][256], brlev2[3][256];
   double      costf = cos(toff), sintf = sin(toff);
   double      a1, a2, b1, b2;

   pixbuf_poop(91)
   pixbuf_poop(92)
   
   for (kk = 0; kk < 3; kk++)                                              //  clear distributions
   for (ii = 0; ii < 256; ii++)
      brdist1[kk][ii] = brdist2[kk][ii] = 0;
      
   npix = 0;

   for (py9 = pyL; py9 < pyH; py9++)                                       //  scan image1/image2 pixels parallel
   for (px9 = pxL; px9 < pxH; px9++)                                       //  use entire overlap area
   {
      px1 = costf * px9 - sintf * (py9 - yoff);                            //  image1 pixel, after offsets   v.5.3 
      py1 = costf * py9 + sintf * (px9 - xoff);
      pix1 = vpixel(ppix91,px1,py1,ww91,hh91,rs91);
      if (! pix1) continue;                                                //  does not exist
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = costf * (px9 - xoff) + sintf * (py9 - yoff);                   //  corresponding image2 pixel
      py2 = costf * (py9 - yoff) - sintf * (px9 - xoff);
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;                                                //  does not exist
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;                      //  ignore black pixels

      ++npix;                                                              //  count overlapping pixels
      
      for (kk = 0; kk < 3; kk++)                                           //  accumulate brightness distributions
      {                                                                    //    by color in 256 bins
         ++brdist1[kk][pix1[kk]];
         ++brdist2[kk][pix2[kk]];
      }
   }
   
   npix1 = npix / 256;                                                     //  1/256th of total pixels
   
   for (kk = 0; kk < 3; kk++)                                              //  get brlev1[kk][N] = mean brightness
   for (ii = jj = 0; jj < 256; jj++)                                       //    for Nth group of image1 pixels
   {                                                                       //      for color kk
      brlev1[kk][jj] = 0;
      npix2 = npix1;                                                       //  1/256th of total pixels

      while (npix2 > 0 && ii < 256)                                        //  next 1/256th group from distribution
      {
         npix3 = brdist1[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev1[kk][jj] += ii * npix3;                                     //  brightness * (pixels with bright.)
         brdist1[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev1[kk][jj] = brlev1[kk][jj] / npix1;                             //  mean brightness for group, 0-255
   }

   for (kk = 0; kk < 3; kk++)                                              //  do same for image2
   for (ii = jj = 0; jj < 256; jj++)
   {
      brlev2[kk][jj] = 0;
      npix2 = npix1;

      while (npix2 > 0 && ii < 256)
      {
         npix3 = brdist2[kk][ii];
         if (npix3 == 0) { ++ii; continue; }
         if (npix3 > npix2) npix3 = npix2;
         brlev2[kk][jj] += ii * npix3;
         brdist2[kk][ii] -= npix3;
         npix2 -= npix3;
      }

      brlev2[kk][jj] = brlev2[kk][jj] / npix1;
   }

   for (kk = 0; kk < 3; kk++)                                              //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev1 brightness, 0 to 255
   {
      while (ii > brlev2[kk][jj] && jj < 256) ++jj;                        //  find matching brlev2 brightness
      a2 = brlev2[kk][jj];                                                 //  next higher value
      b2 = brlev1[kk][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev2[kk][jj-1];                                         //  next lower value
            b1 = brlev1[kk][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0) Bratios2[kk][ii] = b2 / a2;
         else Bratios2[kk][ii] = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;    //  interpolate
      }
      else Bratios2[kk][ii] = 1;

      if (Bratios2[kk][ii] < 0.1) Bratios2[kk][ii] = 0.1;                  //  contain outlyers    v.43
      if (Bratios2[kk][ii] > 10) Bratios2[kk][ii] = 10;
   }

   for (kk = 0; kk < 3; kk++)                                              //  color
   for (ii = jj = 0; ii < 256; ii++)                                       //  brlev2 brightness, 0 to 255
   {
      while (ii > brlev1[kk][jj] && jj < 256) ++jj;                        //  find matching brlev1 brightness
      a2 = brlev1[kk][jj];                                                 //  next higher value
      b2 = brlev2[kk][jj];
      if (a2 > 0 && b2 > 0) {
         if (jj > 0) {
            a1 = brlev1[kk][jj-1];                                         //  next lower value
            b1 = brlev2[kk][jj-1];
         }
         else   a1 = b1 = 0;
         if (ii == 0) Bratios1[kk][ii] = b2 / a2;
         else Bratios1[kk][ii] = (b1 + (ii-a1)/(a2-a1) * (b2-b1)) / ii;    //  interpolate
      }
      else Bratios1[kk][ii] = 1;

      if (Bratios1[kk][ii] < 0.1) Bratios1[kk][ii] = 0.1;                  //  contain outlyers    v.43
      if (Bratios1[kk][ii] > 10) Bratios1[kk][ii] = 10;
   }

   return;
}


//  Set color matching factors                                             //  v.43
//     on:   Bratios are used for color matching the two images
//     off:  Bratios are not used - use 1.0 instead
//  In both cases, manual settings Radjust/Gadjust/Badjust are used

void set_colormatch(int state)
{
   int      ii;

   if (state) 
   {
      for (ii = 0; ii < 256; ii++)
      {
         colormatch1R[ii] = sqrt(Bratios1[0][ii]) / Radjust * ii;          //  use sqrt(ratio) so that adjustment
         colormatch1G[ii] = sqrt(Bratios1[1][ii]) / Gadjust * ii;          //    can be applied to both images
         colormatch1B[ii] = sqrt(Bratios1[2][ii]) / Badjust * ii;          //      (image1 --, image2 ++)    v.43

         colormatch2R[ii] = sqrt(Bratios2[0][ii]) * Radjust * ii;
         colormatch2G[ii] = sqrt(Bratios2[1][ii]) * Gadjust * ii;
         colormatch2B[ii] = sqrt(Bratios2[2][ii]) * Badjust * ii;
      }
   }

   else 
   {
      for (ii = 0; ii < 256; ii++)
      {
         colormatch1R[ii] = 1.0 / Radjust * ii;
         colormatch1G[ii] = 1.0 / Gadjust * ii;
         colormatch1B[ii] = 1.0 / Badjust * ii;

         colormatch2R[ii] = Radjust * ii;
         colormatch2G[ii] = Gadjust * ii;
         colormatch2B[ii] = Badjust * ii;
      }
   }
   
   return;
}


//  find pixels of greatest contrast within overlap area
//  flag high-contrast pixels to use in each image compare region

void flag_edge_pixels(GdkPixbuf *pxb9)
{
   void  flag_edge_pixels2(GdkPixbuf *, int pxL, int pxH, int pyL, int pyH, int samp);

   pixel       ppix9;
   int         ww9, hh9, rs9, samp;

   pixbuf_poop(9)

   if (BMpixels) bitmap_delete(BMpixels);                                  //  bitmap to flag alignment pixels
   BMpixels = bitmap_new(ww9*hh9);

   if (blend == 0) {                                                       //  HDR: use 4 regions
      samp = pixsamp / 4;
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyL, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxL, pxH/2, pyH/2, pyH, samp);
      flag_edge_pixels2(pxb9, pxH/2, pxH, pyH/2, pyH, samp);
   }
   else {                                                                  //  pano: use 4 regions in blend stripe
      samp = pixsamp / 4;                                                  //  v.32
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyL, pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/4, pyH/2, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH/2, pyH-pyH/4, samp);
      flag_edge_pixels2(pxb9, pxmL, pxmH, pyH-pyH/4, pyH, samp);
   }

   return;
}


//  Find the highest contrast pixels meeting sample size
//    within the specified sub-region of image overlap.

void flag_edge_pixels2(GdkPixbuf *pxb9, int pxL, int pxH, int pyL, int pyH, int samp)
{
   pixel       ppix9;
   int         px9, py9, ww9, hh9, rs9;
   int         ii, jj, npix;
   int         red1, green1, blue1, red2, green2, blue2, tcon;
   int         Hdist[766], Vdist[766], Hmin, Vmin;
   double      costf = cos(toff), sintf = sin(toff);
   double      px1, py1;
   uchar       *Hcon, *Vcon;
   pixel       pix1, pix2;
   
   pixbuf_poop(9)

   npix = (pxH - pxL) * (pyH - pyL);                                       //  overlapping pixels
   if (npix < 100) return;                                                 //  insignificant
   if (samp > npix / 4) samp = npix / 4;                                   //  use max. 1/4 of pixels       v.32

   Hcon = (uchar *) zmalloc(npix);                                         //  horizontal pixel contrast 0-255
   Vcon = (uchar *) zmalloc(npix);                                         //  vertical pixel contrast 0-255

   for (py9 = pyL; py9 < pyH-2; py9++)                                     //  scan image pixels in sub-region
   for (px9 = pxL; px9 < pxH-2; px9++)
   {
      ii = (py9-pyL) * (pxH-pxL) + (px9-pxL);
      Hcon[ii] = Vcon[ii] = 0;                                             //  horiz. = vert. contrast = 0

      px1 = costf * px9 - sintf * (py9 - yoff);                            //  image1 pixel
      py1 = costf * py9 + sintf * (px9 - xoff);
      pix1 = vpixel(ppix9,px1,py1,ww9,hh9,rs9);
      if (! pix1) continue;                                                //  does not exist
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      if (!red1 && !green1 && !blue1) continue;                            //  ignore if black

      pix2 = vpixel(ppix9,px1+4,py1,ww9,hh9,rs9);                          //  4 pixels to right   v.5.3
      if (! pix2) continue;                                                //  reject if off edge
      if (! pix2[0] && ! pix2[1] && ! pix2[2]) continue;
      pix2 = vpixel(ppix9,px1+2,py1,ww9,hh9,rs9);                          //  2 pixels to right
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  horizontal contrast
      if (tcon > 10) Hcon[ii] = tcon / 3;

      pix2 = vpixel(ppix9,px1,py1+4,ww9,hh9,rs9);                          //  4 pixels below     v.5.3
      if (! pix2) continue;
      if (! pix2[0] && ! pix2[1] && ! pix2[2]) continue;
      pix2 = vpixel(ppix9,px1,py1+2,ww9,hh9,rs9);                          //  2 pixels below
      red2 = pix2[0];
      green2 = pix2[1];
      blue2 = pix2[2];
      tcon = abs(red1-red2) + abs(green1-green2) + abs(blue1-blue2);       //  vertical contrast
      if (tcon > 10) Vcon[ii] = tcon / 3;
   }

   for (ii = 0; ii < 766; ii++) Hdist[ii] = Vdist[ii] = 0;                 //  clear contrast distributions

   for (py9 = pyL; py9 < pyH-2; py9++)                                     //  scan image pixels
   for (px9 = pxL; px9 < pxH-2; px9++)
   {                                                                       //  build contrast distributions
      ii = (py9-pyL) * (pxH-pxL) + (px9-pxL);
      ++Hdist[Hcon[ii]];
      ++Vdist[Vcon[ii]];
   }
   
   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  find minimum contrast needed to get
   {                                                                       //    enough pixels for sample size
      npix += Hdist[ii];                                                   //      (horizontal contrast pixels)
      if (npix > samp) break; 
   }
   Hmin = ii; 

   for (npix = 0, ii = 765; ii > 0; ii--)                                  //  (verticle contrast pixels)
   {
      npix += Vdist[ii];
      if (npix > samp) break;
   }
   Vmin = ii; 

   for (py9 = pyL; py9 < pyH-2; py9++)                                     //  scan image pixels
   for (px9 = pxL; px9 < pxH-2; px9++)
   {
      ii = (py9-pyL) * (pxH-pxL) + (px9-pxL);
      jj = py9 * ww9 + px9;

      if (Hcon[ii] > Hmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag horizontal group of 3
         bitmap_set(BMpixels,jj+1,1);
         bitmap_set(BMpixels,jj+2,1);
      }

      if (Vcon[ii] > Vmin) {
         bitmap_set(BMpixels,jj,1);                                        //  flag verticle group of 3
         bitmap_set(BMpixels,jj+ww9,1);
         bitmap_set(BMpixels,jj+2*ww9,1);
      }
   }
   
   zfree(Hcon);
   zfree(Vcon);
   return;
}


//  Compare two images in overlapping areas.
//  (image2 is overlayed on image1, offset by xoff, yoff, toff)
//  If blend > 0, compare verticle stripe in the middle, width = blend.
//  Use pixels with contrast >= minimum needed to reach enough pixels.
//  return: 1 = perfect match, 0 = total mismatch (black/white)

double match_images(GdkPixbuf *pxb91, GdkPixbuf *pxb92)
{
   pixel       ppix91, ppix92, pix1, pix2;
   int         ww91, hh91, ww92, hh92, rs91, rs92;
   int         px9, py9, ii, pixcount, ymid;
   double      px1, py1, px2, py2;
   double      costf = cos(toff), sintf = sin(toff);
   double      match, cmatch, maxcmatch;
   double      yspan, weight;
   
   pixbuf_poop(91)
   pixbuf_poop(92)
   
   cmatch = maxcmatch = pixcount = 0;

   if (blend > 0) goto match_blend;

   for (py9 = pyL; py9 < pyH; py9++)                                       //  scan image1/image2 pixels parallel
   for (px9 = pxL; px9 < pxH; px9++)
   {
      ii = py9 * ww91 + px9;                                               //  skip low-contrast pixels
      if (! bitmap_get(BMpixels,ii)) continue;

      px1 = costf * px9 - sintf * (py9 - yoff);                            //  image1 pixel
      py1 = costf * py9 + sintf * (px9 - xoff);
      pix1 = vpixel(ppix91,px1,py1,ww91,hh91,rs91);
      if (! pix1) continue;                                                //  does not exist
      if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                      //  ignore black pixels

      px2 = costf * (px9 - xoff) + sintf * (py9 - yoff);                   //  corresponding image2 pixel
      py2 = costf * (py9 - yoff) - sintf * (px9 - xoff);
      pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
      if (! pix2) continue;
      if (!pix2[0] && !pix2[1] && !pix2[2]) continue;

      match = match_pixels(pix1,pix2);                                     //  compare brightness adjusted
      cmatch += match;                                                     //  accumulate total match
      pixcount++;
   }

   return cmatch / pixcount;

match_blend:

   if (pxM > ww91) return 0;                                               //  blend runs off of image, no match

   yspan = 1.0 / (pyH - pyL);                                              //  prepare params for weight calc.
   ymid = (pyL + pyH) / 2;                                                 //  (weight central pixels more)

   for (py9 = pyL; py9 < pyH; py9++)                                       //  step through image1 pixels, rows
   {
      weight = 1 - yspan * abs(py9 - ymid);                                //  weight: 0.5 .. 1.0 .. 0.5   v.32

      for (px9 = pxmL; px9 < pxmH; px9++)                                  //  step through image1 pixels, cols
      {
         ii = py9 * ww91 + px9;                                            //  skip low-contrast pixels
         if (! bitmap_get(BMpixels,ii)) continue;

         px1 = costf * px9 - sintf * (py9 - yoff);                         //  image1 pixel
         py1 = costf * py9 + sintf * (px9 - xoff);
         pix1 = vpixel(ppix91,px1,py1,ww91,hh91,rs91);
         if (! pix1) continue;                                             //  does not exist
         if (!pix1[0] && !pix1[1] && !pix1[2]) continue;                   //  ignore black pixels

         px2 = costf * (px9 - xoff) + sintf * (py9 - yoff);                //  corresponding image2 pixel
         py2 = costf * (py9 - yoff) - sintf * (px9 - xoff);
         pix2 = vpixel(ppix92,px2,py2,ww92,hh92,rs92);
         if (! pix2) continue;
         if (!pix2[0] && !pix2[1] && !pix2[2]) continue;

         match = match_pixels(pix1,pix2);                                  //  compare brightness adjusted
         cmatch += match * weight;                                         //  accumulate total match
         maxcmatch += weight;
         pixcount++;
      }
   }

   return cmatch / maxcmatch;
}


//  Compare 2 pixels using precalculated brightness ratios
//  1.0 = perfect match   0 = total mismatch (black/white)

double match_pixels(pixel pix1, pixel pix2)
{
   double   red1, green1, blue1, red2, green2, blue2;
   double   reddiff, greendiff, bluediff, match;

   red1 = colormatch1R[pix1[0]];                                           //  adjust both images       v.43
   green1 = colormatch1G[pix1[1]];
   blue1 = colormatch1B[pix1[2]];

   red2 = colormatch2R[pix2[0]];
   green2 = colormatch2G[pix2[1]];
   blue2 = colormatch2B[pix2[2]];

   reddiff = 0.00392 * fabs(red1-red2);                                    //  0 = perfect match
   greendiff = 0.00392 * fabs(green1-green2);                              //  1 = total mismatch
   bluediff = 0.00392 * fabs(blue1-blue2);
   
   match = (1.0 - reddiff) * (1.0 - greendiff) * (1.0 - bluediff);
   return match;
}


/**************************************************************************
         supporting functions
***************************************************************************/


//  get the brightness of a pixel, 0-255                                   //  v.5.5

double brightness(pixel pix)
{
   double bright = 0.25 * pix[0] + 0.65 * pix[1] + 0.10 * pix[2];
   return bright;
}


//  get the redness of a pixel
//  ratio of red to overall brightness, 0-100                              //  v.5.5

double redness(pixel pix)
{
   double bright = brightness(pix);
   double redpart = 0.25 * pix[0] / (bright + 1);
   return 100 * redpart;
}


//  set pixel redness to given value

void setredness(pixel ppix, double redness)                                //  v.5.6
{
   ppix[0] = int(redness * (0.65 * ppix[1] + 0.10 * ppix[2] + 1) 
                     / (25 - 0.25 * redness));
   return;
}


//  query if modifications should be kept or discarded
//  return:  1 to keep mods,  0 to discard mods

int mod_keep()
{
   if (Fmod3 == 0) return 0;                                               //  no mods
   if (zmessageYN(ZTX("discard modifications?"))) return 0;                //  OK to discard
   return 1;
}


//  Save some current state information at exit, reload at startup
//  saved state data file: $HOME/.fotoxx/saved_state

void save_fotoxx_state()                                                   //  v.5.6
{
   FILE           *fid;
   char           dirbuff[maxfcc];

   snprintf(dirbuff,maxfcc-1,"%s/saved_state",get_zuserdir());
   fid = fopen(dirbuff,"w");
   if (! fid) return;
   fputs(imagedirk,fid);                                                   //  current image directory
   fputs("\n",fid);
   fputs(topdirk,fid);                                                     //  top image directory
   fputs("\n",fid);
   fputs(fname1,fid);                                                      //  current file name
   fputs("\n",fid);
   snprintf(dirbuff,20," %d %d \n",wwD,hhD);                               //  window size
   fputs(dirbuff,fid);
   snprintf(dirbuff,20," %d %d \n",image_navi::xwinW,image_navi::xwinH);   //  thumbnail index window size
   fputs(dirbuff,fid);
   snprintf(dirbuff,20," %d \n",image_navi::thumbsize);                    //  thumbnail size
   fputs(dirbuff,fid);
   fclose(fid);
   return;
}

void load_fotoxx_state()
{
   int            err, ww, hh;
   FILE           *fid;
   struct stat    statdat;
   char           dirbuff[maxfcc], *pp;

   pp = getcwd(imagedirk,maxfcc-1);                                        //  defaults are current directory
   pp = getcwd(topdirk,maxfcc-1);

   snprintf(dirbuff,maxfcc-1,"%s/saved_state",get_zuserdir());             //  saved file
   fid = fopen(dirbuff,"r");
   if (! fid) return;

   pp = fgets_trim(dirbuff,maxfcc-1,fid,1);                                //  current directory
   if (pp) {
      err = stat(dirbuff,&statdat);                                        //  valid directory?
      if (! err && S_ISDIR(statdat.st_mode))
         strcpy(imagedirk,dirbuff);                                        //  yes, use it
   }

   pp = fgets_trim(dirbuff,maxfcc-1,fid,1);                                //  top directory
   if (pp) {
      err = stat(dirbuff,&statdat);                                        //  valid directory?
      if (! err && S_ISDIR(statdat.st_mode))
         strcpy(topdirk,dirbuff);                                          //  yes, use it
   }

   pp = fgets_trim(dirbuff,maxfcc-1,fid,1);                                //  image file name
   if (pp) strncpy0(fname1,pp,100);
   
   pp = fgets(dirbuff,maxfcc-1,fid);                                       //  window size
   if (pp) {
      ww = hh = 0;
      sscanf(dirbuff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 2000) wwD = ww;
      if (hh > 200 && hh < 1600) hhD = hh;                                 //  menu + toolbar adder
   }

   pp = fgets(dirbuff,maxfcc-1,fid);                                       //  thumbnail index window size
   if (pp) {
      ww = hh = 0;
      sscanf(dirbuff," %d %d ",&ww,&hh);
      if (ww > 200 && ww < 2000) image_navi::xwinW = ww;
      if (hh > 200 && hh < 1600) image_navi::xwinH = hh;                   //  toolbar adder
   }

   pp = fgets_trim(dirbuff,maxfcc-1,fid,1);                                //  thumbnail image size
   if (pp) {
      sscanf(dirbuff," %d ",&ww);
      if (ww > 32 && ww < 256) image_navi::thumbsize = ww;
   }

   fclose(fid);
   return;
}


/**************************************************************************/

//  functions to push image3 into undo stack and pull from stack

void push_image3()                                                         //  push image3 into stack
{
   GdkPixbuf      *pxbX;
   static int     undoftf = 1;

   if (undoftf) {
      undoftf = Nundos = 0;                                                //  first call, clear undo stack
      for (int ii = 0; ii < undomax; ii++) 
            undostack[ii] = 0;
   }

   if (! pxb3) return;
   zlock();
   if (Nundos == undomax) {
      pixbuf_free(undostack[0]);                                           //  lose oldest stack entry
      for (int ii = 1; ii < undomax; ii++)
            undostack[ii-1] = undostack[ii];                               //  push other entries down
      Nundos--;
      undostack[Nundos] = 0;                                               //  last entry vacated
   }
   pxbX = pixbuf_copy(pxb3);
   pixbuf_test(X);
   pixbuf_free(undostack[Nundos]);                                         //  free entry to be overwritten
   undostack[Nundos] = pxbX;                                               //  image3 >> stack
   Nundos++;
   zunlock();
   return;
}

void pull_image3()                                                         //  pull image3 from stack (undo)
{
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (Nundos == 0) return;
   zlock();
   temp = undostack[Nundos-1];                                             //  image3 <==> last stack entry
   undostack[Nundos-1] = pxb3;
   pxb3 = temp;
   pixbuf_poop(3);
   zunlock();
   Nundos--;                                                               //  decrement stack count
   if (Nundos == 0) Fmod3 = 0;                                             //  if 0, have original image
   mwpaint2(); 
   return;
}

void prior_image3()                                                        //  get last image3 from stack
{                                                                          //    without removal from stack
   if (! pxb3) return;
   if (Nundos == 0) return;
   zlock();
   pixbuf_free(pxb3);
   pxb3 = pixbuf_copy(undostack[Nundos-1]);
   pixbuf_test(3);
   pixbuf_poop(3);
   zunlock();
   if (Nundos == 1) Fmod3 = 0;                                             //  1 in stack: original image  v.5.4.1
   mwpaint2();                                                             //  refresh screen   v.5.0
   return;
}

void redo_image3()                                                         //  get last+1 stack entry (redo)
{                                                                          //    and advance stack depth
   GdkPixbuf   *temp;

   if (! pxb3) return;
   if (Nundos == undomax) return;
   if (! undostack[Nundos]) return;                                        //  nothing is there
   zlock();
   temp = undostack[Nundos];                                               //  image3 <==> last+1 stack entry
   undostack[Nundos] = pxb3;
   pxb3 = temp;
   pixbuf_poop(3);
   Nundos++;                                                               //  stack count up
   zunlock();
   Fmod3 = 1;                                                              //  image3 modified
   mwpaint2(); 
   return;
}

void clearmem_image3()                                                     //  clear stack (start new image)
{
   zlock();
   for (int ii = 0; ii < undomax; ii++) 
         pixbuf_free(undostack[ii]);
   zunlock();
   Nundos = 0;
   Fmod3 = 0;                                                              //  image3 not modified
   return;
}


/**************************************************************************/

//  validate an image file and load pixbuf from file        v.33
//  if an alpha channel is present, remove it

GdkPixbuf * load_pixbuf(const char *file)
{
   GdkPixbuf   *pxb91 = 0, *pxb92 = 0;
   int         ww91, hh91, rs91, ww92, hh92, rs92;
   int         px, py, alfa;
   pixel       ppix91, pix91, ppix92, pix92;

   pxb91 = pixbuf_new_from_file(file,gerror);                              //  validate file and load pixbuf
   if (! pxb91) return 0;

   nch = gdk_pixbuf_get_n_channels(pxb91);
   nbits = gdk_pixbuf_get_bits_per_sample(pxb91);
   alfa = gdk_pixbuf_get_has_alpha(pxb91);

   if (nch < 3 || nbits != 8) {                                            //  must be 3 or 4 channels
      pixbuf_free(pxb91);                                                  //  and 8 bits per channel
      return 0;
   }
   
   if (! alfa) return pxb91;                                               //  no alpha channel, 3 channels

   pixbuf_poop(91);
   pxb92 = pixbuf_new(colorspace,0,8,ww91,hh91);
   pixbuf_test(92);
   pixbuf_poop(92);

   for (py = 0; py < hh91; py++)                                           //  copy without 4th channel (alpha)
   for (px = 0; px < ww91; px++)
   {
      pix91 = ppix91 + rs91 * py + 4 * px;
      pix92 = ppix92 + rs92 * py + 3 * px;
      pix92[0] = pix91[0];
      pix92[1] = pix91[1];
      pix92[2] = pix91[2];
   }

   pixbuf_free(pxb91);
   nch = 3;                                                                //  set channels to 3 
   return pxb92;
}


/**************************************************************************/

//  track pixbufs allocated and freed for memory leak detection

int      pixbuf_count = 0;

void incr_pixbufs()
{
   ++pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}

void pixbuf_free(GdkPixbuf *&pixbuf)
{
   if (! pixbuf) return;
   zlock();
   g_object_unref(pixbuf);
   pixbuf = 0;
   zunlock();
   --pixbuf_count;
   if (debug) printf("pixbufs: %d \n",pixbuf_count);
   return;
}


