#include <glib.h>
#include "brightness.h"
#include "config.h"


void iv_brightness_apply(
	guint8 *data,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gfloat brightness_coeff,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Applies brightness to the image.
 *
 *	The data, width, height, bpp, and bpl specifies the image data
 *	to apply contast to.
 *
 *	The brightness_coeff specifies the brightness coefficient from
 *	-1.0 to 1.0.
 */
void iv_brightness_apply(
	guint8 *data,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gfloat brightness_coeff,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	const gint _bpl = (bpl > 0) ? bpl : (width * bpp);
	gint v, y;
	guint8 *data_ptr, *data_end;

	if((data == NULL) || (width <= 0) || (height <= 0) ||
	   (bpp <= 0) || (brightness_coeff == 0.0f)
	)
	    return;

	/* Calculate the brightness delta value in bytes */
	v = (gint)(0xff * brightness_coeff);

	switch(bpp)
	{
	  case 4:	/* RGBA */
	  case 3:	/* RGB */
	    for(y = 0; y < height; y++)
	    {
		data_ptr = data + (y * _bpl);
		data_end = data_ptr + (width * bpp);

		while(data_ptr < data_end)
		{
                    data_ptr[0] = (guint8)CLIP(
                        ((gint)data_ptr[0] + v),
                        0x00, 0xff
                    );
                    data_ptr[1] = (guint8)CLIP(
                        ((gint)data_ptr[1] + v),
                        0x00, 0xff
                    );
                    data_ptr[2] = (guint8)CLIP(
                        ((gint)data_ptr[2] + v),
                        0x00, 0xff
                    );
		    data_ptr += bpp;
		}

		if(progress_cb != NULL)
		{
		    if(progress_cb(
			(gulong)(y + 1),
			(gulong)height,
			progress_data
		    ))
			break;
		}
	    }
	    break;

	  case 2:	/* Greyscale Alpha */
	  case 1:	/* Greyscale */
	    for(y = 0; y < height; y++)
	    {
		data_ptr = data + (y * _bpl);
		data_end = data_ptr + (width * bpp);

		while(data_ptr < data_end)
		{
                    data_ptr[0] = (guint8)CLIP(
                        ((gint)data_ptr[0] + v),
                        0x00, 0xff
                    );
		    data_ptr += bpp;
		}

		if(progress_cb != NULL)
		{
		    if(progress_cb(
			(gulong)(y + 1),
			(gulong)height,
			progress_data
		    ))
			break;
		}
	    }
	    break;
	}
}
