#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "guiutils.h"
#include "guirgbimg.h"
#include "pulist.h"
#include "progressdialog.h"
#include "imgview.h"
#include "colorlevels.h"
#include "colorlevelsdlg.h"
#include "config.h"

#include "images/icon_color_levels_32x32.xpm"
#include "images/icon_color_levels_48x48.xpm"
#include "images/icon_revert_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"


typedef struct _Dialog			Dialog;
#define DIALOG(p)			((Dialog *)(p))
typedef struct _CLDProgressData		CLDProgressData;
#define CLD_PROGRESS_DATA(p)		((CLDProgressData *)(p))


typedef enum {
	CLD_CHANNEL_RED			= (1 << 0),
	CLD_CHANNEL_GREEN		= (1 << 1),
	CLD_CHANNEL_BLUE		= (1 << 2),
	CLD_CHANNEL_ALPHA		= (1 << 3)
} CLDChannel;


typedef enum {
	CLD_MARKER_MIN,
	CLD_MARKER_MID,
	CLD_MARKER_MAX
} CLDMarker;


static gint apply_iv_image_progress_cb(
	const gulong i, const gulong m, gpointer data
);
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void changed_cb(GtkWidget *widget, gpointer data);
static void set_cb(GtkWidget *widget, gpointer data);
static void cancel_cb(GtkWidget *widget, gpointer data);
static void channel_changed_cb(
	pulistbox_struct *pulistbox, gint i, gpointer data
);
static void spin_changed_cb(GtkWidget *widget, gpointer data);
static void preview_toggled_cb(GtkWidget *widget, gpointer data);
static void reset_cb(GtkWidget *widget, gpointer data);
static gint histogram_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint gradient_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint apply_iv_image_tocb(gpointer data);

static CLDChannel get_current_channels(Dialog *d);

static void get_current_marker_values(
	Dialog *d,
	gint *min_rtn,
	gint *mid_rtn,
	gfloat *mid_coeff_rtn,
	gint *max_rtn
);
static void set_marker_value(
	Dialog *d,
	const CLDMarker marker,
	const gint v
);

static void reset_all_marker_values(
	Dialog *d,
	const gboolean apply_to_iv_image,
	const gboolean interruptable
);

static void histogram_draw(Dialog *d);
static void gradient_draw(Dialog *d);
static void update_draw_all(
	Dialog *d,
	const gboolean apply_to_iv_image,
	const gboolean interruptable
);

static void apply_iv_image(
	Dialog *d,
	const gboolean verbose,
	const gboolean interruptable,
	const gboolean cur_frame_only
);
static void queue_apply_iv_image(Dialog *d);
static void restore_iv_image(Dialog *d);

static void make_histogram(
	guint *histogram_data,		/* 256 guints */
	guint *histogram_data_max,
	const guint8 *buf,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const CLDChannel channels
);
static void rerender_gradient_data(Dialog *d);

static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
);

gboolean iv_color_levels(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gboolean *all_frames
);


/*
 *	Dialog:
 */
struct _Dialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gboolean	has_changes,
			got_response;
	imgview_struct	*iv;

	GdkCursor	*translate_cur;
	GdkGC		*gc;

	pulistbox_struct	*channel_pulistbox;

	GtkWidget	*ok_btn,
			*cancel_btn,
			*close_btn,
			*histogram_da,
			*gradient_da,
			*min_marker_spin,
			*mid_marker_spin,
			*max_marker_spin,
			*preview_check,
			*all_frames_check,
			*reset_btn;

	GdkPixmap	*histogram_pm,
			*gradient_pm;

	guint		histogram_data[(gint)((guint8)-1) + 1];
	guint		histogram_data_max;

	guint8		*gradient_img_data;	/* RGB */
	gint		gradient_img_width,
			gradient_img_height;

	gint		red_min_marker;		/* In bytes */
	gfloat		red_mid_marker_coeff;	/* Coefficient */
	gint		red_max_marker;		/* In bytes */
	gint		green_min_marker;	/* In bytes */
	gfloat		green_mid_marker_coeff;	/* Coefficient */
	gint		green_max_marker;	/* In bytes */
	gint		blue_min_marker;	/* In bytes */
	gfloat		blue_mid_marker_coeff;	/* Coefficient */
	gint		blue_max_marker;	/* In bytes */
	gint		alpha_min_marker;	/* In bytes */
	gfloat		alpha_mid_marker_coeff;	/* Coefficient */
	gint		alpha_max_marker;	/* In bytes */

	GdkModifierType	drag_buttons;
	gint		drag_x,
			drag_y;
	CLDMarker	drag_marker;

	guint		apply_to_iv_idle_id;

	/* Copy of the original frame's data */
	guint8		*orig_img_data;
	gint		cur_frame;
};


struct _CLDProgressData {
	gint		cur_frame,
			nframes;
	gfloat		unit_frame_coeff;
	gboolean	aborted;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Apply IV Image progress callback.
 */
static gint apply_iv_image_progress_cb(
	const gulong i, const gulong m, gpointer data
)
{
	gfloat v;
	CLDProgressData *d = CLD_PROGRESS_DATA(data);
	if(d == NULL)
	    return(-2);

	v = (d->cur_frame * d->unit_frame_coeff) +
	    ((m > 0) ? (d->unit_frame_coeff * (gfloat)i / (gfloat)m) : 0.0f);

	if(ProgressDialogIsQuery())
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		v,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		d->aborted = TRUE;
		return(-4);
	    }
	}

	return(0);
}


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	d->got_response = FALSE;
	gtk_main_quit();

	return(TRUE);
}

/*
 *	Changed callback.
 */
static void changed_cb(GtkWidget *widget, gpointer data)
{
        Dialog *d = DIALOG(data);
        if(d == NULL)
            return;

        if(d->freeze_count > 0)
            return;

        d->freeze_count++;

        if(!d->has_changes)
        {
            gtk_widget_set_sensitive(d->ok_btn, TRUE);
            gtk_widget_show(d->cancel_btn);
            gtk_widget_hide(d->close_btn);
            d->has_changes = TRUE;
        }

        d->freeze_count--;
}

/*
 *	Set callback.
 */
static void set_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Cancel callback.
 */
static void cancel_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}

/*
 *	Channel Popup List Box "changed" signal callback.
 */
static void channel_changed_cb(
	pulistbox_struct *pulistbox, gint i, gpointer data
)
{
	CLDChannel channels;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Restore the level values for the new channel */
	channels = get_current_channels(d);
	if(channels == CLD_CHANNEL_RED)
	{
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->min_marker_spin),
		d->red_min_marker
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		d->red_mid_marker_coeff
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->max_marker_spin),
		d->red_max_marker
	    );
	}
	else if(channels == CLD_CHANNEL_GREEN)
	{
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->min_marker_spin),
		d->green_min_marker
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		d->green_mid_marker_coeff
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->max_marker_spin),
		d->green_max_marker
	    );
	}
	else if(channels == CLD_CHANNEL_BLUE)
	{
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->min_marker_spin),
		d->blue_min_marker
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		d->blue_mid_marker_coeff
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->max_marker_spin),
		d->blue_max_marker
	    );
	}
	else if(channels == CLD_CHANNEL_ALPHA)
	{
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->min_marker_spin),
		d->alpha_min_marker
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		d->alpha_mid_marker_coeff
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->max_marker_spin),
		d->alpha_max_marker
	    );
	}
	else
	{
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->min_marker_spin),
		(gfloat)0x00
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		1.0f
	    );
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->max_marker_spin),
		(gfloat)0xff
	    );
	}

	d->freeze_count--;

	/* Update the histogram with the new set of channels */
	if(channels != 0)
	{
	    imgview_image_struct *img = ImgViewGetImage(d->iv);
	    if(img != NULL)
		make_histogram(
		    d->histogram_data,
		    &d->histogram_data_max,
		    d->orig_img_data,
		    img->width, img->height, img->bpp, img->bpl,
		    channels
		);
	}

	update_draw_all(d, TRUE, TRUE);
}

/*
 *	GtkSpinButton "changed" signal callback.
 */
static void spin_changed_cb(GtkWidget *widget, gpointer data)
{
	CLDChannel channels;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	/* Get the current color channels */
	channels = get_current_channels(d);

	/* Get the marker values based on the current color channels */
	if(channels & CLD_CHANNEL_RED)
	{
	    get_current_marker_values(
		d,
		&d->red_min_marker,
		NULL,
		&d->red_mid_marker_coeff,
		&d->red_max_marker
	    );
	}
	if(channels & CLD_CHANNEL_GREEN)
	{
	    get_current_marker_values(
		d,
		&d->green_min_marker,
		NULL,
		&d->green_mid_marker_coeff,
		&d->green_max_marker
	    );
	}
	if(channels & CLD_CHANNEL_BLUE)
	{
	    get_current_marker_values(
		d,
		&d->blue_min_marker,
		NULL,
		&d->blue_mid_marker_coeff,
		&d->blue_max_marker
	    );
	}
	if(channels & CLD_CHANNEL_ALPHA)
	{
	    get_current_marker_values(
		d,
		&d->alpha_min_marker,
		NULL,
		&d->alpha_mid_marker_coeff,
		&d->alpha_max_marker
	    );
	}

	update_draw_all(d, TRUE, TRUE);
}

/*
 *	Preview GtkToggleButton "toggled" signal callback.
 */
static void preview_toggled_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->preview_check))
	    queue_apply_iv_image(d);
}

/*
 *	Reset callback.
 */
static void reset_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	reset_all_marker_values(d, TRUE, FALSE);

	if(d->has_changes)
	{
	    gtk_widget_set_sensitive(d->ok_btn, FALSE);
	    gtk_widget_hide(d->cancel_btn);
	    gtk_widget_show(d->close_btn);
	    d->has_changes = FALSE;
	}
}

/*
 *	Histogram GtkDrawingArea event signal callback.
 */
static gint histogram_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean key_press;
	guint keyval, keystate;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    if(d->iv != NULL)
	    {
		imgview_image_struct *img = ImgViewGetImage(d->iv);
		GDK_PIXMAP_UNREF(d->histogram_pm);
		d->histogram_pm = gdk_pixmap_new(
		    widget->window, configure->width, configure->height, -1
		);
		if(img != NULL)
		    make_histogram(
			d->histogram_data,
			&d->histogram_data_max,
			d->orig_img_data,
			img->width, img->height, img->bpp, img->bpl,
			get_current_channels(d)
		    );
		gtk_widget_queue_draw(widget);
	    }
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    histogram_draw(d);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget)) 
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
	    keystate = key->state;
#define STOP_EMIT(_w_)				\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  key_press ?					\
   "key_press_event" : "key_release_event"	\
 );
	    switch(keyval)
	    {
	      case GDK_Tab:
	      case GDK_KP_Tab:
	      case GDK_ISO_Left_Tab:
	      case GDK_3270_BackTab:
		if(key_press)
		{
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MIN:
			if(keystate & GDK_SHIFT_MASK)
			{
			    /* Do not handle or stop the signal
			     * emit and let the previous widget be
			     * focused
			     */
			}
			else
			{
			    d->drag_marker = CLD_MARKER_MID;
			    gtk_widget_queue_draw(d->histogram_da);
			    gtk_widget_queue_draw(d->gradient_da);
			    STOP_EMIT(widget);
			    status = TRUE;
			}
			break;
		      case CLD_MARKER_MID:
			if(keystate & GDK_SHIFT_MASK)
			{
			    d->drag_marker = CLD_MARKER_MIN;
			    gtk_widget_queue_draw(d->histogram_da);
			    gtk_widget_queue_draw(d->gradient_da);
			    STOP_EMIT(widget);
			    status = TRUE;
			}
			else
			{
			    d->drag_marker = CLD_MARKER_MAX;
			    gtk_widget_queue_draw(d->histogram_da);
			    gtk_widget_queue_draw(d->gradient_da);
			    STOP_EMIT(widget);
			    status = TRUE;
			}
			break;
		      case CLD_MARKER_MAX:
			if(keystate & GDK_SHIFT_MASK)
			{
			    d->drag_marker = CLD_MARKER_MID;
			    gtk_widget_queue_draw(d->histogram_da);
			    gtk_widget_queue_draw(d->gradient_da);
			    STOP_EMIT(widget);
			    status = TRUE;
			}
			else
			{
			    /* Do not handle or stop the signal
			     * emit and let the next widget be
			     * focused
			     */
			}
			break;
		    }
		}
		break;

	      case GDK_Up:
	      case GDK_KP_Up:
	      case GDK_Left:
	      case GDK_KP_Left:
		if(key_press)
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = (keystate & GDK_SHIFT_MASK) ?
			    adj->page_increment : adj->step_increment;
			gfloat v = adj->value - dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);
			changed_cb(widget, data);
		    }
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;

	      case GDK_Down:
	      case GDK_KP_Down:
	      case GDK_Right:
	      case GDK_KP_Right:
		if(key_press)
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = (keystate & GDK_SHIFT_MASK) ?
			    adj->page_increment : adj->step_increment;
			gfloat v = adj->value + dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);
			changed_cb(widget, data);
		    }
		}
		STOP_EMIT(widget);
		status = TRUE;
		break;
	    }



#undef STOP_EMIT
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(GTK_WIDGET_CAN_FOCUS(widget) && !GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    switch(button->button)
	    {
	      case GDK_BUTTON1:		/* Drag the nearest marker */
		if(!(d->drag_buttons & GDK_BUTTON1_MASK))
		{
		    /* Set up the pointer grab for the nearest marker */
		    const gint	x = (gint)button->x,
				x_start = (widget->allocation.width - 256) / 2;
		    gint min, mid, max, d_min, d_mid, d_max;
		    get_current_marker_values(d, &min, &mid, NULL, &max);

		    /* Calculate the absolute distances from the
		     * pointer to each marker
		     */
		    d_min = x - (x_start + min);
		    if(d_min < 0)
			d_min *= -1;
		    d_mid = x - (x_start + mid);
		    if(d_mid < 0)
			d_mid *= -1;
		    d_max = x - (x_start + max);
		    if(d_max < 0)
			d_max *= -1;

		    /* Determine which marker the pointer is
		     * closest to
		     */
		    if(d_max < d_min)
			d->drag_marker = (d_max < d_mid) ?
			    CLD_MARKER_MAX : CLD_MARKER_MID;
		    else
			d->drag_marker = (d_min < d_mid) ?
			    CLD_MARKER_MIN : CLD_MARKER_MID;

		    /* Set the marker value and update the
		     * displays
		     */
		    set_marker_value(d, d->drag_marker, x - x_start);
		    update_draw_all(d, TRUE, TRUE);

		    if(!gdk_pointer_is_grabbed())
			gdk_pointer_grab(
			    widget->window,
			    FALSE,
			    GDK_BUTTON_RELEASE_MASK |
				GDK_POINTER_MOTION_MASK,
			    NULL,
			    d->translate_cur,
			    button->time
			);

		    d->drag_buttons |= GDK_BUTTON1_MASK;
		    d->drag_x = (gint)button->x;
		    d->drag_y = (gint)button->y;
		}
		break;

	      case GDK_BUTTON4:		/* Decrement the selected marker */
		if(!(d->drag_buttons & GDK_BUTTON4_MASK))
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = MAX(
			    (adj->page_increment * 0.25f), 0.01f
			);
			gfloat v = adj->value - dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);
			changed_cb(widget, data);
		    }
		}
		break;

	      case GDK_BUTTON5:		/* Increment the selected marker */
		if(!(d->drag_buttons & GDK_BUTTON5_MASK))
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = MAX(
			    (adj->page_increment * 0.25f), 0.01f
			);
			gfloat v = adj->value + dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);
			changed_cb(widget, data);
		    }
		}
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:		/* Stop dragging the nearest marker */
		if(d->drag_buttons & GDK_BUTTON1_MASK)
		{
		    const gint x_start = (widget->allocation.width - 256) / 2;
		    const gint x = (gint)button->x;

		    /* Set the marker value and update the
		     * displays
		     */
		    set_marker_value(d, d->drag_marker, x - x_start);
		    update_draw_all(d, TRUE, FALSE);

		    /* Ungrab the pointer */
		    if(gdk_pointer_is_grabbed())
			gdk_pointer_ungrab(button->time);

		    d->drag_buttons &= ~GDK_BUTTON1_MASK;

		    changed_cb(widget, data);
		}
		break;

	      case GDK_BUTTON4:
		if(d->drag_buttons & GDK_BUTTON4_MASK)
		{
		    d->drag_buttons &= ~GDK_BUTTON4_MASK;
		}
		break;

	      case GDK_BUTTON5:
		if(d->drag_buttons & GDK_BUTTON5_MASK)
		{
		    d->drag_buttons &= ~GDK_BUTTON5_MASK;
		}
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(d->drag_buttons & GDK_BUTTON1_MASK)
	    {
		const gint x_start = (widget->allocation.width - 256) / 2;
		gint x, y;
		GdkModifierType mask;
		GdkWindow *window = motion->window;

		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
			window, &x, &y, &mask
		    );
		}
		else
		{
		    x = (gint)motion->x;
		    y = (gint)motion->y;
		}
		d->drag_x = x;
		d->drag_y = y;

		set_marker_value(d, d->drag_marker, x - x_start);
		update_draw_all(d, TRUE, TRUE);
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Gradient GtkDrawingArea event signal callback.
 */
static gint gradient_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean key_press;
	guint keyval;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;

	    GDK_PIXMAP_UNREF(d->gradient_pm);
	    d->gradient_pm = gdk_pixmap_new(
		widget->window, configure->width, configure->height, -1
	    );

	    d->gradient_img_width = configure->width;
	    d->gradient_img_height = configure->height;
	    g_free(d->gradient_img_data);
	    d->gradient_img_data = (guint8 *)g_malloc(
		d->gradient_img_width * 3 *
		d->gradient_img_height
	    );
	    rerender_gradient_data(d);

	    gtk_widget_queue_draw(widget);

	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    gradient_draw(d);
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget)) 
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		gtk_widget_queue_draw(widget);
		status = TRUE;
	    }
	    break;

	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    keyval = key->keyval;
#define STOP_EMIT(_w_)				\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(_w_),				\
  key_press ?					\
   "key_press_event" : "key_release_event"	\
 );
	    switch(keyval)
	    {




	    }

#undef STOP_EMIT
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(GTK_WIDGET_CAN_FOCUS(widget) && !GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    switch(button->button)
	    {
	      case GDK_BUTTON1:		/* Drag the nearest marker */
		if(!(d->drag_buttons & GDK_BUTTON1_MASK))
		{
		    /* Set up the pointer grab for the nearest marker */
		    const gint	x = (gint)button->x,
				x_start = (widget->allocation.width - 256) / 2;
		    gint min, mid, max, d_min, d_mid, d_max;
		    get_current_marker_values(d, &min, &mid, NULL, &max);

		    /* Calculate the absolute distances from the
		     * pointer to each marker
		     */
		    d_min = x - (x_start + min);
		    if(d_min < 0)
			d_min *= -1;
		    d_mid = x - (x_start + mid);
		    if(d_mid < 0)
			d_mid *= -1;
		    d_max = x - (x_start + max);
		    if(d_max < 0)
			d_max *= -1;

		    /* Determine which marker the pointer is
		     * closest to
		     */
		    if(d_max < d_min)
			d->drag_marker = (d_max < d_mid) ?
			    CLD_MARKER_MAX : CLD_MARKER_MID;
		    else
			d->drag_marker = (d_min < d_mid) ?
			    CLD_MARKER_MIN : CLD_MARKER_MID;

		    /* Set the marker value and update the
		     * displays
		     */
		    set_marker_value(d, d->drag_marker, x - x_start);
		    update_draw_all(d, TRUE, TRUE);

		    if(!gdk_pointer_is_grabbed())
			gdk_pointer_grab(
			    widget->window,
			    FALSE,
			    GDK_BUTTON_RELEASE_MASK |
				GDK_POINTER_MOTION_MASK,
			    NULL,
			    d->translate_cur,
			    button->time
			);

		    d->drag_buttons |= GDK_BUTTON1_MASK;
		    d->drag_x = (gint)button->x;
		    d->drag_y = (gint)button->y;
		}
		break;

	      case GDK_BUTTON4:		/* Decrement the selected marker */
		if(!(d->drag_buttons & GDK_BUTTON4_MASK))
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = MAX(
			    (adj->page_increment * 0.25f), 0.01f
			);
			gfloat v = adj->value - dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);

			changed_cb(widget, data);
		    }
		}
		break;

	      case GDK_BUTTON5:		/* Increment the selected marker */
		if(!(d->drag_buttons & GDK_BUTTON5_MASK))
		{
		    GtkWidget *w = NULL;
		    switch(d->drag_marker)
		    {
		      case CLD_MARKER_MAX:
			w = d->max_marker_spin;
			break;
		      case CLD_MARKER_MID:
			w = d->mid_marker_spin;
			break;
		      case CLD_MARKER_MIN:
			w = d->min_marker_spin;
			break;
		    }
		    if(w != NULL)
		    {
			GtkAdjustment *adj = gtk_spin_button_get_adjustment(
			    GTK_SPIN_BUTTON(w)
			);
			const gfloat dv = MAX(
			    (adj->page_increment * 0.25f), 0.01f
			);
			gfloat v = adj->value + dv;
			if(v > (adj->upper - adj->page_size))
			    v = adj->upper - adj->page_size;
			if(v < adj->lower)
			    v = adj->lower;
			gtk_spin_button_set_value(
			    GTK_SPIN_BUTTON(w), v
			);

			changed_cb(widget, data);
		    }
		}
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:		/* Stop dragging the nearest marker */
		if(d->drag_buttons & GDK_BUTTON1_MASK)
		{
		    const gint x_start = (widget->allocation.width - 256) / 2;
		    const gint x = (gint)button->x;

		    /* Set the marker value and update the
		     * displays
		     */
		    set_marker_value(d, d->drag_marker, x - x_start);
		    update_draw_all(d, TRUE, FALSE);

		    /* Ungrab the pointer */
		    if(gdk_pointer_is_grabbed())
			gdk_pointer_ungrab(button->time);

		    d->drag_buttons &= ~GDK_BUTTON1_MASK;

		    changed_cb(widget, data);
		}
		break;

	      case GDK_BUTTON4:
		if(d->drag_buttons & GDK_BUTTON4_MASK)
		{
		    d->drag_buttons &= ~GDK_BUTTON4_MASK;
		}
		break;

	      case GDK_BUTTON5:
		if(d->drag_buttons & GDK_BUTTON5_MASK)
		{
		    d->drag_buttons &= ~GDK_BUTTON5_MASK;
		}
		break;
	    }
	    status = TRUE;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(d->drag_buttons & GDK_BUTTON1_MASK)
	    {
		const gint x_start = (widget->allocation.width - 256) / 2;
		gint x, y;
		GdkModifierType mask;
		GdkWindow *window = motion->window;

		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
			window, &x, &y, &mask
		    );
		}
		else
		{
		    x = (gint)motion->x;
		    y = (gint)motion->y;
		}
		d->drag_x = x;
		d->drag_y = y;

		set_marker_value(d, d->drag_marker, x - x_start);
		update_draw_all(d, TRUE, TRUE);
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Apply to ImgView timeout callback.
 */
static gint apply_iv_image_tocb(gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	d->apply_to_iv_idle_id = 0;

	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	/* Apply the color levels to the ImgView's current frame and
	 * allow interruption
	 */
	apply_iv_image(
	    d,
	    FALSE,			/* Not verbose */
	    TRUE,			/* Interruptable */
	    TRUE			/* Current frame only */
	);

	return(FALSE);
}


/*
 *	Gets the current color channels.
 */
static CLDChannel get_current_channels(Dialog *d)
{
	return((CLDChannel)PUListGetItemData(
	    PUListBoxGetPUList(d->channel_pulistbox),
	    PUListBoxGetSelected(d->channel_pulistbox)
	));
}

/*
 *	Gets the marker values in units of bytes.
 */
static void get_current_marker_values(
	Dialog *d,
	gint *min_rtn,
	gint *mid_rtn,
	gfloat *mid_coeff_rtn,
	gint *max_rtn
)
{
	const gfloat mid_coeff = GTK_ENTRY_GET_VALUEF(d->mid_marker_spin);
	const gint min = GTK_ENTRY_GET_VALUEI(d->min_marker_spin),
	max = MAX(GTK_ENTRY_GET_VALUEI(d->max_marker_spin), min),
	mid = (gint)((gfloat)min + (mid_coeff / 2.0f *
		(gfloat)(max - min)
	    )
	);

	if(min_rtn != NULL)
	    *min_rtn = min;
	if(mid_rtn != NULL)
	    *mid_rtn = mid;
	if(mid_coeff_rtn != NULL)
	    *mid_coeff_rtn = mid_coeff;
	if(max_rtn != NULL)
	    *max_rtn = max;
}

/*
 *	Sets the marker to the value specified by i but does not
 *	apply the changes to the current ImgView image frame.
 *
 *	The marker specifies which marker to set.
 *
 *	The v specifies the marker's value in bytes.
 */
static void set_marker_value(
	Dialog *d,
	const CLDMarker marker,
	const gint v
)
{
	d->freeze_count++;

	/* Mid marker? */
	if(marker == CLD_MARKER_MID)
	{
	    /* When setting the mid marker from a byte value the
	     * byte value must be converted into a coefficient
	     * value because the mid marker is displayed as a
	     * coeffcient
	     */
	    gint min, mid, max;
	    gfloat mid_coeff;

	    get_current_marker_values(d, &min, &mid, NULL, &max);

	    mid_coeff = (gfloat)(v - min) / (gfloat)MAX((max - min), 1);

	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->mid_marker_spin),
		mid_coeff * 2.0f
	    );
	}
	else
	{
	    /* Max marker? */
	    if(marker == CLD_MARKER_MAX)
		gtk_spin_button_set_value(
		    GTK_SPIN_BUTTON(d->max_marker_spin),
		    (gfloat)v
		);
	    else
		gtk_spin_button_set_value(
		    GTK_SPIN_BUTTON(d->min_marker_spin),
		    (gfloat)v
		);
	}

	d->freeze_count--;

	spin_changed_cb(NULL, d);
}


/*
 *	Resets all the marker values.
 */
static void reset_all_marker_values(
	Dialog *d,
	const gboolean apply_to_iv_image,
	const gboolean interruptable
)
{
	d->red_min_marker = 0x00;
	d->red_mid_marker_coeff = 1.0f;
	d->red_max_marker = 0xff;
	d->green_min_marker = 0x00;
	d->green_mid_marker_coeff = 1.0f;
	d->green_max_marker = 0xff;
	d->blue_min_marker = 0x00;
	d->blue_mid_marker_coeff = 1.0f;
	d->blue_max_marker = 0xff;
	d->alpha_min_marker = 0x00;
	d->alpha_mid_marker_coeff = 1.0f;
	d->alpha_max_marker = 0xff;

	d->freeze_count++;
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->min_marker_spin),
	    (gfloat)0x00
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->mid_marker_spin),
	    (gfloat)1.0f
	);
	gtk_spin_button_set_value(
	    GTK_SPIN_BUTTON(d->max_marker_spin),
	    (gfloat)0xff
	);
	d->freeze_count--;

	update_draw_all(d, apply_to_iv_image, interruptable);
}


/*
 *	Redraws the Histogram GtkDrawingArea.
 */
static void histogram_draw(Dialog *d)
{
	const gint	arrow_width = 6,
			arrow_height = 6;
	gboolean has_focus;
	gint i, x, y, width, height, min, mid, max;
	gfloat coeff, m;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GdkDrawable *drawable;
	GdkGC *gc = d->gc;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w = d->histogram_da;

	window = w->window;
	has_focus = GTK_WIDGET_HAS_FOCUS(w);
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	pixmap = d->histogram_pm;
	drawable = (GdkDrawable *)((pixmap != NULL) ? pixmap : window);
	gdk_window_get_size(drawable, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	/* Draw the background */
	gdk_draw_rectangle(
	    drawable, style->base_gc[state], TRUE,
	    0, 0, width, height
	);

	/* Get the bounding positions in units of bytes */
	get_current_marker_values(d, &min, &mid, NULL, &max);

	/* Calculate x starting position so that the histogram is
	 * centered
	 */
	x = (width - 256) / 2;

	/* Get the highest histogram value */
	m = MAX((gfloat)d->histogram_data_max, 1.0f);

	/* Draw the histogram */
	for(i = 0; i < 256; i++)
	{
	    coeff = (gfloat)d->histogram_data[i] / (gfloat)m;

	    y = height - (gint)(height * coeff);

	    /* Draw value line */
	    gdk_draw_line(
		drawable, style->fg_gc[state],
		x, y, x, height
	    );

	    x++;
	}

	/* Draw the min, mid, and max markers */
	x = (width - 256) / 2;

	gdk_gc_set_function(gc, GDK_INVERT);

	/* Min marker line */
	if(d->drag_marker == CLD_MARKER_MIN)
	    gdk_draw_line(
		drawable, style->fg_gc[GTK_STATE_SELECTED],
		x + min, 0,
		x + min, height
	    );
	gdk_draw_line(
	    drawable, gc,
	    x + min, 0,
	    x + min, height
	);

	/* Mid marker line */
	if(d->drag_marker == CLD_MARKER_MID)
	    gdk_draw_line(
		drawable, style->fg_gc[GTK_STATE_SELECTED],
		x + mid, 0,
		x + mid, height
	    );
	gdk_draw_line(
	    drawable, gc,
	    x + mid, 0,
	    x + mid, height
	);

	/* Max marker line */
	if(d->drag_marker == CLD_MARKER_MAX)
	    gdk_draw_line(
		drawable, style->fg_gc[GTK_STATE_SELECTED],
		x + max, 0,
		x + max, height
	    );
	gdk_draw_line(
	    drawable, gc,
	    x + max, 0,
	    x + max, height
	);

	gdk_gc_set_function(gc, GDK_COPY);

	/* Min arrow */
	gtk_draw_arrow(
	    style,
	    drawable,
	    (has_focus && (d->drag_marker == CLD_MARKER_MIN)) ?
		GTK_STATE_SELECTED : state,
	    GTK_SHADOW_OUT,
	    GTK_ARROW_DOWN,
	    TRUE,
	    x + min - (arrow_width / 2),
	    0,
	    arrow_width, arrow_height
	);

	/* Mid arrow */
	gtk_draw_arrow(
	    style,
	    drawable,
	    (has_focus && (d->drag_marker == CLD_MARKER_MID)) ?
		GTK_STATE_SELECTED : state,
	    GTK_SHADOW_OUT,
	    GTK_ARROW_DOWN,
	    TRUE,
	    x + mid - (arrow_width / 2),
	    0,
	    arrow_width, arrow_height
	);

	/* Max arrow */
	gtk_draw_arrow(
	    style,
	    drawable,
	    (has_focus && (d->drag_marker == CLD_MARKER_MAX)) ?
		GTK_STATE_SELECTED : state,
	    GTK_SHADOW_OUT,
	    GTK_ARROW_DOWN,
	    TRUE,
	    x + max - (arrow_width / 2),
	    0,
	    arrow_width, arrow_height
	);


	/* Send drawable to window if drawable is not the window */
	if(drawable != window)
	    gdk_draw_pixmap(
		window, style->fg_gc[state], drawable,
		0, 0, 0, 0, width, height
	    );
}

/*
 *	Redraws the Gradient GtkDrawingArea.
 */
static void gradient_draw(Dialog *d)
{
	gint x, width, height, min, mid, max;
	GdkPixmap *pixmap;
	GdkWindow *window;
	GdkDrawable *drawable;
	GdkGC *gc = d->gc;
	GtkStateType state;
	GtkStyle *style;
	GtkWidget *w = d->gradient_da;

	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	pixmap = d->gradient_pm;
	drawable = (GdkDrawable *)((pixmap != NULL) ? pixmap : window);
	gdk_window_get_size(drawable, &width, &height);
	if((width <= 0) || (height <= 0))
	    return;

	/* Draw the background */
	gdk_draw_rectangle(
	    drawable, style->base_gc[state], TRUE,
	    0, 0, width, height
	);

	/* Get bounding positions in units of bytes */
	get_current_marker_values(d, &min, &mid, NULL, &max);

	/* Draw the gradient */
	if(d->gradient_img_data != NULL)
	    gdk_draw_rgb_image(
		drawable, gc,
		0, 0,
		d->gradient_img_width, d->gradient_img_height,
		GDK_RGB_DITHER_NONE,
		(guchar *)d->gradient_img_data,
		d->gradient_img_width * 3
	    );

	/* Draw the min, mid, and max markers */
	x = (width - 256) / 2;
	gdk_gc_set_function(gc, GDK_INVERT);
	gdk_draw_line(
	    drawable, gc,
	    x + min, 0,
	    x + min, height
	);
	gdk_draw_line(
	    drawable, gc,
	    x + mid, 0,
	    x + mid, height
	);
	gdk_draw_line(
	    drawable, gc,
	    x + max, 0,
	    x + max, height
	);
	gdk_gc_set_function(gc, GDK_COPY);

	/* Send drawable to window if drawable is not the window */
	if(drawable != window)
	    gdk_draw_pixmap(
		window, style->fg_gc[state], drawable,
		0, 0, 0, 0, width, height
	    );
}

/*
 *	Updates the gradient data, queues the drawing of the histogram
 *	and gradient GtkDrawingAreas, and updates the IV image (if
 *	preview is specified).
 *
 *	If interruptable is TRUE then this function will return
 *	when a pending GTK event detected and the color leveling will
 *	not be completed.
 */
static void update_draw_all(
	Dialog *d,
	const gboolean apply_to_iv_image,
	const gboolean interruptable
)
{
	/* Update the gradient data */
	rerender_gradient_data(d);

	/* Draw the histogram and gradient GtkDrawingAreas */
	gtk_widget_queue_draw(d->histogram_da);
	gtk_widget_queue_draw(d->gradient_da);

	/* Update the IV image if specified and that preview is
	 * specified
	 */
	if(apply_to_iv_image &&
	   GTK_TOGGLE_BUTTON_GET_ACTIVE(d->preview_check)
	)
	{
	    if(interruptable)
		queue_apply_iv_image(d);
	    else
		apply_iv_image(
		    d,
		    FALSE,		/* Not verbose */
		    FALSE,		/* Not interruptable */
		    TRUE		/* Current frame only */
		);
	}
}


/*
 *	Applies the color levels to the ImgView's current frame
 *	and redraws the ImgView.
 *
 *	If verbose is TRUE then the progress dialog will be used to
 *	display the progress. This will also make this call
 *	interruptable by the user (but not when new GTK events are
 *	pending unless interruptable is also set to TRUE).
 *
 *	If interruptable is TRUE then the call to iv_color_levels_apply()
 *	function will return before completely applying the color levels
 *	to the ImgView's current frame if/when pending GTK events are
 *	detected.
 *
 *	If cur_frame_only is TRUE then the color levels will only be
 *	applied to the current frame regardless of if "All Frames"
 *	has been checked or not.
 */
static void apply_iv_image(
	Dialog *d,
	const gboolean verbose,
	const gboolean interruptable,
	const gboolean cur_frame_only
)
{
	imgview_struct *iv = d->iv;
	imgview_image_struct *img = ImgViewGetImage(iv);
	gint (*progress_cb)(const gulong, const gulong, gpointer);
	CLDProgressData *progress_data;

	if(img == NULL)
	    return;

	/* Allocate the progress callback data if verbose */
	if(verbose)
	{
	    progress_cb = apply_iv_image_progress_cb;
	    progress_data = CLD_PROGRESS_DATA(g_malloc0(
		sizeof(CLDProgressData)
	    ));
	}
	else
	{
	    progress_cb = NULL;
	    progress_data = NULL;
	}

	/* Apply to all frames? */
	if(!cur_frame_only &&
	   GTK_TOGGLE_BUTTON_GET_ACTIVE(d->all_frames_check)
	)
	{
	    /* Apply color levels to all frames */
	    const gint m = ImgViewGetTotalFrames(iv);
	    gint i;

	    /* Need to restore the original image data before applying
	     * the color leveling to all frames
	     */
	    imgview_frame_struct *frame = ImgViewImageGetFrame(
		img, d->cur_frame
	    );
	    if(frame != NULL)
	    {
		if((frame->buf != NULL) && (d->orig_img_data != NULL) &&
		   (img->bpl > 0) && (img->height > 0)
		)
		    memcpy(
			frame->buf,
			d->orig_img_data,
			img->bpl * img->height
		    );
	    }

	    /* Map the progress dialog and set up the progress callback
	     * data
	     */
	    if((progress_data != NULL) && (m > 0))
	    {
		gchar *msg = (m > 1) ? g_strdup_printf(
"Applying color levels to frame %i of %i...",
		    0 + 1, m
		) : STRDUP(
"Applying color levels..."
		);
		GtkWidget *toplevel = d->toplevel;
		if(ProgressDialogIsQuery())
		    ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
		    "Applying Color Levels",
		    msg,
		    (const guint8 **)icon_color_levels_32x32_xpm,
		    "Stop"
		);
		g_free(msg);
		gdk_flush();

		progress_data->nframes = m;
		progress_data->unit_frame_coeff = 1.0f / (gfloat)m;
	    }

	    /* Iterate through each frame */
	    for(i = 0; i < m; i++)
	    {
		frame = ImgViewImageGetFrame(img, i);
		if(frame == NULL)
		    continue;

		if(progress_data != NULL)
		{
		    gchar *msg;

		    if(progress_data->aborted)
			break;
		    progress_data->cur_frame = i;

		    msg = (m > 1) ? g_strdup_printf(
"Applying color levels to frame %i of %i...",
			i + 1, m
		    ) : STRDUP(
"Applying color levels..."
		    );
		    ProgressDialogUpdate(
			NULL, msg, NULL, NULL,
			i * progress_data->unit_frame_coeff,
#ifdef PROGRESS_BAR_DEF_TICKS
			PROGRESS_BAR_DEF_TICKS,
#else
			25,
#endif
			TRUE
		    );
		    g_free(msg);
		}

		iv_color_levels_apply(
		    frame->buf, frame->buf,
		    img->width, img->height,
		    img->bpp, img->bpl,
		    d->red_min_marker, d->red_mid_marker_coeff, d->red_max_marker,
		    d->green_min_marker, d->green_mid_marker_coeff, d->green_max_marker,
		    d->blue_min_marker, d->blue_mid_marker_coeff, d->blue_max_marker,
		    d->alpha_min_marker, d->alpha_mid_marker_coeff, d->alpha_max_marker,
		    interruptable,
		    progress_cb, progress_data
		);
	    }
	}
	else
	{
	    /* Apply color levels to only the current frame */
	    imgview_frame_struct *frame = ImgViewImageGetFrame(
		img, d->cur_frame
	    );
	    if(frame != NULL)
	    {
		/* Map the progress dialog and set up the progress
		 * callback data
		 */
		if(progress_data != NULL)
		{
		    gchar *msg = STRDUP(
"Applying color levels..."
		    );
		    GtkWidget *toplevel = d->toplevel;
		    if(ProgressDialogIsQuery())
			ProgressDialogBreakQuery(TRUE);
		    ProgressDialogSetTransientFor(toplevel);
		    ProgressDialogMap(
			"Applying Color Levels",
			msg,
			(const guint8 **)icon_color_levels_32x32_xpm,
			"Stop"
		    );
		    g_free(msg);
		    gdk_flush();

		    progress_data->nframes = 1;
		    progress_data->unit_frame_coeff = 1.0f;
		}

		iv_color_levels_apply(
		    d->orig_img_data, frame->buf,
		    img->width, img->height,
		    img->bpp, img->bpl,
		    d->red_min_marker, d->red_mid_marker_coeff, d->red_max_marker,
		    d->green_min_marker, d->green_mid_marker_coeff, d->green_max_marker,
		    d->blue_min_marker, d->blue_mid_marker_coeff, d->blue_max_marker,
		    d->alpha_min_marker, d->alpha_mid_marker_coeff, d->alpha_max_marker,
		    interruptable,
		    progress_cb, progress_data
		);
	    }
	}

	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	g_free(progress_data);

	/* Redraw the ImgView */
	ImgViewQueueDrawView(iv);
}

/*
 *	Queues a call to apply_iv_image() that is interruptable
 *	and only to the current frame.
 */
static void queue_apply_iv_image(Dialog *d)
{
	GTK_TIMEOUT_REMOVE(d->apply_to_iv_idle_id);
#if 0
/* Do not use idle, it's too soon and a user event may follow */
	d->apply_to_iv_idle_id = gtk_idle_add_priority(
	    G_PRIORITY_LOW,
	    apply_iv_image_tocb, d
	);
#else
	d->apply_to_iv_idle_id = gtk_timeout_add(
	    400l,
	    apply_iv_image_tocb, d
	);
#endif
}

/*
 *	Restores the ImgView's current frame and redraws the ImgView.
 */
static void restore_iv_image(Dialog *d)
{
	imgview_struct *iv = d->iv;
	imgview_image_struct *img = ImgViewGetImage(iv);
	imgview_frame_struct *frame = ImgViewImageGetFrame(
	    img, d->cur_frame
	);
	if(frame == NULL)
	    return;

	/* Copy our original image data to the ImgView's current
	 * frame
	 */
	if((d->orig_img_data != NULL) && (frame->buf != NULL) &&
	   (img->bpl > 0) && (img->height > 0)
	)
	    memcpy(
		frame->buf,
		d->orig_img_data,
		img->bpl * img->height
	    );

	/* Redraw the ImgView */
	ImgViewQueueDrawView(iv);
}



static void make_histogram_from_rgba(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	    g = ((guint)pix_ptr[0] + (guint)pix_ptr[1] +
		(guint)pix_ptr[2] + (guint)pix_ptr[3]) / 4;
	    break;
	  case 3:
	    g = ((guint)pix_ptr[0] + (guint)pix_ptr[1] +
		(guint)pix_ptr[2]) / 3;
	    break;
	  case 2:
	    g = ((guint)pix_ptr[0] + (guint)pix_ptr[1]) / 2;
	    break;
	  case 1:
	    g = (guint)pix_ptr[0];
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}

static void make_histogram_from_rgb(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	  case 3:
	    g = ((guint)pix_ptr[0] + (guint)pix_ptr[1] +
		(guint)pix_ptr[2]) / 3;
	    break;
	  case 2:
	  case 1:
	    g = (guint)pix_ptr[0];
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}

static void make_histogram_from_red(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	  case 3:
	    g = (guint)pix_ptr[0];
	    break;
	  case 2:
	  case 1:
	    g = (guint)pix_ptr[0];
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}

static void make_histogram_from_green(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	  case 3:
	    g = (guint)pix_ptr[1];
	    break;
	  case 2:
	  case 1:
	    g = (guint)pix_ptr[0];
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}

static void make_histogram_from_blue(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	  case 3:
	    g = (guint)pix_ptr[2];
	    break;
	  case 2:
	  case 1:
	    g = (guint)pix_ptr[0];
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}

static void make_histogram_from_alpha(
	guint *histogram_data,
	guint *histogram_data_max,
	const guint8 *pix_ptr,
	const gint bpp
)
{
	guint v, g;

	switch(bpp)
	{
	  case 4:
	    g = (guint)pix_ptr[3];
	    break;
	  case 3:
	    return;
	    break;
	  case 2:
	    g = (guint)pix_ptr[1];
	    break;
	  case 1:
	    return;
	    break;
	  default:
	    return;
	    break;
	}

	v = histogram_data[g] + 1;
	if(v > *histogram_data_max)
	    *histogram_data_max = v;
	histogram_data[g] = v;
}


/*
 *	Creates a histogram from the specified image data.
 *
 *	The histogram_data specifies the histogram data which
 *	must be allocated to 256 * sizeof(guint).
 *
 *	The histogram_data_max specifies the pointer to the
 *	resulting max return value.
 *
 *	The buf, width, height, bpp, and bpl specifies the image
 *	data to create the histogram from.
 *
 *	The channels specifies which channel(s) to scan on the
 *	image to create the histogram from.
 */
static void make_histogram(
	guint *histogram_data,		/* 256 guints */
	guint *histogram_data_max,
	const guint8 *buf,
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const CLDChannel channels
)
{
	gint y;
	const guint8 *src_ptr, *src_end;
	void (*pix_func)(
	    guint *histogram_data,
	    guint *histogram_data_max,
	    const guint8 *pix_ptr,
	    const gint bpp
	);

	if((histogram_data == NULL) || (histogram_data_max == NULL) ||
	   (buf == NULL)
	)
	    return;

	if((channels & CLD_CHANNEL_RED) &&
	   (channels & CLD_CHANNEL_GREEN) &&
	   (channels & CLD_CHANNEL_BLUE) &&
	   (channels & CLD_CHANNEL_ALPHA)
	)
	    pix_func = make_histogram_from_rgba;
	else if((channels & CLD_CHANNEL_RED) &&
		(channels & CLD_CHANNEL_GREEN) &&
		(channels & CLD_CHANNEL_BLUE)
	)
	    pix_func = make_histogram_from_rgb;
	else if(channels == CLD_CHANNEL_RED)
	    pix_func = make_histogram_from_red;
	else if(channels == CLD_CHANNEL_GREEN)
	    pix_func = make_histogram_from_green;
	else if(channels == CLD_CHANNEL_BLUE)
	    pix_func = make_histogram_from_blue;
	else if(channels == CLD_CHANNEL_ALPHA)
	    pix_func = make_histogram_from_alpha;
	else
	    pix_func = NULL;

	/* Clear the histogram */
	*histogram_data_max = 0;
	memset(histogram_data, 0x00, 256 * sizeof(guint));

	if(pix_func == NULL)
	    return;

	/* Create the histogram from the image data */
	for(y = 0; y < height; y++)
	{
	    src_ptr = buf + (y * bpl);
	    src_end = src_ptr + (width * bpp);
	    while(src_ptr < src_end)
	    {
		pix_func(
		    histogram_data,
		    histogram_data_max,
		    src_ptr,
		    bpp
		);
		src_ptr += bpp;
	    }
	}
}


/*
 *	Renders the gradient image data.
 *
 *	The gradient data must already be allocated.
 */
static void rerender_gradient_data(Dialog *d)
{
	const gint bpp = 3;
	gint	i, x, y,
		width, height,
		bpl,
		min_marker, max_marker;
	guint8 *tar, *tar_ptr;
	gfloat mid_marker_coeff;
	CLDChannel channels;

	/* Get the gradient image data, must already be allocated */
	tar = d->gradient_img_data;
	if(tar == NULL)
	    return;

	channels = get_current_channels(d);
	width = d->gradient_img_width;
	height = d->gradient_img_height;
	bpl = width * bpp;

	/* Clear the image */
	memset(tar, 0x00, bpl * height);

	/* Begin drawing the gradient bars
	 *
	 * The gradient bars will be originated from the center
	 */
	x = (width - 256) / 2;
	for(i = 0; i < 256; i++)
	{
	    if(x < 0)
		continue;
	    if(x >= width)
		break;

	    for(y = 0; y < height; y++)
	    {
		tar_ptr = tar + (y * bpl) + (x * bpp);
		if(channels == CLD_CHANNEL_RED)
		{
		    tar_ptr[0] = (guint8)i;
		    tar_ptr[1] = (guint8)0;
		    tar_ptr[2] = (guint8)0;
		}
		else if(channels == CLD_CHANNEL_GREEN)
		{
		    tar_ptr[0] = (guint8)0;
		    tar_ptr[1] = (guint8)i;
		    tar_ptr[2] = (guint8)0;
		}
		else if(channels == CLD_CHANNEL_BLUE)
		{
		    tar_ptr[0] = (guint8)0;
		    tar_ptr[1] = (guint8)0;
		    tar_ptr[2] = (guint8)i;
		}
		else
		{
		    /* All else assume all visual channels or alpha */
		    tar_ptr[0] = (guint8)i;
		    tar_ptr[1] = (guint8)i;
		    tar_ptr[2] = (guint8)i;
		}
	    }

	    x++;
	}

	/* Get the marker values based on the current color channel */
	if(channels == CLD_CHANNEL_RED)
	{
	    min_marker = d->red_min_marker;
	    mid_marker_coeff = d->red_mid_marker_coeff;
	    max_marker = d->red_max_marker;
	}
	else if(channels == CLD_CHANNEL_GREEN)
	{
	    min_marker = d->green_min_marker;
	    mid_marker_coeff = d->green_mid_marker_coeff;
	    max_marker = d->green_max_marker;
	}
	else if(channels == CLD_CHANNEL_BLUE)
	{
	    min_marker = d->blue_min_marker;
	    mid_marker_coeff = d->blue_mid_marker_coeff;
	    max_marker = d->blue_max_marker;
	}
	else if(channels == CLD_CHANNEL_ALPHA)
	{
	    min_marker = d->alpha_min_marker;
	    mid_marker_coeff = d->alpha_mid_marker_coeff;
	    max_marker = d->alpha_max_marker;
	}
	else
	{
	    if(channels & CLD_CHANNEL_RED)
	    {
		min_marker = d->red_min_marker;
		mid_marker_coeff = d->red_mid_marker_coeff;
		max_marker = d->red_max_marker;
	    }
	    else if(channels & CLD_CHANNEL_GREEN)
	    {
		min_marker = d->green_min_marker;
		mid_marker_coeff = d->green_mid_marker_coeff;
		max_marker = d->green_max_marker;
	    }
	    else
	    {
		min_marker = d->blue_min_marker;
		mid_marker_coeff = d->blue_mid_marker_coeff;
		max_marker = d->blue_max_marker;
	    }
	}

	/* Apply the color levels to the gradient image data
	 *
	 * Each min, mid, and max color level value for red, green
	 * and blue will be passed as the same because we do not
	 * need to modify a specific channel since the color of the
	 * image has already been drawn such that modifying all
	 * the channels will only affect the drawn color
	 */
	iv_color_levels_apply(
	    tar, tar,
	    width, height,
	    bpp, bpl,
	    min_marker, mid_marker_coeff, max_marker,
	    min_marker, mid_marker_coeff, max_marker,
	    min_marker, mid_marker_coeff, max_marker,
	    0x00, 1.0f, 0xff,
	    FALSE,
	    NULL, NULL
	);
}


/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;

	/* GtkAccelGroup */
	d->accelgrp = accelgrp = gtk_accel_group_new();

	/* Toplevel GtkWindow */
	d->toplevel = toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_color_levels_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Set Button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Conjunto",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Serie",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Satz",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Serie",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Het Stel",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Jogo",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Sett",
#else
	    "Set",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(set_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);

	/* Cancel Button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close Button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
	    "Close",
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);
}


/*
 *	Queries user to edit color levels.
 *
 *	Returns TRUE on Set or FALSE on Cancel.
 */
gboolean iv_color_levels(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename,
	gboolean *all_frames
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gboolean got_response;
	gchar *s;
	GdkWindow *window;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	imgview_image_struct *img;
	imgview_frame_struct *frame;
	Dialog *d;

	if(iv == NULL)
	    return(FALSE);

	/* Create the dialog */
	d = DIALOG(g_malloc0(sizeof(Dialog)));
	if(ImgViewGetTotalFrames(iv) > 1)
	    s = g_strdup_printf(
		"Color Levels: %s: Frame %i",
		STRISEMPTY(filename) ? "Untitled" : filename,
		ImgViewGetCurrentFrame(iv) + 1
	    );
	else
	    s = g_strconcat(
		"Color Levels: ",
		STRISEMPTY(filename) ? "Untitled" : filename,
		NULL
	    );
	build_std_dialog(
	    d,
	    400, -1,
	    s,				/* Title */
	    ref_toplevel,
	    &toplevel, &client_vbox
	);
	g_free(s);
	window = (toplevel != NULL) ? toplevel->window : NULL;
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	d->iv = iv;
	d->gc = gdk_gc_new(window);
	d->histogram_pm = NULL;
	d->gradient_pm = NULL;
	memset(d->histogram_data, 0x00, 256 * sizeof(guint));
	d->histogram_data_max = 0;
	d->orig_img_data = NULL;
	d->cur_frame = ImgViewGetCurrentFrame(iv);
	d->drag_buttons = 0;
	d->drag_x = 0;
	d->drag_y = 0;
	d->drag_marker = CLD_MARKER_MIN;
	d->apply_to_iv_idle_id = 0;

	d->freeze_count++;

	/* Create a copy of the current frame */
	img = ImgViewGetImage(iv);
	frame = ImgViewImageGetFrame(img, d->cur_frame);
	if(frame != NULL)
	{
	    d->orig_img_data = (guint8 *)g_malloc(
		img->bpl * img->height
	    );
	    if((d->orig_img_data != NULL) && (frame->buf != NULL))
		memcpy(
		    d->orig_img_data,
		    frame->buf,
		    img->bpl * img->height
		);
	}

	/* Create the dialog's widgets */
	if(client_vbox != NULL)
	{
	    gint i, value_width;
	    GdkWindow *window = toplevel->window;
	    GtkAdjustment *adj;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget	*w,
			*parent = client_vbox,
			*parent2, *parent3, *parent4;
	    pulist_struct *pulist;
	    pulistbox_struct *pulistbox;

	    d->translate_cur = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent = w;


	    /* Left column GtkVBox */
	    w = gtk_vbox_new(TRUE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		window,
		(style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		(guint8 **)icon_color_levels_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Channel GtkHBox */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Channel GtkLabel */
	    w = gtk_label_new("Channel:");
	    gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Channel Popup List Box */
	    d->channel_pulistbox = pulistbox = PUListBoxNew(
		parent3,
		-1, -1
	    );
	    pulist = PUListBoxGetPUList(pulistbox);
	    PUListBoxSetChangedCB(
		pulistbox,
		channel_changed_cb, d
	    );
	    i = PUListAddItem(
		pulist, "Red, Green, Blue"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)(CLD_CHANNEL_RED |
		    CLD_CHANNEL_GREEN |
		    CLD_CHANNEL_BLUE
		)
	    );
	    i = PUListAddItem(
		pulist, "Red, Green, Blue, Alpha"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)(CLD_CHANNEL_RED |
		    CLD_CHANNEL_GREEN |
		    CLD_CHANNEL_BLUE |
		    CLD_CHANNEL_ALPHA
		)
	    );
	    i = PUListAddItem(
		pulist, "Red"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)CLD_CHANNEL_RED
	    );
	    i = PUListAddItem(
		pulist, "Green"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)CLD_CHANNEL_GREEN
	    );
	    i = PUListAddItem(
		pulist, "Blue"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)CLD_CHANNEL_BLUE
	    );
	    i = PUListAddItem(
		pulist, "Alpha"
	    );
	    PUListSetItemData(
		pulist, i,
		(gpointer)CLD_CHANNEL_ALPHA
	    );
	    PUListBoxSetLinesVisible(pulistbox, i + 1);
	    PUListBoxSelect(pulistbox, 0);
	    PUListBoxMap(pulistbox);

	    /* Histogram GtkFrame */
	    w = gtk_frame_new(NULL);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Histogram GtkDrawingArea */
	    d->histogram_da = w = gtk_drawing_area_new();
	    GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS);
	    gtk_widget_set_usize(w, -1, 100);
	    gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(histogram_event_cb), d
	    );
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_realize(w);
	    gtk_widget_grab_focus(w);	/* Grab focus */
	    gtk_widget_show(w);

	    w = gtk_frame_new(NULL);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Gradient GtkDrawingArea */
	    d->gradient_da = w = gtk_drawing_area_new();
/*	    GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS); */
	    gtk_widget_set_usize(w, -1, 15);
	    gtk_widget_add_events(
		w,
		GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
		GDK_FOCUS_CHANGE_MASK |
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_in_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "focus_out_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(gradient_event_cb), d
	    );
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_realize(w);
	    gtk_widget_show(w);


	    /* GtkHBox for the min, mid, and max GtkSpinButtons */
	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    value_width = 60;

	    /* Min */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Min:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w); 
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, -256.0f, 255.0f, 1.0f, 10.0f, 0.0f  
	    );
	    d->min_marker_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);

	    /* Mid */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Mid:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		1.0f, 0.0f, 2.0f, 0.01f, 0.1f, 0.0f
	    );
	    d->mid_marker_spin = w = gtk_spin_button_new(adj, 0.0, 2);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);

	    /* Max */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    w = gtk_label_new("Max:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* GtkSpinButton */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		255.0f, 0.0f, 511.0f, 1.0f, 10.0f, 0.0f
	    );
	    d->max_marker_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, value_width, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(changed_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "changed",
		GTK_SIGNAL_FUNC(spin_changed_cb), d
	    );
	    GUIEditableEndowPopupMenu(w, 0);
	    gtk_widget_show(w);


	    /* GtkHBox for the preview and reset buttons */
	    w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    d->preview_check = w = gtk_check_button_new_with_label(
		"Preview"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(preview_toggled_cb), d
	    );
	    GUISetWidgetTip(
		w,
		"Check this to update the image with the\
 color level values as they are being modified"
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(w, TRUE);
	    gtk_widget_show(w);

	    d->all_frames_check = w = gtk_check_button_new_with_label(
		"All Frames"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(
		w,
		"Check this to apply the changes to all the\
 frames instead of just the current frame"
	    );
	    if(all_frames != NULL)
		GTK_TOGGLE_BUTTON_SET_ACTIVE(w, *all_frames);
	    gtk_widget_set_sensitive(
		w,
		(ImgViewGetTotalFrames(iv) > 1) ? TRUE : FALSE
	    );
	    gtk_widget_show(w);

	    d->reset_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_revert_20x20_xpm,
		"Reset",
		NULL
	    );
	    gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(reset_cb), d
	    );
	    GUISetWidgetTip(
		w,
		"Click this to reset all the color levels to their\
 default values"
	    );
	    gtk_widget_show(w);

	}

	d->freeze_count--;

	reset_all_marker_values(d, FALSE, FALSE);

	/* Map the dialog */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	got_response = d->got_response;
	if(got_response)
	{
	    if(all_frames != NULL)
		*all_frames = GTK_TOGGLE_BUTTON_GET_ACTIVE(d->all_frames_check);

	    apply_iv_image(
		d,
		TRUE,		/* Verbose */
		FALSE,		/* Not interruptable */
		FALSE		/* Not current frame only */
	    );
	}
	else
	{
	    restore_iv_image(d);
	}

	/* Destroy widgets and related resources */
	GTK_TIMEOUT_REMOVE(d->apply_to_iv_idle_id);
	d->apply_to_iv_idle_id = 0;

	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(d->preview_check);
	GTK_WIDGET_DESTROY(d->all_frames_check);
	GTK_WIDGET_DESTROY(d->reset_btn);
	GTK_WIDGET_DESTROY(d->min_marker_spin);
	GTK_WIDGET_DESTROY(d->mid_marker_spin);
	GTK_WIDGET_DESTROY(d->max_marker_spin);
	GTK_WIDGET_DESTROY(d->gradient_da);
	GTK_WIDGET_DESTROY(d->histogram_da);
	PUListBoxDelete(d->channel_pulistbox);
	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	GDK_GC_UNREF(d->gc);
	GDK_PIXMAP_UNREF(d->gradient_pm);
	GDK_PIXMAP_UNREF(d->histogram_pm);
	GDK_CURSOR_DESTROY(d->translate_cur);
	g_free(d->gradient_img_data);
	g_free(d->orig_img_data);
	g_free(d);

	return(got_response);
}
