#include <string.h>
#include <gtk/gtk.h>
#include "guiutils.h"
#include "guirgbimg.h"
#include "desktop.h"


void iv_desktop_mesg(
	GdkWindow *root_window, GdkGC *gc,
	GtkStyle *style, GtkJustification justify,
	const gchar *title,
	GdkPixmap *icon_pixmap, GdkBitmap *icon_mask,
	const gchar *s
);

static guint8 *iv_desktop_combine_image_to_bg(
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
);

void iv_desktop_put_image_centered(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
);
void iv_desktop_put_image_tofit(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
);
void iv_desktop_put_image_tocover(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
);
void iv_desktop_put_image_tiled(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Displays a message on the root window.
 */
void iv_desktop_mesg(
	GdkWindow *root_window, GdkGC *gc,
	GtkStyle *style, GtkJustification justify,
	const gchar *title,
	GdkPixmap *icon_pixmap, GdkBitmap *icon_mask,
	const gchar *mesg
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gint	x, y, width, height,
		root_width, root_height,
		title_width = 0, title_height = 0,
		icon_width = 0, icon_height = 0,
		mesg_width = 0, mesg_height = 0;
	gint title_text_len = 0;
	GdkVisual *vis;
	GdkColormap *colormap;
	GdkColor fg, bg, light, dark, text, base, bg_active;
	GdkFont *font;
	GdkPixmap *pixmap;
	GdkDrawable *drawable;
	GtkStateType state = GTK_STATE_NORMAL;

	if((root_window == NULL) || (gc == NULL) || (style == NULL))
	    return;

	vis = gdk_window_get_visual(root_window);
	font = style->font;
	gdk_window_get_size(root_window, &root_width, &root_height);
	colormap = gdk_window_get_colormap(root_window);
	if((vis == NULL) || (colormap == NULL) ||
	   (root_width <= 0) || (root_height <= 0)
	)
	    return;

	/* Get title size */
	if((font != NULL) && !STRISEMPTY(title))
	{
	    gint line_len;
	    const gchar *s = title, *s_end;
	    GdkTextBounds b;

	    /* Seek s_end to end of line or end of string */
	    s_end = strchr(s, '\n');
	    if(s_end == NULL)
		s_end = s + STRLEN(s);

	    /* Calculate length of line */
	    line_len = s_end - s;

	    /* Get line bounds and update message size */
	    if(line_len > 0)
	    {
		gdk_text_bounds(font, s, line_len, &b);
		if(b.width > title_width)
		    title_width = b.width;
		title_height += font->ascent + font->descent;
		title_text_len = line_len;
	    }

	    /* Add padding */
	    title_width += 2 * border_minor;
	    title_height += 2 * border_minor;
	}

	/* Get icon size */
	if(icon_pixmap != NULL)
	    gdk_window_get_size(icon_pixmap, &icon_width, &icon_height);

	/* Get message size */
	if((font != NULL) && !STRISEMPTY(mesg))
	{
	    gint line_len;
	    const gchar *s = mesg, *s_end;
	    GdkTextBounds b;

	    do
	    {
		/* Seek s_end to end of line or end of string */
		s_end = strchr(s, '\n');
		if(s_end == NULL)
		    s_end = s + STRLEN(s);

		/* Calculate length of line */
		line_len = s_end - s;

		/* Get line bounds and update message size */
		if(line_len > 0)
		{
		    gdk_text_bounds(font, s, line_len, &b);
		    if(b.width > mesg_width)
			mesg_width = b.width;
		}
		mesg_height += font->ascent + font->descent;

		/* Seek s to start of next line or end of string */
		if(*s_end == '\0')
		    s = s_end;
		else
		    s = s_end + 1;
	    } while(*s != '\0');
	}

	/* Begin calculating total size */
	width = icon_width +
	    ((icon_width > 0) ? border_major : 0) +
	    mesg_width;
	if(title_width > width)
	    width = title_width;

	height = title_height +
	    ((title_height > 0) ? border_major : 0) +
	    ((icon_height > mesg_height) ? icon_height : mesg_height);

	/* Title width must span entire width */
	if((title_width > 0) && (title_width < width))
	    title_width = width;

	/* Calculate position */
	x = (root_width / 2) - (width / 2);
	y = (root_height / 2) - (height / 2);

	/* Create pixmap for drawing */
	pixmap = gdk_pixmap_new(
	    root_window, root_width, root_height,
	    (vis != NULL) ? vis->depth : -1
	);
	if(pixmap == NULL)
	    return;

	drawable = pixmap;

	/* Allocate colors */
	memcpy(&fg, &style->fg[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &fg);
	memcpy(&bg, &style->bg[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &bg);
	memcpy(&light, &style->light[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &light);
	memcpy(&dark, &style->dark[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &dark);
	memcpy(&text, &style->text[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &text);
	memcpy(&base, &style->base[state], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &base);

	memcpy(&bg_active, &style->bg[GTK_STATE_ACTIVE], sizeof(GdkColor));
	GDK_COLORMAP_ALLOC_COLOR(colormap, &bg_active);


	/* Set up GC */
	gdk_gc_set_foreground(gc, &fg);
	gdk_gc_set_foreground(gc, &bg);
	gdk_gc_set_font(gc, font);
	gdk_gc_set_function(gc, GDK_COPY);
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_gc_set_subwindow(gc, GDK_CLIP_BY_CHILDREN);

	/* Get current root window graphics and put it on to the
	 * pixmap
	 */
	gdk_window_copy_area(
	    drawable, gc,
	    0, 0,
	    root_window,
	    0, 0,
	    root_width, root_height
	);

	/* Draw background */
	if(TRUE)
	{
	    gint	x2 = x - border_major,
			y2 = y - border_major,
			width2 = width + (2 * border_major),
			height2 = height + (2 * border_major);

	    gdk_gc_set_foreground(gc, &bg);
	    gdk_draw_rectangle(
		drawable, gc, TRUE,
		x2, y2, width2, height2
	    );
	    gdk_gc_set_foreground(gc, &light);
	    gdk_draw_line(
		drawable, gc,
		x2, y2 + height2 - 1,
		x2, y2
	    );
	    gdk_draw_line(
		drawable, gc,
		x2, y2,
		x2 + width2 - 1, y2
	    );
	    gdk_gc_set_foreground(gc, &dark);
	    gdk_draw_line(
		drawable, gc,
		x2 + width2, y2 + 1,
		x2 + width2, y2 + height2 - 1
	    );
	    gdk_draw_line(
		drawable, gc,
		x2 + 1, y2 + height2 - 1,
		x2 + width2 - 1, y2 + height2 - 1
	    );
	}

	/* Draw title */
	if(!STRISEMPTY(title) && (title_text_len > 0))
	{
	    gint	x2 = x,
			y2 = y;
	    GdkTextBounds b;

	    gdk_gc_set_foreground(gc, &bg_active);
	    gdk_draw_rectangle(
		drawable, gc, TRUE,
		x2 - border_minor,
		y2 - border_minor,
		title_width + (2 * border_minor),
		title_height + (2 * border_minor)
	    );
	    gdk_gc_set_foreground(gc, &fg);
	    gdk_draw_rectangle(
		drawable, gc, FALSE,
		x2 - border_minor,
		y2 - border_minor,
		title_width + (2 * border_minor) - 1,
		title_height + (2 * border_minor) - 1
	    );

	    x2 = x + border_minor;
	    y2 = y + border_minor;
	    gdk_text_bounds(font, title, title_text_len, &b);
	    gdk_gc_set_foreground(gc, &text);
	    gdk_draw_text(
		drawable, font, gc,
		x2 + b.lbearing,
		y2 + font->ascent,
		title, title_text_len
	    );
	}

	/* Draw icon */
	if((icon_pixmap != NULL) && (icon_width > 0) && (icon_height > 0))
	{
	    gint	x2 = x,
			y2 = y + title_height +
		((title_height > 0) ? border_major : 0);

	    if(mesg_height > icon_height)
		y2 += (mesg_height / 2) - (icon_height / 2);

	    gdk_gc_set_clip_mask(gc, icon_mask);
	    gdk_gc_set_clip_origin(gc, x2, y2);
	    gdk_draw_pixmap(
		drawable, gc, icon_pixmap,
		0, 0,
		x2, y2,
		icon_width, icon_height
	    );
	    gdk_gc_set_clip_mask(gc, NULL);
	}

	/* Draw message */
	if(!STRISEMPTY(mesg))
	{
	    gint	x2 = x + icon_width +
		((icon_width > 0) ? border_major : 0),
			y2 = y + title_height +
		((title_height > 0) ? border_major : 0);
	    gint line_len;
	    const gchar *s = mesg, *s_end;
	    GdkTextBounds b;

	    if(icon_height > mesg_height)
		y2 += (icon_height / 2) - (mesg_height / 2);

	    gdk_gc_set_foreground(gc, &text);

	    do
	    { 
		/* Seek s_end to end of line or end of string */
		s_end = strchr(s, '\n');
		if(s_end == NULL)
		    s_end = s + STRLEN(s);

		/* Calculate length of line */
		line_len = s_end - s;

		/* Get line bounds and update message size */
		if(line_len > 0)
		{
		    gdk_text_bounds(font, s, line_len, &b);
		    gdk_draw_text(
			drawable, font, gc,
			x2 + b.lbearing,
			y2 + font->ascent,
			s, line_len
		    );
		}

		y2 += font->ascent + font->descent;
		 
		/* Seek s to start of next line or end of string */
		if(*s_end == '\0')
		    s = s_end;
		else
		    s = s_end + 1;
	    } while(*s != '\0');
	}




	/* Set pixmap to the root window and clear */
	gdk_window_set_back_pixmap(root_window, pixmap, FALSE);
	gdk_window_clear(root_window);

	GDK_PIXMAP_UNREF(pixmap);
	GDK_COLORMAP_FREE_COLOR(colormap, &fg);
	GDK_COLORMAP_FREE_COLOR(colormap, &bg);
	GDK_COLORMAP_FREE_COLOR(colormap, &light);
	GDK_COLORMAP_FREE_COLOR(colormap, &dark);
	GDK_COLORMAP_FREE_COLOR(colormap, &text);
	GDK_COLORMAP_FREE_COLOR(colormap, &base);
	GDK_COLORMAP_FREE_COLOR(colormap, &bg_active);
}


/*
 *	Combines the specified image data to the specified background
 *	and returns a new image
 *
 *	Inputs assumed valid.
 */
static guint8 *iv_desktop_combine_image_to_bg(
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
)
{
	const guint8 *src_line, *src_line_end, *src_ptr, *src_end;
	guint8 *tar_data, *tar_line, *tar_ptr;
	guint32 def_bg_color = 0xff000000;

	if(bg_color == NULL)
	    bg_color = (const guint8 *)&def_bg_color;

	if(bpl <= 0)
	    bpl = width * bpp;

	/* Allocate target image */
	tar_data = (guint8 *)g_malloc(bpl * height);
	if(tar_data == NULL)
	    return(NULL);

	if(bpp == 3)
	{
	    for(src_line = data,
		src_line_end = src_line + (bpl * height),
		tar_line = tar_data;
		src_line < src_line_end;
		src_line += bpl,
	        tar_line += bpl
	    )
	    {
	        src_ptr = src_line;
	        src_end = src_ptr + (width * bpp);
	        tar_ptr = tar_line;

	        while(src_ptr < src_end)
	        {
		    *tar_ptr++ = *src_ptr++;
		    *tar_ptr++ = *src_ptr++;
		    *tar_ptr++ = *src_ptr++;
		}
	    }
	}
	else if(bpp == 4)
	{
	    guint src_alpha, tar_alpha;
	    guint32 bg_color32 = *(const guint32 *)bg_color;
	    guint	bg_r = bg_color[0],
			bg_g = bg_color[1],
			bg_b = bg_color[2];

	    for(src_line = data,
		src_line_end = src_line + (bpl * height),
		tar_line = tar_data;
		src_line < src_line_end;
		src_line += bpl,
		tar_line += bpl
	    )
	    {
		src_ptr = src_line;
		src_end = src_ptr + (width * bpp);
		tar_ptr = tar_line;

		while(src_ptr < src_end)
		{
		    src_alpha = src_ptr[3];
		    if(src_alpha == 0xff)
		    {
			*(guint32 *)tar_ptr = *(const guint32 *)src_ptr;
			src_ptr += bpp;
			tar_ptr += bpp;
		    }   
		    else if(src_alpha > 0x00)
		    {
			tar_alpha = 0xff - src_alpha;
			*tar_ptr++ = (guint8)(
			    ((guint)(*src_ptr++) * src_alpha / 0xff) +
			    (bg_r * tar_alpha / 0xff)
			);
			*tar_ptr++ = (guint8)(
			    ((guint)(*src_ptr++) * src_alpha / 0xff) +
			    (bg_g * tar_alpha / 0xff)  
			);
			*tar_ptr++ = (guint8)(
			    ((guint)(*src_ptr++) * src_alpha / 0xff) +
			    (bg_b * tar_alpha / 0xff)  
			);
			*tar_ptr++ = 0xff;
			src_ptr++;
		    }
		    else
		    {
			*(guint32 *)tar_ptr = bg_color32;
			src_ptr += bpp;
			tar_ptr += bpp;
		    }
		}
	    }
	}

	return(tar_data);
}


/*
 *	Puts the image data to the root window centered.
 */
void iv_desktop_put_image_centered(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
)
{
	gint rx, ry, rwidth, rheight, rdepth;
	guint8 *comb_data;
	GdkRgbDither dither;
	GdkPixmap *pixmap;
	GdkGC *gc;

	if((root_window == NULL) || (data == NULL) ||
	   (width <= 0) || (height <= 0) || (bpp <= 0)
	)
	    return;

	if(bpl <= 0)
	    bpl = width * bpp;

	switch(quality)
	{
	  case 2:
	    dither = GDK_RGB_DITHER_MAX;
	    break;
	  case 1:
	    dither = GDK_RGB_DITHER_NORMAL;
	    break;
	  default:
	    dither = GDK_RGB_DITHER_NONE;
	    break;
	}

	/* Combine the specified image data to the background */
	comb_data = iv_desktop_combine_image_to_bg(
	    data, width, height, bpp, bpl,
	    bg_color
	);
	if(comb_data == NULL)
	    return;

	/* Get the size of the root GdkWindow */
	gdk_window_get_geometry(
	    root_window, &rx, &ry, &rwidth, &rheight, &rdepth
	);
	if((rwidth <= 0) || (rheight <= 0))
	{
	    g_free(comb_data);
	    return;
	}

	/* Create the GdkPixmap */
	pixmap = gdk_pixmap_new(root_window, rwidth, rheight, rdepth);
	if(pixmap == NULL)
	{
	    g_free(comb_data);
	    return;
	}

	/* Create the GdkGC */
	gc = gdk_gc_new(root_window);
	if(gc == NULL)
	{
	    g_free(comb_data);
	    GDK_PIXMAP_UNREF(pixmap);
	    return;
	}

	/* Clear the GdkPixmap with the background color */
	if(bg_color != NULL)
	{
	    GdkColormap *colormap = gdk_window_get_colormap(root_window);
	    GdkColor bg, *c = &bg;
	    GDK_COLOR_SET_BYTE(
		c, bg_color[0], bg_color[1], bg_color[2]
	    );
	    GDK_COLORMAP_ALLOC_COLOR(colormap, c);
	    gdk_gc_set_foreground(gc, c);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_draw_rectangle(
		(GdkDrawable *)pixmap, gc, TRUE,
		0, 0, rwidth, rheight
	    );
	    GDK_COLORMAP_FREE_COLOR(colormap, c);
	}

	/* Put the combined RGBA image data to the GdkPixmap */
	switch(bpp)
	{
	  case 4:
	    gdk_draw_rgb_32_image(
		(GdkDrawable *)pixmap, gc,
		(rwidth - width) / 2,
		(rheight - height) / 2,
		width, height,
		dither,
		(guchar *)comb_data,
		bpl
	    );
	    break;
	  case 3:
	    gdk_draw_rgb_image(
		(GdkDrawable *)pixmap, gc,
		(rwidth - width) / 2,
		(rheight - height) / 2,
		width, height,
		dither,
		(guchar *)comb_data,
		bpl
	    );
	    break;
	}

	g_free(comb_data);

	/* Set the GdkPixmap as the root GdkWindow's background */
	gdk_window_set_back_pixmap(root_window, pixmap, FALSE);

	GDK_PIXMAP_UNREF(pixmap);

	/* Redraw the root GdkWindow */
	gdk_window_clear(root_window);

	GDK_GC_UNREF(gc);
}

/*
 *	Puts the image data to the root window to fit, keeping
 *	original aspect.
 */
void iv_desktop_put_image_tofit(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
)
{
	guint8 *comb_data, *resized_data;
	gfloat itor_zoom;
	gint rx, ry, rwidth, rheight, rdepth;
	gint twidth, theight, tbpl, tbpp;
	GdkRgbDither dither;
	GdkPixmap *pixmap;
	GdkGC *gc;

	if((root_window == NULL) || (data == NULL) ||
	   (width <= 0) || (height <= 0) || (bpp <= 0)
	)
	    return;

	if(bpl <= 0)
	    bpl = width * bpp;

	switch(quality)
	{
	  case 2:
	    dither = GDK_RGB_DITHER_MAX;
	    break;
	  case 1:
	    dither = GDK_RGB_DITHER_NORMAL;
	    break;
	  default:
	    dither = GDK_RGB_DITHER_NONE;
	    break;
	}

	/* Combine the specified image data to the background */
	comb_data = iv_desktop_combine_image_to_bg(
	    data, width, height, bpp, bpl,
	    bg_color
	);
	if(comb_data == NULL)
	    return;

	/* Get the size of the root GdkWindow */
	gdk_window_get_geometry(
	    root_window, &rx, &ry, &rwidth, &rheight, &rdepth
	);
	if((rwidth <= 0) || (rheight <= 0))
	{
	    g_free(comb_data);
	    return;
	}

	/* Calculate image to root window zoom coefficient */
	itor_zoom = (gfloat)rwidth / (gfloat)width;
	if((itor_zoom * height) > rheight)
	    itor_zoom = (gfloat)rheight / (gfloat)height;

	/* Calculate the resized image geometry */
	twidth = (gint)(itor_zoom * width);
	theight = (gint)(itor_zoom * height);

	/* Allocate the resized image data */
	tbpp = bpp;
	tbpl = twidth * tbpp;
	resized_data = (guint8 *)g_malloc(tbpl * theight);
	if(resized_data == NULL)
	{
	    g_free(comb_data);
	    return;
	}

	/* Copy/resize the combined image data to the resized image
	 * data
	 */
	GUIImageBufferResize(
	    bpp,
	    comb_data, width, height, bpl,
	    resized_data, twidth, theight, tbpl,
	    NULL, NULL
	);

	g_free(comb_data);

	/* Create the GdkPixmap */
	pixmap = gdk_pixmap_new(root_window, rwidth, rheight, rdepth);
	if(pixmap == NULL)
	{
	    g_free(resized_data);
	    return;
	}

	/* Create the GdkGC */
	gc = gdk_gc_new(root_window);
	if(gc == NULL)
	{
	    g_free(resized_data);
	    GDK_PIXMAP_UNREF(pixmap);
	    return;
	}

	/* Clear the GdkPixmap with the background color */
	if(bg_color != NULL)
	{
	    GdkColormap *colormap = gdk_window_get_colormap(root_window);
	    GdkColor bg, *c = &bg;
	    GDK_COLOR_SET_BYTE(
		c, bg_color[0], bg_color[1], bg_color[2]
	    );
	    GDK_COLORMAP_ALLOC_COLOR(colormap, c);
	    gdk_gc_set_foreground(gc, c);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	    gdk_draw_rectangle(
		(GdkDrawable *)pixmap, gc, TRUE,
		0, 0, rwidth, rheight
	    );
	    GDK_COLORMAP_FREE_COLOR(colormap, c);
	}

	/* Put the resized image data to the GdkPixmap */
	switch(bpp)
	{
	  case 4:
	    gdk_draw_rgb_32_image(
		(GdkDrawable *)pixmap, gc,
		(rwidth - twidth) / 2,
		(rheight - theight) / 2,
		twidth, theight,
		dither,
		resized_data,
		tbpl
	    );
	    break;
	  case 3:
	    gdk_draw_rgb_image(
		(GdkDrawable *)pixmap, gc,
		(rwidth - twidth) / 2,
		(rheight - theight) / 2,
		twidth, theight,
		dither,
		resized_data,
		tbpl
	    );
	    break;
	}

	g_free(resized_data);

	/* Set the GdkPixmap as the root GdkWindow's background */
	gdk_window_set_back_pixmap(root_window, pixmap, FALSE);

	GDK_PIXMAP_UNREF(pixmap);

	/* Redraw the root GdkWindow */
	gdk_window_clear(root_window);

	GDK_GC_UNREF(gc);
}

/*
 *	Puts the image data to the root window to cover it completly,
 *	keeping original aspect but may obscure some parts of the image.
 */
void iv_desktop_put_image_tocover(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
)
{
	guint8 *comb_data, *resized_data;
	gfloat itor_zoom;
	gint rx, ry, rwidth, rheight, rdepth;
	gint twidth, theight, tbpl, tbpp;
	GdkRgbDither dither;
	GdkPixmap *pixmap;
	GdkGC *gc;

	if((root_window == NULL) || (data == NULL) ||
	   (width <= 0) || (height <= 0) || (bpp <= 0)
	)
	    return;

	if(bpl <= 0)
	    bpl = width * bpp;

	switch(quality)
	{
	  case 2:
	    dither = GDK_RGB_DITHER_MAX;
	    break;
	  case 1:
	    dither = GDK_RGB_DITHER_NORMAL;
	    break;
	  default:
	    dither = GDK_RGB_DITHER_NONE;
	    break;
	}

	/* Combine the specified image data to the background */
	comb_data = iv_desktop_combine_image_to_bg(
	    data, width, height, bpp, bpl,
	    bg_color
	);
	if(comb_data == NULL)
	    return;

	/* Get the size of the root GdkWindow */
	gdk_window_get_geometry(
	    root_window, &rx, &ry, &rwidth, &rheight, &rdepth
	);
	if((rwidth <= 0) || (rheight <= 0))
	{
	    g_free(comb_data);
	    return;
	}

	/* Calculate image to root window zoom coefficient */
	itor_zoom = (gfloat)rwidth / (gfloat)width;
	if((itor_zoom * height) < rheight)
	    itor_zoom = (gfloat)rheight / (gfloat)height;

	/* Calculate target image geometry */
	twidth = (gint)(itor_zoom * width);
	theight = (gint)(itor_zoom * height);

	/* Allocate the resized image data */
	tbpp = bpp;
	tbpl = twidth * tbpp;
	resized_data = (guint8 *)g_malloc(tbpl * theight);
	if(resized_data == NULL)
	{
	    g_free(comb_data);
	    return;
	}

	/* Copy/resize the combined image data to the resized image
	 * data
	 */
	GUIImageBufferResize(
	    bpp,
	    comb_data, width, height, bpl,
	    resized_data, twidth, theight, tbpl,
	    NULL, NULL
	);

	g_free(comb_data);

	/* Create tempory GdkPixmap the exact size to match the root
	 * window.
	 */
	pixmap = gdk_pixmap_new(root_window, rwidth, rheight, rdepth);
	if(pixmap == NULL)
	{
	    g_free(resized_data);
	    return;
	}

	/* Create the GdkGC */
	gc = gdk_gc_new(root_window);
	if(gc == NULL)
	{
	    g_free(resized_data);
	    GDK_PIXMAP_UNREF(pixmap);
	    return;
	}

	/* Put the resized image data to the GdkPixmap */
	switch(bpp)
	{
	  case 4:
	    gdk_draw_rgb_32_image(
		pixmap, gc,
		(rwidth - twidth) / 2,
		(rheight - theight) / 2,
		twidth, theight,
		dither,
		resized_data,
		tbpl
	    );
	    break;
	  case 3:
	    gdk_draw_rgb_image(
		pixmap, gc,
		(rwidth - twidth) / 2,
		(rheight - theight) / 2,
		twidth, theight,
		dither,
		resized_data,
		tbpl
	    );
	    break;
	}

	g_free(resized_data);

	/* Set the GdkPixmap as the root GdkWindow's background */
	gdk_window_set_back_pixmap(root_window, pixmap, FALSE);

	GDK_PIXMAP_UNREF(pixmap);

	/* Redraw the root GdkWindow */
	gdk_window_clear(root_window);

	GDK_GC_UNREF(gc);
}

/*
 *      Put given image data to the root window tiled.
 */
void iv_desktop_put_image_tiled(
	GdkWindow *root_window,
	gint quality,			/* 0 to 2 */
	const guint8 *data,
	gint width, gint height, gint bpp, gint bpl,
	const guint8 *bg_color
)
{
	guint8 *comb_data;
	gint rx, ry, rwidth, rheight, rdepth;
	GdkRgbDither dither;
	GdkPixmap *pixmap;
	GdkGC *gc;

	if((root_window == NULL) || (data == NULL) ||
	   (width <= 0) || (height <= 0) || (bpp <= 0)
	)
	    return;

	if(bpl <= 0)
	    bpl = width * bpp;

	switch(quality)
	{
	  case 2:
	    dither = GDK_RGB_DITHER_MAX;
	    break;
	  case 1:
	    dither = GDK_RGB_DITHER_NORMAL;
	    break;
	  default:
	    dither = GDK_RGB_DITHER_NONE;
	    break;
	}

	/* Combine the specified image data to the background */
	comb_data = iv_desktop_combine_image_to_bg(
	    data, width, height, bpp, bpl,
	    bg_color
	);
	if(comb_data == NULL)
	    return;

	/* Get the size of the root GdkWindow */
	gdk_window_get_geometry(
	    root_window, &rx, &ry, &rwidth, &rheight, &rdepth
	);
	if((rwidth <= 0) || (rheight <= 0))
	{
	    g_free(comb_data);
	    return;
	}

	/* Create the GdkPixmap */
	pixmap = gdk_pixmap_new(root_window, rwidth, rheight, rdepth);
	if(pixmap == NULL)
	{
	    g_free(comb_data);
	    return;
	}

	/* Create the GdkGC */
	gc = gdk_gc_new(root_window);
	if(gc == NULL)
	{
	    g_free(comb_data);
	    GDK_PIXMAP_UNREF(pixmap);
	    return;
	}

	if(TRUE)
	{
	    const gint	x_start = (((rwidth / 2) - (width / 2)) % width) -
			     width,
			y_start = (((rheight / 2) - (height / 2)) % height) -
			     height;
	    gint x, y;

	    /* Iterate through tiles */
	    for(y = y_start; y < rheight; y += height)
	    {
		for(x = x_start; x < rwidth; x += width)
		{
		    /* Put given image to tiled position */
		    switch(bpp)
		    {
		      case 4:
			gdk_draw_rgb_32_image(
			    (GdkDrawable *)pixmap, gc,
			    x, y, width, height,
			    dither,
			    (guchar *)comb_data,
			    bpl
			);
			break;
		      case 3:
			gdk_draw_rgb_image(
			    (GdkDrawable *)pixmap, gc,
			    x, y, width, height,
			    dither,
			    (guchar *)comb_data,
			    bpl
			);
			break;
		    }
		}
	    }
	}

	g_free(comb_data);

	/* Set the GdkPixmap as the root GdkWindow's background */
	gdk_window_set_back_pixmap(root_window, pixmap, FALSE);

	GDK_PIXMAP_UNREF(pixmap);

	/* Redraw the root GdkWindow */
	gdk_window_clear(root_window);

	GDK_GC_UNREF(gc);
}
