#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "imgio.h"
#include "guiutils.h"
#include "guirgbimg.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "fb.h"
#include "pdialog.h"

#include "imgview.h"
#include "framesdlg.h"
#include "config.h"

#include "images/icon_animate_48x48.xpm"
#include "images/icon_animate_32x32.xpm"

#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_play_20x20.xpm"
#include "images/icon_pause_20x20.xpm"
#include "images/icon_close_20x20.xpm"


typedef struct _Dialog			Dialog;
#define DIALOG(p)			((Dialog *)(p))
typedef struct _FrameData		FrameData;
#define FRAME_DATA(p)			((FrameData *)(p))
typedef struct _ProgressData		ProgressData;
#define PROGRESS_DATA(p)		((ProgressData *)(p))


/*
 *	Image fitting types:
 */
typedef enum {
	IMG_FIT_RESIZE,			/* Resize and do not maintain aspect */
	IMG_FIT_RESIZE_MAINTAIN_ASPECT,
	IMG_FIT_RESIZE_MAINTAIN_ASPECT_TO_COVER,
	IMG_FIT_CENTERED,
	IMG_FIT_UPPER_LEFT,
	IMG_FIT_UPPER_RIGHT,
	IMG_FIT_LOWER_RIGHT,
	IMG_FIT_LOWER_LEFT
} FDImgFitType;
#define IMG_FIT_STRINGS	{				\
	"Resize",					\
	"Resize & Maintain Aspect",			\
	"Resize & Maintain Aspect To Cover",		\
	"Centered",					\
	"Upper Left",					\
	"Upper Right",					\
	"Lower Right",					\
	"Lower Left",					\
	NULL						\
}


/* Callbacks */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void add_cb(GtkWidget *widget, gpointer data);
static void edit_cb(GtkWidget *widget, gpointer data);
static void remove_cb(GtkWidget *widget, gpointer data);
static void shift_up_cb(GtkWidget *widget, gpointer data);
static void shift_down_cb(GtkWidget *widget, gpointer data);
static void play_cb(GtkWidget *widget, gpointer data);
static void pause_cb(GtkWidget *widget, gpointer data);
static void close_cb(GtkWidget *widget, gpointer data);
static gint clist_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void clist_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void clist_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void clist_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void clist_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static gchar *pdialog_browse_cb(
	gpointer pdialog, gpointer data, gint prompt_num
);
static gint imgio_progress_cb(
	gpointer data,
	gint pi, gint pm,
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba
);
static gint resize_progress_cb(
	const gulong i, const gulong m, gpointer data
);

/* GtkCList */
static void clist_row_update(
	GtkCList *clist, const gint row,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const gulong delay
);
static void clist_row_update_frame_num(
	GtkCList *clist, const gint row
);
static gint clist_row_insert(
	GtkCList *clist, const gint row,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const gulong delay
);

/* Frame image setting */
static gint set_frame_from_image_file(
	Dialog *d,
	imgview_image_struct *img,
	const gint frame_num, imgview_frame_struct *frame,
	const gchar *path,
	const FDImgFitType img_fit,
	const gboolean append_multi,
	const gchar *progress_title,
	const gboolean verbose
);

/* Frame ops */
static void add_frame(Dialog *d);
static void edit_frame(Dialog *d, const gboolean from_add);
static void remove_frame(Dialog *d);
static void shift_frame_up(Dialog *d);
static void shift_frame_down(Dialog *d);

/* Dialog Updating */
static void set_busy(Dialog *d, const gboolean busy);
static void update_stats_label(Dialog *d);
static void update(Dialog *d);

/* Utils */
static FrameData *frame_data_new(
	const gulong delay
);
static void frame_data_delete(FrameData *fd);
static FDImgFitType get_img_fit_type_from_s(const gchar *s);
static gchar *img_fit_string(const FDImgFitType img_fit);

static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn,
	GtkAccelGroup **accelgrp_rtn
);

gboolean iv_edit_frames(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename
);


/*
 *	Dialog:
 */
struct _Dialog {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count,
			busy_count;
	gboolean	has_changes;
	imgview_struct	*iv;		/* Pointer back to the ImgView */
	GdkCursor	*busy_cur;

	GtkWidget	*clist,
			*stats_label,
			*add_btn,
			*edit_btn,
			*remove_btn,
			*shift_up_btn,
			*shift_down_btn,
			*play_btn,
			*pause_btn,
			*menu,
			*add_mi,
			*edit_mi,
			*remove_mi,
			*shift_up_mi,
			*shift_down_mi,
			*play_mi,
			*pause_mi;

	FDImgFitType	img_fit;
};


/*
 *	Frame Data:
 */
struct _FrameData {
	gulong		delay;
};

/*
 *	Progress Data:
 */
struct _ProgressData {
	Dialog		*d;
	gint		cur_stage,
			nstages;
	gfloat		unit_stage_coeff;
	gboolean	aborted;
};


/* Size of frame icon in the GtkCList */
#define FRAME_ICON_WIDTH		48
#define FRAME_ICON_HEIGHT		48


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return(FALSE);

	gtk_main_quit();
	return(TRUE);
}

/*
 *	Add callback.
 */
static void add_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	add_frame(d);
}

/*
 *	Edit callback.
 */
static void edit_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	edit_frame(d, FALSE);
}

/*
 *	Remove callback.
 */
static void remove_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	remove_frame(d);
}

/*
 *	Shift up callback.
 */
static void shift_up_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	shift_frame_up(d);
}

/*
 *	Shift down callback.
 */
static void shift_down_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	shift_frame_down(d);
}

/*
 *	Play callback.
 */
static void play_cb(GtkWidget *widget, gpointer data)
{
	imgview_struct *iv;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	iv = d->iv;
	if(iv == NULL)
	    return;

	if(!ImgViewIsPlaying(iv) && (ImgViewGetTotalFrames(iv) > 1))
	    ImgViewPlay(iv);

	update(d);
}

/*
 *	Pause callback.
 */
static void pause_cb(GtkWidget *widget, gpointer data)
{
	imgview_struct *iv;
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	iv = d->iv;
	if(iv == NULL)
	    return;

	if(ImgViewIsPlaying(iv))
	    ImgViewPause(iv);

	update(d);
}

/*
 *	Close callback.
 */
static void close_cb(GtkWidget *widget, gpointer data)
{
	Dialog *d = DIALOG(data);
	if(d == NULL)
	    return;

	gtk_main_quit();
}

/*
 *	GtkCList event signal callback.
 */
static gint clist_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(button->button == GDK_BUTTON3)
	    {
		GtkWidget *w = d->menu;
		if(w != NULL)
		    gtk_menu_popup(
			GTK_MENU(w), NULL, NULL,
			NULL, NULL,
			button->button, button->time
		    );
		status = TRUE;
	    }
	    break;
	}

	return(status);
}

/*
 *	GtkCList "drag_data_get" signal callback.
 */
static void clist_drag_data_get_cb(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint buf_len;
	guint8 *buf;
	GList *glist;
	GtkCList *clist;
	Dialog *d = DIALOG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->clist);

	/* Format the data as:
	 *
	 *	<frames_clist>		GtkWidget *
	 *	frame_num_1		gint
	 *	frame_num_2		gint
	 *	frame_num_#		gint
	 *	...
	 */

	buf_len = sizeof(GtkWidget *) +
	    (g_list_length(clist->selection) * sizeof(gint));

	buf = (guint8 *)g_malloc(buf_len);
	if(buf != NULL)
	{
	    guint8 *buf_ptr = buf;

	    *((GtkWidget **)buf_ptr) = GTK_WIDGET(clist);
	    buf_ptr += sizeof(GtkWidget *);

	    for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		*(gint *)buf_ptr = (gint)glist->data;
		buf_ptr += sizeof(gint);
	    }

	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		buf,			/* Data */
		buf_len			/* Length */
	    );
	    data_sent = TRUE;
	    g_free(buf);
	}
	 
	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,			/* Bits Per Character */
		s,			/* Data */
		STRLEN(s)               /* Length */
	    );
	}
}

/*
 *	GtkCList "drag_data_received" signal callback.
 */
static void clist_drag_data_received_cb(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row, column;
	GtkCList *clist;
	imgview_struct *iv;
	imgview_image_struct *img;
	Dialog *d = DIALOG(data);
	if((widget == NULL) || (dc == NULL) || (selection_data == NULL) ||
	   (d == NULL) || CDialogIsQuery() || FileBrowserIsQuery() ||
	   PDialogIsQuery()
	)
	    return;

	iv = d->iv;
	clist = GTK_CLIST(d->clist);

	if((selection_data->data == NULL) || (iv == NULL))
	    return;

	img = ImgViewGetImage(iv);
	if(img == NULL)
	    return;

	set_busy(d, TRUE);

	/* Calculate the row and column based on the drop coordinates */
	if(!gtk_clist_get_selection_info(
	    clist,
	    x,
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	))
	{
	    row = -1;
	    column = 0;
	}

	/* Handle by data type */
	if(info == IV_DND_INFO_FRAME_INDEX)
	{
	    gint buf_len = selection_data->length;
	    const guint8	*buf_ptr = (const guint8 *)selection_data->data,
				*buf_end = buf_ptr + buf_len;
	    GtkWidget *src_w;

	    if((buf_ptr + sizeof(GtkWidget *)) <= buf_end)
	    {
		src_w = *(GtkWidget **)buf_ptr;
		buf_ptr += sizeof(GtkWidget *);
	    }
	    else
		src_w = NULL;

	    if(src_w == GTK_WIDGET(clist))
	    {
		gint	i, n,
			frame_num, ins_frame_num, new_frame_num,
			new_row;
		GList *glist, *frames_list;
		imgview_frame_struct *frame;

		/* Create the selected frames list */
		frames_list = NULL;
		while((buf_ptr + sizeof(gint)) <= buf_end)
		{
		    frame_num = *((gint *)buf_ptr);
		    frame = ImgViewFrameCopy(
			img,
			ImgViewImageGetFrame(img, frame_num)
		    );
		    if(frame != NULL)
			frames_list = g_list_append(
			    frames_list,
			    frame
			);
		    buf_ptr += sizeof(gint);
		}

		gtk_clist_freeze(clist);

		/* Insert each frame */
		ins_frame_num = row;
		for(glist = frames_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    frame = IMGVIEW_FRAME(glist->data);
		    if(frame == NULL)
			continue;

		    new_frame_num = ImgViewImageInsertFrame(
			img, ins_frame_num,
			frame->buf,		/* Transfered */
			frame->delay
		    );
		    frame->buf = NULL;
		    if(new_frame_num < 0)
			break;

		    frame = ImgViewImageGetFrame(img, new_frame_num);
		    if(frame == NULL)
			break;

		    new_row = clist_row_insert(
			clist, ins_frame_num,
			frame->buf, img->width, img->height, img->bpl,
			frame->delay
		    );
		    if(new_row < 0)
			break;

		    if(ins_frame_num > -1)
			ins_frame_num++;
		}		

		/* Update all the rows' frame number */
		n = clist->rows;
		for(i = 0; i < n; i++)
		    clist_row_update_frame_num(clist, i);

		gtk_clist_thaw(clist);

		/* Delete the selected frames list */
		if(frames_list != NULL)
		{
		    g_list_foreach(
			frames_list, (GFunc)ImgViewFrameDelete, NULL
		    );
		    g_list_free(frames_list);
		}

		d->has_changes = TRUE;

		if(!ImgViewIsPlaying(iv))
		{
		    iv->cur_frame_num = -1;
		    if(clist->selection_end != NULL)
			ImgViewSeek(iv, (gint)clist->selection_end->data);
		}

		update_stats_label(d);
		update(d);
	    }
	}
	else if((info == DND_INFO_TEXT_PLAIN) ||
	        (info == DND_INFO_TEXT_URI_LIST) ||
	        (info == DND_INFO_STRING)
	)
	{
	    const gchar *img_fit_strings[] = IMG_FIT_STRINGS;
	    gboolean got_response;
	    gint i, strc;
	    const gchar	*pfx = "file://",
			*url = selection_data->data,
			*path,
			*img_fit_s = NULL;
	    gchar *mesg, **strv;
	    gulong delay = 0l;
	    GList *img_fit_list;
	    GtkWidget *toplevel = d->toplevel;

	    /* Parse the path from the url */
	    if(g_strcasepfx(url, pfx))
		path = url + STRLEN(pfx);
	    else
		path = url;

	    /* Query the user for the add frames values */
	    PDialogDeleteAllPrompts();
	    gtk_window_set_transient_for(
		GTK_WINDOW(PDialogGetToplevel()),
		GTK_WINDOW(toplevel)
	    );
	    PDialogSetSize(350, -1);
	    img_fit_list = NULL;
	    for(i = 0; img_fit_strings[i] != NULL; i++)
		img_fit_list = g_list_append(
		    img_fit_list, STRDUP(img_fit_strings[i])
		);
	    PDialogAddPromptPopupList(
		NULL, "Fit Image:",
		img_fit_list,
		d->img_fit,
		-1
	    );
	    PDialogSetPromptTip(
		0,
"Select the method in which the image will be resized and\
 positioned in the frame"
	    );
	    g_list_foreach(img_fit_list, (GFunc)g_free, NULL);
	    g_list_free(img_fit_list);
	    PDialogAddPromptSpin(  
		NULL, "Delay:",
		(gfloat)250.0f,
		0.0f,
		(gfloat)((guint)-1),	/* Reasonable max delay, right? */
		1.0f, 5.0f,
		1.0f, 0
	    );
	    PDialogSetPromptTip(
		1,
"Set the delay in milliseconds which specifies how long this\
 frame will be displayed before displaying the next frame"
	    );
	    mesg = g_strdup_printf("File: %s", g_basename(path));
	    strv = PDialogGetResponse(
		"Insert New Frame",
		mesg, NULL,
		PDIALOG_ICON_EDIT,
		"Insert", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	    );
	    g_free(mesg);
	    if(strv != NULL)
	    {
		got_response = TRUE;
		img_fit_s = (strc > 0) ? strv[0] : NULL;
		delay = (strc > 1) ? ATOL(strv[1]) : 250l;

		/* Get the image fit type */
		d->img_fit = get_img_fit_type_from_s(img_fit_s);
	    }
	    else
	    {
		got_response = FALSE;
	    }

	    PDialogSetTransientFor(NULL);
	    PDialogDeleteAllPrompts();

	    if(got_response)
	    {
		gint i, n, frame_num;
		imgview_frame_struct *frame;

		/* Warn if the delay is set to 0 */
		if(delay == 0l)
		{
		    CDialogSetTransientFor(toplevel);
		    CDialogGetResponse(
			"Delay Warning",
"The delay value is set to 0, which means that\n\
this frame will never be visible when played.",
			NULL,
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK,
			CDIALOG_BTNFLAG_OK
		    );
		    CDialogSetTransientFor(NULL);
		}

	        /* Mark has changes */
	        d->has_changes = TRUE;

	        /* Insert or append a new frame */
	        frame_num = ImgViewImageInsertFrame(
		    img, row,
		    (guint8 *)g_malloc0(img->bpl * img->height),
		    delay
	        );

	        /* Get the new frame */
	        frame = ImgViewImageGetFrame(img, frame_num);
	        if(frame != NULL)
	        {
		    gint new_row;

		    /* Need to update the alpha channel flags on the
		     * ImgView to note that the alpha channel is
		     * defined because creating a new frame usually
		     * has transparency
		     */
		    if(!(iv->alpha_flags & IMGVIEW_ALPHA_DEFINED))
		    {
			iv->alpha_flags |= IMGVIEW_ALPHA_DEFINED;
		    }

		    gtk_clist_freeze(clist);

		    /* Insert a new row on the frames GtkCList
		     * for the new frame
		     */
		    new_row = clist_row_insert(
			clist, row,
			frame->buf, img->width, img->height, img->bpl,
			frame->delay
		    );

		    /* Update all the rows' frame number */
		    n = clist->rows;
		    for(i = new_row + 1; i < n; i++)
			clist_row_update_frame_num(clist, i);

	            /* Select the new frame */
	            gtk_clist_unselect_all(clist);
	            gtk_clist_select_row(clist, new_row, 0);

		    /* Open the image specified by the path to the
		     * new frame
		     */
		    if(set_frame_from_image_file(
		        d, img, frame_num, frame, path, d->img_fit,
			TRUE,			/* Append multiple frames */
			"Adding Frame",
			TRUE			/* Verbose */
		    ))
		    {
			/* Unable to set the new frame, remove
			 * this new frame
			 */
			ImgViewImageDeleteFrame(img, frame_num);
			gtk_clist_remove(clist, new_row);
			iv->cur_frame_num = -1;
			n = clist->rows;
			for(i = new_row; i < n; i++)
			    clist_row_update_frame_num(clist, i);
			gtk_clist_unselect_all(clist);
			gtk_clist_select_row(clist, new_row, 0);
			if(!ImgViewIsPlaying(iv))
			{
			    iv->cur_frame_num = -1;
			    ImgViewSeek(iv, frame_num);
			}
		    }
		    else
		    {
			frame = ImgViewImageGetFrame(img, frame_num);
			if(frame != NULL)
			    clist_row_update(
				clist, frame_num,
				frame->buf, img->width, img->height, img->bpl,
				frame->delay
			    );
		    }

		    gtk_clist_thaw(clist);
	        }

		if(!ImgViewIsPlaying(iv))
		{
		    iv->cur_frame_num = -1;
		    ImgViewSeek(iv, frame_num);
		}

		update_stats_label(d);
	        update(d);
	    }
	}

	set_busy(d, FALSE);
}

/*
 *	GtkCList "drag_data_delete" signal callback.
 */
static void clist_drag_data_delete_cb(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	gint i, n, frame_num;
	GtkCList *clist;
	imgview_image_struct *img;
	imgview_struct *iv;
	Dialog *d = DIALOG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->clist);
	iv = d->iv;
	img = ImgViewGetImage(iv);

	/* Delete the selected commands */
	gtk_clist_freeze(clist);
	while(clist->selection != NULL)
	{
	    frame_num = (gint)clist->selection->data;

	    if(frame_num == iv->cur_frame_num)
		iv->cur_frame_num = -1;

	    ImgViewImageDeleteFrame(img, frame_num);
	    gtk_clist_remove(clist, frame_num);
	}

	/* Update all the rows' frame number */
	n = clist->rows;
	for(i = 0; i < n; i++)
	    clist_row_update_frame_num(clist, i);

	gtk_clist_thaw(clist);

	d->has_changes = TRUE;

	if(ImgViewIsPlaying(iv))
	{
	    if(ImgViewGetTotalFrames(iv) <= 1)
		ImgViewPause(iv);
	}
	else
	    ImgViewQueueDrawView(iv);
	update_stats_label(d);
	update(d);
}

/*
 *	GtkCList "select_row" signal callback.
 */
static void clist_select_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap = NULL;
	GtkCellType cell_type;
	imgview_struct *iv;
	Dialog *d = DIALOG(data);
	if((clist == NULL) || (event == NULL) || (d == NULL))
	    return;

	iv = d->iv;
	if(iv == NULL)
	    return;

	set_busy(d, TRUE);

	/* Scroll to the selected row if it is not fully visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	    GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Set the DND icon */
	cell_type = gtk_clist_get_cell_type(clist, row, 1);
	if(cell_type == GTK_CELL_PIXTEXT)
	{
	    gchar *text;
	    guint8 spacing;
	    gtk_clist_get_pixtext(
		clist,
		row, 1,
		&text,
		&spacing,
		&pixmap, &mask
	    );
	}
	else if(cell_type == GTK_CELL_PIXMAP)
	{
	    gtk_clist_get_pixmap(
		clist,
		row, 1,
		&pixmap, &mask
	    );
	}
	if(pixmap != NULL)
	{
	    gint w = 15, h = 15;
	    gdk_window_get_size(pixmap, &w, &h);
	    GUIDNDSetDragIcon(
		pixmap, mask,
		w / 2, h / 2
	    );
	}
	else
	{
	    GUIDNDSetDragIcon(NULL, NULL, 0, 0);
	}

	/* Seek the ImgView to the selected frame if it is not
	 * playing
	 */
	if(!ImgViewIsPlaying(iv))
	    ImgViewSeek(iv, row);

	update(d);

	/* Edit on double-click */
	if(event->type == GDK_2BUTTON_PRESS)
	    edit_frame(d, FALSE);

	set_busy(d, FALSE);
}

/*
 *	GtkCList "unselect_row" signal callback.
 */
static void clist_unselect_row_cb(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	Dialog *d = DIALOG(data);
	if((clist == NULL) || (event == NULL) || (d == NULL))
	    return;

	update(d);
}

/*
 *	PDialog browse callback.
 */
static gchar *pdialog_browse_cb(
	gpointer pdialog, gpointer data, gint prompt_num
)
{
	const gchar *img_list[] =
#ifdef IV_IMAGE_LIST
	    IV_IMAGE_LIST;
#else
	    { NULL, NULL };
#endif
	gboolean response;
	gint i;
	gint npaths = 0, total_ftypes = 0;
	gchar **paths_list;
	fb_type_struct *type_rtn, **ftype = NULL;
	static gchar path[PATH_MAX + NAME_MAX];
	Dialog *d = DIALOG(data);

	if((d == NULL) || FileBrowserIsQuery())
	    return(NULL);

	set_busy(d, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All Images"
	);
	for(i = 0; img_list[i] != NULL; i += 2)
	    FileBrowserTypeListNew(
		&ftype, &total_ftypes,
		img_list[i], img_list[i + 1]
	    );

	*path = '\0';

	/* Need to set transient for manually to avoid setting modal */
	gtk_window_set_transient_for(
	    GTK_WINDOW(FileBrowserGetToplevel()),
	    GTK_WINDOW(PDialogGetToplevel())
	);

	/* Prompt user for image */
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja La Imagen",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Choisir L'Image",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie Bildnis Aus",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere L'Immagine",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Selecteer Beeld",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione Imagem",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut Avbilde",
	    "Velg", "Kanseler",
#else
	    "Select Image",
	    "Select", "Cancel",
#endif
	    NULL,
	    ftype, total_ftypes,
	    &paths_list, &npaths,
	    &type_rtn
	);

	FileBrowserSetTransientFor(NULL);

	if(response)
	{
	    strncpy(
		path,
		paths_list[0],
		sizeof(path)
	    );
	    path[sizeof(path) - 1] = '\0';
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	set_busy(d, FALSE);

	return(path);
}

/*
 *	ImgIO progress callback.
 */
static gint imgio_progress_cb(
	gpointer data,
	gint pi, gint pm,
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba
)
{
	ProgressData *progress_data = PROGRESS_DATA(data);
	if(progress_data == NULL)
	    return(FALSE);

	if(ProgressDialogIsQuery() && (pm > 0))
	{
	    const gfloat uv = progress_data->unit_stage_coeff;
	    const gfloat v = (uv * (gfloat)progress_data->cur_stage) +
		(uv * ((gfloat)pi / (gfloat)pm));

	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		v,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		progress_data->aborted = TRUE;
		return(FALSE);
	    }
	}

	return(TRUE);
}

/*
 *	Resize/reposition progress callback.
 */
static gint resize_progress_cb(
	const gulong i, const gulong m, gpointer data
)
{
	ProgressData *progress_data = PROGRESS_DATA(data);
	if(progress_data == NULL)
	    return(-2);

	if(ProgressDialogIsQuery() && (m > 0l))
	{
	    const gfloat uv = progress_data->unit_stage_coeff;
	    const gfloat v = (uv * (gfloat)progress_data->cur_stage) +
		(uv * ((gfloat)i / (gfloat)m));

	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		v,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		progress_data->aborted = TRUE;
		return(-4);
	    }
	}

	return(0);
}


/*
 *	Updates the GtkCList's row values and frame data.
 *
 *	The rgba, width, height, and bpl specifies the image data that
 *	will be used as the icon. If rgba is NULL then the icon will
 *	not be changed.
 */
static void clist_row_update(
	GtkCList *clist, const gint row,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const gulong delay
)
{
	const gint	iwidth = FRAME_ICON_WIDTH,
			iheight = FRAME_ICON_HEIGHT,
			bpp = 4;
	gint _bpl = bpl;
	gchar *s;
	GdkWindow *window;
	GtkStyle *style;
	FrameData *fd;

	if(clist == NULL)
	    return;

	if(_bpl <= 0)
	    _bpl = width * bpp;

	window = GTK_WIDGET(clist)->window;
	style = gtk_widget_get_style(GTK_WIDGET(clist));
	if(style == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Frame number */
	clist_row_update_frame_num(clist, row);

	/* Icon */
	if(rgba != NULL)
	{
	    GdkBitmap *mask;
	    GdkPixmap *pixmap;

	    if((width > iwidth) || (height > iheight))
	    {
		const gfloat aspect = (height > 0) ?
		    ((gfloat)width / (gfloat)height) : 1.0f;
		gint width2, height2;
		guint8 *rgba2;

		width2 = iwidth;
		height2 = (aspect > 0.0f) ? (gint)(iheight / aspect) : iheight;
		if(height2 > iheight)
		{
		    height2 = iheight;
		    width2 = (gint)(iwidth * aspect);
		}

		rgba2 = (guint8 *)g_malloc(width2 * height2 * bpp);
		GUIImageBufferResize(
		    bpp,
		    rgba, width, height, _bpl,
		    rgba2, width2, height2, width2 * bpp,
		    NULL, NULL
		);
		pixmap = gdk_pixmap_new(window, width2, height2, -1);
		if((pixmap != NULL) && (rgba2 != NULL))
		    gdk_draw_rgb_32_image(
			(GdkDrawable *)pixmap,
			style->black_gc,
			0, 0, width2, height2,
			GDK_RGB_DITHER_NONE,
			(guchar *)rgba2,
			width2 * bpp
		    );
		mask = GUICreateBitmapFromDataRGBA(
		    width2, height2, width2 * bpp,
		    rgba2, 0x80,
		    window
		);
		g_free(rgba2);
	    }
	    else
	    {
		pixmap = gdk_pixmap_new(window, width, height, -1);
		if(pixmap != NULL)
		    gdk_draw_rgb_32_image(
			(GdkDrawable *)pixmap,
			style->black_gc,
			0, 0, width, height,
			GDK_RGB_DITHER_NONE,
			(guchar *)rgba,
			_bpl
		    );
		mask = GUICreateBitmapFromDataRGBA(
		    width, height, _bpl,
		    rgba, 0x80,
		    window
		);
	    }
	    gtk_clist_set_pixmap(
		clist, row, 1, pixmap, mask
	    );
	    GDK_PIXMAP_UNREF(pixmap);
	    GDK_BITMAP_UNREF(mask);
	}

	/* Delay */
	s = g_strdup_printf("%ld ms", delay);
	gtk_clist_set_text(
	    clist, row, 2, s
	);
	g_free(s);

	/* Update the frame data */
	fd = FRAME_DATA(gtk_clist_get_row_data(clist, row));
	if(fd != NULL)
	{
	    fd->delay = delay;
	}

	gtk_clist_thaw(clist);
}

/*
 *	Updates the GtkCList's row frame number value.
 */
static void clist_row_update_frame_num(
	GtkCList *clist, const gint row
)
{
	gchar *s;

	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Frame number */
	s = g_strdup_printf("#%i", row + 1);
	gtk_clist_set_text(
	    clist, row, 0, s
	);
	g_free(s);

	gtk_clist_thaw(clist);
}

/*
 *	Inserts a new row to the GtkCList with the specified frame
 *	data.
 *
 *	The row specifies the row index to insert the new row at
 *	or -1 to append a new row.
 *
 *	The rgba, width, height, and bpl specifies the frame data.
 *
 *	The delay specifies the delay in milliseconds.
 */
static gint clist_row_insert(
	GtkCList *clist, const gint row,
	const guint8 *rgba,
	const gint width, const gint height,
	const gint bpl,
	const gulong delay
)
{
	gint i, columns, new_row;
	gchar **strv;
	FrameData *fd;

	if(clist == NULL)
	    return(-1);

	gtk_clist_freeze(clist);

	/* Create a new row */
	columns = clist->columns;
	strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	for(i = 0; i < columns; i++)
	    strv[i] = "";
	if(row > -1)
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);
	g_free(strv);

	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    return(-1);
	}

	/* Set the new row */
	clist_row_update(
	    clist, new_row,
	    rgba, width, height, bpl,
	    delay
	);

	/* Set the new row's data */
	fd = frame_data_new(delay);
	gtk_clist_set_row_data_full(
	    clist, new_row,
	    fd, (GtkDestroyNotify)frame_data_delete
	);

	gtk_clist_thaw(clist);

	return(new_row);
}


/*
 *	Opens the image from file and sets it to the image's frame.
 *
 *	The d specifies the dialog.
 *
 *	The img specifies the image that the frame is from.
 *
 *	The frame_num and frame specifies the frame that the opened
 *	image is to be set to. If append_multi is TRUE Then
 *	subsequent frames from the opened image will be appened
 *	after this frame and the Frames GtkCList will be updated
 *	with additional rows added.
 *
 *	The path specifies the full path to the image file.
 *
 *	The img_fit specifies what modifications (cropping,
 *	positioning, resizing, etc) that is to be performed on the
 *	opened image before it is set to the frame.
 *
 *	If verbose is TRUE then the progress dialog will be mapped
 *	and this operation will be interruptable.
 */
static gint set_frame_from_image_file(
	Dialog *d,
	imgview_image_struct *img,
	const gint frame_num, imgview_frame_struct *frame,
	const gchar *path,
	const FDImgFitType img_fit,
	const gboolean append_multi,
	const gchar *progress_title,
	const gboolean verbose
)
{
	gint		status, width, height, bpp, bpl,
			nframes,
			x, y, base_width, base_height;
	gchar		*creator,
			*title,
			*author,
			*comments;
	gboolean	need_insert_to_frames_list = FALSE,
			use_image_animation_info = FALSE;
	const gchar *resize_progress_msg = NULL;
	guint8 **rgba_list, bg_color[4];
	gulong *delay_list;
	GtkWidget *toplevel;
	GtkCList *clist = GTK_CLIST(d->clist);
	ProgressData *progress_data;

#define DELETE_IMAGE_DATA_ALL	{	\
 gint i;				\
 for(i = 0; i < nframes; i++)		\
  g_free(rgba_list[i]);			\
 g_free(rgba_list);			\
 g_free(delay_list);			\
 g_free(creator);			\
 g_free(title);				\
 g_free(author);			\
 g_free(comments);			\
}

	if((d == NULL) || (img == NULL) || (frame == NULL) ||
	   STRISEMPTY(path)
	)
	    return(-2);

	toplevel = d->toplevel;

	/* Map the progress dialog and allocate the progress
	 * callback data?
	 */
	if(verbose)
	{
	    gchar *msg;

	    progress_data = PROGRESS_DATA(g_malloc0(
		sizeof(ProgressData)
	    ));
	    progress_data->d = d;
	    progress_data->cur_stage = 0;
	    progress_data->nstages = 1;
	    progress_data->unit_stage_coeff = 1.0f /
		(gfloat)progress_data->nstages;
	    progress_data->aborted = FALSE;

	    if(ProgressDialogIsQuery())
		ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(toplevel);
	    msg = g_strdup_printf(
"Opening image \"%s\"...",
		g_basename(path)
	    );
	    ProgressDialogMap(
		progress_title,
		msg,
		(const guint8 **)icon_animate_32x32_xpm,
		"Stop"  
	    );
	    g_free(msg);
	    gdk_flush();
	}
	else
	{
	    progress_data = NULL;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,
	    &x, &y, &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,
	    progress_data,
	    verbose ? imgio_progress_cb : NULL
	);

	if(verbose)
	{
	    /* User aborted? */
	    if(progress_data->aborted)
	    {
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
		g_free(progress_data);
		DELETE_IMAGE_DATA_ALL
		return(-4);
	    }

	    /* Update the progress message and stage values for
	     * the subsequent image fit operations below
	     */
	    progress_data->cur_stage = 0;
	    switch(img_fit)
	    {
	      case IMG_FIT_RESIZE:
		resize_progress_msg =
"Resizing image to fit...";
		progress_data->nstages = 1;
		break;
	      case IMG_FIT_RESIZE_MAINTAIN_ASPECT:
		resize_progress_msg =
"Resizing and repositioning image to fit...";
		progress_data->nstages = 2;
		break;
	      case IMG_FIT_RESIZE_MAINTAIN_ASPECT_TO_COVER:
		resize_progress_msg =
"Resizing and repositioning image to cover...";
		progress_data->nstages = 2;
		break;
	      case IMG_FIT_CENTERED:
	      case IMG_FIT_UPPER_LEFT:
	      case IMG_FIT_UPPER_RIGHT:
	      case IMG_FIT_LOWER_RIGHT:
	      case IMG_FIT_LOWER_LEFT:
		resize_progress_msg =
"Repositioning image...";
		progress_data->nstages = 1;
		break;
	    }
	    if(append_multi)
		progress_data->nstages *= nframes;
	    if(progress_data->nstages > 0)
		progress_data->unit_stage_coeff = 1.0f /
		    (gfloat)progress_data->nstages;
	    ProgressDialogUpdate(
		NULL,
		resize_progress_msg,
		NULL, NULL,
		0.0f,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	}

	/* Was the image opened successfully? */
	if((rgba_list != NULL) && (img->bpp == bpp))
	{
	    const gint	src_width = width,
			src_height = height,
			src_bpl = bpl,
			tar_width = img->width,
			tar_height = img->height,
			tar_bpl = img->bpl;
	    const gfloat	aspect = (src_height > 0) ?
		    ((gfloat)src_width / (gfloat)src_height) : 1.0f;
	    gint cur_frame_num = frame_num;
	    gint i, tmp_width, tmp_height;
	    const guint8 *src;
	    guint8 *tar;

	    /* Check if this image contains multiple frames and
	     * values for each frame
	     */
	    if((delay_list != NULL) ? (delay_list[0] > 0l) : FALSE)
	    {
		if(verbose)
		{
		    gint response;
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse( 
			"Use Image Animation Information?",
"This image contains animation information (such as\n\
delay values).\n\
\n\
Do you want to use the animation information found\n\
in this image?\n\
\n\
Answering \"No\" will use the animation values that\n\
you have specified.",
			NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_YES
		    );
		    CDialogSetTransientFor(NULL);
		    if(response == CDIALOG_RESPONSE_YES)
			use_image_animation_info = TRUE;
		}
	    }

	    /* Iterate through each frame from the opened image and
	     * add them to the image
	     */
	    for(i = 0; i < nframes; i++)
	    {
		src = rgba_list[i];
		if(src == NULL)
		    continue;

		/* Is this not the first frame? */
		if(i > 0)
		{
		    if(!append_multi)
			break;

		    /* Append a new frame to the target image */
		    if(frame != NULL)
			cur_frame_num = ImgViewImageInsertFrame(
			    img,
			    cur_frame_num + 1,
			    (guint8 *)g_malloc(tar_bpl * tar_height),
			    use_image_animation_info ?
				delay_list[i] : frame->delay
			);
		    else
			cur_frame_num = ImgViewImageInsertFrame(
			    img,
			    cur_frame_num + 1,
			    (guint8 *)g_malloc(tar_bpl * tar_height),
			    delay_list[i]
			);
		    frame = ImgViewImageGetFrame(img, cur_frame_num);
		    if(frame == NULL)
		    {
			status = -3;
			break;
		    }

		    /* Mark that the frames GtkCList needs to be
		     * updated afterwards
		     */
		    need_insert_to_frames_list = TRUE;

		    tar = frame->buf;
		}
		else
		{
		    /* First frame */
		    tar = frame->buf;
		    if(use_image_animation_info)
			frame->delay = delay_list[i];
		}

		if(tar == NULL)
		{
		    status = -3;
		    break;
		}

		/* Clear the target frame first */
		memset(tar, 0x00, tar_bpl * tar_height);

#define CHECK_HANDLE_ABORT_INC_STAGE(_tmp_data_) { \
 if(verbose) {					\
  if(progress_data->aborted) {			\
   if((_tmp_data_) != NULL)			\
    g_free(_tmp_data_);				\
						\
   ProgressDialogBreakQuery(TRUE);		\
   ProgressDialogSetTransientFor(NULL);		\
   g_free(progress_data);			\
						\
   DELETE_IMAGE_DATA_ALL			\
						\
   return(-4);					\
  } else {					\
   progress_data->cur_stage++;			\
  }						\
 }						\
}

		/* Copy the source image data to the target image
		 * data based on the specified image fit method
		 */
		switch(img_fit)
		{
		  case IMG_FIT_RESIZE:
		    GUIImageBufferResize(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;

		  case IMG_FIT_RESIZE_MAINTAIN_ASPECT:
		    tmp_width = tar_width;
		    tmp_height = (aspect > 0.0f) ?
			(gint)(tmp_width / aspect) : tmp_width;
		    if(tmp_height > tar_height)
		    {
			tmp_height = tar_height;
			tmp_width = (gint)(tmp_height * aspect);
		    }
		    if((tmp_width > 0) && (tmp_height > 0))
		    {
			const gint tmp_bpl = tmp_width * 4;
			guint8 *tmp = (guint8 *)g_malloc(
			    tmp_bpl * tmp_height
			);
			GUIImageBufferResize(
			    bpp,
			    src, src_width, src_height, src_bpl,
			    tmp, tmp_width, tmp_height, tmp_bpl,
			    verbose ? resize_progress_cb : NULL,
			    progress_data
			);
			CHECK_HANDLE_ABORT_INC_STAGE(tmp);
			GUIImageBufferCopyArea(
			    bpp,
			    tmp, tmp_width, tmp_height, tmp_bpl,
			    tar, tar_width, tar_height, tar_bpl,
			    (tar_width - tmp_width) / 2,
			    (tar_height - tmp_height) / 2,
			    FALSE,		/* Do not blend */
			    verbose ? resize_progress_cb : NULL,
			    progress_data
			);
			CHECK_HANDLE_ABORT_INC_STAGE(NULL);
			g_free(tmp);
		    }
		    break;

		  case IMG_FIT_RESIZE_MAINTAIN_ASPECT_TO_COVER:
		    tmp_width = tar_width;
		    tmp_height = (aspect > 0.0f) ?
			(gint)(tmp_width / aspect) : tmp_width;
		    if(tmp_height < tar_height)
		    {
			tmp_height = tar_height;
			tmp_width = (gint)(tmp_height * aspect);
		    }
		    if((tmp_width > 0) && (tmp_height > 0))
		    {
			const gint tmp_bpl = tmp_width * 4;
			guint8 *tmp = (guint8 *)g_malloc(
			    tmp_bpl * tmp_height 
			);
			GUIImageBufferResize(
			    bpp,
			    src, src_width, src_height, src_bpl,
			    tmp, tmp_width, tmp_height, tmp_bpl,
			    verbose ? resize_progress_cb : NULL,
			    progress_data
			);
			CHECK_HANDLE_ABORT_INC_STAGE(tmp);
			GUIImageBufferCopyArea(
			    bpp,
			    tmp, tmp_width, tmp_height, tmp_bpl,
			    tar, tar_width, tar_height, tar_bpl,
			    (tar_width - tmp_width) / 2,
			    (tar_height - tmp_height) / 2,
			    FALSE,		/* Do not blend */
			    verbose ? resize_progress_cb : NULL,
			    progress_data
			);
			CHECK_HANDLE_ABORT_INC_STAGE(NULL);
			g_free(tmp);
		    }
		    break;

		  case IMG_FIT_CENTERED:
		    GUIImageBufferCopyArea(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			(tar_width - src_width) / 2,
			(tar_height - src_height) / 2,
			FALSE,		/* Do not blend */
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;

		  case IMG_FIT_UPPER_LEFT:
		    GUIImageBufferCopyArea(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			0, 0,
			FALSE,		/* Do not blend */
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;

		  case IMG_FIT_UPPER_RIGHT:
		    GUIImageBufferCopyArea(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			tar_width - src_width,
			0,
			FALSE,		/* Do not blend */
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;

		  case IMG_FIT_LOWER_RIGHT:
		    GUIImageBufferCopyArea(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			tar_width - src_width,
			tar_height - src_height,
			FALSE,		/* Do not blend */
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;

		  case IMG_FIT_LOWER_LEFT:
		    GUIImageBufferCopyArea(
			bpp,
			src, src_width, src_height, src_bpl,
			tar, tar_width, tar_height, tar_bpl,
			0,
			tar_height - src_height,
			FALSE,		/* Do not blend */
			verbose ? resize_progress_cb : NULL,
			progress_data
		    );
		    CHECK_HANDLE_ABORT_INC_STAGE(NULL);
		    break;
		}

		/* Need to add this frame to the frames GtkCList? */
		if(append_multi && need_insert_to_frames_list && (frame != NULL))
		{
		    gtk_clist_freeze(clist);
		    clist_row_insert(
			clist, cur_frame_num,
			frame->buf, img->width, img->height, img->bpl,
			frame->delay
		    );
		    gtk_clist_thaw(clist);
		}

		/* Delete this opened image frame since it is no
		 * longer needed
		 */
		g_free(rgba_list[i]);
		rgba_list[i] = NULL;
#undef CHECK_HANDLE_ABORT_INC_STAGE
	    }

	    /* Need to update all the rows' frame number */
	    if(append_multi && need_insert_to_frames_list)
	    {
		const gint n = clist->rows;
		gint i;
		gtk_clist_freeze(clist);
		for(i = 0; i < n; i++)
		    clist_row_update_frame_num(clist, i);
		gtk_clist_thaw(clist);
	    }
	}
	else
	{
	    /* Unable to open the image */
	    gchar *s = g_strdup_printf(
"Unable to open image:\n\
\n\
    %s\n\
\n\
%s",
		path, ImgLoadGetError()
	    );
	    if(verbose)
	    {
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	    }
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Open Image Failed",
		s, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    g_free(s);
	    CDialogSetTransientFor(NULL);
	}

	/* Delete the callback data? */
	if(verbose)
	{
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);
	    g_free(progress_data);
	}

	DELETE_IMAGE_DATA_ALL

	return(status);
#undef DELETE_IMAGE_DATA_ALL
}

/*
 *	Add frame.
 */
static void add_frame(Dialog *d)
{
	gint i, n, frame_num, row, new_row;
	GList *glist;
	GtkWidget *toplevel = d->toplevel;
	GtkCList *clist;
	imgview_struct *iv = d->iv;
	imgview_frame_struct *frame;
	imgview_image_struct *img;
	if(iv == NULL)
	    return;

	set_busy(d, TRUE);

	img = ImgViewGetImage(iv);
	if(img == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(   
		"Add Frame Failed",
"There is no image loaded.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	clist = GTK_CLIST(d->clist);

	/* Get the last selected row that the frame is to be inserted
	 * after
	 */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

#if 0
	/* Increase the row count so that we insert after instead of
	 * before (the default) because this will make inserting
	 * frames easier for one after another
	 */
	if(row > -1)
	    row++;
	if(row >= clist->rows)
	    row = -1;
#endif

	/* Mark has changes */
	d->has_changes = TRUE;

	/* Insert or append a new frame */
	frame_num = ImgViewImageInsertFrame(
	    img, row,
	    (guint8 *)g_malloc0(img->bpl * img->height),
	    250l
	);
	frame = ImgViewImageGetFrame(img, frame_num);
	if(frame == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Add Frame Failed",
"A new frame could not be added into the image.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	/* Need to update the alpha channel flags on the ImgView to
	 * note that the alpha channel is defined because creating
	 * a new frame usually has transparency
	 */
	if(!(iv->alpha_flags & IMGVIEW_ALPHA_DEFINED))
	{
	    iv->alpha_flags |= IMGVIEW_ALPHA_DEFINED;
	}

	gtk_clist_freeze(clist);

	/* Insert the new frame */
	new_row = clist_row_insert(
	    clist, row,
	    frame->buf, img->width, img->height, img->bpl,
	    frame->delay
	);

	/* Update all the rows' frame number */
	n = clist->rows;
	for(i = new_row + 1; i < n; i++)
	    clist_row_update_frame_num(clist, i);

	/* Select the new frame */
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(clist, new_row, 0);

	gtk_clist_thaw(clist);

	if(!ImgViewIsPlaying(iv))
	    ImgViewQueueDrawView(iv);
	update_stats_label(d);
	update(d);

	/* Edit the new frame */
	edit_frame(d, TRUE);

	set_busy(d, FALSE);
}

/*
 *	Edit frame.
 */
static void edit_frame(Dialog *d, const gboolean from_add)
{
	const gchar *img_fit_strings[] = IMG_FIT_STRINGS;
	gboolean got_response;
	gint i, frame_num, strc;
	gulong delay = 0l;
	const gchar	*path = NULL,
			*img_fit_s = NULL;
	gchar *title, **strv;
	GList *glist, *img_fit_list;
	GtkWidget *toplevel = d->toplevel;
	GtkCList *clist;
	imgview_struct *iv = d->iv;
	imgview_frame_struct *frame;
	imgview_image_struct *img;

	if((iv == NULL) || PDialogIsQuery())
	    return;

	set_busy(d, TRUE);

	img = ImgViewGetImage(iv);
	if(img == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Edit Frame Failed",
"There is no image loaded.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	clist = GTK_CLIST(d->clist);
	glist = clist->selection_end;
	frame_num = (glist != NULL) ? (gint)glist->data : -1;
	if(frame_num < 0)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Edit Frame Failed",
"There is no frame selected.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	/* Get selected frame */
	frame = ImgViewImageGetFrame(img, frame_num);
	if(frame == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Edit Frame Failed",
"Unable to get the selected frame from the image.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	/* Get the image fit strings */
	img_fit_list = NULL;
	for(i = 0; img_fit_strings[i] != NULL; i++)
	    img_fit_list = g_list_append(
		img_fit_list, STRDUP(img_fit_strings[i])
	    );

	/* Set up the PDialog and prompt the user */
	PDialogDeleteAllPrompts();
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()),
	    GTK_WINDOW(toplevel)
	);
	PDialogSetSize(350, -1);
	PDialogAddPromptWithBrowse(
	    NULL,
	    "File:",
	    NULL,
	    d,
	    pdialog_browse_cb
	);
	PDialogSetPromptCompletePath(-1);
	PDialogSetPromptTip(
	    0,
	    from_add ?
"Select the image file to be added to this frame" :
"Select the new image file to be set on this frame, or\
 leave this blank to keep the current frame's data"
	);
	PDialogAddPromptPopupList(
	    NULL, "Fit Image:",
	    img_fit_list,
	    d->img_fit,
	    -1
	);
	PDialogSetPromptTip(
	    1,
"Select the method in which the image will be resized and\
 positioned in the frame"
	);
	g_list_foreach(img_fit_list, (GFunc)g_free, NULL);
	g_list_free(img_fit_list);
	PDialogAddPromptSpin(
	    NULL, "Delay:",
	    (gfloat)frame->delay,
	    0.0f,
	    (gfloat)((guint)-1),	/* Reasonable max delay, right? */
	    1.0f, 5.0f,
	    1.0f, 0
	);
	PDialogSetPromptTip(
	    2,
"Set the delay in milliseconds which specifies how long this\
 frame will be displayed before displaying the next frame"
	);
	title = g_strdup_printf("Frame #%i", frame_num + 1);
	strv = PDialogGetResponse(
	    title, NULL, NULL,
	    PDIALOG_ICON_EDIT,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	g_free(title);
	if(strv != NULL)
	{
	    got_response = TRUE;
	    path = (strc > 0) ? strv[0] : NULL;
	    img_fit_s = (strc > 1) ? strv[1] : NULL;
	    delay = (strc > 2) ? ATOL(strv[2]) : 0l;

	    /* Set the image fit type */
	    d->img_fit = get_img_fit_type_from_s(img_fit_s);
	}
	else
	{
	    got_response = FALSE;
	}

	PDialogSetTransientFor(NULL);
	PDialogDeleteAllPrompts();

	if(got_response)
	{
	    imgview_frame_struct *frame = ImgViewImageGetFrame(
		img, frame_num
	    );

	    /* Warn if the delay is set to 0 */
	    if(delay == 0l)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Delay Warning",
"The delay value is set to 0, which means that\n\
this frame will never be visible when played.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }

	    if(frame != NULL)
	    {
		/* Set new frame values */
		frame->delay = delay;

		gtk_clist_freeze(clist);

		/* Set the new image? */
		if(!STRISEMPTY(path))
		{
		    if(set_frame_from_image_file(
			d, img, frame_num, frame, path, d->img_fit,
			TRUE,			/* Append multiple frames */
			from_add ? "Adding Frame" : "Setting Frame",
			TRUE			/* Verbose */
		    ))
		    {
			/* Unable to set the frame, if this is from
			 * an add then remove the new row
			 */
			if(from_add)
			{
			    gint i, n;
			    ImgViewImageDeleteFrame(img, frame_num);
			    gtk_clist_remove(clist, frame_num);
			    iv->cur_frame_num = -1;
			    n = clist->rows;
			    for(i = frame_num; i < n; i++)
				clist_row_update_frame_num(clist, i);
			    gtk_clist_unselect_all(clist);
			    gtk_clist_select_row(clist, frame_num, 0);

			    if(!ImgViewIsPlaying(iv))
			    {
				iv->cur_frame_num = -1;
				ImgViewSeek(iv, frame_num);
			    }
			}
		    }
		    else
		    {
			frame = ImgViewImageGetFrame(img, frame_num);
			if(frame != NULL)
			    clist_row_update(
				clist, frame_num,
				frame->buf, img->width, img->height, img->bpl,
				frame->delay
			    );
		    }
		}
		else
		{
		    clist_row_update(
			clist, frame_num,
			frame->buf, img->width, img->height, img->bpl,
			frame->delay
		    );
		}

		gtk_clist_thaw(clist);

		/* Mark has changes */
		d->has_changes = TRUE;

		if(!ImgViewIsPlaying(iv))
		{
		    iv->cur_frame_num = -1;
		    ImgViewSeek(iv, frame_num);
		}
		update_stats_label(d);
		update(d);
	    }
	    else
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Edit Frame Failed",
"The selected frame no longer exists.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	}
	else
	{
	    /* User canceled */
	    if(from_add)
	    {
		/* Remove the added frame */
		gint i, n;

		gtk_clist_freeze(clist);

		ImgViewImageDeleteFrame(img, frame_num);
		gtk_clist_remove(clist, frame_num);
		iv->cur_frame_num = -1;

		n = clist->rows;
		for(i = frame_num; i < n; i++)
		    clist_row_update_frame_num(clist, i);

		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, frame_num, 0);

		gtk_clist_thaw(clist);

		if(!ImgViewIsPlaying(iv))
		    ImgViewSeek(iv, frame_num);
	    }
	}

	set_busy(d, FALSE);
}

/*
 *	Remove frame.
 */
static void remove_frame(Dialog *d)
{
	gint i, n, response;
	gchar *s;
	GtkWidget *toplevel = d->toplevel;
	GtkCList *clist = GTK_CLIST(d->clist);
	imgview_struct *iv = d->iv;
	imgview_image_struct *img;

	if((iv == NULL) || CDialogIsQuery())
	    return;

	set_busy(d, TRUE);

	img = ImgViewGetImage(iv);
	if(img == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Remove Frame Failed",
"There are no frames selected.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    set_busy(d, FALSE);
	    return;
	}

	/* Confirm remove */
	i = g_list_length(clist->selection);
	if(i == 1)
	    s = g_strdup_printf(
		"Remove frame #%i?",
		(gint)(clist->selection->data) + 1
	    );
	else
	    s = g_strdup_printf(
		"Remove %i selected frames?",
		i
	    );
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Remove",
	    s, NULL,
	    CDIALOG_ICON_QUESTION,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	g_free(s);
	CDialogSetTransientFor(NULL);
	if(response != CDIALOG_RESPONSE_YES)
	{
	    set_busy(d, FALSE);
	    return;
	}

	gtk_clist_freeze(clist);

	/* Remove each selected frame */
	while(clist->selection != NULL)
	{
	    i = (gint)clist->selection->data;

	    if(i == iv->cur_frame_num)
		iv->cur_frame_num = -1;

	    /* Remove the ImgView image frame */
	    ImgViewImageDeleteFrame(img, i);

	    /* Remove the corresponding GtkCList row */
	    gtk_clist_remove(clist, i);
	}

	/* Update all the rows' frame number */
	n = clist->rows;
	for(i = 0; i < n; i++)
	    clist_row_update_frame_num(clist, i);

	gtk_clist_thaw(clist);


	d->has_changes = TRUE;

	if(ImgViewIsPlaying(iv))
	{
	    if(ImgViewGetTotalFrames(iv) <= 1)
		ImgViewPause(iv);
	}
	else
	    ImgViewQueueDrawView(iv);
	update_stats_label(d);
	update(d);

	set_busy(d, FALSE);
}

/*
 *	Shift up
 */
static void shift_frame_up(Dialog *d)
{
	gint row, nrows, lead_row;
	GtkCList *clist = GTK_CLIST(d->clist);
	imgview_struct *iv = d->iv;
	imgview_image_struct *img = ImgViewGetImage(iv);
	if(img == NULL)
	    return;

	set_busy(d, TRUE);

	/* Nothing to shift? */
	nrows = clist->rows;
	if(nrows <= 1)
	{
	    set_busy(d, FALSE);
	    return;
	}

	/* Already shifted to the top? */
	row = 0;
	if(g_list_find(clist->selection, (gpointer)row) != NULL)
	{
	    set_busy(d, FALSE);
	    return;
	}

	/* Shift the selected rows up */
	lead_row = -1;
	gtk_clist_freeze(clist);
	for(row = 1; row < nrows; row++)
	{
	    if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    {
		ImgViewImageSwapFrames(img, row, row - 1);
		gtk_clist_swap_rows(clist, row, row - 1);

		clist_row_update_frame_num(clist, row - 1);
		clist_row_update_frame_num(clist, row);

		if(lead_row < 0)
		    lead_row = row - 1;
	    }
	}
	gtk_clist_thaw(clist);

	if(lead_row > -1)
	{
	    row = lead_row;
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )  
		gtk_clist_moveto(
		    clist,
		    row, -1,		/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	if(!ImgViewIsPlaying(iv) && (lead_row > -1))
	    ImgViewSeek(iv, lead_row);

	update_stats_label(d);
	update(d);

	set_busy(d, FALSE);
}

/*
 *	Shift down.
 */
static void shift_frame_down(Dialog *d)
{
	gint row, nrows, lead_row;
	GtkCList *clist = GTK_CLIST(d->clist);
	imgview_struct *iv = d->iv;
	imgview_image_struct *img = ImgViewGetImage(iv);
	if(img == NULL)
	    return;

	set_busy(d, TRUE);

	/* Nothing to shift? */
	nrows = clist->rows;
	if(nrows <= 1)
	{
	    set_busy(d, FALSE);
	    return;
	}

	/* Already shifted to the bottom? */
	row = nrows - 1;
	if(g_list_find(clist->selection, (gpointer)row) != NULL)
	{
	    set_busy(d, FALSE);
	    return;
	}

	/* Shift the selected rows down */
	lead_row = -1;
	gtk_clist_freeze(clist);
	for(row = nrows - 2; row >= 0; row--)
	{
	    if(g_list_find(clist->selection, (gpointer)row) != NULL)
	    {
		ImgViewImageSwapFrames(img, row, row + 1);
		gtk_clist_swap_rows(clist, row, row + 1);

		clist_row_update_frame_num(clist, row + 1);
		clist_row_update_frame_num(clist, row);

		if(lead_row < 0)
		    lead_row = row + 1;
	    }
	}
	gtk_clist_thaw(clist);

	if(lead_row > -1)
	{
	    row = lead_row;
	    if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	    )  
		gtk_clist_moveto(
		    clist,
		    row, -1,		/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	if(!ImgViewIsPlaying(iv) && (lead_row > -1))
	    ImgViewSeek(iv, lead_row);

	update_stats_label(d);
	update(d);

	set_busy(d, FALSE);
}

/*
 *	Sets the dialog busy or ready.
 */
static void set_busy(Dialog *d, const gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w = d->toplevel;
	if(busy)
	{
	    d->busy_count++;
	    if(d->busy_count > 1)
		return;
	    cur = d->busy_cur;
	}
	else
	{   
	    d->busy_count--;
	    if(d->busy_count < 0)
		d->busy_count = 0;
	    if(d->busy_count > 0)
		return;
	    cur = NULL;			/* Use the default cursor */
	}
	if(w->window != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush();
	}
}

/*
 *	Updates the stats label.
 */
static void update_stats_label(Dialog *d)
{
	gchar *s;
	GtkLabel *label = GTK_LABEL(d->stats_label);
	GtkCList *clist = GTK_CLIST(d->clist);
	imgview_struct *iv = d->iv;
	const gint nframes = clist->rows;
	imgview_image_struct *img = ImgViewGetImage(iv);

	if(img == NULL)
	    return;

	if(nframes > 0)
	{
	    imgview_frame_struct *frame;
	    gulong total_time_ms = 0l;
	    GList *glist = img->frames_list;
	    for(; glist != NULL; glist = g_list_next(glist))
	    {
		frame = IMGVIEW_FRAME(glist->data);
		if(frame == NULL)
		    continue;

		total_time_ms += frame->delay;
	    }
	    if(total_time_ms > 60000l)
	    {
		const gulong secs = total_time_ms / 1000l;
		s = g_strdup_printf(
		    "%i frame%s - %ld mins %ld secs",
		    nframes,
		    (nframes == 1) ? "" : "s",
		    secs / 60l,
		    secs % 60l
		);
	    }
	    else if(total_time_ms >= 10000l)
		s = g_strdup_printf(
		    "%i frame%s - %.1f secs",
		    nframes,
		    (nframes == 1) ? "" : "s",
		    (gfloat)total_time_ms / 1000.0f
		);
	    else if(total_time_ms >= 1000l)
		s = g_strdup_printf(
		    "%i frame%s - %.2f secs",
		    nframes,
		    (nframes == 1) ? "" : "s",
		    (gfloat)total_time_ms / 1000.0f
		);
	    else
		s = g_strdup_printf(
		    "%i frame%s - %ld ms",
		    nframes,
		    (nframes == 1) ? "" : "s",
		    total_time_ms
		);
	}
	else
	{
	    s = g_strdup("No frames");
	}
	gtk_label_set_text(label, s);
	g_free(s);
}

/*
 *	Updates the Dialog.
 */
static void update(Dialog *d)
{
	gboolean sensitive;
	gint row, nframes, nselected;
	GList *glist;
	GtkCList *clist;
	imgview_struct *iv;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->clist);
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	iv = d->iv;
	nframes = clist->rows;

	d->freeze_count++;

	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->add_btn, sensitive);
	gtk_widget_set_sensitive(d->add_mi, sensitive);
	sensitive = (nselected == 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->edit_btn, sensitive);
	gtk_widget_set_sensitive(d->edit_mi, sensitive);
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->remove_btn, sensitive);
	gtk_widget_set_sensitive(d->remove_mi, sensitive);

	sensitive = ((nselected >= 1) &&
	    (g_list_find(clist->selection, (gpointer)0) == NULL)) ?
		TRUE : FALSE;
	gtk_widget_set_sensitive(d->shift_up_btn, sensitive);
	gtk_widget_set_sensitive(d->shift_up_mi, sensitive);
	sensitive = ((nselected >= 1) &&
	    (g_list_find(clist->selection, (gpointer)(nframes - 1)) == NULL)) ?
		TRUE : FALSE;
	gtk_widget_set_sensitive(d->shift_down_btn, sensitive);
	gtk_widget_set_sensitive(d->shift_down_mi, sensitive);

	sensitive = (!ImgViewIsPlaying(iv) && (nframes > 1)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->play_btn, sensitive);
	gtk_widget_set_sensitive(d->play_mi, sensitive);
	sensitive = ImgViewIsPlaying(iv);
	gtk_widget_set_sensitive(d->pause_btn, sensitive); 
	gtk_widget_set_sensitive(d->pause_mi, sensitive);

	d->freeze_count--;
}


/*
 *	Creates a new frame data.
 */
static FrameData *frame_data_new(
	const gulong delay
)
{
	FrameData *fd = FRAME_DATA(g_malloc0(sizeof(FrameData)));
	if(fd == NULL)
	    return(NULL);

	fd->delay = delay;

	return(fd);
}

/*
 *	Deletes the frame data.
 */
static void frame_data_delete(FrameData *fd)
{
	if(fd == NULL)
	    return;

	g_free(fd);
}


/*
 *	Returns the image fit type from the specified string.
 */
static FDImgFitType get_img_fit_type_from_s(const gchar *s)
{
	const gchar *list[] = IMG_FIT_STRINGS;
	gint i;

	if(s == NULL)
	    return(IMG_FIT_RESIZE);

	for(i = 0; list[i] != NULL; i++)
	{
	    if(!g_strcasecmp(list[i], s))
		return((FDImgFitType)i);
	}

	return(IMG_FIT_RESIZE);
}

/*
 *	Returns a dynamically allocated string describing the
 *	specified image fit type.
 */
static gchar *img_fit_string(const FDImgFitType img_fit)
{
	const gchar *list[] = IMG_FIT_STRINGS;
	gint i;

	for(i = 0; list[i] != NULL; i++)
	{
	    if(i == (gint)img_fit)
		return(STRDUP(list[i]));
	}

	return(NULL);
}


/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	Dialog *d,
	const gint width, const gint height,
	const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn,
	GtkAccelGroup **accelgrp_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;


	/* Create keyboard accelerator group */
	accelgrp = gtk_accel_group_new();
	if(accelgrp_rtn != NULL)
	    *accelgrp_rtn = accelgrp;

	/* Create toplevel */
	toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_animate_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : FALSE
	)
	{
#if 0
/* Cannot set modal since the file browser wil not function properly */
	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);
#endif
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Close Button */
	w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cierre",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Fin",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Nah",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Vicino",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Einde",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Prximo",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Nr",
#else
	    "Close",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(close_cb), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);
}


/*
 *	Queries user to edit frames.
 *
 *	Always returns TRUE.
 */
gboolean iv_edit_frames(
	GtkWidget *ref_toplevel,
	imgview_struct *iv,
	const gchar *filename
)
{
	const gint	border_major = 5;
	gboolean has_changes;
	gchar *s;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;

	/* Create the frames dialog */
	Dialog *d = DIALOG(g_malloc0(sizeof(Dialog)));
	s = g_strdup_printf(
	    "Frames: %s",
	    STRISEMPTY(filename) ? "Untitled" : filename
	);
	build_std_dialog(
	    d,
	    400, -1,
	    s,				/* Title */
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp
	);
	g_free(s);
	d->toplevel = toplevel;
	d->accelgrp = accelgrp;
	d->freeze_count = 0;
	d->busy_count = 0;
	d->iv = iv;
	d->busy_cur = gdk_cursor_new(GDK_WATCH);
	d->img_fit = IMG_FIT_RESIZE_MAINTAIN_ASPECT;
	d->has_changes = FALSE;

	d->freeze_count++;

	if(client_vbox != NULL)
	{
	    const gint	columns = 3,
			bw = GUI_BUTTON_HLABEL_WIDTH,
			bh = GUI_BUTTON_HLABEL_HEIGHT;
	    const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	DND_INFO_STRING},
{IV_DND_TARGET_FRAME_INDEX,	0,	IV_DND_INFO_FRAME_INDEX}
	    };
	    const GtkTargetEntry dnd_src_types[] = {
{IV_DND_TARGET_FRAME_INDEX,	0,	IV_DND_INFO_FRAME_INDEX}
	    };
	    gchar **strv;
	    GdkWindow *window = toplevel->window;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget	*w,
			*parent = client_vbox,
			*parent2, *parent3;
	    GtkCList *clist;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column GtkVBox */
	    w = gtk_vbox_new(TRUE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		window, &style->bg[GTK_STATE_NORMAL],
		(guint8 **)icon_animate_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Center column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* GtkScrolledWindow for the GtkCList */
	    w = gtk_scrolled_window_new(NULL, NULL);
	    gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* GtkCList */
	    strv = (gchar **)g_malloc(columns * sizeof(gchar *));
	    strv[0] = "Frame";
	    strv[1] = "Image";
	    strv[2] = "Delay";
	    d->clist = w = gtk_clist_new_with_titles(columns, strv);
	    g_free(strv);
	    clist = GTK_CLIST(w);
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(clist_event_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(clist_event_cb), d
	    );
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_realize(w);
	    gtk_widget_set_usize(w, -1, 250);
	    gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	    gtk_clist_column_titles_passive(clist);
	    gtk_clist_set_column_auto_resize(
		clist, 0, TRUE
	    );
	    gtk_clist_set_column_resizeable(
		clist, 0, FALSE
	    );
	    gtk_clist_set_column_justification(
		clist, 0, GTK_JUSTIFY_RIGHT
	    );
	    gtk_clist_set_column_min_width(
		clist, 0, 10
	    );
	    gtk_clist_set_column_auto_resize(
		clist, 1, FALSE
	    );
	    gtk_clist_set_column_resizeable(
		clist, 1, FALSE
	    );
	    gtk_clist_set_column_justification(
		clist, 1, GTK_JUSTIFY_CENTER
	    );
	    gtk_clist_set_column_min_width(
		clist, 1, FRAME_ICON_WIDTH
	    );
	    gtk_clist_set_column_width(
		clist, 1, FRAME_ICON_WIDTH
	    );
	    gtk_clist_set_column_auto_resize(
		clist, 2, TRUE
	    );
	    gtk_clist_set_column_resizeable(
		clist, 2, FALSE
	    );
	    gtk_clist_set_column_justification(
		clist, 2, GTK_JUSTIFY_LEFT
	    );
	    gtk_clist_set_column_min_width(
		clist, 2, 10
	    );
	    gtk_clist_set_row_height(clist, FRAME_ICON_HEIGHT);
	    gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	    gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(clist_select_row_cb), d
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(clist_unselect_row_cb), d
	    );
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		clist_drag_data_get_cb,
		clist_drag_data_delete_cb,
		NULL,
		d
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_COPY,		/* Default action */
		clist_drag_data_received_cb,
		d
	    );
	    gtk_widget_show(w);

	    w = gtk_frame_new(NULL);
	    gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    w = gtk_hbox_new(FALSE, 0);
	    gtk_widget_set_usize(w, -1, 20);
	    gtk_container_add(GTK_CONTAINER(parent3), w);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Stats GtkLabel */
	    d->stats_label = w = gtk_label_new("");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Right column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_vbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Add Button */
	    d->add_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_add_20x20_xpm,
		"Add...", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(add_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_a);
	    gtk_widget_show(w);

	    /* Edit Button */
	    d->edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
		"Edit...", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(edit_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_e, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_e);
	    gtk_widget_show(w);

	    /* Remove Button */
	    d->remove_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_remove_20x20_xpm,
		"Remove", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(remove_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_r);
	    gtk_widget_show(w);


	    w = gtk_vbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Shift Up Button */
	    d->shift_up_btn = w = GUIButtonArrowLabelH(
		GTK_ARROW_UP, 20, 20, "Shift Up", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(shift_up_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_u, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_u);
	    gtk_widget_show(w);

	    /* Shift Down Button */
	    d->shift_down_btn = w = GUIButtonArrowLabelH(
		GTK_ARROW_DOWN, 20, 20, "Shift Down", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(shift_down_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_d, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"                             
	    );
	    GUIButtonLabelUnderline(w, GDK_d);
	    gtk_widget_show(w);


	    w = gtk_vbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Play Button */
	    d->play_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_play_20x20_xpm,
		"Play", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(play_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_p);
	    gtk_widget_show(w);

	    /* Pause Button */
	    d->pause_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_pause_20x20_xpm,
		"Pause", NULL
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_set_usize(w, bw, bh);
	    gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(pause_cb), d
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_s);
	    gtk_widget_show(w);


	    /* Right-click menu */
	    if(TRUE)
	    {
		GtkWidget *menu = (GtkWidget *)GUIMenuCreate();
		guint8 **icon;
		const gchar *label;
		guint accel_key, accel_mods;
		gpointer data = d;
		void (*func_cb)(GtkWidget *w, gpointer);

#define DO_ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  data, func_cb					\
 );						\
}
#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
		d->menu = menu;

		icon = (guint8 **)icon_add_20x20_xpm;
		label = "Add...";
		accel_key = GDK_a;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = add_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->add_mi = w;

		icon = (guint8 **)icon_edit_20x20_xpm;
		label = "Edit...";
		accel_key = GDK_e;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = edit_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->edit_mi = w;

		icon = (guint8 **)icon_remove_20x20_xpm;
		label = "Remove";
		accel_key = GDK_r;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = remove_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->remove_mi = w;

		DO_ADD_MENU_SEP

		icon = NULL;
		label = "Shift Up";
		accel_key = GDK_u;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = shift_up_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->shift_up_mi = w;

		icon = NULL;
		label = "Shift Down";
		accel_key = GDK_d;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = shift_down_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->shift_down_mi = w; 

		DO_ADD_MENU_SEP

		icon = (guint8 **)icon_play_20x20_xpm;
		label = "Play";
		accel_key = GDK_p;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = play_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->play_mi = w;

		icon = (guint8 **)icon_pause_20x20_xpm;
		label = "Pause";
		accel_key = GDK_s;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = pause_cb;
		DO_ADD_MENU_ITEM_LABEL
		d->pause_mi = w;
	    }
	}

	d->freeze_count--;

	/* Map the dialog */
	gtk_widget_show_raise(toplevel);
	gtk_widget_grab_focus(d->clist);

	/* Add the image's frames to the Frames GtkCList */
	if(iv != NULL)
	{
	    GtkCList *clist = GTK_CLIST(d->clist);
	    imgview_image_struct *img;

	    set_busy(d, TRUE);

	    /* Add the frames to the GtkCList */
	    img = ImgViewGetImage(iv);
	    if((img != NULL) ? (img->bpp == 4) : FALSE)
	    {
		const gint	cur_frame = ImgViewGetCurrentFrame(iv),
				m = ImgViewGetTotalFrames(iv);
		gint row;
		GList *glist;
		imgview_frame_struct *frame;

		gtk_clist_freeze(clist);

		/* Map the progress dialog for multiple frames */
		if(m > 1)
		{
		    gchar *msg = g_strdup_printf(
"Loading %i frames...",
			m
		    );
		    if(ProgressDialogIsQuery())
			ProgressDialogBreakQuery(TRUE);
		    ProgressDialogSetTransientFor(ref_toplevel);
		    ProgressDialogMap(
			"Loading Frames",
			msg,
			(const guint8 **)icon_animate_32x32_xpm,
			"Stop"
		    );
		    g_free(msg);
		    gdk_flush();
		}

		/* Iterate through each frame on the image and append
		 * each frame to the GtkCList
		 */
		row = 0;
		for(glist = img->frames_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    frame = IMGVIEW_FRAME(glist->data);
		    if(frame == NULL)
			continue;

		    if(ProgressDialogIsQuery() && (m > 0))
		    {
			gchar *msg = g_strdup_printf(
"Loading frame #%i...",
			    row + 1
			);
			ProgressDialogUpdate(
			    NULL, msg, NULL, NULL,
			    (gdouble)row / (gdouble)m,
#ifdef PROGRESS_BAR_DEF_TICKS
			    PROGRESS_BAR_DEF_TICKS,
#else
			    25,
#endif
			    TRUE
			);
			g_free(msg);
			if(ProgressDialogStopCount() > 0)
			    break;
		    }

		    /* Append this frame */
		    row = clist_row_insert(
			clist, -1,
			frame->buf, img->width, img->height, img->bpl,
			frame->delay
		    );

		    if(ProgressDialogIsQuery() && (m > 0))
		    {
			ProgressDialogUpdate(
			    NULL, NULL, NULL, NULL,
			    (gdouble)(row + 1) / (gdouble)m,
#ifdef PROGRESS_BAR_DEF_TICKS
			    PROGRESS_BAR_DEF_TICKS,
#else
			    25,
#endif
			    TRUE
			);
			if(ProgressDialogStopCount() > 0)
			    break;
		    }
		}

		/* Select the current frame if not playing */
		if(clist->rows > 0)
		{
		    if(!ImgViewIsPlaying(iv))
			gtk_clist_select_row(clist, cur_frame, 0);
		}

		gtk_clist_thaw(clist);

		/* Unmap the progress dialog */
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	    }

	    update_stats_label(d);
	    update(d);

	    set_busy(d, FALSE);
	}

	/* Push a GTK main level and wait for user response */
	gtk_main();

	/* Get values from widgets if user clicked on save */
	has_changes = d->has_changes;
	if(has_changes)
	{

	}

	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
/*	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE); */
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	gtk_clist_freeze(GTK_CLIST(d->clist));
	gtk_clist_clear(GTK_CLIST(d->clist));
	gtk_clist_thaw(GTK_CLIST(d->clist));

	GTK_WIDGET_DESTROY(d->menu);
	GTK_WIDGET_DESTROY(d->clist);
	GTK_WIDGET_DESTROY(d->stats_label);
	GTK_WIDGET_DESTROY(d->add_btn);
	GTK_WIDGET_DESTROY(d->edit_btn);
	GTK_WIDGET_DESTROY(d->remove_btn);
	GTK_WIDGET_DESTROY(d->shift_up_btn);
	GTK_WIDGET_DESTROY(d->shift_down_btn);
	GTK_WIDGET_DESTROY(d->play_btn);
	GTK_WIDGET_DESTROY(d->pause_btn);
	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	GDK_CURSOR_DESTROY(d->busy_cur);
	g_free(d);

	return(has_changes);
}
