#ifdef HAVE_LIBGIF
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <gif_lib.h>

#include "rgba_to_cidx.h"
#include "imgio.h"


/* Last error message pointer */
extern const char	*imgio_last_load_error,
			*imgio_last_write_error;


typedef struct _ImgGIFReadData		ImgGIFReadData;
#define IMG_GIF_READ_DATA(p)		((ImgGIFReadData *)(p))
typedef struct _RGBAToCIdxData		RGBAToCIdxData;
#define RGBA_TO_CIDX_DATA(p)		((RGBAToCIdxData *)(p))


/* GIF Library Version */
void ImgGIFVersion(int *major, int *minor, int *release); 

mode_t ImgGIFGetFileMode(const char *filename);

/* GIF Reading */
static int ImgGIFReadCB(GifFileType *ft, GifByteType *buf, int buf_len);
int ImgGIFReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t ***rgba_rtn, unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,		/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	int *user_aborted
);


/* GIF Dithering */
static int ImgGIFRGBAToCIdxCB(int i, int m, void *data);
static int ImgGIFDitherRGBA(
	const int frame_num, const int nframes,
	u_int8_t *rgba,
	const int width, const int height, const int bpl,
	const u_int8_t *bg_color, int *bg_color_num,
	int *trans_color_num,
	ColorMapObject **colormap_rtn,
	GifByteType **data_rtn, int *data_bpl_rtn,
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
);
/* GIF Write Looping Control Block */
static int ImgWriteGIFFileLoopingBlock(GifFileType *ft);
/* GIF Writing */
static int ImgWriteGIFFileRGBAToColor(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
);
static int ImgWriteGIFFileRGBAToGreyscale(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
);
static int ImgWriteGIFFileRGBAToBW(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
);
int ImgWriteGIFFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,		/* bpp must be 4 */
	u_int8_t **rgba, unsigned long *delay_list,
	int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	int interlaced,			/* 0 or 1 */
	int format,			/* 0 = B&W
					 * 1 = Greyscale
					 * 2 = Color */
	int transparency,		/* 0 or 1 */
	int looping,			/* 0 = no looping, 1 = looping */
	void *client_data,
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	)
);


/*
 *	Graphic Control Flags:
 */
#define GIF_GRAPHIC_CONTROL_TRANSPARENCY	(1 << 0)
#define GIF_GRAPHIC_CONTROL_USER_INPUT		(1 << 1)
#define GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK	((1 << 2) | (1 << 3) | (1 << 4))


/*
 *	Disposal Method Flags:
 *
 *	Obtained from the graphic control flags as follows:
 *
 *	disposal_method_flags = ((graphic_control_flags &
 *		GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK) >> 2);
 */
#define GIF_DISPOSAL_METHOD_NONE		0
#define GIF_DISPOSAL_METHOD_DO_NOTHING		(1 << 0)	/* Same as GIF_DISPOSAL_METHOD_NONE */
#define GIF_DISPOSAL_METHOD_CLEAR_WITH_BG	(1 << 1)	/* Clear with the background color */
#define GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV	(1 << 2)	/* Clear with the previous image */
#define GIF_DISPOSAL_METHOD_MASK		((1 << 0) | (1 << 1) | (1 << 2))


/*
 *	GIF Read Callback Data:
 */
struct _ImgGIFReadData {
	FILE		*fp;
	struct stat	stat_buf;
	int		width, height,
			bpp, bpl;
	u_int8_t	*rgba;
	int		(*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	);
	void		*progress_data;
	int		*user_aborted;
};

/*
 *	RGBAToCIdx Callback Data:
 */
struct _RGBAToCIdxData {

	int		(*progress_cb)(
		void *,		/* Data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	);
	void		*data;
	int		frame_num,
			nframes;
	u_int8_t	*rgba;
	int		width,
			height,
			bpp,
			bpl;
	int		*user_aborted;
};


static int	gif_interlace_offset[] = { 0, 4, 2, 1 },
		gif_interlace_jumps[] = { 8, 8, 4, 2 };


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the GIF library's version.
 */
void ImgGIFVersion(int *major, int *minor, int *release) 
{
	const char *ver_str = GIF_LIB_VERSION, *s;

	s = strstr(ver_str, "Version");
	while((*s != '\0') && (*s != ' '))
	    s++;
	while(*s == ' ')
	    s++;
	if(major != NULL)
	    *major = ATOI(s);

	while((*s != '\0') && (*s != '.'))
	    s++;
	while(*s == '.')
	    s++;
	if(minor != NULL)
	    *minor = ATOI(s);

	while((*s != '\0') && (*s != '.'))
	    s++;
	while(*s == '.')
	    s++;
	if(release != NULL)
	    *release = ATOI(s);
}


/*
 *	Returns the permissions of the specified file.
 *
 *	If the file does not exist then the default permissions are
 *	returned.
 *
 *	This is needed for the GIF write functions since they do not
 *	set the file permissions on a newly created file.
 */
mode_t ImgGIFGetFileMode(const char *filename)
{
	struct stat stat_buf;

	if(stat(filename, &stat_buf))
	{
	    const mode_t m = umask(0);
	    umask(m);
	    return(
		(~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
	    );
	}
	else
	{
	    return(stat_buf.st_mode);
	}
}


/*
 *	GIF read callback.
 */
static int ImgGIFReadCB(GifFileType *ft, GifByteType *buf, int buf_len)
{
	int i, bytes_read, total_bytes_read, read_len;
	ImgGIFReadData *d = IMG_GIF_READ_DATA(ft->UserData);
	if((d == NULL) || (buf == NULL) || (buf_len <= 0))
	    return(0);
	if(d->fp == NULL)
	    return(0);

	total_bytes_read = 0;
	read_len = MAX((int)d->stat_buf.st_blksize, 1);
	i = 0;
	while(i < buf_len)
	{
	    bytes_read = fread(
		buf + i,
		sizeof(GifByteType),
		MIN((buf_len - i), read_len),
		d->fp
	    );
	    if(bytes_read <= 0)
		break;

	    total_bytes_read += bytes_read;

	    if(d->progress_cb != NULL)
	    {
		if(!d->progress_cb(
		    d->progress_data,
		    (int)ftell(d->fp), (int)d->stat_buf.st_size,
		    d->width, d->height,
		    d->bpl, d->bpp,
		    d->rgba
		))
		    *d->user_aborted = 1;
	    }

	    i += bytes_read;
	}

	return(total_bytes_read);
}

/*
 *	Reads the GIF file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value (invalid format/not a gif file)
 *	-3	Systems error
 *	-4	User abort
 */
int ImgGIFReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t ***rgba_rtn, unsigned long **delay_list_rtn,
	int *nframes_rtn,
	u_int8_t *bg_color,		/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	), 
	int *user_aborted
)
{
	int		i, status,
			scr_width, scr_height,
			scr_bpp, scr_bpl,
			scr_buf_len,
			bg_color_num, trans_color_num = -1;
	char		*comment = NULL;
	u_int8_t	disposal_method = GIF_DISPOSAL_METHOD_NONE,
			prev_disposal_method = disposal_method;
	const u_int8_t	*cwp_rgba = NULL;	/* Clear With Previous RGBA image data */
	GifFileType *ft;
	GifRowType *scr_buf;		/* Main GIF screen buffer */
	GifRecordType record_type;
	ImgGIFReadData *read_data;

	/* Reset last load error */
	imgio_last_load_error = NULL;

	/* Skip if user aborted */
	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}

	/* Allocate the GIF read callback data */
	read_data = IMG_GIF_READ_DATA(calloc(1, sizeof(ImgGIFReadData)));
	if(read_data == NULL)
	{
	    imgio_last_load_error = "Memory allocation error";
	    return(-3);
	}
	read_data->fp = fopen(filename, "rb");
	if(read_data->fp == NULL)
	{
	    free(read_data);
	    imgio_last_load_error = "Unable to open the file for reading";
	    return(-1);
	}
	fstat(fileno(read_data->fp), &read_data->stat_buf);
	read_data->progress_cb = progress_cb;
	read_data->progress_data = client_data;
	read_data->user_aborted = user_aborted;

	ft = DGifOpen(read_data, ImgGIFReadCB);
	if(ft == NULL)
	{
	    fclose(read_data->fp);
	    free(read_data);
	    imgio_last_load_error = "Unable to open the file for reading";
	    return(-1);
	}

	/* Get the GIF screen values */
	scr_width = ft->SWidth;
	scr_height = ft->SHeight;
	bg_color_num = ft->SBackGroundColor;

	read_data->width = scr_width;
	read_data->height = scr_height;
	read_data->bpp = 4;
	read_data->bpl = read_data->width * read_data->bpp;
	read_data->rgba = NULL;

	/* Get the GIF background color */
	if((ft->SColorMap != NULL) && (bg_color != NULL))
	{
	    const ColorMapObject *colormap = ft->SColorMap;
	    const GifColorType *c = &colormap->Colors[bg_color_num];
	    bg_color[0] = (u_int8_t)c->Red;
	    bg_color[1] = (u_int8_t)c->Green;
	    bg_color[2] = (u_int8_t)c->Blue;
	    bg_color[3] = 0xff;
	}

	/* Allocate the main GIF screen buffer, an array of
	 * GifRowType *, each image in the GIF file will be first
	 * opened to this before being rendered to the RGBA image
	 * data
	 */
	scr_buf_len = scr_height * sizeof(GifRowType *);
	scr_buf = (scr_buf_len > 0) ?
	    (GifRowType *)malloc(scr_buf_len) : NULL;
	if(scr_buf == NULL)
	{
	    DGifCloseFile(ft);
	    fclose(read_data->fp);
	    free(read_data);
	    imgio_last_load_error = "Memory allocation error";
	    return(-3);
	}

	/* Calculate the main GIF screen buffer's bytes per line */
	scr_bpp = sizeof(GifPixelType);
	scr_bpl = scr_width * scr_bpp;

        /* Allocate the main GIF screen buffer's rows and initialize
	 * them to the background color
	 */
        scr_buf[0] = (GifRowType)malloc(scr_bpl);
        for(i = 0; i < scr_width; i++) 
            scr_buf[0][i] = bg_color_num;

        for(i = 1; i < scr_height; i++)
        {
            scr_buf[i] = (GifRowType)malloc(scr_bpl);
            memcpy(scr_buf[i], scr_buf[0], scr_bpl);
        }


	/* Begin reading the GIF file and open the images
	 *
	 * Each image will be opened to the main GIF screen buffer
	 * and then rendered to the RGBA image data
	 */
	status = 0;
	do
	{
	    if(*read_data->user_aborted)
	    {
		status = -4;
		break;
	    }

	    /* Get the next record type */
	    if(DGifGetRecordType(ft, &record_type) != GIF_OK)
	    {
		imgio_last_load_error = "Unable to obtain the GIF record type";
		status = -1;
		break;
	    }

	    if(*read_data->user_aborted)
	    {
		status = -4;
		break;
	    }

	    switch(record_type)
	    {
	      case UNDEFINED_RECORD_TYPE:
		break;

	      case SCREEN_DESC_RECORD_TYPE:
		break;

	      case IMAGE_DESC_RECORD_TYPE:
		if(DGifGetImageDesc(ft) == GIF_OK)
		{
		    const int	rgba_bpp = read_data->bpp,
				rgba_bpl = read_data->bpl,
				frame_num = *nframes_rtn,
				prev_frame_num = frame_num - 1;
		    GifImageDesc *img = &ft->Image;
		    int		x, y, j, c_num,
				top = img->Top,
				left = img->Left,
				width = img->Width,
				height = img->Height;
		    const u_int8_t	*prev_rgba = (prev_frame_num >= 0) ?
			(*rgba_rtn)[prev_frame_num] : NULL;
		    u_int8_t *rgba, *rgba_ptr;
		    const GifColorType *c;
		    GifRowType scr_row;
		    const ColorMapObject *colormap = (img->ColorMap != NULL) ?
			img->ColorMap : ft->SColorMap;

		    if(*read_data->user_aborted)
		    {
			status = -4;
			break;
		    }

		    /* No colormap defined by local image or system? */
		    if((colormap != NULL) ? (colormap->ColorCount <= 0) : TRUE)
		    {
			imgio_last_load_error = "No GIF colormap found";
			status = -2;
			break;
		    }

		    /* Allocate the RGBA image data */
		    read_data->rgba = rgba = (u_int8_t *)malloc(
			rgba_bpl * scr_height
		    );
		    if(rgba == NULL)
		    {
			imgio_last_load_error = "Memory allocation error";
			status = -3;
			break;
		    }

		    /* Open this GIF image to the main GIF screen buffer
		     *
		     * Interlaced?
		     */
		    if(img->Interlace)
		    {
			/* Interlace needs 4 passes on the image */
			for(i = 0; i < 4; i++)
			{
			    for(j = top + gif_interlace_offset[i];
				j < (top + height);
				j += gif_interlace_jumps[i]
			    )
			    {
				if(DGifGetLine(
				    ft,
				    (GifPixelType *)&scr_buf[j][left],
				    width
				) != GIF_OK)
				{
				    status = -1;
				    break;
				}

				/* Check for user abort */
				if(*read_data->user_aborted)
				{
				    status = -4;
				    break;
				}
			    }
			}
		    }
		    else
		    {
			/* Read each line sequentially */
			for(i = 0; i < height; i++)
			{
			    if(DGifGetLine(
				ft,
				(GifPixelType *)&scr_buf[top++][left],
				width
			    ) != GIF_OK)
			    {
				status = -1;
				break;
			    }

			    /* Check for user abort */
			    if(*read_data->user_aborted)
			    {
				status = -4;
				break;
			    }
			}
		    }
		    if(*read_data->user_aborted)
		    {
			free(rgba);
			status = -4;
			break;
		    }

		    /* Copy/render the entire GIF screen buffer to the
		     * RGBA image data
		     */
		    for(y = 0; y < scr_height; y++)
		    {
			scr_row = scr_buf[y];
			rgba_ptr = rgba + (y * rgba_bpl);
			for(x = 0; x < scr_width; x++)
			{
			    /* Current pixel value is colormap index */
			    c_num = (int)scr_row[x];

			    /* Not transparent? */
			    if(c_num != trans_color_num)
			    {
				c = &colormap->Colors[c_num];
				*rgba_ptr++ = (u_int8_t)c->Red;
				*rgba_ptr++ = (u_int8_t)c->Green;
				*rgba_ptr++ = (u_int8_t)c->Blue;
				*rgba_ptr++ = def_alpha_value;
			    }
			    else
			    {
				/* Is transparent, handle by the previous
				 * image's disposal method
				 */
				switch(prev_disposal_method)
				{
				  case GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV:
				    /* Use the clear with previous image's pixel */
				    if(cwp_rgba != NULL)
				    {
					memcpy(
					    rgba_ptr,
					    cwp_rgba + (y * rgba_bpl) +
						(x * rgba_bpp),
					    rgba_bpp
					);
					rgba_ptr += rgba_bpp;
				    }
				    else
				    {
					/* No clear with previous image,
					 * set to transparent
					 */
					c = &colormap->Colors[c_num];
					*rgba_ptr++ = (u_int8_t)c->Red;
					*rgba_ptr++ = (u_int8_t)c->Green;
					*rgba_ptr++ = (u_int8_t)c->Blue;
					*rgba_ptr++ = 0x00;
				    }
				    break;

				  /* Clear with background */
				  case GIF_DISPOSAL_METHOD_CLEAR_WITH_BG:
				    /* Set to transparent */
				    c = &colormap->Colors[c_num];
				    *rgba_ptr++ = (u_int8_t)c->Red;
				    *rgba_ptr++ = (u_int8_t)c->Green;
				    *rgba_ptr++ = (u_int8_t)c->Blue;
				    *rgba_ptr++ = 0x00;
				    break;

				  case GIF_DISPOSAL_METHOD_DO_NOTHING:
				  case GIF_DISPOSAL_METHOD_NONE:
				  default:
				    /* Use the previous image's pixel */
				    if(prev_rgba != NULL)
				    {
					memcpy(
					    rgba_ptr,
					    prev_rgba + (y * rgba_bpl) +
						(x * rgba_bpp),
					    rgba_bpp
					);
					rgba_ptr += rgba_bpp;
				    }
				    else
				    {
					/* No previous image,
					 * set to transparent
					 */
					c = &colormap->Colors[c_num];
					*rgba_ptr++ = (u_int8_t)c->Red;
					*rgba_ptr++ = (u_int8_t)c->Green;
					*rgba_ptr++ = (u_int8_t)c->Blue;
					*rgba_ptr++ = 0x00;
				    }
				    break;
				}
			    }
			}
		    }

		    /* Append this RGBA image data to the RGBA images list */
		    *nframes_rtn = frame_num + 1;
		    *rgba_rtn = (u_int8_t **)realloc(
			*rgba_rtn,
			(*nframes_rtn) * sizeof(u_int8_t *)
		    );
		    if(*rgba_rtn != NULL)
		    {
			(*rgba_rtn)[frame_num] = rgba;
		    }
		    else
		    {
			*nframes_rtn = 0;
			free(rgba);
			imgio_last_load_error = "Memory allocation error";
			status = -3;
		    }

		    /* Clear the GIF screen buffer's background? */
                    if(disposal_method & GIF_DISPOSAL_METHOD_CLEAR_WITH_BG)
                    {
			int i;

                        /* Clear the first line */
                        for(i = 0; i < scr_width; i++) 
                            scr_buf[0][i] = bg_color_num;
                        /* Copy the first line to the subsequent lines */
                        for(i = 1; i < scr_height; i++)
                            memcpy(scr_buf[i], scr_buf[0], scr_bpl);
                    }

		    /* If this frame's disposal method is not
		     * clear with previous then we need to
		     * unset the Clear With Previous RGBA
		     * image data pointer
		     */
		    if(!(disposal_method & GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV))
			cwp_rgba = NULL;
		}
		break;

	      case EXTENSION_RECORD_TYPE:
		if(1)
		{
		    u_int8_t get_error_need_break;
		    int code, ext_len;
		    GifByteType *ext;

		    /* Get the first extension */
		    if(DGifGetExtension(ft, &code, &ext) == GIF_OK)
		    {
			/* Iterate through the extensions */
			while((ext != NULL) && !(*read_data->user_aborted))
			{
			    /* Get this extension's length */
			    ext_len = (int)ext[0];

			    /* Reset the get extension error marker */
			    get_error_need_break = 0;

			    /* Handle by extension code */
			    switch(code)
			    {
			      case COMMENT_EXT_FUNC_CODE:	/* Comment */
				if(ext_len > 0)
				{
				    const int comment_len = STRLEN(comment);
				    if(comment != NULL)
				    {
					comment = (char *)realloc(
					    comment,
					    comment_len + ext_len + 1
					);
				    }
				    else
				    {
					comment = (char *)malloc(ext_len + 1);
					*comment = '\0';
				    }
				    if(comment != NULL)
				    {
					memcpy(
					    comment + comment_len,
					    ext + 1,
					    ext_len
					);
					comment[comment_len + ext_len] = '\0';
				    }
				}
				break;

			      case GRAPHICS_EXT_FUNC_CODE:	/* Graphics Control */
				/* Format (in bytes):
				 *
				 * <len>
				 * <flags> <delay_time> <delay_time_cnt>
				 * <transparent_color>
				 */
				if(ext_len >= 4)
				{
				    const int nframes = *nframes_rtn;
				    const u_int8_t flags = (u_int8_t)ext[1];
				    char user_input;

				    /* Get the delay in milliseconds
				     *
				     * Needed to add 1 to nframes since
				     * Graphic Controls come before
				     * each frame
				     *
				     * Delay stored on GIF files are in
				     * hundreths of a second, need to
				     * * 10 to convert to milliseconds
				     */
				    *delay_list_rtn = (unsigned long *)realloc(
					*delay_list_rtn,
					(nframes + 1) * sizeof(unsigned long)
				    );
				    if(*delay_list_rtn != NULL)
					(*delay_list_rtn)[nframes] = (unsigned long)(
					    (u_int16_t)ext[2] |
					    ((u_int16_t)ext[3] << 8)
					) * 10l;

				    /* Get the transparent color number */
				    trans_color_num = (flags & GIF_GRAPHIC_CONTROL_TRANSPARENCY) ?
					(int)ext[4] : -1;

				    /* Get user input */
				    user_input = (flags & GIF_GRAPHIC_CONTROL_USER_INPUT) ?
					1 : 0;
				    if(user_input)
				    {
					/* Wait for user input */

				    }

				    /* Record the previous disposal
				     * method and get the disposal
				     * method for this image
				     */
				    prev_disposal_method = disposal_method;
				    disposal_method = ((flags &
					GIF_GRAPHIC_CONTROL_DISPOSAL_METHOD_MASK) >> 2);

				    /* If this disposal method is clear
				     * with prevous then update the
				     * Clear With Previous RGBA image
				     * data pointer if it was not set
				     * so that subsequent clear with
				     * previous disposal methods use
				     * this RGBA image data
				     */
				    if(disposal_method & GIF_DISPOSAL_METHOD_CLEAR_WITH_PREV)
				    {
					if(cwp_rgba == NULL)
					{
					    const int frame_num = (*nframes_rtn) - 1;
					    if(frame_num >= 0)
						cwp_rgba = (*rgba_rtn)[frame_num];
					}
				    }
				}
				break;

			      case PLAINTEXT_EXT_FUNC_CODE:
				break;

			      case APPLICATION_EXT_FUNC_CODE:
				if(ext_len > 0)
				{
				    /* Netscape 2.0 looping control? */
				    if((ext_len >= 11) ?
					!memcmp(ext + 1, "NETSCAPE2.0", 11) : 0
				    )
				    {
					/* Get the value from the next extension */
					if(DGifGetExtensionNext(ft, &ext) != GIF_OK)
					{
					    get_error_need_break = 1;
					    break;
					}
					ext_len = (int)ext[0];
					if(ext_len >= 3)
					{
#if 0
					    const int	v1 = (int)ext[1],	/* v1 is always 1 */
							loop_count = (int)ext[2] |
							    ((int)ext[3] << 8);
#endif
					}
				    }
				}
				break;
			    }

			    /* Get extension error? */
			    if(get_error_need_break)
				break;

			    /* Get the next extension */
			    if(DGifGetExtensionNext(ft, &ext) != GIF_OK)
				break;
			}
		    }
		}
		break;

	      case TERMINATE_RECORD_TYPE:
		break;
	    }
	}
	while(record_type != TERMINATE_RECORD_TYPE);


	/* Close the GIF file and delete the read data */
	DGifCloseFile(ft);
	fclose(read_data->fp);
	free(read_data);

	/* Delete the GIF screen buffer */
	for(i = 0; i < scr_height; i++)
	    free(scr_buf[i]);
	free(scr_buf);

	/* Update the returns */
	if(width_rtn != NULL)
	    *width_rtn = scr_width;
	if(height_rtn != NULL)
	    *height_rtn = scr_height;
	if(bpp_rtn != NULL)
	    *bpp_rtn = 4;
	if(bpl_rtn != NULL)
	    *bpl_rtn = scr_width * 4;
	if(base_width_rtn != NULL)
	    *base_width_rtn = scr_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = scr_height;
	if(comments_rtn != NULL)
	{
	    free(*comments_rtn);
	    *comments_rtn = comment;
	    comment = NULL;
	}

	/* Delete values that were not transfered to the returns */
	free(comment);

	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}


/*
 *	RGBAToCIdx progress callback.
 */
static int ImgGIFRGBAToCIdxCB(int i, int m, void *data)
{
	RGBAToCIdxData *d = (RGBAToCIdxData *)data;
	if(d == NULL)
	    return(1);

	if((d->progress_cb != NULL) && ((i % 5) == 0))
	{
	    if(!d->progress_cb(
		d->data,
		(d->frame_num * m) + (i / 2),
		(d->nframes * m),
		d->width, d->height,
		d->bpl, d->bpp,
		d->rgba
	    ))
		*d->user_aborted = 1;
	}

	return(*d->user_aborted);
}

/*
 *	Creates a GIF ColorMapObject and GIF image from the specified
 *	RGBA image data.
 *
 *	If bg_color and bg_color_num are not NULL then the background
 *	color specified by bg_color will be added to the colormap.
 *	*bg_color_num will be updated with the background color number.
 *
 *	If trans_color_num is not NULL then transparency will be
 *	processed and a transparent color will be added to the colormap.
 *	*trans_color_num will be updated with the transparent color
 *	number.
 *
 *	Inputs assumed valid except as noted above.
 */
static int ImgGIFDitherRGBA(
	const int frame_num, const int nframes,
	u_int8_t *rgba,
	const int width, const int height, const int bpl,
	const u_int8_t *bg_color, int *bg_color_num,
	int *trans_color_num,
	ColorMapObject **colormap_rtn,
	GifByteType **data_rtn, int *data_bpl_rtn,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
)
{
	const int bpp = 4;
	int		i, n, data_bpl, ncolors,
			max_colors = 256;
	u_int8_t *cidx;
	GifColorType *gif_c;
	ColormapColor *c, *colors_list;
	ColorMapObject *colormap;
	GifByteType *data;
	RGBAToCIdxData *d;

	/* Reduce the maximum number of allowed dither colors for
	 * the background and transparent colors
	 */
	if((bg_color != NULL) || (trans_color_num != NULL))
	    max_colors--;

	/* Allocate the RGBAToCIdx() callback data */
	d = (RGBAToCIdxData *)calloc(1, sizeof(RGBAToCIdxData));
	d->progress_cb = progress_cb;
	d->data = progress_data;
	d->frame_num = frame_num;
	d->nframes = nframes;
	d->width = width;
	d->height = height;
	d->bpp = bpp;
	d->bpl = bpl;
	d->rgba = rgba;
	d->user_aborted = user_aborted;

	/* Generate the color index image data and the corresponding
	 * colormap
	 */
 	RGBAToCIdx(
	    rgba, width, height, bpl,
	    max_colors,
	    &cidx,
	    &colors_list,
	    &ncolors,
	    ImgGIFRGBAToCIdxCB,
	    d
	);
	free(d);

	/* User aborted? */
	if(*user_aborted)
	{
	    free(cidx);
	    free(colors_list);
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}

	/* Get the generated color index image as the GIF image */
	data = (GifByteType *)cidx;
	data_bpl = width * sizeof(GifByteType);
	if(data == NULL)
	{
	    free(colors_list);
	    imgio_last_write_error =
		"Unable to generate color index image";
	    return(-1);
	}

	/* Unable to generate the colors list? */
	if(colors_list == NULL)
	{
	    free(data);
	    imgio_last_write_error =
		"Unable to generate color palette";
	    return(-1);
	}

	/* Append the background color? */
	if(bg_color != NULL)
	{
	    const int i = ncolors;
	    ncolors = i + 1;
	    colors_list = (ColormapColor *)realloc(
		colors_list,
		ncolors * sizeof(ColormapColor)
	    );
	    if(colors_list == NULL)
	    {
		free(data);
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }
	    c = &colors_list[i];
	    c->r = bg_color[0];
	    c->g = bg_color[1];
	    c->b = bg_color[2];
	    if(bg_color_num != NULL)
		*bg_color_num = i;
	}

	/* Need transparent color and background color available? */
	if((trans_color_num != NULL) && (bg_color != NULL) &&
	   (bg_color_num != NULL)
	)
	{
	    /* Use background color as the transparent color */
	    *trans_color_num = *bg_color_num;
	}
	/* Append transparent color? */
	else if(trans_color_num != NULL)
	{
	    const int i = ncolors;
	    ncolors = i + 1;
	    colors_list = (ColormapColor *)realloc(
		colors_list,
		ncolors * sizeof(ColormapColor)
	    );
	    if(colors_list == NULL)
	    {
		imgio_last_write_error = "Memory allocation error";
		return(-3);
	    }
	    c = &colors_list[i];
	    c->r = 0x00;
	    c->g = 0x00;
	    c->b = 0x00;
	    *trans_color_num = i;
	}

	/* Need to apply the transparency to the color index image
	 * data?
	 */
	if(trans_color_num != NULL)
	{
	    const int	cidx_bpp = sizeof(GifByteType),
			cidx_bpl = width * cidx_bpp;
	    const GifByteType v = (GifByteType)(*trans_color_num);
	    int y;
	    const u_int8_t alpha_threshold = 0x80;
	    const u_int8_t *rgba_ptr;
	    GifByteType *cidx_ptr, *cidx_end;

	    /* Scan the RGBA image data, set all values on the color
	     * index image to the transparent color number v that
	     * correspond to a transparent pixel on the RGBA image
	     * data
	     */
	    for(y = 0; y < height; y++)
	    {
		rgba_ptr = rgba + (y * bpl);
		cidx_ptr = data + (y * cidx_bpl);
	        cidx_end = cidx_ptr + (width * cidx_bpp);
	        while(cidx_ptr < cidx_end)
	        {
		    /* Transparent? */
		    if(rgba_ptr[3] < alpha_threshold)
			*cidx_ptr = v;
		    rgba_ptr += 4;
		    cidx_ptr += 1;
		}
	    }
	}

	/* Create the GIF colormap */
	colormap = MakeMapObject(
	    256,		/* Must be 256 or else fails */
	    NULL
	);
	if(colormap == NULL)
	{
	    free(data);
	    free(colors_list);
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Copy the colors from the colors list to the GIF colormap */
	n = MIN(ncolors, colormap->ColorCount);
	for(i = 0; i < n; i++)
	{
	    c = &colors_list[i];
	    gif_c = &colormap->Colors[i];
	    gif_c->Red = (GifByteType)c->r;
	    gif_c->Green = (GifByteType)c->g;
	    gif_c->Blue = (GifByteType)c->b;
	}

	/* Delete the colors list */
	free(colors_list);

	*data_rtn = data;
	*data_bpl_rtn = data_bpl;
	*colormap_rtn = colormap;

	return(0);
}


/*
 *	Writes a Netscape 2.0 looping application control block.
 *
 *	This should be called right after each EGifPutScreenDesc().
 */
static int ImgWriteGIFFileLoopingBlock(GifFileType *ft)
{
	const u_int8_t buf[] = {
		'N',			/* Bytes 0 to 7 = "NETSCAPE" */
		'E',
		'T',
		'S',
		'C',
		'A',
		'P',
		'E',
		'2',			/* Bytes 8 to 10 = "2.0" */
		'.',
		'0',
	};
	const u_int8_t buf2[] = {
		0x01,			/* 0x01 (unknown why as to this value) */
		0x00,			/* Lo byte of a u_int16_t loop count */
		0x00			/* Hi byte of a u_int16_t loop count */
	    };

	if(EGifPutExtensionFirst(
	    ft, APPLICATION_EXT_FUNC_CODE,
	    sizeof(buf), buf
	) != GIF_OK)
	    return(-1);

	if(EGifPutExtensionLast(
	    ft, APPLICATION_EXT_FUNC_CODE,
	    sizeof(buf2), buf2
	) != GIF_OK)
	    return(-1);

	return(0);
}


/*
 *	Writes the specified image data to a GIF file in color.
 */
static int ImgWriteGIFFileRGBAToColor(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
)
{
	int	status, status2, frame_num, cidx_bpl,
		color_resolution = 8,
		bg_color_num = 0,
		trans_color_num = -1;
	char user_input = 0;
	unsigned long delay;
	u_int8_t disposal_method, *rgba_cur;
	const mode_t mode = ImgGIFGetFileMode(filename);
	ColorMapObject *colormap = NULL;
	GifByteType *cidx = NULL;
	GifFileType *ft;

	/* Open the GIF file for writing */
	ft = EGifOpenFileName(filename, 0);
	if(ft == NULL)
	{
	    imgio_last_write_error = "Unable to open the file for writing";
	    return(-1);
	}

	status = 0;

	/* Iterate through each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame */
	    delay = (delay_list != NULL) ? delay_list[frame_num] : 0l;

	    /* Dither this RGBA image data frame and get the GIF
	     * colormap and the GIF color index image data
	     */
	    status2 = ImgGIFDitherRGBA(
		frame_num, nframes,
		rgba_cur, width, height, bpl,
		bg_color, &bg_color_num,
		&trans_color_num,
		&colormap,
		&cidx, &cidx_bpl,
		progress_cb, progress_data, user_aborted
	    );
	    if((status2 != 0) || (colormap == NULL) || (cidx == NULL))
	    {
		if(colormap != NULL)
		    FreeMapObject(colormap);
		free(cidx);
		continue;
	    }
	    if(*user_aborted)
	    {
		FreeMapObject(colormap);
		free(cidx);
		status = -4;
		break;
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		if(EGifPutScreenDesc(
		    ft,
		    width, height,	/* Screen width & height */
		    color_resolution,	/* Color resolution */
		    bg_color_num,	/* Background color number */
		    colormap		/* Screen colormap */
		) != GIF_OK)
		{
		    FreeMapObject(colormap);
		    free(cidx);
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && looping)
		{
		    if(ImgWriteGIFFileLoopingBlock(ft))
		    {
			FreeMapObject(colormap);
			free(cidx);
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(transparency)
		disposal_method = GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Write the graphic control as needed */
	    if(transparency || user_input || (delay > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (transparency ? GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
	        };
		if(EGifPutExtension(
		    ft, GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		) != GIF_OK)
		{
		    FreeMapObject(colormap);
		    free(cidx);
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    if(EGifPutImageDesc(
		ft,
		0, 0,		/* Top & left */
		width, height,	/* Width & height */
		interlaced,	/* Interlace */
		/* Specify the local colormap only if multiple frames */
		(nframes > 1) ? colormap : NULL
	    ) != GIF_OK)
	    {
		FreeMapObject(colormap);
		free(cidx);
		status = -1;
		break;
	    }

	    /* Delete the GIF colormap */
	    FreeMapObject(colormap);

	    /* Write the image data for this frame */
	    if(interlaced)
	    {
		const int	po = height,
				pm = 2 * height;
		int i, y, lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
			y < height;
			y += gif_interlace_jumps[i]
		    )
		    {
			if(EGifPutLine(
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			) != GIF_OK)
			{
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((progress_cb != NULL) && ((lines_written % 5) == 0))
			{
			    if(!progress_cb(
				progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*user_aborted = 1;
				status = -4;
				break;
			    }
			} 
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		const int	po = height,
				pm = 2 * height;
		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    if(EGifPutLine(
			ft, (GifPixelType *)cidx_ptr, cidx_bpl
		    ) != GIF_OK)
		    {
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((progress_cb != NULL) && ((y % 5) == 0))
		    {
			if(!progress_cb(
			    progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
	    {
		free(cidx);
		break;
	    }

	    /* Delete the GIF color index image data */
	    free(cidx);
	}

	/* Write the comments */
	if(!STRISEMPTY(comments) &&
	   (status == 0) && !(*user_aborted)
	)
	{
	    if(EGifPutExtension(
		ft, COMMENT_EXT_FUNC_CODE,
		STRLEN(comments), comments
	    ) != GIF_OK)
		status = -1;
	}

	/* Close the GIF file */
	EGifCloseFile(ft);

	/* Set/restore the file permissions */
	chmod(filename, mode);

	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

/*
 *	Writes the specified image data to a GIF file in greyscale.
 */
static int ImgWriteGIFFileRGBAToGreyscale(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
)
{
	int	status, frame_num, cidx_bpl,
		color_resolution = 8,
		colormap_ncolors_set = 256,	/* Must be 256 or else fails */
		bg_color_num = 0,
		trans_color_num = -1;
	char user_input = 0;
	unsigned long delay;
	u_int8_t disposal_method, *rgba_cur;
	const mode_t mode = ImgGIFGetFileMode(filename);
	ColorMapObject *colormap;
	GifByteType *cidx;
	GifFileType *ft;

	/* Create the GIF colormap */
	colormap = (ColorMapObject *)malloc(sizeof(ColorMapObject));
	colormap->ColorCount = colormap_ncolors_set;
	colormap->BitsPerPixel = 8;
	colormap->Colors = (GifColorType *)calloc(
	    colormap->ColorCount, sizeof(GifColorType)
	);
	/* Generate the greyscale colors */
	if(colormap->Colors != NULL)
	{
	    GifByteType v_bg;

	    /* Calculate the greyscale background color */
	    if(bg_color != NULL)
		v_bg = (GifByteType)(
		    ((int)bg_color[0] + (int)bg_color[1] +
		    (int)bg_color[2]
		) / 3);
	    else
		v_bg = 0x00;

	    if(transparency)
	    {
		/* Generate 255 colors from black to white, leaving
		 * color 255 for the transparent pixel
		 */
		const int	m = colormap->ColorCount - 1,
				n = MAX(m - 1, 1);
		int i;
		GifColorType *c;
		for(i = 0; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue =
			(GifByteType)(i * 0xff / n);

		    /* If this color matches the background color then
		     * set this color as the background color
		     */
		    if(c->Red == v_bg)
			bg_color_num = i;
		}

		/* Set the transparent color */
		trans_color_num = m;
		c = &colormap->Colors[trans_color_num];
		c->Red = c->Green = c->Blue = 0xff;
	    }
	    else
	    {
		/* Generate 256 colors from black to white */
		int i, m = colormap->ColorCount, n = MAX(m - 1, 1);
		GifColorType *c;
		for(i = 0; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue =
			(GifByteType)(i * 0xff / n);

		    /* If this color matches the background color then
		     * set this color as the background color
		     */
		    if(c->Red == v_bg)
			bg_color_num = i;
		}
	    }
	}

	/* Create the GIF color index image data */
	cidx_bpl = width * sizeof(GifByteType);
	cidx = (GifByteType *)malloc(cidx_bpl * height);
	if(cidx == NULL)
	{
	    FreeMapObject(colormap);
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Open the GIF file for writing */
	ft = EGifOpenFileName(filename, 0);
	if(ft == NULL)
	{
	    FreeMapObject(colormap);
	    free(cidx);
	    imgio_last_write_error = "Unable to open the file for writing";
	    return(-1);
	}

	status = 0;
	 
	/* Iterate through each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame */
	    delay = (delay_list != NULL) ? delay_list[frame_num] : 0l;

	    /* Copy/convert the RGBA image data to the GIF color
	     * index image data in greyscale
	     */
	    if(transparency && (trans_color_num > -1))
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (cidx_bpl * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
	        {
		    /* Solid? */
		    if(rgba_ptr[3] >= 0x80)
			*cidx_ptr++ = (GifByteType)(
		((rgba_ptr[0] + rgba_ptr[1] + rgba_ptr[2]) / 3) *
			    255 / 256
			);
		    else
		        *cidx_ptr++ = (GifByteType)trans_color_num;
		    rgba_ptr += 4;
	        }
	    }
	    else
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr + (cidx_bpl * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
		{
		    *cidx_ptr++ = (GifByteType)(
		    (rgba_ptr[0] + rgba_ptr[1] + rgba_ptr[2]) / 3
		    );
		    rgba_ptr += 4;
		}
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		if(EGifPutScreenDesc(
		    ft,
		    width, height,	/* Screen width & height */
		    color_resolution,	/* Color resolution */
		    bg_color_num,	/* Background color number */
		    colormap		/* Screen colormap */
		) != GIF_OK)
		{
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && looping)
		{
		    if(ImgWriteGIFFileLoopingBlock(ft))
		    {
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(transparency)
		disposal_method = GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Need to write the graphic control? */
	    if(transparency || user_input || (delay > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (transparency ? GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
		};
		if(EGifPutExtension(
		    ft, GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		) != GIF_OK)
		{
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    if(EGifPutImageDesc(
		ft,
		0, 0,		/* Top & left */
		width, height,	/* Width & height */
		interlaced,	/* Interlace */
		NULL		/* No local colormap */
	    ) != GIF_OK)
	    {
		status = -1;
		break;
	    }

	    /* Write image data */
	    if(interlaced)
	    {
		const int	po = 0,
				pm = height;
		int i, y, lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
			y < height;
			y += gif_interlace_jumps[i]
		    )
		    {
			if(EGifPutLine( 
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			) != GIF_OK)
			{
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((progress_cb != NULL) && ((lines_written % 5) == 0))
			{
			    if(!progress_cb(
			        progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*user_aborted = 1;
				status = -4;
				break;
			    }
			}
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		const int	po = 0,
				pm = height;
 		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    if(EGifPutLine(ft, (GifPixelType *)cidx_ptr, cidx_bpl) != GIF_OK)
		    {
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((progress_cb != NULL) && ((y % 5) == 0))
		    {
			if(!progress_cb(
			    progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
		break;
	}

	/* Write the comments */
	if(!STRISEMPTY(comments) &&
	   (status == 0) && !(*user_aborted)
	)
	{
	    if(EGifPutExtension(
		ft, COMMENT_EXT_FUNC_CODE,
		STRLEN(comments), comments
	    ) != GIF_OK)
		status = -1;
	}

	/* Close the GIF file */ 
	EGifCloseFile(ft);

	/* Set/restore file permissions */
	chmod(filename, mode);

	/* Delete the GIF colormap */
	FreeMapObject(colormap);

	/* Delete the GIF color index image data */
	free(cidx);

	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

/*
 *	Writes the specified image data to a GIF file in black & white.
 */
static int ImgWriteGIFFileRGBAToBW(
	const char *filename,
	const int width, const int height,
	const int bpl, const int bpp,
	u_int8_t **rgba, unsigned long *delay_list,
	const int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	const int interlaced,		/* 0 or 1 */
	const int transparency,		/* 0 or 1 */
	const int looping,		/* 0 = no looping, 1 = looping */
	int (*progress_cb)(
		void *,			/* Client data */
		int, int,		/* Progress current & end values */
		int, int,		/* Width & height */
		int, int,		/* Bytes per line & bytes per pixel */
		u_int8_t *		/* RGBA data */
	),
	void *progress_data,
	int *user_aborted
)
{
	int	status, frame_num, cidx_bpl,
		color_resolution = 8,
		colormap_ncolors_set = 256,	/* Must be 256 or else fails */
		bg_color_num = 0,
		trans_color_num = -1;
	char user_input = 0;
	unsigned long delay;
	u_int8_t disposal_method, *rgba_cur;
	const mode_t mode = ImgGIFGetFileMode(filename);
	ColorMapObject *colormap;
	GifByteType *cidx;
	GifFileType *ft;

	/* Create the GIF colormap */
	colormap = (ColorMapObject *)malloc(sizeof(ColorMapObject));
	colormap->ColorCount = colormap_ncolors_set;
	colormap->BitsPerPixel = 8;
	colormap->Colors = (GifColorType *)calloc(
	    colormap->ColorCount, sizeof(GifColorType)
	);
	/* Generate black and white colors */
	if(colormap->Colors != NULL)
	{
	    GifByteType v_bg;

	    /* Calculate the black and white background color */
	    if(bg_color != NULL)
	    {
		const GifByteType v = (GifByteType)((
		    (int)bg_color[0] + (int)bg_color[1] +
		    (int)bg_color[2]
		) / 3);
		v_bg = (v >= 0x80) ? 0xff : 0x00;
	    }
	    else
	    {
		v_bg = 0x00;
	    }

	    if(transparency)
	    {
		/* Generate 3 colors; black, white, and the rest
		 * are black (including color 2 for the transparent
		 * pixel)
		 */

		/* Set color 0 to black */
		int i = 0, m = colormap->ColorCount;
		GifColorType *c = &colormap->Colors[i];
		c->Red = c->Green = c->Blue = (GifByteType)0x00;
		if(c->Red == v_bg)
		    bg_color_num = i;

		/* Set color 1 to white */
		i = 1;
		if(i < m)
		{
		    c = &colormap->Colors[i]; 
		    c->Red = c->Green = c->Blue = (GifByteType)0xff;
		    if(c->Red == v_bg)
			bg_color_num = i;
		}

		/* Set color 2 (transparent) to black */
		trans_color_num = i = 2;
		if(i < m)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}

		/* Set the rest of the colors to black */
		for(i++; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}
	    }
	    else
	    {
		/* Set color 0 to black */
		int i = 0, m = colormap->ColorCount;
		GifColorType *c = &colormap->Colors[i];
		c->Red = c->Green = c->Blue = (GifByteType)0x00;

		/* Set color 1 to white */
		i = 1;
		if(i < m)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0xff;
		}

		/* Set the rest of the colors to black */
		for(i++; i < m; i++)
		{
		    c = &colormap->Colors[i];
		    c->Red = c->Green = c->Blue = (GifByteType)0x00;
		}
	    }
	}

	/* Create the GIF color index image data */
	cidx_bpl = width * sizeof(GifByteType);
	cidx = (GifByteType *)malloc(cidx_bpl * height);
	if(cidx == NULL)
	{
	    FreeMapObject(colormap);
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	/* Open the GIF file for writing */
	ft = EGifOpenFileName(filename, 0);
	if(ft == NULL)
	{
	    FreeMapObject(colormap);
	    free(cidx);
	    imgio_last_write_error = "Unable to open the file for writing";
	    return(-1);
	}

	status = 0;

	/* Iterate through each frame */
	for(frame_num = 0; frame_num < nframes; frame_num++)
	{
	    if(*user_aborted)
	    {
		status = -4;
		break;
	    }

	    rgba_cur = rgba[frame_num];
	    if(rgba_cur == NULL)
		continue;

	    /* Get the delay for this frame */
	    delay = (delay_list != NULL) ? delay_list[frame_num] : 0l;

	    /* Copy/convert the RGBA image data to the GIF color
	     * index image data in black & white
	     */
	    if(transparency && (trans_color_num > -1))
	    {
	        GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr +
				    (width * height);
		const u_int8_t *rgba_ptr = rgba_cur;
		while(cidx_ptr < cidx_end)
		{
		    /* Solid? */
		    if(rgba_ptr[3] >= 0x80)
			*cidx_ptr++ = (GifByteType)(
	(((rgba_ptr[0] + rgba_ptr[1] + rgba_ptr[2]) / 3) >= 0x80) ?
			    1 : 0
			);
		    else
		        *cidx_ptr++ = (GifByteType)trans_color_num;
		    rgba_ptr += 4;
	        }
	    }
	    else
	    {
		GifByteType	*cidx_ptr = cidx,
				*cidx_end = cidx_ptr +
				    (width * height);
		const u_int8_t *rgba_ptr = rgba_cur;
	        while(cidx_ptr < cidx_end)
	        {
		    *cidx_ptr++ = (GifByteType)(
	(((rgba_ptr[0] + rgba_ptr[1] + rgba_ptr[2]) / 3) > 0x80) ?
			1 : 0
		    );
		    rgba_ptr += 4;
	        }
	    }

	    /* Write the GIF screen descriptor for the first frame */
	    if(frame_num == 0)
	    {
		if(EGifPutScreenDesc(
		    ft,
		    width, height,	/* Screen width & height */
		    color_resolution,	/* Color resolution */
		    bg_color_num,	/* Background color number */
		    colormap		/* Screen colormap */
		) != GIF_OK)
		{
		    status = -1;
		    break;
		}

		/* If there is more than 1 frame and looping is
		 * specified then write the looping block
		 */
		if((nframes > 1) && looping)
		{
		    if(ImgWriteGIFFileLoopingBlock(ft))
		    {
			status = -1;
			break;
		    }
		}
	    }

	    /* Set the disposal method based on the existance of
	     * transparency
	     */
	    if(transparency)
		disposal_method = GIF_DISPOSAL_METHOD_CLEAR_WITH_BG;
	    else
		disposal_method = GIF_DISPOSAL_METHOD_DO_NOTHING;

	    /* Need to write the graphic control? */
	    if(transparency || user_input || (delay > 0l))
	    {
		u_int16_t delay_hs = (u_int16_t)(delay / 10l);	/* Delay in hundreth of a second */
		const u_int8_t buf[] = {
		    /* Flags */
		    (transparency ? GIF_GRAPHIC_CONTROL_TRANSPARENCY : 0) |
		    (user_input ? GIF_GRAPHIC_CONTROL_USER_INPUT : 0) |
		    (disposal_method << 2),
		    /* Delay in 100th of a second (lower byte) */
		    (u_int8_t)(delay_hs & 0x00ff),
		    /* Delay in 100th of a second (upper byte) */
		    (u_int8_t)((delay_hs & 0xff00) >> 8),
		    /* Transparent color number */
		    (u_int8_t)trans_color_num
		};
		if(EGifPutExtension(
		    ft, GRAPHICS_EXT_FUNC_CODE,
		    sizeof(buf), buf
		) != GIF_OK)
		{
		    status = -1;
		    break;
		}
	    }

	    /* Write the image descriptor */
	    if(EGifPutImageDesc(
		ft,
		0, 0,		/* Top & left */
		width, height,	/* Width & height */
		interlaced,	/* Interlace */
		NULL		/* No local colormap */
	    ) != GIF_OK)
	    {
		status = -1;
		break;
	    }

	    /* Write image data */
	    if(interlaced)
	    {
		const int	po = 0,
				pm = height;
		int i, y, lines_written = 0;

		/* Interlace needs 4 passes on the image */
		for(i = 0; i < 4; i++)
		{
		    for(y = gif_interlace_offset[i];
		        y < height;
		        y += gif_interlace_jumps[i]
		    )
		    {
			if(EGifPutLine(
			    ft,
			    (GifPixelType *)(cidx + (y * cidx_bpl)),
			    cidx_bpl
			) != GIF_OK)
			{
			    status = -1;
			    break;
			}

			lines_written++;

			/* Report progress */
			if((progress_cb != NULL) && ((lines_written % 5) == 0))
			{
			    if(!progress_cb(
				progress_data,
				(frame_num * pm) + (po + lines_written),
				(nframes * pm),
				width, height,
				bpl, bpp,
				rgba_cur
			    ))
			    {
				*user_aborted = 1;
				status = -4;
				break;
			    }
			}
		    }
		    if(status != 0)
			break;
		}
	    }
	    else
	    {
		const int	po = 0,
				pm = height;
		int y;
		GifByteType *cidx_ptr = cidx;
		for(y = 0; y < height; y++)
		{
		    if(EGifPutLine(ft, (GifPixelType *)cidx_ptr, cidx_bpl) != GIF_OK)
		    {
			status = -1;
			break;
		    }

		    cidx_ptr += width;

		    /* Report progress */
		    if((progress_cb != NULL) && ((y % 5) == 0))
		    {
			if(!progress_cb(
			    progress_data,
			    (frame_num * pm) + (po + y),
			    (nframes * pm),
			    width, height,
			    bpl, bpp,
			    rgba_cur
			))
			{
			    *user_aborted = 1;
			    status = -4;
			    break;
			}
		    }
		}
	    }
	    if(status != 0)
		break;
	}

	/* Write the comments */
	if(!STRISEMPTY(comments) &&
	   (status == 0) && !(*user_aborted)
	)
	{
	    if(EGifPutExtension(
		ft, COMMENT_EXT_FUNC_CODE,
		STRLEN(comments), comments
	    ) != GIF_OK)
		status = -1;
	}

	/* Close the GIF file */
	EGifCloseFile(ft);

	/* Set/restore file permissions */
	chmod(filename, mode);

	/* Delete the GIF colormap */
	FreeMapObject(colormap);

	/* Delete the GIF color index image data */
	free(cidx);

	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

/*
 *	Writes the specified image data to a GIF file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value
 *	-3	Systems error
 *	-4	User abort
 */
int ImgWriteGIFFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t **rgba, unsigned long *delay_list,
	int nframes,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	const char *comments,
	int interlaced,		/* 0 or 1 */
	int format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Color */
	int transparency,	/* 0 or 1 */
	int looping,		/* 0 = no looping, 1 = looping */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int status;
	int user_aborted = 0;

	/* Reset global last write error message pointer */
	imgio_last_write_error = NULL;

	if(STRISEMPTY(filename) || (rgba == NULL) || (nframes <= 0) ||
	   (width <= 0) || (height <= 0) ||
	   (bpp != 4)
	)
	{
	    imgio_last_write_error = "Invalid value used to describe image";
	    return(-2);
	}

	/* Automatically calculate bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		rgba[0]
	    ))
		user_aborted = 1;
	}

	/* Save by format */
	switch(format)
	{
	  case 2:
	    status = ImgWriteGIFFileRGBAToColor(
		filename,
		width, height, bpl, bpp,
		rgba, delay_list, nframes,
		bg_color,
		comments,
		interlaced,
		transparency,
		looping,
		progress_cb, client_data, &user_aborted
	    );
	    break;

	  case 1:
	    status = ImgWriteGIFFileRGBAToGreyscale(
		filename,
		width, height, bpl, bpp,
		rgba, delay_list, nframes,
		bg_color,
		comments,
		interlaced,
		transparency,
		looping,
		progress_cb, client_data, &user_aborted
	    );
	    break;

	  case 0:
	    status = ImgWriteGIFFileRGBAToBW(
		filename,
		width, height, bpl, bpp,
		rgba, delay_list, nframes,
		bg_color,
		comments,
		interlaced,
		transparency,
		looping,
		progress_cb, client_data, &user_aborted
	    );
	    break;

	  default:
	    imgio_last_write_error = "Unsupported GIF color format";
	    status = -2;
	    break;
	}

	/* Report final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		rgba[nframes - 1]
	    ))
		user_aborted = 1;
	}

	if(user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

#endif	/* HAVE_LIBGIF */
