#ifdef HAVE_LIBJPEG
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <setjmp.h>
#include <jpeglib.h>
#include "../include/fio.h"
#include "imgio.h"

#if !defined(_WIN32)
#warning It is safe to ignore setjmp() and longjmp() warnings for this module
#endif


/* Last error message pointer */
extern const char *imgio_last_load_error;
extern const char *imgio_last_write_error;

/*
 *	IO Context:
 */
typedef struct _io_context_struct	io_context_struct;
struct _io_context_struct {

	/* Public (order is important) */
	struct jpeg_error_mgr	pub;

	jmp_buf		setjmp_buffer;	/* for return to caller */

	/* Private */
	/* Image Info & Data */
	int		width, height,
			bpp, bpl;
	u_int8_t	*rgba;		/* Allocated when reading,
					 * shared when writing (do not
					 * delete shared)
					 */
	FILE		*fp;
	int		quality;	/* 0 - 100 (used for writing) */
};
#define IO_CONTEXT(p)	((io_context_struct *)(p))


/* JPEG Library Version */
void ImgJPEGVersion(int *major, int *minor, int *release);


/* Callbacks */
static void ImgJPEGReadErrorCB(j_common_ptr jinfo);
static void ImgJPEGWriteErrorCB(j_common_ptr jinfo);


/* JPEG Reading */
int ImgJPEGReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,     /* 4 bytes in RGBA format, will be modified */
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);


/* JPEG Writing */
int ImgWriteJPEGFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int quality,		/* 0 to 100 */
	int format,             /* 0 = Greyscale
				 * 1 = Color */         
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the JPEG library's version.
 */
void ImgJPEGVersion(int *major, int *minor, int *release) 
{
	if(major != NULL)
	    *major = JPEG_LIB_VERSION / 10;
	if(minor != NULL)
	    *minor = JPEG_LIB_VERSION % 10;
	if(release != NULL)
	    *release = 0;
}


/*
 *	JPEG read error callback.
 */
static void ImgJPEGReadErrorCB(j_common_ptr jinfo)
{
	io_context_struct *ctx = (jinfo != NULL) ?
	    IO_CONTEXT(jinfo->err) : NULL;
	if(ctx == NULL)
	    return;

#if 0
	free(ctx->last_error_mesg);
	ctx->last_error_mesg = (char *)malloc(JMSG_LENGTH_MAX);
	(*jinfo->err->format_message)(jinfo, last_error_mesg);
#endif

	imgio_last_load_error =
"JPEG library encountered an error while reading the image";

	/* Return control to the setjmp point */
	longjmp(ctx->setjmp_buffer, 1);
}

/*
 *	JPEG write error callback.
 */
static void ImgJPEGWriteErrorCB(j_common_ptr jinfo)
{
	io_context_struct *ctx = (jinfo != NULL) ?
	    IO_CONTEXT(jinfo->err) : NULL;
	if(ctx == NULL)
	    return;

	imgio_last_write_error = 
"JPEG library encountered an error while writing the image";

	/* Return control to the setjmp point */ 
	longjmp(ctx->setjmp_buffer, 1);
}


/*
 *	JPEG library read front end for RGBA (4 bpp) image data.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error               
 *	-2	Bad value (invalid format/not a jpeg file)
 *	-3	Systems error
 *	-4	User abort
 */
int ImgJPEGReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	j_decompress_ptr jdecomp = (j_decompress_ptr)(
	    calloc(1, sizeof(struct jpeg_decompress_struct))
	);
	io_context_struct *ctx = IO_CONTEXT(
	    calloc(1, sizeof(io_context_struct))
	);
	JSAMPARRAY jpeg_row_buf;
	int jpeg_samples_per_row, jpeg_compoents_per_pixel;
	J_COLOR_SPACE jpeg_color_space;

#define FREE_ALL	{		\
 /* Delete JPEG Decompression Object */	\
 if(jdecomp != NULL) {			\
  jpeg_destroy_decompress(jdecomp);	\
  free(jdecomp);			\
  jdecomp = NULL;			\
 }					\
					\
 /* Delete IO Context */		\
 if(ctx != NULL) {			\
  if(ctx->fp != NULL)			\
   FClose(ctx->fp);			\
  free(ctx->rgba);			\
  free(ctx);				\
  ctx = NULL;				\
 }					\
}

	/* Reset last load error */
	imgio_last_load_error = NULL;

	if((jdecomp == NULL) || (ctx == NULL))
	{
	    FREE_ALL
	    imgio_last_load_error = "Memory allocation error";
	    return(-3);
	}

	/* Skip if user aborted */
	if(*user_aborted)
	{
	    FREE_ALL
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}

	/* Set up JPEG error callbacks */
	jdecomp->err = jpeg_std_error(&ctx->pub);
	ctx->pub.error_exit = ImgJPEGReadErrorCB;
	ctx->width = 0;
	ctx->height = 0;
	ctx->bpp = 0;
	ctx->bpl = 0;
	ctx->rgba = NULL;
	ctx->fp = NULL;
	ctx->quality = 0;	/* Ignored when reading */

	/* Establish the setjmp return context for the error callback */
	if(setjmp(ctx->setjmp_buffer))
	{
	    FREE_ALL
	    return(-1);
	}

	/* Open JPEG file for reading */
	ctx->fp = FOpen(filename, "rb");
	if(ctx->fp == NULL)
	{
	    FREE_ALL
	    imgio_last_load_error = "Unable to open the file for reading";
	    return(-1);
	}

	/* Initialize the JPEG decompression object */
	jpeg_create_decompress(jdecomp);

	/* Set decompression input as the file */
	jpeg_stdio_src(jdecomp, ctx->fp);

	/* Read the JPEG header */
	jpeg_read_header(jdecomp, TRUE);


	/* Modify output values */

	/* Check the color space (JPEG image data format) specified by
	 * the JPEG header, the output color space must be RGB or
	 * Greyscale
	 */
	switch(jdecomp->jpeg_color_space)
	{
	  case JCS_UNKNOWN:
	  case JCS_RGB:
	  case JCS_YCbCr:	/* YUV */
	  case JCS_CMYK:
	  case JCS_YCCK:
	    jdecomp->out_color_space = JCS_RGB;
	    break;
	  case JCS_GRAYSCALE:
	    jdecomp->out_color_space = JCS_GRAYSCALE;
	    break;
	}

	/* Add other output options here */


	/* With the output values set we can now start decompressing
	 * the JPEG data
	 */
	jpeg_start_decompress(jdecomp);

	/* Calculate the number of JSAMPLEs per row in the output
	 * buffer
	 */
	jpeg_color_space = jdecomp->jpeg_color_space;
	jpeg_compoents_per_pixel = jdecomp->output_components;
	jpeg_samples_per_row = jdecomp->output_width * jpeg_compoents_per_pixel;

	/* Create a one-row-high sample array that will go away when
	 * done with image
	 */
	jpeg_row_buf = (*jdecomp->mem->alloc_sarray)(
	    (j_common_ptr)jdecomp, JPOOL_IMAGE, jpeg_samples_per_row, 1
	);

	/* Create target RGBA image */
	ctx->width = (int)jdecomp->output_width;
	ctx->height = (int)jdecomp->output_height;
	ctx->bpp = 4;
	ctx->bpl = ctx->width * ctx->bpp;
	ctx->rgba = (u_int8_t *)realloc(
	    ctx->rgba, ctx->bpl * ctx->height
	);
	if(ctx->rgba == NULL)
	{
	    FREE_ALL
	    imgio_last_load_error = "Memory allocation error";
	    return(-4);
	}

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, ctx->height,
		ctx->width, ctx->height,
		ctx->bpl, ctx->bpp,
		ctx->rgba
	    ))
		*user_aborted = 1;
	}

	/* Read JPEG image */
	if(!(*user_aborted))
	{
	    int y;
	    JSAMPROW src_ptr;
	    u_int8_t *rgba_ptr, *rgba_end;

	    /* Begin reading each line from the JPEG decompressor
	     *
	     * Use the JPEG library's state variable
	     * jdecomp->output_scanline as the row counter
	     */
	    while(jdecomp->output_scanline < jdecomp->output_height)
	    {
		y = jdecomp->output_scanline;

		/* Read next line */
		jpeg_read_scanlines(jdecomp, jpeg_row_buf, 1);

		/* Copy the line to the target RGBA buffer */
		rgba_ptr = ctx->rgba + (y * ctx->bpl);
		rgba_end = rgba_ptr + (ctx->width * ctx->bpp);
		switch(jpeg_compoents_per_pixel)
		{
		  case 4:
		    src_ptr = jpeg_row_buf[0];
		    while(rgba_ptr < rgba_end)
		    {
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
		    }
		    break;
		  case 3:
		    src_ptr = jpeg_row_buf[0];
		    while(rgba_ptr < rgba_end)
		    {
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = 0xff;
		    }
		    break;
		  case 1:
		    src_ptr = jpeg_row_buf[0];
		    while(rgba_ptr < rgba_end)
		    {
			*rgba_ptr++ = *src_ptr;
			*rgba_ptr++ = *src_ptr;
			*rgba_ptr++ = *src_ptr++;
			*rgba_ptr++ = 0xff;
		    }
		    break;
		}

		/* Report progress */
		if((progress_cb != NULL) && ((y % 5) == 0))
		{
		    if(!progress_cb(
			client_data,
			y, ctx->height,
		        ctx->width, ctx->height,
	                ctx->bpl, ctx->bpp,
			ctx->rgba
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}
 	    }
	}

	/* End decompression */
	jpeg_finish_decompress(jdecomp);

	/* Update returns */
	if(width_rtn != NULL)
	    *width_rtn = ctx->width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->height;
	if(bpp_rtn != NULL)
	    *bpp_rtn = ctx->bpp;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->bpl;
	if(rgba_rtn != NULL)
	{
	    *rgba_rtn = ctx->rgba;	/* Transferer pointer */
	    ctx->rgba = NULL;
	}

	/* Report final progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		ctx->height, ctx->height,
		ctx->width, ctx->height,
		ctx->bpl, ctx->bpp,
		ctx->rgba
	    ))
		*user_aborted = 1;
	}

	FREE_ALL

	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(0);
#undef FREE_ALL
}


/*
 *	Writes the JPEG image to file using libjpeg.
 *
 *	If progress_cb is not NULL then it will be called during
 *	loading, if it returns 0 then loading will be aborted.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value
 *	-3	Systems error
 *	-4	User abort
 */
int ImgWriteJPEGFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int quality,		/* 0 to 100 */
	int format,		/* 0 = Greyscale
				 * 1 = Color */ 
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int *user_aborted = (int *)malloc(sizeof(int));

	j_compress_ptr jcomp = (j_compress_ptr)(
	    calloc(1, sizeof(struct jpeg_compress_struct))
	);
	io_context_struct *ctx = IO_CONTEXT(
	    calloc(1, sizeof(io_context_struct))
	);

#define FREE_ALL	{		\
 /* Delete JPEG Compression Object */	\
 if(jcomp != NULL) {			\
  jpeg_destroy_compress(jcomp);		\
  free(jcomp);				\
  jcomp = NULL;				\
 }					\
					\
 /* Delete IO Context */		\
 if(ctx != NULL) {			\
  if(ctx->fp != NULL)			\
   FClose(ctx->fp);			\
/*ctx->rgba is shared when writing */	\
  free(ctx);				\
  ctx = NULL;				\
 }					\
}

	/* Reset global last write error message pointer */
	imgio_last_write_error = NULL;
	*user_aborted = 0;

	if((jcomp == NULL) || (ctx == NULL))
	{
	    FREE_ALL
	    imgio_last_write_error = "Memory allocation error";
	    return(-3);
	}

	if(STRISEMPTY(filename) || (rgba == NULL) ||
	   (width <= 0) || (height <= 0) ||
	   (bpp != 4)
	)
	{
	    FREE_ALL
	    imgio_last_write_error = "Invalid value used to describe image";
	    return(-1);
	}

	/* Automatically calculate bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		rgba
	    ))   
		*user_aborted = 1;
	}

	/* Set up JPEG error callbacks */
	jcomp->err = jpeg_std_error(&ctx->pub);
	ctx->pub.error_exit = ImgJPEGWriteErrorCB;
	ctx->width = width;
	ctx->height = height;
	ctx->bpp = bpp;
	ctx->bpl = bpl;
	ctx->rgba = rgba;	/* Image data is shared when writing */
	ctx->fp = NULL;
	ctx->quality = CLIP(quality, 0, 100);

	/* Establish the setjmp return context for the error callback */
	if(setjmp(ctx->setjmp_buffer))
	{
	    FREE_ALL
	    return(-1);
	} 

	/* Open JPEG file for writing */ 
	ctx->fp = FOpen(filename, "wb");
	if(ctx->fp == NULL)
	{
	    FREE_ALL
	    imgio_last_write_error = "Unable to open the file for writing";
	    return(-1);
	}

	/* Initialize the JPEG compression object */
	jpeg_create_compress(jcomp);

	/* Set compression input as the file */
	jpeg_stdio_dest(jcomp, ctx->fp);

	/* Set mandatory image description values */
	jcomp->image_width = width;
	jcomp->image_height = height;
	switch(format)
	{
	  case 1:	/* Color */
	    jcomp->input_components = 3;
	    jcomp->in_color_space = JCS_RGB;
	    break;
	  default:	/* Greyscale */
	    jcomp->input_components = 1;
	    jcomp->in_color_space = JCS_GRAYSCALE;
	    break;
	}

	/* Set compression parameters to default values */
	jpeg_set_defaults(jcomp);

	/* Begin setting specific compression parameters */

	/* Quality */
	jpeg_set_quality(
	    jcomp,
	    ctx->quality,	/* 1 to 100 */
	    TRUE		/* limit to baseline JPEG values */
	);


	/* Start compressing the JPEG data */
	jpeg_start_compress(jcomp, TRUE);

	/* Report progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		0, ctx->height,
		ctx->width, ctx->height,
		ctx->bpl, ctx->bpp,
		ctx->rgba
	    ))
		*user_aborted = 1;
	}

	/* Write JPEG image */
	if(!(*user_aborted))
	{
	    /* Allocate JSAMPLEs row buffer for use in writing the JPEG
	     * image
	     *
	     * Each line from the source RGBA image data will be
	     * copy/converted to this JSAMPLEs buffer and written to the
	     * JPEG image using jpeg_write_scanlines()
	     */
	    int samples_per_pixel = jcomp->input_components;
	    JSAMPROW row_ptr = (JSAMPROW)malloc(ctx->width * samples_per_pixel);
	    JSAMPARRAY buf = (JSAMPARRAY)malloc(1 * sizeof(JSAMPROW));
	    if((buf != NULL) && (row_ptr != NULL))
	    {
		u_int8_t	*src_line = ctx->rgba,
				*src_line_end = src_line + (ctx->bpl * ctx->height),
				*src_ptr, *src_end;
		JSAMPROW	tar_ptr;

		buf[0] = row_ptr;

		/* Handle by format */
		switch(format)
		{
		  case 1:	/* Color */
		    /* Iterate through each line */
		    while(src_line < src_line_end)
		    {
			/* Copy source RGBA line to target RGB line */
			src_ptr = src_line;
			src_end = src_ptr + (ctx->width * ctx->bpp);
			tar_ptr = row_ptr;
			while(src_ptr < src_end)
			{
			    *tar_ptr++ = (JSAMPLE)*src_ptr++;
			    *tar_ptr++ = (JSAMPLE)*src_ptr++;
			    *tar_ptr++ = (JSAMPLE)*src_ptr++;
			    src_ptr++;
			}

			/* Write this line */
			jpeg_write_scanlines(jcomp, buf, 1);

			/* Report progress */
			if(progress_cb != NULL)
			{
			    const int i = (int)(src_line - ctx->rgba);
			    if((i % 5) == 0)
			    {
			        if(!progress_cb(
				    client_data,
				    i, (int)(src_line_end - ctx->rgba),
				    ctx->width, ctx->height,
				    ctx->bpl, ctx->bpp,
				    ctx->rgba
			        ))
			        {
				    *user_aborted = 1;
				    break;
				}
			    }
			}

			/* Seek to next line */
			src_line += ctx->bpl;
		    }
		    break;

		  default:	/* Greyscale */
		    /* Iterate through each line */
		    while(src_line < src_line_end)
		    {
			/* Copy source RGBA line to target Greyscale line */
			src_ptr = src_line;
			src_end = src_ptr + (ctx->width * ctx->bpp);
			tar_ptr = row_ptr;
			while(src_ptr < src_end)
			{
			    *tar_ptr++ = (JSAMPLE)(
		((int)src_ptr[0] + (int)src_ptr[1] + (int)src_ptr[2]) / 3
			    );
			    src_ptr += ctx->bpp;
			}

			/* Write this line */
			jpeg_write_scanlines(jcomp, buf, 1);

			/* Report progress */
			if(progress_cb != NULL)
			{
			    const int i = (int)(src_line - ctx->rgba);
			    if((i % 5) == 0)
			    {
			        if(!progress_cb(
				    client_data,
				    i, (int)(src_line_end - ctx->rgba),
				    ctx->width, ctx->height,
				    ctx->bpl, ctx->bpp,
				    ctx->rgba
			        ))
			        {
				    *user_aborted = 1;
				    break;
				}
			    }
			}

			/* Seek to next line */
			src_line += ctx->bpl;
		    }
		    break;
		}
	    }

	    free(row_ptr);
	    free(buf);
	}

	/* Finish compression */
	jpeg_finish_compress(jcomp);

	/* Report final progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		ctx->height, ctx->height,
		ctx->width, ctx->height,
		ctx->bpl, ctx->bpp,
		ctx->rgba
	    ))
		*user_aborted = 1;
	}

	FREE_ALL

	if(*user_aborted)
	{
	    free(user_aborted);
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	{
	    free(user_aborted);
	    return(0);
	}
#undef FREE_ALL
}

#endif
