#ifdef HAVE_LIBPNG
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
/* #include <setjmp.h> png.h does not like this */
#include <png.h>
#if defined(_WIN32)
# include "../include/string.h"
#endif
#include "imgio.h"

#if !defined(_WIN32)
#warning It is safe to ignore setjmp() and longjmp() warnings for this module
#endif


/* Last error message pointer */
extern const char	*imgio_last_load_error,
			*imgio_last_write_error;


typedef struct _ImgPNGReadData		ImgPNGReadData;
#define IMG_PNG_READ_DATA(p)		((ImgPNGReadData *)(p))


/*
 *	Local PNG message buffer:
 *
 *	For recording png read or write error messages.
 */
static char		img_png_local_msg[1024];


/*	png_jmpbuf() macro used in error handling
 *
 *	This macro became available in libpng version 1.0.6
 *
 *	Notes: If you want to be able to run your code with older
 *	versions of libpng, then you must define the macro yourself (but
 *	only if it is not already defined by libpng)
 */
#ifndef png_jmpbuf
# define png_jmpbuf(png_ptr)	((png_ptr)->jmpbuf)
#endif


/* PNG Library Version */
void ImgPNGVersion(int *major, int *minor, int *release);


/* PNG Reading */
static void ImgPNGReadErrorCB(png_structp png_ptr, png_const_charp mesg);
static void ImgPNGReadWarningCB(png_structp png_ptr, png_const_charp mesg);
static void ImgPNGReadConvertBuffer(
	int png_color_type,
	int width,			/* For target and also source */
	int height,			/* Total rows in tar_data */
	u_int8_t *tar_data,
	int tar_bpp, int tar_bpl,
	const u_int8_t *src_data,
	int src_bpp, int src_bpl,
	int src_row_num,		/* Source row number, must be < height */
	u_int8_t *trans_color,		/* 4 byte transparent color */
	u_int8_t def_alpha_value
);
static void ImgPNGReadInfoCB(png_structp png_ptr, png_infop png_info_ptr);
static void ImgPNGReadRowCB(
	png_structp png_ptr, png_bytep new_row,
	png_uint_32 row_num, int pass
);
static void ImgPNGReadEndCB(png_structp png_ptr, png_infop png_info_ptr);

int ImgPNGReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);


/* PNG Writing */
static int ImgPNGWriteRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int x, int y,
	int base_width, int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	int compression_level,	/* 0 to 9 */
	int interlaced,		/* 0 or 1 */
	int format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB
				 * 4 = RGBA */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
int ImgWritePNGFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int x, int y,
	int base_width, int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	int compression_level,	/* 0 to 9 */
	int interlaced,		/* 0 or 1 */
	int format,		/* 0 = B&W  
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB  
				 * 4 = RGBA */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
);


/*
 *	Libpng IO Context:
 */
struct _ImgPNGReadData {

	char		*filename;
	FILE		*fp;

	png_structp	png_ptr;
	png_infop	png_info_ptr;
	int		png_bpp, png_bpl,	/* Geometry of PNG data */
			png_width, png_height,
			png_x, png_y;
	int		png_color_type;		/* One of PNG_COLOR_TYPE_* */
	int		png_last_row;
	png_bytep	png_last_row_ptr;	/* Pointer to last row, used
						 * for interlaced loading */
	int		png_encountered_rows;	/* Highest number of rows
						 * encountered, think of it
						 * as the *current* allocated
						 * height */

	/* PNG read buffer, contains data from the png image in
	 * multiple rows (total number of rows indicated by png_height) 
	 * and each row is allocated to the size of
	 * MAX(png_width * 4, png_bpl) (whichever is bigger)
	 *
	 * This pointer array and each row must be deleted after use
	 */
	png_bytep	*png_data;

	/* Our allocated image buffer, size is calculated with
	 * data_bpl * png_encountered_rows.
	 */
	u_int8_t	*data;
	int		data_bpp, data_bpl;	/* Geometry of member data,
						 * the width and height matches
						 * png_width and png_height */
	int		interlace_type;		/* One of PNG_INTERLACE_* */
	int		interlace_pass;		/* Interlace pass number */
	int		interlace_passes;	/* Total interlace passes */

	/* Transparent color (4 bytes RGBA, only first 3 bytes are used
	 * though)
	 *
	 * If it is NULL then it hints that there is no transparency
	 *
	 * This needs to be deleted if it is not NULL
	 */
	u_int8_t	*trans_color;

	u_int8_t	def_alpha_value;
	u_int8_t	bg_color[4];		/* Background color in RGBA format */

	int		operation_completed;

	/* Progress callback */
	void		*client_data;
	int		(*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	);

};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))     

#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : 1)


/*
 *	Gets the PNG library's version.
 */
void ImgPNGVersion(int *major, int *minor, int *release)
{
	if(major != NULL)   
	    *major = PNG_LIBPNG_VER_MAJOR;
	if(minor != NULL)
	    *minor = PNG_LIBPNG_VER_MINOR;
	if(release != NULL)
	    *release = PNG_LIBPNG_VER_RELEASE;
}


/*
 *	PNG read error callback.
 *
 *	Records the error message on the local png message buffer.
 */
static void ImgPNGReadErrorCB(png_structp png_ptr, png_const_charp mesg)
{
	if(mesg == NULL)
	    return;

	/* Record the message to our local message buffer */
	strncpy(
	    img_png_local_msg, (const char *)mesg,
	    sizeof(img_png_local_msg)
	);
	img_png_local_msg[sizeof(img_png_local_msg) - 1] = '\0';

	/* Set the last read error message to this one */
	imgio_last_load_error = img_png_local_msg;
}

/*
 *	PNG read warning callback.
 *
 *	Records the warning message on the local png message buffer.
 */
static void ImgPNGReadWarningCB(png_structp png_ptr, png_const_charp mesg)
{
/* IGNORE */
	return;
}

/*
 *	Copies the PNG row data specified by src_data into a row inside
 *	the buffer tar_data.
 *
 *	Format of source buffer must be in libpng's format.
 *
 *	Format of target buffer can be either RGBA or RGB.
 */
static void ImgPNGReadConvertBuffer(
	int png_color_type,
	int width,			/* For target and also source */
	int height,			/* Total rows in tar_data */
	u_int8_t *tar_data,
	int tar_bpp, int tar_bpl,
	const u_int8_t *src_data,
	int src_bpp, int src_bpl,
	int src_row_num,		/* Source row number, must be < height */
	u_int8_t *trans_color,		/* 4 byte transparent color */
	u_int8_t def_alpha_value
)
{
	int bc, bc_min;
	int x, y = src_row_num;
	u_int8_t *tar_ptr;
	const u_int8_t *src_ptr;

	/* Allocation positive? */
	if((tar_bpl <= 0) || (tar_bpp <= 0) || (height <= 0) ||
	   (src_bpl <= 0) || (src_bpp <= 0)
	)
	    return;

	/* Source row number out of bounds? */
	if((y < 0) || (y >= height))
	    return;

	/* Buffers allocated? */
	if((tar_data == NULL) || (src_data == NULL))
	    return;

	/* Handle by color type */
	switch(png_color_type)
	{
	  case PNG_COLOR_TYPE_GRAY:
	    /* Assume 1 byte per pixel */
	    src_bpp = 1;
	    if(1)
	    {
		u_int8_t src_byte;

		/* Get smaller of target bytes per pixel or 3 bytes
		 * per pixel.
		 */
		bc_min = MIN(tar_bpp, 3);

		/* Begin copying line */
		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_data + (y * tar_bpl) + (x * tar_bpp);
		    src_ptr = src_data + (x * src_bpp);

		    /* Copy source pixel bytes to the target pixel */
		    src_byte = *src_ptr;
		    for(bc = 0; bc < bc_min; bc++)
			tar_ptr[bc] = src_byte;

		    /* Set rest of target pixel bytes to the default alpha
		     * value
		     */
		    for(; bc < tar_bpp; bc++)
			tar_ptr[bc] = def_alpha_value;
		}
	    }
	    break;

	  case PNG_COLOR_TYPE_PALETTE:
	    /* Note that we expanded palette color to RGB triplets
	     * so for color type PNG_COLOR_TYPE_PALETTE, the src_bpp
	     * is actually 3 bpp (or 4 bpp if trans_color is not NULL,
	     * suggesting that there is transparency and an alpha channel)
	     */
	    if(trans_color != NULL)
		src_bpp = 4;
	    else
		src_bpp = 3;
	    if(1)
	    {
		/* Get smaller of the two pixel sizes */
		bc_min = MIN(tar_bpp, src_bpp);

		/* Begin copying line */
		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_data + (y * tar_bpl) + (x * tar_bpp);
		    src_ptr = src_data + (x * src_bpp);

		    /* Copy source pixel bytes to the target pixel */
		    for(bc = 0; bc < bc_min; bc++)
			tar_ptr[bc] = src_ptr[bc];

		    /* Set rest of target pixel bytes to def alpha value */
		    for(; bc < tar_bpp; bc++)
			tar_ptr[bc] = def_alpha_value;
		}
	    }
	    break;

	  case PNG_COLOR_TYPE_RGB:
	    if(src_bpp == 3)
	    {
		/* Get smaller of the two pixel sizes */
		bc_min = MIN(tar_bpp, src_bpp);

		/* Begin copying line */
		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_data + (y * tar_bpl) + (x * tar_bpp);
		    src_ptr = src_data + (x * src_bpp);

		    /* Copy source pixel bytes to the target pixel */
		    for(bc = 0; bc < bc_min; bc++)
			tar_ptr[bc] = src_ptr[bc];

		    /* Set rest of target pixel bytes to def alpha value */
		    for(; bc < tar_bpp; bc++)
			tar_ptr[bc] = def_alpha_value;
		}
	    }
	    break;

#ifdef PNG_COLOR_TYPE_RGBA
	  case PNG_COLOR_TYPE_RGBA:
#else
	  case PNG_COLOR_TYPE_RGB_ALPHA:
#endif
	    if(src_bpp == 4)
	    {
		/* Get smaller of the two pixel sizes */
		bc_min = MIN(tar_bpp, src_bpp);

		/* Begin copying line */
		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_data + (y * tar_bpl) + (x * tar_bpp);
		    src_ptr = src_data + (x * src_bpp);

		    /* Copy source pixel bytes to the target pixel */
		    for(bc = 0; bc < bc_min; bc++)
			tar_ptr[bc] = src_ptr[bc];

		    /* Set rest of target pixel bytes to def alpha value */
		    for(; bc < tar_bpp; bc++)
			tar_ptr[bc] = def_alpha_value;
		}
	    }
	    break;

#ifdef PNG_COLOR_TYPE_GA
	  case PNG_COLOR_TYPE_GA:
#else
	  case PNG_COLOR_TYPE_GRAY_ALPHA:
#endif
	    /* Assume 2 bytes per pixel */
	    src_bpp = 2;
	    if(1)
	    {
		u_int8_t src_byte;


		/* Get smaller of target bytes per pixel or 3 bytes
		 * per pixel.
		 */
		bc_min = MIN(tar_bpp, 3);

		/* Begin copying line */
		for(x = 0; x < width; x++)
		{
		    tar_ptr = tar_data + (y * tar_bpl) + (x * tar_bpp);
		    src_ptr = src_data + (x * src_bpp);

		    /* Copy source pixel bytes to the target pixel */
		    src_byte = src_ptr[0];
		    for(bc = 0; bc < bc_min; bc++)
			tar_ptr[bc] = src_byte;

		    /* Set rest of target pixel to source pixel's alpha
		     * value
		     */
		    src_byte = src_ptr[1];
		    for(; bc < tar_bpp; bc++)
			tar_ptr[bc] = src_byte;
		}
	    }
	    break;
	}
}

/*
 *	PNG library read info callback.
 */
static void ImgPNGReadInfoCB(png_structp png_ptr, png_infop png_info_ptr)
{
	int color_type, bit_depth;
	int buf_len;
	ImgPNGReadData *ctx = IMG_PNG_READ_DATA(
	    png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
	    return;

	/* Update PNG read instance pointer on core structure */
	ctx->png_ptr = png_ptr;
	ctx->png_info_ptr = png_info_ptr;

	/* Get color type */
	ctx->png_color_type = color_type = ((png_info_ptr != NULL) ?
	    png_info_ptr->color_type : PNG_COLOR_TYPE_GRAY
	);
	bit_depth = (png_info_ptr != NULL) ? png_info_ptr->bit_depth : 1;


	/* Insert transformations here */

	/* Swap bytes of 16 bit files to least significant byte first */
/*	png_set_swap(png_ptr); */

	/* Reduce 16 bits data down to 8 bits data */
	if(bit_depth == 16)
	    png_set_strip_16(png_ptr);

	/* swap the RGBA or GA data to ARGB or AG (or BGRA to ABGR) */
/*	png_set_swap_alpha(png_ptr); */

	/* Flip the RGB pixels to BGR (or RGBA to BGRA) */
/*
	if(color_type & PNG_COLOR_MASK_COLOR)
	    png_set_bgr(png_ptr);
 */

	/* Extract multiple pixels with bit depths of 1, 2, and 4 from
	 * a single byte into separate bytes (useful for paletted and
	 * grayscale images)
	 */
/*	png_set_packing(png_ptr); */

	/* Change the order of packed pixels to least significant bit
	 * first (not useful if you are using png_set_packing)
	 */
/*	png_set_packswap(png_ptr); */



	/* Expand paletted colors into true RGB triplets, this will
	 * make the actual bpp value to be 3 instead of the reported 1.
	 */
	if(color_type == PNG_COLOR_TYPE_PALETTE)
	    png_set_expand(png_ptr);
	/* Expand grayscale images to the full 8 bits from 1, 2, or 4
	 * bits per pixel.
	 */
	if((color_type == PNG_COLOR_TYPE_GRAY) && (bit_depth < 8))
	    png_set_expand(png_ptr);

	/* Expand paletted or RGB images with transparency to full alpha
	 * channels so the PNG data will be available as RGBA
	 */
	if(png_get_valid(png_ptr, png_info_ptr, PNG_INFO_tRNS))
	{
	    /* This will give us an alpha channel */
	    png_set_expand(png_ptr);

	    /* Set transparent color */
	    ctx->trans_color = (u_int8_t *)realloc(
		ctx->trans_color,
		4 * sizeof(u_int8_t)
	    );
	    if(ctx->trans_color != NULL)
	    {
#if defined(PNG_tRNS_SUPPORTED) 
		int ncolors;
		png_bytep trans;
		png_color_16p trans_values;
		u_int8_t *trans_color = ctx->trans_color;

		png_get_tRNS(
		    png_ptr, png_info_ptr,
		    &trans, &ncolors, &trans_values
		);

		/* Palette image? */
		if(color_type == PNG_COLOR_TYPE_PALETTE)
		{
#if 0
		    int i;
		    png_color *palette;
/* TODO apply transparent colors to colormap */
		    png_get_PLTE(png_ptr, info_ptr, &palette, &ncolors);

		    for(i = 0; i < ncolors; i++)
		    {
			colormap->c[i].a = 0xff - (u_int8_t)trans[i];
			colormap->c[i].r = (u_int8_t)palette[i].red;
			colormap->c[i].g = (u_int8_t)palette[i].green;
			colormap->c[i].b = (u_int8_t)palette[i].blue;
		    }
#endif
		}
		else if(trans_values != NULL)
		{
		    /* Get single transparent color */
		    trans_color[0] = (u_int8_t)(trans_values->red >> 8);
		    trans_color[1] = (u_int8_t)(trans_values->green >> 8);
		    trans_color[2] = (u_int8_t)(trans_values->blue >> 8);
		    trans_color[3] = ctx->def_alpha_value;
		}
#endif
	    }
	}

#if defined(PNG_bKGD_SUPPORTED)
	/* Get background color */
	if(png_get_valid(png_ptr, png_info_ptr, PNG_INFO_bKGD))
	{
	    png_color_16p pBackground;
	    u_int8_t *bg_color = ctx->bg_color;

	    /* It is not obvious from the libpng documentation, but this
	     * function takes a pointer to a pointer, and it always
	     * returns valid red, green and blue values, regardless of
	     * color_type
	     */
	    png_get_bKGD(png_ptr, png_info_ptr, &pBackground);
	    /* However, it always returns the raw bKGD data, regardless of
	     * any bit-depth transformations, so check depth and adjust if
	     * necessary
	     */
	    if(bit_depth == 16)
	    {
		bg_color[0] = (u_int8_t)(pBackground->red >> 8);
		bg_color[1] = (u_int8_t)(pBackground->green >> 8);
		bg_color[2] = (u_int8_t)(pBackground->blue >> 8);
		bg_color[3] = ctx->def_alpha_value;
	    }
	    else if((color_type == PNG_COLOR_TYPE_GRAY) && (bit_depth < 8))
	    {
		if(bit_depth == 1)
		    bg_color[0] = bg_color[1] = bg_color[2] =
			pBackground->gray ? 255 : 0;
		else if(bit_depth == 2)
		    bg_color[0] = bg_color[1] = bg_color[2] =
			(255/3) * pBackground->gray;
		else /* bit_depth == 4 */
		    bg_color[0] = bg_color[1] = bg_color[2] =
			(255/15) * pBackground->gray;
		bg_color[3] = ctx->def_alpha_value;
	    }
	    else
	    {
		bg_color[0] = (u_int8_t)pBackground->red;
		bg_color[1] = (u_int8_t)pBackground->green;
		bg_color[2] = (u_int8_t)pBackground->blue;
		bg_color[3] = ctx->def_alpha_value;
	    }
	}
#endif

	/* Set interlace handling */
	ctx->interlace_passes = png_set_interlace_handling(png_ptr);

	/* Get interlace type */
	ctx->interlace_type = png_get_interlace_type(png_ptr, png_info_ptr);


	/* Get size of image and byte units */
	ctx->png_width = (int)png_get_image_width(
	    png_ptr, png_info_ptr
	);
	ctx->png_height = (int)png_get_image_height(
	    png_ptr, png_info_ptr
	);
	/* Update data_bpl when the png_width is obtained, note that
	 * that data_bpp is set at the creation of this context
	 */
	ctx->data_bpl = ctx->data_bpp * ctx->png_width;
	/* Get bytes per row of one PNG line */
	ctx->png_bpl = png_get_rowbytes(
	    png_ptr, png_info_ptr
	);
	/* Get bytes per pixel of one PNG pixel */
	if(color_type & PNG_COLOR_MASK_COLOR)
	{
	    if(color_type & PNG_COLOR_MASK_ALPHA)
		ctx->png_bpp = 4;
	    else
		ctx->png_bpp = 3;
	}
	else
	{
	    if(color_type & PNG_COLOR_MASK_ALPHA)
		ctx->png_bpp = 2;
	    else
		ctx->png_bpp = 1;
	}

#if defined(PNG_oFFs_SUPPORTED)
	/* Get offset */
	if(png_get_valid(png_ptr, png_info_ptr, PNG_INFO_oFFs))
	{
	    png_int_32 x, y;
	    int unit_type;
	    png_get_oFFs(png_ptr, png_info_ptr, &x, &y, &unit_type);
	    switch(unit_type)
	    {
	      case PNG_OFFSET_PIXEL:
		ctx->png_x = (int)x;
		ctx->png_y = (int)y;
		break;
	      case PNG_OFFSET_MICROMETER:
/* TODO Offset in micrometers (1/10^6 meter) */
		break;
	    }
	}
#endif

	/* Allocate buffer for reading libpng format data in
	 * ImgPNGRowCB(). Note that each row is uninitialized at 
	 * allocation
	 */
	/* If already allocated, then free it first just in case. This
	 * situation should never occure
	 */
	if(ctx->png_data != NULL)
	{
	    int i;
	    for(i = 0; i < ctx->png_height; i++)
		free(ctx->png_data[i]);
	    free(ctx->png_data);
	    ctx->png_data = NULL;
	}
	/* Now allocate read buffer pointer array and each row, each row
	 * is of length 4 bytes per pixel of the width or the length
	 * specified by png_bpl (whichever is bigger)
	 */
	buf_len = MAX(ctx->png_width * 4, ctx->png_bpl) *
	    sizeof(png_byte);
	if((buf_len > 0) && (ctx->png_height > 0))
	{
	    ctx->png_data = (png_bytep *)malloc(
		ctx->png_height * sizeof(png_bytep)
	    );
	    if(ctx->png_data != NULL)
	    {
		int i;
		for(i = 0; i < ctx->png_height; i++)
		    ctx->png_data[i] = (png_bytep)malloc(buf_len);
	    }
	}


	/* Allocate image data buffer for storing the png image data in
	 * our data format and eventually passed to calling function.
	 */
	buf_len = ctx->data_bpl * ctx->png_height;
	if(buf_len > 0)
	    ctx->data = (u_int8_t *)realloc(
		ctx->data, buf_len
	    );


	/* setjmp() doesn't make sense here, because we'd either have to
	 * exit(), longjmp() ourselves, or return control to libpng, which
	 * doesn't want to see us again.  By not doing anything here,
	 * libpng will instead jump to readpng2_decode_data(), which can
	 * return an error value to the main program
	 */

	/* Transformations set, now begin reading by combining
	 * background with palette (this is required).
	 */
	png_start_read_image(png_ptr);
}

/*
 *	PNG library read row callback.
 */
static void ImgPNGReadRowCB(
	png_structp png_ptr, png_bytep new_row,
	png_uint_32 row_num, int pass
)
{
	int png_last_row;
	png_bytep png_last_row_ptr;
	ImgPNGReadData *ctx = IMG_PNG_READ_DATA(
	    png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
	    return;

	/* Update PNG read instance pointer on core structure */
	ctx->png_ptr = png_ptr;

	/* Update interlace pass number */
	ctx->interlace_pass = pass;

	/* Get previous row (might be NULL) */
	png_last_row = ctx->png_last_row;
	png_last_row_ptr = ctx->png_last_row_ptr;


	/* Increment total number of rows encountered? */
	if((int)row_num >= ctx->png_encountered_rows)
	    ctx->png_encountered_rows = MAX((int)row_num + 1, 0);

	/* Combine rows */
	if((ctx->png_data != NULL) && (new_row != NULL))
	{
	    int i = (int)row_num;
	    if((i >= 0) && (i < ctx->png_height))
	    {
		png_bytep old_row = ctx->png_data[i];
		if(old_row != NULL)
		{
		    png_progressive_combine_row(
			png_ptr, old_row, new_row
		    );

		    /* Update buffer's row that corresponds with row_num
		     * with the new_row data.
		     */
		    ImgPNGReadConvertBuffer(
			ctx->png_color_type,
			ctx->png_width,		/* Tar and src width */
			ctx->png_encountered_rows,	/* Tar height */
			ctx->data,
			ctx->data_bpp, ctx->data_bpl,
			(const u_int8_t *)old_row,
			ctx->png_bpp, ctx->png_bpl,
			i,				/* Row number */
			ctx->trans_color,
			ctx->def_alpha_value
		    );
		}
	    }
	}

	/* Update last row information with the current row index
	 * and pointer
	 */
	ctx->png_last_row = (int)row_num;
	ctx->png_last_row_ptr = new_row;
}

/*
 *	PNG library read end callback.
 */
static void ImgPNGReadEndCB(png_structp png_ptr, png_infop png_info_ptr)
{
	ImgPNGReadData *ctx = IMG_PNG_READ_DATA(
	    png_get_progressive_ptr(png_ptr)
	);
	if(ctx == NULL)
	    return;

	/* Update PNG read instance pointer on core structure */
	ctx->png_ptr = png_ptr;
	ctx->png_info_ptr = png_info_ptr;

	/* Mark PNG reading as done */
	ctx->operation_completed = 1;
}

/*
 *	Reads the PNG file.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value or invalid format/not a png file)
 *	-3	Systems error
 *	-4	User abort
 */
int ImgPNGReadRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes in RGBA format, will be modified */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	const int rbuf_len = 256;
	char *rbuf = NULL;
	size_t bytes_read;
	ImgPNGReadData *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 /* Delete the callback data */				\
 if(ctx != NULL) {					\
  /* Delete the PNG read structure and info pair */	\
  if(ctx->png_ptr != NULL) {				\
   png_destroy_read_struct(				\
    &ctx->png_ptr,					\
    (ctx->png_info_ptr != NULL) ?			\
     &ctx->png_info_ptr : NULL,				\
    (png_infopp)NULL					\
   );							\
  }							\
							\
  /* Delete image data in libpng data format */		\
  if(ctx->png_data != NULL) {				\
   const int height = ctx->png_height;			\
   int y;						\
   for(y = 0; y < height; y++)				\
    free(ctx->png_data[y]);				\
   free(ctx->png_data);					\
  }							\
							\
  /* Delete the return RGBA format image data */	\
  free(ctx->data);					\
							\
  /* Delete the transparent color */			\
  free(ctx->trans_color);				\
							\
  /* Close the PNG file */				\
  if(ctx->fp != NULL)					\
   fclose(ctx->fp);					\
							\
  free(ctx->filename);					\
							\
  free(ctx);						\
 }							\
							\
 /* Delete the libpng read buffer */			\
 free(rbuf);						\
							\
 return(_v_);						\
}

	/* Reset global load error string pointer */
	imgio_last_load_error = NULL;

	/* Skip entirly if user aborted */
	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    CLEANUP_RETURN(-4);
	}

	/* Allocate callback data structure for PNG reading */
	ctx = IMG_PNG_READ_DATA(calloc(1, sizeof(ImgPNGReadData)));
	if(ctx == NULL)
	{
	    imgio_last_load_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->filename = STRDUP(filename);
	ctx->fp = NULL;
	ctx->png_ptr = NULL;
	ctx->png_info_ptr = NULL;
	ctx->png_width = 0;		/* Obtained in ImgPNGInfoCB() */
	ctx->png_height = 0;
	ctx->png_bpp = 0;
	ctx->png_bpl = 0;
	ctx->png_x = 0;
	ctx->png_y = 0;
	ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	ctx->png_last_row_ptr = NULL;
	ctx->png_encountered_rows = 0;
	ctx->png_data = NULL;	/* Allocated in ImgPNGInfoCB() */
	ctx->data = NULL;		/* Allocated in ImgPNGInfoCB() */
	ctx->data_bpp = 4;		/* Start off with 4 bytes, RGBA */
	ctx->data_bpl = 0;		/* Calculate later */
	ctx->interlace_type = PNG_INTERLACE_NONE;
	ctx->interlace_pass = 0;
	ctx->interlace_passes = 0;
	ctx->trans_color = NULL;	/* Allocated in ImgPNGInfoCB() */
	ctx->def_alpha_value = def_alpha_value;
	if(bg_color != NULL)
	    memcpy(ctx->bg_color, bg_color, 4 * sizeof(u_int8_t));
	ctx->operation_completed = 0;
	ctx->client_data = client_data;
	ctx->progress_cb = progress_cb;


	/* Open PNG file for reading in binary mode */
	ctx->fp = fopen(ctx->filename, "rb");
	if(ctx->fp == NULL)
	{
	    imgio_last_load_error = "Unable to open the file for reading";
	    CLEANUP_RETURN(-1);
	}

	/* Read first 8 bytes to check for PNG signature to see if this
	 * really is a PNG file
	 */
	if(1)
	{
	    unsigned char sig_buf[8];

	    fread(sig_buf, sizeof(unsigned char), 8, ctx->fp);
	    if(!png_check_sig(sig_buf, 8))
	    {
		imgio_last_load_error = "Data does not appear to be in PNG format";
		CLEANUP_RETURN(-2);
	    }

	    /* Rewind so libpng can get the beginning on formal read */
	    rewind(ctx->fp);
	}

	/* Create instance of PNG read */
	ctx->png_ptr = png_create_read_struct(
	    PNG_LIBPNG_VER_STRING,
	    NULL, ImgPNGReadErrorCB, ImgPNGReadWarningCB
	);
	if(ctx->png_ptr == NULL)
	{
	    imgio_last_load_error = "Unable to create a new PNG read structure";
	    CLEANUP_RETURN(-3);
	}

	/* Create info structure for paring with PNG read instance */
	ctx->png_info_ptr = png_create_info_struct(ctx->png_ptr);
	if(ctx->png_info_ptr == NULL)
	{
	    imgio_last_load_error = "Unable to create a new PNG info structure";
	    CLEANUP_RETURN(-3);
	}

	/* Allocate read buffer */
	rbuf = (char *)realloc(rbuf, rbuf_len * sizeof(char));
	if(rbuf == NULL)
	{
	    imgio_last_load_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}

	/* setjmp() must be called in every function that calls a
	 * PNG-reading libpng function
	 */
	if(setjmp(png_jmpbuf((ctx->png_ptr))))
	{
	    if(imgio_last_load_error == NULL)
		imgio_last_load_error = "libpng failed to open the image";
	    CLEANUP_RETURN(-1);
	}

	/* Set up the libpng progressive reading callbacks */
	png_set_progressive_read_fn(
	    ctx->png_ptr,
	    ctx,			/* User data */
	    ImgPNGReadInfoCB,		/* Read info callback */
	    ImgPNGReadRowCB,		/* Read row callback */
	    ImgPNGReadEndCB		/* Read end callback */
	);

	/* Begin reading the image data */
	do
	{
	    /* Read a segment of bytes from the PNG file and loaded
	     * it into the libpng read buffer
	     */
	    bytes_read = fread(
		rbuf, sizeof(char), rbuf_len, ctx->fp
	    );
	    if((bytes_read > 0) && (bytes_read <= (size_t)rbuf_len))
	    {
		/* Process this segment of the buffer, note that
		 * that we pass the addresses to the PNG pointer
		 * and info pointer pair from the callback data so
		 * that they get updated as needed.
		 */
		png_process_data(
		    ctx->png_ptr,
		    ctx->png_info_ptr,
		    rbuf, bytes_read
		);

		/* Report the progress */
		if(ctx->progress_cb != NULL)
		{
		    /* Report by interlace type */
		    if(ctx->interlace_type == PNG_INTERLACE_ADAM7)
		    {
			/* Give the current interlace pass and the total
			 * number of interlace passes needed as the min
			 * and max
			 */
			const int i = (ctx->interlace_pass * ctx->png_height) +
			    ctx->png_last_row;
			if((i % 5) == 0)
			{
			    if(!ctx->progress_cb(
			        ctx->client_data,
				i, ctx->interlace_passes * ctx->png_height,
			        ctx->png_width, ctx->png_encountered_rows,
				ctx->data_bpl, ctx->data_bpp,
				ctx->data
			    ))
			    {
			        *user_aborted = 1;
			        break;
			    }
			}
		    }
		    else
		    {
			/* No interlacing, give the current number of rows
			 * encountered (current height) and the expected
			 * height as the min and max
			 */
			const int i = ctx->png_encountered_rows;
			if((i % 5) == 0)
			{
			    if(!ctx->progress_cb(
			        ctx->client_data,
			        i, ctx->png_height,
			        ctx->png_width, ctx->png_encountered_rows,
			        ctx->data_bpl, ctx->data_bpp,
			        ctx->data
			    ))
			    {
			        *user_aborted = 1;
			        break;
			    }
			}
		    }
		}
	    }
	} while(!ctx->operation_completed && (bytes_read > 0));


	/* Get text values for creator, title, author, and comments */
	if(ctx->png_info_ptr != NULL)
	{
	    int i, total_text;
	    const char *keyword, *value;
	    png_textp text, text_ptr;

	    /* Get the text values from the PNG info */
	    png_get_text(
		ctx->png_ptr, ctx->png_info_ptr,
		&text, &total_text
	    );
	    for(i = 0; i < total_text; i++)
	    {
		text_ptr = &text[i];

		if(text_ptr->compression != PNG_TEXT_COMPRESSION_NONE)
		    continue;

		keyword = (const char *)text_ptr->key;
		value = (const char *)text_ptr->text;
		if(STRISEMPTY(keyword) || STRISEMPTY(value))
		    continue;

		/* Title */
		if(!strcasecmp(keyword, "Title") ||
		   !strcasecmp(keyword, "Name")
		)
		{
		    if(title_rtn != NULL)
		    {
			free(*title_rtn);
			*title_rtn = STRDUP(value);
		    }
		}
		/* Author */
		else if(!strcasecmp(keyword, "Author") ||
			!strcasecmp(keyword, "Artist")
		)
		{
		    if(author_rtn != NULL)
		    {
			free(*author_rtn);
			*author_rtn = STRDUP(value);
		    }
		}
		/* E-mail */
		else if(!strcasecmp(keyword, "E-mail"))
		{
		    if(author_rtn != NULL)
		    {
			/* Set author field as e-mail only if author
			 * field has not been set yet
			 */
			if(*author_rtn == NULL)
			    *author_rtn = STRDUP(value);
		    }
		}
		/* Creator */
		else if(!strcasecmp(keyword, "Creator") ||
			!strcasecmp(keyword, "Generator")
		)
		{
		    if(creator_rtn != NULL)
		    {
			free(*creator_rtn);
			*creator_rtn = STRDUP(value);
		    }
		}
		/* Comments */
		else if(!strcasecmp(keyword, "Comments") ||
			!strcasecmp(keyword, "Comment")
		)
		{
		    if(comments_rtn != NULL)
		    {
			free(*comments_rtn);
			*comments_rtn = STRDUP(value);
		    }
		}
		/* Description */
		else if(!strcasecmp(keyword, "Description"))
		{
		    if(comments_rtn != NULL)
		    {
			/* Set comments field as description only if
			 * the comments field has not been set yet
			 */
			if(*comments_rtn == NULL)
			    *comments_rtn = STRDUP(value);
		    }
		}
	    }
	}

	/* Write up comments if there isn't one already on the return */
#if 0
/* Comments should not be generated, any comments should be obtained
 * from image data only
 */
	if((*comments_rtn == NULL) && (ctx->png_info_ptr != NULL))
	{
	    char s[256];

	    *s = '\0';
	    png_info_ptr = ctx->png_info_ptr;
	    switch(png_info_ptr->color_type)
	    {
	      case PNG_COLOR_TYPE_GRAY:
		if(ctx->interlace_passes > 1)
		    sprintf(
			s,
"Greyscale, interlaced (%i passes)",
			ctx->interlace_passes
		    );
		else
		    sprintf(
			s,
"Greyscale, non-interlaced"
		    );
		break;

	      case PNG_COLOR_TYPE_PALETTE:
		if(ctx->interlace_passes > 1)
		    sprintf(
			s,
"Palette with %i colors, interlaced (%i passes)",
			png_info_ptr->num_palette,
			ctx->interlace_passes
		    );
		else
		    sprintf(
			s,
"Palette with %i colors, non-interlaced",
			png_info_ptr->num_palette
		    );
		break;

	      case PNG_COLOR_TYPE_RGB:
		if(ctx->interlace_passes > 1)
		    sprintf(
			s,
"Rgb, interlaced (%i passes)",
			ctx->interlace_passes
		    );
		else
		    sprintf(
			s,
"Rgb, non-interlaced"
		    );
		break;

#ifdef PNG_COLOR_TYPE_RGBA
	      case PNG_COLOR_TYPE_RGBA:
#else
	      case PNG_COLOR_TYPE_RGB_ALPHA:
#endif
		if(ctx->interlace_passes > 1)
		    sprintf(
			s,
"Rgb with alpha channel, interlaced (%i passes)",
			ctx->interlace_passes
		    );
		else
		    sprintf(
			s,
"Rgb with alpha channel, non-interlaced"
		    );
		break;

#ifdef PNG_COLOR_TYPE_GA
	      case PNG_COLOR_TYPE_GA:
#else
	      case PNG_COLOR_TYPE_GRAY_ALPHA:
#endif
		if(ctx->interlace_passes > 1)
		    sprintf(
			s,
"Greyscale with alpha channel, interlaced (%i passes)",
			ctx->interlace_passes
		    );
		else
		    sprintf(
			s,
"Greyscale with alpha channel, non-interlaced"
		    );
		break;
	    }
	    /* If comments string was properly formatted, then copy it to
	     * the comments return
	     */
	    if(!STRISEMPTY(s))
		*comments_rtn = STRDUP(s);


	    /* Needs to be NULL when we return control from this case */
	    png_info_ptr = NULL;
	}
#endif

	/* Update the returns */
	if(width_rtn != NULL)
	    *width_rtn = ctx->png_width;
	if(height_rtn != NULL)
	    *height_rtn = ctx->png_height;
	if(bpp_rtn != NULL)              
	    *bpp_rtn = ctx->data_bpp;
	if(bpl_rtn != NULL)
	    *bpl_rtn = ctx->data_bpl;
	if(x_rtn != NULL)          
	    *x_rtn = ctx->png_x;
	if(y_rtn != NULL)          
	    *y_rtn = ctx->png_y;
	if(base_width_rtn != NULL)
	    *base_width_rtn = ctx->png_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = ctx->png_height;

	if(rgba_rtn != NULL)
	{
	    /* Transfer the rgba data to the return */
	    free(*rgba_rtn);
	    *rgba_rtn = ctx->data;

	    /* Mark it NULL so it dosen't get deleted when callback
	     * data is deleted
	     */
	    ctx->data = NULL;
	}

	if(bg_color != NULL)
	    memcpy(bg_color, ctx->bg_color, 4 * sizeof(u_int8_t));

	if(*user_aborted)
	{
	    imgio_last_load_error = "User aborted operation";
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    CLEANUP_RETURN(0);
	}

#undef CLEANUP_RETURN
}

/*
 *	Writes the PNG file.
 *
 *	Return values:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error.
 *	-4	User aborted.
 */
static int ImgPNGWriteRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int x, int y,
	int base_width, int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	int compression_level,	/* 0 to 9 */
	int interlaced,		/* 0 or 1 */
	int format,		/* 0 = B&W  
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB  
				 * 4 = RGBA */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	ImgPNGReadData *ctx = NULL;

#define CLEANUP_RETURN(_v_)	{			\
 /* Delete the callback data */				\
 if(ctx != NULL) {					\
  /* Delete the PNG write structure and info pair */	\
  if(ctx->png_ptr != NULL) {				\
   png_destroy_write_struct(				\
    &ctx->png_ptr,					\
    (ctx->png_info_ptr != NULL) ?			\
     &ctx->png_info_ptr : NULL				\
   );							\
  }							\
							\
  /* Delete the image data in libpng data format */	\
  if(ctx->png_data != NULL) {				\
   const int height = ctx->png_height;			\
   int y;						\
   for(y = 0; y < height; y++)				\
    free(ctx->png_data[y]);				\
   free(ctx->png_data);					\
  }							\
							\
  /* Delete the return RGBA format image data */	\
  free(ctx->data);					\
							\
  /* Delete the transparent color */			\
  free(ctx->trans_color);				\
							\
  /* Close the PNG file */				\
  if(ctx->fp != NULL)					\
   fclose(ctx->fp);					\
							\
  free(ctx->filename);					\
							\
  free(ctx);						\
 }							\
							\
 return(_v_);						\
}


	/* Reset global write error string pointer */
	imgio_last_write_error = NULL;

	/* Check for user abort */
	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    CLEANUP_RETURN(-4);
	}

	if((rgba == NULL) || (width <= 0) || (height <= 0) ||
	   (bpp != 4)
	)
	{
	    imgio_last_write_error = "Invalid description of the image data";
	    CLEANUP_RETURN(-2);
	}

	/* Need to calculate the bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;
	if(bpl <= 0)
	{
	    imgio_last_write_error = "Invalid description of the image data";
	    CLEANUP_RETURN(-2);
	}


	/* Allocate the libpng write callback data */
	ctx = IMG_PNG_READ_DATA(calloc(1, sizeof(ImgPNGReadData)));
	if(ctx == NULL)
	{
	    imgio_last_load_error = "Memory allocation error";
	    CLEANUP_RETURN(-3);
	}
	ctx->filename = STRDUP(filename);
	ctx->fp = NULL;
	ctx->png_ptr = NULL;
	ctx->png_info_ptr = NULL;
	ctx->png_width = width;
	ctx->png_height = height;
	ctx->png_bpl = 0;		/* Not used */
	ctx->png_bpp = 0;		/* Not used */
	ctx->png_x = x;
	ctx->png_y = y;
	ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	ctx->png_last_row_ptr = NULL;
	ctx->png_encountered_rows = 0;
	ctx->png_data = NULL;	/* Allocated later */
	ctx->data = NULL;		/* Never used or allocated */
	ctx->data_bpl = bpl;
	ctx->data_bpp = bpp;	/* Must be 4 bytes RGBA */
	ctx->interlace_type = PNG_INTERLACE_NONE;
	ctx->interlace_pass = 0;
	ctx->interlace_passes = 0;
	ctx->trans_color = NULL;	/* Allocated in ImgPNGInfoCB() */
	ctx->def_alpha_value = 0xff;	/* Not used */
	if(bg_color != NULL)
	    memcpy(ctx->bg_color, bg_color, 4 * sizeof(u_int8_t));
	ctx->operation_completed = 0;
	ctx->client_data = client_data;
	ctx->progress_cb = progress_cb;


	/* Set color_type depending on the given format, format can be:
	 *
	 *	0	B&W
	 *	1	Greyscale
	 *	2	Greyscale with alpha
	 *	3	RGB
	 *	4	RGBA
	 */
	switch(format)
	{
	  case 0:
	  case 1:
	    ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	    break;
	  case 2:
#ifdef PNG_COLOR_TYPE_GA
	    ctx->png_color_type = PNG_COLOR_TYPE_GA;
#else
	    ctx->png_color_type = PNG_COLOR_TYPE_GRAY_ALPHA;
#endif
	    break;
	  case 3:
	    ctx->png_color_type = PNG_COLOR_TYPE_RGB;
	    break;
	  case 4:
#ifdef PNG_COLOR_TYPE_RGBA
	    ctx->png_color_type = PNG_COLOR_TYPE_RGBA;
#else
	    ctx->png_color_type = PNG_COLOR_TYPE_RGB_ALPHA;
#endif
	    break;

	  default:
	    ctx->png_color_type = PNG_COLOR_TYPE_GRAY;
	    break;
	}

	/* Set interlace type */
	if(interlaced)
	    ctx->interlace_type = PNG_INTERLACE_ADAM7;
	else
	    ctx->interlace_type = PNG_INTERLACE_NONE;

	/* Allocate libpng format image data in multiple rows */
	ctx->png_data = (png_bytep *)realloc(
	    ctx->png_data, ctx->png_height * sizeof(png_bytep)
	);
	if(ctx->png_data != NULL)
	{
	    int x, y, png_row_len;
	    png_bytep png_row_ptr;
	    const u_int8_t *src_ptr;
	    png_bytep *png_data = ctx->png_data;


	    /* Set color_type depending on the given format, format can be:
	     *
	     *	0	B&W
	     *	1	Greyscale
	     *	2	Greyscale with alpha
	     *	3	RGB
	     *	4	RGBA
	     */
	    switch(format)
	    {
	      case 0:
	      case 1:
		/* Greyscale */
		png_row_len = 1 * ctx->png_width * sizeof(png_byte);
		for(y = 0; y < ctx->png_height; y++)
		{
		    png_data[y] = png_row_ptr = (png_bytep)malloc(png_row_len);
		    if(png_row_ptr == NULL)
			continue;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % 5) == 0))
		    {
			/* Report progress from 0 to 50% */
			if(!ctx->progress_cb(
			    ctx->client_data,
			    (int)(y / 2.0f), ctx->png_height,
			    ctx->png_width, ctx->png_height,
			    ctx->data_bpl, ctx->data_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }

		    for(x = 0; x < ctx->png_width; x++)
		    {
			src_ptr = (const u_int8_t *)&rgba[
			    (y * ctx->data_bpl) + (x * ctx->data_bpp)
			];
			*png_row_ptr++ = (png_byte)(
			    (0.299 * src_ptr[0]) + (0.587 * src_ptr[1]) +
			    (0.114 * src_ptr[2])
			);
		    }
		}
		break;

	      case 2:
		/* Greyscale Alpha */
		png_row_len = 2 * ctx->png_width * sizeof(png_byte);
		for(y = 0; y < ctx->png_height; y++)
		{
		    png_data[y] = png_row_ptr = (png_bytep)malloc(png_row_len);
		    if(png_row_ptr == NULL)
			continue;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % 5) == 0))
		    {
			/* Report progress from 0 to 50% */
			if(!ctx->progress_cb(
			    ctx->client_data,
			    (int)(y / 2.0f), ctx->png_height,
			    ctx->png_width, ctx->png_height,
			    ctx->data_bpl, ctx->data_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }

		    for(x = 0; x < ctx->png_width; x++)
		    {
			src_ptr = (const u_int8_t *)&rgba[
			    (y * ctx->data_bpl) + (x * ctx->data_bpp)
			];
			*png_row_ptr++ = (png_byte)(
			    (0.299 * src_ptr[0]) + (0.587 * src_ptr[1]) +
			    (0.114 * src_ptr[2])
			);
			*png_row_ptr++ = (png_byte)src_ptr[3];
		    }
		}
		break;

	      case 3:
		/* RGB */
		png_row_len = 3 * ctx->png_width * sizeof(png_byte);
		for(y = 0; y < ctx->png_height; y++)
		{
		    png_data[y] = png_row_ptr = (png_bytep)malloc(png_row_len);
		    if(png_row_ptr == NULL)
			continue;

		    /* Report progress */
		    if((progress_cb != NULL) && ((y % 5) == 0))
		    {
			/* Report progress from 0 to 50% */
			if(!ctx->progress_cb(
			    ctx->client_data,
			    (int)(y / 2.0f), ctx->png_height,
			    ctx->png_width, ctx->png_height,
			    ctx->data_bpl, ctx->data_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }

		    for(x = 0; x < ctx->png_width; x++)
		    {
			src_ptr = (const u_int8_t *)&rgba[
			    (y * ctx->data_bpl) + (x * ctx->data_bpp)
			];
			*png_row_ptr++ = src_ptr[0];
			*png_row_ptr++ = src_ptr[1];
			*png_row_ptr++ = src_ptr[2];
		    }
		}
		break;

	      case 4:
		/* RGBA */
		png_row_len = 4 * ctx->png_width * sizeof(png_byte);
		for(y = 0; y < ctx->png_height; y++)
		{
		    png_data[y] = png_row_ptr = (png_bytep)malloc(png_row_len);
		    if(png_row_ptr == NULL)
			continue;

		    /* Report progress */
		    if((ctx->progress_cb != NULL) && ((y % 5) == 0))
		    {
			/* Report progress from 0 to 50% */
			if(!ctx->progress_cb(
			    ctx->client_data,
			    (int)(y / 2.0), ctx->png_height,
			    ctx->png_width, ctx->png_height,
			    ctx->data_bpl, ctx->data_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }
		    for(x = 0; x < ctx->png_width; x++)
		    {
			src_ptr = (const u_int8_t *)&rgba[
			    (y * ctx->data_bpl) + (x * ctx->data_bpp)
			];
			*png_row_ptr++ = src_ptr[0];
			*png_row_ptr++ = src_ptr[1];
			*png_row_ptr++ = src_ptr[2];
			*png_row_ptr++ = src_ptr[3];
		    }
		}
		break;
	    }
	}

	/* Check for user abort */
	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    CLEANUP_RETURN(-4);
	}

	/* Open the PNG file for writing */
	ctx->fp = fopen(ctx->filename, "wb");
	if(ctx->fp == NULL)
	{
	    imgio_last_write_error = "Unable to open the file for writing";
	    CLEANUP_RETURN(-1);
	}

	/* Create the libpng write structure */
	ctx->png_ptr = png_create_write_struct(
	    PNG_LIBPNG_VER_STRING, NULL, NULL, NULL
	);
	if(ctx->png_ptr == NULL)
	{
	    imgio_last_write_error = "Unable to create a new PNG write structure";
	    CLEANUP_RETURN(-3);
	}

	/* Create the libpng info structure */
	ctx->png_info_ptr = png_create_info_struct(ctx->png_ptr);
	if(ctx->png_info_ptr == NULL)
	{
	    imgio_last_write_error = "Unable to create a new PNG info structure";
	    CLEANUP_RETURN(-3);
	}

	/* setjmp() must be called in every function that calls a
	 * PNG-writing libpng function.
	 */
	if(setjmp(png_jmpbuf(ctx->png_ptr)))
	{
	    if(imgio_last_write_error)
		imgio_last_write_error = "libpng failed to save the image";
	    CLEANUP_RETURN(-1);
	}

	/* Set output stream */
	png_init_io(ctx->png_ptr, ctx->fp);

	/* Begin setting write parameters */

	/* Compression level (0 to 9) */
	png_set_compression_level(
	    ctx->png_ptr,
	    CLIP(compression_level, 0, 9)
	);

#if 0
/* this is default for no filtering; Z_FILTERED is default otherwise: */
	png_set_compression_strategy(ctx->png_ptr, Z_DEFAULT_STRATEGY);
/* these are all defaults: */
	png_set_compression_mem_level(ctx->png_ptr, 8);
	png_set_compression_window_bits(ctx->png_ptr, 15);
	png_set_compression_method(ctx->png_ptr, 8);
#endif

	/* Set up the PNG image header */
	png_set_IHDR(
	    ctx->png_ptr, ctx->png_info_ptr,
	    ctx->png_width, ctx->png_height,	/* Size */
	    8,			/* Depth per pixel compoent */
	    ctx->png_color_type, ctx->interlace_type,
	    PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT
	);
#if 0
	if(gamma > 0.0f)
	    png_set_gAMA(png_ptr, png_info_ptr, gamma);
#endif

#if defined(PNG_bKGD_SUPPORTED)
	/* Set the background color */
	if(ctx->bg_color != NULL)
	{
	    png_color_16 bg;

	    /* Format in 8 bit RGBA */
	    bg.red = ctx->bg_color[0];
	    bg.green = ctx->bg_color[1];
	    bg.blue = ctx->bg_color[2];
/*	    ??? = ctx->bg_color[3];	skip alpha */
	    bg.gray = ((int)ctx->bg_color[0] +
		(int)ctx->bg_color[1] + (int)ctx->bg_color[2]) / 3;
	    png_set_bKGD(ctx->png_ptr, ctx->png_info_ptr, &bg);
	}
#endif

#if defined(PNG_oFFs_SUPPORTED)
	/* Set the offsets */
	if(1)
	{
	    png_int_32	offset_x = (png_int_32)ctx->png_x,
			offset_y = (png_int_32)ctx->png_y;
	    int unit_type = PNG_OFFSET_PIXEL;
	    png_set_oFFs(
		ctx->png_ptr, ctx->png_info_ptr,
		offset_x, offset_y, unit_type
	    );
	}
#endif

#if 0
	/* Set the modified time */
	if(mainprog_ptr->have_time)
	{
	    png_time modtime;

	    png_convert_from_time_t(&modtime, mainprog_ptr->modtime);
	    png_set_tIME(ctx->png_ptr, ctx->png_info_ptr, &modtime);
	}
#endif

	/* Write the text fields */
	if(1)
	{
#define total_text	7
	    png_text text[total_text];
	    int num_text = 0;

	    if(!STRISEMPTY(creator))
	    {
		/* PNG dosen't have a standard keyword (or any keyword
		 * that comes close) to suggest creator, so implicitly
		 * use a keyword called "Creator"
		 */
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "Creator";
		text[num_text].text = (char *)creator;
		++num_text;
	    }
	    if(!STRISEMPTY(title))
	    {
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "Title";
		text[num_text].text = (char *)title;
		++num_text;
	    }
	    if(!STRISEMPTY(author))
	    {
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "Author";
		text[num_text].text = (char *)author;
		++num_text;
	    }
	    if(!STRISEMPTY(comments))
	    {
		/* Save comments under PNG keyword "Description" */
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "Description";
		text[num_text].text = (char *)comments;
		++num_text;
	    }
	    if(0)
	    {
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "Copyright";
		text[num_text].text = "By Author";
		++num_text;
	    }
	    if(0)
	    {
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "E-mail";
		text[num_text].text = "Undefined";
		++num_text;
	    }
	    if(0)
	    {
		text[num_text].compression = PNG_TEXT_COMPRESSION_NONE;
		text[num_text].key = "URL";
		text[num_text].text = "None";
		++num_text;
	    }
	    png_set_text(
		ctx->png_ptr, ctx->png_info_ptr,
		text, num_text
	    );

#undef total_text
	}

	/* Write all chunks up to (but not including) first IDAT */
	png_write_info(ctx->png_ptr, ctx->png_info_ptr);


	/* Set up the transformations:  for now, just pack low-bit-depth
	 * pixels into bytes (one, two or four pixels per byte)
	 */
	png_set_packing(ctx->png_ptr);
#if 0
	/* to scale low-bit-depth values */
	png_set_shift(ctx->png_ptr, &sig_bit);
#endif



	/* Begin writing the image data */

	if((ctx->png_data != NULL) && !(*user_aborted))
	{
	    /* Handle by interlaced or non-interlacted */
	    if(ctx->interlace_type == PNG_INTERLACE_NONE)
	    {
		int y;

		/* For non-interlaced, each row is written at a
		 * at a time
		 */
		for(y = 0; y < ctx->png_height; y++)
		{
		    /* Report the progress? */
		    if((ctx->progress_cb != NULL) && ((y % 5) == 0))
		    {
			/* Report the progress from 50% to 100% */
			if(!ctx->progress_cb(
			    ctx->client_data,
			    (int)(y / 2.0) + (ctx->png_height / 2), ctx->png_height,
			    ctx->png_width, ctx->png_height,
			    ctx->data_bpl, ctx->data_bpp,
			    rgba
			))
			{
			    *user_aborted = 1;
			    break;
			}
		    }
		    /* Write this row */
		    png_write_row(ctx->png_ptr, ctx->png_data[y]);
		}
	    }
	    else
	    {
		/* For interlacing, the entire image must be written
		 * at once
		 */
		png_write_image(ctx->png_ptr, ctx->png_data);
	    }
	}

	/* Since that's it, we also close out the end of the PNG file
	 * now
	 *
	 * If we had any text or time info to write after the IDATs,
	 * second argument would be info_ptr, but we optimize slightly
	 * by sending NULL pointer
	 */
	png_write_end(ctx->png_ptr, NULL);

	if(*user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    CLEANUP_RETURN(-4);
	}
	else
	{
	    CLEANUP_RETURN(0);
	}
#undef CLEANUP_RETURN
}

/*
 *	Writes the PNG image to file using libpng.
 *
 *	If progress_cb is not NULL then it will be called during
 *	loading, if it returns 0 then loading will be aborted.
 *
 *	Return values:
 *
 *	0	Success
 *	-1	General error
 *	-2	Bad value
 *	-3	Systems error
 *	-4	User abort
 */
int ImgWritePNGFileRGBA(
	const char *filename,
	int width, int height,
	int bpl, int bpp,	/* bpp must be 4 */
	u_int8_t *rgba,
	const u_int8_t *bg_color,	/* 4 bytes in RGBA format */
	int x, int y,
	int base_width, int base_height,
	const char *creator, const char *title,
	const char *author, const char *comments,
	int compression_level,	/* 0 to 9 */
	int interlaced,		/* 0 or 1 */
	int format,		/* 0 = B&W  
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB  
				 * 4 = RGBA */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	)
)
{
	int status;
	int user_aborted = 0;


	/* Reset global last write error message pointer */
	imgio_last_write_error = NULL;

	if(STRISEMPTY(filename) || (rgba == NULL) ||
	   (width <= 0) || (height <= 0) ||
	   (bpp != 4)
	)
	{
	    imgio_last_write_error = "Invalid value used to describe image";
	    return(-1);
	}

	/* Automatically calculate bytes per line? */
	if(bpl <= 0)
	    bpl = width * bpp;

	/* Report the initial progress */
	if(progress_cb != NULL)
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	/* Call the PNG writing function for RGBA image data format */
	status = ImgPNGWriteRGBA(
	    filename,
	    width, height, bpl, bpp,
	    rgba,
	    bg_color,
	    x, y, base_width, base_height,
	    creator, title, author, comments,
	    compression_level,
	    interlaced,
	    format,		/* 0 = b&w, 1 = grey, 2 = greyalpha
				 * 3 = rgb, 4 = rgba.
				 */
	    client_data, progress_cb,
	    &user_aborted
	);


	/* Report the final progress */
	if((progress_cb != NULL) && !user_aborted)
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		user_aborted = 1;
	}

	if(user_aborted)
	{
	    imgio_last_write_error = "User aborted operation";
	    return(-4);
	}
	else
	    return(status);
}

#endif	/* HAVE_LIBPNG */
