#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "../include/string.h"
#include "../include/strexp.h"
#include "../include/fio.h"
#include "guiutils.h"
#include "guirgbimg.h"
#include "libps.h"


#ifndef DEBUG_LEVEL
# define DEBUG_LEVEL		0
#endif


/*
 *	PostScript Visuals:
 */
typedef enum {
	PSVisualGrey,
	PSVisualRGB,
	PSVisualRGBA
} ps_visual;


static const char *ps_last_error;


/* Stack */
static const char *stack_nth(char **sv, int sc, int n);
#define STACK_NTH(_n_)		stack_nth(stackv, stackc, (_n_))
static const char *stack_get_fp(char ***sv, int *sc, FILE *fp);
#define STACK_GET_FP(_fp_)	stack_get_fp(&stackv, &stackc, (_fp_))
static void stack_pop(char ***sv, int *sc);
#define STACK_POP()		stack_pop(&stackv, &stackc)
static void stack_clear(char ***sv, int *sc);
#define STACK_CLEAR()		stack_clear(&stackv, &stackc)


/* Error handling */
const char *PSLastError(void);


/* PostScript Reading */
static u_int8_t *PSReadRGBAColorImage(
	FILE *fp,
	int bits_per_comp,	/* Compoent size in bits */
	int ncomp,		/* Compoents per pixel */
	int width, int height,	/* Size of image data from stream fp */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted, int *status
);
static void PSTransformRGBAColorImage(
	u_int8_t **rgba,
	int *width_cur, int *height_cur, int *bpl_cur,
	int *x_cur, int *y_cur,
	int sx, int sy,		/* Size scale divisor */
	int tx, int ty,		/* Translate */
	int *status
);

int PSReadRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);
int PSReadFileRGBA(
	const char *filename,        
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
);


/* PostScript Writing */
static int PSWriteFilePS3Any(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *img_data,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* Image data */
	),
	int *user_aborted,
	ps_visual visual
);
int PSWriteFilePS3RGBA(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,  /* Size of image in pixels */
	int bpl,                /* Bytes per line */
	u_int8_t *rgba,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,         /* Client data */
		int, int,       /* Progress current & end values */
		int, int,       /* Width & height */
		int, int,       /* Bytes per line & bytes per pixel */
		u_int8_t *      /* Image data */
	), 
	int *user_aborted
);
int PSWriteFilePS3RGB(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *rgb,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* Image data */
	), 
	int *user_aborted
);
int PSWriteFilePS3Grey(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *grey,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* Image data */
	),
	int *user_aborted
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)     (((x) < 0) ? ((x) * -1) : (x))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : 1)

#define ISSTRTRUE(s)	(((s) != NULL) ? (tolower(*(s)) == 't') : 0)
#define ISSTRYES(s)	(((s) != NULL) ? (tolower(*(s)) == 'y') : 0)
#define ISSTRON(s)	(((s) != NULL) ? (!strcasecmp((s), "on")) : 0)


/*
 *	Gets the nth value on the stack.
 *
 *	If n is negative then nth is relative from the top of the
 *	stack, for example n = -1 means the top most value and n = -2
 *	means second to the top most value.
 */
static const char *stack_nth(char **sv, int sc, int n)
{
	/* Relative? */
	if(n < 0)
	{
	    n = sc + n;
	    if(n >= 0)
		return(sv[n]);
	    else
		return(NULL);	/* Stack underflow */
	}
	else
	{
	    if(n < sc)
		return(sv[n]);
	    else
		return(NULL);	/* Stack overflow */
	}
}

/*
 *	Reads the next stack value from fp and pushes it on to the
 *	stack.
 *
 *	Returns the new stack value or NULL if EOF is reached.
 */
static const char *stack_get_fp(char ***sv, int *sc, FILE *fp)
{
	int brace_level = 0;
	const int stack_item_len_max = 1024;
	int c, i;

	char *s = (char *)malloc(stack_item_len_max * sizeof(char));
	if(s == NULL)
	    return(NULL);

	/* Seek past spaces */
	do
	{
	    c = fgetc(fp);
	    if(c == EOF)
	    {
		free(s);
		return(NULL);
	    }
	}
	while(ISSPACE(c));
	fseek(fp, -1, SEEK_CUR);

	/* Read value */
	for(i = 0; i < stack_item_len_max; i++)
	{
	    c = fgetc(fp);
	    if(c == EOF)
	    {
		free(s);
		return(NULL);
	    }
	    else if(ISSPACE(c) && (brace_level <= 0))
	    {
		s[i] = '\0';
		fseek(fp, -1, SEEK_CUR);	/* So next read starts at space */
		break;
	    }
	    else
	    {
		if((c == '[') || (c == '{') || (c == '\"'))
		{
		    brace_level++;
		    i--;
		}
		else if((c == ']') || (c == '}') || (c == '\"'))
		{
		    brace_level--;
		    i--;
		}
		else
		{
		    s[i] = (char)c;
		}
	    }
	}
	if(i < stack_item_len_max)
	    s = (char *)realloc(s, (i + 1) * sizeof(char));
	else
	    s[stack_item_len_max - 1] = '\0';
	strstrip(s);

	/* Append value s to list */
	i = MAX(*sc, 0);
	*sc = i + 1;
	*sv = (char **)realloc(*sv, (*sc) * sizeof(char *));
	if(*sv == NULL)
	{
	    *sc = 0;
	    free(s);
	    return(NULL);
	}
	(*sv)[i] = s;

	return(s);
}

/*
 *	Pops the top most value off the stack.
 */
static void stack_pop(char ***sv, int *sc)
{
	int i = MAX((*sc) - 1, 0);

	if(i < *sc)
	    free((*sv)[i]);

	*sc = i;
	if(i > 0)
	{
	    *sv = (char **)realloc(*sv, i * sizeof(char *));
	    if(*sv == NULL)
		*sc = 0;
	}
	else
	{
	    free(*sv);
	    *sv = NULL;
	}
}

/*
 *	Deletes the entire stack.
 */
static void stack_clear(char ***sv, int *sc)
{
	free(*sv);
	*sv = NULL;
	*sc = 0;
}


/*
 *      Returns a statically allocated string describing the last   
 *      error that occured or NULL if no error occured.   
 */
const char *PSLastError(void)
{
	return(ps_last_error);
}


/*
 *	Reads from the stream fp a series of 6 character RRGGBB
 *	bytes that represent a PostScript colorimage image and returns
 *	the image data as RGBA.
 */
static u_int8_t *PSReadRGBAColorImage(
	FILE *fp,
	int bits_per_comp,	/* Compoent size in bits */
	int ncomp,		/* Compoents per pixel */
	int width, int height,	/* Size of image data from stream fp */
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted, int *status
)
{
	int i, n, s_len;
	const int bpp = 4;
	int y, bpl;
	char *s;
	u_int8_t *rgba, *line, *line_end, *ptr, *ptr_end;

	if((width <= 0) || (height <= 0) ||
	   (ncomp <= 0) || (bits_per_comp <= 0)
	)
	{
	    *status = PSBadValue;
	    ps_last_error = "Invalid value given to primitive \"colorimage\"";
	    return(NULL);
	}

	/* Seek fp past spaces */
	do {

	    i = fgetc(fp);
	    if(i == EOF)
	    {
		*status = PSBadValue;
		ps_last_error = "Unexpected end of file encountered";
		return(NULL);
	    }

	} while(ISSPACE(i));
	fseek(fp, -1, SEEK_CUR);


	/* Calculate bytes per line */
	bpl = width * bpp;

	/* Allocate return image as the size of the bounding box
	 * and use calloc() so that the alpha channel is 0x00
	 */
	rgba = (u_int8_t *)malloc(bpl * height);
	if(rgba == NULL)
	{
	    *status = PSErrorSystem;
	    ps_last_error = "Memory allocation error";
	    return(NULL);
	}

	/* Allocate buffer for reading one PostScript pixel based on
	 * the specified number of compoents per pixel * 2
	 */
	s_len = ncomp * 2;
	s = (char *)malloc(s_len);
	if(s == NULL)
	{
	    free(rgba);
	    *status = PSErrorSystem;
	    ps_last_error = "Memory allocation error";
	    return(NULL);
	}

	/* Report initial progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		*user_aborted = 1;
	}

	/* Iterate through image */
	for(y = 0,
	    line = rgba,
	    line_end = line + (bpl * height);
	    line < line_end;
	    y++,
	    line += bpl
	)
	{
	    ptr = line;
	    ptr_end = line + (width * bpp);
	    while(ptr < ptr_end)
	    {
		/* Read s_len characters for the hex pixel string from
		 * the PostScript data
		 */
		n = 0;
		do
		{
		    i = fgetc(fp);
		    if(i == EOF)
		    {
			*status = PSBadValue;
			ps_last_error = "Unexpected end of file encountered";
			return(rgba);
		    }
		    if(ISSPACE(i))
			continue;
		    s[n] = tolower(i);
		    n++;
		} while(n < s_len);

#define HEX_CTOI(_c_)	(u_int8_t)(((_c_) >= 'a') ?	\
 ((_c_) - 'a' + 10) : ((_c_) - '0')			\
)
		/* Convert hex pixel values to image data's current pixel */
		switch(ncomp)
		{
		  case 4:
		    *ptr++ = (HEX_CTOI(tolower(s[0])) * 16) +
			HEX_CTOI(tolower(s[1]));
		    *ptr++ = (HEX_CTOI(tolower(s[2])) * 16) +
			HEX_CTOI(tolower(s[3]));
		    *ptr++ = (HEX_CTOI(tolower(s[4])) * 16) +
			HEX_CTOI(tolower(s[5]));
		    *ptr++ = (HEX_CTOI(tolower(s[6])) * 16) +
			HEX_CTOI(tolower(s[7]));
		    break;

		  case 3:
		    *ptr++ = (HEX_CTOI(tolower(s[0])) * 16) +
			HEX_CTOI(tolower(s[1]));
		    *ptr++ = (HEX_CTOI(tolower(s[2])) * 16) +
			HEX_CTOI(tolower(s[3]));
		    *ptr++ = (HEX_CTOI(tolower(s[4])) * 16) +
			HEX_CTOI(tolower(s[5]));
		    *ptr++ = 0xff;
		    break;

		  case 1:
		    ptr[0] = ptr[1] = ptr[2] =
			(HEX_CTOI(tolower(s[0])) * 16) +
			HEX_CTOI(tolower(s[1]));
		    ptr[3] = 0xff;
		    ptr += bpp;
		    break;

		  default:
		    ptr += bpp;
		    break;
		}
#undef HEX_CTOI
	    }

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(!progress_cb(
		    client_data,
		    y, height,
		    width, height,
		    bpl, bpp,
		    rgba
		))
		{
		    *user_aborted = 1;
		    break;
		}
	    }
	}

	/* Delete buffer for reading one PostScript pixel */
	free(s);

	/* Report final progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		rgba
	    ))
		*user_aborted = 1;
	}

	return(rgba);
}

/*
 *	Applies the transformation matrix specified by sx, sy, tx, and
 *	ty to the specified image.
 */
static void PSTransformRGBAColorImage(
	u_int8_t **rgba,
	int *width_cur, int *height_cur, int *bpl_cur,    
	int *x_cur, int *y_cur,
	int sx, int sy,		/* Size scale divisor */
	int tx, int ty,		/* Translate */
	int *status
)
{
	const int bpp = 4;
	int	x = *x_cur,
		y = *y_cur,
		width = *width_cur,
		height = *height_cur,
		bpl = *bpl_cur;

#if (DEBUG_LEVEL >= 1)
printf(
 "PSTransformRGBAColorImage():\
 size_scale_divisor: %i %i  translate: %i %i\n",
 sx, sy,
 tx, ty
);
#endif

	if((*rgba == NULL) || (sx == 0) || (sy == 0))
	{
	    *status = PSBadValue;
	    ps_last_error = "Invalid value given to primitive \"colorimage\"";
	    return;
	}

	if(bpl <= 0)
	    bpl = width * bpp;

	/* Mirror horizontally if matrix width scale sx is negative */
	if(sx < 0)
	{
	    u_int8_t	*src_data = *rgba,
			*tar_data = (u_int8_t *)malloc(bpl * height);

	    /* Mirror image horizontally */
	    GUIImageBufferMirrorH(
	        bpp,
		src_data,
		width, height, bpl,
		tar_data,          
		width, height, bpl
	    );                    
	    free(src_data);
	    *rgba = tar_data;

	    x += sx;	/* Adjust x due to mirroring horizontally */
	}
	else
	{
	    /* Do not adjust x when not mirroring horizontally */
	}

	/* Mirror vertically if matrix height scale sy is positive */
	if(sy > 0)
	{
	    u_int8_t	*src_data = *rgba,
			*tar_data = (u_int8_t *)malloc(bpl * height);

	    /* Mirror image vertically */
	    GUIImageBufferMirrorV(
		bpp,
		src_data,              
		width, height, bpl,
		tar_data,
		width, height, bpl 
	    );                    
	    free(src_data);
	    *rgba = tar_data;

	    /* Do not adjust y when mirroring vertically */
	}
	else
	{
	    y += sy;	/* Adjust y due to not mirroring vertically */
	}

	/* Resize as needed */
	if((ABSOLUTE(sx) != width) ||
	   (ABSOLUTE(sy) != height)
	)
	{
	    float	width_coeff = ABSOLUTE((float)width / (float)sx),
			height_coeff = ABSOLUTE((float)height / (float)sy);
 	    int	rs_width = (int)(width * width_coeff),
		rs_height = (int)(height * height_coeff),
		rs_bpl = rs_width * bpp;
	    u_int8_t	*src_data = *rgba,
			*rs_data = ((rs_bpl * rs_height) > 0) ?
			    (u_int8_t *)malloc(rs_bpl * rs_height) : NULL;

	    if(rs_data != NULL)
	    {
#if (DEBUG_LEVEL >= 1)
printf(
 "PSTransformRGBAColorImage():\
 resize from: %ix%i  to: %ix%i\n",
 width, height,
 rs_width, rs_height
);
#endif

		/* Since y is lower-left oriented we need to adjust it
		 * due to the size change of the image
		 */                   
		y += height - rs_height;

		/* Resize */
	        GUIImageBufferResize(
		    bpp,
		    src_data,
		    width, height, bpl,
		    rs_data,
		    rs_width, rs_height, rs_bpl,
		    NULL, NULL
		);

		free(*rgba);
		*rgba = src_data = rs_data;
		width = rs_width;
		height = rs_height;
		bpl = rs_bpl;
	    }
	    else
	    {
		*status = PSErrorSystem;
		ps_last_error = "Memory allocation error";
	    }
	}

	/* Translate coordinates */
	x += tx;
	y += ty;

	/* Update returns */
	*width_cur = width;
	*height_cur = height;
	*bpl_cur = bpl;
	*x_cur = x;
	*y_cur = y;
}


/*
 *	Reads a PostScript image from the specified stream.
 *
 *	The returned image data will be 4 bytes per pixel in RGBA
 *	format, the calling function must delete the returned image
 *	data.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if the stream does not contain valid
 *	PostScript data.
 */
int PSReadRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,     /* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */  
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */       
	),
	int *user_aborted
)
{
	int status = PSSuccess;
	const char *s;
	char *comment;
	char **stackv;
	int stackc;
	int cur_page, pages;
	int landscape = 0;
	int x, y;			/* Translation from lower-left of 0,0 */
	gfloat scale_width, scale_height;	/* From unit size */
	int bb_x[2], bb_y[2];		/* Bounding box coordinates */
	int width, height;		/* Size of image in pixels */
	int bb_width, bb_height;	/* Size of bounding box */
	int bpp, bpl;
	u_int8_t *rgba;

	/* Reset globals */
	ps_last_error = NULL;

	if(*user_aborted)
	    return(PSAbort);

	if(fp == NULL)
	{
	    ps_last_error = "Stream pointer is NULL";
	    return(PSError);
	}

#if (DEBUG_LEVEL >= 1)
 printf(
  "PSReadRGBA(): Reading from descriptor: %i\n",
  fileno(fp)
 );
#endif                                   

	/* Check identifier */
	if(1)
	{
	    char *s, buf[10];
	    int bytes_read = fread(
		buf, sizeof(char), sizeof(buf) / sizeof(char), fp
	    );
	    if(bytes_read == sizeof(buf))
	    {
		if(memcmp(buf, "%!PS-Adobe", sizeof(buf)))
		    return(PSBadValue);
	    }
	    else
	    {
		return(PSBadValue);
	    }

	    /* Seek past identifier */
	    s = FGetStringLiteral(fp);
	    free(s);
	}

	/* Reset values */
	cur_page = 0;
	pages = 0;
	bb_x[0] = 0;
	bb_x[1] = 0;
	bb_y[0] = 611;
	bb_x[1] = 791;
	bb_width = 612;		/* Default to 850x1100 at 72 DPI */
	bb_height = 792;
	x = 0;
	y = 0;
	scale_width = 0.0f;
	scale_height = 0.0f;
	width = 0;
	height = 0;
	bpp = 4;		/* Always 4 bytes since reading RGBA */
	bpl = 0;
	rgba = NULL;

	/* Read subsequent lines of the PostScript file */
	stackv = NULL;
	stackc = 0;
	while(1)
	{
	    s = STACK_GET_FP(fp);
	    if(s == NULL)
		break;

	    /* Comment? */
	    if(*s == '%')
	    {
		int i;

		/* Seek past blanks */
		do
		{
		    i = fgetc(fp);
		    if(i == EOF)
			break;
		}
		while(ISBLANK(i));
		fseek(fp, -1, SEEK_CUR);

		comment = FGetStringLiteral(fp);
		if(comment == NULL)
		{
		    STACK_POP();
		    continue;
		}

#if (DEBUG_LEVEL >= 2)
 printf("PSReadRGBA(): Comment: \"%s%s\"\n", s, comment);
#endif                                   

		/* Creator */
		if(!strcasecmp(s, "%%Creator:"))
	        {
		    if(creator_rtn != NULL)
		    {
			free(*creator_rtn);
			*creator_rtn = STRDUP(comment);
		    }
		}
	        /* Title */
	        else if(!strcasecmp(s, "%%Title:"))
	        {
		    if(title_rtn != NULL)
		    {
			free(*title_rtn);
			*title_rtn = STRDUP(comment);
		    }
		}
		/* Author */
	        else if(!strcasecmp(s, "%%Author:") ||
		        !strcasecmp(s, "%%Copyright:")
	        )
	        {
		    if(author_rtn != NULL)
		    {
			free(*author_rtn);
			*author_rtn = STRDUP(s);
		    }
		}
		/* Comment */
		else if(!strcasecmp(s, "%%Comment:") || 
			!strcasecmp(s, "%%Comments:")
		)
		{
		    if(comments_rtn != NULL)
		    {
			free(*comments_rtn);
			*comments_rtn = STRDUP(s);
		    }
		}
		/* Orientation */
		else if(!strcasecmp(s, "%%Orientation:"))
		{
		    if(!strcasecmp(comment, "landscape"))
			landscape = 1;
		    else
			landscape = 0;
		}
	        /* Pages */
		else if(!strcasecmp(s, "%%Pages:"))
		{
		    if(!strcasecmp(comment, "(atend)"))
		    {

		    }
		    else
		    {
			sscanf(
			    comment, "%i",
			    &pages
			);
		    }
	        }
		/* BoundingBox */
	        else if(!strcasecmp(s, "%%BoundingBox:"))
	        {
		    sscanf(
		        comment, "%i %i %i %i",
		        &bb_x[0], &bb_y[0],
		        &bb_x[1], &bb_y[1]
		    );
		    bb_width = bb_x[1] - bb_x[0] + 1;
		    bb_height = bb_y[1] - bb_y[0] + 1;
	        }
	        /* Page */
	        else if(!strcasecmp(s, "%%Page:"))
	        {
		    sscanf(
			comment, "%i %i",
			&cur_page, &pages
		    );
	        }

		free(comment);

		STACK_POP();
	    }
	    else
	    {
		/* Handle by current stack value */
#if (DEBUG_LEVEL >= 2)
 printf("PSReadRGBA(): Stack %i: \"%s\"\n", stackc, s);
#endif
		/* SetBBox */
		if(!strcmp(s, "setbbox"))
		{
		    /* Ignore */
		    STACK_CLEAR();
		}
		/* Define */
		else if(!strcmp(s, "def"))
		{
		    int i;
		    const char *sn;

		    for(i = stackc - 1; i >= 0; i--)
		    {
			sn = STACK_NTH(i);
			if(sn != NULL)
			{
			    if(*sn == '/')
			    {
#if (DEBUG_LEVEL >= 1)
 printf(
  "PSReadRGBA(): define \"%s\"\n",
  sn + 1
 );
#endif
				break;
			    }
			}
		    }
		    STACK_CLEAR();
		}
		/* Translate */
		else if(!strcmp(s, "translate"))
		{
		    /* Get values relative to lower-left corner */
		    x = ATOI(STACK_NTH(-3));
		    y = ATOI(STACK_NTH(-2));
#if (DEBUG_LEVEL >= 1)
 printf(
  "PSReadRGBA(): translate: %i %i\n",
  x, y
 );
#endif
		    STACK_CLEAR();
		}
		/* Scale */
		else if(!strcmp(s, "scale"))
		{
		    /* Get scale coefficient from unit size */
		    scale_width = (gfloat)ATOF(STACK_NTH(-3));
		    scale_height = (gfloat)ATOF(STACK_NTH(-2));
#if (DEBUG_LEVEL >= 1) 
 printf(
  "PSReadRGBA(): scale: %f %f\n",
  scale_width, scale_height
 );   
#endif
		    STACK_CLEAR();
		}
		/* Colorimage */
		else if(!strcmp(s, "colorimage"))
		{
		    int bits_per_comp, multi, ncomp;
		    const char *matrix_s, *data_src_s;

		    width = ATOI(STACK_NTH(-8));
		    height = ATOI(STACK_NTH(-7));
		    bits_per_comp = ATOI(STACK_NTH(-6));
		    matrix_s = STACK_NTH(-5);
		    data_src_s = STACK_NTH(-4);
		    multi = ISSTRTRUE(STACK_NTH(-3));
		    ncomp = ATOI(STACK_NTH(-2));

		    bpl = width * bpp;

#if (DEBUG_LEVEL >= 1)    
 printf(
  "PSReadRGBA(): colorimage:\
 %ix%i  bits_per_comp: %i  multi: %s  ncomp: %i\n",
  width, height,
  bits_per_comp,
  multi ? "true" : "false",
  ncomp
 );
#endif

		    /* Delete existing image data if any and begin
		     * reading image data from file
		     */
		    free(rgba);
		    rgba = PSReadRGBAColorImage(
			fp,
			bits_per_comp, ncomp,
			width, height,
			client_data,
			progress_cb,
			user_aborted, &status
		    );

		    /* Modify with transformation matrix? */
		    if(matrix_s != NULL)
		    {
			int strc;
			char **strv = strexp(matrix_s, &strc);
			if(strc >= 6)
			{
			    PSTransformRGBAColorImage(
				&rgba,
				&width, &height, &bpl,
				&x, &y,
				ATOI(strv[0]), ATOI(strv[3]),
				ATOI(strv[4]), ATOI(strv[5]),
				&status
			    );
			}
			strlistfree(strv, strc);
		    }

		    STACK_CLEAR();
		}
	    }
	}

	STACK_CLEAR();

#if (DEBUG_LEVEL >= 1)
if(rgba != NULL)
 printf(
  "PSReadRGBA():\
 Got image: %ix%i  Bounding Box: %ix%i  Upper-Left Translate: %i %i\n",
  width, height,
  bb_width, bb_height,
  x, bb_height - y - height
 );
#endif

	/* Update returns */
	if(width_rtn != NULL)
	    *width_rtn = width;
	if(height_rtn != NULL)
	    *height_rtn = height;
	if(bpp_rtn != NULL)
	    *bpp_rtn = bpp;
	if(bpl_rtn != NULL)
	    *bpl_rtn = bpl;
	if(rgba_rtn != NULL)
	{
	    free(*rgba_rtn);
	    *rgba_rtn = rgba;		/* Transfer to return */
	    rgba = NULL;
	}

	if(x_rtn != NULL)
	    *x_rtn = x;
	if(y_rtn != NULL)
	    *y_rtn = bb_height - y - height;

	if(base_width_rtn != NULL)
	    *base_width_rtn = bb_width;
	if(base_height_rtn != NULL)
	    *base_height_rtn = bb_height;


	/* Delete image data if it was not transfered to return */
	free(rgba);

	if(*user_aborted)
	{
	    ps_last_error = "User aborted";
	    return(PSAbort);
	}
	else
	    return(status);
}

/*
 *	Reads a PostScript image from the specified file.
 *
 *	The returned image data will be 4 bytes per pixel in RGBA
 *	format, the calling function must delete the returned image
 *	data.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if the file is not a PostScript image.
 */
int PSReadFileRGBA(
	const char *filename,
	int *width_rtn, int *height_rtn, int *bpp_rtn, int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,                     
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* RGBA data */
	),
	int *user_aborted
)
{
	int status;
	FILE *fp = FOpen(filename, "rb");
	if(fp == NULL)
	{
	    ps_last_error = "Unable to open the file for reading";
	    return(PSError);
	}

	status = PSReadRGBA(
	    fp,
	    width_rtn, height_rtn, bpp_rtn, bpl_rtn,
	    rgba_rtn,
	    bg_color,
	    x_rtn, y_rtn, base_width_rtn, base_height_rtn,
	    creator_rtn, title_rtn, author_rtn, comments_rtn,
	    def_alpha_value,
	    client_data, progress_cb, user_aborted
	);
	FClose(fp);

	return(status);
}


/*
 *	Write PostScript nexus.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
static int PSWriteFilePS3Any(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *img_data,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,		/* Client data */
		int, int,	/* Progress current & end values */
		int, int,	/* Width & height */
		int, int,	/* Bytes per line & bytes per pixel */
		u_int8_t *	/* Image data */
	),
	int *user_aborted,
	ps_visual visual
)
{
	const unsigned int flags = fmt->flags;
	int	bpp = 1,		/* Bytes per pixel of given buffer */
		bpp_out = 1,		/* Bytes per pixel of output data */
		dpi_x = PSDefaultDPIX,
		dpi_y = PSDefaultDPIY,
		paper_width = 320,
		paper_height = 240,
		paper_widthm,		/* DPI modified paper size */
		paper_heightm,
		output_x = 0,
		output_y = 0;

	if(flags & PSFlagDPIX)
	    dpi_x = fmt->dpi_x;
	if(flags & PSFlagDPIY)
	    dpi_y = fmt->dpi_y;

	if(flags & PSFlagPaperWidth)
	    paper_width = fmt->paper_width;
	if(flags & PSFlagPaperHeight)
	    paper_height = fmt->paper_height;

	if(flags & PSFlagOutputX)
	    output_x = fmt->output_x;
	if(flags & PSFlagOutputY)
	    output_y = fmt->output_y;

	/* Calculate paper size with DPI applied */
	paper_widthm = (int)(paper_width * dpi_x / 100);
	paper_heightm = (int)(paper_height * dpi_y / 100);

	/* Determine bytes per pixel from visual */
	switch(visual)
	{
	  case PSVisualRGBA:
	    bpp = 4;
	    bpp_out = 3;
	    break;
	  case PSVisualRGB:
	    bpp = 3;
	    bpp_out = 3;
	    break;
	  case PSVisualGrey:
	    bpp = 1;
	    bpp_out = 1;
	    break;
	}

	/* If the given bytes per line bpl is 0 then that implies they
	 * want us to calculate it. In which we will assume it is
	 * width * bpp
	 */
	if(bpl <= 0)
	    bpl = width * bpp;
	if(bpl <= 0)
	{
	    ps_last_error = "Invalid image bytes per line value";
	    return(PSBadValue);
	}

	/* Report initial progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		0, height,
		width, height,
		bpl, bpp,
		img_data
	    ))
		*user_aborted = 1;
	}

	/* Begin writing PostScript data to the stream fp */

	/* Header */

	/* Format identifier */
	fprintf(
	    fp,
"%%!PS-Adobe-3.0\n"
	);
	/* Creator */
	if(!STRISEMPTY(creator))
	{
	    int len;
	    char *s = STRDUP(creator), *s2;
	    for(s2 = s; *s2 != '\0'; s2++)
	    {
		if((*s2 == '\n') || (*s2 == '\r'))
		{
		    *s2 = '\0';
		    break;
		}
	    }
	    len = STRLEN(s);
	    if(len > 256)
		s[256] = '\0';
	    fprintf(
		fp,
"%%%%Creator: %s\n",
		s
	    );
	    free(s);
	}
	/* Title */
	if(!STRISEMPTY(title))
	{                  
	    int len;       
	    char *s = STRDUP(title), *s2;
	    for(s2 = s; *s2 != '\0'; s2++)
	    {
		if((*s2 == '\n') || (*s2 == '\r'))
		{
		    *s2 = '\0';
		    break;
		}
	    }
	    len = STRLEN(s);
	    if(len > 256)   
		s[256] = '\0'; 
	    fprintf(
		fp,
"%%%%Title: %s\n",
		s
	    );
	}
	/* Author */
	if(!STRISEMPTY(author))
	{
	    int len;
	    char *s = STRDUP(author), *s2;
	    for(s2 = s; *s2 != '\0'; s2++)
	    {
		if((*s2 == '\n') || (*s2 == '\r'))
		{
		    *s2 = '\0';
		    break;
		}   
	    }
	    len = STRLEN(s);
	    if(len > 256)
		s[256] = '\0';
	    fprintf(
		fp,
"%%%%Author: %s\n",
		s
	    );
	    free(s);
	}
	/* Comments */
	if(!STRISEMPTY(comments))
	{
	    int len;
	    char *s = STRDUP(comments), *s2;
	    for(s2 = s; *s2 != '\0'; s2++)
	    {
		if((*s2 == '\n') || (*s2 == '\r'))
		{
		    *s2 = '\0';
		    break;
		}
	    }
	    len = STRLEN(s);
	    if(len > 256)
		s[256] = '\0';
	    fprintf(
		fp,
"%%%%Comments: %s\n",
		s
	    );
	    free(s);
	}
	/* Document data type */
	fprintf(
	    fp,
"%%%%DocumentData: Clean7Bit\n"
	);

	/* Pages */
	fprintf(
	    fp,
"%%%%Pages: %i\n",
	    1
	);

	/* Bounding box (in units of pixels) */
	fprintf(
	    fp,
"%%%%BoundingBox: %i %i %i %i\n",
	    0, 0,
	    paper_widthm - 1, paper_heightm - 1
	);

	/* End comments */
	fprintf(
	    fp,
"%%%%EndComments\n"
	);


	/* Begin prolong */
	fprintf(
	    fp,
"%%%%BeginProlog\n\n"
	);

	fprintf(
	    fp,
"%% Use own dictionary to avoid conflicts\n"
	);
	fprintf(
	    fp,
"5 dict begin\n"
	);

	/* End prolong */
	fprintf(
	    fp,
"%%%%EndProlog\n\n"
	);


	/* Page set */
	fprintf(
	    fp,
"%%%%Page: %i %i\n\n",
	    1, 1
	);

	/* Translate for offset */
	fprintf(
	    fp,
"%% Translate for offset (origin at lower-left corner)\n"
	);     
	fprintf(
	    fp,
"%i %i translate\n",
	    output_x,
	    paper_heightm - output_y - height
	);

	/* Scale */
	fprintf(
	    fp,
"%% Scale of image from unit size\n"
	);
	fprintf( 
	    fp,
"%.5f %.5f scale\n\n",
	    (float)width, (float)height
	);

	/* Scan line buffer */
	fprintf(
	    fp,
"%% Buffer to store one scan line (width * bytes_per_pixel)\n"
	);
	fprintf(
	    fp,
"/scanline %i %i mul string def\n",
	    width, bpp_out
	);

	/* Image geometry */
	fprintf(
	    fp,
"%% Image geometry\n"
	);
	fprintf(
	    fp,
"%i %i %i\n",
	    width, height, 8
	);

	/* Transformatino matrix */
	fprintf(
	    fp,
"%% Transformation matrix\n"
	);
	fprintf(
	    fp,
"[ %i %i %i %i %i %i ]\n",
	    width,
	    0,
	    0,
	    height * -1,
	    0,
	    height
	);


	switch(visual)
	{
	  case PSVisualRGBA:
	  case PSVisualRGB:
	    fprintf(
		fp,
"{ currentfile scanline readhexstring pop }\n"
	    );
	    fprintf(
		fp,
"false 3 colorimage\n"
	    );
	    break;
	  case PSVisualGrey:
	    fprintf(
		fp,
"{ currentfile scanline readhexstring pop }\n"
	    );
	    fprintf(
		fp,
"image\n"
	    );
	    break;
	}

	/* Begin writing image data */
	if(!(*user_aborted))
	{
	    int		x, y,
			col = 0,
			line_max = 70;
	    const u_int8_t *src_ptr;

	    for(y = 0; y < height; y++)
	    {
		/* Report progress */
		if((progress_cb != NULL) && !(*user_aborted))
		{
		    if(!progress_cb(
			client_data,
			y, height,
			width, height,
			bpl, bpp,
			img_data
		    ))
		    {
			*user_aborted = 1;
			break;
		    }
		}

		for(x = 0; x < width; x++)
		{
		    src_ptr = img_data + (y * bpl) + (x * bpp);

		    /* Write pixel depending on visual */
		    switch(visual)
		    {
		      case PSVisualRGBA:
			/* Red */
			fprintf(
			    fp, "%.2x",
			    src_ptr[0]
			);
			/* Green */
			fprintf(
			    fp, "%.2x",
			    src_ptr[1]
			);
			/* Blue */
			fprintf(
			    fp, "%.2x",
			    src_ptr[2]
			);
			/* Skip alpha */

			col += 6;
			break;

		      case PSVisualRGB:
			/* Red */
			fprintf(fp, "%.2x",
			    src_ptr[0]
			);
			/* Green */
			fprintf(fp, "%.2x",
			    src_ptr[1]
			);
			/* Blue */
			fprintf(fp, "%.2x",
			    src_ptr[2]
			);

			col += 6;
			break;

		      case PSVisualGrey:
			/* Grey */
			fprintf(fp, "%.2x",
			    src_ptr[0]
			);

			col += 2;
			break;
		    }

		    /* Columns exceeded? */
		    if(col >= line_max)
		    {
			col = 0;
			fputc('\n', fp);
		    }
		}

		col = 0;
		fputc('\n', fp);
	    }
	}

	fputc('\n', fp);


	/* Footer */
	fprintf(
	    fp,
"showpage\n\
%%%%Trailer\n\
end\n\
%%%%EOF\n"
	);

	/* Report final progress */
	if((progress_cb != NULL) && !(*user_aborted))
	{
	    if(!progress_cb(
		client_data,
		height, height,
		width, height,
		bpl, bpp,
		img_data
	    ))     
		*user_aborted = 1;
	}

	if(*user_aborted)
	{
	    ps_last_error = "User aborted";
	    return(PSAbort);
	}
	else
	    return(PSSuccess);
}

/*
 *	Writes a PostScript image to the specified stream.
 *      
 *	The specified image data must be 4 bytes per pixel RGBA format
 *	and the PostScript image file will be written in RGB color.             
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
int PSWriteFilePS3RGBA(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,  /* Size of image in pixels */
	int bpl,                /* Bytes per line */
	u_int8_t *rgba,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,         /* Client data */
		int, int,       /* Progress current & end values */
		int, int,       /* Width & height */
		int, int,       /* Bytes per line & bytes per pixel */
		u_int8_t *      /* Image data */
	),
	int *user_aborted
)
{
	/* Reset globals */
	ps_last_error = NULL;

	if(fp == NULL)
	{
	    ps_last_error = "Stream pointer is NULL";
	    return(PSError);
	}
	if(fmt == NULL)
	{
	    ps_last_error = "PostScript format was not given";
	    return(PSBadValue);
	}
	if(rgba == NULL)
	{
	    ps_last_error = "Image data pointer is NULL";
	    return(PSBadValue);
	}
	if((width <= 0) || (height <= 0))
	{
	    ps_last_error = "Invalid image size";
	    return(PSBadValue);
	}

	return(PSWriteFilePS3Any(
	    fp, fmt,
	    width, height, bpl, rgba,
	    creator, title, author, comments,
	    client_data, progress_cb,
	    user_aborted,
	    PSVisualRGBA
	));
}

/*
 *	Writes a PostScript image to the specified stream.
 *      
 *	The specified image data must be 3 bytes per pixel RGB format
 *	and the PostScript image file will be written in RGB color.             
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
int PSWriteFilePS3RGB(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,  /* Size of image in pixels */
	int bpl,                /* Bytes per line */
	u_int8_t *rgb,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,         /* Client data */
		int, int,       /* Progress current & end values */
		int, int,       /* Width & height */
		int, int,       /* Bytes per line & bytes per pixel */
		u_int8_t *      /* Image data */
	),
	int *user_aborted
)
{
	/* Reset globals */
	ps_last_error = NULL;

	if(fp == NULL)
	{
	    ps_last_error = "Stream pointer is NULL";
	    return(PSError);
	}
	if(fmt == NULL)
	{
	    ps_last_error = "PostScript format was not given";
	    return(PSBadValue);
	}
	if(rgb == NULL)
	{
	    ps_last_error = "Image data pointer is NULL";
	    return(PSBadValue);
	}
	if((width <= 0) || (height <= 0))
	{
	    ps_last_error = "Invalid image size";   
	    return(PSBadValue);
	}

	return(PSWriteFilePS3Any(
	    fp, fmt,
	    width, height, bpl, rgb,
	    creator, title, author, comments,
	    client_data, progress_cb,
	    user_aborted,
	    PSVisualRGB
	));
}

/*
 *	Writes a PostScript image to the specified stream.
 *
 *	The specified image data must be 1 bytes per pixel Greyscale
 *	format and the PostScript image file will be written in RGB
 *	color.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
int PSWriteFilePS3Grey(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,  /* Size of image in pixels */
	int bpl,                /* Bytes per line */
	u_int8_t *grey,
	const char *creator, const char *title,
	const char *author, const char *comments,
	void *client_data,
	int (*progress_cb)(
		void *,         /* Client data */
		int, int,       /* Progress current & end values */
		int, int,       /* Width & height */
		int, int,       /* Bytes per line & bytes per pixel */
		u_int8_t *      /* Image data */
	),
	int *user_aborted
)
{
	/* Reset globals */
	ps_last_error = NULL;

	if(fp == NULL)
	{
	    ps_last_error = "Stream pointer is NULL";
	    return(PSError);
	}
	if(fmt == NULL)
	{              
	    ps_last_error = "PostScript format was not given";
	    return(PSBadValue);
	}
	if(grey == NULL)
	{
	    ps_last_error = "Image data pointer is NULL";
	    return(PSBadValue);
	}
	if((width <= 0) || (height <= 0))
	{
	    ps_last_error = "Invalid image size";   
	    return(PSBadValue);
	}

	return(PSWriteFilePS3Any(
	    fp, fmt,
	    width, height, bpl, grey,
	    creator, title, author, comments,
	    client_data, progress_cb,
	    user_aborted,
	    PSVisualGrey
	));
}
