#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <signal.h>
#include <math.h>
#if defined(HAVE_IMLIB)
# include <Imlib.h>
#elif defined(HAVE_IMLIB2)
# include <X11/Xlib.h>
# include <Imlib2.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#if defined(_WIN32)
# include <gdk/gdkwin32.h>
# include <direct.h>
#else
# include <gdk/gdkx.h>
# define HAVE_X
# include <unistd.h>
#endif
#ifdef HAVE_LIBENDEAVOUR2
# include <endeavour2.h>
#endif

#include "libps.h"
#include "imgio.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "cdialog.h"
#include "fprompt.h"
#include "csd.h"
#include "fsd.h"
#include "fb.h"
#include "pdialog.h"
#include "progressdialog.h"

#include "fullscreenimg.h"
#include "desktop.h"
#if !defined(_WIN32)
# include "screenshot.h"
# include "print_values.h"
# include "print.h"
#endif

#include "tview.h"
#include "hview.h"
#include "imgview.h"

#include "resizedlg.h"
#include "colorlevelsdlg.h"
#include "textdlg.h"
#include "headerdlg.h"
#include "framesdlg.h"
#if !defined(_WIN32)
# include "screenshotdlg.h"
#endif
#include "savedlg.h"
#include "imginfodlg.h"
#if !defined(_WIN32)
# include "print_dlg.h"
#endif

#include "ivtitle.h"

#include "config.h"
#include "compiletimeinfo.h"

#include "images/icon_edit_20x20.xpm"
#include "images/icon_resize_20x20.xpm"
#include "images/icon_color_levels_20x20.xpm"
#include "images/icon_fonts_20x20.xpm"
#include "images/icon_animate_20x20.xpm"
#include "images/icon_camera_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_save_as_20x20.xpm"
#include "images/icon_info_20x20.xpm"
#include "images/icon_monitor2_20x20.xpm"
#include "images/icon_desktop_20x20.xpm"
#include "images/icon_root_fit_20x20.xpm"
#include "images/icon_root_centered_20x20.xpm"
#include "images/icon_root_tiled_20x20.xpm"
#include "images/icon_root_cover_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_print2_20x20.xpm"
#include "images/icon_iv_20x20.xpm"

#include "images/icon_error_32x32.xpm"
#include "images/icon_edit_48x48.xpm"


#ifdef HAVE_IMLIB
/*
 *	Imlib handle:
 */
gpointer	imlib_handle = NULL;
#endif


/*
 *	Has changes string, displayed on title
 */
#define HAS_CHANGES_STR		"(*)"


typedef struct _iv_imgio_cb_struct	iv_imgio_cb_struct;
#define IV_IMGIO_CB(p)			((iv_imgio_cb_struct *)(p))
typedef struct _iv_info_struct		iv_info_struct;
#define IV_INFO(p)			((iv_info_struct *)(p))
typedef struct _iv_image_viewer_struct	iv_image_viewer_struct;
#define IV_IMAGE_VIEWER(p)		((iv_image_viewer_struct *)(p))
typedef struct _iv_data_viewer_struct	iv_data_viewer_struct;
#define IV_DATA_VIEWER(p)		((iv_data_viewer_struct *)(p))
typedef struct _iv_core_struct		iv_core_struct;
#define IV_CORE(p)			((iv_core_struct *)(p))


/*
 *	ImgIO Callback:
 */
struct _iv_imgio_cb_struct {

	GtkWidget		*ref_toplevel;
	iv_image_viewer_struct	*viewer;
	gchar			*path;

};

/*
 *	Image Info:
 */
struct _iv_info_struct {

	gchar		*filename;	/* Can be NULL if never been saved */
	gint		width,		/* Size in pixels */
			height;
	gint		nframes;
	gint		x, y;		/* Translation from upper-left in pixels */
	gint		base_width,	/* Base/boundingbox/screen size in pixels */
			base_height;
	gulong		filesize;	/* Size in bytes on disk, can be 0
					 * if not been saved yet */
	gboolean	has_alpha;	/* TRUE if alpha channel exists */
	gchar		*creator;	/* Program that created the image */
	gchar		*title;		/* Title of image */
	gchar		*author;	/* Author */
	gchar		*comments;	/* Comments/description of image */
	gulong		time_modified;	/* Last modified time in systime seconds*/

};

/*
 *	IV Image Viewer:
 */
struct _iv_image_viewer_struct {

	imgview_struct	*iv;
	iv_core_struct  *core;
	gboolean	processing,
			has_changes;

	/* Submenus */
	GtkWidget	*edit_submenu,
			*desktop_submenu;

	/* Menu items (added to the ImgView's menu) */
	GtkWidget	*set_bg_mi,
			*edit_mi,
			*resize_mi,
			*color_levels_mi,
			*add_text_mi,
			*edit_header_mi,
			*edit_frames_mi,
			*open_mi,
			*save_as_mi,
			*screenshot_mi,
			*information_mi,
			*fullscreen_mi,
			*desktop_mi,
			*clear_mi,
			*print_mi,
			*new_viewer_mi;

	/* Image info */
	iv_info_struct	info;

	/* Foreground & Background Colors
	 *
	 * 4 bytes in RGBA format
	 */
	guint8		fg_color[4],
			bg_color[4];

};

/*
 *	IV Data Viewer:
 */
struct _iv_data_viewer_struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	iv_core_struct	*core;

	tview_struct	*tv;
	hview_struct	*hv;

	GtkWidget	*text_tb,
			*hex_tb,
			*close_btn;
};

/*
 *	UV Core:
 */
struct _iv_core_struct {

	/* Current directory */
	gchar		*cur_dir;

	/* Image Viewers */
	GList		*iv_viewer;

	/* Data Viewers */
	GList		*iv_fviewer;

	/* Image Info Dialogs */
	GList		*imginfodlg;

	/* Close all windows marker */
	gboolean	close_all_windows;

	/* Untitled image counter, increments when a new image is
	 * created or obtained from a screenshot */
	gint		untitled_count;

	/* Cursors */
	GdkCursor	*grab_cur;

	/* GtkWindow used for grab window operations, pointer events are
	 * specified to be relative to this window
	 */
	GtkWidget	*grab_window;

	/* Options */
	gboolean	show_tooltips;

	/* Options set from the command line at startup, they do not
	 * reflect current values, so do not modify
	 *
	 * Check imgview for current values
	 */
	gboolean	show_toolbar,
			show_values,
			show_statusbar,
			show_image_on_wm_icon;
	gint		quality;		/* 0 to 2 */
	guint8		bg_color[4];		/* 4 bytes RGBA */
	gboolean	bg_color_set;		/* Background color set from command line? */
 	gchar		*font_name,		/* User defined font name */
 			*wmname,		/* User defined WM name */
			*wmclass,		/* User defined WM class */
			*title;			/* User defined title */
	GdkRectangle	*geometry;		/* User defined geometry */
	gboolean	resize_on_open,		/* Resize IV on load */
			resize_maintain_aspect;
	gint		imlib_fallback;		/* 0 = No Fallbacks
						 * 1 = Use ImageMagick & NETPBM */
	gboolean	auto_hide_iv_nonimage,
			simple_title;

	/* File Format Save Options */
	gboolean	gif_interlaced;
	gint		gif_format;		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	gboolean	gif_transparency,
			gif_looping;

	gfloat		jpeg_quality;		/* 0.0 to 1.0 */
	gboolean	jpeg_color;		/* TRUE = Color
						 * FALSE = Greyscale */

	gint		png_compression;	/* 0 to 9 */
	gint		png_format;		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
	gboolean	png_interlaced;

	gint		tga_format;		/* 0 = Greyscale
						 * 1 = RGB
						 * 2 = RGBA */

	gint		xpm_format;		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
	gint		xpm_max_colors;		/* -1 for no limit */
	guint8		xpm_threshold;		/* 0x00 to 0xff */

	/* Screenshot Options */
	gulong		screenshot_delay;	/* In milliseconds */
	gboolean	screenshot_hide_windows;
	gboolean	screenshot_include_decorations;

	/* Color Levels Options */
	gboolean	color_levels_all_frames;

	/* Add Text Options */
	gchar		*add_text_font_name;
	GdkColor	add_text_color;
	gboolean	add_text_outline;
	GdkColor	add_text_outline_color;
	gfloat		add_text_opacity;
	gboolean	add_text_all_frames;

	/* Print Options */
#if !defined(_WIN32)
	print_values_struct *print_values;
#endif

#ifdef HAVE_LIBENDEAVOUR2
	edv_context_struct	*edv2_ctx;
#endif

};


static iv_core_struct		*iv_core;


/* System Signal Callback */
static void iv_signal_cb(int s);

/* Image Viewer General Callbacks */
static gint iv_viewer_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint iv_viewer_key_event_cb(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);
static void iv_viewer_dnd_recieved_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);

/* Data Viewer General Callbacks */
static gint iv_fviewer_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void iv_fviewer_display_format_toggle_cb(
	GtkWidget *widget, gpointer data
);
static void iv_fviewer_close_cb(GtkWidget *widget, gpointer data);

/* Main Timeout Callback */
static gint iv_main_timeout_cb(gpointer data);

/* File Selector Callbacks */
static void iv_file_selector_created_cb(
	const gchar *path, gpointer data
);
static void iv_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
);
static void iv_file_selector_deleted_cb(
	const gchar *path, gpointer data
);

/* Core */
static void iv_core_reset(iv_core_struct *core);
static void iv_core_delete(iv_core_struct *core);

/* Image Viewer New/Delete */
static iv_image_viewer_struct *iv_viewer_new(
	iv_core_struct *core,
	const gboolean show_toolbar,
	const gboolean show_values,
	const gboolean show_statusbar,
	const gboolean show_image_on_wm_icon,
	const gint quality,		/* Image quality, 0 to 2 */
	const guint8 *bg_color,		/* 4 bytes RGBA */
	const GdkRectangle *geometry,
	const gboolean load_title
);
static void iv_viewer_delete(iv_image_viewer_struct *v);

/* Data Viewer New/Delete */
static iv_data_viewer_struct *iv_fviewer_new(
	iv_core_struct *core,
	GdkRectangle *geometry
);
static void iv_fviewer_delete(iv_data_viewer_struct *fv);

/* Image Viewer Operations */
static void iv_viewer_update_image_info(
	iv_image_viewer_struct *v,
	const gchar *path,		/* Can be NULL */
	const struct stat *stat_buf,	/* Can be NULL */
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gint nframes,
	const gboolean has_alpha,
	const gint x, const gint y,
	const gint base_width,		/* Can be 0 for no base */
	const gint base_height,
	const gchar *creator,		/* Can be NULL */
	const gchar *title,		/* Can be NULL */
	const gchar *author,		/* Can be NULL */
	const gchar *comments,		/* Can be NULL */
	const gulong time_modified	/* In systime seconds */
);
static void iv_viewer_open(
	iv_image_viewer_struct *v,
	const gchar *path,
	const gint frame_num,
	const gboolean resize_on_open
);
static void iv_viewer_save(
	iv_image_viewer_struct *v,
	const gchar *path,
	const gchar *ext
);

/* Command Line Only Operations */
static gint iv_open_fullscreen(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
);
static gint iv_open_to_root_centered(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
);
static gint iv_open_to_root_tofit(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
);
static gint iv_open_to_root_tocover(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
);
static gint iv_open_to_root_tiled(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
);
static gint iv_open_print(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color		/* 4 bytes in RGBA format */
);

/* Open With New Data Viewer */
static iv_data_viewer_struct *iv_fviewer_new_open_file(
	iv_core_struct *core,
	const gchar *path,
	const gboolean resize_on_open,
	GtkWidget *toplevel
);

/* Image Viewer Check Query Save Changes */
static gboolean iv_viewer_check_query_save_changes(iv_image_viewer_struct *v);

/* Image Viewer Callbacks */
static void iv_imgview_changed_cb(
	imgview_struct *iv,
	imgview_image_struct *image,
	gpointer data
);
static void iv_viewer_set_bg_color_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_resize_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_color_levels_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_add_text_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_header_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_frames_cb(GtkWidget *widget, gpointer data);
#if !defined(_WIN32)
static void iv_viewer_screenshot_cb(GtkWidget *widget, gpointer data);
#endif
static gint iv_open_progress_cb(
	gpointer data,
	gint min, gint max,		/* Progress position & maximum */
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba_data
);
static gint iv_save_progress_cb(
	gpointer data,
	gint min, gint max,		/* Progress position & maximum */
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba_data
);
static void iv_viewer_open_image_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_open_other_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_save_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_display_on_root_centered_cb(
	GtkWidget *widget, gpointer data
);
static void iv_viewer_display_on_root_tofit_cb(
	GtkWidget *widget, gpointer data
);
static void iv_viewer_display_on_root_tocover_cb(
	GtkWidget *widget, gpointer data
);
static void iv_viewer_display_on_root_tiled_cb(
	GtkWidget *widget, gpointer data
);
static void iv_viewer_clear_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_information_cb(GtkWidget *widget, gpointer data);
static void iv_viewer_fullscreen_cb(GtkWidget *widget, gpointer data);
#if !defined(_WIN32)
static void iv_viewer_print_cb(GtkWidget *widget, gpointer data);
#endif
static void iv_viewer_new_viewer_cb(GtkWidget *widget, gpointer data);

/* Image Viewer Updating */
static void iv_viewer_update_title(
	iv_image_viewer_struct *v, const gchar *new_path
);
static void iv_viewer_update(iv_image_viewer_struct *v);
static void iv_viewer_resize(
	iv_image_viewer_struct *v,
	const gint width, const gint height,
	const gboolean keep_visible_on_desktop
);
static void iv_clear_info(iv_info_struct *info);

/* Command Line Utilities */
static gboolean extcmp(const gchar *s, const gchar *ext);
static void get_args_value(
	char *arg, char **parm_rtn, char **val_rtn
);
static gboolean stristrue(const gchar *s);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ABSOLUTE_VALUE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define UNLINK(p)	((!STRISEMPTY(p)) ? unlink(p) : -1)


#define PRINT_ERROR_TO_DESKTOP(_s_)	{		\
 GdkWindow *root_win = GDK_ROOT_PARENT();		\
 GdkBitmap *mask;					\
 GdkPixmap *pixmap;					\
 GdkGC *gc = gdk_gc_new(root_win);			\
 GtkStyle *style = gtk_widget_get_default_style();	\
							\
 pixmap = gdk_pixmap_create_from_xpm_d(			\
  root_win, &mask,					\
  (style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,\
  (gchar **)icon_error_32x32_xpm			\
 );							\
							\
 iv_desktop_mesg(					\
  root_win, gc, style, GTK_JUSTIFY_LEFT,		\
  PROG_NAME_FULL " Version " PROG_VERSION,		\
  pixmap, mask, (_s_)					\
 );							\
							\
 GDK_GC_UNREF(gc);					\
 GDK_PIXMAP_UNREF(pixmap);				\
 GDK_BITMAP_UNREF(mask);				\
}


/*
 *	System signal callback.
 */
static void iv_signal_cb(int s)
{
	switch(s)
	{
#ifdef SIGHUP
	  case SIGHUP:
	    if(iv_core != NULL)
	    {



	    }
	    break;
#endif
#ifdef SIGINT
	  case SIGINT:
	    if(iv_core != NULL)
	    {
		iv_core_struct *core = iv_core;

		/* Is there a screenshot in progress? */
		if(core->grab_window != NULL)
		{
		    /* Destroy the grab window to break the grab */
		    gtk_widget_hide(core->grab_window);
		    GTK_WIDGET_DESTROY(core->grab_window);
		    core->grab_window = NULL;
		}

		core->close_all_windows = TRUE;
	    }
	    break;
#endif
#ifdef SIGTERM
	  case SIGTERM:
	    exit(0);
	    break;
#endif
#ifdef SIGQUIT
	  case SIGQUIT:
	    if(iv_core != NULL)
	    {
		iv_core_struct *core = iv_core;

		/* Is there a screenshot in progress? */
		if(core->grab_window != NULL)
		{
		    /* Destroy the grab window to break the grab */
		    gtk_widget_hide(core->grab_window);
		    GTK_WIDGET_DESTROY(core->grab_window);
		    core->grab_window = NULL;
		}

		core->close_all_windows = TRUE;
	    }
	    break;
#endif
#ifdef SIGSEGV
	  case SIGSEGV:
	    exit(1);
	    break;
#endif
#ifdef SIGPIPE
	  case SIGPIPE:
	    break;
#endif
	}
}


/*
 *	Image Viewer toplevel GtkWindow "delete_event" signal callback.
 */
static gint iv_viewer_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	imgview_struct *iv;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	iv_core_struct *core;
	if(v == NULL)
	    return(FALSE);

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return(TRUE);

	/* Check if there are any unsaved changes and query user to
	 * save those changes
	 */
	if(!iv_viewer_check_query_save_changes(v))
	    return(TRUE);

	/* Unmap the ImgView */
	ImgViewUnmap(iv);

	return(TRUE);
}

/*
 *	Image View "key_press_event" or "key_release_event" signal
 *	callback.
 */
static gint iv_viewer_key_event_cb(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gboolean keypress;
	guint keyval, keystate;
	imgview_struct *iv;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	iv_core_struct *core;
	if(v == NULL)
	    return(status);

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return(status);

	keypress = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	keystate = key->state;

	/* Handle by key value */
	switch(keyval)
	{
	  case GDK_q:
	    /* Exit/Quit */
	    if(keystate & GDK_CONTROL_MASK)
	    {
		if(keypress)
		{
		    GList *glist;

		    /* Check for any Image Viewer that still has
		     * unsaved changes
		     */
		    for(glist = core->iv_viewer;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			if(glist->data != NULL)
			{
			    iv_image_viewer_struct *v = IV_IMAGE_VIEWER(glist->data);

			    /* Check if there are any unsaved changes
			     * and query user to save those changes
			     */
			    if(!iv_viewer_check_query_save_changes(v))
				break;
			}
		    }
		    /* Set close_all_windows to TRUE only if there were
		     * no Image Viewers that have unsaved changes
		     */
		    core->close_all_windows = (glist == NULL) ? TRUE : FALSE;
		}
		status = TRUE;
	    }
	    break;

	  case GDK_Escape:
	    /* Close */
	    if(keypress)
		iv_viewer_delete_event_cb(
		    ImgViewGetToplevelWidget(iv), NULL, data
		);
	    status = TRUE;
	    break;

	  case GDK_w:
	    /* Close */
	    if(keystate & GDK_CONTROL_MASK)
	    {
		if(!keypress)
		    iv_viewer_delete_event_cb(
			ImgViewGetToplevelWidget(iv), NULL, data
		    );
		status = TRUE;
	    }
	    break;

	  case GDK_F3:
	    if(keystate & GDK_SHIFT_MASK)
	    {
		/* Open File */
		if(keypress)
		    iv_viewer_open_other_cb(v->open_mi, data);
		status = TRUE;
	    }
	    else
	    {
	        /* Open the image */
	        if(keypress)
		    iv_viewer_open_image_cb(v->open_mi, data);
		status = TRUE;
	    }
	    break;

	  case GDK_o:
	    if(keystate & GDK_SHIFT_MASK)
	    {
		/* Open Other File */
		if(keypress)
		    iv_viewer_open_other_cb(v->open_mi, data);
		status = TRUE;
	    }
	    break;

	  case GDK_a:
	    /* Save As Image */
	    if(keystate & GDK_CONTROL_MASK)
	    {
		if(keypress)
		    iv_viewer_save_cb(v->save_as_mi, data);
		status = TRUE;
	    }
	    break;

	  case GDK_p:
	    /* Play or pause */
	    if(keypress)
	    {
		const gint nframes = ImgViewGetTotalFrames(iv);
		if(ImgViewIsPlaying(iv))
		    ImgViewPause(iv);
		else if(nframes > 1)
		    ImgViewPlay(iv);
	    }
	    status = TRUE;
	    break;

	  case GDK_s:
	    /* Stop */
	    if(keypress)
	    {
		if(ImgViewIsPlaying(iv))
		    ImgViewPause(iv);
	    }
	    status = TRUE;
	    break;

	  case '[':
	  case '{':
	  case '<':
	  case ',':
	  case GDK_b:
	    /* Seek previous */
	    if(keypress)
	    {
		const gint nframes = ImgViewGetTotalFrames(iv);
		if(nframes > 1)
		{
		    gint frame_num = ImgViewGetCurrentFrame(iv);
		    frame_num--;
		    if(frame_num < 0)
			frame_num = nframes - 1;
		    ImgViewSeek(iv, frame_num);
		}
	    }
	    status = TRUE;
	    break;

	  case ']':
	  case '}':
	  case '>':
	  case '.':
	  case GDK_n:
	  case GDK_space:
	    /* Seek next */
	    if(keypress)
	    {
		const gint nframes = ImgViewGetTotalFrames(iv);
		if(nframes > 1)
		{ 
		    gint frame_num = ImgViewGetCurrentFrame(iv);
		    frame_num++;
		    if(frame_num >= nframes)
			frame_num = 0;
		    ImgViewSeek(iv, frame_num);
		}
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Image View "drag_data_recieved" signal callback.
 */
static void iv_viewer_dnd_recieved_cb(
	GtkWidget *widget, GdkDragContext *dc, gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	GtkWidget *w;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((widget == NULL) || (v == NULL) || (dc == NULL))
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	/* Mark viewer as processing */
	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;
	
	/* Get ImgView's view widget */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);

	/* View widget the same as the callback widget? */
	if(w == widget)
	{
	    /* Check if the DND target type is one that we support */
	    if((info == DND_INFO_TEXT_PLAIN) ||
	       (info == DND_INFO_TEXT_URI_LIST) ||
	       (info == DND_INFO_STRING)
	    )
	    {
		const gchar *s;
		const gchar *url = (const gchar *)selection_data->data;
		gchar *full_path;

		/* The DND data should be a URL string, we need to parse
		 * it and get the full path portion of that URL string.
		 */
		s = strstr(url, "://");
		if(s != NULL)
		    s += strlen("://");
		else
		    s = url;

		s = strchr(s, '/');
		full_path = STRDUP(s);

		/* Got the full path to object to be loaded? */
		if(full_path != NULL)
		{
		    gboolean status;

		    /* Set processing to FALSE so that the user
		     * can be queried to save changes
		     */
		    v->processing = FALSE;

		    /* Check if there are any unsaved changes and
		     * if so then query the user to save those changes
		     */
		    status = iv_viewer_check_query_save_changes(v);

		    v->processing = TRUE;

		    if(status)
			iv_viewer_open(
			    v,
			    full_path,
			    -1,			/* Frame unspecified */
			    core->resize_on_open
			);
		}

		g_free(full_path);
	    }
	}

	/* Mark viewer as no longer processing */
	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);

	/* Update Image Viewer's widgets */
	iv_viewer_update(v);
}


/*
 *	Data Viewer toplevel GtkWindow "delete_event" signal callback.
 */
static gint iv_fviewer_delete_event_cb(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	iv_data_viewer_struct *fv = IV_DATA_VIEWER(data);
	if(fv == NULL)
	    return(FALSE);

	if(fv->freeze_count > 0)
	    return(FALSE);

	/* Unmap the Data Viewer */
	gtk_widget_hide(fv->toplevel);

	return(TRUE);
}

/*
 *	Data Viewer Display Format GtkToggleButton "toggled" signal
 *	callback.
 */
static void iv_fviewer_display_format_toggle_cb(
	GtkWidget *widget, gpointer data
)
{
	GtkToggleButton *tb;
	tview_struct *tv;
	hview_struct *hv;
	iv_data_viewer_struct *fv = IV_DATA_VIEWER(data);
	if((widget == NULL) || (fv == NULL))
	    return;

	if(!GTK_IS_TOGGLE_BUTTON(widget) ||
	   (fv->freeze_count > 0)
	)
	    return;

	fv->freeze_count++;

	tb = GTK_TOGGLE_BUTTON(widget);

	/* Ignore untoggled signals */
	if(!gtk_toggle_button_get_active(tb))
	{
	    gtk_toggle_button_set_active(tb, TRUE);
	    fv->freeze_count--;
	    return;
	}

	tv = fv->tv;
	hv = fv->hv;

	/* Check which widget was toggled */
	if(widget == fv->text_tb)
	{
	    HViewUnmap(hv);
	    TViewMap(tv);
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(fv->hex_tb), FALSE
	    );
	}
	else if(widget == fv->hex_tb)
	{
	    TViewUnmap(tv);
	    HViewMap(hv);
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(fv->text_tb), FALSE
	    );
	}

	fv->freeze_count--;
}

/*
 *	Data Viewer close callback.
 */
static void iv_fviewer_close_cb(GtkWidget *widget, gpointer data)
{
	iv_data_viewer_struct *fv = IV_DATA_VIEWER(data);
	if(fv == NULL)
	    return;

	if(fv->freeze_count > 0)
	    return;

	gtk_widget_hide(fv->toplevel);
}


/*
 *	Main timeout callback.
 */
static gint iv_main_timeout_cb(gpointer data)
{
	gint windows = 0;
	gboolean close_all_windows;
	GtkWidget *grab_window;
	GList *glist;
	iv_core_struct *core = IV_CORE(data);
	if(core == NULL)
	    return(FALSE);

	close_all_windows = core->close_all_windows;

	/* Count Dialogs */
	if(CDialogIsQuery())
	    windows++;
	if(CSDIsQuery())
	    windows++;
	if(FSDIsQuery())
	    windows++;
	if(FileBrowserIsQuery())
	    windows++;
	if(PDialogIsQuery())
	    windows++;
	if(ProgressDialogIsQuery())
	    windows++;

	/* Count grab GtkWindow if it exists (mapped or not) */
	grab_window = core->grab_window;
	if(grab_window != NULL)
	    windows++;

	/* Manage Image Viewers */
	for(glist = core->iv_viewer;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data != NULL)
	    {
		iv_image_viewer_struct *v = IV_IMAGE_VIEWER(glist->data);
		imgview_struct *iv = v->iv;

		if(close_all_windows)
		{
		    iv_viewer_delete(v);
		    glist->data = NULL;
		    continue;
		}

		if((iv != NULL) ? iv->map_state : FALSE)
		{
		    windows++;
		}
		/* Delete the unmapped Image Viewer only when there
		 * is no grab window so that no Image Viewers get
		 * deleted during a screenshot
		 */
		else if(grab_window == NULL)
		{
		    iv_viewer_delete(v);
		    glist->data = NULL;
		    continue;
		}
	    }
	}

	/* Manage Data Viewers */
	for(glist = core->iv_fviewer;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data != NULL)
	    {
		iv_data_viewer_struct *fv = IV_DATA_VIEWER(glist->data);

		if(close_all_windows)
		{
		    iv_fviewer_delete(fv);
		    glist->data = NULL;
		    continue;
		}

		if(GTK_WIDGET_MAPPED(fv->toplevel))
		{
		    windows++;
		}
		else
		{
		    iv_fviewer_delete(fv);
		    glist->data = NULL;
		    continue;
		}
	    }
	}

	/* Count number of Image Info Dialogs */
	for(glist = core->imginfodlg;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data != NULL)
	    {
		imginfodlg_struct *d = IMGINFODLG(glist->data);

		if(close_all_windows)
		{
		    ImgInfoDlgUnmap(d);
		    ImgInfoDlgDelete(d);
		    glist->data = NULL;
		    continue;
		}

		if(ImgInfoDlgIsMapped(d))
		{
		    windows++;
		}
		else
		{
		    ImgInfoDlgUnmap(d);
		    ImgInfoDlgDelete(d);
		    glist->data = NULL;
		    continue;
		}
	    }
	}

	/* If there are no windows left then break out of the main
	 * loop
	 */
	if(windows == 0)
	{
	    gtk_main_quit();
	    return(FALSE);
	}

	return(TRUE);
}


/*
 *	File selector created callback.
 */
static void iv_file_selector_created_cb(
	const gchar *path, gpointer data
)
{
	iv_core_struct *core = IV_CORE(data);
#ifdef HAVE_LIBENDEAVOUR2
	EDVNotifyQueueObjectAdded(core->edv2_ctx, path);
	EDVContextFlush(core->edv2_ctx);
#endif
}

/*
 *	File selector changed callback.
 */
static void iv_file_selector_modified_cb(
	const gchar *old_path,
	const gchar *new_path,
	gpointer data
)
{
	iv_core_struct *core = IV_CORE(data);
#ifdef HAVE_LIBENDEAVOUR2
	EDVNotifyQueueObjectModified(core->edv2_ctx, old_path, new_path);
	EDVContextFlush(core->edv2_ctx);
#endif
}

/*
 *	File selector deleted callback.
 */
static void iv_file_selector_deleted_cb(
	const gchar *path, gpointer data
)
{
	iv_core_struct *core = IV_CORE(data);
#ifdef HAVE_LIBENDEAVOUR2
	EDVNotifyQueueObjectRemoved(core->edv2_ctx, path);
	EDVContextFlush(core->edv2_ctx);
#endif
}


/*
 *	Resets the IV Core's configuration to default values.
 */
static void iv_core_reset(iv_core_struct *core)
{
	guint8 bg_color[4] = { 0x00, 0x00, 0x00, 0xff };
#if !defined(_WIN32)
	print_values_struct *pv;
#endif

	if(core == NULL)
	    return;

	g_free(core->cur_dir);
	core->cur_dir = STRDUP(g_get_current_dir());
	core->close_all_windows = FALSE;
	core->untitled_count = 0;

	/* Options */
	core->show_tooltips = TRUE;
	core->show_toolbar = TRUE;
	core->show_values = TRUE;
	core->show_statusbar = TRUE;
	core->show_image_on_wm_icon = FALSE;
	core->quality = 1;
	memcpy(core->bg_color, bg_color, sizeof(bg_color));
	core->bg_color_set = FALSE;
	g_free(core->font_name);
	core->font_name = NULL;
	g_free(core->wmname);
	core->wmname = NULL;
	g_free(core->wmclass);
	core->wmclass = NULL;
	g_free(core->title);
	core->title = NULL;
	g_free(core->geometry);
	core->geometry = NULL;

	core->resize_on_open = TRUE;
	core->resize_maintain_aspect = TRUE;

	core->imlib_fallback = 1;

	core->auto_hide_iv_nonimage = FALSE;
	core->simple_title = FALSE;

	/* Save Options */
	core->gif_interlaced = GIF_DEF_INTERLACED;
	core->gif_format = GIF_DEF_FORMAT;
	core->gif_transparency = GIF_DEF_TRANSPARENCY;
	core->gif_looping = GIF_DEF_LOOPING;
	core->jpeg_quality = JPEG_DEF_QUALITY;
	core->jpeg_color = JPEG_DEF_IS_COLOR;
	core->png_compression = PNG_DEF_COMPRESSION;
	core->png_format = PNG_DEF_FORMAT;
	core->png_interlaced = PNG_DEF_INTERLACED;
	core->tga_format = TGA_DEF_FORMAT;
	core->xpm_format = XPM_DEF_FORMAT;
	core->xpm_max_colors = XPM_DEF_MAX_COLORS;
	core->xpm_threshold = XPM_DEF_THRESHOLD;

	/* Screenshot Options */
	core->screenshot_delay = SCREENSHOT_DEF_DELAY;
	core->screenshot_hide_windows = SCREENSHOT_DEF_HIDE_WINDOWS;
	core->screenshot_include_decorations = SCREENSHOT_DEF_INCLUDE_DECORATIONS;

	/* Color Levels Options */
	core->color_levels_all_frames = COLOR_LEVELS_DEF_ALL_FRAMES;

	/* Add Text Options */
	g_free(core->add_text_font_name);
	core->add_text_font_name = STRDUP(ADD_TEXT_DEF_FONT_NAME);
	{ GdkColor c = ADD_TEXT_DEF_COLOR;
	  memcpy(&core->add_text_color, &c, sizeof(GdkColor));
	}
	core->add_text_outline = ADD_TEXT_DEF_OUTLINE;
	{ GdkColor c = ADD_TEXT_DEF_OUTLINE_COLOR;
	  memcpy(&core->add_text_outline_color, &c, sizeof(GdkColor));
	}
	core->add_text_opacity = ADD_TEXT_DEF_OPACITY;
	core->add_text_all_frames = ADD_TEXT_DEF_ALL_FRAMES;

#if !defined(_WIN32)
	/* Print Options */
	PrintValuesDelete(core->print_values);
	core->print_values = pv = PrintValuesNew();
	PrintValuesReset(pv);
#endif	/* !_WIN32 */
}

/*
 *	Deletes the IV Core.
 */
static void iv_core_delete(iv_core_struct *core)
{
	if(core == NULL)
	    return;

	g_free(core->cur_dir);

	/* Delete all the Image Viewers */
	g_list_foreach(
	    core->iv_viewer, (GFunc)iv_viewer_delete, NULL
	);
	g_list_free(core->iv_viewer);

	/* Delete all the Data Viewers */
	g_list_foreach(
	    core->iv_fviewer, (GFunc)iv_fviewer_delete, NULL
	);
	g_list_free(core->iv_fviewer);

	/* Delete all the Image Info Dialogs */
	g_list_foreach(
	    core->imginfodlg, (GFunc)ImgInfoDlgDelete, NULL
	);
	g_list_free(core->imginfodlg);

	/* Grab GtkWindow */
	if(core->grab_window != NULL)
	{
	    gtk_widget_hide(core->grab_window);
	    GTK_WIDGET_DESTROY(core->grab_window);
	}

	/* Destroy all GdkCursors */
	GDK_CURSOR_DESTROY(core->grab_cur);

	g_free(core->add_text_font_name);

	PrintValuesDelete(core->print_values);

	g_free(core->font_name);
	g_free(core->wmname);
	g_free(core->wmclass);
	g_free(core->title);
	g_free(core->geometry);

#ifdef HAVE_LIBENDEAVOUR2
	EDVContextDelete(core->edv2_ctx);
#endif

	g_free(core);
}


/*
 *	Creates a new Image Viewer and adds it to the windows list.
 */
static iv_image_viewer_struct *iv_viewer_new(
	iv_core_struct *core,
	const gboolean show_toolbar,
	const gboolean show_values,
	const gboolean show_statusbar,
	const gboolean show_image_on_wm_icon,
	const gint quality,		/* Image quality, 0 to 2 */
	const guint8 *bg_color,		/* 4 bytes RGBA */
	const GdkRectangle *geometry,
	const gboolean load_title
)
{
	GList *glist;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent;
	GtkWidget *toplevel = NULL;
	GtkWidget *iv_toplevel, *menu;
	imgview_struct *iv = NULL;
	iv_image_viewer_struct *v = NULL;


	/* Create new Image Viewer */
	v = IV_IMAGE_VIEWER(g_malloc0(sizeof(iv_image_viewer_struct)));
	v->core = core;
	v->processing = FALSE;
	v->has_changes = FALSE;
	v->fg_color[0] = 0x00;
	v->fg_color[1] = 0x00;
	v->fg_color[2] = 0x00;
	v->fg_color[3] = 0xff;
	if(bg_color != NULL)
	{
	    memcpy(v->bg_color, bg_color, 4 * sizeof(guint8));
	}
	else
	{
	    v->bg_color[0] = 0xff;
	    v->bg_color[1] = 0xff;
	    v->bg_color[2] = 0xff;
	    v->bg_color[3] = 0xff;
	}
	iv_clear_info(&v->info);

	/* Begin creating image viewer */

	/* Create ImgView as a toplevel GtkWindow */
	v->iv = iv = ImgViewNew(
	    show_toolbar,		/* Show Tool Bar */
	    show_values,		/* Show values */
	    show_statusbar,		/* Show Status Bar */
	    show_image_on_wm_icon,	/* Show image on WM icon */
	    quality,			/* Quality, 0 to 2 */
	    TRUE,			/* Toplevel is to be a GtkWindow */
	    &iv_toplevel
	);
	if(iv == NULL)
	{
	    g_free(v);
	    return(NULL);
	}

	/* Get accelgrp */
	accelgrp = ImgViewGetAccelGroup(iv);

	/* Get the ImgView's toplevel GtkWidget */
	parent = w = toplevel = iv_toplevel;

	gtk_widget_set_name(w, IV_IMGVIEW_TOPLEVEL_WIDGET_NAME);

	/* Allow cropping */
	ImgViewAllowCrop(iv, TRUE);

	/* Set the ImgView's changed callback */
	ImgViewSetChangedCB(
	    iv, iv_imgview_changed_cb, v
	);

	/* Set the background color */
	if(v->bg_color != NULL)
	{
	    gint i;
	    GdkColor c[5];
	    for(i = 0; i < 5; i++)
	    {
		GDK_COLOR_SET_BYTE(
		    &(c[i]),
		    v->bg_color[0],
		    v->bg_color[1],
		    v->bg_color[2]
		);
	    }
	    ImgViewSetViewBG(iv, c);
	}

	/* Set up the ImgView's toplevel GtkWindow */
	w = iv_toplevel;
	if(w != NULL)
	{
	    /* Font */
	    if(core->font_name != NULL)
	    {
		GtkRcStyle *rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(core->font_name);
		gtk_widget_modify_style_recursive(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
	    }

	    /* Geometry */
	    if(geometry != NULL)
	    {
		gtk_widget_set_uposition(
		    w, geometry->x, geometry->y
		);
		gtk_widget_set_usize(
		    w, geometry->width, geometry->height
		);
	    }

	    /* Title */
	    gtk_window_set_title(
		GTK_WINDOW(w),
		(core->title != NULL) ?
		    core->title :
		    PROG_NAME_FULL " Version " PROG_VERSION
	    );

	    /* Events */
	    gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(iv_viewer_delete_event_cb), v
	    );
	}

	/* Set up ImgView's view GtkDrawingArea */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w != NULL)
	{
	    /* Drag & Drop */
	    const GtkTargetEntry dnd_tar_types[] = {
		{"text/plain",		0,	DND_INFO_TEXT_PLAIN},
		{"text/uri-list",	0,	DND_INFO_TEXT_URI_LIST},
		{"STRING",		0,	DND_INFO_STRING}
	    };
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_DEFAULT |		/* Drag actions */
		GDK_ACTION_LINK |
		GDK_ACTION_MOVE |
		GDK_ACTION_COPY,
		GDK_ACTION_MOVE,		/* Def act if same */
		GDK_ACTION_COPY,		/* Def act */
		iv_viewer_dnd_recieved_cb,
		v
	    );

	    /* Events */
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(iv_viewer_key_event_cb), v
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(iv_viewer_key_event_cb), v
	    );
	}


	/* Get Image Viewer's GtkMenu */
	menu = (GtkWidget *)ImgViewGetMenuWidget(iv);
	if(menu != NULL)
	{
	    GtkWidget *submenu;
	    guint accel_key, accel_mods;
	    guint8 **icon;
	    const gchar *label;
	    gpointer mclient_data = v;
	    void (*func_cb)(GtkWidget *w, gpointer) = NULL;

#define DO_ADD_MENU_ITEM_LABEL		{	\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  mclient_data, func_cb				\
 );						\
}
#define DO_ADD_MENU_ITEM_SUBMENU	{	\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SUBMENU, accelgrp,\
  icon, label, accel_key, accel_mods, NULL,	\
  mclient_data, func_cb				\
 );						\
 if(w != NULL)					\
  GUIMenuItemSetSubMenu(w, submenu);		\
}
#define DO_ADD_MENU_SEP			{	\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}

	    icon = NULL;
	    label = "Background Color...";
	    accel_key = GDK_b;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_set_bg_color_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->set_bg_mi = w;

	    /* Edit Submenu */
	    v->edit_submenu = submenu = GUIMenuCreate();
	    if(submenu != NULL)
	    {
		GtkWidget *menu;	/* Overload */

		menu = submenu;

	        icon = (guint8 **)icon_resize_20x20_xpm;
	        label = "Resize...";
	        accel_key = GDK_r;
	        accel_mods = GDK_CONTROL_MASK;
	        func_cb = iv_viewer_resize_cb;
	        DO_ADD_MENU_ITEM_LABEL
	        v->resize_mi = w;

		icon =  (guint8 **)icon_color_levels_20x20_xpm;
		label = "Color Levels...";
		accel_key = GDK_l;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = iv_viewer_color_levels_cb;
		DO_ADD_MENU_ITEM_LABEL
		v->color_levels_mi = w;

		icon = (guint8 **)icon_fonts_20x20_xpm;
		label = "Add Text...";
		accel_key = GDK_t;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = iv_viewer_add_text_cb;
		DO_ADD_MENU_ITEM_LABEL
		v->add_text_mi = w;

	        icon = (guint8 **)icon_edit_20x20_xpm;
	        label = "Header...";
	        accel_key = GDK_h;
	        accel_mods = GDK_CONTROL_MASK;
	        func_cb = iv_viewer_header_cb;
	        DO_ADD_MENU_ITEM_LABEL
	        v->edit_header_mi = w;

		icon = (guint8 **)icon_animate_20x20_xpm;
		label = "Frames...";
		accel_key = GDK_f;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = iv_viewer_frames_cb;
		DO_ADD_MENU_ITEM_LABEL
		v->edit_frames_mi = w;
	    }
	    icon = (guint8 **)icon_edit_20x20_xpm;
	    label = "Edit";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = NULL;
	    DO_ADD_MENU_ITEM_SUBMENU
	    v->edit_mi = w;

	    DO_ADD_MENU_SEP

	    icon = (guint8 **)icon_open_20x20_xpm;
	    label = "Open...";
	    accel_key = GDK_o;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_open_image_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->open_mi = w;

	    icon = (guint8 **)icon_save_as_20x20_xpm;
	    label = "Save As...";
	    accel_key = GDK_s;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_save_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->save_as_mi = w;

	    DO_ADD_MENU_SEP

#if !defined(_WIN32)
	    icon = (guint8 **)icon_camera_20x20_xpm;
	    label = "Screenshot...";
	    accel_key = GDK_g;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_screenshot_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->screenshot_mi = w;

	    DO_ADD_MENU_SEP
#endif

	    icon = (guint8 **)icon_info_20x20_xpm;
	    label = "Information";
	    accel_key = GDK_i;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_information_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->information_mi = w;

	    icon = (guint8 **)icon_monitor2_20x20_xpm;
	    label = "Fullscreen";
	    accel_key = GDK_F11;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_fullscreen_cb;
	    DO_ADD_MENU_ITEM_LABEL 
	    v->fullscreen_mi = w; 

	    /* Desktop submenu */
	    v->desktop_submenu = submenu = GUIMenuCreate();
	    if(submenu != NULL)
	    {
		GtkWidget *menu;	/* Overload */

		menu = submenu;

		icon = (guint8 **)icon_root_centered_20x20_xpm;
		label = "Centered";
		accel_key = GDK_d;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = iv_viewer_display_on_root_centered_cb;
		DO_ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_root_fit_20x20_xpm;
		label = "To Fit";
		accel_key = GDK_d;
		accel_mods = GDK_MOD1_MASK | GDK_CONTROL_MASK;
		func_cb = iv_viewer_display_on_root_tofit_cb;
		DO_ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_root_cover_20x20_xpm;
		label = "To Cover";
		accel_key = GDK_d;
		accel_mods = GDK_SHIFT_MASK | GDK_CONTROL_MASK;
		func_cb = iv_viewer_display_on_root_tocover_cb;
		DO_ADD_MENU_ITEM_LABEL

		icon = (guint8 **)icon_root_tiled_20x20_xpm;
		label = "Tiled";
		accel_key = GDK_d;
		accel_mods = GDK_CONTROL_MASK;
		func_cb = iv_viewer_display_on_root_tiled_cb;
		DO_ADD_MENU_ITEM_LABEL
	    }
	    icon = (guint8 **)icon_desktop_20x20_xpm;
	    label = "Desktop";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = NULL;
	    DO_ADD_MENU_ITEM_SUBMENU
	    v->desktop_mi = w;

	    icon = (guint8 **)icon_clear_20x20_xpm;
	    label = "Clear";
	    accel_key = GDK_k;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_clear_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->clear_mi = w;

	    DO_ADD_MENU_SEP

#if !defined(_WIN32)
	    icon = (guint8 **)icon_print2_20x20_xpm;
	    label = "Print...";
	    accel_key = GDK_p;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_print_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->print_mi = w;

	    DO_ADD_MENU_SEP
#endif

	    icon = (guint8 **)icon_iv_20x20_xpm;
	    label = "New Image Viewer";
	    accel_key = GDK_1;
	    accel_mods = GDK_CONTROL_MASK;
	    func_cb = iv_viewer_new_viewer_cb;
	    DO_ADD_MENU_ITEM_LABEL
	    v->new_viewer_mi = w;

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_SUBMENU
#undef DO_ADD_MENU_SEP
	}


	/* Record window */
	for(glist = core->iv_viewer;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data == NULL)
	    {
		glist->data = v;
		break;
	    }
	}
	if(glist == NULL)
	    core->iv_viewer = g_list_append(core->iv_viewer, v);


	/* Load title image? */
	if(load_title)
	    iv_load_title(
		iv, core->resize_on_open, core->simple_title
	    );

	/* Update Image Viewer's widgets */
	iv_viewer_update(v);

	/* Map the ImgView and the toplevel GtkWindow */
	ImgViewMap(iv);
	gtk_widget_show(toplevel);

	/* Have the ImgView's view widget grab focus */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w != NULL)
	    gtk_widget_grab_focus(w);

	return(v);
}

/*
 *	Deletes the Image Viewer.
 */
static void iv_viewer_delete(iv_image_viewer_struct *v)
{
	imgview_struct *iv;

	if(v == NULL)
	    return;

	iv = v->iv;
	if(ImgViewIsPlaying(iv))
	    ImgViewPause(iv);

	/* Begin destroying widgets that we created */

	v->set_bg_mi = NULL;
	v->edit_mi = NULL;
	v->resize_mi = NULL;
	v->color_levels_mi = NULL;
	v->add_text_mi = NULL;
	v->edit_header_mi = NULL;
	v->edit_frames_mi = NULL;
	v->open_mi = NULL;
	v->save_as_mi = NULL;
	v->information_mi = NULL;
	v->desktop_mi = NULL;
	v->clear_mi = NULL;
	v->screenshot_mi = NULL;
	v->print_mi = NULL;
	v->new_viewer_mi = NULL;

	/* Submenus */
	GTK_WIDGET_DESTROY(v->edit_submenu);
	v->edit_submenu = NULL;
	GTK_WIDGET_DESTROY(v->desktop_submenu);
	v->desktop_submenu = NULL;

	/* Delete the ImgView */
	ImgViewUnmap(iv);
	ImgViewDelete(iv);
	v->iv = iv = NULL;

	/* Delete the image info values */
	iv_clear_info(&v->info);

	g_free(v);
}


/*
 *	Creates a new Data Viewer and adds it to the windows list.
 */
static iv_data_viewer_struct *iv_fviewer_new(
	iv_core_struct *core,
	GdkRectangle *geometry
)
{
	const gint border_major = 5;
	GList *glist;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*toplevel;
	tview_struct *tv;
	hview_struct *hv;
	iv_data_viewer_struct *fv = IV_DATA_VIEWER(
	    g_malloc0(sizeof(iv_data_viewer_struct))
	);
	if(fv == NULL)
	    return(NULL);

	fv->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	fv->accelgrp = accelgrp = gtk_accel_group_new();
	fv->freeze_count = 0;
	fv->core = core;

	fv->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_widget_set_name(w, IV_DATAVIEW_TOPLEVEL_WIDGET_NAME);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	if(geometry != NULL)
	{
	    gtk_widget_set_usize(w, geometry->width, geometry->height);
	    gtk_widget_set_uposition(w, geometry->x, geometry->y);
	}
	else
	{
	    gtk_widget_set_usize(w, 640, 480);
	}
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w),
	    (core->wmname != NULL) ? core->wmname : "dataview",
	    (core->wmclass != NULL) ? core->wmclass : PROG_NAME
	);
#endif
#ifdef PROG_NAME
	gtk_window_set_title(
	    GTK_WINDOW(w),
	    (core->title != NULL) ? core->title : PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL) 
	{
	    GdkGeometry geometry;

	    geometry.min_width = 100;
	    geometry.min_height = 70;
	    geometry.base_width = 0;
	    geometry.base_height = 0;
	    geometry.width_inc = 1;
	    geometry.height_inc = 1;
	    gdk_window_set_geometry_hints(
		window, &geometry,
		GDK_HINT_MIN_SIZE |
		GDK_HINT_BASE_SIZE |
		GDK_HINT_RESIZE_INC
	    );

	    GUISetWMIcon( 
		window,
		(guint8 **)icon_edit_48x48_xpm
	    );
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(iv_fviewer_delete_event_cb), fv
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Text View & Hex View GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Create the Text View */
	fv->tv = tv = TViewNew(parent2);
	gtk_widget_set_name(tv->text, IV_DATAVIEW_TEXT_VIEW_WIDGET_NAME);
	TViewSetReadOnly(tv, TRUE);

	/* Create the Hex View */
	fv->hv = hv = HViewNew(parent2);
	gtk_widget_set_name(hv->view_da, IV_DATAVIEW_HEX_VIEW_WIDGET_NAME);
	HViewShowStatusBar(hv, FALSE);
	HViewSetReadOnly(hv, TRUE);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Text & Hex toggle buttons GtkVBox */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Text & Hex toggle buttons GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Text GtkToggleButton */
	fv->text_tb = w = GUIToggleButtonPixmapLabelH(
	    NULL, "ASCII", NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(iv_fviewer_display_format_toggle_cb), fv
	);
	gtk_widget_show(w);
	/* Hex GtkToggleButton */
	fv->hex_tb = w = GUIToggleButtonPixmapLabelH(
	    NULL, "Hex", NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(iv_fviewer_display_format_toggle_cb), fv
	);
	gtk_widget_show(w);

	/* Close Button */
	fv->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
	    "Close",
	    NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(iv_fviewer_close_cb), fv
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);


	/* Record window */
	for(glist = core->iv_fviewer;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    if(glist->data == NULL)
	    {
		glist->data = fv;
		break;
	    }
	}
	if(glist == NULL)
	    core->iv_fviewer = g_list_append(core->iv_fviewer, fv);

	/* Map the Data Viewer */
	gtk_widget_show_raise(toplevel);

	fv->freeze_count--;

	return(fv);
}

/*
 *	Deletes the Data Viewer.
 */
static void iv_fviewer_delete(iv_data_viewer_struct *fv)
{
	if(fv == NULL)
	    return;

	if(fv->toplevel != NULL)
	    gtk_widget_hide(fv->toplevel);

	TViewDelete(fv->tv);
	HViewDelete(fv->hv);
	GTK_WIDGET_DESTROY(fv->text_tb);
	GTK_WIDGET_DESTROY(fv->hex_tb);
	GTK_WIDGET_DESTROY(fv->close_btn);
	GTK_WIDGET_DESTROY(fv->toplevel);
	GTK_ACCEL_GROUP_UNREF(fv->accelgrp);
	g_free(fv);
}


/*
 *	Update Image Viewer's image info.
 */
static void iv_viewer_update_image_info(
	iv_image_viewer_struct *v,
	const gchar *path,			/* Can be NULL */
	const struct stat *stat_buf,		/* Can be NULL */
	const gint width, const gint height,
	const gint bpp, const gint bpl,
	const gint nframes,
	const gboolean has_alpha,
	const gint x, const gint y,
	const gint base_width,			/* Can be 0 for no base */
	const gint base_height,
	const gchar *creator,			/* Can be NULL */
	const gchar *title,			/* Can be NULL */
	const gchar *author,			/* Can be NULL */
	const gchar *comments,			/* Can be NULL */
	const gulong time_modified		/* In systime seconds */
)
{
	iv_info_struct *info;

	if(v == NULL)
	    return;

	info = &v->info;

	/* Delete any existing info values */
	iv_clear_info(info);

	/* Set the file name */
	g_free(info->filename);
	info->filename = STRDUP(path);

	/* Get the file information if specified */
	if(stat_buf != NULL)
	{
	    info->filesize = (gulong)stat_buf->st_size;
	    /* Do not get time modified from the file stats */
	}

	/* Set the rest of the info values */
	info->width = width;
	info->height = height;
	info->nframes = nframes;
	info->has_alpha = has_alpha;
	info->x = x;
	info->y = y;
	info->base_width = base_width;
	info->base_height = base_height;

	/* Set the creator only if changed */
	if(creator != info->creator)
	{
	    g_free(info->creator);
	    info->creator = STRDUP(creator);
	}
	/* Set the title only if changed */
	if(title != info->title)
	{
	    g_free(info->title);
	    info->title = STRDUP(title);
	}
	/* Set the author only if changed */
	if(author != info->author)
	{
	    g_free(info->author);
	    info->author = STRDUP(author);
	}
	/* Set the comments only if changed */
	if(comments != info->comments)
	{
	    g_free(info->comments);
	    info->comments = STRDUP(comments);
	}

	info->time_modified = time_modified;
}


/*
 *	Opens the image to the Image Viewer.
 *
 *	The path specifies the full path to the image.
 *
 *	The frame_num specifies the frame to initially display or -1
 *	for unspecified (in which case if there are multiple frames
 *	in the image then the Image Viewer will automatically start
 *	playing).
 *
 *	If resize_on_open is TRUE then the Image Viewer will be
 *	resized to fit the opened image.
 */
static void iv_viewer_open(
	iv_image_viewer_struct *v,
	const gchar *path,
	const gint frame_num,
	const gboolean resize_on_open
)
{
	gboolean use_file_viewer = FALSE;
	gint		i, status,
			x, y, width, height,
			bpp, bpl, nframes,
			base_width, base_height;
	guint8		**data_rgba,
			old_bg_color[4];
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	struct stat stat_buf;
	GtkWidget *toplevel, *w;
	imgview_struct *iv;
	iv_imgio_cb_struct *cb_data;
	iv_core_struct *core;

	if((v == NULL) || STRISEMPTY(path))
	    return;

	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Get the ImgView's view widget */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w == NULL)
	    return;

	/* Check if the image file does not exist */
	if(stat(path, &stat_buf))
	{
	    const gint error_code = (gint)errno;
	    gchar	*s = STRDUP(g_strerror(error_code)),
			*msg;
	    if(s == NULL)
		s = STRDUP("No such file");
	    *s = toupper(*s);
	    msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		s, path
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Open Failed",
		msg,
"Please check if the specified file exists and that\n\
permissions allow the reading of that file.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    g_free(s);
	    return;
	}
#ifdef S_ISDIR
	/* Check if the specified path is a directory */
	if(S_ISDIR(stat_buf.st_mode))
	{
	    gchar *buf = g_strdup_printf(
"The specified file is a directory:\n\
\n\
    %s",
		path
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Open Failed",
		buf,
"The specified path refers to a directory and\n\
directories may not be opened as files.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    return;
	}
#endif  /* S_ISDIR */

	/* Record the current background color */
	memcpy(old_bg_color, v->bg_color, sizeof(old_bg_color));

	/* Clear the existing image (if any) on Image Viewer so that
	 * the ImgView is cleared and redrawn
	 *
	 * Note that the redraw event will be handled when the events
	 * are handled in the progress callback while opening the
	 * image below
	 */
	ImgViewClear(iv);

	/* Clear image info */
	iv_clear_info(&v->info);

	/* Reset has changes mark */
	v->has_changes = FALSE;


	/* Allocate the open image callback data */
	cb_data = IV_IMGIO_CB(g_malloc0(sizeof(iv_imgio_cb_struct)));
	cb_data->ref_toplevel = toplevel;
	cb_data->viewer = v;
	cb_data->path = STRDUP(path);

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &data_rgba, &delay_list,
	    &nframes,
	    v->bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,			/* Default alpha value */
	    cb_data,			/* Callback data */
	    iv_open_progress_cb		/* Callback function */
	);

	/* Delete the open image callback data */
	g_free(cb_data->path);
	g_free(cb_data);
	cb_data = NULL;

	/* Reset the ImgView's progress bar */
	ImgViewProgressUpdate(iv, 0.0f, FALSE);

	/* Warn if the opened image data was not RGBA */
	if(((status == 0) || (status == -4)) && (bpp != 4))
	{
	    gchar *msg = g_strdup_printf(
"The image data that was opened is %i bytes per pixel,\n\
the image data must be 4 bytes per pixel RGBA format\n\
in order for it to be displayed.\n\
\n\
The image data will be displayed in a Data Viewer\n\
instead.",
		bpp
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Warning",
		msg, NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL); 
	    g_free(msg);
	}

	/* Loaded image data successfully? */
	if((data_rgba != NULL) && (bpp == 4))
	{
	    /* Create a new ImgView image and transfer the image data
	     * to it
	     */
	    imgview_image_struct *img = ImgViewImageNew(
		width, height, bpp, bpl
	    );
	    if(img != NULL)
	    {
		for(i = 0; i < nframes; i++)
		{
		    ImgViewImageAppendFrame(
			img,
			data_rgba[i],
			(delay_list != NULL) ?
			    delay_list[i] : 0l
		    );
		    data_rgba[i] = NULL;
		}
	    }

	    /* Update the image info */
	    iv_viewer_update_image_info(
		v,
		path,
		&stat_buf,
		width, height, bpp, bpl, nframes,
		TRUE, x, y, base_width, base_height,
		creator, title, author, comments,
		stat_buf.st_mtime
	    );

	    /* Update the ImgView's background color if it has changed */
	    if(memcmp(old_bg_color, v->bg_color, sizeof(old_bg_color)))
	    {
		GdkColor c[5];
		for(i = 0; i < 5; i++)
		{
		    GDK_COLOR_SET_BYTE(
			&(c[i]),
			v->bg_color[0],
			v->bg_color[1],
			v->bg_color[2]
		    );
		}
		ImgViewSetViewBG(iv, c);
	    }

	    /* Resize and reposition the ImgView as needed */
	    if(resize_on_open)
		iv_viewer_resize(v, width, height, TRUE);

	    /* Set the new image to the ImgView
	     *
	     * Note that the image passed to this function should not
	     * be referenced again after this call
	     */
	    ImgViewSetImage(iv, img);
	    img = NULL;

	    /* More than one frame on this image? */
	    if(nframes > 1)
	    {
		/* If there was a specific frame specified to be
		 * displayed then seek to it, otherwise start
		 * playing
		 */
		if(frame_num > -1)
		    ImgViewSeek(iv, frame_num);
		else
		    ImgViewPlay(iv);
	    }

	    /* Update the ImgView's title */
	    iv_viewer_update_title(v, path);
	}
	else
	{
	    /* Failed to open the image
	     *
	     * Try to open the image as a binary or text file in a new
	     * Data Viewer
	     */
	    use_file_viewer = TRUE;
	    iv_fviewer_new_open_file(core, path, resize_on_open, toplevel);
	}

	/* Delete the opened image data and other resources */
	for(i = 0; i < nframes; i++)
	    g_free(data_rgba[i]);
	g_free(data_rgba);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	/* Update the Image Viewer's widgets */
	iv_viewer_update(v);

	/* Hide the Image Viewer when a non-image file was opened? */
	if(use_file_viewer && core->auto_hide_iv_nonimage)
	    ImgViewUnmap(iv);
}

/*
 *	Saves an image to a different format.
 *
 *	The path specifies the full path to the image to save to.
 *
 *	The ext specifies the extension which determines the file
 *	format to save as. If ext is NULL, "*", or "*.*" then the
 *	format will be determined by the extension specified by
 *	the path.
 */
static void iv_viewer_save(
	iv_image_viewer_struct *v,
	const gchar *path,
	const gchar *ext
)
{
	gboolean	file_exists;
	gint		i, status,
			width, height, bpp, bpl, nframes,
			x, y, base_width, base_height;
	gchar		*orig_path = NULL,
			*new_path = NULL; 
	gulong *delay_list = NULL;
	guint8 *rgba, **rgba_list = NULL;
	GList *glist;
	struct stat stat_buf;
	GtkWidget *w;
	iv_info_struct *info;
	imgview_format format = IMGVIEW_FORMAT_RGBA;
	imgview_frame_struct *frame;
	imgview_image_struct *img;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_imgio_cb_struct *cb_data = NULL;

#define CLENAUP_RETURN	{		\
 if(cb_data != NULL) {			\
  g_free(cb_data->path);		\
  g_free(cb_data);			\
 }					\
					\
 g_free(rgba_list);			\
 g_free(delay_list);			\
 g_free(orig_path);			\
 g_free(new_path);			\
					\
 return;				\
}

	if((v == NULL) || STRISEMPTY(path))
	{
	    CLENAUP_RETURN;
	}

	info = &v->info;
	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	{
	    CLENAUP_RETURN;
	}

	/* No image loaded? */
	if(!ImgViewIsLoaded(iv))
	{
	    CLENAUP_RETURN;
	}

	/* Get image data */
	img = ImgViewGetImage(iv);
	if(img == NULL)
	{
	    CLENAUP_RETURN;
	}
	nframes = img->nframes;
	if((nframes < 1) || (img->frames_list == NULL))
	{
	    CLENAUP_RETURN;
	}

	rgba_list = (guint8 **)g_malloc0(nframes * sizeof(guint8 *));
	delay_list = (gulong *)g_malloc0(nframes * sizeof(gulong));
	for(i = 0,
	    glist = img->frames_list;
	    glist != NULL;
	    i++,
	    glist = g_list_next(glist)
	)
	{
	    frame = IMGVIEW_FRAME(glist->data);
	    if(frame == NULL)
		continue;

	    if(i < nframes)
	    {
		rgba_list[i] = frame->buf;
		delay_list[i] = frame->delay;
	    }
	}
	width = img->width;
	height = img->height;
	bpp = img->bpp;
	bpl = img->bpl;
	rgba = rgba_list[0];
	x = info->x;
	y = info->y;
	base_width = info->base_width;
	base_height = info->base_height;

	/* If no extension given, then use generic string "*.*" as
	 * the extension
	 */
	if(STRISEMPTY(ext))
	    ext = "*.*";

	/* Format the new path from the specified path and extension */
	new_path = STRDUP(path);
	/* Special case for match any extension */
	if(!g_strcasecmp(ext, "*.*") || !g_strcasecmp(ext, "*"))
	{
	    /* Leave new_path as is, but update the extension to
	     * reflect the actual extension
	     */
	    ext = strrchr(path, '.');
	}
	else
	{
	    /* Need to add the extension to the path? */
	    if(!extcmp(path, ext))
	    {
		g_free(new_path);
		new_path = g_strconcat(
		    path,
		    (*ext != '.') ? "." : "",
		    ext,
		    NULL
		);
	    }
	}
	/* Unable to generate the new path? */
	if(new_path == NULL)
	{
	    CLENAUP_RETURN;
	}

	/* At this point ext contains the string of the file type 
	 * extension, event if it was given as "*.*" it will be updated
	 * above to reflect the correct extension or ext can be NULL
	 * on failure
	 */


	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    w = ImgViewGetToplevelWidget(iv);
	else
	    w = NULL;

	/* Check if the file already exists */
	file_exists = stat(new_path, &stat_buf) ? FALSE : TRUE;
	if(file_exists)
	{
	    /* Confirm overwrite */
	    gchar *msg = g_strdup_printf(
		"Overwrite existing file:\n\n    %s",
		new_path
	    );
	    CDialogSetTransientFor(w);
	    status = CDialogGetResponse(
		"Confirm Overwrite",
		msg,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    switch(status)
	    {
	      case CDIALOG_RESPONSE_YES:
	      case CDIALOG_RESPONSE_YES_TO_ALL:
	      case CDIALOG_RESPONSE_OK:
		break;
	      default:
		CLENAUP_RETURN;
		break;
	    }
	}


	/* Get the original file name (if any), this may be needed
	 * for some image libraries
	 */
	orig_path = STRDUP(info->filename);

	/* Allocate the save callback data */
	cb_data = IV_IMGIO_CB(g_malloc0(sizeof(iv_imgio_cb_struct)));
	cb_data->ref_toplevel = w;
	cb_data->viewer = v;
	cb_data->path = STRDUP(new_path);


	/* Reset status to -1, it will be updated when (and if) an
	 * image saving routine is called
	 */
	status = -1;

	/* Save file type by extension, check if we have a valid
	 * extension
	 */
	if(ext != NULL)
	{
	    gboolean	got_response,
			save_attempted = FALSE,
			saved_all_frames = FALSE;

	    /* Save by extension */
#if defined(HAVE_LIBGIF)
	    /* GIF */
	    if(!g_strcasecmp(ext, FTYPE_EXT_GIF))
	    {
		save_attempted = TRUE;
		saved_all_frames = TRUE;

		/* Query user for GIF save options */
		got_response = iv_query_gif_opts(
		    w,
		    new_path,
		    &core->gif_interlaced,
		    &core->gif_format,
		    &core->gif_transparency,
		    &core->gif_looping
		);
		/* Got positive user response and source image data format
		 * is RGBA?
		 */
		if(got_response && (format == IMGVIEW_FORMAT_RGBA))
		{
		    /* Save GIF */
		    status = ImgWriteGIFFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba_list,		/* Source RGBA images */
			delay_list,		/* Delay list */
			nframes,		/* Total frames */
 			v->bg_color,		/* Background color */
			info->comments,		/* Comments */
			core->gif_interlaced,
			core->gif_format,	/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
			core->gif_transparency,
			core->gif_looping,
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
As a Compuserve Graphics Interchange Format (GIF) image.\n\
\n\
%s",
			    new_path,
			    !STRISEMPTY(error_msg) ? error_msg : 
				"Unknown error occured."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed",
			    s,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}
	    }
#endif
#if defined(HAVE_LIBJPEG)
	    /* JPEG */
	    if(!g_strcasecmp(ext, FTYPE_EXT_JPG) ||
	       !g_strcasecmp(ext, FTYPE_EXT_JPEG)
	    )
	    {
		save_attempted = TRUE;

		/* Query user for JPEG save options */
		got_response = iv_query_jpeg_opts(
		    w,
		    new_path,
		    &core->jpeg_quality,	/* 0.0 to 1.0 */
		    &core->jpeg_color
		);
		/* Got positive user response and source image data format
		 * is RGBA?
		 */
		if(got_response && (format == IMGVIEW_FORMAT_RGBA))
		{
		    /* Save JPEG using Imlib */
		    status = ImgWriteJPEGFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba,			/* Source RGBA data */
			v->bg_color,		/* Background color */
#if defined(_WIN32)
			(gint)(core->jpeg_quality * 100.0f),
#else
			rint(core->jpeg_quality * 100.0f), /* Quality 0 to 100 */
#endif
			core->jpeg_color ? 1 : 0,	/* Format:
							 * 0 = Greyscale
							 * 1 = Color */
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
As a JPEG (JPG) image.\n\
\n\
%s",
			    new_path,
			    !STRISEMPTY(error_msg) ? error_msg :
				"Unknown error occured."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed", 
			    s,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}
	    }
#endif
#if defined(HAVE_LIBPNG)
	    /* PNG */
	    if(!g_strcasecmp(ext, FTYPE_EXT_PNG) ||
	       !g_strcasecmp(ext, FTYPE_EXT_MNG)
	    )
	    {
		save_attempted = TRUE;

		/* Query user for PNG save options */
		got_response = iv_query_png_opts(
		    w,
		    new_path,
		    &core->png_compression,	/* 0 to 9 */
		    &core->png_format,	/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Greyscale Alpha
						 * 3 = RGB
						 * 4 = RGBA */
		    &core->png_interlaced
		);
		/* Got positive user response and source image data
		 * format is RGBA?
		 */
		if(got_response && (format == IMGVIEW_FORMAT_RGBA))
		{
		    status = ImgWritePNGFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba,			/* Source RGBA data */
			v->bg_color,		/* Background color */
			info->x, info->y,
			info->base_width, info->base_height,
			PROG_NAME_FULL " " PROG_VERSION,	/* Creator */
			info->title,			/* Title */
			info->author,			/* Author */
			info->comments,			/* Comments */
			core->png_compression,	/* 0 to 9 */
			core->png_interlaced,
			core->png_format,
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
As a Portable Network Graphics (PNG) image.\n\
\n\
%s",
			    new_path,
			    !STRISEMPTY(error_msg) ? error_msg :
				"Unknown error occured."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed", 
			    s, NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}
	    }
#endif
#if !defined(_WIN32)
	    /* PS */
	    if(!g_strcasecmp(ext, FTYPE_EXT_PS) ||
	       !g_strcasecmp(ext, FTYPE_EXT_PS2) ||
	       !g_strcasecmp(ext, FTYPE_EXT_PS3)
	    )
	    {
		/* Set up the print options and call the print callback
		 * to save Postscript images
		 */
		gboolean prev_processing = v->processing;
		print_values_struct *pv = core->print_values;

		save_attempted = TRUE;

		/* Need to remove the existing file (if any) since the
		 * print callback will confirm overwrite again
		 */
		UNLINK(new_path);

		/* Modify print option values */
#if defined(HAVE_XPRINT)
#else
		pv->output = PRINT_OUTPUT_TO_FILE;
		if(info->base_width > 0)
		    pv->paper_width = info->base_width;
		if(info->base_height > 0)
		    pv->paper_height = info->base_height;
#endif
		pv->output_x = info->x;
		pv->output_y = info->y;

		/* Unset processing marker before calling the print
		 * callback or else the print callback will not function
		 * when it detects the image viewer as processing
		 */
		if(prev_processing)
		    v->processing = FALSE;
		iv_viewer_print_cb(v->print_mi, v);
		if(prev_processing)
		    v->processing = TRUE;

		status = 0;
	    }
#endif
	    /* TGA */
	    if(!g_strcasecmp(ext, FTYPE_EXT_TGA))
	    {
		save_attempted = TRUE;

		/* Query user for TGA save options */
		got_response = iv_query_tga_opts(
		    w,
		    new_path,
		    &core->tga_format	/* 0 = Greyscale
					 * 1 = RGB
					 * 2 = RGBA */
		);
		/* Got positive user response and source image data
		 * format is RGBA?
		 */
		if(got_response && (format == IMGVIEW_FORMAT_RGBA))
		{
		    status = ImgWriteTGAFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba,			/* Source RGBA data */
			v->bg_color,
			info->x, info->y,
			info->base_width, info->base_height,
			info->comments,
			core->tga_format,
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
As a Targa (TGA) image.\n\
\n\
%s",
			    new_path,
			    !STRISEMPTY(error_msg) ? error_msg :
				"Unknown error occured."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed",
			    s,
			    NULL, 
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}
	    }
#if defined(HAVE_LIBXPM)
	    /* XPM */
	    if(!g_strcasecmp(ext, FTYPE_EXT_XPM))
	    {
		gchar *s, *c_id;

		save_attempted = TRUE;

		/* Get the C ID from file name */
		s = g_basename(new_path);
		c_id = (s != NULL) ? STRDUP(s) : STRDUP(new_path);

		/* Replace unacceptable characters in the C ID */
		for(s = c_id; *s != '\0'; s++)
		{
		    if(!isalnum(*s))
			*s = '_';
		}

		/* Query user for XPM save options */
		got_response = iv_query_xpm_opts(
		    w,
		    new_path,
		    &c_id,
		    &core->xpm_format,		/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
		    &core->xpm_max_colors,	/* -1 for no limit */
		    &core->xpm_threshold	/* 0x00 to 0xff */
		);
		/* Got positive user response and source image data format
		 * is RGBA?
		 */
		if(got_response && (format == IMGVIEW_FORMAT_RGBA))
		{
		    status = ImgXPMWriteFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba,			/* Source RGBA data */
			v->bg_color,
			info->x, info->y,
			info->base_width, info->base_height,
			c_id, info->comments,
			core->xpm_format,	/* 0 = B&W
						 * 1 = Greyscale
						 * 2 = Color */
			core->xpm_max_colors,	/* -1 for no limit */
			core->xpm_threshold,	/* 0x00 to 0xff */
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
As an XPixmap (XPM) image.\n\
\n\
%s",
			    new_path,
			    !STRISEMPTY(error_msg) ? error_msg :
				"Unknown error occured."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed",
			    s,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}

		g_free(c_id);
	    }
#endif

	    /* Was a save attempted? */
	    if(save_attempted)
	    {
		if(status == 0)
		{
		    /* Warn if not all the frames were saved */
		    if(!saved_all_frames && (nframes > 1))
		    {
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Warning",
"This image contains multiple frames and the\n\
image format that was saved to does not support\n\
multiple frames.\n\
\n\
Try using the GIF image format to save multiple\n\
frames.",
			    NULL,
			    CDIALOG_ICON_WARNING,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
		    }
		}
	    }
	    else	/* All else try to save using Imlib */
	    {
		/* A save was not attempted using a specific
		 * image library format, so now we use Imlib to
		 * handle all other image format cases
		 */
		save_attempted = TRUE;

		if(format == IMGVIEW_FORMAT_RGBA)
		{
		    /* Save image using Imlib */
		    status = ImgWriteImlibFileRGBA(
			new_path,
			width, height, bpl, bpp,
			rgba,		/* Source RGBA data */
			v->bg_color,	/* Background color */
			info->comments,	/* Comments */
			1.0f,		/* Quality, 0.0 to 1.0 */
			1,		/* Save as color (1 = color) */
			0xff,		/* Default alpha value */
			core->imlib_fallback,
			cb_data, iv_save_progress_cb
		    );
		    /* Save failed? */
		    if((status != 0) && (status != -4))
		    {
			const gchar *error_msg = ImgWriteGetError();
			gchar *s = g_strdup_printf(
"Failed to save image:\n\
\n\
    %s\n\
\n\
With %s.\n\
\n\
%s",
			    new_path,
#if defined(HAVE_IMLIB)
			    "Imlib",
#elif defined(HAVE_IMLIB2)
			    "Imlib2",
#else
			    "unknown library",
#endif
			    !STRISEMPTY(error_msg) ? error_msg :
"Please check if the image format and the specified save\n\
options (if any) are supported by this application."
			);
			CDialogSetTransientFor(w);
			CDialogGetResponse(
			    "Save Failed",
			    s,
			    NULL,
			    CDIALOG_ICON_ERROR,
			    CDIALOG_BTNFLAG_OK,
			    CDIALOG_BTNFLAG_OK
			);
			CDialogSetTransientFor(NULL);
			g_free(s);
		    }
		}
	    }
	}	/* if(ext != NULL) */

	/* Save successful? */
	if(status == 0)
	{
	    const gulong last_modified = (gulong)time(NULL);
	    gchar	*creator,
			*title,
			*author,
			*comments;

	    /* Reset the has changes marker */
	    v->has_changes = FALSE;

	    /* Get the stats for newly saved image file */
	    if(stat((const char *)new_path, &stat_buf))
		memset(&stat_buf, 0x00, sizeof(struct stat));

	    /* Update image info on viewer structure with the info from
	     * the newly saved image
	     *
	     * Note that the creator is set to this program's name and
	     * version
	     */
	    creator = STRDUP(PROG_NAME_FULL " " PROG_VERSION);
	    title = STRDUP(info->title);
	    author = STRDUP(info->author);
	    comments = STRDUP(info->comments);
	    iv_viewer_update_image_info(
		v,
		new_path,
		&stat_buf,
		width, height, bpp, bpl, nframes,
		TRUE, x, y, base_width, base_height,
		creator, title, author, comments,
		last_modified
	    );
	    g_free(creator);
	    g_free(title);
	    g_free(author);
	    g_free(comments);

	    /* Update the ImgView's title */
	    iv_viewer_update_title(v, new_path);
	}

	/* Reset the image viewer's progress bar */
	ImgViewProgressUpdate(iv, 0.0f, FALSE);

	/* Update the Image Viewer's widgets */
	iv_viewer_update(v);

#ifdef HAVE_LIBENDEAVOUR2
	/* Report the saved file */
	if(file_exists)
	    EDVNotifyQueueObjectModified(
		core->edv2_ctx, new_path, new_path
	    );
	else
	    EDVNotifyQueueObjectAdded(
		core->edv2_ctx, new_path
	    );
	EDVContextSync(core->edv2_ctx);
#endif	/* HAVE_LIBENDEAVOUR2 */

	CLENAUP_RETURN;
#undef CLENAUP_RETURN
}


/*
 *	Opens the image, displays it fullscreen, and waits for user
 *	response before returning.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 *
 *	The quality specifies the image data to window rendering
 *	dither level. Where 2 is maximum dithering, 1 is normal
 *	dithering, and 0 is no dithering.
 */
static gint iv_open_fullscreen(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Use local default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    /* Display a specific frame? */
	    if(frame_num > -1)
	    {
		const gint s_frame_num = MIN(frame_num, (nframes - 1));
		guint8 *s_rgba_list[1];
		gulong s_delay_list[1];

		s_rgba_list[0] = rgba_list[s_frame_num];
		s_delay_list[0] = delay_list[s_frame_num];

		/* Display the specified frame of the image
		 * fullscreen and wait for user response
		 */
		FSImgNew(
		    s_rgba_list, s_delay_list,
		    1,			/* 1 frame */
		    width, height, bpl,
		    bg_color,		/* 4 bytes RGBA */
		    quality,		/* Quality, 0 to 2 */
		    TRUE,		/* Block until user response */
		    NULL
		);
	    }
	    else
	    {
		/* Display the image fullscreen and wait for user
		 * response
		 */
		FSImgNew(
		    rgba_list, delay_list,
		    nframes,
		    width, height, bpl,
		    bg_color,		/* 4 bytes RGBA */
		    quality,		/* Quality, 0 to 2 */
		    TRUE,		/* Block until user response */
		    NULL
		);
	    }
	}

	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(status);
}

/*
 *	Opens the image and displays it on the desktop centered.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 *
 *	The quality specifies the image data to window rendering
 *	dither level. Where 2 is maximum dithering, 1 is normal
 *	dithering, and 0 is no dithering.
 */
static gint iv_open_to_root_centered(
	const gchar *path,
	const gint frame_num,
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Set default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    const gint s_frame_num = (frame_num > -1) ?
		MIN(frame_num, (nframes - 1)) : 0;
	    guint8 *rgba = rgba_list[s_frame_num];
	    GdkWindow *root_win = GDK_ROOT_PARENT();

	    /* Display the image on the desktop centered */
	    iv_desktop_put_image_centered(
		root_win,
		quality,
		rgba,
		width, height, bpp, bpl,
		bg_color
	    );

	    /* Manage any pending events to ensure that the image is
	     * displayed properly before proceeding with additional
	     * operations 
	     */
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
	else
	{
	    /* Failed to open the image, print error message on to the
	     * desktop   
	     */
	    const gchar *error_msg = ImgLoadGetError();
	    gchar *s;
	    if(error_msg != NULL)
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s\n\
\n\
%s",
		    path, error_msg
		);
	    else
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s",
		    path
		);
	    PRINT_ERROR_TO_DESKTOP(s);
	    g_free(s);
	}

	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(status);
}

/*
 *	Opens the image and displays it scaled to fit (preserving
 *	aspect ratio) on the desktop.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 *
 *	The quality specifies the image data to window rendering
 *	dither level. Where 2 is maximum dithering, 1 is normal
 *	dithering, and 0 is no dithering.
 */
static gint iv_open_to_root_tofit(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Set default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    const gint s_frame_num = (frame_num > -1) ?
		MIN(frame_num, (nframes - 1)) : 0;
	    guint8 *rgba = rgba_list[s_frame_num];
	    GdkWindow *root_win = GDK_ROOT_PARENT();

	    /* Display the image on the desktop to fit */
	    iv_desktop_put_image_tofit(
		root_win,
		quality,
		rgba,
		width, height, bpp, bpl,
		bg_color
	    );

	    /* Manage any pending events to ensure that the image is
	     * displayed properly before proceeding with additional
	     * operations 
	     */
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
	else
	{
	    /* Failed to open the image, print error message on to the
	     * desktop   
	     */
	    const gchar *error_msg = ImgLoadGetError();
	    gchar *s;
	    if(error_msg != NULL)
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s\n\
\n\
%s",
		    path, error_msg
		);
	    else
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s",
		    path
		);
	    PRINT_ERROR_TO_DESKTOP(s)
	    g_free(s);
	}

	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(status);
}

/*
 *	Opens the image and displays it scaled to cover (preserving 
 *	aspect ratio) on the desktop.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 *
 *	The quality specifies the image data to window rendering
 *	dither level. Where 2 is maximum dithering, 1 is normal
 *	dithering, and 0 is no dithering.
 */
static gint iv_open_to_root_tocover(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Set default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    const gint s_frame_num = (frame_num > -1) ?
		MIN(frame_num, (nframes - 1)) : 0;
	    guint8 *rgba = rgba_list[s_frame_num];
	    GdkWindow *root_win = GDK_ROOT_PARENT();

	    /* Display the image on the desktop to cover */
	    iv_desktop_put_image_tocover(
		root_win,
		quality,
		rgba,
		width, height, bpp, bpl,
		bg_color
	    );

	    /* Manage any pending events to ensure that the image is
	     * displayed properly before proceeding with additional
	     * operations 
	     */
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
	else
	{
	    /* Failed to open the image, print error message on to the
	     * desktop
	     */
	    const gchar *error_msg = ImgLoadGetError();
	    gchar *s;
	    if(error_msg != NULL)
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s\n\
\n\
%s",
		    path, error_msg
		);
	    else
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s",
		    path
		);
	    PRINT_ERROR_TO_DESKTOP(s)
	    g_free(s);
	}

	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(status);
}

/*
 *	Opens the image and displays it tiled on the desktop.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 *
 *	The quality specifies the image data to window rendering
 *	dither level. Where 2 is maximum dithering, 1 is normal
 *	dithering, and 0 is no dithering.
 */
static gint iv_open_to_root_tiled(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color,		/* 4 bytes in RGBA format */
	const gint quality		/* Quality, 0 to 2 */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Set default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    const gint s_frame_num = (frame_num > -1) ?
		MIN(frame_num, (nframes - 1)) : 0;
	    guint8 *rgba = rgba_list[s_frame_num];
	    GdkWindow *root_win = GDK_ROOT_PARENT();

	    /* Display the image on the desktop tiled */
	    iv_desktop_put_image_tiled(
		root_win,
		quality,
		rgba,
		width, height, bpp, bpl,
		bg_color
	    );

	    /* Manage any pending events to ensure that the image is
	     * displayed properly before proceeding with additional
	     * operations
	     */
	    while(gtk_events_pending() > 0)
		gtk_main_iteration();
	}
	else
	{
	    /* Failed to open the image, print error message on to the
	     * desktop   
	     */
	    const gchar *error_msg = ImgLoadGetError();
	    gchar *s;
	    if(error_msg != NULL)
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s\n\
\n\
%s",
		    path, error_msg
		);
	    else
		s = g_strdup_printf(
"Unable to open the image:\n\
\n\
    %s",
		    path
		);
	    PRINT_ERROR_TO_DESKTOP(s)
	    g_free(s);
	}

 	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
 	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	return(status);
}

/*
 *	Opens the image and queries the user to print it using the
 *	Print Dialog. If the user selects print then the image is
 *	printed.
 *
 *	The path specifies the full path to the image to be printed.
 *
 *	The frame_num specifies the index of the image's frame to
 *	be printed. If frame_num is -1 then the default frame will
 *	be printed.
 *
 *	The bg_color specifies the background color in 4 bytes RGBA
 *	format.
 */
static gint iv_open_print(
	const gchar *path,
	const gint frame_num,		/* Can be -1 for unspecified */
	guint8 *bg_color		/* 4 bytes in RGBA format */
)
{
	gint		i, status, nframes,
			width, height, bpp, bpl,
			x, y, base_width, base_height;
	guint8 **rgba_list;
	gulong *delay_list;
	gchar		*creator,
			*title,
			*author,
			*comments;
	guint8 lbg_color[4];

	/* Need to initialize confirmation and progress dialogs for
	 * use during this call
	 */
	ProgressDialogInit();
	CDialogInit();

	/* Set default background color? */
	if(bg_color == NULL)
	{
	    bg_color = lbg_color;
	    bg_color[0] = 0x00;
	    bg_color[1] = 0x00;
	    bg_color[2] = 0x00;
	    bg_color[3] = 0xff;
	}

	/* Open the image */
	status = ImgLoadFileRGBA(
	    path,
	    &width, &height, &bpp, &bpl,
	    &rgba_list, &delay_list, &nframes,
	    bg_color,		/* 4 bytes RGBA (will be modified) */
	    &x, &y,
	    &base_width, &base_height,
	    &creator, &title, &author, &comments,
	    0xff,		/* Default alpha value */
	    NULL, NULL		/* No progress callback */
	);

	/* Image opened successfully? */
	if((status == 0) && (rgba_list != NULL) && (nframes > 0))
	{
	    const gchar *name = (path != NULL) ? g_basename(path) : NULL;
	    imgview_image_struct *img;
	    print_values_struct *pv;
#ifdef HAVE_LIBENDEAVOUR2
	    edv_context_struct *edv2_ctx = EDVContextNew();
#endif

#ifdef HAVE_LIBENDEAVOUR2
            EDVContextLoadConfigurationFile(edv2_ctx, NULL);
#endif

            /* Create a new ImgView image and transfer the image data
             * to it
             */
            img = ImgViewImageNew(
                width, height, bpp, bpl
            );
            if(img != NULL)
            {
                for(i = 0; i < nframes; i++)   
                {
                    ImgViewImageAppendFrame(
                        img,
                        rgba_list[i],
                        (delay_list != NULL) ?
                            delay_list[i] : 0l
                    );
                    rgba_list[i] = NULL;
                }
            }

	    /* Create the print values */
	    pv = PrintValuesNew();
	    PrintValuesReset(pv);
	    pv->frame = CLIP(frame_num, 0, (nframes - 1));
	    pv->output_width = img->width;
	    pv->output_height = img->height;

            /* Query user to print the image */
#ifdef HAVE_LIBENDEAVOUR2
            if(PrintDlgGetResponse(
                img,
		name,
		pv,			/* Print values, will be modified */
		bg_color,		/* 4 bytes RGBA */
		NULL,			/* No toplevel widget */
		edv2_ctx
            ))
#else
            if(PrintDlgGetResponse(
                img,
                name,
                pv,			/* Print values, will be modified */
		bg_color,		/* 4 bytes RGBA */
		NULL			/* No toplevel widget */
	    ))
#endif
	    {
                /* Print */
#ifdef HAVE_LIBENDEAVOUR2
                iv_print(
                    img,
                    pv,
                    bg_color,		/* 4 bytes RGBA */
		    NULL,		/* No toplevel widget */
		    edv2_ctx
		);
#else
                iv_print(
                    img,
                    pv,
                    bg_color,		/* 4 bytes RGBA */
		    NULL		/* No toplevel widget */
		);
#endif
	    }

	    while(gtk_events_pending() > 0)
		gtk_main_iteration();

	    PrintValuesDelete(pv);

	    ImgViewImageDelete(img);

	    while(gtk_events_pending() > 0)
		gtk_main_iteration();

#ifdef HAVE_LIBENDEAVOUR2
            EDVContextDelete(edv2_ctx);
#endif
	}
	else
	{
	    /* Failed to open the image, print error message */
	    const gchar *error_msg = ImgLoadGetError();
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		error_msg,
		path
	    );
            CDialogSetTransientFor(NULL);
            CDialogGetResponse(
                "Open Failed",
                msg,
"Please check if the specified file exists and that\n\
permissions allow the reading of that file.",
                CDIALOG_ICON_WARNING,
                CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
                CDIALOG_BTNFLAG_OK
            );
            CDialogSetTransientFor(NULL);
            g_free(msg);
	}

 	for(i = 0; i < nframes; i++)
	    g_free(rgba_list[i]);
 	g_free(rgba_list);
	g_free(delay_list);
	g_free(creator);
	g_free(title);
	g_free(author);
	g_free(comments);

	ProgressDialogShutdown();
	CDialogShutdown();

	return(status);
}


/*
 *	Opens the file as a binary or text file in a new Data Viewer.
 *
 *	This is often used as a fallback when iv_viewer_open fails.
 */
static iv_data_viewer_struct *iv_fviewer_new_open_file(
	iv_core_struct *core,
	const gchar *path,
	const gboolean resize_on_open,
	GtkWidget *toplevel
)
{
	gboolean is_ascii = TRUE;
	const gchar *name;
	gchar *title;
	guint8 *data = NULL, *data_ascii_end;
	gulong data_len = 0l;
	FILE *fp;
	struct stat stat_buf;
	iv_data_viewer_struct *fv;

	if((core == NULL) || STRISEMPTY(path))
	    return(NULL);

	/* Open file for reading */
	fp = fopen(path, "rb");
	if(fp == NULL)
	{
	    const gint error_code = (gint)errno;
	    gchar	*s = STRDUP(g_strerror(error_code)),
			*msg;
	    if(s == NULL)
		s = STRDUP("Unable to open the file");
	    *s = toupper(*s);
	    msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		s, path
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Open Failed",
		msg,
"Please check if the specified file exists and that\n\
permissions allow the reading of that file.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    g_free(s);
	    return(NULL);
	}

	/* Get the file's statistics */
	if(!fstat(fileno(fp), &stat_buf))
	{
	    /* Allocate data buffer */
	    data_len = (gulong)stat_buf.st_size;
	    data = (data_len > 0l) ?
		(guint8 *)g_malloc(data_len * sizeof(guint8)) : NULL;
	    if(data != NULL)
	    {
#ifdef _WIN32
		const gulong blksize = 1024l;
#else
		const gulong blksize = MAX(stat_buf.st_blksize, 1);
#endif
		gint bytes_read;
		guint8	*ptr = data,
			*end = ptr + data_len;

		/* Begin loading data to the buffer */
		while(ptr < end)
		{
		    bytes_read = fread(
			ptr, sizeof(guint8),
			MIN(blksize, (gulong)(end - ptr)), fp
		    );
		    if(bytes_read <= 0)
			break;

		    ptr += bytes_read;
		}
	    }
	}

	/* Close file */
	fclose(fp);

	/* Check file format to see if it is ASCII or binary */
	data_ascii_end = NULL;
	if(data_len > 0l)
	{
	    guint8	*ptr = data,
			*end = ptr + data_len;
	    while(ptr < end)
	    {
		if(!isascii(*ptr))
		{
		    is_ascii = FALSE;
		}
		if(*ptr == '\0')
		{
		    data_ascii_end = ptr;
		    is_ascii = FALSE;
		    break;
		}
		ptr++;
	    }
	}
	if(data_ascii_end == NULL)
	    data_ascii_end = data + data_len;

	/* Get name */
	name = g_basename(path);

	/* Format title */
	title = g_strdup_printf(
	    "%s: %s (%ld bytes)",
	    (core->title != NULL) ?
		core->title : PROG_NAME_FULL,
	    name,
	    data_len
	);

	/* Create Data Viewer */
	fv = iv_fviewer_new(core, core->geometry);
	if(fv != NULL)
	{
	    GtkWidget *w = fv->toplevel;
	    tview_struct *tv = fv->tv;
	    hview_struct *hv = fv->hv;

	    gtk_window_set_title(GTK_WINDOW(w), title);

	    TViewSetBusy(tv, TRUE);
	    HViewSetBusy(hv, TRUE);

	    if(is_ascii)
	    {
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(fv->text_tb), TRUE
		);
		TViewOpenData(tv, data, data_len);
		HViewOpenData(hv, data, data_len);
	    }
	    else
	    {
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(fv->hex_tb), TRUE
		);
		TViewOpenData(tv, data, data_ascii_end - data);
		HViewOpenData(hv, data, data_len);
	    }

	    TViewSetBusy(tv, FALSE);
	    HViewSetBusy(hv, FALSE);
	}

	g_free(data);
	g_free(title);

	return(fv);
}

/*
 *	Checks of there are changes on the Image Viewer and queries
 *	the user to save those changes.
 *
 *	This should be called whenever the Image Viewer is about to
 *	discard the image data.
 *
 *	Warning: If iv->processing is TRUE then This function will not
 *	be able to save.
 *
 *	A return value of TRUE means that the calling function can
 *	continue, otherwise FALSE indicates that the Image VIewer
 *	should not perform the operation.
 */
static gboolean iv_viewer_check_query_save_changes(iv_image_viewer_struct *v)
{
	gint response;
	GtkWidget *toplevel;
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return(TRUE);

	if(CDialogIsQuery())
	    return(FALSE);

	/* No changes that need to be saved? */
	if(!v->has_changes)
	    return(TRUE);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Query user to save changes before closing? */
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"?Salva Los Cambios?",
"?Los datos contienen los cambios que tienen no mas\n\
fue salvados, usted quiere salvar estos cambios?",
#elif defined(PROG_LANGUAGE_FRENCH)
"Epargne Des Changements?",
"Les donne'es contiennent des changements qui pas\n\
encore ont e'te' e'pargne's, vous voulez e'pargner\n\
ces changements?",
#elif defined(PROG_LANGUAGE_GERMAN)
"Auber Anderungen?",
"Enthalt die daten anderungen die nicht noch sind\n\
gespart worden, sie wollen sparen diese anderungen?",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmia I Cambiamenti?",
"I dati contiene dei cambiamenti che non sono stato\n\
ancora risparmiati, lei vuole risparmiare questi\n\
cambiamenti?",
#elif defined(PROG_LANGUAGE_DUTCH)
"Behalve Veranderingen?",
"Bevatten de gegevens veranderingen die niet nog zijn\n\
gered worden, u wil redden deze veranderingen?",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exceto Mudancas?",
"Os dados contem mudancas que nao mas foi poupado,\n\
quer poupar estas mudancas?",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt Forandringer?",
"Dataene inneholder forandringer som ikke enna\n\
spart, gjr De sparer disse forandringene?",
#else
"Save Changes?",
"The data contains changes that have not been\n\
saved, do you want to save these changes?",
#endif
	    NULL,
	    CDIALOG_ICON_WARNING,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO | CDIALOG_BTNFLAG_CANCEL,
	    CDIALOG_BTNFLAG_YES
	);
	CDialogSetTransientFor(NULL);
	switch(response)
	{
	  case CDIALOG_RESPONSE_CANCEL:
	    return(FALSE);	/* Cancel, do not continue */
	    break;

	  case CDIALOG_RESPONSE_NO:
	    return(TRUE);	/* No, discard changes */
	    break;

	  default:		/* Yes, save the image data */
	    /* Call the save image callback to prompt the user to save
	     * the image data
	     */
	    iv_viewer_save_cb(v->save_as_mi, v);

	    /* Check if the save failed by checking if v->has_changes
	     * is still TRUE
	     */
	    return(v->has_changes ? FALSE : TRUE);
	    break;
	}
}


/*
 *	Image Viewer image changed callback.
 *
 *	This is called whenever the image viewer has changed its image
 *	data from its own side invoked by the user.
 */
static void iv_imgview_changed_cb(
	imgview_struct *iv,
	imgview_image_struct *image,
	gpointer data
)
{
	gboolean size_changed = FALSE;
	gint width, height;
	iv_core_struct *core;
	iv_info_struct *info;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((iv == NULL) || (image == NULL) || (v == NULL))
	    return;

	info = &v->info;
	core = v->core;
	if(core == NULL)
	    return;

	/* Update image information on viewer structure with the 
	 * information from the given image
	 */
	width = image->width;
	height = image->height;
	if((info->width != width) || (info->height != height))
	    size_changed = TRUE;
	v->has_changes = TRUE;
	info->width = width;
	info->height = height;

	/* Resize the ImgView's view widget as needed */
	if(core->resize_on_open && size_changed)
	{
	    iv_viewer_resize(v, width, height, TRUE);
	    ImgViewZoomOneToOne(iv);
	}

	/* Update the ImgView's title */
	iv_viewer_update_title(v, NULL);

	/* Update Image Viewer's widgets */
	iv_viewer_update(v);
}

/*
 *	Set view background color callback.
 */
static void iv_viewer_set_bg_color_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	GtkWidget *toplevel;
	csd_color_struct start_color, *color_rtn = NULL;
	imgview_image_struct *img;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((v == NULL) || CSDIsQuery())
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Set up initial color */
	start_color.r = (gfloat)v->bg_color[0] / (gfloat)0xff;
	start_color.g = (gfloat)v->bg_color[1] / (gfloat)0xff;
	start_color.b = (gfloat)v->bg_color[2] / (gfloat)0xff;
	start_color.a = (gfloat)v->bg_color[3] / (gfloat)0xff;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	/* Prompt user to select new color */
	CSDSetTransientFor(toplevel);
	response = CSDGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Color Fijo De Fondo",
	    "Conjunto", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Regler La Couleur De Fond",
	    "Serie", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Feste Hintergrundfarbe",
	    "Satz", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Regolare Il Colore Di Fondo",
	    "Serie", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "De Vaste Achtergrondkleur",
	    "Het Stel", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cor Fixa De Fundo",
	    "Jogo", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Fast BakgrunnFarge",
	    "Sett", "Kanseler",
#else
	    "Set Background Color",
	    "Set", "Cancel",
#endif
	    &start_color, &color_rtn,
	    NULL, NULL
	);
	CSDSetTransientFor(NULL);

	if(response)
	{
	    if(color_rtn != NULL)
	    {
		/* Got user response, set new background color */
		gint i;
		GdkColor c[5];

		for(i = 0; i < 5; i++)
		{
		    GDK_COLOR_SET_COEFF(
			&(c[i]),
			color_rtn->r, color_rtn->g, color_rtn->b
		    );
		}
		ImgViewSetViewBG(iv, c);

		/* Record new background color on view structure */
		v->bg_color[0] = (guint8)(color_rtn->r * 0xff);
		v->bg_color[1] = (guint8)(color_rtn->g * 0xff);
		v->bg_color[2] = (guint8)(color_rtn->b * 0xff);
		v->bg_color[3] = (guint8)(color_rtn->a * 0xff);

		/* Notify change if an image was loaded */
		img = ImgViewGetImage(iv);
		if(img != NULL)
		    iv_imgview_changed_cb(
			iv,			/* ImgView */
			img,			/* Image */
			v			/* Data */
		    );
	    }
	}

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Resize callback.
 */
static void iv_viewer_resize_cb(GtkWidget *widget, gpointer data)
{
	gboolean was_playing;
	gint src_width, src_height, src_bpl, src_bpp;
	const gchar *filename, *name;
	imgview_format format;
	const guint8 *src_data;
	gint tar_width, tar_height;
	GtkWidget *toplevel;
	iv_info_struct *info;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	was_playing = ImgViewIsPlaying(iv);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Get the loaded image's data */
	src_data = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &src_width, &src_height, &src_bpp, &src_bpl, &format
	);
	if(src_data == NULL)
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	info = &v->info;
	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Set initial size values */
	tar_width = src_width;
	tar_height = src_height;

	/* Map the Resize Dialog to query the user to resize */
	if(iv_resize(
	    toplevel,
	    iv,
	    name,
	    &tar_width, &tar_height,
	    &core->resize_maintain_aspect
	))
	{
	    imgview_image_struct *tar_img = ImgViewGetImage(iv);

	    if(was_playing)
		ImgViewPlay(iv);

	    /* Notify change */
	    iv_imgview_changed_cb(
		iv,			/* ImgView */
		tar_img,		/* Image */
		v			/* Data */
	    );
	}

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Color Levels callback.
 */
static void iv_viewer_color_levels_cb(GtkWidget *widget, gpointer data)
{
	gboolean was_playing;
	const gchar *filename, *name;
	GtkWidget *toplevel;
	imgview_struct *iv;
	iv_info_struct *info;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	was_playing = ImgViewIsPlaying(iv);
	if(was_playing)
	    ImgViewPause(iv);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;


	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	info = &v->info;
	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Map the Color Levels Dialog and query the user to modify
	 * color levels
	 */
	if(iv_color_levels(
	    toplevel,
	    iv,
	    name,
	    &core->color_levels_all_frames
	))
	{
	    imgview_image_struct *img = ImgViewGetImage(iv);
	    if(img != NULL)
	    {
		/* Notify change */
		iv_imgview_changed_cb(
		    iv,			/* ImgView */
		    img,		/* Image */
		    v			/* Data */
		);
	    }
	}

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);

	if(was_playing)
	    ImgViewPlay(iv);
}

/*
 *	Add Text callback.
 */
static void iv_viewer_add_text_cb(GtkWidget *widget, gpointer data)
{
	gboolean was_playing;
	const gchar *filename, *name;
	GtkWidget *toplevel;
	imgview_struct *iv;
	iv_info_struct *info;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	/* If the image is playing then it needs to be stopped */
	was_playing = ImgViewIsPlaying(iv);
	if(was_playing)
	    ImgViewPause(iv);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

/*	ImgViewSetBusy(iv, TRUE); */
	v->processing = TRUE;

	info = &v->info;
	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Map the Text Dialog and query user to add text */
	if(iv_add_text(
	    toplevel,
	    iv,
	    name,
	    &core->add_text_font_name,
	    &core->add_text_color,
	    &core->add_text_outline,
	    &core->add_text_outline_color,
	    &core->add_text_opacity,
	    &core->add_text_all_frames
	))
	{
	    imgview_image_struct *img = ImgViewGetImage(iv);

	    /* Notify change */
	    iv_imgview_changed_cb(
		iv,			/* ImgView */
		img,			/* Image */
		v			/* Data */
	    );
	}

	v->processing = FALSE;
/*	ImgViewSetBusy(iv, FALSE); */

	if(was_playing) 
	    ImgViewPlay(iv); 
}

/*
 *	Edit Header callback.
 */
static void iv_viewer_header_cb(GtkWidget *widget, gpointer data)
{
	const gchar *filename, *name;
	GtkWidget *toplevel;
	iv_info_struct *info;
	imgview_frame_struct *frame;
	imgview_image_struct *img;
	imgview_struct *iv;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	info = &v->info;
	iv = v->iv;
	if(v->processing || (iv == NULL))
	    return;

	/* Get the current image and the current frame */
	img = ImgViewGetImage(iv);
	frame = ImgViewImageGetFrame(img, ImgViewGetCurrentFrame(iv));
	if(frame == NULL)
	    frame = ImgViewImageGetFrame(img, 0);
	if(frame == NULL)
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Query user to edit header */
	if(iv_edit_header(
	    toplevel,
	    name,
	    frame->buf,
	    img->width, img->height, img->bpl,
	    img->nframes,
	    v->bg_color,
	    &info->title,
	    &info->author,
	    &info->comments,
	    &info->x, &info->y,
	    &info->base_width, &info->base_height
	))
	{
	    /* Mark that we have changes */
	    v->has_changes = TRUE;

	    /* Update the ImgView's title */
	    iv_viewer_update_title(v, NULL);

	    /* Update Image Viewer's widgets */
	    iv_viewer_update(v);
	}

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Edit Frames callback.
 */
static void iv_viewer_frames_cb(GtkWidget *widget, gpointer data)
{
	const gchar *filename, *name;
	GtkWidget *toplevel;
	iv_info_struct *info;
	imgview_struct *iv;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	if(v->processing || (iv == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	info = &v->info;
	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Query user to edit frames */
	if(iv_edit_frames(
	    toplevel,
	    iv,
	    name
	))
	{
	    /* Mark that we have changes */
	    v->has_changes = TRUE;

	    /* Update the ImgView's title */
	    iv_viewer_update_title(v, NULL);

	    /* Update Image Viewer's widgets */
	    iv_viewer_update(v);
	}

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

#if !defined(_WIN32)
/*
 *	Screenshot callback.
 */
static void iv_viewer_screenshot_cb(GtkWidget *widget, gpointer data)
{
	gint status;
	gint width, height;
	GtkWidget *toplevel, *grab_window;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	if(v->processing)
	    return;

	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Check if there are any unsaved changes and if so then
	 * query the user to save those changes
	 */
	if(!iv_viewer_check_query_save_changes(v))
	{
	    ImgViewSetBusy(iv, FALSE);
	    return;
	}

	v->processing = TRUE;

	/* Get Screenshot options */
	if(!iv_screenshot_query(
	    toplevel,
	    &core->screenshot_delay,		/* In milliseconds */
	    &core->screenshot_hide_windows,
	    &core->screenshot_include_decorations
	))
	{
	    /* User canceled */
	    v->processing = FALSE;
	    ImgViewSetBusy(iv, FALSE);
	    return;
	}

	/* Create grab GtkWindow (it should be NULL) */
	grab_window = core->grab_window;
	if(grab_window == NULL)
	{
	    GtkWidget *w = gtk_window_new(GTK_WINDOW_POPUP);
	    gtk_widget_add_events(
		w,
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	    );
	    gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	    gtk_widget_set_uposition(w, 0, 0);
	    gtk_widget_set_usize(w, 1, 1);
	    gtk_widget_realize(w);

	    core->grab_window = grab_window = w;
	}
	if(grab_window == NULL)
	{
	    v->processing = FALSE;
	    ImgViewSetBusy(iv, FALSE);
	    return;
	}


	/* Unmap ImgView? */
	if(iv->map_state && core->screenshot_hide_windows)
	    ImgViewUnmap(iv);

	/* Map the grab GtkWindow */
	gtk_widget_show(grab_window);


	/* Need to flush GTK+ events, so new grab window creation and
	 * unmapping of other windows are realized
	 */
	gdk_flush();
	while(gtk_events_pending() > 0)
	    gtk_main_iteration();

	/* Block input and do screenshot */
	status = iv_screenshot_start(
	    iv, &width, &height,
	    core->screenshot_delay,
	    core->screenshot_include_decorations,
	    core->resize_on_open,
	    GTK_WIDGET(grab_window),
	    core->grab_cur
	);

	/* Remap ImgView? */
	if(!iv->map_state)
	    ImgViewMap(iv);

	/* Destroy the grab GtkWindow */
	core->grab_window = NULL;
	if(grab_window != NULL)
	{
	    gtk_widget_hide(grab_window);
	    GTK_WIDGET_DESTROY(grab_window);
	}

	/* Screenshot user aborted? */
	if(status == -4)
	{
	    /* User aborted, do nothing */
	}
	/* Screenshot other error? */
	else if(status)
	{
	    /* Show screenshot error message to user */
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Screenshot Failed",
		iv_screenshot_get_error(),
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}
	/* Screenshot successful */
	else
	{
	    iv_info_struct *info = &v->info;

	    /* Increment untitled count on core structure since a new
	     * grabbed image was obtained
	     */
	    core->untitled_count++;


	    /* Begin updating image information */

	    /* Mark new grabbed image as not being saved yet */
	    v->has_changes = TRUE;

	    /* Clear image info structure */
	    iv_clear_info(info);

	    info->width = width;
	    info->height = height;

	    info->has_alpha = FALSE;

	    /* Make a note that this is a grabbed image */
	    g_free(info->comments);
	    info->comments = STRDUP("From a screenshot");

	    /* Update the ImgView's title */
	    iv_viewer_update_title(v, NULL);
	}

	/* Update Image Viewer's widgets */
	iv_viewer_update(v);

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}
#endif


/*
 *	Open image progress callback.
 *
 *	Called by ImgIO*() functions when opening an image.
 */
gint iv_open_progress_cb(
	gpointer data,
	gint min, gint max,		/* Progress position & maximum */
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba_data
)
{
	iv_image_viewer_struct *v;
	iv_imgio_cb_struct *cb_data = IV_IMGIO_CB(data);
	if(cb_data == NULL)
	    return(FALSE);

	/* Get the Image Viewer */
	v = cb_data->viewer;
	if(v != NULL)
	{
	    /* Update the progress bar */
	    if(max > 0)
		ImgViewProgressUpdate(
		    v->iv,
		    (gfloat)min / (gfloat)max,
		    TRUE
		);
	}

	return(TRUE);
}

/*
 *	Save image progress callback.
 *
 *	Called by ImgIO*() functions when saving an image.
 */
static gint iv_save_progress_cb(
	gpointer data,
	gint min, gint max,		/* Progress position & maximum */
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba_data
)
{
	iv_image_viewer_struct *v;
	iv_imgio_cb_struct *cb_data = IV_IMGIO_CB(data);
	if(cb_data == NULL)
	    return(FALSE);

	/* Get the Viewer */
	v = cb_data->viewer;
	if(v != NULL)
	{
	    /* Update the progress bar */
	    if(max > 0)
		ImgViewProgressUpdate(
		    v->iv,
		    (gfloat)min / (gfloat)max,
		    TRUE
		);
	}

	return(TRUE);
}


/*
 *	Open Image callback.
 */
static void iv_viewer_open_image_cb(GtkWidget *widget, gpointer data)
{
	const gchar *img_list[] = 
#ifdef IV_IMAGE_LIST 
	    IV_IMAGE_LIST;
#else
	    { NULL, NULL };
#endif
	gboolean response;
	gint i, npaths, total_ftypes = 0;
	gchar **paths_list;
	fb_type_struct *type_rtn, **ftype = NULL;
	GtkWidget *toplevel;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((v == NULL) || FileBrowserIsQuery())
	    return;

	if(v->processing)
	    return;

	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Check if there are any unsaved changes and if so then
	 * query the user to save those changes
	 */
	if(!iv_viewer_check_query_save_changes(v))
	{
	    ImgViewSetBusy(iv, FALSE);
	    return;
	}

	/* Get the current directory */
	g_free(core->cur_dir);
	core->cur_dir = STRDUP(g_get_current_dir());

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All Images"
	);
	for(i = 0; img_list[i] != NULL; i += 2)
	    FileBrowserTypeListNew(
		&ftype, &total_ftypes,
		img_list[i], img_list[i + 1]
	    );

	/* Query user to open image */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Abra La Imagen",
	    "Abra", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Image Ouverte",
	    "Ouvrir", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Offenes Bildnis",
	    "Offen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Aprire L'Immagine",
	    "Aprire", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Open Beeld",
	    "Open", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Abra Imagem",
	    "Abra", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Apn Avbilde",
	    "Apn", "Kanseler",
#else
	    "Open Image",
	    "Open", "Cancel",
#endif
	    core->cur_dir,
	    ftype, total_ftypes,
	    &paths_list, &npaths,
	    &type_rtn
	);
	FileBrowserSetTransientFor(NULL);

	ImgViewSetBusy(iv, FALSE);

	/* Got image to open? */
	if(response)
	{
	    gchar *parent;
	    const gchar *path;

	    for(i = 0; i < npaths; i++)
	    {
		path = paths_list[i];
		if(STRISEMPTY(path))
		    continue;

		/* Set the current directory */
		parent = g_dirname(path);
		if(parent != NULL)
		{
		    g_free(core->cur_dir);
		    core->cur_dir = STRDUP(parent);
		    chdir(parent);
		    g_free(parent);
		}

		/* First image? */
		if(i == 0)
		{
		    /* Use given Image Viewer to open the first image */
		    ImgViewSetBusy(iv, TRUE);
		    v->processing = TRUE;

		    /* Open image to the given Image Viewer */
		    iv_viewer_open(v, path, -1, core->resize_on_open);

		    v->processing = FALSE;
		    ImgViewSetBusy(iv, FALSE);
		}
		else
		{
		    /* Create a new Image Viewer and open subsequent
		     * images
		     */
		    iv_image_viewer_struct *v2 = iv_viewer_new(
			core,
			IMGVIEW_TOOL_BAR_MAPPED(iv),
			IMGVIEW_SHOW_VALUES(iv),
			IMGVIEW_STATUS_BAR_MAPPED(iv),
			IMGVIEW_SHOW_IMAGE_ON_WM_ICON(iv),
			iv->quality,
			core->bg_color_set ? core->bg_color : NULL,
			core->geometry,
			FALSE			/* Do not load title */
		    );
		    if(v2 == NULL)
			break;

		    ImgViewSetBusy(v2->iv, TRUE);
		    v2->processing = TRUE;

		    /* Open image to the new Image Viewer */
		    iv_viewer_open(v2, path, -1, core->resize_on_open);

		    v2->processing = FALSE;
		    ImgViewSetBusy(v2->iv, FALSE);
		}
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
}

/*
 *	Open Other callback.
 *
 *	Opens a file and displays it on the Data Viewer.
 */
static void iv_viewer_open_other_cb(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint npaths, total_ftypes = 0;
	gchar **paths_list;
	fb_type_struct *type_rtn, **ftype = NULL;
	GtkWidget *toplevel;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((v == NULL) || FileBrowserIsQuery())
	    return;

	if(v->processing)
	    return;

	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Get the current directory */
	g_free(core->cur_dir);
	core->cur_dir = STRDUP(g_get_current_dir());

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All Files"
	);

	/* Query user to open file */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Abra El Archivo",
	    "Abra", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Dossier Ouvert",
	    "Ouvrir", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Offene Akte",
	    "Offen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN) 
	    "Aprire Il File",
	    "Aprire", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Het Open Bestand",
	    "Open", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Abra Arquivo",
	    "Abra", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Apn Arkiv",
	    "Apn", "Kanseler",
#else
	    "Open File",
	    "Open", "Cancel",
#endif
	    core->cur_dir,
	    ftype, total_ftypes,
	    &paths_list, &npaths,
	    &type_rtn
	);
	FileBrowserSetTransientFor(NULL);

	ImgViewSetBusy(iv, FALSE);

	if(response)
	{
	    gint i;
	    gchar *parent;
	    const gchar *path;

	    for(i = 0; i < npaths; i++)
	    {
		path = paths_list[i];
		if(STRISEMPTY(path))
		    continue;

		/* Set the current directory */
		parent = g_dirname(path);
		if(parent != NULL)
		{
		    g_free(core->cur_dir);
		    core->cur_dir = STRDUP(parent);
		    chdir(parent);
		    g_free(parent);
		}

		/* Open file in new Data Viewer */
		iv_fviewer_new_open_file(
		    core,
		    path,
		    core->resize_on_open,
		    toplevel
		);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
}

/*
 *	Save Image callback.
 */
static void iv_viewer_save_cb(GtkWidget *widget, gpointer data)
{
	const gchar *img_list[] = 
#ifdef IV_IMAGE_LIST 
	    IV_IMAGE_LIST;
#else                     
	    { NULL, NULL };
#endif
	gboolean response;
	gint i, npaths, total_ftypes = 0;
	gchar **paths_list;
	fb_type_struct *type_rtn, **ftype = NULL;
	GtkWidget *toplevel;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if((v == NULL) || FileBrowserIsQuery())
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	if(!ImgViewIsLoaded(iv))
	    return;

	ImgViewSetBusy(iv, TRUE);

	/* Get the current directory */
	g_free(core->cur_dir);
	core->cur_dir = STRDUP(g_get_current_dir());

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "Automatically detect format"
	);
	for(i = 0; img_list[i] != NULL; i += 2)
	    FileBrowserTypeListNew(
		&ftype, &total_ftypes,
		img_list[i], img_list[i + 1]
	    );

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Query user to save image as */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Salve La Imagen Como",
	    "Salve", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Epargner L'Image Comme",
	    "Epargner", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "AuBer Blidnis Als",
	    "Sparen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Risparmiare L'Immagine Come",
	    "Risparm", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Behalve Beeld Als",
	    "Red", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Exceto Imagem Como",
	    "Poupe", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Untatt Avbilde Som",
	    "Untatt", "Kanseler",
#else
	    "Save Image As",
	    "Save", "Cancel",
#endif

	    core->cur_dir,
	    ftype, total_ftypes,
	    &paths_list, &npaths,
	    &type_rtn
	);
	FileBrowserSetTransientFor(NULL);

	ImgViewSetBusy(iv, FALSE);

	if(response)
	{
	    /* Copy the list of paths so if any changes to the
	     * response paths during the saves will not cause
	     * problems
	     */
	    gchar *path, *ext;
	    GList *paths_list2 = NULL, *glist;

	    for(i = 0; i < npaths; i++)
		paths_list2 = g_list_append(
		    paths_list2, STRDUP(paths_list[i])
		);

	    /* Copy the extension */
	    ext = STRDUP((type_rtn != NULL) ? type_rtn->ext : "*.*");

	    /* Save each path */
	    for(glist = paths_list2;
	        glist != NULL;
	        glist = g_list_next(glist)
	    )
	    {
		path = (gchar *)glist->data;
		if(!STRISEMPTY(path))
	        {
		    /* Set the current directory */
		    gchar *parent = g_dirname(path);
		    if(parent != NULL)
		    {
			g_free(core->cur_dir);
			core->cur_dir = STRDUP(parent);
		        chdir(parent);
		        g_free(parent);
		    }

		    ImgViewSetBusy(iv, TRUE);
		    v->processing = TRUE;

		    /* Save image */
		    iv_viewer_save(v, path, ext);

		    v->processing = FALSE;
		    ImgViewSetBusy(iv, FALSE);
		}
	    }

	    g_free(ext);

	    g_list_foreach(paths_list2, (GFunc)g_free, NULL);
	    g_list_free(paths_list2);
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Reset the file browser listing due to change in objects */
	FileBrowserReset();
}

/*
 *	Display On Root Centered callbacks.
 */
static void iv_viewer_display_on_root_centered_cb(
	GtkWidget *widget, gpointer data
)
{
	const guint8 *data_rgba;
	gint width, height, bpp, bpl;
	imgview_format format;
	GdkWindow *root_win;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	iv = v->iv;
	root_win = GDK_ROOT_PARENT();
	if(v->processing || (iv == NULL) || (root_win == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	/* Get the image data of the image's current frame */
	data_rgba = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &width, &height, &bpp, &bpl, &format
	);

	/* Display the image data to the desktop */
	iv_desktop_put_image_centered(
	    root_win,
	    iv->quality,
	    data_rgba,
	    width, height, bpp, bpl,
	    v->bg_color
	);

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Display On Root To Fit callback.
 */
static void iv_viewer_display_on_root_tofit_cb(
	GtkWidget *widget, gpointer data
)
{
	const guint8 *data_rgba;
	gint width, height, bpp, bpl;
	imgview_format format;
	GdkWindow *root_win;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	iv = v->iv;
	root_win = GDK_ROOT_PARENT();
	if(v->processing || (iv == NULL) || (root_win == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	/* Get the image data of the image's current frame */
	data_rgba = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &width, &height, &bpp, &bpl, &format
	);

	/* Display the image data to the desktop */
	iv_desktop_put_image_tofit(
	    root_win,
	    iv->quality,
	    data_rgba,
	    width, height, bpp, bpl,
	    v->bg_color
	);

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Display On Root To Cover callback.
 */
static void iv_viewer_display_on_root_tocover_cb(
	GtkWidget *widget, gpointer data
)
{
	const guint8 *data_rgba;
	gint width, height, bpp, bpl;
	imgview_format format;
	GdkWindow *root_win;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	iv = v->iv;
	root_win = GDK_ROOT_PARENT();
	if(v->processing || (iv == NULL) || (root_win == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	/* Get the image data of the image's current frame */
	data_rgba = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &width, &height, &bpp, &bpl, &format
	);

	/* Display the image data to the desktop */
	iv_desktop_put_image_tocover(
	    root_win,
	    iv->quality,
	    data_rgba,
	    width, height, bpp, bpl,
	    v->bg_color
	);

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Display On Root Tiled callback.
 */
static void iv_viewer_display_on_root_tiled_cb(
	GtkWidget *widget, gpointer data
)
{
	const guint8 *data_rgba;
	gint width, height, bpp, bpl;
	imgview_format format;
	GdkWindow *root_win;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	iv = v->iv;
	root_win = GDK_ROOT_PARENT();
	if(v->processing || (iv == NULL) || (root_win == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);
	v->processing = TRUE;

	/* Get the image data of the image's current frame */
	data_rgba = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &width, &height, &bpp, &bpl, &format
	);

	/* Display the image data to the desktop */
	iv_desktop_put_image_tiled(
	    root_win,
	    iv->quality,
	    data_rgba,
	    width, height, bpp, bpl,
	    v->bg_color
	);

	v->processing = FALSE;
	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Clear image callback.
 */
static void iv_viewer_clear_cb(GtkWidget *widget, gpointer data)
{
	imgview_struct *iv;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	if(v->processing || (iv == NULL))
	    return;

	ImgViewSetBusy(iv, TRUE);

	/* Check if there are any unsaved changes and if so then
	 * query the user to save those changes
	 */
	if(!iv_viewer_check_query_save_changes(v))
	{
	    ImgViewSetBusy(iv, FALSE);
	    return;
	}

	/* Begin clearing image */

	/* Unload image on image viewer */
	ImgViewClear(iv);

	/* Begin resetting information values on viewer structure */
	v->has_changes = FALSE;

	/* Delete image info values */
	iv_clear_info(&v->info);

	/* Update the ImgView's title */
	iv_viewer_update_title(v, NULL);

	/* Update Image Viewer's widgets */
	iv_viewer_update(v);

	ImgViewSetBusy(iv, FALSE);
}

/*
 *	Image information callback.
 */
static void iv_viewer_information_cb(GtkWidget *widget, gpointer data)
{
	const guint8 *rgba;
	gint width, height, bpp, bpl, nframes;
	imgview_format format;
	const gchar *filename, *ext_ptr, *name_ptr;
	gint alpha_channel;
	guint vis_id = 0x00000000;
	const gchar *dpy_name, *vis_name = NULL;
	gint depth = 0;
	GtkWidget *w, *toplevel;
	iv_info_struct *info;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	info = &v->info;
	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Get viewer widget */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if((w != NULL) ? (w->window != NULL) : FALSE)
	{
	    GdkWindow *window = w->window;
	    if(window != NULL)
	    {
		GdkVisual *vis = gdk_window_get_visual(window);

		/* Get Visual information */
		if(vis != NULL)
		{
		    vis_id = (guint)vis;
		    switch(vis->type)
		    {
		      case GDK_VISUAL_STATIC_GRAY:
			vis_name = "StaticGray";
			break;
		      case GDK_VISUAL_GRAYSCALE:
			vis_name = "GreyScale";
			break;
		      case GDK_VISUAL_STATIC_COLOR:
			vis_name = "StaticColor";
			break;
		      case GDK_VISUAL_PSEUDO_COLOR:
			vis_name = "PseudoColor";
			break;
		      case GDK_VISUAL_TRUE_COLOR:
			vis_name = "TrueColor";
			break;
		      case GDK_VISUAL_DIRECT_COLOR:
			vis_name = "DirectColor";
			break;
		    }
		    depth = vis->depth;
		}
	    }
	}

	/* Get the current image data */
	rgba = ImgViewGetImageData(
	    iv, ImgViewGetCurrentFrame(iv),
	    &width, &height, &bpp, &bpl, &format
	);

	nframes = ImgViewGetTotalFrames(iv);

	/* Get the ImgView's alpha channel info */
	if((format == IMGVIEW_FORMAT_RGBA) ||
	   (format == IMGVIEW_FORMAT_GREYSCALEA32)
	)
	{
	    /* There is an allocated alpha channel, but check if
	     * it has defined data
	     */
	    alpha_channel = (iv->alpha_flags & IMGVIEW_ALPHA_DEFINED) ?
		2 : 1;
	}
	else
	{
	    alpha_channel = 0;
	}

	/* Get other information about image */

	/* Filename */
	filename = info->filename;

	/* Extension */
	ext_ptr = (filename != NULL) ?
	    strrchr(filename, '.') : NULL;

	/* Get file's name without parent path */
	name_ptr = (filename != NULL) ? g_basename(filename) : NULL;
	if(name_ptr == NULL)
	    name_ptr = filename;

	/* Get display name */
	dpy_name = (const gchar *)gdk_get_display();

	/* Create a new Image Info Dialog to display the newly obtained
	 * image information
	 */
	if(TRUE)
	{
	    GList *glist;
	    imginfodlg_struct *d = ImgInfoDlgNew(toplevel);
	    ImgInfoDlgSetValues(
		d,
		rgba,
		width, height,
		bpp,			/* Bytes Per Pixel */
		bpl,			/* Bytes Per Line */
		nframes,		/* Number Of Frames */
		(gulong)bpl *		/* Total Bytes */
		    (gulong)height *
		    (gulong)nframes,
		alpha_channel,
		v->bg_color,		/* 4 bytes RGBA */
		info->x, info->y,
		info->base_width, info->base_height,
		v->has_changes,
		name_ptr,
		ext_ptr,
		info->filesize,		/* Bytes */
		info->title,
		info->author,
		info->creator,
		info->comments,
		info->time_modified,
		dpy_name,		/* X Display Address */
		vis_name,		/* X Visual Type */
		depth			/* X Depth In Bits Per Pixel */
	    );
	    ImgInfoDlgMap(d);

	    /* Record window */
	    for(glist = core->imginfodlg;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		if(glist->data == NULL)
		{
		    glist->data = d;
		    break;
		}
	    }
	    if(glist == NULL)
		core->imginfodlg = g_list_append(
		    core->imginfodlg, d
		);
	}
}

/*
 *	Fullscreen callback.
 */
static void iv_viewer_fullscreen_cb(GtkWidget *widget, gpointer data)
{
	gboolean was_playing;
	gint i;
	guint8 **rgba_list;
	gulong *delay_list;
	GList *glist;
	GtkWidget *toplevel;
	imgview_frame_struct *frame;
	imgview_image_struct *img;
	imgview_struct *iv;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if((iv == NULL) || (core == NULL))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	/* Get the current image */
	img = ImgViewGetImage(iv);
	if(img == NULL)
	    return;

	rgba_list = (guint8 **)g_malloc0(img->nframes * sizeof(guint8 *));
	delay_list = (gulong *)g_malloc0(img->nframes * sizeof(gulong));
	if((rgba_list == NULL) || (delay_list == NULL))
	{
	    g_free(rgba_list);
	    g_free(delay_list);
	    return;
	}
	for(i = 0, glist = img->frames_list;
	    glist != NULL;
	    i++, glist = g_list_next(glist)
	)
	{
	    frame = IMGVIEW_FRAME(glist->data);
	    if(frame == NULL)
		continue;

	    rgba_list[i] = frame->buf;
	    delay_list[i] = frame->delay;
	}

	/* Pause while displaying full screen */
	was_playing = ImgViewIsPlaying(iv);
	if(was_playing)
	    ImgViewPause(iv);

	/* Display the image full screen and wait for user reponse */
	FSImgNew(
	    rgba_list, delay_list,
	    img->nframes,
	    img->width, img->height, img->bpl,
	    v->bg_color,		/* 4 bytes RGBA */
	    iv->quality,		/* Quality, 0 to 2 */
	    TRUE,			/* Block until user response */
	    toplevel
	);

	g_free(rgba_list);
	g_free(delay_list);

	if(was_playing)
	    ImgViewPlay(iv);
}

#if !defined(_WIN32)
/*
 *	Print callback.
 */
static void iv_viewer_print_cb(GtkWidget *widget, gpointer data)
{
	gboolean was_playing;
	gint frame_num;
	const gchar *filename, *name;
	GtkWidget *toplevel;
	print_values_struct *pv;
	imgview_struct *iv;
	imgview_image_struct *img;
	iv_info_struct *info;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	was_playing = ImgViewIsPlaying(iv);
	if(was_playing)
	    ImgViewPause(iv);

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;

	info = &v->info;
	filename = info->filename;
	name = (filename != NULL) ? g_basename(filename) : NULL;

	/* Get image that we want to print */
	img = ImgViewGetImage(iv);
	if(img == NULL)
	    return;
	frame_num = ImgViewGetCurrentFrame(iv);

	/* Get and set up print values */
	pv = core->print_values;
	pv->frame = frame_num;
	pv->output_width = img->width;
	pv->output_height = img->height;

	/* Query user to print image */
#ifdef HAVE_LIBENDEAVOUR2
	if(PrintDlgGetResponse(
	    img,			/* Image */
	    name,			/* File name */
	    pv,				/* Print values, will be modified */
	    v->bg_color,		/* 4 bytes RGBA */
	    toplevel,
	    core->edv2_ctx
	))
#else
	if(PrintDlgGetResponse(
	    img,			/* Image */
	    name,			/* File name */
	    pv,				/* Print values, will be modified */
	    v->bg_color,		/* 4 bytes RGBA */
	    toplevel
	))
#endif
	{
	    ImgViewSetBusy(iv, TRUE);
	    v->processing = TRUE;

	    /* Print */
#ifdef HAVE_LIBENDEAVOUR2
	    iv_print(
		img,
		pv,			/* Print values */
		v->bg_color,		/* Background color RGBA */
		toplevel,
		core->edv2_ctx
	    );
#else
	    iv_print(
		img,
		pv,
		v->bg_color,		/* Print values */
		toplevel		/* Background color RGBA */
	    );
#endif

	    v->processing = FALSE;
	    ImgViewSetBusy(iv, FALSE);
	}

	if(was_playing)
	    ImgViewPlay(iv);
}
#endif	/* !_WIN32 */

/*
 *	New Image Viewer callback.
 */
static void iv_viewer_new_viewer_cb(GtkWidget *widget, gpointer data)
{
	imgview_struct *iv;
	iv_image_viewer_struct *v2;
	iv_core_struct *core;
	iv_image_viewer_struct *v = IV_IMAGE_VIEWER(data);
	if(v == NULL)
	    return;

	iv = v->iv;
	core = v->core;
	if(v->processing || (iv == NULL) || (core == NULL))
	    return;

	/* Create new Image Viewer */
	v2 = iv_viewer_new(
	    core,
	    IMGVIEW_TOOL_BAR_MAPPED(iv),
	    IMGVIEW_SHOW_VALUES(iv),
	    IMGVIEW_STATUS_BAR_MAPPED(iv),
	    IMGVIEW_SHOW_IMAGE_ON_WM_ICON(iv),
	    iv->quality,
	    core->bg_color_set ? core->bg_color : NULL,
	    core->geometry,
	    TRUE		/* Always load title when creating a new
				 * viewer */
	);

	/* Update the given Image Viewer's widgets */
	iv_viewer_update(v);
}


/*
 *	Updates the Image Viewer's ImgView toplevel GtkWindow title.
 *
 *	If new_path is not NULL then it will be used as the new file
 *	name instead of v->filename.
 */
static void iv_viewer_update_title(
	iv_image_viewer_struct *v, const gchar *new_path
)
{
	const gchar *path, *name;
	gchar *title;
	GtkWidget *w;
	iv_info_struct *info;
	iv_core_struct *core;
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	info = &v->info;
	core = v->core;
	if((core == NULL))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    w = ImgViewGetToplevelWidget(iv);
	else
	    w = NULL;
	if(w == NULL)
	    return;

	/* Get the new path if it is specified or else get the
	 * current path from the info
	 */
	path = (new_path != NULL) ? new_path : info->filename;

	/* Get the name portion of the path */
	name = (path != NULL) ? g_basename(path) : NULL;

	/* Format the title */
	if(!STRISEMPTY(path))
	    title = g_strdup_printf(
		"%s: %s %s",
		(core->title != NULL) ?
		    core->title : PROG_NAME_FULL,
		name,
		(v->has_changes) ? HAS_CHANGES_STR : ""
	    );
	else
	    title = g_strdup_printf(
		"%s: Untitled #%i %s",
		(core->title != NULL) ?
		    core->title : PROG_NAME_FULL,
		core->untitled_count,
		(v->has_changes) ? HAS_CHANGES_STR : ""
	    );

	/* Set title */
	gtk_window_set_title(GTK_WINDOW(w), title);

	g_free(title);
}

/*
 *	Updates the Image Viewer's widgets to reflect current values.
 *
 *	Does not update the ImgView's widgets.
 */
static void iv_viewer_update(iv_image_viewer_struct *v)
{
	gboolean sensitive;
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if(iv == NULL)
	    return;

	sensitive = ImgViewIsLoaded(iv);
/*	GTK_WIDGET_SET_SENSITIVE(v->edit_mi, sensitive); */
	GTK_WIDGET_SET_SENSITIVE(v->resize_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->color_levels_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->add_text_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->edit_header_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->edit_frames_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->save_as_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->fullscreen_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->desktop_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->clear_mi, sensitive);
	GTK_WIDGET_SET_SENSITIVE(v->print_mi, sensitive);
}

/*
 *	Resizes the Image Viewer.
 *
 *	The v specifies the Image Viewer.
 *
 *	The width and height specifies the size to resize the Image
 *	Viewer to.
 *
 *	If keep_visible_on_desktop is TRUE then the position and
 *	the width and height may be automatically adjusted to keep
 *	the Image Viewer visible on the desktop.
 */
static void iv_viewer_resize(
	iv_image_viewer_struct *v,
	const gint width, const gint height,
	const gboolean keep_visible_on_desktop
)
{
	const gint	wm_frame_width = 8,	/* Assumed values */
			wm_title_height = 20;
	gint		_width = width,
			_height = height,
			left_pad, right_pad,
			top_pad, bottom_pad;
	GdkWindow *window, *root_win;
	GtkWidget *w, *toplevel;
	imgview_struct *iv = (v != NULL) ? v->iv : NULL;
	if((iv == NULL) || (_width <= 0) || (_height <= 0))
	    return;

	/* Get the ImgView's toplevel widget if it is a GtkWindow */
	if(ImgViewIsToplevelWindow(iv))
	    toplevel = ImgViewGetToplevelWidget(iv);
	else
	    toplevel = NULL;
	if(toplevel == NULL)
	    return;

	/* If the Image Viewer does not need to be visible on the
	 * desktop then just set the size explicitly here
	 */
	if(!keep_visible_on_desktop)
	{
	    GtkWidget *w = (GtkWidget *)ImgViewGetViewWidget(iv);
	    if(w != NULL)
	    {
		gtk_widget_set_usize(w, _width, _height);
		gtk_widget_queue_resize(w);
	    }
	    gtk_widget_queue_resize(toplevel);
	    return;
	}

	/* Calculate the paddings from the ImgView's view GtkWidget
	 * to the edge of the ImgView's toplevel GtkWindow
	 *
	 * All padding values are absolute values
	 */

	/* Left padding */
	left_pad = wm_frame_width;

	/* Right padding */
	right_pad = wm_frame_width;
	w = iv->vscrollbar;
	if(w != NULL)
	    right_pad += 2 + w->allocation.width;

	/* Top padding */
	top_pad = wm_frame_width + wm_title_height;
	w = IMGVIEW_TOOL_BAR_MAPPED(iv) ?
	    iv->tool_bar_toplevel : NULL;
	if(w != NULL)
	    top_pad += 2 + w->allocation.height;

	/* Bottom padding */
	bottom_pad = wm_frame_width;
	w = iv->hscrollbar;
	if(w != NULL)
	    bottom_pad += 2 + w->allocation.height;
	w = IMGVIEW_STATUS_BAR_MAPPED(iv) ?
	    iv->status_bar_toplevel : NULL;
	if(w != NULL)
	    bottom_pad += 2 + w->allocation.height;

	/* Get the ImgView's view GtkWidget */
	w = (GtkWidget *)ImgViewGetViewWidget(iv);
	if(w == NULL)
	    return;

	/* Get the ImgView toplevel's GdkWindow and desktop's
	 * GdkWindow
	 */
	window = toplevel->window;
	root_win = GDK_ROOT_PARENT();

	/* If we got the root GdkWindow then make sure that the entire
	 * ImgView does not exceed the desktop's size if possible
	 */
	if(root_win != NULL)
	{
	    const gint	toplevel_width = _width + left_pad + right_pad,
			toplevel_height = _height + top_pad + bottom_pad;
	    gint	root_width, root_height;

	    gdk_window_get_size(root_win, &root_width, &root_height);

	    if(toplevel_width > root_width)
		_width = root_width - left_pad - right_pad;
	    if(toplevel_height > root_height)
		_height = root_height - top_pad - bottom_pad;
	    if((_width <= 0) || (_height <= 0))
		return;
	}
	/* Resize the ImgView's view and toplevel GtkWidgets */
	gtk_widget_set_usize(w, _width, _height);
	gtk_widget_queue_resize(w);
	gtk_widget_queue_resize(toplevel);

	/* Adjust the position to keep the image on the desktop
	 *
	 * Since the toplevel's size is not updated yet, we need to
	 * calculate it based on the view and other widgets' size
	 * then adjust the toplevel's position as needed
	 */
	if((window != NULL) && (root_win != NULL))
	{
	    gboolean need_move = FALSE;
	    gint	x, y,
			root_width, root_height;

	    gdk_window_get_root_origin(window, &x, &y);
	    gdk_window_get_size(root_win, &root_width, &root_height);

	    if((x + (_width + right_pad)) > root_width)
	    {
		x = root_width - (_width + right_pad);
		need_move = TRUE;
	    }
	    if(x < left_pad)
	    {
		x = left_pad;
		need_move = TRUE;
	    }

	    if((y + (_height + bottom_pad)) > root_height)
	    {
		y = root_height - (_height + bottom_pad);
		need_move = TRUE;
	    }
	    if(y < top_pad)
	    {
		y = top_pad;
		need_move = TRUE;
	    }

	    if(need_move)
	    {
		/* Convert view position to toplevel position */
		gint	toplevel_x = x,
			toplevel_y = y;
		GtkWidget *w = IMGVIEW_TOOL_BAR_MAPPED(iv) ?
		    iv->tool_bar_toplevel : NULL;
		if(w != NULL)
		    toplevel_y -= 2 + w->allocation.height;

		gtk_widget_set_uposition(
		    toplevel, toplevel_x, toplevel_y
		);
	    }
	}
}


/*
 *	Deletes and resets the Image Viewer's info.
 */
static void iv_clear_info(iv_info_struct *info)
{
	if(info == NULL)
	    return;

	g_free(info->filename);
	g_free(info->creator);
	g_free(info->title);
	g_free(info->author);
	g_free(info->comments);

	memset(info, 0x00, sizeof(iv_info_struct));
}


/*
 *	Checks if ext is an extension (postfix) of s (case insensitive).
 */
static gboolean extcmp(const gchar *s, const gchar *ext)
{
	gint s_len, ext_len;
	const gchar *s_ptr, *ext_ptr;

	if(STRISEMPTY(s) || STRISEMPTY(ext))
	    return(FALSE);

	/* Get lengths */
	s_len = strlen(s);
	ext_len = strlen(ext);

	/* Get end pointers */
	s_ptr = s + s_len;
	ext_ptr = ext + ext_len;

	/* See backwards from ends, stop when one or both pointers
	 * reach the beginning of the string (whichever comes first)
	 */
	while((s_ptr >= s) && (ext_ptr >= ext))
	{
	    /* Characters do not match? */
	    if(toupper(*s_ptr--) != toupper(*ext_ptr--))
		break;
	}

	/* Matched entire extension string? */
	return((ext_ptr < ext) ? TRUE : FALSE);
}


/*
 *	Parses the parameter and value from an argument, returning a
 *	statically allocated string or a string pointing within arg for
 *	both the parm_rtn and val_rtn.
 *
 *	The given arg will not be modified.
 */
static void get_args_value(
	char *arg, char **parm_rtn, char **val_rtn
)
{
	gchar *s;
	static gchar parm_str[256];


	if(parm_rtn != NULL)
	    *parm_rtn = "";
	if(val_rtn != NULL)
	    *val_rtn = "";

	if(arg == NULL)
	    return;

	/* If first char is not a '-' or '+' then skip as there is
	 * no parameter or value.
	 */
	if((*arg != '-') && (*arg != '+'))
	    return;

	/* Seek past initial '-' or '+' characters */
	while((*arg == '-') || (*arg == '+'))
	    arg++;

	/* Get parameter and store it in our parm_str */
	strncpy(parm_str, arg, sizeof(parm_str));
	parm_str[sizeof(parm_str) - 1] = '\0';

	s = strchr(parm_str, '=');
	if(s != NULL)
	    *s = '\0';
	s = strchr(parm_str, '-');
	if(s != NULL)
	    *s = '\0';

	/* Update parm return */
	if(parm_rtn != NULL)
	    *parm_rtn = parm_str;


	/* Get value */
	s = strchr(arg, '=');
	if(s == NULL)
	    s = strchr(arg, '-');
	if(s == NULL)
	{
	    if(val_rtn != NULL)
		*val_rtn = "";
	}
	else
	{
	    if(val_rtn != NULL)
		*val_rtn = s + 1;
	}
}

/*
 *	Returns TRUE if the given value is true, accepted values are
 *	"yes" "y" "on" "active" "enabled" or any non-zero value. Any
 *	other value will return FALSE.
 */
static gboolean stristrue(const gchar *s)
{
	gint c;

	if(s == NULL)
	    return(FALSE);

	c = (int)(*s);
	if((c == 'Y') || (c == 'y'))
	    return(TRUE);

	if(!g_strcasecmp(s, "on"))
	    return(TRUE);

	if(!g_strcasecmp(s, "active"))
	    return(TRUE);

	if(!g_strcasecmp(s, "enabled"))
	    return(TRUE);

	c = ATOI(s);
	if(c != 0)
	    return(TRUE);

	return(FALSE);
}


#if defined(_WIN32)
int WINAPI WinMain(
	HINSTANCE hInstance,		// Instance
	HINSTANCE hPrevInstance,	// Previous Instance
	LPSTR lpCmdLine,		// Command Line Parameters
	int nCmdShow			// Window Show State
)
#else
int main(int argc, char *argv[])
#endif
{
#ifdef _WIN32
	/* Need to explode arguments for Win32 */
	gint argc;
	gchar **argv = g_strsplit(lpCmdLine, " ", -1);
#endif
	gint rtn_val = 0;
	gboolean	initialized_gtk = FALSE,
			initialized_imlib = FALSE,
			initialized_dialogs = FALSE;

	gboolean	resize_on_open = TRUE,
			show_tooltips = TRUE,
			show_toolbar = TRUE,
			show_values = TRUE,
			show_statusbar = TRUE,
			show_image_on_wm_icon = FALSE,
			startup_screenshot = FALSE;
	gint		quality = 1,		/* 0 to 2 */
			startup_frame_num = -1;
	guint8		bg_color[4] = { 0x00, 0x00, 0x00, 0xff };
	gboolean	bg_color_set = FALSE;
	gint		imlib_fallback = 1;	/* 0 or 1 */
	gboolean	auto_hide_iv_nonimage = FALSE,
			simple_title = FALSE;
	const gchar	*fullscreen_path = NULL,
			*display_on_root_centered_path = NULL,
			*display_on_root_tofit_path = NULL,
			*display_on_root_tocover_path = NULL,
			*display_on_root_tiled_path = NULL,
			*print_path = NULL;
	gchar		*font_name = NULL,
			*wmname = NULL,
			*wmclass = NULL,
			*title = NULL;
	GdkRectangle	*geometry = NULL;

	gint i;
	GList *file_list = NULL;
	iv_core_struct *core;


	/* Reset the core */
	iv_core = NULL;

	/* Set the system signal callbacks */
#ifdef SIGHUP
	signal(SIGHUP, iv_signal_cb);
#endif
#ifdef SIGINT
	signal(SIGINT, iv_signal_cb);
#endif
#ifdef SIGTERM
	signal(SIGTERM, iv_signal_cb);
#endif
#ifdef SIGQUIT
	signal(SIGQUIT, iv_signal_cb);
#endif
#ifdef SIGSEGV
	signal(SIGSEGV, iv_signal_cb);
#endif
#ifdef SIGPIPE
	signal(SIGPIPE, iv_signal_cb);
#endif


#ifdef _WIN32
	/* Count arguments */
	for(argc = 0; argv[argc] != NULL; argc++);
#endif                                              


/* Deletes the values obtained from the command line */
#define DELETE_FETCHED_CMD_LINE_VALUES	{	\
 g_free(font_name);				\
 g_free(wmname);				\
 g_free(wmclass);				\
 g_free(title);					\
 g_free(geometry);				\
						\
 g_list_foreach(file_list, (GFunc)g_free, NULL);\
 g_list_free(file_list);			\
 file_list = NULL;				\
}

	/* Parse arguments for help and version only */
	for(i = 1; i < argc; i++)
	{
	    gchar *parm, *val;
	    gchar *arg = (gchar *)argv[i];
	    if(arg == NULL)
		continue;

	    get_args_value(arg, &parm, &val);

	    /* Help? */
	    if(!g_strcasecmp(parm, "help") ||
	       !g_strcasecmp(parm, "h") ||
	       !g_strcasecmp(parm, "?")
	    )
	    {
#if defined(_WIN32)
		g_print("Usage: iv [files]");
#else
		g_print(PROG_USAGE);
#endif
		DELETE_FETCHED_CMD_LINE_VALUES;
		return(rtn_val);
	    }
	    /* Version? */
	    else if(!g_strcasecmp(parm, "version"))
	    {
		gint major, minor, release;

		/* This program's version and copyright */
		g_print(
		    "%s Version %s\n%s",
		    PROG_NAME_FULL, PROG_VERSION,
		    PROG_COPYRIGHT
		);
		g_print("\nSupported Libraries:\n");

#if defined(HAVE_X) && defined(HAVE_XF86_VIDMODE)
		g_print(
"        XF86VidMode\n"
		);
#endif
#if defined(HAVE_IMLIB) && defined(IMLIB_VERSION)
		g_print(
"        Imlib Version " IMLIB_VERSION "\n"
		);
#endif
#if defined(HAVE_IMLIB2) && defined(IMLIB2_VERSION)
		g_print(
"        Imlib2 Version " IMLIB2_VERSION "\n"
		);
#endif
#ifdef GTK_VERSION
		g_print(
"        GTK+ Version " GTK_VERSION "\n"
		);
#endif
#ifdef HAVE_LIBGIF
		ImgGIFVersion(&major, &minor, &release);
		g_print(
"        Uncompressed GIF Library Version %i.%i.%i\n",
		    major, minor, release
		);
#endif
#ifdef HAVE_LIBJPEG
		ImgJPEGVersion(&major, &minor, &release);
		g_print(
"        JPEG Library Version %i.%i.%i\n",
		    major, minor, release
		);
#endif
#ifdef HAVE_LIBPNG
		ImgPNGVersion(&major, &minor, &release);
		g_print(
"        PNG Library Version %i.%i.%i\n",
		    major, minor, release
		);
#endif
#ifdef HAVE_LIBXPM
		ImgXPMVersion(&major, &minor, &release);
		g_print(
"        XPM Library Version %i.%i.%i\n",
		    major, minor, release
		);
#endif            
		DELETE_FETCHED_CMD_LINE_VALUES;
		return(rtn_val);
	    }
	}

	/* Set GTK locale */
	gtk_set_locale();

	/* Initialize GTK as needed */
	if(!initialized_gtk)
	{
	    if(!gtk_init_check(&argc, &argv))
	    {
		g_printerr("Unable to initialize GTK.\n");
		DELETE_FETCHED_CMD_LINE_VALUES;
		rtn_val = 1;
		return(rtn_val);
	    }

	    /* Initialize GDK RGB buffers */
	    gdk_rgb_init();

	    initialized_gtk = TRUE;
	}


	/* Parse arguments */
	for(i = 1; i < argc; i++)
	{
	    gchar *parm, *val;
	    gchar *arg = (gchar *)argv[i];
	    if(arg == NULL)
		continue;

	    get_args_value(arg, &parm, &val);

	    /* Enable */
	    if(!g_strcasecmp(parm, "enable") ||
	       !g_strcasecmp(parm, "with")
	    )
	    {
		if(!g_strcasecmp(val, "resize_on_open"))
		    resize_on_open = TRUE;
		else if(!g_strcasecmp(val, "show_image_on_wm_icon"))
		    show_image_on_wm_icon = TRUE;
		else if(!g_strcasecmp(val, "show_values"))
		    show_values = TRUE;
		else if(!g_strcasecmp(val, "show_toolbar"))
		    show_toolbar = TRUE;
		else if(!g_strcasecmp(val, "show_statusbar"))
		    show_statusbar = TRUE;
		else if(!g_strcasecmp(val, "show_tooltips"))
		    show_tooltips = TRUE;
	    }
	    /* Disable */
	    else if(!g_strcasecmp(parm, "disable") ||
		    !g_strcasecmp(parm, "without")
	    )
	    {
		if(!g_strcasecmp(val, "resize_on_open"))
		    resize_on_open = FALSE;
		else if(!g_strcasecmp(val, "show_image_on_wm_icon"))
		    show_image_on_wm_icon = FALSE;
		else if(!g_strcasecmp(val, "show_values"))
		    show_values = FALSE;
		else if(!g_strcasecmp(val, "show_toolbar"))
		    show_toolbar = FALSE;
		else if(!g_strcasecmp(val, "show_statusbar"))
		    show_statusbar = FALSE;
		else if(!g_strcasecmp(val, "show_tooltips"))
		    show_tooltips = FALSE;
	    }
	    /* Startup Frame Index */
	    else if(!g_strcasecmp(parm, "frame") ||
		    !g_strcasecmp(parm, "f")
	    )
            {
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
                if(!STRISEMPTY(val))
                {
		    startup_frame_num = ATOI(val);
                }
	    }
	    /* Do Not Resize On Load */
	    else if(!g_strcasecmp(parm, "r"))
	    {
		resize_on_open = FALSE;
	    }
	    /* Show/Hide Tooltips */
	    else if(!g_strcasecmp(parm, "tooltips"))
	    {
		if(stristrue(val))
		    show_tooltips = TRUE;
		else
		    show_tooltips = FALSE;
	    }
	    /* Hide Tooltips */
	    else if(!g_strcasecmp(parm, "m"))
	    {
		show_tooltips = FALSE;
	    }
	    /* Show/Hide Tool Bar */
	    else if(!g_strcasecmp(parm, "toolbar"))
	    {
		if(stristrue(val))
		    show_toolbar = TRUE;
		else
		    show_toolbar = FALSE;
	    }
	    else if(!g_strcasecmp(parm, "t"))
	    {
		show_toolbar = FALSE;
	    }
	    /* Show/Hide Values */
	    else if(!g_strcasecmp(parm, "show_values"))
	    {
		if(stristrue(val))
		    show_values = TRUE;
		else
		    show_values = FALSE;
	    }
	    else if(!g_strcasecmp(parm, "v"))
	    {
		show_values = FALSE;
	    }
	    /* Show/Hide Status Bar */
	    else if(!g_strcasecmp(parm, "show_statusbar"))
	    {
		if(stristrue(val))
		    show_statusbar = TRUE;
		else
		    show_statusbar = FALSE;
	    }
	    else if(!g_strcasecmp(parm, "s"))
	    {
		show_statusbar = FALSE;
	    }
	    /* Show Image On WM Icon */
	    else if(!g_strcasecmp(parm, "imageonwmicon"))
	    {
		if(stristrue(val))
		    show_image_on_wm_icon = TRUE;
		else
		    show_image_on_wm_icon = FALSE;
	    }
	    else if(!g_strcasecmp(parm, "i"))
	    {
		show_image_on_wm_icon = TRUE;
	    }
	    /* Quality */
	    else if(!g_strcasecmp(parm, "q"))
	    {
		/* Value from 0 to 2 (2 being best/slowest) */
		quality = CLIP(ATOI(val), 0, 2);
	    }
	    /* Background Color */
	    else if(!g_strcasecmp(parm, "bg") ||
		    !g_strcasecmp(parm, "bg_color") ||
		    !g_strcasecmp(parm, "bgcolor")
	    )
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    GdkColor c;
		    if(gdk_color_parse(val, &c))
		    {
			bg_color[0] = (guint8)(c.red >> 8);
			bg_color[1] = (guint8)(c.green >> 8);
			bg_color[2] = (guint8)(c.blue >> 8);
			bg_color[3] = 0xff;
			bg_color_set = TRUE;
		    }
		    else
		    {
			g_printerr(
"%s: Unable to parse color \"%s\".\n",
			    parm, val
			);
		    }
		}
	    }
	    /* Imlib Fallback */
	    else if(!g_strcasecmp(parm, "b"))
	    {
		/* Value of 0 or 1 */
		imlib_fallback = CLIP(ATOI(val), 0, 1);
	    }
	    /* Automatically Hide IV when viewing Non-Image files */
	    else if(!g_strcasecmp(parm, "auto_hide_iv_nonimage") ||
		    !g_strcasecmp(parm, "autohideivnonimage") ||
		    !g_strcasecmp(parm, "k")
	    )
	    {
		auto_hide_iv_nonimage = TRUE;
	    }
	    /* Display Simple Non-graphical Title On Startup */
	    else if(!g_strcasecmp(parm, "simple_title") ||
		    !g_strcasecmp(parm, "simpletitle")
	    )
	    {
		simple_title = TRUE;
	    }
	    /* Screenshot On Startup */
	    else if(!g_strcasecmp(parm, "screenshot") ||
		    !g_strcasecmp(parm, "grab") ||
		    !g_strcasecmp(parm, "g")
	    )
	    {
		startup_screenshot = TRUE;
	    }
	    /* Fullscreen */
	    else if(!g_strcasecmp(parm, "fullscreen") ||
		    !g_strcasecmp(parm, "fs")
	    )
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm 
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    fullscreen_path = val;
		}
	    }
	    /* Display Image On Desktop */
	    else if(!g_strcasecmp(parm, "root") ||
		    !g_strcasecmp(parm, "desktop")
	    )
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    if(strstr(arg, "centered") ||
		       strstr(arg, "center")
		    )
			display_on_root_centered_path = val;
		    else if(strstr(arg, "tofit"))
			display_on_root_tofit_path = val;
		    else if(strstr(arg, "tocover"))
			display_on_root_tocover_path = val;
		    else if(strstr(arg, "tiled") ||
			    strstr(arg, "tile")
		    )
			display_on_root_tiled_path = val;
		}
	    }
	    /* Print */
	    else if(!g_strcasecmp(parm, "print"))
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    print_path = val;
		}
	    }
	    /* Font */
	    else if(!g_strcasecmp(parm, "fontname") ||
		    !g_strcasecmp(parm, "font") ||
		    !g_strcasecmp(parm, "fn")
	    )
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    g_free(font_name);
		    font_name = STRDUP(val);
		}
	    }
	    /* WM Name */
	    else if(!g_strcasecmp(parm, "wmname"))
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    g_free(wmname);
		    wmname = STRDUP(val);
		}
	    }
	    /* WM Class */
	    else if(!g_strcasecmp(parm, "wmclass"))
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm   
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    g_free(wmclass);
		    wmclass = STRDUP(val);
		}
	    }
	    /* Title */
	    else if(!g_strcasecmp(parm, "title"))
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
			val = argv[i];
		    }
		    else 
		    {
			val = "";
			g_printerr(
"%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    g_free(title);
		    title = STRDUP(val);
		}
	    }
	    /* Geometry */
	    else if(!g_strcasecmp(parm, "geometry"))
	    {
		/* Need to get argument from next argument? */
		if(STRISEMPTY(val))
		{
		    i++;
		    if(i < argc)
		    {
		        val = argv[i];
		    }
		    else
		    {
			val = "";
			g_printerr(
			    "%s: Requires argument.\n",
			    parm
			);
		    }
		}
		/* Argument specified? */
		if(!STRISEMPTY(val))
		{
		    gint x, y, width, height;
		    GdkGeometryFlags flags = gdk_parse_geometry(
			val, &x, &y, &width, &height
		    );
		    if(flags)
		    {
			if(geometry == NULL)
			    geometry = (GdkRectangle *)g_malloc0(
				sizeof(GdkRectangle)
			    );
			geometry->x = (flags & GDK_GEOMETRY_X) ? x : 0;
			geometry->y = (flags & GDK_GEOMETRY_Y) ? y : 0;
			geometry->width =
			    (flags & GDK_GEOMETRY_WIDTH) ? width : 0;
			geometry->height =
			    (flags & GDK_GEOMETRY_HEIGHT) ? height : 0;

			/* Setting geometry implies that we need to
			 * disable resize_on_open
			 */
			resize_on_open = FALSE;
		    }
		}
	    }

	    /* File name (check if arg is not a toggle)? */
	    else if((*arg != '-') &&
		    (*arg != '+')
	    )
	    {
		/* Add this argument as a file in the list of files
		 * to open on startup
		 */
		file_list = g_list_append(file_list, STRDUP(arg));
	    }
	    /* Unsupported argument */
	    else
	    {
		g_printerr(
"%s: Unsupported argument.\n",
		    parm
		);
		DELETE_FETCHED_CMD_LINE_VALUES;
		rtn_val = 2;
		return(rtn_val);
	    }
	}


	/* Create the core */
	core = IV_CORE(g_malloc0(sizeof(iv_core_struct)));
	if(core == NULL)
	{
	    DELETE_FETCHED_CMD_LINE_VALUES;
	    rtn_val = 3;
	    return(rtn_val);
	}

#undef DELETE_FETCHED_CMD_LINE_VALUES

	/* Reset the configuration values on the core */
	iv_core_reset(core);

	core->cur_dir = STRDUP(g_get_current_dir());
	core->iv_viewer = NULL;
	core->iv_fviewer = NULL;
	core->imginfodlg = NULL;
	core->close_all_windows = FALSE;
	core->untitled_count = 0;
	core->grab_cur = gdk_cursor_new(GDK_CROSSHAIR);
	core->grab_window = NULL;

	/* Options */
	core->show_tooltips = show_tooltips;
	core->show_toolbar = show_toolbar;
	core->show_values = show_values;
	core->show_statusbar = show_statusbar;
	core->show_image_on_wm_icon = show_image_on_wm_icon;
	core->quality = quality;
	memcpy(core->bg_color, bg_color, sizeof(bg_color));
	core->bg_color_set = bg_color_set;
	core->font_name = font_name;
	core->wmname = wmname;
	core->wmclass = wmclass;
	core->title = title;
	core->geometry = geometry;

	core->resize_on_open = resize_on_open;
	core->resize_maintain_aspect = TRUE;

	core->imlib_fallback = imlib_fallback;

	core->auto_hide_iv_nonimage = auto_hide_iv_nonimage;
	core->simple_title = simple_title;

#ifdef HAVE_LIBENDEAVOUR2
	/* Initialize the Endeavour 2 API context */
	core->edv2_ctx = EDVContextNew();
	EDVContextLoadConfigurationFile(core->edv2_ctx, NULL);
#endif	/* HAVE_LIBENDEAVOUR2 */


	/* Set the pointer to the main IV Core */
	iv_core = core;

/* Deletes the IV Core, all other resources, and returns */
#define DELETE_ALL_AND_RETURN(_status_)	{	\
						\
 /* Delete the core */				\
 iv_core = NULL;				\
 if(core != NULL) {				\
  iv_core_delete(core);				\
  core = NULL;					\
 }						\
						\
 /* Shutdown the dialogs */			\
 if(initialized_dialogs) {			\
  FSDShutdown();				\
  CSDShutdown();				\
  PDialogShutdown();				\
  FileBrowserShutdown();			\
  ProgressDialogShutdown();			\
  CDialogShutdown();				\
  FPromptShutdown();				\
  initialized_dialogs = FALSE;			\
 }						\
						\
 /* Delete the list of startup files */		\
 g_list_foreach(file_list, (GFunc)g_free, NULL);\
 g_list_free(file_list);			\
 file_list = NULL;				\
						\
 return(_status_);				\
}


#if defined(HAVE_IMLIB)
	/* Initialize Imlib as needed */
	if(!initialized_imlib)
	{
	    imlib_handle = Imlib_init(GDK_DISPLAY());
	    if(imlib_handle != NULL)
		initialized_imlib = TRUE;
	    else
		g_printerr("Unable to initialize Imlib.\n");
	}
#endif	/* HAVE_IMLIB */


	/* Check and handle Special Startup cases first */

	/* Fullscreen */
	if(!STRISEMPTY(fullscreen_path))
	{
	    const gint status = iv_open_fullscreen(
		fullscreen_path,
		startup_frame_num,
		core->bg_color,
		quality
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}
	/* Display image centered on the desktop and exit? */
	else if(!STRISEMPTY(display_on_root_centered_path))
	{
	    const gint status = iv_open_to_root_centered(
		display_on_root_centered_path,
		startup_frame_num,
		core->bg_color,
		quality
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}
	/* Display image to fit on the desktop and exit? */
	else if(!STRISEMPTY(display_on_root_tofit_path))
	{
	    const gint status = iv_open_to_root_tofit(
		display_on_root_tofit_path,
		startup_frame_num,
		core->bg_color,
		quality
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}
	/* Display image to cover the desktop and exit? */
	else if(!STRISEMPTY(display_on_root_tocover_path))
	{
	    const gint status = iv_open_to_root_tocover(
		display_on_root_tocover_path,
		startup_frame_num,
		core->bg_color,
		quality
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}
	/* Display image tiled on the desktop and exit? */
	else if(!STRISEMPTY(display_on_root_tiled_path))
	{
	    const gint status = iv_open_to_root_tiled(
		display_on_root_tiled_path,
		startup_frame_num,
		core->bg_color,
		quality
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}
	/* Print? */
	else if(!STRISEMPTY(print_path))
	{
	    const gint status = iv_open_print(
		print_path,
		startup_frame_num,
		core->bg_color
	    );
	    rtn_val = ABSOLUTE_VALUE(status);
	    DELETE_ALL_AND_RETURN(rtn_val);
	}


	/* If this point is reached then there were no Special Startup
	 * cases, so continue with Standard Startup
	 */

	/* Update the global tooltips state */
	GUISetGlobalTipsState(core->show_tooltips);

	/* Initialize the dialogs */
	if(!initialized_dialogs)
	{
	    FPromptInit(); 
	    CDialogInit();
	    ProgressDialogInit();
	    FileBrowserInit();
	    PDialogInit();
	    CSDInit();
	    FSDInit();
	    initialized_dialogs = TRUE;

	    /* Set the file selector callbacks */
	    FileBrowserSetObjectCreatedCB(
		iv_file_selector_created_cb, core
	    );
	    FileBrowserSetObjectModifiedCB(
		iv_file_selector_modified_cb, core
	    );
	    FileBrowserSetObjectDeletedCB(
		iv_file_selector_deleted_cb, core
	    );
	}

	/* Create the Image Viewer(s) */
	i = (gint)g_list_length(file_list);
	if(i > 0)
	{
	    /* One or more files to open on startup, create an Image
	     * Viewer for each file in the startup files list
	     */
	    const gchar *path;
	    GList *glist;
	    imgview_struct *iv;
	    iv_image_viewer_struct *v;

	    /* Warn if there are more than 25 objects to open */
	    if(i > 25)
	    {
		gchar *msg = g_strdup_printf(
"There are %i objects that have been specified\n\
to be opened, are you sure you want to open all\n\
of them?",
		    i
		);
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
		    "Confirm Open Multiple Objects",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);
		rtn_val = 4;
		DELETE_ALL_AND_RETURN(rtn_val);
	    }

#if !defined(_WIN32)
	    /* Warn if opening objects and doing screenshot at startup */
	    if(startup_screenshot)
	    {
		CDialogSetTransientFor(NULL);
		CDialogGetResponse(
		    "Screenshot Failed",
"Unable to take screenshot and open objects at\n\
startup. Please specify only one of these options\n\
at startup.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		rtn_val = 2;
		DELETE_ALL_AND_RETURN(rtn_val);
	    }
#endif	/* Not _WIN32 */

	    /* Create an Image Viewer and open each object */
	    for(glist = file_list; glist != NULL; glist = g_list_next(glist))
	    {
		path = (const gchar *)glist->data;
		if(STRISEMPTY(path))
		    continue;

		/* Create a new Image Viewer */
		v = iv_viewer_new(
		    core,
		    show_toolbar,
		    show_values,
		    show_statusbar,
		    show_image_on_wm_icon,
		    quality,
		    bg_color_set ? bg_color : NULL,
		    geometry,
		    FALSE			/* Do not load title */
		);
		if(v == NULL)
		    break;

		iv = v->iv;

		/* Update current directory if the path is absolute */
		if(g_path_is_absolute(path))
		{
		    gchar *parent = g_dirname(path);
		    if(parent != NULL)
		    {
			g_free(core->cur_dir);
			core->cur_dir = STRDUP(parent);
			chdir(parent);
			g_free(parent);
		    }
		}

		ImgViewSetBusy(iv, TRUE);
		v->processing = TRUE;

		/* Open the image to the new Image Viewer */
		iv_viewer_open(
		    v,
		    path,
		    startup_frame_num,
		    core->resize_on_open
		);

		v->processing = FALSE;
		ImgViewSetBusy(iv, FALSE);
 	    }
	}
	else
	{
	    /* No files to open on startup, create one Image Viewer
	     * and have it display the title image
	     */
	    iv_image_viewer_struct *v = iv_viewer_new(
		core,
		show_toolbar,
		show_values,
		show_statusbar,
		show_image_on_wm_icon,
		quality,
		bg_color_set ? bg_color : NULL,
		geometry,
		TRUE				/* Display title image */
	    );
	    if(v != NULL)
	    {
#if !defined(_WIN32)
		/* Map the Screenshot Dialog on startup? */
		if(startup_screenshot)
		    iv_viewer_screenshot_cb(v->screenshot_mi, v);
#endif

	    }
	}

	/* Set the main timeout callback */
	gtk_timeout_add(
	    1000l,
	    iv_main_timeout_cb, core
	);

	/* Enter main loop */
	gtk_main();

#if defined(HAVE_IMLIB)
	/* Imlib does not need to be shut down */
	imlib_handle = NULL;
#endif  /* HAVE_IMLIB */

	DELETE_ALL_AND_RETURN(rtn_val);
#undef DELETE_ALL_AND_RETURN
}
