#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include "guiutils.h"
#include "guirgbimg.h"

#include "imgview.h"
#include "print_values.h"
#include "print.h"
#include "config.h"


typedef struct _PrintProgressData	PrintProgressData;
#define PRINT_PROGRESS_DATA(p)		((PrintProgressData *)(p))


struct _PrintProgressData {
	gulong		cur_stage,
			nstages,
			m,			/* nunits per stage */
			m2;			/* m * nstages */
	gboolean	aborted;
	gint		(*progress_cb)(const gulong, const gulong, gpointer);
	gpointer	progress_data;
};


static gint PrintProgressCB(
	const gulong i, const gulong m, gpointer data
);

imgview_image_struct *PrintRenderImageFromValues(
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,			/* 4 bytes RGBA */
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);
void PrintAdjustImageColor(
	imgview_image_struct *img,
	const print_values_struct *pv,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Print progress callback.
 */
static gint PrintProgressCB(
	const gulong i, const gulong m, gpointer data
)
{
	PrintProgressData *d = PRINT_PROGRESS_DATA(data);
	if(d == NULL)
	    return(-2);

	if(d->progress_cb != NULL)
	{
	    if(d->progress_cb(
		(d->cur_stage * d->m) + i,
		d->m2,
		d->progress_data
	    ))
	    {
		d->aborted = TRUE;
		return(-4);
	    }
	    else
	    {
		return(0);
	    }
	}
	else
	{
	    return(0);
	}
}


/*
 *	Creates/renders a new image from the specified values.
 *
 *	The img specifies the source image.
 *
 *	The pv specifies the values in which to render the image
 *	with. The values that will be used are; visual, contrast,
 *	brightness, orientation, output_width, and output_height.
 *
 *	The bg_color specifies the 4 byte RGBA background color.
 *
 *	Returns the rendered image or NULL on error.
 */
imgview_image_struct *PrintRenderImageFromValues(
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,			/* 4 bytes RGBA */
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	guint vflags;
	gint i, frame_num, bpp;
#if !defined(HAVE_XPRINT)
	print_orientation orientation = PRINT_ORIENTATION_PORTRAIT;
#endif
	gint	output_width = 0,
		output_height = 0;
	imgview_frame_struct	*src_frame,
				*tar_frame;
	imgview_image_struct	*src_img = img,
				*tar_img;
	PrintProgressData *d;

	if((src_img == NULL) || (pv == NULL))
	    return(NULL);

	frame_num = (pv->flags & PRINT_VALUE_FRAME) ?
	    pv->frame : 0;
	src_frame = ImgViewImageGetFrame(src_img, frame_num);
	if(src_frame == NULL)
	    return(NULL);
	if(src_frame->buf == NULL)
	    return(NULL);

	bpp = src_img->bpp;

	/* Get the print values */
	vflags = pv->flags;
#if !defined(HAVE_XPRINT)
	if(vflags & PRINT_VALUE_ORIENTATION)
	    orientation = pv->orientation;
#endif
	if(vflags & PRINT_VALUE_OUTPUT_WIDTH)
	    output_width = pv->output_width;
	if(vflags & PRINT_VALUE_OUTPUT_HEIGHT)
	    output_height = pv->output_height;

	/* Output size too small? */
	if((output_width <= 0) || (output_height <= 0))
	    return(NULL);

	/* Allocate the progress callback data */
	d = PRINT_PROGRESS_DATA(g_malloc0(
	    sizeof(PrintProgressData)
	));
	if(d == NULL)
	    return(NULL);

	d->cur_stage = 0;
	d->nstages = 3;
	d->m = output_height;
	d->m2 = d->m * d->nstages;
	d->aborted = FALSE;
	d->progress_cb = progress_cb;
	d->progress_data = progress_data;

	/* Create the target image as the rendered image */
	tar_img = ImgViewImageNew(
	    output_width, output_height, bpp, 0
	);
	if(tar_img == NULL)
	{
	    g_free(d);
	    return(NULL);
	}
	i = ImgViewImageAppendFrameNew(tar_img);
	tar_frame = ImgViewImageGetFrame(tar_img, i);


#if !defined(HAVE_XPRINT)
	/* Change the orientation? */
	if(orientation == PRINT_ORIENTATION_LANDSCAPE)
	{
	    /* Create a tempory image for rotating */
	    imgview_frame_struct *tmp_frame;
	    imgview_image_struct *tmp_img = ImgViewImageNew(
	        src_img->height, src_img->width, bpp, 0
	    );
	    i = ImgViewImageAppendFrameNew(tmp_img);
	    tmp_frame = ImgViewImageGetFrame(tmp_img, i);
	    if(tmp_frame != NULL)
	    {
		/* Copy/rotate to the tempory image */
		GUIImageBufferRotateCCW90(
		    bpp,
		    src_frame->buf,
		    src_img->width, src_img->height, src_img->bpl,
		    tmp_frame->buf,
		    tmp_img->width, tmp_img->height, tmp_img->bpl
		);

		/* Copy/resize back to the target image */
	        GUIImageBufferResize(
		    bpp,
		    tmp_frame->buf,
		    tmp_img->width, tmp_img->height, tmp_img->bpl,
		    tar_frame->buf,
		    tar_img->width, tar_img->height, tar_img->bpl,
		    PrintProgressCB, d
	        );
	    }

	    /* Delete the tempory image */
	    ImgViewImageDelete(tmp_img);
	}
	else
#endif
	{
	    /* Copy/resize */
	    GUIImageBufferResize(
		bpp,
		src_frame->buf,
		src_img->width, src_img->height, src_img->bpl,
		tar_frame->buf,
		tar_img->width, tar_img->height, tar_img->bpl,
		PrintProgressCB, d
	    );
	}

	/* User aborted? */
	if(d->aborted)
	{
	    g_free(d);
	    return(tar_img);
	}

	d->cur_stage++;

	/* Apply the background color to the target image */
	GUIImageBufferFlattenWithBG(
	    bpp,
	    tar_frame->buf,
	    tar_img->width, tar_img->height, tar_img->bpl,
	    bg_color,
	    PrintProgressCB, d
	);

	/* User aborted? */
	if(d->aborted)
	{
	    g_free(d);
	    return(tar_img);
	}

	d->cur_stage++;

	/* Modify the target image's color, brightness, and contrast */
	PrintAdjustImageColor(
	    tar_img,
	    pv,
	    PrintProgressCB, d
	);

	/* Delete the progress callback data */
	g_free(d);

	return(tar_img);
}


/*
 *	Modifies the image's color, brightness, and contrast.
 *
 *	The img specifies the image.
 *
 *	The pv specifies the color, brightness, and contrast.
 */
void PrintAdjustImageColor(
	imgview_image_struct *img,
	const print_values_struct *pv,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	guint vflags;
	print_visual visual = PRINT_VISUAL_BW;
	gfloat	brightness = 0.0f,
		contrast = 0.0f;
	gint	v, y,
		width, height,
		bpp, bpl,
		brightness_delta;
	guint8 *data, *data_ptr, *data_end;
	imgview_frame_struct *frame;

	if((img == NULL) || (pv == NULL))
	    return;

	/* Get image values */
	width = img->width;
	height = img->height;
	bpp = img->bpp;
	bpl = img->bpl;
	frame = ImgViewImageGetFrame(img, 0);
	data = (frame != NULL) ? frame->buf : NULL;
	if(data == NULL)
	    return;

	if(progress_cb != NULL)
	{
	    if(progress_cb(0l, (gulong)height, progress_data))
		return;
	}

	vflags = pv->flags;
	if(vflags & PRINT_VALUE_VISUAL)
	    visual = pv->visual;
	if(vflags & PRINT_VALUE_BRIGHTNESS)
	    brightness = pv->brightness;
	if(vflags & PRINT_VALUE_CONTRAST)
	    contrast = pv->contrast;

	/* Amplify the positive contrast if the contrast is positive */
	if(contrast > 0.0f)
	    contrast *= 5.0f;

	/* Calculate the brightness delta byte value */
	brightness_delta = (gint)(0xff * brightness);

	switch(bpp)
	{
	  case 4:	/* RGBA */
	  case 3:	/* RGB */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}

		data_ptr = data + (y * bpl);
		data_end = data_ptr + (width * bpp);

		while(data_ptr < data_end)
		{
		    /* Adjust the visual */
		    switch(visual)
		    {
		     case PRINT_VISUAL_BW:
			v = (gint)(
			    (0.299 * data_ptr[0]) +
			    (0.587 * data_ptr[1]) +
			    (0.114 * data_ptr[2])
			);
			v = (v >= 0x80) ? 0xff : 0x00;
			data_ptr[0] = (guint8)v;
			data_ptr[1] = (guint8)v;
			data_ptr[2] = (guint8)v;
			break;

		      case PRINT_VISUAL_GREYSCALE:
			v = (gint)(
			    (0.299 * data_ptr[0]) +
			    (0.587 * data_ptr[1]) +
			    (0.114 * data_ptr[2])
			);
			data_ptr[0] = (guint8)v;
			data_ptr[1] = (guint8)v;
			data_ptr[2] = (guint8)v;
			break;

		      case PRINT_VISUAL_RGB:
			break;
		    }

		    /* Adjust the brightness? */
		    if(brightness_delta != 0)
		    {
			data_ptr[0] = (guint8)CLIP(
			    ((gint)data_ptr[0] + brightness_delta),
			    0x00, 0xff
			);
			data_ptr[1] = (guint8)CLIP(
			    ((gint)data_ptr[1] + brightness_delta),
			    0x00, 0xff
			);
			data_ptr[2] = (guint8)CLIP(
			    ((gint)data_ptr[2] + brightness_delta),
			    0x00, 0xff
			);
		    }
		    /* Adjust the contrast? */
		    if(contrast != 0.0f)
		    {
			v = (gint)(data_ptr[0]);
			data_ptr[0] = (guint8)CLIP(
			    (v + (gint)((v - (gint)0x80) * contrast)),
			    0x00, 0xff
			);
			v = (gint)(data_ptr[1]);
			data_ptr[1] = (guint8)CLIP(
			    (v + (gint)((v - (gint)0x80) * contrast)),
			    0x00, 0xff
			);
			v = (gint)(data_ptr[2]);
			data_ptr[2] = (guint8)CLIP(
			    (v + (gint)((v - (gint)0x80) * contrast)),
			    0x00, 0xff
			);
		    }
		    data_ptr += bpp;
		}
	    }
	    break;

	  case 2:	/* Greyscale Alpha */
	  case 1:	/* Greyscale */
	    for(y = 0; y < height; y++)
	    {
		if((progress_cb != NULL) && ((y % 10) == 0))
		{
		    if(progress_cb((gulong)y, (gulong)height, progress_data))
			return;
		}

		data_ptr = data + (y * bpl);
		data_end = data_ptr + (width * bpp);

		while(data_ptr < data_end)
		{
		    /* Adjust the visual */
		    switch(visual)
		    {
		     case PRINT_VISUAL_BW:
			data_ptr[0] = (data_ptr[0] >= 0x80) ? 0xff : 0x00;
			break;

		      case PRINT_VISUAL_GREYSCALE:
			break;

		      case PRINT_VISUAL_RGB:
			break;
		    }

		    /* Adjust the brightness? */
		    if(brightness_delta != 0)
		    {
			data_ptr[0] = (guint8)CLIP(
			    ((gint)data_ptr[0] + brightness_delta),
			    0x00, 0xff
			);
		    }
		    /* Adjust the contrast? */
		    if(contrast != 0.0f)
		    {
			v = (gint)data_ptr[0];
			data_ptr[0] = (guint8)CLIP(
			    (v + (gint)((v - (gint)0x80) * contrast)),
			    0x00, 0xff
			);
		    }
		    data_ptr += bpp;
		}
	    }
	    break;
	}

	if(progress_cb != NULL)
	{
	    if(progress_cb((gulong)height, (gulong)height, progress_data))
		return;
	}
}
