#if !defined(_WIN32) && !defined(HAVE_XPRINT)
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#ifdef HAVE_LIBENDEAVOUR2
# include <endeavour2.h>
#endif
#include <unistd.h>

#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "libps.h"
#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "imgview.h"
#include "print_values.h"
#include "print.h"
#include "config.h"

#include "images/icon_print2_32x32.xpm"


typedef struct _ps_progress_data_struct	ps_progress_data_struct;
#define PS_PROGRESS_DATA(p)	((ps_progress_data_struct *)(p))


/*
 *	Progress Data:
 *
 *	Used for the progress callback data while writing PostScript
 *	files
 */
struct _ps_progress_data_struct {

	gchar		*filename,
			*message;

	GtkWidget	*toplevel;
};


static gint print_lpr_ps_write_progress_cb(
	gpointer data,
	gint cur, gint max,
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba
);


#ifdef HAVE_LIBENDEAVOUR2
static gint iv_print_lpr_to_file(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
);
#else
static gint iv_print_lpr_to_file(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
);
#endif
static gint iv_print_lpr_to_printer(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
);

#ifdef HAVE_LIBENDEAVOUR2
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
);
#else
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
);
#endif


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	PostScript file write progress callback.
 *
 *	Called when writing a PostScript file.
 *
 *	A non-zero return means to abort the operation.
 */
static gint print_lpr_ps_write_progress_cb(
	gpointer data,
	gint cur, gint max,
	gint width, gint height,
	gint bpl, gint bpp,
	guint8 *rgba
)
{
	gfloat progress_coeff;
	ps_progress_data_struct *cb_data = PS_PROGRESS_DATA(data);
	if(cb_data == NULL)
	    return(FALSE);

	/* Calculate progress coefficient from 0.0 to 1.0 */
	progress_coeff = CLIP(
	    ((max > 0) ? (gfloat)cur / (gfloat)max : 0.0f),
	    0.0f, 1.0f
	);

	/* If progress dialog is not mapped, then map it and set
	 * initial values
	 */
	if(!ProgressDialogIsQuery())
	{
	    ProgressDialogSetTransientFor(cb_data->toplevel);
	    ProgressDialogMap(
		"Printing",
		cb_data->message,
		(const guint8 **)icon_print2_32x32_xpm,
		"Stop"
	    );
	}

	/* Update the progress */
	ProgressDialogUpdate(
	    NULL, NULL, NULL, NULL,
	    progress_coeff, 25, TRUE
	);

	gdk_flush();

	return((ProgressDialogStopCount() > 0) ? FALSE : TRUE);
}


/*
 *	Generates a PostScript file but does not send it to the printer.
 */
#ifdef HAVE_LIBENDEAVOUR2
static gint iv_print_lpr_to_file(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
)
#else
static gint iv_print_lpr_to_file(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
)
#endif
{
	struct stat stat_buf;
	gboolean file_exists;
	const guint vflags = pv->flags;
	gint		status,
			output_x = 0,
			output_y = 0,
			paper_width = 320,
			paper_height = 240,
			paper_widthm = 320 * 0.72f,
			paper_heightm = 240 * 0.72f;
	const gchar	*filename = NULL,
			*ext_ptr,
			*creator = PROG_NAME " Version " PROG_VERSION;
	imgview_frame_struct *frame = ImgViewImageGetFrame(img, 0);
	if(frame == NULL)
	    return(-1);

	if(vflags & PRINT_VALUE_FILE_NAME)
	    filename = pv->file_name;

	if(vflags & PRINT_VALUE_OUTPUT_X)
	    output_x = pv->output_x;
	if(vflags & PRINT_VALUE_OUTPUT_Y)
	    output_y = pv->output_y;

	if((vflags & PRINT_VALUE_PAPER_WIDTH) &&
	   (vflags & PRINT_VALUE_DPI_X)
	)
	{
	    paper_width = pv->paper_width;
	    paper_widthm = (gint)(
		(gfloat)paper_width * (gfloat)pv->dpi_x / 100.0f
	    );
	}
	if((vflags & PRINT_VALUE_PAPER_HEIGHT) &&
	   (vflags & PRINT_VALUE_DPI_Y)
	)
	{
	    paper_height = pv->paper_height;
	    paper_heightm = (gint)(
		(gfloat)paper_height * (gfloat)pv->dpi_y / 100.0f
	    );
	}

	/* Paper size valid? */
	if((paper_width <= 0) || (paper_height <= 0))
	    return(-2);

	/* Did we get a file name to save to from the print values? */
	if(filename == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
		"The file name was not specified.",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return(-1);
	}

	/* The extension of the file name must be one of the accepted
	 * post script formats
	 */
	ext_ptr = strrchr(filename, '.');
	if(ext_ptr == NULL)
	    ext_ptr = "";
	/* Check extension to see if it matches atleast one of the known
	 * PostScript file format extensions
	 */
	if(strcasecmp(ext_ptr, FTYPE_EXT_PS) &&
	   strcasecmp(ext_ptr, FTYPE_EXT_PS2) &&
	   strcasecmp(ext_ptr, FTYPE_EXT_PS3)
	)
	{
	    /* Given file name does not have the proper extension to
	     * indicate that it is a PostScript file
	     */
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
"Print Failed",
"Invalid file type extension.",
"The file name's file type extension must indicate\n\
that it is a PostScript file. Try specifying .ps as\n\
the file name's extension.",
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return(-2);
	}

	/* Check if image already exists and if so then prompt for
	 * overwrite
	 */
	file_exists = stat(filename, &stat_buf) ? FALSE : TRUE;
	if(file_exists)
	{
	    /* Confirm overwrite */
	    gchar *buf = g_strdup_printf(
		"Overwrite existing file:\n\n    %s",
		filename
	    );
	    CDialogSetTransientFor(toplevel);
	    status = CDialogGetResponse(
		"Confirm Overwrite",
		buf,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    switch(status)
	    {
	      case CDIALOG_RESPONSE_YES:
	      case CDIALOG_RESPONSE_YES_TO_ALL:
	      case CDIALOG_RESPONSE_OK:
		break;

	      default:
		return(-4);
		break;
	    }
	}


	/* Reset status to -1 to indicate error, it will be set to 0
	 * later if things are successful
	 */
	status = -1;

	/* Write PostScript file */
	while(TRUE)
	{
	    gint user_aborted = FALSE;
	    const gchar *mesg;
	    PSFormatStruct fmt;
	    ps_progress_data_struct *cb_data;

	    /* Open the PostScript file for writing */
	    FILE *fp = FOpen(filename, "wb");
	    if(fp == NULL)
	    {
		gchar *s = g_strdup_printf(
"Unable to open the file for writing:\n\n\
    %s",
		    filename
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    s, NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(s);
		status = -1;
		break;
	    }

	    /* Set up the PostScript format values */
	    fmt.flags = PSFlagDPIX | PSFlagDPIY |
		PSFlagPaperWidth | PSFlagPaperHeight |
		PSFlagOutputX | PSFlagOutputY;
	    /* DPI resolution */
	    fmt.dpi_x = (vflags & PRINT_VALUE_DPI_X) ?
		pv->dpi_x : 72;
	    fmt.dpi_y = (vflags & PRINT_VALUE_DPI_Y) ?
		pv->dpi_y : 72;
	    /* Paper size without dpi applied */
	    fmt.paper_width = paper_width;
	    fmt.paper_height = paper_height;
	    /* Output image offset */
	    fmt.output_x = output_x;
	    fmt.output_y = output_y;

	    /* Set up the progress callback */
	    cb_data = PS_PROGRESS_DATA(
		g_malloc0(sizeof(ps_progress_data_struct))
	    );
	    if(cb_data != NULL)
	    {
		cb_data->toplevel = toplevel;
		cb_data->filename = g_strdup(filename);
		cb_data->message = g_strdup_printf(
"Generating PostScript file:\n\
\n\
    %s\n",
		    filename
		);
	    }

	    /* Write the PostScript file based on the source image's
	     * bytes per pixel
	     */
	    switch(img->bpp)
	    {
	      case 4:
		status = PSWriteFilePS3RGBA(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      case 3:
		status = PSWriteFilePS3RGB(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      case 1:
		status = PSWriteFilePS3Grey(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      default:
		status = PSError;
		break;
	    }

	    /* Close the PostScript file */
	    FClose(fp);

	    /* Unmap progress dialog which might have been mapped when
	     * print_lpr_ps_write_progress_cb() was called
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Delete the callback data */
	    if(cb_data != NULL)
	    {
		g_free(cb_data->filename);
		g_free(cb_data->message);
		g_free(cb_data);
	    }

	    /* Check for error */
	    mesg = NULL;
	    switch(status)
	    {
	      case PSSuccess:
		status = 0;
		break;
	      case PSError:
		mesg = "Error writing PostScript file.";
		status = -1;
		break;
	      case PSBadValue:
		mesg = "Bad value used to write PostScript file.";
		status = -2;
		break;
	      case PSErrorSystem:
		mesg = "Systems error occured while writing PostScript file.";
		status = -3;
		break;
	      case PSAbort:
		status = -4;
		break;
	      case PSErrorInternal:
		mesg = "Systems error occured while writing PostScript file.";
		status = -3;		/* Treat as systems error */
		break;
	      default:
		status = -1;
		break;
	    }

	    if(mesg != NULL)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    mesg,
		    NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }

#ifdef HAVE_LIBENDEAVOUR2
	    /* Report the new file */
            if(file_exists)
                EDVNotifyQueueObjectModified(
                    edv2_ctx, filename, filename
                );
            else  
                EDVNotifyQueueObjectAdded(
                    edv2_ctx, filename
                );
            EDVContextSync(edv2_ctx);
#endif	/* HAVE_LIBENDEAVOUR2 */

	    /* Break when we reach the end of this big while() loop */
	    break;
	}

	return(status);
}


/*
 *	Generate PostScript file and send it to the printer to print it.
 */
static gint iv_print_lpr_to_printer(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
)
{
	gint status, pid;
	const guint vflags = pv->flags;
	gint	output_x = 0,
		output_y = 0,
		paper_width = 320,
		paper_height = 240,
		paper_widthm = 320 * 0.72f,
		paper_heightm = 240 * 0.72,
		ncoppies = 1;
	const gchar *print_command = NULL;
	gchar	*filename = NULL,
		*command = NULL;
	const gchar *creator = PROG_NAME " Version " PROG_VERSION;
	imgview_frame_struct *frame = ImgViewImageGetFrame(img, 0);
	if(frame == NULL)
	    return(-1);

#define CLEANUP_RETURN(_v_)	{		\
 if(filename != NULL) {				\
  unlink(filename);				\
  g_free(filename);				\
 }						\
						\
 g_free(command);				\
						\
 return(_v_);					\
}

	/* Get print values */
	if(vflags & PRINT_VALUE_PRINT_COMMAND)
	    print_command = pv->print_command;

	if(vflags & PRINT_VALUE_OUTPUT_X)
	    output_x = pv->output_x;
	if(vflags & PRINT_VALUE_OUTPUT_Y)
	    output_y = pv->output_y;

	if((vflags & PRINT_VALUE_PAPER_WIDTH) &&
	   (vflags & PRINT_VALUE_DPI_X)
	)
	{
	    paper_width = pv->paper_width;
	    paper_widthm = (gint)(
		(gfloat)paper_width * (gfloat)pv->dpi_x / 100.0f
	    );
	}
	if((vflags & PRINT_VALUE_PAPER_HEIGHT) &&
	   (vflags & PRINT_VALUE_DPI_Y)
	)
	{
	    paper_height = pv->paper_height;
	    paper_heightm = (gint)(
		(gfloat)paper_height * (gfloat)pv->dpi_y / 100.0f
	    );
	}

	if(vflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	    ncoppies = pv->output_ncoppies;

	/* Paper size valid? */
	if((paper_width <= 0) || (paper_height <= 0))
	{
	    CLEANUP_RETURN(-2);
	}

	/* No command specified? */
	if(print_command == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
		"Print command is not set.",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    CLEANUP_RETURN(-1);
	}

	/* Begin formulating a tempory file name and the print command */

	/* Generate tempory file name */
	if(filename == NULL)
	{
	    int fd;
	    const gchar *tmp_dir = g_getenv("TMPDIR");
	    if(STRISEMPTY(tmp_dir))
#if defined(P_tmpdir)
		tmp_dir = P_tmpdir;
#elif defined(_WIN32)
		tmp_dir = "C:\\TEMP";
#else
		tmp_dir = "/tmp";
#endif
	    filename = g_strconcat(
		tmp_dir,
		G_DIR_SEPARATOR_S,
		PROG_NAME "XXXXXX",
		NULL
	    );
	    fd = mkstemp((char *)filename);
	    if(fd > -1)
		close(fd);
	}
	/* Failed to generate file name? */
	if(filename == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
"Unable to generate a tempory filename for the PostScript file.\n",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    CLEANUP_RETURN(-1);
	}

	/* Format the print command */
	if(command == NULL)
	{
	    if(ncoppies > 1)
		command = g_strdup_printf(
		    "%s -#%i \"%s\"",
		    print_command, ncoppies, filename
		);
	    else
		command = g_strconcat(
		    print_command, " \"", filename, "\"", NULL
		);
	}
	/* Failed to generate the print command? */
	if(command == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
		"Unable to generate print command.",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    CLEANUP_RETURN(-1);
	}

	/* Reset the status to -1 to indicate error, it will be set to 0
	 * later if things are successful
	 */
	status = -1;

	/* Write the PostScript file */
	while(TRUE)
	{
	    gint user_aborted = FALSE;
	    const gchar *mesg;
	    PSFormatStruct fmt;
	    ps_progress_data_struct *cb_data;

	    /* Open the PostScript file for writing */
	    FILE *fp = FOpen(filename, "wb");
	    if(fp == NULL)
	    {
		gchar *buf = g_strdup_printf(
"Unable to open the file for writing:\n\
\n\
    %s",
		    filename
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    buf, NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK, CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(buf);
		break;
	    }

	    /* Set up PostScript format */
	    fmt.flags = PSFlagDPIX | PSFlagDPIY |
		PSFlagPaperWidth | PSFlagPaperHeight |
		PSFlagOutputX | PSFlagOutputY;
	    /* DPI resolution */
	    fmt.dpi_x = (vflags & PRINT_VALUE_DPI_X) ?
		pv->dpi_x : 72;
	    fmt.dpi_y = (vflags & PRINT_VALUE_DPI_Y) ?
		pv->dpi_y : 72;
	    /* Paper size without dpi applied */
	    fmt.paper_width = paper_width;
	    fmt.paper_height = paper_height;
	    /* Output image offset */
	    fmt.output_x = output_x;
	    fmt.output_y = output_y;

	    /* Set up ps progress callback structure */
	    cb_data = PS_PROGRESS_DATA(
		g_malloc0(sizeof(ps_progress_data_struct))
	    );
	    if(cb_data != NULL)
	    {
		cb_data->toplevel = toplevel;
		cb_data->filename = STRDUP(filename);
		cb_data->message = g_strdup_printf(
"Generating output image for printing..."
		);
	    }

	    /* Write depending on source image bytes per pixel */
	    switch(img->bpp)
	    {
	      case 4:
		status = PSWriteFilePS3RGBA(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      case 3:
		status = PSWriteFilePS3RGB(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      case 1:
		status = PSWriteFilePS3Grey(
		    fp, &fmt,
		    img->width, img->height, img->bpl,
		    frame->buf,
		    creator, NULL, NULL, NULL,
		    cb_data,
		    print_lpr_ps_write_progress_cb,
		    &user_aborted
		);
		break;

	      default:
		status = PSError;
		break;
	    }

	    /* Close the PostScript file */
	    FClose(fp);

	    /* Unmap progress dialog which might have been mapped when
	     * print_lpr_ps_write_progress_cb() was called
	     */
	    ProgressDialogBreakQuery(TRUE);
	    ProgressDialogSetTransientFor(NULL);

	    /* Delete the callback data */
	    if(cb_data != NULL)
	    {
		g_free(cb_data->filename);
		g_free(cb_data->message);
		g_free(cb_data);
		cb_data = NULL;
	    }

	    /* Check for error */
	    mesg = NULL;
	    switch(status)
	    {
	      case PSSuccess:
		status = 0;
		break;
	      case PSError:
		mesg = "Error writing PostScript file.";
		status = -1;
		break;
	      case PSBadValue:
		mesg = "Bad value used to write PostScript file.";
		status = -2;
		break;
	      case PSErrorSystem:
		mesg = "Systems error occured while writing PostScript file.";
		status = -3;
		break;
	      case PSAbort:
		status = -4;
		break;
	      case PSErrorInternal:
		mesg = "Systems error occured while writing PostScript file.";
		status = -3;		/* Treat as systems error */
		break;
	      default:
		status = -1;
		break;
	    }
	    if(mesg != NULL)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    mesg,
		    NULL,
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }

	    /* Break when we reach the end of this big while() loop */
	    break;
	}

	/* Failed to write PS image? */
	if(status == -4)
	{
	    /* Aborted, which means print procedure must be aborted */
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Aborted",
		"User aborted print",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    CLEANUP_RETURN(0);
	}
	if(status)
	{
	    /* Other error occured while the writing the PostScript
	     * file
	     */
	    CLEANUP_RETURN(-1);
	}


	/* Execute the print command */
	pid = Exec(command);
	if(pid <= 0)
	{
	    gchar *msg = g_strdup_printf(
"Unable to execute print command:\n\
\n\
    %s\n",
		command
	    );
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
		msg,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(msg);
	    CLEANUP_RETURN(-1);
	}


	/* Map progress dialog and wait for print process to finish */

	/* Set up and map progress dialog */
	ProgressDialogSetTransientFor(toplevel);
	/* While print process is running... */
	while(ExecProcessExists(pid))
	{
	    /* If progress dialog is not mapped, then map it and set
	     * the initial values
	     */
	    if(!ProgressDialogIsQuery())
		ProgressDialogMap(
		    "Printing",
		    "Sending output image to printer...",
		    (const guint8 **)icon_print2_32x32_xpm,
		    "Stop"
		);

	    /* Update progress */
	    ProgressDialogUpdateUnknown(NULL, NULL, NULL, NULL, TRUE);

	    gdk_flush();

	    /* User abort? */
	    if(ProgressDialogStopCount() > 0)
		break;

	    usleep(8000l);
	}
	/* Unmap progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* If the process is still running then it */
	if(ExecProcessExists(pid))
	    kill(pid, SIGINT);
	pid = 0;

	CLEANUP_RETURN(0);
#undef CLEANUP_RETURN
}

/*
 *	Prints the image.
 *
 *	The img specifies the image to print.
 *
 *	The pv specifies the print values that will be used as the
 *	parameters for printing the image.
 *
 *	The bg_color specifies the background color. It must contain
 *	four bytes describing the RGBA values.
 *
 *	The toplevel specifies the reference toplevel GtkWindow.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error.
 *	-4	User aborted.
 */
#ifdef HAVE_LIBENDEAVOUR2
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
)
#else
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
)
#endif
{
	gint status = -1;
	guint vflags;
	print_output output = PRINT_OUTPUT_TO_PRINTER;
	imgview_image_struct *output_img;

	if((img == NULL) || (pv == NULL))
	    return(-2);

	/* Create output image to generate the PostScript file
	 * with
	 *
	 * This image will be properly rotated and colored with respect
	 * to the specified print values
	 */
	output_img = PrintRenderImageFromValues(
	    img, pv, bg_color,
	    NULL, NULL
	);
	if(output_img == NULL)
	{
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Print Failed",
		"Unable to generate output image",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return(-3);
	}

	/* Print to printer or file */
	vflags = pv->flags;
	if(vflags & PRINT_VALUE_OUTPUT_TYPE)
	    output = pv->output;
	switch(output)
	{
	  case PRINT_OUTPUT_TO_FILE:
#ifdef HAVE_LIBENDEAVOUR2
	    status = iv_print_lpr_to_file(
		output_img, pv, bg_color, toplevel, edv2_ctx
	    );
#else
	    status = iv_print_lpr_to_file(
		output_img, pv, bg_color, toplevel
	    );
#endif
	    break;
	  case PRINT_OUTPUT_TO_PRINTER:
	    status = iv_print_lpr_to_printer(
		output_img, pv, bg_color, toplevel
	    );
	    break;
	}

	/* Delete output image */
	ImgViewImageDelete(output_img);

	return(status);
}

#endif
