#if !defined(_WIN32) && !defined(HAVE_XPRINT)
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifdef HAVE_LIBENDEAVOUR2
# include <endeavour2.h>
#endif

#include "../include/string.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "fb.h"
#include "imgview.h"
#include "print_values.h"
#include "print.h"
#include "print_dlg.h"
#include "config.h"

#include "images/icon_print2_48x48.xpm"
#include "images/icon_print2_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"

#include "images/icon_minus_16x16.xpm"
#include "images/icon_plus_16x16.xpm"
#include "images/icon_brightness_16x16.xpm"
#include "images/icon_contrast_16x16.xpm"


typedef struct _print_dlg_struct	print_dlg_struct;
#define PRINT_DLG(p)			((print_dlg_struct *)(p))

#define PAPER_SIZE(p)			((paper_size_struct *)(p))


/*
 *	Paper Size:
 */
typedef struct {

	gchar		*name,
			*desc;
	gint		width,		/* Size of paper in pixels */
			height;

} paper_size_struct;


/*
 *	Print Dialog:
 */
struct _print_dlg_struct {

	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gboolean	map_state,
			got_response;
	gint		freeze_count,
			busy_count,
			main_level;

	/* Cursors */
	GdkCursor	*busy_cur,
			*translate_cur,
			*size_cur;

	/* Source image data that user wants to print */
	imgview_image_struct	*src_img;
	gint			src_frame_num;

	/* Output radios */
	GtkWidget	*output_printer_radio,
			*output_file_radio;
	/* Output entries */
	GtkWidget	*output_printer_entry,	/* Print command */
			*output_file_entry;	/* File name */
	GtkWidget	*output_file_browse_btn;

	/* Visual scales */
	GtkWidget	*brightness_scale;
	GtkAdjustment	*brightness_adj;
	GtkWidget	*contrast_scale;
	GtkAdjustment	*contrast_adj;
	/* Visual radios */
	GtkWidget	*visual_bw_radio,
			*visual_greyscale_radio,
			*visual_color_radio;

	/* Orientation radios */
	GtkWidget	*orientation_portrait_radio,
			*orientation_landscape_radio;

	/* DPI entries, axis relative to paper (not image) */
	GtkWidget	*dpi_x_entry,
			*dpi_y_entry;

	/* Paper size clist */
	GtkWidget	*paper_size_clist;


	/* Output geometry entries */
	GtkWidget	*output_x_spin,
			*output_y_spin,
			*output_width_spin,
			*output_height_spin,
			*output_maintain_aspect_check,
			*output_ncoppies_spin;

	/* Preview drawing area */
	GtkWidget	*preview_da;

	/* Graphic contexts for the preview drawnig area */
	GdkGC		*gc_base,
			*gc_invert;

	/* Preview ImgView image
	 *
	 * This image is post processed with the modified window
	 * coordinate size, color, brightness, rotations, etc which
	 * makes it suitable and ready to be drawn on the preview_da
	 */
	imgview_image_struct	*preview_img;

	/* Original Preview ImgView image
	 *
	 * It matches the size of the preview_img but does not have
	 * brightness or contrast modifications made to it, this is so
	 * that it can be quickly copied to the preview_img with
	 * modifications to the contrast and brightness
	 */
	imgview_image_struct	*orig_preview_img;

	/* Preview menu */
	GtkWidget	*menu;

	/* Image output geometry (not allocated size of preview image), in
	 * pixels. Relative to upper left corner and assumed orientation
	 * to be PRINT_ORIENTATION_PORTRAIT
	 *
	 * When orientation is set to PRINT_ORIENTATION_LANDSCAPE the
	 * geometry values still represent their values as if the
	 * orientation was PRINT_ORIENTATION_PORTRAIT
	 */
	gint		output_x, output_y;
	gint		output_width, output_height;
	gboolean	output_maintain_aspect;

	guint8		bg_color[4];	/* 4 bytes RGBA */

	/* Buttons */
	GtkWidget	*print_btn,
			*cancel_btn;


	/* Keyboard modifiers mask */
#define PRINT_MODIFIER_ALT		(1 << 0)
#define PRINT_MODIFIER_CTRL		(1 << 1)
#define PRINT_MODIFIER_SHIFT		(1 << 2)
	guint		modifiers;

	/* Preview drag mode */
#define PRINT_DRAG_NONE			0
#define PRINT_DRAG_TRANSLATE		1
#define PRINT_DRAG_SIZE			2
	gint		drag_mode;
	gint		drag_last_x, drag_last_y;

	/* Rectangular drag outline in window coordinates, for drawing
	 * a rectangular rubber band on the preview widget while dragging
	 */
	gboolean	only_draw_outline;	/* Set by PrintDlgPreviewEventCB() and
						 * cleared by PrintDlgPreviewExposeCB()
						 */
	gint		drag_outline_x[2],
			drag_outline_y[2];
	gboolean	have_prev_outline;
	gint		prev_outline_x[2],
			prev_outline_y[2];

};


/* Callbacks */
static void PrintDlgPaperSizeRowDataDeleteCB(gpointer data);
static gint PrintDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void PrintDlgSelectRowCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
);
static void PrintDlgPrintCB(GtkWidget *widget, gpointer data);
static void PrintDlgCancelCB(GtkWidget *widget, gpointer data);
static void PrintDlgToggledCB(GtkWidget *widget, gpointer data);
static void PrintDlgAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
);
static void PrintDlgDPIChangedCB(GtkWidget *widget, gpointer data);
static void PrintDlgOutputChangedCB(GtkWidget *widget, gpointer data);
static void PrintDlgPrintFileBrowseCB(GtkWidget *widget, gpointer data);
static gint PrintDlgPreviewEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint PrintDlgPreviewExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);


/* Utilities */
static void PrintDlgSyncOutputWidgets(print_dlg_struct *d);
static print_visual PrintDlgVisual(print_dlg_struct *d);
static print_orientation PrintDlgOrientation(print_dlg_struct *d);
static gboolean PrintDlgGetDPI(
	print_dlg_struct *d,
	gint *dpi_x, gint *dpi_y
);
static gboolean PrintDlgPaperSize(
	print_dlg_struct *d,
	gint *width, gint *height,
	gboolean apply_dpi
);
static gfloat PrintDlgSourceImageAspect(print_dlg_struct *d);
static gboolean PrintDlgOutputOffsetW(
	print_dlg_struct *d, gint *x, gint *y
);

static void PrintDlgRevertImages(
	imgview_image_struct *src_img, imgview_image_struct *tar_img
);
static void PrintDlgRevertPreview(print_dlg_struct *d);

static void PrintDlgUpdatePreviewVisual(print_dlg_struct *d);
static void PrintDlgRecreatePreview(print_dlg_struct *d);
static void PrintDlgResizePreview(print_dlg_struct *d);
static void PrintDlgPreviewDraw(print_dlg_struct *d);

static print_dlg_struct *PrintDlgNew(const gchar *filename);
static void PrintDlgMap(print_dlg_struct *d);
static void PrintDlgUnmap(print_dlg_struct *d);
static void PrintDlgDelete(print_dlg_struct *d);

/* Front ends */

#ifdef HAVE_LIBENDEAVOUR2
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
);
#else
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
);
#endif


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0) 
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f) 
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)	(((x) < 0) ? ((x) * -1) : (x))

#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#define PRINT_DEF_TITLE			"Print"
#define PRINT_DEF_WIDTH			640
#define PRINT_DEF_HEIGHT		480

/* Nominal size of preview drawing area, the nominal size must be
 * square in shape
 */
#define PRINT_PREVIEW_WIDTH		200
#define PRINT_PREVIEW_HEIGHT		200

#define PRINT_CLIST_ROW_SPACING		20


/* Tooltips */
#define PRINT_TTM_PRINT_COMMAND		"\
Command to run the print client plus any arguments"

#define PRINT_TTM_BRIGHTNESS		"Brightness"
#define PRINT_TTM_CONTRAST		"Contrast"

#define PRINT_TTM_DPI_X			"\
Dots per inch (DPI) resolution of paper media width,\
 value must match with value set on printer driver"

#define PRINT_TTM_DPI_Y			"\
Dots per inch (DPI) resolution of paper media height,\
 value must match with value set on printer driver"

#define PRINT_TTM_PREVIEW		"\
Press and hold Button1 to adjust the position, press and hold SHIFT +\
 Button1 (or Button2) to adjust the size"



/*
 *	Paper Size list row destroy callback.
 */
static void PrintDlgPaperSizeRowDataDeleteCB(gpointer data)
{
	paper_size_struct *psize = PAPER_SIZE(data);
	if(psize == NULL)
	    return;

	g_free(psize->name);
	g_free(psize->desc);
	g_free(psize);
}

/*
 *	Toplevel GtkWindow "delete_event" signal callback.
 */
static gint PrintDlgDeleteEventCB(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	PrintDlgCancelCB(widget, data);
	return(TRUE);
}

/*
 *	Paper size GtkCList "select_row" signal callback.
 */
static void PrintDlgSelectRowCB(
	GtkWidget *widget, gint row, gint column,
	GdkEventButton *event, gpointer data
)
{
	GtkCList *clist;
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	clist = GTK_CLIST(widget);

	if(d->map_state)
	{
	    /* Scroll to row if not visable */
	    if(gtk_clist_row_is_visible(clist, row) != GTK_VISIBILITY_FULL)
	        gtk_clist_moveto(
		    clist,
		    row, 0,	/* Row, column */
		    0.5f, 0.0f	/* Row, column */
	        );
	}

	/* Is this is the paper size clist? */
	if((GtkWidget *)clist == d->paper_size_clist)
	{
	    /* Since paper size has changed, we need to update the
	     * preview widget size and recreate the preview images
	     */
	    PrintDlgResizePreview(d);
	}
}


/*
 *	Print callback.
 */
static void PrintDlgPrintCB(GtkWidget *widget, gpointer data)
{
	print_dlg_struct *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	/* Update response code */
	d->got_response = TRUE;

	/* Break out of highest GTK+ main loop */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}

/*
 *	Cancel callback.
 */
static void PrintDlgCancelCB(GtkWidget *widget, gpointer data)
{
	print_dlg_struct *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	/* Update response code */
	d->got_response = FALSE;

	/* Break out of highest GTK+ main loop */
	if(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}
}


/*
 *	All purpose "toggled" signal callback for all radio and
 *	toggle buttons.
 */
static void PrintDlgToggledCB(GtkWidget *widget, gpointer data)
{
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	/* Only handle toggled to TRUE signals */
	if(GTK_IS_TOGGLE_BUTTON(widget) ?
	    !GTK_TOGGLE_BUTTON_GET_ACTIVE(widget) : TRUE
	)
	    return;

	d->freeze_count++;

	/* Output to printer? */
	if(widget == d->output_printer_radio)
	{
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_printer_entry, TRUE
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_file_entry, FALSE
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_file_browse_btn, FALSE
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_ncoppies_spin, TRUE
	    );
	}
	/* Output to file? */
	else if(widget == d->output_file_radio)
	{
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_printer_entry, FALSE
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_file_entry, TRUE
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_file_browse_btn, TRUE 
	    );
	    GTK_WIDGET_SET_SENSITIVE(
		d->output_ncoppies_spin, FALSE
	    );
	}

	/* Visual Black & White? */
	else if(widget == d->visual_bw_radio)
	{
	    PrintDlgUpdatePreviewVisual(d);
	    PrintDlgPreviewDraw(d);
	}
	/* Visual Greyscale? */
	else if(widget == d->visual_greyscale_radio)
	{
	    PrintDlgUpdatePreviewVisual(d);
	    PrintDlgPreviewDraw(d);
	}
	/* Visual Color? */
	else if(widget == d->visual_color_radio)
	{
	    PrintDlgUpdatePreviewVisual(d);
	    PrintDlgPreviewDraw(d);
	}

	/* Orientation Portrait? */
	else if(widget == d->orientation_portrait_radio)
	{
	    /* Need to swap output size */
	    const gint tmp_height = d->output_width;
	    d->output_width = d->output_height;
	    d->output_height = tmp_height;

	    PrintDlgSyncOutputWidgets(d);
	    PrintDlgRecreatePreview(d);
	    PrintDlgPreviewDraw(d);
	}
	/* Orientation Landscape? */
	else if(widget == d->orientation_landscape_radio)
	{
	    /* Need to swap output size */
	    const gint tmp_height = d->output_width;
	    d->output_width = d->output_height;
	    d->output_height = tmp_height;

	    PrintDlgSyncOutputWidgets(d);
	    PrintDlgRecreatePreview(d);
	    PrintDlgPreviewDraw(d);
	}

	d->freeze_count--;
}

/*
 *	Adjustment "value_changed" signal callback.
 */
static void PrintDlgAdjustmentValueChangedCB(
	GtkAdjustment *adj, gpointer data
)
{
	print_dlg_struct *d = PRINT_DLG(data);
	if((adj == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Assume brightness or contrast adjustment change */
	PrintDlgUpdatePreviewVisual(d);

	/* Redraw */
	if(d->map_state)
	    PrintDlgPreviewDraw(d);

	d->freeze_count--;
}


/*
 *	DPI entry "changed" signal callback.
 */
void PrintDlgDPIChangedCB(GtkWidget *widget, gpointer data)
{
	const gchar *s;
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Get text from entry widget and check if it contains
	 * data
	 */
	s = gtk_entry_get_text(GTK_ENTRY(widget));
	if((s != NULL) ?
	    ((*s != '\0') && !ISBLANK(*s)) : FALSE
	)
	{
	    /* The DPI has changed so the preview widget needs to be
	     * resized and the preview image needs to be recreated to
	     * match the new size of the preview widget
	     */
	    PrintDlgResizePreview(d);
	}

	d->freeze_count--;
}


/*
 *	Output geometry widget "changed" or "toggled" signal callback.
 */
static void PrintDlgOutputChangedCB(GtkWidget *widget, gpointer data)
{
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(widget == d->output_x_spin)
	{
	    d->output_x = GTK_ENTRY_GET_VALUEI(widget);
	    PrintDlgPreviewDraw(d);
	}
	else if(widget == d->output_y_spin)
	{
	    d->output_y = GTK_ENTRY_GET_VALUEI(widget);
	    PrintDlgPreviewDraw(d);
	}
	else if(widget == d->output_width_spin)
	{
	    d->output_width = GTK_ENTRY_GET_VALUEI(widget);
	    if(d->output_width > 0)
	    {
		/* Need to adjust height if maintaining aspect? */
		if(d->output_maintain_aspect)
		{ 
		    gfloat aspect = PrintDlgSourceImageAspect(d);
		    if(aspect > 0.0f)
		    {
			if(PrintDlgOrientation(d) == PRINT_ORIENTATION_LANDSCAPE)
			    d->output_height = d->output_width * aspect;
			else
			    d->output_height = d->output_width / aspect;
		    }
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_height_spin),
			(gfloat)d->output_height
		    );
		}
		/* Recreate preview image due to resize and redraw */
		PrintDlgRecreatePreview(d);
		PrintDlgPreviewDraw(d);
	    }
	}
	else if(widget == d->output_height_spin)
	{
	    d->output_height = GTK_ENTRY_GET_VALUEI(widget);
	    if(d->output_height > 0)
	    {
		/* Need to adjust width if maintaining aspect? */
		if(d->output_maintain_aspect)
		{
		    gfloat aspect = PrintDlgSourceImageAspect(d);
		    if(aspect > 0.0f)
		    {
			if(PrintDlgOrientation(d) == PRINT_ORIENTATION_LANDSCAPE)
			    d->output_width = d->output_height / aspect;
			else
			    d->output_width = d->output_height * aspect;
		    }
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_width_spin),
			(gfloat)d->output_width
		    );
		}
		/* Recreate preview image due to resize and redraw */
		PrintDlgRecreatePreview(d);
		PrintDlgPreviewDraw(d);
	    }
	}
	else if(widget == d->output_maintain_aspect_check)
	{
	    d->output_maintain_aspect = GTK_TOGGLE_BUTTON_GET_ACTIVE(widget);
	    if(d->output_maintain_aspect)
	    {
		gfloat aspect = PrintDlgSourceImageAspect(d);
		if(aspect > 0.0f)
		{
		    /* Update size values */
		    if(PrintDlgOrientation(d) == PRINT_ORIENTATION_LANDSCAPE)
			d->output_height = d->output_width * aspect;
		    else
			d->output_height = d->output_width / aspect;
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_width_spin),
			(gfloat)d->output_width
		    );
		    gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(d->output_height_spin),
			(gfloat)d->output_height
		    );
		    /* Recreate preview image due to resize and redraw */
		    PrintDlgRecreatePreview(d);
		    PrintDlgPreviewDraw(d);
		}
	    }
	}

	d->freeze_count--;
}

/*
 *	Print to file prompt browse button callback.
 */
static void PrintDlgPrintFileBrowseCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint npaths, total_ftypes = 0;
	const gchar *last_path;
	gchar **paths_list;
	GtkWidget *w, *toplevel;
	GtkEntry *entry;
	fb_type_struct **ftype = NULL, *ftype_rtn;
	print_dlg_struct *d = PRINT_DLG(data);
	if(d == NULL)
	    return;

	w = d->output_file_entry;
	toplevel = d->toplevel;
	if(w == NULL)
	    return;

	entry = GTK_ENTRY(w);

	/* Get last path */
	last_path = gtk_entry_get_text(entry);

	/* Set up file extension types list */
	ftype = NULL;
	total_ftypes = 0;
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    FTYPE_EXT_PS, "Adobe ISO PostScript"
	);
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    FTYPE_EXT_PS2, "Adobe ISO PostScript 2"
	);
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    FTYPE_EXT_PS3, "Adobe ISO PostScript 3"
	);
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All Files"
	);

	/* Unset the print dialog as modal */
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);

	/* Map file browser and query user for response */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
	    "Escoja El PostScript File",
	    "Escoja", "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Privilgi PostScript File",
	    "Privilgi", "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Wahlen Sie PostScript File",
	    "Wahlen", "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Scegliere PostScript File",
	    "Sceglier", "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH) 
	    "Selecteer PostScript File",
	    "Selecter", "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Selecione PostScript File",
	    "Selecion", "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Velg Ut PostScript File",
	    "Velg", "Kanseler",
#else
	    "Select PostScript File",
	    "Select", "Cancel",
#endif
	    last_path,
	    ftype, total_ftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Set the print dialog as modal */
	gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);

	if(response)
	{
	    const gchar *path = (npaths > 0) ? paths_list[0] : NULL;
	    if(path != NULL)
		gtk_entry_set_text(entry, path);
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);
}


/*
 *	Print window's preview_da event handler.
 */
static gint PrintDlgPreviewEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint etype, status = FALSE;
	gboolean need_grab_pointer = FALSE;
	GdkCursor *cur = NULL;
	GdkWindow *window;
	GdkEventConfigure *configure;
	GdkEventCrossing *crossing;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	gboolean keystate;
	guint keyval;
	imgview_image_struct *preview_img;
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	/* Get view widget GdkWindow */
	if(GTK_WIDGET_NO_WINDOW(widget))
	    window = NULL;
	else
	    window = widget->window;

	/* Get preview image (may be NULL) */
	preview_img = d->preview_img;

	/* Get event type */
	etype = *(gint *)event;

	/* Handle by event type */
	switch(etype)
	{
	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    PrintDlgRecreatePreview(d);
	    if(d->map_state)
		PrintDlgPreviewDraw(d);
	    status = TRUE;
	    break;

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    if(!crossing->send_event)
	    {
#if 0
		/* Need to reset all modifiers */
		if(d->modifiers)
		{
		    d->modifiers = 0;
		    if(window != NULL)
			gdk_window_set_cursor(window, NULL);
		}
#endif
		/* Set up widget to grab focus so "key_press_event" and
		 * and "key_release_event" events will be sent to it
		 */
		gtk_widget_grab_focus(widget);

		/* Force tooltip to show, this will send a synthetic
		 * GDK_ENTER_NOTIFY event which we will ignore
		 */
		GUIShowTipsNow(widget);
		status = TRUE;
	    }
	    break;

	  case GDK_LEAVE_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    status = TRUE;
	    break;

	  case GDK_KEY_PRESS: case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    keyval = key->keyval;
	    keystate = ((etype == GDK_KEY_PRESS) ? TRUE : FALSE);
	    /* Handle by keyval */
	    /* ALT? */
	    if((keyval == GDK_Alt_L) || (keyval == GDK_Alt_R))
	    {
		if(keystate)
		    d->modifiers |= PRINT_MODIFIER_ALT;
		else
		    d->modifiers &= ~PRINT_MODIFIER_ALT;
		status = TRUE;
	    }
	    /* CTRL? */
	    else if((keyval == GDK_Control_L) || (keyval == GDK_Control_R))
	    {
		if(keystate)
		    d->modifiers |= PRINT_MODIFIER_CTRL;
		else
		    d->modifiers &= ~PRINT_MODIFIER_CTRL;
		status = TRUE;
	    }
	    /* SHIFT? */
	    else if((keyval == GDK_Shift_L) || (keyval == GDK_Shift_R))
	    {
		if(keystate)
		    d->modifiers |= PRINT_MODIFIER_SHIFT;
		else
		    d->modifiers &= ~PRINT_MODIFIER_SHIFT;
		status = TRUE;
	    }
	    if(status)
	    {
		if(d->drag_mode == PRINT_DRAG_NONE)
		{
		    if(d->modifiers & PRINT_MODIFIER_SHIFT)
		    {
			cur = d->size_cur;
		    }

		    /* Update cursor */
		    if(window != NULL)
			gdk_window_set_cursor(window, cur);
		}
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    /* Handle by which button number was pressed */
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		if(d->modifiers & (PRINT_MODIFIER_ALT | PRINT_MODIFIER_SHIFT))
		{
		    /* Size, same as if GDK_BUTTON2 was pressed */
		    cur = d->size_cur;
		    need_grab_pointer = TRUE;
		    d->drag_mode = PRINT_DRAG_SIZE;
		    d->drag_last_x = (gint)button->x;
		    d->drag_last_y = (gint)button->y;

		    if(!PrintDlgOutputOffsetW(
			d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		    ))
		    {
			d->drag_outline_x[0] = 0;
			d->drag_outline_y[0] = 0;
		    }
		    if(preview_img != NULL)
		    {
			d->drag_outline_x[1] = d->drag_outline_x[0] +
			    preview_img->width;
			d->drag_outline_y[1] = d->drag_outline_y[0] +
			    preview_img->height;
		    }
		    /* If mapped, then draw only the outline */
		    if(d->map_state)
		    {
			d->have_prev_outline = FALSE;
			d->only_draw_outline = TRUE;;
			PrintDlgPreviewDraw(d);
		    }
		}
		else
		{
		    /* No keyboard modifiers held, then just translate */
		    cur = d->translate_cur;
		    need_grab_pointer = TRUE;
		    d->drag_mode = PRINT_DRAG_TRANSLATE;
		    d->drag_last_x = (gint)button->x;
		    d->drag_last_y = (gint)button->y;

		    if(!PrintDlgOutputOffsetW(
		        d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		    ))
		    {
		        d->drag_outline_x[0] = 0;
		        d->drag_outline_y[0] = 0;
		    }
		    if(preview_img != NULL)
		    {
		        d->drag_outline_x[1] = d->drag_outline_x[0] +
			    preview_img->width;
			d->drag_outline_y[1] = d->drag_outline_y[0] +
			    preview_img->height;
		    }
		    /* If mapped, then draw only the outline */
		    if(d->map_state)
		    {
		        d->have_prev_outline = FALSE;
		        d->only_draw_outline = TRUE;;
			PrintDlgPreviewDraw(d);
		    }
		}
		status = TRUE;
		break;

	      case GDK_BUTTON2:
		/* Size */
		cur = d->size_cur;
		need_grab_pointer = TRUE;
		d->drag_mode = PRINT_DRAG_SIZE;
		d->drag_last_x = (gint)button->x;
		d->drag_last_y = (gint)button->y;

		if(!PrintDlgOutputOffsetW(
		    d, &d->drag_outline_x[0], &d->drag_outline_y[0]
		))
		{
		    d->drag_outline_x[0] = 0;
		    d->drag_outline_y[0] = 0;
		}
		if(preview_img != NULL)
		{
		    d->drag_outline_x[1] = d->drag_outline_x[0] +
			preview_img->width;
		    d->drag_outline_y[1] = d->drag_outline_y[0] +
			preview_img->height;
		}
		/* If mapped, then draw only the outline */
		if(d->map_state)
		{
		    d->have_prev_outline = FALSE;
		    d->only_draw_outline = TRUE;;
		    PrintDlgPreviewDraw(d);
		}
		status = TRUE;
		break;
	    }
	    /* Set window cursor if specified */
	    if((window != NULL) && (cur != NULL))
		gdk_window_set_cursor(window, cur);

	    /* Grab pointer? */
	    if(need_grab_pointer && (window != NULL))
	    {
		gdk_pointer_grab(
		    window,
		    FALSE,
		    GDK_BUTTON_PRESS_MASK |
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK,
		    NULL,
		    GDK_NONE,
		    button->time
		);
		need_grab_pointer = FALSE;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;

	    /* Ungrab pointer */
	    if(gdk_pointer_is_grabbed())
		gdk_pointer_ungrab(button->time);

	    /* Handle by current drag mode */
	    switch(d->drag_mode)
	    {
	      case PRINT_DRAG_TRANSLATE:
		if(TRUE)
		{
		    gint	ox = MIN(
			d->drag_outline_x[0], d->drag_outline_x[1]
		    ),
				oy = MIN(
			d->drag_outline_y[0], d->drag_outline_y[1]
		    );
		    gint paper_width, paper_height, ww, wh;
		    gfloat wtoo_coeff;


		    ww = widget->allocation.width;
		    wh = widget->allocation.height;

		    if(!PrintDlgPaperSize(d, &paper_width, &paper_height, TRUE))
		    {
			paper_width = 320;
			paper_height = 240;
		    }

		    if(ww > 0)
			wtoo_coeff = (gfloat)paper_width / (gfloat)ww;
		    else
			wtoo_coeff = 0.0f;
		    d->output_x = ox * wtoo_coeff;

		    if(wh > 0)
			wtoo_coeff = (gfloat)paper_height / (gfloat)wh;
		    else
			wtoo_coeff = 0.0f;
		    d->output_y = oy * wtoo_coeff;

		    /* Update output entry widgets to reflect new position */
		    PrintDlgSyncOutputWidgets(d);
		}
		/* Reset drag mode and redraw */
		d->drag_mode = PRINT_DRAG_NONE;
		if(d->map_state)
		    PrintDlgPreviewDraw(d);
		status = TRUE;
		break;

	      case PRINT_DRAG_SIZE:
		if(TRUE)
		{
		    gint	ow = ABSOLUTE(
			d->drag_outline_x[1] - d->drag_outline_x[0]
		    ),
				oh = ABSOLUTE(
			d->drag_outline_y[1] - d->drag_outline_y[0]
		    );
		    gint	ww = widget->allocation.width,
				wh = widget->allocation.height;

		    gint paper_width, paper_height;
		    gfloat wtoo_coeff;

		    /* Get paper size */
		    if(!PrintDlgPaperSize(
			d, &paper_width, &paper_height, TRUE
		    ))
		    {
			paper_width = 320;
			paper_height = 240;
		    }

		    /* Calculate output size */
		    wtoo_coeff = (ww > 0) ?
			((gfloat)paper_width / (gfloat)ww) : 0.0f;
		    if((ow > 0) && (wtoo_coeff > 0.0f))
			d->output_width = ow * wtoo_coeff;

		    wtoo_coeff = (wh > 0) ?
			((gfloat)paper_height / (gfloat)wh) : 0.0f;
		    if((oh > 0) && (wtoo_coeff > 0.0f))
			d->output_height = oh * wtoo_coeff;

		    /* Update output entry widgets to reflect new size */
		    PrintDlgSyncOutputWidgets(d);

		    /* Need to recreate the preview image since it changed
		     * size
		     */
		    PrintDlgRecreatePreview(d);
		}
		/* Reset drag mode and redraw */
		d->drag_mode = PRINT_DRAG_NONE;
		if(d->map_state)
		    PrintDlgPreviewDraw(d);
		status = TRUE;
		break;

	      default:
		/* Always reset drag mode on button release */
		d->drag_mode = PRINT_DRAG_NONE;
		break;
	    }
	    /* If no keyboard modifiers are being held then reset cursor */
	    if(!d->modifiers)
	    {
		if(window != NULL)
		    gdk_window_set_cursor(window, NULL);
	    }
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(d->drag_mode != PRINT_DRAG_NONE)
	    {
		gint x, y, dx, dy;
		GdkModifierType mask;

		if(motion->is_hint)
		{
		    gdk_window_get_pointer(
			motion->window, &x, &y, &mask
		    );
		}
		else
		{
		    x = (gint)motion->x;
		    y = (gint)motion->y;
		    mask = motion->state;
		}

		/* Handle by the drag mode */
		switch(d->drag_mode)
		{
		   case PRINT_DRAG_TRANSLATE:
		    /* Calculate delta values */
		    dx = x - d->drag_last_x;
		    dy = y - d->drag_last_y;

		    /* Apply delta values to translate drag outline */
		    d->drag_outline_x[0] += dx;
		    d->drag_outline_y[0] += dy;
		    d->drag_outline_x[1] += dx;
		    d->drag_outline_y[1] += dy;

		    /* Draw only the outline if mapped */
		    if(d->map_state)
		    {
		        d->only_draw_outline = TRUE;
			PrintDlgPreviewDraw(d);
		    }

		    /* Update last drag positions */
		    d->drag_last_x = x;
		    d->drag_last_y = y;

		    status = TRUE;
		    break;

		  case PRINT_DRAG_SIZE:
		    /* Calculate delta values */
		    dx = x - d->drag_last_x;
		    dy = y - d->drag_last_y;

		    /* Apply delta values to resize drag outline based
		     * on maintaining aspect or not
		     */
		    if(d->output_maintain_aspect)
		    {
			/* Maintaining aspect, so get aspect and resize
			 * drag output while maintaining aspect
			 */
			gfloat aspect = PrintDlgSourceImageAspect(d);
			print_orientation orientation = PrintDlgOrientation(d);
			gint len;

			if(aspect <= 0.0f)
			    aspect = 1.0f;

			/* Swap aspect if orientation is set to landscape */
			if(orientation == PRINT_ORIENTATION_LANDSCAPE)
			    aspect = 1 / aspect;

			/* Use greater of the two compoent movements */
			if(ABSOLUTE(dx) > ABSOLUTE(dy))
			{
			    d->drag_outline_x[1] += dx;
			    len = MAX(
				d->drag_outline_x[1] - 
				d->drag_outline_x[0], 0
			    );
			    d->drag_outline_y[1] = d->drag_outline_y[0] +
				(len / aspect);
			}
			else
			{
			    d->drag_outline_y[1] += dy;
			    len = MAX(
				d->drag_outline_y[1] - 
				d->drag_outline_y[0], 0 
			    );
			    d->drag_outline_x[1] = d->drag_outline_x[0] +
				(len * aspect);
			}
		    }
		    else
		    {
			d->drag_outline_x[1] += dx;
			d->drag_outline_y[1] += dy;
		    }
		    /* Make sure drag outline does not get smaller than
		     * its origin
		     */
		    if(d->drag_outline_x[1] < d->drag_outline_x[0])
			d->drag_outline_x[1] = d->drag_outline_x[0];
		    if(d->drag_outline_y[1] < d->drag_outline_y[0])
			d->drag_outline_y[1] = d->drag_outline_y[0];

		    /* Draw only the outline if mapped */
		    if(d->map_state)
		    {
			d->only_draw_outline = TRUE;
			PrintDlgPreviewDraw(d);
		    }

		    /* Update last drag positions */
		    d->drag_last_x = x;
		    d->drag_last_y = y;
		    status = TRUE;
		    break;
		}
	    }
	    break;
	}

	return(status);
}

/*
 *	Print window's preview_da widget "expose_event" callback.
 */
static gint PrintDlgPreviewExposeCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	GdkGC *gc_base, *gc_invert;
	gboolean only_draw_outline;
	GdkWindow *window;
	GdkDrawable *drawable;
	GtkWidget *w;
	gint ww, wh;
	imgview_frame_struct *frame;
	imgview_image_struct *img;
	print_dlg_struct *d = PRINT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return(FALSE);

	/* Get marker for if we only want to draw the drag outlines */
	only_draw_outline = d->only_draw_outline;
	d->only_draw_outline = FALSE;	/* Always reset it in this function */

	/* Get preview drawing area widget */
	w = d->preview_da;
	if(w != widget)
	    return(FALSE);

	ww = w->allocation.width;
	wh = w->allocation.height;
	if((ww <= 0) || (wh <= 0))
	    return(FALSE);

	/* Get pointers to the GCs */
	gc_base = d->gc_base;
	gc_invert = d->gc_invert;

	/* Must have the base GC */
	if(gc_base == NULL)
	    return(FALSE);

	/* Get preview drawing area widget's window */
	if(GTK_WIDGET_NO_WINDOW(w))
	    window = NULL;
	else
	    window = w->window;
	if(window == NULL)
	    return(FALSE);

	drawable = (GdkDrawable *)window;


	/* Clear background as needed */
	if(!only_draw_outline)
	{
	    gdk_draw_rectangle(drawable, gc_base, TRUE, 0, 0, ww, wh);
	}

	/* Get preview image and check if it contains data? */
	img = d->preview_img;
	frame = ImgViewImageGetFrame(img, 0);
	if((frame != NULL) ?
	    ((frame->buf != NULL) && !only_draw_outline) : FALSE
	)
	{
	    gint x, y;

	    if(!PrintDlgOutputOffsetW(d, &x, &y))
	    {
		x = 0;
		y = 0;
	    }

	    switch(img->bpp)
	    {
	      case 4:
		gdk_draw_rgb_32_image(
		    drawable, gc_base,
		    x, y,
		    img->width, img->height,
		    GDK_RGB_DITHER_NORMAL,
		    (guchar *)frame->buf,
		    img->bpl
		);
		break;

	      case 3:
		gdk_draw_rgb_image(
		    drawable, gc_base,
		    x, y,
		    img->width, img->height,
		    GDK_RGB_DITHER_NORMAL,
		    (guchar *)frame->buf,
		    img->bpl
		);
		break;
	    }
	}


	/* Clear previous drag marks if defined and if only drawing
	 * for drag outlines.
	 */
	if(d->have_prev_outline && only_draw_outline && (gc_invert != NULL))
	{
	     gint        x = MIN(d->prev_outline_x[0], d->prev_outline_x[1]),
			 y = MIN(d->prev_outline_y[0], d->prev_outline_y[1]);
	     gint        width = ABSOLUTE(
		d->prev_outline_x[1] - d->prev_outline_x[0]
			 ),
			 height = ABSOLUTE(
		d->prev_outline_y[1] - d->prev_outline_y[0]
			 );

	    if((width > 0) && (height > 0))
		gdk_draw_rectangle(
		    drawable, gc_invert, FALSE, x, y, width, height
		);

	    d->have_prev_outline = FALSE;
	}

	/* Draw new drag marks? */
	if((d->drag_mode != PRINT_DRAG_NONE) && (gc_invert != NULL))
	{
	    gint	x = MIN(d->drag_outline_x[0], d->drag_outline_x[1]),
			y = MIN(d->drag_outline_y[0], d->drag_outline_y[1]);
	    gint	width = ABSOLUTE(
		d->drag_outline_x[1] - d->drag_outline_x[0]
			),
			height = ABSOLUTE(
		d->drag_outline_y[1] - d->drag_outline_y[0]
			);

	    if((width > 0) && (height > 0))
	    {
		gdk_draw_rectangle(
		    drawable, gc_invert, FALSE, x, y, width, height
		);

		/* Record newly drawn drag outline marks */
		d->have_prev_outline = TRUE;
		d->prev_outline_x[0] = d->drag_outline_x[0];
		d->prev_outline_y[0] = d->drag_outline_y[0];
		d->prev_outline_x[1] = d->drag_outline_x[1];
		d->prev_outline_y[1] = d->drag_outline_y[1];
	    }
	}

	return(TRUE);
}


/*
 *	Sets the Print Window's output widget values to match the current
 *	output geometry values.
 */
static void PrintDlgSyncOutputWidgets(print_dlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	d->freeze_count++;

	/* Begin updating output entry widgets */
	w = d->output_x_spin;
	if(w != NULL)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(w), (gfloat)d->output_x
	    );
	w = d->output_y_spin;
	if(w != NULL)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(w), (gfloat)d->output_y
	    );
	w = d->output_width_spin;
	if(w != NULL)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(w), (gfloat)d->output_width
	    );
	w = d->output_height_spin;
	if(w != NULL)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(w), (gfloat)d->output_height
	    );
	w = d->output_maintain_aspect_check;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w), d->output_maintain_aspect
	    );

	d->freeze_count--;
}

/*
 *	Gets the Print Window's print visual.
 */
static print_visual PrintDlgVisual(print_dlg_struct *d)
{
	if(d == NULL)
	    return(PRINT_VISUAL_BW);

	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_bw_radio
	))
	    return(PRINT_VISUAL_BW);
	else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_greyscale_radio
	))
	    return(PRINT_VISUAL_GREYSCALE);
	else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->visual_color_radio
	))
	    return(PRINT_VISUAL_RGB);
	else
	    return(PRINT_VISUAL_BW);
}

/*
 *	Gets the Print Window's print orientation.
 */
static print_orientation PrintDlgOrientation(print_dlg_struct *d)
{
	GtkWidget *w = (d != NULL) ?
	    d->orientation_portrait_radio : NULL;
	if(w == NULL)
	    return(PRINT_ORIENTATION_PORTRAIT);

	return(GTK_TOGGLE_BUTTON_GET_ACTIVE(w) ?
	    PRINT_ORIENTATION_PORTRAIT : PRINT_ORIENTATION_LANDSCAPE
	);
}

/*
 *	Gets teh DPI (dots per inch) value, returns TRUE if the value
 *	was obtained or FALSE on failure.
 */
static gboolean PrintDlgGetDPI(
	print_dlg_struct *d,
	gint *dpi_x, gint *dpi_y
)
{
	GtkWidget *w;

	if(dpi_x != NULL)
	    (*dpi_x) = -1;
	if(dpi_y != NULL)
	    (*dpi_y) = -1;

	if(d == NULL)
	    return(FALSE);


	w = d->dpi_x_entry;
	if((w != NULL) ? GTK_IS_ENTRY(w) : 0)
	{
	    const gchar *s = gtk_entry_get_text(
		GTK_ENTRY(w)
	    );
	    if((s != NULL) && (dpi_x != NULL))
		(*dpi_x) = atoi(s);
	}

	w = d->dpi_y_entry;
	if((w != NULL) ? GTK_IS_ENTRY(w) : 0)
	{
	    const gchar *s = gtk_entry_get_text(
		GTK_ENTRY(w)
	    );
	    if((s != NULL) && (dpi_y != NULL))
		(*dpi_y) = atoi(s);
	}

	return(TRUE);
}

/*
 *	Gets the paper size, returns TRUE if the value was obtained or
 *	FALSE on failure.
 *
 *	If apply_dpi is TRUE then the dpi will be applied to the paper
 *	size before it's returned.
 */
static gboolean PrintDlgPaperSize(
	print_dlg_struct *d,
	gint *width, gint *height,
	gboolean apply_dpi
)
{
	GList *glist;
	gint row;
	const paper_size_struct *psize;
	gint orig_width, orig_height;
	GtkCList *clist = (d != NULL) ?
	    (GtkCList *)d->paper_size_clist : NULL;

	if(width != NULL)
	    *width = -1;
	if(height != NULL)
	    *height = -1;

	if(clist == NULL)
	    return(FALSE);

	/* Get last selected row */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Get paper size from the selected row */
	if((row >= 0) && (row < clist->rows))
	    psize = PAPER_SIZE(
		gtk_clist_get_row_data(clist, row)
	    );
	else
	    psize = NULL;
	if(psize == NULL)
	    return(FALSE);

	/* Get original paper size */
	orig_width = psize->width;
	orig_height = psize->height;
	if((orig_width <= 0) || (orig_height <= 0))
	    return(FALSE);

	/* Apply DPI to paper size? */
	if(apply_dpi)
	{
	    gint dpi_x, dpi_y;

	    /* Get DPI */
	    if(PrintDlgGetDPI(d, &dpi_x, &dpi_y))
	    {
		/* Modify width and height to match DPI */
		orig_width = (gfloat)orig_width * (gfloat)dpi_x / 100.0f;
		orig_height = (gfloat)orig_height * (gfloat)dpi_y / 100.0f;
	    }
	}

	/* Update returns */
	if(width != NULL)
	    *width = orig_width;
	if(height != NULL)
	    *height = orig_height;

	return(TRUE);
}

/*
 *	Returns the Print Window's source image aspect.
 */
static gfloat PrintDlgSourceImageAspect(print_dlg_struct *d)
{
	const imgview_image_struct *img = (d != NULL) ? d->src_img : NULL;
	if(img == NULL)
	    return(0.0f);

	if(img->height <= 0)
	    return(0.0f);
	else
	    return((gfloat)img->width / (gfloat)img->height);
}


/*
 *	Returns the Print Window's output image upper-left corner offset from
 *	the window's upper-left corner (regardless of orientation).
 *
 *	Returns TRUE if the output x and y offsets were obtained
 *	or FALSE on error.
 */
static gboolean PrintDlgOutputOffsetW(
	print_dlg_struct *d, gint *x, gint *y
)
{
	gfloat otow_coeff;
	gint paper_width, paper_height;
	GtkWidget *w = (d != NULL) ? d->preview_da : NULL;
	if(w == NULL)
	    return(FALSE);

	if(!PrintDlgPaperSize(
	    d, &paper_width, &paper_height, TRUE
	))
	    return(FALSE);

	if((paper_width <= 0) || (paper_height <= 0))
	{
	    if(x != NULL)
		*x = 0;
	    if(y != NULL)
		*y = 0;
	    return(TRUE);
	}

	/* Calculate offsets */
	otow_coeff = (gfloat)w->allocation.width / (gfloat)paper_width;
	if(x != NULL)
	    *x = d->output_x * otow_coeff;
	otow_coeff = (gfloat)w->allocation.height / (gfloat)paper_height;
	if(y != NULL)
	    *y = d->output_y * otow_coeff;

	return(TRUE);
}


/*
 *	Copies the image data from src_img to tar_img.
 *
 *	Both images must have the same size and depth.
 */
static void PrintDlgRevertImages(
	imgview_image_struct *src_img, imgview_image_struct *tar_img
)
{
	gint bpp, src_bpl, tar_bpl, min_bpl, height;
	const guint8 *src_data, *src_line, *src_line_end;
	guint8 *tar_data, *tar_line;
	imgview_frame_struct *src_frame, *tar_frame;

	if((src_img == NULL) || (tar_img == NULL))
	    return;

	/* Source image geometry and bpp must match the target image */
	if((src_img->bpp != tar_img->bpp) ||
	   (src_img->width != tar_img->width) ||
	   (src_img->height != tar_img->height)
	)
	{
	    g_printerr(
"PrintDlgRevertImages(): BPP or geometry mismatch, cannot copy.\n"
	    );
	    return;
	}

	bpp = src_img->bpp;
	height = src_img->height;

	src_bpl = src_img->bpl;
	tar_bpl = tar_img->bpl;
	min_bpl = src_img->width * bpp;

	src_frame = ImgViewImageGetFrame(src_img, 0);
	tar_frame = ImgViewImageGetFrame(tar_img, 0);

	/* Get pointers to both image's buffers, both must be allocated */
	src_data = src_frame->buf;
	tar_data = tar_frame->buf;
	if((src_data == NULL) || (tar_data == NULL))
	    return;

	for(src_line = src_data,
	    src_line_end = src_line + (src_bpl * height),
	    tar_line = tar_data;
	    src_line < src_line_end;
	    src_line += src_bpl,
	    tar_line += tar_bpl
	)
	    memcpy(tar_line, src_line, min_bpl);
}


/*
 *	Resets the image data in preview_img with the data from 
 *	orig_preview_img.
 */
static void PrintDlgRevertPreview(print_dlg_struct *d)
{
	if(d == NULL)
	    return;

	/* Copy data from the original preview image to the preview
	 * image that will be ready for drawing on preview_da
	 */
	PrintDlgRevertImages(
	    d->orig_preview_img,	/* Source */
	    d->preview_img		/* Target */
	);
}


/*
 *	Updates the Print Window's preview_img to reflect the current
 *	visual values.
 *
 *	PrintDlgRevertPreview() will be called to revert the data.
 */
static void PrintDlgUpdatePreviewVisual(print_dlg_struct *d)
{
	print_values_struct *v;

	if(d == NULL)
	    return;

	/* Copy data from orig_preview_img to preview_img */
	PrintDlgRevertPreview(d);

	/* Set up print values to match the print dialog's current
	 * print values
	 */
	v = PRINT_VALUES(
	    g_malloc0(sizeof(print_values_struct))
	);
	v->flags = PRINT_VALUE_BRIGHTNESS | PRINT_VALUE_CONTRAST |
	    PRINT_VALUE_VISUAL;
	v->brightness = GTK_ADJUSTMENT_GET_VALUE(d->brightness_adj);
	v->contrast = GTK_ADJUSTMENT_GET_VALUE(d->contrast_adj);
	v->visual = PrintDlgVisual(d);

	/* Modify preview_img's color, brightness, and contrast */
	PrintAdjustImageColor(d->preview_img, v, NULL, NULL);

	g_free(v);
}

/*
 *	Recreates the original preview image orig_preview_img and
 *	the preview image preview_img.
 *
 *	The new preview_img will contain the same data as 
 *	orig_preview_img.
 *
 *	This function will call PrintDlgUpdatePreviewVisual() to make sure
 *	the new images are updated and ready to be drawn.
 */
static void PrintDlgRecreatePreview(print_dlg_struct *d)
{
	print_visual visual;
	print_orientation orientation;
	gint i;
	gint paper_width, paper_height;
	gint output_width, output_height;
	gint new_width, new_height;
	gfloat x_coeff, y_coeff;
	imgview_frame_struct	*src_frame,
				*tar_frame;
	imgview_image_struct	*tar_img,
				*src_img;
	GtkWidget *w = (d != NULL) ? d->preview_da : NULL;
 	if(w == NULL)
	    return;

	/* Delete existing preview images */
	ImgViewImageDelete(d->preview_img);
	d->preview_img = NULL;
	ImgViewImageDelete(d->orig_preview_img);
	d->orig_preview_img = NULL;

	/* Get the source image */
	src_img = d->src_img;
	src_frame = ImgViewImageGetFrame(src_img, d->src_frame_num);
	if((src_frame != NULL) ? (src_frame->buf == NULL) : TRUE)
	    return;

	/* Get current print values from the print dialog */
	orientation = PrintDlgOrientation(d);
	visual = PrintDlgVisual(d);
	if(!PrintDlgPaperSize(d, &paper_width, &paper_height, TRUE))
	    return;

	/* Get output image size in pixels, we can ignore the
	 * orientation because the output size should already be
	 * updated current orientation
	 */
	output_width = d->output_width;
	output_height = d->output_height;
	if((output_width <= 0) || (output_height <= 0))
	    return;

	/* Calculate coefficients that represent the amount of
	 * the output image visible on the paper
	 */
	x_coeff = (gfloat)output_width / (gfloat)paper_width;
	y_coeff = (gfloat)output_height / (gfloat)paper_height;

	/* Calculate the size of the new preview image */
	new_width = (gint)(w->allocation.width * x_coeff);
	new_height = (gint)(w->allocation.height * y_coeff);

	if((new_width <= 0) || (new_height <= 0))
	    return;


	/* Create the new preview images */
	d->orig_preview_img = tar_img = ImgViewImageNew(
	    new_width, new_height, src_img->bpp, 0
	);
	i = ImgViewImageAppendFrameNew(tar_img);
	tar_frame = ImgViewImageGetFrame(tar_img, i);

	d->preview_img = ImgViewImageNew(
	    new_width, new_height, src_img->bpp, 0
	);
	i = ImgViewImageAppendFrameNew(d->preview_img);

	if((tar_frame != NULL) ? (tar_frame->buf != NULL) : FALSE)
	{
	    const gint bpp = src_img->bpp;
	    if(bpp != tar_img->bpp)
	    {
		g_printerr(
"PrintDlgRecreatePreview(): Error:\
Source and target images do not have the same bytes per pixel.\n"
		);
		return;
	    }

	    /* Copy/resize by orientation */
	    if(orientation == PRINT_ORIENTATION_LANDSCAPE)
	    {
		/* Create a tempory image rotated 90 degrees of the
		 * target image's size
		 */
		imgview_frame_struct *tmp_frame;
		imgview_image_struct *tmp_img = ImgViewImageNew(
		    tar_img->height, tar_img->width,
		    tar_img->bpp, 0
		);
		i = ImgViewImageAppendFrameNew(tmp_img);
		tmp_frame = ImgViewImageGetFrame(tmp_img, i);
		if(tmp_frame != NULL)
		{
		    /* Copy/resize source image to tempory image */
		    GUIImageBufferResize(
			bpp,
			src_frame->buf,
			src_img->width, src_img->height, src_img->bpl,
			tmp_frame->buf,
			tmp_img->width, tmp_img->height, tmp_img->bpl,
			NULL, NULL
		    );

		    /* Copy/rotate tempory image to target image */
		    GUIImageBufferRotateCCW90(
			bpp,
			tmp_frame->buf,
			tmp_img->width, tmp_img->height, tmp_img->bpl,
			tar_frame->buf,
			tar_img->width, tar_img->height, tar_img->bpl
		    );
		}

		/* Delete the tempory image */
		ImgViewImageDelete(tmp_img);
	    }
	    else
	    {
		/* Copy/resize the source image to the target image */
		GUIImageBufferResize(
		    bpp,
		    src_frame->buf,
		    src_img->width, src_img->height, src_img->bpl,
		    tar_frame->buf,
		    tar_img->width, tar_img->height, tar_img->bpl,
		    NULL, NULL
		);
	    }

	    /* Apply background to the target image */
	    GUIImageBufferFlattenWithBG(
		bpp,
		tar_frame->buf,
		tar_img->width, tar_img->height, tar_img->bpl,
		d->bg_color,
		NULL, NULL
	    );
	}

	/* This will call PrintDlgRevertPreview() to first copy the data
	 * from orig_preview_img to preview_img and modify preview_img to
	 * reflect the current visual settings
	 */
	PrintDlgUpdatePreviewVisual(d);
}

/*
 *	Resizes the Print Window's preview GtkDrawingArea.
 */
static void PrintDlgResizePreview(print_dlg_struct *d)
{
	gint width, height;	/* Nominal size of preview_da */
	gint new_width, new_height;
	gint paper_width, paper_height;
	gfloat aspect;
	GtkWidget *w = (d != NULL) ? d->preview_da : NULL;
	if(w == NULL)
	    return;

	/* Set nominal display size of preview drawing area */
	width = PRINT_PREVIEW_WIDTH;
	height = PRINT_PREVIEW_HEIGHT;

	/* Get new paper size */
	if(!PrintDlgPaperSize(d, &paper_width, &paper_height, TRUE))
	    return;
	if(paper_height <= 0)
	    return;

	/* Calculate aspect ratio of paper */
	aspect = (gfloat)paper_width / (gfloat)paper_height;
	if(aspect <= 0.0f)
	    return;

	/* Calculate new size for preview drawing area */
	if(aspect >= 1.0f)
	{
	    /* Width is longer than height */
	    new_width = width;
	    new_height = height / aspect;
	}
	else
	{
	    /* Height is longer than width */
	    new_width = width * aspect;
	    new_height = height;
	}

	/* Set new size of preview drawing area */
	w = d->preview_da;
	if(w != NULL)
	    gtk_widget_set_usize(w, new_width, new_height);

	/* Notify toplevel about resize */
	w = d->toplevel;
	if(w != NULL)
	    gtk_widget_queue_resize(w);

	/* We do not need to recreate the preview images since a
	 * "configure_event" will be generated and the signal handler
	 * will see it and recreate the preview images
	 */
}


/*
 *	Redraws the Print Window's preview.
 */
static void PrintDlgPreviewDraw(print_dlg_struct *d)
{
	if(d == NULL)
	    return;

	PrintDlgPreviewExposeCB(d->preview_da, NULL, d);
}


/*
 *	Creates a new print dialog.
 */
static print_dlg_struct *PrintDlgNew(const gchar *filename)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5,
		border_minor = 2;
	gchar *s, *heading[2];
	GSList *gslist;
	GdkColormap *colormap;
	GdkWindow *window;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkAdjustment *adj;
	GtkWidget	*w, *parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *main_vbox, *scroll_parent;
	GtkCList *clist;
	gpointer entry, browse_btn;
	print_dlg_struct *d = PRINT_DLG(
	    g_malloc0(sizeof(print_dlg_struct))
	);
	if(d == NULL)
	    return(NULL);

	d->accelgrp = accelgrp = gtk_accel_group_new();

	d->map_state = FALSE;
	d->got_response = FALSE;
	d->freeze_count = 0;
	d->busy_count = 0;
	d->main_level = 0;
	d->src_img = NULL;
	d->preview_img = NULL;
	d->orig_preview_img = NULL;
	d->drag_mode = PRINT_DRAG_NONE;

	d->busy_cur = gdk_cursor_new(GDK_WATCH);
	d->translate_cur = gdk_cursor_new(GDK_FLEUR);
	d->size_cur = gdk_cursor_new(GDK_SIZING);

	/* Create the toplevel GtkWindow */
	gtk_widget_push_visual(gdk_rgb_get_visual());
	gtk_widget_push_colormap(gdk_rgb_get_cmap());
	d->toplevel = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
	gtk_window_set_policy(
	    GTK_WINDOW(w), TRUE, TRUE, TRUE
	);
	gtk_widget_set_usize(w, PRINT_DEF_WIDTH, PRINT_DEF_HEIGHT);
	s = g_strdup_printf(
	    PRINT_DEF_TITLE ": %s",
	    STRISEMPTY(filename) ? "Untitled" : filename
	);
	gtk_window_set_title(GTK_WINDOW(w), s);
	g_free(s);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_print2_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(PrintDlgDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	parent = w;


	/* Main vbox */
	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Vbox for child widgets */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent = w;


	/* Output widgets */
	w = gtk_frame_new("Print To");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Hbox for print command widgets */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Print to printer radio */
	gslist = NULL;
	d->output_printer_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Printer");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_set_usize(w, 75, -1);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Print command prompt */
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "Print Command:",
	    NULL, &entry
	);
	d->output_printer_entry = (GtkWidget *)entry;
	GUISetWidgetTip((GtkWidget *)entry, PRINT_TTM_PRINT_COMMAND);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	/* Hbox for print to file widgets */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Radio button */
	d->output_file_radio = w =
	    gtk_radio_button_new_with_label(gslist, "File");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_set_usize(w, 75, -1);
	gtk_widget_show(w);
	/* Prompt with browse */
	w = GUIPromptBarWithBrowse(
	    NULL, "Name:",
	    NULL, &entry, &browse_btn,
	    (gpointer)d, PrintDlgPrintFileBrowseCB
	);
	d->output_file_entry = (GtkWidget *)entry;
	d->output_file_browse_btn = (GtkWidget *)browse_btn;
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);


	/* Hbox to divide into two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left side vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Visual */
	/* Frame */
	w = gtk_frame_new("Visual");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Vbox for brightness ticks and scale */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Hbox for ticks */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_minus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_brightness_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_plus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Brightness scale */
	d->brightness_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Initial value */
	    -1.0f,		/* Minimum */
	    1.0f,		/* Maximum */
	    0.1f,		/* Step inc */
	    0.1f,		/* Page inc */
	    0.0f		/* Page size */
	);
	/* Scale */
	gtk_object_ref(GTK_OBJECT(adj));	/* Keep adj around */
	d->brightness_scale = w = gtk_hscale_new(adj);
	gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	gtk_scale_set_value_pos(GTK_SCALE(w), GTK_POS_RIGHT);
	gtk_scale_set_digits(GTK_SCALE(w), 2);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(PrintDlgAdjustmentValueChangedCB), d
	);
	GUISetWidgetTip(w, PRINT_TTM_BRIGHTNESS);
	gtk_widget_show(w);

	/* Vbox for contrast ticks and scale */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Hbox for ticks */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_minus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_contrast_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_pixmap_new_from_xpm_d(
	    window, NULL,
	    (guint8 **)icon_plus_16x16_xpm
	);
	gtk_widget_set_usize(w, 16, 16);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Contrast scale */
	d->contrast_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,		/* Initial value */
	    -1.0f,		/* Minimum */
	    1.0f,		/* Maximum */
	    0.1f,		/* Step inc */
	    0.1f,		/* Page inc */
	    0.0f		/* Page size */
	);
	/* Scale */
	gtk_object_ref(GTK_OBJECT(adj));	/* Keep adj around */
	d->contrast_scale = w = gtk_hscale_new(adj);
	gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	gtk_scale_set_value_pos(GTK_SCALE(w), GTK_POS_RIGHT);
	gtk_scale_set_digits(GTK_SCALE(w), 2);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(adj), "value_changed",
	    GTK_SIGNAL_FUNC(PrintDlgAdjustmentValueChangedCB), d
	);
	GUISetWidgetTip(w, PRINT_TTM_CONTRAST);
	gtk_widget_show(w);


	/* Hbox for greyscale and color radios */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Black & White Radio */
	gslist = NULL;
	d->visual_bw_radio = w =
	    gtk_radio_button_new_with_label(gslist, "B&W");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Greyscale Radio */
	d->visual_greyscale_radio = w =
	    gtk_radio_button_new_with_label(gslist, "GreyScale");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Color Radio */
	d->visual_color_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Color");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);


	/* Orientation */
	/* Frame */
	w = gtk_frame_new("Orientation");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Hbox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Portrait radio */
	gslist = NULL;
	d->orientation_portrait_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Portrait");
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	/* Landscape radio */
	d->orientation_landscape_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Landscape");
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgToggledCB), d
	);
	gtk_widget_show(w);

	/* Paper size */
	w = gtk_frame_new("Paper Size");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Scrolled window for clist */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	scroll_parent = w;
	/* Clist */
	heading[0] = "Type";
	heading[1] = "Size";
	d->paper_size_clist = w = gtk_clist_new_with_titles(2, heading);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, 250, -1);
	gtk_container_add(GTK_CONTAINER(scroll_parent), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_hide(clist);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, PRINT_CLIST_ROW_SPACING);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 0, 90);
	gtk_clist_set_column_width(clist, 1, 10);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(PrintDlgSelectRowCB), d
	);
	gtk_widget_show(w);
	if(clist->columns >= 2)
	{
	    gint row;
	    paper_size_struct paper_size[] = PAPER_SIZE_LIST;
	    const paper_size_struct *src_psize = paper_size;
	    paper_size_struct *tar_psize;

	    while(src_psize->name != NULL)
	    {
		tar_psize = PAPER_SIZE(g_malloc0(sizeof(paper_size_struct)));
		if(tar_psize == NULL)
		    break;

		/* Set row cell values */
		heading[0] = src_psize->name;
		heading[1] = src_psize->desc;

		/* Append a new row to the clist */
		row = gtk_clist_append(GTK_CLIST(w), heading);
		if(row > -1)
		{
		    tar_psize->name = STRDUP(src_psize->name);
		    tar_psize->desc = STRDUP(src_psize->desc);
		    tar_psize->width = src_psize->width;
		    tar_psize->height = src_psize->height;
		    gtk_clist_set_row_data_full(
			GTK_CLIST(w), row,
			tar_psize, PrintDlgPaperSizeRowDataDeleteCB
		    );
		}
		else
		{
		    g_free(tar_psize);
		}

		src_psize++;
	    }
	}

	/* Hbox for resolution */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* DPI X */
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "DPI X:",
	    NULL, &entry
	);
	d->dpi_x_entry = (GtkWidget *)entry;
	GUISetWidgetTip((GtkWidget *)entry, PRINT_TTM_DPI_X);
	gtk_widget_set_usize((GtkWidget *)entry, 50, -1);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    (GtkObject *)entry, "changed",
	    GTK_SIGNAL_FUNC(PrintDlgDPIChangedCB), d
	);
	gtk_widget_show(w);
	/* DPI Y */
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "DPI Y:",
	    NULL, &entry
	);
	d->dpi_y_entry = (GtkWidget *)entry;
	GUISetWidgetTip((GtkWidget *)entry, PRINT_TTM_DPI_Y);
	gtk_widget_set_usize((GtkWidget *)entry, 50, -1);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    (GtkObject *)entry, "changed",
	    GTK_SIGNAL_FUNC(PrintDlgDPIChangedCB), d
	);
	gtk_widget_show(w);




	/* Right side vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Output geometry */
	w = gtk_frame_new("Output");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Output X */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("X:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    -(gfloat)((guint16)-1 / 2),
	    (gfloat)((guint16)-1 / 2),
	    1.0f,
	    10.0f,
	    0.0f
	);
	d->output_x_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);
	/* Output Y */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("Y:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    -(gfloat)((guint16)-1 / 2),
	    (gfloat)((guint16)-1 / 2),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_y_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);
	/* Output Width */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("W:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    0.0f,
	    (gfloat)((guint16)-1),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_width_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);
	/* Output Height */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("H:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    0.0f,
	    (gfloat)((guint16)-1),
	    1.0f,
	    10.0f,
	    0.0f  
	);
	d->output_height_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);
	/* Hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Maintain Aspect */
	d->output_maintain_aspect_check = w = gtk_check_button_new_with_label(
	    "Maintain Aspect"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintDlgOutputChangedCB), d
	);
	gtk_widget_show(w);
	/* Output Number Of Coppies */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	w = gtk_label_new("Number Of Coppies:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f,
	    0.0f,
	    (gfloat)((guint32)-1),
	    1.0f,
	    5.0f,
	    0.0f  
	);
	d->output_ncoppies_spin = w = gtk_spin_button_new(
	    adj, 1.0, 0
	);
	gtk_widget_set_usize(w, 65, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Preview */
	w = gtk_frame_new("Preview");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Vbox */
	w = gtk_vbox_new(TRUE, border_minor);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;
	/* Hbox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Frame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_border_width(GTK_CONTAINER(w), border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Nother vbox */
	w = gtk_vbox_new(TRUE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);
	parent5 = w;
	/* Drawing area */
	d->preview_da = w = gtk_drawing_area_new();
	/* Needs to accept focus for keyboard modifier events */
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_FOCUS | GTK_CAN_DEFAULT);
	gtk_widget_add_events(
	    w,
	    GDK_EXPOSURE_MASK | GDK_ENTER_NOTIFY_MASK |
	    GDK_LEAVE_NOTIFY_MASK | GDK_KEY_PRESS_MASK |
	    GDK_KEY_RELEASE_MASK | GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
	    GDK_POINTER_MOTION_HINT_MASK | GDK_STRUCTURE_MASK
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewExposeCB), d
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(PrintDlgPreviewEventCB), d
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GUISetWidgetTip(w, PRINT_TTM_PREVIEW);
	gtk_widget_show(w);

	/* Create preview graphic contexts */
	style = gtk_widget_get_style(w);
	if(style != NULL)
	{
	    GdkGC *gc;

	    /* Use drawing area's base GC */
	    gc = style->base_gc[GTK_STATE_NORMAL];
	    if(gc != NULL)
	    {
		gdk_gc_ref(gc);
		d->gc_base = gc;
	    }
	}
	/* Create invert GC */
	colormap = gtk_widget_get_colormap(w);
	if(colormap == NULL)
	    colormap = gdk_colormap_get_system();
	if(colormap != NULL)
	{
	    GdkColor *cf, *cb;
	    GdkGCValues gcv;
	    GdkGCValuesMask gcv_mask = (
		GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | 
		GDK_GC_FUNCTION | GDK_GC_FILL | GDK_GC_LINE_WIDTH |
		GDK_GC_LINE_STYLE | GDK_GC_CAP_STYLE | GDK_GC_JOIN_STYLE
	    );
	    cf = &gcv.foreground;
	    GDK_COLOR_SET_COEFF(cf, 0.0f, 0.0f, 0.0f)
	    GDK_COLORMAP_ALLOC_COLOR(colormap, cf)
	    cb = &gcv.background;
	    GDK_COLOR_SET_COEFF(cb, 1.0f, 1.0f, 1.0f)
	    GDK_COLORMAP_ALLOC_COLOR(colormap, cb)
	    gcv.function = GDK_INVERT;
	    gcv.fill = GDK_SOLID;
	    gcv.line_width = 1;
	    gcv.line_style = GDK_LINE_SOLID;
	    gcv.cap_style = GDK_CAP_NOT_LAST;
	    gcv.join_style = GDK_JOIN_MITER;
	    d->gc_invert = gdk_gc_new_with_values(
		window, &gcv, gcv_mask
	    );
	    GDK_COLORMAP_FREE_COLOR(colormap, cf)
	    GDK_COLORMAP_FREE_COLOR(colormap, cb)
	}


	/* Separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Print button */
	d->print_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_print2_20x20_xpm, "Print", NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintDlgPrintCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_p);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Cancele", 
#elif defined(PROG_LANGUAGE_FRENCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
	    "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
	    "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
	    "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	    "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	    "Kanseler",
#else
	    "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"                             
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);




	/* Resize the preview drawing area */
	PrintDlgResizePreview(d);

	/* Recreate the preview image */
	PrintDlgRecreatePreview(d);

	return(d);
}

/*
 *	Maps the print dialog.
 */
static void PrintDlgMap(print_dlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	if(!d->map_state)
	{
	    w = d->print_btn;
	    if(w != NULL)
	    {
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	    }

	    w = d->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

	    d->map_state = TRUE;
	}
}

/*
 *	Unmaps the print dialog.
 */
static void PrintDlgUnmap(print_dlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	if(d->map_state)
	{
	    w = d->toplevel;
	    if(w != NULL)
		gtk_widget_hide(w);

	    d->map_state = FALSE;
	}
}

/*
 *	Destroys the print dialog.
 */
static void PrintDlgDelete(print_dlg_struct *d)
{
	if(d == NULL)
	    return;

	d->src_img = NULL;

	/* Delete preview images */
	gdk_flush();
	ImgViewImageDelete(d->preview_img);
	d->preview_img = NULL;
	ImgViewImageDelete(d->orig_preview_img);
	d->orig_preview_img = NULL;


	d->freeze_count++;

	/* Begin destroying widgets */
	GTK_WIDGET_DESTROY(d->menu);
	GTK_WIDGET_DESTROY(d->paper_size_clist);

	GTK_WIDGET_DESTROY(d->output_printer_radio);
	GTK_WIDGET_DESTROY(d->output_file_radio);
	GTK_WIDGET_DESTROY(d->output_printer_entry);
	GTK_WIDGET_DESTROY(d->output_file_entry);
	GTK_WIDGET_DESTROY(d->output_file_browse_btn);

	GTK_WIDGET_DESTROY(d->brightness_scale);
	GTK_WIDGET_DESTROY(d->contrast_scale);

	GTK_WIDGET_DESTROY(d->visual_bw_radio);
	GTK_WIDGET_DESTROY(d->visual_greyscale_radio);
	GTK_WIDGET_DESTROY(d->visual_color_radio);

	GTK_WIDGET_DESTROY(d->orientation_portrait_radio);
	GTK_WIDGET_DESTROY(d->orientation_landscape_radio);

	GTK_WIDGET_DESTROY(d->dpi_x_entry);
	GTK_WIDGET_DESTROY(d->dpi_y_entry);

	GTK_WIDGET_DESTROY(d->output_x_spin);
	GTK_WIDGET_DESTROY(d->output_y_spin);
	GTK_WIDGET_DESTROY(d->output_width_spin);
	GTK_WIDGET_DESTROY(d->output_height_spin);
	GTK_WIDGET_DESTROY(d->output_maintain_aspect_check);
	GTK_WIDGET_DESTROY(d->output_ncoppies_spin);

	GTK_WIDGET_DESTROY(d->preview_da);

	GTK_WIDGET_DESTROY(d->print_btn);
	GTK_WIDGET_DESTROY(d->cancel_btn);

	GTK_WIDGET_DESTROY(d->toplevel);

	GTK_ACCEL_GROUP_UNREF(d->accelgrp);

	GTK_OBJECT_UNREF(d->brightness_adj);
	GTK_OBJECT_UNREF(d->contrast_adj);

	GDK_CURSOR_DESTROY(d->busy_cur);
	GDK_CURSOR_DESTROY(d->translate_cur);
	GDK_CURSOR_DESTROY(d->size_cur);

	GDK_GC_UNREF(d->gc_base);
	GDK_GC_UNREF(d->gc_invert);

	d->freeze_count--;

	g_free(d);
}


/*
 *	Queries user to print the specified ImgView image.
 *
 *	Creates a new print dialog and blocks output until the user
 *	responds.
 *
 *	The specified values *v will be used to initialize the print
 *	values and *v will be modified if TRUE is returned, otherwise
 *	*v should be considered garbage after this call.
 */
#ifdef HAVE_LIBENDEAVOUR2
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
)
#else
gboolean PrintDlgGetResponse(
	imgview_image_struct *img,		/* Image to print */
	const gchar *filename,			/* File name */
	print_values_struct *v,			/* Print values (will be modified) */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel
)
#endif
{
	gboolean got_response;
	gint frame_num;
	GtkWidget *w;
	guint vflags;
	print_dlg_struct *d;

	if(v == NULL)
	    return(FALSE);

	/* Create a new print dialog */
	d = PrintDlgNew(filename);
	if(d == NULL)
	    return(FALSE);

	/* Begin setting values */
	vflags = v->flags;

	frame_num = (vflags & PRINT_VALUE_FRAME) ?
	    v->frame : 0;

	/* Source ImgView image that user wants to print */
	d->src_img = img;
	d->src_frame_num = frame_num;

	/* Background color */
	if(bg_color != NULL)
	    memcpy(&d->bg_color, bg_color, 4);
	else
	    memset(&d->bg_color, 0xffffffff, 4);

	/* Output image geometry */
	if(vflags & PRINT_VALUE_OUTPUT_X)
	    d->output_x = v->output_x;
	if(vflags & PRINT_VALUE_OUTPUT_Y)
	    d->output_y = v->output_y;
	if(vflags & PRINT_VALUE_OUTPUT_WIDTH)
	    d->output_width = v->output_width;
	if(vflags & PRINT_VALUE_OUTPUT_HEIGHT)
	    d->output_height = v->output_height;
	if(vflags & PRINT_VALUE_OUTPUT_MAINTAIN_ASPECT)
	    d->output_maintain_aspect = v->output_maintain_aspect;

	/* Update output entry widgets with the new output geometry
	 * values
	 */
	PrintDlgSyncOutputWidgets(d);

	/* Print command */
	if(vflags & PRINT_VALUE_PRINT_COMMAND)
	{
	    w = d->output_printer_entry;
	    if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
		gtk_entry_set_text(
		    GTK_ENTRY(w),
		    (v->print_command != NULL) ? v->print_command : ""
		);
	}
	/* File name */
	if(vflags & PRINT_VALUE_FILE_NAME)
	{
	    w = d->output_file_entry;
	    if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
		gtk_entry_set_text(
		    GTK_ENTRY(w),
		    (v->file_name != NULL) ? v->file_name : ""
		);
	}
	/* Print output type */
	if(vflags & PRINT_VALUE_OUTPUT_TYPE)
	{
	    if(v->output == PRINT_OUTPUT_TO_FILE)
	    {
	        w = d->output_printer_radio;
	        if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;
		w = d->output_file_radio;
		if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;

		w = d->output_file_radio;
		if(w != NULL)
		    gtk_signal_emit_by_name(GTK_OBJECT(w), "toggled");
	    }
	    else
	    {
		w = d->output_printer_radio;
		if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		w = d->output_file_radio;
		if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;

		w = d->output_printer_radio;
		if(w != NULL)
		    gtk_signal_emit_by_name(GTK_OBJECT(w), "toggled");
	    }
	}
	/* Brightness */
	if(vflags & PRINT_VALUE_BRIGHTNESS)
	{
	    GtkAdjustment *adj = d->brightness_adj;
	    if(adj != NULL)
	    {
		adj->value = v->brightness;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		if(adj->value > adj->upper)
		    adj->value = adj->upper;
		gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    }
	}
	/* Contrast */
	if(vflags & PRINT_VALUE_CONTRAST)
	{
	    GtkAdjustment *adj = d->contrast_adj;
	    if(adj != NULL)
	    {
		adj->value = v->contrast;
		if(adj->value < adj->lower)
		    adj->value = adj->lower;
		if(adj->value > adj->upper)
		    adj->value = adj->upper;
		gtk_signal_emit_by_name(GTK_OBJECT(adj), "changed");
	    }
	}
	/* Visual */
	if(vflags & PRINT_VALUE_VISUAL)
	{
	    w = d->visual_bw_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = d->visual_greyscale_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = d->visual_color_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;

	    switch(v->visual)
	    {
	      case PRINT_VISUAL_BW:
		w = d->visual_bw_radio;
		break;
	      case PRINT_VISUAL_GREYSCALE:
		w = d->visual_greyscale_radio;
		break;
	      case PRINT_VISUAL_RGB:
		w = d->visual_color_radio;
		break;
	    }
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
	    {
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
/*		gtk_signal_emit_by_name(GTK_OBJECT(w), "toggled"); */
	    }
	}
	/* Orientation */
	if(vflags & PRINT_VALUE_ORIENTATION)
	{
	    w = d->orientation_portrait_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = d->orientation_landscape_radio;
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;

	    switch(v->orientation)
	    {
	      case PRINT_ORIENTATION_PORTRAIT:
		w = d->orientation_portrait_radio;
		break;
	      case PRINT_ORIENTATION_LANDSCAPE:
		w = d->orientation_landscape_radio;
		/* Need to swap the output size for the landscape
		 * orientation
		 */
		if((vflags & PRINT_VALUE_OUTPUT_WIDTH) &&
		   (vflags & PRINT_VALUE_OUTPUT_HEIGHT)
		)
		{
		    const gint i = d->output_width;
		    d->output_width = d->output_height;
		    d->output_height = i;
		}
		break;
	    }
	    if((w != NULL) ? GTK_IS_TOGGLE_BUTTON(w) : FALSE)
	    {
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
/*		gtk_signal_emit_by_name(GTK_OBJECT(w), "toggled"); */
	    }
	}
	/* DPI X */
	if(vflags & PRINT_VALUE_DPI_X)
	{
	    gchar num_str[40];
	    g_snprintf(num_str, sizeof(num_str), "%i", v->dpi_x);
	    gtk_entry_set_text(GTK_ENTRY(d->dpi_x_entry), num_str);
	}
	/* DPI Y */
	if(vflags & PRINT_VALUE_DPI_Y)
	{
	    gchar num_str[40];
	    g_snprintf(num_str, sizeof(num_str), "%i", v->dpi_y);
	    gtk_entry_set_text(GTK_ENTRY(d->dpi_y_entry), num_str);
	}
	/* Paper size */
	if((vflags & PRINT_VALUE_PAPER_WIDTH) &&
	   (vflags & PRINT_VALUE_PAPER_HEIGHT)
	)
	{
	    GtkCList *clist = (GtkCList *)d->paper_size_clist;
	    if(clist != NULL)
	    {
		gint row;
		const paper_size_struct *psize;

		/* Select the row who's paper size matches the
		 * specified paper size
		 */
		for(row = 0; row < clist->rows; row++)
		{
		    psize = PAPER_SIZE(
			gtk_clist_get_row_data(clist, row)
		    );
		    if(psize == NULL)
			continue;

		    if((v->paper_width == psize->width) &&
		       (v->paper_height == psize->height)
		    )
		    {
			gtk_clist_select_row(clist, row, 0);
			break;
		    }
		}
		/* No row selected? */
		if(row >= clist->rows)
		{
		    /* select first row if possible */
		    if(clist->rows > 0)
			gtk_clist_select_row(clist, 0, 0);
		}
	    }
	}
	/* Note that selecting a paper size above with 
	 * gtk_clist_select_row() would have called the "select_row"
	 * callback and update the preview widget and preview image
	 */
	/* Number Of Coppies */
	if(vflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	    gtk_spin_button_set_value(
		GTK_SPIN_BUTTON(d->output_ncoppies_spin),
		(gfloat)v->output_ncoppies
	    );

	PrintDlgSyncOutputWidgets(d);


	/* Set reference toplevel? */
	if((toplevel != NULL) && (d->toplevel != NULL))
	{
	    if(GTK_IS_WINDOW(toplevel))
	    {
		gtk_window_set_modal(
		    GTK_WINDOW(d->toplevel), TRUE
	        );
	        gtk_window_set_transient_for(
		    GTK_WINDOW(d->toplevel), GTK_WINDOW(toplevel)
	        );
	    }
	}


	/* Map print dialog */
	PrintDlgMap(d);

	/* Enter main loop and wait for user response */
	d->main_level++;
	gtk_main();

	/* Break out of any additional main loop levels */
	while(d->main_level > 0)
	{
	    d->main_level--;
	    gtk_main_quit();
	}


	/* Got response? */
	got_response = d->got_response;
	if(got_response)
	{
	    /* Get the values from the print dialog and update the
	     * specified print values with them
	     */

	    /* Print output type */
	    if(vflags & PRINT_VALUE_OUTPUT_TYPE)
	    {
		w = d->output_file_radio;
		if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
		    v->output = PRINT_OUTPUT_TO_FILE;
		else
		    v->output = PRINT_OUTPUT_TO_PRINTER;
	    }
	    /* Print command */
	    if(vflags & PRINT_VALUE_PRINT_COMMAND)
	    {
		w = d->output_printer_entry;
		if(w != NULL)
		{
		    g_free(v->print_command);
		    v->print_command = STRDUP(
			gtk_entry_get_text(GTK_ENTRY(w))
		    );
		}
	    }
	    /* Print to file name */
	    if(vflags & PRINT_VALUE_FILE_NAME)
	    {
		w = d->output_file_entry;
		if(w != NULL)
		{
		    g_free(v->file_name);
		    v->file_name = STRDUP(
			gtk_entry_get_text(GTK_ENTRY(w))
		    );
		}
	    }
	    /* Brightness */
	    if(vflags & PRINT_VALUE_BRIGHTNESS)
	    {
		GtkAdjustment *adj = d->brightness_adj;
		if(adj != NULL)
		{
		    v->brightness = adj->value;
		}
	    }
	    /* Contrast */
	    if(vflags & PRINT_VALUE_CONTRAST)
	    {
		GtkAdjustment *adj = d->contrast_adj;
		if(adj != NULL)
		{
		    v->contrast = adj->value;
		}
	    }
	    /* Visual */
	    if(vflags & PRINT_VALUE_VISUAL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->visual_bw_radio
		))
		    v->visual = PRINT_VISUAL_BW;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(   
		    d->visual_greyscale_radio 
		)) 
		    v->visual = PRINT_VISUAL_GREYSCALE; 
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->visual_color_radio 
		)) 
		    v->visual = PRINT_VISUAL_RGB;
	    }
	    /* Orientation */
	    if(vflags & PRINT_VALUE_ORIENTATION)
	    {
		w = d->orientation_landscape_radio;
		if((w != NULL) ? GTK_TOGGLE_BUTTON(w)->active : FALSE)
		    v->orientation = PRINT_ORIENTATION_LANDSCAPE;
		else
		    v->orientation = PRINT_ORIENTATION_PORTRAIT;
	    }
	    /* DPI X */
	    if(vflags & PRINT_VALUE_DPI_X)
	    {
		w = d->dpi_x_entry;
		if((w != NULL) ? GTK_IS_ENTRY(w) : FALSE)
		{
		    const gchar *s = gtk_entry_get_text(GTK_ENTRY(w));
		    if(s != NULL)
			v->dpi_x = atoi(s);
		}
	    }
	    /* DPI Y */
	    if(vflags & PRINT_VALUE_DPI_Y)
	    {
		w = d->dpi_y_entry;
		if((w != NULL) ? GTK_IS_ENTRY(w) : 0)
		{
		    const gchar *s = gtk_entry_get_text(GTK_ENTRY(w));
		    if(s != NULL)
			v->dpi_y = atoi(s);
		}
	    }
	    /* Paper size */
	    if((vflags & PRINT_VALUE_PAPER_WIDTH) &&
	       (vflags & PRINT_VALUE_PAPER_HEIGHT)
	    )
	    {
		/* Get paper size (without DPI applied) */
		PrintDlgPaperSize(d, &v->paper_width, &v->paper_height, FALSE);
	    }

	    /* Output image geometry */
	    if(vflags & PRINT_VALUE_OUTPUT_X)
		v->output_x = d->output_x;
	    if(vflags & PRINT_VALUE_OUTPUT_Y)
		v->output_y = d->output_y;
	    if(vflags & PRINT_VALUE_OUTPUT_WIDTH)
		v->output_width = d->output_width;
	    if(vflags & PRINT_VALUE_OUTPUT_HEIGHT)
		v->output_height = d->output_height;

	    if(vflags & PRINT_VALUE_OUTPUT_MAINTAIN_ASPECT)
		v->output_maintain_aspect = d->output_maintain_aspect;

	    if(vflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	    {
		w = d->output_ncoppies_spin;
		if((w != NULL) ? GTK_IS_SPIN_BUTTON(w) : FALSE)
		    v->output_ncoppies = gtk_spin_button_get_value_as_int(
			GTK_SPIN_BUTTON(w)
		    );
	    }
	}


	/* Unset reference toplevel */
	if(d->toplevel != NULL)
	{
	    gtk_widget_hide(d->toplevel);
	    gtk_window_set_modal(
		GTK_WINDOW(d->toplevel), FALSE
	    );
	    gtk_window_set_transient_for(
		GTK_WINDOW(d->toplevel), NULL
	    );
	}

	/* Delete the print dialog */
	PrintDlgDelete(d);

	return(got_response);
#undef PRINT_COMMAND_MAX
}

#endif
