#if !defined(_WIN32) && defined(HAVE_XPRINT)
#include <stdio.h>
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/Print.h>
#include <gtk/gtk.h>
#include "print_values.h"
#include "print.h"
#include "cdialog.h"
#include "progressdialog.h"
#include "config.h"

#include "images/icon_print2_32x32.xpm"


static gchar *get_string(
	const gchar *s,
	const gchar **next_s,
	const gchar *delim_chars
);

static void pixel_set_rgb_to_rgb(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_rgba(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_bgr(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_bgra(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_grey_alpha(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_alpha_grey(guint8 *tar, const guint8 *src);
static void pixel_set_rgb_to_grey(guint8 *tar, const guint8 *src);


/* Connect/Disconnect From X Print Server */
static gint iv_print_xprint_connect(
	Display **dpy_rtn,
	int *xp_event_base_rtn, int *xp_error_base_rtn,
	const gchar *address
);
static void iv_print_xprint_disconnect(Display *dpy);

/* Color */
static unsigned long iv_print_xprint_get_pixel_rgb(
	Display *dpy, Colormap cmap,
	guint8 r, guint8 g, guint8 b
);

/* X Visual */
static Visual *iv_print_xprint_get_best_visual(
	Display *dpy,
	int visual_class,
	int *depth_rtn
);

/* Printers */
iv_print_xprint_printer_struct *iv_print_xprint_printer_new(void);
void iv_print_xprint_printer_delete(iv_print_xprint_printer_struct *m);
void iv_print_xprint_printers_list_delete(GList *glist);

/* Mediums */
iv_print_xprint_medium_struct *iv_print_xprint_medium_new(void);
void iv_print_xprint_medium_delete(iv_print_xprint_medium_struct *m);
void iv_print_xprint_mediums_list_delete(GList *glist);

/* Get Printer Attributes */
GList *iv_print_xprint_get_printers_list(
	const gchar *xp_address
);
GList *iv_print_xprint_get_printer_trays(
	const gchar *xp_address,
	const gchar *printer_name
);
GList *iv_print_xprint_get_printer_mediums(
	const gchar *xp_address,
	const gchar *printer_name
);
GList *iv_print_xprint_get_printer_orientations(
	const gchar *xp_address,
	const gchar *printer_name
);
GList *iv_print_xprint_get_printer_resolutions(
	const gchar *xp_address,
	const gchar *printer_name
);

/* Get Medium Size */
gint iv_print_xprint_get_medium_size(
	const gchar *xp_address,
	const gchar *printer_name,
	const gchar *tray_name,
	const gchar *medium_name,
	const gchar *orientation_name,
	const gint dpi,
	gint *width_rtn, gint *height_rtn
);

/* Callbacks */
static int iv_print_x_error_cb(Display *dpy, XErrorEvent *error);
static gint iv_print_xprint_progress_cb(
	const gulong i, const gulong m, gpointer data
);

/* Render Image For Printing */
static XImage *iv_print_xprint_render_ximage(
	Display *dpy, Visual *visual, const int depth,
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
);

/* Print Main */
static gint iv_print_xprint_main_iteration(
	Display *dpy, Visual *visual, const int depth,
	Window w, GC gc,
	XPContext xp_ctx,
	int xp_event_base, int xp_error_base,
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	const gboolean verbose, const gboolean interactive,
	GtkWidget *toplevel
);
static gint iv_print_xprint_main(
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	const gboolean verbose, const gboolean interactive,
	GtkWidget *toplevel
);

#ifdef HAVE_LIBENDEAVOUR2
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
);
#else
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel 
);
#endif


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the string value.
 *
 *	The string value will be obtained from the position specified
 *	by s up until one of the deliminator characters is reached or
 *	the null byte. If the string value starts with a " or '
 *	character then all characters up until the ending " or '
 *	character will be obtained.
 *
 *	The s specifies the current string position to obtain the
 *	string value from.
 *
 *	The next_s specifies the return value for the position
 *	after the obtained value.
 *
 *	The delim_chars specifies the list of deliminator characters.
 *
 *	The returned string will be dynamically allocated.
 */
static gchar *get_string(
	const gchar *s, 
	const gchar **next_s_rtn,
	const gchar *delim_chars
)
{
	gchar *s_rtn;

	if(*s == '"')
	{
	    const gchar *s_end;
	    gint len;

	    s++;	/* Skip the double quote character */

	    /* Seek to the second double quote character */
	    s_end = (const gchar *)strchr(
		(const char *)s, '"'
	    );

	    /* Calculate the number of characters between the first
	     * and second double quote characters or to the end of the
	     * string
	     */
	    len = (s_end != NULL) ?
		(gint)(s_end - s) : STRLEN(s);

	    /* Get the characters (if any) between double quote
	     * characters as the as the return string
	     */
	    if(len > 0)
	    {
		s_rtn = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		memcpy(s_rtn, s, len * sizeof(gchar));
		s_rtn[len] = '\0';
	    }
	    else
	    {
		s_rtn = NULL;
	    }

	    /* Set the string position return past the second double
	     * quote character or to the end of the string (whichever
	     * occured)
	     */
	    *next_s_rtn = (s_end != NULL) ?
		(s_end + 1) : (s + len);
	}
	else if(*s == '\'')
	{
	    const gchar *s_end;
	    gint len;

	    s++;	/* Skip the quote character */

	    /* Seek to the second quote character */
	    s_end = (const gchar *)strchr(
		(const char *)s, '\''
	    );

	    /* Calculate the number of characters between the first
	     * and second quote characters or to the end of the string
	     */
	    len = (s_end != NULL) ?
		(gint)(s_end - s) : STRLEN(s);

	    /* Get the characters (if any) between quote characters
	     * as the as the return string
	     */
	    if(len > 0)
	    {
		s_rtn = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		memcpy(s_rtn, s, len * sizeof(gchar));
		s_rtn[len] = '\0';
	    }
	    else
	    {
		s_rtn = NULL;
	    }

	    /* Set the string position return past the second quote
	     * character or to the end of the string (whichever occured)
	     */
	    *next_s_rtn = (s_end != NULL) ?
		(s_end + 1) : (s + len);
	}
	else
	{
	    /* Seek to the first occurance of one of the specified
	     * deliminator characters
	     */
	    const gchar *s_end = (const gchar *)strpbrk(
		(const char *)s, (const char *)delim_chars
	    );

	    /* Calculate the number of characters to the first
	     * occurance of one of the specified deliminator
	     * characters or to the end of the string
	     */
	    const gint len = (s_end != NULL) ?
		(gint)(s_end - s) : STRLEN(s);

	    /* Get the characters (if any) as the as the return
	     * string
	     */
	    if(len > 0)
	    {
		s_rtn = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		memcpy(s_rtn, s, len * sizeof(gchar));
		s_rtn[len] = '\0';
	    }
	    else
	    {
		s_rtn = NULL;
	    }

	    /* Set the string position return past the first
	     * occurance of the specified deliminator characters or
	     * to the end of the string (whichever occured)
	     */
	    *next_s_rtn = (s_end != NULL) ?
		s_end : (s + len);
	}

	return(s_rtn);
}


/*
 *	Pixel set functions.
 *
 *	The src is always assumed to be at least 3 bytes in RGB
 *	format.
 */
static void pixel_set_rgb_to_rgb(guint8 *tar, const guint8 *src)
{
	tar[0] = src[0];
	tar[1] = src[1];
	tar[2] = src[2];
}

static void pixel_set_rgb_to_rgba(guint8 *tar, const guint8 *src)
{
	tar[0] = src[0];
	tar[1] = src[1];
	tar[2] = src[2];
	tar[3] = 0xff;
}

static void pixel_set_rgb_to_bgr(guint8 *tar, const guint8 *src)
{
	tar[0] = src[2];
	tar[1] = src[1];
	tar[2] = src[0];
}

static void pixel_set_rgb_to_bgra(guint8 *tar, const guint8 *src)
{
	tar[0] = src[2];
	tar[1] = src[1];
	tar[2] = src[0];
	tar[3] = 0xff;
}

static void pixel_set_rgb_to_grey_alpha(guint8 *tar, const guint8 *src)
{
	tar[0] = src[0];
	tar[1] = src[1];
}

static void pixel_set_rgb_to_alpha_grey(guint8 *tar, const guint8 *src)
{
	tar[0] = src[1];
	tar[1] = src[0];
}

static void pixel_set_rgb_to_grey(guint8 *tar, const guint8 *src)
{
	tar[0] = src[0];
}


/*
 *	Connects to the X Print server.
 *
 *	The address specifies the address to the X Print server. If
 *	address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	Returns 0 on success and non-zero on error.
 */
static gint iv_print_xprint_connect(
	Display **dpy_rtn,
	int *xp_event_base_rtn, int *xp_error_base_rtn,
	const gchar *address
)
{
	Display *dpy;

	/* Connect to the X server */
	if(address != NULL)
	{
	    dpy = XOpenDisplay(address);
	}
	else
	{
	    address = g_getenv("XPSERVERLIST");
	    dpy = XOpenDisplay(address);
	}

	/* Failed to connect to the X Server? */
	if(dpy == NULL)
	{
	    *dpy_rtn = NULL;
	    *xp_event_base_rtn = 0;
	    *xp_error_base_rtn = 0;
	    return(-1);
	}

	*dpy_rtn = dpy;

	/* Set the X error handler */
	XSetErrorHandler(iv_print_x_error_cb);

	/* Check if X Print is supported on this X Server */
	if(!XpQueryExtension(
	    dpy, xp_event_base_rtn, xp_error_base_rtn
	))
	{
	    XCloseDisplay(dpy);
	    *dpy_rtn = NULL;
	    return(-2);
	}

	return(0);
}

/*
 *	Disconnects from the X Print server.
 */
static void iv_print_xprint_disconnect(Display *dpy)
{
	if(dpy != NULL)
	    XCloseDisplay(dpy);
}


/*
 *	Gets the pixel from the colormap by RGB values.
 */
static unsigned long iv_print_xprint_get_pixel_rgb(
	Display *dpy, Colormap cmap,
	guint8 r, guint8 g, guint8 b
)
{
	XColor xc;
	xc.flags = DoRed | DoGreen | DoBlue;
	xc.red = ((unsigned short)r << 8) |
	    (unsigned short)r;
	xc.green = ((unsigned short)g << 8) |
	    (unsigned short)g;
	xc.blue = ((unsigned short)b << 8) |
	    (unsigned short)b;
	XAllocColor(dpy, cmap, &xc);
	return(xc.pixel);
}


/*
 *	Gets the best X Visual suitable for printing.
 *
 *	Returns the X Visual or NULL if no suitable X Visual was
 *	found.
 */
static Visual *iv_print_xprint_get_best_visual(
	Display *dpy,
	int visual_class,
	int *depth_rtn
)
{
	int screen = DefaultScreen(dpy);
	XVisualInfo visual_info;

	/* Begin checking for a suitable X Visual */

	/* TrueColor */
	if(visual_class == TrueColor)
	{
	    /* 32 bit depth TrueColor */
	    if(XMatchVisualInfo(
		dpy, screen,
		32,			/* 32 bit depth */
		visual_class,
		&visual_info
	    ))
	    {
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }

	    /* 24 bit depth TrueColor */
	    if(XMatchVisualInfo(
		dpy, screen,
		24,			/* 24 bit depth */
		visual_class,
		&visual_info
	    ))
	    { 
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }
	}
	/* GrayScale */
	else if(visual_class == GrayScale)
	{
	    /* 16 bit depth GrayScale */
	    if(XMatchVisualInfo(
		dpy, screen,
		16,			/* 16 bit depth */
		visual_class,
		&visual_info
	    ))
	    {
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }

	    /* 8 bit depth GrayScale */
	    if(XMatchVisualInfo(
		dpy, screen,
		8,			/* 8 bit depth */
		visual_class,
		&visual_info
	    ))
	    { 
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }

	    /* Resort to TrueColor when a GrayScale X Visual is
	     * not available
	     */

	    /* 32 bit depth TrueColor */
	    if(XMatchVisualInfo(
		dpy, screen,
		32,			/* 32 bit depth */
		TrueColor,
		&visual_info
	    ))
	    {
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }

	    /* 24 bit depth TrueColor */
	    if(XMatchVisualInfo(
		dpy, screen,
		24,			/* 24 bit depth */
		TrueColor,
		&visual_info
	    ))
	    { 
		*depth_rtn = visual_info.depth;
		return(visual_info.visual);
	    }
	}

	return(NULL);
}


/*
 *	Creates a new X Print printer.
 */
iv_print_xprint_printer_struct *iv_print_xprint_printer_new(void)
{
	return(IV_PRINT_XPRINT_PRINTER(g_malloc0(
	    sizeof(iv_print_xprint_printer_struct)
	)));
}

/*
 *	Deletes the X Print printer..
 */
void iv_print_xprint_printer_delete(iv_print_xprint_printer_struct *m)
{
	if(m == NULL)
	    return;

	g_free(m->name);
	g_free(m->description);
	g_free(m);
}


/*
 *	Deletes the list of X Print printers.
 */
void iv_print_xprint_printers_list_delete(GList *glist)
{
	if(glist == NULL)
	    return;

	g_list_foreach(glist, (GFunc)iv_print_xprint_printer_delete, NULL);
	g_list_free(glist);
}


/*
 *	Creates a new X Print medium.
 */
iv_print_xprint_medium_struct *iv_print_xprint_medium_new(void)
{
	return(IV_PRINT_XPRINT_MEDIUM(g_malloc0(
	    sizeof(iv_print_xprint_medium_struct)
	)));
}

/*
 *	Deletes the X Print medium.
 */
void iv_print_xprint_medium_delete(iv_print_xprint_medium_struct *m)
{
	if(m == NULL)
	    return;

	g_free(m->name);
	g_free(m->tray);
	g_free(m);
}

/*
 *	Deletes the list of X Print mediums.
 */
void iv_print_xprint_mediums_list_delete(GList *glist)
{
	if(glist == NULL)
	    return;

	g_list_foreach(glist, (GFunc)iv_print_xprint_medium_delete, NULL);
	g_list_free(glist);
}


/*
 *	Returns a list of printers.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	Returns a glist of iv_print_xprint_printer_struct describing
 *	each printer It must be deleted by the calling
 *	iv_print_xprint_printers_list_delete().
 */
GList *iv_print_xprint_get_printers_list(
	const gchar *xp_address
)
{
	int xp_event_base, xp_error_base, i, nprinters;
	Display *dpy;
	XPPrinterList xp, xp_printers_list;
	GList *printers_list;
	iv_print_xprint_printer_struct *p;

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(NULL);

	/* Get the printers list from the X Print server */
	xp_printers_list = XpGetPrinterList(dpy, NULL, &nprinters);
	if(xp_printers_list == NULL)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Parse the printers list */
	printers_list = NULL;
	for(i = 0; i < nprinters; i++)
	{
	    xp = &xp_printers_list[i];
	    p = iv_print_xprint_printer_new();
	    if(p != NULL)
	    {
		p->name = STRDUP(xp->name);
		p->description = STRDUP(xp->desc);
		printers_list = g_list_append(printers_list, p);
	    }
	}

	/* Delete the printers list obtained from the X Print server */
	XpFreePrinterList(xp_printers_list);
	iv_print_xprint_disconnect(dpy);

	return(printers_list);
}

/*
 *	Gets the list of trays supported by the printer.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	The printer_name specifies the name of the printer obtained
 *	from iv_print_xprint_get_printers_list(). This value may
 *	not be NULL.
 *
 *	Returns a glist of strings describing each tray.
 */
GList *iv_print_xprint_get_printer_trays(
	const gchar *xp_address,
	const gchar *printer_name
)
{
	int xp_event_base, xp_error_base;
	gint level;
	char *xp_trays_list;
	const gchar *s;
	Display *dpy;
	XPContext xp_ctx;
	GList *trays_list;

	if(printer_name == NULL)
	    return(NULL);

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(NULL);

	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Get the list of trays supported by the specified
	 * printer from the X Print server
	 */
	xp_trays_list = XpGetOneAttribute(
	    dpy, xp_ctx,
	    XPPrinterAttr,
	    "input-trays-supported"
	);
	if(xp_trays_list == NULL)
	{
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Parse the trays list */
	trays_list = NULL;
	level = 0;
	s = (const gchar *)xp_trays_list;
	while(*s != '\0')
	{
	    while((*s == ' ') || (*s == '\t'))
		s++;

	    if(*s != '\0')
	    {
		const gchar *s_end;
		gchar *name = get_string(
		    s, &s_end, " \t{}"
		);
		s = s_end;

		trays_list = g_list_append(
		    trays_list, name
		);
	    }
	}

	XFree(xp_trays_list);
	XpDestroyContext(dpy, xp_ctx);
	iv_print_xprint_disconnect(dpy);

	return(trays_list);
}

/*
 *	Gets the list of mediums supported by the printer.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	The printer_name specifies the name of the printer obtained
 *	from iv_print_xprint_get_printers_list(). This value may
 *	not be NULL.
 *
 *	Returns a glist of iv_print_xprint_medium_struct describing
 *	each medium. It must be deleted by the calling
 *	iv_print_xprint_mediums_list_delete().
 */
GList *iv_print_xprint_get_printer_mediums(
	const gchar *xp_address,
	const gchar *printer_name
)
{
	int xp_event_base, xp_error_base;
	gint level;
	char *xp_mediums_list;
	const gchar *s;
	gchar *tray_name;
	Display *dpy;
	XPContext xp_ctx;
	GList *mediums_list;
	iv_print_xprint_medium_struct *m;

	if(printer_name == NULL)
	    return(NULL);

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(NULL);

	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Get the list of mediums supported by the specified printer
	 * from the X Print server
	 */
	xp_mediums_list = XpGetOneAttribute(
	    dpy, xp_ctx,
	    XPPrinterAttr,
	    "medium-source-sizes-supported"
	);
	if(xp_mediums_list == NULL)
	{
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Parse the mediums list */
	mediums_list = NULL;
	m = NULL;
	tray_name = NULL;
	level = 0;
	s = (const gchar *)xp_mediums_list;
	while(*s != '\0')
	{
	    if(*s == '{')
	    {
		level++;
		s++;

		/* Tray level */
		if(level == 1)
		{
		    const gchar *s_end;

		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* Tray */
		    g_free(tray_name);
		    tray_name = get_string(
			s, &s_end, " \t{}"
		    );
		    s = s_end;

		    while((*s == ' ') || (*s == '\t'))
			s++;
		    if((*s == '{') || (*s == '}'))
			continue;
		}
		/* Medium level */
		else if(level == 2)
		{
		    const gchar *s_end;

		    /* Create a new medium and append it to the list */
		    m = iv_print_xprint_medium_new();
		    m->tray = STRDUP(tray_name);
		    mediums_list = g_list_append(mediums_list, m);

		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* Name */
		    g_free(m->name);
		    m->name = get_string(
			s, &s_end, " \t{}"
		    );
		    s = s_end;

		    while((*s == ' ') || (*s == '\t'))
			s++;
		    if((*s == '{') || (*s == '}'))
			continue;

		    /* True/False */
		    s_end = s;
		    while((*s_end != '\0') &&
		          (*s_end != ' ') && (*s_end != '\t') &&
		          (*s_end != '{') && (*s_end != '}')
		    )
			s_end++;



		    s = s_end;
		    while((*s == ' ') || (*s == '\t'))
			s++;
		    if((*s == '{') || (*s == '}'))
			continue;
		}
		/* Medium geometry level */
		else if(level == 3)
		{
		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* X */
		    if(m != NULL)
			m->x_mm = atof(s);
		    while((*s != '\0') &&
		          (*s != ' ') && (*s != '\t') &&
		          (*s != '{') && (*s != '}')
		    )
			s++;

		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* Width */
		    if(m != NULL)
			m->width_mm = atof(s);
		    while((*s != '\0') &&
		          (*s != ' ') && (*s != '\t') &&
		          (*s != '{') && (*s != '}')
		    )
			s++;

		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* Y */
		    if(m != NULL)
			m->y_mm = atof(s);
		    while((*s != '\0') &&
		          (*s != ' ') && (*s != '\t') &&
		          (*s != '{') && (*s != '}')
		    )
			s++;

		    while((*s == ' ') || (*s == '\t'))
			s++;

		    /* Height */
		    if(m != NULL)
			m->height_mm = atof(s);
		    while((*s != '\0') &&
		          (*s != ' ') && (*s != '\t') &&
		          (*s != '{') && (*s != '}')
		    )
			s++;
		}
	    }
	    else if(*s == '}')
	    {
		level--;
		s++;
	    }
	    else
	    {
		s++;
	    }
	}

	g_free(tray_name);
	XFree(xp_mediums_list);
	XpDestroyContext(dpy, xp_ctx);
	iv_print_xprint_disconnect(dpy);

	return(mediums_list);
}

/*
 *	Gets the list of orientations supported by the printer.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	The printer_name specifies the name of the printer obtained
 *	from iv_print_xprint_get_printers_list(). This value may
 *	not be NULL.
 *
 *	Returns a glist of strings describing each orientation by name.
 */
GList *iv_print_xprint_get_printer_orientations(
	const gchar *xp_address,
	const gchar *printer_name
)
{
	int xp_event_base, xp_error_base;
	gint level;
	char *xp_orientations_list;
	const gchar *s;
	Display *dpy;
	XPContext xp_ctx;
	GList *orientations_list;

	if(printer_name == NULL)
	    return(NULL);

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(NULL);

	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Get the list of orientations supported by the specified
	 * printer from the X Print server
	 */
	xp_orientations_list = XpGetOneAttribute(
	    dpy, xp_ctx,
	    XPPrinterAttr,
	    "content-orientations-supported"
	);
	if(xp_orientations_list == NULL)
	{
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Parse the orientations list */
	orientations_list = NULL;
	level = 0;
	s = (const gchar *)xp_orientations_list;
	while(*s != '\0')
	{
	    while((*s == ' ') || (*s == '\t'))
		s++;

	    if(*s != '\0')
	    {
		const gchar *s_end;
		gchar *name = get_string(
		    s, &s_end, " \t{}"
		);
		s = s_end;
		orientations_list = g_list_append(
		    orientations_list, name
		);
	    }
	}

	XFree(xp_orientations_list);
	XpDestroyContext(dpy, xp_ctx);
	iv_print_xprint_disconnect(dpy);

	return(orientations_list);
}

/*
 *	Gets the list of resolutions supported by the printer.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	The printer_name specifies the name of the printer obtained
 *	from iv_print_xprint_get_printers_list(). This value may
 *	not be NULL.
 *
 *	Returns a glist of gints describing each resolution by dpi.
 */
GList *iv_print_xprint_get_printer_resolutions(
	const gchar *xp_address,
	const gchar *printer_name
)
{
	int xp_event_base, xp_error_base;
	char *xp_resolutions_list;
	gchar **strv;
	Display *dpy;
	XPContext xp_ctx;
	GList *resolutions_list;

	if(printer_name == NULL)
	    return(NULL);

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(NULL);

	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Get the list of resolutions supported by the specified
	 * printer from the X Print server
	 */
	xp_resolutions_list = XpGetOneAttribute(
	    dpy, xp_ctx,
	    XPPrinterAttr,
	    "printer-resolutions-supported"
	);
	if(xp_resolutions_list == NULL)
	{
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(NULL);
	}

	/* Parse the resolutions list */
	resolutions_list = NULL;
	strv = g_strsplit(
	    (const gchar *)xp_resolutions_list,
	    " ",
	    -1
	);
	if(strv != NULL)
	{
	    gint i = 0;
	    while(strv[i] != NULL)
	    {
		resolutions_list = g_list_append(
		    resolutions_list,
		    (gpointer)ATOI(strv[i])
		);
		i++;
	    }
	    g_strfreev(strv);
	}

	XFree(xp_resolutions_list);
	XpDestroyContext(dpy, xp_ctx);
	iv_print_xprint_disconnect(dpy);

	return(resolutions_list);
}

/*
 *	Gets the size of the medium.
 *
 *	The xp_address specifies the address to the X Print server. If
 *	xp_address is NULL then the address specified by the environment
 *	variable "XPSERVERLIST" will be used instead.
 *
 *	The printer_name specifies the name of the printer obtained
 *	from iv_print_xprint_get_printers_list(). This value may
 *	not be NULL.
 *
 *	The tray_name specifies the name of the tray obtained
 *	from iv_print_xprint_get_trays_list(). If tray_name is NULL
 *	then the tray will not be specified when obtaining the medium
 *	size from the X Print server.
 *
 *	The medium_name specifies the name of the medium obtained from
 *	iv_print_xprint_get_mediums(). If medium_name is NULL then the
 *	medium will not be specified when obtaining the medium size from
 *	the X Print server.
 *
 *	The orientation_name specifies the name of the orientation
 *	obtained from iv_print_xprint_get_orientations(). If
 *	orientation_name is NULL then the orientation will not be
 *	specified when obtaining the medium size from the X Print
 *	server.
 *
 *	The dpi specifies the dots per inch resolution obtained from
 *	iv_print_xprint_get_resolutions(). If dpi is 0 or negative
 *	then the resolution will not be specified when obtaining the
 *	medium size from teh X Print server.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint iv_print_xprint_get_medium_size(
	const gchar *xp_address,
	const gchar *printer_name,
	const gchar *tray_name,
	const gchar *medium_name,
	const gchar *orientation_name,
	const gint dpi,
	gint *width_rtn, gint *height_rtn
)
{
	int xp_event_base, xp_error_base;
	Display *dpy;
	unsigned short width, height;
	XRectangle rect;
	XPContext xp_ctx;

	if(width_rtn != NULL)
	    *width_rtn = 0;
	if(height_rtn != NULL)
	    *height_rtn = 0;

	if((printer_name == NULL) || (medium_name == NULL) ||
	   (orientation_name == NULL)
	)
	    return(-2);

	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xp_address
	))
	    return(-1);

	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)
	{
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

#if 0
	/* Set the resolution for XPutImage() requests */
	XpSetImageResolution(dpy, xp_ctx, dpi, &resulting_dpi);
#endif

	/* Set the tray */
	if(!STRISEMPTY(tray_name))
	{
	    gchar *s = g_strconcat(
		"*default-input-tray: ",
		tray_name,
		NULL
	    );
	    XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
	    g_free(s);
	}

	/* Set the medium */
	if(!STRISEMPTY(medium_name))
	{
	    gchar *s = g_strconcat(
		"*default-medium: ",
		medium_name,
		NULL
	    );
	    XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
	    g_free(s);
	}

	/* Set the orientation */
	if(!STRISEMPTY(orientation_name))
	{
	    gchar *s = g_strconcat(
		"*content-orientation: ",
		orientation_name,
		NULL
	    );
	    XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
	    g_free(s);
	}

	/* Set the resolution */
	if(dpi > 0)
	{
	    gchar *s = g_strdup_printf(
		"*default-printer-resolution: %i",
		dpi
	    );
	    XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
	    g_free(s);
	}

	/* Get the size of the medium */
	if(XpGetPageDimensions(
	    dpy, xp_ctx,
	    &width, &height,
	    &rect
	))
	{
	    if(width_rtn != NULL)
		*width_rtn = (gint)rect.width;
	    if(height_rtn != NULL)
		*height_rtn = (gint)rect.height;

	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);

	    return(0);
	}
	else
	{
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);

	    return(-1);
	}
}


/*
 *	X error callback.
 */
static int iv_print_x_error_cb(Display *dpy, XErrorEvent *error)
{
	char buf[1024];

	if(error == NULL)
	    return(0);

	*buf = '\0';
	XGetErrorText(dpy, error->error_code, buf, sizeof(buf));

	g_printerr(
"X Error: %i  Code: %i (%s)\n",
	    error->type,
	    error->error_code,
	    buf
	);

	return(0);
}

/*
 *	Progress callback.
 */
static gint iv_print_xprint_progress_cb(
	const gulong i, const gulong m, gpointer data
)
{
	if(m > 0l)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		(gdouble)i / (gdouble)m,
#ifdef PROGRESS_BAR_DEF_TICKS
		PROGRESS_BAR_DEF_TICKS,
#else
		25,
#endif
		TRUE
	    );
	}
	else
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	}
	return(ProgressDialogStopCount());
}


/*
 *	Creates an X Image from the image with resizing and color
 *	modifications applied.
 *
 *	The visual specifies the X Visual which must be of class
 *	TrueColor.
 *
 *	The img specifies the image.
 *
 *	The pv specifies the print values which will be used to resize
 *	and apply color modifications.
 *
 *	The bg_color specifies the background color. It must contain
 *	four bytes describing the RGBA values.
 *
 *	Returns the X Image.
 */
static XImage *iv_print_xprint_render_ximage(
	Display *dpy, Visual *visual, const int depth,
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	gint (*progress_cb)(const gulong, const gulong, gpointer),
	gpointer progress_data
)
{
	XImage *ximg;
	imgview_frame_struct *frame;

	/* Render the output image */
	imgview_image_struct *output_img = PrintRenderImageFromValues(
	    img, pv, bg_color,
	    progress_cb, progress_data
	);
	if(output_img == NULL)
	    return(NULL);

	/* The output image must be 3 bpp or greater in RGB* format */
	if(output_img->bpp < 3)
	{
	    ImgViewImageDelete(output_img);
	    return(NULL);
	}

	frame = ImgViewImageGetFrame(output_img, 0);
	if(frame == NULL)
	{
	    ImgViewImageDelete(output_img);
	    return(NULL);
	}                

	/* Create the X Image */
	ximg = XCreateImage(
	    dpy, visual, depth,
	    ZPixmap,
	    0,					/* X pixel offset */
	    NULL,
	    output_img->width, output_img->height,
	    BitmapPad(dpy),
	    0
	);
	if(ximg == NULL)
	{
	    ImgViewImageDelete(output_img);
	    return(NULL);
	}

	/* Allocate the X Image's data as needed */
	if((ximg->data == NULL) && (ximg->bytes_per_line > 0) &&
	   (ximg->height > 0)
	)
	    ximg->data = (char *)g_malloc(
		ximg->height * ximg->bytes_per_line
	    );
	if(ximg->data == NULL)
	{
	    XDestroyImage(ximg);
	    ImgViewImageDelete(output_img);
	    return(NULL);
	}

	/* Copy/convert the rendered image's data to the X Image */
	if(frame->buf != NULL)
	{
	    gint y;
	    const gint	width = MIN(ximg->width, output_img->width),
			height = MIN(ximg->height, output_img->height),
			output_img_bpp = output_img->bpp,
			output_img_bpl = output_img->bpl,
			ximg_bpp = ximg->bits_per_pixel / 8,
			ximg_bpl = ximg->bytes_per_line;
	    gulong	ximg_red_mask = ximg->red_mask,
			ximg_green_mask = ximg->green_mask,
			ximg_blue_mask = ximg->blue_mask;
	    const guint8 *src_data = frame->buf, *src_ptr, *src_end;
	    guint8 *tar_data = (guint8 *)ximg->data, *tar_ptr;
	    void (*pix_func)(guint8 *, const guint8 *);

	    if((ximg_red_mask   == 0xff0000) &&
	       (ximg_green_mask == 0x00ff00) &&
	       (ximg_blue_mask  == 0x0000ff)
	    )
	    {
		if(ximg_bpp == 4)
		    pix_func = pixel_set_rgb_to_bgra;
		else
		    pix_func = pixel_set_rgb_to_bgr;
	    }
	    else if((ximg_red_mask   == 0x0000ff) &&
                    (ximg_green_mask == 0x00ff00) &&
                    (ximg_blue_mask  == 0xff0000)
            )
	    {
		if(ximg_bpp == 4)
		    pix_func = pixel_set_rgb_to_rgba;
		else
		    pix_func = pixel_set_rgb_to_rgb;
	    }
	    else
	    {
		if(ximg_bpp >= 2)
		    pix_func = pixel_set_rgb_to_grey_alpha;
		else
		    pix_func = pixel_set_rgb_to_grey;
	    }

	    for(y = 0; y < height; y++)
	    {
		src_ptr = src_data + (y * output_img_bpl);
		src_end = src_ptr + (width * output_img_bpp);
		tar_ptr = tar_data + (y * ximg_bpl);
		while(src_ptr < src_end)
		{
		    pix_func(tar_ptr, src_ptr);
		    src_ptr += output_img_bpp;
		    tar_ptr += ximg_bpp;
		}
#if 0
/* Do not report progress for this */
		if(progress_cb != NULL)
		{
		    if(progress_cb(
			(gulong)y + 1,
			(gulong)height,
			progress_data
		    ))
			break;
		}
#endif
	    }
	}

	/* Delete the rendered image */
	ImgViewImageDelete(output_img);

	return(ximg);
}

/*
 *	Print operation iteration.
 *
 *	Performs one iteration of the print operation by printing the
 *	page with the specified image. The X Window to print to,
 *	X Graphic Context, and the X Print job must already be
 *	created prior to calling this function.
 *
 *	The w specifies the X Window to render the image to.
 *
 *	The gc specifies the X Graphic Context to use to render the
 *	image and to put the image to the X Window.
 *
 *	The img specifies the image to print.
 *
 *	The pv specifies the print values.
 *
 *	The bg_color specifies the background color. It must contain
 *	four bytes describing the RGBA values.
 */
static gint iv_print_xprint_main_iteration(
	Display *dpy, Visual *visual, const int depth,
	Window w, GC gc,
	XPContext xp_ctx,
	int xp_event_base, int xp_error_base,
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	const gboolean verbose, const gboolean interactive,
	GtkWidget *toplevel
)
{
	gboolean waiting;
	XImage *ximg;
	XEvent event;
	const print_value_flags pvflags = pv->flags;
	const gint	output_x = (pvflags & PRINT_VALUE_OUTPUT_X) ?
		pv->output_x : 0,
			output_y = (pvflags & PRINT_VALUE_OUTPUT_Y) ?
		pv->output_y : 0;

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Starting a new page...",
		NULL, NULL,
		TRUE
	    );

	/* Start the page */
	XpStartPage(dpy, w);

	/* Wait for the page to start */
	waiting = True;
	while(waiting)
	{
	    XNextEvent(dpy, &event);
	    if(event.type == (xp_event_base + XPPrintNotify))
	    {
		XPPrintEvent *pev = (XPPrintEvent *)&event;
		switch(pev->detail)
		{
		  case XPStartPageNotify:
		    waiting = False;
		    break;
		}
	    }
	    if(verbose)
		ProgressDialogUpdateUnknown(
		    NULL, NULL, NULL, NULL,
		    TRUE
		);
	}

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Rendering the image for printing...",
		NULL, NULL,
		TRUE
	    );

	/* Render the X Image */
	ximg = iv_print_xprint_render_ximage(
	    dpy, visual, depth,
	    img,
	    pv,
	    bg_color,
	    verbose ? iv_print_xprint_progress_cb : NULL,
	    NULL
	);
	if(ximg == NULL)
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to render the image for printing.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    XpCancelPage(dpy, xp_ctx);
	    return(-1);
	}

	/* Update progress message and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL,
		"Sending the rendered image to the page...",
		NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelPage(dpy, xp_ctx);
		XDestroyImage(ximg);
		return(-4);
	    }
	}

	/* Set the resolution for XPutImage() operations */
	if(pvflags & PRINT_VALUE_RESOLUTION)
	{
	    if(pv->resolution > 0)
	    {
#if 0
/* Not needed since the image is already scaled */
		int prev_resolution;
		XpSetImageResolution(
		    dpy, xp_ctx,
		    pv->resolution,
		    &prev_resolution
		);
#endif
	    }
	}

	/* Put the X Image to the X Window */
	XPutImage(
	    dpy, (Drawable)w, gc,
	    ximg,
	    0, 0,
	    (int)output_x, (int)output_y,
	    ximg->width, ximg->height
	);

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelPage(dpy, xp_ctx);
		XDestroyImage(ximg);
		return(-4);
	    }
	}

	/* End the page */
	XpEndPage(dpy);

	/* Wait for the page to end */
	waiting = True;
	while(waiting)
	{
	    XNextEvent(dpy, &event);
	    if(event.type == (xp_event_base + XPPrintNotify))
	    {
		XPPrintEvent *pev = (XPPrintEvent *)&event;
		switch(pev->detail)
		{
		  case XPEndPageNotify:
		    waiting = False;
		    break;
		}
	    }
	    if(verbose)
		ProgressDialogUpdateUnknown(
		    NULL, NULL, NULL, NULL,
		    TRUE
		);
	}

	/* Delete the X Image */
	XDestroyImage(ximg);

	return(0);
}

/*
 *	Performs the print operation.
 */
static gint iv_print_xprint_main(
	imgview_image_struct *img,
	const print_values_struct *pv,
	const guint8 *bg_color,
	const gboolean verbose, const gboolean interactive,
	GtkWidget *toplevel
)
{
	unsigned short page_width, page_height;
	int visual_class, depth, xp_event_base, xp_error_base;
	XRectangle page_rect;
	Colormap cmap;
	Screen *screen_ptr;
	Visual *visual;
	Window w, root;
	Display *dpy;
	XSetWindowAttributes w_set_attr;
	GC gc;
	XPContext xp_ctx;
	gboolean waiting;
	gint status;
	const print_value_flags pvflags = pv->flags;
	const gchar *xprint_server_address = (pvflags & PRINT_VALUE_XPRINT_SERVER_ADDRESS) ?
	    pv->xprint_server_address : NULL;
	const gchar *printer_name = (pvflags & PRINT_VALUE_PRINTER) ?
	    pv->printer : NULL;

	/* Printer not specified? */
	if(STRISEMPTY(printer_name))
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    "Printer not specified.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    return(-2);
	}

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Connecting to the X Print Server...",
		NULL, NULL,
		TRUE
	    );

	/* Connect to the X Print server */
	if(iv_print_xprint_connect(
	    &dpy,
	    &xp_event_base, &xp_error_base,
	    xprint_server_address
	))
	{
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
"Unable to connect to the X Print Server at address:\n\
\n\
    %s",
		    xprint_server_address
		);
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
		    msg,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
		g_free(msg);
	    }
	    return(-1);
	}

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		iv_print_xprint_disconnect(dpy);
		return(-4);
	    }
	}

	/* Create an X Print context for the specified printer */
	xp_ctx = XpCreateContext(dpy, (char *)printer_name);
	if(xp_ctx == None)                                  
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to create a new X Print context.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Setting the print parameters...",
		NULL, NULL,
		TRUE
	    );

#define REPORT_PROGRESS_CHECK_ABORT(_verbose_)	{	\
 if(_verbose_) {					\
  ProgressDialogUpdateUnknown(				\
   NULL, NULL, NULL, NULL, TRUE				\
  );							\
  if(ProgressDialogStopCount() > 0) {			\
   XpDestroyContext(dpy, xp_ctx);			\
   iv_print_xprint_disconnect(dpy);			\
   return(-4);						\
  }							\
 }							\
}

	/* Set the tray */
	if(pvflags & PRINT_VALUE_INPUT_TRAY)
	{
	    if(!STRISEMPTY(pv->input_tray))
	    {
		gchar *s = g_strconcat(
		    "*default-input-tray: ",
		    pv->input_tray,
		    NULL
		);
		XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
		g_free(s);
	    }
	}

	REPORT_PROGRESS_CHECK_ABORT(verbose);

	/* Set the medium */
	if(pvflags & PRINT_VALUE_MEDIUM)
	{
	    if(!STRISEMPTY(pv->medium))
	    {
		gchar *s = g_strconcat(
		    "*default-medium: ",
		    pv->medium,
		    NULL
		);
		XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
		g_free(s);
	    }
	}

	REPORT_PROGRESS_CHECK_ABORT(verbose);

	/* Set the orientation */
	if(pvflags & PRINT_VALUE_ORIENTATION)
	{
	    if(!STRISEMPTY(pv->orientation))
	    {
		gchar *s = g_strconcat(
		    "*content-orientation: ",
		    pv->orientation,
		    NULL
		);
		XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
		g_free(s);
	    }
	}

	REPORT_PROGRESS_CHECK_ABORT(verbose);

	/* Set the resolution */
	if(pvflags & PRINT_VALUE_RESOLUTION)
	{
	    if(pv->resolution > 0)
	    {
		gchar *s = g_strdup_printf(
		    "*default-printer-resolution: %i",
		    pv->resolution
		);
		XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
		g_free(s);
	    }
	}

	REPORT_PROGRESS_CHECK_ABORT(verbose);

	/* Set the number of coppies */
	if(pvflags & PRINT_VALUE_OUTPUT_NCOPPIES)
	{
	    if(pv->output_ncoppies > 0)
	    {
		gchar *s = g_strdup_printf(
		    "*copy-count: %i",
		    pv->output_ncoppies
		);
		XpSetAttributes(dpy, xp_ctx, XPDocAttr, s, XPAttrMerge);
		g_free(s);
	    }
	}

	REPORT_PROGRESS_CHECK_ABORT(verbose);
#undef REPORT_PROGRESS_CHECK_ABORT

	/* Monitor the printing procedure since the process is async */
	XpSelectInput(dpy, xp_ctx, XPPrintMask);

	/* Set the X print context */
	XpSetContext(dpy, xp_ctx);

	/* Get the screen associated with this X print context */
	screen_ptr = XpGetScreenOfContext(dpy, xp_ctx);
	root = RootWindowOfScreen(screen_ptr);

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Starting a new print job...",
		NULL, NULL,
		TRUE
	    );

	/* Start the print job in spool mode */
	XpStartJob(dpy, XPSpool);

	/* Wait for start job */
	waiting = True;
	while(waiting)
	{
	    XEvent event;
	    XNextEvent(dpy, &event);
	    if(event.type == (xp_event_base + XPPrintNotify))
	    {
		XPPrintEvent *pev = (XPPrintEvent *)&event;
		switch(pev->detail)
		{
		  case XPStartJobNotify:
		    waiting = False;
		    break;
		}
	    }    
	    if(verbose)
		ProgressDialogUpdateUnknown(
		    NULL, NULL, NULL, NULL,
		    TRUE
		);
	}

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelJob(dpy, xp_ctx);
		XpDestroyContext(dpy, xp_ctx);
		iv_print_xprint_disconnect(dpy);
		return(-4);
	    }
	}

	/* Get the page dimensions in pixels */
	if(!XpGetPageDimensions(  
	    dpy, xp_ctx,
	    &page_width, &page_height,
	    &page_rect
	))
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to get the page dimensions from the X Print Server.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    XpCancelJob(dpy, xp_ctx);
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

	/* Get the best X Visual for printing */
	visual_class = TrueColor;
	if(pvflags & PRINT_VALUE_VISUAL)
	{
	    switch(pv->visual)
	    {
	      case PRINT_VISUAL_RGB:
		visual_class = TrueColor;
		break;
	      case PRINT_VISUAL_GREYSCALE:
		visual_class = GrayScale;
		break;
	      case PRINT_VISUAL_BW:
		visual_class = GrayScale;
		break;
	    }
	}
	visual = iv_print_xprint_get_best_visual(
	    dpy,
	    visual_class,		/* X Visual class */
	    &depth
	);
	if(visual == NULL)
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to obtain a suitable X Visual on the X Print\n\
Server for printing.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    XpCancelJob(dpy, xp_ctx);
	    XpDestroyContext(dpy, xp_ctx);
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Creating the job resources...",
		NULL, NULL,
		TRUE
	    );

	/* Create the colormap */
	cmap = XCreateColormap(dpy, root, visual, AllocNone);

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelJob(dpy, xp_ctx);
		XpDestroyContext(dpy, xp_ctx);
		XFreeColormap(dpy, cmap);
		iv_print_xprint_disconnect(dpy);
		return(-4);
	    }
	}

	/* Create the window to render the image to */
	w_set_attr.background_pixel = iv_print_xprint_get_pixel_rgb(
	    dpy, cmap,
	    0xff, 0xff, 0xff
	);
	w_set_attr.border_pixel = w_set_attr.background_pixel;
	w_set_attr.colormap = cmap;
	w = XCreateWindow(
	    dpy, root,
	    page_rect.x, page_rect.y,
	    page_rect.width, page_rect.height, 0,
	    depth,
	    CopyFromParent,		/* IO Type */
	    visual,
	    CWBackPixel | CWBorderPixel | CWColormap,
	    &w_set_attr
	);
	if(w == None)
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to create a new X Window for rendering the\n\
image to on the X Print Server.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    XpCancelJob(dpy, xp_ctx);
	    XpDestroyContext(dpy, xp_ctx);
	    XFreeColormap(dpy, cmap);
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelJob(dpy, xp_ctx);
		XpDestroyContext(dpy, xp_ctx);
		XDestroyWindow(dpy, w);
		XFreeColormap(dpy, cmap);
		iv_print_xprint_disconnect(dpy);
		return(-4);
	    }
	}

	/* Create the graphic context */
	gc = XCreateGC(dpy, (Drawable)w, 0l, NULL);
	if(gc == None)
	{
	    if(verbose)
	    {
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Print Failed",
"Unable to create a new X Graphic Context for\n\
rendering the image with on the X Print Server.",
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		CDialogSetTransientFor(NULL);
	    }
	    XpCancelJob(dpy, xp_ctx);
	    XpDestroyContext(dpy, xp_ctx);
	    XDestroyWindow(dpy, w);
	    XFreeColormap(dpy, cmap);
	    iv_print_xprint_disconnect(dpy);
	    return(-1);
	}

	/* Report progress and check for abort */
	if(verbose)
	{
	    ProgressDialogUpdateUnknown(
		NULL, NULL, NULL, NULL,
		TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
	    {
		XpCancelJob(dpy, xp_ctx);
		XpDestroyContext(dpy, xp_ctx);
		XFreeGC(dpy, gc);
		XDestroyWindow(dpy, w);
		XFreeColormap(dpy, cmap);
		iv_print_xprint_disconnect(dpy);
		return(-4);
	    }
	}

	/* Render the image and print the page */
	status = iv_print_xprint_main_iteration(
	    dpy, visual, depth,
	    w, gc,
	    xp_ctx, xp_event_base, xp_error_base,
	    img,
	    pv,
	    bg_color,
	    verbose, interactive, toplevel
	);
	if(status != 0)
	{
	    XpCancelJob(dpy, xp_ctx);
	    XpDestroyContext(dpy, xp_ctx);
	    XFreeGC(dpy, gc);
	    XDestroyWindow(dpy, w);
	    XFreeColormap(dpy, cmap);
	    iv_print_xprint_disconnect(dpy);
	    return(status);
	}

	/* Update the progress dialog message */
	if(verbose)
	    ProgressDialogUpdateUnknown(
		NULL,
		"Sending the print job to the printer...",
		NULL, NULL,
		TRUE
	    );

	/* End the print job */
	XpEndJob(dpy);

	/* Wait for the print job to end */
	waiting = True;
	while(waiting)
	{
	    XEvent event;
	    XNextEvent(dpy, &event);
	    if(event.type == (xp_event_base + XPPrintNotify))
	    {
		XPPrintEvent *pev = (XPPrintEvent *)&event;
		switch(pev->detail)
		{
		  case XPEndJobNotify:
		    waiting = False;
		    break;
		}
	    }
	    if(verbose)
		ProgressDialogUpdateUnknown(
		    NULL, NULL, NULL, NULL,
		    TRUE
		);
	}

	/* Delete the X Print context and disconnect from the X Print
	 * server
	 */
	XpDestroyContext(dpy, xp_ctx);
	XFreeGC(dpy, gc);
	XDestroyWindow(dpy, w);
	XFreeColormap(dpy, cmap);
	iv_print_xprint_disconnect(dpy);

	return(status);
}

/*
 *	Prints the image.
 *
 *	The img specifies the image to print.
 *
 *	The pv specifies the print values that will be used as the
 *	parameters for printing the image.
 *
 *	The bg_color specifies the background color. It must contain
 *	four bytes describing the RGBA values.
 *
 *	The toplevel specifies the reference toplevel GtkWindow.
 *
 *      Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error.
 *	-4	User aborted.
 */
#ifdef HAVE_LIBENDEAVOUR2
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel,
	edv_context_struct *edv2_ctx
)
#else
gint iv_print(
	imgview_image_struct *img,		/* Image to print */
	const print_values_struct *pv,		/* Print values */
	const guint8 *bg_color,			/* 4 bytes RGBA */
	GtkWidget *toplevel 
)
#endif
{
	gint status;

	if((img == NULL) || (pv == NULL))
	    return(-2);

	if(ProgressDialogIsQuery())
	    ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(toplevel);
        ProgressDialogMap(
            "Printing",
            "Printing image...",
            (const guint8 **)icon_print2_32x32_xpm,
            "Stop"
        );
	gdk_flush();

	status = iv_print_xprint_main(
	    img,
	    pv,
	    bg_color,
	    TRUE,			/* Verbose */
	    TRUE,			/* Interactive */
	    toplevel
	);

	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	return(status);
}


#endif	/* HAVE_XPRINT */
