#include <stdlib.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "savedlg.h"
#include "config.h"

#include "images/icon_save_20x20.xpm"
#include "images/icon_save_32x32.xpm"
#include "images/icon_save_48x48.xpm"
#include "images/icon_cancel_20x20.xpm"


static gint delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data);
static void save_btn_cb(GtkWidget *widget, gpointer data);
static void cancel_btn_cb(GtkWidget *widget, gpointer data);

static void scale_changed_cb(GtkAdjustment *adjustment, gpointer data);
static void entry_changed_adj_cb(GtkEntry *entry, gpointer data);

static void toggle_glist_widget_set_sensitive_cb(
	GtkWidget *widget, gpointer data
);
static void toggle_glist_widget_set_insensitive_cb(
	GtkWidget *widget, gpointer data
);

static void build_std_dialog(
	gint width, gint height, const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn,
	GtkAccelGroup **accelgrp_rtn
);


gint iv_query_gif_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *interlaced,
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Color */
	gboolean *transparency,
	gboolean *looping
);
gint iv_query_jpeg_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gfloat *quality,	/* Quality value from 0.0 to 1.0 */
	gboolean *save_as_color
);
gint iv_query_png_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gint *compression,	/* Value from 0 to 9 */
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB
				 * 4 = RGBA */
	gboolean *interlaced
);
gint iv_query_tga_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gint *format		/* 0 = Greyscale
				 * 1 = RGB
				 * 2 = RGBA */
);
gint iv_query_xpm_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gchar **c_id,
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Color */ 
	gint *max_colors,	/* -1 for no limit */
	guint8 *threshold	/* 0x00 to 0xff */
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define ABSOLUTE(x)     (((x) < 0) ? ((x) * -1) : (x))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/* Tooltip messages */
#define SAVE_OPTS_TTM_JPEG_QUALITY	"Select the percentage of data\
 retained, higher percentages improve the quality and reduce the amount\
 of data discarded but increases the file size"

#define SAVE_OPTS_TTM_PNG_COMPRESSION	"Select the level of\
 compression, the optimal compression level is 6 and the maximum\
 compression level is 9. PNG compression is loss-less so it never\
 discards data"
#define SAVE_OPTS_TTM_PNG_ALPHA		"Check this to save the alpha\
 channel, the alpha channel is used to store PNG transparency and\
 blending information"
#define SAVE_OPTS_TTM_PNG_INTERLACED	"Check this to save the image\
 using Adam 7 interlacing (no data will be discarded)"

#define SAVE_OPTS_TTM_XPM_C_ID		"Enter the C identifier for\
 this XPM image, the C identifier is used to identify this XPM image\
 when it is used in C source code"
#define SAVE_OPTS_TTM_XPM_MAX_COLORS	"Enter the maximum number of\
 colors that will be generated for this XPM image's colormap"
#define SAVE_OPTS_TTM_XPM_NO_MAX_COLORS	"Check this to generate as\
 many colors as needed in order to preserve all the colors from the\
 original image"
#define SAVE_OPTS_TTM_XPM_THRESHOLD	"Set the alpha threshold,\
 the alpha threshold determines which colors will be solid and\
 which colors will be transparent. Colors with an alpha value less\
 than the threshold will be completely transparent"


static gboolean		got_response;


/*
 *	Closes the given toplevel window given by data and breaks
 *	the highest gtk_main level.
 */
static gint delete_event_cb(GtkWidget *widget, GdkEvent *event, gpointer data)
{
        got_response = FALSE;
        gtk_main_quit();
	return(TRUE);
}

/*
 *	Save button callback.
 */
static void save_btn_cb(GtkWidget *widget, gpointer data)
{
	got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Cancel button callback.
 */
static void cancel_btn_cb(GtkWidget *widget, gpointer data)
{
	got_response = FALSE;
	gtk_main_quit();
}


/*
 *	Scale value changed.
 */
static void scale_changed_cb(GtkAdjustment *adjustment, gpointer data)
{
	GtkAdjustment *adj = adjustment;
	GtkWidget *w = (GtkWidget *)data;
	if((adj == NULL) || (w == NULL))
	    return;

	if(GTK_IS_ENTRY(w))
	{
	    gchar *s, num_str[80];
	    GtkEntry *entry = GTK_ENTRY(w);
	    g_snprintf(num_str, sizeof(num_str), "%.0f", adj->value);

	    s = gtk_entry_get_text(entry);
	    if((s != NULL) ? strcmp(s, num_str) : TRUE)
		gtk_entry_set_text(entry, num_str);
	}
}

/*
 *	Entry value changed with adjustment as data callback.
 */
void entry_changed_adj_cb(GtkEntry *entry, gpointer data)
{
	const gchar *s;
	GtkAdjustment *adj = (GtkAdjustment *)data;
	if((entry == NULL) || (adj == NULL))
	    return;

	s = gtk_entry_get_text(entry);
	if(!STRISEMPTY(s))
	    gtk_adjustment_set_value(adj, (gfloat)ATOF(s));
}


/*
 *	Sets all widgets in the list sensitive if toggled.
 */
static void toggle_glist_widget_set_sensitive_cb(
	GtkWidget *widget, gpointer data  
)
{
	gboolean sensitive;
	GList *glist = (GList *)data;
	if((widget == NULL) || (glist == NULL))
	    return;

	sensitive = GTK_TOGGLE_BUTTON_GET_ACTIVE(widget);
	while(glist != NULL)
	{
	    GTK_WIDGET_SET_SENSITIVE(
		(GtkWidget *)glist->data, sensitive
	    );
	    glist = g_list_next(glist);
	}
}

/*
 *	Sets all widgets in the list insensitive if toggled.
 */
static void toggle_glist_widget_set_insensitive_cb(
	GtkWidget *widget, gpointer data
)
{
	gboolean sensitive;
	GList *glist = (GList *)data;
	if((widget == NULL) || (glist == NULL))
	    return;

	sensitive = !GTK_TOGGLE_BUTTON_GET_ACTIVE(widget);
	while(glist != NULL)
	{
	    GTK_WIDGET_SET_SENSITIVE(
		(GtkWidget *)glist->data, sensitive
	    );
	    glist = g_list_next(glist);
	}
}


/*
 *	Builds a standard toplevel dialog and returns the pointers to
 *	the created widgets.
 */
static void build_std_dialog(
	gint width, gint height, const gchar *title,
	GtkWidget *ref_toplevel,
	GtkWidget **toplevel_rtn, GtkWidget **client_vbox_rtn,
	GtkAccelGroup **accelgrp_rtn
)
{
	gint	bw = GUI_BUTTON_HLABEL_WIDTH_DEF,
		bh = GUI_BUTTON_HLABEL_HEIGHT_DEF,
		border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;


	/* Create keyboard accelerator group */
	accelgrp = gtk_accel_group_new();
	if(accelgrp_rtn != NULL)
	    (*accelgrp_rtn) = accelgrp;

	/* Create toplevel */
	toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_widget_set_usize(w, width, height);
	gtk_window_set_title(GTK_WINDOW(w), title);
#ifdef PROG_NAME
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
#endif
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_save_48x48_xpm);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(delete_event_cb), toplevel
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	if((ref_toplevel != NULL) ? GTK_IS_WINDOW(GTK_OBJECT(ref_toplevel)) : 0)
	{
	    gtk_window_set_modal(GTK_WINDOW(toplevel), TRUE);
	    gtk_window_set_transient_for(
		GTK_WINDOW(toplevel), GTK_WINDOW(ref_toplevel)
	    );
	}
	if(toplevel_rtn != NULL)
	    *toplevel_rtn = toplevel;

	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;
	if(client_vbox_rtn != NULL)
	    *client_vbox_rtn = w;

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Save button */
	w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_save_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
	    "Salve",
#elif defined(PROG_LANGUAGE_FRENCH)
            "Epargner",
#elif defined(PROG_LANGUAGE_GERMAN)
            "Sparen",
#elif defined(PROG_LANGUAGE_ITALIAN)
            "Risparmi",
#elif defined(PROG_LANGUAGE_DUTCH)
            "Red",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
            "Poupe",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
            "Untatt",
#else
	    "Save",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(save_btn_cb), toplevel
	);
        gtk_accel_group_add(
            accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_grab_focus(w);
	gtk_widget_grab_default(w);
	gtk_widget_show(w);

	/* Cancel button */
	w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
            "Cancele",
#elif defined(PROG_LANGUAGE_FRENCH)
            "Annuler",
#elif defined(PROG_LANGUAGE_GERMAN)
            "Heben",
#elif defined(PROG_LANGUAGE_ITALIAN)
            "Annullar",
#elif defined(PROG_LANGUAGE_DUTCH)
            "Annuler",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
            "Cancela",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
            "Kanseler",
#else
            "Cancel",
#endif
	    NULL
	);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(cancel_btn_cb), toplevel
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
        gtk_accel_group_add(
            accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
            GTK_OBJECT(w), "clicked"
        );
        GUIButtonLabelUnderline(w, GDK_c); 
	gtk_widget_show(w);
}


/*
 *	Queries user for GIF format save options.
 */
gint iv_query_gif_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *interlaced,
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Color */
	gboolean *transparency,
	gboolean *looping
)
{
	gint    width = 320,
		height = -1,
		border_major = 5,
		border_minor = 2;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	GtkWidget	*format_bw_radio = NULL,
			*format_greyscale_radio = NULL,
			*format_color_radio = NULL,
			*interlaced_toggle = NULL,
			*transparency_toggle = NULL,
			*looping_toggle = NULL;

	/* Reset the global got response flag */
	got_response = FALSE;

	/* Create the dialog base widgets */
	build_std_dialog(
	    width, height,
	    "GIF Format Options",
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp
	);

	/* Create GIF format option widgets */
	if(client_vbox != NULL)
	{
	    GSList *gslist;
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column GtkVBox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		toplevel->window, NULL,
		(guint8 **)icon_save_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column GtkVBox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;


	    /* Format GtkHBox */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* B&W GtkRadioButton */
	    gslist = NULL;
	    format_bw_radio = w =
		gtk_radio_button_new_with_label(gslist, "Black & White");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	    /* Greyscale GtkRadioButton */
	    format_greyscale_radio = w =
		gtk_radio_button_new_with_label(gslist, "Greyscale");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	    /* Color GtkRadioButton */
	    format_color_radio = w =
		gtk_radio_button_new_with_label(gslist, "Color");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Set the initial format */
	    switch((format != NULL) ? *format : 0)
	    { 
	      case 0:
		w = format_bw_radio;
		break;
	      case 1:
		w = format_greyscale_radio;
		break;
	      default:
		w = format_color_radio;
		break;
	    }
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);


	    /* Interlaced, Transparency, and Looping GtkHBox */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Interlaced GtkCheckButton */
	    interlaced_toggle = w = gtk_check_button_new_with_label(
		"Interlaced"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    if((interlaced != NULL) ? *interlaced : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    gtk_widget_show(w);

	    /* Transparency GtkCheckButton */
	    transparency_toggle = w = gtk_check_button_new_with_label(
		"Transparency"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    if((transparency != NULL) ? *transparency : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    gtk_widget_show(w);

	    /* Looping GtkCheckButton */
	    looping_toggle = w = gtk_check_button_new_with_label(
		"Looping"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    if((looping != NULL) ? *looping : FALSE)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    gtk_widget_show(w);
	}


	/* Map the dialog */
	gtk_widget_show_raise(toplevel);


	/* Wait for user response */
	gtk_main();


	/* Get the values */
	if(got_response)
	{
	    if(format != NULL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_bw_radio))
		    *format = 0;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_greyscale_radio))
		    *format = 1;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_color_radio))
		    *format = 2;
		else
		    *format = 2;
	    }

	    if(interlaced != NULL)
		*interlaced = GTK_TOGGLE_BUTTON_GET_ACTIVE(interlaced_toggle);

	    if(transparency != NULL)
		*transparency = GTK_TOGGLE_BUTTON_GET_ACTIVE(transparency_toggle);

	    if(looping != NULL)
		*looping = GTK_TOGGLE_BUTTON_GET_ACTIVE(looping_toggle);
	}


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(toplevel);
	GTK_ACCEL_GROUP_UNREF(accelgrp);

	return(got_response);
}


/*
 *	Queries user for JPEG format save options.
 */
gint iv_query_jpeg_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gfloat *quality,	/* Quality value from 0.0 to 1.0 */
	gboolean *save_as_color
)
{
	gint	width = 320,
		height = -1,
		border_major = 5,
		border_minor = 2;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	GtkAdjustment	*quality_adj = NULL;
	GtkWidget	*save_as_greyscale_radio = NULL,
			*save_as_color_radio = NULL;


	/* Reset global got response flag */
	got_response = FALSE;

	/* Create dialog base widgets */
	build_std_dialog(
	    width, height,
	    "JPEG Format Options",
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp
	);

	/* Create JPEG format option widgets */
	if(client_vbox != NULL)
	{
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3;
	    GtkEntry *entry;
	    GtkAdjustment *adj;
	    GSList *gslist;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		toplevel->window, NULL,
		(guint8 **)icon_save_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Quality */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Label */
	    w = gtk_label_new("Quality:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Adjustment and hscale */
	    quality_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		(quality != NULL) ? (*quality) * 100.0f : 0.0f,
		0.0f,		/* Minimum */
		100.0f,		/* Maximum */
		1.0f,		/* Step inc */
		1.0f,		/* Page inc */
		0.0f		/* Page size */
	    );
	    w = gtk_hscale_new(adj);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_JPEG_QUALITY);
	    gtk_widget_show(w);
	    /* Entry */
	    w = gtk_entry_new_with_max_length(80);
	    entry = GTK_ENTRY(w);
	    gtk_widget_set_usize(w, 30, -1);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_JPEG_QUALITY);
	    gtk_widget_show(w);
	    /* Units label */
	    w = gtk_label_new("%");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Connect value changed signals */
	    gtk_signal_connect(
		GTK_OBJECT(adj), "value_changed",
		GTK_SIGNAL_FUNC(scale_changed_cb), entry
	    );
	    gtk_signal_connect(
		GTK_OBJECT(entry), "changed",
		GTK_SIGNAL_FUNC(entry_changed_adj_cb), adj
	    );
	    /* Update entry widget value */
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");


	    /* Hbox for Greyscale and Color */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Greyscale Radio Button */
	    gslist = NULL;
	    save_as_greyscale_radio = w =
		gtk_radio_button_new_with_label(gslist, "GreyScale");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	    /* Color Radio Button */
	    save_as_color_radio = w =
		gtk_radio_button_new_with_label(gslist, "Color");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Set initial radio state */
	    if((save_as_color != NULL) ? *save_as_color : FALSE)
		w = save_as_color_radio;
	    else
		w = save_as_greyscale_radio;
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

	}


	/* Map toplevel */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	if(got_response)
	{
	    GtkWidget *w;
	    GtkAdjustment *adj = quality_adj;
	    if((quality != NULL) && (adj != NULL))
		*quality = CLIP(
		    adj->value, adj->lower, adj->upper
		) / 100.0f;

	    w = save_as_color_radio;
	    if((save_as_color != NULL) && (w != NULL))
		*save_as_color = GTK_TOGGLE_BUTTON_GET_ACTIVE(w);
	}


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(toplevel);
	GTK_ACCEL_GROUP_UNREF(accelgrp);

	return(got_response);
}

/*
 *	Queries user for PNG format save options.
 */
gint iv_query_png_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gint *compression,	/* Value from 0 to 9 */
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Greyscale Alpha
				 * 3 = RGB
				 * 4 = RGBA */
	gboolean *interlaced
)
{
	gint    width = 320,
		height = -1,
		border_minor = 2,
		border_major = 5;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	GtkAdjustment *compression_adj = NULL;
	GtkWidget       *format_greyscale_radio = NULL,
			*format_rgb_radio = NULL,
			*alpha_toggle = NULL;
	GtkWidget *interlaced_toggle = NULL;


	/* Reset global got response flag */
	got_response = FALSE;

	/* Create dialog base widgets */
	build_std_dialog(
	    width, height,
	    "PNG Format Options",
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp
	);

	/* Create PNG format option widgets */
	if(client_vbox != NULL)
	{
	    GSList *gslist;
	    GtkAdjustment *adj;
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3, *parent4;
	    GtkEntry *entry;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		toplevel->window, NULL,
		(guint8 **)icon_save_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Compression */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Label */
	    w = gtk_label_new("Compression:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Adjustment and hscale */
	    compression_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		(gfloat)((compression != NULL) ? (*compression) : 0),
		0.0f,		/* Minimum */
		9.0f,		/* Maximum */
		1.0f,		/* Step inc */
		1.0f,		/* Page inc */
		0.0f		/* Page size */
	    );
	    w = gtk_hscale_new(adj);
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_PNG_COMPRESSION);
	    gtk_widget_show(w);
	    /* Entry */
	    w = gtk_entry_new_with_max_length(80);
	    entry = GTK_ENTRY(w);
	    gtk_widget_set_usize(w, 25, -1);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_PNG_COMPRESSION);
	    gtk_widget_show(w);
	    /* Connect value changed signals */
	    gtk_signal_connect(
		GTK_OBJECT(adj), "value_changed",
		GTK_SIGNAL_FUNC(scale_changed_cb), entry
	    );
	    gtk_signal_connect(
		GTK_OBJECT(entry), "changed",
		GTK_SIGNAL_FUNC(entry_changed_adj_cb), adj
	    );
	    /* Update entry widget value */
	    gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");


	    /* Hbox to hold format and alpha widgets in two separate sub
	     * hboxes
	     */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for format greyscale and rgb radios */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Format greyscale and rgb radios */
	    /* Greyscale radio */
	    gslist = NULL;
	    format_greyscale_radio = w =
		gtk_radio_button_new_with_label(gslist, "GreyScale");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	    /* RGB radio */
	    format_rgb_radio = w =
		gtk_radio_button_new_with_label(gslist, "RGB");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Set initial radio state */
	    switch((format != NULL) ? *format : 0)
	    {
	      /* B&W, Greyscale, or Greyscale Alpha */
	      case 0: case 1: case 2:
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(format_greyscale_radio), TRUE
		);
		break;
	      /* RGB or RGBA */
	      case 3: case 4:
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(format_rgb_radio), TRUE
		);
		break;
	    }

	    /* Hbox for alpha channel check */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Alpha channel check */
	    alpha_toggle = w = gtk_check_button_new_with_label("Alpha");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    /* Set initial state */
	    switch((format != NULL) ? *format : 0)
	    {
	      /* Greyscale Alpha or RGBA */
	      case 2: case 4:
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
		break;
	    }
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_PNG_ALPHA);
	    gtk_widget_show(w);


	    /* Interlaced */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Check button */
	    interlaced_toggle = w = gtk_check_button_new_with_label("Interlaced");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    if((interlaced != NULL) ? *interlaced : 0)
		GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_PNG_INTERLACED);
	    gtk_widget_show(w);
	}


	/* Map toplevel */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	if(got_response)
	{
	    GtkWidget *w;
	    GtkAdjustment *adj = compression_adj;
	    if((compression != NULL) && (adj != NULL))
		*compression = (gint)CLIP(
		    adj->value, adj->lower, adj->upper
		);

	    w = alpha_toggle;
	    if((format != NULL) && (w != NULL))
	    {
		/* RGB or RGBA? */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_rgb_radio))
		{
		    /* RGBA? */
		    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(w))
			*format = 4;
		    else
			*format = 3;
		}
		/* Greyscale or Greyscale Alpha */
		else
		{
		    /* Greyscale Alpha? */
		    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(w))
			*format = 2;
		    else
			*format = 1;
		}
	    }

	    w = interlaced_toggle;
	    if((interlaced != NULL) && (w != NULL))
		*interlaced = GTK_TOGGLE_BUTTON_GET_ACTIVE(w);
	}


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(toplevel);
	GTK_ACCEL_GROUP_UNREF(accelgrp);

	return(got_response);
}

/*
 *	Queries user for TGA format save options.
 */
gint iv_query_tga_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gint *format            /* 0 Greyscale
				 * 1 = RGB
				 * 2 = RGBA */
)
{
	gint    width = 320,
		height = -1,
		border_minor = 2,
		border_major = 5;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	GtkWidget       *format_greyscale_radio = NULL,
			*format_rgb_radio = NULL,
			*format_rgba_radio = NULL;


	/* Reset global got response flag */
	got_response = FALSE;

	/* Create dialog base widgets */
	build_std_dialog(
	    width, height,
	    "TGA Format Options",
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp
	);

	/* Create TGA format option widgets */
	if(client_vbox != NULL)
	{
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3, *parent4;
	    GSList *gslist;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		toplevel->window, NULL,
		(guint8 **)icon_save_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for format greyscale, rgb, and rgba radios */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Greyscale radio */
	    gslist = NULL;
	    format_greyscale_radio = w =
		gtk_radio_button_new_with_label(gslist, "GreyScale");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	    /* RGB radio */
	    format_rgb_radio = w =
		gtk_radio_button_new_with_label(gslist, "RGB");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));
	    /* RGBA radio */
	    format_rgba_radio = w =
		gtk_radio_button_new_with_label(gslist, "RGBA");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);

	    /* Set initial radio state */
	    switch((format != NULL) ? *format : 0)
	    {
	      case 2:
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(format_rgba_radio), TRUE
		);
		break;
	      case 1:
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(format_rgb_radio), TRUE
		);
		break;
	      default:
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(format_greyscale_radio), TRUE
		);
		break;
	    }
	}


	/* Map toplevel */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	if(got_response)
	{
	    if(format != NULL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_rgba_radio))
		    *format = 2;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_rgb_radio))
		    *format = 1;
		else
		    *format = 0;
	    }
	}

	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(toplevel);
	GTK_ACCEL_GROUP_UNREF(accelgrp);

	return(got_response);
}

/*
 *	Queries user for XPM format save options.
 */
gint iv_query_xpm_opts(
	GtkWidget *ref_toplevel,
	const gchar *path,
	gchar **c_id,
	gint *format,		/* 0 = B&W
				 * 1 = Greyscale
				 * 2 = Color */ 
	gint *max_colors,	/* -1 for no limit */
	guint8 *threshold	/* 0x00 to 0xff */
)
{
	gint    width = 320,
		height = -1,
		border_major = 5,
		border_minor = 2;
	GtkAccelGroup *accelgrp = NULL;
	GtkWidget *toplevel = NULL, *client_vbox = NULL;
	GList		*max_color_widgets_glist = NULL;
	GtkAdjustment	*threshold_adj = NULL;
	GtkWidget       *c_id_entry = NULL,
			*format_bw_radio = NULL,
			*format_greyscale_radio = NULL,
			*format_color_radio = NULL,
			*max_colors_spin = NULL,
			*no_max_colors_check = NULL,
			*threshold_scale = NULL,
			*threshold_entry = NULL;

	/* Reset global got response flag */
	got_response = FALSE;

	/* Create dialog base widgets */
	build_std_dialog(
	    width, height,
	    "XPM Format Options",
	    ref_toplevel,
	    &toplevel, &client_vbox, &accelgrp   
	);

	/* Create XPM format option widgets */
	if(client_vbox != NULL)
	{
	    GSList *gslist;
	    GtkAdjustment *adj;
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3,
		*parent4;

	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent = w;

	    /* Left column vbox */
	    w = gtk_vbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Icon */
	    w = gtk_pixmap_new_from_xpm_d(
		toplevel->window, NULL,
		(guint8 **)icon_save_32x32_xpm
	    );
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Right column vbox */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Hbox for C ID */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* C ID */
	    w = gtk_label_new("C ID:");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    /* Entry */
	    c_id_entry = w = gtk_entry_new();
	    gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_XPM_C_ID);
	    gtk_widget_show(w);


	    /* Hbox for Format */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
		   
	    /* B&W Radio Button */
	    gslist = NULL;
	    format_bw_radio = w =
		gtk_radio_button_new_with_label(gslist, "Black & White");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	    /* Greyscale Radio Button */
	    format_greyscale_radio = w =
		gtk_radio_button_new_with_label(gslist, "Greyscale");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	    /* Color Radio Button */
	    format_color_radio = w =
		gtk_radio_button_new_with_label(gslist, "Color");
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);


	    /* Hbox for Max Colors */
	    w = gtk_hbox_new(FALSE, border_major);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	    /* Hbox for Max Colors Spin */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent4 = w;
	    /* Label */
	    w = gtk_label_new("Maximum Colors:");
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    max_color_widgets_glist = g_list_append(
		max_color_widgets_glist, w
	    );
	    /* Spin */
	    adj = (GtkAdjustment *)gtk_adjustment_new(
		0.0f, 0.0f, (gfloat)((guint32)-1),
		1.0f, 10.0f, 0.0f
	    );
	    max_colors_spin = w = gtk_spin_button_new(adj, 1.0, 0);
	    gtk_widget_set_usize(w, 80, -1);
	    gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
            GUISetWidgetTip(w, SAVE_OPTS_TTM_XPM_MAX_COLORS);
	    gtk_widget_show(w);
	    max_color_widgets_glist = g_list_append(
		max_color_widgets_glist, w
	    );

	    /* No Max Colors Check */ 
	    no_max_colors_check = w = gtk_check_button_new_with_label(
		"No Limit"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    gtk_signal_connect(
		GTK_OBJECT(w), "toggled",
		GTK_SIGNAL_FUNC(toggle_glist_widget_set_insensitive_cb),
		max_color_widgets_glist
	    );
	    GUISetWidgetTip(w, SAVE_OPTS_TTM_XPM_NO_MAX_COLORS);
	    gtk_widget_show(w);


            /* Hbox for Threshold */
            w = gtk_hbox_new(FALSE, border_minor);
            gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            parent3 = w;
            /* Label */
            w = gtk_label_new("Threshold:");
            gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
            gtk_widget_show(w);
            /* Scale */ 
            threshold_adj = adj = (GtkAdjustment *)gtk_adjustment_new(
		(gfloat)0x00, (gfloat)0x00, (gfloat)0xff,
		1.0f, 1.0f, 0.0f
            );
            threshold_scale = w = gtk_hscale_new(adj);
            gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
            gtk_scale_set_draw_value(GTK_SCALE(w), FALSE);
            GUISetWidgetTip(w, SAVE_OPTS_TTM_XPM_THRESHOLD);
            gtk_widget_show(w);
            /* Entry */
            threshold_entry = w = gtk_entry_new_with_max_length(80);
            gtk_widget_set_usize(w, 30, -1);
            gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUIEditableEndowPopupMenu(w, 0);
            GUISetWidgetTip(w, SAVE_OPTS_TTM_XPM_THRESHOLD);
            gtk_widget_show(w);
	    /* Set up threshold widget signals */
            gtk_signal_connect(
                GTK_OBJECT(threshold_adj), "value_changed",
                GTK_SIGNAL_FUNC(scale_changed_cb), threshold_entry
            );
            gtk_signal_connect(
                GTK_OBJECT(threshold_entry), "changed",
                GTK_SIGNAL_FUNC(entry_changed_adj_cb), threshold_adj
            );


	    /* Set initial values */

	    /* C ID */
	    if((c_id != NULL) ? !STRISEMPTY(*c_id) : FALSE)
	    {
		gtk_entry_set_text(GTK_ENTRY(c_id_entry), *c_id);
		gtk_entry_set_position(GTK_ENTRY(c_id_entry), -1);
	    }

	    /* Format */
	    switch((format != NULL) ? *format : 0)
	    {
	      case 0:
		w = format_bw_radio;
		break;
	      case 1:
		w = format_greyscale_radio;
		break;
	      default:
		w = format_color_radio;
		break;
	    }
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);

	    /* Max Colors */
	    if(max_colors != NULL)
	    {
		w = no_max_colors_check;
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(w),
		    (*max_colors > 0) ? FALSE : TRUE
		);

		w = max_colors_spin;
		gtk_spin_button_set_value(
		    GTK_SPIN_BUTTON(w), (gfloat)(*max_colors)
		);
	    }

	    /* Threshold */
	    if(threshold != NULL)
	    {
		gtk_adjustment_set_value(
		    GTK_ADJUSTMENT(threshold_adj), (gfloat)(*threshold)
		);
	    }

	}


	/* Map toplevel */
	gtk_widget_show_raise(toplevel);


	/* Push main loop to wait for user response */
	gtk_main();


	/* Get values from widgets if user clicked on save */
	if(got_response)
	{
	    /* C ID */
	    if(c_id != NULL)
	    {
		g_free(*c_id);
		*c_id = STRDUP(gtk_entry_get_text(GTK_ENTRY(c_id_entry)));
	    }

	    /* Format */
	    if(format != NULL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_bw_radio))
		    *format = 0;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_greyscale_radio))
		    *format = 1;
		else if(GTK_TOGGLE_BUTTON_GET_ACTIVE(format_color_radio))
		    *format = 2;
		else
		    *format = 2;
	    }

	    /* Max Colors */
	    if(max_colors != NULL)
	    {
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(no_max_colors_check))
		    *max_colors = -1;
		else
		    *max_colors = GTK_ENTRY_GET_VALUEI(max_colors_spin);
	    }

            /* Threshold */
            if(threshold != NULL)
            {
		*threshold = (guint8)GTK_ADJUSTMENT_GET_VALUE(threshold_adj);
	    }
	}


	/* Destroy widgets and related resources */
	gtk_widget_hide(toplevel);
	gtk_window_set_modal(GTK_WINDOW(toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(toplevel), NULL);

	GTK_WIDGET_DESTROY(toplevel);
	GTK_ACCEL_GROUP_UNREF(accelgrp);
	g_list_free(max_color_widgets_glist);

	return(got_response);
}
