// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, read to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Implements the PNG reader K-3D object, which reads bitmap images using the libpng API
		\author Timothy M. Shead
*/

#include <k3dsdk/i18n.h>
#include <k3dsdk/ibitmap_read_format.h>
#include <k3dsdk/ideletable.h>
#include <k3dsdk/ifile_format.h>
#include <k3dsdk/module.h>
#include <k3dsdk/string_modifiers.h>

#include <boost/filesystem/path.hpp>

#include <png.h>

namespace libk3dpng
{

/////////////////////////////////////////////////////////////////////////////
// png_reader

class png_reader :
	public k3d::ifile_format,
	public k3d::ibitmap_read_format,
	public k3d::ideletable
{
public:
	png_reader()
	{
	}

	unsigned long priority()
	{
		return 128;
	}

	bool query_can_handle(const boost::filesystem::path& File)
	{
		return_val_if_fail(!File.empty(), false);
		return "png" == k3d::file_extension(File);
	}

	static void user_error_function(png_structp png_ptr, png_const_charp error_msg)
	{
		k3d::log() << critical << __PRETTY_FUNCTION__ << ": " << error_msg << std::endl;
	}

	static void user_warning_function(png_structp png_ptr, png_const_charp warning_msg)
	{
		k3d::log() << warning << __PRETTY_FUNCTION__ << ": " << warning_msg << std::endl;
	}

	bool read_file(const boost::filesystem::path& File, k3d::bitmap& Bitmap)
	{
		// Sanity checks ...
		return_val_if_fail(!File.empty(), false);

		k3d::log() << info << "Read " << File.native_file_string() << " using PNGReader" << std::endl;

		png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, 0, &user_error_function, &user_warning_function);
		return_val_if_fail(png_ptr, false);

		png_infop info_ptr = png_create_info_struct(png_ptr);
		if(!info_ptr)
			{
				png_destroy_read_struct(&png_ptr, 0, 0);
				return_val_if_fail(0, false);
			}

		png_infop end_info = png_create_info_struct(png_ptr);
		if(!end_info)
			{
				png_destroy_read_struct(&png_ptr, &info_ptr, 0);
				return_val_if_fail(0, false);
			}

		FILE* const file = fopen(File.native_file_string().c_str(), "rb");
		if(!file)
			{
				k3d::log() << error << "Error opening [" << File.native_file_string() << "] for PNG input" << std::endl;
				png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);
				return false;
			}

		png_init_io(png_ptr, file);
		png_read_png(png_ptr, info_ptr, PNG_TRANSFORM_IDENTITY, 0);
		png_bytep* row_pointers = png_get_rows(png_ptr, info_ptr);

		Bitmap.reset(png_get_image_width(png_ptr, info_ptr), png_get_image_height(png_ptr, info_ptr));

		switch(png_get_color_type(png_ptr, info_ptr))
		{
			case PNG_COLOR_TYPE_GRAY:
				switch(png_get_bit_depth(png_ptr, info_ptr))
				{
					case 8:
						k3d::log() << info << "Loading 8-bit grayscale image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_luma<boost::uint8_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					case 16:
						k3d::log() << info << "Loading 16-bit grayscale image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_luma<boost::uint16_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					default:
						k3d::log() << error << "Unsupported bit depth" << std::endl;
						break;
				}
				break;
			case PNG_COLOR_TYPE_GRAY_ALPHA:
				switch(png_get_bit_depth(png_ptr, info_ptr))
				{
					case 8:
						k3d::log() << info << "Loading 8-bit grayscale image with alpha" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_luma_alpha<boost::uint8_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					case 16:
						k3d::log() << info << "Loading 16-bit grayscale image with alpha" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_luma_alpha<boost::uint16_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					default:
						k3d::log() << error << "Unsupported bit depth" << std::endl;
						break;
				}
				break;
			case PNG_COLOR_TYPE_RGB:
				switch(png_get_bit_depth(png_ptr, info_ptr))
				{
					case 8:
						k3d::log() << info << "Loading 8-bit RGB image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_rgb<boost::uint8_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					case 16:
						k3d::log() << info << "Loading 16-bit RGB image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_rgb<boost::uint16_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					default:
						k3d::log() << error << "Unsupported bit depth" << std::endl;
						break;
				}
				break;
			case PNG_COLOR_TYPE_RGB_ALPHA:
				switch(png_get_bit_depth(png_ptr, info_ptr))
				{
					case 8:
						k3d::log() << info << "Loading 8-bit RGBA image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_rgba<boost::uint8_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					case 16:
						k3d::log() << info << "Loading 16-bit RGBA image" << std::endl;
						for(unsigned long row = 0; row != Bitmap.height(); ++row)
							{
								typedef k3d::basic_rgba<boost::uint16_t> integer_pixel_t;
								integer_pixel_t* const begin = reinterpret_cast<integer_pixel_t*>(row_pointers[row]);
								integer_pixel_t* const end = reinterpret_cast<integer_pixel_t*>(row_pointers[row]) + Bitmap.width();
								std::copy(begin, end, Bitmap.row_begin(row));
							}
						break;
					default:
						k3d::log() << error << "Unsupported bit depth" << std::endl;
						break;
				}
				break;
			default:
				k3d::log() << error << "Unsupported color type" << std::endl;
				break;
		}

		fclose(file);
		png_destroy_read_struct(&png_ptr, &info_ptr, &end_info);

		return true;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::application_plugin<png_reader>,
			k3d::interface_list<k3d::ibitmap_read_format> > factory(
				k3d::uuid(0xac17627d, 0xaa8848fd, 0xb621bd81, 0x4ba02136),
				"PNGReader",
				_("PNG (*.png)"),
				"Bitmap BitmapReader");

		return factory;
	}
};

/////////////////////////////////////////////////////////////////////////////
// png_reader_factory

k3d::iplugin_factory& png_reader_factory()
{
	return png_reader::get_factory();
}

} // namespace libk3dpng


