/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// mainwindow.cpp

#include "mainwindow.h"

#include "effectconfig.h"
#include "textstyleconfig.h"
#include "imageconfig.h"
#include "case.h"
#include "pref.h"
#include "settings.h"

#include "cdinfo.h"
#ifdef ENABLE_AUDIOCD
# include "cddbquery.h"
#endif

#include <kaccel.h>
#include <kaction.h>
#include <kapplication.h>
#include <kcolorbutton.h>
#include <kconfig.h>
#include <kdeversion.h>
#include <keditlistbox.h>
#include <kfiledialog.h>
#include <kfontdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <kio/netaccess.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kprinter.h>
#include <kpushbutton.h>
#include <kstatusbar.h>
#include <kstdaccel.h>
#include <kstdaction.h>
#include <ktempfile.h>
#include <kurldrag.h>
#include <kurl.h>
#include <kurlrequesterdlg.h>
#include <kurlrequester.h>

#include <qapplication.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qdragobject.h>
#include <qfileinfo.h>
#include <qfont.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qpaintdevicemetrics.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>
#include <qscrollview.h>
#include <qslider.h>
#include <qtabbar.h>
#include <qtextedit.h>
#include <qtimer.h>
#include <qtooltip.h>
#include <qwidgetstack.h>

#include <iostream>
#include <assert.h>


#define KOVERARTIST_GROUP_SPACING 8
#define KOVERARTIST_BUTTONBLOCK_SPACING 6

enum
{
   KOA_TABID_PROJECT=0, KOA_TABID_OPTIONS,
   /*must be last:*/ KOA_TABID_DISC
};



MainWindow::MainWindow()
:KMainWindow(0, "MainWindow")
,mPrinter(0)
,mRecentFiles(0)
,mProject()
,mRender(new KoverArtist::Renderer(&mProject))
#ifdef ENABLE_AUDIOCD
,mCddbQuery(new CddbQuery(this))
#else
,mCddbQuery(0)
#endif
,mCaptionLblFont()
,mIdActiveDisc(-1)
,mNumTabsNoDisc(0)
,mCddbDiscId(-1)
,mLocalFileName()
,mPixPreview()
,mUpdPreviewFront(true)
,mUpdPreviewBack(true)
,mRendering(false)
,mPreview(0)
,mUpdateTimer(new QTimer(this, "update-preview-timer"))
,mClearStatusbarTimer(new QTimer(this, "clear-statusbar-timer"))
,mTabBar(0)
,mTabStack(0)
,mTabProject(0)
,mTabOptions(0)
,mBaseProject(0)
,mBaseOptions(0)
,mBaseDisc(0)
{
   int i;

#ifdef ENABLE_AUDIOCD
   connect(mCddbQuery, SIGNAL(status(const QString&)), this, SLOT(changeStatusbar(const QString&)));
   connect(mCddbQuery, SIGNAL(finished()), this, SLOT(cddbQueryAudioCdDone()));
#endif /*ENABLE_AUDIOCD*/

   KoverArtist::Case *c = KoverArtist::Case::findCase("cd-standard");
   if (c) mProject.setBox(*c);
   else mProject.setBox(KoverArtist::Case::standardCase());

   for (i=0; i<_NUM_ACTIONS; ++i)
      mAction[i] = 0;

   setAutoSaveSettings();

   connect(mUpdateTimer, SIGNAL(timeout()), this, SLOT(updatePreviewTimeout()));
   connect(mClearStatusbarTimer, SIGNAL(timeout()), statusBar(), SLOT(clear()));

   mCaptionLblFont.setBold(true);

   QWidget *central = new QWidget(this, "central");
   setCentralWidget(central);

   mPreview = new QLabel(central, "preview");
   mPreview->setMinimumSize(400, 500);
   mPreview->setFrameStyle(QFrame::StyledPanel|QFrame::Sunken);
   mPreview->setEraseColor(QColor("#eeeeee"));
   mPreview->setMargin(4);

   mTabBar = new QTabBar(central, "tabbar");
   connect(mTabBar, SIGNAL(selected(int)), this, SLOT(switchToTab(int)));

   mTabStack = new QWidgetStack(central, "tabstack");

   setupProjectTab();
   setupOptionsTab();
   setupDiscTab();

   mNumTabsNoDisc = mTabBar->count();

   QGridLayout *grid = new QGridLayout(central, 2, 2);
   grid->addMultiCellWidget(mTabBar, 0, 0, 0, 1);
   grid->addWidget(mTabStack, 1, 0);
   grid->addWidget(mPreview, 1, 1);
   grid->setColStretch(1, 5);
   grid->setRowStretch(1, 1000);
   grid->activate();

   setAcceptDrops(true);
   setupActions();
   statusBar()->show();

#if KDE_IS_VERSION(3,3,0)
   setupGUI();
#else
   createGUI();
#endif

   numDiscsChanged();
   updatePreview(250);

   KConfig *cfg = KGlobal::config();
   if (mRecentFiles && cfg) mRecentFiles->loadEntries(cfg);
}


MainWindow::~MainWindow()
{
   KConfig *cfg = KGlobal::config();
   if (mRecentFiles && cfg) mRecentFiles->saveEntries(cfg);

   delete mRender;

#ifdef ENABLE_AUDIOCD
   delete mCddbQuery;
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::setupActions()
{
   KAction *act;

   // File
   KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
   KStdAction::open(this, SLOT(fileOpen()), actionCollection());
   mRecentFiles = KStdAction::openRecent(this, SLOT(load(const KURL&)), actionCollection());
   mAction[ACT_PROJ_REVERT] = KStdAction::revert(this, SLOT(fileRevert()), actionCollection());
   mAction[ACT_PROJ_SAVE] = KStdAction::save(this, SLOT(fileSave()), actionCollection());
   KStdAction::saveAs(this, SLOT(fileSaveAs()), actionCollection());
   KStdAction::print(this, SLOT(filePrint()), actionCollection());

   KStdAction::quit(this, SLOT(close()), actionCollection());

#if KDE_IS_VERSION(3,5,0)
   setStandardToolBarMenuEnabled(true);
#else
/*  mToolbarAction = KStdAction::showToolbar(this, SLOT(optionsShowToolbar()),
                                           actionCollection());*/
#endif

   // Disc
#ifdef DONT_COMPILE
   I18N_NOOP("&Disc");
#endif
   mAction[ACT_DISC_NEW] = new KAction(i18n("&New Disc"), "filenew",
            0, this, SLOT(discNew()), actionCollection(), "disc_new");
   mAction[ACT_DISC_DELETE] = new KAction(i18n("&Delete Disc"), "editdelete",
            0, this, SLOT(discDelete()), actionCollection(), "disc_delete");
   mAction[ACT_DISC_CLEAR] = new KAction(i18n("&Clear Disc"), "editclear",
            0, this, SLOT(discClear()), actionCollection(), "disc_clear");
   mAction[ACT_DISC_FRONT] = new KAction(i18n("Move Disc to &Front"), "back",
            0, this, SLOT(discFront()), actionCollection(), "disc_front");
   mAction[ACT_DISC_BACK] = new KAction(i18n("Move Disc to &Back"), "forward",
            0, this, SLOT(discBack()), actionCollection(), "disc_back");
   mAction[ACT_DISC_IMPORT] = new KAction(i18n("&Import Disc"), "fileimport",
            0, this, SLOT(discImport()), actionCollection(), "disc_import");

#ifdef ENABLE_AUDIOCD
   // Use audio-cd icon if available, else use generic cd icon
   KIconLoader* ildr = KGlobal::iconLoader();
   QPixmap pix = ildr->loadIcon("cdaudio_unmount", KIcon::Toolbar, 0,
                                KIcon::DefaultState, 0, true);
   if (pix.isNull()) pix = ildr->loadIcon("cd", KIcon::Toolbar);

   mAction[ACT_DISC_IMPORT_CDDB] = new KAction(i18n("Import &Audio-CD"), pix,
            0, this, SLOT(discImportCddb()), actionCollection(), "disc_import_cddb");
#endif /*ENABLE_AUDIOCD*/

   mAction[ACT_DISC_EXPORT] = new KAction(i18n("&Export Disc"), "fileexport",
            0, this, SLOT(discExport()), actionCollection(), "disc_export");

   // Extras
#ifdef DONT_COMPILE
   I18N_NOOP("&Extras");
#endif
   new KAction(i18n("&Search for cover images..."), "filefind", 0, this,
      SLOT(searchCoverImages()), actionCollection(), "search_cover_images");

   // Others
   KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection());

   new KAction(i18n("&Redisplay"), "reload", 0, this,
               SLOT(redisplay()), actionCollection(), "redisplay");

   act = new KAction(i18n("New Window"), "window_new", 0, this, SLOT(windowNew()),
                     actionCollection(), "window_new");
}


void MainWindow::setupProjectTab()
{
   QBoxLayout *box, *vbox;
   QWidget *base;
   QLabel *lbl;
   int h, i;

   mTabProject = new QTab(i18n("Project"));
   mTabProject->setIdentifier(KOA_TABID_PROJECT);
   mTabBar->addTab(mTabProject);
   mBaseProject = new QWidget(mTabStack, "base-project");
   mTabStack->addWidget(mBaseProject, KOA_TABID_PROJECT);
   base = mBaseProject;

   vbox = new QVBoxLayout(base, 4, 2, "project-vbox");

   //
   // Number of discs
   //
   lbl = createCaptionLabel(i18n("Number of Discs"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   box = new QHBoxLayout(vbox, 2, "box-discs");
   mEdtNumDiscs = new QLineEdit(base, "edt-discs");
   mEdtNumDiscs->setText("9999");
   mEdtNumDiscs->setFixedSize(mEdtNumDiscs->sizeHint());
   mEdtNumDiscs->setText("1");
   connect(mEdtNumDiscs, SIGNAL(returnPressed()), this, SLOT(slotDiscsEdited()));
   box->addWidget(mEdtNumDiscs);
   mBtnDiscsInc = new QPushButton("+", base, "btn-discsinc");
   mBtnDiscsInc->setFixedSize(mBtnDiscsInc->sizeHint());
   connect(mBtnDiscsInc, SIGNAL(clicked()), SLOT(slotDiscsInc()));
   box->addWidget(mBtnDiscsInc);
   mBtnDiscsDec = new QPushButton("-", base, "btn-discsdec");
   mBtnDiscsDec->setFixedSize(mBtnDiscsDec->sizeHint());
   connect(mBtnDiscsDec, SIGNAL(clicked()), SLOT(slotDiscsDec()));
   box->addWidget(mBtnDiscsDec);

   box->addStretch(1000);
   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Case
   //
   lbl = createCaptionLabel(i18n("Case"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mCboCase = new QComboBox(base, "cbo-title-effect");
   QValueVector<KoverArtist::Case>& cases = KoverArtist::Case::cases();
   for (i=0; i<(int)cases.size(); ++i)
      mCboCase->insertItem(i18n(cases[i].label()));
   connect(mCboCase, SIGNAL(activated(int)), this, SLOT(slotCaseActivated(int)));
   vbox->addWidget(mCboCase, Qt::AlignLeft);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Title
   //
   lbl = createCaptionLabel(i18n("Title"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mTscTitle = new TextStyleConfig(&mProject.titleStyle(), true, true, base, "tsc-title");
   mTscTitle->setFixedSize(mTscTitle->sizeHint());
   connect(mTscTitle, SIGNAL(changed()), this, SLOT(updatePreviewFront()));
   connect(mTscTitle, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mTscTitle, Qt::AlignLeft);

   mEdtProjTitle = new QTextEdit(base, "edt-projtitle");
   connect(mEdtProjTitle, SIGNAL(textChanged()), SLOT(slotTitleChanged()));
   mEdtProjTitle->setTextFormat(Qt::PlainText);
   h = mEdtProjTitle->fontMetrics().height()<<2;
   mEdtProjTitle->setFixedHeight(h);
   vbox->addWidget(mEdtProjTitle);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Sub Title
   //
   lbl = createCaptionLabel(i18n("Sub Title"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mTscSubTitle = new TextStyleConfig(&mProject.subTitleStyle(), true, false, base, "tsc-subtitle");
   mTscSubTitle->setFixedSize(mTscSubTitle->sizeHint());
   connect(mTscSubTitle, SIGNAL(changed()), this, SLOT(updatePreviewFront()));
   connect(mTscSubTitle, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mTscSubTitle, Qt::AlignLeft);

   mEdtProjSubTitle = new QTextEdit(base, "edt-projsubtitle");
   connect(mEdtProjSubTitle, SIGNAL(textChanged()), SLOT(slotSubTitleChanged()));
   mEdtProjSubTitle->setTextFormat(Qt::PlainText);
   mEdtProjSubTitle->setFixedHeight(h);
   vbox->addWidget(mEdtProjSubTitle);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Text Effect for Title and Sub Title
   //
   lbl = createCaptionLabel(i18n("Title and Sub Title Effect"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mEfcTitle = new EffectConfig(&mProject.titleEffect(), base, "efc-title");
   connect(mEfcTitle, SIGNAL(changed()), this, SLOT(updatePreviewFront()));
   connect(mEfcTitle, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mEfcTitle);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Finish setup
   //
   vbox->addStretch(1000);
   vbox->activate();
}


void MainWindow::setupOptionsTab()
{
   QBoxLayout *vbox, *box;
   QWidget *base;
   QLabel *lbl;

   mTabOptions = new QTab(i18n("Options"));
   mTabOptions->setIdentifier(KOA_TABID_OPTIONS);
   mTabBar->addTab(mTabOptions);

   mBaseOptions = new QWidget(mTabStack, "base-options");
   mTabStack->addWidget(mBaseOptions, KOA_TABID_OPTIONS);
   base = mBaseOptions;

   vbox = new QVBoxLayout(base, 4, 1, "options-vbox");


   //
   // General
   //
   lbl = createCaptionLabel(i18n("General"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   box = new QHBoxLayout(vbox, 2);

   lbl = new QLabel(i18n("Background")+":", base);
   lbl->setFixedSize(lbl->sizeHint());
   box->addWidget(lbl);

   mBtnBackground = new KColorButton(Qt::white, base, "btn-background");
   mBtnBackground->setFixedSize(mBtnBackground->sizeHint());
   connect(mBtnBackground, SIGNAL(changed(const QColor&)), SLOT(setBackgroundColor(const QColor&)));
   QToolTip::add(mBtnBackground, i18n("Change the background color"));
   box->addWidget(mBtnBackground);

   box->addSpacing(8);
   box->addStretch(100);

   lbl = new QLabel(i18n("Outline")+":", base);
   lbl->setFixedSize(lbl->sizeHint());
   box->addWidget(lbl);

   mBtnOutline = new KColorButton(Qt::white, base, "btn-outline");
   mBtnOutline->setFixedSize(mBtnOutline->sizeHint());
   connect(mBtnOutline, SIGNAL(changed(const QColor&)), SLOT(setOutlineColor(const QColor&)));
   QToolTip::add(mBtnOutline, i18n("Change the outline color"));
   box->addWidget(mBtnOutline);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Front picture
   //
   lbl = createCaptionLabel(i18n("Front Picture"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);
//    vbox->addSpacing(2);

   mImcFront = new ImageConfig(&mProject.imgFront(), base);
   vbox->addWidget(mImcFront);
   connect(mImcFront, SIGNAL(changed()), this, SLOT(modified()));
   connect(mImcFront, SIGNAL(changed()), this, SLOT(updatePreview()));

   box = new QHBoxLayout(vbox, 0);
   mCbxFrontImageWrap = new QCheckBox(i18n("Wrap image around front and back side"), base);
   mCbxFrontImageWrap->setFixedSize(mCbxFrontImageWrap->sizeHint());
   connect(mCbxFrontImageWrap, SIGNAL(toggled(bool)), this, SLOT(setFrontImageWrap(bool)));
   box->addSpacing(4);
   box->addWidget(mCbxFrontImageWrap);
   box->addStretch(1000);

   box = new QHBoxLayout(vbox, 0);
   mCbxFrontTitleHide = new QCheckBox(i18n("Hide title text on front side"), base);
   mCbxFrontTitleHide->setFixedSize(mCbxFrontTitleHide->sizeHint());
   connect(mCbxFrontTitleHide, SIGNAL(toggled(bool)), this, SLOT(setFrontTitleHide(bool)));
   box->addSpacing(4);
   box->addWidget(mCbxFrontTitleHide);
   box->addStretch(1000);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Back picture
   //
   lbl = createCaptionLabel(i18n("Back Picture"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);
//    vbox->addSpacing(2);

   mImcBack = new ImageConfig(&mProject.imgBack(), base);
   vbox->addWidget(mImcBack);
   connect(mImcBack, SIGNAL(changed()), this, SLOT(modified()));
   connect(mImcBack, SIGNAL(changed()), this, SLOT(updatePreviewBack()));

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Disc Contents Text
   //
   lbl = createCaptionLabel(i18n("Disc Contents Text"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);
//    vbox->addSpacing(2);

   mTscContents = new TextStyleConfig(&mProject.contentsStyle(), true, false, base, "tsc-contents");
   mTscContents->setFixedSize(mTscContents->sizeHint());
   connect(mTscContents, SIGNAL(changed()), this, SLOT(updatePreviewBack()));
   connect(mTscContents, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mTscContents, Qt::AlignLeft);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Disc Contents Effect
   //
   lbl = createCaptionLabel(i18n("Disc Contents Effect"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mEfcContents = new EffectConfig(&mProject.contentsEffect(), base);
   connect(mEfcContents, SIGNAL(changed()), this, SLOT(updatePreviewBack()));
   connect(mEfcContents, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mEfcContents);
   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Side Text Style
   //
   lbl = createCaptionLabel(i18n("Side Text"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mTscSide = new TextStyleConfig(&mProject.sideStyle(), true, false, base, "tsc-side");
   mTscSide->setFixedSize(mTscSide->sizeHint());
   connect(mTscSide, SIGNAL(changed()), this, SLOT(updatePreview()));
   connect(mTscSide, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mTscSide, Qt::AlignLeft);

   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Side Text Effect
   //
   lbl = createCaptionLabel(i18n("Side Text Effect"), base);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mEfcSide = new EffectConfig(&mProject.sideEffect(), base, "efc-side");
   connect(mEfcSide, SIGNAL(changed()), this, SLOT(updatePreview()));
   connect(mEfcSide, SIGNAL(changed()), this, SLOT(modified()));
   vbox->addWidget(mEfcSide);

//    vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   // Finish setup
   vbox->addStretch(1000);
   vbox->activate();
}


void MainWindow::setupDiscTab()
{
   QBoxLayout *vbox;
   QLabel *lbl;

   mBaseDisc = new QWidget(mTabStack, "base-disc");
   mTabStack->addWidget(mBaseDisc, KOA_TABID_DISC);

   vbox = new QVBoxLayout(mBaseDisc, 4, 2, "disc-vbox");

   //
   // Disc Title
   //
   lbl = createCaptionLabel(i18n("Disc Title (optional)"), mBaseDisc);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mEdtDiscTitle = new QTextEdit(mBaseDisc, "edt-disctitle");
   connect(mEdtDiscTitle, SIGNAL(textChanged()), SLOT(slotDiscTitleChanged()));
   mEdtDiscTitle->setTextFormat(Qt::PlainText);
   mEdtDiscTitle->setFixedHeight(mEdtDiscTitle->fontMetrics().height()<<1);
   vbox->addWidget(mEdtDiscTitle, Qt::AlignLeft);
   vbox->addSpacing(KOVERARTIST_GROUP_SPACING);

   //
   // Disc Contents
   //
   lbl = createCaptionLabel(i18n("Disc Contents"), mBaseDisc);
   vbox->addWidget(lbl, Qt::AlignLeft);

   mElbDiscContents = new KEditListBox(mBaseDisc, "elb-disc-contents", true,
                                       KEditListBox::All);
   connect(mElbDiscContents, SIGNAL(changed()), this, SLOT(slotDiscContentsChanged()));
   vbox->addWidget(mElbDiscContents);

   // Finish setup
   //vbox->addStretch(1000);
   vbox->activate();
}


QPushButton* MainWindow::createButton(const char* aIcon, QWidget* aParent,
                                      const char* aSlot, QBoxLayout* aBox) const
{
   QPushButton *btn;
   btn = new QPushButton(BarIconSet(aIcon), 0, aParent, "btn-toggle");
   btn->setFixedSize(btn->sizeHint());
   connect(btn, SIGNAL(clicked()), aSlot);

   if (aBox) aBox->addWidget(btn);

   return btn;
}


QLabel* MainWindow::createCaptionLabel(const QString& aText, QWidget* aParent,
                                       const char* aName) const
{
   QLabel* lbl = new QLabel(aText, aParent, aName);
   lbl->setFont(mCaptionLblFont);
   lbl->setFixedSize(lbl->sizeHint());
   return lbl;
}


void MainWindow::updateFromProject()
{
   bool mod = mProject.isModified();
   bool ue = isUpdatesEnabled();
   setUpdatesEnabled(false);

   mEdtProjTitle->setText(mProject.title());
   mEdtProjSubTitle->setText(mProject.subTitle());

   mImcFront->update();
   mCbxFrontImageWrap->setChecked(mProject.imgFrontWrapAround());
   mCbxFrontTitleHide->setChecked(mProject.frontTitleHide());

   mImcBack->update();

   mBtnBackground->setColor(mProject.backgroundColor());
   mBtnOutline->setColor(mProject.outlineColor());

   mCboCase->setCurrentItem(KoverArtist::Case::indexOfCase(mProject.box().name()));

   mEfcContents->update();
   mEfcTitle->update();
   if (mEfcSide) mEfcSide->update();

   mTscTitle->update();
   mTscSubTitle->update();
   mTscContents->update();
   mTscSide->update();

   numDiscsChanged();

   setUpdatesEnabled(ue);
   if (!mod) mProject.clearModified();

   changeCaption();
   updatePreview(100);
}


void MainWindow::updateDiscFromProject()
{
   bool upd = isUpdatesEnabled();
   bool mod = mProject.isModified();
   setUpdatesEnabled(false);

   // Calling mElbDiscContents->clear() always triggers the changed()
   // signal (KDE 3.5.1) and this deletes the project's contents. So we
   // call clear of the listbox only
   mElbDiscContents->listBox()->clear();

   if (mIdActiveDisc<0)
   {
      mEdtDiscTitle->setText("");
   }
   else
   {
      KoverArtist::Disc& disc = mProject.disc(mIdActiveDisc);
      mEdtDiscTitle->setText(disc.title());
      mElbDiscContents->insertStringList(disc.entries());
   }

   setUpdatesEnabled(upd);
   if (!mod) mProject.clearModified();
}


bool MainWindow::load(const KURL& aUrl)
{
   bool upd = isUpdatesEnabled();
   setUpdatesEnabled(false);
   qApp->setOverrideCursor(Qt::WaitCursor);

   bool ok = mProject.load(aUrl, this);
   bool mod = mProject.isModified();
   updateFromProject();
   updateDiscFromProject();
   if (!mod) mProject.clearModified();
   changeCaption();

   if (ok && mRecentFiles)
      mRecentFiles->addURL(aUrl);

   qApp->restoreOverrideCursor();
   setUpdatesEnabled(upd);
   updatePreview(100);

   return ok;
}


bool MainWindow::save(const KURL& aUrl)
{
   if (!mProject.save(aUrl, this)) return false;

   if (mRecentFiles)
      mRecentFiles->addURL(aUrl);

   mUpdateTimer->stop();
   changeCaption();

   return true;
}


void MainWindow::saveProperties(KConfig* aConfig)
{
   if (!mProject.url().isEmpty())
      aConfig->writePathEntry("lastURL", mProject.url().url());
}


void MainWindow::readProperties(KConfig *aConfig)
{
   QString url = aConfig->readPathEntry("lastURL");
   if (!url.isEmpty()) load(KURL(url));
}


void MainWindow::modified()
{
   if (mProject.isModified()) return;
   mProject.modified();
   changeCaption();
}


bool MainWindow::queryClose()
{
   if (mProject.isModified())
   {
      QString nm = mProject.url().fileName();
      if (nm.isEmpty()) nm = "unnamed.koap";
      int ret = KMessageBox::questionYesNoCancel(this,
         i18n("The project \"%1\" has been modified.\n"
              "Do you want to save your changes or discard them?").arg(nm),
         i18n("Confirm Close")+" - KoverArtist",
         KStdGuiItem::save(), KStdGuiItem::discard(), "query-close");
      if (ret==KMessageBox::Cancel) return false;
      if (ret==KMessageBox::Yes)
         if (!fileSave()) return false;
   }

   return true;
}


bool MainWindow::isImageFile(const QString& aFileName) const
{
   const char* fmt;
   QStrList fmts = QImageIO::inputFormats();

   QCString ext = KoverArtist::Project::fileExt(aFileName).upper().local8Bit();
   if (ext=="JPG") ext = "JPEG";

   for (fmt=fmts.first(); fmt; fmt=fmts.next())
      if (ext==fmt) return true;

   return false;
}


bool MainWindow::canDecode(const KURL& aUrl) const
{
   QString fname = aUrl.fileName();
   if (KoverArtist::Project::canLoad(fname)) return true;
   if (KoverArtist::Disc::canLoad(fname)) return true;

#ifdef ENABLE_AUDIOCD
   if (CddbQuery::canDecode(aUrl)) return true;
#endif /*ENABLE_AUDIOCD*/

   return isImageFile(fname);
}


void MainWindow::dragEnterEvent(QDragEnterEvent *e)
{
   bool ok = KURLDrag::canDecode(e);
   if (ok)
   {
      KURL::List urls;
      if (KURLDrag::decode(e, urls) && !urls.isEmpty())
         ok = canDecode(urls.first());
   }

   e->accept(ok);
}


bool MainWindow::imageDropped(const KURL &aUrl, const QPoint& aPos)
{
   QPopupMenu mnu(this);
   mnu.insertItem(i18n("Use for front side"), 1);
   mnu.insertItem(i18n("Use for back side"), 2);
   mnu.insertSeparator();
   mnu.insertItem(BarIconSet("cancel"), i18n("Cancel"), 0, 0,
                  QKeySequence(Qt::Key_Escape), 0);
   int ret = mnu.exec(aPos);
   if (ret<=0) return false;

   if (ret==1) mProject.imgFront().load(aUrl);
   else mProject.imgBack().load(aUrl);

   updateFromProject();
   modified();

   return true;
}


void MainWindow::dropEvent(QDropEvent *e)
{
   KURL::List urls;
   bool ok = false;

   e->accept(false);

   if (!KURLDrag::decode(e, urls) || urls.isEmpty())
      return;

   const KURL &url = urls.first();
   KMimeType::Ptr mt = KMimeType::findByURL(url);

//    std::cout<<"Dropped "<<url.prettyURL()<<" mime type "<<mt->name()<<std::endl;

#ifdef ENABLE_AUDIOCD
   if (CddbQuery::canDecode(url.url()))
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;

      if (!mProject.disc(mCddbDiscId).isEmpty() && !confirmDiscNotEmpty(mCddbDiscId))
         return;

      e->accept(mCddbQuery->fromUrl(url));
      return;
   }
#endif /*ENABLE_AUDIOCD*/

   // Remote protocols do not get detected properly (at least with images in
   // Kde 3.5.1) so we need to download the file to determine the type in that
   // case
   if (!url.isLocalFile() && mt->name()=="application/octet-stream")
   {
      QString fname;
      if (!KIO::NetAccess::download(url, fname, this))
      {
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
         return;
      }
      mt = KMimeType::findByPath(fname);
      KIO::NetAccess::removeTempFile(fname);
   }

   QString mtName = mt->name();
   int idx = mtName.find('/');
   if (idx>0) mtName = mtName.left(idx);

   if (mtName=="image" || mt->is("image/png") || mt->is("image/jpeg"))
      ok = imageDropped(url, mapToGlobal(e->pos()));
   else if (KoverArtist::Project::canLoad(url.fileName()))
      ok = load(url);
   else if (KoverArtist::Disc::canLoad(url.fileName()))
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;
      KoverArtist::Disc& d = mProject.disc(mCddbDiscId);
      if (!d.isEmpty() && !confirmDiscNotEmpty(mCddbDiscId)) return;
      if (d.load(url, this))
      {
         modified();
         updatePreviewBack();
         updateDiscFromProject();
      }
   }

   e->accept(ok);
}


void MainWindow::windowNew()
{
   MainWindow *mw = new MainWindow;
   mw->show();
}


void MainWindow::fileNew()
{
   if (!queryClose()) return;

   mProject.clear();

   KoverArtist::Case *c = KoverArtist::Case::findCase("cd-standard");
   if (c) mProject.setBox(*c);
   else mProject.setBox(KoverArtist::Case::standardCase());

   updateFromProject();
}


void MainWindow::fileRevert()
{
   if (!mProject.url().isValid()) return;
   if (!queryClose()) return;
   KURL url = mProject.url();
   load(url);
}


void MainWindow::fileOpen()
{
   if (!queryClose()) return;

   QString filter = QString("*.koap *.kiap *.kmf|%1\n*.koap|%2\n*.kiap|%3\n"
      "*.kmf|%4\n*|%5")
      .arg(i18n("All Supported Files"))
      .arg(i18n("KoverArtist Projects"))
      .arg(i18n("KinoArtist Projects"))
      .arg(i18n("KMediaFactory Projects"))
      .arg(i18n("All Files"));

   KFileDialog dlg(":KoverArtist::Project", filter, this, "dlg-open", true);
   dlg.setCaption(i18n("Open Project"));
   dlg.setOperationMode(KFileDialog::Opening);
   dlg.setMode(KFile::File);

   int ret = dlg.exec();
   if (ret!=QDialog::Accepted) return;

   KURL url = dlg.selectedURL();
   if (!url.isEmpty()) load(url);
}


bool MainWindow::fileSave()
{
   if (mProject.url().isValid()) return save(mProject.url());
   return fileSaveAs();
}


bool MainWindow::fileSaveAs()
{
   QString filter = QString("*.koap|%1")
      .arg(i18n("KoverArtist Projects"));

   KFileDialog dlg(":KoverArtist::Project", filter, this, "dlg-save", true);
   dlg.setCaption(i18n("Save Project"));
   dlg.setOperationMode(KFileDialog::Saving);
   dlg.setMode(KFile::File);
   dlg.setURL(mProject.url());
   dlg.setSelection(mProject.url().fileName());

   int ret = dlg.exec();
   if (ret!=QDialog::Accepted) return false;

   KURL url = dlg.selectedURL();
   if (url.isEmpty() || !url.isValid()) return false;

   if (url.isLocalFile())
   {
      if (QFile::exists(url.path()))
      {
         int ret = KMessageBox::warningYesNo(this,
            i18n("A file '%1' already exists.\nDo you want to overwrite it?").arg(url.fileName()),
            i18n("Save file")+" - KoverArtist", KStdGuiItem::cont(),
            KStdGuiItem::cancel());
	 if (ret==KMessageBox::Cancel) return false;
      }
   }

   return save(url);
}


void MainWindow::filePrint()
{
   if (!mPrinter)
   {
      mPrinter = new KPrinter;
      mPrinter->setOrientation(KPrinter::Portrait);
//       mPrinter->setResolution(200);
   }

   if (mPrinter->setup(this))
   {
      std::cout<<"Printer resolution: "<<mPrinter->resolution()<<"dpi"<<std::endl;
      KoverArtist::Renderer render(&mProject);
      render.render(mPrinter, false);
   }
}


void MainWindow::discClear()
{
   if (mIdActiveDisc<0) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc& disc = discs[mIdActiveDisc];

   if (!disc.isEmpty())
   {
      int ret = KMessageBox::warningContinueCancel(this,
         i18n("The current disc is not empty.\nContinue and clear it?"),
         i18n("Clear disc")+" - KoverArtist",
         KStdGuiItem::cont(), "confirm_disc_clear");
      if (ret!=KMessageBox::Continue) return;
   }

   disc = KoverArtist::Disc();
   updateDiscFromProject();
   modified();
   updatePreviewBack();
}


void MainWindow::discNew()
{
   KoverArtist::DiscVector& discs = mProject.discs();
   if (mIdActiveDisc<0)
   {
      discs.append(KoverArtist::Disc());
   }
   else
   {
      KoverArtist::DiscVector::iterator it = mProject.discIt(mIdActiveDisc);
      discs.insert(it, KoverArtist::Disc());
   }
   numDiscsChanged();
   if (mIdActiveDisc>0) updateDiscFromProject();
   modified();
   updatePreviewBack();
}


void MainWindow::discDelete()
{
   if (mIdActiveDisc<0) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::DiscVector::iterator it = mProject.discIt(mIdActiveDisc);

   if (!it->isEmpty())
   {
      int ret = KMessageBox::warningContinueCancel(0,
         i18n("The current disc is not empty.\nContinue and delete it?"),
         i18n("Delete disc")+" - KoverArtist",
         KStdGuiItem::cont(), "confirm_disc_delete");
      if (ret!=KMessageBox::Continue) return;
   }

   discs.erase(it);

   if (mIdActiveDisc>=int(discs.count()))
      mIdActiveDisc = discs.count()-1;

   mTabBar->setCurrentTab(KOA_TABID_DISC+mIdActiveDisc);
   numDiscsChanged();
   updateDiscFromProject();
   modified();
   updatePreviewBack();
}


void MainWindow::discFront()
{
   if (mIdActiveDisc<1) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc d = discs[mIdActiveDisc];
   discs[mIdActiveDisc] = discs[mIdActiveDisc-1];
   discs[mIdActiveDisc-1] = d;
   mTabBar->setCurrentTab(KOA_TABID_DISC+mIdActiveDisc-1);
   modified();
   updatePreviewBack();
}


void MainWindow::discBack()
{
   if (mIdActiveDisc<0 || mIdActiveDisc>=mProject.count()-1) return;
   KoverArtist::DiscVector& discs = mProject.discs();
   KoverArtist::Disc d = discs[mIdActiveDisc];
   discs[mIdActiveDisc] = discs[mIdActiveDisc+1];
   discs[mIdActiveDisc+1] = d;
   mTabBar->setCurrentTab(KOA_TABID_DISC+mIdActiveDisc+1);
   modified();
   updatePreviewBack();
}


bool MainWindow::confirmDiscNotEmpty(int aDiscIdx)
{
   int ret = KMessageBox::warningContinueCancel(this,
         i18n("Disc #%1 is not empty.\nContinue and replace it?").arg(aDiscIdx+1),
         i18n("Import disc")+" - KoverArtist",
         KStdGuiItem::cont(), "confirm_disc_import");

   return ret==KMessageBox::Continue;
}


void MainWindow::cddbQueryAudioCdDone()
{
#ifdef ENABLE_AUDIOCD
   int num = mProject.count();
   if (mCddbDiscId<0)
   {
      mCddbDiscId = mIdActiveDisc;
      if (mCddbDiscId<0) mCddbDiscId = 0;
   }
   if (mCddbDiscId>=num)
   {
      mProject.resize(num+1);
      mCddbDiscId = num;
   }

   mProject.disc(mCddbDiscId) = mCddbQuery->disc();

   if (mProject.title().isEmpty() &&
       KoverArtist::Settings::instance()->cddbAutoSetTitle)
   {
      QString t = mCddbQuery->disc().title();
      int idx = t.find('/');
      if (idx>0)
      {
         mProject.setTitle(t.mid(idx+1).stripWhiteSpace());
         mProject.setSubTitle(t.left(idx).stripWhiteSpace());
      }
      else mProject.setTitle(t);

      updateFromProject();
      updatePreview();
   }

   modified();
   updatePreviewBack();
   updateDiscFromProject();
   mCddbDiscId = -1;
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::discImportCddb()
{
#ifdef ENABLE_AUDIOCD
   mCddbDiscId = mIdActiveDisc;
   if (mCddbDiscId<0) mCddbDiscId = 0;

   if (!mProject.disc(mCddbDiscId).isEmpty() && !confirmDiscNotEmpty(mCddbDiscId))
      return;

   mCddbQuery->cddbQueryAudioCd();
#endif /*ENABLE_AUDIOCD*/
}


void MainWindow::discImport()
{
   int id = mIdActiveDisc;
   if (id<0) id = 0;
   KoverArtist::Disc& disc = mProject.disc(id);

   if (!disc.isEmpty() && !confirmDiscNotEmpty(id))
      return;

   QString filter = QString("*.koad *.kiap *.kmf *.txt|%1\n*.koad|%2\n*.kiap|%3\n"
      "*.kmf|%4\n*.txt|%5\n*|%6")
      .arg(i18n("All Supported Files"))
      .arg(i18n("KoverArtist Discs"))
      .arg(i18n("KinoArtist Projects"))
      .arg(i18n("KMediaFactory Projects"))
      .arg(i18n("Text Files"))
      .arg(i18n("All Files"));

   KURL url = KFileDialog::getOpenURL(QString::null, filter, this,
                                      i18n("Import Disc")+" - KoverArtist");
   if (url.isEmpty()) return;

   if (!discImport(disc, url)) return;

   if (mProject.title().isEmpty())
   {
      mProject.setTitle(disc.title());
      updateFromProject();
   }

   int num = disc.snapShots().count();
   if (mProject.imgFront().isNull() && num>0)
   {
      mProject.imgFront().load(disc.snapShots()[0]);
      updatePreviewFront();
   }
   if (mProject.imgBack().isNull() && num>1)
      mProject.imgBack().load(disc.snapShots()[1]);

   updateDiscFromProject();
   modified();
   updatePreviewBack();
}


bool MainWindow::discImport(KoverArtist::Disc& aDisc, const KURL& aUrl)
{
   QString fileName;
   bool ok = false;

   if (aUrl.isLocalFile()) fileName = aUrl.path();
   else
   {
      if (!KIO::NetAccess::download(aUrl, fileName, this))
      {
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
         return false;
      }
   }

   QFileInfo fi(fileName);
   if (fi.exists() && fi.isReadable() && !fi.isDir())
   {
      ok = aDisc.loadFile(fileName);
   }
   else
   {
      KMessageBox::error(this, KIO::NetAccess::lastErrorString());
   }

   if (!aUrl.isLocalFile())
      KIO::NetAccess::removeTempFile(fileName);

   return ok;
}


void MainWindow::discExport()
{
   QString filter = QString("*.koad|%1\n*.txt|%2")
      .arg(i18n("KoverArtist Discs"))
      .arg(i18n("Text Files"));

   KURL url = KFileDialog::getSaveURL(QString::null, filter, this,
                                      i18n("Export disc")+" - KoverArtist");
   if (url.isEmpty()) return;

   int id = mIdActiveDisc;
   if (id<0) id = 0;
   KoverArtist::Disc& disc = mProject.disc(id);

   discExport(disc, url);
}


bool MainWindow::discExport(const KoverArtist::Disc& aDisc, const KURL& aUrl)
{
   QString filename;
   KTempFile tempFile;
   tempFile.setAutoDelete(true);

   if (aUrl.isLocalFile()) filename = aUrl.path();
   else filename = tempFile.name();

   if (!aDisc.saveFile(filename)) return false;

   if (!aUrl.isLocalFile())
   {
      if (!KIO::NetAccess::upload(filename, aUrl, this))
         KMessageBox::error(this, KIO::NetAccess::lastErrorString());
   }

   return true;
}


void MainWindow::searchCoverImages()
{
   QString search = mEdtProjTitle->text()+" "+mEdtProjSubTitle->text();
   search = search.stripWhiteSpace().simplifyWhiteSpace();
   search.replace(' ', '+');

   if (search.isEmpty())
   {
      KMessageBox::error(this, i18n("Please set a title and try again."));
      mEdtProjTitle->setFocus();
      mTabBar->setCurrentTab(mTabProject);
      return;
   }

   KMessageBox::information(this,
         i18n("The cover search is performed by querying an "
         "internet search engine for images, using the "
         "title and subtitle as keywords. You can then "
         "drag&drop the selected image(s) from the browser "
         "into the KoverArtist window."),
         i18n("Search For Cover Images")+" - KoverArtist",
         "search_cover_images", 0);

   kapp->invokeBrowser(KoverArtist::Settings::instance()
      ->coverImageSearchUrl.arg(search));

   changeStatusbar(i18n("Starting external browser..."));
}


void MainWindow::numDiscsChanged()
{
   int i, idx, tabs = mTabBar->count();

   int numDiscs = mProject.count();
   mEdtNumDiscs->setText(QString("%1").arg(numDiscs));

   while (tabs-mNumTabsNoDisc > numDiscs)
      mTabBar->removeTab(mTabBar->tabAt(--tabs));

   while (tabs-mNumTabsNoDisc < numDiscs)
   {
      idx = ++tabs - mNumTabsNoDisc;
      mTabBar->addTab(new QTab(BarIconSet("cd"), QString("%1").arg(idx)));
   }

   // Set identifers of the other tabs
   mTabBar->tabAt(0)->setIdentifier(mTabStack->id(mBaseProject));
   mTabBar->tabAt(1)->setIdentifier(mTabStack->id(mBaseOptions));

   // Set identifers of the disc tabs
   tabs = mTabBar->count();
   for (i=mNumTabsNoDisc,idx=KOA_TABID_DISC; i<tabs; ++i,++idx)
      mTabBar->tabAt(i)->setIdentifier(idx);

   modified();
   updateDiscFromProject();
}


void MainWindow::slotDiscsInc()
{
   int num = mProject.count()+1;
   if (num<=999)
   {
      mProject.resize(num);
      numDiscsChanged();
      modified();
   }
}


void MainWindow::slotDiscsDec()
{
   int num = mProject.count()-1;
   if (num>=1)
   {
      mProject.resize(num);
      numDiscsChanged();
      modified();
   }
}


void MainWindow::slotDiscsEdited()
{
   int num = mEdtNumDiscs->text().toInt();
   if (num<1) num = 1;
   else if (num>999) num = 999;
   if (num!=mProject.count())
   {
      mProject.resize(num);
      numDiscsChanged();
      modified();
   }
}


void MainWindow::switchToTab(int aId)
{
   bool discActive = false;

   if (aId>=KOA_TABID_DISC)
   {
      discActive = true;
      mIdActiveDisc = aId-KOA_TABID_DISC;
      updateDiscFromProject();
      mTabStack->raiseWidget(KOA_TABID_DISC);
      changeCaption();
   }
   else
   {
      mIdActiveDisc = -1;
      mTabStack->raiseWidget(aId);
   }

   mAction[ACT_DISC_NEW]->setEnabled(true);
   mAction[ACT_DISC_DELETE]->setEnabled(discActive && mProject.count()>1);
   mAction[ACT_DISC_IMPORT]->setEnabled(discActive);
#ifdef ENABLE_AUDIOCD
   mAction[ACT_DISC_IMPORT_CDDB]->setEnabled(mProject.count()>0);
#endif /*ENABLE_AUDIOCD*/
   mAction[ACT_DISC_EXPORT]->setEnabled(discActive);
   mAction[ACT_DISC_CLEAR]->setEnabled(discActive);
   mAction[ACT_DISC_FRONT]->setEnabled(discActive && mIdActiveDisc>0);
   mAction[ACT_DISC_BACK]->setEnabled(discActive && mIdActiveDisc<mProject.count()-1);
}


void MainWindow::optionsPreferences()
{
   // popup some sort of preference dialog, here
   KoverArtistPreferences dlg;
   if (dlg.exec())
   {
      // redo your settings
   }
}


void MainWindow::changeStatusbar(const QString& text)
{
   // display the text on the statusbar
   statusBar()->message(text);
   mClearStatusbarTimer->start(5000, true);
}


void MainWindow::changeCaption(const QString& text)
{
   bool mod = mProject.isModified();
   QString str = text.isNull() ? mProject.url().fileName() : text;
   if (str.isEmpty()) str = i18n("unnamed.koap");
   setCaption(kapp->makeStdCaption(str, true, mod));
   mAction[ACT_PROJ_SAVE]->setEnabled(mod);
   mAction[ACT_PROJ_REVERT]->setEnabled(mProject.url().isValid());
}


void MainWindow::scheduleUpdatePreview(int aTmout)
{
   if (mRendering) mRender->abortRendering();
   mUpdateTimer->stop();
   mUpdateTimer->start(aTmout>0 ? aTmout : 250, true);
}


void MainWindow::updatePreview(int aTmout)
{
   mUpdPreviewFront = true;
   mUpdPreviewBack = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewFront(int aTmout)
{
   mUpdPreviewFront = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewBack(int aTmout)
{
   mUpdPreviewBack = true;
   scheduleUpdatePreview(aTmout);
}


void MainWindow::updatePreviewNow()
{
   mUpdPreviewFront = true;
   mUpdPreviewBack = true;
   updatePreviewTimeout();
}


void MainWindow::updatePreviewTimeout()
{
   if (!isUpdatesEnabled() || (!mUpdPreviewFront && !mUpdPreviewBack))
      return;

   if (mRendering)
   {
      scheduleUpdatePreview();
      return;
   }

//    std::cout<<"Updating preview ";
//    if (mUpdPreviewFront) std::cout<<"front ";
//    if (mUpdPreviewBack) std::cout<<"back";
//    std::cout<<std::endl;

   mRendering = true;
   qApp->setOverrideCursor(Qt::BusyCursor);

   int wp = mPreview->contentsRect().width()-4;
   int hp = mPreview->contentsRect().height()-4;

   if (mPixPreview.width()!=wp || mPixPreview.height()!=hp)
   {
      mPixPreview = QPixmap(wp, hp);
      mUpdPreviewFront = true;
      mUpdPreviewBack = true;
   }

   if (mUpdPreviewFront && mUpdPreviewBack)
      mPixPreview.fill(mPreview->backgroundColor());

   mRender->setEnabledFront(mUpdPreviewFront);
   mRender->setEnabledBack(mUpdPreviewBack);

   mRender->render(&mPixPreview, true);
   mPreview->setPixmap(mPixPreview);

   qApp->restoreOverrideCursor();
   mUpdPreviewFront = false;
   mUpdPreviewBack = false;
   mRendering = false;

   if (mUpdateTimer->isActive())
   {
      mUpdateTimer->stop();
      mUpdateTimer->start(500, true);
   }
}


void MainWindow::slotTitleChanged()
{
   const QString& str = mEdtProjTitle->text();
   if (str==mProject.title()) return;

   mProject.setTitle(str);
   modified();
   updatePreview(800);
}


void MainWindow::slotSubTitleChanged()
{
   const QString& str = mEdtProjSubTitle->text();
   if (str==mProject.subTitle()) return;

   mProject.setSubTitle(str);
   modified();
   updatePreview(800);
}


void MainWindow::slotDiscTitleChanged()
{
   if (mIdActiveDisc<0) return;

   KoverArtist::Disc& disc = mProject.disc(mIdActiveDisc);
   const QString& str = mEdtDiscTitle->text();
   if (disc.title()==str) return;

   disc.setTitle(mEdtDiscTitle->text());
   modified();
   updatePreviewBack(800);
}


void MainWindow::slotDiscContentsChanged()
{
   if (mIdActiveDisc<0) return;

   KoverArtist::Disc& disc = mProject.disc(mIdActiveDisc);
   disc.entries() = mElbDiscContents->items();

   modified();
   updatePreviewBack(800);
}


void MainWindow::slotCaseActivated(int aIndex)
{
   if (aIndex<0) aIndex = 0;
   mProject.setBox(KoverArtist::Case::cases()[aIndex]);
   modified();
   updatePreview();
}


void MainWindow::setFrontImageWrap(bool aWrap)
{
   mProject.setImgFrontWrapAround(aWrap);

   mImcBack->setEnabled(!aWrap);

   modified();
   updatePreview();
}


void MainWindow::setFrontTitleHide(bool aHide)
{
   mProject.setFrontTitleHide(aHide);
   modified();
   updatePreviewFront();
}


void MainWindow::setBackgroundColor(const QColor& c)
{
   mProject.setBackgroundColor(c);
   modified();
   updatePreview();
}


void MainWindow::setOutlineColor(const QColor& c)
{
   mProject.setOutlineColor(c);
   modified();
   updatePreview();
}


void MainWindow::redisplay()
{
   updateFromProject();
   updatePreviewNow();
}


#include "mainwindow.moc"
