/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// textstyleconfig.cpp

#include "textstyleconfig.h"
#include "textstyle.h"

#include <qpushbutton.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qlabel.h>

#include <kcolorbutton.h>
#include <kfontdialog.h>
#include <kiconloader.h>
#include <klocale.h>

#define KATS_BUTTONBLOCK_SPACING 6


TextStyleConfig::TextStyleConfig(KoverArtist::TextStyle* aStyle,
                                 bool aHAlign, bool aVAlign,
                                 QWidget* aParent, const char* aName)
:Inherited(aParent, aName)
,mStyle(aStyle)
,mBtnAlignLeft(0)
,mBtnAlignHCenter(0)
,mBtnAlignRight(0)
,mBtnAlignTop(0)
,mBtnAlignVCenter(0)
,mBtnAlignBottom(0)
,mBtnColor(0)
{
   QWidget *base = this;
   QPushButton *btn;

   QHBoxLayout *box = new QHBoxLayout(base, 4, 2, "box");

   btn = new QPushButton(BarIconSet("fonts"), 0, base, "btn-title-font");
   btn->setFixedSize(btn->sizeHint());
   QToolTip::add(btn, i18n("Change the font"));
   connect(btn, SIGNAL(clicked()), SLOT(changeFont()));
   box->addWidget(btn);

   mBtnColor = new KColorButton(Qt::black, base, "btn-title-text-color");
   mBtnColor->setFixedSize(mBtnColor->sizeHint());
   connect(mBtnColor, SIGNAL(changed(const QColor&)), SLOT(setColor(const QColor&)));
   QToolTip::add(mBtnColor, i18n("Change the text color"));
   box->addWidget(mBtnColor);

   box->addSpacing(KATS_BUTTONBLOCK_SPACING);

   if (aHAlign)
   {
      mBtnAlignLeft = createButton("text_left", base, SLOT(alignLeft()), box);
      mBtnAlignLeft->setToggleButton(true);

      mBtnAlignHCenter = createButton("text_center", base, SLOT(alignHCenter()), box);
      mBtnAlignHCenter->setToggleButton(true);

      mBtnAlignRight = createButton("text_right", base, SLOT(alignRight()), box);
      mBtnAlignRight->setToggleButton(true);

      box->addSpacing(KATS_BUTTONBLOCK_SPACING);
   }

   if (aVAlign)
   {
      mBtnAlignTop = createButton("1uparrow", base, SLOT(alignTop()), box);
      mBtnAlignTop->setToggleButton(true);

      mBtnAlignVCenter = createButton("move", base, SLOT(alignVCenter()), box);
      mBtnAlignVCenter->setToggleButton(true);

      mBtnAlignBottom = createButton("1downarrow", base, SLOT(alignBottom()), box);
      mBtnAlignBottom->setToggleButton(true);
   }

//    box->addStretch(1000);
   box->activate();
}


TextStyleConfig::~TextStyleConfig()
{
}


void TextStyleConfig::update()
{
   mBtnColor->setColor(mStyle->color());

   if (mBtnAlignLeft)
   {
      int hf = (mStyle->alignment() & Qt::AlignHorizontal_Mask);
      mBtnAlignLeft->setOn(hf==Qt::AlignLeft);
      mBtnAlignHCenter->setOn(hf==Qt::AlignHCenter);
      mBtnAlignRight->setOn(hf==Qt::AlignRight);
   }

   if (mBtnAlignTop)
   {
      int vf = (mStyle->alignment() & Qt::AlignVertical_Mask);
      mBtnAlignTop->setOn(vf==Qt::AlignTop);
      mBtnAlignVCenter->setOn(vf==Qt::AlignVCenter);
      mBtnAlignBottom->setOn(vf==Qt::AlignBottom);
   }
}


QPushButton* TextStyleConfig::createButton(const char* aIcon, QWidget* aParent,
                                           const char* aSlot, QBoxLayout* aBox) const
{
   QPushButton *btn = new QPushButton(BarIconSet(aIcon), 0, aParent);
   btn->setFixedSize(btn->sizeHint());
   connect(btn, SIGNAL(clicked()), aSlot);
   if (aBox) aBox->addWidget(btn);
   return btn;
}


void TextStyleConfig::changeFont()
{
   QFont font = mStyle->font();
   int ret = KFontDialog::getFont(font, false, this, false, 0);
   if (ret==KFontDialog::Accepted)
   {
      mStyle->setFont(font);
      if (isUpdatesEnabled()) emit changed();
   }
}


void TextStyleConfig::setColor(const QColor& c)
{
   mStyle->setColor(c);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignLeft()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignLeft));
   mBtnAlignHCenter->setOn(false);
   mBtnAlignRight->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignHCenter()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignHCenter));
   mBtnAlignLeft->setOn(false);
   mBtnAlignRight->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignRight()
{
   int f = int(mStyle->alignment() & Qt::AlignVertical_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignRight));
   mBtnAlignLeft->setOn(false);
   mBtnAlignHCenter->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignTop()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignTop));
   mBtnAlignVCenter->setOn(false);
   mBtnAlignBottom->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignVCenter()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignVCenter));
   mBtnAlignTop->setOn(false);
   mBtnAlignBottom->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


void TextStyleConfig::alignBottom()
{
   int f = int(mStyle->alignment() & Qt::AlignHorizontal_Mask);
   mStyle->setAlignment(Qt::AlignmentFlags(f | Qt::AlignBottom));
   mBtnAlignTop->setOn(false);
   mBtnAlignVCenter->setOn(false);
   if (isUpdatesEnabled()) emit changed();
}


#include "textstyleconfig.moc"
