#!/bin/sh
#
# Download and install a known-good version of the ukai.ttf font for Wine. This
# works around a problem with some versions of the font. For more information,
# see: http://www.codeweavers.com/support/wiki/faq/fontissues
#
# Copyright 2008 Google Inc. All Rights Reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# * Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
# * Redistributions in binary form must reproduce the above
#   copyright notice, this list of conditions and the following disclaimer
#   in the documentation and/or other materials provided with the
#   distribution.
# * Neither the name of Google Inc. nor the names of its
#   contributors may be used to endorse or promote products derived from
#   this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

# Include common functions.
export SCRIPTDIR=$(readlink -f $(dirname "$0"))
. "$SCRIPTDIR/common.sh"

FONT="ukai.ttf"
PKGBASE="ttf-arphic-ukai-0.1.20060108"
FONTPKG="$PKGBASE.tar.gz"
FONTURL="http://dl.google.com/linux/src/$FONTPKG"

usage() {
  SEP="-------------------------------------------------------"
  prompt_info "
$SEP
This program attempts to install a Wine-friendly version of the ukai.ttf font,
to avoid the problems described in:
http://www.codeweavers.com/support/wiki/faq/fontissues
$SEP
Usage: $(basename $0) [-n] [-p message] [-c config_file]  [-v] -d font_directory
  -d  Directory where the font should be installed.
  -c  Config file to store preferences (i.e. 'don't ask again')
  -n  Don't actually install, just show what would happen.
  -p  Prompt to display before installing.
  -v  Be verbose, even when no actions taken or errors encountered.
  -h  Show this help message.
$SEP"
}

trap cleanup HUP INT QUIT ABRT TERM EXIT
cleanup() {
  set +ex
  close_progress
}

install_font() {
  WORKDIR=$(mktemp -d)
  if [ $? -gt 0 ]; then
    prompt_err "Couldn't create working directory."
    return 1
  fi
  SAVEFILE="$WORKDIR/$FONTPKG"

  do_fetch "$FONTURL" "$SAVEFILE"
  if [ $? -ne 0 ]; then
    return 1
  fi

  prompt_progress "Installing font.
Enter your administrative password if prompted."
  if [ ! $DRYRUN ]; then
    tar -C "$WORKDIR" -zxf "$SAVEFILE"
    FONTFILE="$WORKDIR/$PKGBASE/$FONT"
    if [ ! -f "$FONTFILE" ]; then
      prompt_err "Failed to extract $FONT from $SAVEFILE."
      return 1
    fi
    chmod 644 "$FONTFILE"
    run_as_root cp -fvp "$FONTFILE" "$FONTDIR"/
    if [ $? -ne 0 ]; then
      return 1
    fi
  fi
  # cleanup
  rm -rf "$WORKDIR"
}

is_supressed() {
  PROMPTID=$1
  grep -qs "$PROMPTID 1 $(hostname -f)" "$CFGFILE"
  return $?
}

#=========
# MAIN
#=========
while getopts ":nc:d:p:vh" OPTNAME
do
  case $OPTNAME in
    n )
      enable_dry_run
      ;;
    c )
      CFGFILE="$OPTARG"
      ;;
    d )
      FONTDIR="$OPTARG"
      ;;
    p )
      PROMPTMSG="$OPTARG"
      ;;
    v )
      VERBOSE=1
      ;;
    h )
      usage
      exit 0
      ;;
    \: )
      prompt_err "'-$OPTARG' needs an argument."
      usage
      exit 1
      ;;
    * )
      prompt_err "invalid command-line option: $OPTARG"
      usage
      exit 1
      ;;
  esac
done
shift $(($OPTIND - 1))

if [ ! -d "$FONTDIR" ]; then
  prompt_err "You must specify an existing directory with the '-d' option."
  usage
  exit 1
fi

# No point going farther if we don't have a way to download the files we need.
prep_fetch
if [ $? -gt 0 ]; then
  exit 1
fi

# Now that we're past general usage errors, we'll allow GUI output.
if [ ! $DRYRUN ]; then
  enable_gui_output
fi

if [ "$PROMPTMSG" ]; then
  PROMPTID=$(echo "$PROMPTMSG" | md5sum | cut -d ' ' -f 1)
  is_supressed "$PROMPTID"
  if [ $? -eq 0 ]; then
    exit 0
  fi
  # Don't offer the "don't ask again" option if we can't save the preference.
  touch "$CFGFILE" > /dev/null 2>&1
  if [ $? -eq 0 ]; then
    prompt_yesno_never "$PROMPTMSG"
  else
    prompt_yesno "$PROMPTMSG"
  fi
  RES=$?
  if [ $RES -gt 0 ]; then
    if [ $RES -eq 2 ]; then
      echo "$PROMPTID 1 $(hostname -f)" >> "$CFGFILE"
    fi
    exit 1
  fi
fi

install_font
exit $?
