/***************************************************************************
 *   Copyright (C) 2005 by Antonio Fasolato                                *
 *   Antonio.Fasolato@poste.it                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "pixelview.h"
#include "pixelview.moc"
#include "qmessagebox.h"

#include <math.h>

PixelView::PixelView(QWidget *parent, const char *name, WFlags f )
 : QScrollView(parent, name, f)
{
	//gui construction
	setVScrollBarMode(QScrollView::AlwaysOn);
	setHScrollBarMode(QScrollView::AlwaysOn);
	label = new QLabel(viewport());
	label->hide();
	label->setGeometry(0,0,0,0);
	label->clear();
	addChild(label); //making the label visible in THIS (QScrollView)
	
	theThresholdDialog = new thresholdDialog(this);
	connect( theThresholdDialog, SIGNAL( valuesChanged(bool, double, bool) ), this, SLOT( setOptions(bool, double, bool) ) );
	
	//standaed values
	internImagePresent = FALSE;
	internThreshold = 128;
	internThresholdPercent = 50;
	internThresholdingEnabled = FALSE;
}


PixelView::~PixelView()
{
}


/*!
 *   Deletes the image and clears the widget.
 */
void PixelView::deleteImage()
{
    internImagePresent = FALSE;
    label->hide();
    label->setGeometry(0,0,0,0);
    label->clear();
    originalImage.reset();
    changedImage.reset();
}


/*!
    Returns, if aktually an image is present.
 */
bool PixelView::imagePresent() const
{
    return internImagePresent;
}


/*!
    \fn PixelView::displayedImage()
    Returns the internally used QImage, in which the displayed image is stored.
 */
QImage PixelView::displayedImage() const
{
    return changedImage;
}


/*!
    \fn PixelView::changeToImage(QImage &image)
    Produces an internal copy of image and uses this.
 */
void PixelView::changeToImage(const QImage &image)
{
    deleteImage();
    internThreshold = 128;
    internThresholdPercent = 50;
    internThresholdingEnabled = FALSE;
    internInvert = FALSE;
    
    originalImage = image;
    originalImage.detach();
    changedImage = originalImage;
    internImagePresent = TRUE;
    
    //calculate zoom for fitting image in window and paint
    fitToPage();
}


/*!
    \fn PixelView::thresholdingEnabled() const
 */
bool PixelView::thresholdingEnabled() const
{
    return internThresholdingEnabled;
}


/*!
    \fn PixelView::setThresholdingEnabled(bool enabled)
 */
void PixelView::setThresholdingEnabled(bool enabled)
{
    setOptions(enabled, internThresholdPercent, internInvert);
}


/*!
    \fn PixelView::thresholdPercent() const
 */
double PixelView::thresholdPercent() const
{
    return internThresholdPercent;
}


/*!
    \fn PixelView::setThresholdPercent(double)
 */
void PixelView::setThresholdPercent(double value)
{
    setOptions(internThresholdingEnabled, value, internInvert);
}


/*!
    \fn PixelView::invert() const
 */
bool PixelView::invert() const
{
    return internInvert;
}


/*!
    \fn PixelView::setInvert(bool value)
 */
void PixelView::setInvert(bool value)
{
    setOptions(internThresholdingEnabled, internThresholdPercent, value);
}


void PixelView::setOptions(bool m_thresholdingEnabled, double m_thresholdPercent, bool m_invert)
{

   //correct thresholdPercent to 0 <= thresholdPercent <=100 if necesarry, and save it
   if (m_thresholdPercent < 0) m_thresholdPercent = 0;
   else if (m_thresholdPercent > 100) m_thresholdPercent = 100;   
   internThresholdPercent = m_thresholdPercent;
   
   //initialize variables
	unsigned int m_threshold = static_cast<unsigned int>(round(255*m_thresholdPercent/100));
   
   //do the necessary recalculations
   if (  //if new thresholding is nesecarry. This is the case when
        ( internThresholdingEnabled != m_thresholdingEnabled ) or //thresholding was enabled or disabled OR
        ( internThreshold != m_threshold )                        //another threshold is given.
      )
   {
      internThresholdingEnabled = m_thresholdingEnabled;
      internThreshold = m_threshold;
      internInvert = m_invert;
      
      //calc start
      changedImage = originalImage;
      if (internThresholdingEnabled)
      {
	 changedImage.detach();
         KImageEffect::threshold(changedImage,internThreshold);
         changedImage=changedImage.convertDepth(1,Qt::MonoOnly|Qt::ThresholdDither);
	 if (internInvert) changedImage.invertPixels();
      }
      internUpdatePixmap();
      //calc end
      
   }
   else
      if (internInvert != m_invert)
      {
         internInvert = m_invert;
         if (m_thresholdingEnabled)
         {
            //calc start
            changedImage.invertPixels();
            internUpdatePixmap();
            //calc end
         }
      };
}


/*!
    \fn PixelView::internUpdatePixmap
 */
void PixelView::internUpdatePixmap()
{
   if (internImagePresent) {
	if (internZoom < 100)
		scaledImage=changedImage.smoothScale(originalImage.width()*internZoom/100, originalImage.height()*internZoom/100, QImage::ScaleMin);
	else
		if (internZoom > 100)
			scaledImage=changedImage.scale(originalImage.width()*internZoom/100, originalImage.height()*internZoom/100, QImage::ScaleMin);
		else scaledImage=changedImage;
	label->setGeometry(0,0,0,0);
	label->setPixmap(pixmapIO.convertToPixmap(scaledImage));
	label->setGeometry(0,0,scaledImage.width(),scaledImage.height());
	label->show();
   }
}

/*!
    \fn PixelView::thresholdUI()
 */
void PixelView::thresholdUI()
{
	theThresholdDialog->show(internThresholdingEnabled, internThresholdPercent, internInvert);
}




/*!
    \fn PixelView::fitToPage()
 */
void PixelView::fitToPage()
{
	//extremly unexact!!!
    internZoom = visibleWidth()*100/originalImage.width();
    unsigned int temp = visibleHeight()*100/originalImage.height();
    if (temp < internZoom) internZoom=temp;
    
    if (internZoom < 1) internZoom=1;
    
    internUpdatePixmap();
}


/*!
    \fn PixelView::fitToWidth()
 */
void PixelView::fitToWidth()
{
    internZoom = visibleWidth()*100/originalImage.width();
    if (internZoom < 1) internZoom=1;
    internUpdatePixmap();
}


/*!
    \fn PixelView::fitToHeight()
 */
void PixelView::fitToHeight()
{
    internZoom = visibleHeight()*100/originalImage.height();
    if (internZoom < 1) internZoom=1;
    internUpdatePixmap();
}


/*!
    \fn PixelView::zoomIn()
 */
void PixelView::zoomIn()
{
	unsigned int newZoom=static_cast<unsigned int>(internZoom*1.25);
    if (newZoom > internZoom) internZoom=newZoom;
    else internZoom=internZoom+1;  //possible for little values
    
    internUpdatePixmap();
}


/*!
    \fn PixelView::zoomOut()
 */
void PixelView::zoomOut()
{
	unsigned int newZoom=static_cast<unsigned int>(internZoom*0.8);
    if (newZoom > 0)
    { internZoom=newZoom;
      internUpdatePixmap();
    }
}


/*!
    \fn PixelView::zoom()
 */
void PixelView::zoom()
{
    /// @todo implement me
}
