#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "phish.h"
#include "phish_settings.h"

#define BUF_SIZE 1024

#define MODE_FIELD              "Mode"
#define ONLINE_MODE_VALUE       "Online"
#define OFFLINE_MODE_VALUE      "Offline"
#define SITE_QUERY_URL_FIELD    "SiteQuery"
#define COUNTRY_QUERY_URL_FIELD "CountryQuery"
#define REPORT_SITE_URL_FIELD   "SiteReport"
#define REMOTE_XML_URL_FIELD    "RemoteXMLURL"
#define REMOTE_XML_ETAG_FIELD   "RemoteXMLETag"
#define LOCAL_XML_FILE_FIELD    "SitesXMLFile"
#define SAFE_LIST_FILE_FIELD    "SafeListFile"

#define DEFAULT_MODE              PHISH_OFFLINE_MODE
#define DEFAULT_SITE_QUERY_URL    "http://opdb.berlios.de/cgi-bin/query.pl"
#define DEFAULT_COUNTRY_QUERY_URL "http://opdb.berlios.de/cgi-bin/country.pl"
#define DEFAULT_REPORT_SITE_URL   "http://opdb.berlios.de/cgi-bin/report.pl"
#define DEFAULT_REMOTE_XML_URL    "http://opdb.berlios.de/opdbsites.xml"
#define DEFAULT_LOCAL_XML_FILE    "opdbsites.xml"
#define DEFAULT_SAFE_LIST_FILE    "safesites"


void phish_settings_init(phish_settings_t *settings)
{
  settings->mode = -1;
  settings->site_query_url = NULL;
  settings->country_query_url = NULL;
  settings->report_site_url = NULL;
  settings->remote_xml_url = NULL;
  settings->remote_xml_etag = NULL;
  settings->local_xml_file = NULL;
  settings->safe_list_file = NULL;
}

void phish_settings_free(phish_settings_t *settings)
{
  free(settings->site_query_url);
  free(settings->country_query_url);
  free(settings->report_site_url);
  free(settings->remote_xml_url);
  free(settings->remote_xml_etag);
  free(settings->local_xml_file);
  free(settings->safe_list_file);
}

phish_mode_t phish_settings_runningMode(phish_settings_t *settings)
{
  if (settings->mode == -1)
    return DEFAULT_MODE;
  else
    return settings->mode;
}

const char *phish_settings_siteQueryURL(phish_settings_t *settings)
{
  if (settings->site_query_url == NULL)
    return DEFAULT_SITE_QUERY_URL;
  else
    return settings->site_query_url;
}

const char *phish_settings_countryQueryURL(phish_settings_t *settings)
{
  if (settings->country_query_url == NULL)
    return DEFAULT_COUNTRY_QUERY_URL;
  else
    return settings->country_query_url;
}

const char *phish_settings_reportSiteURL(phish_settings_t *settings)
{
  if (settings->report_site_url == NULL)
    return DEFAULT_REPORT_SITE_URL;
  else
    return settings->report_site_url;
}

const char *phish_settings_remoteXMLURL(phish_settings_t *settings)
{
  if (settings->remote_xml_url == NULL)
    return DEFAULT_REMOTE_XML_URL;
  else
    return settings->remote_xml_url;
}

const char *phish_settings_remoteXMLETag(phish_settings_t *settings)
{
  return settings->remote_xml_etag;
}

const char *phish_settings_localXMLFile(phish_settings_t *settings)
{
  if (settings->local_xml_file == NULL)
    return DEFAULT_LOCAL_XML_FILE;
  else
    return settings->local_xml_file;
}

const char *phish_settings_safeListFile(phish_settings_t *settings)
{
  if (settings->safe_list_file == NULL)
    return DEFAULT_SAFE_LIST_FILE;
  else
    return settings->safe_list_file;
}

void phish_settings_setRunningMode(phish_settings_t *settings,
                                   phish_mode_t mode)
{
  settings->mode = mode;
}

static void setSetting(char **setting, const char *value)
{
  free(*setting);
  *setting = malloc(strlen(value) + 1);
  strcpy(*setting, value);
}

void phish_settings_setSiteQueryURL(phish_settings_t *settings,
                                    const char *url)
{
  setSetting(&settings->site_query_url, url);
}

void phish_settings_setCountryQueryURL(phish_settings_t *settings,
                                       const char *url)
{
  setSetting(&settings->country_query_url, url);
}

void phish_settings_setReportSiteURL(phish_settings_t *settings,
                                     const char *url)
{
  setSetting(&settings->report_site_url, url);
}

void phish_settings_setRemoteXMLURL(phish_settings_t *settings,
                                    const char *url)
{
  setSetting(&settings->remote_xml_url, url);
}

void phish_settings_setRemoteXMLETag(phish_settings_t *settings,
                                     const char *etag)
{
  setSetting(&settings->remote_xml_etag, etag);
}

void phish_settings_setLocalXMLFile(phish_settings_t *settings,
                                    const char *path)
{
  setSetting(&settings->local_xml_file, path);
}

void phish_settings_setSafeListFile(phish_settings_t *settings,
                                    const char *path)
{
  setSetting(&settings->safe_list_file, path);
}

static void readSettingsEntry(phish_settings_t *settings,
                              const char *line)
{
  int pos = 0;
  int eq_pos = -1;

  while (line[pos] != '\n' && line[pos] != '\0')
  {
    /* find index of '=' separator */
    if (line[pos] == '=')
    {
      eq_pos = pos;
      break;
    }
    
    pos++;
  }
  
  if (eq_pos == -1)
    return;

  if (strncmp(line, MODE_FIELD, eq_pos - 1) == 0)
  {
    if (strcmp(line + eq_pos + 1, ONLINE_MODE_VALUE) == 0)
      phish_settings_setRunningMode(settings, PHISH_ONLINE_MODE);
    else if (strcmp(line + eq_pos + 1, OFFLINE_MODE_VALUE) == 0)
      phish_settings_setRunningMode(settings, PHISH_OFFLINE_MODE);
  }
  else if (strncmp(line, SITE_QUERY_URL_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setSiteQueryURL(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, COUNTRY_QUERY_URL_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setCountryQueryURL(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, REPORT_SITE_URL_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setReportSiteURL(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, REMOTE_XML_URL_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setRemoteXMLURL(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, REMOTE_XML_ETAG_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setRemoteXMLETag(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, LOCAL_XML_FILE_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setLocalXMLFile(settings, line + eq_pos + 1);
  }
  else if (strncmp(line, SAFE_LIST_FILE_FIELD, eq_pos - 1) == 0)
  {
    phish_settings_setSafeListFile(settings, line + eq_pos + 1);
  }
}

phish_result_t phish_settings_load(phish_settings_t *settings,
                                   const char *settings_file_name)
{
  FILE *settings_file;
  char buf[BUF_SIZE];

  settings_file = fopen(settings_file_name, "r");
  if (settings_file != NULL)
  {
    while (fgets(buf, BUF_SIZE, settings_file) != NULL)
    {
      char *nl = strchr(buf, '\n');
      if (nl != NULL)
        *nl = '\0'; /* remove newline character */
      
      readSettingsEntry(settings, buf);
    }

    fclose(settings_file);
  }

  return PHISH_SUCCESS;
}


static void writeSettingsEntry(FILE *settings_file,
                               const char *field, const char *value)
{
  if (value != NULL)
    fprintf(settings_file, "%s=%s\n", field, value);
}

phish_result_t phish_settings_write(phish_settings_t *settings,
                                    const char *settings_file_name)
{

  FILE *settings_file;
  
  if ((settings_file = fopen(settings_file_name, "w")) == NULL)
  {
    return PHISH_ERR_FILE_WRITE;
  }
  
  switch(settings->mode)
  {
    case PHISH_ONLINE_MODE:
      writeSettingsEntry(settings_file, MODE_FIELD, ONLINE_MODE_VALUE);
      break;
    case PHISH_OFFLINE_MODE:
      writeSettingsEntry(settings_file, MODE_FIELD, OFFLINE_MODE_VALUE);
      break;
    default:
      break;
  }

  writeSettingsEntry(settings_file, SITE_QUERY_URL_FIELD,
                           settings->site_query_url);
  
  writeSettingsEntry(settings_file, COUNTRY_QUERY_URL_FIELD,
                           settings->country_query_url);
  
  writeSettingsEntry(settings_file, REPORT_SITE_URL_FIELD,
                           settings->report_site_url);
  
  writeSettingsEntry(settings_file, REMOTE_XML_URL_FIELD,
                           settings->remote_xml_url);
  
  writeSettingsEntry(settings_file, REMOTE_XML_ETAG_FIELD,
                           settings->remote_xml_etag);
  
  writeSettingsEntry(settings_file, LOCAL_XML_FILE_FIELD,
                           settings->local_xml_file);
  
  writeSettingsEntry(settings_file, SAFE_LIST_FILE_FIELD,
                           settings->safe_list_file);

  fclose(settings_file);
  
  return PHISH_SUCCESS;
}

