/***************************************************************************
 *   Copyright (C) 2005 Meni Livne <livne@kde.org>                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef __PHISH_UTIL_NET_H
#define __PHISH_UTIL_NET_H


#include <stdlib.h>

#include "phish.h"


#define HTTP_STATUS_OK           200
#define HTTP_STATUS_NOT_MODIFIED 304


/** Headers returned from HTTP reply */
typedef struct
{
  /** Status code, e.g. 200 for O.K */
  int status_code;

  /** Length of content */
  size_t content_length;

  /** ETag field */
  char *etag;
} phish_util_http_headers_t;


/** Resolves a hostname to its IP address.
 *  @param host hostname to look up
 *  @param ip pointer to string which will be allocated and assigned with
 *            corresponding IP address
 */
phish_result_t phish_util_hostToIP(const char *host, char **ip);

/** Initiates a TCP connection to a server.
 *  @param sock socket to connect on
 *  @param server server to connect to
 *  @param port port on server to connect to
 */
phish_result_t phish_util_tcpConnect(int sock, const char *server, int port);

/** Read a line from a connected socket up to a new line character.
 *  @param sock socket to read from
 *  @param line buffer to fill with data read. Carriage return and new line
 *              will be read but won't be added to buffer
 *  @param line_length size of line argument
 */
phish_result_t phish_util_sockReadLine(int sock, char *line,
                                       size_t line_length);

/** Writes a line to a connected socket, including a carriage return and line
 *  feed at end.
 *  @param sock socket to write to
 *  @param line null-terminated string to write.
 */
phish_result_t phish_util_sockWriteLine(int sock, const char *line);

/** Sends an HTTP GET request to a socket connected to an HTTP server.
 *  @param sock socket connected to HTTP server to send request to
 *  @param path path requested
 *  @param version version string of HTTP request (e.g. "1.1")
 *  @param host "Host" field of request, NULL if not to send such a field
 *  @param user_agent User Agent field of request, NULL if not to send such
 *                    a field
 *  @param referrer Referrer field of request, NULL of not to send such a field
 *  @param if_none_match "If-None-Match" field of request, NULL of not to send
 *                       such a field
 *  @param keep-alive 1 if to keep connection to server alive after request, 0
 *                    if to close connection after request
 */
phish_result_t phish_util_httpGet(int sock, const char *path,
                                  const char *version, const char *host,
                                  const char *user_agent,
                                  const char *referrer,
                                  const char *if_none_match, int keep_alive);

/** Reads headers of HTTP reply from a socket connected to an HTTP server and
 *  parses them.
 *  @param sock socket connected to HTTP server to read reply from
 *  @param result pointer to phish_util_http_headers_t structure which will
 *                be filled results of reply read
 */
phish_result_t phish_util_parseHTTPReply(int sock,
                                         phish_util_http_headers_t *result);


#endif /* __PHISH_UTIL_NET_H */

