/* Implementation of Template functions from template.h
*
* Written 2/8/03 by Clay Dowling
* Lazarus Internet Development
*/

#ifdef __WIN32__
/*#include "libtemplatedll.h"*/
#include <windows.h>
#endif

#include <stdlib.h>
#include <string.h> 


#include <sys/types.h>
#include <sys/stat.h>


#include "template.h"

struct tpl_t* tpl_new() {

  struct tpl_t* t;
  
  t = (struct tpl_t*)calloc(1, sizeof(struct tpl_t));
  return t;
  
}

void tpl_delete(struct tpl_t* t) {
  
  struct tpl_node_t* n;
  struct tpl_node_t* tmp;
  
  if (!t) return;
  if (t->name) {
    fprintf(stderr, "tpl_delete - name\n");
    free(t->name);
    t->name = 0;
  }
  for(n = t->nodes; n; n = tmp) {
    tmp = n->next;
    tpl_node_delete(n);
  }
  fprintf(stderr, "tpl_delete\n");
  free(t);
 
}

struct tpl_node_t* tpl_node_new() {
  
  struct tpl_node_t* tn;
  
  tn = (struct tpl_node_t*)calloc(1, sizeof(struct tpl_node_t));
  return tn;
  
}

void tpl_node_delete(struct tpl_node_t* n) {
  
  if (!n) return;
  if (n->value) {
    free(n->value);
    n->value = 0;
  }
  free(n);
  
}

DLLIMPORT struct tpl_t* tpl_compile(char* buffer) {
  
  struct tpl_t* tmpl;
  struct tpl_node_t* node;
  struct tpl_node_t* last;
  char* node_begin;
  char* node_end;
  char* space;
  char target;
  
  tmpl = tpl_new();
  if (!tmpl) return 0;
  
  node_begin = buffer;
  while(node_begin) {
    node = tpl_node_new();
    if (!node) return 0;
    if (*node_begin == '{') {
      space = strpbrk(node_begin, " \t\n\r");
      node_end = strchr(node_begin, '}');
      if (!node_end) node->type = NT_LITERAL;
      if (space && space < node_end)
	node->type = NT_LITERAL;
      else {
	node->type = NT_ELEMENT;
	node_begin++;
      }
    }
    else
      node->type = NT_LITERAL;
    
    if (node->type == NT_LITERAL)
      target = '{';
    else
      target = '}';
    
    node_end = strchr(node_begin + 1, target);
    if (!node_end) node->length = strlen(node_begin);
    else 
      node->length = node_end - node_begin;
    node->length++;
    
    node->value = (char*)calloc(1, node->length);
    strncpy(node->value, node_begin, node->length - 1);
    
    /* Recalculating the length because it gets effed up when
     * we have NT_LITERAL nodes under Linux */
    node->length = strlen(node->value);
    
    if (tmpl->nodes) {
      last->next = node;
      last = node;
    }
    else {
      tmpl->nodes = node;
      last = node;
    }
    tmpl->count++;
    
    if (node_end) {
      if (*node_end == '}')
	node_begin = node_end + 1;
      else
	node_begin = node_end;
    }
    else
      node_begin = 0;
    
  }
  
  return tmpl;
  
}

DLLIMPORT struct tpl_engine* tpl_engine_new() {
  
  struct tpl_engine* t;
  
  t = (struct tpl_engine*)calloc(1, sizeof(struct tpl_engine));
  t->templates = forest_new();
  t->elements = forest_new();
  
  return t;
  
}

DLLIMPORT void tpl_engine_delete(struct tpl_engine* t) {

  forest_delete(t->elements, tpl_element_node_delete);
  forest_delete(t->templates, tpl_delete);
  free(t);
  
}

struct tpl_element_node_t* tpl_element_node_new() {
  
  struct tpl_element_node_t* n;
  
  n = (struct tpl_element_node_t*)calloc(1, sizeof(struct tpl_element_node_t));
  return n;
  
};

void tpl_element_node_delete(struct tpl_element_node_t* n) {
  
  if (!n) return;
  if (n->value) {
    free(n->value);
    n->value = 0;
  }
  /* if (n->name) free(n->name); Don't free this, it's a constant */
  free(n);
  
}

struct tpl_element_node_t* tpl_element_node_join(struct tpl_element_node_t** data) {
  
  struct tpl_element_node_t* n;
  long size = 0;
  int i;
  char* start;
  
  for(i=0; data[i]; i++)
    size += data[i]->length;
  
  n = tpl_element_node_new();
  if (!n) return 0;
  n->value = (char*)calloc(1, size + 1);
  if (!n->value) return 0;
  n->length = size;
  
  start = n->value;
  for(i=0; data[i]; i++) {
    strncpy(start, data[i]->value, data[i]->length);
    start = &start[data[i]->length];
  }
  
  return n;
  
}

int tpl_element_compare(const void* left, const void* right) {
  
  struct tpl_element_node_t* lhs;
  struct tpl_element_node_t* rhs;
  
  lhs = (struct tpl_element_node_t*)left;
  rhs = (struct tpl_element_node_t*)right;
  
  if (lhs->name == 0)
    return 1;
  if (rhs->name == 0)
    return -1;
  
  return strcmp(lhs->name, rhs->name);
  
}

DLLIMPORT void tpl_element_set(struct tpl_engine* t, const char* element, const char* value) {
  
  struct tpl_element_node_t* insnode;
  
  insnode = tpl_element_node_get(t, element);
  if (insnode) {
    if (insnode->value) {
      free(insnode->value);
      insnode->value = 0;
    }
    insnode->length = strlen(value);
    insnode->value = (char*)calloc(1, insnode->length + 1);
    strcpy(insnode->value, value);
  }
  else {
    insnode = tpl_element_node_new();
    insnode->name = (char*)calloc(1, strlen(element) + 1);
    strcpy(insnode->name, element);
    insnode->length = strlen(value);
    insnode->value = (char*)calloc(1, insnode->length + 1);
    strcpy(insnode->value, value);
    tree_insert(t->elements, (void*)insnode, tpl_element_compare);
  }
  
}

struct tpl_element_node_t* tpl_element_node_get(struct tpl_engine* t, const char* element) {
  
  struct tpl_element_node_t seek;
  struct tree_t* leaf;
  struct tpl_element_node_t* result;

  seek.name = (char*)calloc(1, strlen(element) + 1);
  strcpy(seek.name, element);
  leaf = tree_find(t->elements, (void*)&seek, tpl_element_compare);
  if (!leaf) return 0;
  
  result = (struct tpl_element_node_t*)leaf->data;
  return result;
  
}

DLLIMPORT char* tpl_element_get(struct tpl_engine* t, const char* element) {
  
  struct tpl_element_node_t* seek;
  
  seek = tpl_element_node_get(t, element);
  if (seek) return seek->value;
  
  return 0;
  
}

DLLIMPORT int tpl_compare(const void* left, const void* right) {

  struct tpl_t* lhs;
  struct tpl_t* rhs;
  
  lhs = (struct tpl_t*)left;
  rhs = (struct tpl_t*)right;

  return strcmp(lhs->name, rhs->name);

}

struct tpl_element_node_t* tpl_parse_private(struct tpl_engine* t, const char* name) {

  struct tpl_element_node_t** work;
  struct tpl_element_node_t* ins;
  struct tpl_element_node_t* result;
  struct tpl_element_node_t* holder;
  struct tpl_element_node_t elementseek;
  struct tpl_t* tpl;
  struct tpl_t seek;
  struct tpl_node_t* node;
  struct tree_t* leaf;
  int i;
  
  seek.name = (char*)calloc(1, strlen(name) + 1);
  strcpy(seek.name, name);
  leaf = tree_find(t->templates, (void*)&seek, tpl_compare);
  free(seek.name);
  seek.name = 0;
  if (!leaf) {
    t->tplerrno = tpl_not_found;
    return 0;
  }
  tpl = leaf->data;
  if (!tpl) {
    t->tplerrno = tpl_empty;
    return 0;
  }

  work = (struct tpl_element_node_t**)calloc(tpl->count + 1, sizeof(struct tpl_element_node_t*));
  if (!work)
    return 0;
  
  i = 0;
  for(node = tpl->nodes; node; node = node->next) {
    ins = tpl_element_node_new();
    if (node->type == NT_ELEMENT) {
      elementseek.name = node->value;
      holder = tpl_element_node_get(t, node->value);
      if (holder) {
	ins->value = holder->value;
	ins->length = holder->length;
      }
      else {
	ins->length = 0;
	ins->value = "";
      }
    }
    else {
      ins->value = node->value;
      ins->length = node->length;
    }
    work[i] = ins;
    i++;
  }
  
  /* Terminate the list */
  work[i] = 0;
  result = tpl_element_node_join(work);
  
  /* Clean up our workspace */
  for(i=0; work[i]; i++) {
    free(work[i]);
    work[i] = 0;
  }
  
  return result;

}

DLLIMPORT void tpl_parse(struct tpl_engine* t, const char* name, const char* newname, int append) {
  
  struct tpl_element_node_t* newelement;
  struct tpl_element_node_t* cradle[3];
  struct tpl_element_node_t* joinresult;
  struct tree_t* leaf;
  struct tpl_element_node_t* found=0;
  struct tpl_element_node_t seek;
  
  newelement = tpl_parse_private(t, name);
  if (!newelement) return;

  newelement->name = (char*)calloc(1, strlen(newname) + 1);
  strcpy(newelement->name, newname);
  seek.name = newelement->name;
  leaf = tree_find(t->elements, (void*)&seek, tpl_element_compare);
  if (leaf) found = leaf->data;
  if (!found) {
    tree_insert(t->elements, (void*)newelement, tpl_element_compare);
    return;
  }
  
  if (append) {
    cradle[0] = found;
    cradle[1] = newelement;
    cradle[2] = 0;
    joinresult = tpl_element_node_join(cradle);
    joinresult->name = (char*)calloc(1, strlen(newname) + 1);
    strcpy(joinresult->name, newname);
    leaf->data = joinresult;
    tpl_element_node_delete(found);
    tpl_element_node_delete(newelement);
  }
  else {
    leaf->data = newelement;
    tpl_element_node_delete(found);
  }
  
}

char* tpl_contents_extract(char* buffer, char** name, char** newstart) {
  
  char* end;
  char* start;
  char* namestart;
  char* nameend;
  char* tagstart;
  char* contents;
  
  tagstart = strstr(buffer, "<template");
  if (!tagstart) return 0;
  
  namestart = strstr(tagstart, "name=\"");
  if (!namestart) return 0;
  namestart += 6;
  nameend = strchr(namestart, '\"');
  if (!nameend) return 0;

  start = strchr(nameend, '>');
  if (!start) return 0;
  end = strstr(start, "</template>");
  if (!end) return 0;
  
  *newstart = end + strlen("</template>");
  *name = (char*)calloc(1, nameend - namestart + 1);
  strncpy(*name, namestart, nameend - namestart);

  contents = (char*)calloc(1, end - start);
  strncpy(contents, start + 1, end - start - 1);

  return contents;  

}

DLLIMPORT void tpl_file_load(struct tpl_engine* t, const char* filename) {

  char* start;
  char* content;
  char* name;
  FILE* fp;
  char* buffer;
  struct tpl_t* tmp;
  struct stat sb;

  if (stat(filename, &sb)) {
    t->tplerrno = tpl_error_external;
    return;
  }

  buffer = (char*)calloc(1, sb.st_size);
  if (!buffer) {
    t->tplerrno = tpl_error_external;
    return;
  }
  fp = fopen(filename, "r");
  fread((void*)buffer, sb.st_size, 1, fp);
  fclose(fp);

  start = buffer;
  content = buffer;
  while (content) {
    content = tpl_contents_extract(start, &name, &start);
    if (content) {
      tmp = tpl_compile(content);
      tmp->name = name;
      tree_insert(t->templates, (void*)tmp, tpl_compare);
    }
  }

  free(buffer);

}

#ifdef __WIN32__
BOOL APIENTRY
DllMain (
  HINSTANCE hInst     /* Library instance handle. */ ,
  DWORD reason        /* Reason this function is being called. */ ,
  LPVOID reserved     /* Not used. */ )
{
    switch (reason)
    {
      case DLL_PROCESS_ATTACH:
        break;

      case DLL_PROCESS_DETACH:
        break;

      case DLL_THREAD_ATTACH:
        break;

      case DLL_THREAD_DETACH:
        break;
    }

    /* Returns TRUE on success, FALSE on failure */
    return TRUE;
}
#endif
