/* Implementation of an AVL tree.  Used by the alias module to make
 * storage and searching faster.  As a nice side effect, it keeps alias
 * lists from flipping top to bottom when you click back to an alias list.
 *
 * Written by Clay Dowling <clay@lazarusid.com>
 */

#include <stdlib.h>
#include <stdio.h>
#include "tree.h"

#define MIN(a,b)  ( a<b?a:b )
#define MAX(a,b)  ( a>b?a:b )

struct tree_t* tree_find(struct forest* top, void* data, int (*compare)(const void*, const void*)) {

  struct tree_t* cur;
  int direction;

  cur = top->root;
  while(cur) {
    direction = compare(data, cur->data);
    if (direction == 0)
      return cur;
    if (direction < 0)
      cur = cur->left;
    if (direction > 0)
      cur = cur->right;
  }
  
  return 0;

}

DLLIMPORT struct tree_t* tree_insert(struct forest* top, void* data, int (*compare)(const void*, const void*)) {

  struct tree_t* n = top->root;  /* current node */
  struct tree_t* p = 0;  /* parent node */
  struct tree_t* q = 0;  /* parent node held for special processing */
  int direction;

  while(n) {
    if (n->data == 0) break;

    direction = compare(data, n->data);

    /* Replace data and get out of here */
    if (direction == 0) {
      n->data = data;
      return n;
    }

    p = n;  /* current now becomes the parent */
    if (p->balance != 0) q = p;  /* There was an existing minor inbalance */

    if (direction < 0)
      n = n->left;
    else
      n = n->right;
      
  }

  /* Do the actual insertion */
  n = tree_new();
  if (!n) return 0;

  top->count++;  /* Let forest know it has a new tree */
  n->data = data;
  n->left = 0; 
  n->right = 0;
  n->parent = p;
  n->balance = 0;
  if (n->parent == 0) top->root = n; /* Probably already covered, but 
					 * a good idea to make sure. */
  else {
    if (direction < 0)
      p->left = n;
    else
      p->right = n;
  }


  /* Adjust balance up the tree, up to the last minor imbalance */
  while (p != q) {
    if (p->left == n) p->balance = -1;
    else p->balance = +1;
    n = p;
    p = p->parent;
  }

  tree_balance(top, q, n);

  return n;

}

void tree_balance(struct forest* top, struct tree_t* q, struct tree_t* n) {

  if (q == 0) return;

  if (q->left == n) {
    q->balance--;
    if (q->balance == -2) {
      if (q->left->balance > 0) 
	tree_rotleft(top, q->left); 
      tree_rotright(top, q);
    }
  }
  if (q->right == n) {
    q->balance++;
    if (q->balance == 2) {
      if (q->right->balance < 0)
	tree_rotright(top, q->right);
      tree_rotleft(top, q);
    }
  }

}

void tree_print(struct forest* top, FILE* out, void (*print)(FILE*, const void*)) {

  tree_print_worker(top->root, out, print);

}

void tree_print_worker(struct tree_t* root, FILE* out, void (*print)(FILE*, const void*)) {

  if (root == 0) return;
  
  tree_print_worker(root->left, out, print);
  print(out, root->data);
  tree_print_worker(root->right, out, print);

}

void tree_rotleft(struct forest* top, struct tree_t* n) {

  struct tree_t* r;
  struct tree_t* p;

  r = n->right;
  n->right = r->left;
  if (r->left) r->left->parent = n;
  p = n->parent; 
  r->parent = p;
  if (p == 0) top->root = r;
  if (p != 0) {
    if ( p->left == n) p->left = r;
    else
      p->right = r;
  }
  r->left = n;
  n->parent = r;

  /* Reset balance factors */
  n->balance = n->balance - (1 + MAX(r->balance, 0));
  r->balance = r->balance - (1 - MIN(n->balance, 0));

}

void tree_rotright(struct forest* top, struct tree_t* n) {
  struct tree_t* l;
  struct tree_t* p;

  l = n->left;
  n->left = l->right;
  if (l->right) l->right->parent = n;
  p = n->parent;
  l->parent = p;
  if (p == 0) top->root = l;
  if (p != 0) {
    if (p->left == n)
      p->left = l;
    else
      p->right = l;
  }
  l->right = n;
  n->parent = l;

  /* Reset balance factors */
  n->balance = n->balance + (1 - MIN(l->balance, 0));
  l->balance = l->balance + (1 + MAX(n->balance, 0));

}

struct tree_t* tree_new() {

  struct tree_t* t;

  t = (struct tree_t*)calloc(1, sizeof(struct tree_t));
  return t;

}

void tree_delete(struct tree_t* n, void (*dealloc)(void*)) {

  if (!n) 
    return;

  tree_delete(n->left, dealloc);
  tree_delete(n->right, dealloc);

  if (n->data)
    dealloc(n->data);

}

void** tree_span(struct forest* top) {

  struct tree_t** node;
  void** span;
  char* touched;
  int cspan = 0;
  int i;

  if (top->root == 0) return 0;

  node = (struct tree_t**)calloc(top->count + 1, sizeof(struct tree_t*));
  span = (void**)calloc(top->count + 1, sizeof(void*));
  touched = (char*)calloc(top->count + 1, sizeof(char));

  node[cspan++] = top->root;

  for(i=0; i < top->count; i++) {
    if (node[i] && !touched[i]) {
      if (node[i]->left) node[cspan++] = node[i]->left;
      if (node[i]->right) node[cspan++] = node[i]->right;
    }
  }

  cspan=0;
  for(i=0; i < top->count; i++) 
    if (node[i] && node[i]->data)
      span[cspan++] = node[i]->data;

  free(node);
  free(touched);

  return span;

}

struct forest* forest_new() {

  struct forest* f;

  f = (struct forest*)calloc(1, sizeof(struct forest));
  /* f->root = tree_new(); */

  return f;

}

void forest_delete(struct forest* f, void (*dealloc)(void*)) {

  tree_delete(f->root, dealloc);
  free(f);

}
