#ifndef MOO_DRIVER_HH
#define MOO_DRIVER_HH

#include <string>
#include <glib.h>
#include "types.hh"
#include "mucipher.h"

namespace Moo
{
  class Driver 
  {
    public:

      typedef std::map<std::string, std::string> Domain;
      typedef std::map<std::string, Domain> Config;
      
      Driver (const std::string& hostname,
              const std::string& port,
              const std::string& password) 

          : m_stream        (0),
            m_server_host   (hostname),
            m_server_port   (port),
            m_password      (password)

      {
        lock = g_mutex_new ();
      }

      ~Driver () { if (m_stream) disconnect(); g_mutex_free (lock); }

      // Core Functions
      void credentials (const std::string& hostname,
                        const std::string& port,
                        const std::string& password,
                        bool  reconnect = false) 
      { 
        m_server_host = hostname;
        m_server_port = port;
        m_password    = password;

        if (reconnect)
          {
            disconnect ();
            connect ();
          }
      }

      bool connect    ();
      void disconnect ();

      void search_start (MuseekSearchType type, const std::string &query);
      void search_cancel (unsigned int ticket);

      void download_start  (const XFERKey& key);

      void transfer_abort  (const XFERKey& key, bool upload = false);
      void transfer_remove (const XFERKey& key, bool upload = false);

      void request_transfers  ();

      //Config
      std::string
      get_key (const std::string& domain, const std::string& key)
      {
        Config::iterator i = config.find (domain);
        if (i == config.end()) return std::string();

        Domain::iterator d = i->second.find (key);        
        if (d == i->second.end ()) return std::string();

        return d->second;
      }

      //Helper Stuff
      static std::string xfer_status_description (Moo::TransferState state);

      // Signals
      SConnected&           s_connected               () { return s_connected_; }
      SDisconnected&        s_disconnected            () { return s_disconnected_; }
      SStatusMessage&       s_status_message          () { return s_status_message_; }
      SServerState&         s_server_state            () { return s_server_state_; }
      SLogin&               s_login                   () { return s_login_; }
      SSearch&              s_search                  () { return s_search_; }
      SSearchResult&        s_search_result           () { return s_search_result_; }
      STransferUpdate&      s_transfer_update         () { return s_transfer_update_; }

    private:

      SConnected          s_connected_;
      SDisconnected       s_disconnected_;
      SStatusMessage      s_status_message_;
      SServerState        s_server_state_;
      SLogin              s_login_;
      SSearch             s_search_;
      SSearchResult       s_search_result_;
      STransferUpdate     s_transfer_update_;

      static gboolean
      m_stream_read (GIOChannel     *source,
                     GIOCondition    condition,
                     gpointer        data);

      GIOChannel  *m_stream;
      guint        m_stream_source_id;
      GMutex      *lock; 

      std::string  m_server_host;
      std::string  m_server_port;
      std::string  m_password;

      Transfers         m_transfers;
      SearchesBySearchT m_searches_by_search;
      SearchesByTicketT m_searches_by_ticket;

      CipherContext mContext; 

      Config config;
  };
}
#endif
