// MooDriver - (C) 2006 M.Derezynski
//
// Code based on;
//
// ---
//
// giFTcurs - curses interface to giFT
// Copyright (C) 2001, 2002, 2003 Göran Weinholt <weinholt@dtek.chalmers.se>
// Copyright (C) 2003 Christian Häggström <chm@c00.info>
//

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <unistd.h>
#include <errno.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <iostream>
#include <istream>
#include <ostream>
#include <sstream>

#include "xcommon.hh"
#include "util.hh"

#include <moo/types.hh>
#include <moo/driver.hh>
#include <moo/mucipher.h>

#include "moo-io.hh"
#include "moo-types-private.hh"

namespace
{
    const char* xfer_states[] =
    {
      N_("Finished"),
      N_("Running"),
      N_("Negotiate"),
      N_("Waiting"),
      N_("Establishing"),
      N_("Initiating"),
      N_("Connecting"),
      N_("Queued"),
      N_("Getting Address"),
      N_("Getting Status"),
      N_("User Offline"),
      N_("Transfer Closed"),
      N_("Unable To Connect"),
      N_("Transfer Aborted"),
      N_("Error"),
    };
}

namespace Moo
{
    //Helper Stuff
    std::string
    Driver::xfer_status_description (Moo::TransferState state)
    {
      if ( state > int(G_N_ELEMENTS (xfer_states))) return std::string();
      return std::string(xfer_states[state]);
    } 

    gboolean
    Driver::m_stream_read (GIOChannel     *source,
                           GIOCondition    condition,
                           gpointer        data)
    {
      Moo::Driver *driver = reinterpret_cast<Moo::Driver *>(data);

      if (!g_mutex_trylock (driver->lock)) return FALSE;

      if (condition == G_IO_HUP)
        {
          g_mutex_unlock (driver->lock);
          driver->disconnect ();
          return FALSE;
        }
      else if (condition == G_IO_ERR)
        {
          g_message ("Condition: %d", condition); 
        }
      else if (condition == G_IO_IN)
        {
          guint msize = Moo::IO::read_uint (source);

          if (!msize)
            {
              g_mutex_unlock (driver->lock);
              driver->disconnect ();
              return FALSE;
            }

          size_t  expect = msize,
                  length = 0;
          char   *data   = g_new0 (char, expect); 

          GError *error = 0;
          g_io_channel_read_chars (source, data, expect, &length, &error);

          if (error)
            {
              g_warning ("Error occured reading from the iochannel: %s", error->message);
              g_error_free (error);
              abort ();
            }

          Moo::Data read_data;
          if (expect != length)
            {
              g_free (data);
              g_mutex_unlock (driver->lock);
              driver->disconnect ();
              return FALSE;
            }
          else
            {
              for (guint n = 0; n < length; ++n)
                {
                  read_data.push_back ((unsigned char)data[n]);
                }
              g_free (data);
            }

          guint offset = 0;    
          guint mtype = Moo::IO::unpack_uint (read_data, offset);

          if (msize) switch (mtype)
            {
              case Moo::M_CHALLENGE:
              {
                Moo::IO::unpack_uint (read_data, offset); //NOTE: We need to read off those 4 first bytes or the rest data is scrambled
                std::string challenge = Moo::IO::unpack_string (read_data, offset);

                challenge += driver->m_password; 
                
                std::string resp      = Util::md5_hex ((char*)challenge.c_str(), strlen (challenge.c_str())); 
                std::string algo      = "MD5";

                // Attempt Login
                guint response_size =  (sizeof(guint)+strlen(algo.c_str())) +  // Algorithm; we use MD5
                                       (sizeof(guint)+strlen(resp.c_str())) +  // Challenge response
                                       (sizeof(guint)) +                       // Mask
                                       (sizeof(guint));                        // Message ID;

                Moo::IO::write_uint (source, response_size); 
                Moo::IO::write_uint (source, (guint)(Moo::M_LOGIN));
                Moo::IO::write_string (source, algo.c_str());
                Moo::IO::write_string (source, resp.c_str()); 
                Moo::IO::write_uint (source, 0x04 | 0x40); 
                g_io_channel_flush (source, NULL);
                break;
              }

              case Moo::M_STATUSMSG:
              {
                bool type = Moo::IO::unpack_bool (read_data, offset);
                std::string message = Moo::IO::unpack_string (read_data, offset);
                driver->s_status_message_.emit (type, message);
                break;
              }

              case Moo::M_SERVER_STATE:     
              {
                bool connected = Moo::IO::unpack_bool (read_data, offset);
                std::string username  = Moo::IO::unpack_string (read_data, offset);
                driver->s_server_state_.emit (connected, username);
                break;
              }

              case Moo::M_LOGIN:
              { 
                bool success = Moo::IO::unpack_bool (read_data, offset);
                std::string reason;
                if (!success)
                  {
                    reason = Moo::IO::unpack_string (read_data, offset);
                  }
                else
                  {
                    cipherKeySHA256(&(driver->mContext), (char*)driver->m_password.c_str(), strlen(driver->m_password.c_str()));
                  }
                driver->s_login_.emit (success, reason);
                break;
              }

              case Moo::M_CONFSTATE:
              {
                unsigned int n = Moo::IO::unpack_uint (read_data, offset);

                while (n)
                {
                  std::string domain = Moo::CipherIO::decipher (read_data, offset, &driver->mContext);
                  driver->config[domain];
                  unsigned int o = Moo::IO::unpack_uint (read_data, offset);
                  while (o)
                  {
                    std::string key = Moo::CipherIO::decipher (read_data, offset, &driver->mContext);
                    std::string val = Moo::CipherIO::decipher (read_data, offset, &driver->mContext);
                    driver->config[domain][key] = val;
                    o--;
                  }
                  n--;
                }
                break;
              };

              case Moo::M_XFER_UPDATE:
              {
                Transfer xfer;

                xfer.upload     = Moo::IO::unpack_bool (read_data, offset);
                xfer.user       = Moo::IO::unpack_string (read_data, offset);
                xfer.path       = Moo::IO::unpack_string (read_data, offset);
                xfer.queuepos   = Moo::IO::unpack_uint (read_data, offset);
                xfer.state      = Moo::IO::unpack_uint (read_data, offset);
                xfer.error      = Moo::IO::unpack_string (read_data, offset);
                xfer.position   = Moo::IO::unpack_off_t (read_data, offset);
                xfer.size       = Moo::IO::unpack_off_t (read_data, offset);
                xfer.rate       = Moo::IO::unpack_uint (read_data, offset);

                XFERKey key (xfer.user, xfer.path); 

                driver->m_transfers[key] = xfer; 
                driver->s_transfer_update_.emit (key, xfer);

                break;
              }

              case Moo::M_XFER_STATE:
              {
                guint nTransfers = Moo::IO::unpack_uint (read_data, offset);
                Transfers xfers; 

                for (guint n = 0; n < nTransfers; ++n)
                {
                  Transfer xfer;

                  xfer.upload     = Moo::IO::unpack_bool (read_data, offset);
                  xfer.user       = Moo::IO::unpack_string (read_data, offset);
                  xfer.path       = Moo::IO::unpack_string (read_data, offset);
                  xfer.queuepos   = Moo::IO::unpack_uint (read_data, offset);
                  xfer.state      = Moo::IO::unpack_uint (read_data, offset);
                  xfer.error      = Moo::IO::unpack_string (read_data, offset);
                  xfer.position   = Moo::IO::unpack_off_t (read_data, offset);
                  xfer.size       = Moo::IO::unpack_off_t (read_data, offset);
                  xfer.rate       = Moo::IO::unpack_uint (read_data, offset);

                  XFERKey key (xfer.user, xfer.path); 
                  driver->m_transfers[key] = xfer; 
                  driver->s_transfer_update_.emit (key, xfer);
                }
                break;
              }

              case Moo::M_SEARCH:
              {
                std::string search = Moo::IO::unpack_string (read_data, offset);
                guint ticket = Moo::IO::unpack_uint (read_data, offset);
                driver->m_searches_by_search.insert (std::make_pair (search, ticket));
                driver->m_searches_by_ticket.insert (std::make_pair (ticket, search));
                driver->s_search_.emit (search, ticket);
                break;
              }

              case Moo::M_SEARCH_REPLY:
              {
                SearchResult result;

                result.ticket   = Moo::IO::unpack_uint    (read_data, offset);
                result.user     = Moo::IO::unpack_string  (read_data, offset);
                result.slot     = Moo::IO::unpack_bool    (read_data, offset);
                result.speed    = Moo::IO::unpack_uint    (read_data, offset);
                result.queue    = Moo::IO::unpack_uint    (read_data, offset);
                
                guint numFiles = Moo::IO::unpack_uint (read_data, offset); 
                for (guint n = 0; n < numFiles; ++n)
                {
                    File file;
                    file.name = Moo::IO::unpack_string (read_data, offset);
                    file.size = Moo::IO::unpack_off_t (read_data, offset);
                    file.ext  = Moo::IO::unpack_string (read_data, offset);
                    guint numAttrs = Moo::IO::unpack_uint (read_data, offset);
                    if (numAttrs) for (guint n = 0; n < numAttrs; ++n)
                      {
                        file.attrs.push_back (Moo::IO::unpack_uint (read_data, offset));
                      }
                    result.files.push_back (file); 
                }
                driver->s_search_result_.emit (result);
                break;
              }

            }
        }
      g_mutex_unlock (driver->lock);
      return TRUE;
    } 
}

namespace Moo
{
    //------------ Searching
    void
    Driver::search_start (MuseekSearchType        type,
                          const std::string      &query)  
    {
      guint mSize = ((3*sizeof(guint))+strlen(query.c_str())); 
      Moo::IO::write_uint   (m_stream, mSize); 

      Moo::IO::write_uint   (m_stream, (guint)(M_SEARCH));
      Moo::IO::write_uint   (m_stream, (guint)(type));
      Moo::IO::write_string (m_stream, query.c_str());
      g_io_channel_flush    (m_stream, NULL);
    }

    void
    Driver::search_cancel (unsigned int ticket) 
    {
      guint mSize = (2*sizeof(guint));
      Moo::IO::write_uint   (m_stream, mSize); 

      Moo::IO::write_uint   (m_stream, (guint)(M_SEARCH_REPLY));
      Moo::IO::write_uint   (m_stream, (guint)(ticket));

      g_io_channel_flush    (m_stream, NULL);

      std::string search;
      SearchesByTicketT::iterator i = m_searches_by_ticket.find (ticket);
      if (i != m_searches_by_ticket.end())
        {
          search = i->second;
          m_searches_by_ticket.erase (i);
        }
      m_searches_by_search.erase (search);
    }

    void
    Driver::request_transfers ()
    {
      guint mSize = (1*sizeof(guint));
      Moo::IO::write_uint   (m_stream, mSize); 
      Moo::IO::write_uint   (m_stream, (guint)(M_XFER_STATE));
      g_io_channel_flush    (m_stream, NULL);
    }

    //------------ Downloads 
    void
    Driver::download_start (const XFERKey& key)
    {
      guint mSize = (3*sizeof(guint)) + strlen(key.first.c_str()) + strlen (key.second.c_str());
      Moo::IO::write_uint   (m_stream, mSize); 

      Moo::IO::write_uint   (m_stream, (guint)(M_XFER_DOWNLOAD)); 
      Moo::IO::write_string (m_stream, key.first.c_str()); 
      Moo::IO::write_string (m_stream, key.second.c_str()); 

      g_io_channel_flush    (m_stream, NULL);
    }

    void
    Driver::transfer_abort (const XFERKey& key, bool upload)
    {
      guint mSize = (3*sizeof(guint)) + strlen(key.first.c_str()) + strlen (key.second.c_str()) + (sizeof(unsigned char));
      Moo::IO::write_uint   (m_stream, mSize); 

      Moo::IO::write_uint   (m_stream, (guint)(M_XFER_ABORT)); 
      Moo::IO::write_bool   (m_stream, (unsigned char)(upload)); 
      Moo::IO::write_string (m_stream, key.first.c_str()); 
      Moo::IO::write_string (m_stream, key.second.c_str()); 

      g_io_channel_flush    (m_stream, NULL);
    }

    void
    Driver::transfer_remove (const XFERKey& key, bool upload)
    {
      guint mSize = (3*sizeof(guint)) + strlen(key.first.c_str()) + strlen (key.second.c_str()) + (sizeof(unsigned char));
      Moo::IO::write_uint   (m_stream, mSize); 

      Moo::IO::write_uint   (m_stream, (guint)(M_XFER_REMOVE)); 
      Moo::IO::write_bool   (m_stream, (unsigned char)(upload)); 
      Moo::IO::write_string (m_stream, key.first.c_str()); 
      Moo::IO::write_string (m_stream, key.second.c_str()); 

      g_io_channel_flush    (m_stream, NULL);
    }

    //------------ Connection with museekd 
    bool
    Driver::connect ()
    {
      if (m_stream) return true;

      int fd = xconnect_ip (m_server_host.c_str(), m_server_port.c_str());
      if (fd < 0)
      {
        g_log (MOO_LOG_DOMAIN,
               G_LOG_LEVEL_INFO,
               _("Can't connect to the daemon at %s:%s, is museekd running?"), m_server_host.c_str(), m_server_port.c_str());
        s_disconnected_.emit();
        return false; 
      }

      m_stream = g_io_channel_unix_new (fd);
      g_io_channel_set_encoding (m_stream, NULL, NULL);
      m_stream_source_id = g_io_add_watch (m_stream, GIOCondition (G_IO_IN | G_IO_ERR | G_IO_HUP), Driver::m_stream_read, this);

      g_log (MOO_LOG_DOMAIN,
             G_LOG_LEVEL_INFO,
             _("Connected to museekd at %s:%s"), m_server_host.c_str(), m_server_port.c_str());

      s_connected_.emit();
      return true;
    }

    void
    Driver::disconnect ()
    {
      if (!m_stream) return;
      if (!g_mutex_trylock(lock))  
        {
          while (!g_mutex_trylock(lock)) while (g_main_context_pending(NULL)) g_main_context_iteration (NULL, TRUE);
        }
      g_source_destroy (g_main_context_find_source_by_id (NULL, m_stream_source_id));
      g_io_channel_shutdown (m_stream, TRUE, NULL);
      g_io_channel_unref (m_stream);
      m_stream = NULL;
      g_mutex_unlock (lock);
      s_disconnected_.emit();
    }
}
