/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmdrawable
 * @short_description: A base class used for embedding media types.
 * @see_also: #PgmImage, #PgmText, #PgmCanvas.
 *
 * <refsect2>
 * <para>
 * #PgmDrawable represents a visual object on which you can embed media types
 * such as text, images and video, or 2D graphics. It is meant to be a base
 * abstract class for more specific visual objects such as #PgmImage or
 * #PgmText. Drawables are not visible until they are added to a #PgmCanvas.
 * A drawable never knows about physical on screen coordinates such as pixels,
 * and use floating point canvas coordinates. Drawables are positioned in 3D
 * and then projected on a #PgmViewport.
 * </para>
 * <title>Manipulating drawables</title>
 * <para>
 * The base class implements simple drawing methods that are common to the
 * different subclasses. For example if you want to draw a left aligned green
 * text you would do something like that:
 * <example id="pigment-drawable-functions">
 * <title>Calling drawing functions on a drawable</title>
 * <programlisting language="c">
 * pgm_drawable_set_fg_color (text, 0, 255, 0, 255);
 * pgm_drawable_set_bg_color (text, 255, 0, 0, 255);
 * pgm_text_set_label (PGM_TEXT (text), "Hello world!");
 * </programlisting>
 * </example>
 * You are basically using a mix of drawable calls and text calls on the
 * same object at different level of the object hierarchy.
 * </para>
 * <title>Signal connections</title>
 * <para>
 * The <link linkend="PgmDrawable-changed">changed</link> signal is fired
 * whenever a property of a drawable is changed. There are also signals for
 * handling mouse picking. For instance, you just need to connect a callback to
 * the <link linkend="PgmDrawable-changed">clicked</link> signal to be notified
 * of a click on a drawable.
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-09-25 (0.3.1)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "pgmdrawable.h"
#include "pgmcanvas.h"
#include "pgmmarshal.h"
#include "pgmenumtypes.h"

/* Double-click delay in milliseconds */
#define DOUBLE_CLICK_DELAY 250

GST_DEBUG_CATEGORY_STATIC (pgm_drawable_debug);
#define GST_CAT_DEFAULT pgm_drawable_debug

/* Drawable signals */
enum {
  CHANGED,
  PRESSED,
  RELEASED,
  CLICKED,
  DOUBLE_CLICKED,
  SCROLLED,
  DRAG_BEGIN,
  DRAG_MOTION,
  DRAG_END,
  LAST_SIGNAL
};

static GstObjectClass *parent_class = NULL;
static guint pgm_drawable_signals[LAST_SIGNAL] = { 0 };

/* Private methods */

/* Store the intersection between the plane defined by the drawable and a
 * line given by two points. Returns whether or not the intersection belongs
 * to the drawable. */
static gboolean
get_line_intersection (PgmDrawable *drawable,
                       PgmVec3 *intersection,
                       PgmVec3 *p1,
                       PgmVec3 *p2)
{
  gboolean belongs = FALSE;
  PgmVec3 p, pu, pv;
  PgmVec3 *v;

  /* Init plane */
  pgm_vec3_set_from_scalars (&p, drawable->x, drawable->y, drawable->z);
  pgm_vec3_set_from_scalars (&pu, drawable->width, 0.0f, 0.0f);
  pgm_vec3_set_from_scalars (&pv, 0.0f, drawable->height, 0.0f);

  /* Get intersection with the plane */
  v = pgm_intersection_line_plane (p1, p2, &p, &pu, &pv);
  if (!v)
    return FALSE;

  /* Store the intersection */
  pgm_vec3_set_from_vec3 (intersection, v);

  /* Test if the intersection belongs to the drawable rectangle */
  belongs = pgm_point_belongs_rectangle (v, &p, &pu, &pv);

  pgm_vec3_free (v);

  return belongs;
}

/* GObject stuff */

G_DEFINE_ABSTRACT_TYPE (PgmDrawable, pgm_drawable, GST_TYPE_OBJECT);

static void
pgm_drawable_dispose (GObject *object)
{
  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_drawable_class_init (PgmDrawableClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmDrawable::changed:
   * @drawable: the #PgmDrawable
   * @property: the #PgmDrawableProperty changed in @drawable
   *
   * Will be emitted after @property of @drawable is changed.
   */
  pgm_drawable_signals[CHANGED] =
    g_signal_new ("changed", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST,
                  G_STRUCT_OFFSET (PgmDrawableClass, changed),
                  NULL, NULL, pgm_marshal_VOID__ENUM, G_TYPE_NONE,
                  1, PGM_TYPE_DRAWABLE_PROPERTY);
  /**
   * PgmDrawable::pressed:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the click on @drawable
   * @y: the y coordinate of the point intersecting the click on @drawable
   * @z: the z coordinate of the point intersecting the click on @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the button press event
   *
   * Will be emitted after @drawable is pressed.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[PRESSED] =
    g_signal_new ("pressed", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, pressed),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::released:
   * @drawable: the #PgmDrawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the button release event
   *
   * Will be emitted after @drawable is released.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[RELEASED] =
    g_signal_new ("released", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, released),
                  NULL, NULL, pgm_marshal_BOOLEAN__ENUM_UINT, G_TYPE_BOOLEAN,
                  2, PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::clicked:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the click on @drawable
   * @y: the y coordinate of the point intersecting the click on @drawable
   * @z: the z coordinate of the point intersecting the click on @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the button press event
   *
   * Will be emitted after @drawable is clicked.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[CLICKED] =
    g_signal_new ("clicked", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, clicked),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::double-clicked:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the double-click on
   * @drawable
   * @y: the y coordinate of the point intersecting the double-click on
   * @drawable
   * @z: the z coordinate of the point intersecting the double-click on
   * @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the button press event
   *
   * Will be emitted after @drawable is clicked.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[DOUBLE_CLICKED] =
    g_signal_new ("double-clicked", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, double_clicked),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::scrolled:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the scroll on @drawable
   * @y: the y coordinate of the point intersecting the scroll on @drawable
   * @z: the z coordinate of the point intersecting the scroll on @drawable
   * @direction: the #PgmScrollDirection
   * @time: the timestamp of the scroll event
   *
   * Will be emitted after @drawable is scrolled.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[SCROLLED] =
    g_signal_new ("scrolled", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, scrolled),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_SCROLL_DIRECTION, G_TYPE_UINT);
  /**
   * PgmDrawable::drag-begin:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the initial click on
   * @drawable
   * @y: the y coordinate of the point intersecting the initial click on
   * @drawable
   * @z: the z coordinate of the point intersecting the initial click on
   * @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the motion event
   *
   * Will be emitted after @drawable is initially dragged. You can for instance
   * use that signal to change @drawable properties or store the intersecting
   * point for further computations during the "drag-motion" signal emission.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[DRAG_BEGIN] =
    g_signal_new ("drag-begin", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, drag_begin),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::drag-motion:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the cursor position on
   * the dragged drawable @drawable
   * @y: the y coordinate of the point intersecting the cursor position on
   * the dragged drawable @drawable
   * @z: the z coordinate of the point intersecting the cursor position on
   * the dragged drawable @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the motion event
   *
   * Will be emitted after @drawable is dragged.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[DRAG_MOTION] =
    g_signal_new ("drag-motion", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, drag_motion),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);
  /**
   * PgmDrawable::drag-end:
   * @drawable: the #PgmDrawable
   * @x: the x coordinate of the point intersecting the release event on
   * the dragged drawable @drawable at 
   * @y: the y coordinate of the point intersecting the release event on
   * @drawable
   * @z: the z coordinate of the point intersecting the release event on
   * @drawable
   * @button: the #PgmButtonType
   * @time: the timestamp of the motion event
   *
   * Will be emitted after @drawable is dropped.
   *
   * Returns: TRUE to stop other handlers from being invoked for the event.
   * FALSE to propagate the event further.
   */
  pgm_drawable_signals[DRAG_END] =
    g_signal_new ("drag-end", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmDrawableClass, drag_end),
                  NULL, NULL, pgm_marshal_BOOLEAN__FLOAT_FLOAT_FLOAT_ENUM_UINT,
                  G_TYPE_BOOLEAN, 5, G_TYPE_FLOAT, G_TYPE_FLOAT, G_TYPE_FLOAT,
                  PGM_TYPE_BUTTON_TYPE, G_TYPE_UINT);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_drawable_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_drawable_debug, "pgm_drawable", 0,
                           "drawable object");
}

static void
pgm_drawable_init (PgmDrawable *drawable)
{
  drawable->layer = PGM_DRAWABLE_UNBOUND;
  /* Position */
  drawable->x = 0.0f;
  drawable->y = 0.0f;
  drawable->z = 0.0f;
  /* Size */
  drawable->width = 1.0f;
  drawable->height = 1.0f;
  /* Colors */
  drawable->fg_r = 255;
  drawable->fg_g = 255;
  drawable->fg_b = 255;
  drawable->fg_a = 255;
  drawable->bg_r = 255;
  drawable->bg_g = 255;
  drawable->bg_b = 255;
  drawable->bg_a = 255;
  /* Opacity */
  drawable->opacity = 255;

  /* Private data */
  pgm_vec3_set_from_scalars (&drawable->press_point, 0.0f, 0.0f, 0.0f);
  drawable->drag_mask = 0;
  drawable->press_mask = 0;
  drawable->release_time = 0;
  drawable->release_button = 0;
}

/* Public methods */

/**
 * pgm_drawable_hide:
 * @drawable: a #PgmDrawable object.
 *
 * Makes @drawable invisible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_hide (PgmDrawable *drawable)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_FLAG_UNSET (drawable, PGM_DRAWABLE_FLAG_VISIBLE);
  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->hide)
    ret = klass->hide (drawable);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_VISIBILITY);

  return ret;
}

/**
 * pgm_drawable_show:
 * @drawable: a #PgmDrawable object.
 *
 * Makes @drawable visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_show (PgmDrawable *drawable)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  GST_OBJECT_FLAG_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE);
  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->show)
    ret = klass->show (drawable);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_VISIBILITY);

  return ret;
}

/**
 * pgm_drawable_is_visible:
 * @drawable: a #PgmDrawable object.
 * @visible: a pointer to a #gboolean where the visible state of the drawable
 * is going to be stored.
 *
 * Retrieves whether @drawable is visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_is_visible (PgmDrawable *drawable,
                         gboolean *visible)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (visible != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  *visible = GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE);
  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_set_size:
 * @drawable: a #PgmDrawable object.
 * @width: the width of the canvas.
 * @height: the height of the canvas.
 *
 * Defines @drawable size in canvas units to @width x @height.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_set_size (PgmDrawable *drawable,
                       gfloat width,
                       gfloat height)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  drawable->width = MAX (0.0f, width);
  drawable->height = MAX (0.0f, height);
  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->set_size)
    ret = klass->set_size (drawable, width, height);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_SIZE);

  return ret;
}

/**
 * pgm_drawable_get_size:
 * @drawable: a #PgmDrawable object.
 * @width: a pointer to a #gfloat where the width of the drawable is going to be
 * stored.
 * @height: a pointer to a #gfloat where the height of the drawable is going to
 * be stored.
 *
 * Gets @drawable size in canvas units.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_get_size (PgmDrawable *drawable,
                       gfloat *width,
                       gfloat *height)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  *width = drawable->width;
  *height = drawable->height;

  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_set_position:
 * @drawable: a #PgmDrawable object.
 * @x: the position of @drawable on the x axis.
 * @y: the position of @drawable on the y axis.
 * @z: the position of @drawable on the z axis.
 *
 * Sets position of @drawable in canvas units.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_set_position (PgmDrawable *drawable,
                           gfloat x,
                           gfloat y,
                           gfloat z)
{
  PgmError ret = PGM_ERROR_OK;
  gboolean reordered = FALSE;
  PgmDrawableClass *klass;
  GstObject *canvas;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  drawable->x = x;
  drawable->y = y;
  if (drawable->z != z)
    {
      drawable->z = z;
      reordered = TRUE;
    }

  canvas = GST_OBJECT_PARENT (drawable);

  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->set_position)
    ret = klass->set_position (drawable, x, y, z);

  /* Reorder the drawable inside the canvas in the case that z has changed */
  if (reordered && canvas)
    _pgm_canvas_update_order (PGM_CANVAS (canvas), drawable);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_POSITION);

  return ret;
}

/**
 * pgm_drawable_get_position:
 * @drawable: a #PgmDrawable object.
 * @x: a pointer to a #gfloat where the x position of @drawable is going to
 * be stored.
 * @y: a pointer to a #gfloat where the y position of @drawable is going to
 * be stored.
 * @z: a pointer to a #gfloat where the z position of @drawable is going to
 * be stored.
 *
 * Gets @drawable position in canvas units.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_get_position (PgmDrawable *drawable,
                           gfloat *x,
                           gfloat *y,
                           gfloat *z)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (x != NULL, PGM_ERROR_X);
  g_return_val_if_fail (y != NULL, PGM_ERROR_X);
  g_return_val_if_fail (z != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  *x = drawable->x;
  *y = drawable->y;
  *z = drawable->z;

  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_set_fg_color:
 * @drawable: a #PgmDrawable object.
 * @r: the red foreground color.
 * @g: the green foreground color.
 * @b: the blue foreground color.
 * @a: the alpha foreground color.
 *
 * Set the color used for drawing operations on @drawable to (@r,@g,@b,@a).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_set_fg_color (PgmDrawable *drawable,
                           guchar r,
                           guchar g,
                           guchar b,
                           guchar a)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  drawable->fg_r = r;
  drawable->fg_g = g;
  drawable->fg_b = b;
  drawable->fg_a = a;

  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->set_fg_color)
    ret = klass->set_fg_color (drawable, r, g, b, a);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_FG_COLOR);

  return ret;
}

/**
 * pgm_drawable_get_fg_color:
 * @drawable: a #PgmDrawable object.
 * @r: a pointer to a #guchar where the red foreground color is going to be
 * stored.
 * @g: a pointer to a #guchar where the green foreground color is going to be
 * stored.
 * @b: a pointer to a #guchar where the blue foreground color is going to be
 * stored.
 * @a: a pointer to a #guchar where the alpha foreground color is going to be
 * stored.
 *
 * Retrieves the color used for drawing operations on @drawable in
 * (@r,@g,@b,@a).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_get_fg_color (PgmDrawable *drawable,
                           guchar *r,
                           guchar *g,
                           guchar *b,
                           guchar *a)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (r != NULL, PGM_ERROR_X);
  g_return_val_if_fail (g != NULL, PGM_ERROR_X);
  g_return_val_if_fail (b != NULL, PGM_ERROR_X);
  g_return_val_if_fail (a != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  *r = drawable->fg_r;
  *g = drawable->fg_g;
  *b = drawable->fg_b;
  *a = drawable->fg_a;

  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_set_bg_color:
 * @drawable: a #PgmDrawable object.
 * @r: the red background color.
 * @g: the green background color.
 * @b: the blue background color.
 * @a: the alpha background color.
 *
 * Sets the color used to fill the background of @drawable to (@r,@g,@b,@a).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_set_bg_color (PgmDrawable *drawable,
                           guchar r,
                           guchar g,
                           guchar b,
                           guchar a)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  drawable->bg_r = r;
  drawable->bg_g = g;
  drawable->bg_b = b;
  drawable->bg_a = a;

  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->set_bg_color)
    ret = klass->set_bg_color (drawable, r, g, b, a);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_BG_COLOR);

  return ret;
}

/**
 * pgm_drawable_get_bg_color:
 * @drawable: A #PgmDrawable object.
 * @r: a pointer to a #guchar where the red background color is going to be
 * stored.
 * @g: a pointer to a #guchar where the green background color is going to be
 * stored.
 * @b: a pointer to a #guchar where the blue background color is going to be
 * stored.
 * @a: a pointer to a #guchar where the alpha background color is going to be
 * stored.
 *
 * Retrieves the color used to fill the background of @drawable in
 * (@r,@g,@b,@a).
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_get_bg_color (PgmDrawable *drawable,
                           guchar *r,
                           guchar *g,
                           guchar *b,
                           guchar *a)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (r != NULL, PGM_ERROR_X);
  g_return_val_if_fail (g != NULL, PGM_ERROR_X);
  g_return_val_if_fail (b != NULL, PGM_ERROR_X);
  g_return_val_if_fail (a != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  *r = drawable->bg_r;
  *g = drawable->bg_g;
  *b = drawable->bg_b;
  *a = drawable->bg_a;

  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_set_opacity:
 * @drawable: a #PgmDrawable object.
 * @opacity: the opacity of @drawable.
 *
 * Defines the opacity of @drawable to @opacity.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_set_opacity (PgmDrawable *drawable,
                          guchar opacity)
{
  PgmError ret = PGM_ERROR_OK;
  PgmDrawableClass *klass;

  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);

  drawable->opacity = opacity;

  GST_OBJECT_UNLOCK (drawable);

  klass = PGM_DRAWABLE_GET_CLASS (drawable);

  if (klass->set_opacity)
    ret = klass->set_opacity (drawable, opacity);

  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_OPACITY);

  return ret;
}

/**
 * pgm_drawable_regenerate:
 * @drawable: a #PgmDrawable object.
 *
 * Forces a regeneration of the drawable.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_regenerate (PgmDrawable *drawable)
{
  _pgm_drawable_emit_changed (drawable, PGM_DRAWABLE_REGENERATE);

  return PGM_ERROR_OK;
}

/**
 * pgm_drawable_get_opacity:
 * @drawable: a #PgmDrawable object.
 * @opacity: a pointer to a #guchar where the opacity is going to be stored.
 *
 * Retrieves the opacity of @drawable in @opacity.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_drawable_get_opacity (PgmDrawable *drawable,
                          guchar *opacity)
{
  g_return_val_if_fail (PGM_IS_DRAWABLE (drawable), PGM_ERROR_X);
  g_return_val_if_fail (opacity != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (drawable);
  *opacity = drawable->opacity;
  GST_OBJECT_UNLOCK (drawable);

  return PGM_ERROR_OK;
}

/* Protected methods */

/**
 * _pgm_drawable_emit_changed:
 * @drawable: the #PgmDrawable object.
 * @property: the #PgmDrawablePoroperty that has changed.
 *
 * This function is only used internally by #PgmDrawable and its subclasses to
 * emit the 'changed' signal on @drawable.
 *
 * MT safe.
 */
void
_pgm_drawable_emit_changed (PgmDrawable *drawable,
                            PgmDrawableProperty property)
{
  g_signal_emit (G_OBJECT (drawable), pgm_drawable_signals[CHANGED], 0,
                 property);
}

/**
 * _pgm_drawable_do_press_event:
 * @drawable: the #PgmDrawable object.
 * @event: the #PgmEventButton.
 * @p1: the 1st line point in canvas coordinates.
 * @p2: the 2nd line point in canvas coordinates.
 * @emission_mask: the mask to inhibit signal emission.
 *
 * This function is only used internally in #PgmViewport to propagate mouse
 * press events to drawables so that they can properly handle and emit the
 * corresponding picking signals.
 *
 * MT safe.
 */
void
_pgm_drawable_do_press_event (PgmDrawable *drawable,
                              PgmEventButton *event,
                              PgmVec3 *p1,
                              PgmVec3 *p2,
                              guint16 *emission_mask)
{
  PgmVec3 intersect;

  /* Do not process invisible drawables */
  if (!GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE))
    return;

  /* Update properties and emit the signal if the cursor is over the drawable */
  if (get_line_intersection (drawable, &intersect, p1, p2))
    {
      drawable->press_mask |= event->button;
      pgm_vec3_set_from_vec3 (&drawable->press_point, &intersect);

      if (*emission_mask & PGM_DRAWABLE_PICKING_PRESSED)
        {
          gboolean stop_emission;

          GST_DEBUG_OBJECT (drawable, "emitting 'pressed' signal");
          g_signal_emit (G_OBJECT (drawable), pgm_drawable_signals[PRESSED], 0,
                         intersect.v[0], intersect.v[1], intersect.v[2],
                         event->button, event->time, &stop_emission);
          if (stop_emission)
            *emission_mask &= ~PGM_DRAWABLE_PICKING_PRESSED;
        }
    }
}

/**
 * _pgm_drawable_do_release_event:
 * @drawable: the #PgmDrawable object.
 * @event: the #PgmEventButton.
 * @p1: the 1st line point in canvas coordinates.
 * @p2: the 2nd line point in canvas coordinates.
 * @emission_mask: the mask to inhibit signal emission.
 *
 * This function is only used internally in #PgmViewport to propagate mouse
 * release events to drawables so that they can properly handle and emit the
 * corresponding picking signals.
 *
 * MT safe.
 */
void
_pgm_drawable_do_release_event (PgmDrawable *drawable,
                                PgmEventButton *event,
                                PgmVec3 *p1,
                                PgmVec3 *p2,
                                guint16 *emission_mask)
{
  /* Do not process invisible drawables */
  if (!GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE)
      && !drawable->drag_mask)
    return;

  /* First check if the button is pressed before going to further computations */
  if (drawable->press_mask & event->button)
    {
      gboolean cached = FALSE;
      gboolean picked = FALSE;
      gboolean stop_emission;
      PgmVec3 intersect;

      /* Emit the release signal */
      if (*emission_mask & PGM_DRAWABLE_PICKING_RELEASED)
        {
          GST_DEBUG_OBJECT (drawable, "emitting 'released' signal");
          g_signal_emit (G_OBJECT (drawable), pgm_drawable_signals[RELEASED], 0,
                         event->button, event->time, &stop_emission);
          if (stop_emission)
            *emission_mask &= ~PGM_DRAWABLE_PICKING_RELEASED;
        }

      /* Emit a click if the cursor is over the drawable */
      if (*emission_mask & PGM_DRAWABLE_PICKING_CLICKED)
        {
          picked = get_line_intersection (drawable, &intersect, p1, p2);

          if (picked)
            {
              GST_DEBUG_OBJECT (drawable, "emitting 'clicked' signal");
              g_signal_emit (G_OBJECT (drawable), pgm_drawable_signals[CLICKED],
                             0, intersect.v[0], intersect.v[1], intersect.v[2],
                             event->button, event->time, &stop_emission);
              if (stop_emission)
                *emission_mask &= ~PGM_DRAWABLE_PICKING_CLICKED;
            }
          cached = TRUE;
        }

      /* Emit a double-click if the cursor is over the drawable, the last
       * released button is the same and the last click happened recently */
      if (*emission_mask & PGM_DRAWABLE_PICKING_DOUBLE_CLICKED)
        {
          if (!cached)
            picked = get_line_intersection (drawable, &intersect, p1, p2);

          if (picked)
            {
              if (drawable->release_button & event->button
                  && event->time - drawable->release_time < DOUBLE_CLICK_DELAY)
                {
                  GST_DEBUG_OBJECT (drawable,
                                    "emitting 'double-clicked' signal");
                  g_signal_emit (G_OBJECT (drawable),
                                 pgm_drawable_signals[DOUBLE_CLICKED], 0,
                                 intersect.v[0], intersect.v[1], intersect.v[2],
                                 event->button, event->time, &stop_emission);
                  if (stop_emission)
                    *emission_mask &= ~PGM_DRAWABLE_PICKING_DOUBLE_CLICKED;

                  drawable->release_time = 0;
                  drawable->release_button = 0;
                }
              else
                {
                  drawable->release_time = event->time;
                  drawable->release_button |= event->button;
                }
            }
          cached = TRUE;
        }

      /* Emit a drag end if there was an on-going dragging */
      if (drawable->drag_mask & event->button)
        {
          if (*emission_mask & PGM_DRAWABLE_PICKING_DRAG_END)
            {
              if (!cached)
                get_line_intersection (drawable, &intersect, p1, p2);

              GST_DEBUG_OBJECT (drawable, "emitting 'drag-end' signal");
              g_signal_emit (G_OBJECT (drawable),
                             pgm_drawable_signals[DRAG_END], 0,
                             intersect.v[0], intersect.v[1], intersect.v[2],
                             event->button, event->time, &stop_emission);
              if (stop_emission)
                *emission_mask &= ~PGM_DRAWABLE_PICKING_DRAG_END;
            }
          drawable->drag_mask &= ~event->button;
        }

      drawable->press_mask &= ~event->button;
    }
}

/**
 * _pgm_drawable_do_motion_event:
 * @drawable: the #PgmDrawable object.
 * @event: the #PgmEventButton.
 * @p1: the 1st line point in canvas coordinates.
 * @p2: the 2nd line point in canvas coordinates.
 * @emission_mask: the mask to inhibit signal emission.
 *
 * This function is only used internally in #PgmViewport to propagate mouse
 * motion events to drawables so that they can properly handle and emit the
 * corresponding picking signals.
 *
 * MT safe.
 */
void
_pgm_drawable_do_motion_event (PgmDrawable *drawable,
                               PgmEventMotion *event,
                               PgmVec3 *p1,
                               PgmVec3 *p2,
                               guint16 *emission_mask)
{
  /* Do not process invisible drawables */
  if (!GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE))
    return;

  /* Avoid further computations if there's no button pressed */
  if (drawable->press_mask)
    {
      guint16 button;

      /* Check for pressed buttons */
      for (button = 1; button <= PGM_BUTTON_RIGHT; button <<= 1)
        {
          if (drawable->press_mask & button)
            {
              gboolean stop_emission;

              /* Emit the drag begin signal if there's no on-going drag */
              if (!(drawable->drag_mask & button))
                {
                  drawable->drag_mask |= button;

                  if (*emission_mask & PGM_DRAWABLE_PICKING_DRAG_BEGIN)
                    {
                      GST_DEBUG_OBJECT (drawable,
                                        "emitting 'drag-begin' signal");
                      g_signal_emit (G_OBJECT (drawable),
                                     pgm_drawable_signals[DRAG_BEGIN],
                                     0, drawable->press_point.v[0],
                                     drawable->press_point.v[1],
                                     drawable->press_point.v[2],
                                     button, event->time, &stop_emission);
                      if (stop_emission)
                        *emission_mask &= ~PGM_DRAWABLE_PICKING_DRAG_BEGIN;
                    }
                }

              /* Emit the drag motion signal */
              if (*emission_mask & PGM_DRAWABLE_PICKING_DRAG_MOTION)
                {
                  PgmVec3 intersect;

                  get_line_intersection (drawable, &intersect, p1, p2);

                  GST_DEBUG_OBJECT (drawable, "emitting 'drag-motion' signal");
                  g_signal_emit (G_OBJECT (drawable),
                                 pgm_drawable_signals[DRAG_MOTION],
                                 0, intersect.v[0], intersect.v[1],
                                 intersect.v[2], button, event->time,
                                 &stop_emission);
                  if (stop_emission)
                    *emission_mask &= ~PGM_DRAWABLE_PICKING_DRAG_MOTION;
                }
            }
        }
    }
}

/**
 * _pgm_drawable_do_scroll_event:
 * @drawable: the #PgmDrawable object.
 * @event: the #PgmEventButton.
 * @p1: the 1st line point in canvas coordinates.
 * @p2: the 2nd line point in canvas coordinates.
 * @emission_mask: the mask to inhibit signal emission.
 *
 * This function is only used internally in #PgmViewport to propagate mouse
 * scroll events to drawables so that they can properly handle and emit the
 * corresponding picking signals.
 *
 * MT safe.
 */
void
_pgm_drawable_do_scroll_event (PgmDrawable *drawable,
                               PgmEventScroll *event,
                               PgmVec3 *p1,
                               PgmVec3 *p2,
                               guint16 *emission_mask)
{
  PgmVec3 intersect;

  /* Do not process invisible drawables */
  if (!GST_OBJECT_FLAG_IS_SET (drawable, PGM_DRAWABLE_FLAG_VISIBLE))
    return;

  /* Update properties and emit the signal if the cursor is over the drawable */
  if (get_line_intersection (drawable, &intersect, p1, p2))
    {
      pgm_vec3_set_from_vec3 (&drawable->press_point, &intersect);

      if (*emission_mask & PGM_DRAWABLE_PICKING_SCROLLED)
        {
          gboolean stop_emission;

          GST_DEBUG_OBJECT (drawable,
                                        "emitting 'scrolled' signal");
          g_signal_emit (G_OBJECT (drawable), pgm_drawable_signals[SCROLLED], 0,
                         intersect.v[0], intersect.v[1], intersect.v[2],
                         event->direction, event->time, &stop_emission);
          if (stop_emission)
            *emission_mask &= ~PGM_DRAWABLE_PICKING_SCROLLED;
        }
    }
}
