/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmviewportfactory
 * @short_description: A factory creating viewports.
 * @see_also: #PgmViewport.
 *
 * #PgmViewportFactory is used to create instances of #PgmViewport.
 *
 * Use the pgm_viewport_factory_new() and pgm_viewport_factory_create()
 * functions to create viewport instances or use pgm_viewport_factory_make()
 * as a convenient shortcut.
 *
 * The following code example shows you how to create an OpenGL based viewport.
 *
 * <example id="pigment-factory-creation">
 * <title>Using a viewport factory</title>
 * <programlisting language="c">
 * PgmViewportFactory *factory;
 * PgmViewport *viewport;
 * &nbsp;
 * pgm_init (&amp;argc, &amp;argv);
 * &nbsp;
 * factory = pgm_viewport_factory_new ("opengl");
 * pgm_viewport_factory_create (factory, &amp;viewport);
 * </programlisting>
 * </example>
 *
 * Last reviewed on 2007-04-12 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include "pgmviewportfactory.h"

GST_DEBUG_CATEGORY_STATIC (pgm_viewport_factory_debug);
#define GST_CAT_DEFAULT pgm_viewport_factory_debug

static PgmViewportFactoryClass *parent_class = NULL;

/* Already created factories*/
static GHashTable *factory_hash = NULL;

/* GObject stuff */

static void
pgm_viewport_factory_dispose (GObject *object)
{
  PgmViewportFactory *factory = PGM_VIEWPORT_FACTORY (object);

  if (factory->module)
    {
      g_object_unref (factory->module);
      factory->module = NULL;
    }

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_viewport_factory_class_init (PgmViewportFactoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_viewport_factory_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_viewport_factory_debug, "pgm_viewport_factory",
                           0, "Pigment ViewportFactory object");
}

static void
pgm_viewport_factory_base_class_init (PgmViewportFactoryClass *klass)
{
  factory_hash = g_hash_table_new_full (g_str_hash, g_str_equal, g_free,
                                        g_object_unref);
}

static void
pgm_viewport_factory_base_class_finalize (PgmViewportFactoryClass *klass)
{
  g_hash_table_unref (factory_hash);
}

static void
pgm_viewport_factory_init (PgmViewportFactory *factory)
{
  factory->module = NULL;
}

GType
pgm_viewport_factory_get_type (void)
{
  static GType pgm_viewport_factory_type = 0;

  if (G_UNLIKELY (pgm_viewport_factory_type == 0))
    {
      static const GTypeInfo viewport_factory_info = {
        sizeof (PgmViewportFactoryClass),
        (GBaseInitFunc) pgm_viewport_factory_base_class_init,
        (GBaseFinalizeFunc) pgm_viewport_factory_base_class_finalize,
        (GClassInitFunc) pgm_viewport_factory_class_init,
        NULL,
        NULL,
        sizeof (PgmViewportFactory),
        0,
        (GInstanceInitFunc) pgm_viewport_factory_init,
      };

      pgm_viewport_factory_type =
        g_type_register_static (GST_TYPE_OBJECT, "PgmViewportFactory",
                                &viewport_factory_info, 0);
    }

  return pgm_viewport_factory_type;
}

/* Public methods */

/**
 * pgm_viewport_factory_new:
 * @name: the name of the factory to create.
 *
 * Creates a new #PgmViewportFactory instance of the given @name.
 * #PgmViewportFactory is a singleton, if a factory of the given name has
 * already been created, this constructor returns that one with a incremented
 * refcount.
 * This function does not detect if the name passed as argument is an invalid
 * name (i.e. if there is no plugin of that name). Such an error will only be
 * detected when trying to use the factory (e.g. with
 * pgm_viewport_factory_create() or with one of the pgm_viewport_factory_get*()
 * functions.
 *
 * MT safe.
 *
 * Returns: a #PgmViewportFactory instance or NULL if an error occurred.
 */

PgmViewportFactory *
pgm_viewport_factory_new (const gchar *name)
{
  PgmViewportFactory *factory = NULL;
  PgmModule *module = NULL;

  /* Checks whether we already have a factory of that name. We check if the
   * factory hash-table exists, since if no factory has been created before,
   * it is not initialised. */
  if (G_LIKELY (factory_hash))
    {
      factory = g_hash_table_lookup (factory_hash, name);
      if (factory)
        return gst_object_ref (factory);
    }

  module = pgm_module_new (name);

  /* A plugin has been found */
  if (G_LIKELY (module))
    {
      /* Let's create and initialize the factory */
      factory = g_object_new (PGM_TYPE_VIEWPORT_FACTORY, NULL);
      factory->module = module;
      module = NULL;
      GST_DEBUG_OBJECT (factory, "Created new viewport factory");

      g_hash_table_insert (factory_hash, g_strdup (name),
                           gst_object_ref (factory));
    }
  else
    GST_WARNING ("Can't find any plugin with name '%s'", name);

  return factory;
}

/**
 * pgm_viewport_factory_get_description:
 * @factory: a #PgmViewportFactory object.
 * @description: a pointer to a pointer to a #gchar where the description
 * string is going to be be stored. g_free() after use.
 *
 * Retrieves the description of the viewports managed by @factory in
 * @description.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_description (PgmViewportFactory *factory,
                                      gchar **description)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (description != NULL, PGM_ERROR_X);
  g_return_val_if_fail (factory->module, PGM_ERROR_X);

  if (!g_type_module_use (G_TYPE_MODULE (factory->module)))
    return PGM_ERROR_X;

  GST_OBJECT_LOCK (factory);

  *description = g_strdup (factory->module->plugin_desc->description);

  GST_OBJECT_UNLOCK (factory);

  g_type_module_unuse (G_TYPE_MODULE (factory->module));

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_license:
 * @factory: a #PgmViewportFactory object.
 * @license: a pointer to a pointer to a #gchar where the license string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the license of the viewports managed by @factory in @license.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_license (PgmViewportFactory *factory,
                                  gchar **license)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (license != NULL, PGM_ERROR_X);

  if (!g_type_module_use (G_TYPE_MODULE (factory->module)))
    return PGM_ERROR_X;

  GST_OBJECT_LOCK (factory);

  *license = g_strdup (factory->module->plugin_desc->license);

  GST_OBJECT_UNLOCK (factory);

  g_type_module_unuse (G_TYPE_MODULE (factory->module));

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_origin:
 * @factory: a #PgmViewportFactory object.
 * @origin: a pointer to a pointer to a #gchar where the origin string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the origin of the viewports managed by @factory in @origin.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_origin (PgmViewportFactory *factory,
                                 gchar **origin)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (origin != NULL, PGM_ERROR_X);

  if (!g_type_module_use (G_TYPE_MODULE (factory->module)))
    return PGM_ERROR_X;

  GST_OBJECT_LOCK (factory);

  *origin = g_strdup (factory->module->plugin_desc->origin);

  GST_OBJECT_UNLOCK (factory);

  g_type_module_unuse (G_TYPE_MODULE (factory->module));

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_get_author:
 * @factory: a #PgmViewportFactory object.
 * @author: a pointer to a pointer to a #gchar where the author string is
 * going to be be stored. g_free() after use.
 *
 * Retrieves the author of the viewports managed by @factory in @author.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_get_author (PgmViewportFactory *factory,
                                 gchar **author)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (author != NULL, PGM_ERROR_X);


  if (!g_type_module_use (G_TYPE_MODULE (factory->module)))
    return PGM_ERROR_X;

  GST_OBJECT_LOCK (factory);

  *author = g_strdup (factory->module->plugin_desc->author);

  GST_OBJECT_UNLOCK (factory);

  g_type_module_unuse (G_TYPE_MODULE (factory->module));

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_create:
 * @factory: a #PgmViewportFactory object.
 * @viewport: a pointer to a #PgmViewport's pointer where the created viewport
 * is going to be be stored.
 *
 * Creates a new viewport of the type defined by @factory in @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_create (PgmViewportFactory *factory,
                             PgmViewport **viewport)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT_FACTORY (factory), PGM_ERROR_X);
  g_return_val_if_fail (viewport != NULL, PGM_ERROR_X);

  /* FIXME: we should return a more specific error if the module loading fails
   */
  if (!g_type_module_use (G_TYPE_MODULE (factory->module)))
    return PGM_ERROR_X;

  GST_OBJECT_LOCK (factory);

  *viewport = factory->module->plugin_desc->create ();

  (*viewport)->factory = gst_object_ref (GST_OBJECT_CAST (factory));

  GST_OBJECT_UNLOCK (factory);

  g_type_module_unuse (G_TYPE_MODULE (factory->module));

  GST_DEBUG_OBJECT (*viewport, "created new viewport");

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_factory_make:
 * @name: the name of the viewport to make.
 * @viewport: a pointer to a #PgmViewport's pointer where the created viewport
 * is going to be be stored.
 *
 * Creates a new viewport of the given @name in @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_factory_make (const gchar *name,
                           PgmViewport **viewport)
{
  PgmViewportFactory *factory;

  g_return_val_if_fail (viewport != NULL, PGM_ERROR_X);

  factory = pgm_viewport_factory_new (name);
  g_return_val_if_fail (factory != NULL, PGM_ERROR_X);

  pgm_viewport_factory_create (factory, viewport);
  gst_object_unref (GST_OBJECT_CAST (factory));

  return PGM_ERROR_OK;
}
