/***************************************************************************
 * kguitartab.cpp: implementation of KGuitarTab class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#include "kguitartab.h"
#include "tabsong.h"

KGuitarTab::KGuitarTab()
{
}

KGuitarTab::~KGuitarTab()
{
}

TabSong KGuitarTab::loadFile(const QString& fileName)
{
	KZip zipFile(fileName);
	QFile file(fileName);
	QDomDocument doc("KGuitar");
	QDomNode node;
	TabSong tabSong("", 120);
	
	if (zipFile.open(IO_ReadOnly) == false)
		oldKGuitar(file, tabSong);
	else {
		QDir homeDir(QDir::home());
		QFileInfo info(fileName);
		const KArchiveDirectory* dir = zipFile.directory();
		const KArchiveEntry* entry = dir->entry("/" + info.fileName() + ".xml");
		
		// Create ktabedit dir where the the file is 
		// unzip, the file is deleted after the parsing.
		//
		if (!homeDir.cd(".ktabedit"))
			if (homeDir.mkdir(".ktabedit") == false)
				throw "Unable to create .ktabedit directory";
		else {
			// Clean up the dir
			//
			QStringList removeList = homeDir.entryList("*.kg.xml");
			
			for (uint i = 0; i < removeList.count(); i++)
				homeDir.remove(removeList[i]);
		}
		
		if (entry) {
			dir->copyTo(homeDir.absPath() + "/");
			QFile file(homeDir.absPath() + "/" + info.fileName() + ".xml");
			
			QDomDocument doc("KTabEdit");
			QDomNode node;
			
			if (file.open(IO_ReadOnly) == false)
				throw "Can't open the file";
			
			if (doc.setContent(&file) == false) {
				file.close();
				file.remove();
				
				throw "Can't set the content of the file in the QDomDocument class";
			}
			
			file.close();
			
			if (!file.remove())
				kdDebug() << "Impossible to delete the xml file in " << file.name() << " \n";
			
			doc.documentElement();
			
			node = doc.firstChild();
			while (node.isNull() == false) {
				// Find the kguitar header !!
				//
				if (node.isElement()) {
					QDomElement element = node.toElement();
					
					if ((element.tagName() == "KTabEdit") && (element.attribute("version") == "0.5")) {
						readSongInformation(element.firstChild(), tabSong);
					} else {
						throw "Bad XML file\n";
					}
				}
				
				node = node.nextSibling();
			}

		} else {
			QString err("Don't find /" + info.fileName() + ".xml" + "\n");
			
			throw err;
		}
		
	}
	
	return tabSong;
}

void KGuitarTab::readSongInformation(QDomNode node, TabSong& tabSong)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "SongProperties")
				readSongProperties(element.firstChild(), tabSong);
			else if (element.tagName() == "Tracks")
				readTrackInformation(element.firstChild(), tabSong);
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::oldKGuitar(QFile& file, TabSong& tabSong)
{
	QDomDocument doc("KGuitar");
	QDomNode node;
	
	if (file.open(IO_ReadOnly) == false)
		throw "Can't open the file";
	
	if (doc.setContent(&file) == false) {
		file.close();
		throw "Can't set the content of the file in the QDomDocument class";
	}
	
	file.close();
	
	doc.documentElement();
	
	node = doc.firstChild();
	while (node.isNull() == false) {
		// Find the kguitar header !!
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if ((element.tagName() == "KGuitar") && (element.attribute("version") == "0.1")) {
				kguitarElement(element.firstChild(), tabSong);
			}
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::kguitarElement(QDomNode node, TabSong& tabSong)
{
	while (node.isNull() == false) {
		// Find SongProperties, Tracks
		//
		if (node.isElement()){
			QDomElement element = node.toElement();
			
			
			
			if (element.tagName() == "SongProperties") {
				songPropertiesElement(element.firstChild(), tabSong);
			} else if (element.tagName() == "Tracks") {
				QString string = element.attribute("number", "0");
				uint nbTracks = string.toUInt();
				
				if (nbTracks == 0)
					throw "Wrong number of tracks";
				
				tracksElement(element.firstChild(), tabSong, nbTracks);
			}
			
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::songPropertiesElement(QDomNode node, TabSong& tabSong)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			
			
			if (element.tagName() == "Title")
				tabSong.setTitle(element.text());
			else if (element.tagName() == "SubTitle")
				tabSong.setSubTitle(element.text());
			else if (element.tagName() == "Artist")
				tabSong.setArtist(element.text());
			else if (element.tagName() == "Album")
				tabSong.setAlbum(element.text());
			else if (element.tagName() == "Author")
				tabSong.setAuthor(element.text());
			else if (element.tagName() == "Copyright")
				tabSong.setCopyright(element.text());
			else if (element.tagName() == "Transcriber")
				tabSong.setTranscriber(element.text());
			else if (element.tagName() == "Instructions")
				tabSong.setInstructions(element.text());
			else if (element.tagName() == "Tempo")
				tabSong.setTempo(element.text().toUInt());
			
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::tracksElement(QDomNode node, TabSong& tabSong, uint nbTracks)
{
	// Create tracks
	//
	for (uint i = 0; i < nbTracks; i++)
		tabSong.append(new TabTrack);
	
	while (node.isNull() == false) {
		// Find TrackProperties, TrackSong
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "TrackProperties") {
				QString string = element.attribute("number", "0");
				uint i = string.toUInt();
				
				trackPropertiesElement(element.firstChild(), tabSong, i);
			} else if (element.tagName() == "TrackSong") {
				QString string = element.attribute("number", "0");
				uint i = string.toUInt();
				
				trackSongElement(element.firstChild(), tabSong, i);
			}
			
			node = node.nextSibling();
		}
	}
}

void KGuitarTab::trackPropertiesElement(QDomNode node, TabSong& tabSong, uint nbTracks)
{
	TrackProperties* properties = tabSong.at(nbTracks);
	
	while (node.isNull() == false) {
		// Find mode, name, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Mode")
				properties->setMode(static_cast<TrackMode>(element.text().toUInt()));
			else if (element.tagName() == "Name")
				properties->setName(element.text());
			else if (element.tagName() == "NbStrings")
				properties->setNbStrings(element.text().toUInt());
			else if (element.tagName() == "NbFrets")
				properties->setNbFrets(element.text().toUInt());
			else if (element.tagName() == "Tune") {
				QString string = element.attribute("chord", "0");
				uint i = string.toUInt();
				uchar c = element.text().toUInt();
				
				properties->setTune(i, c);
			} else if (element.tagName() == "TimeSignature") {
				QString string = element.text();
				int i = 4, j = 4, k = 0;
				
				k = string.find('/');
				
				if (k != -1) {
					i = string.mid(0, k).toUInt();
					j = string.mid(k+1, string.length() - k).toUInt();
				}
				
				properties->setTimeSignature(0, i);
				properties->setTimeSignature(1, j);
			} else if (element.tagName() == "Channel")
				properties->setChannel(element.text().toUInt());
			else if (element.tagName() == "Patch")
				properties->setPatch(element.text().toUInt());
			else if (element.tagName() == "Bank")
				properties->setBank(element.text().toUInt());
			else if (element.tagName() == "Volume")
				properties->setVolume(element.text().toUInt());
			else if (element.tagName() == "Balance")
				properties->setBalance(element.text().toUInt());
			else if (element.tagName() == "Chorus")
				properties->setChorus(element.text().toUInt());
			else if (element.tagName() == "Reverb")
				properties->setReverb(element.text().toUInt());
			else if (element.tagName() == "Phaser")
				properties->setPhaser(element.text().toUInt());
			else if (element.tagName() == "Tremolo")
				properties->setTremolo(element.text().toUInt());
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::trackSongElement(QDomNode node, TabSong& tabSong, uint nbTrack)
{
	TabTrack* track = tabSong.at(nbTrack);
	
	while (node.isNull() == false) {
		// Find bar.
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Bar") {
				QString string = element.attribute("number", "0");
				uint i = string.toUInt();
				
				trackBarElement(element.firstChild(), track, i);
			}
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::trackBarElement(QDomNode node, TabTrack* track, uint nbBar)
{
	while (node.isNull() == false) {
		// Find time and key signature, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "TimeSignature") {
				QString string = element.text();
				int i = 4, j = 4, k = 0;
				
				k = string.find('/');
				
				if (k != -1) {
					i = string.mid(0, k).toUInt();
					j = string.mid(k+1, string.length() - k).toUInt();
				}
				
				TabBar bar(i,j);
				
				track->addBar(bar);
			} else if (element.tagName() == "KeySignature") {
				QString string = element.text();
				uint i = string.toUInt();
				TabBar bar = track->bar(nbBar);
				
				bar.setKeySignature(i);
				
				track->setBar(nbBar, bar);
			} else if (element.tagName() == "Note") {
				QString string = element.attribute("duration", "0");
				uint i = string.toUInt();
				TabBar bar = track->bar(nbBar);
				
				trackNoteElement(element.firstChild(), bar, i);
				
				track->setBar(nbBar, bar);
			}
		}
		node = node.nextSibling();
	}
}

void KGuitarTab::trackNoteElement(QDomNode node, TabBar& bar, uint duration)
{
	TabTimes times;
	times.setDuration(static_cast<Duration>(duration));
	
	while (node.isNull() == false) {
		// Find chord or rest.
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Fret") {
				QString string = element.attribute("chord", "0");
				uint i = string.toUInt();
				
				TabNote note;
				note.setFret(element.text().toUInt());
				times.setNotes(i, note);
			} else if (element.tagName() == "Rest") {
				times.setRest(true);
			} else if (element.tagName() == "Effect") {
				QString string = element.attribute("kind", "");
				
				if (string != "") {
					QString string = element.attribute("chord", "0");
					uint i = string.toUInt();
					
					if (string == "bend") {
						TabBend bend;
						
						bend.setBendType(TabBend::Bend);
						bend.setBendValue(150);
						
						QValueList<TabBendPoint> list;
						TabBendPoint point;
						
						point.setPosition(0);
						point.setHeight(0);
						list.append(point);
						
						point.setPosition(30);
						point.setHeight(150);
						list.append(point);
						
						point.setPosition(60);
						point.setHeight(150);
						list.append(point);
						
						bend.setBendPoint(list);
						
						TabNote note = times.notes(i);
						
						*dynamic_cast<TabBend*>(&note) = bend;
						
						times.setNotes(i, note);
					} else if (string == "vibrato") {
						times.setVibrato(true);
					} else if (string == "harmonic") {
						times.setNaturalHarmonic(true);
					} else if (string == "ring") {
						TabNote note = times.notes(i);
						
						note.setTieNote(true);
						
						times.setNotes(i, note);
					} else if (string == "others") {
						TabNote note = times.notes(i);
						
						switch (element.text().toUInt()) {
							case 0:
								break;
							case 1:
								note.setSlide(true);
								break;
							case 2:
								note.setHammerOnPullOff(true);
								break;
							case 3:
								// Trill
								break;
							case 4:
								times.setTapping(true);
								break;
							case 5:
							case 6:
								// Muffled string
								note.setPalmMute(true);
								break;
							case 7:
								// Pick slide
								note.setSlide(true);
								break;
							case 8:
								note.setTremoloPicking(true);
								break;
						}
// 						times.setEffect(i, static_cast<effectOthers>(element.text().toUInt()));
					}
				}
			}
		}
		
		node = node.nextSibling();
	}
	
	bar.insertTimes(times);
}

bool KGuitarTab::saveFile(const QString& fileName, const TabSong& tabSong)
{
	QFile file(fileName + ".xml");
	QString string;
	QPtrListIterator<TabTrack> it(tabSong);
	TabTrack* track;
	TabBar bar;
	TabTimes times;
	unsigned int countTrack = 0;
	
	if (file.open(IO_WriteOnly) == false)
		return false;
	
	QTextStream stream(&file);
	
	stream << "<?xml version='1.0' encoding='UTF-8'?>" << "\n";
	stream << "<KTabEdit version='0.5'>" << "\n";
	
	stream << "\t" << "<SongProperties>" << "\n";
	writeSongInformation("\t\t", stream, tabSong);
	stream << "\t" << "</SongProperties>" << "\n";
	
	stream << "\t" << "<Tracks>" << "\n";
	
	while((track = it.current()) != 0) {
		++it;
		
		stream << "\t\t" << "<TrackProperties>" << "\n";
		writeTrackInformation("\t\t\t", stream, *track);
		stream << "\t\t" << "</TrackProperties>" << "\n";
		
		stream << "\t\t" << "<TrackSong>" << "\n";
		
		for (uint j = 0; j < track->count(); j++) {
			stream << "\t\t\t" << "<Bar>" << "\n";
			
			bar = track->bar(j);
			
			writeBarInformation("\t\t\t\t", stream, bar);
			
			for (uint k = 0; k < bar.count(); k++) {
				times = bar.times(k);
				
				stream << "\t\t\t\t" << "<Time>" << "\n";
				
				writeTimesInformation("\t\t\t\t\t", stream, times);
				
				if (!times.isRest()) {
					
					for (uint l = 0; l < track->nbStrings(); l++) {
						QString temp;
						
						stream << "\t\t\t\t\t" << "<Note chord=\"" << temp.setNum(l) << "\">" << "\n";
						
						if (!times.notes(l).isEmpty()) {
							writeNoteInformation("\t\t\t\t\t\t", stream, times.notes(l));
						}
						
						stream << "\t\t\t\t\t" << "</Note>" << "\n";
					}
				}
				
				stream << "\t\t\t\t" << "</Time>" << "\n";
			}
			
			stream << "\t\t\t" << "</Bar>" << "\n";
		}
		stream << "\t\t" << "</TrackSong>" << "\n";
		
		countTrack++;
	}
	stream << "\t" << "</Tracks>" << "\n";
	
	stream << "</KTabEdit>" << "\n";
	
	file.close();
	
	file.open(IO_ReadOnly);
	
	QFileInfo info(file);
	
	KZip fileZip(fileName);
	
	char* buffer = new char[info.size()];
	QDataStream fstream(&file);
	fstream.readRawBytes(reinterpret_cast<char *>(buffer),info.size());
	
	file.remove();
	
	fileZip.setCompression(KZip::DeflateCompression);
	fileZip.open(IO_WriteOnly);
	if (!fileZip.writeFile(info.fileName(), info.owner(), info.group(), info.size(), buffer))
		return false;
	
	fileZip.close();
	
	delete buffer;
	
	return true;
}

void KGuitarTab::writeSongInformation(const QString& fill, QTextStream& stream, const TabSong& tabSong)
{
	stream << fill << "<Title>" << tabSong.title().utf8() << "</Title>" << "\n";
	stream << fill << "<SubTitle>" << tabSong.subTitle().utf8() << "</SubTitle>" << "\n";
	stream << fill << "<Artist>" << tabSong.artist().utf8() << "</Artist>" << "\n";
	stream << fill << "<Album>" << tabSong.album().utf8() << "</Album>" << "\n";
	stream << fill << "<Author>" << tabSong.author().utf8() << "</Author>" << "\n";
	stream << fill << "<Copyright>" << tabSong.copyright().utf8() << "</Copyright>" << "\n";
	stream << fill << "<Transcriber>" << tabSong.transcriber().utf8() << "</Transcriber>" << "\n";
	stream << fill << "<Instructions>" << tabSong.instructions().utf8() << "</Instructions>" << "\n";
	stream << fill << "<Comments>" << tabSong.comments().utf8() << "</Comments>" << "\n";
	stream << fill << "<Shuffle>" << tabSong.isShuffle() << "</Shuffle>" << "\n";
	stream << fill << "<Tempo>" << tabSong.tempo() << "</Tempo>" << "\n";
	stream << fill << "<Octave>" << tabSong.octave() << "</Octave>" << "\n";
	stream << fill << "<Key>" << tabSong.key() << "</Key>" << "\n";
	stream << fill << "<Minor>" << tabSong.isMinor() << "</Minor>" << "\n";
	
	stream << fill << "<LyricsProperties>\n";
	writeLyricsInformation(fill + "\t", stream, tabSong);
	stream << fill << "</LyricsProperties>\n";
}

void KGuitarTab::readSongProperties(QDomNode node, TabProperties& tabProperties)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Title")
				tabProperties.setTitle(element.text());
			else if (element.tagName() == "SubTitle")
				tabProperties.setSubTitle(element.text());
			else if (element.tagName() == "Artist")
				tabProperties.setArtist(element.text());
			else if (element.tagName() == "Album")
				tabProperties.setAlbum(element.text());
			else if (element.tagName() == "Author")
				tabProperties.setAuthor(element.text());
			else if (element.tagName() == "Copyright")
				tabProperties.setCopyright(element.text());
			else if (element.tagName() == "Transcriber")
				tabProperties.setTranscriber(element.text());
			else if (element.tagName() == "Instructions")
				tabProperties.setInstructions(element.text());
			else if (element.tagName() == "Comments")
				tabProperties.setComments(element.text());
			else if (element.tagName() == "Shuffle")
				tabProperties.setShuffle(element.text().toUInt());
			else if (element.tagName() == "Tempo")
				tabProperties.setTempo(element.text().toUInt());
			else if (element.tagName() == "Octave")
				tabProperties.setOctave(element.text().toUInt());
			else if (element.tagName() == "Key")
				tabProperties.setKey(element.text().toUInt());
			else if (element.tagName() == "Minor")
				tabProperties.setMinor(element.text().toUInt());
			else if (element.tagName() == "LyricsProperties")
				readLyricsProperties(element.firstChild(), tabProperties);
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeLyricsInformation(const QString& fill, QTextStream& stream, const TabLyrics& tabLyrics)
{
	stream << fill << "<Track>" << tabLyrics.lyricsTrack() << "</Track>" << "\n";
	
	for (int i = 0; i < 5; i++)
		stream << fill << "<Measure number=\"" << i << "\">" << tabLyrics.lyricsMeasure(i) << "</Measure>" << "\n";
	
	for (int i = 0; i < 5; i++)
		stream << fill << "<Lyrics number=\"" << i << "\">" << tabLyrics.lyrics(i).utf8() << "</Lyrics>" << "\n";
}

void KGuitarTab::readLyricsProperties(QDomNode node, TabLyrics& tabLyrics)
{
	uint measure = 0, lyrics = 0;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Track")
				tabLyrics.setLyricsTrack(element.text().toUInt());
			else if (element.tagName() == "Measure") {
				tabLyrics.setLyricsMeasure(measure, element.text().toUInt());
				measure++;
			} else if (element.tagName() == "Lyrics") {
				tabLyrics.setLyrics(lyrics, element.text());
				lyrics++;
			}
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeTrackInformation(const QString& fill, QTextStream& stream, const TabTrack& tabTrack)
{
	stream << fill << "<Mode>" << tabTrack.mode() << "</Mode>" << "\n";
	stream << fill << "<Name>" << tabTrack.name().utf8() << "</Name>" << "\n";
	stream << fill << "<Color>" << tabTrack.color().name().utf8() << "</Color>" << "\n";
	stream << fill << "<NbStrings>" << tabTrack.nbStrings() << "</NbStrings>" << "\n";
	stream << fill << "<NbFrets>" << tabTrack.nbFrets() << "</NbFrets>" << "\n";
	
	for (int i = 0; i < tabTrack.nbStrings(); i++)
		stream << fill << "<Tune fret=\"" << i << "\">" << tabTrack.tune(i) << "</Tune>" << "\n";
	
	stream << fill << "<Capo>" << tabTrack.capo() << "</Capo>" << "\n";
	stream << fill << "<TimeSignature numerator=\"" << tabTrack.timeSignature(0) << "\" denominator=\"" << tabTrack.timeSignature(1) << "\">" <<  "</TimeSignature>" << "\n";
	stream << fill << "<Patch>" << tabTrack.patch() << "</Patch>" << "\n";
	stream << fill << "<Channel>" << tabTrack.channel() << "</Channel>" << "\n";
	stream << fill << "<ChannelEffects>" << tabTrack.channelEffects() << "</ChannelEffects>" << "\n";
	stream << fill << "<MidiPort>" << tabTrack.midiPort() << "</MidiPort>" << "\n";
	stream << fill << "<Bank>" << tabTrack.bank() << "</Bank>" << "\n";
	stream << fill << "<Volume>" << tabTrack.volume() << "</Volume>" << "\n";
	stream << fill << "<Balance>" << tabTrack.balance() << "</Balance>" << "\n";
	stream << fill << "<Chorus>" << tabTrack.chorus() << "</Chorus>" << "\n";
	stream << fill << "<Reverb>" << tabTrack.reverb() << "</Reverb>" << "\n";
	stream << fill << "<Phaser>" << tabTrack.phaser() << "</Phaser>" << "\n";
	stream << fill << "<Tremolo>" << tabTrack.tremolo() << "</Tremolo>" << "\n";
}

void KGuitarTab::readTrackInformation(QDomNode node, TabSong& tabSong)
{
	TabTrack* tabTrack = 0;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "TrackProperties") {
				tabTrack = new TabTrack;
				readTrackProperties(element.firstChild(), *tabTrack);
			} else if (element.tagName() == "TrackSong") {
				if (tabTrack == 0)
					throw "Bad Track format in XML file !";
				
				readTrackSong(element.firstChild(), *tabTrack);
				
				tabSong.append(tabTrack);
				
				tabTrack = 0;
			}
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::readTrackProperties(QDomNode node, TrackProperties& trackProperties)
{
	uint tune = 0;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Mode")
				trackProperties.setMode(static_cast<TrackMode>(element.text().toUInt()));
			else if (element.tagName() == "Name")
				trackProperties.setName(element.text());
			else if (element.tagName() == "Color")
				trackProperties.setColor(element.text());
			else if (element.tagName() == "NbStrings")
				trackProperties.setNbStrings(element.text().toUInt());
			else if (element.tagName() == "NbFrets")
				trackProperties.setNbFrets(element.text().toUInt());
			else if (element.tagName() == "Tune") {
				trackProperties.setTune(tune, element.text().toUInt());
				tune++;
			} else if (element.tagName() == "TimeSignature" && element.hasAttribute("numerator") && element.hasAttribute("denominator")) {
				trackProperties.setTimeSignature(0, element.attribute("numerator").toUInt());
				trackProperties.setTimeSignature(1, element.attribute("denominator").toUInt());
			} else if (element.tagName() == "Patch")
				trackProperties.setPatch(element.text().toUInt());
			else if (element.tagName() == "Channel")
				trackProperties.setChannel(element.text().toUInt());
			else if (element.tagName() == "ChannelEffects")
				trackProperties.setChannelEffects(element.text().toUInt());
			else if (element.tagName() == "MidiPort")
				trackProperties.setMidiPort(element.text().toUInt());
			else if (element.tagName() == "Bank")
				trackProperties.setBank(element.text().toUInt());
			else if (element.tagName() == "Volume")
				trackProperties.setVolume(element.text().toUInt());
			else if (element.tagName() == "balance")
				trackProperties.setBalance(element.text().toUInt());
			else if (element.tagName() == "Chorus")
				trackProperties.setChorus(element.text().toUInt());
			else if (element.tagName() == "Reverb")
				trackProperties.setReverb(element.text().toUInt());
			else if (element.tagName() == "Phaser")
				trackProperties.setPhaser(element.text().toUInt());
			else if (element.tagName() == "Tremolo")
				trackProperties.setTremolo(element.text().toUInt());
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::readTrackSong(QDomNode node, TabTrack& tabTrack)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Bar")
				tabTrack.addBar(readBar(element.firstChild()));
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeBarInformation(const QString& fill, QTextStream& stream, const TabBar& tabBar)
{
	stream << fill << "<TimeSignature numerator=\"" << tabBar.timeSignature(0) << "\" denominator=\"" << tabBar.timeSignature(1) << "\">" <<  "</TimeSignature>" << "\n";
	stream << fill << "<KeySignature>" << tabBar.keySignature() << "</KeySignature>" << "\n";
	stream << fill << "<Color>" << tabBar.color().name().utf8() << "</Color>" << "\n";
	stream << fill << "<Marker>" << tabBar.marker().utf8() << "</Marker>" << "\n";
	stream << fill << "<RepeatBar>" << tabBar.isRepeatBar() << "</RepeatBar>" << "\n";
	stream << fill << "<CloseBar>" << tabBar.isCloseBar() << "</CloseBar>" << "\n";
	stream << fill << "<AltBar>" << tabBar.isAltBar() << "</AltBar>" << "\n";
	stream << fill << "<RepeatStart>" << tabBar.isRepeatStart() << "</RepeatStart>" << "\n";
	stream << fill << "<Repetitions>" << tabBar.repetitions() << "</Repetitions>" << "\n";
	stream << fill << "<AlternateEndings>" << tabBar.isAlternateEndings() << "</AlternateEndings>" << "\n";
	stream << fill << "<DoubleBar>" << tabBar.isDoubleBar() << "</DoubleBar>" << "\n";
	stream << fill << "<Key>" << tabBar.key() << "</Key>" << "\n";
	stream << fill << "<Minor>" << tabBar.isMinor() << "</Minor>" << "\n";
}

TabBar KGuitarTab::readBar(QDomNode node)
{
	TabBar tabBar;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "TimeSignature" && element.hasAttribute("numerator") && element.hasAttribute("denominator")) {
				tabBar.setTimeSignature(0, element.attribute("numerator").toUInt());
				tabBar.setTimeSignature(1, element.attribute("denominator").toUInt());
			} else if (element.tagName() == "KeySignature")
				tabBar.setKeySignature(element.text().toUInt());
			else if (element.tagName() == "Color")
				tabBar.setColor(element.text());
			else if (element.tagName() == "Marker")
				tabBar.setMarker(element.text());
			else if (element.tagName() == "RepeatStart")
				tabBar.setRepeatStart(element.text().toUInt());
			else if (element.tagName() == "Repetitions")
				tabBar.setRepetitions(element.text().toUInt());
			else if (element.tagName() == "AlternateEndings")
				tabBar.setAlternateEndings(element.text().toUInt());
			else if (element.tagName() == "DoubleBar")
				tabBar.setDoubleBar(element.text().toUInt());
			else if (element.tagName() == "Key")
				tabBar.setKey(element.text().toUInt());
			else if (element.tagName() == "KeySignature")
				tabBar.setMinor(element.text().toUInt());
			else if (element.tagName() == "Time")
				tabBar.insertTimes(readTime(element.firstChild()));
		}
		
		node = node.nextSibling();
	}
	
	return tabBar;
}

void KGuitarTab::writeTimesInformation(const QString& fill, QTextStream& stream, const TabTimes& tabTimes)
{
	stream << fill << "<Duration>" << tabTimes.duration() << "</Duration>" << "\n";
	stream << fill << "<Rest>" << tabTimes.isRest() << "</Rest>" << "\n";
	stream << fill << "<Dotted>" << tabTimes.isDotted() << "</Dotted>" << "\n";
	stream << fill << "<NTuplet>" << tabTimes.nTuplet() << "</NTuplet>" << "\n";
	stream << fill << "<Text>" << tabTimes.text().utf8() << "</Text>" << "\n";
	
	stream << fill << "<Effect>\n";
	writeTimesEffectInformation(fill + "\t", stream, tabTimes);
	stream << fill << "</Effect>\n";
	
	stream << fill << "<Bend>\n";
	writeBendInformation(fill + "\t", stream, tabTimes);
	stream << fill << "</Bend>\n";
	
	if (!tabTimes.isChord())
		return ;
	
	stream << fill << "<Chord>\n";
	writeChordInformation(fill + "\t", stream, tabTimes);
	stream << fill << "</Chord>\n";
}

TabTimes KGuitarTab::readTime(QDomNode node)
{
	TabTimes tabTime;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Duration")
				tabTime.setDuration(static_cast<Duration>(element.text().toUInt()));
			else if (element.tagName() == "Rest")
				tabTime.setRest(element.text().toUInt());
			else if (element.tagName() == "Dotted")
				tabTime.setDotted(element.text().toUInt());
			else if (element.tagName() == "NTuplet")
				tabTime.setNTuplet(element.text().toUInt());
			else if (element.tagName() == "Text")
				tabTime.setText(element.text());
			else if (element.tagName() == "Effect")
				readTimesEffect(element.firstChild(), tabTime);
			else if (element.tagName() == "Bend")
				readBendEffect(element.firstChild(), tabTime);
			else if (element.tagName() == "Chord")
				readChordEffect(element.firstChild(), tabTime);
			else if (element.tagName() == "Note" && element.hasAttribute("chord"))
				tabTime.setNotes(element.attribute("chord").toUInt(), readNote(element.firstChild()));
		}
		
		node = node.nextSibling();
	}
	
	return tabTime;
}

void KGuitarTab::writeTimesEffectInformation(const QString& fill, QTextStream& stream, const TabBeatEffects& timesEffect)
{
	stream << fill << "<Rasgueado>" << timesEffect.isRasgueado() << "</Rasgueado>" << "\n";
	stream << fill << "<Pickstroke>" << timesEffect.pickstroke() << "</Pickstroke>" << "\n";
	stream << fill << "<Tremolo>" << timesEffect.tremolo() << "</Tremolo>" << "\n";
	stream << fill << "<Tapping>" << timesEffect.isTapping() << "</Tapping>" << "\n";
	stream << fill << "<Popping>" << timesEffect.isPopping() << "</Popping>" << "\n";
	stream << fill << "<Slapping>" << timesEffect.isSlapping() << "</Slapping>" << "\n";
	stream << fill << "<Stroke>" << timesEffect.isStroke() << "</Stroke>" << "\n";
	stream << fill << "<UpStroke>" << timesEffect.upStroke() << "</UpStroke>" << "\n";
	stream << fill << "<DownStroke>" << timesEffect.downStroke() << "</DownStroke>" << "\n";
	stream << fill << "<Vibrato>" << timesEffect.isVibrato() << "</Vibrato>" << "\n";
	stream << fill << "<WideVibrato>" << timesEffect.isWideVibrato() << "</WideVibrato>" << "\n";
	stream << fill << "<NaturalHarmonic>" << timesEffect.isNaturalHarmonic() << "</NaturalHarmonic>" << "\n";
	stream << fill << "<ArtificialHarmonic>" << timesEffect.isArtificialHarmonic() << "</ArtificialHarmonic>" << "\n";
	stream << fill << "<FadeIn>" << timesEffect.isFadeIn() << "</FadeIn>" << "\n";
}

void KGuitarTab::readTimesEffect(QDomNode node, TabBeatEffects& tabEffect)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Rasgueado")
				tabEffect.setRasgueado(element.text().toUInt());
			else if (element.tagName() == "PickStroke")
				tabEffect.setPickStroke(static_cast<PickStroke>(element.text().toUInt()));
			else if (element.tagName() == "Tremolo")
				tabEffect.setTremolo(element.text().toUInt());
			else if (element.tagName() == "Tapping")
				tabEffect.setTapping(element.text().toUInt());
			else if (element.tagName() == "Popping")
				tabEffect.setPopping(element.text().toUInt());
			else if (element.tagName() == "Slapping")
				tabEffect.setSlapping(element.text().toUInt());
			else if (element.tagName() == "Stroke")
				tabEffect.setStroke(element.text().toUInt());
			else if (element.tagName() == "UpStroke")
				tabEffect.setUpStroke(element.text().toUInt());
			else if (element.tagName() == "DownStroke")
				tabEffect.setDownStroke(element.text().toUInt());
			else if (element.tagName() == "Vibrato")
				tabEffect.setVibrato(element.text().toUInt());
			else if (element.tagName() == "WideVibrato")
				tabEffect.setWideVibrato(element.text().toUInt());
			else if (element.tagName() == "NaturalHarmonic")
				tabEffect.setNaturalHarmonic(element.text().toUInt());
			else if (element.tagName() == "ArtificialHarmonic")
				tabEffect.setArtificialHarmonic(element.text().toUInt());
			else if (element.tagName() == "FadeIn")
				tabEffect.setFadeIn(element.text().toUInt());
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeBendInformation(const QString& fill, QTextStream& stream, const TabBend& tabBend)
{
	stream << fill << "<Type>" << tabBend.bendType() << "</Type>" << "\n";
	stream << fill << "<Value>" << tabBend.bendValue() << "</Value>" << "\n";
	
	QValueList<TabBendPoint> list = tabBend.bendPoint();
	
	for (uint i = 0; i < list.count(); i++) {
		stream << fill << "<Point>\n";
		stream << fill << "\t" << "<Position>" << list[i].position() << "</Position>" << "\n";
		stream << fill << "\t" << "<Height>" << list[i].height() << "</Height>" << "\n";
		stream << fill << "\t" << "<Vibrato>" << list[i].tabVibrato() << "</Vibrato>" << "\n";
		stream << fill << "</Point>\n";
	}
}

void KGuitarTab::readBendEffect(QDomNode node, TabBend& tabBend)
{
	QValueList<TabBendPoint> pointList;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Type")
				tabBend.setBendType(static_cast<TabBend::BendType>(element.text().toUInt()));
			else if (element.tagName() == "Value")
				tabBend.setBendValue(element.text().toUInt());
			else if (element.tagName() == "Point")
				pointList.append(readPointBend(element.firstChild()));
		}
		
		node = node.nextSibling();
	}
	
	tabBend.setBendPoint(pointList);
}

TabBendPoint KGuitarTab::readPointBend(QDomNode node)
{
	TabBendPoint point;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Position")
				point.setPosition(element.text().toUInt());
			else if (element.tagName() == "Height")
				point.setHeight(element.text().toUInt());
			else if (element.tagName() == "Vibrato")
				point.setVibrato(static_cast<TabVibrato>(element.text().toUInt()));
		}
		
		node = node.nextSibling();
	}
	
	return point;
}

void KGuitarTab::writeChordInformation(const QString& fill, QTextStream& stream, const TabChord& tabChord)
{
	stream << fill << "<Name>" << tabChord.name().utf8() << "</Name>" << "\n";
	stream << fill << "<Sharp>" << tabChord.isSharp() << "</Sharp>" << "\n";
	stream << fill << "<Root>" << tabChord.root() << "</Root>" << "\n";
	stream << fill << "<Type>" << tabChord.type() << "</Type>" << "\n";
	stream << fill << "<NineElevenThirteen>" << tabChord.nineElevenThirteen() << "</NineElevenThirteen>" << "\n";
	stream << fill << "<Bass>" << tabChord.bass() << "</Bass>" << "\n";
	stream << fill << "<AugmentedDiminished>" << tabChord.augmentedDiminished() << "</AugmentedDiminished>" << "\n";
	stream << fill << "<AddedNote>" << tabChord.addedNote() << "</AddedNote>" << "\n";
	stream << fill << "<TonalityFive>" << tabChord.tonalityFive() << "</TonalityFive>" << "\n";
	stream << fill << "<TonalityNine>" << tabChord.tonalityNine() << "</TonalityNine>" << "\n";
	stream << fill << "<TonalityEleven>" << tabChord.tonalityEleven() << "</TonalityEleven>" << "\n";
	stream << fill << "<BaseFret>" << tabChord.baseFret() << "</BaseFret>" << "\n";
	
	for (int i = 0; i < 6; i++) {
		stream << fill << "<Fret chord=\"" << i << "\">" << tabChord.fret(i) << "</Fret>" << "\n";
	}
	
	stream << fill << "<NumBarres>" << tabChord.numBarres() << "</NumBarres>" << "\n";
	
	for (int i = 0; i < 5; i++) {
		stream << fill << "<BarreFret chord=\"" << i << "\">" << tabChord.barreFret(i) << "</BarreFret>" << "\n";
	}
	
	for (int i = 0; i < 5; i++) {
		stream << fill << "<BarreStart chord=\"" << i << "\">" << tabChord.barreStart(i) << "</BarreStart>" << "\n";
	}
	
	for (int i = 0; i < 5; i++) {
		stream << fill << "<BarreEnd chord=\"" << i << "\">" << tabChord.barreEnd(i) << "</BarreEnd>" << "\n";
	}
	
	for (int i = 0; i < 7; i++) {
		stream << fill << "<Fingering chord=\"" << i << "\">" << tabChord.fingering(i) << "</Fingering>" << "\n";
	}
	
	stream << fill << "<Show>" << tabChord.isShowDiagram() << "</Show>" << "\n";
}

void KGuitarTab::readChordEffect(QDomNode node, TabChord& tabChord)
{
	QString temp;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Name")
				tabChord.setName(element.text());
			else if (element.tagName() == "Sharp")
				tabChord.setSharp(element.text().toUInt());
			else if (element.tagName() == "Root")
				tabChord.setRoot(element.text().toUInt());
			else if (element.tagName() == "Root")
				tabChord.setRoot(element.text().toUInt());
			else if (element.tagName() == "Type")
				tabChord.setType(element.text().toUInt());
			else if (element.tagName() == "NineElevenThirteen")
				tabChord.setNineElevenThirteen(static_cast<TabChord::ExtentionChord>(element.text().toUInt()));
			else if (element.tagName() == "Bass")
				tabChord.setBass(element.text().toUInt());
			else if (element.tagName() == "AugmentedDiminished")
				tabChord.setAugmentedDiminished(static_cast<TabChord::AugmentedDiminished>(element.text().toUInt()));
			else if (element.tagName() == "AddedNote")
				tabChord.setAddedNote(element.text().toUInt());
			else if (element.tagName() == "TonalityFive")
				tabChord.setTonalityFive(element.text().toUInt());
			else if (element.tagName() == "TonalityNine")
				tabChord.setTonalityNine(element.text().toUInt());
			else if (element.tagName() == "TonalityEleven")
				tabChord.setTonalityEleven(element.text().toUInt());
			else if (element.tagName() == "BaseFret")
				tabChord.setBaseFret(element.text().toUInt());
			else if (element.tagName() == "Fret" && element.hasAttribute("chord"))
				tabChord.setFret(element.attribute("chord").toUInt(), element.text().toUInt());
			else if (element.tagName() == "NumBarres")
				tabChord.setNumBarres(element.text().toUInt());
			else if (element.tagName() == "BarreFret" && element.hasAttribute("chord"))
				tabChord.setBarreFret(element.attribute("chord").toUInt(), element.text().toUInt());
			else if (element.tagName() == "BarreStart" && element.hasAttribute("chord"))
				tabChord.setBarreStart(element.attribute("chord").toUInt(), element.text().toUInt());
			else if (element.tagName() == "BarreEnd" && element.hasAttribute("chord"))
				tabChord.setBarreEnd(element.attribute("chord").toUInt(), element.text().toUInt());
			else if (element.tagName() == "Fingering" && element.hasAttribute("chord"))
				tabChord.setBarreEnd(element.attribute("chord").toUInt(), element.text().toUInt());
			else if (element.tagName() == "Show")
				tabChord.setShowDiagram(element.text().toUInt());
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeNoteInformation(const QString& fill, QTextStream& stream, const TabNote& tabNote)
{
	stream << fill << "<Fret>" << tabNote.fret() << "</Fret>" << "\n";
	stream << fill << "<Duration>" << tabNote.duration() << "</Duration>" << "\n";
	stream << fill << "<NTuplet>" << tabNote.nTuplet() << "</NTuplet>" << "\n";
	stream << fill << "<Accentuated>" << tabNote.isAccentuated() << "</Accentuated>" << "\n";
	stream << fill << "<Dotted>" << tabNote.isDotted() << "</Dotted>" << "\n";
	stream << fill << "<Ghost>" << tabNote.isGhostNote() << "</Ghost>" << "\n";
	stream << fill << "<Tie>" << tabNote.isTieNote() << "</Tie>" << "\n";
	stream << fill << "<Dead>" << tabNote.isDeadNote() << "</Dead>" << "\n";
	stream << fill << "<Hammer>" << tabNote.isHammerOnPullOff() << "</Hammer>" << "\n";
	stream << fill << "<LetRing>" << tabNote.isLetRing() << "</LetRing>" << "\n";
	stream << fill << "<Dynamic>" << tabNote.dynamic() << "</Dynamic>" << "\n";
	stream << fill << "<FingeringLeft>" << tabNote.fingeringLeft() << "</FingeringLeft>" << "\n";
	stream << fill << "<FingeringRight>" << tabNote.fingeringRight() << "</FingeringRight>" << "\n";
	stream << fill << "<LeftHandVibrato>" << tabNote.isLeftHandVibrato() << "</LeftHandVibrato>" << "\n";
	stream << fill << "<PalmMute>" << tabNote.isPalmMute() << "</PalmMute>" << "\n";
	stream << fill << "<Staccato>" << tabNote.isStaccato() << "</Staccato>" << "\n";
	stream << fill << "<Slide>" << tabNote.isSlide() << "</Slide>" << "\n";
	stream << fill << "<SlideType>" << tabNote.slide() << "</SlideType>" << "\n";
	stream << fill << "<TremoloPicking>" << tabNote.tremoloPicking() << "</TremoloPicking>" << "\n";
	stream << fill << "<Harmonic>" << tabNote.harmonic() << "</Harmonic>" << "\n";
	
	stream << fill << "<Bend>\n";
	writeBendInformation(fill + "\t", stream, tabNote);
	stream << fill << "</Bend>\n";
	
	stream << fill << "<Trill>\n";
	writeTrillInformation(fill + "\t", stream, tabNote);
	stream << fill << "</Trill>\n";
	
	stream << fill << "<GraceNote>\n";
	writeGraceNoteInformation(fill + "\t", stream, tabNote);
	stream << fill << "</GraceNote>\n";
}

TabNote KGuitarTab::readNote(QDomNode node)
{
	TabNote note;
	
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Fret")
				note.setFret(element.text().toUInt());
			else if (element.tagName() == "Duration")
				note.setDuration(static_cast<Duration>(element.text().toUInt()));
			else if (element.tagName() == "NTuplet")
				note.setNTuplet(element.text().toUInt());
			else if (element.tagName() == "Accentuated")
				note.setAccentuated(element.text().toUInt());
			else if (element.tagName() == "Dotted")
				note.setDotted(element.text().toUInt());
			else if (element.tagName() == "Ghost")
				note.setGhostNote(element.text().toUInt());
			else if (element.tagName() == "Tie")
				note.setTieNote(element.text().toUInt());
			else if (element.tagName() == "Hammer")
				note.setHammerOnPullOff(element.text().toUInt());
			else if (element.tagName() == "LetRing")
				note.setLetRing(element.text().toUInt());
			else if (element.tagName() == "Dynamic")
				note.setDynamic(static_cast<Dynamic>(element.text().toUInt()));
			else if (element.tagName() == "FingeringLeft")
				note.setFingeringLeft(static_cast<TabNote::Fingering>(element.text().toUInt()));
			else if (element.tagName() == "FingeringRight")
				note.setFingeringRight(static_cast<TabNote::Fingering>(element.text().toUInt()));
			else if (element.tagName() == "LeftHandVibrato")
				note.setLeftHandVibrato(element.text().toUInt());
			else if (element.tagName() == "PalmMute")
				note.setPalmMute(element.text().toUInt());
			else if (element.tagName() == "Staccato")
				note.setStaccato(element.text().toUInt());
			else if (element.tagName() == "Slide")
				note.setSlide(static_cast<bool>(element.text().toUInt()));
			else if (element.tagName() == "SlideType")
				note.setSlide(static_cast<Slide>(element.text().toUInt()));
			else if (element.tagName() == "TremoloPicking")
				note.setTremoloPicking(element.text().toUInt());
			else if (element.tagName() == "Harmonic")
				note.setHarmonic(static_cast<Harmonic>(element.text().toUInt()));
			else if (element.tagName() == "Bend")
				readBendEffect(element.firstChild(), note);
			else if (element.tagName() == "Trill")
				readTrillEffect(element.firstChild(), note);
			else if (element.tagName() == "GraceNote")
				readGraceNoteEffect(element.firstChild(), note);
		}
		
		node = node.nextSibling();
	}
	
	return note;
}

void KGuitarTab::writeTrillInformation(const QString& fill, QTextStream& stream, const TabTrill& tabTrill)
{
	stream << fill << "<Fret>"<< tabTrill.fretTrill() << "</Fret>\n";
	stream << fill << "<Period>"<< tabTrill.fretTrill() << "</Period>\n";
}

void KGuitarTab::readTrillEffect(QDomNode node, TabTrill& tabTrill)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Fret")
				tabTrill.setFretTrill(element.text().toUInt());
			else if (element.tagName() == "Period")
				tabTrill.setPeriod(element.text().toUInt());
		}
		
		node = node.nextSibling();
	}
}

void KGuitarTab::writeGraceNoteInformation(const QString& fill, QTextStream& stream, const TabGraceNote& tabGraceNote)
{
	stream << fill << "<Duration>"<< tabGraceNote.graceNoteDuration() << "</Duration>\n";
	stream << fill << "<Dynamic>"<< tabGraceNote.graceNoteDynamic() << "</Dynamic>\n";
	stream << fill << "<Fret>"<< tabGraceNote.graceNoteFret() << "</Fret>\n";
	stream << fill << "<Transition>"<< tabGraceNote.graceNoteTransition() << "</Transition>\n";
}

void KGuitarTab::readGraceNoteEffect(QDomNode node, TabGraceNote& graceNote)
{
	while (node.isNull() == false) {
		// Find title, subtitle, ...
		//
		if (node.isElement()) {
			QDomElement element = node.toElement();
			
			if (element.tagName() == "Duration")
				graceNote.setGraceNoteDuration(element.text().toUInt());
			else if (element.tagName() == "Dynamic")
				graceNote.setGraceNoteDynamic(static_cast<Dynamic>(element.text().toUInt()));
			else if (element.tagName() == "Fret")
				graceNote.setGraceNoteFret(element.text().toUInt());
			else if (element.tagName() == "Transition")
				graceNote.setGraceNoteTransition(static_cast<GraceNoteTransition>(element.text().toUInt()));
		}
		
		node = node.nextSibling();
	}
}
