#include "powertab.h"
#include "tabsong.h"

const uint PowerTab::nullTag = 0;
const uint PowerTab::newTagClass = 0xffff;
const uint PowerTab::tagClass = 0x8000;
const uint PowerTab::BigTagClass = 0x80000000;
const uint PowerTab::objectTag = 0x7fff;
const uint PowerTab::MaxMapCount = 0x3ffffffe;

PowerTab::PowerTab()
: version(0)
{
}

PowerTab::~PowerTab()
{
}

inline unsigned char PowerTab::readUChar(QFile& file) throw (const char*)
{
	int temp = 0;
	
	if ((temp = file.getch()) == -1)
		throw "[readUChar] Unable to read the file\n";
	
	return temp;
}

template <class T> inline const T PowerTab::readUCType(QFile& file) throw (const char*)
{
	int temp = 0;
	
	if ((temp = file.getch()) == -1)
		throw "[readUType] Unable to read the file\n";
	
	return static_cast<T>(temp);
}

inline unsigned int PowerTab::readUInt(QFile& file) throw (const char*)
{
	unsigned int temp = 0;
	
	if (file.readBlock(reinterpret_cast<char*>(&temp), 2) != 2)
		throw "[readUInt] Unable to read the file\n";
	
	return temp;
}

template <class T> inline const T PowerTab::readUIType(QFile& file) throw (const char*)
{
	return static_cast<T>(readUInt(file));
}

inline unsigned long PowerTab::readULong(QFile& file) throw (const char*)
{
	unsigned long temp = 0;
	
	if (file.readBlock(reinterpret_cast<char*>(&temp), 4) != 4)
		throw "[readULong] Unable to read the file\n";
	
	return temp;
}

inline unsigned long PowerTab::readMFCLength(QFile& file)
{
	unsigned long length = readUChar(file);
	if (length < 0xff)
		return length;
	
	length = readUInt(file);
	if (length < 0xffff)
		return length;
	
	length = readULong(file);
	
	return length;
}

inline QString PowerTab::readMFCString(QFile& file) throw (const char*)
{
	unsigned long length = readMFCLength(file);
	
	if (length) {
		QString tempString;
		char* temp = new char[length + 1];
		
		if (file.readBlock(temp, length) != length)
			 throw "[readMFCString] Unable to read the file\n";
		
		tempString = QString::fromLocal8Bit(temp, length);
		
		delete temp;
		
		return tempString;
	}
	
	return "";
}

int PowerTab::readHeaderItems(QFile& file, QString& name) throw (const char*)
{
	uint nbItems = 0;
	uint header = 0;
	
	name = "";
	
	nbItems = readUInt(file);
	
	if (nbItems == 0)
		return 0;
	
	header = readUInt(file);
	
	if (header == newTagClass) {
		QString section;
		int constValue = 0;
		int length = 0;
		
		constValue = readUInt(file);
		
		if (constValue != 1) {
			throw "[readHeaderItems] Unable to read the file\n";
		}
		
		length = readUInt(file);
		
		if (file.readLine(section, length + 1) == - 1) {
			throw "[readHeaderItems] Unable to read the file\n";
		}
		
		section =  QString::fromLocal8Bit(section, length + 1);
		
		name = section;
	}
	
	return nbItems;
}

inline void PowerTab::readItemNext(QFile& file) throw (const char*)
{
	int temp = readUInt(file);
	
	if (!(temp & 0x8000))
		throw "[readItemNext] Unable to read the file\n";
}

inline void PowerTab::callSection(QFile& file, int nbCall, TabSong& tabSong, void (PowerTab::*pf)(QFile&, TabSong&)) throw (const char*)
{
	for (int i = 0; i < nbCall; i++) {
		(this->*pf)(file, tabSong);
		
		if (i < nbCall - 1)
			readItemNext(file);
	}
}

inline void PowerTab::callSection(QFile& file, void (PowerTab::*pf)(QFile&)) throw (const char*)
{
	QString itemName;
	int itemCount = readHeaderItems(file, itemName);
	
	// CChordDiagram section
	//
	for (int i = 0; i < itemCount; i++) {
		(this->*pf)(file);
		
		if (i < itemCount - 1)
			readItemNext(file);
	}
}

void PowerTab::checkHeader(QFile& file) throw (const char*)
{
	const QString headerString("ptab");
	
	// Check if the header string is ptab
	//
	for (int i = 0; i < 4; i++) {
		if (headerString.ascii()[i] != file.getch())
			throw "[checkHeader] Unable to read the file\n";
	}
	
	version = readUInt(file);
	
	if (version < 1 || version > 4)
		throw "[checkHeader] Unable to read the file\n";
}

void PowerTab::readHeaderVersion1(QFile& file, TabSong& tabSong)
{
	bool live = false;
	int year = 0;
	QString comments, temp;
	release1::releaseType1 release;
	release1::authorType1 author;
	
	tabSong.setTitle(readMFCString(file));
	kdDebug() << "title = " << tabSong.title() << "\n";
	
	tabSong.setArtist(readMFCString(file));
	kdDebug() << "artist = " << tabSong.artist() << "\n";
	
	release = readUCType<release1::releaseType1>(file);
	kdDebug() << "releaseOn = " << release << "\n";
	
	comments = "Release type is ";
	switch (release) {
		case release1::Single:
			comments+="single.\n";
			break;
		case release1::Ep:
			comments+="ep.\n";
			break;
		case release1::Lp:
			comments+="lp.\n";
			break;
		case release1::DoubleLp:
			comments+="double lp.\n";
			break;
		case release1::TripleLp:
			comments+="triple lp.\n";
			break;
		case release1::Boxset:
			comments+="boxset.\n";
			break;
		case release1::BootLeg:
			comments+="bootleg.\n";
			break;
		case release1::Demo:
			comments+="demo.\n";
			break;
		case release1::Soundtrack:
			comments+="sountrack.\n";
			break;
		case release1::Video:
			comments+="video.\n";
			break;
		case release1::None:
			comments+="no type.\n";
			break;
	}
	
	tabSong.setSubTitle(readMFCString(file));
	kdDebug() << "release title = " << tabSong.subTitle() << "\n";
	
	if (live = readUChar(file))
		comments+="This was a live.\n";
	
	kdDebug() << "live = " << live << "\n";
	
	tabSong.setAuthor(readMFCString(file));
	kdDebug() << "composer = " << tabSong.author() << "\n";
	
	comments+="The lyricist name is " + readMFCString(file) + ".\n";
	kdDebug() << "lyricist = " << comments << "\n";
	
	comments+="The arranger name is " + readMFCString(file) + ".\n";
	kdDebug() << "arranger = " << comments << "\n";
	
	tabSong.setTranscriber(readMFCString(file));
	kdDebug() << "guitar score transcriber = " << tabSong.transcriber() << "\n";
	
	year = readUInt(file);
	kdDebug() << "year = " << year << "\n";
	
	comments+="Year is " + temp.setNum(year) + ".\n";
	
	author = readUCType<release1::authorType1>(file);
	kdDebug() << "author type = " << author << "\n";
	
	if (author == release1::Traditional)
		comments+="Author is traditionnal.\n";
	
	tabSong.setCopyright(readMFCString(file));
	kdDebug() << "copyright = " << tabSong.copyright() << "\n";
	
	comments+="The lyrics :\n" + readMFCString(file) + ".\n";
	kdDebug() << "lyrics = " << comments << "\n";
	
	tabSong.setInstructions(readMFCString(file));
	kdDebug() << "score notes = " << tabSong.instructions() << "\n";
	
	tabSong.setComments(comments);
}

void PowerTab::readHeaderVersion15(QFile& file, TabSong& tabSong)
{
	bool live = false;
	int year = 0;
	release1::releaseType1 release;
	release1::authorType1 author;
	QString comments, temp;
	
	tabSong.setTitle(readMFCString(file));
	kdDebug() << "title = " << tabSong.title() << "\n";
	
	tabSong.setArtist(readMFCString(file));
	kdDebug() << "artist = " << tabSong.artist() << "\n";
	
	release = readUCType<release1::releaseType1>(file);
	kdDebug() << "releaseOn = " << release << "\n";
	
	comments = "Release type is ";
	switch (release) {
	case release1::Single:
		comments+="single.\n";
		break;
	case release1::Ep:
		comments+="ep.\n";
		break;
	case release1::Lp:
		comments+="lp.\n";
		break;
	case release1::DoubleLp:
		comments+="double lp.\n";
		break;
	case release1::TripleLp:
		comments+="triple lp.\n";
		break;
	case release1::Boxset:
		comments+="boxset.\n";
		break;
	case release1::BootLeg:
		comments+="bootleg.\n";
		break;
	case release1::Demo:
		comments+="demo.\n";
		break;
	case release1::Soundtrack:
		comments+="sountrack.\n";
		break;
	case release1::Video:
		comments+="video.\n";
		break;
	case release1::None:
		comments+="no type.\n";
		break;
	}
	
	tabSong.setSubTitle(readMFCString(file));
	kdDebug() << "release title = " << tabSong.subTitle() << "\n";
	
	if (live = readUChar(file))
		comments+="This was a live.\n";
	
	kdDebug() << "live = " << live << "\n";
	
	tabSong.setAuthor(readMFCString(file));
	kdDebug() << "composer = " << tabSong.author() << "\n";
	
	comments+="The lyricist name is " + readMFCString(file) + ".\n";
	kdDebug() << "lyricist = " << comments << "\n";
	
	comments+="The arranger name is " + readMFCString(file) + ".\n";
	kdDebug() << "arranger = " << comments << "\n";
	
	year = readUInt(file);
	kdDebug() << "year = " << year << "\n";
	
	comments+="Year is " + temp.setNum(year) + ".\n";
	
	author = readUCType<release1::authorType1>(file);
	kdDebug() << "author type = " << author << "\n";
	
	if (author == release1::Traditional)
		comments+="Author is traditionnal.\n";
	    
	tabSong.setCopyright(readMFCString(file));
	kdDebug() << "copyright = " << tabSong.copyright() << "\n";
	
	comments+="The lyrics :\n" + readMFCString(file) + ".\n";
	kdDebug() << "lyrics = " << comments << "\n";
	
	tabSong.setComments(comments);
}

void PowerTab::readHeaderVersion17(QFile& file, TabSong& tabSong)
{
	release17::classificationSong classe;
	release1::authorType1 author;
	QString comments, temp;
	
	classe = readUCType<release17::classificationSong>(file);
	kdDebug() << "classification " << classe << "\n";
	
	switch(classe) {
	case release17::song: {
		release17::contentType content;
		release17::releaseType release;
		
		content = readUCType<release17::contentType>(file);
		kdDebug() << "content = " << content << "\n";
		
		comments = "This song contents ";
		switch (content) {
		case release17::None:
			comments+="no tracks.\n";
			break;
		case release17::Guitar:
			comments+="a guitar track.\n";
			break;
		case release17::Bass:
			comments+="a bass track.\n";
			break;
		case release17::Percussion:
			comments+="a percussion track.\n";
			break;
		case release17::All:
			comments+="a guitar, bass, and percussion track.\n";
			break;
		}
		
		tabSong.setTitle(readMFCString(file));
		kdDebug() << "title = " << tabSong.title() << "\n";
		
		tabSong.setArtist(readMFCString(file));
		kdDebug() << "artist = " << tabSong.artist() << "\n";
		
		release = readUCType<release17::releaseType>(file);
		kdDebug() << "release type = " << release << "\n";
		
		comments+= "Release type is ";
		switch (release) {
		case release17::publicAudio: {
			int year = 0;
			bool live = false;
			release17::audioRelease audio;
			
			comments+="public audio.\n";
			
			audio = readUCType<release17::audioRelease>(file);
			kdDebug() << "audio type = " << audio << "\n";
			
			comments+="The audio type is ";
			
			switch(audio) {
			case release17::Single:
				comments+="a single.\n";
				break;
			case release17::Ep:
				comments+="an EP.\n";
				break;
			case release17::Album:
				comments+="an album.\n";
				break;
			case release17::DoubleAlbum:
				comments+="a double album.\n";
				break;
			case release17::TripleAlbum:
				comments+="a triple album.\n";
				break;
			case release17::Boxset:
				comments+="a boxset.\n";
				break;
			}
			
			tabSong.setTitle(readMFCString(file));
			kdDebug() << "album title = " << tabSong.title() << "\n";
			
			year = readUInt(file);
			kdDebug() << "audio year = " << year << "\n";
			
			comments+="Year is " + temp.setNum(year) + ".\n";
			
			if (live = readUChar(file))
				comments+="This was a live.\n";
			
			kdDebug() << "is live recording = " << live << "\n";
		} break;
		case release17::publicVideo: {
			bool live = false;
			
			comments+="public video.\n";
			
			tabSong.setTitle(readMFCString(file));
			kdDebug() << "video title = " << tabSong.title() << "\n";
			
			if (live = readUChar(file))
				comments+="This was a live.\n";
			
			kdDebug() << "is live recording = " << live << "\n";
		} break;
		case release17::bootleg: {
			bool live = false;
			int day = 0, month = 0, year = 0;
			QDate date;
			
			comments+="bootleg.\n";
			
			tabSong.setTitle(readMFCString(file));
			kdDebug() << "bootleg title = " << tabSong.title() << "\n";
			
			if (live = readUChar(file))
				comments+="This was a live.\n";
			
			kdDebug() << "is live recording = " << live << "\n";
			
			day = readUInt(file);
			month = readUInt(file);
			year = readUInt(file);
			
			date.setYMD(year, month, day);
			
			comments+="Date of recording " + date.toString("dddd MMMM d yy") + ".\n";
			
			kdDebug() << "date = " << date.toString("dddd MMMM d yy") << "\n";
		} break;
		}
		
		author = readUCType<release1::authorType1>(file);
		kdDebug() << "is original author unknown = " << author << "\n";
		
		if (author == release1::AuthorUnknown) {
			tabSong.setAuthor(readMFCString(file));
			kdDebug() << "composer = " << tabSong.author() << "\n";
			
			comments+="The lyricist name is " + readMFCString(file) + ".\n";
			kdDebug() << "lyricist = " << comments << "\n";
		}
		
		comments+="The arrenger is " + readMFCString(file) + ".\n";
		kdDebug() << "arrenged by = " << comments << "\n";
		
		tabSong.setTranscriber(readMFCString(file));
		kdDebug() << "guitar score transcriber = " << tabSong.transcriber() << "\n";
		
		comments+="The bass transcriber is " + readMFCString(file) + ".\n";
		kdDebug() << "bass transcribed by = " << comments << "\n";
		
		tabSong.setCopyright(readMFCString(file));
		kdDebug() << "copyright = " << tabSong.copyright() << "\n";
		
		comments+="The lyrics :\n" + readMFCString(file) + ".\n";
		kdDebug() << "lyrics = " << comments << "\n";
		
		tabSong.setInstructions("Guitar : \n" + readMFCString(file) + ".\n\n");
		kdDebug() << "guitar notes = " << tabSong.instructions() << "\n";
		
		tabSong.setInstructions(tabSong.instructions() + "Bass : \n" + readMFCString(file) + ".\n\n");
		kdDebug() << "bass notes = " << tabSong.instructions() << "\n";
	} break;
	case release17::lesson: {
		release17::musicStyle style;
		release17::lessonLevel level;
		
		comments = "This is a lesson.\n";
		
		tabSong.setTitle(readMFCString(file));
		kdDebug() << "title = " << tabSong.title() << "\n";
		
		tabSong.setArtist(readMFCString(file));
		kdDebug() << "artist = " << tabSong.artist() << "\n";
		
		style = readUIType<release17::musicStyle>(file);
		kdDebug() << "lesson style = " << style << "\n";
		
		comments+="The style of the lesson is ";
		switch(style) {
		case release17::Alternative:
			comments+="alternative.\n";
			break;
		case release17::BlueGrass:
			comments+="blue grass.\n";
			break;
		case release17::Blues:
			comments+="blues.\n";
			break;
		case release17::Country:
			comments+="country.\n";
			break;
		case release17::FingerPick:
			comments+="finger picking.\n";
			break;
		case release17::Flamenco:
			comments+="flamenco.\n";
			break;
		case release17::Folk:
			comments+="folk.\n";
			break;
		case release17::Funk:
			comments+="funk.\n";
			break;
		case release17::Fusion: 
			comments+="fusion.\n";
			break;
		case release17::General:
			comments+="general.\n";
			break;
		case release17::Jazz:
			comments+="jazz.\n";
			break;
		case release17::Metal:
			comments+="metal.\n";
			break;
		case release17::Other:
			comments+="other.\n";
			break;
		case release17::Pop:
			comments+="pop.\n";
			break;
		case release17::Progressive:
			comments+="progressive.\n";
			break;
		case release17::Punk:
			comments+="punk.\n";
			break;
		case release17::Reggae:
			comments+="reggae.\n";
			break;
		case release17::Rock:
			comments+="rock.\n";
			break;
		case release17::Swing:
			comments+="swing.\n";
			break;
		}
		
		level = readUCType<release17::lessonLevel>(file);
		kdDebug() << "lesson level = " << level << "\n";
		
		comments+="Lesson level ";
		switch(level) {
		case release17::Beginner:
			comments+="beginner.\n";
			break;
		case release17::Intermediate:
			comments+="intermediate.\n";
			break;
		case release17::Advanced:
			comments+="advanced.\n";
			break;
		}
		
		tabSong.setAuthor(readMFCString(file));
		kdDebug() << "lesson author = " << tabSong.author() << "\n";
		
		tabSong.setInstructions("Guitar : " + readMFCString(file) + ".\n");
		kdDebug() << "guitar notes = " << tabSong.instructions() << "\n";
		
		tabSong.setCopyright(readMFCString(file));
		kdDebug() << "copyright = " << tabSong.copyright() << "\n";
	} break;
	}
	
	tabSong.setComments(comments);
}

void PowerTab::readDataInstruments(QFile& file, TabSong& tabSong, uint trackNb) throw (const char*)
{
	int itemCount = 0;
	QString itemName;
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CGuitar section
	//
	callSection(file, itemCount, tabSong, &PowerTab::readTrackInfo);
/*	for (int j = 0; j < itemCount; j++) {
		readTrackInfo(file, tabSong);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readChord);
	
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CChordDiagram section
	//
	for (int j = 0; j < itemCount; j++) {
		readChord(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readFloattingText);
	
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CFloatingText section
	//
	for (int j = 0; j < itemCount; j++) {
		readFloattingText(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readGuitarIn);
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CGuitarIn section
	//
	for (int j = 0; j < itemCount; j++) {
		readGuitarIn(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readTempoMarker);
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CTempoMarker
	//
	for (int j = 0; j < itemCount; j++) {
		readTempoMarker(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readDynamic);
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CDynamic section
	//
	for (int j = 0; j < itemCount; j++) {
		readDynamic(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	callSection(file, &PowerTab::readSectionSymbol);
/*	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CSectionSymbol section
	//
	for (int j = 0; j < itemCount; j++) {
		readSectionSymbol(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}*/
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	kdDebug() << "item name " << itemName << "\n";
	
	// CSection section
	//
	for (int j = 0; j < itemCount; j++) {
		readSection15(file, *tabSong.at(trackNb));
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
}

void PowerTab::readTrackInfo(QFile& file, TabSong& tabSong) throw (const char*)
{
	uchar trackNumber = 0;
	TabTrack* tabTrack = new TabTrack();
	
	// Track initialisation
	//
	tabTrack->setMode(FretTab);
	tabTrack->setNbFrets(21);
	
	// Track info
	//
	trackNumber = readUChar(file);
	kdDebug() << "number = " << trackNumber << "\n";
	
	tabTrack->setName(readMFCString(file));
	kdDebug() << "description = " << tabTrack->name() << "\n";
	
	tabTrack->setPatch(readUChar(file));
	kdDebug() << "preset = " << tabTrack->patch() << "\n";
	
	tabTrack->setVolume(readUChar(file));
	kdDebug() << "initial volume = " << tabTrack->volume() << "\n";
	
	tabTrack->setBalance(readUChar(file));
	kdDebug() << "pan = " << tabTrack->balance() << "\n";
	
	tabTrack->setReverb(readUChar(file));
	kdDebug() << "reverb = " << tabTrack->reverb() << "\n";
	
	tabTrack->setChorus(readUChar(file));
	kdDebug() << "chorus = " << tabTrack->chorus() << "\n";
	
	tabTrack->setTremolo(readUChar(file));
	kdDebug() << "tremolo = " << tabTrack->tremolo() << "\n";
	
	tabTrack->setPhaser(readUChar(file));
	kdDebug() << "phaser = " << tabTrack->phaser() << "\n";
	
	tabTrack->setCapo(readUChar(file));
	kdDebug() << "capo = " << tabTrack->capo() << "\n";
	
	// Tuning
	//
	kdDebug() << "tuning name = " << readMFCString(file) << "\n";
	kdDebug() << "bit 7 = Music notation offset sign, bits 6 to 1 = Music notation offset value, bit 0 = display sharps or flats = " << readUChar(file) << "\n";
	
	tabTrack->setNbStrings(readUChar(file));
	kdDebug() << "string count = " << tabTrack->nbStrings() << "\n";
	
	for (uint i = 0; i < tabTrack->nbStrings(); i++) {
		tabTrack->setTune(tabTrack->nbStrings()- i - 1, readUChar(file));
		kdDebug() << "note = " << tabTrack->tune(i) << "\n";
	}
	
	tabSong.append(tabTrack);
}

void PowerTab::readChord(QFile& file)
{
	uchar stringCount = 0;
	int chordModification = 0;
	unsigned long chordKey = 0;
	
	// Chord key and bass note (hiword = tonic, loword = bass note; bits 5-6 = variation, bottom 4 bits = key)
	//
	chordKey = readULong(file);
	kdDebug() << "Chord key and bass note (hiword = tonic, loword = bass note; bits 5-6 = variation, bottom 4 bits = key) = " << chordKey << "\n";
	
	// Core formula + flags (see formula and formulaFlags enums for values)
	//
	kdDebug() << "Core formula + flags (see formula and formulaFlags enums for values) = " << readUChar(file) << "\n";
	
	// Stores the formula modifications (see formulaModifications enum for values)
	//
	chordModification = readULong(file);
	kdDebug() << "Stores the formula modifications (see formulaModifications enum for values) = " << chordModification << "\n";
	
	// Stores the type and fret position data (top 3 bits = type, bottom 5 bits = position)
	//
	kdDebug() << "Stores the type and fret position data (top 3 bits = type, bottom 5 bits = position) = " << readUChar(file) << "\n";
	
	// Fret represented at the top of the chord diagram
	//
	kdDebug() << "Fret represented at the top of the chord diagram = " << readUChar(file) << "\n";
	
	stringCount = readUChar(file);
	
	for (uint j = 0; j < stringCount; j++) {
		// List of fret number offsets
		//
		kdDebug() << "List of fret number offsets = " << readUChar(file) << "\n";
	}
}

void PowerTab::readFontSetting(QFile& file)
{
	int pointSize = 0;
	int weight = 0;
	int color = 0;
	
	kdDebug() << "font name = " << readMFCString(file) << "\n";
	
	pointSize = readULong(file);
	kdDebug() << "point size = " << pointSize << "\n";
	
	weight = readULong(file);
	kdDebug() << "weigth = " << weight << "\n";
	
	kdDebug() << "italic = " << readUChar(file) << "\n";
	kdDebug() << "underline = " << readUChar(file) << "\n";
	kdDebug() << "strikeout = " << readUChar(file) << "\n";
	
	color = readULong(file);
	kdDebug() << "color = " << color << "\n";
}

void PowerTab::readFloattingText(QFile& file)
{
	unsigned long left = 0, top = 0, right = 0, bottom = 0;
	
	// Floating text
	//
	kdDebug() << "floating text = " << readMFCString(file) << "\n";
	
	// Read mfc rect
	//
	left   = readULong(file);
	top    = readULong(file);
	right  = readULong(file);
	bottom = readULong(file);
	
	kdDebug() << "mfc rect = " << left << " " << top << " " << right << " " << bottom << "\n";
	
	// Flags representing alignment and borders (see flags below)
	//
	kdDebug() << "Flags = " << readUChar(file) << "\n";
	
	// Font settings
	//
	readFontSetting(file);
}

void PowerTab::readGuitarIn(QFile& file)
{
	int system = 0;
	int data = 0;
	
	// Zero-based index of the system where the guitar in is anchored
	//
	system = readUInt(file);
	kdDebug() << "Zero-based index of the system where the guitar in is anchored = " << system << "\n";
	
	// Zero-based index of the staff within the system where the guitar in is anchored
	//
	kdDebug() << "Zero-based index of the staff within the system where the guitar in is anchored = " << readUChar(file) << "\n";
	
	// Zero-based index of the position within the system where the guitar in is anchored
	//
	kdDebug() << "Zero-based index of the position within the system where the guitar in is anchored = " << readUChar(file) << "\n";
	
	// Bit map representing the guitar number of the active guitars (bit 1 = guitar 0, bit 2 = guitar 1, etc.)
	// High byte = staff guitars, low byte = rhythm slash guitars
	//
	data = readUInt(file);
	kdDebug() << "Bit map representing the guitar number of the active guitars (bit 1 = guitar 0, bit 2 = guitar 1, etc.) = " << data << "\n";
}

void PowerTab::readTempoMarker(QFile& file)
{
	int system = 0;
	int data = 0;
	
	file.readBlock((char*)&system, 2);
	kdDebug() << "Zero-based index of the system the symbol is anchored : " << system << "\n";
	
	kdDebug() << "Zero-based index of the position within the system where the symbol is anchored = " << file.getch() << "\n";
	
	file.readBlock((char*)&data, 4);
	kdDebug() << "Data used by the symbol (different for each symbol) : " << data << "\n";
	
	kdDebug() << "description : " << readMFCString(file) << "\n";
}

void PowerTab::readDynamic(QFile& file)
{
	int system = 0;
	int data = 0;
	
	file.readBlock((char*)&system, 2);
	kdDebug() << "Zero-based index of the system where the dynamic is anchored : " << system << "\n";
	
	kdDebug() << "Zero-based index of the staff within the system where the dynamic is anchored : " << file.getch() << "\n";
	
	kdDebug() << "Zero-based index of the position within the system where the dynamic is anchored : " << file.getch() << "\n";
	
	file.readBlock((char*)&data, 2);
	kdDebug() << "Volume level (see volumeLevels enum for values; top byte = staff volume, bottom byte = dynamic volume) : " << data << "\n";
}

void PowerTab::readDirection(QFile& file)
{
	// Direction
	//
	kdDebug() << "Zero-based index of the position within the system where the direction is anchored : " << file.getch() << "\n";
	
	int symboleCount = 0;
	symboleCount = file.getch();
	
	kdDebug() << "symbole count : " << symboleCount << "\n";
	
	for (int i = 0; i < symboleCount; i++) {
		int word = 0;
		
		file.readBlock((char*)&word, 2);
		kdDebug() << "Array of direction symbols (symbol: top byte = symbol type, next 2 bits = active symbol, next 6 bits = repeat number) : " << word << "\n";
	}
}

void PowerTab::readNote(QFile& file, TabTimes& time) throw (const char*)
{
	uchar position = 0;
	int simpleData = 0;
	unsigned long complexSymbol = 0;
	TabNote note;
	
	position = readUChar(file);
	note.setFret(position & 0x1f);
	kdDebug() << "Top 3 bits = string, bottom 5 bits = fret number : " << position << "\n";
	
	file.readBlock((char*)&simpleData, 2);
	kdDebug() << "Contains simple symbol flags : " << simpleData << "\n";
	
	int symbolCount = 0;
	symbolCount = file.getch();
	
	kdDebug() << "symbol count " << symbolCount << "\n";
	
	for (int i = 0; i < symbolCount; i++) {
		file.readBlock((char*)&complexSymbol, 4);
		kdDebug() << "Contains simple symbol flags : " << complexSymbol << "\n";
	}
	
	time.setNotes((position & 0xe0) >> 5, note);
}

void PowerTab::readPosition(QFile& file, TabBar& bar) throw (const char*)
{
	int itemCount = 0;
	QString itemName;
	TabTimes time;
	// Position
	//
	unsigned int beaming = 0;
	unsigned long positionData = 0;
	
	kdDebug() << "Zero-based index of the position within the system where the position is anchored : " << readUChar(file) << "\n";
	
	beaming = readUInt(file);
	kdDebug() << "Beaming and irregular grouping timing data : " << beaming << "\n";
	
	positionData = readULong(file);
	kdDebug() << "Duration and simple symbol flags : " << positionData << "\n";
	
	int complexCount = 0;
	complexCount = readUChar(file);
	kdDebug() << "complexCount = "  << complexCount << "\n";
	
	for (int i = 0; i < complexCount; i++) {
		int symbolArray = 0;
		
		symbolArray = readULong(file);
		kdDebug() << "Symbole array : " << symbolArray << "\n";
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CLineData
	//
	for (int j = 0; j < itemCount; j++) {
		readNote(file, time);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	if (itemCount == 0)
		time.setRest(true);
	
	// Set the duration
	//
	switch ((positionData & 0xff000000) >> 24) {
	case 1:
		time.setDuration(Whole);
		break;
	case 2:
		time.setDuration(Half);
		break;
	case 4:
		time.setDuration(Quarter);
		break;
	case 8:
		time.setDuration(Eighth);
		break;
	case 16:
		time.setDuration(Sixteenth);
		break;
	}
	
	// Check if time is dotted
	//
	if (positionData & 0x01)
		time.setDotted(true);
	
	bar.insertTimes(time);
}

void PowerTab::readSectionSymbol(QFile& file)
{
	int system = 0;
	int data = 0;
	
	file.readBlock((char*)&system, 2);
	kdDebug() << "Zero-based index of the system the symbol is anchored : " << system << "\n";
	
	kdDebug() << "Zero-based index of the position within the system where the symbol is anchored = " << file.getch() << "\n";
	
	file.readBlock((char*)&data, 4);
	kdDebug() << "Data used by the symbol (different for each symbol) : " << data << "\n";
}

void PowerTab::readChordText(QFile& file)
{
	// Chordtext
	//
	kdDebug() << "Zero-based index of the position within the system where the chord text is anchored : " << file.getch() << "\n";
	
	int key = 0;
	file.readBlock((char*)&key, 2);
	kdDebug() << "Chord key and bass note (hiword = tonic, loword = bass note; bits 5-6 = variation, bottom 4 bits = key) : " << key << "\n";
	
	kdDebug() << "Core formula + flags (see formula and formulaFlags enums for values) ; " << file.getch() << "\n";
	
	int formulaModification = 0;
	file.readBlock((char*)&formulaModification, 2);
	kdDebug() << "Stores the formula modifications (see formulaModifications enum for values) ; " << formulaModification << "\n";
	
	kdDebug() << "Stores the type and fret position data (top 3 bits = type, bottom 5 bits = position) : " << file.getch() << "\n";
}

void PowerTab::readRhythmSlash(QFile& file)
{
	int rhythmData = 0;
	
	kdDebug() << "Zero-based index of the position within the system where the rhythm slash is anchored : " << file.getch() << "\n";
	kdDebug() << "Beaming and triplet data (top 3 bits = triplet flags, bottom 5 bits = beaming flags) : " << file.getch() << "\n";
	
	file.readBlock((char*)&rhythmData, 4);
	kdDebug() << " Top byte = single note data, next 3 bits = duration type, remaining bits are flags: " << rhythmData << "\n";
}

void PowerTab::readStaff(QFile& file, TabTrack& tabTrack) throw (const char*)
{
	TabBar bar;
	int itemCount = 0;
	QString itemName;
	
	// Staff
	//
	kdDebug() << "Top 4 bits = clef type, bottom 4 bits = tablature type : " << file.getch() << "\n";
	kdDebug() << "Amount of space alloted from the top line of the standard notation staff : " << file.getch() << "\n";
	kdDebug() << "Amount of space alloted from the last line of the standard notation staff : " << file.getch() << "\n";
	kdDebug() << "Amount of space alloted for symbols located between the standard notation and tablature staff : " << file.getch() << "\n";
	kdDebug() << "Amount of space alloted from the last line of the tablature staff : " << file.getch() << "\n";
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CPosition section
	//
	for (int j = 0; j < itemCount; j++) {
		readPosition(file, bar);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CPosition section
	//
	for (int j = 0; j < itemCount; j++) {
		readPosition(file, bar);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	tabTrack.addBar(bar);
}

void PowerTab::readKeySignature(QFile& file)
{
	kdDebug() << "Stores all data required by the key signature (see the flags enum for the meaning of the individual bits) : " << file.getch() << "\n";
}

void PowerTab::readTimeSignature(QFile& file)
{
	unsigned long data = 0;
	
	file.readBlock((char*)&data, 4);
	kdDebug() << "Stores the meter, beaming pattern and any flags (see flags enum for bit breakdown) : " << data << "\n";
	kdDebug() << "Number of pulses in a measure : " << file.getch() << "\n";
}

void PowerTab::readRehearsalSign(QFile& file)
{
	kdDebug() << "The letter used to uniquely identify the rehearsal sign (i.e. A, B, F, etc. - must be a capital letter) : " << file.getch() << "\n";
	
	kdDebug() << "A description that indicates the passage the rehearsal sign is marking (i.e. Chorus, Intro, etc.) : " << readMFCString(file) << "\n";
}

void PowerTab::readBarLine(QFile& file)
{
	kdDebug() << "Zero-based index of the position within the system where the barline is anchored : " << file.getch() << "\n";
	kdDebug() << "Top 3 bits = type, bottom 5 = repeat number : " << file.getch() << "\n";
	
	readKeySignature(file);
	readTimeSignature(file);
	readRehearsalSign(file);
}

void PowerTab::readSection15(QFile& file, TabTrack& tabTrack) throw (const char*)
{
	int left = 0, top = 0, right = 0, bottom = 0;
	int itemCount = 0;
	QString itemName;
	
	// Read mfc rect
	//
	file.readBlock((char*)&left, 4);
	file.readBlock((char*)&top, 4);
	file.readBlock((char*)&right, 4);
	file.readBlock((char*)&bottom, 4);
	
	kdDebug() << "mfc rect = " << left << " " << top << " " << right << " " << bottom << "\n";
	
	kdDebug() << "end bar " << file.getch() << "\n";
	
	kdDebug() << "Bounding rect for the system : " << file.getch() << "\n";
	
	kdDebug() << "Spacing between each position in the system : " << file.getch() << "\n";
	
	kdDebug() << "Spacing above the rhythm slashes : " << file.getch() << "\n";
	
	kdDebug() << "Spacing below the rhythm slashes : " << file.getch() << "\n";
	
	// BarLine
	//
	kdDebug() << "Zero-based index of the position within the system where the barline is anchored : " << file.getch() << "\n";
	
	kdDebug() << "Top 3 bits = type, bottom 5 = repeat number : " << file.getch() << "\n";
	
	// KeySignature
	//
	kdDebug() << "Stores all data required by the key signature (see the flags enum for the meaning of the individual bits) : " << file.getch() << "\n";
	
	// TimeSignature
	//
	int data = 0;
	
	file.readBlock((char*)&data, 4);
	kdDebug() << "Stores the meter, beaming pattern and any flags (see flags enum for bit breakdown) : " << data << "\n";
	
	kdDebug() << "Number of pulses in a measure : " << file.getch() << "\n";
	
	// ReHearsalSign
	//
	readRehearsalSign(file);
	
	// Direction array
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CDirection section
	//
	for (int j = 0; j < itemCount; j++) {
		readDirection(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// ChordText array
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CChordText section
	//
	for (int j = 0; j < itemCount; j++) {
		readChordText(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// Rythmslash
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CRhythmSlash section
	//
	for (int j = 0; j < itemCount; j++) {
		readRhythmSlash(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// Staff
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CStaff section
	//
	for (int j = 0; j < itemCount; j++) {
		readStaff(file, tabTrack);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// MusicBar
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CMusicBar section
	//
	for (int j = 0; j < itemCount; j++) {
		readBarLine(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
}

void PowerTab::readOldTimeSignature(QFile& file)
{
	uint system = 0;
	uchar position = 0;
	unsigned long data = 0;
	
	system = readUInt(file);
	position = readUChar(file);
	data = readULong(file);
}

void PowerTab::readDynamic1(QFile& file)
{
	uint system = 0;
	uchar staff = 0;
	uchar position = 0;
	uchar staffVolume = 0;
	
	system = readUInt(file);
	staff  = readUChar(file);
	position = readUChar(file);
	staffVolume = readUChar(file);
}

void PowerTab::readOldRehearsalSign(QFile& file)
{
	uint system = 0;
	uchar position = 0;
	unsigned long data = 0;
	uchar letter = 0;
	QString description;
	
	system = readUInt(file);
	position = readUChar(file);
	data = readULong(file);
	letter = readUChar(file);
	description = readMFCString(file);
}

void PowerTab::readSection1(QFile& file, TabTrack& tabTrack)
{
	int left = 0, top = 0, right = 0, bottom = 0;
	int itemCount = 0;
	QString itemName;
	
	// Read mfc rect
	//
	left = readULong(file);
	top = readULong(file);
	right = readULong(file);
	bottom = readULong(file);
	
	kdDebug() << "mfc rect = " << left << " " << top << " " << right << " " << bottom << "\n";
	
	kdDebug() << "key " << readUChar(file) << "\n";
	
	kdDebug() << "end bar " << file.getch() << "\n";
	
	kdDebug() << "Bounding rect for the system : " << file.getch() << "\n";
	
	kdDebug() << "Spacing between each position in the system : " << file.getch() << "\n";
	
	kdDebug() << "Spacing above the rhythm slashes : " << file.getch() << "\n";
	
	kdDebug() << "Spacing below the rhythm slashes : " << file.getch() << "\n";
	
	// Direction array
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CDirection section
	//
	for (int j = 0; j < itemCount; j++) {
		readDirection(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// ChordText array
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CChordText section
	//
	for (int j = 0; j < itemCount; j++) {
		readChordText(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// Rythmslash
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CRhythmSlash section
	//
	for (int j = 0; j < itemCount; j++) {
		readRhythmSlash(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// Staff
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CStaff section
	//
	for (int j = 0; j < itemCount; j++) {
		readStaff(file, tabTrack);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	// MusicBar
	//
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "item name " << itemName << "\n";
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CMusicBar section
	//
	for (int j = 0; j < itemCount; j++) {
		readBarLine(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
}

void PowerTab::readFormatVersion1(QFile& file, TabSong& tabSong, uint trackNb)
{
	int itemCount = 0;
	QString itemName;
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CGuitar section
	//
	for (int j = 0; j < itemCount; j++) {
		readTrackInfo(file, tabSong);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CChordDiagram section
	//
	for (int j = 0; j < itemCount; j++) {
		readChord(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CFloatingText section
	//
	for (int j = 0; j < itemCount; j++) {
		readFloattingText(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CGuitarIn section
	//
	for (int j = 0; j < itemCount; j++) {
		readGuitarIn(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CTempoMarker section
	//
	for (int j = 0; j < itemCount; j++) {
		readTempoMarker(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CTimeSignature
	//
	for (int j = 0; j < itemCount; j++) {
		readOldTimeSignature(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CDynamic
	//
	for (int j = 0; j < itemCount; j++) {
		readDynamic1(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CSectionSymbol section
	//
	for (int j = 0; j < itemCount; j++) {
		readSectionSymbol(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CRehearsalSign section
	//
	for (int j = 0; j < itemCount; j++) {
		readOldRehearsalSign(file);
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
	
	itemCount = readHeaderItems(file, itemName);
	
	kdDebug() << "nbItems " << itemCount << "\n";
	
	// CSection section
	//
	for (int j = 0; j < itemCount; j++) {
		readSection1(file, *tabSong.at(trackNb));
		
		if (j < itemCount - 1)
			readItemNext(file);
	}
}

TabSong PowerTab::loadFile(const QString &name) throw (const char*)
{
	// Open file
	//
	QFile file(name);
	TabSong tabSong;
	
	if (!file.open(IO_ReadOnly))
		throw "[loadFile] Unable to open the file\n";
	
	checkHeader(file);
	
	switch (version) {
	case 1:
	case 2: {
		unsigned long lineSpace = 0;
		
		kdDebug() << "Version 1.00 or 1.2\n";
		
		readHeaderVersion1(file, tabSong);
		
		lineSpace = readULong(file);
		kdDebug() << "line spaces = " << lineSpace << "\n";
		
		// Fonts info
		//
		for (int i = 0; i < 3; i++) {
			readFontSetting(file);
		}
		
		readFormatVersion1(file, tabSong, 0);
		
		readFormatVersion1(file, tabSong, 1);
		
		// Fade in Fade out
		//
	} break;
	case 3:
		kdDebug() << "Version 1.5\n";
		
		readHeaderVersion15(file, tabSong);
		
		// Guitar
		//
		readDataInstruments(file, tabSong, 0);
		
		// Bass
		//
		readDataInstruments(file, tabSong, 1);
		break;
	case 4:
		kdDebug() << "Version 1.7\n";
		
		readHeaderVersion17(file, tabSong);
		
		// Guitar
		//
		readDataInstruments(file, tabSong, 0);
		
		// Bass
		//
		readDataInstruments(file, tabSong, 1);
		break;
	};
	
	return tabSong;
}

bool PowerTab::saveFile(const QString&, const TabSong&)
{
	return false;
}
