/***************************************************************************
 * kguitar_part.h: implementation of KGuitarPart class
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#ifndef KGUITARPART_H
#define KGUITARPART_H

#include <header.h>

class SongView;
class TabTrack;
class TimidityLauncher;

// KGuitar KPart - main class of application, uses everything else.
/*
	Deals mostly with basic KDE GUI and configuration stuff. Real
	editor is composed in SongView, that, in turn, is composed from
	several song editing widgets.
*/
class KGuitarPart: public KParts::ReadWritePart
{
	Q_OBJECT
public:
	/*
		Inits the custom main widget and set up the actions.
	*/
	KGuitarPart(QWidget *parentWidget, const char * /*widgetName*/, QObject *parent, const char *name, const QStringList & /*args*/);
	
	/*
		Save the options and free the memory.
	*/
	virtual ~KGuitarPart();
	
	/*
		A shell will use this to inform this Part if it should act
		read-only
	*/
	virtual void setReadWrite(bool);
	
	/*
		Provide "About..." credits data, required by KPart
	*/
	static KAboutData *createAboutData();
	
public slots:
	virtual bool save();
	void filePrint();
	void viewMelodyEditor();
	void viewScore();
	void updateStatusBar(const QString& msg);
	
private slots:
	void fileOpenAs();
	void fileSaveAs();
	void clipboardDataChanged();
	void options();
	void saveOptions();
	void updateToolbars(TabTrack *);
	virtual void setModified(bool);
	
private:
	void setupActions();
	void setupAccels();
	void updateMenu();
	void setWinCaption(const QString &);
	void readOptions();
	void readMidiNames();
	bool exportOptionsDialog(QString);
	
	/*
		Configure the midi card with TSE3
	*/
	void midiConfigure();

	/*
		Main widget - the song view. Accessible from other places with
		widget() from KPart.
	*/
	SongView* songView;
	
	KAction *preferencesAct, *saveOptionAct, *pasteAct;
	KToggleAction *viewMelodyEditorAct, *viewScoreAct;
	KCommandHistory* cmdHist;
	
	/*
		This class will generate an appropriate MidiScheduler for the current platform.
		It provides a measure of platform indepenance in the generation of MidiSchedulers.
	*/
	TSE3::MidiSchedulerFactory factory;
	/*
		A MIDI interface providing scheduling facilities. 
		The MidiScheduler provides the interface to the MIDI system (be it a software or hardware driver).
		On top of this it implements a timed stream for reception and transmission of MidiEvents.
	*/
	TSE3::MidiScheduler* scheduler;
	
	/*
		Manager of transport facilities.
		This object handles the playback/record mechanism. It is the central control for musical input and output.
		Facilities provided by the Transport class include: 
			Play, Record into PhraseEdit, Fast forward and rewind, 
			Ability to attach callbacks which are kept informed of MIDI activity coming into or leaving the Transport object,
			Uses Panic objects (for start and stop reset sequences), a MidiEcho object,
			Synchro start (playback starts when MIDI input recieved),
			Uses a FlagTrack to move around the Playable easily, 
			Auto stop at the end of the Playable object,
			Lead in values for play and record ,
			Adaptive look ahead, Ability to 'inject' a MidiCommand into the Transport stream.
	*/
	TSE3::Transport* transport;
	
	/*
		This class provides the functionality of a metronome.
		The metronome remembers whether it should be produced in playing and recording modes,
		but it doesn't enforce this itself - it is the responsibility of the Transport playback routines.
	*/
	TSE3::Metronome* metronome;
	
	/*
		A timidity launcher.
	*/
	TimidityLauncher* timidityLauncher;
protected:
	virtual bool openFile();
	virtual bool saveFile();
};

#endif
