/***************************************************************************
 * musicxml.cpp: implementation of MusicXML classes
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2004 the KGuitar development team
 *
 * Copyright of the MusicXML file format:
 * (C) Recordare LLC. All rights reserved. http://www.recordare.com
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

// LVIFIX missing features:
// harmonics

// LVIFIX:
// improve error reporting

// LVIFIX:
// add bounds checking on all toInt() results
// check <score-partwise> (score-timewise is not supported)

// LVIFIX:
// MIDI bank, channel and instrument handling: where is it 0- or 1-based ?
// MusicXML 0.6 common.dtd:
// bank numbers range from 1 to 16,384.
// channel numbers range from 1 to 16.
// note numbers range from 1 to 128.
// program numbers range from 1 to 128.
// MIDI spec:
// channel 0..15
// patch (== program): 0..127
// KGuitar ???
// Note: the KGuitar user interface does not limit bank, channel and program
// to specific ranges and the instrument name is just a text input field

// LVIFIX:
// saving a file with empty "author" property results in an empty <encoder>,
// which is read back as three blanks

// LVIFIX:
// reading an xml file with size 0 results in sig 11
// reading an xml file without part-list results in sig 11
// reading an xml file without midi-instrument results in chn=bank=patch=0

// LVIFIX:
// all tracks are written in the guitar-specific format using two staves:
// - standard notation with clef-octave-change = -1
// - TAB (in alternate staff)

// LVIFIX:
// clef-octave-change, although present in the MusicXML file written,
// is not really supported:
// it is ignored when reading the file
// it is not used throughout KGuitar

// LVIFIX:
// accidentals are ignored in staff-tuning

// LVIFIX:
// check value of <backup>

// remarks on Qt's error handling (tested on Qt 2.3.1 and 3.0.5)
// - MusicXMLErrorHandler::warning(), error() and errorString() are not called
//   by Qt's parser, only fatalError() is called
// - when one of MusicXMLParser's handlers returns false, fatalError is called
//   with msg="error triggered by consumer"
// - a single error may result in many fatalError() calls
// - when fatalError() is called, the parseException contains valid columnnr,
//   linenr and message, but public and systemId are empty
// - failure to properly match start en end elements (e.g. <aaa></bbb>)
//   results in a "tag mismatch" error. To be able to report which tags
//   don't match, startElement/endElement would have to maintain a stack.

#include "global.h"
#include "accidentals.h"
#include "musicxml.h"
#include "tabsong.h"
#include "tabtrack.h"

using namespace std;		// for cout and friends

// local conversion functions

static QString kgNoteLen2Mxml(const int);
static int mxmlStr2Kg(const int, const int);
static int mxmlNoteType2Kg(const QString&);

// convert KGuitar notelength to MusicXML note type

static QString kgNoteLen2Mxml(const int kgNoteLen)
{
	switch (kgNoteLen)
	{
	case 480:
		return "whole";
	case 240:
		return "half";
	case 120:
		return "quarter";
	case  60:
		return "eighth";
	case  30:
		return "16th";
	case  15:
		return "32th";
	default:
		return "";
	}
}

// convert MusicXML string number to KGuitar string number
// this transformation is symmetrical, but depends on # strings
//
// bass(5):  EADGB
// MusicXML: 54321
// KGuitar:  01234
//
// guitar:   EADGBE
// MusicXML: 654321
// KGuitar:  012345

static int mxmlStr2Kg(const int mxmlStr, const int nStrings)
{
	return nStrings - mxmlStr;
}

// convert MusicXML note type to KGuitar notelength

static int mxmlNoteType2Kg(const QString& mxmlNoteType)
{
	if (mxmlNoteType == "whole") {
		return 480;
	} else if (mxmlNoteType == "half") {
		return 240;
	} else if (mxmlNoteType == "quarter") {
		return 120;
	} else if (mxmlNoteType == "eighth") {
		return  60;
	} else if (mxmlNoteType == "16th") {
		return  30;
	} else if (mxmlNoteType == "32th") {
		return  15;
	} else {
		return 0;
	}
}

// given pitch, allocate string/fret
// might also come in handy in the MIDI load function
// in:  pitch, tabtrack, current column
// out: result (TRUE=OK), (mxml)string, fret

// simple version:
// will only allocate notes on the highest possible string
// no conflict solving capabilities
// example: an A will always be allocated on the A string
// if a B is also needed, this will fail
// possible solutions would include:
// put B on 7th fret on E string
// put A on 5th fret on E string, B on 2nd fret on A string

// strange things will happen if strings are tuned in reverse order

static bool allocStrFrt(int pitch, TabTrack * trk, int col,
                        unsigned int& str, unsigned int& frt)
{
/*	// remove some boundary conditions
	// if no strings at all, we're out of luck
	if (trk->getNbStrings() == 0) {
		return FALSE;
	}
	// if pitch is lower than lowest string, we're out of luck
	if (pitch < trk->getTune(0)) {
		return FALSE;
	}

	int kgStr;
	if (trk->getNbStrings() == 1) {
		// must be on the one-and-only string
		kgStr = 0;
	} else {
		// assume on highest string
		kgStr = (trk->getNbStrings())-1;
		// but search for lower string to use
		// note that # strings >= 2
		for (int i = 0; i < (trk->getNbStrings())-1; i++) {
			if ((trk->getTune(i) <= pitch) && (pitch < trk->getTune(i+1))) {
				kgStr = i;
			}
		}
	}

	// check string not in use
	if (trk->getVectTabColumn(col).getNbFret(kgStr) >= 0) {
		return FALSE;
	}

	str = mxmlStr2Kg(kgStr, trk->getNbStrings());
	frt = pitch - trk->getTune(kgStr);
	return TRUE;*/
}


// Class MusicXMLErrorHandler

MusicXMLErrorHandler::MusicXMLErrorHandler()
{
/*	fatalReported = false;
	parser = 0;*/
}

bool MusicXMLErrorHandler::warning(const QXmlParseException& exception)
{
/*	cerr << "MusicXMLErrorHandler::warning"
		<< " col=" << exception.columnNumber()
		<< " line=" << exception.lineNumber()
		<< " msg=" << exception.message()
		<< " pid=" << exception.publicId()
		<< " sid=" << exception.systemId()
		<< endl;
	return true;	// continue parsing*/
}

bool MusicXMLErrorHandler::error(const QXmlParseException& exception)
{
/*	cerr << "MusicXMLErrorHandler::error"
		<< " col=" << exception.columnNumber()
		<< " line=" << exception.lineNumber()
		<< " msg=" << exception.message()
		<< " pid=" << exception.publicId()
		<< " sid=" << exception.systemId()
		<< endl;
	return true;	// continue parsing*/
}

bool MusicXMLErrorHandler::fatalError(const QXmlParseException& exception)
{
/*	if (exception.message() == "error triggered by consumer") {
		// no need to report this: should already have been done
		// by MusicXMLParser's handler
		fatalReported = true;
	} else {
		if (!fatalReported) {
			if (parser) {
				parser->reportError(exception.message());
			} else {
				cerr << "MusicXMLErrorHandler::fatalError"
					<< " parser=0" << endl;
			}
			fatalReported = true;
		}
	}
	return false;	// do not continue parsing*/
}

QString MusicXMLErrorHandler::errorString()
{
// 	return "KGuitar musicxmlimport error string";
}

void MusicXMLErrorHandler::setParser(MusicXMLParser * p)
{
// 	parser = p;
}


// Class MusicXMLParser

// MusicXMLParser constructor
//
MusicXMLParser::MusicXMLParser(TabSong * tsp)
	: QXmlDefaultHandler()
{
	// need to remember tabsong for callbacks
// 	ts = tsp;
}

// set document locator handler

void MusicXMLParser::setDocumentLocator(QXmlLocator *locator)
{
// 	lctr = locator;
}

// start of document handler

bool MusicXMLParser::startDocument()
{
/*	// init tabsong
	ts->setTempo(120);			// default start tempo
	ts->clear();				// no tracks read yet: clear track list
	ts->setTitle("");				// default title
	ts->setAuthor("");			// default author
	ts->setTranscriber("");		// default transcriber
	ts->setInstructions("");			// default comments
// 	ts->filename = "";			// is set in KGuitarPart::slotOpenFile()
	// init global variables: clear part list
	partIds.clear();
	// init global variables: characters collected
	stCha = "";
	// init global variables: identification
	stCrt = "";
	stEnc = "";
	stTtl = "";
	// init global variables: measure, default to 4/4
	// do note re-init between measures
	stBts = "4";
	stBtt = "4";
	stDiv = "";
	stFif = "";
	iDiv = 0;
	return TRUE;*/
}

// start of element handler

// Note: on reading the following input
//
// <score-part id="P1">
//   <part-name></part-name>
//     <score-instrument id="P1-I1">
//       <instrument-name>Voice</instrument-name>
//
// the parser calls
//
// startElement("score-part")
// characters("\n")
// characters("  ")
// startElement("part-name")
// endElement("part-name")
// characters("\n")
// characters("    ")
// startElement("score-instrument")
// characters("\n")
// characters("      ")
// startElement("instrument-name")
// characters("Voice")
// endElement("instrument-name")
//
// As characters() is not called between startElement("part-name") and
// endElement("part-name"), stCha needs to be cleared at each startElement().
// Failing to do so results in reading (previous) whitespace for empty
// elements such as the part-name in this example.

bool MusicXMLParser::startElement( const QString&, const QString&,
                                   const QString& qName,
                                   const QXmlAttributes& attributes)
{
/*	stCha = "";		// see note above
	if (false) {
	} else if (qName == "glissando") {
		QString tp = attributes.value("type");
		if (tp == "start") {
			stGls = TRUE;
		}
	} else if (qName == "hammer-on") {
		QString tp = attributes.value("type");
		if (tp == "start") {
			stHmr = TRUE;
		}
	} else if (qName == "measure") {
		// add a bar (measure) to the track
		// if not first bar: copy attributes from previous bar
		// note: first bar's default attributes set in TabTrack's constructor
		// LVIFIX: maybe don't add first measure here
		// (already done in TabTrack's constructor) ?
		if (trk) {
			TabBar tab = trk->getVectTabBar(bar - 1);
			
			bar++;
			trk->tabBarResize(bar);
			
// 			tab.start = x;
			//trk->getTabBar(bar - 1).start=x;
			
			if (bar > 1)
			{
				tab.time1 = trk->getVectTabBar(bar - 2).time1;
				tab.time2 = trk->getVectTabBar(bar - 2).time2;
// 				trk->getTabBar(bar - 1).time1=trk->getTabBar(bar - 2).time1;
// 				trk->getTabBar(bar - 1).time2=trk->getTabBar(bar - 2).time2;
			}
			
			trk->setVectTabBar(bar - 1, tab);
		}
		tStartCur = -1;			// undefined
    } else if (qName == "note") {
    	// re-init note specific variables
		initStNote();
	} else if (qName == "part") {
		// start of track data found
    	// use part id to switch to correct track
		QString id = attributes.value("id");
		int index = -1;
		for (unsigned int i = 0; i < partIds.size(); i++) {
			if (id.compare(*partIds.at(i)) == 0) {
				index = i;
			}
		}
		if (index == -1) {
			// track id not found
			trk = NULL;
		} else {
			// init vars for track reading
			x = 0;
			bar = 0;
			ts->at(index);
			trk = ts->current();
			tEndCur = 0;
		}
	} else if (qName == "pull-off") {
		QString tp = attributes.value("type");
		if (tp == "start") {
			stPlo = TRUE;
		}
	} else if (qName == "score-part") {
		// start of track definition found
	    // re-init score part specific variables
		initStScorePart();
	    stPid = attributes.value("id");
	} else if (qName == "sound") {
		ts->setTempo(attributes.value("tempo").toInt());
	} else if (qName == "staff-tuning") {
	    // re-init staff tuning specific variables
		initStStaffTuning();
	    stPtl = attributes.value("line");
	} else if (qName == "tie") {
		QString tp = attributes.value("type");
		if (tp == "stop") {
			stTie = TRUE;
		}
	} else {
	    // others (silently) ignored
	}
	return TRUE;*/
}

// end of element handler

bool MusicXMLParser::endElement( const QString&, const QString&, const QString& qName)
{
	return TRUE;
}

// character(s) handler

bool MusicXMLParser::characters(const QString& ch)
{
/*	stCha = ch;*/
	return TRUE;
}

// add a note to the current track

bool MusicXMLParser::addNote()
{
    // string conversions
/*	bool ok1;
	bool ok2;
	bool ok3;
	bool ok4;
	bool ok5;
	bool ok6;
	unsigned int frt = stFrt.toUInt(&ok1);
	unsigned int str = stStr.toUInt(&ok2);
	unsigned int ano = stAno.toUInt(&ok3);
	unsigned int nno = stNno.toUInt(&ok4);
	         int alt = stAlt.toInt( &ok5);
	unsigned int oct = stOct.toUInt(&ok6);
	int          len = mxmlNoteType2Kg(stTyp);

	// sanity checks
	if ((trk == NULL) || (len == 0)) {
		initStNote();
		return TRUE;			// LVIFIX: how to report error ?
	}

	int  nnDur   = len;			// new note duration incl dot/triplet
	uint nnFlags = 0;			// new note flags

	// handle dot (LVIFIX: more than one not supported by KGuitar)
    if (stDts) {
		nnDur = nnDur * 3 / 2;
	    nnFlags |= FLAG_DOT;
	}

	// handle triplets
	if (ok3 && ok4 && (ano == 3) && (nno == 2)) {
		nnDur = nnDur * 2 / 3;
		nnFlags |= FLAG_TRIPLET;
	}

	// append note to current track
	if (stCho) {
		if (tStartCur < 0) {
			// LVIFIX: report error ?
			kdDebug() << "<chord> at start of measure of after backup/forward"
			          << endl;
			// pretend to be appending
			tStartCur = tEndCur;
		}
		tEndCur = tStartCur + nnDur;
	} else {
		tStartCur = tEndCur;
		tEndCur += nnDur;
	}
	int ncols = trk->insertColumn(tStartCur, tEndCur);
	x = trk->getTabColumn() + 1;

	if (stRst) {
		// kdDebug() << "rest, l=" << len << endl;
	}

	// if not rest or tie: fill in fret (if rest: frets stay -1)
	// placed here on purpose:
	// in case of failure, fret stays -1 which is a safe value
	if (!stRst && !stTie) {
		if (!ok1 || !ok2) {
			// no valid frt/str: try if stp/alt/oct can be used instead
			// note: alt may be missing
			if ((stStp == "") || !ok6) {
				// no valid stp/alt/oct
				initStNote();
				return TRUE;	// LVIFIX: how to report error ?
			} else {
				Accidentals acc;
				int pitch = acc.sao2Pitch(stStp, alt, oct);
				if (!allocStrFrt(pitch, trk, x-1, str, frt)) {
					kdDebug() << "MusicXMLParser::addNote() ";
					kdDebug() << "string/fret allocation failed, ";
					kdDebug() << "column=" << x << endl;
				}
			}
		}
		// set string/fret
		// LVIFIX: check valid range for frt and str
		int kgStr = mxmlStr2Kg(str, trk->getNbStrings());
		
		TabColumn tab = trk->getVectTabColumn(x - 1);
		
		tab.setNbFret(kgStr, frt);
// 		trk->getTabColumn(x - 1).getNbFret(kgStr) = frt;
		
		// if note spans multiple columns, then set ringing
		if (ncols > 1) 
		{
			tab.setEffect(kgStr, EFFECT_LETRING);
// 			trk->getTabColumn(x - 1).getEffect(kgStr) = EFFECT_LETRING;
			
			// stop ringing at column x+ncols-1 (if it exists)
			// needed only if x-1 has note and x+ncols-1 hasn't
			if ((unsigned) x < (trk->getTabColumnSize() - ncols + 1)) 
			{
				if (trk->getVectTabColumn(x + ncols - 1).getNbFret(kgStr) < 0) 
				{
					TabColumn tabCol = trk->getVectTabColumn(x + ncols - 1);
// 					trk->getTabColumn(x + ncols - 1).getEffect(kgStr) = EFFECT_STOPRING;
					tabCol.setEffect(kgStr, EFFECT_STOPRING);
					trk->setVectTabColumn(x + ncols - 1, tabCol);
				}
			}
		}
		
		// handle slide, hammer-on and pull-off
		// last two get higher priority
		// (EFFECT_LEGATO overwrites EFFECT_SLIDE)
		if (stGls) 
		{
			tab.setEffect(kgStr, EFFECT_SLIDE);
// 			trk->getTabColumn(x - 1).getEffect(kgStr) = EFFECT_SLIDE;
		}
		
		if (stHmr || stPlo) 
		{
			tab.setEffect(kgStr, EFFECT_LEGATO);
// 			trk->getTabColumn(x - 1).getEffect(kgStr) = EFFECT_LEGATO;
		}
		
		trk->setVectTabColumn(x - 1, tab);
	}

	TabColumn tab = trk->getVectTabColumn(x - 1);
	
	// handle tie
	if (stTie && (x>0)) 
	{
		tab.setFlags(tab.getFlags() | FLAG_ARC);
	}

	trk->setVectTabColumn(x - 1, tab);
    // re-init note specific variables
	initStNote();*/
	return TRUE;
}

// add a track to the current song

bool MusicXMLParser::addTrack()
{
	// new track found, append it
	// note: TabTracks contructor initializes all required member variables,
	// tune[0..5] is set to guitar standard tuning
	// LVIFIX (in tabtrack.cpp): init all other members of TabTrack.tune[]
	// current code gives uninitialized tuning if #strings > 6, but no tuning
	// specified
/*	TabTrack * trk = new TabTrack(
		FretTab,      // _tm LVIFIX: no support for drumtrack
		stPnm,                  // _name
		stPmc.toInt(),          // _channel
		stPmb.toInt(),          // _bank
		stPmp.toInt(),          // _patch (=program)
		6,                      // _string (default value)
		24                      // _frets (default value)
	);
	ts->append(trk);
	// don't want any columns yet, as that would interfere
	// with the first note's timing
	// LVIFIX: add a single column to empty tracks after loading mxml
	trk->tabColumnResize(0);
	// remember part id to track nr mapping
	QString *sp = new QString(stPid);
	int sz = partIds.size();
	partIds.resize(sz+1);
	partIds.insert(sz, sp);
	return TRUE;*/
}

// initialize note state variables

void MusicXMLParser::initStNote()
{
/*	stAlt = "";
	stAno = "";
	stCho = FALSE;
	stDts = 0;
	stDur = "";
	stFrt = "";
	stGls = FALSE;
	stHmr = FALSE;
	stNno = "";
	stOct = "";
	stPlo = FALSE;
	stRst = FALSE;
	stStp = "";
	stStr = "";
	stTie = FALSE;
	stTyp = "";*/
}

// initialize part state variables

void MusicXMLParser::initStScorePart()
{
/*    stPid = "";
    stPmb = "";
    stPmc = "";
    stPmp = "";
    stPnm = "";*/
}

// initialize tuning state variables

void MusicXMLParser::initStStaffTuning()
{
/*    stPtl = "";
//    stPtn = "";
    stPto = "";
    stPts = "";*/
}


// helpers for the parser

// report all (fatal and non-fatal) errors
// LVIFIX: in future, might show a dialog

void MusicXMLParser::reportAll(const QString& lvl, const QString& err)
{
//	QString filename(parser_params.fname);
/*	QString filename("<add filename>");	// LVIFIX
	QString fullErr;
	QString linenr;
	linenr.setNum(lctr->lineNumber());
	fullErr  = "";
	fullErr += lvl;
	fullErr += ": In ";
	fullErr += filename;
	fullErr += " line ";
	fullErr += linenr;
	fullErr += ": ";
	fullErr += err;
	fullErr += "\n";
	cerr << fullErr;*/
}


// report a warning (non-fatal error, i.e. one which allows parsing to continue)

void MusicXMLParser::reportWarning(const QString& err)
{
// 	reportAll("Warning", err);
}


// report a fatal error

void MusicXMLParser::reportError(const QString& err)
{
// 	reportAll("Error", err);
}


// helpers for NMusicXMLExport::calcDivisions

typedef QValueList<int> IntVector;
static IntVector integers;
static IntVector primes;

static void addInt(int len) {
	IntVector::Iterator it = integers.find(len);
	if (it == integers.end()) {
		integers.append(len);
	}
}
	
// check if all integers can be divided by div

static bool canDivideBy(int div) {
	bool res = true;
	for (unsigned int i = 0; i < integers.count(); i++) {
		if ((integers[i] <= 1) || ((integers[i] % div) != 0)) {
			res = false;
		}
	}
	return res;
}

// divide all integers by div

static void divideBy(int div) {
	for (unsigned int i = 0; i < integers.count(); i++) {
		integers[i] /= div;
	}
}

// Loop over all voices in all staffs and determine a suitable value for divisions.

// Length of time in MusicXML is expressed in "units", which should allow expressing all time values
// as an integral number of units. Divisions contains the number of units in a quarter note.
// The way KGuitar stores note length meets this requirement, but sets divisions to a very
// large number: length of a quarter note is 120. Solution is to collect all time values required,
// and divide them by the highest common denominator, which is implemented as a series of
// divisions by prime factors. Initialize the list with 120 to make sure a quarter note can always
// be written as an integral number of units.

void MusicXMLWriter::calcDivisions() {
}

// MusicXMLWriter constructor

MusicXMLWriter::MusicXMLWriter(TabSong * tsp)
{
  // need to remember tabsong
  ts = tsp;
}

// write tabsong to QTextStream os

void MusicXMLWriter::write(QTextStream& os)
{
/*	calcDivisions();
	os << "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>"
	   << endl;
	os << "<!DOCTYPE score-partwise PUBLIC" << endl;
	os << "    \"-//Recordare//DTD MusicXML 1.0 Partwise//EN\"" << endl;
	os << "    \"http://www.musicxml.org/dtds/partwise.dtd\">" << endl;
	os << endl;
	os << "<score-partwise>\n";
	os << "\t<work>\n";
	os << "\t\t<work-title>" << ts->getTitle() << "</work-title>\n";
	os << "\t</work>\n";

// identification
	os << "\n";
	os << "\t<identification>\n";
	os << "\t\t<creator type=\"composer\">" << ts->getAuthor() << "</creator>\n";
	os << "\t\t<encoding>\n";
	os << "\t\t\t<encoder>" << ts->getTranscriber() << "</encoder>\n";
	os << "\t\t\t<software>KGuitar</software>\n";
	os << "\t\t</encoding>\n";
	os << "\t</identification>\n";

// part list
	os << "\n";
	os << "\t<part-list>\n";
	// loop over all tracks
	for (unsigned int it = 0; it < ts->count(); it++) {
		os << "\t\t<score-part id=\"P" << it+1 << "\">\n";
		os << "\t\t\t<part-name>" << ts->at(it)->getName() << "</part-name>\n";
		// LVIFIX: fill-in real instrument-name instead of "Guitar"
		// note: in DTD 0.6 score-instrument may appear zero or more times
		//       within a score-part
		// note: in DTD 0.7a score-instrument apparently is required
		os << "\t\t\t<score-instrument id=\"P" << it+1
		   << "-I" << it+1 << "\">\n";
		os << "\t\t\t\t<instrument-name>" << "Guitar"
		   << "</instrument-name>\n";
		os << "\t\t\t</score-instrument>\n";
		os << "\t\t\t<midi-instrument id=\"P" << it+1
		   << "-I" << it+1 << "\">\n";
		os << "\t\t\t\t<midi-channel>" << ts->at(it)->getChannel()
		   << "</midi-channel>\n";
		os << "\t\t\t\t<midi-bank>" << ts->at(it)->getBank()
		   << "</midi-bank>\n";
		os << "\t\t\t\t<midi-program>" << ts->at(it)->getPatch()
		   << "</midi-program>\n";
		os << "\t\t\t</midi-instrument>\n";
		os << "\t\t</score-part>\n";
	} // end for (unsigned int it = 0; ...
	os << "\t</part-list>\n";

// parts
	TabTrack *trk;
	// loop over all tracks
	for (unsigned int it = 0; it < ts->count(); it++) {
		trk = ts->at(it);
		trk->calcVoices();
		trk->calcStepAltOct();
		trk->calcBeams();
		os << "\n";
		os << "\t<part id=\"P" << it+1 << "\">\n";

		// loop over all bars
		for (uint ib = 0; ib < trk->getTabBarSize(); ib++) {
			os << "\t\t<measure number=\"" << ib + 1 << "\">\n";
			if (ib == 0) {
				// First bar: write all attributes
				os << "\t\t\t<attributes>\n";
				os << "\t\t\t\t<divisions>" << divisions << "</divisions>\n";
				os << "\t\t\t\t<key>\n";
				os << "\t\t\t\t\t<fifths>" << trk->getVectTabBar(0).keysig << "</fifths>\n";
				// LVIFX: re-enable when KGuitar supports major/minor modes
				// os << "\t\t\t\t\t<mode>major</mode>\n";
				os << "\t\t\t\t</key>\n";
				writeTime(os, trk->getVectTabBar(0).time1, trk->getVectTabBar(0).time2);
				os << "\t\t\t\t<staves>2</staves>\n";
				os << "\t\t\t\t<clef number=\"1\">\n";
				os << "\t\t\t\t\t<sign>G</sign>\n";
				os << "\t\t\t\t\t<line>2</line>\n";
				os << "\t\t\t\t\t<clef-octave-change>-1</clef-octave-change>\n";
				os << "\t\t\t\t</clef>\n";
				os << "\t\t\t\t<clef number=\"2\">\n";
				os << "\t\t\t\t\t<sign>TAB</sign>\n";
				os << "\t\t\t\t\t<line>5</line>\n";
				os << "\t\t\t\t</clef>\n";
				writeStaffDetails(os, trk);
				os << "\t\t\t</attributes>\n";
				os << "\t\t\t<sound tempo=\"" << ts->getTempo() << "\"/>\n";
			} else {
				// LVIFIX write time sig if changed
			}

			os << "\t\t</measure>\n";
			os << "\n";
		} // end for (uint ib = 0; ...

		os << "\t</part>\n";
	} // end for (unsigned int it = 0; ...
	os << "\n";
	os << "</score-partwise>\n";*/
}

// write accidental of midi note n to QTextStream os

QString MusicXMLWriter::strAccid(Accidentals::Accid acc)
{
/*	QString s;
	switch (acc) {
		case Accidentals::Natural: s = "natural"; break;
		case Accidentals::Sharp:   s = "sharp";   break;
		case Accidentals::Flat:    s = "flat";    break;
		default:                   s = "unknown"; break;
	}*/
	return "";
}

// write beam bm at level l to QTextStream os

static void writeBeam(QTextStream& os, int l, char bm)
{
/*	if (bm == 'n') {
		return;
	}
	os << "\t\t\t\t<beam number=\"";
	os << l << "\">";
	switch (bm) {
		case 'b': os << "backward hook"; break;
		case 'c': os << "continue";      break;
		case 'e': os << "end";           break;
		case 'f': os << "forward hook";  break;
		case 's': os << "begin";         break;
		default:  /* nothing  ;        break;
	}
	os << "</beam>\n";*/
}

// write beams of voice v of column x of TabTrack trk to QTextStream os

void MusicXMLWriter::writeBeams(QTextStream& os, TabTrack * trk, int x, int v)
{
/*	const StemInfo * stxt = 0;
	if (v == 0) {
		stxt = & trk->getVectTabColumn(x).getLowerStems();
	} else {
		stxt = & trk->getVectTabColumn(x).getUpperStems();
	}*/
	/*
	kdDebug()
		<< "writeBeams()"
		<< " x=" << x
		<< " v=" << v
		<< " l1..3=" << stxt->l1 << stxt->l2 << stxt->l3 << endl;
	*/
/*	writeBeam(os, 1, stxt->l1);
	writeBeam(os, 2, stxt->l2);
	writeBeam(os, 3, stxt->l3);*/
}

// write voice v of column x of TabTrack trk to QTextStream os
// if wrt is true then write MusicXML else calculate timing only
// update triplet state
// return ncols used

// writeCol must write rest:
// single voice: only in voice 1
// multi voice: only in voice 0
// LVIFIX: cause of this is that in a single voice track all notes
// are allocated to voice 1 to force stem up when printing

int MusicXMLWriter::writeCol(QTextStream& os, TabTrack * trk, uint x, int v, bool wrt)
{
	// debug: dump this column
	/*
	os << "x=" << x;
	os << " a[i]/e[i]/v[i]=";
	for (int i = 0; i < trk->getNbStrings(); i++) {
		os << " " << (int) trk->getTabColumn(x).getNbFret(i)
		   << "/" << (int) trk->getTabColumn(x).getEffect(i)
		   << "/" << (int) trk->getTabColumn(x).getVoiceToNote(i);
	}
	os << " fl=" << (int) trk->getTabColumn(x).getFlags();
	os << endl;
	*/
	// end debug: dump this column

/*	if (trk->getVectTabBar(trk->barNr(x)).start == x) {
		// do start of bar specific init
		tEndPrev  = 0;
		trpCnt    = 0;
		tStartCur = 0;
		if ((v != 0) && trk->hasMultiVoices()) {
			if (wrt) {
				// write backup
				os << "\t\t\t<backup>" << endl;
				os << "\t\t\t\t<duration>"
				   << trk->currentBarDuration() * divisions / 120
				   << "</duration>\n";
				os << "\t\t\t</backup>" << endl;
			} else {
				addInt(trk->currentBarDuration());
//				cout << "backup " << trk->currentBarDuration() << endl;
			}
		}
	}*/
	// debug info
	/*
	os << "tEndPrev=" << tEndPrev << " tStartCur=" << tStartCur;
	if (tStartCur - tEndPrev) {
		os << " -> forward: " << tStartCur - tEndPrev;
	}
	os << endl;
	*/
	// end debug info
/*	if (tStartCur - tEndPrev) {
		if (wrt) {
			os << "\t\t\t<forward>" << endl;
			os << "\t\t\t\t<duration>"
			   << (tStartCur - tEndPrev) * divisions / 120
			   << "</duration>\n";
			os << "\t\t\t\t<voice>" << v + 1 << "</voice>\n";
			os << "\t\t\t</forward>" << endl;
		} else {
			addInt(tStartCur - tEndPrev);
//			cout << "forward " << tStartCur - tEndPrev << endl;
		}
		tEndPrev = tStartCur;
	}

	int dots = 0;				// # dots
	bool triplet = false;		// triplet flag
	int duration = 0;			// note duration (incl. dot/triplet)
	int fret;
	int length = 0;				// note length (excl. dot/triplet)
	int nCols = 1;				// # columns used (default 1)
	int nNotes = 0;				// # notes printed in this column
	int nRests = 0;				// # rests printed in this column
	int res;

	// LVIFIX: error handling ?
	res = trk->getNoteTypeAndDots(x, v, length, dots, triplet);*/
/*
	if (!wrt) {
		cout
		<< "x=" << x
		<< " res=" << res
		<< " tp=" << length
		<< " dt=" << dots
		<< " tr=" << triplet
		<< endl;
	}
*/

	// tie handling:
	// if the next column is linked with this one, start a tie
	// if the this column is linked with the previous one, end a tie
	// LVIFIX:
	// KGuitar stores the second column of a tie as a rest (an empty column),
	// while MusicXML requires notes there. Therefore take the notes from the
	// previous column.
	// See also: songprint.cpp SongPrint::drawBar()
	// LVIFIX:
	// "previous" should be "first column of the set of tied columns"
	// (there may be more than two)
/*	bool tieStart = FALSE;
	bool tieStop  = FALSE;
	int  xt = x;				// x where tie starts
	if (((unsigned)(x+1) < trk->getTabColumnSize()) && (trk->getVectTabColumn(x + 1).getFlags() & FLAG_ARC)) {
		tieStart = TRUE;
	}
	if ((x > 0) && (trk->getVectTabColumn(x).getFlags() & FLAG_ARC)) {
		tieStop = TRUE;
		xt = x - 1;				// LVIFIX: handle more than one tie
	}

	// triplet handling:
	// - reset after third note of triplet
	// - count notes while inside triplet
	if (trpCnt >= 3) {
		trpCnt = 0;
	}
	if (triplet) {
		trpCnt++;
	}

	// print all notes
	for (int i = trk->getNbStrings() - 1; i >= 0 ; i--) {
		if ((trk->getVectTabColumn(xt).getNbFret(i) > -1) && (trk->getVectTabColumn(x).getVoiceToNote(i) == v)) {
			nNotes++;
			if (nNotes == 1) {
				// first note: calc duration etc.
				// for regular column, include ringing etc.
				// for column linked to previous, use column length
				if (trk->getVectTabColumn(x).getFlags() & FLAG_ARC) {
					length = trk->getVectTabColumn(x).getDuration();
					duration = length;
					// LVIFIX: dot and triplet handling required here ?
					// E.g. use trk->getNoteTypeAndDots()
					dots = 0;
					triplet = false;
					nCols = 1;
				} else {
					duration = length;
					// LVIFIX: allow more than one dot ?
					if (dots) {
						duration = duration * 3 / 2;
					}
					if (triplet) {
						duration = duration * 2 / 3;
					}
					nCols = trk->noteNrCols(x, i);
				}
			}
			fret = trk->getVectTabColumn(xt).getNbFret(i);
			// if this note has an effect legato, start slur
			// if previous note has an effect legato, stop slur
			// EFFECT_LEGATO means hammer-on/pull-off, depending on
			// if the next note's pitch is higher or lower than this note's
			// MusicXML requires both the <hammer-on>/<pull-off> and <slur>
			// EFFECT_SLIDE is assumed to mean a slide on the fretboard
			// (not a bottleneck slide). Use a <glissando> in MusicXML.
			bool legStart = (((unsigned)(x+1) < trk->getTabColumnSize())
							  && (trk->getVectTabColumn(x).getEffect(i) == EFFECT_LEGATO));
			bool legStop  = ((x > 0)
							  && (trk->getVectTabColumn(x - 1).getEffect(i) == EFFECT_LEGATO));
			QString legStartType;
			QString legStopType;
			if (((unsigned)(x+1) < trk->getTabColumnSize())
				&& (trk->getVectTabColumn(x).getNbFret(i) < trk->getVectTabColumn(x + 1).getNbFret(i))) {
				legStartType = "hammer-on";
			} else {
				legStartType = "pull-off";
			}
			if ((x > 0)
				&& (trk->getVectTabColumn(x - 1).getNbFret(i) < trk->getVectTabColumn(x).getNbFret(i))) {
				legStopType = "hammer-on";
			} else {
				legStopType = "pull-off";
			}
			bool sliStart = (((unsigned)(x+1) < trk->getTabColumnSize())
							  && (trk->getVectTabColumn(x).getEffect(i) == EFFECT_SLIDE));
			bool sliStop  = ((x>0)
							  && (trk->getVectTabColumn(x - 1).getEffect(i) == EFFECT_SLIDE));
			if (wrt) {
				os << "\t\t\t<note>\n";
				if (nNotes > 1) {
					os << "\t\t\t\t<chord/>\n";
				}
				os << "\t\t\t\t<pitch>\n";
				os << "\t\t\t\t\t<step>" << trk->getVectTabColumn(xt).getStep(i) << "</step>\n";
				if (trk->getVectTabColumn(xt).getAlter(i) != '\0')
					os << "\t\t\t\t\t<alter>" << (int) trk->getVectTabColumn(xt).getAlter(i)
						<< "</alter>\n";
				os << "\t\t\t\t\t<octave>" << (int) trk->getVectTabColumn(xt).getOctave(i)
					<< "</octave>\n";
				os << "\t\t\t\t</pitch>\n";
				os << "\t\t\t\t<duration>" << duration * divisions / 120 << "</duration>\n";
				if (tieStart) {
					os << "\t\t\t\t<tie type=\"start\"/>\n";
				}
				if (tieStop) {
					os << "\t\t\t\t<tie type=\"stop\"/>\n";
				}
				os << "\t\t\t\t<voice>" << v + 1 << "</voice>\n";
				os << "\t\t\t\t<type>" << kgNoteLen2Mxml(length) << "</type>\n";
				if (dots) {
					os << "\t\t\t\t<dot/>\n";
				}
				if (trk->getVectTabColumn(x).getAccidental(i) != Accidentals::None) {
					os << "\t\t\t\t<accidental>" << strAccid(trk->getVectTabColumn(x).getAccidental(i))
						<< "</accidental>\n";
				}
				if (trpCnt) {
					os << "\t\t\t\t<time-modification>\n";
					os << "\t\t\t\t\t<actual-notes>3</actual-notes>\n";
					os << "\t\t\t\t\t<normal-notes>2</normal-notes>\n";
					os << "\t\t\t\t</time-modification>\n";
				}
				if (v == 0) {
					os << "\t\t\t\t<stem>down</stem>\n";
				} else {
					os << "\t\t\t\t<stem>up</stem>\n";
				}
				writeBeams(os, trk, x, v);
				os << "\t\t\t\t<notations>\n";
				if (legStop) {
					os << "\t\t\t\t\t<slur type=\"stop\"/>\n";
				}
				if (legStart) {
					os << "\t\t\t\t\t<slur type=\"start\"/>\n";
				}
				if (sliStop) {
					os << "\t\t\t\t\t<glissando type=\"stop\"/>\n";
				}
				if (sliStart) {
					os << "\t\t\t\t\t<glissando type=\"start\""
							" line-type=\"solid\"/>\n";
				}
				if (tieStart) {
					os << "\t\t\t\t\t<tied type=\"start\"/>\n";
				}
				if (tieStop) {
				os << "\t\t\t\t\t<tied type=\"stop\"/>\n";
				}
				if (trpCnt == 1) {
					os << "\t\t\t\t\t<tuplet type=\"start\"/>\n";
				}
				if (trpCnt == 3) {
					os << "\t\t\t\t\t<tuplet type=\"stop\"/>\n";
				}
				os << "\t\t\t\t\t<technical>\n";
				os << "\t\t\t\t\t\t<string>" << mxmlStr2Kg(i, trk->getNbStrings())
				   << "</string>\n";
				os << "\t\t\t\t\t\t<fret>" << fret << "</fret>\n";
				if (legStop) {
					os << "\t\t\t\t\t\t<" << legStopType
					   << " type=\"stop\"/>\n";
				}
				if (legStart) {
					os << "\t\t\t\t\t\t<" << legStartType
					   << " type=\"start\"/>\n";
				}
				os << "\t\t\t\t\t</technical>\n";
				os << "\t\t\t\t</notations>\n";
				os << "\t\t\t</note>\n";
			} else {
				addInt(duration);
//				cout << "note " << duration << endl;
			}
		}
	}
	// if no notes in this column, it is a rest
	// rests are printed:
	// single voice: only in voice 1
	// multi voice: only in voice 0
	if (nNotes == 0) {
		length = trk->getVectTabColumn(x).getDuration();
		// note scaling: quarter note = 48
		// LVIFIX: use divisions instead
		duration = length;
		// LVIFIX: dot and triplet handling required here ?
		if (trk->getVectTabColumn(x).getFlags() & FLAG_DOT) {
			duration = duration * 3 / 2;
		}
		if (trk->getVectTabColumn(x).getFlags() & FLAG_TRIPLET) {
			duration = duration * 2 / 3;
		}
		if (((v == 1) && !trk->hasMultiVoices())
			|| ((v == 0) && trk->hasMultiVoices())) {
			if (wrt) {
				os << "\t\t\t<note>\n";
				os << "\t\t\t\t<rest/>\n";
				os << "\t\t\t\t<duration>" << duration * divisions / 120 << "</duration>\n";
				os << "\t\t\t\t<voice>" << v + 1 << "</voice>\n";
				os << "\t\t\t\t<type>" << kgNoteLen2Mxml(length) << "</type>\n";
				if (trk->getVectTabColumn(x).getFlags() & FLAG_DOT) {
					os << "\t\t\t\t<dot/>\n";
				}
				os << "\t\t\t</note>\n";
			} else {
				addInt(duration);
//				cout << "rest " << duration << endl;
			}
			nRests++;
		}
	}

	tStartCur += duration;
	if (nNotes || nRests) {
		tEndPrev += duration;
	}

	if (trk->lastColumn(trk->barNr(x)) == x) {
		// end of bar specifics: forward if necessary
		if (v != 0) {
			// write forward
			// debug info
			
			os << "forward needed ? ";
			os << "tEndPrev=" << tEndPrev << " tStartCur=" << tStartCur;
			if (tStartCur - tEndPrev) {
				os << " -> forward: " << tStartCur - tEndPrev;
			}
			os << endl;
			
			// end debug info
			if (tStartCur - tEndPrev) {
				if (wrt) {
					os << "\t\t\t<forward>" << endl;
					os << "\t\t\t\t<duration>"
					   << (tStartCur - tEndPrev) * divisions / 120
					   << "</duration>\n";
					os << "\t\t\t\t<voice>" << v + 1 << "</voice>\n";
					os << "\t\t\t</forward>" << endl;
				} else {
					addInt(tStartCur - tEndPrev);
//					cout << "forward " << tStartCur - tEndPrev << endl;
				}
			}
		}
	}

	return nCols;*/
}

// write midi note number as step/alter/octave to QTextStream os

void MusicXMLWriter::writePitch(QTextStream& os,
                                int n, QString tabs, QString prfx)
{
/*	int alt = 0;
	int oct = 0;
	Accidentals::Accid acc = Accidentals::None;
	QString nam = "";

	accSt.getNote(n, nam, alt, oct, acc);
	os << tabs << "<" << prfx << "step>" << nam
	   << "</" << prfx << "step>\n";
	if (alt) {
		os << tabs << "<" << prfx << "alter>" << alt
		   << "</" << prfx << "alter>\n";
	}
	os << tabs << "<" << prfx << "octave>" << oct
	   << "</" << prfx << "octave>\n";*/
}

// write staff details of TabTrack trk to QTextStream os

void MusicXMLWriter::writeStaffDetails(QTextStream& os, TabTrack * trk)
{
/*	// note: writePitch uses accSt, which has to be initialized first
	// Initialize the accidentals
	accSt.resetToKeySig();
	// calculate accidentals
	accSt.startChord();
	for (int i = 0; i < trk->getNbStrings(); i++) {
		accSt.addPitch(trk->getTune(i));
	}
	accSt.calcChord();
	os << "\t\t\t\t<staff-details number=\"2\">\n";
	os << "\t\t\t\t\t<staff-type>alternate</staff-type>\n";
	os << "\t\t\t\t\t<staff-lines>" << (int) trk->getNbStrings() << "</staff-lines>\n";
	for (int i = 0; i < trk->getNbStrings(); i++) {
		os << "\t\t\t\t\t<staff-tuning line=\""
		   << i + 1 << "\">\n";
		writePitch(os, trk->getTune(i),"\t\t\t\t\t\t", "tuning-");
		os << "\t\t\t\t\t</staff-tuning>\n";
	}
	os << "\t\t\t\t</staff-details>\n";*/
}

// write time signature to QTextStream os

void MusicXMLWriter::writeTime(QTextStream& os, int bts, int btt)
{
/*	os << "\t\t\t\t<time>\n";
	os << "\t\t\t\t\t<beats>" << bts << "</beats>\n";
	os << "\t\t\t\t\t<beat-type>" << btt << "</beat-type>\n";
	os << "\t\t\t\t</time>\n";*/
}
