/***************************************************************************
 * musicxml.h: definition of MusicXML classes
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2004 the KGuitar development team
 *
 * Copyright of the MusicXML file format:
 * (C) Recordare LLC. All rights reserved. http://www.recordare.com
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#ifndef MUSICXML_H
#define MUSICXML_H

#include <header.h>

#include "accidentals.h"

class TabSong;
class TabTrack;


class MusicXMLParser;

//! Class MusicXMLErrorHandler
class MusicXMLErrorHandler : public QXmlErrorHandler
{
public:
	/*!
		Inits the class.
	*/
	MusicXMLErrorHandler();
	
	/*!
		Display a warning message.
	*/
	bool warning(const QXmlParseException& exception);
	
	/*!
		Display an error message.
	*/
	bool error(const QXmlParseException& exception);
	
	/*!
		Display a fatal erro message.
	*/
	bool fatalError(const QXmlParseException& exception);
	
	/*!
		Return an error string.
	*/
	QString errorString();
	
	/*!
		Set the parser.
	*/
	void setParser(MusicXMLParser * p);
private:
	bool fatalReported;
	MusicXMLParser * parser;
};

//! Class MusicXMLParser
class MusicXMLParser : public QXmlDefaultHandler
{
public:
	/*!
		MusicXMLParser constructor
	*/
	MusicXMLParser(TabSong *);
	
	/*!
		start of document handler
	*/
	bool startDocument();
	
	/*!
		start of element handler
	*/
	bool startElement(const QString&, const QString&, const QString&,
	                  const QXmlAttributes&);
	/*!
		end of element handler
	*/
	bool endElement(const QString&, const QString&, const QString&);
	
	/*!
		character(s) handler
	*/
	bool characters(const QString& ch);
	
	/*!
		report a fatal error
	*/
	void reportError(const QString& err);
	
	/*!
		set document locator handler
	*/
	void setDocumentLocator(QXmlLocator * locator);

private:
	bool addNote();
	bool addTrack();
	void initStNote();
	void initStScorePart();
	void initStStaffTuning();
	void reportAll(const QString& lvl, const QString& err);
	void reportWarning(const QString& err);
	
	TabSong * ts;				// pointer to calling tabsong
	TabTrack * trk;				// pointer to current track
	QVector<QString> partIds;	// part (track) id's
	int x;						// current column
	int bar;					// current bar
	Accidentals accSt;			// accidental state
	int iDiv;					// divisions
	int tStartCur;				// start time current note
	int tEndCur;				// end time current note
	QXmlLocator * lctr;

	// state variables for parsing
	// general -- initialized in startDocument()
	QString stCha;				// characters collected
	// identification -- initialized in startDocument()
	QString stCrt;				// creator
	QString stEnc;				// encoder
	QString stTtl;				// title
	// measure -- initialized in startDocument()
	QString stBts;				// beats
	QString stBtt;				// beat-type
	QString stDiv;				// divisions
	QString stFif;				// fifths
	// note (including forward/backup) -- initialized in initStNote()
	QString stAlt;				// alter
	QString	stAno;				// actual notes
	bool    stCho;				// chord with previous note
	int     stDts;				// dots (count)
	QString stDur;				// duration
	QString stFrt;				// fret
	bool    stGls;				// glissando
	bool    stHmr;				// hammer-on
	QString	stNno;				// normal notes
	QString stOct;				// octave
	bool    stPlo;				// pull-off
	bool    stRst;				// rest
	QString stStp;				// step
	QString stStr;				// string
	bool    stTie;				// tie stop
	QString stTyp;				// type
	// part (== track) -- initialized in initStScorePart()
	QString stPid;				// ID
	QString stPmb;				// MIDI bank
	QString stPmc;				// MIDI channel
	QString stPmp;				// MIDI program
	QString stPnm;				// name
	// tuning -- initialized in initStStaffTuning()
	QString stPtl;				// tuning line
	QString stPtn;				// tuning number of lines
	QString stPto;				// tuning octave
	QString stPts;				// tuning step
};

/*!
	TODO
*/
class MusicXMLWriter
{
public:
	/*!
		TODO
	*/
	MusicXMLWriter(TabSong *);
	
	/*!
		TODO
	*/
	void write(QTextStream&);
private:
	void calcDivisions();
	QString strAccid(Accidentals::Accid);
	void writeBeams(QTextStream&, TabTrack *, int, int);
	int writeCol(QTextStream&, TabTrack *, uint, int, bool);
	void writePitch(QTextStream&, int, QString, QString);
	void writeStaffDetails(QTextStream&, TabTrack *);
	void writeTime(QTextStream&, int, int);
	Accidentals accSt;			// accidental state
	int divisions;				// divisions
	TabSong * ts;				// pointer to calling tabsong
	// following variables are used by writeCol only
	int tEndPrev;				// end time of previous note
	int trpCnt;					// triplet count (0=none, 1=1st...3=3rd)
	int tStartCur;				// start time current note
};

#endif
