/***************************************************************************
 * trackviewcommands.h: implementation of SetLengthCommand class
 *                                        InsertTabCommand
 *                                        MoveFingerCommand
 *                                        AddFXCommand
 *                                        SetFlagCommand
 *                                        DeleteNoteCommand
 *                                        AddColumnCommand
 *                                        SetTimeSigCommand
 *                                        InsertStrumCommand
 *                                        InsertRythm
 *
 * This file is part of KGuitar, a KDE tabulature editor
 *
 * copyright (C) 2002-2003 the KGuitar development team
 ***************************************************************************/

/***************************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * See the file COPYING for more information.
 ***************************************************************************/

#ifndef TRACKVIEWCOMMANDS_H
#define TRACKVIEWCOMMANDS_H

#include <header.h>

#include "tabduration.h"
#include "trackpos.h"
#include "trackeditview.h"

class TabNote;
class TabTimes;
class TabTrack;
class TrackPos;

/*!
	Set dead note to true.
*/
class TrackEditView::DeadNoteCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _t The new note.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	DeadNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old note.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TabNote tab;
	const TabNote oldtab;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	Set rest note to true.
*/
class TrackEditView::RestNoteCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _t The new note.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	RestNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old note.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TabNote tab;
	const TabTimes oldTimes;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	This class can set the duration for the notes and undo the operation.
*/
class TrackEditView::SetLengthCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _bar The bar number in the current track.
		\param _note The note number.
		\param length The new length.
	*/
	SetLengthCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos, Duration newLength);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old length.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	Duration oldLength;
	const Duration length;
	const TabTimes oldTimes;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	Set dotted note to true.
*/
class TrackEditView::DottedNoteCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _t The new note.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	DottedNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old note.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TabNote tab;
	const TabNote oldtab;
	const TabTimes oldtimes;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	This class can insert tabs from keyboard and undo the operation.
*/
class TrackEditView::InsertTabCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _t The new note.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	InsertTabCommand(TrackEditView& tv, TabTrack& trk, int num, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old note.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TabNote tab;
	const TabNote oldtab;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	Set dotted note to true.
*/
class TrackEditView::BendNoteCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note.
		\param _t The new note.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	BendNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos, const TabBend& effect);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old note.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TabNote tab;
	const TabNote oldtab;
	const TabTimes oldtimes;
	TrackEditView& trackView;
	TabTrack& tabTrack;
	const TabBend& bend;
};

/*!
	This class can moves the finger and undo the operation.
*/
class TrackEditView::MoveFingerCommand: public KNamedCommand {
public:
	/*!
		TODO
	*/
	MoveFingerCommand(TrackView *, TabTrack *, int _from, int _to, int /*_tune*/);
	
	/*!
		TODO
	*/
	virtual void execute();
	
	/*!
		TODO
	*/
	virtual void unexecute();
	
private:
};

/*!
	This class can insert note and undo the operation.
*/
class TrackEditView::InsertNoteCommand : public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note to delete.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	InsertNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old flag.
	*/
	virtual void unexecute();
	
private:
	TrackPos& trackPos;
	const TrackPos oldPos;
	TrackEditView &trackView;
	TabBar oldBar;
	TabTrack &tabTrack;
};

/*!
	This class can delete note and undo the operation.
*/
class TrackEditView::DeleteNoteCommand : public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the note to delete.
		\param _bar The bar number.
		\param _note The note number.
		\param _fret The position of the fret.
	*/
	DeleteNoteCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old flag.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TrackEditView &trackView;
	TabBar oldBar;
	TabTrack &tabTrack;
};

/*!
	This class can delete column and undo the operation.
*/
class TrackEditView::DeleteColumnCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the bar to delete.
		\param _bar The bar number.
	*/
	DeleteColumnCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old bar.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos, oldPos;
	int countBar;
	TabBar oldBar;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

/*!
	This class can set time sig and undo the operation.
*/
class TrackEditView::SetTimeSigCommand : public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the bar(s).
		\param _bar The bar number.
		\param _toend We must apply the changing up to the end.
		\param _time1 The time signature.
		\param _time2 The time signature.
	*/
	SetTimeSigCommand(TrackEditView& tv, TabTrack& trk, int _bar, bool _toend, int _time1, int _time2);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Restore the old time signature.
	*/
	virtual void unexecute();

private:
	int bar, time1, time2;
	int oldTime1, oldTime2;
	int defaultOldTime1, defaultOldTime2;
	bool sel, toend;
	TabTrack& tabTrack;
	TrackEditView& trackView;
};

/*!
	This class can insert a column at cursor pos and undo the operation.
*/
class TrackEditView::InsertColumnCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the bar(s).
		\param _bar The bar number.
	*/
	InsertColumnCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Remove the new bar.
	*/
	virtual void unexecute();
	
private:
	TrackPos& trackPos, oldPos;
	TabTrack& tabTrack;
	TrackEditView& trackView;
};

/*!
	This class can insert strum and undo the operation.
*/
class TrackEditView::InsertStrumCommand: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the bar(s).
		\param _tab DO NOTHING.
		\param _bar The bar number.
		\param _note The note number.
		\param _sch I DON'T KNOW !!!
		\param _chord The chord to insert.
	*/
	InsertStrumCommand(TrackEditView& tv, TabTrack& trk, TrackPos& pos, QMemArray<int> &_chord);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Remove the new chord.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	const QMemArray<int> chord;
	const TabTimes oldTimes;
	TrackEditView& trackView;
	TabTrack& tabTrack;
};

class QListBox;

/*!
	This class can insert rhythm from rhythmer and undo the operation.
*/
class TrackEditView::InsertRhythm: public KNamedCommand {
public:
	/*!
		The constructor inits the variables of the class.
		\param tv TrackView is used to emit a signal.
		\param trk TabTrack contains the bar(s).
		\param quantized The rythm.
		\param _bar The bar number.
		\param _note The note number.
	*/
	InsertRhythm(TrackEditView& tv, TabTrack& trk, QListBox& quantized, TrackPos& pos);
	
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Remove the new chord.
	*/
	virtual void unexecute();
	
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	QMemArray<Duration> newdur;
	TabTrack& tabTrack;
	TrackEditView& trackView;
	TabBar oldBar;
};

class TrackEditView::NTupletCommand: public KNamedCommand {
public:
	NTupletCommand(TrackEditView &_tv, TabTrack &_trk, TrackPos& pos, uint n);
	/*!
		Execute the operation.
	*/
	virtual void execute();
	
	/*!
		Remove the new chord.
	*/
	virtual void unexecute();
private:
	TrackPos& currentPos;
	const TrackPos oldPos;
	TrackEditView &tv;
	TabTrack &tabTrack;
	TabTimes oldTime;
	uint ntuplet;
};

#endif
