#include "kguitar.h"

#include <kiconloader.h>
#include <kstddirs.h>
#include <kapp.h>
#include <kaction.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <klibloader.h>
#include <kstdaction.h>
#include <kedittoolbar.h>
#include <kaccel.h>
#include <kdebug.h>
#include <kcommand.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kdebug.h>

#include <qfileinfo.h>

KGuitar::KGuitar()
: KParts::MainWindow(0, "KTabEdit", WDestructiveClose | WGroupLeader)
{
	// set the shell's ui resource file
	//
	setXMLFile("ktabedit_shell.rc");
	
	// setup shell-specific actions
	//
	setupActions();
	
	// and a status bar
	//
	statusBar()->insertItem(QString(i18n("Bar: ")) + "1", 1);
	statusBar()->show();
	
	// this routine will find and load KGuitar KPart.
	//
	KLibFactory *factory = KLibLoader::self()->factory("libktabeditpart");
	
	if (factory) {
		// now that the Part is loaded, we cast it to a Part to get
		// our hands on it
		//
		kgpart = static_cast<KParts::ReadWritePart *>(factory->create(this, "ktabedit_part", "KParts::ReadWritePart"));
	} else {
		// if we couldn't find our Part, we exit since the Shell by
		// itself can't do anything useful
		//
		KMessageBox::error(this, i18n("Could not find KTabEdit KPart! Check your installation!"));
		
		kapp->quit();
		
		// we return here, cause kapp->quit() only means "exit the
		// next time we enter the event loop...
		//
		return;
	}
	
	// Set the central widget for this main windows
	//
	setCentralWidget(kgpart->widget());
	createGUI(kgpart);
	
	// undo / redo
	//
	cmdHistory = new KCommandHistory(actionCollection());
	
	// Title of the window
	//
	setCaption(i18n("Unnamed"));
	
	toolBar("mainToolBar")->setText(i18n("Main Toolbar"));
	toolBar("editToolBar")->setText(i18n("Edit Toolbar"));
	
	setAutoSaveSettings();
	readOptions(kapp->config());
}

KGuitar::~KGuitar()
{
	delete cmdHistory;
	delete kgpart;
}

void KGuitar::readOptions(KConfig* config)
{
	config->setGroup("General");
	openRecentAct->loadEntries(config, "Recent Files");
}

void KGuitar::saveOptions(KConfig* config)
{
	config->setGroup("General");
	openRecentAct->saveEntries(config, "Recent Files");
}

bool KGuitar::queryClose()
{
	saveProperties(kapp->config());
	
	if (kgpart->isModified() == false)
		return true;
	
	kgpart->save();
	
	return true;
}

void KGuitar::setupActions()
{
	//Connect the file operations with their slots
	//
	KStdAction::openNew(this, SLOT(fileNew()), actionCollection());
	KStdAction::open(this, SLOT(fileOpen()), actionCollection());
	KStdAction::quit(this, SLOT(quit()), actionCollection());
	
	openRecentAct = KStdAction::openRecent(this, SLOT(load(const KURL&)), actionCollection(), "file_openRecent");
	openRecentAct->setWhatsThis(i18n("This lists files which you have opened recently, and allows you to easily open them again."));
	openRecentAct->setMaxItems(5);
	
	showMainTBAct = KStdAction::showToolbar(this, SLOT(slotToggleMainTB()), actionCollection(), "tog_mainTB");
	showMainTBAct->setText(i18n("Main Toolbar"));
	
	showEditTBAct = KStdAction::showToolbar(this, SLOT(slotToggleEditTB()), actionCollection(), "tog_editTB");
	showEditTBAct->setText(i18n("Edit Toolbar"));
	
	showStatusbarAct = KStdAction::showStatusbar(this, SLOT(optionsShowStatusbar()), actionCollection());
	
	KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection());
	KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection());
}

// Call KPart's URL opening and maintain recent files list
//
void KGuitar::load(const KURL& url)
{
	if (kgpart->openURL(url)) {
		openRecentAct->addURL(url);
		saveOptions(kapp->config());
	}
}

// Call KPart's saving URL and maintain recent files list
//
void KGuitar::saveURL(const KURL& url)
{
	if (kgpart->saveAs(url)) {
		openRecentAct->addURL(url);
		saveOptions(kapp->config());
	}
}

void KGuitar::fileNew()
{
	if (!kgpart->url().isEmpty() || kgpart->isModified())
		(new KGuitar)->show();
}

void KGuitar::fileOpen()
{
	KURL file_name = KFileDialog::getOpenURL( QString::null,
					"*.kg *.gtp *.gp? *.mid *.tab *.xml *.ptb|" + i18n("All music files") + "\n"
					"*.kg|" + i18n("KGuitar files") + " (*.kg)\n"
					"*.tab|" + i18n("ASCII files") + " (*.tab)\n"
					"*.mid|" + i18n("MIDI files") + " (*.mid)\n"
					"*.ptb|" + i18n("PowerTab files") + " (*.ptb)\n"
					"*.gtp|" + i18n("Guitar Pro files") + " (*.gtp)\n"
					"*.gp?|" + i18n("Guitar Pro files") + " (*.gp?)\n"
					"*.xml|" + i18n("MusicXML files") + " (*.xml)\n"
					"*|" + i18n("All files"), this);
	
	if (file_name.isEmpty() == false) {
		// if the document is not in its initial state, we open a new window
		if (kgpart->url().isEmpty() && !kgpart->isModified()) {
			load(file_name);
		} else {
			// we open the file in a new window...
			KGuitar* newWin = new KGuitar;
			
			newWin->load(KURL(file_name));
			newWin->show();
		}
	}
}

void KGuitar::optionsShowStatusbar()
{
	if (showStatusbarAct->isChecked())
		statusBar()->show();
	else
		statusBar()->hide();
}

void KGuitar::slotToggleMainTB()
{
	KToolBar *bar = toolBar("mainToolBar");
	
	if (bar!=0L) {
		if (showMainTBAct->isChecked())
			bar->show();
		else
			bar->hide();
	}
}

void KGuitar::slotToggleEditTB()
{
	KToolBar *bar = toolBar("editToolBar");
	
	if (bar!=0L) {
		if (showEditTBAct->isChecked())
			bar->show();
		else
			bar->hide();
	}
}

void KGuitar::optionsConfigureKeys()
{
	KKeyDialog::configure(actionCollection(), "ktabedit_shell.rc");
}

void KGuitar::optionsConfigureToolbars()
{
	saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
	
	// use the standard toolbar editor
	//
	KEditToolbar dlg(factory());
	
	connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(applyNewToolbarConfig()));
	
	dlg.exec();
}

void KGuitar::quit()
{
	close();
}
