/*
 * Created on 18-Apr-2006
 * Created by Paul Gardner
 * Copyright (C) 2006 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SAS au capital de 46,603.30 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package com.aelitis.azureus.plugins.upnpmediaserver;

import java.util.*;

import java.io.IOException;
import java.net.Socket;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;

import org.gudy.azureus2.core3.util.AESemaphore;
import org.gudy.azureus2.core3.util.AEThread2;
import org.gudy.azureus2.core3.util.Debug;
import org.gudy.azureus2.plugins.utils.PooledByteBuffer;

import com.aelitis.azureus.core.networkmanager.VirtualChannelSelector;
import com.aelitis.azureus.core.networkmanager.VirtualChannelSelector.VirtualSelectorListener;

public class 
UPnPMediaChannel 
{
	private static final int	READ_TIMEOUT	= 30*1000;

	private final static VirtualChannelSelector read_selector	= new VirtualChannelSelector( "UPnPMediaServer", VirtualChannelSelector.OP_READ, false );
	private final static VirtualChannelSelector write_selector 	= new VirtualChannelSelector( "UPnPMediaServer", VirtualChannelSelector.OP_WRITE, false );

	private static final int	BUFFER_LIMIT	= 3;
	
	static{
		new AEThread2( "UPnPMediaChannel:writeSelector", true )
			{
				public void
				run()
				{
					Thread.currentThread().setPriority( Thread.MAX_PRIORITY );
					
					selectLoop( write_selector, 50 );
				}
			}.start();
			
		new AEThread2( "UPnPMediaChannel:readSelector", true )
			{
				public void
				run()
				{
					Thread.currentThread().setPriority( Thread.MAX_PRIORITY );

					selectLoop( read_selector, 50 );
				}
			}.start();
	}
	
	static void
	selectLoop(
		VirtualChannelSelector	selector,
		int						timeout )
	{
		while( true ){
			
			selector.select( timeout );
		}
	}
	
	private final Object	read_lock	= new Object();
	private final Object	write_lock	= new Object();
	
	private SocketChannel	channel;
		
	private List			pending_read_bytes = new ArrayList();
	
	private List			write_buffers = new ArrayList();
	private IOException		write_error;
	
	protected
	UPnPMediaChannel(
		Socket		socket )
	
		throws IOException
	{
		channel	= socket.getChannel();
		
		channel.configureBlocking( false );
		
		try{			
			socket.setSendBufferSize( 65536 );
			
		}catch ( SocketException e ){
		}
	}
	
	public void
	read(
		byte[]		buffer )
	
		throws IOException
	{
		read( ByteBuffer.wrap( buffer ));
	}
	
	public void
	read(
		final ByteBuffer	buffer )
	
		throws IOException
	{
		try{
			synchronized( read_lock ){
				
				Iterator	it = pending_read_bytes.iterator();
				
				while( it.hasNext() && buffer.hasRemaining()){
					
					buffer.put((byte[])it.next());
					
					it.remove();
				}
				
				if ( !buffer.hasRemaining()){
					
					return;
				}
				
				channel.read( buffer );
				
				if ( buffer.hasRemaining()){
			
					final IOException[]	error = { null };
					
					final AESemaphore	sem = new AESemaphore( "UPnPMediaChannel::read" );
					
					read_selector.register( 
							channel, 
							new VirtualSelectorListener()
							{
								public boolean 
								selectSuccess(
									VirtualChannelSelector 	selector, 
									SocketChannel 			sc,
									Object 					attachment)
								{
									try{
										int	len = channel.read( buffer );
										
										if ( !buffer.hasRemaining()){
											
											read_selector.cancel( channel );
											
											sem.release();
										}
										
										return( len > 0 );
										
									}catch( IOException e ){
										
										error[0] = e;
										
										read_selector.cancel( channel );

										sem.release();
										
										return( false );
									}
								}
			
								public void 
								selectFailure(
									VirtualChannelSelector 	selector, 
									SocketChannel 			sc, 
									Object 					attachment, 
									Throwable 				msg )
								{
									error[0] = msg instanceof IOException?(IOException)msg:new IOException(msg.getMessage());

									read_selector.cancel( channel );
									
									sem.release();
								}
							},
							null );
						
					
					if ( !sem.reserve( READ_TIMEOUT )){
					
						throw( new IOException( "Read timeout" ));
					}
					
					if ( error[0] != null ){
						
						throw( error[0] );
					}
				}
			}
		}catch( IOException e ){
	
			close();
			
			throw( e );
		}
	}
	
	public void
	write(
		byte[]		buffer )
	
		throws IOException
	{
		writeSupport( ByteBuffer.wrap( buffer ));
	}
	
	public void
	write(
		final PooledByteBuffer		buffer )
	
		throws IOException
	{
		writeSupport( buffer );
	}
	
	protected void
	writeSupport(
		Object	buffer )
	
		throws IOException
	{
		try{
			synchronized( write_lock ){
				
				write_buffers.add( buffer );
				
				// System.out.println( "buffers = " + write_buffers.size());
				
				if ( write_error != null ){
					
					throw( write_error );
				}
				
				if ( write_buffers.size() == 1 ){
										
					write_selector.register( 
							channel, 
							new VirtualSelectorListener()
							{
								public boolean 
								selectSuccess(
									VirtualChannelSelector 	selector, 
									SocketChannel 			sc,
									Object 					attachment)
								{
									long	total_written	= 0;
								
									while( true ){
								
										Object		current;
										ByteBuffer	buffer;
										
										synchronized( write_lock ){
											
											if ( write_buffers.size() == 0 ){
												
												write_selector.cancel( channel );
												
												return( false );
											}
											
											current = write_buffers.get(0);
											
											if ( current instanceof ByteBuffer ){
												
												buffer = (ByteBuffer)current;
												
											}else{
												
												buffer = ((PooledByteBuffer)current).toByteBuffer();
												
											}
										}
									
										try{
											total_written += channel.write( buffer );
												
											if ( buffer.hasRemaining()){
												
												break;
											}
											
											synchronized( write_lock ){

												write_buffers.remove(current);
												
												if ( current instanceof PooledByteBuffer ){
													
													((PooledByteBuffer)current).returnToPool();
												}
												
												if ( write_buffers.size() == 0 ){
													
													write_selector.cancel( channel );
													
													break;
													
												}else if( write_buffers.size() == BUFFER_LIMIT - 1 ){
													
													write_lock.notify();
												}
											}
										}catch( IOException e ){
											
											write_selector.cancel( channel );
	
											synchronized( write_lock ){
												
												write_error = e;
	
												write_lock.notifyAll();
											}
											
											return( false );
										}
									}
									
									return( total_written > 0 );
								}
			
								public void 
								selectFailure(
									VirtualChannelSelector 	selector, 
									SocketChannel 			sc, 
									Object 					attachment, 
									Throwable 				msg )
								{
									write_selector.cancel( channel );

									synchronized( write_lock ){
										
										write_error = msg instanceof IOException?(IOException)msg:new IOException(msg.getMessage());

										write_lock.notifyAll();
									}
								}
							},
							null );
						
				}else if ( write_buffers.size() == BUFFER_LIMIT ){
										
					try{
						write_lock.wait();
	
						if ( write_error != null ){
							
							throw( write_error );
						}
	
					}catch( InterruptedException e ){
						
						throw( new IOException( "interrupted" ));
					}
				}
			}
		}catch( IOException e ){
	
			close();
			
			throw( e );
		}
	}
	
	public void
	flush()
	
		throws IOException
	{
		try{
	
			while( true ){
			
				synchronized( write_lock ){
					
					if ( write_error != null ){
						
						throw( write_error );
					}
					
					if ( write_buffers.size() == 0 ){
						
						break;
					}
				}
				
				try{
					Thread.sleep(100);
					
				}catch( Throwable e ){
					
					throw( new IOException( "interrupted" ));
				}
			}
		}catch( IOException e ){
			
			close();
			
			throw( e );
		}
	}
	
	public boolean
	isClosed()
	{
		try{
			synchronized( read_lock ){
				
					// bah, the only way I can find to pick up a channel that has died it to try and
					// read a byte from it (zero byte ops do nothing, as do selects)
					// of course we then have to deal with the fact that we might actually read a byte...
				
				byte[]	buffer = new byte[1];
				
				if ( channel.read( ByteBuffer.wrap( buffer )) == 1 ){
			
					pending_read_bytes.add( buffer );
				}
				
				return( false );
			}
		}catch( IOException e ){
	
			return( true );
		}
	}
	
	public void
	close()
	{
		try{
			channel.close();
			
		}catch( Throwable e ){
			
			Debug.printStackTrace(e);
		}
		
		synchronized( write_lock ){

			if ( write_error == null ){
				
				write_error = new IOException( "channel closed" );
			
				write_lock.notifyAll();
			}
			
			read_selector.cancel( channel );
			write_selector.cancel( channel );
			
			Iterator	it = write_buffers.iterator();
			
			while( it.hasNext()){
				
				Object	o = it.next();
		
				if ( o instanceof PooledByteBuffer ){
					
					((PooledByteBuffer)o).returnToPool();
				}
				
				it.remove();
			}
		}
	}
}
