/*
 * Created on 23-Mar-2006
 * Created by Paul Gardner
 * Copyright (C) 2006 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SAS au capital de 46,603.30 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package com.aelitis.azureus.plugins.upnpmediaserver;

import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.Socket;
import java.net.URL;
import java.net.URLEncoder;
import java.util.*;

import org.gudy.azureus2.core3.util.AESemaphore;
import org.gudy.azureus2.core3.util.ByteFormatter;
import org.gudy.azureus2.core3.util.Constants;
import org.gudy.azureus2.core3.util.Debug;
import org.gudy.azureus2.core3.util.SystemTime;
import org.gudy.azureus2.core3.util.TimeFormatter;
import org.gudy.azureus2.plugins.*;
import org.gudy.azureus2.plugins.disk.DiskManagerFileInfo;
import org.gudy.azureus2.plugins.download.Download;
import org.gudy.azureus2.plugins.download.DownloadException;
import org.gudy.azureus2.plugins.download.DownloadManager;
import org.gudy.azureus2.plugins.download.DownloadManagerListener;
import org.gudy.azureus2.plugins.ipc.IPCException;
import org.gudy.azureus2.plugins.ipc.IPCInterface;
import org.gudy.azureus2.plugins.logging.LoggerChannel;
import org.gudy.azureus2.plugins.logging.LoggerChannelListener;
import org.gudy.azureus2.plugins.torrent.Torrent;
import org.gudy.azureus2.plugins.torrent.TorrentAttribute;
import org.gudy.azureus2.plugins.tracker.Tracker;
import org.gudy.azureus2.plugins.tracker.web.TrackerAuthenticationListener;
import org.gudy.azureus2.plugins.tracker.web.TrackerWebContext;
import org.gudy.azureus2.plugins.tracker.web.TrackerWebPageGenerator;
import org.gudy.azureus2.plugins.tracker.web.TrackerWebPageRequest;
import org.gudy.azureus2.plugins.tracker.web.TrackerWebPageResponse;
import org.gudy.azureus2.plugins.ui.UIInstance;
import org.gudy.azureus2.plugins.ui.UIManager;
import org.gudy.azureus2.plugins.ui.UIManagerListener;
import org.gudy.azureus2.plugins.ui.config.ActionParameter;
import org.gudy.azureus2.plugins.ui.config.BooleanParameter;
import org.gudy.azureus2.plugins.ui.config.IntParameter;
import org.gudy.azureus2.plugins.ui.config.Parameter;
import org.gudy.azureus2.plugins.ui.config.ParameterListener;
import org.gudy.azureus2.plugins.ui.config.StringParameter;
import org.gudy.azureus2.plugins.ui.menus.MenuItem;
import org.gudy.azureus2.plugins.ui.menus.MenuItemFillListener;
import org.gudy.azureus2.plugins.ui.menus.MenuItemListener;
import org.gudy.azureus2.plugins.ui.model.BasicPluginConfigModel;
import org.gudy.azureus2.plugins.ui.model.BasicPluginViewModel;
import org.gudy.azureus2.plugins.ui.tables.TableContextMenuItem;
import org.gudy.azureus2.plugins.ui.tables.TableManager;
import org.gudy.azureus2.plugins.ui.tables.TableRow;
import org.gudy.azureus2.plugins.utils.LocaleUtilities;
import org.gudy.azureus2.plugins.utils.UTTimer;
import org.gudy.azureus2.plugins.utils.UTTimerEvent;
import org.gudy.azureus2.plugins.utils.UTTimerEventPerformer;
import org.gudy.azureus2.plugins.utils.Utilities;
import org.gudy.azureus2.plugins.utils.resourcedownloader.ResourceDownloaderFactory;
import org.gudy.azureus2.plugins.utils.xml.simpleparser.SimpleXMLParserDocument;
import org.gudy.azureus2.plugins.utils.xml.simpleparser.SimpleXMLParserDocumentException;
import org.gudy.azureus2.plugins.utils.xml.simpleparser.SimpleXMLParserDocumentNode;
import org.gudy.azureus2.ui.swt.plugins.UISWTInstance;

import com.aelitis.azureus.core.content.AzureusContent;
import com.aelitis.azureus.core.content.AzureusContentDirectory;
import com.aelitis.azureus.core.content.AzureusContentDirectoryManager;
import com.aelitis.azureus.core.util.UUIDGenerator;
import com.aelitis.azureus.plugins.upnpmediaserver.ui.UPnPMediaServerUI;
import com.aelitis.azureus.plugins.upnpmediaserver.ui.swt.UPnPMediaServerUISWT;
import com.aelitis.net.upnp.UPnP;
import com.aelitis.net.upnp.UPnPAdapter;
import com.aelitis.net.upnp.UPnPFactory;
import com.aelitis.net.upnp.UPnPListener;
import com.aelitis.net.upnp.UPnPRootDevice;
import com.aelitis.net.upnp.UPnPRootDeviceListener;
import com.aelitis.net.upnp.UPnPSSDP;
import com.aelitis.net.upnp.UPnPSSDPListener;

public class 
UPnPMediaServer 
	implements UnloadablePlugin
{
	private static final boolean DISABLE_MENUS_FOR_INCOMPLETE	= true;
	
	private static final int EVENT_TIMEOUT_SECONDS	= 30*60;
	
	private static final String	CM_SOURCE_PROTOCOLS = "http-get:*:*:*";
	private static final String	CM_SINK_PROTOCOLS 	= "";
	

	private static final String	CONTENT_UNKNOWN	= "object.item";
	//private static final String	CONTENT_VIDEO	= "object.item.videoItem";
	private static final String	CONTENT_VIDEO	= "object.item.videoItem.movie";
	private static final String	CONTENT_AUDIO	= "object.item.audioItem.musicTrack";
	private static final String	CONTENT_IMAGE	= "object.item.imageItem.photo";

	private static final String[][]	mime_mappings = {
		
			// Microsoft
		
		{ "asf", "video/x-ms-asf",				CONTENT_VIDEO },
		{ "asx", "video/x-ms-asf",				CONTENT_VIDEO },
		{ "nsc", "video/x-ms-asf",				CONTENT_VIDEO },
		{ "wax", "audio/x-ms-wax",				CONTENT_AUDIO },
		{ "wm",  "video/x-ms-wm",				CONTENT_VIDEO },
		{ "wma", "audio/x-ms-wma",				CONTENT_AUDIO },
		{ "wmv", "video/x-ms-wmv",				CONTENT_VIDEO },
		{ "wmx", "video/x-ms-wmx",				CONTENT_VIDEO },
		{ "wvx", "video/x-ms-wvx",				CONTENT_VIDEO },
		
			// other video
		
		{ "avi",  "video/avi",					CONTENT_VIDEO },
		{ "mp2",  "video/mpeg", 				CONTENT_VIDEO },
		{ "mpa",  "video/mpeg", 				CONTENT_VIDEO },
		{ "mpe",  "video/mpeg", 				CONTENT_VIDEO },
		{ "mpeg", "video/mpeg", 				CONTENT_VIDEO },
		{ "mpg",  "video/mpeg", 				CONTENT_VIDEO },
		{ "mpv2", "video/mpeg", 				CONTENT_VIDEO },
		{ "mov",  "video/quicktime", 			CONTENT_VIDEO },
		{ "qt",   "video/quicktime", 			CONTENT_VIDEO },
		{ "lsf",  "video/x-la-asf", 			CONTENT_VIDEO },
		{ "lsx",  "video/x-la-asf", 			CONTENT_VIDEO },
		{ "movie","video/x-sgi-movie", 			CONTENT_VIDEO },
		{ "mkv",  "video/x-matroska", 			CONTENT_VIDEO },
		{ "mp4",  "video/mp4", 					CONTENT_VIDEO },
		{ "mpg4", "video/mp4", 					CONTENT_VIDEO },

			// audio
		
		{ "au",   "audio/basic",				CONTENT_AUDIO },
		{ "snd",  "audio/basic", 				CONTENT_AUDIO },
		{ "mid",  "audio/mid",  				CONTENT_AUDIO },
		{ "rmi",  "audio/mid", 					CONTENT_AUDIO },
		{ "mp3",  "audio/mpeg" ,				CONTENT_AUDIO },
		{ "aif",  "audio/x-aiff", 				CONTENT_AUDIO },
		{ "aifc", "audio/x-aiff", 				CONTENT_AUDIO },
		{ "aiff", "audio/x-aiff", 				CONTENT_AUDIO },
		{ "m3u",  "audio/x-mpegurl", 			CONTENT_AUDIO },
		{ "ra",   "audio/x-pn-realaudio",		CONTENT_AUDIO },
		{ "ram",  "audio/x-pn-realaudio",		CONTENT_AUDIO },
		{ "wav",  "audio/x-wav", 				CONTENT_AUDIO },
		{ "flac", "audio/flac",					CONTENT_AUDIO },
		{ "mka",  "audio/x-matroska",			CONTENT_AUDIO },
		{ "m4a",  "audio/mp4",                  CONTENT_AUDIO },
		
			// image
		
		{ "bmp",  "image/bmp", 					CONTENT_IMAGE },
		{ "cod",  "image/cis-cod",				CONTENT_IMAGE }, 
		{ "gif",  "image/gif", 					CONTENT_IMAGE },
		{ "ief",  "image/ief", 					CONTENT_IMAGE },
		{ "jpe",  "image/jpeg", 				CONTENT_IMAGE },
		{ "jpeg", "image/jpeg", 				CONTENT_IMAGE },
		{ "jpg",  "image/jpeg", 				CONTENT_IMAGE },
		{ "jfif", "image/pipeg",		 		CONTENT_IMAGE },
		{ "tif",  "image/tiff", 				CONTENT_IMAGE },
		{ "tiff", "image/tiff", 				CONTENT_IMAGE },
		{ "ras",  "image/x-cmu-raster", 		CONTENT_IMAGE },
		{ "cmx",  "image/x-cmx", 				CONTENT_IMAGE },
		{ "ico",  "image/x-icon", 				CONTENT_IMAGE },
		{ "pnm",  "image/x-portable-anymap", 	CONTENT_IMAGE }, 
		{ "pbm",  "image/x-portable-bitmap", 	CONTENT_IMAGE },
		{ "pgm",  "image/x-portable-graymap",	CONTENT_IMAGE },
		{ "ppm",  "image/x-portable-pixmap", 	CONTENT_IMAGE },
		{ "rgb",  "image/x-rgb", 				CONTENT_IMAGE },
		{ "xbm",  "image/x-xbitmap", 			CONTENT_IMAGE },
		{ "xpm",  "image/x-xpixmap", 			CONTENT_IMAGE },
		{ "xwd",  "image/x-xwindowdump", 		CONTENT_IMAGE },
		
			// other
		
		{ "ogg",   "application/ogg",			CONTENT_AUDIO },
		{ "ogm",   "application/ogg",			CONTENT_VIDEO },
				
	};
	
	private static Map	ext_lookup_map = new HashMap();
	
	static{
		for (int i=0;i<mime_mappings.length;i++){
			
			ext_lookup_map.put( mime_mappings[i][0], mime_mappings[i] );
		}
	}
	
	
	private String 		UUID_rootdevice;
	private String		service_name;
	
	private String[]	upnp_entities;

	private PluginInterface		plugin_interface;
	private LoggerChannel		logger;
	
	private UPnPSSDP 				ssdp;
	private UPnPSSDPListener 		ssdp_listener;
	
	private UPnP					upnp;
	private UPnPListener			upnp_listener;
	
	private boolean	alive	= true;
	private int		next_oid		= 0;
	private Map		content_map 	= new HashMap();
		
	private Map		events					= new HashMap();
	private List	new_events				= new ArrayList();
	private Set		container_update_events;
	
	private contentContainer		root_container 			= new contentContainer( "Azureus" );
	private contentContainer		downloads_container;
	private contentContainer		movies_container;
	private contentContainer		music_container;
	private contentContainer		pictures_container;

	private UPnPMediaServerContentServer	content_server;
	
	private int		system_update_id;
	
	private Set		renderers = new HashSet();
	
	private int		stream_id_next;
	
	private boolean enable_upnp			= true;
	private boolean enable_lan_publish	= true;
	private int		stream_port			= 0;
	
	private UPnPMediaServerUI		media_server_ui;
	
	private List	logged_searches  = new ArrayList();

	private boolean quickTimeAvail = false;
	
	private TorrentAttribute	ta_unique_name;
	
	private BasicPluginConfigModel 	config_model;
	private BasicPluginViewModel	view_model;
	private List					menus = new ArrayList();
	private UTTimer					timer2;
	private UTTimerEvent 			timer2_event;
	private UTTimer					timer3;
	private UTTimerEvent 			timer3_event;
	private DownloadManagerListener	dm_listener;
	private TrackerWebContext		web_context;
	
	private Method 	TrackerWebContent_destroy;
	
	private volatile boolean		unloaded;
	
	private Object	startup_lock	= new Object();
	private boolean	starting		= false;
	private AESemaphore			startup_sem = new AESemaphore( "UPnPMediaServer:Startup" );
	
	public void
	initialize(
		PluginInterface		_plugin_interface )
	{
		plugin_interface		= _plugin_interface;

			// we can't unload the plugin until core support is released for TrackerWebContext.destroy
				
		try{
			TrackerWebContent_destroy = TrackerWebContext.class.getMethod( "destroy", new Class[0] );
			
		}catch( Throwable e ){
		}
		
		if ( TrackerWebContent_destroy == null ){
			
			plugin_interface.getPluginProperties().setProperty( "plugin.unload.disabled", "true" );
		}
		
		logger				= plugin_interface.getLogger().getTimeStampedChannel( "MediaServer" ); 

		LocaleUtilities loc_utils = plugin_interface.getUtilities().getLocaleUtilities();

		loc_utils.integrateLocalisedMessageBundle( "com.aelitis.azureus.plugins.upnpmediaserver.internat.Messages" );

		ta_unique_name	= plugin_interface.getTorrentManager().getPluginAttribute( "unique_name");
		
		Utilities utils = plugin_interface.getUtilities();
		
		timer2 	= utils.createTimer( "alive", true );
		timer3 	= utils.createTimer( "eventDispatch", false );
				
		PluginConfig	config = plugin_interface.getPluginconfig();
		
		UUID_rootdevice = config.getPluginStringParameter( "uuid", "" );
		
		if ( UUID_rootdevice.length() == 0 ){
			
			UUID_rootdevice = "uuid:" + UUIDGenerator.generateUUIDString();
		
			config.setPluginParameter( "uuid", UUID_rootdevice );
		}
			
		upnp_entities = new String[]{
				"upnp:rootdevice",
				"urn:schemas-upnp-org:device:MediaServer:1",
				"urn:schemas-upnp-org:service:ConnectionManager:1",
				"urn:schemas-upnp-org:service:ContentDirectory:1",
				UUID_rootdevice
			};
		
		final UIManager	ui_manager = plugin_interface.getUIManager();
		
		view_model = ui_manager.createBasicPluginViewModel( "upnpmediaserver.name" );
		
		view_model.getActivity().setVisible( false );
		view_model.getProgress().setVisible( false );
		
		logger.addListener(
				new LoggerChannelListener()
				{
					public void
					messageLogged(
						int		type,
						String	content )
					{
						view_model.getLogArea().appendText( content + "\n" );
					}
					
					public void
					messageLogged(
						String		str,
						Throwable	error )
					{
						if ( str.length() > 0 ){
							view_model.getLogArea().appendText( str + "\n" );
						}
						
						StringWriter sw = new StringWriter();
						
						PrintWriter	pw = new PrintWriter( sw );
						
						error.printStackTrace( pw );
						
						pw.flush();
						
						view_model.getLogArea().appendText( sw.toString() + "\n" );
					}
				});		
	
		view_model.setConfigSectionID( "upnpmediaserver.name" );
		
		logger.log( "RootDevice: " + UUID_rootdevice );
		
		config_model = 
			ui_manager.createBasicPluginConfigModel( "upnpmediaserver.name" );
		
		config_model.addLabelParameter2( "upnpmediaserver.info" );
		
		final BooleanParameter enable_upnp_p = config_model.addBooleanParameter2( "upnpmediaserver.enable_upnp", "upnpmediaserver.enable_upnp", true );

		enable_upnp = enable_upnp_p.getValue();
			
		final IntParameter stream_port_p = config_model.addIntParameter2( "upnpmediaserver.stream.port", "upnpmediaserver.stream.port", 0 );

		stream_port = stream_port_p.getValue();
		
		stream_port_p.addListener(
			new ParameterListener()
			{
				public void
				parameterChanged(
					Parameter	param )
				{
					int	port = stream_port_p.getValue();
					
					if ( port != stream_port ){
						
						stream_port = port;
						
						createContentServer();
					}
				}
			});
		
		final StringParameter snp = config_model.addStringParameter2( "upnpmediaserver.service_name", "upnpmediaserver.service_name", "Azureus" );
		
		service_name = snp.getValue();
		
		snp.addListener(
			new ParameterListener()
			{
				public void
				parameterChanged(
					Parameter	param )
				{
					service_name = snp.getValue();
				}
			});
		
		final BooleanParameter enable_publish_p = config_model.addBooleanParameter2( "upnpmediaserver.enable_publish", "upnpmediaserver.enable_publish", false );

		enable_lan_publish = enable_publish_p.getValue();
		
		enable_publish_p.addListener(
				new ParameterListener()
				{
					public void
					parameterChanged(
						Parameter	param )
					{
						enable_lan_publish = enable_publish_p.getValue();
					}
				});
		
		enable_upnp_p.addListener(
				new ParameterListener()
				{
					public void
					parameterChanged(
						Parameter	param )
					{
						enable_upnp = enable_upnp_p.getValue();
						snp.setEnabled( enable_upnp );
						enable_publish_p.setEnabled( enable_upnp );
					}
				});
		
		if ( !enable_upnp ){
			
			snp.setEnabled( false );
			
			enable_publish_p.setEnabled( false );
		}
		
		BooleanParameter menu_param = config_model.addBooleanParameter2("upnpmediaserver.enable_menus", "upnpmediaserver.enable_menus", true);
		
		if (menu_param.getValue()) {buildMenu();}
		
		config_model.addHyperlinkParameter2("upnpmediaserver.web_link", "http://www.azureuswiki.com/index.php/UG_Plugins#UPnP_Media_Server");

		ActionParameter print = config_model.addActionParameter2( "upnpmediaserver.printcd.label", "upnpmediaserver.printcd.button" );
		
		print.addListener(
			new ParameterListener()
			{
				public void 
				parameterChanged(
					Parameter param ) 
				{
					root_container.print( "" );
				}
			});
		
		ui_manager.addUIListener(
			new UIManagerListener()
			{
				public void
				UIAttached(
					UIInstance		instance )
				{
					if ( instance instanceof UISWTInstance ){	
				
						ui_manager.removeUIListener( this );
						
						log( "SWT user interface bound" );
						
						media_server_ui = new UPnPMediaServerUISWT();													
					}
				}
				
				public void
				UIDetached(
					UIInstance		instance )
				{
					
				}
			});
		
		createContentServer();
		
			// need to do this here due to 15 second backoff below...
		
		music_container 	= new contentContainer( "Music" );
		
		root_container.addChild( music_container );
		
		pictures_container 	= new contentContainer( "Pictures" );
		
		root_container.addChild( pictures_container );

		movies_container 	= new contentContainer( "Movies" );
		
		root_container.addChild( movies_container );

		downloads_container 	= new contentContainer( "Downloads" );
		
		root_container.addChild( downloads_container );


		plugin_interface.addListener(
			new PluginListener()
			{
				public void
				initializationComplete()
				{
					Thread t = 
						new Thread( "UPnPMediaServer::init" )
						{
							public void
							run()
							{
								try{
									Thread.sleep( 15000 );
									
								}catch( Throwable e ){
								}
								
								UPnPMediaServer.this.start();
							}
						};
					
					t.setPriority( Thread.MIN_PRIORITY );
					
					t.setDaemon( true );
					
					t.start();
				}
				
				public void
				closedownInitiated()
				{	
					if ( ssdp != null ){
						
						stop();
					}
				}
				
				public void
				closedownComplete()
				{
				}
			});
	}
	
	private void
	createContentServer()
	{
		if ( content_server != null ){
			
			log( "Destroying existing content server on port " + content_server.getPort());
			
			content_server.destroy();
			
			content_server = null;
		}
		
		try{
			content_server = new UPnPMediaServerContentServer( this );
	
			log("Content port = " + content_server.getPort());
		
		}catch( Throwable e ){
			
			log( "Failed to initialise content server", e );
		}
	}
	
	private void buildMenu() {
		MenuItemFillListener	menu_fill_simple_listener = 
			new MenuItemFillListener()
			{
				public void
				menuWillBeShown(
					MenuItem	menu,
					Object		_target )
				{
					Object	obj = null;
					
					if ( _target instanceof TableRow ){
						
						obj = ((TableRow)_target).getDataSource();
	
					}else{
						
						TableRow[] rows = (TableRow[])_target;
					     
						if ( rows.length > 0 ){
						
							obj = rows[0].getDataSource();
						}
					}
					
					if ( obj == null ){
						
						menu.setEnabled( false );

						return;
					}
					
					Download				download;
					DiskManagerFileInfo		file;
					
					if ( obj instanceof Download ){
					
						download = (Download)obj;

						if ( DISABLE_MENUS_FOR_INCOMPLETE && !download.isComplete()){
							
							menu.setEnabled( false );

							return;
						}

					}else{
						
						file = (DiskManagerFileInfo)obj;
						
						if ( DISABLE_MENUS_FOR_INCOMPLETE && file.getDownloaded() != file.getLength()){
							
							menu.setEnabled( false );

							return;
						}
					}
					
					menu.setEnabled( true );
				}
			};
		
		{
			// play
		
			MenuItemListener	menu_listener = 
				new MenuItemListener()
				{
					public void
					selected(
						MenuItem		_menu,
						Object			_target )
					{
						Object	obj = ((TableRow)_target).getDataSource();
						
						if ( obj == null ){
							
							return;
						}
						
						Download				download;
						DiskManagerFileInfo		file;
						
						if ( obj instanceof Download ){
						
							download = (Download)obj;
		
							file	= download.getDiskManagerFileInfo()[0];
						
						}else{
							
							file = (DiskManagerFileInfo)obj;
							
							try{
								download	= file.getDownload();
								
							}catch( DownloadException e ){	
								
								Debug.printStackTrace(e);
								
								return;
							}
						}
						
						String	id = createResourceID( download, file );
						
						System.out.println( "play: " + id );
	
						contentItem item = getContentFromResourceID( id );
						
						if ( item != null ){
							
							Iterator	it;
							
							synchronized( renderers ){
								
								it = new HashSet( renderers ).iterator();
							}
								
							while( it.hasNext()){
									
								UPnPMediaRenderer	renderer = (UPnPMediaRenderer)it.next();
								
								if ( !renderer.isBusy()){
									
									int	stream_id;
									
									synchronized( UPnPMediaServer.this ){
										
										stream_id	= stream_id_next++;
									}
									
									renderer.play( item, stream_id );
								}
							}
						}
					}
				};
				
				MenuItemListener	menu_listener_play_external = 
					new MenuItemListener()
					{
						public void
						selected(
							MenuItem		_menu,
							Object			_target )
						{
							Object	obj = ((TableRow)_target).getDataSource();
							
							if ( obj == null ){
								
								return;
							}
							
							try{
								play(obj);
								
							}catch( IPCException e ){
								
								log( "Failed to play '" + obj + "'", e );
							}
							
						}
					};
					
				// top level menus
					
			TableContextMenuItem menu_item_itorrents;
			if ( DISABLE_MENUS_FOR_INCOMPLETE ){
				menu_item_itorrents = null;
			}else{
				menu_item_itorrents = plugin_interface.getUIManager().getTableManager().addContextMenuItem(TableManager.TABLE_MYTORRENTS_INCOMPLETE, "upnpmediaserver.contextmenu");
				menus.add( menu_item_itorrents );
			}
			TableContextMenuItem menu_item_ctorrents 	= plugin_interface.getUIManager().getTableManager().addContextMenuItem(TableManager.TABLE_MYTORRENTS_COMPLETE, "upnpmediaserver.contextmenu");
			menus.add( menu_item_ctorrents );
			TableContextMenuItem menu_item_files 		= plugin_interface.getUIManager().getTableManager().addContextMenuItem(TableManager.TABLE_TORRENT_FILES, "upnpmediaserver.contextmenu");
			menus.add( menu_item_files );
			
				// set menu style
			
			if ( menu_item_itorrents != null ){
				menu_item_itorrents.setStyle(TableContextMenuItem.STYLE_MENU);
			}
			menu_item_ctorrents.setStyle(TableContextMenuItem.STYLE_MENU);
			menu_item_files.setStyle(TableContextMenuItem.STYLE_MENU);
			
				// create play-external items
			
			TableContextMenuItem menup1;
			if ( DISABLE_MENUS_FOR_INCOMPLETE ){
				menup1 = null;
			}else{
				menup1 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_itorrents, 	"upnpmediaserver.contextmenu.playExternal" );
			}
			TableContextMenuItem menup2 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_ctorrents, "upnpmediaserver.contextmenu.playExternal" );
			TableContextMenuItem menup3 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_files, "upnpmediaserver.contextmenu.playExternal" );

				// create play items
			
			TableContextMenuItem menu1;
			if ( DISABLE_MENUS_FOR_INCOMPLETE ){
				menu1 = null;
			}else{
				menu1 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_itorrents, 	"upnpmediaserver.contextmenu.play" );
			}
			TableContextMenuItem menu2 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_ctorrents, "upnpmediaserver.contextmenu.play" );
			TableContextMenuItem menu3 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_files, "upnpmediaserver.contextmenu.play" );
		
			MenuItemFillListener	menu_fill_listener = 
				new MenuItemFillListener()
				{
					public void
					menuWillBeShown(
						MenuItem	menu,
						Object		_target )
					{
						Object	obj = null;
						
						if ( _target instanceof TableRow ){
							
							obj = ((TableRow)_target).getDataSource();
		
						}else{
							
							TableRow[] rows = (TableRow[])_target;
							     
							if ( rows.length > 0 ){
							
								obj = rows[0].getDataSource();
							}
						}
						
						if ( obj == null ){
							
							menu.setEnabled( false );

							return;
						}
						
						Download				download;
						DiskManagerFileInfo		file;
						
						if ( obj instanceof Download ){
						
							download = (Download)obj;

							if ( DISABLE_MENUS_FOR_INCOMPLETE && !download.isComplete()){
								
								menu.setEnabled( false );

								return;
							}

						}else{
							
							file = (DiskManagerFileInfo)obj;
							
							if ( DISABLE_MENUS_FOR_INCOMPLETE && file.getDownloaded() != file.getLength()){
								
								menu.setEnabled( false );

								return;
							}
						}
						
						synchronized( renderers ){
							
							boolean	enabled = false;
							
							Iterator	it = renderers.iterator();
							
							while( it.hasNext()){
								
								UPnPMediaRenderer	renderer = (UPnPMediaRenderer)it.next();
								
								if ( !renderer.isBusy()){
									
									enabled	= true;
									break;
								}
							}
							
							menu.setEnabled( enabled );
						}
					}
				};
			
			if ( menu1 != null ){
				menu1.addListener( menu_listener );
			}
			menu2.addListener( menu_listener );
			menu3.addListener( menu_listener );
			
			if ( menu1 != null ){
				menu1.addFillListener( menu_fill_listener );
			}
			menu2.addFillListener( menu_fill_listener );
			menu3.addFillListener( menu_fill_listener );
			
			if ( menup1 != null ){
				menup1.addListener( menu_listener_play_external );
			}
			menup2.addListener( menu_listener_play_external );
			menup3.addListener( menu_listener_play_external );
			
			menup2.addFillListener( menu_fill_simple_listener );
			menup3.addFillListener( menu_fill_simple_listener );
		
			// copy to clip
		
			menu_listener = 
				new MenuItemListener()
				{
					public void
					selected(
						MenuItem		_menu,
						Object			_target )
					{
						Object	obj = ((TableRow)_target).getDataSource();
						
						if ( obj == null ){
							
							return;
						}
						
						Download				download;
						DiskManagerFileInfo		file;
						
						if ( obj instanceof Download ){
						
							download = (Download)obj;
		
							file	= download.getDiskManagerFileInfo()[0];
						
						}else{
							
							file = (DiskManagerFileInfo)obj;
							
							try{
								download	= file.getDownload();
								
							}catch( DownloadException e ){	
								
								Debug.printStackTrace(e);
								
								return;
							}
						}
						
						String	id = createResourceID( download, file );
							
						contentItem item = getContentFromResourceID( id );
						
						if ( item != null ){
						
							try{
								plugin_interface.getUIManager().copyToClipBoard( item.getURI( "127.0.0.1", -1 ));
								
							}catch( Throwable e ){
								
								log( "Failed to copy to clipboard", e);
							}
						}
					}
				};
				
			if ( DISABLE_MENUS_FOR_INCOMPLETE ){
				menu1 = null;
			}else{
				menu1 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_itorrents, 	"upnpmediaserver.contextmenu.toclipboard" );
			}
			menu2 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_ctorrents, 	"upnpmediaserver.contextmenu.toclipboard" );
			menu3 = plugin_interface.getUIManager().getTableManager().addContextMenuItem(menu_item_files, 			"upnpmediaserver.contextmenu.toclipboard" );
			
			if ( menu1 != null ){
				menu1.addFillListener( menu_fill_simple_listener );
			}
			menu2.addFillListener( menu_fill_simple_listener );
			menu3.addFillListener( menu_fill_simple_listener );

			if ( menu1 != null ){
				menu1.addListener( menu_listener );
			}
			menu2.addListener( menu_listener );
			menu3.addListener( menu_listener );
		}
	}

	public void
	unload()
	{
		if ( !unloaded ){
			
			log( "Unloading" );
		}
		
		unloaded = true;
	
		if ( timer2_event != null ){
			
			timer2_event.cancel();
		}
		
		if ( timer3_event != null ){
			
			timer3_event.cancel();
		}
	
		timer2.destroy();
		timer3.destroy();
		
		config_model.destroy();
		view_model.destroy();
		
		for (int i=0;i<menus.size();i++){
			
			((TableContextMenuItem)menus.get(i)).remove();
		}
		
		if ( upnp != null && upnp_listener != null ){
			
			upnp.removeRootDeviceListener( upnp_listener );
		}
		
		if ( ssdp != null && ssdp_listener != null ){
			
			ssdp.removeListener( ssdp_listener );
		}
		
		if ( dm_listener != null ){
			
			plugin_interface.getDownloadManager().removeListener( dm_listener );
		}
		
		if ( web_context != null ){
			
			try{
				TrackerWebContent_destroy.invoke( web_context, new Object[0] );
				
			}catch( Throwable e ){
			}
		}
		
		content_server.destroy();
	}
	
		// ***** plugin public interface starts
	
	public int
	addLocalRenderer(
		IPCInterface	callback )
	{
		UPnPMediaRendererLocal	renderer = new UPnPMediaRendererLocal( callback );
		
		synchronized( renderers ){
			
			renderers.add( renderer );
		}
		
		return( renderer.getID());
	}
	
	public long
	getStreamBufferBytes(
		int		content_id,
		int		stream_id )
	
		throws IPCException
	{
		content	c = getContentFromID( content_id );
		
		if ( c == null || !( c instanceof contentItem )){
			
			throw( new IPCException( "Content id " + content_id + " not found" ));
		}
		
		contentItem	item = (contentItem)c;

		DiskManagerFileInfo	file = item.getFile();
		
			// if already complete then simple
		
		if ( file.getLength() == file.getDownloaded()){
			
			return( Long.MAX_VALUE );
		}
		
		UPnPMediaServerContentServer.streamInfo	stream_info = content_server.getStreamInfo( stream_id );
		
		if ( stream_info == null ){
			
			return( -1 );
		}
		
		long	remaining 	= stream_info.getRemaining();
		long	available	= stream_info.getAvailableBytes();
		
		if ( remaining == available ){
			
			if ( available == -1 ){
				
				return( -1 );
			}
			
				// indicate all available
			
			return( Long.MAX_VALUE );
		}
		
		return( available );
	}
	
	public int
	getStreamType(
		int		content_id )
	
		throws IPCException
	{
		content	c = getContentFromID( content_id );
		
		if ( c == null || !( c instanceof contentItem )){
			
			throw( new IPCException( "Content id " + content_id + " not found" ));
		}
		
		contentItem	item = (contentItem)c;

		String	cla = item.getContentClass();
		
		if ( cla == CONTENT_AUDIO ){
			
			return( 0 );
			
		}else if ( cla == CONTENT_VIDEO ){
			
			return( 1 );
			
		}else{
			
			return( -1 );
		}
	}
	
	public void
	playDownload(
		Download			download )
	
		throws IPCException
	{
		play( download );
	}
	
	
	public void
	playFile(
		DiskManagerFileInfo	file )
	
		throws IPCException
	{
		play( file );
	}
	
	public String
	getContentURL(
		Download download)
	
		throws IPCException
	{
		return( getContentURL( download.getDiskManagerFileInfo()[0]));
	}
	
	public String
	getContentURL(
		DiskManagerFileInfo	file )
	
		throws IPCException
	{
		try{
			Download	download = file.getDownload();
			
			file	= download.getDiskManagerFileInfo()[0];
			
			String	id = createResourceID( download, file );

			contentItem item = getContentFromResourceID( id );

			if ( item != null ){

				try{
					return item.getURI( "127.0.0.1", -1 );

				}catch( Throwable e ){

					log( "Failed to get URI", e);
				}
			}
			
			return "";
			
		} catch (Throwable t) {
			
			throw new IPCException(t);
		}
	}

	public void setQuickTimeAvailable(
		boolean avail )
	{
		quickTimeAvail = avail;
	}
	
	public boolean
	isRecognisedMedia(
		DiskManagerFileInfo	file )
	{
		try{
			String	id = createResourceID( file.getDownload(), file );
		
			contentItem item = getContentFromResourceID( id );
			
			if ( item == null ){
				
				return( false );
			}
			
			String cla = item.getContentClass();
		
			return( cla == CONTENT_AUDIO || cla == CONTENT_IMAGE || cla == CONTENT_VIDEO );
			
		}catch( Throwable e ){
			
			logger.log( e );
		}
		
		return( false );
	}

		// **** plugin public interface ends
	
	protected PluginInterface
	getPluginInterface()
	{
		return( plugin_interface );
	}
	
	protected String
	getServerName()
	{
		return( System.getProperty( "os.name" ) + "/" + System.getProperty("os.version") + " UPnP/1.0 " +
				Constants.AZUREUS_NAME + "/" + Constants.AZUREUS_VERSION );
	}
	
	protected void
	log(
		String		str )
	{
		logger.log( str );
	}
	
	protected void
	log(
		String		str,
		Throwable	e )
	{
		logger.log( str, e );
	}
	
	protected void
	logAlert(
		String		str )
	{
		logger.logAlertRepeatable( LoggerChannel.LT_ERROR, str );
	}
	protected void
	start()
	{
		synchronized( startup_lock ){
			
			if ( starting ){
				
				return;
			}
			
			starting	= true;
		}
		
		logger.log( "Server starts: upnp enabled=" + enable_upnp );
		
		try{
			if ( enable_upnp ){
				
				UPnPAdapter adapter = 
					new UPnPAdapter()
					{
						public SimpleXMLParserDocument
						parseXML(
							String	data )
						
							throws SimpleXMLParserDocumentException
						{
							return( plugin_interface.getUtilities().getSimpleXMLParserDocumentFactory().create( data ));
						}
						
						public ResourceDownloaderFactory
						getResourceDownloaderFactory()
						{
							return( plugin_interface.getUtilities().getResourceDownloaderFactory());
						}
						
						public UTTimer
						createTimer(
							String	name )
						{
							return( plugin_interface.getUtilities().createTimer( name ));
						}
						
						public void
						createThread(
							String				name,
							final Runnable		runnable )
						{
							plugin_interface.getUtilities().createThread( name, runnable );
						}
						
						public Comparator
						getAlphanumericComparator()
						{
							return( plugin_interface.getUtilities().getFormatters().getAlphanumericComparator( true ));
						}
	
						public void
						log(
							Throwable	e )
						{
							Debug.printStackTrace(e);
						}
						
						public void
						trace(
							String	str )
						{
							// System.out.println( str );
						}
						
						public void
						log(
							String	str )
						{
							// System.out.println( str );
						}
						
						public String
						getTraceDir()
						{
							return( plugin_interface.getPluginDirectoryName());
						}
					};
					
	
				ssdp = UPnPFactory.getSSDP( 
									adapter, 
									UPnPSSDP.SSDP_GROUP_ADDRESS, 
									UPnPSSDP.SSDP_GROUP_PORT, 
									0, 
									null );
	
				int	ssdp_port = ssdp.getControlPort();
			
				createContent();
				
					// bail out early in to reduce possibility of breaking a reload
					// in progress
				
				if ( unloaded ){
					
					return;
				}
				
				web_context = 
					plugin_interface.getTracker().createWebContext(
							ssdp_port,
							Tracker.PR_HTTP );
				
				web_context.addAuthenticationListener(
					new TrackerAuthenticationListener()
					{
						public boolean
						authenticate(
							URL			resource,
							String		user,
							String		password )
						{
							return( true );
						}
						
						public byte[]
						authenticate(
							URL			resource,
							String		user )
						{
							return( null );
						}
					});
				
				web_context.addPageGenerator(
					new TrackerWebPageGenerator()
					{
						public boolean
						generate(
							TrackerWebPageRequest		request,
							TrackerWebPageResponse		response )
						
							throws IOException
						{
							String	url = request.getURL();
							
							Map	headers = request.getHeaders();
							
							String	action = (String)headers.get( "soapaction" );					
	
							if ( 	url.equals( "/ConnectionManager/Event" )||
									url.equals( "/ContentDirectory/Event" )){					
									
								// System.out.println( request.getHeader());
	
								Map	headers_out = new HashMap();
								
								String result = 
									processEvent( 
										url.equals( "/ContentDirectory/Event" ),
										request.getClientAddress2(),
										headers_out,	
										request.getHeader().startsWith( "SUBSCRIBE" ),
										(String)headers.get( "callback" ),
										(String)headers.get( "sid" ),
										(String)headers.get( "timeout" ));
										
								response.setHeader( "SERVER", getServerName());
								
								Iterator	it = headers_out.keySet().iterator();
								
								while( it.hasNext()){
									
									String	name 	= (String)it.next();
									String	value	= (String)headers_out.get(name);
									
									response.setHeader( name, value );
								}
								
								response.useStream( "xml", new ByteArrayInputStream( result.getBytes( "UTF-8" )));
	
								return( true );
															
							}else if ( action == null ){
								
									// unfortunately java let's getSourceAsStream escape from the
									// package hierarchy and onto the file system, at least when
									// loading the class from FS instead of .jar
								
									// first we don't support ..
								
								url = url.trim();
								
								if ( url.indexOf( ".." ) == -1 && !url.endsWith( "/" )){
								
										// second we don't support nested resources
									
									int	 pos = url.lastIndexOf("/");
									
									if ( pos != -1 ){
										
										url = url.substring( pos );
									}
									
									String resource = "/com/aelitis/azureus/plugins/upnpmediaserver/resources" + url;
						
									InputStream stream = getClass().getResourceAsStream( resource );
									
									if ( stream != null ){
										
										try{
											if ( url.equalsIgnoreCase( "/RootDevice.xml" )){
												
												byte[]	buffer = new byte[1024];
												
												String	content = "";
												
												while( true ){
													
													int	len = stream.read( buffer );
													
													if ( len <= 0 ){
														
														break;
													}
													
													content += new String( buffer, 0, len, "UTF-8" );
												}
												
												content = content.replaceAll( "%UUID%", UUID_rootdevice );
												content = content.replaceAll( "%SERVICENAME%", service_name );
												
												stream.close();
												
												stream = new ByteArrayInputStream( content.getBytes( "UTF-8" ));
											}
											
											response.useStream( "xml", stream );
										
											return( true );
											
										}finally{
												
											stream.close();
										}
									}
								}
								
								logger.log( "HTTP: no match for " + url + ":" + request.getHeader());
								
								return( false );
								
							}else{
													
								String	host = (String)headers.get( "host" );
								
								if ( host == null ){
									
									host = "127.0.0.1";
									
								}else{
									
									int	pos = host.indexOf(':');
									
									if ( pos != -1 ){
										
										host	= host.substring(0,pos);
									}
									
									host	= host.trim();
								}
								
								String	client = (String)headers.get( "user-agent" );
								
								if ( client == null ){
									
									client = request.getClientAddress();
									
								}else{
									
									client = client + ":" + request.getClientAddress();
								}
								
								try{
										// Philips steamium sends invalid XML (trailing 0 byte screws the parser) so
										// sanitise it here. Note it has to be in utf-8 as this is part of the UPnP 
										// spec.
									
									String	action_content = "";
									
									byte[]	buffer = new byte[1024];
									
									InputStream	is = request.getInputStream();
									
									while( true ){
										
										int	len = is.read( buffer );
										
										if ( len < 0 ){
											
											break;
										}
										
										action_content += new String( buffer, 0, len, "UTF-8" );
									}
									
									action_content	= action_content.trim();
									
									String	result = 
										processAction(
												host,
												client,
												url,
												action,
												plugin_interface.getUtilities().getSimpleXMLParserDocumentFactory().create( 
														action_content ));
										
									if ( result == null ){
										
											// not implemented
										
										response.setReplyStatus( 501 );
										
									}else{
									
										response.setHeader( "SERVER", getServerName());
										
										response.useStream( "xml", new ByteArrayInputStream( result.getBytes( "UTF-8" )));
									}
									
									return( true );
									
								}catch( Throwable e ){
									
									Debug.printStackTrace(e);
									
									logger.log( e );
									
									return( false );
								}
							}
						}
					});
				
				ssdp_listener = 
						new UPnPSSDPListener()
						{
							public void
							receivedResult(
								NetworkInterface	network_interface,
								InetAddress			local_address,
								InetAddress			originator,
								String				USN,
								URL					location,
								String				ST,
								String				AL )
							{
								// System.out.println( "receivedResult: " + location + "/" + ST + "/" + AL );
							}
							
							public void
							receivedNotify(
								NetworkInterface	network_interface,
								InetAddress			local_address,
								InetAddress			originator,
								String				USN,
								URL					location,
								String				NT,
								String				NTS )
							{
								// System.out.println( "receivedNotify: " + USN + "/" + location + "/" + NT + "/" + NTS );					
							}
	
							public String[]
							receivedSearch(
								NetworkInterface	network_interface,
								InetAddress			local_address,
								InetAddress			originator,
								String				ST )
							{
								for (int i=0;i<upnp_entities.length;i++){
									
									if ( ST.equals( upnp_entities[i] )){
										
										if ( i == 0 ){
											
											synchronized( logged_searches ){
												
												if ( !logged_searches.contains( originator )){
													
													logged_searches.add( originator );
													
													logger.log( "SSDP: search from " + originator );
												}
											}
										}
										
										return( new String[]{ 
												UUID_rootdevice,
												"RootDevice.xml" });
									}
								}
								
								return( null );
							}
							
							public void
							interfaceChanged(
								NetworkInterface	network_interface )
							{	
							}
						};
				
				ssdp.addListener( ssdp_listener );

				sendAlive();
	
				upnp = UPnPFactory.getSingleton( adapter, null );
				
				upnp_listener = 
					new UPnPListener()
					{
						public boolean
						deviceDiscovered(
							String		USN,
							URL			location )
						{
							return( true );
						}
						
						public void
						rootDeviceFound(
							UPnPRootDevice		device )
						{
							if ( device.getDevice().getDeviceType().equals("urn:schemas-upnp-org:device:MediaRenderer:1")){
								
								final UPnPMediaRenderer	renderer = new UPnPMediaRendererRemote( UPnPMediaServer.this, device );
								
								synchronized( renderers ){
									
									renderers.add( renderer );
								}
								
								device.addListener(
									new UPnPRootDeviceListener()
									{
										public void
										lost(
											UPnPRootDevice	root,
											boolean			replaced )
										{
											synchronized( renderers ){
												
												renderers.remove( renderer );
											}
										}
									});
							}
						}
					};
				
				upnp.addRootDeviceListener( upnp_listener );

				timer2_event = timer2.addPeriodicEvent(
						60*1000,
						new UTTimerEventPerformer()
						{
							public void
							perform(
								UTTimerEvent		event )
							{
								sendAlive();
							}
						});
				
				timer3_event = timer3.addPeriodicEvent(
						2*1000,
						new UTTimerEventPerformer()
						{
							int	ticks = 0;
							
							public void
							perform(
								UTTimerEvent		event )
							{
								ticks++;
								
								sendEvents( ticks % 30 == 0 );
							}
						});
				
			}else{
				
				createContent();
			}
		}catch( Throwable e ){
			
		}finally{
			
			startup_sem.releaseForever();
			
			if ( unloaded ){
				
				unload();
			}
		}
	}
	
	protected void
	ensureStarted()
	{
			// we defer startup on init to help speed up AZ init. However, we can
			// get requests for content items before this delay has completed so
			// we need to force earlier init if needed
		
		if ( startup_sem.isReleasedForever()){
			
			return;
		}
		
		start();
		
		if ( !startup_sem.reserve( 30000 )){
			
			log( "Timeout waiting for startup" );
		}
	}
	
	protected boolean
	isUserSelectedContentPort()
	{
		return( stream_port != 0 );
	}
	
	protected int
	getContentPort()
	{
		if ( stream_port != 0 ){
			
			return( stream_port );
		}
		
		PluginConfig	config = plugin_interface.getPluginconfig();
		
		
		return( config.getPluginIntParameter( "content_port", 0 ));
	}
	
	protected void
	setContentPort(
		int		port )
	{
		PluginConfig	config = plugin_interface.getPluginconfig();
		
		config.setPluginParameter( "content_port", port, true );
	}
	
	protected void
	createContent()
	{
		dm_listener =
			new DownloadManagerListener()
			{
				public void
				downloadAdded(
					Download	download )
				{		
					if ( download.getTorrent() == null ){
						
						return;
					}
					
					addContent( download );
				}
				
				public void
				downloadRemoved(
					Download	download )
				{
					if ( download.getTorrent() == null ){
						
						return;
					}
					
					removeContent( download );
				}
			};
		
		plugin_interface.getDownloadManager().addListener( dm_listener );
	}
	
	protected void
	addContent(
		Download			download )
	{
		synchronized( downloads_container ){
			
			DiskManagerFileInfo[]	files = download.getDiskManagerFileInfo();
						
			if ( files.length == 1 ){
						
				String title = getUniqueName( download, files[0].getFile().getName());
				
				contentItem	item = new contentItem( download, files[0], title );

				downloads_container.addChild( item );
				
				addToFilters( item );
				
			}else{
				
				contentContainer container = 
					new contentContainer( getUniqueName( download, download.getName()));
				
				downloads_container.addChild( container );
				
				Set	name_set = new HashSet();
				
				boolean	duplicate = false;
				
				for (int j=0;j<files.length;j++){

					DiskManagerFileInfo	file = files[j];
									
					String	title = file.getFile().getName();
					
					if ( name_set.contains( title )){
						
						duplicate = true;
						
						break;
					}
					
					name_set.add( title );
				}
				
				for (int j=0;j<files.length;j++){
					
					DiskManagerFileInfo	file = files[j];
							
						// keep these unique within the container
					
					String	title = file.getFile().getName();
					
					if ( duplicate ){
						
						title =  ( j + 1 ) + ". " + title;
					}
					
					contentItem	item = new contentItem( download, file, title );
		
					container.addChild( item );
				}
				
				addToFilters( container );
			}
		}
	}
	
	protected void
	removeContent(
		Download			download )

	{
		synchronized( downloads_container ){

			DiskManagerFileInfo[]	files = download.getDiskManagerFileInfo();
			
			String	unique_name;
			
			if ( files.length == 1 ){

				unique_name = getUniqueName( download, files[0].getFile().getName());
								
			}else{
				
				unique_name = getUniqueName( download, download.getName());
			}
			
			content container = downloads_container.removeChild( unique_name );

			removeUniqueName( download );
			
			if ( container != null ){
				
				removeFromFilters( container );
			}
		}
	}
	
	private String
	findPrimaryContentType(
		content		con )
	{
		if ( con instanceof contentItem ){
			
			return(((contentItem)con).getContentClass());
			
		}else{
	
			String	type = CONTENT_UNKNOWN;
			
			contentContainer container = (contentContainer)con;
			
			List kids = container.getChildren();
			
			for (int i=0;i<kids.size();i++){
				
				String	t = findPrimaryContentType((content)kids.get(i));
				
				if ( t == CONTENT_VIDEO ){
					
					return( t );
				}
				
				if ( t == CONTENT_AUDIO ){
					
					type = t;
					
				}else if ( t == CONTENT_IMAGE && type == CONTENT_UNKNOWN ){
					
					type = t;
					
				}
			}
			
			return( type );
		}
	}
	
	private void
	addToFilters(
		content		con )
	{
		String type = findPrimaryContentType( con );
		
		if ( type == CONTENT_VIDEO ){
			
			movies_container.addLink( con );
			
		}else if ( type == CONTENT_AUDIO ){
			
			music_container.addLink( con );
	
		}else if ( type == CONTENT_IMAGE ){
			
			pictures_container.addLink( con );
		}
	}
	
	private void
	removeFromFilters(
		content		con )
	{
		movies_container.removeLink( con.getName());
		
		pictures_container.removeLink( con.getName());
		
		music_container.removeLink( con.getName());
	}
	
	private Map unique_name_map	= new HashMap();
	private Set unique_names 	= new HashSet();
	
	protected String
	getUniqueName(
		Download		dl,
		String			name )
	{
		String result = (String)unique_name_map.get( dl );

		if ( result != null ){
			
			return( result );
		}
			// ensure that we have a unique name for the download
		
		result = dl.getAttribute( ta_unique_name );
		
		if ( result == null || result.length() == 0 ){
			
			result = name;
		}
		
		int	num = 1;
		
		while( unique_names.contains( result )){
			
			result = (num++) + ". " + name; 
		}
	
			// if we had to make one up, record it persistently
		
		if ( num > 1 ){
		
			dl.setAttribute( ta_unique_name, result );
		}
		
		unique_names.add( result );
		
		unique_name_map.put( dl, result );
		
		return( result );
	}
	
	protected void
	removeUniqueName(
		Download		dl )
	{
		String name = (String)unique_name_map.remove( dl );
		
		if ( name != null ){
			
			unique_names.remove( name );
		}
	}
	
	protected contentItem
	getContentFromID(
		int		id )
	{
		ensureStarted();
		
		synchronized( content_map ){
			
			return((contentItem)content_map.get( new Integer( id )));
		}
	}
	
	protected String
	createResourceID(
		Download				download,
		DiskManagerFileInfo		file )
	{
		String	res =
			ByteFormatter.encodeString(download.getTorrent().getHash()) + "-" + file.getIndex();
		
		String	name = file.getFile().toString();
		
		int	pos = name.lastIndexOf('.');
		
		if ( pos != -1 && !name.endsWith(".")){
			
			res += name.substring( pos );
		}
		
		return( res );
	}
	
	protected contentItem
	getContentFromResourceID(
		String		id )
	{
		ensureStarted();
		
		int	pos = id.indexOf( "-" );
		
		if ( pos == -1 ){
			
			log( "Failed to decode resource id '" + id + "'" );
			
			return( null );
		}
		
		byte[]	hash = ByteFormatter.decodeString( id.substring( 0, pos ));
		
		String	rem = id.substring( pos+1 );
		
		pos = rem.indexOf( "." );
		
		if ( pos != -1 ){
			
			rem = rem.substring( 0, pos );
		}
		
		try{
			int file_index = Integer.parseInt( rem );
			
			return( getExistingContentFromHashAndFileIndex( hash, file_index ));
			
		}catch( Throwable e ){
			
			log( "Failed to decode resource id '" + id + "'", e );
			
			return( null );
		}
	}
	
	protected contentItem
	getContentFromHash(
		byte[]		hash )
	{
		contentItem	item = getExistingContentFromHash( hash );
		
		if ( item == null ){
			
			AzureusContentDirectory[]	directories = AzureusContentDirectoryManager.getDirectories();
			
			Map	lookup = new HashMap();
			
			lookup.put( AzureusContentDirectory.AT_BTIH, hash );
			
			for (int i=0;i<directories.length;i++){
				
				AzureusContentDirectory	directory = directories[i];
				
				AzureusContent	content = directory.lookupContent( lookup );
				
				if ( content != null ){
					
					Torrent	torrent = content.getTorrent();
					
					if ( torrent != null ){
						
						DownloadManager	dm = plugin_interface.getDownloadManager();
						
							// hmm, when to resume...
						
						dm.pauseDownloads();
						
						try{
							Download download = dm.addDownload( torrent );
							
							addContent( download );
									
							item = getExistingContentFromHash( hash );

							int	sleep 	= 100;
							int	max		= 20000;
							
								// need to wait for things to get started else file might not
								// yet exist and we bork
							
							for (int j=0;j<max/sleep;j++){
								
								int	state = download.getState();
								
								if ( 	state == Download.ST_DOWNLOADING || 
										state == Download.ST_SEEDING ||
										state == Download.ST_ERROR ||
										state == Download.ST_STOPPED ){
									
									break;
								}
								
								Thread.sleep(sleep);
							}
							
							break;
							
						}catch( Throwable e ){
							
							log( "Failed to add download", e );
						}
					}
				}
			}
		}
		
		return( item );
	}
	
	protected contentItem
	getExistingContentFromHash(
		byte[]		hash )
	{
		return( getExistingContentFromHashAndFileIndex( hash, 0 ));
	}
	
	protected contentItem
	getExistingContentFromHashAndFileIndex(
		byte[]		hash,
		int			file_index )
	{
		ensureStarted();
		
		synchronized( content_map ){

			Iterator	it = content_map.values().iterator();
			
			while( it.hasNext()){
				
				content	content = (content)it.next();
				
				if ( content instanceof contentItem ){
					
					contentItem	item = (contentItem)content;
					
					DiskManagerFileInfo	file = item.getFile();
					
					if ( file.getIndex() == file_index ){
					
						try{
							Download	dl = item.getFile().getDownload();
							
							Torrent	torrent = dl.getTorrent();
							
							if ( torrent != null ){
								
								if ( Arrays.equals( torrent.getHash(), hash )){
									
									return( item );
								}
							}
						}catch( Throwable e ){
							
							log( "hmm", e );
						}
					}
				}
			}
		}
		
		return( null );
	}
	
	protected String
	processAction(
		String						host,
		String						client,
		String						url,
		String						action,
		SimpleXMLParserDocument		doc )
	{
		String	xmlns;
		
		boolean	directory;
		
		if ( url.equals( "/ContentDirectory/Control" )){
			
			xmlns	= "urn:schemas-upnp-org:service:ContentDirectory:1";
			
			directory	= true;
			
		}else{
			
			xmlns	= "urn:schemas-upnp-org:service:ConnectionManager:1";
			
			directory = false;
		}
		
		// System.out.println( "Process action '" + action + "'" );
		// doc.print();
		
		SimpleXMLParserDocumentNode	body = doc.getChild( "Body" );

		SimpleXMLParserDocumentNode	call = body.getChildren()[0];
		
		String	command = call.getName();				
		
		SimpleXMLParserDocumentNode[]	args = call.getChildren();
		
		Map	arg_map = new HashMap();
		
		String	arg_str = "";
		
		for (int i=0;i<args.length;i++){
			
			SimpleXMLParserDocumentNode	arg = args[i];
			
			arg_map.put( arg.getName(), arg.getValue());
			
			arg_str += (i==0?"":",") + arg.getName() + " -> " + arg.getValue();
		}
	
		logger.log( "Action (client=" + client + "): " + url + ":" + command +", " + arg_str );
		
		resultEntries	result = new resultEntries();
		
		if ( directory ){
			
			if ( command.equals( "GetSearchCapabilities")){
				
				getSearchCapabilities( arg_map, result );
				
			}else if ( command.equals( "GetSortCapabilities" )){
				
				getSortCapabilities( arg_map, result );
	
			}else if ( command.equals( "GetSystemUpdateID" )){
	
				getSystemUpdateID( arg_map, result);
	
			}else if ( command.equals( "Browse" )){
	
				int num = browse( host, arg_map, result );
				
				logger.log( "    -> " + num + " entries" );
				
			}else{
				
				return( null );
			}
		}else{
			
			if ( command.equals( "GetProtocolInfo")){

				getProtocolInfo( arg_map, result );
				
			}else if ( command.equals( "GetCurrentConnectionIDs")){

				getCurrentConnectionIDs( arg_map, result );
				
			}else if ( command.equals( "GetCurrentConnectionInfo")){

				getCurrentConnectionInfo( arg_map, result );
				
			}else{
				
				return( null );
			}
		}
		
		String	response =
			"<?xml version=\"1.0\" encoding=\"utf-8\"?>"+
			"<s:Envelope xmlns:s=\"http://schemas.xmlsoap.org/soap/envelope/\" s:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\">"+
				"<s:Body>";
				
			// todo we should support faults here for errors - see http://www.upnp.org/specs/arch/UPnP-DeviceArchitecture-v1.0.pdf page 50
		
		response += "<u:" + command + "Response xmlns:u=\"" + xmlns + "\">";
			
		List	result_args = result.getArgs();
		List	result_vals	= result.getValues();
		
		for (int i=0;i<result_args.size();i++){
			
			String	arg = (String)result_args.get(i);
			String	val	= (String)result_vals.get(i);
			
			response += "<" + arg + ">" + val + "</" + arg + ">";
		}
		
		response += "</u:" + command + "Response>";

		response += "</s:Body>"+
					"</s:Envelope>";
		
		return( response );
	}
	
	protected void
	getProtocolInfo(
		Map				args,
		resultEntries	result )
	{
		result.add( "Source", CM_SOURCE_PROTOCOLS );
		result.add( "Sink", CM_SINK_PROTOCOLS );
	}
	
	protected void
	getCurrentConnectionIDs(
		Map				args,
		resultEntries	result )
	{
		result.add( "ConnectionIDs", "" );
	}
	
	protected void
	getCurrentConnectionInfo(
		Map				args,
		resultEntries	result )
	{
		result.add( "RcsID", "0" );
		result.add( "AVTransportID", "0" );
		result.add( "ProtocolInfo", "" );
		result.add( "PeerConnectionManager", "" );
		result.add( "PeerConnectionID", "-1" );
		result.add( "Direction", "Input" );
		result.add( "Status", "Unknown" );
	}
	
	protected void
	getSearchCapabilities(
		Map				args,
		resultEntries	result )
	{
		result.add( "SearchCaps", "" );
	}
	
	protected void
	getSortCapabilities(
		Map				args,
		resultEntries	result )
	{
		result.add( "SortCaps", "" );
	}
	
	protected int
	getSystemUpdateID()
	{
		return( system_update_id );
	}
	
	protected void
	getSystemUpdateID(
		Map				args,
		resultEntries	result )
	{
		result.add( "Id",String.valueOf(system_update_id) );
	}
	
	protected int
	browse(
		String			host,
		Map				args,
		resultEntries	result )
	{
		String	oid			= (String)args.get( "ObjectID" );
		String	browseFlag 	= (String)args.get( "BrowseFlag" );
		String	start_str	= (String)args.get( "StartingIndex" );
		String	request_str	= (String)args.get( "RequestedCount" );
		
		int	start_index	 	= start_str==null?0:Integer.parseInt( start_str );
		int	requested_count	= request_str==null?0:Integer.parseInt( request_str );
		
		String	didle 			= "";
		int		num_returned	= 0;
		int		total_matches	= 0;
		int		update_id		= getSystemUpdateID();
		
		content	cont;
		
		synchronized( content_map ){
			
			cont = (content)content_map.get( new Integer( oid.length()==0?"0":oid ));
		}
		
		if ( cont == null ){
			
				// todo error case
			
			log( "Object '" + oid + "' not found" );
			
		}else{
			
			if ( cont instanceof contentContainer ){
				
				update_id = ((contentContainer)cont).getUpdateID();
			}
			
			if ( browseFlag.equals( "BrowseMetadata")){
						
				didle = getDIDL( cont, host );
				
				num_returned	= 1;
				total_matches	= 1;
				
			}else if ( browseFlag.equals( "BrowseDirectChildren")){
				
				boolean	ok = false;
				
				if ( enable_lan_publish ){
					
					ok = true;
					
				}else{
					
					try{
						ok = InetAddress.getByName( host ).isLoopbackAddress();
						
					}catch( Throwable e ){
						
						logger.log( e );
					}
				}
				
				if ( ok ){
					
					contentContainer	container = (contentContainer)cont;
	
					List	children = container.getChildren();
					
					int	added = 0;
					
					for (int i=0;i<children.size();i++){
						
						if ( start_index > 0 ){
							
							start_index--;
							
						}else{
							
							content	child = (content)children.get(i);
							
							didle += getDIDL( child, host ); 
							
							added++;
							
							if ( added == requested_count ){
								
								break;
							}
						}
					}
				
					num_returned	= added;
					total_matches	= children.size();
					
				}else{
					
					num_returned	= 0;
					total_matches	= 0;
				}
			}
		}
		
			// always return the DIDL element as if we don't this borks PS3 for empty folders
		//if ( didle.length() > 0 ){
			
			didle = 
				"<DIDL-Lite xmlns:dc=\"http://purl.org/dc/elements/1.1/\" " + 
				"xmlns:upnp=\"urn:schemas-upnp-org:metadata-1-0/upnp/\" " +
				"xmlns=\"urn:schemas-upnp-org:metadata-1-0/DIDL-Lite/\">" +
				
				didle + 
				
				"</DIDL-Lite>";
		//}
		
		result.add( "Result", didle );
		
		result.add( "NumberReturned", "" + num_returned );
		result.add( "TotalMatches", "" + total_matches );
		result.add( "UpdateID", "" + update_id  );
		
		return( num_returned );
	}
	
	protected String
	getDIDL(
		content	con,
		String	host )
	{
		if ( con instanceof contentContainer ){
			
			contentContainer	child_container = (contentContainer)con;
			
			return(
				"<container id=\"" + child_container.getID() + "\" parentID=\"" + child_container.getParentID() + "\" childCount=\"" + child_container.getChildren().size() + "\" restricted=\"false\" searchable=\"true\">" +
				
					"<dc:title>" + escapeXML(child_container.getName()) + "</dc:title>" +
					"<upnp:class>object.container.storageFolder</upnp:class>" +
					"<upnp:storageUsed>" + child_container.getStorageUsed() + "</upnp:storageUsed>" +
					"<upnp:writeStatus>WRITABLE</upnp:writeStatus>" +
	
				"</container>" );
			
		
		}else{
			contentItem	child_item = (contentItem)con;
			
			return( 
				"<item id=\"" + child_item.getID() + "\" parentID=\"" + child_item.getParentID() + "\" restricted=\"false\">" +
					child_item.getDIDL( host ) + 
				"</item>" );
		}
	}
	
	
	protected String
	processEvent(
		boolean				content_directory,
		InetSocketAddress	client,
		Map					headers,
		boolean				subscribe,
		String				callback,
		String				sid,
		String				timeout )
	{
		// System.out.println( "ProcessEvent:" + client + "," + subscribe + "," + callback + "," + sid + "," + timeout );
		
		if ( subscribe ){
			
			/*
			 * HTTP/1.1 200 OK
				DATE: when response was generated
				SERVER: OS/version UPnP/1.0 product/version
				SID: uuid:subscription-UUID
				TIMEOUT: Second-actual subscription duration
			 */
			
			if ( sid == null ){
				
				sid = "uuid:" + UUIDGenerator.generateUUIDString();
			
				event ev = new event( client, callback, sid, content_directory );
				
				synchronized( events ){
				
					events.put( sid, ev );
					
					new_events.add( ev );
				}
				
				logger.log( "Event: Subscribe - " + ev.getString());
				
			}else{
					// renew
				
				synchronized( events ){
					
					event	ev = (event)events.get(sid);
					
					if ( ev == null ){
						
						System.out.println( "can't renew event subscription for " + sid + ", not found" );
						
					}else{
						
						ev.renew();
						
						logger.log( "Event: Renew - " + ev.getString());
					}
				}
			}
		

			headers.put( "DATE", TimeFormatter.getHTTPDate( SystemTime.getCurrentTime()));
			headers.put( "SID", sid );
			headers.put( "TIMEOUT", "Second-" + EVENT_TIMEOUT_SECONDS );
			
		}else{
			
			if ( sid != null ){
				
				synchronized( events ){
					
					event	ev = (event)events.remove( sid );
					
					logger.log( "Event: Unsubscribe - " + ev.getString());
				}
			}
		}
		
		return( "" );
	}
	
	protected void
	stop()
	{
		sendDead();
	}
	
	
	protected void
	play(Object obj) 
	
		throws IPCException
	{
		Download				download;
		DiskManagerFileInfo		file;
		
		if ( media_server_ui == null ){
			
			throw( new IPCException( "Media server UI not bound" ));
		}
		
		if ( obj instanceof Download ){
		
			download = (Download)obj;

			file	= download.getDiskManagerFileInfo()[0];
		
		}else{
			
			file = (DiskManagerFileInfo)obj;
			
			try{
				download	= file.getDownload();
				
			}catch( DownloadException e ){	
				
				throw( new IPCException( "Failed to get download from file", e ));
			}
		}
		
		String	id = createResourceID( download, file );
		
		System.out.println( "play: " + id );

		contentItem item = getContentFromResourceID( id );
		
		if ( item == null ){
			
			throw( new IPCException( "Failed to find item for id '" + id + "'" ));
		}
		
		String url = item.getURI( "127.0.0.1", -1 );
		
		/* no point in this - it'll match the "." in the host name...
		int lastDot = url.lastIndexOf(".");
		
		if ( lastDot == -1 ){
			
			throw( new IPCException( "For an item to be playable it has to have a file extension (" + url + ")" ));
		}
		*/
		
			//String extension = url.substring(lastDot);
			//Program program = Program.findProgram(".asx");
    
		try {
			String user_dir = plugin_interface.getUtilities().getAzureusUserDir();
       
			File playList;
			String playCode;
			
			boolean isQuickTime = item.content_type.equals("video/quicktime");
			boolean forceWMP = false;
			if ((plugin_interface.getUtilities().isOSX() && (item.content_type.equals("video/mpeg")
					|| isQuickTime || item.content_type.equals("video/x-ms-wmv")))
					|| (isQuickTime && quickTimeAvail)) {
				
				File playLists = new File(user_dir, "playlists");
				if(!playLists.exists()) playLists.mkdir();
				String fileName = "azplay" + (SystemTime.getCurrentTime() / 1000) + ".qtl";
				playList = new File( playLists, fileName);
				
				playCode = "<?xml version=\"1.0\"?>" +
						"<?quicktime type=\"application/x-quicktime-media-link\"?>" +
						"<embed autoplay=\"true\" fullscreen=\"full\" moviename=\"" + item.title + "\" " +
						"src=\"" + url + "\" />";
			} else {
				playList = new File( user_dir, "azplay.asx");
				playCode = "<ASX version = \"3.0\">" +
				"<Entry>" +
				"<Ref href = \"" + url + "\" />" +
				"</Entry>" +
				"</ASX>";
				forceWMP = Constants.isWindows;
			}
			
			BufferedWriter bw = new BufferedWriter(new FileWriter(playList));
			
			
			bw.write(playCode);
			
			bw.close();
			
			//program.execute(playList.getAbsolutePath());

			if (forceWMP) {
				media_server_ui.runInMediaPlayer( playList.getAbsolutePath(), true );
			} else {
				media_server_ui.play( playList );
			}
			
		} catch(Exception e) {
			
			throw( new IPCException( "Play operation failed", e ));
		}							
	}
	
	protected void
	sendAlive()
	{
		synchronized( this ){
			
			if ( alive ){
		
				sendNotify( "ssdp:alive" );
			}
		}
	}
	
	protected void
	sendDead()
	{
		synchronized( this ){

			alive	= false;
			
			sendNotify( "ssdp:byebye" );
		}
	}
	
	protected void
	sendNotify(
		String	status )
	{
		if ( ssdp != null ){
			
			for (int i=0;i<upnp_entities.length;i++){
				
				ssdp.notify( upnp_entities[i], status, upnp_entities[i]==UUID_rootdevice?null:UUID_rootdevice, "RootDevice.xml" );
			}
		}
	}
	
	protected void
	contentContainerUpdated(
		contentContainer		c )
	{
		synchronized( events ){
			
			if ( container_update_events == null ){
				
				container_update_events	= new HashSet();
			}
			
			container_update_events.add( c );
		}
	}
	
	protected void
	sendEvents(
		boolean	check_timeouts )
	{
		long	now = plugin_interface.getUtilities().getCurrentSystemTime();
		
		List	new_dir_targets = new ArrayList();
		List	new_con_targets = new ArrayList();

		synchronized( events ){
			
			Iterator	it = new_events.iterator();
			
			while( it.hasNext()){
				
				event	ev = (event)it.next();
				
				if ( ev.getStartTime() > now || now - ev.getStartTime() > 1000 ){
					
					it.remove();
										
					if ( ev.isContentDirectory()){
						
						new_dir_targets.add( ev );
						
					}else{
						
						new_con_targets.add( ev );
					}
				}
			}
			
			if ( check_timeouts ){
				
				it = events.values().iterator();
				
				while( it.hasNext()){
					
					event ev = (event)it.next();
					
					long	t = ev.getStartTime();
					
					if ( t > now ){
						
						ev.renew();
						
					}else if ( now - t > EVENT_TIMEOUT_SECONDS*1000 + 10*60*1000 ){
						
						logger.log( "Event: Timeout - " + ev.getString());
						
						it.remove();
					}
				}
			}
		}
		
		if ( new_dir_targets.size() > 0 ){

			sendDirEvents( new_dir_targets, ""+root_container.getUpdateID(), "", "" );
		}
		
		if ( new_con_targets.size() > 0 ){

			sendConEvents( new_con_targets );
		}
		
		Set		to_send;
		List	targets;
		
		synchronized( events ){
			
			to_send = container_update_events;
		
			container_update_events	= null;
			
			if ( to_send == null || to_send.size() == 0 || events.size() == 0 ){
				
				return;
			}
			
			targets = new ArrayList( events.values());
		}
	
			// currently only really support content directory events
		
		Iterator	it = targets.iterator();
		
		while( it.hasNext()){
			
			event	ev = (event)it.next();
			
			if ( !ev.isContentDirectory()){
				
				it.remove();
			}
		}
		
		if ( targets.size() == 0 ){
			
			return;
		}
		
		it = to_send.iterator();
		
		String	system_update 		= null;
		String	container_update	= null;
			
		while( it.hasNext()){
			
			contentContainer	container = (contentContainer)it.next();
			
			int	id 			= container.getID();
			int	update_id	= container.getUpdateID();
			
			if ( container.getParent() == null ){
				
				system_update = "" + update_id;
			}
			
			if ( container_update == null ){
				container_update = "";
			}
			
			container_update += (container_update.length()==0?"":",") + id + "," + update_id;
		}
		
		sendDirEvents( targets, system_update, container_update, null );
	}
	
	protected void
	sendDirEvents(
		List	targets,
		String	system_update,
		String	container_update,
		String	transfer_ids )
	{
		String	xml =
			"<e:propertyset xmlns:e=\"urn:schemas-upnp-org:event-1-0\">";
		
		if ( system_update != null ){
			xml += 
				"<e:property>"+
					"<SystemUpdateID>" + system_update + "</SystemUpdateID>" +
				"</e:property>";
		}
		
		if ( container_update != null ){
			xml += 
				"<e:property>"+
					"<ContainerUpdateIDs>" + container_update + "</ContainerUpdateIDs>" +
				"</e:property>";
		}
		
		if ( transfer_ids != null ){
			xml += 
				"<e:property>"+
					"<TransferIDs>" + "" + "</TransferIDs>" +
				"</e:property>";
		}
		
		xml += "</e:propertyset>";
		  	
		try{
			byte[]	xml_bytes = xml.getBytes( "UTF-8" );
			
			Iterator it = targets.iterator();
			
			while( it.hasNext()){
				
				event	ev = (event)it.next();
				
				ev.sendEvent( xml_bytes );
			}	
		}catch( Throwable e ){
			
			Debug.printStackTrace(e);
		}
	}
	
	protected void
	sendConEvents(
		List	targets )
	{
		String	xml =
			"<e:propertyset xmlns:e=\"urn:schemas-upnp-org:event-1-0\">";
		
		xml += 
			"<e:property>"+
				"<SourceProtocolInfo>" + CM_SOURCE_PROTOCOLS + "</SourceProtocolInfo>" +
			"</e:property>";
		
		xml += 
			"<e:property>"+
				"<SinkProtocolInfo>" + CM_SINK_PROTOCOLS + "</SinkProtocolInfo>" +
			"</e:property>";
	
		xml += 
			"<e:property>"+
				"<CurrentConnectionIDs>" + "" + "</CurrentConnectionIDs>" +
			"</e:property>";

		
		xml += "</e:propertyset>";
		  	
		try{
			byte[]	xml_bytes = xml.getBytes( "UTF-8" );
			
			Iterator it = targets.iterator();
			
			while( it.hasNext()){
				
				event	ev = (event)it.next();
				
				ev.sendEvent( xml_bytes );
			}	
		}catch( Throwable e ){
			
			Debug.printStackTrace(e);
		}
	}
	
	protected String
	escapeXML(
		String	str )
	{
		if ( str == null ){
			
			return( "" );
			
		}
		str = str.replaceAll( "&", "&amp;" );
		str = str.replaceAll( ">", "&gt;" );
		str = str.replaceAll( "<", "&lt;" );
		// str = str.replaceAll( "\"", "&quot;" );	breaks Philips streamium...
		
		return( str );
	}
	
	protected abstract class
	content
		implements Cloneable
	{
		private int						id;
		private contentContainer		parent;
		
		protected
		content()
		{
			id	= next_oid++;
			
			if ( next_oid == 1 ){
				
				next_oid = (int)( SystemTime.getCurrentTime()/1000);
			}
			
			synchronized( content_map ){
			
				content_map.put( new Integer( id ), this );
			}
		}
		
		protected int
		getID()
		{
			return( id );
		}
		
		protected int
		getParentID()
		{
			if ( parent == null ){
				
				return( -1 );
				
			}else{
				
				return( parent.getID());
			}
		}
		protected void
		setParent(
			contentContainer	_parent )
		{
			parent	= _parent;
		}
		
		protected contentContainer
		getParent()
		{
			return( parent );
		}

		protected abstract content
		getCopy();

		protected abstract String
		getName();
		
		protected void
		deleted()
		{
			synchronized( content_map ){
				
				content_map.remove( new Integer( id ));
			}
		}
		
		protected abstract long
		getStorageUsed();
		
		protected abstract void
		print(
			String	indent );
	}
	
	protected class
	contentContainer
		extends content
	{
		private String		name;
		private List		children 	= new ArrayList();
		private int			update_id	= 0;
		
		protected
		contentContainer(
			String		_name )
		{
			name	= _name;
		}
		
		protected content
		getCopy()
		{
			contentContainer copy = new contentContainer( name );
			
			for (int i=0;i<children.size();i++){
				
				copy.addChild(((content)children.get(i)).getCopy());
			}
			
			return( copy );
		}
		
		protected void
		addLink(
			content		child )
		{
			//logger.log( "Container: adding link '" + child.getName() + "' to '" + getName() + "'" );
				
			child = child.getCopy();
			
			child.setParent( this );
			
			children.add( child );
			
			updated();
		}
		
		protected content
		removeLink(
			String	child_name )
		{
			//logger.log( "Container: removing link '" + child_name + "' from '" + getName() + "'" );

			Iterator	it = children.iterator();
				
			while( it.hasNext()){
				
				content	con = (content)it.next();
				
				String	c_name = con.getName();
				
				if ( c_name.equals( child_name )){
						
					it.remove();
						
					updated();
						
					con.deleted();
					
					return( con );
				}
			}
			
			return( null );
		}
		
		protected void
		addChild(
			content		child )
		{
			//logger.log( "Container: adding child '" + child.getName() + "' to '" + getName() + "'" );
			
			child.setParent( this );
			
			children.add( child );
			
			updated();
		}
		
		protected content
		removeChild(
			String	child_name )
		{
			//logger.log( "Container: removing child '" + child_name + "' from '" + getName() + "'" );

			Iterator	it = children.iterator();
				
			while( it.hasNext()){
				
				content	con = (content)it.next();
				
				String	c_name = con.getName();
				
				if ( c_name.equals( child_name )){
						
					it.remove();
						
					updated();
						
					con.deleted();
					
					return( con );
				}
			}
			
			logger.log( "    child not found" );
			
			return( null );
		}
		
		protected content
		getChild(
			String	child_name )
		{
			Iterator	it = children.iterator();
			
			while( it.hasNext()){
				
				content	con = (content)it.next();
				
				String	c_name = con.getName();
				
				if ( c_name.equals( child_name )){

					return( con );
				}
			}
			
			return( null );
		}
		
		protected void
		updated()
		{
			update_id++;
		
			contentContainerUpdated( this );

			if ( getParent() != null ){
					
				getParent().updated();
				
			}else{
				
				system_update_id++;
			}
		}
		
		protected void
		deleted()
		{
			super.deleted();
			
			Iterator	it = children.iterator();
			
			while( it.hasNext()){

				((content)it.next()).deleted();
			}
		}
		
		protected String
		getName()
		{
			return( name );
		}
		
		protected List
		getChildren()
		{
			return( children );
		}
		
		protected int
		getUpdateID()
		{
			return( update_id );
		}
		
		protected long
		getStorageUsed()
		{
			long	res = 0;
			
			Iterator	it = children.iterator();
			
			while( it.hasNext()){

				content	con = (content)it.next();
				
				res += con.getStorageUsed();
			}
			
			return( res );
		}
		
		protected void
		print(
			String	indent )
		{
			log( indent + name + ", id=" + getID());
			
			indent += "    ";
			
			Iterator	it = children.iterator();
			
			while( it.hasNext()){

				content	con = (content)it.next();

				con.print( indent );
			}
		}
	}
	
	protected class
	contentItem
		extends content
		implements Cloneable
	{
		private Download				download;
		private DiskManagerFileInfo		file;
		
		private boolean		valid;
		private String		title;
		private String		creator;
		private String		duration;
		private String		content_type;
		private String		item_class;
		
		protected 
		contentItem(
			Download			_download,
			DiskManagerFileInfo	_file,
			String				_title )
		{
			download	= _download;
			file		= _file;
			title		= _title;
						
			String	file_name = file.getFile().getName();
						
				// TODO: ID tag extraction
			
			creator			= "Unknown";
			duration		= "0:00:00";
			
			int	pos = file_name.lastIndexOf('.');
			
			if ( pos != -1 && !file_name.endsWith( "." )){
				
				String[]	entry = (String[])ext_lookup_map.get( file_name.substring( pos+1 ).toLowerCase());
			
				if ( entry != null ){
					
					content_type	= entry[1];
					item_class		= entry[2];
					
					valid	= true;
				}
			}
			
			if ( !valid ){
				
				content_type	= "unknown/unknown";
				item_class		= CONTENT_UNKNOWN;
			}
		}
				
		protected content
		getCopy()
		{
			try{
				return((content)clone());
				
			}catch( Throwable e ){
				
				e.printStackTrace();
				
				return( null );
			}
		}
		
		protected DiskManagerFileInfo
		getFile()
		{
			return( file );
		}
		
		protected String
		getTitle()
		{
			return( title );
		}
		
		protected String
		getCreator()
		{
			return( creator );
		}
		
		protected String
		getProtocolInfo()
		{
			return( "http-get:*:" + content_type + ":*" );
		}
		
		protected String
		getURI(
			String	host,
			int		stream_id )
		{
			return( "http://" + host + ":" +content_server.getPort() + "/Content/" + createResourceID( download, file ) + (stream_id==-1?"":("?sid=" + stream_id ))); 
		}
		
		protected String
		getResource(
			String		host )
		{
			LinkedHashMap	attributes = new LinkedHashMap();
			
			attributes.put( "protocolInfo", getProtocolInfo());
			attributes.put( "size", String.valueOf( file.getLength()));
			attributes.put( "duration", duration );
			
			String	resource = "<res ";
			
			Iterator	it = attributes.keySet().iterator();
			
			while( it.hasNext()){
				
				String	key = (String)it.next();
				
				resource += key + "=\"" + attributes.get(key) + "\"" + (it.hasNext()?" ":"");
			}
			
			resource += ">" + getURI( host, -1 ) + "</res>";
			
			return( resource );
		}
		
		protected String
		getDIDL(
			String	host )
		{
				// for audio: dc:date 2005-11-07
				//			upnp:genre Rock/Pop
				//			upnp:artist
				// 			upnp:album
				//			upnp:originalTrackNumber
				
			String	didle = 
				"<dc:title>" + escapeXML( getTitle()) + "</dc:title>" +
				"<dc:creator>" +  escapeXML(getCreator()) + "</dc:creator>" +
				"<upnp:class>" + item_class + "</upnp:class>" +
				getResource( host );
			
			return( didle );
		}
		
		protected String
		getName()
		{
			return( title );
		}
		
		protected String
		getContentClass()
		{
			return( item_class );
		}
		
		protected String
		getContentType()
		{
			return( content_type );
		}
		
		protected void
		deleted()
		{
			super.deleted();
		}
		
		protected long
		getStorageUsed()
		{
			return( file.getLength());
		}
		
		protected void
		print(
			String	indent )
		{
			log( indent + title + ", id=" + getID() + ", class=" + item_class + ", type=" + content_type );
		}
	}
	
	protected class
	resultEntries
	{
		private List		names	= new ArrayList();
		private List		values	= new ArrayList();
		
		public void
		add(
			String		name,
			String		value )
		{
			names.add( name );
			values.add( escapeXML( value ));
		}
		
		public List
		getArgs()
		{
			return( names );
		}
		
		public List
		getValues()
		{
			return( values );
		}
	}
	
	protected class
	event
	{
		private static final String	NL			= "\015\012";
		
		private boolean content_directory;
		private String	sid;
		private long	create_renew_time;
		private long	event_seq;
		
		private List	callbacks	= new ArrayList();
		
		protected
		event(
			InetSocketAddress	address,
			String				callback,
			String				_sid,
			boolean				_content_directory )
		{
			sid					= _sid;
			content_directory	= _content_directory;
			
			create_renew_time	= plugin_interface.getUtilities().getCurrentSystemTime();
			
			StringTokenizer	tok = new StringTokenizer( callback, ">" );
			
			while( tok.hasMoreTokens()){
				
				String	c = tok.nextToken().trim().substring(1);
				
				if ( !c.toLowerCase().startsWith( "http" )){
					
					if ( c.startsWith( "/" )){
						
						c = c.substring(1);
					}
					
					c = "http://" + address.getAddress().getHostAddress() + ":" + address.getPort() + "/" + c;				
				}
				
				try{
					URL	url = new URL( c );
					
					callbacks.add( url );
					
				}catch( Throwable e ){
					
					Debug.printStackTrace(e);
				}
			}
		}
		
		protected boolean
		isContentDirectory()
		{
			return( content_directory );
		}
		
		protected void
		renew()
		{
			create_renew_time	= plugin_interface.getUtilities().getCurrentSystemTime();
		}
		
		protected long
		getStartTime()
		{
			return( create_renew_time );
		}
		
		protected String
		getString()
		{	
			String	cb = "";
			
			for (int i=0;i<callbacks.size();i++){
				
				cb += (i==0?"":",") + callbacks.get(i);
			}
			
			return( "sid=" + sid + ",callbacks=" + cb + ",seq=" + event_seq + ",cd=" + content_directory );
		}
		
		protected void
		sendEvent(
			byte[]	bytes )
		{
			/*
			NOTIFY delivery path HTTP/1.1
			HOST: delivery host:delivery port
			CONTENT-TYPE: text/xml
			CONTENT-LENGTH: Bytes in body
			NT: upnp:event
			NTS: upnp:propchange
			SID: uuid:subscription-UUID
			SEQ: event key
			*/
			
			for (int i=0;i<callbacks.size();i++){
				
				URL	url = (URL)callbacks.get(i);
				
				Socket	socket	= null;
				
				try{
					
					socket = new Socket();
					
					socket.setSoTimeout( 10000 );
					
					socket.connect( 
							new InetSocketAddress( 
									url.getHost(), 
									url.getPort()==-1?url.getDefaultPort():url.getPort()),
							10000 );
							
					OutputStream	os = socket.getOutputStream();
					
					write( os, "NOTIFY " + URLEncoder.encode(url.getPath(), "UTF-8") + " HTTP/1.1" + NL );
					write( os, "HOST: " + url.getHost() + ":" + socket.getPort() + NL );
					write( os, "CONTENT-TYPE: text/xml" + NL );
					write( os, "CONTENT-LENGTH: " + bytes.length + NL );
					write( os, "NT: upnp:event" + NL );
					write( os, "NTS: upnp:propchange" + NL );
					write( os, "SID: " + sid + NL );
					write( os, "SEQ: " + event_seq + NL + NL );
					
					os.write( bytes );
					
					os.flush();
					
					InputStream	is = socket.getInputStream();
					
					is.read();
					
					logger.log( "Event: sent to  " + url );
					
				}catch( Throwable e ){
					
					// e.printStackTrace();
					
				}finally{
					
					try{
						if ( socket != null ){
							
							socket.close();
						}
					}catch( Throwable e ){
					
					}
				}
			}
		
				// first event must be 0
			
			event_seq++;
		}
		
		protected void
		write(
			OutputStream	os,
			String			str )
		
			throws IOException
		{
			os.write( str.getBytes( "UTF-8" ));
		}
	}
}
