/*
 * Created on 29-Mar-2006
 * Created by Paul Gardner
 * Copyright (C) 2006 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SAS au capital de 46,603.30 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package com.aelitis.azureus.plugins.upnpmediaserver;

import java.net.*;
import java.nio.ByteBuffer;
import java.nio.channels.ServerSocketChannel;
import java.io.*;
import java.util.*;

import org.gudy.azureus2.core3.util.AEThread2;
import org.gudy.azureus2.core3.util.Average;
import org.gudy.azureus2.core3.util.Base32;
import org.gudy.azureus2.core3.util.Debug;
import org.gudy.azureus2.core3.util.ThreadPool;
import org.gudy.azureus2.core3.util.ThreadPoolTask;
import org.gudy.azureus2.plugins.PluginInterface;
import org.gudy.azureus2.plugins.disk.DiskManagerChannel;
import org.gudy.azureus2.plugins.disk.DiskManagerEvent;
import org.gudy.azureus2.plugins.disk.DiskManagerFileInfo;
import org.gudy.azureus2.plugins.disk.DiskManagerListener;
import org.gudy.azureus2.plugins.disk.DiskManagerRequest;
import org.gudy.azureus2.plugins.utils.PooledByteBuffer;


public class 
UPnPMediaServerContentServer 
{
	private static final String	NL			= "\r\n";

	private static final int	MAX_CONNECTIONS_PER_ENDPOINT	= 16;
	
	private UPnPMediaServer	plugin;
	private int				port;
	
	private ThreadPool		thread_pool;
	private PluginInterface	plugin_interface;
	
	private List		close_queue	= new ArrayList();
	
	private List		active_processors	= new ArrayList();
	
	private Map			stream_map	= new HashMap();
	
	private ServerSocket	server_socket;
	
	private volatile boolean	destroyed;
	
	protected
	UPnPMediaServerContentServer(
		UPnPMediaServer		_plugin )
	
		throws IOException
	{
		plugin	= _plugin;
		
		plugin_interface = plugin.getPluginInterface();
		
		thread_pool = new ThreadPool( "UPnPMediaServer:processor", 64 );
		
		Random	random = new Random();
		
		port	= plugin.getContentPort();
		
		ServerSocketChannel	ssc	= null;
		
		if ( port == 0 ){
			
			port = random.nextInt(20000) + 40000;
		}
		
		boolean	warned = false;
		
		for (int i=0;i<1024;i++){
			
			try{
				ssc = ServerSocketChannel.open();

				ssc.socket().bind( new InetSocketAddress(port), 1024 );

				break;
				
			}catch( Throwable e ){
				
				if ( ssc != null ){
					
					try{
						ssc.close();
						
					}catch( Throwable f ){
						
						Debug.printStackTrace(e);
					}
					
					ssc = null;
				}
				
				if ( plugin.isUserSelectedContentPort()){
					
					if ( !warned ){
						
						plugin.logAlert( "Unable to bind to user selected stream port " + port + "; reverting to random port");
						
						warned = true;
					}
				}
				
				port = random.nextInt(20000) + 40000;
			}
		}
			
		if ( ssc == null ){
			
			ssc = ServerSocketChannel.open();

			ssc.socket().bind( new InetSocketAddress(0), 1024 );
			
			port = ssc.socket().getLocalPort();
		}
		
		ServerSocket ss	= ssc.socket();
		
		ss.setReuseAddress(true);
		
		plugin.setContentPort( port );
		
		server_socket = ss;
		
		new AEThread2( "UPnPMediaServer:accepter", true )
			{
				public void
				run()
				{
					int	processor_num = 0;
					
					try{

						
						long	successfull_accepts = 0;
						long	failed_accepts		= 0;
						
						while( !destroyed ){
							
							try{				
								Socket socket = server_socket.accept();
									
								successfull_accepts++;
								
								String	ip = socket.getInetAddress().getHostAddress();			
									
									// TODO: permission checks???
								
								processor	proc = new processor( ip, socket, processor_num++ );
								
								thread_pool.run( proc );
									
							}catch( Throwable e ){
								
								if ( !destroyed ){
									
									failed_accepts++;
									
									plugin.log( "listener failed on port " + getPort(), e ); 
									
									if ( failed_accepts > 100 && successfull_accepts == 0 ){
				
											// looks like its not going to work...
											// some kind of socket problem
				
										plugin.log( "    too many listen fails, giving up" );
								
										break;
									}
								}
							}
						}
					}catch( Throwable e ){
					}
				}
			}.start();
			
		new AEThread2( "UPnPMediaServer:closer", true )
		{
			public void
			run()
			{
				List	pending	= new ArrayList();
				
				while( !( destroyed && active_processors.size() == 0 )){
					
					try{
						Thread.sleep(10*1000);
					
					}catch( Throwable e ){
						
						e.printStackTrace();
						
						break;
					}
					
					Iterator	it = pending.iterator();
					
					while( it.hasNext()){
					
						try{
							
							((UPnPMediaChannel)it.next()).close();
							
						}catch( Throwable e ){
							
							it.remove();
						}
					}
					
					synchronized( close_queue ){
						
						pending.addAll( close_queue );
						
						close_queue.clear();
					}
					
					synchronized( active_processors ){
						
						Map	conn_map = new HashMap();
						
						for (int i=0;i<active_processors.size();i++){
							
							processor	proc = (processor)active_processors.get(i);
															
							DiskManagerRequest	req = proc.getActiveRequest();
								
							if ( req != null ){
									
								UPnPMediaChannel	channel = proc.getChannel();
														
								if ( channel.isClosed()){

									// System.out.println( "Cancelling active request on closed socket" );
								
									req.cancel();
									
								}else{
									
									List	conns = (List)conn_map.get( proc.getIP());
									
									if ( conns == null ){
										
										conns = new ArrayList();
										
										conn_map.put( proc.getIP(), conns );
									}
									
									conns.add( proc );
								}
							}
						}
						
							// some devices don't close down connections properly and we end up with a
							// load of CLOSE_WAIT sockets. Limit the number of open sockets per end point
							// to put an upper limit on this
						
						it = conn_map.values().iterator();
						
						while( it.hasNext()){
						
							List	conns = (List)it.next();
							
							for (int i=0;i<conns.size()-MAX_CONNECTIONS_PER_ENDPOINT;i++){
								
								processor	proc = (processor)conns.get(i);
								
								DiskManagerRequest	req = proc.getActiveRequest();
								
								if ( req != null ){
		
									// System.out.println( "Cancelling active request - client has too many open connections" );
									
									req.cancel();
								}
							}
						}
					}
				}
			}
		}.start();
	}
	
	protected void
	destroy()
	{
		destroyed	= true;
		
		try{
			server_socket.close();
			
		}catch( Throwable e ){
			
		}
	}
	
	protected streamInfo
	getStreamInfo(
		int		stream_id )
	{
		synchronized( active_processors ){
				
			return((processor)stream_map.get( new Integer( stream_id )));
		}
	}
	
	protected int
	getPort()
	{
		return( port );
	}
	
	protected class
	processor
		extends ThreadPoolTask
		implements streamInfo
	{
		private String				ip;
		private Socket				socket;
		private UPnPMediaChannel	channel;
		
		private int			processor_num;
		
		private long		last_write_time;
		private long		last_write_offset;
		
		private long		last_blocked_offset;
		
		private int			stream_id	= -1;
		
		private volatile DiskManagerRequest	active_request;
		
		protected
		processor(
			String					_ip,
			Socket					_socket,
			int						_processor_num )
		{			
			ip				= _ip;
			socket			= _socket;
			processor_num	= _processor_num;
			
			last_write_time	= plugin_interface.getUtilities().getCurrentSystemTime();
		}

		protected String
		getIP()
		{
			return( ip );
		}
		
		protected long
		getLastWriteTime()
		{
			return( last_write_time );
		}
		
		protected long
		getLastWriteOffset()
		{
			return( last_write_offset );
		}
		
		public long
		getPosition()
		{
			return( getLastWriteOffset());
		}
		
		public long
		getAvailableBytes()
		{
			DiskManagerRequest request = active_request;
			
			if ( request == null ){
				
				return( -1 );
			}
			
			return( request.getAvailableBytes());
		}
		
		public long
		getRemaining()
		{
			DiskManagerRequest request = active_request;
			
			if ( request == null ){
				
				return( -1 );
			}
			
			return( request.getRemaining());
		}
		
		protected int
		getStreamID()
		{
			return( stream_id );
		}
		
		protected UPnPMediaChannel
		getChannel()
		{
			return( channel );
		}
		
		protected DiskManagerRequest
		getActiveRequest()
		{
			return( active_request );
		}
		
		protected void
		log(
			String		str )
		{
			plugin.log( "[" + processor_num + "] " + str );
		}
		
		public void
		runSupport()
		{
			boolean	close_now	= false;
		
			try{
		
				synchronized( active_processors ){
					
					active_processors.add( this );
				}
			
				// System.out.println( "Processor " + processor_num  + " starts" );
				
				setTaskState( "entry" );
				
				channel			= new UPnPMediaChannel( socket );
				
				process();

				//System.out.println( "closing media server channel now" );
				// Mplayer on OSX needs the stream to be closed when complete otherwise we
				// get a hang at end of content
				
				Thread.sleep(100);
				
				close_now = true;
				
			}catch( Throwable e ){
		
				close_now	= true;
				
				if ( ! (e instanceof SocketTimeoutException )){
					
					e.printStackTrace();
				}
				
			}finally{
				
				synchronized( active_processors ){
					
					active_processors.remove( this );
					
					if ( stream_id != -1 ){
						
						if ( getStreamInfo( stream_id ) == this ){
							
							stream_map.remove( new Integer( stream_id ));
						}
					}
				}
				
				// System.out.println( "Processor " + processor_num  + " ends" );
				
				if ( close_now ){
				
					try{
						channel.close();
					
					}catch( Throwable f ){
					}
				}else{
				
					synchronized( close_queue ){
						
						close_queue.add( channel );
					}
				}
			}
		}
		
		protected void
		process()
		
			throws IOException
		{
			int	loop_count = 0;
			
			while( true ){
			
				String	command	= null;
				Map		headers	= new HashMap();

				while( true ){
	
					String	line = "";
				
					while( !line.endsWith( NL )){
						
						byte[]	buffer = new byte[1];
						
						channel.read( buffer );
						
						line += new String( buffer );
					}
				
					line = line.trim();
					
					if ( line.length() == 0 ){
						
						break;
					}
					
					if ( command == null ){
						
						command	= line;
						
					}else{
						
						int	pos = line.indexOf(':');
						
						if ( pos == -1 ){
							
							return;
						}
						
						String	lhs = line.substring(0,pos).trim().toLowerCase();
						String	rhs = line.substring(pos+1).trim();
						
						headers.put( lhs, rhs );
					}
				}
				
				// System.out.println( "command: " + command );

				String	url;
				boolean	head	= false;
				
				if ( command.startsWith( "GET " )){
					
					url = command.substring( 4 );

				}else if ( command.startsWith( "HEAD " )){
					
					url = command.substring( 5 );
					
					head	= true;
					
				}else{
					
					log( "Unhandled HTTP request: " + command );
					
					return;
				}
				
				
				int	pos = url.indexOf( ' ' );
				
				if ( pos == -1 ){
					
					return;
				}
				
				String	http_version = "HTTP/1.1";	// always return this url.substring( pos ).trim();
								
				url = URLDecoder.decode( url.substring(0,pos), "ISO8859-1" );
				
				UPnPMediaServer.contentItem	item			= null;
				
				if ( url.startsWith( "/Platform" )){
										
					int	q_pos = url.indexOf('?');
					
					String	content_id = null;
					
					if ( q_pos != -1 ){

						StringTokenizer	tok = new StringTokenizer( url.substring( q_pos+1 ), "&" );
						
						while( tok.hasMoreTokens()){
							
							String	token = tok.nextToken();
							
							int	e_pos = token.indexOf('=');
							
							if ( e_pos != -1 ){
							
								String	lhs = token.substring( 0, e_pos );
								String	rhs = token.substring( e_pos+1 );
								
								if ( lhs.equals( "cid" )){
									
									content_id = rhs;
									
									break;
								}
							}
						}
					}
					
					if ( content_id != null ){
						
						byte[]	hash = Base32.decode( content_id );
						
						item = plugin.getContentFromHash( hash );
					}
				}else if ( url.startsWith( "/Content/" )){
										
					String	content = url.substring( 9 );
					
					int	q_pos = content.indexOf('?');
					
					if ( q_pos != -1 ){

						String	params = content.substring(q_pos+1);
						
						content = content.substring(0,q_pos);
						
						StringTokenizer tok = new StringTokenizer( params, "&" );
						
						while( tok.hasMoreTokens()){
							
							String	param = tok.nextToken();
							
							int	e_pos = param.indexOf('=');
							
							if ( e_pos != -1 ){
								
								String	lhs = param.substring(0,e_pos);
								String	rhs = param.substring(e_pos+1);
								
								if ( lhs.equals( "sid" )){
									
									try{
										stream_id = Integer.parseInt( rhs );
										
										synchronized( active_processors ){
											
											stream_map.put( new Integer( stream_id ), this );
										}
										
									}catch( Throwable  e){
									}
								}
							}
						}
					}
					
					item = plugin.getContentFromResourceID( content );
				}
				
				if ( item == null ){
							
					plugin.log( "Unknown content: " + url );
					
					write( http_version + " 404 Not Found" + NL + NL );

				}else{
					
					if ( head ){
						
						long	content_len = item.getStorageUsed();
					
						write( http_version + " 200 OK" + NL );
						write( "Server: Azureus Media Server 1.0" + NL ); 
						write( "Accept-Ranges: bytes" + NL );
						write( "Content-Length: " + content_len + NL );
						write( "Content-Range: 0-" + content_len + "/" + content_len + NL );
						write( "Content-Type: " + item.getContentType() + NL + NL );

					}else{
													
						try{
							if ( !process( http_version, headers, item )){
								
								return;
							}
						}catch( Throwable e ){
							
								// IOException not interesting as we get when stream closed
							
							if ( !( e instanceof IOException )){
							 
								e.printStackTrace();
							}
							
							return;
							
						}						
					}
				}

				if ( loop_count == 0 ){
					
					return;	// disable keep-alive
					
					/* 
					String	keep_alive = (String)headers.get( "connection" );
					
					if ( keep_alive == null || !keep_alive.equalsIgnoreCase( "Keep-Alive" )){
						
						return;
					}
					*/
				}
								
				loop_count++;
			}
		}
		
		protected boolean
		process(
			String							http_version,
			Map								headers,
			UPnPMediaServer.contentItem		content_item )
		
			throws Throwable
		{
			/*
				Iterator	it = headers.entrySet().iterator();
				
				while( it.hasNext()){
					
					Map.Entry	entry = (Map.Entry)it.next();
					
					System.out.println( "    "  + entry.getKey() + " -> "  + entry.getValue());
				}
			*/
			
			DiskManagerFileInfo	file = content_item.getFile();
			
			if ( !file.getFile().exists()){
			
				write( http_version + " 404 Not Found" + NL + NL );
				
				return( true );
			}
			
			DiskManagerChannel	disk_channel = file.createChannel();
			
			final long	piece_size = file.getDownload().getTorrent().getPieceSize();
			
			try{
				String	ranges = (String)headers.get( "range" );
				
				final DiskManagerRequest 	request = disk_channel.createRequest();

				request.setUserAgent((String)headers.get( "user-agent" ));
				
				if ( ranges == null ){
					
					log( "Streaming starts for " + content_item.getName() + "  [complete file]" );
					
					write( http_version + " 200 OK" + NL );
					write( "Server: Azureus Media Server 1.0" + NL ); 
					write( "Content-Type: " + content_item.getContentType() + NL );
					write( "Connection: close" + NL );
					write( "Accept-Ranges: bytes" + NL );
					write( "Content-Range: 0-" + file.getLength() + "/" + file.getLength() + NL );
					write( "Content-Length: " + file.getLength() + NL + NL );
								
					request.setType( DiskManagerRequest.REQUEST_READ );
					last_write_offset	= 0;
					request.setOffset( 0 );
					request.setLength( file.getLength());
					
				}else{
					
					ranges = ranges.toLowerCase();
					
					log( "Streaming starts for " + content_item.getName() + "[" + ranges + "]" );

					if ( !ranges.startsWith("bytes=")){
						
						throw( new IOException( "invalid range: " + ranges ));
					}
					
					ranges = ranges.substring( 6 );
					
					StringTokenizer	tok = new StringTokenizer( ranges, "," );
					
					if ( tok.countTokens() != 1 ){
						
						throw( new IOException( "invalid range - only single supported: " + ranges ));
					}
					
					String	range = tok.nextToken();
					
					int pos	= range.indexOf('-');
					
					long	start;
					long	end;
					
					long	length = file.getLength();
					
					if ( pos < range.length()-1 ){
						
						end = Long.parseLong( range.substring(pos+1));
						
					}else{
						end = length-1;
					}
					
					if ( pos > 0 ){
						
						start = Long.parseLong( range.substring(0,pos));
						
					}else{
							// -500 = last 500 bytes of file
						
						start 	= length-end;
						end		= length-1;
					}
	
					long	rem = ( end - start ) + 1;
					
						// prevent seeking too far
					
					if ( rem < 0 ){
						
						write( http_version + " 416 Requested range not satisfiable" + NL );

						return( true );
						
					}else{
					
						write( http_version + " 206 Partial content" + NL );
						write( "Content-Type: " + content_item.getContentType() + NL );
						write( "Server: Azureus Media Server 1.0" + NL ); 
						write( "Connection: close" + NL );
						write( "Content-Range: bytes " + start + "-" + end + "/" + file.getLength() + NL );
						write( "Content-Length: " + rem + NL + NL );
		
						request.setType( DiskManagerRequest.REQUEST_READ );
						last_write_offset	= start;
						request.setOffset( start );
						request.setLength( rem );
					}
				}
								
				final Throwable[]	error = { null };
				
				request.addListener(
					new DiskManagerListener()
					{
						private Average	write_speed = Average.getInstance(1000,10);
						private long	start_time	= plugin_interface.getUtilities().getCurrentSystemTime();
						private long	last_log	= start_time;
						
						private long	total_written;
						
						public void
						eventOccurred(
							DiskManagerEvent	event )
						{
							int	type = event.getType();
							
							if ( type == DiskManagerEvent.EVENT_TYPE_FAILED ){
								
								error[0]	= event.getFailure();
								
									// we need to close the channel here as we might be stuck trying to write 
									// to it below and thus without closing the channel we won't pick up
									// this error and terminate 
								
								channel.close();
								
							}else if ( type == DiskManagerEvent.EVENT_TYPE_SUCCESS ){
								
								PooledByteBuffer	buffer = null;
								
								//System.out.println( "[" + processor_num + "] writing at " + event.getOffset() + ", length " + event.getLength());

								try{
									buffer	= event.getBuffer();
										
									int	length = event.getLength();
									
									ByteBuffer	bb = buffer.toByteBuffer();
									
									bb.position( 0 );

									channel.write( buffer );
																		
									write_speed.addValue( length );
									
									total_written += length;
																		
									last_write_time		= plugin_interface.getUtilities().getCurrentSystemTime();
									last_write_offset	= event.getOffset();
									
									if ( last_write_time - last_log > 5000 ){
										
										// System.out.println( "[" + processor_num + "] write speed = " + write_speed.getAverage() + ", total = " + total_written );
											
										last_log = last_write_time;
									}
										
										// bit crap this - if this is a local renderer then limit speed for
										// the first second to prevent vlc from overbufferring
								
									if ( stream_id != -1 ){
										
										if ( last_write_time - start_time < 1000 ){
											
											Thread.sleep(100);
											
										}else{
											
											start_time = 0;	// prevent recur on clock change
										}
									}

									// System.out.println( "[" + processor_num + "] wrote " + event.getLength() + " bytes, total = " + total_written );
									
								}catch( Throwable e ){
									
									request.cancel();
									
									error[0] = e;
									
								}
							}else if ( type == DiskManagerEvent.EVENT_TYPE_BLOCKED ){
								
								//System.out.println( "[" + processor_num + "] blocked at " + event.getOffset());
								
								long	offset = event.getOffset();
								
								if ( offset != last_blocked_offset ){
									
									last_blocked_offset	= offset;
									
									long	piece_num 		= offset/piece_size;
									long	piece_offset	= offset - (piece_num * piece_size );
									
									log( "Blocked reading data at piece " + piece_num + ", offset " + piece_offset );
								}
							}
						}
					});
				
				int	priority = Thread.currentThread().getPriority();
				
				try{
					active_request 	= request;
				
					Thread.currentThread().setPriority( Thread.MAX_PRIORITY );
					
					request.run();

					channel.flush();
					
				}finally{
					
					Thread.currentThread().setPriority( priority );
					
					active_request	= null;
				}

				if ( error[0] != null ){
					
					throw( error[0] );
				}
				
			}finally{
				
				
				log( "Streaming ends for " + content_item.getName());

				disk_channel.destroy();
			}
			
			return( true );
		}
		
		protected void
		write(
			String			str )
		
			throws IOException
		{
			channel.write( str.getBytes());
		}
		
		public void
		interruptTask()
		{
		}
	}
	
	interface 
	streamInfo
	{
		public long
		getPosition();
		
		public long
		getAvailableBytes();
		
		public long
		getRemaining();
	}
}
