#
# toolbar.py
#
# Copyright (C) 2007, 2008 Andrew Resch <andrewresch@gmail.com>
#
# Deluge is free software.
#
# You may redistribute it and/or modify it under the terms of the
# GNU General Public License, as published by the Free Software
# Foundation; either version 3 of the License, or (at your option)
# any later version.
#
# deluge is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with deluge.    If not, write to:
# 	The Free Software Foundation, Inc.,
# 	51 Franklin Street, Fifth Floor
# 	Boston, MA    02110-1301, USA.
#


import pygtk
pygtk.require('2.0')
import gtk, gtk.glade
import gobject

import deluge.component as component
from deluge.log import LOG as log
from deluge.common import TORRENT_STATE
from deluge.ui.client import aclient as client
from deluge.configmanager import ConfigManager

class ToolBar(component.Component):
    def __init__(self):
        component.Component.__init__(self, "ToolBar")
        log.debug("ToolBar Init..")
        self.window = component.get("MainWindow")
        self.toolbar = self.window.main_glade.get_widget("toolbar")
        self.config = ConfigManager("gtkui.conf")
        ### Connect Signals ###
        self.window.main_glade.signal_autoconnect({
            "on_toolbutton_add_clicked": self.on_toolbutton_add_clicked,
            "on_toolbutton_remove_clicked": self.on_toolbutton_remove_clicked,
            "on_toolbutton_pause_clicked": self.on_toolbutton_pause_clicked,
            "on_toolbutton_resume_clicked": self.on_toolbutton_resume_clicked,
            "on_toolbutton_preferences_clicked": \
                self.on_toolbutton_preferences_clicked,
            "on_toolbutton_connectionmanager_clicked": \
                self.on_toolbutton_connectionmanager_clicked,
            "on_toolbutton_queue_up_clicked": self.on_toolbutton_queue_up_clicked,
            "on_toolbutton_queue_down_clicked": self.on_toolbutton_queue_down_clicked
        })
        self.change_sensitivity = [
            "toolbutton_add",
            "toolbutton_remove",
            "toolbutton_pause",
            "toolbutton_resume",
            "toolbutton_queue_up",
            "toolbutton_queue_down"
        ]

        # Set the Remove Torrent toolbuttons drop-down menu
        tb_remove = self.window.main_glade.get_widget("toolbutton_remove")
        tb_remove.set_menu(
            component.get("MenuBar").torrentmenu_glade.get_widget("remove_torrent_menu"))

        if self.config["classic_mode"]:
            self.window.main_glade.get_widget("toolbutton_connectionmanager").hide()

        # Hide if necessary
        self.visible(self.config["show_toolbar"])

    def start(self):
        for widget in self.change_sensitivity:
            self.window.main_glade.get_widget(widget).set_sensitive(True)
        self.update_buttons()

    def stop(self):
        for widget in self.change_sensitivity:
            self.window.main_glade.get_widget(widget).set_sensitive(False)

    def visible(self, visible):
        if visible:
            self.toolbar.show()
        else:
            self.toolbar.hide()

        self.config["show_toolbar"] = visible

    def add_toolbutton(self, callback, label=None, image=None, stock=None,
                                                         tooltip=None):
        """Adds a toolbutton to the toolbar"""
        # Create the button
        toolbutton = gtk.ToolButton()
        if stock is not None:
            toolbutton.set_stock_id(stock)
        if label is not None:
            toolbutton.set_label(label)
        if image is not None:
            toolbutton.set_icon_widget(image)
        # Set the tooltip
        if tooltip is not None:
            toolbutton.set_tooltip_text(tooltip)
        # Connect the 'clicked' event callback
        toolbutton.connect("clicked", callback)

        # Append the button to the toolbar
        self.toolbar.insert(toolbutton, -1)

        # Show the new toolbutton
        toolbutton.show_all()

        return toolbutton

    def add_separator(self, position=None):
        """Adds a separator toolitem"""
        sep = gtk.SeparatorToolItem()
        if position is not None:
            self.toolbar.insert(sep, position)
        else:
            # Append the separator
            self.toolbar.insert(sep, -1)

        sep.show()

        return sep

    def remove(self, widget):
        """Removes a widget from the toolbar"""
        self.toolbar.remove(widget)

    ### Callbacks ###
    def on_toolbutton_add_clicked(self, data):
        log.debug("on_toolbutton_add_clicked")
        # Use the menubar's callback
        component.get("MenuBar").on_menuitem_addtorrent_activate(data)

    def on_toolbutton_remove_clicked(self, data):
        log.debug("on_toolbutton_remove_clicked")
        # Use the menubar's callbacks
        component.get("MenuBar").on_menuitem_remove_session_activate(data)

    def on_toolbutton_pause_clicked(self, data):
        log.debug("on_toolbutton_pause_clicked")
        # Use the menubar's callbacks
        component.get("MenuBar").on_menuitem_pause_activate(data)

    def on_toolbutton_resume_clicked(self, data):
        log.debug("on_toolbutton_resume_clicked")
        # Use the menubar's calbacks
        component.get("MenuBar").on_menuitem_resume_activate(data)

    def on_toolbutton_preferences_clicked(self, data):
        log.debug("on_toolbutton_preferences_clicked")
        # Use the menubar's callbacks
        component.get("MenuBar").on_menuitem_preferences_activate(data)

    def on_toolbutton_connectionmanager_clicked(self, data):
        log.debug("on_toolbutton_connectionmanager_clicked")
        # Use the menubar's callbacks
        component.get("MenuBar").on_menuitem_connectionmanager_activate(data)

    def on_toolbutton_queue_up_clicked(self, data):
        log.debug("on_toolbutton_queue_up_clicked")
        component.get("MenuBar").on_menuitem_queue_up_activate(data)

    def on_toolbutton_queue_down_clicked(self, data):
        log.debug("on_toolbutton_queue_down_clicked")
        component.get("MenuBar").on_menuitem_queue_down_activate(data)

    def update_buttons(self, action=None, torrent_id=None):
        if action == None:
            # If all the selected torrents are paused, then disable the 'Pause'
            # button.
            # The same goes for the 'Resume' button.
            pause = False
            resume = False

            selected = component.get('TorrentView').get_selected_torrents()
            if not selected:
                selected = []

            for torrent in selected:
                try:
                    status = component.get("TorrentView").get_torrent_status(torrent)['state']
                except KeyError, e:
                    log.debug("Error getting torrent state: %s", e)
                    continue
                if status == "Paused" or status == "Error":
                    resume = True
                else:
                    pause = True
                if pause and resume:
                    break

            # Enable the 'Remove Torrent' button only if there's some selected
            # torrent.
            remove = (len(selected) > 0)

            for name, sensitive in (("toolbutton_pause", pause),
                                    ("toolbutton_resume", resume),
                                    ("toolbutton_remove", remove)):
                self.window.main_glade.get_widget(name).set_sensitive(sensitive)


            return False

        pause = False
        resume = False
        if action == "paused":
            pause = False
            resume = True
        elif action == "resumed":
            pause = True
            resume = False
        selected = component.get('TorrentView').get_selected_torrents()
        if torrent_id == None or (torrent_id in selected and len(selected) == 1):
            self.window.main_glade.get_widget("toolbutton_pause").set_sensitive(pause)
            self.window.main_glade.get_widget("toolbutton_resume").set_sensitive(resume)
        else:
            self.update_buttons()

        return False
