// BEGIN FLOCK GPL
//
// Copyright Flock Inc. 2005-2008
// http://flock.com
//
// This file may be used under the terms of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
//
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
//
// END FLOCK GPL

const CC = Components.classes;
const CI = Components.interfaces;
const CR = Components.results;
const CU = Components.utils;

CU.import("resource:///modules/FlockXPCOMUtils.jsm");
FlockXPCOMUtils.debug = false;

CU.import("resource:///modules/FlockSvcUtils.jsm");
CU.import("resource:///modules/FlockDeliciousApi.jsm");
CU.import("resource:///modules/FlockOnlineFavsUtils.jsm");
CU.import("resource:///modules/FlockStringBundleHelpers.jsm");

const MODULE_NAME = "Magnolia";

const CLASS_NAME = "Flock Magnolia Service";
const CLASS_SHORT_NAME = "magnolia"
const CLASS_TITLE = "Magnolia"
const CLASS_ID = Components.ID("{7c9aa278-85ed-4cbe-b0bc-f35df0978ade}");
const CONTRACT_ID = "@flock.com/magnolia-service;1";

const SERVICE_ENABLED_PREF = "flock.service.magnolia.enabled";

const MAGNOLIA_STRING_BUNDLE = "chrome://flock/locale/services/magnolia.properties";
const MAGNOLIA_FAVICON = "chrome://flock/content/services/magnolia/favicon.png";
const MAGNOLIA_API_URL = "https://ma.gnolia.com/api/mirrord/v1/";
const MAGNOLIA_USER_URL_BASE = "http://ma.gnolia.com/people";

// One minute, expressed in nsITimer terms.
const TIMER_SECOND = 1000;

/**************************************************************************
 * Component: Flock Magnolia Service
 **************************************************************************/

// Constructor.
function flockMagnoliaService() {
  this._init();

  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "url");
  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "isHidden");
  FlockSvcUtils.flockIWebService.addDefaultMethod(this, "getStringBundle");

  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "loginURL");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getAccount");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getAuthenticatedAccount");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getAccounts");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "logout");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "docRepresentsSuccessfulLogin");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getAccountIDFromDocument");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getCredentialsFromForm");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "ownsDocument");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "ownsLoginForm");
  FlockSvcUtils.flockILoginWebService
               .addDefaultMethod(this, "getSessionValue");
}


/**************************************************************************
 * Flock Magnolia Service: XPCOM Component Creation
 **************************************************************************/

flockMagnoliaService.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME,
  CLASS_ID,
  CONTRACT_ID,
  flockMagnoliaService,
  CI.nsIClassInfo.SINGLETON,
  [
    CI.nsIObserver,
    CI.nsITimerCallback,
    CI.flockIWebService,
    CI.flockILoginWebService,
    CI.flockIBookmarkWebService,
    CI.flockIPollingService,
    CI.flockIMigratable
  ]
);

// FlockXPCOMUtils.genericModule() categories
flockMagnoliaService.prototype._xpcom_categories = [
  { category: "wsm-startup" },
  { category: "flockMigratable" },
  { category: "flockWebService", entry: CLASS_SHORT_NAME }
];


/**************************************************************************
 * Flock Magnolia Service: Private Data and Functions
 **************************************************************************/

// Member variables.

flockMagnoliaService.prototype._init =
function MagnoliaSvc__init() {
  FlockSvcUtils.getLogger(this).info(".init()");

  // Determine whether this service has been disabled, and unregister if so.
  prefService = CC["@mozilla.org/preferences-service;1"]
                .getService(CI.nsIPrefBranch);
  if (prefService.getPrefType(SERVICE_ENABLED_PREF) &&
     !prefService.getBoolPref(SERVICE_ENABLED_PREF))
  {
    this._logger.info("Pref " + SERVICE_ENABLED_PREF
                              + " set to FALSE... not initializing.");
    this.deleteCategories();
    return;
  }

  profiler = CC["@flock.com/profiler;1"].getService(CI.flockIProfiler);
  var evtID = profiler.profileEventStart("magnolia-init");

  var obs = CC["@mozilla.org/observer-service;1"]
            .getService(CI.nsIObserverService);
  obs.addObserver(this, "xpcom-shutdown", false);

  this._api = new DeliciousAPI(MAGNOLIA_API_URL, this._logger);
  this._accountClassCtor = flockMagnoliaAccount;
  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);
  this._c_svc = this._coop.get(this.urn);
  if (!this._c_svc) {
    this._c_svc = new this._coop.Service(this.urn, {
      name: CLASS_SHORT_NAME,
      desc: CLASS_TITLE
    });
  }
  this._c_svc.serviceId = CONTRACT_ID;
  this._c_svc.domains = FlockSvcUtils.getWD(this)
                        .getString("magnolia", "domains", "gnolia.com");

  // Initialize member that specifies path for localized string bundle
  this._stringBundlePath = MAGNOLIA_STRING_BUNDLE;

  profiler.profileEventEnd(evtID, "");
}

// Helper to retrieve tags on server.
flockMagnoliaService.prototype._getAllTags =
function MagnoliaSvc__getAllTags(aUrn, aAccountId,
                                 aPassword, aPollingListener) {
  this._logger.debug("getAllTags(...)");
  var svc = this;

  var flockListener = {
    onSuccess: function MagnoliaSvc__getAllTags_onSuccess(aSubject, aTopic) {
      svc._logger.info("tags/get SUCCEEDED: " + aTopic);

      var tags = [];
      for (var i = 0; i < aSubject.length; i++) {
        tags.push(aSubject[i].tag);
      }
      onlineFavoritesBackend.updateTags(svc, aAccountId, tags);

      // Tell the poller we're done
      if (aPollingListener) {
        aPollingListener.onResult();
      }
    },
    onError: function MagnoliaSvc__getAllTags_onError(aFlockError, aTopic) {
      svc._logger.error("tags/get FAILED");
      // BUG: 5705 -  report error via notification bar?
      // onlineFavoritesBackend.showNotification(message);
      if (aPollingListener) {
        aPollingListener.onError(aFlockError);
      }
    }
  }

  this._api.tagsGet(flockListener, null);
}


// Migrate from Cormorant, Grackle, Grouse
flockMagnoliaService.prototype._migrateFromCoop =
function MagnoliaSvc__migrateFromCoop(aContext) {
  var accounts = this._acUtils.getAccountsForService(CONTRACT_ID);

  while (accounts.hasMoreElements()) {
    var account = accounts.getNext();
    if (account instanceof CI.flockIWebServiceAccount) {
      // This call will just update the existing account
      onlineFavoritesBackend.createAccount(this,
                                           account.getParam("accountId"),
                                           false);
    }
  }

  yield true;
}

/**************************************************************************
 * Flock Magnolia Service: flockIWebService Implementation
 **************************************************************************/

// readonly attribute AString urn;
flockMagnoliaService.prototype.urn = "urn:" + CLASS_SHORT_NAME + ":service";

// readonly attribute AString title;
flockMagnoliaService.prototype.title = CLASS_TITLE;

// readonly attribute AString icon;
flockMagnoliaService.prototype.icon = MAGNOLIA_FAVICON;

// readonly attribute AString contractId;
// ALMOST duplicated by nsIClassInfo::contractID
// with different case: contractId != contractID
flockMagnoliaService.prototype.contractId = CONTRACT_ID;

// readonly attribute AString shortName;
flockMagnoliaService.prototype.shortName = CLASS_SHORT_NAME;

// readonly attribute boolean needPassword;
flockMagnoliaService.prototype.needPassword = true;

// flockIWebServiceAccount addAccount(in AString aAccountID,
//                                    in boolean aIsTransient,
//                                    in flockIListener aFlockListener);
flockMagnoliaService.prototype.addAccount =
function MagnoliaSvc_addAccount(aAccountID, aIsTransient, aFlockListener) {
  this._logger.debug("addAccount('" + aAccountID + "', "
                     + aIsTransient + ", aFlockListener)");
  var acct = onlineFavoritesBackend.createAccount(this,
                                                  aAccountID, aIsTransient);
  if (aFlockListener) {
    aFlockListener.onSuccess(acct, "addAccount");
  }
  return acct;
}

// void removeAccount(in AString aUrn);
flockMagnoliaService.prototype.removeAccount =
function MagnoliaSvc_removeAccount(aUrn) {
  this._logger.debug("removeAccount('" + aUrn + "')");
  onlineFavoritesBackend.removeAccount(this, aUrn);
}

// DEFAULT: flockIWebServiceAccount getAccount(in AString aUrn);
// DEFAULT: nsISimpleEnumerator getAccounts();
// DEFAULT: void logout();


/**************************************************************************
 * Flock Magnolia Service: flockILoginWebService Implementation
 **************************************************************************/

// DEFAULT: boolean docRepresentsSuccessfulLogin(in nsIDOMHTMLDocument aDocument);
// DEFAULT: AString getAccountIDFromDocument(in nsIDOMHTMLDocument aDocument);
// DEFAULT: flockILoginInfo getCredentialsFromForm(in nsIDOMHTMLFormElement aForm);
// DEFAULT: boolean ownsDocument(in nsIDOMHTMLDocument aDocument);
// DEFAULT: boolean ownsLoginForm(in nsIDOMHTMLFormElement aForm);

/**
 * @see flockILoginWebService#updateAccountStatusFromDocument
 */
flockMagnoliaService.prototype.updateAccountStatusFromDocument =
function MagnoliaSvc_updateAccountStatusFromDocument(aDocument,
                                                     aAcctURN,
                                                     aFlockListener)
{
  this._logger.debug("updateAccountStatusFromDocument('" + aAcctURN + "')");
  if (aAcctURN) {
    // We're logged in to this account
    if (!this._acUtils.ensureOnlyAuthenticatedAccount(aAcctURN)) {
      if (aFlockListener) {
        aFlockListener.onSuccess(this.getAccount(aAcctURN), null);
      }
    }
  } else if (this._WebDetective
                 .detect("magnolia", "loggedout", aDocument, null))
  {
    // We're logged out (of all accounts)
    this._acUtils.markAllAccountsAsLoggedOut(CONTRACT_ID);
  }
}


/**************************************************************************
 * Flock Magnolia Service: flockIBookmarkWebService Implementation
 **************************************************************************/

// void publish (in flockIListener aFlockListener, in AString aAccountId,
//               in flockIBookmark aBookmark);
flockMagnoliaService.prototype.publish =
function MagnoliaSvc_publish(aFlockListener, aAccountId, aBookmark) {
  this._logger.info("Publish(" + aBookmark.URL + "," + aBookmark.name
                    + ") to " + aAccountId + "@Magnolia");
  var accountUrn = this.urn + ":" + aAccountId;
  var password = this._acUtils.getPassword(accountUrn);
  var svc = this;
  var tags = "";
  if (aBookmark.tags) {
    // Remove the FLOCK_UNFILED tag, that we add to favorites that have no tag.
    // Theorically favorites shouldn't have FLOCK_UNFILED with other tags,
    // but even when that happen we want to be sure not to push
    // FLOCK_UNFILED to the service.
    var tagsArray = aBookmark.tags.split(/[\s,]/).sort();
    if (tagsArray[0] && tagsArray[0] == onlineFavoritesBackend.FLOCK_UNFILED) {
      tagsArray.pop();
    }
    tags = tagsArray.join(",").replace(/^ /, "");
  }
  var args = {
    url: aBookmark.URL,
    description: aBookmark.name,
    extended: aBookmark.description,
    tags: tags,
    shared: aBookmark.shared ? "yes" : "no"
  };

  var deliciousApiListener = {
    onError: function MagnoliaSvc_publish_onError(aFlockError) {
      svc._logger.error("posts/add FAILED");
      // BUG: 5705 -  report error via notification bar?
      // onlineFavoritesBackend.showNotification(message);
      if (aFlockListener) {
        aFlockListener.onError(aFlockError, "publish");
      }
    },
    onSuccess: function MagnoliaSvc_publish_onSuccess(aXML) {
      svc._logger.info("posts/add SUCCEEDED");

      // Validate the nsIDOMDocument response.
      if (!svc._api.isExpectedResponse(aXML, "result")) {
        svc._logger.error("posts/add succeeded - but with invalid xml response");
        // BUG: 5705 -  report error via notification bar?
        // onlineFavoritesBackend.showNotification(message);
        if (aFlockListener) {
          var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
          // FIXME: fill in this flockIError.
          error.errorCode = CI.flockIError.FAVS_UNKNOWN_ERROR;
          aFlockListener.onError(error, "publish");
        }
        return;
      } else {
        // Process the validated response.
        var result = aXML.documentElement.getAttribute("code");
        svc._logger.debug(" Result for posts/add: " + result);
        if (result == "done") {
          var account = svc.getAccount(accountUrn);
          onlineFavoritesBackend.updateBookmark(svc,
                                                account,
                                                aBookmark);
          if (aFlockListener) {
            aFlockListener.onSuccess(svc, "publish");
          }
        } else if (aFlockListener) {
          var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
          // FIXME: fill in this flockIError.
          error.errorCode = CI.flockIError.FAVS_UNKNOWN_ERROR;
          aFlockListener.onError(error, "publish");
        }
      }
    }
  }

  this._api.call("posts/add", args, deliciousApiListener, password);
}

// void publishList (in flockIListener aFlockListener, in AString aAccountId,
//                   in nsISimpleEnumerator aBookmarkList);
flockMagnoliaService.prototype.publishList =
function MagnoliaSvc_publishList(aFlockListener, aAccountId, aBookmarkList) {
  var svc = this;
  var i = 0;
  var bookmarks = [];

  var syncCallback = {
    notify: function publishListCallback_notify() {
      var bm = bookmarks.shift();
      svc.publish(aFlockListener, aAccountId, bm);
    }
  }

  while (aBookmarkList.hasMoreElements()) {
    var bookmark = aBookmarkList.getNext().QueryInterface(CI.flockIBookmark);
    // Duplicate it because it's going to be removed too early
    bookmarks[i] = {};
    bookmarks[i].URL = bookmark.URL;
    bookmarks[i].name = bookmark.name;
    bookmarks[i].description = bookmark.description;
    bookmarks[i].tags = bookmark.tags;
    bookmarks[i].time = bookmark.time;
    bookmarks[i].shared = bookmark.shared;

    this._logger.debug("Set a timer to " + i * TIMER_SECOND
                       + " for " + bookmarks[i].URL);
    var publishTimer = CC["@mozilla.org/timer;1"]
                       .createInstance(CI.nsITimer);
    // 1 second between each publish
    publishTimer.initWithCallback(syncCallback, i * TIMER_SECOND,
                                  CI.nsITimer.TYPE_ONE_SHOT);
    i++;
  }
}

// void remove (in flockIListener aFlockListener, in AString aAccountId,
//              in AString aBookmark);
flockMagnoliaService.prototype.remove =
function MagnoliaSvc_remove(aFlockListener, aAccountId, aBookmark) {
  this._logger.info("Remove " + aBookmark + " from "
                    + aAccountId + "@Magnolia");
  var password = this._acUtils.getPassword(this.urn + ":" + aAccountId);
  var svc = this;
  var args = { url: aBookmark };

  var deliciousApiListener = {
    onError: function MagnoliaSvc_remove_onError(aFlockError) {
      svc._logger.error("posts/delete FAILED");
      // BUG: 5705 -  report error via notification bar?
      // onlineFavoritesBackend.showNotification(message);
      if (aFlockListener) {
        aFlockListener.onError(aFlockError, svc);
      }
    },
    onSuccess: function MagnoliaSvc_remove_onSuccess(aXML) {
      svc._logger.info("posts/delete SUCCEEDED");

      // Validate the nsIDOMDocument response.
      if (!svc._api.isExpectedResponse(aXML, "result")) {
        svc._logger.error("posts/delete succeeded - but with invalid xml response");
        // BUG: 5705 -  report error via notification bar?
        // onlineFavoritesBackend.showNotification(message);
        if (aFlockListener) {
          var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
          // FIXME: fill in this flockIError.
          error.errorCode = CI.flockIError.FAVS_UNKNOWN_ERROR;
          aFlockListener.onError(error, svc);
        }
      } else {
        // Process the validated response.
        var result = aXML.documentElement.getAttribute("code");
        svc._logger.debug("Result for posts/delete: " + result);
        if (result == "done") {
          onlineFavoritesBackend.destroyBookmark(svc, aAccountId, aBookmark);
          CC["@flock.com/poller-service;1"].getService(CI.flockIPollerService)
          .forceRefresh(this.urn + ":" + aAccountId + ":bookmarks");
          if (aFlockListener) {
            aFlockListener.onSuccess(svc, "remove");
          }
        } else if (aFlockListener) {
          var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
          // FIXME: fill in this flockIError.
          error.errorCode = CI.flockIError.FAVS_UNKNOWN_ERROR;
          aFlockListener.onError(error, "remove");
        }
      }
    }
  }

  this._api.call("posts/delete", args, deliciousApiListener, password);
}

// boolean exists (in AString aAccountId, in AString aURL);
flockMagnoliaService.prototype.exists =
function MagnoliaSvc_exists(aAccountId, aURL) {
  return onlineFavoritesBackend.favoriteExists(this, aAccountId, aURL);
}

// boolean isPrivate (in AString aAccountId, in AString aURL);
flockMagnoliaService.prototype.isPrivate =
function MagnoliaSvc_isPrivate(aAccountId, aURL) {
  return onlineFavoritesBackend.isPrivate(this, aAccountId, aURL);
}

// AString getUserURL (in AString aAccountId);
flockMagnoliaService.prototype.getUserUrl =
function MagnoliaSvc_getUserUrl(aAccountId) {
  return MAGNOLIA_USER_URL_BASE + "/" + aAccountId;
}


/**************************************************************************
 * Flock Magnolia Service: flockIPollingService Implementation
 **************************************************************************/

// void refresh(in AString aUrn, in flockIPollingListener aPollingListener);
flockMagnoliaService.prototype.refresh =
function MagnoliaSvc_refresh(aURN, aPollingListener) {
  var svc = this;
  this._logger.info("refresh(" + aURN + ")");

  if (!this._coop.Account.exists(aURN)) {
    throw "flockMagnoliaService: " + aURN + " can't be found";
  }

  var coopAccount = this._coop.get(aURN);
  var accountId = coopAccount.accountId;
  var accountUrn = this.urn + ":" + accountId;
  // nsILoginInfo for auth for this sync
  var password = this._acUtils.getPassword(accountUrn);

  var flockListener = {
    onSuccess: function MagnoliaSvc_refresh_onSuccess(aSubject, aTopic) {
      svc._logger.info("posts/all SUCCEEDED: " + aTopic);

      if (aTopic != "nonew") {
        onlineFavoritesBackend.updateLocal(svc, aSubject, aTopic, accountUrn);
      }
      aPollingListener.onResult();
    },
    onError: function MagnoliaSvc_refresh_onError(aFlockError, aTopic) {
      svc._logger.error("posts/all FAILED: ["
                        + aFlockError.errorCode + "] "
                        + aFlockError.errorString + " (["
                        + aFlockError.serviceErrorCode + "] "
                        + aFlockError.serviceErrorString + ")");
      if (aPollingListener) {
        aPollingListener.onError(aFlockError);
      }
    }
  }

  // even for the first refresh, we need to get the last update time from 
  // the server. Bruno
  var lastUpdate = coopAccount.accountParamList
                              .getParam("last_update_time")
                              .value;
  this._api.postsUpdate(flockListener,
                        password,
                        lastUpdate);
}

/**************************************************************************
 * Flock Magnolia Service: flockIMigratable Implementation
 **************************************************************************/

flockMagnoliaService.prototype.__defineGetter__("migrationName",
function MagnoliaSvc_getter_migrationName() {
  return flockGetString("common/migrationNames", "migration.name.magnolia");
});

// boolean needsMigration(in string aOldVersion);
flockMagnoliaService.prototype.needsMigration =
function MagnoliaSvc_needsMigration(aOldVersion) {
  var major = parseInt(aOldVersion[0], 10);
  var middle = parseInt(aOldVersion[2], 10); // aOldVersion[1] == "."

  if (major == 0 && middle == 7) {
    // Cardinal: Magnolia was not supported
    return false;
  } else if ((major == 0) || // Cormorant
             (major == 1 && middle <= 2)) // Grackle, Grouse
  {
    return true;
  }
  return false;
}

// nsISupports startMigration(in string aOldVersion,
//                            in flockIMigrationProgressListener aListener);
flockMagnoliaService.prototype.startMigration =
function DeliciousSvc_startMigration(aOldVersion,
                                     aFlockMigrationProgressListener)
{
  var ctxt = {
    oldVersion: aOldVersion,
    listener: aFlockMigrationProgressListener
  };

  return { wrappedJSObject: ctxt };
}

// boolean doMigrationWork(in nsISupports aMigrationContext);
flockMagnoliaService.prototype.doMigrationWork =
function DeliciousSvc_doMigrationWork(aMigrationContext) {
  var context = aMigrationContext.wrappedJSObject;

  if (!context.magnoliaMigrator) {
    context.magnoliaMigrator = this._migrateFromCoop(context);
  }
  if (context.magnoliaMigrator.next()) {
    context.magnoliaMigrator = null;
  }

  return Boolean(context.magnoliaMigrator);
}

// void finishMigration(in nsISupports aMigrationContext);
flockMagnoliaService.prototype.finishMigration =
function DeliciousSvc_finishMigration(aMigrationContext) {
}

/**************************************************************************
 * Flock Magnolia Service: nsIObserver Implementation
 **************************************************************************/

flockMagnoliaService.prototype.observe =
function MagnoliaSvc_observe(aSubject, aTopic, aState) {
  switch (aTopic) {
    case "xpcom-shutdown":
      var obs = CC["@mozilla.org/observer-service;1"]
                .getService(CI.nsIObserverService);
      obs.removeObserver(this, "xpcom-shutdown");
      break;
  }
}

/**************************************************************************
 * Component: Flock Magnolia Account
 **************************************************************************/

function flockMagnoliaAccount() {
  FlockSvcUtils.getLogger(this).info(".init()");
  FlockSvcUtils.getACUtils(this);
  FlockSvcUtils.getCoop(this);

  var wsa = FlockSvcUtils.flockIWebServiceAccount;
  wsa.addDefaultMethod(this, "getService");
  wsa.addDefaultMethod(this, "login");
  wsa.addDefaultMethod(this, "logout");
  wsa.addDefaultMethod(this, "getParam");
  wsa.addDefaultMethod(this, "setParam");
  wsa.addDefaultMethod(this, "getCustomParam");
  wsa.addDefaultMethod(this, "getAllCustomParams");
  wsa.addDefaultMethod(this, "setCustomParam");
  wsa.addDefaultMethod(this, "isAuthenticated");
}

/**************************************************************************
 * Flock Magnolia Account: XPCOM Component Creation
 **************************************************************************/

// Use genericComponent() for the goodness it provides (QI, nsIClassInfo, etc),
// but do NOT add flockMagnoliaAccount to the list of constructors passed
// to FlockXPCOMUtils.genericModule().
flockMagnoliaAccount.prototype = new FlockXPCOMUtils.genericComponent(
  CLASS_NAME + " Account",
  "",
  "",
  flockMagnoliaAccount,
  0,
  [
    CI.flockIWebServiceAccount,
    CI.flockIBookmarkAccount
  ]
  );

/**************************************************************************
 * Flock Magnolia Account: flockIWebServiceAccount Implementation
 **************************************************************************/

// readonly attribute AString urn;
flockMagnoliaAccount.prototype.urn = "";

// DEFAULT: void login(in flockIListener aListener);
// DEFAULT: void logout(in flockIListener aListener);

// void keep();
flockMagnoliaAccount.prototype.keep =
function MagnoliaAcct_keep() {
  this._logger.debug("keep()");
  this._coop.get(this.urn).isTransient = false;
  this._acUtils.makeTempPasswordPermanent(this.urn);
}


/**************************************************************************
 * Flock Magnolia Account: flockIBookmarkAccount Implementation
 **************************************************************************/

/* No methods or properties on this interface! */


/**************************************************************************
 * XPCOM Support - Module Construction
 **************************************************************************/

// Create array of components.
var gComponentsArray = [flockMagnoliaService];

// Generate a module for XPCOM to find.
var NSGetModule = FlockXPCOMUtils.generateNSGetModule(MODULE_NAME,
                                                      gComponentsArray);

/**************************************************************************
 * END XPCOM Support
 **************************************************************************/
