// BEGIN FLOCK GPL
// 
// Copyright Flock Inc. 2005-2007
// http://flock.com
// 
// This file may be used under the terms of of the
// GNU General Public License Version 2 or later (the "GPL"),
// http://www.gnu.org/licenses/gpl.html
// 
// Software distributed under the License is distributed on an "AS IS" basis,
// WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
// for the specific language governing rights and limitations under the
// License.
// 
// END FLOCK GPL

const CC = Components.classes;
const CI = Components.interfaces;

var EXPORTED_SYMBOLS = ["FlockPrefsUtils"];

/**
 * A set of helper wrapping Mozilla's preferences service.
 * In particular, the preference service throws exceptions when trying
 * to access a value that is not set. This modules returns null instead.
 */

var FlockPrefsUtils = {

  _getPrefSVC: function FPU__getPrefSVC() {
    if (!this._prefSVC) {
      this._prefSVC = CC['@mozilla.org/preferences-service;1']
                      .getService(CI.nsIPrefBranch);
    }
    return this._prefSVC;
  },

  clearPref: function FPU_clearPref(aKey) {
    var psvc = this._getPrefSVC();
    psvc.clearUserPref(aKey);
    psvc.deleteBranch(aKey);
  },

  getCharPref: function FPU_getCharPref(aKey) {
    try {
      return this._getPrefSVC().getCharPref(aKey);
    } catch (ex) {
      // The preference is not set
      return null;
    }
  },

  setCharPref: function FPU_setCharPref(aKey, aValue) {
    this._getPrefSVC().setCharPref(aKey, aValue);
  },

  getIntPref: function FPU_getIntPref(aKey) {
    try {
      return this._getPrefSVC().getIntPref(aKey);
    } catch (ex) {
      // The preference is not set
      return null;
    }
  },

  setIntPref: function FPU_setIntPref(aKey, aValue) {
    this._getPrefSVC().setIntPref(aKey, aValue);
  },

  getBoolPref: function FPU_getBoolPref(aKey) {
    try {
      return this._getPrefSVC().getBoolPref(aKey);
    } catch (ex) {
      // The preference is not set
      return null;
    }
  },

  getLocalizedPref: function FPU_getLocalizedPref(aPref, aDefault) {
    try {
      return this._getPrefSVC()
                 .getComplexValue(aPref, CI.nsIPrefLocalizedString).data;
    } catch (ex) {
      // Fall through...
    }
    try {
      return this._getPrefSVC().getCharPref(aPref);
    } catch (ex) {
      // Fall through...
    }
    return aDefault;
  },

  setLocalizedPref: function FPU_setLocalizedPref(aPref, aValue) {
    try {
      var pls = CC["@mozilla.org/pref-localizedstring;1"]
                .createInstance(CI.nsIPrefLocalizedString);
      pls.data = aValue;
      this._getPrefSVC().setComplexValue(aPref, CI.nsIPrefLocalizedString, pls);
    } catch (ex) {
      // TODO XXX: This probably doesn't need to be in a try/catch.
      //           Review and remove in a general cleanup later.
    }
  },

  setBoolPref: function FPU_setBoolPref(aKey, aValue) {
    this._getPrefSVC().setBoolPref(aKey, aValue);
  },

  prefHasUserValue: function FPU_prefHasUserValue(aKey) {
    return this._getPrefSVC().prefHasUserValue(aKey);
  }

}
